/* Copyright (c) 2019-2024 Advanced Micro Devices, Inc.

 Permission is hereby granted, free of charge, to any person obtaining a copy
 of this software and associated documentation files (the "Software"), to deal
 in the Software without restriction, including without limitation the rights
 to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 copies of the Software, and to permit persons to whom the Software is
 furnished to do so, subject to the following conditions:

 The above copyright notice and this permission notice shall be included in
 all copies or substantial portions of the Software.

 THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 THE SOFTWARE. */

/** \mainpage AMD Debugger API Specification
 *
 * \section introduction Introduction
 *
 * The amd-dbgapi is a library that implements an AMD GPU debugger application
 * programming interface (API).  It provides the support necessary for a client
 * of the library to control the execution and inspect the state of supported
 * commercially available AMD GPU devices.
 *
 * The term \e client is used to refer to the application that uses this API.
 *
 * The term \e library is used to refer to the implementation of this interface
 * being used by the client.
 *
 * The term <em>AMD GPU</em> is used to refer to commercially available AMD GPU
 * devices supported by the library.
 *
 * The term \e inferior is used to refer to the process being debugged.
 *
 * The library does not provide any operations to perform symbolic mappings,
 * code object decoding, or stack unwinding.  The client must use the AMD GPU
 * code object ELF ABI defined in [User Guide for AMDGPU Backend - Code Object]
 * (https://llvm.org/docs/AMDGPUUsage.html#code-object), together with the AMD
 * GPU debug information DWARF and call frame information CFI ABI define in
 * [User Guide for AMDGPU Backend - Code Object - DWARF]
 * (https://llvm.org/docs/AMDGPUUsage.html#dwarf) to perform those tasks.
 *
 * The library does not provide operations for inserting or managing
 * breakpoints.  The client must write the architecture specific breakpoint
 * instruction provided by the
 * ::AMD_DBGAPI_ARCHITECTURE_INFO_BREAKPOINT_INSTRUCTION query into the loaded
 * code object memory to set breakpoints.  For resuming from breakpoints the
 * client must use the displaced stepping mechanism provided by
 * ::amd_dbgapi_displaced_stepping_start and
 * ::amd_dbgapi_displaced_stepping_complete in conjunction with the
 * ::amd_dbgapi_wave_resume in single step mode.  In order to determine the
 * location of stopped waves the client must read the architecture specific
 * program counter register available using the
 * ::AMD_DBGAPI_ARCHITECTURE_INFO_PC_REGISTER query and adjust it by the amount
 * specified by the
 * ::AMD_DBGAPI_ARCHITECTURE_INFO_BREAKPOINT_INSTRUCTION_PC_ADJUST query.
 *
 * The client is responsible for checking that only a single thread at a time
 * invokes a function provided by the library.  A callback (see \ref
 * callbacks_group) invoked by the library must not itself invoke any function
 * provided by the library.
 *
 * The library implementation uses the native operating system to inspect and
 * control the inferior.  Therefore, the library must be executed on the same
 * machine as the inferior.
 *
 * A library instance is defined as the period between a call to
 * ::amd_dbgapi_initialize and a matching call to ::amd_dbgapi_finalize.
 *
 * The library uses opaque handles to refer to the entities that it manages.  A
 * handle value should not be modified directly.  See the handle definitions
 * for information on the lifetime and scope of handles of that type.  Handles
 * are invalidated outside their lifetime, scope, or single library instance.
 * If a handle is returned by an operation in one library instance which then
 * becomes invalidated, then any operation using that handle in the same
 * library instance will return an invalid handle error code.  However, it is
 * undefined to use a handle created by an operation in one library instance in
 * the operations of another library instance.  A handle value is globally
 * unique within each library instance.  This is true even if the handle
 * becomes invalidated: handle values are not reused within a library instance.
 * Every handle with \p handle value of 0 is reserved to indicate the handle
 * does not reference an entity.
 *
 * When the library is first loaded it is in the uninitialized state with the
 * logging level set to ::AMD_DBGAPI_LOG_LEVEL_NONE.
 *
 * \section amd_gpu_execution_model AMD GPU Execution Model
 *
 * In this section the AMD GPU execution model is described to provide
 * background to the reader if they are not familiar with this environment.
 * The AMD GPU execution model is more complicated than that of a traditional
 * CPU because of how GPU hardware is used to accelerate and schedule the very
 * large number of threads of execution that are created on GPUs.
 *
 * Chapter 2 of the [HSA Programmer's Reference Manual][hsa-prm] provides an
 * introduction to this execution model.  Note that the AMD ROCm compilers
 * compile directly to ISA and do not use the HSAIL intermediate language.
 * However, the ROCr low-level runtime and ROCgdb debugger use the same
 * terminology.
 *
 * In this model, a CPU process may interact with multiple AMD GPU devices,
 * which are termed agents.  A Process Address Space Identifier (PASID) is
 * created for each process that interacts with agents.  An agent can be
 * executing code for multiple processes at once.  This is achieved by mapping
 * the PASID to one of a limited set of Virtual Memory Identifiers (VMIDs).
 * Each VMID is associated with its own page table.
 *
 * The AMD GPU device driver for Linux, termed the Kernel Mode Driver (KMD),
 * manages the page tables used by each GPU so they correlate with the CPU page
 * table for the corresponding process.  The CPU and GPU page tables do not
 * necessarily map all the same memory pages but pages they do have in common
 * have the same virtual address.  Therefore, the CPU and GPUs have a unified
 * address space.
 *
 * Each GPU includes one or more Microcode Engines (ME) that can execute
 * microcode firmware.  This firmware includes a Hardware Scheduler (HWS) that,
 * in collaboration with the KMD, manages which processes, identified by a
 * PASID, are mapped onto the GPU using one of the limited VMIDs.  This mapping
 * configures the VMID to use the GPU page table that corresponds to the PASID.
 * In this way, the code executing on the GPU from different processes is
 * isolated.
 *
 * Multiple software submission queues may be created for each agent.  The GPU
 * hardware has a limited number of pipes, each of which has a fixed number of
 * hardware queues.  The HWS, in collaboration with the KMD, is responsible for
 * mapping software queues onto hardware queues.  This is done by multiplexing
 * the software queues onto hardware queues using time slicing.  The software
 * queues provide a virtualized abstraction, allowing for more queues than are
 * directly supported by the hardware.  Each ME manages its own set of pipes
 * and their associated hardware queues.
 *
 * To execute code on the GPU, a packet must be created and placed in a
 * software queue.  This is achieved using regular user space atomic memory
 * operations.  No Linux kernel call is required.  For this reason, the queues
 * are termed user mode queues.
 *
 * The AMD ROCm platform uses the Asynchronous Queuing Language (AQL) packet
 * format defined in the [HSA Platform System Architecture
 * Specification][hsa-sysarch].  Packets can request GPU management actions
 * (for example, manage memory coherence) and the execution of kernel
 * functions.  The ME firmware includes the Command Processor (CP) which,
 * together with fixed-function hardware support, is responsible for detecting
 * when packets are added to software queues that are mapped to hardware
 * queues.  Once detected, CP is responsible for initiating actions requested
 * by the packet, using the appropriate VMID when performing all memory
 * operations.
 *
 * Dispatch packets are used to request the execution of a kernel function.
 * Each dispatch packet specifies the address of a kernel descriptor, the
 * address of the kernel argument block holding the arguments to the kernel
 * function, and the number of threads of execution to create to execute the
 * kernel function.  The kernel descriptor describes how the CP must configure
 * the hardware to execute the kernel function and the starting address of the
 * kernel function code.  The compiler generates a kernel descriptor in the
 * code object for each kernel function and determines the kernel argument
 * block layout.  The number of threads of execution is specified as a grid,
 * such that each thread of execution can identify its position in the grid.
 * Conceptually, each of these threads executes the same kernel code, with the
 * same arguments.
 *
 * The dispatch grid is organized as a three-dimensional collection of
 * workgroups, where each workgroup is the same size (except for potential
 * boundary partial workgroups).  The workgroups form a three-dimensional
 * collection of work-items.  The work-items are the threads of execution.  The
 * position of a work-item is its zero-based three-dimensional position in a
 * workgroup, termed its work-item ID, plus its workgroup's three-dimensional
 * position in the dispatch grid, termed its workgroup ID.  These
 * three-dimensional IDs can also be expressed as a zero-based one-dimensional
 * ID, termed a flat ID, by simply numbering the elements in a natural manner
 * akin to linearizing a multi-dimensional array.
 *
 * Consecutive work-items, in flat work-item ID order, of a workgroup are
 * organized into fixed size wavefronts, or waves for short.  Each work-item
 * position in the wave is termed a lane, and has a zero-base lane ID.  The
 * hardware imposes an upper limit on the number of work-items in a workgroup
 * but does not limit the number of workgroups in a dispatch grid.  The
 * hardware executes instructions for waves independently.  But the lanes of a
 * wave all execute the same instruction jointly.  This is termed Single
 * Instruction Multiple Thread (SIMT) execution.
 *
 * Each hardware wave has a set of registers that are shared by all lanes of
 * the wave, termed scalar registers.  There is only one set of scalar
 * registers for the whole wave.  Instructions that act on the whole wave,
 * which typically use scalar registers, are termed scalar instructions.
 *
 * Additionally, each wave also has a set of vector registers that are
 * replicated so each lane has its own copy.  A set of vector registers can be
 * viewed as a vector with each element of the vector belonging to the
 * corresponding lane of the wave.  Instructions that act on vector registers,
 * which produce independent results for each lane, are termed vector
 * instructions.
 *
 * Each hardware wave has an execution mask that controls if the execution of a
 * vector instruction should change the state of a particular lane.  If the
 * lane is masked off, no changes are made for that lane and the instruction is
 * effectively ignored.  The compiler generates code to update the execution
 * mask which emulates independent work-item execution.  However, the lanes of
 * a wave do not execute instructions independently.  If two subsets of lanes
 * in a wave need to execute different code, the compiler will generate code to
 * set the execution mask to execute the subset of lanes for one path, then
 * generate instructions for that path.  The compiler will then generate code
 * to change the execution mask to enable the other subset of lanes, then
 * generate code for those lanes.  If both subsets of lanes execute the same
 * code, the compiler will generate code to set the execution mask to include
 * both subsets of lanes, then generate code as usual.  When only a subset of
 * lanes is enabled, they are said to be executing divergent control flow.
 * When all lanes are enabled, they are said to be executing wave uniform
 * control flow.
 *
 * Not all MEs have the hardware to execute kernel functions.  One such ME is
 * used to execute the HWS microcode and to execute microcode that manages a
 * service queue that is used to update GPU state.  If the ME does support
 * kernel function execution it uses fixed-function hardware to initiate the
 * creation of waves.  This is accomplished by sending requests to create
 * workgroups to one or more Compute Units (CUs).  Requests are sent to create
 * all the workgroups of a dispatch grid.  Each CU has resources to hold a
 * fixed number of waves and has fixed-function hardware to schedule execution
 * of these waves.  The scheduler may execute multiple waves concurrently and
 * will hide latency by switching between the waves that are ready to execute.
 * At any point of time, a subset of the waves belonging to workgroups in a
 * dispatch may be actively executing.  As waves complete, the waves of
 * subsequent workgroup requests are created.
 *
 * Each CU has a fixed amount of memory from which it allocates vector and
 * scalar registers.  The kernel descriptor specifies how many registers to
 * allocate for a wave.  There is a tradeoff between how many waves can be
 * created on a CU and the number of registers each can use.
 *
 * The CU also has a fixed size Local Data Store (LDS).  A dispatch packet
 * specifies how much LDS each workgroup is allocated.  All waves in a
 * workgroup are created on the same CU.  This allows the LDS to be used to
 * share data between the waves of the same workgroup.  There is a tradeoff
 * between how much LDS a workgroup can allocate, and the number of workgroups
 * that can fit on a CU.  The address of a location in a workgroup LDS
 * allocation is zero-based and is a different address space than the global
 * virtual memory.  There are specific instructions that take an LDS address to
 * access it.  There are also flat address instructions that map the LDS
 * address range into an unused fixed aperture range of the global virtual
 * address range.  An LDS address can be converted to or from a flat address by
 * offsetting by the base of the aperture.  Note that a flat address in the LDS
 * aperture only accesses the LDS workgroup allocation for the wave that uses
 * it.  The same address will access different LDS allocations if used by waves
 * in different workgroups.
 *
 * The dispatch packet specifies the amount of scratch memory that must be
 * allocated for a work-item.  This is used for work-item private memory.
 * Fixed-function hardware in the CU manages per wave allocation of scratch
 * memory from pre-allocated global virtual memory mapped to GPU device memory.
 * Like an LDS address, a scratch address is zero-based, but is per work-item
 * instead of per workgroup.  It maps to an aperture in a flat address.  The
 * hardware swizzles this address so that adjacent lanes access adjacent DWORDs
 * (4 bytes) in global memory for better cache performance.
 *
 * For an AMD Radeon Instinct™ MI60 GPU the workgroup size limit is 1,024
 * work-items, the wave size is 64, and the CU count is 64.  A CU can hold up
 * to 40 waves (this is limited to 32 if using scratch memory).  Therefore, a
 * workgroup can comprise between 1 and 16 waves inclusive, and there can be up
 * to 2,560 waves, making a maximum of 163,840 work-items.  A CU is organized
 * as 4 Execution Units (EUs) also referred to as Single Instruction Multiple
 * Data units (SIMDs) that can each hold 10 waves.  Each SIMD has 256 64-wide
 * DWORD vector registers and each CU has 800 DWORD scalar registers.  A single
 * wave can access up to 256 64-wide vector registers and 112 scalar registers.
 * A CU has 64KiB of LDS.
 *
 * \section supported_amd_gpu_architectures Supported AMD GPU Architectures
 *
 * The following AMD GPU architectures are supported:
 *
 * - gfx900 (AMD Vega 10)
 * - gfx906 (AMD Vega 7nm also referred to as AMD Vega 20)
 * - gfx908 (AMD Instinct™ MI100 accelerator)
 * - gfx90a (Aldebaran)
 * - gfx940 (Aqua Vanjaram)
 * - gfx941
 * - gfx942
 * - gfx1010 (Navi10)
 * - gfx1011 (Navi12)
 * - gfx1012 (Navi14)
 * - gfx1030 (Sienna Cichlid)
 * - gfx1031 (Navy Flounder)
 * - gfx1032 (Dimgrey Cavefish)
 * - gfx1100 (Plum Bonito)
 * - gfx1101 (Wheat Nas)
 * - gfx1102 (Hotpink Bonefish)
 * - gfx1200
 * - gfx1201
 *
 * The following generic AMD GPU architectures are supported:
 *
 * - gfx9-generic
 * - gfx9-4-generic
 * - gfx10-1-generic
 * - gfx10-3-generic
 * - gfx11-generic
 * - gfx12-generic
 *
 * For more information about the AMD ROCm ecosystem, please refer to:
 *
 * - https://rocm.docs.amd.com/
 *
 * \section known_limitations Known Limitations and Restrictions
 *
 * The AMD Debugger API library implementation currently has the following
 * restrictions.  Future releases aim to address these restrictions.
 *
 * 1. The following *_get_info queries are not yet implemented:
 *
 *    - ::AMD_DBGAPI_QUEUE_INFO_ERROR_REASON
 *    - ::AMD_DBGAPI_QUEUE_INFO_STATE
 *
 * 2. On a ::AMD_DBGAPI_STATUS_FATAL error the library does fully reset the
 *    internal state and so subsequent functions may not operate correctly.
 *
 * 3. ::amd_dbgapi_process_next_pending_event returns
 *    ::AMD_DBGAPI_EVENT_KIND_WAVE_STOP events only for AQL queues.  PM4 queues
 *    that launch wavefronts are not supported.
 *
 * 4. ::amd_dbgapi_queue_packet_list returns packets only for AQL queues.
 *
 * 5. Generation of the ::AMD_DBGAPI_EVENT_KIND_QUEUE_ERROR event, the
 *    ::AMD_DBGAPI_EVENT_INFO_QUEUE query, and the generation of
 *    ::AMD_DBGAPI_EVENT_KIND_WAVE_COMMAND_TERMINATED events for waves with
 *    pending single step requests when a queue enters the queue error state,
 *    have not been implemented.
 *
 * 6. By default, for some architectures, the AMD GPU device driver for Linux
 *    causes all wavefronts created when the library is not attached to the
 *    process by ::amd_dbgapi_process_attach to be unable to query the
 *    wavesfront's ::AMD_DBGAPI_WAVE_INFO_DISPATCH,
 *    ::AMD_DBGAPI_WAVE_INFO_WORKGROUP_COORD, or
 *    ::AMD_DBGAPI_WAVE_INFO_WAVE_NUMBER_IN_WORKGROUP, or workgroup's
 *    ::AMD_DBGAPI_WORKGROUP_INFO_DISPATCH, or
 *    ::AMD_DBGAPI_WORKGROUP_INFO_WORKGROUP_COORD.  This does not affect
 *    wavefronts and workgroups created while the library is attached to the
 *    process which are always capable of reporting this information.
 *
 *    If the \p HSA_ENABLE_DEBUG environment variable is set to "1" when the
 *    inferior's runtime is successfully enabled (see
 *    ::AMD_DBGAPI_EVENT_KIND_RUNTIME), then this information will be available
 *    for all architecture even for wavefronts created when the library was not
 *    attached to the process.  Setting this environment variable may very
 *    marginally reduce wavefront launch latency for some architectures for
 *    very short lived wavefronts.
 *
 *    \sa ::amd_dbgapi_wave_get_info
 *
 * 7. The \p AMD_DBGAPI_WAVE_STOP_REASON_FP_* and \p
 *    AMD_DBGAPI_WAVE_STOP_REASON_INT-* stop reasons (see
 *    ::amd_dbgapi_wave_stop_reasons_t) are not reported for enabled arithmetic
 *    exceptions if the \p DX10_CLAMP bit in the \p MODE register is set.  This
 *    happens if the \p DX10_CLAMP kernel descriptor field is set.
 *
 * 8. The library does not support single root I/O virtualization (SR-IOV) on
 *    any AMD GPU architecture that supports it.  That includes gfx1030,
 *    gfx1031, and gfx1032.
 *
 * 9. The library does not support debugging programs that use cooperative
 *    groups or CU masking for gfx1100, gfx1101, and gfx11102.  A restriction
 *    will be reported when attaching to a process that has already created
 *    cooperative group queues or CU masked queues.  Any attempt by the
 *    process to create a cooperative queue or CU masked queue when attached
 *    will fail.
 *
 * 10. On gfx1100, gfx1101 and gfx1102, the library cannot debug a program
 *     past a "s_sendmsg sendmsg(MSG_DEALLOC_VGPRS)" instruction.  If an
 *     exception is delivered to a wave in an attached process after the
 *     wave has executed this instruction, the wave is killed.
 *
 * \section references References
 *
 * 1. Advanced Micro Devices: [www.amd.com] (https://www.amd.com/)
 * 2. AMD ROCm Ecosystem: [rocm.docs.amd.com] (https://rocm.docs.amd.com/)
 * 3. Bus:Device.Function (BDF) Notation:
 *    [wiki.xen.org/wiki/Bus:Device.Function_(BDF)_Notation]
 *    (https://wiki.xen.org/wiki/Bus:Device.Function_(BDF)_Notation)
 * 4. HSA Platform System Architecture Specification:
 *    [https://hsafoundation.com/wp-content/uploads/2021/02/HSA-SysArch-1.2.pdf]
 *    (https://hsafoundation.com/wp-content/uploads/2021/02/HSA-SysArch-1.2.pdf)
 * 5. HSA Programmer's Reference Manual:
 *    [https://hsafoundation.com/wp-content/uploads/2021/02/HSA-PRM-1.2.pdf]
 *    (https://hsafoundation.com/wp-content/uploads/2021/02/HSA-PRM-1.2.pdf)
 * 6. Semantic Versioning: [semver.org] (https://semver.org)
 * 7. The LLVM Compiler Infrastructure: [llvm.org] (https://llvm.org/)
 * 8. User Guide for AMDGPU LLVM Backend: [llvm.org/docs/AMDGPUUsage.html]
 *    (https://llvm.org/docs/AMDGPUUsage.html)
 *
 * \section disclaimer Legal Disclaimer and Copyright Information
 *
 * AMD ROCm software is made available by Advanced Micro Devices, Inc. under
 * the open source license identified in the top-level directory for the
 * library in the repository on [Github.com](https://github.com/) (Portions of
 * AMD ROCm software are licensed under MITx11 and UIL/NCSA.  For more
 * information on the license, review the \p license.txt in the top-level
 * directory for the library on [Github.com](https://github.com/)).  The
 * additional terms and conditions below apply to your use of AMD ROCm
 * technical documentation.
 *
 * ©2019-2024 Advanced Micro Devices, Inc.  All rights reserved.
 *
 * The information presented in this document is for informational purposes
 * only and may contain technical inaccuracies, omissions, and typographical
 * errors.  The information contained herein is subject to change and may be
 * rendered inaccurate for many reasons, including but not limited to product
 * and roadmap changes, component and motherboard version changes, new model
 * and/or product releases, product differences between differing
 * manufacturers, software changes, BIOS flashes, firmware upgrades, or the
 * like.  Any computer system has risks of security vulnerabilities that cannot
 * be completely prevented or mitigated.  AMD assumes no obligation to update
 * or otherwise correct or revise this information.  However, AMD reserves the
 * right to revise this information and to make changes from time to time to
 * the content hereof without obligation of AMD to notify any person of such
 * revisions or changes.
 *
 * THIS INFORMATION IS PROVIDED "AS IS."  AMD MAKES NO REPRESENTATIONS OR
 * WARRANTIES WITH RESPECT TO THE CONTENTS HEREOF AND ASSUMES NO RESPONSIBILITY
 * FOR ANY INACCURACIES, ERRORS, OR OMISSIONS THAT MAY APPEAR IN THIS
 * INFORMATION.  AMD SPECIFICALLY DISCLAIMS ANY IMPLIED WARRANTIES OF
 * NON-INFRINGEMENT, MERCHANTABILITY, OR FITNESS FOR ANY PARTICULAR PURPOSE.
 * IN NO EVENT WILL AMD BE LIABLE TO ANY PERSON FOR ANY RELIANCE, DIRECT,
 * INDIRECT, SPECIAL, OR OTHER CONSEQUENTIAL DAMAGES ARISING FROM THE USE OF
 * ANY INFORMATION CONTAINED HEREIN, EVEN IF AMD IS EXPRESSLY ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGES.
 *
 * AMD, the AMD Arrow logo, AMD Instinct™, Radeon™, AMD ROCm™, and combinations
 * thereof are trademarks of Advanced Micro Devices, Inc.  Linux® is the
 * registered trademark of Linus Torvalds in the U.S. and other countries.
 * PCIe® is a registered trademark of PCI-SIG Corporation.  Other product names
 * used in this publication are for identification purposes only and may be
 * trademarks of their respective companies.
 */

/**
 * [amd]:
 * https://www.amd.com/
 * "Advanced Micro Devices"
 *
 * [bdf]:
 * https://wiki.xen.org/wiki/Bus:Device.Function_(BDF)_Notation
 * "[Bus:Device.Function (BDF) Notation]"
 *
 * [hsa-prm]:
 * https://hsafoundation.com/wp-content/uploads/2021/02/HSA-PRM-1.2.pdf
 * "HSA Programmer's Reference Manual"
 *
 * [hsa-sysarch]:
 * https://hsafoundation.com/wp-content/uploads/2021/02/HSA-SysArch-1.2.pdf
 * "HSA Platform System Architecture Specification"
 *
 * [llvm]:
 * https://llvm.org/
 * "The LLVM Compiler Infrastructure"
 *
 * [llvm-amdgpu]:
 * https://llvm.org/docs/AMDGPUUsage.html
 * "User Guide for AMDGPU LLVM Backend"
 *
 * [rocm]:
 * https://rocm.docs.amd.com/
 * "AMD ROCm Ecosystem"
 *
 * [semver]:
 * https://semver.org/
 * "Semantic Versioning"
 */

/**
 * \file
 * AMD debugger API interface.
 */

#ifndef AMD_DBGAPI_H
#define AMD_DBGAPI_H 1

/* Placeholder for calling convention and import/export macros */
#if !defined(AMD_DBGAPI_CALL)
#define AMD_DBGAPI_CALL
#endif /* !defined (AMD_DBGAPI_CALL) */

#if !defined(AMD_DBGAPI_EXPORT_DECORATOR)
#if defined(__GNUC__)
#define AMD_DBGAPI_EXPORT_DECORATOR __attribute__ ((visibility ("default")))
#elif defined(_MSC_VER)
#define AMD_DBGAPI_EXPORT_DECORATOR __declspec(dllexport)
#endif /* defined (_MSC_VER) */
#endif /* !defined (AMD_DBGAPI_EXPORT_DECORATOR) */

#if !defined(AMD_DBGAPI_IMPORT_DECORATOR)
#if defined(__GNUC__)
#define AMD_DBGAPI_IMPORT_DECORATOR
#elif defined(_MSC_VER)
#define AMD_DBGAPI_IMPORT_DECORATOR __declspec(dllimport)
#endif /* defined (_MSC_VER) */
#endif /* !defined (AMD_DBGAPI_IMPORT_DECORATOR) */

#define AMD_DBGAPI_EXPORT AMD_DBGAPI_EXPORT_DECORATOR AMD_DBGAPI_CALL
#define AMD_DBGAPI_IMPORT AMD_DBGAPI_IMPORT_DECORATOR AMD_DBGAPI_CALL

#if !defined(AMD_DBGAPI)
#if defined(AMD_DBGAPI_EXPORTS)
#define AMD_DBGAPI AMD_DBGAPI_EXPORT
#else /* !defined (AMD_DBGAPI_EXPORTS) */
#define AMD_DBGAPI AMD_DBGAPI_IMPORT
#endif /* !defined (AMD_DBGAPI_EXPORTS) */
#endif /* !defined (AMD_DBGAPI) */

#if __cplusplus >= 201103L
/* c++11 allows extended initializer lists.  */
#define AMD_DBGAPI_HANDLE_LITERAL(type, value) (type{ value })
#elif __STDC_VERSION__ >= 199901L
/* c99 allows compound literals.  */
#define AMD_DBGAPI_HANDLE_LITERAL(type, value) ((type){ value })
#else /* !__STDC_VERSION__ >= 199901L */
#define AMD_DBGAPI_HANDLE_LITERAL(type, value) {value}
#endif /* !__STDC_VERSION__ >= 199901L */

#if defined(__cplusplus) && __cplusplus >= 201402L
#define DEPRECATED [[deprecated]]
#else
#define DEPRECATED
#endif

#if defined(__cplusplus)
extern "C" {
#endif /* defined (__cplusplus) */

#if defined(__linux__)
#include <sys/types.h>
#endif /* __linux__ */

#include <stddef.h>
#include <stdint.h>

/** \defgroup symbol_versions_group Symbol Versions
 *
 * The names used for the shared library versioned symbols.
 *
 * Every function is annotated with one of the version macros defined in this
 * section.  Each macro specifies a corresponding symbol version string.  After
 * dynamically loading the shared library with \p dlopen, the address of each
 * function can be obtained using \p dlvsym with the name of the function and
 * its corresponding symbol version string.  An error will be reported by \p
 * dlvsym if the installed library does not support the version for the
 * function specified in this version of the interface.
 *
 * @{
 */

/**
 * The function was introduced in version 0.54 of the interface and has the
 * symbol version string of ``"AMD_DBGAPI_0.54"``.
 */
#define AMD_DBGAPI_VERSION_0_54

/**
 * The function was introduced in version 0.56 of the interface and has the
 * symbol version string of ``"AMD_DBGAPI_0.56"``.
 */
#define AMD_DBGAPI_VERSION_0_56

/**
 * The function was introduced in version 0.58 of the interface and has the
 * symbol version string of ``"AMD_DBGAPI_0.58"``.
 */
#define AMD_DBGAPI_VERSION_0_58

/**
 * The function was introduced in version 0.62 of the interface and has the
 * symbol version string of ``"AMD_DBGAPI_0.62"``.
 */
#define AMD_DBGAPI_VERSION_0_62

/**
 * The function was introduced in version 0.64 of the interface and has the
 * symbol version string of ``"AMD_DBGAPI_0.64"``.
 */
#define AMD_DBGAPI_VERSION_0_64

/**
 * The function was introduced in version 0.67 of the interface and has the
 * symbol version string of ``"AMD_DBGAPI_0.67"``.
 */
#define AMD_DBGAPI_VERSION_0_67

/**
 * The function was introduced in version 0.68 of the interface and has the
 * symbol version string of ``"AMD_DBGAPI_0.68"``.
 */
#define AMD_DBGAPI_VERSION_0_68

/**
 * The function was introduced in version 0.70 of the interface and has the
 * symbol version string of ``"AMD_DBGAPI_0.70"``.
 */
#define AMD_DBGAPI_VERSION_0_70

/**
 * The function was introduced in version 0.76 of the interface and has the
 * symbol version string of ``"AMD_DBGAPI_0.76"``.
 */
#define AMD_DBGAPI_VERSION_0_76

/**
 * The function was introduced in version 0.77 of the interface and has the
 * symbol version string of ``"AMD_DBGAPI_0.77"``.
 */
#define AMD_DBGAPI_VERSION_0_77

/** @} */

/** \ingroup callbacks_group
 * Forward declaration of callbacks used to specify services that must be
 * provided by the client.
 */
typedef struct amd_dbgapi_callbacks_s amd_dbgapi_callbacks_t;

/** \defgroup basic_group Basic Types
 *
 * Types used for common properties.
 *
 * Note that in some cases enumeration types are used as output parameters for
 * functions using pointers.  The C language does not define the underlying
 * type used for enumeration types.  This interface requires that:
 *
 * - For all enumeration types the underlying type used by the client will be
 *   \p int with a size of 32 bits.
 *
 * In addition, it requires that enumeration types passed by value to
 * functions, or returned as values from functions, will have the platform
 * function ABI representation.
 *
 * @{
 */

/**
 * Integral type used for a global virtual memory address in the inferior
 * process.
 */
typedef uint64_t amd_dbgapi_global_address_t;

/**
 * Integral type used for sizes, including memory allocations, in the inferior.
 */
typedef uint64_t amd_dbgapi_size_t;

/**
 * Indication of if a value has changed.
 */
typedef enum
{
  /**
   * The value has not changed.
   */
  AMD_DBGAPI_CHANGED_NO = 0,
  /**
   * The value has changed.
   */
  AMD_DBGAPI_CHANGED_YES = 1
} amd_dbgapi_changed_t;

/**
 * Native operating system process ID.
 *
 * This is the process ID used by the operating system that is executing the
 * library.  It is used in the implementation of the library to interact with
 * the operating system AMD GPU device driver.
 */
#if defined(__linux__)
typedef pid_t amd_dbgapi_os_process_id_t;
#endif /* __linux__ */

/**
 * Type used to notify the client of the library that a process may have
 * pending events.
 *
 * A notifier is created when ::amd_dbgapi_process_attach is used to
 * successfully attach to a process.  It is obtained using the
 * ::AMD_DBGAPI_PROCESS_INFO_NOTIFIER query.  If the notifier indicates there
 * may be pending events, then ::amd_dbgapi_process_next_pending_event can be
 * used to retrieve them.  The same notifier may be returned when attaching to
 * different processes.
 *
 * For Linux<sup>&reg;</sup> this is a file descriptor number that can be used
 * with the \p poll call to wait on events from multiple sources.  The file
 * descriptor is made to have data available when events may be added to the
 * pending events.  The client can flush the file descriptor and read the
 * pending events until none are available.  Note that the file descriptor may
 * become ready spuriously when no pending events are available, in which case
 * the client should simply wait again.  If new pending events are added while
 * reading the pending events, then the file descriptor will again have data
 * available.  The amount of data on the file descriptor is not an indication
 * of the number of pending events as the file may become full and so no
 * further data will be added.  The file descriptor is simply a robust way to
 * determine if there may be some pending events.
 */
#if defined(__linux__)
typedef int amd_dbgapi_notifier_t;
#endif /* __linux__ */

/**
 * Native operating system agent ID.
 *
 * This is the agent ID used by the operating system AMD GPU device driver that
 * is executing the library to specify the AMD GPU agents accessible to a
 * process.
 */
#if defined(__linux__)
typedef uint64_t amd_dbgapi_os_agent_id_t;
#endif /* __linux__ */

/**
 * Native operating system queue ID.
 *
 * This is the queue ID used by the operating system AMD GPU device driver that
 * is executing the library to specify the AMD GPU queues of a process.
 */
#if defined(__linux__)
typedef uint64_t amd_dbgapi_os_queue_id_t;
#endif /* __linux__ */

/**
 * Native operating system queue packet ID.
 *
 * This is the queue packet ID used by the operating system AMD GPU device
 * driver that is executing the library to specify the AMD GPU packets of a
 * queue of a process.  The meaning of the queue packet ID is dependent on the
 * queue type.  See ::amd_dbgapi_os_queue_type_t.
 */
#if defined(__linux__)
typedef uint64_t amd_dbgapi_os_queue_packet_id_t;
#endif /* __linux__ */

/**
 * Native operating system queue type.
 *
 * This is used by the operating system AMD GPU device driver that is executing
 * the library to specify the AMD GPU queue mechanics supported by the queues
 * of a process.
 */
#if defined(__linux__)
typedef enum
{
  /**
   * Unknown queue type.
   */
  AMD_DBGAPI_OS_QUEUE_TYPE_UNKNOWN = 0,
  /**
   * Queue supports the HSA AQL protocol.
   */
  AMD_DBGAPI_OS_QUEUE_TYPE_HSA_AQL = 1,
  /**
   * Queue supports the AMD PM4 protocol.
   */
  AMD_DBGAPI_OS_QUEUE_TYPE_AMD_PM4 = 257,
  /**
   * Queue supports the AMD SDMA protocol.
   */
  AMD_DBGAPI_OS_QUEUE_TYPE_AMD_SDMA = 513,
  /**
   * Queue supports the AMD SDMA XGMI protocol.
   */
  AMD_DBGAPI_OS_QUEUE_TYPE_AMD_SDMA_XGMI = 514
} amd_dbgapi_os_queue_type_t;
#endif /* __linux__ */

/** @} */

/** \defgroup status_codes_group Status Codes
 *
 * Most operations return a status code to indicate success or error.
 *
 * @{
 */

/**
 * AMD debugger API status codes.
 */
typedef enum
{
  /**
   * The function has executed successfully.
   */
  AMD_DBGAPI_STATUS_SUCCESS = 0,
  /**
   * A generic error has occurred.
   */
  AMD_DBGAPI_STATUS_ERROR = -1,
  /**
   * A fatal error has occurred.
   *
   * The library encountered an error from which it cannot recover.  All
   * processes are detached.  All breakpoints inserted by
   * amd_dbgapi_callbacks_s::insert_breakpoint are attempted to be removed.
   * All handles are invalidated.  The library is left in an uninitialized
   * state.  The logging level is reset to ::AMD_DBGAPI_LOG_LEVEL_NONE.
   *
   * To resume using the library the client must re-initialize the
   * library; re-attach to any processes; re-fetch the list of code objects,
   * agents, queues, dispatches, and waves; and update the state of all waves
   * as appropriate.  While in the uninitialized state the inferior processes
   * will continue executing but any execution of a breakpoint instruction will
   * put the queue into an error state, aborting any executing waves.  Note
   * that recovering from a fatal error most likely will require the user of
   * the client to re-start their session.
   *
   * The cause of possible fatal errors is that resources became exhausted or
   * unique handle numbers became exhausted.
   */
  AMD_DBGAPI_STATUS_FATAL = -2,
  /**
   * The operation is not currently implemented.  This error may be reported by
   * any function.  Check the \ref known_limitations section to determine the
   * status of the library implementation of the interface.
   */
  AMD_DBGAPI_STATUS_ERROR_NOT_IMPLEMENTED = -3,
  /**
   * The requested information is not available.
   */
  AMD_DBGAPI_STATUS_ERROR_NOT_AVAILABLE = -4,
  /**
   * The operation is not supported.
   */
  AMD_DBGAPI_STATUS_ERROR_NOT_SUPPORTED = -5,
  /**
   * An invalid argument was given to the function.
   */
  AMD_DBGAPI_STATUS_ERROR_INVALID_ARGUMENT = -6,
  /**
   * An invalid combination of arguments was given to the function.
   */
  AMD_DBGAPI_STATUS_ERROR_INVALID_ARGUMENT_COMPATIBILITY = -7,
  /**
   * The library is already initialized.
   */
  AMD_DBGAPI_STATUS_ERROR_ALREADY_INITIALIZED = -8,
  /**
   * The library is not initialized.
   */
  AMD_DBGAPI_STATUS_ERROR_NOT_INITIALIZED = -9,
  /**
   * There is a restriction error that prevents the operation to complete.
   *
   * Reasons which could prevent debugging the process include:
   *
   * - The AMD GPU driver is not installed.
   * - The installed AMD GPU driver version is not compatible with the library.
   * - The installed AMD GPU driver's debug support version is not compatible
   *   with the library.
   * - A limitation on the number of debuggers that can be active for an
   *   AMD GPU agent has been exceeded.
   * - The process has the same address space as another process to which the
   *   library is already attached.  For example, attaching to a process
   *   created by the Linux \p vfork system call while attached to the parent
   *   process.
   *
   * On some configurations, this error is returned instead of creating a core
   * dump containing an ambiguous state.
   */
  AMD_DBGAPI_STATUS_ERROR_RESTRICTION = -10,
  /**
   * The process is already attached to the given inferior process.
   */
  AMD_DBGAPI_STATUS_ERROR_ALREADY_ATTACHED = -11,
  /**
   * The architecture handle is invalid.
   */
  AMD_DBGAPI_STATUS_ERROR_INVALID_ARCHITECTURE_ID = -12,
  /**
   * The bytes being disassembled are not a legal instruction.
   */
  AMD_DBGAPI_STATUS_ERROR_ILLEGAL_INSTRUCTION = -13,
  /**
   * The code object handle is invalid.
   */
  AMD_DBGAPI_STATUS_ERROR_INVALID_CODE_OBJECT_ID = -14,
  /**
   * The ELF AMD GPU machine value is invalid or unsupported.
   */
  AMD_DBGAPI_STATUS_ERROR_INVALID_ELF_AMDGPU_MACHINE = -15,
  /**
   * The process handle is invalid.
   */
  AMD_DBGAPI_STATUS_ERROR_INVALID_PROCESS_ID = -16,
  /**
   * The native operating system process associated with a client process has
   * exited.
   */
  AMD_DBGAPI_STATUS_ERROR_PROCESS_EXITED = -17,
  /**
   * The agent handle is invalid.
   */
  AMD_DBGAPI_STATUS_ERROR_INVALID_AGENT_ID = -18,
  /**
   * The queue handle is invalid.
   */
  AMD_DBGAPI_STATUS_ERROR_INVALID_QUEUE_ID = -19,
  /**
   * The dispatch handle is invalid.
   */
  AMD_DBGAPI_STATUS_ERROR_INVALID_DISPATCH_ID = -20,
  /**
   * The wave handle is invalid.
   */
  AMD_DBGAPI_STATUS_ERROR_INVALID_WAVE_ID = -21,
  /**
   * The wave is not stopped.
   */
  AMD_DBGAPI_STATUS_ERROR_WAVE_NOT_STOPPED = -22,
  /**
   * The wave is stopped.
   */
  AMD_DBGAPI_STATUS_ERROR_WAVE_STOPPED = -23,
  /**
   * The wave has an outstanding stop request.
   */
  AMD_DBGAPI_STATUS_ERROR_WAVE_OUTSTANDING_STOP = -24,
  /**
   * The wave cannot be resumed.
   */
  AMD_DBGAPI_STATUS_ERROR_WAVE_NOT_RESUMABLE = -25,
  /**
   * The displaced stepping handle is invalid.
   */
  AMD_DBGAPI_STATUS_ERROR_INVALID_DISPLACED_STEPPING_ID = -26,
  /**
   * No more displaced stepping buffers are available that are suitable for the
   * requested wave.
   */
  AMD_DBGAPI_STATUS_ERROR_DISPLACED_STEPPING_BUFFER_NOT_AVAILABLE = -27,
  /**
   * The wave has an active displaced stepping buffer.
   */
  AMD_DBGAPI_STATUS_ERROR_DISPLACED_STEPPING_ACTIVE = -28,
  /**
   * The wave cannot be resumed in the manner requested due to displaced
   * stepping restrictions.
   */
  AMD_DBGAPI_STATUS_ERROR_RESUME_DISPLACED_STEPPING = -29,
  /**
   * The watchpoint handle is invalid.
   */
  AMD_DBGAPI_STATUS_ERROR_INVALID_WATCHPOINT_ID = -30,
  /**
   * No more watchpoints available.
   */
  AMD_DBGAPI_STATUS_ERROR_NO_WATCHPOINT_AVAILABLE = -31,
  /**
   * The register class handle is invalid.
   */
  AMD_DBGAPI_STATUS_ERROR_INVALID_REGISTER_CLASS_ID = -32,
  /**
   * The register handle is invalid.
   */
  AMD_DBGAPI_STATUS_ERROR_INVALID_REGISTER_ID = -33,
  /**
   * The lane handle is invalid.
   */
  AMD_DBGAPI_STATUS_ERROR_INVALID_LANE_ID = -34,
  /**
   * The address class handle is invalid.
   */
  AMD_DBGAPI_STATUS_ERROR_INVALID_ADDRESS_CLASS_ID = -35,
  /**
   * The address space handle is invalid.
   */
  AMD_DBGAPI_STATUS_ERROR_INVALID_ADDRESS_SPACE_ID = -36,
  /**
   * An error occurred while trying to access memory in the inferior.
   */
  AMD_DBGAPI_STATUS_ERROR_MEMORY_ACCESS = -37,
  /**
   * The segment address cannot be converted to the requested address space.
   */
  AMD_DBGAPI_STATUS_ERROR_INVALID_ADDRESS_SPACE_CONVERSION = -38,
  /**
   * The event handle is invalid.
   */
  AMD_DBGAPI_STATUS_ERROR_INVALID_EVENT_ID = -39,
  /**
   * The breakpoint handle is invalid.
   */
  AMD_DBGAPI_STATUS_ERROR_INVALID_BREAKPOINT_ID = -40,
  /**
   * A callback to the client reported an error.
   */
  AMD_DBGAPI_STATUS_ERROR_CLIENT_CALLBACK = -41,
  /**
   * The client process handle is invalid.
   */
  AMD_DBGAPI_STATUS_ERROR_INVALID_CLIENT_PROCESS_ID = -42,
  /**
   * The symbol was not found.
   */
  AMD_DBGAPI_STATUS_ERROR_SYMBOL_NOT_FOUND = -43,
  /**
   * The register handle is valid, but specifies a register that is not
   * allocated in the associated wave.
   */
  AMD_DBGAPI_STATUS_ERROR_REGISTER_NOT_AVAILABLE = -44,
  /**
   * The workgroup handle is invalid.
   */
  AMD_DBGAPI_STATUS_ERROR_INVALID_WORKGROUP_ID = -45,
  /**
   * The current process state is not compatible with the requested operation.
   */
  AMD_DBGAPI_STATUS_ERROR_INCOMPATIBLE_PROCESS_STATE = -46,
  /**
   * This operation is not allowed when the process is frozen.
   */
  AMD_DBGAPI_STATUS_ERROR_PROCESS_FROZEN = -47,
  /**
   * The process is already frozen.
   */
  AMD_DBGAPI_STATUS_ERROR_PROCESS_ALREADY_FROZEN = -48,
  /**
   * This operation is not allowed when the process is not frozen.
   */
  AMD_DBGAPI_STATUS_ERROR_PROCESS_NOT_FROZEN = -49,
} amd_dbgapi_status_t;

/**
 * Query a textual description of a status code.
 *
 * This function can be used even when the library is uninitialized.
 *
 * \param[in] status Status code.
 *
 * \param[out] status_string A NUL terminated string that describes the
 * status code.  The string is read only and owned by the library.
 *
 * \retval ::AMD_DBGAPI_STATUS_SUCCESS The function has been executed
 * successfully.  \p status_string has been updated.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ARGUMENT \p status is an invalid
 * status code or \p status_string is NULL.  \p status_string is unaltered.
 */
amd_dbgapi_status_t AMD_DBGAPI amd_dbgapi_get_status_string (
    amd_dbgapi_status_t status,
    const char **status_string) AMD_DBGAPI_VERSION_0_54;

/** @} */

/** \defgroup versioning_group Versioning
 *
 * Version information about the interface and the associated installed
 * library.
 *
 * @{
 */

/**
 * The semantic version of the interface following
 * [semver.org][semver] rules.
 *
 * A client that uses this interface is only compatible with the installed
 * library if the major version numbers match and the interface minor version
 * number is less than or equal to the installed library minor version number.
 */

/**
 * The major version of the interface as a macro so it can be used by the
 * preprocessor.
 */
#define AMD_DBGAPI_VERSION_MAJOR 0

/**
 * The minor version of the interface as a macro so it can be used by the
 * preprocessor.
 */
#define AMD_DBGAPI_VERSION_MINOR 77

/**
 * Query the version of the installed library.
 *
 * Return the version of the installed library.  This can be used to check if
 * it is compatible with this interface version.  This function can be used
 * even when the library is not initialized.
 *
 * \param[out] major The major version number is stored if non-NULL.
 *
 * \param[out] minor The minor version number is stored if non-NULL.
 *
 * \param[out] patch The patch version number is stored if non-NULL.
 */
void AMD_DBGAPI amd_dbgapi_get_version (
    uint32_t *major, uint32_t *minor, uint32_t *patch) AMD_DBGAPI_VERSION_0_54;

/**
 * Query the installed library build name.
 *
 * This function can be used even when the library is not initialized.
 *
 * \return Returns a string describing the build version of the library.  The
 * string is owned by the library.
 */
const char AMD_DBGAPI *
amd_dbgapi_get_build_name (void) AMD_DBGAPI_VERSION_0_54;

/** @} */

/** \defgroup initialization_group Initialization and Finalization
 *
 * Operations to control initializing and finalizing the library.
 *
 * When the library is first loaded it is in the uninitialized state.  Before
 * any operation can be used, the library must be initialized.  The exception
 * is the status operation in \ref status_codes_group and the version
 * operations in \ref versioning_group which can be used regardless of whether
 * the library is initialized.
 *
 * @{
 */

/**
 * Initialize the library.
 *
 * Initialize the library so that the library functions can be used to control
 * the AMD GPU devices accessed by processes.
 *
 * Initializing the library does not change the logging level (see
 * \ref logging_group).
 *
 * \param[in] callbacks A set of callbacks must be provided.  These
 * are invoked by certain operations.  They are described in
 * ::amd_dbgapi_callbacks_t.
 *
 * \retval ::AMD_DBGAPI_STATUS_SUCCESS The function has been executed
 * successfully and the library is now initialized.
 *
 * \retval ::AMD_DBGAPI_STATUS_FATAL A fatal error occurred.  The library
 * remains uninitialized.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_ALREADY_INITIALIZED The library is
 * already initialized.  The library is left initialized and the callbacks are
 * not changed.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ARGUMENT \p callbacks is NULL
 * or has fields that are NULL.  The library remains uninitialized.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_CLIENT_CALLBACK This will be
 * reported if any of the amd_dbgapi_callbacks_s callbacks used return an
 * error.  The library remains uninitialized.
 */
amd_dbgapi_status_t AMD_DBGAPI amd_dbgapi_initialize (
    amd_dbgapi_callbacks_t *callbacks) AMD_DBGAPI_VERSION_0_76;

/**
 * Finalize the library.
 *
 * Finalizing the library invalidates all handles previously returned by any
 * operation.  It is undefined to use any such handle even if the library is
 * subsequently initialized with ::amd_dbgapi_initialize.  Finalizing the
 * library implicitly detaches from any processes currently attached.  It is
 * allowed to initialize and finalize the library multiple times.  Finalizing
 * the library does not changed the logging level (see \ref logging_group).
 *
 * \retval ::AMD_DBGAPI_STATUS_SUCCESS The function has been executed
 * successfully and the library is now uninitialized.
 *
 * \retval ::AMD_DBGAPI_STATUS_FATAL A fatal error occurred.  The library
 * is left uninitialized.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_NOT_INITIALIZED The library is not
 * initialized.  The library is left uninitialized.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_CLIENT_CALLBACK This will be reported if
 * any of the ::amd_dbgapi_callbacks_s callbacks used return an error.  The
 * library is still left uninitialized, but the client may be in
 * an inconsistent state.
 */
amd_dbgapi_status_t AMD_DBGAPI amd_dbgapi_finalize (void)
    AMD_DBGAPI_VERSION_0_54;

/** @} */

/** \defgroup architecture_group Architectures
 *
 * Operations related to AMD GPU architectures.
 *
 * The library supports a family of AMD GPU devices.  Each device has its own
 * architectural properties.  The operations in this section provide
 * information about the supported architectures.
 *
 * @{
 */

/**
 * Opaque architecture handle.
 *
 * There is an architecture handle for each AMD GPU model supported by the
 * library.
 *
 * Globally unique for a single library instance.
 */
typedef struct
{
  uint64_t handle;
} amd_dbgapi_architecture_id_t;

/**
 * The NULL architecture handle.
 */
#define AMD_DBGAPI_ARCHITECTURE_NONE                                          \
    AMD_DBGAPI_HANDLE_LITERAL (amd_dbgapi_architecture_id_t, 0)

/**
 * Architecture queries that are supported by
 * ::amd_dbgapi_architecture_get_info.
 *
 * Each query specifies the type of data returned in the \p value argument to
 * ::amd_dbgapi_architecture_get_info.
 */
typedef enum
{
  /**
   * Return the architecture name.  The type of this attribute is a
   * pointer to a NUL terminated \p char*.  It is allocated by the
   * amd_dbgapi_callbacks_s::allocate_memory callback and is owned by the
   * client.
   */
  AMD_DBGAPI_ARCHITECTURE_INFO_NAME = 1,
  /**
   * Return the AMD GPU ELF \p EF_AMDGPU_MACH value corresponding to the
   * architecture.  This is defined as a bit field in the \p e_flags AMD GPU
   * ELF header.  See [User Guide for AMDGPU Backend - Code Object - Header]
   * (https://llvm.org/docs/AMDGPUUsage.html#header).  The type of this
   * attribute is \p uint32_t.
   */
  AMD_DBGAPI_ARCHITECTURE_INFO_ELF_AMDGPU_MACHINE = 2,
  /**
   * Return the largest instruction size in bytes for the architecture.  The
   * type of this attribute is ::amd_dbgapi_size_t.
   */
  AMD_DBGAPI_ARCHITECTURE_INFO_LARGEST_INSTRUCTION_SIZE = 3,
  /**
   * Return the minimum instruction alignment in bytes for the architecture.
   * The returned value will be a power of two.  The type of this attribute is
   * ::amd_dbgapi_size_t.
   */
  AMD_DBGAPI_ARCHITECTURE_INFO_MINIMUM_INSTRUCTION_ALIGNMENT = 4,
  /**
   * Return the breakpoint instruction size in bytes for the architecture.  The
   * type of this attribute is ::amd_dbgapi_size_t.
   */
  AMD_DBGAPI_ARCHITECTURE_INFO_BREAKPOINT_INSTRUCTION_SIZE = 5,
  /**
   * Return the breakpoint instruction for the architecture.  The type of this
   * attribute is pointer to \p N bytes where \p N is the value returned by the
   * ::AMD_DBGAPI_ARCHITECTURE_INFO_BREAKPOINT_INSTRUCTION_SIZE query.  It is
   * allocated by the amd_dbgapi_callbacks_s::allocate_memory callback and is
   * owned by the client.
   */
  AMD_DBGAPI_ARCHITECTURE_INFO_BREAKPOINT_INSTRUCTION = 6,
  /**
   * Return the number of bytes to subtract from the PC after stopping due to a
   * breakpoint instruction to get the address of the breakpoint instruction
   * for the architecture.  The type of this attribute is ::amd_dbgapi_size_t.
   */
  AMD_DBGAPI_ARCHITECTURE_INFO_BREAKPOINT_INSTRUCTION_PC_ADJUST = 7,
  /**
   * Return the register handle for the PC for the architecture.  The type of
   * this attribute is ::amd_dbgapi_register_id_t.
   */
  AMD_DBGAPI_ARCHITECTURE_INFO_PC_REGISTER = 8
} amd_dbgapi_architecture_info_t;

/**
 * Query information about an architecture.
 *
 * ::amd_dbgapi_architecture_info_t specifies the queries supported and the
 * type returned using the \p value argument.
 *
 * \param[in] architecture_id The architecture being queried.
 *
 * \param[in] query The query being requested.
 *
 * \param[in] value_size Size of the memory pointed to by \p value.  Must be
 * equal to the byte size of the query result.
 *
 * \param[out] value Pointer to memory where the query result is stored.
 *
 * \retval ::AMD_DBGAPI_STATUS_SUCCESS The function has been executed
 * successfully and the result is stored in \p value.
 *
 * \retval ::AMD_DBGAPI_STATUS_FATAL A fatal error occurred.  The library is
 * left uninitialized and \p value is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_NOT_INITIALIZED The library is not
 * initialized.  The library is left uninitialized and \p value is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ARCHITECTURE_ID \p
 * architecture_id is invalid.  \p value is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ARGUMENT \p value is NULL or
 * \p query is invalid.  \p value is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ARGUMENT_COMPATIBILITY \p
 * value_size does not match the size of the \p query result.  \p value is
 * unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_CLIENT_CALLBACK This will be
 * reported if the amd_dbgapi_callbacks_s::allocate_memory callback used to
 * allocate \p value returns NULL.  \p value is unaltered.
 */
amd_dbgapi_status_t AMD_DBGAPI amd_dbgapi_architecture_get_info (
    amd_dbgapi_architecture_id_t architecture_id,
    amd_dbgapi_architecture_info_t query, size_t value_size,
    void *value) AMD_DBGAPI_VERSION_0_54;

/**
 * Get an architecture from the AMD GPU ELF \p EF_AMDGPU_MACH value
 * corresponding to the architecture.  This is defined as a bit field in the \p
 * e_flags AMD GPU ELF header.  See [User Guide for AMDGPU Backend - Code
 * Object
 * - Header] (https://llvm.org/docs/AMDGPUUsage.html#header).
 *
 * \param[in] elf_amdgpu_machine The AMD GPU ELF \p EF_AMDGPU_MACH value.
 *
 * \param[out] architecture_id The corresponding architecture.
 *
 * \retval ::AMD_DBGAPI_STATUS_SUCCESS The function has been executed
 * successfully and the result is stored in \p architecture_id.
 *
 * \retval ::AMD_DBGAPI_STATUS_FATAL A fatal error occurred.  The library
 * is left uninitialized and \p architecture_id is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_NOT_INITIALIZED The library is not
 * initialized.  The library is left uninitialized and \p architecture_id is
 * unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ELF_AMDGPU_MACHINE \p
 * elf_amdgpu_machine is invalid or unsupported.  \p architecture_id is
 * unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ARGUMENT \p architecture_id is
 * NULL.  \p architecture_id is unaltered.
 */
amd_dbgapi_status_t AMD_DBGAPI amd_dbgapi_get_architecture (
    uint32_t elf_amdgpu_machine,
    amd_dbgapi_architecture_id_t *architecture_id) AMD_DBGAPI_VERSION_0_54;

/**
 * Opaque client symbolizer handle.
 *
 * A pointer to client data associated with a symbolizer.  This pointer is
 * passed to the ::amd_dbgapi_disassemble_instruction \p symbolizer callback.
 */
typedef struct amd_dbgapi_symbolizer_id_s *amd_dbgapi_symbolizer_id_t;

/**
 * Disassemble a single instruction.
 *
 * \param[in] architecture_id The architecture to use to perform the
 * disassembly.
 *
 * \param[in] address The address of the first byte of the instruction.
 *
 * \param[in,out] size Pass in the number of bytes available in \p memory which
 * must be greater than 0.  Return the number of bytes consumed to decode the
 * instruction.
 *
 * \param[in] memory The bytes to decode as an instruction.  Must point to an
 * array of at least \p size bytes.  The
 * ::AMD_DBGAPI_ARCHITECTURE_INFO_LARGEST_INSTRUCTION_SIZE query for \p
 * architecture_id can be used to determine the number of bytes of the largest
 * instruction.  By making \p size at least this size ensures that the
 * instruction can be decoded if legal.  However, \p size may need to be
 * smaller if no memory exists at the address of \p address plus \p size.
 *
 * \param[out] instruction_text If NULL then only the instruction \p size is
 * returned.
 *
 * If non-NULL then set to a pointer to a NUL terminated string that contains
 * the disassembled textual representation of the instruction.  The memory is
 * allocated using the amd_dbgapi_callbacks_s::allocate_memory callback and is
 * owned by the client.
 *
 * \param[in] symbolizer_id The client handle that is passed to any invocation
 * of the \p symbolizer callback made while disassembling the instruction.
 *
 * \param[in] symbolizer A callback that is invoked for any operand of the
 * disassembled instruction that is a memory address.  It allows the client to
 * provide a symbolic representation of the address as a textual symbol that
 * will be used in the returned \p instruction_text.
 *
 * If \p symbolizer is NULL, then no symbolization will be performed and any
 * memory addresses will be shown as their numeric address.
 *
 * If \p symbolizer is non-NULL, the \p symbolizer function will be called with
 * \p symbolizer_id having the value of the above \p symbolizer_id operand, and
 * with \p address having the value of the address of the disassembled
 * instruction's operand.
 *
 * If the \p symbolizer callback wishes to report a symbol text it must
 * allocate and assign memory for a non-empty NUL terminated \p char* string
 * using a memory allocator that can be deallocated using the
 * amd_dbgapi_callbacks_s::deallocate_memory callback.  If must assign the
 * pointer to \p symbol_text, and return ::AMD_DBGAPI_STATUS_SUCCESS.
 *
 * If the \p symbolizer callback does not wish to report a symbol it must
 * return ::AMD_DBGAPI_STATUS_ERROR_SYMBOL_NOT_FOUND.
 *
 * Any \p symbol_text strings returned by the \p symbolizer callbacks reporting
 * ::AMD_DBGAPI_STATUS_SUCCESS are deallocated using the
 * amd_dbgapi_callbacks_s::deallocate_memory callback before
 * ::amd_dbgapi_disassemble_instruction returns.
 *
 * \retval ::AMD_DBGAPI_STATUS_SUCCESS The function has been executed
 * successfully and the result is stored in \p size and \p instruction_text.
 *
 * \retval ::AMD_DBGAPI_STATUS_FATAL A fatal error occurred.  The library is
 * left uninitialized and \p size and \p instruction_text are unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_NOT_INITIALIZED The library is not
 * initialized.  The library is left uninitialized and \p size and \p
 * instruction_text are unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ARCHITECTURE_ID \p architecture_id
 * is invalid.  \p size and \p instruction_text are unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ARGUMENT \p size or \p memory are
 * NULL, \p *size is 0, or \p address is not aligned on the value returned by
 * the ::AMD_DBGAPI_ARCHITECTURE_INFO_MINIMUM_INSTRUCTION_ALIGNMENT query.  \p
 * size and \p * instruction_text are unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR Encountered an error disassembling the
 * instruction, a \p symbolizer callback returned ::AMD_DBGAPI_STATUS_SUCCESS
 * with a NULL or empty \p symbol_text string. The bytes may or may not be a
 * legal instruction. \p size and \p instruction_text are unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_ILLEGAL_INSTRUCTION The bytes starting at
 * \p address, when up to \p size bytes are available, are not a legal
 * instruction for the architecture.  \p size and \p instruction_text are
 * unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_CLIENT_CALLBACK This will be reported if
 * the amd_dbgapi_callbacks_s::allocate_memory callback used to allocate \p
 * instruction_text returns NULL, or a \p symbolizer callback returns a status
 * other than ::AMD_DBGAPI_STATUS_SUCCESS and
 * ::AMD_DBGAPI_STATUS_ERROR_SYMBOL_NOT_FOUND.  \p size and \p instruction_text
 * are unaltered.
 */
amd_dbgapi_status_t AMD_DBGAPI amd_dbgapi_disassemble_instruction (
    amd_dbgapi_architecture_id_t architecture_id,
    amd_dbgapi_global_address_t address, amd_dbgapi_size_t *size,
    const void *memory, char **instruction_text,
    amd_dbgapi_symbolizer_id_t symbolizer_id,
    amd_dbgapi_status_t (*symbolizer) (
        amd_dbgapi_symbolizer_id_t symbolizer_id,
        amd_dbgapi_global_address_t address,
        char **symbol_text)) AMD_DBGAPI_VERSION_0_54;

/**
 * The kinds of instruction classifications.
 */
typedef enum
{
  /**
   * The instruction classification is unknown.  The instruction has no
   * information.
   */
  AMD_DBGAPI_INSTRUCTION_KIND_UNKNOWN = 0,
  /**
   * The instruction executes sequentially.  It performs no control flow and
   * the next instruction executed is the following one.  The instruction has
   * no information.
   */
  AMD_DBGAPI_INSTRUCTION_KIND_SEQUENTIAL = 1,
  /**
   * The instruction unconditionally branches to a literal address.  The
   * instruction information is of type ::amd_dbgapi_global_address_t with the
   * value of the target address of the branch.
   */
  AMD_DBGAPI_INSTRUCTION_KIND_DIRECT_BRANCH = 2,
  /**
   * The instruction conditionally branches to a literal address.  If the
   * condition is not satisfied then the next instruction is the following one.
   * The instruction information is of type ::amd_dbgapi_global_address_t with
   * the value of the target address of the branch if taken.
   */
  AMD_DBGAPI_INSTRUCTION_KIND_DIRECT_BRANCH_CONDITIONAL = 3,
  /**
   * The instruction unconditionally branches to an address held in a pair of
   * registers.  The instruction information is of type
   * ::amd_dbgapi_register_id_t[2] with the value of the register IDs for the
   * registers.  The first register holds the least significant address bits,
   * and the second register holds the most significant address bits.
   */
  AMD_DBGAPI_INSTRUCTION_KIND_INDIRECT_BRANCH_REGISTER_PAIR = 4,
  /**
   * The instruction conditionally branches to an address held in a pair of
   * registers.  If the condition is not satisfied then the next instruction
   * is the following one.  The instruction information is of type
   * ::amd_dbgapi_register_id_t[2] with the value of the register IDs for the
   * registers holding the value of the target address of the branch if taken.
   * The register with index 0 holds the least significant address bits, and
   * the register with index 1 holds the most significant address bits.
   */
  AMD_DBGAPI_INSTRUCTION_KIND_INDIRECT_BRANCH_CONDITIONAL_REGISTER_PAIR = 5,
  /**
   * The instruction unconditionally branches to a literal address and the
   * address of the following instruction is saved in a pair of registers.  The
   * instruction information is of type
   * ::amd_dbgapi_direct_call_register_pair_information_t with the value of the
   * target address of the call followed by the value of the saved return
   * address register IDs.  The saved return address register with index 0
   * holds the least significant address bits, and the register with index 1
   * holds the most significant address bits.
   */
  AMD_DBGAPI_INSTRUCTION_KIND_DIRECT_CALL_REGISTER_PAIR = 6,
  /**
   * The instruction unconditionally branches to an address held in a pair of
   * source registers and the address of the following instruction is saved in
   * a pair of destination registers.  The instruction information is of type
   * ::amd_dbgapi_register_id_t[4] with the source register IDs in indices 0
   * and 1, and the destination register IDs in indices 2 and 3. The registers
   * with indices 0 and 2 hold the least significant address bits, and the
   * registers with indices 1 and 3 hold the most significant address bits.
   */
  AMD_DBGAPI_INSTRUCTION_KIND_INDIRECT_CALL_REGISTER_PAIRS = 7,
  /**
   * The instruction terminates the wave execution.  The instruction has no
   * information.
   */
  AMD_DBGAPI_INSTRUCTION_KIND_TERMINATE = 8,
  /**
   * The instruction enters the trap handler.  The trap handler may return to
   * resume execution, may put the wave into the halt state and create an event
   * for ::amd_dbgapi_process_next_pending_event to report, or may terminate
   * the wave.  The library cannot report execution in the trap handler.  If
   * single stepping the trap instruction reports the
   * ::AMD_DBGAPI_WAVE_STOP_REASON_SINGLE_STEP reason, then the program counter
   * will be at the instruction following the trap instruction, it will not be
   * at the first instruction of the trap handler. It is undefined to set a
   * breakpoint in the trap handler, and will likely cause the inferior to
   * report errors and stop executing correctly.  The instruction information
   * is of type \p uint64_t with the value of the trap code.
   */
  AMD_DBGAPI_INSTRUCTION_KIND_TRAP = 9,
  /**
   * The instruction unconditionally halts the wave.  The instruction has no
   * information.
   */
  AMD_DBGAPI_INSTRUCTION_KIND_HALT = 10,
  /**
   * The instruction performs some kind of execution barrier which may result
   * in the wave being halted until other waves allow it to continue.  Such
   * instructions include wave execution barriers, wave synchronization
   * barriers, and wave semaphores.  The instruction has no information.
   */
  AMD_DBGAPI_INSTRUCTION_KIND_BARRIER = 11,
  /**
   * The instruction causes the wave to stop executing for some period of time,
   * before continuing execution with the next instruction.  The instruction
   * has no information.
   */
  AMD_DBGAPI_INSTRUCTION_KIND_SLEEP = 12,
  /**
   * The instruction has some form of special behavior not covered by any of
   * the other instruction kinds.  This likely makes it unsuitable to assume it
   * will execute sequentially.  This may include instructions that can affect
   * the execution of other waves waiting at wave synchronization barriers,
   * that may send interrupts, and so forth.  The instruction has no
   * information.
   */
  AMD_DBGAPI_INSTRUCTION_KIND_SPECIAL = 13
} amd_dbgapi_instruction_kind_t;

/**
 * A bit mask of the properties of an instruction.
 */
typedef enum
{
  /**
   * The instruction has no properties.
   */
  AMD_DBGAPI_INSTRUCTION_PROPERTY_NONE = 0
} amd_dbgapi_instruction_properties_t;

/**
 * Classify a single instruction.
 *
 * \param[in] architecture_id The architecture to use to perform the
 * classification.
 *
 * \param[in] address The address of the first byte of the instruction.
 *
 * \param[in,out] size Pass in the number of bytes available in \p memory which
 * must be greater than 0.  Return the number of bytes consumed to decode the
 * instruction.
 *
 * \param[in] memory The bytes to decode as an instruction.  Must point to an
 * array of at least \p size bytes.  The
 * ::AMD_DBGAPI_ARCHITECTURE_INFO_LARGEST_INSTRUCTION_SIZE query for \p
 * architecture_id can be used to determine the number of bytes of the largest
 * instruction.  By making \p size at least this size ensures that the
 * instruction can be decoded if legal.  However, \p size may need to be
 * smaller if no memory exists at the address of \p address plus \p size.
 *
 * \param[out] instruction_kind The classification kind of the instruction.
 *
 * \param[out] instruction_properties Pointer to the instruction properties.
 * ::amd_dbgapi_instruction_properties_t defines the type of the instruction
 * properties.  If NULL, no value is returned.
 *
 * \param[out] instruction_information Pointer to the instruction information
 * that corresponds to the value of \p instruction_kind.
 * ::amd_dbgapi_instruction_kind_t defines the type of the instruction
 * information for each instruction kind value.  If the instruction has no
 * information then NULL is returned.  The memory is allocated using the
 * amd_dbgapi_callbacks_s::allocate_memory callback and is owned by the client.
 * If NULL, no value is returned.
 *
 * \retval ::AMD_DBGAPI_STATUS_SUCCESS The function has been executed
 * successfully; and the result is stored in \p instruction_kind, \p
 * instruction_properties, and \p instruction_information.
 *
 * \retval ::AMD_DBGAPI_STATUS_FATAL A fatal error occurred.  The library is
 * left uninitialized; and \p size, \p instruction_kind, \p
 * instruction_properties, and \p instruction_information are unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_NOT_INITIALIZED The library is not
 * initialized.  The library is left uninitialized; and \p size, \p
 * instruction_kind, \p instruction_properties, and \p instruction_information
 * are unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ARCHITECTURE_ID \p architecture_id
 * is invalid.  \p size, \p instruction_kind, \p instruction_properties, and \p
 * instruction_information are unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ARGUMENT \p size, \p memory, or \p
 * instruction_kind are NULL, \p size is 0, or \p address is not aligned on the
 * value returned by the
 * ::AMD_DBGAPI_ARCHITECTURE_INFO_MINIMUM_INSTRUCTION_ALIGNMENT query.  \p
 * size, \p instruction_kind, \p instruction_properties, and \p
 * instruction_information are unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR Encountered an error disassembling the
 * instruction.  The bytes may or may not be a legal instruction.  \p size, \p
 * instruction_kind, \p instruction_properties, and \p instruction_information
 * are unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_ILLEGAL_INSTRUCTION The bytes starting at
 * \p address, when up to \p size bytes are available, are not a legal
 * instruction for the architecture.  \p size, \p instruction_kind, \p
 * instruction_properties, and \p instruction_information are unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_CLIENT_CALLBACK This will be reported if
 * the amd_dbgapi_callbacks_s::allocate_memory callback used to allocate \p
 * instruction_text and \p address_operands returns NULL.  \p size, \p
 * instruction_kind, \p instruction_properties, and \p instruction_information
 * are unaltered.
 */
amd_dbgapi_status_t AMD_DBGAPI amd_dbgapi_classify_instruction (
    amd_dbgapi_architecture_id_t architecture_id,
    amd_dbgapi_global_address_t address, amd_dbgapi_size_t *size,
    const void *memory, amd_dbgapi_instruction_kind_t *instruction_kind,
    amd_dbgapi_instruction_properties_t *instruction_properties,
    void **instruction_information) AMD_DBGAPI_VERSION_0_58;

/** @} */

/** \defgroup process_group Processes
 *
 * Operations related to establishing AMD GPU debug control of a process.
 *
 * The library supports AMD GPU debug control of multiple operating system
 * processes.  Each process can have access to multiple AMD GPU devices, but
 * each process uses the AMD GPU devices independently of other processes.
 *
 * @{
 */

/**
 * Opaque client process handle.
 *
 * A pointer to client data associated with a process.  This pointer is passed
 * to the process specific callbacks (see \ref callbacks_group) to allow the
 * client of the library to identify the process.  Each process must have a
 * single unique value.
 */
typedef struct amd_dbgapi_client_process_s *amd_dbgapi_client_process_id_t;

/**
 * Opaque process handle.
 *
 * All operations that control an AMD GPU specify the process that is using the
 * AMD GPU with the process handle.  It is undefined to use handles returned by
 * operations performed for one process, with operations performed for a
 * different process.
 *
 * Globally unique for a single library instance.
 */
typedef struct
{
  uint64_t handle;
} amd_dbgapi_process_id_t;

/**
 * The NULL process handle.
 */
#define AMD_DBGAPI_PROCESS_NONE                                               \
    AMD_DBGAPI_HANDLE_LITERAL (amd_dbgapi_process_id_t, 0)

/**
 * Byte endianness encoding.
 */
typedef enum
{
  /** Encoding is done using big endian.  */
  AMD_DBGAPI_ENDIAN_BIG = 0,

  /** Encoding is done using little endian.  */
  AMD_DBGAPI_ENDIAN_LITTLE = 1
} amd_dbgapi_endianness_t;

/**
 * AMDGPU corefile state data for a process.
 */
typedef struct
{
  /** Endianness encoding of the core state.  */
  amd_dbgapi_endianness_t endianness;
  /** Size, in bytes, of the buffer pointed by
   * ::amd_dbgapi_core_state_data_t::data.
   */
  size_t size;
  /**
   * Pointer to the buffer containing the core state data.
   *
   * The buffer is ::amd_dbgapi_core_state_data_t::size bytes long.  See
   * [User Guide for AMDGPU Backend - Core file notes]
   * (https://llvm.org/docs/AMDGPUUsage.html#amdgpu-corefile-note).
   */
  const void *data;
} amd_dbgapi_core_state_data_t;

/**
 * Process queries that are supported by ::amd_dbgapi_process_get_info.
 *
 * Each query specifies the type of data returned in the \p value argument to
 * ::amd_dbgapi_process_get_info.
 */
typedef enum
{
  /**
   * The notifier for the process that indicates if pending events are
   * available.  The type of this attributes is ::amd_dbgapi_notifier_t.
   */
  AMD_DBGAPI_PROCESS_INFO_NOTIFIER = 1,
  /**
   * Return the number of data watchpoints supported by the process.  Zero
   * is returned if data watchpoints are not supported.  The type of this
   * attribute is \p size_t.
   */
  AMD_DBGAPI_PROCESS_INFO_WATCHPOINT_COUNT = 2,
  /**
   * Return how watchpoints are shared between processes.  The type of this
   * attribute is \p uint32_t with the values defined by
   * ::amd_dbgapi_watchpoint_share_kind_t.
   */
  AMD_DBGAPI_PROCESS_INFO_WATCHPOINT_SHARE = 3,
  /**
   * Return if the architectures of all the agents of a process support
   * controlling memory precision.  The type of this attribute is \p uint32_t
   * with the values defined by ::amd_dbgapi_memory_precision_t.
   */
  AMD_DBGAPI_PROCESS_INFO_PRECISE_MEMORY_SUPPORTED = 4,
  /**
   * Return if the architectures of all the agents of a process support
   * controlling ALU exceptions reporting precision.  The type of this
   * attribute is \p uint32_t with the values defined by
   * ::amd_dbgapi_alu_exceptions_precision_t.
   */
  AMD_DBGAPI_PROCESS_INFO_PRECISE_ALU_EXCEPTIONS_SUPPORTED = 5,
  /**
   * Native operating system process ID.  The type of this attribute is
   * ::amd_dbgapi_os_process_id_t.
   *
   * If the native operating system process was exited when
   * ::amd_dbgapi_process_attach attached to the process, then
   * ::amd_dbgapi_process_get_info returns the
   * ::AMD_DBGAPI_STATUS_ERROR_NOT_AVAILABLE error.
   *
   * If the process image was rebuilt using a core dump, then
   * ::amd_dbgapi_process_get_info returns the
   * ::AMD_DBGAPI_STATUS_ERROR_NOT_AVAILABLE error.
   */
  AMD_DBGAPI_PROCESS_INFO_OS_ID = 6,
  /**
   * Return a blob containing the content to put in the state note when
   * generating a core dump.  The content of the note is allocated using the
   * amd_dbgapi_callbacks_s::allocate_memory callback and is owned by the
   * client.
   *
   * If allocation fails, then ::amd_dbgapi_process_get_info returns the
   * ::AMD_DBGAPI_STATUS_ERROR_CLIENT_CALLBACK error.
   */
  AMD_DBGAPI_PROCESS_INFO_CORE_STATE = 7,
} amd_dbgapi_process_info_t;

/**
 * Query information about a process.
 *
 * ::amd_dbgapi_process_info_t specifies the queries supported and the
 * type returned using the \p value argument.
 *
 * \param[in] process_id The process being queried.
 *
 * \param[in] query The query being requested.
 *
 * \param[in] value_size Size of the memory pointed to by \p value.  Must be
 * equal to the byte size of the query result.
 *
 * \param[out] value Pointer to memory where the query result is stored.
 *
 * \retval ::AMD_DBGAPI_STATUS_SUCCESS The function has been executed
 * successfully and the result is stored in \p value.
 *
 * \retval ::AMD_DBGAPI_STATUS_FATAL A fatal error occurred.  The library is
 * left uninitialized and \p value is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_NOT_INITIALIZED The library is not
 * initialized.  The library is left uninitialized and \p value is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_PROCESS_ID \p process_id is
 * invalid.  \p value is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ARGUMENT \p value is NULL or
 * \p query is invalid.  \p value is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ARGUMENT_COMPATIBILITY \p
 * value_size does not match the size of the \p query result.  \p value is
 * unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_NOT_AVAILABLE \p The requested information
 * is not available.  See ::amd_dbgapi_process_info_t for queries that can
 * produce this error.  \p value is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_CLIENT_CALLBACK This will be
 * reported if the amd_dbgapi_callbacks_s::allocate_memory callback used to
 * allocate \p value returns NULL.  \p value is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_PROCESS_NOT_FROZEN The request is
 * ::AMD_DBGAPI_PROCESS_INFO_CORE_STATE but the process is not frozen.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_RESTRICTION The request is
 * ::AMDGPU_DBGAPI_PROCESS_INFO_CORE_STATE but the process configuration does
 * not permit the creation of a reliable core dump.
 */
amd_dbgapi_status_t AMD_DBGAPI amd_dbgapi_process_get_info (
    amd_dbgapi_process_id_t process_id, amd_dbgapi_process_info_t query,
    size_t value_size, void *value) AMD_DBGAPI_VERSION_0_77;

/**
 * Attach to a process in order to provide debug control of the AMD GPUs it
 * uses.
 *
 * Attaching can be performed on processes that have not started executing, as
 * well as those that are already executing.
 *
 * The process progress is initialized to ::AMD_DBGAPI_PROGRESS_NORMAL.  All
 * agents accessed by the process are configured to
 * ::AMD_DBGAPI_MEMORY_PRECISION_NONE and
 * ::AMD_DBGAPI_ALU_EXCEPTIONS_PRECISION_NONE.
 *
 * When attaching to a live process, the client process handle must have been
 * associated with a native operating system process, and the
 * amd_dbgapi_callbacks_s::client_process_get_info callback with the
 * ::AMD_DBGAPI_CLIENT_PROCESS_INFO_OS_PID request is used to obtain it.
 *
 * When attaching to a process image (core dump), the client process handle
 * has not been associated with a native operating system, and the
 * amd_dbgapi_callbacks_s::client_process_get_info request must return
 * ::AMD_DBGAPI_STATUS_ERROR_NOT_AVAILABLE.
 *
 * It is the client's responsibility to fetch the current code object list
 * using ::amd_dbgapi_process_code_object_list as the
 * ::AMD_DBGAPI_EVENT_KIND_CODE_OBJECT_LIST_UPDATED event is only reported when
 * a thread is in the process of changing the code object list.
 *
 * The ::AMD_DBGAPI_EVENT_KIND_RUNTIME event will be reported if the inferior's
 * runtime support is already enabled.
 *
 * If the associated native operating system process exits while the library is
 * attached to it, appropriate actions are taken to reflect that the inferior
 * process no longer has any state.  For example, pending events are created
 * for wave command termination if there are pending wave stop or wave single
 * step requests; a pending code object list updated event is created if there
 * were codes objects previously loaded; a pending runtime event is created to
 * indicate the inferior's runtime support has been unloaded if previously
 * loaded; and queries on agents, queues, dispatches, waves, and code objects
 * will report none exist.  The process handle remains valid until
 * ::amd_dbgapi_process_detach is used to detach from the client process.
 *
 * If the associated native operating system process has already exited when
 * attaching, then the attach is still successful, but any queries on agents,
 * queues, dispatches, waves, and code objects will report none exist.
 *
 * If the associated native operating system process exits while a library
 * operation is being executed, then the operation behaves as if the process
 * exited before it was invoked.  For example, a wave operation will report an
 * invalid wave handle, a list query will report an empty list, and so forth.
 *
 * It is undefined to use any library operation except
 * ::amd_dbgapi_process_detach on a process that has its virtual address space
 * replaced.  After detach, the same process can be attached again to continue
 * accessing the process if desired.  For example, in Linux an ``exec`` system
 * call replaces the virtual address space which causes all information about
 * agents, queues, dispatches, and waves to become invalid, and the ability to
 * read and write memory may also no longer be allowed by the operating system.
 *
 * If after attaching to a process it spawns another process, the library
 * continues to be attached to the parent process.  If desired, the client can
 * always use ::amd_dbgapi_process_attach to attach to the child process and
 * ::amd_dbgapi_process_detach to detach from the parent process.
 *
 * \param[in] client_process_id The client handle for the process.  It is
 * passed as an argument to any callbacks performed to indicate the process
 * being requested.
 *
 * \param[out] process_id The process handle to use for all operations related
 * to this process.
 *
 * \retval ::AMD_DBGAPI_STATUS_SUCCESS The function has been executed
 * successfully and the process is now attached returning \p process_id.
 *
 * \retval ::AMD_DBGAPI_STATUS_FATAL A fatal error occurred.  The library is
 * left uninitialized and \p process_id is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_NOT_INITIALIZED The library is not
 * initialized.  The library is left uninitialized and \p process_id is
 * unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_ALREADY_ATTACHED The process is already
 * attached.  The process remains attached and \p process_id is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_RESTRICTION There is a restriction error
 * that prevents debugging process \p client_process_id.  See
 * ::AMD_DBGAPI_STATUS_ERROR_RESTRICTION for possible reasons.  The process is
 * not attached and \p process_id is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ARGUMENT \p client_process_id or
 * \p process_id are NULL.  The process is not attached and \p process_id is
 * unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR Encountered some other error while
 * attaching to the process.  The process is not attached and \p process_id is
 * unaltered.
 */
amd_dbgapi_status_t AMD_DBGAPI amd_dbgapi_process_attach (
    amd_dbgapi_client_process_id_t client_process_id,
    amd_dbgapi_process_id_t *process_id) AMD_DBGAPI_VERSION_0_56;

/**
 * Detach from a process and no longer have debug control of the AMD GPU
 * devices it uses.
 *
 * If the associated native operating system process has already exited, or
 * exits while being detached, then the process is trivially detached.
 *
 * Otherwise, detaching causes execution of the associated native operating
 * system process to continue unaffected by the library.  Any waves with a
 * displaced stepping buffer are stopped and the displaced stepping buffer
 * completed.  Any data watchpoints are removed.  All agents are configured to
 * ::AMD_DBGAPI_MEMORY_PRECISION_NONE and
 * ::AMD_DBGAPI_ALU_EXCEPTIONS_PRECISION_NONE.  Any waves in the stopped or
 * single step state are resumed in non-single step mode.  Any pending events
 * for the process are discarded, and no further events will be generated for
 * the process.  If the process is in the frozen state, it is unfrozen.  The
 * wave creation mode is restored to ::AMD_DBGAPI_WAVE_CREATION_NORMAL.
 *
 * After detaching, the process handle, and all handles associated with
 * entities relating to the process, become invalid.
 *
 * A native operating system process can be attached and detached multiple
 * times.  Each attach returns a unique process handle even for the same native
 * operating system process.
 *
 * The client is responsible for removing any inserted breakpoints before
 * detaching.  Failing to do so will cause execution of a breakpoint
 * instruction to put the queue into an error state, aborting any executing
 * waves for dispatches on that queue.
 *
 * \param process_id The process handle that is being detached.
 *
 * \retval ::AMD_DBGAPI_STATUS_SUCCESS The function has been executed
 * successfully and the process has been detached from the associated native
 * operating system process, or the associated native operating system process
 * has already exited.
 *
 * \retval ::AMD_DBGAPI_STATUS_FATAL A fatal error occurred.  The library is
 * left uninitialized.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_NOT_INITIALIZED The library is not
 * initialized.  The library is left uninitialized.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_PROCESS_ID The \p process_id is
 * invalid.  No process is detached.
 */
amd_dbgapi_status_t AMD_DBGAPI amd_dbgapi_process_detach (
    amd_dbgapi_process_id_t process_id) AMD_DBGAPI_VERSION_0_54;

/**
 * The kinds of progress supported by the library.
 *
 * In performing operations, the library may make both waves it needs to
 * access, as well as other waves, unavailable for hardware execution.  After
 * completing the operation, it will make all waves available for hardware
 * execution.  This is termed pausing and unpausing wave execution
 * respectively.  Pausing and unpausing waves for each command separately works
 * but can result in longer latency than if several commands could be performed
 * while the waves are paused.  Debugging the very large number of waves that
 * can exist on an AMD GPU can involve many operations, making batching
 * commands even more beneficial.  The progress setting allows controlling this
 * behavior.
 */
typedef enum
{
  /**
   * Normal progress is needed.  Commands are issued immediately.  After
   * completing each command all non-stopped waves will be unpaused.  Switching
   * from another progress mode to this will unpause any waves that are paused.
   */
  AMD_DBGAPI_PROGRESS_NORMAL = 0,
  /**
   * No forward progress is needed.  Commands are issued immediately.  After
   * completing each command, non-stopped waves may be left paused.  The waves
   * left paused may include both the wave(s) the command operates on, as well
   * as other waves.  While in ::AMD_DBGAPI_PROGRESS_NO_FORWARD mode, paused
   * waves may remain paused, or may be unpaused at any point.  Only by leaving
   * ::AMD_DBGAPI_PROGRESS_NO_FORWARD mode will the library not leave any
   * waves paused after completing a command.
   *
   * Note that the events that ::amd_dbgapi_wave_stop causes to be reported
   * will occur when in ::AMD_DBGAPI_PROGRESS_NO_FORWARD mode.  It is not
   * necessary to change the progress mode to ::AMD_DBGAPI_PROGRESS_NORMAL for
   * those events to be reported.
   *
   * This can result in a series of commands completing far faster than in
   * ::AMD_DBGAPI_PROGRESS_NORMAL mode.  Also, any queries for lists such as
   * ::amd_dbgapi_process_wave_list may return \p unchanged as true more often,
   * reducing the work needed to parse the lists to determine what has changed.
   * With large lists this can be significant.  If the client needs a wave to
   * complete a single step resume, then it must leave
   * ::AMD_DBGAPI_PROGRESS_NO_FORWARD mode in order to prevent that wave from
   * remaining paused.
   */
  AMD_DBGAPI_PROGRESS_NO_FORWARD = 1
} amd_dbgapi_progress_t;

/**
 * Set the progress required for a process.
 *
 * \param[in] process_id If ::AMD_DBGAPI_PROCESS_NONE then set the progress for
 * all processes currently attached.  Otherwise, set the progress for the
 * process \p process_id.
 *
 * \param[in] progress The progress being set.
 *
 * \retval ::AMD_DBGAPI_STATUS_SUCCESS The function has been executed
 * successfully and the progress has been set.
 *
 * \retval ::AMD_DBGAPI_STATUS_FATAL A fatal error occurred.  The library is
 * left uninitialized.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_NOT_INITIALIZED The library is not
 * initialized.  The library is left uninitialized.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_PROCESS_ID \p process_id is
 * invalid.  The progress setting is not changed.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ARGUMENT \p progress is invalid.
 * The progress setting is not changed.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_PROCESS_FROZEN The process is frozen.  The
 * progress setting cannot be changed and must remain
 * ::AMD_DBGAPI_PROGRESS_NO_FORWARD.
 */
amd_dbgapi_status_t AMD_DBGAPI amd_dbgapi_process_set_progress (
    amd_dbgapi_process_id_t process_id,
    amd_dbgapi_progress_t progress) AMD_DBGAPI_VERSION_0_76;

/**
 * The kinds of wave creation supported by the hardware.
 *
 * The hardware creates new waves asynchronously as it executes dispatch
 * packets.  If the client requires that all waves are stopped, it needs to
 * first request that the hardware stops creating new waves, followed by
 * halting all already created waves.  The wave creation setting allows
 * controlling how the hardware creates new waves for dispatch packets on
 * queues associated with agents belonging to a specific process.  It has no
 * affect on waves that have already been created.
 */
typedef enum
{
  /**
   * Normal wave creation allows new waves to be created.
   */
  AMD_DBGAPI_WAVE_CREATION_NORMAL = 0,
  /**
   * Stop wave creation prevents new waves from being created.
   */
  AMD_DBGAPI_WAVE_CREATION_STOP = 1
} amd_dbgapi_wave_creation_t;

/**
 * Set the wave creation mode for a process.
 *
 * The setting applies to all agents of the specified process.
 *
 * \param[in] process_id The process being controlled.
 *
 * \param[in] creation The wave creation mode being set.
 *
 * \retval ::AMD_DBGAPI_STATUS_SUCCESS The function has been executed
 * successfully and the wave creation mode has been set.
 *
 * \retval ::AMD_DBGAPI_STATUS_FATAL A fatal error occurred.  The library is
 * left uninitialized.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_NOT_INITIALIZED The library is not
 * initialized.  The library is left uninitialized.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_PROCESS_ID \p process_id is
 * invalid.  The wave creation mode setting is not changed.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ARGUMENT \p creation is invalid.
 * The wave creation setting is not changed.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_PROCESS_FROZEN The process is frozen.
 * The wave creation mode setting cannot be changed and must remain
 * ::AMD_DBGAPI_WAVE_CREATION_STOP.
 */
amd_dbgapi_status_t AMD_DBGAPI amd_dbgapi_process_set_wave_creation (
    amd_dbgapi_process_id_t process_id,
    amd_dbgapi_wave_creation_t creation) AMD_DBGAPI_VERSION_0_76;

/** \defgroup coredump_support Generating a core dump of a process
 *
 * Operations related to generating and using core dumps.
 *
 * When the client needs to generate a core dump of a process, the following
 * steps are followed:
 *
 * -# The client suspends the execution of all host threads.
 * -# The client sets the progress mode for the process to
 *    ::AMD_DBGAPI_PROGRESS_NO_FORWARD and stops wave creation.
 * -# The client calls ::amd_dbgapi_process_freeze to suspend execution on
 *    the agents.
 * -# The client queries ::AMD_DBGAPI_PROCESS_INFO_CORE_STATE and stores
 *    the content of the returned buffer in a note in the generated core file.
 * -# The client includes in the core dump all the information required to
 *    form a core dump of the host process.
 * -# The client unfreezes the process using ::amd_dbgapi_process_unfreeze.
 * -# The client can set progress to ::AMD_DBGAPI_PROGRESS_NORMAL and resume
 *    the execution of host threads as desired.
 *
 * @{
 */

/**
 * Freeze the process identified by \p process_id.
 *
 * The library ensures that all queues belonging to process \p process_id are
 * stopped.  All waves are stopped and left in a state suitable to be
 * discovered by another instance of the library.  Any cached updates to
 * memory or registers are flushed.
 *
 * It is required that the client sets the process's progress to
 * ::AMD_DBGAPI_PROGRESS_NO_FORWARD and the wave creation mode to
 * ::AMD_DBGAPI_WAVE_CREATION_STOP before calling this procedure.
 *
 * It is expected that all displaced stepping buffers are disabled before
 * calling this operation.  If displaced stepping buffers are still enabled
 * when performing this operation, then another instance of the library will
 * see an invalid program counter for the associated wave(s).
 *
 * It is expected that all host threads are suspended by the client before
 * calling this method as executing threads might create queues and submit
 * dispatches.  If any thread is running on the host process, the behavior
 * and state of the library are undefined.
 *
 * \param[in] process_id The client handle of the process to freeze.
 *
 * \retval ::AMD_DBGAPI_STATUS_SUCCESS The function has been executed
 * successfully and the process is in the frozen state.
 *
 * \retval ::AMD_DBGAPI_STATUS_FATAL A fatal error occurred.  The library is
 * left uninitialized.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_NOT_INITIALIZED The library is not
 * initialized.  The library is left uninitialized.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_PROCESS_ID \p process_id is
 * invalid.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INCOMPATIBLE_PROCESS_STATE if either progress
 * mode for the process identified by \p process_id is not
 * ::AMD_DBGAPI_PROGRESS_NO_FORWARD or if wave creation mode for the process
 * identified by \p process_id is not ::AMD_DBGAPI_WAVE_CREATION_STOP.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_PROCESS_ALREADY_FROZEN The process \p
 * process_id is already frozen.  The process is not changed.
 *
 */
amd_dbgapi_status_t AMD_DBGAPI amd_dbgapi_process_freeze (
    amd_dbgapi_process_id_t process_id) AMD_DBGAPI_VERSION_0_76;

/**
 * Unfreeze the process identified by \p process_id.
 *
 * After calling this, the library is allowed to keep writes to registers and
 * memory in an internal cache until the effects are needed to resume
 * execution.
 *
 * \param[in] process_id The client handle of the process to unfreeze.
 *
 * \retval ::AMD_DBGAPI_STATUS_SUCCESS The function has been executed
 * successfully and the process is unfrozen.
 *
 * \retval ::AMD_DBGAPI_STATUS_FATAL A fatal error occurred.  The library is
 * left uninitialized.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_NOT_INITIALIZED The library is not
 * initialized.  The library is left uninitialized.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_PROCESS_ID \p process_id is
 * invalid.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_PROCESS_NOT_FROZEN The process \p
 * process_id is not frozen.  The process is not changed.
 */
amd_dbgapi_status_t AMD_DBGAPI amd_dbgapi_process_unfreeze (
    amd_dbgapi_process_id_t process_id) AMD_DBGAPI_VERSION_0_76;

/** @}
  * @} */

/** \defgroup code_object_group Code Objects
 *
 * Operations related to AMD GPU code objects loaded into a process.
 *
 * AMD GPU code objects are standard ELF shared libraries defined in
 * [User Guide for AMDGPU Backend - Code Object]
 * (https://llvm.org/docs/AMDGPUUsage.html#code-object).
 *
 * AMD GPU code objects can be embedded in the host executable code object
 * that is loaded into memory or be in a separate file in the file system.
 * The AMD GPU loader supports loading either from memory or from files.  The
 * loader selects the segments to put into memory that contain the code and
 * data necessary for AMD GPU code execution.  It allocates global memory to
 * map these segments and performs necessary relocations to create the loaded
 * code object.
 *
 * @{
 */

/**
 * Opaque code object handle.
 *
 * Globally unique for a single library instance.
 */
typedef struct
{
  uint64_t handle;
} amd_dbgapi_code_object_id_t;

/**
 * The NULL code object handle.
 */
#define AMD_DBGAPI_CODE_OBJECT_NONE                                           \
    AMD_DBGAPI_HANDLE_LITERAL (amd_dbgapi_code_object_id_t, 0)

/**
 * Code object queries that are supported by ::amd_dbgapi_code_object_get_info.
 *
 * Each query specifies the type of data returned in the \p value argument to
 * ::amd_dbgapi_code_object_get_info.
 */
typedef enum
{
  /**
   * Return the process to which this code object belongs.  The type of this
   * attribute is ::amd_dbgapi_process_id_t.
   */
  AMD_DBGAPI_CODE_OBJECT_INFO_PROCESS = 1,
  /**
   * The URI name of the ELF shared object from which the code object was
   * loaded.  Note that the code object is the in memory loaded relocated form
   * of the ELF shared object.  Multiple code objects may be loaded at
   * different memory addresses in the same process from the same ELF shared
   * object.
   *
   * The type of this attribute is a NUL terminated \p char*.  It is allocated
   * by the amd_dbgapi_callbacks_s::allocate_memory callback and is owned by
   * the client.
   *
   * The URI name syntax is defined by the following BNF syntax:
   *
   *     code_object_uri ::== file_uri | memory_uri
   *     file_uri        ::== "file://" file_path
   *                          [ range_specifier ]
   *     memory_uri      ::== "memory://" process_id
   *                          range_specifier
   *     range_specifier ::== ( "#" | "?" )
   *                          "offset=" number
   *                          "&" "size=" number
   *     file_path       ::== URI_ENCODED_OS_FILE_PATH
   *     process_id      ::== DECIMAL_NUMBER
   *     number          ::== HEX_NUMBER
   *                        | DECIMAL_NUMBER
   *                        | OCTAL_NUMBER
   *
   * ``DECIMAL_NUMBER`` is a decimal C integral literal, ``HEX_NUMBER`` is a
   * hexadecimal C integral literal with a "0x" or "0X" prefix, and
   * ``OCTAL_NUMBER`` is an octal C integral literal with a "0" prefix.
   *
   * ``URI_ENCODED_OS_FILE_PATH`` is a file path specified as a URI encoded
   * UTF-8 string.  In URI encoding, every character that is not in the regular
   * expression ``[a-zA-Z0-9/_.~-]`` is encoded as two uppercase hexadecimal
   * digits proceeded by "%".  Directories in the path are separated by "/".
   *
   * ``offset`` is a 0-based byte offset to the start of the code object.  For
   * a file URI, it is from the start of the file specified by the
   * ``file_path``, and if omitted defaults to 0. For a memory URI, it is the
   * memory address and is required.
   *
   * ``size`` is the number of bytes in the code object.  For a file URI, if
   * omitted it defaults to the size of the file.  It is required for a memory
   * URI.
   *
   * ``process_id`` is the identity of the process owning the memory.  For
   * Linux it is the C unsigned integral decimal literal for the process ID
   * (PID).
   *
   * For example:
   *
   *     file:///dir1/dir2/file1
   *     file:///dir3/dir4/file2#offset=0x2000&size=3000
   *     memory://1234#offset=0x20000&size=3000
   */
  AMD_DBGAPI_CODE_OBJECT_INFO_URI_NAME = 2,
  /**
   * The difference between the address in the ELF shared object and the
   * address the code object is loaded in memory.  The type of this attributes
   * is \p ptrdiff_t.
   */
  AMD_DBGAPI_CODE_OBJECT_INFO_LOAD_ADDRESS = 3
} amd_dbgapi_code_object_info_t;

/**
 * Query information about a code object.
 *
 * ::amd_dbgapi_code_object_info_t specifies the queries supported and the
 * type returned using the \p value argument.
 *
 * \param[in] code_object_id The handle of the code object being queried.
 *
 * \param[in] query The query being requested.
 *
 * \param[in] value_size Size of the memory pointed to by \p value.  Must be
 * equal to the byte size of the query result.
 *
 * \param[out] value Pointer to memory where the query result is stored.
 *
 * \retval ::AMD_DBGAPI_STATUS_SUCCESS The function has been executed
 * successfully and the result is stored in \p value.
 *
 * \retval ::AMD_DBGAPI_STATUS_FATAL A fatal error occurred.  The library is
 * left uninitialized and \p value is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_NOT_INITIALIZED The library is not
 * initialized.  The library is left uninitialized and \p value is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_CODE_OBJECT_ID \p code_object_id
 * is invalid.  \p value is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ARGUMENT \p value is NULL or
 * \p query is invalid.  \p value is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ARGUMENT_COMPATIBILITY \p
 * value_size does not match the size of the \p query result.  \p value is
 * unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_CLIENT_CALLBACK This will be
 * reported if the amd_dbgapi_callbacks_s::allocate_memory callback used to
 * allocate \p value returns NULL.  \p value is unaltered.
 */
amd_dbgapi_status_t AMD_DBGAPI amd_dbgapi_code_object_get_info (
    amd_dbgapi_code_object_id_t code_object_id,
    amd_dbgapi_code_object_info_t query, size_t value_size,
    void *value) AMD_DBGAPI_VERSION_0_54;

/**
 * Return the list of loaded code objects.
 *
 * The order of the code object handles in the list is unspecified and can vary
 * between calls.
 *
 * \param[in] process_id If ::AMD_DBGAPI_PROCESS_NONE then the code object list
 * for all processes is requested.  Otherwise, the code object list for process
 * \p process_id is requested.
 *
 * \param[out] code_object_count The number of code objects currently loaded.
 *
 * \param[out] code_objects If \p changed is not NULL and the code object list
 * of all of the processes requested have not changed since the last call(s) to
 * ::amd_dbgapi_process_code_object_list for each of them, then return NULL.
 * Otherwise, return a pointer to an array of ::amd_dbgapi_code_object_id_t
 * with \p code_object_count elements.  It is allocated by the
 * amd_dbgapi_callbacks_s::allocate_memory callback and is owned by the client.
 *
 * \param[in,out] changed If NULL then left unaltered.  If non-NULL, set to
 * ::AMD_DBGAPI_CHANGED_NO if the list of code objects for each requested
 * process is the same as when ::amd_dbgapi_process_code_object_list was last
 * called for them.  Otherwise, set to ::AMD_DBGAPI_CHANGED_YES.
 *
 * \retval ::AMD_DBGAPI_STATUS_SUCCESS The function has been executed
 * successfully and the result is stored in \p changed, \p code_object_count,
 * and \p code_objects.
 *
 * \retval ::AMD_DBGAPI_STATUS_FATAL A fatal error occurred.  The library is
 * left uninitialized; and \p code_object_count, \p code_objects, and \p
 * changed are unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_NOT_INITIALIZED The library is not
 * initialized.  The library is left uninitialized; and \p code_object_count,
 * \p code_objects, and \p changed are unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_PROCESS_ID \p process_id is
 * invalid.  \p code_object_count, \p code_objects, and \p changed are
 * unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ARGUMENT \p code_object_count or
 * \p code_objects are NULL, or \p changed is invalid.  \p code_object_count,
 * \p code_objects, and \p changed are unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_CLIENT_CALLBACK This will be reported if
 * the amd_dbgapi_callbacks_s::allocate_memory callback used to allocate \p
 * code_objects returns NULL.  \p code_object_count, \p code_objects, and \p
 * changed are unaltered.
 */
amd_dbgapi_status_t AMD_DBGAPI amd_dbgapi_process_code_object_list (
    amd_dbgapi_process_id_t process_id, size_t *code_object_count,
    amd_dbgapi_code_object_id_t **code_objects,
    amd_dbgapi_changed_t *changed) AMD_DBGAPI_VERSION_0_54;

/** @} */

/** \defgroup agent_group Agents
 *
 * Operations related to AMD GPU agents accessible to a process.
 *
 * Agent is the term for AMD GPU devices that can be accessed by the process.
 *
 * @{
 */

/**
 * Opaque agent handle.
 *
 * Globally unique for a single library instance.
 */
typedef struct
{
  uint64_t handle;
} amd_dbgapi_agent_id_t;

/**
 * The NULL agent handle.
 */
#define AMD_DBGAPI_AGENT_NONE                                                 \
    AMD_DBGAPI_HANDLE_LITERAL (amd_dbgapi_agent_id_t, 0)

/**
 * Agent queries that are supported by ::amd_dbgapi_agent_get_info.
 *
 * Each query specifies the type of data returned in the \p value argument to
 * ::amd_dbgapi_agent_get_info.
 */
typedef enum
{
  /**
   * Return the process to which this agent belongs.  The type of this
   * attribute is ::amd_dbgapi_process_id_t.
   */
  AMD_DBGAPI_AGENT_INFO_PROCESS = 1,
  /**
   * Agent name.  The type of this attribute is a pointer to a NUL terminated
   * \p char*.  It is allocated by amd_dbgapi_callbacks_s::allocate_memory and
   * is owned by the client.
   */
  AMD_DBGAPI_AGENT_INFO_NAME = 2,
  /**
   * Return the architecture of this agent.  The type of this attribute is
   * ::amd_dbgapi_architecture_id_t.
   *
   * If the architecture of the agent is not supported by the library then
   * ::amd_dbgapi_agent_get_info returns the
   * ::AMD_DBGAPI_STATUS_ERROR_NOT_AVAILABLE error.  See the \ref
   * supported_amd_gpu_architectures section.
   */
  AMD_DBGAPI_AGENT_INFO_ARCHITECTURE = 3,
  /**
   * Return the agent state.  The type of this attribute is \p uint32_t with
   * values from ::amd_dbgapi_agent_state_t.
   * */
  AMD_DBGAPI_AGENT_INFO_STATE = 4,
  /**
   * PCI domain the agent is in. The type of this attribute is \p uint16_t.
   */
  AMD_DBGAPI_AGENT_INFO_PCI_DOMAIN = 5,
  /**
   * PCI slot of the agent in BDF format (see [Bus:Device.Function (BDF)
   * Notation][bfd].
   * The type of this attribute is \p uint16_t.
   */
  AMD_DBGAPI_AGENT_INFO_PCI_SLOT = 6,
  /**
   * PCI vendor ID of the agent.  The type of this attribute is \p uint32_t.
   */
  AMD_DBGAPI_AGENT_INFO_PCI_VENDOR_ID = 7,
  /**
   * PCI device ID of the agent.  The type of this attribute is \p uint32_t.
   */
  AMD_DBGAPI_AGENT_INFO_PCI_DEVICE_ID = 8,
  /**
   * Total number of Execution Units (EUs) available in the agent.  The type of
   * this attribute is \p size_t.
   */
  AMD_DBGAPI_AGENT_INFO_EXECUTION_UNIT_COUNT = 9,
  /**
   * Maximum number of waves supported by an execution unit.  The type of this
   * attribute is \p size_t.
   */
  AMD_DBGAPI_AGENT_INFO_MAX_WAVES_PER_EXECUTION_UNIT = 10,
  /**
   * Native operating system agent ID.  The type of this attribute is
   * ::amd_dbgapi_os_agent_id_t.
   */
  AMD_DBGAPI_AGENT_INFO_OS_ID = 11
} amd_dbgapi_agent_info_t;

/**
 * Agent state.
 */
typedef enum
{
  /**
   * Agent supports debugging.
   */
  AMD_DBGAPI_AGENT_STATE_SUPPORTED = 1,
  /**
   * Agent does not support debugging.  Reasons include:
   *
   * - The architecture of the agent is not supported by the library. See the
   *   \ref supported_amd_gpu_architectures section.  If there is such an agent
   *   then some features may be treated conservatively since the library does
   *   not know if the agent really supports the feature.  The conservative
   *   treatment of such features include:
   *
   *   - ::AMD_DBGAPI_PROCESS_INFO_PRECISE_MEMORY_SUPPORTED is conservatively
   *     treated as ::AMD_DBGAPI_MEMORY_PRECISION_NONE.
   *   - ::AMD_DBGAPI_PROCESS_INFO_PRECISE_ALU_EXCEPTIONS_SUPPORTED is
   *     conservatively treated as ::AMD_DBGAPI_ALU_EXCEPTIONS_PRECISION_NONE.
   *   - ::AMD_DBGAPI_PROCESS_INFO_WATCHPOINT_COUNT is conservatively treated
   *     as 0.
   *
   * - The firmware version of the agent is not compatible with the library.
   * - The AMD GPU driver does not support debugging for the the agent's
   *   architecture.
   *
   * No queues, dispatches, or waves will be reported for the agent.
   */
  AMD_DBGAPI_AGENT_STATE_NOT_SUPPORTED = 2
} amd_dbgapi_agent_state_t;

/**
 * Query information about an agent.
 *
 * ::amd_dbgapi_agent_info_t specifies the queries supported and the type
 * returned using the \p value argument.
 *
 * \param[in] agent_id The handle of the agent being queried.
 *
 * \param[in] query The query being requested.
 *
 * \param[in] value_size Size of the memory pointed to by \p value.  Must be
 * equal to the byte size of the query result.
 *
 * \param[out] value Pointer to memory where the query result is stored.
 *
 * \retval ::AMD_DBGAPI_STATUS_SUCCESS The function has been executed
 * successfully and the result is stored in \p value.
 *
 * \retval ::AMD_DBGAPI_STATUS_FATAL A fatal error occurred.  The library is
 * left uninitialized and \p value is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_NOT_INITIALIZED The library is not
 * initialized.  The library is left uninitialized and \p value is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_AGENT_ID \p agent_id is invalid.
 * \p value is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ARGUMENT \p value is NULL or
 * \p query is invalid.  \p value is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ARGUMENT_COMPATIBILITY \p
 * value_size does not match the size of the \p query result.  \p value is
 * unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_NOT_AVAILABLE \p The requested information
 * is not available.  See ::amd_dbgapi_agent_info_t for queries that can
 * produce this error.  \p value is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_CLIENT_CALLBACK This will be
 * reported if the amd_dbgapi_callbacks_s::allocate_memory callback used to
 * allocate \p value returns NULL.  \p value is unaltered.
 */
amd_dbgapi_status_t AMD_DBGAPI amd_dbgapi_agent_get_info (
    amd_dbgapi_agent_id_t agent_id, amd_dbgapi_agent_info_t query,
    size_t value_size, void *value) AMD_DBGAPI_VERSION_0_67;

/**
 * Return the list of agents.
 *
 * The order of the agent handles in the list is unspecified and can vary
 * between calls.
 *
 * All agents of the process are reported, even if they do not support
 * debugging.  See ::AMD_DBGAPI_AGENT_STATE_NOT_SUPPORTED.
 *
 * \param[in] process_id If ::AMD_DBGAPI_PROCESS_NONE then the agent list for
 * all processes is requested.  Otherwise, the agent list of process \p
 * process_id is requested.
 *
 * \param[out] agent_count The number of agents accessed by the process.
 *
 * \param[out] agents If \p changed is not NULL and the agent list of all of
 * the processes requested have not changed since the last call(s) to
 * ::amd_dbgapi_process_agent_list for each of them, then return NULL.
 * Otherwise, return a pointer to an array of ::amd_dbgapi_agent_id_t with \p
 * agent_count elements.  It is allocated by the
 * amd_dbgapi_callbacks_s::allocate_memory callback and is owned by the client.
 *
 * \param[in,out] changed If NULL then left unaltered.  If non-NULL, set to
 * ::AMD_DBGAPI_CHANGED_NO if the list of agents for each requested process is
 * the same as when ::amd_dbgapi_process_agent_list was last called for them.
 * Otherwise, set to ::AMD_DBGAPI_CHANGED_YES.
 *
 * \retval ::AMD_DBGAPI_STATUS_SUCCESS The function has been executed
 * successfully and the result is stored in \p changed, \p agent_count, and \p
 * agents.
 *
 * \retval ::AMD_DBGAPI_STATUS_FATAL A fatal error occurred.  The library is
 * left uninitialized; and \p agent_count, \p agents, and \p changed are
 * unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_NOT_INITIALIZED The library is not
 * initialized.  The library is left uninitialized; and \p agent_count, \p
 * agents, and \p changed are unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_PROCESS_ID \p process_id is
 * invalid.  \p agent_count, \p agents, and \p changed are unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ARGUMENT \p agent_count or \p
 * agents are NULL, or \p changed is invalid.  \p agent_count, \p agents, and
 * \p changed are unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_CLIENT_CALLBACK This will be reported if
 * the amd_dbgapi_callbacks_s::allocate_memory callback used to allocate \p
 * agents returns NULL.  \p agent_count, \p agents, and \p changed are
 * unaltered.
 */
amd_dbgapi_status_t AMD_DBGAPI amd_dbgapi_process_agent_list (
    amd_dbgapi_process_id_t process_id, size_t *agent_count,
    amd_dbgapi_agent_id_t **agents,
    amd_dbgapi_changed_t *changed) AMD_DBGAPI_VERSION_0_54;

/** @} */

/** \defgroup queue_group Queues
 *
 * Operations related to AMD GPU queues.
 *
 * Queues are user mode data structures that allow packets to be inserted that
 * control the AMD GPU agents.  The dispatch packet is used to initiate the
 * execution of a grid of waves.
 *
 * @{
 */

/**
 * Opaque queue handle.
 *
 * Globally unique for a single library instance.
 */
typedef struct
{
  uint64_t handle;
} amd_dbgapi_queue_id_t;

/**
 * The NULL queue handle.
 */
#define AMD_DBGAPI_QUEUE_NONE                                                 \
    AMD_DBGAPI_HANDLE_LITERAL (amd_dbgapi_queue_id_t, 0)

/**
 * Queue queries that are supported by ::amd_dbgapi_queue_get_info.
 *
 * Each query specifies the type of data returned in the \p value argument to
 * ::amd_dbgapi_queue_get_info.
 */
typedef enum
{
  /**
   * Return the agent to which this queue belongs.  The type of this attribute
   * is ::amd_dbgapi_agent_id_t.
   */
  AMD_DBGAPI_QUEUE_INFO_AGENT = 1,
  /**
   * Return the process to which this queue belongs.  The type of this
   * attribute is ::amd_dbgapi_process_id_t.
   */
  AMD_DBGAPI_QUEUE_INFO_PROCESS = 2,
  /**
   * Return the architecture of this queue.  The type of this attribute is
   * ::amd_dbgapi_architecture_id_t.
   */
  AMD_DBGAPI_QUEUE_INFO_ARCHITECTURE = 3,
  /**
   * Return the queue type.  The type of this attribute is \p uint32_t with
   * values from ::amd_dbgapi_os_queue_type_t.
   */
  AMD_DBGAPI_QUEUE_INFO_TYPE = 4,
  /**
   * Return the queue state.  The type of this attribute is \p uint32_t with
   * values from ::amd_dbgapi_queue_state_t.
   */
  AMD_DBGAPI_QUEUE_INFO_STATE = 5,
  /**
   * Return the set of exceptions that caused the queue to enter the queue
   * error state.  If the queue is not in the queue error state then
   * ::AMD_DBGAPI_EXCEPTION_NONE is returned.  The type of this attribute is
   * \p uint32_t with values defined by ::amd_dbgapi_exceptions_t.
   */
  AMD_DBGAPI_QUEUE_INFO_ERROR_REASON = 6,
  /**
   * Return the base address of the memory holding the queue packets.  The type
   * of this attribute is ::amd_dbgapi_global_address_t.
   */
  AMD_DBGAPI_QUEUE_INFO_ADDRESS = 7,
  /**
   * Return the size in bytes of the memory holding the queue packets.  The
   * type of this attribute is ::amd_dbgapi_size_t.
   */
  AMD_DBGAPI_QUEUE_INFO_SIZE = 8,
  /**
   * Native operating system queue ID.  The type of this attribute is
   * ::amd_dbgapi_os_queue_id_t.
   */
  AMD_DBGAPI_QUEUE_INFO_OS_ID = 9
} amd_dbgapi_queue_info_t;

/**
 * Query information about a queue.
 *
 * ::amd_dbgapi_queue_info_t specifies the queries supported and the type
 * returned using the \p value argument.
 *
 * \param[in] queue_id The handle of the queue being queried.
 *
 * \param[in] query The query being requested.
 *
 * \param[out] value Pointer to memory where the query result is stored.
 *
 * \param[in] value_size Size of the memory pointed to by \p value.  Must be
 * equal to the byte size of the query result.
 *
 * \retval ::AMD_DBGAPI_STATUS_SUCCESS The function has been executed
 * successfully and the result is stored in \p value.
 *
 * \retval ::AMD_DBGAPI_STATUS_FATAL A fatal error occurred.  The library is
 * left uninitialized and \p value is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_NOT_INITIALIZED The library is not
 * initialized.  The library is left uninitialized and \p value is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_QUEUE_ID \p queue_id is invalid.
 * \p value is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ARGUMENT \p value is NULL or
 * \p query is invalid.  \p value is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ARGUMENT_COMPATIBILITY \p
 * value_size does not match the size of the \p query result.  \p value is
 * unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_CLIENT_CALLBACK This will be
 * reported if the amd_dbgapi_callbacks_s::allocate_memory callback used to
 * allocate \p value returns NULL.  \p value is unaltered.
 */
amd_dbgapi_status_t AMD_DBGAPI amd_dbgapi_queue_get_info (
    amd_dbgapi_queue_id_t queue_id, amd_dbgapi_queue_info_t query,
    size_t value_size, void *value) AMD_DBGAPI_VERSION_0_68;

/**
 * Queue state.
 */
typedef enum
{
  /**
   * Queue is in a valid state.
   */
  AMD_DBGAPI_QUEUE_STATE_VALID = 1,
  /**
   * Queue is in the queue error state.
   *
   * No further waves will be started on the queue.  All waves that belong to
   * the queue are inhibited from executing further instructions regardless of
   * whether they are in the halt state.
   *
   * When the inferior's runtime puts a queue into the queue error state, a
   * ::AMD_DBGAPI_EVENT_KIND_QUEUE_ERROR event will be reported.  In addition,
   * any waves that belong to the queue that have pending single step requests
   * will cause a ::AMD_DBGAPI_EVENT_KIND_WAVE_COMMAND_TERMINATED event to be
   * generated to indicate the single step has been cancelled.
   */
  AMD_DBGAPI_QUEUE_STATE_ERROR = 2
} amd_dbgapi_queue_state_t;

/**
 * A bit mask of the exceptions that can cause a queue to enter the queue error
 * state.
 */
typedef enum
{
  /**
   * If none of the bits are set, then the queue is not in the error state.
   */
  AMD_DBGAPI_EXCEPTION_NONE = 0,
  /**
   * A wave on the queue executed a trap instruction used to abort a dispatch.
   */
  AMD_DBGAPI_EXCEPTION_WAVE_ABORT = (1 << 0),
  /**
   * A wave on the queue executed an instruction that caused an exception.
   * This includes executing a trap instruction (other than the trap reported
   * as ::AMD_DBGAPI_EXCEPTION_WAVE_ABORT), executing an instruction that
   * causes a fatal halt, executing an instruction that causes an ECC error, or
   * executing an instruction that triggers a watchpoint (normally watchpoints
   * are handled by the library and are never passed to the inferior's runtime
   * to cause this exception).
   */
  AMD_DBGAPI_EXCEPTION_WAVE_TRAP = (1 << 1),
  /**
   * A wave on the queue executed an instruction that had a floating point or
   * integer enabled exception condition.  The conditions include:
   *
   * - Floating point operation is invalid.
   * - Floating point operation had subnormal input that was
   *   rounded to zero.
   * - Floating point operation performed a division by zero.
   * - Floating point operation produced an overflow result.
   *   The result was rounded to infinity.
   * - Floating point operation produced an underflow result.
   *   A subnormal result was rounded to zero.
   * - Floating point operation produced an inexact result.
   * - Integer operation performed a division by zero.
   */
  AMD_DBGAPI_EXCEPTION_WAVE_MATH_ERROR = (1 << 2),
  /**
   * A wave on the queue executed an illegal instruction.
   */
  AMD_DBGAPI_EXCEPTION_WAVE_ILLEGAL_INSTRUCTION = (1 << 3),
  /**
   * A wave on the queue had a memory violation.  This happens when accessing a
   * non-existent memory page or a page without the necessary permission (such
   * as writing to a readonly page or executing a non-execute page).
   */
  AMD_DBGAPI_EXCEPTION_WAVE_MEMORY_VIOLATION = (1 << 4),
  /**
   * A wave on the queue had an exception due to accessing an invalid memory
   * address.  This includes an address that is not suitably aligned (for
   * example, a non-naturally aligned atomic), or is outside the supported
   * address range for global or flat address apertures.
   */
  AMD_DBGAPI_EXCEPTION_WAVE_ADDRESS_ERROR = (1 << 5),
  /** Old deprecated name kept for backward compatibility.  Will be removed in
    * a future release.
    */
  AMD_DBGAPI_EXCEPTION_WAVE_APERTURE_VIOLATION DEPRECATED
    = AMD_DBGAPI_EXCEPTION_WAVE_ADDRESS_ERROR,
  /**
   * A dispatch packet on the queue has an invalid dimension.
   */
  AMD_DBGAPI_EXCEPTION_PACKET_DISPATCH_DIM_INVALID = (1 << 16),
  /**
   * A dispatch packet on the queue has an invalid group segment size.
   */
  AMD_DBGAPI_EXCEPTION_PACKET_DISPATCH_GROUP_SEGMENT_SIZE_INVALID = (1 << 17),
  /**
   * A dispatch packet on the queue has a NULL code address.
   */
  AMD_DBGAPI_EXCEPTION_PACKET_DISPATCH_CODE_INVALID = (1 << 18),
  /**
   * A packet on the queue has an unsupported code.
   */
  AMD_DBGAPI_EXCEPTION_PACKET_UNSUPPORTED = (1 << 20),
  /**
   * A dispatch packet on the queue has an invalid workgroup size.
   */
  AMD_DBGAPI_EXCEPTION_PACKET_DISPATCH_WORKGROUP_SIZE_INVALID = (1 << 21),
  /**
   * A dispatch packet on the queue requires too many registers.
   */
  AMD_DBGAPI_EXCEPTION_PACKET_DISPATCH_REGISTER_COUNT_TOO_LARGE = (1 << 22),
  /**
   * A packet on the queue has an invalid vendor code.
   */
  AMD_DBGAPI_EXCEPTION_PACKET_VENDOR_UNSUPPORTED = (1 << 23),
  /**
   * There was an error preempting the queue.  When the AMD GPU device driver
   * generates this error it may cause all waves associated with the queue to
   * be killed.  Killing a wave causes it to be terminated immediately without
   * reporting any exceptions.  Any killed waves that have a pending single
   * step will report a ::AMD_DBGAPI_EVENT_KIND_WAVE_COMMAND_TERMINATED event
   * to indicate that the single step has been cancelled.
   */
  AMD_DBGAPI_EXCEPTION_QUEUE_PREEMPTION_ERROR = (1 << 31)
} amd_dbgapi_exceptions_t;

/**
 * Return the list of queues.
 *
 * The order of the queue handles in the list is unspecified and can vary
 * between calls.
 *
 * The queues of the process that are associated with agents that do not
 * support debugging are not reported.  See
 * ::AMD_DBGAPI_AGENT_STATE_NOT_SUPPORTED.
 *
 * \param[in] process_id If ::AMD_DBGAPI_PROCESS_NONE then the queue list for
 * all processes is requested.  Otherwise, the queue list of process \p
 * process_id is requested.
 *
 * \param[out] queue_count The number of queues accessed by the process.
 *
 * \param[out] queues If \p changed is not NULL and the queues list of all of
 * the processes requested have not changed since the last call(s) to
 * ::amd_dbgapi_process_queue_list for each of them, then return NULL.
 * Otherwise, return a pointer to an array of ::amd_dbgapi_queue_id_t with \p
 * queue_count elements.  It is allocated by the
 * amd_dbgapi_callbacks_s::allocate_memory callback and is owned by the client.
 *
 * \param[in,out] changed If NULL then left unaltered.  If non-NULL, set to
 * ::AMD_DBGAPI_CHANGED_NO if the list of queues for each requested process is
 * the same as when ::amd_dbgapi_process_queue_list was last called for them.
 * Otherwise set to ::AMD_DBGAPI_CHANGED_YES.
 *
 * \retval ::AMD_DBGAPI_STATUS_SUCCESS The function has been executed
 * successfully and the result is stored in \p changed, \p queue_count, and \p
 * queues.
 *
 * \retval ::AMD_DBGAPI_STATUS_FATAL A fatal error occurred.  The library is
 * left uninitialized; and \p queue_count, \p queues, and \p changed are
 * unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_NOT_INITIALIZED The library is not
 * initialized.  The library is left uninitialized; and \p queue_count, \p
 * queues, and \p changed are unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_PROCESS_ID \p process_id is
 * invalid.  \p queue_count, \p queues, and \p changed are unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ARGUMENT \p queue_count or \p
 * queues are NULL, or \p changed is invalid.  \p queue_count, \p queues, and
 * \p changed are unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_CLIENT_CALLBACK This will be reported if
 * the amd_dbgapi_callbacks_s::allocate_memory callback used to allocate \p
 * queues returns NULL.  \p queue_count, \p queues, and \p changed are
 * unaltered.
 */
amd_dbgapi_status_t AMD_DBGAPI amd_dbgapi_process_queue_list (
    amd_dbgapi_process_id_t process_id, size_t *queue_count,
    amd_dbgapi_queue_id_t **queues,
    amd_dbgapi_changed_t *changed) AMD_DBGAPI_VERSION_0_54;

/**
 * Return the packets for a queue.
 *
 * Since the AMD GPU is asynchronously reading the packets this is only a
 * snapshot of the packets present in the queue, and only includes the packets
 * that the producer has made available to the queue.  In obtaining the
 * snapshot the library may pause the queue processing in order to get a
 * consistent snapshot.
 *
 * The queue packets are returned as a byte block that the client must
 * interpret according to the packet ABI determined by the queue type available
 * using the ::AMD_DBGAPI_QUEUE_INFO_TYPE query.  See
 * ::amd_dbgapi_os_queue_type_t.
 *
 * \param[in] queue_id The queue for which the packet list is requested.
 *
 * \param[out] read_packet_id The packet ID for the next packet to be read from
 * the queue. It corresponds to the first packet in \p packets_bytes.  If \p
 * packets_byte_size is zero, then the packet ID for the next packet added to
 * the queue.
 *
 * \param[out] write_packet_id The packet ID for the next packet to be written
 * to the queue. It corresponds to the next packet after the last packet in \p
 * packets_bytes.  If \p packets_byte_size is zero, then the packet ID for the
 * next packet added to the queue.
 *
 * \param[out] packets_byte_size The number of bytes of packets on the queue.
 *
 * \param[out] packets_bytes If non-NULL, it references a pointer to an array
 * of \p packets_byte_size bytes which is allocated by the
 * amd_dbgapi_callbacks_s::allocate_memory callback and is owned by the client.
 * If NULL, the packet bytes are not returned, just \p packets_byte_size.
 *
 * \retval ::AMD_DBGAPI_STATUS_SUCCESS The function has been executed
 * successfully and the result is stored in \p read_packet_id, \p
 * write_packet_id, \p packets_byte_size and \p packets_bytes.
 *
 * \retval ::AMD_DBGAPI_STATUS_FATAL A fatal error occurred.  The library is
 * left uninitialized; and \p read_packet_id, \p write_packet_id, \p
 * packets_byte_size and \p packets_bytes are unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_NOT_INITIALIZED The library is not
 * initialized.  The library is left uninitialized; and \p read_packet_id, \p
 * write_packet_id, \p packets_byte_size and \p packets_bytes are unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ARGUMENT \p read_packet_id, \p
 * write_packet_id, or \p packets_byte_size are NULL.  \p read_packet_id, \p
 * write_packet_id, \p packets_byte_size and \p packets_bytes are unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_NOT_SUPPORTED \p queue_id has a queue
 * type that is not supported.  \p read_packet_id, \p write_packet_id, \p
 * packets_byte_size and \p packets_bytes are unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR An error was encountered when attempting
 * to access the queue \p queue_id.  For example, the queue may be corrupted.
 * \p read_packet_id, \p write_packet_id, \p packets_byte_size and \p
 * packets_bytes are unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_CLIENT_CALLBACK This will be reported if
 * the amd_dbgapi_callbacks_s::allocate_memory callback used to allocate \p
 * packets_bytes returns NULL.  \p read_packet_id, \p write_packet_id, \p
 * packets_byte_size and \p packets_bytes are unaltered.
 */
amd_dbgapi_status_t AMD_DBGAPI amd_dbgapi_queue_packet_list (
    amd_dbgapi_queue_id_t queue_id,
    amd_dbgapi_os_queue_packet_id_t *read_packet_id,
    amd_dbgapi_os_queue_packet_id_t *write_packet_id,
    size_t *packets_byte_size, void **packets_bytes) AMD_DBGAPI_VERSION_0_54;

/** @} */

/** \defgroup dispatch_group Dispatches
 *
 * Operations related to AMD GPU dispatches.
 *
 * Dispatches are initiated by queue dispatch packets in the format supported
 * by the queue.  See ::amd_dbgapi_os_queue_type_t.  Dispatches are the means
 * that waves are created on the AMD GPU.
 *
 * @{
 */

/**
 * Opaque dispatch handle.
 *
 * Globally unique for a single library instance.
 */
typedef struct
{
  uint64_t handle;
} amd_dbgapi_dispatch_id_t;

/**
 * The NULL dispatch handle.
 */
#define AMD_DBGAPI_DISPATCH_NONE                                              \
    AMD_DBGAPI_HANDLE_LITERAL (amd_dbgapi_dispatch_id_t, 0)

/**
 * Dispatch queries that are supported by ::amd_dbgapi_dispatch_get_info.
 *
 * Each query specifies the type of data returned in the \p value argument to
 * ::amd_dbgapi_queue_get_info.
 */
typedef enum
{
  /**
   * Return the queue to which this dispatch belongs.  The type of this
   * attribute is ::amd_dbgapi_queue_id_t.
   */
  AMD_DBGAPI_DISPATCH_INFO_QUEUE = 1,
  /**
   * Return the agent to which this dispatch belongs.  The type of this
   * attribute is ::amd_dbgapi_agent_id_t.
   */
  AMD_DBGAPI_DISPATCH_INFO_AGENT = 2,
  /**
   * Return the process to which this dispatch belongs.  The type of this
   * attribute is ::amd_dbgapi_process_id_t.
   */
  AMD_DBGAPI_DISPATCH_INFO_PROCESS = 3,
  /**
   * Return the architecture of this dispatch.  The type of this attribute is
   * ::amd_dbgapi_architecture_id_t.
   */
  AMD_DBGAPI_DISPATCH_INFO_ARCHITECTURE = 4,
  /**
   * Return the queue packet ID of the dispatch packet that initiated the
   * dispatch.  The type of this attribute is \p
   * amd_dbgapi_os_queue_packet_id_t.
   */
  AMD_DBGAPI_DISPATCH_INFO_OS_QUEUE_PACKET_ID = 5,
  /**
   * Return the dispatch barrier setting.  The type of this attribute is
   * \p uint32_t with values defined by ::amd_dbgapi_dispatch_barrier_t.
   */
  AMD_DBGAPI_DISPATCH_INFO_BARRIER = 6,
  /**
   * Return the dispatch acquire fence.  The type of this attribute is
   * \p uint32_t with values defined by ::amd_dbgapi_dispatch_fence_scope_t.
   */
  AMD_DBGAPI_DISPATCH_INFO_ACQUIRE_FENCE = 7,
  /**
   * Return the dispatch release fence.  The type of this attribute is
   * \p uint32_t with values defined by ::amd_dbgapi_dispatch_fence_scope_t.
   */
  AMD_DBGAPI_DISPATCH_INFO_RELEASE_FENCE = 8,
  /**
   * Return the dispatch grid dimensionality.  The type of this attribute is
   * \p uint32 with a value of 1, 2, or 3.
   */
  AMD_DBGAPI_DISPATCH_INFO_GRID_DIMENSIONS = 9,
  /**
   * Return the dispatch workgroup size (work-items) in the X, Y, and Z
   * dimensions.  The type of this attribute is \p uint16_t[3].
   */
  AMD_DBGAPI_DISPATCH_INFO_WORKGROUP_SIZES = 10,
  /**
   * Return the dispatch grid size (work-items) in the X, Y, and Z dimensions.
   * The type of this attribute is \p uint32_t[3].
   */
  AMD_DBGAPI_DISPATCH_INFO_GRID_SIZES = 11,
  /**
   * Return the dispatch private segment size in bytes.  The type of this
   * attribute is ::amd_dbgapi_size_t.
   */
  AMD_DBGAPI_DISPATCH_INFO_PRIVATE_SEGMENT_SIZE = 12,
  /**
   * Return the dispatch group segment size in bytes.  The type of this
   * attribute is ::amd_dbgapi_size_t.
   */
  AMD_DBGAPI_DISPATCH_INFO_GROUP_SEGMENT_SIZE = 13,
  /**
   * Return the dispatch kernel argument segment address.  The type of this
   * attribute is ::amd_dbgapi_global_address_t.
   */
  AMD_DBGAPI_DISPATCH_INFO_KERNEL_ARGUMENT_SEGMENT_ADDRESS = 14,
  /**
   * Return the dispatch kernel descriptor address.  The type of this attribute
   * is ::amd_dbgapi_global_address_t.
   */
  AMD_DBGAPI_DISPATCH_INFO_KERNEL_DESCRIPTOR_ADDRESS = 15,
  /**
   * Return the dispatch kernel code entry address.  The type of this attribute
   * is ::amd_dbgapi_global_address_t.
   */
  AMD_DBGAPI_DISPATCH_INFO_KERNEL_CODE_ENTRY_ADDRESS = 16,
  /**
   * Return the dispatch completion event address.  The type of this attribute
   * is ::amd_dbgapi_global_address_t.  The ABI of the completion event varies
   * depending on the queue type available using the
   * ::AMD_DBGAPI_QUEUE_INFO_TYPE query.  See ::amd_dbgapi_os_queue_type_t.  If
   * the queue type does not use completion events, or the dispatch packet does
   * not define a completion event, then ::amd_dbgapi_dispatch_get_info will
   * return ::AMD_DBGAPI_STATUS_ERROR_NOT_SUPPORTED.
   */
  AMD_DBGAPI_DISPATCH_INFO_KERNEL_COMPLETION_ADDRESS = 17
} amd_dbgapi_dispatch_info_t;

/**
 * Query information about a dispatch.
 *
 * ::amd_dbgapi_dispatch_info_t specifies the queries supported and the type
 * returned using the \p value argument.
 *
 * \param[in] dispatch_id The handle of the dispatch being queried.
 *
 * \param[in] query The query being requested.
 *
 * \param[in] value_size Size of the memory pointed to by \p value.  Must be
 * equal to the byte size of the query result.
 *
 * \param[out] value Pointer to memory where the query result is stored.
 *
 * \retval ::AMD_DBGAPI_STATUS_SUCCESS The function has been executed
 * successfully and the result is stored in \p value.
 *
 * \retval ::AMD_DBGAPI_STATUS_FATAL A fatal error occurred.  The library is
 * left uninitialized and \p value is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_NOT_INITIALIZED The library is not
 * initialized.  The library is left uninitialized and \p value is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_DISPATCH_ID \p queue_id is
 * invalid.  \p value is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_NOT_SUPPORTED The requested \p query is
 * not supported for the specified \p dispatch_id.  \p value is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ARGUMENT \p value is NULL or
 * \p query is invalid.  \p value is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ARGUMENT_COMPATIBILITY \p
 * value_size does not match the size of the \p query result.  \p value is
 * unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_CLIENT_CALLBACK This will be
 * reported if the amd_dbgapi_callbacks_s::allocate_memory callback used to
 * allocate \p value returns NULL.  \p value is unaltered.
 */
amd_dbgapi_status_t AMD_DBGAPI amd_dbgapi_dispatch_get_info (
    amd_dbgapi_dispatch_id_t dispatch_id, amd_dbgapi_dispatch_info_t query,
    size_t value_size, void *value) AMD_DBGAPI_VERSION_0_54;

/**
 * Dispatch barrier.
 *
 * Controls when the dispatch will start being executed relative to previous
 * packets on the queue.
 */
typedef enum
{
  /**
   * Dispatch has no barrier.
   */
  AMD_DBGAPI_DISPATCH_BARRIER_NONE = 0,
  /**
   * Dispatch has a barrier.  The dispatch will not be executed until all
   * proceeding packets on the queue have completed.
   */
  AMD_DBGAPI_DISPATCH_BARRIER_PRESENT = 1
} amd_dbgapi_dispatch_barrier_t;

/**
 * Dispatch memory fence scope.
 *
 * Controls how memory is acquired before a dispatch starts executing and
 * released after the dispatch completes execution.
 */
typedef enum
{
  /**
   * There is no fence.
   */
  AMD_DBGAPI_DISPATCH_FENCE_SCOPE_NONE = 0,
  /**
   * There is a fence with agent memory scope.
   */
  AMD_DBGAPI_DISPATCH_FENCE_SCOPE_AGENT = 1,
  /**
   * There is a fence with system memory scope.
   */
  AMD_DBGAPI_DISPATCH_FENCE_SCOPE_SYSTEM = 2
} amd_dbgapi_dispatch_fence_scope_t;

/**
 * Return the list of dispatches.
 *
 * The order of the dispatch handles in the list is unspecified and can vary
 * between calls.
 *
 * \param[in] process_id If ::AMD_DBGAPI_PROCESS_NONE then the dispatch list
 * for all processes is requested.  Otherwise, the dispatch list of process \p
 * process_id is requested.
 *
 * \param[out] dispatch_count The number of dispatches active for a process.
 *
 * \param[out] dispatches If \p changed is not NULL and the dispatch list of
 * all of the processes requested have not changed since the last call(s) to
 * ::amd_dbgapi_process_dispatch_list for each of them, then return NULL.
 * Otherwise, return a pointer to an array of ::amd_dbgapi_dispatch_id_t with
 * \p dispatch_count elements.  It is allocated by the
 * amd_dbgapi_callbacks_s::allocate_memory callback and is owned by the client.
 *
 * \param[in,out] changed If NULL then left unaltered.  If non-NULL, set to
 * ::AMD_DBGAPI_CHANGED_NO if the list of dispatches for each requested process
 * is the same as when ::amd_dbgapi_process_dispatch_list was last called for
 * them.  Otherwise, set to ::AMD_DBGAPI_CHANGED_YES.
 *
 * \retval ::AMD_DBGAPI_STATUS_SUCCESS The function has been executed
 * successfully and the result is stored in \p changed, \p dispatch_count, and
 * \p dispatches.
 *
 * \retval ::AMD_DBGAPI_STATUS_FATAL A fatal error occurred.  The library is
 * left uninitialized; and \p changed, \p dispatch_count, and \p dispatches are
 * unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_NOT_INITIALIZED The library is not
 * initialized.  The library is left uninitialized; and \p changed, \p
 * dispatch_count, and \p dispatches are unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_PROCESS_ID \p process_id is
 * invalid.  \p dispatch_count, \p dispatches, and \p changed are unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ARGUMENT \p dispatch_count or \p
 * dispatches are NULL, or \p changed is invalid.  \p dispatch_count, \p
 * dispatches, and \p changed are unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_CLIENT_CALLBACK This will be reported if
 * the amd_dbgapi_callbacks_s::allocate_memory callback used to allocate \p
 * dispatches returns NULL.  \p dispatch_count, \p dispatches, and \p changed
 * are unaltered.
 */
amd_dbgapi_status_t AMD_DBGAPI amd_dbgapi_process_dispatch_list (
    amd_dbgapi_process_id_t process_id, size_t *dispatch_count,
    amd_dbgapi_dispatch_id_t **dispatches,
    amd_dbgapi_changed_t *changed) AMD_DBGAPI_VERSION_0_54;

/** @} */

/** \defgroup workgroup_group Workgroup
 *
 * Operations related to AMD GPU workgroups.
 *
 * @{
 */

/**
 * Opaque workgroup handle.
 *
 * AMD GPU executes code as waves organized into workgroups.
 *
 * Globally unique for a single library instance.
 */
typedef struct
{
  uint64_t handle;
} amd_dbgapi_workgroup_id_t;

/**
 * The NULL workgroup handle.
 */
#define AMD_DBGAPI_WORKGROUP_NONE                                                  \
    AMD_DBGAPI_HANDLE_LITERAL (amd_dbgapi_workgroup_id_t, 0)

/**
 * Workgroup queries that are supported by ::amd_dbgapi_workgroup_get_info.
 *
 * Each query specifies the type of data returned in the \p value argument to
 * ::amd_dbgapi_workgroup_get_info.
 */
typedef enum
{
  /**
   * Return the dispatch to which this workgroup belongs.  The type of this
   * attribute is ::amd_dbgapi_dispatch_id_t.
   *
   * If the dispatch associated with a workgroup is not available then
   * ::amd_dbgapi_workgroup_get_info returns the
   * ::AMD_DBGAPI_STATUS_ERROR_NOT_AVAILABLE error.  See the \ref
   * known_limitations section.
   */
  AMD_DBGAPI_WORKGROUP_INFO_DISPATCH = 1,
  /**
   * Return the queue to which this workgroup belongs.  The type of this
   * attribute is ::amd_dbgapi_queue_id_t.
   */
  AMD_DBGAPI_WORKGROUP_INFO_QUEUE = 2,
  /**
   * Return the agent to which this workgroup belongs.  The type of this
   * attribute is ::amd_dbgapi_agent_id_t.
   */
  AMD_DBGAPI_WORKGROUP_INFO_AGENT = 3,
  /**
   * Return the process to which this workgroup belongs.  The type of this
   * attribute is ::amd_dbgapi_process_id_t.
   */
  AMD_DBGAPI_WORKGROUP_INFO_PROCESS = 4,
  /**
   * Return the architecture of this workgroup.  The type of this attribute is
   * ::amd_dbgapi_architecture_id_t.
   */
  AMD_DBGAPI_WORKGROUP_INFO_ARCHITECTURE = 5,
  /**
   * The workgroup workgroup coordinate in the dispatch grid dimensions.  The
   * type of this attribute is \p uint32_t[3] with elements 1, 2, and 3
   * corresponding to the X, Y, and Z coordinates respectively.
   *
   * If the dispatch associated with a workgroup is not available then
   * ::amd_dbgapi_workgroup_get_info returns
   * ::AMD_DBGAPI_STATUS_ERROR_NOT_AVAILABLE.  See the \ref known_limitations
   * section.
   */
  AMD_DBGAPI_WORKGROUP_INFO_WORKGROUP_COORD = 6
} amd_dbgapi_workgroup_info_t;

/**
 * Query information about a workgroup.
 *
 * ::amd_dbgapi_workgroup_info_t specifies the queries supported and the type
 * returned using the \p value argument.
 *
 * \param[in] workgroup_id The handle of the workgroup being queried.
 *
 * \param[in] query The query being requested.
 *
 * \param[in] value_size Size of the memory pointed to by \p value.  Must be
 * equal to the byte size of the query result.
 *
 * \param[out] value Pointer to memory where the query result is stored.
 *
 * \retval ::AMD_DBGAPI_STATUS_SUCCESS The function has been executed
 * successfully and the result is stored in \p value.
 *
 * \retval ::AMD_DBGAPI_STATUS_FATAL A fatal error occurred.  The library is
 * left uninitialized and \p value is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_NOT_INITIALIZED The library is not
 * initialized.  The library is left uninitialized and \p value is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_WORKGROUP_ID \p workgroup_id is
 * invalid. \p value is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ARGUMENT \p value is NULL or \p
 * query is invalid.  \p value is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ARGUMENT_COMPATIBILITY \p
 * value_size does not match the size of the \p query result.  \p value is
 * unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_NOT_AVAILABLE \p The requested information
 * is not available.  See ::amd_dbgapi_workgroup_info_t for queries that can
 * produce this error.  \p value is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_CLIENT_CALLBACK This will be reported if
 * the amd_dbgapi_callbacks_s::allocate_memory callback used to allocate \p
 * value returns NULL.  \p value is unaltered.
 */
amd_dbgapi_status_t AMD_DBGAPI amd_dbgapi_workgroup_get_info (
    amd_dbgapi_workgroup_id_t workgroup_id, amd_dbgapi_workgroup_info_t query,
    size_t value_size, void *value) AMD_DBGAPI_VERSION_0_64;

/**
 * Return the list of existing workgroups.
 *
 * The order of the workgroup handles in the list is unspecified and can vary
 * between calls.
 *
 * \param[in] process_id If ::AMD_DBGAPI_PROCESS_NONE then the workgroup list
 * for all processes is requested.  Otherwise, the workgroup list of process \p
 * process_id is requested.
 *
 * \param[out] workgroup_count The number of workgroups executing in the
 * process.
 *
 * \param[out] workgroups If \p changed is not NULL and the workgroup list of
 * all of the processes requested have not changed since the last call(s) to
 * ::amd_dbgapi_process_workgroup_list for each of them, then return NULL.
 * Otherwise, return a pointer to an array of ::amd_dbgapi_workgroup_id_t with
 * \p workgroup_count elements.  It is allocated by the
 * amd_dbgapi_callbacks_s::allocate_memory callback and is owned by the client.
 *
 * \param[in,out] changed If NULL then left unaltered.  If non-NULL, set to
 * ::AMD_DBGAPI_CHANGED_NO if the list of workgroups for each requested process
 * is the same as when ::amd_dbgapi_process_workgroup_list was last called for
 * them.  Otherwise, set to ::AMD_DBGAPI_CHANGED_YES.
 *
 * \retval ::AMD_DBGAPI_STATUS_SUCCESS The function has been executed
 * successfully and the result is stored in \p changed, \p workgroup_count, and
 * \p workgroups.
 *
 * \retval ::AMD_DBGAPI_STATUS_FATAL A fatal error occurred.  The library is
 * left uninitialized; and \p changed, \p workgroup_count, and \p workgroups
 * are unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_NOT_INITIALIZED The library is not
 * initialized.  The library is left uninitialized; and \p workgroup_count, \p
 * workgroups, and \p changed are unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_PROCESS_ID \p process_id is
 * invalid.  \p workgroup_count, \p workgroups, and \p unchanged are unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ARGUMENT \p workgroup_count or \p
 * workgroups are NULL, or \p changed is invalid.  \p workgroup_count, \p
 * workgroups, and \p changed are unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_CLIENT_CALLBACK This will be reported if
 * the amd_dbgapi_callbacks_s::allocate_memory callback used to allocate \p
 * workgroups returns NULL.  \p workgroup_count, \p workgroups, and \p changed
 * are unaltered.
 */
amd_dbgapi_status_t AMD_DBGAPI amd_dbgapi_process_workgroup_list (
    amd_dbgapi_process_id_t process_id, size_t *workgroup_count,
    amd_dbgapi_workgroup_id_t **workgroups,
    amd_dbgapi_changed_t *changed) AMD_DBGAPI_VERSION_0_64;

/** @} */

/** \defgroup wave_group Wave
 *
 * Operations related to AMD GPU waves.
 *
 * @{
 */

/**
 * Opaque wave handle.
 *
 * Waves are the way the AMD GPU executes code.
 *
 * Globally unique for a single library instance.
 */
typedef struct
{
  uint64_t handle;
} amd_dbgapi_wave_id_t;

/**
 * The NULL wave handle.
 */
#define AMD_DBGAPI_WAVE_NONE                                                  \
    AMD_DBGAPI_HANDLE_LITERAL (amd_dbgapi_wave_id_t, 0)

/**
 * Wave queries that are supported by ::amd_dbgapi_wave_get_info.
 *
 * Each query specifies the type of data returned in the \p value argument to
 * ::amd_dbgapi_wave_get_info.
 */
typedef enum
{
  /**
   * Return the wave's state.  The type of this attribute is \p uint32_t with
   * values define by ::amd_dbgapi_wave_state_t.
   */
  AMD_DBGAPI_WAVE_INFO_STATE = 1,
  /**
   * Return the reason the wave stopped as a bit set.  The type of this
   * attribute is \p uint32_t with values defined by
   * ::amd_dbgapi_wave_stop_reasons_t.  The wave must be stopped to make this
   * query.
   */
  AMD_DBGAPI_WAVE_INFO_STOP_REASON = 2,
  /**
   * Return the watchpoint(s) the wave triggered.  The type of this attribute
   * is ::amd_dbgapi_watchpoint_list_t.  The
   * amd_dbgapi_watchpoint_list_t::count field is set to the number of
   * watchpoints that were triggered.  The
   * amd_dbgapi_watchpoint_list_t::watchpoint_ids field is set to a pointer to
   * an array of ::amd_dbgapi_watchpoint_id_t with
   * amd_dbgapi_watchpoint_list_t::count elements comprising the triggered
   * watchpoint handles.  The array is allocated by the
   * amd_dbgapi_callbacks_s::allocate_memory callback and is owned by the
   * client.  The wave must be stopped to make this query.
   */
  AMD_DBGAPI_WAVE_INFO_WATCHPOINTS = 3,
  /**
   * Return the workgroup to which this wave belongs.  The type of this
   * attribute is ::amd_dbgapi_workgroup_id_t.
   *
   * If the workgroup associated with a wave is not available then
   * ::amd_dbgapi_wave_get_info returns the
   * ::AMD_DBGAPI_STATUS_ERROR_NOT_AVAILABLE error.  See the \ref
   * known_limitations section.
   */
  AMD_DBGAPI_WAVE_INFO_WORKGROUP = 4,
  /**
   * Return the dispatch to which this wave belongs.  The type of this
   * attribute is ::amd_dbgapi_dispatch_id_t.
   *
   * If the dispatch associated with a wave is not available then
   * ::amd_dbgapi_wave_get_info returns the
   * ::AMD_DBGAPI_STATUS_ERROR_NOT_AVAILABLE error.  See the \ref
   * known_limitations section.
   */
  AMD_DBGAPI_WAVE_INFO_DISPATCH = 5,
  /**
   * Return the queue to which this wave belongs.  The type of this attribute
   * is ::amd_dbgapi_queue_id_t.
   */
  AMD_DBGAPI_WAVE_INFO_QUEUE = 6,
  /**
   * Return the agent to which this wave belongs.  The type of this attribute
   * is ::amd_dbgapi_agent_id_t.
   */
  AMD_DBGAPI_WAVE_INFO_AGENT = 7,
  /**
   * Return the process to which this wave belongs.  The type of this attribute
   * is ::amd_dbgapi_process_id_t.
   */
  AMD_DBGAPI_WAVE_INFO_PROCESS = 8,
  /**
   * Return the architecture of this wave.  The type of this attribute is
   * ::amd_dbgapi_architecture_id_t.
   */
  AMD_DBGAPI_WAVE_INFO_ARCHITECTURE = 9,
  /**
   * Return the current program counter value of the wave.  The type of this
   * attribute is ::amd_dbgapi_global_address_t.  The wave must be stopped to
   * make this query.
   */
  AMD_DBGAPI_WAVE_INFO_PC = 10,
  /**
   * Return the current execution mask of the wave.  Each bit of the mask maps
   * to a lane with the least significant bit corresponding to the lane with a
   * ::amd_dbgapi_lane_id_t value of 0 and so forth.  If the bit is 1 then the
   * lane is active, otherwise the lane is not active.  The type of this
   * attribute is \p uint64_t.  The wave must be stopped to make this query.
   */
  AMD_DBGAPI_WAVE_INFO_EXEC_MASK = 11,
  /**
   * The wave workgroup coordinate in the dispatch grid dimensions.  The type
   * of this attribute is \p uint32_t[3] with elements 1, 2, and 3
   * corresponding to the X, Y, and Z coordinates respectively.
   *
   * If the dispatch associated with a wave is not available then
   * ::amd_dbgapi_wave_get_info returns
   * ::AMD_DBGAPI_STATUS_ERROR_NOT_AVAILABLE.  See the \ref known_limitations
   * section.
   */
  AMD_DBGAPI_WAVE_INFO_WORKGROUP_COORD = 12,
  /**
   * The wave's number in the workgroup.  The type of this attribute is
   * \p uint32_t.  The work-items of a workgroup are mapped to the lanes of the
   * waves of the workgroup in flattened work-item ID order, with the first
   * work-item corresponding to lane 0 of wave 0, and so forth.
   *
   * If the dispatch associated with a wave is not available then
   * ::amd_dbgapi_wave_get_info returns
   * ::AMD_DBGAPI_STATUS_ERROR_NOT_AVAILABLE.  See the \ref known_limitations
   * section.
   */
  AMD_DBGAPI_WAVE_INFO_WAVE_NUMBER_IN_WORKGROUP = 13,
  /**
   * The number of lanes supported by the wave.  The type of this attribute is
   * \p size_t.
   */
  AMD_DBGAPI_WAVE_INFO_LANE_COUNT = 14
} amd_dbgapi_wave_info_t;

/**
 * Query information about a wave.
 *
 * ::amd_dbgapi_wave_info_t specifies the queries supported and the type
 * returned using the \p value argument.
 *
 * \param[in] wave_id The handle of the wave being queried.
 *
 * \param[in] query The query being requested.
 *
 * \param[in] value_size Size of the memory pointed to by \p value.  Must be
 * equal to the byte size of the query result.
 *
 * \param[out] value Pointer to memory where the query result is stored.
 *
 * \retval ::AMD_DBGAPI_STATUS_SUCCESS The function has been executed
 * successfully and the result is stored in \p value.
 *
 * \retval ::AMD_DBGAPI_STATUS_FATAL A fatal error occurred.  The library is
 * left uninitialized and \p value is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_NOT_INITIALIZED The library is not
 * initialized.  The library is left uninitialized and \p value is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_WAVE_ID \p wave_id is invalid. \p
 * value is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ARGUMENT \p value is NULL or \p
 * query is invalid.  \p value is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ARGUMENT_COMPATIBILITY \p
 * value_size does not match the size of the \p query result.  \p value is
 * unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_NOT_AVAILABLE \p The requested information
 * is not available.  See ::amd_dbgapi_wave_info_t for queries that can produce
 * this error.  \p value is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_WAVE_NOT_STOPPED \p query has a value of
 * ::amd_dbgapi_wave_info_t that requires the wave to be stopped, but the wave
 * is not stopped.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_CLIENT_CALLBACK This will be reported if
 * the amd_dbgapi_callbacks_s::allocate_memory callback used to allocate \p
 * value returns NULL.  \p value is unaltered.
 */
amd_dbgapi_status_t AMD_DBGAPI amd_dbgapi_wave_get_info (
    amd_dbgapi_wave_id_t wave_id, amd_dbgapi_wave_info_t query,
    size_t value_size, void *value) AMD_DBGAPI_VERSION_0_64;

/**
 * The execution state of a wave.
 */
typedef enum
{
  /**
   * The wave is running.
   */
  AMD_DBGAPI_WAVE_STATE_RUN = 1,
  /**
   * The wave is running in single-step mode.  It will execute a single
   * instruction and then stop.
   */
  AMD_DBGAPI_WAVE_STATE_SINGLE_STEP = 2,
  /**
   * The wave is stopped.
   *
   * Note that a wave may stop at any time due to the instructions it executes
   * or because the queue it is executing on enters the error state.  This will
   * cause a ::AMD_DBGAPI_EVENT_KIND_WAVE_STOP event to be created.  However,
   * until ::amd_dbgapi_process_next_pending_event returns the event, the wave
   * will continue to be reported as in the ::AMD_DBGAPI_WAVE_STATE_RUN state.
   * Only when the ::AMD_DBGAPI_EVENT_KIND_WAVE_STOP event is returned by
   * ::amd_dbgapi_process_next_pending_event will the wave be reported in the
   * ::AMD_DBGAPI_WAVE_STATE_STOP state.
   */
  AMD_DBGAPI_WAVE_STATE_STOP = 3
} amd_dbgapi_wave_state_t;

/**
 * A bit mask of the reasons that a wave stopped.
 *
 * The stop reason of a wave is available using the
 * ::AMD_DBGAPI_WAVE_INFO_STOP_REASON query.
 */
typedef enum
{
  /**
   * If none of the bits are set, then ::amd_dbgapi_wave_stop stopped the
   * wave.
   */
  AMD_DBGAPI_WAVE_STOP_REASON_NONE = 0,
  /**
   * The wave stopped due to executing a breakpoint instruction.  Use the
   * ::AMD_DBGAPI_ARCHITECTURE_INFO_BREAKPOINT_INSTRUCTION_PC_ADJUST query to
   * determine the address of the breakpoint instruction.
   */
  AMD_DBGAPI_WAVE_STOP_REASON_BREAKPOINT = (1 << 0),
  /**
   * The wave stopped due to triggering a data watchpoint.  The
   * ::AMD_DBGAPI_WAVE_INFO_WATCHPOINTS query can be used to determine which
   * watchpoint(s) were triggered.
   *
   * The program counter may not be positioned at the instruction that caused
   * the watchpoint(s) to be triggered as the AMD GPU can continue executing
   * instructions after initiating a memory operation.  If the architecture
   * supports it, the ::amd_dbgapi_set_memory_precision can be used to control
   * the precision, but may significantly reduce performance.
   */
  AMD_DBGAPI_WAVE_STOP_REASON_WATCHPOINT = (1 << 1),
  /**
   * The wave stopped due to completing an instruction single-step.
   */
  AMD_DBGAPI_WAVE_STOP_REASON_SINGLE_STEP = (1 << 2),
  /**
   * The wave stopped due to triggering an enabled floating point input
   * denormal exception.
   *
   * This stop reason would normally put the wave's queue into the queue error
   * state and include the ::AMD_DBGAPI_EXCEPTION_WAVE_TRAP queue error reason.
   */
  AMD_DBGAPI_WAVE_STOP_REASON_FP_INPUT_DENORMAL = (1 << 3),
  /**
   * The wave stopped due to triggering an enabled floating point divide by
   * zero exception.
   *
   * This stop reason would normally put the wave's queue into the queue error
   * state and include the ::AMD_DBGAPI_EXCEPTION_WAVE_TRAP queue error reason.
   */
  AMD_DBGAPI_WAVE_STOP_REASON_FP_DIVIDE_BY_0 = (1 << 4),
  /**
   * The wave stopped due to triggering an enabled floating point overflow
   * exception.
   *
   * This stop reason would normally put the wave's queue into the queue error
   * state and include the ::AMD_DBGAPI_EXCEPTION_WAVE_TRAP queue error reason.
   */
  AMD_DBGAPI_WAVE_STOP_REASON_FP_OVERFLOW = (1 << 5),
  /**
   * The wave stopped due to triggering an enabled floating point underflow
   * exception.
   *
   * This stop reason would normally put the wave's queue into the queue error
   * state and include the ::AMD_DBGAPI_EXCEPTION_WAVE_TRAP queue error reason.
   */
  AMD_DBGAPI_WAVE_STOP_REASON_FP_UNDERFLOW = (1 << 6),
  /**
   * The wave stopped due to triggering an enabled floating point inexact
   * exception.
   *
   * This stop reason would normally put the wave's queue into the queue error
   * state and include the ::AMD_DBGAPI_EXCEPTION_WAVE_TRAP queue error reason.
   */
  AMD_DBGAPI_WAVE_STOP_REASON_FP_INEXACT = (1 << 7),
  /**
   * The wave stopped due to triggering an enabled floating point invalid
   * operation exception.
   *
   * This stop reason would normally put the wave's queue into the queue error
   * state and include the ::AMD_DBGAPI_EXCEPTION_WAVE_TRAP queue error reason.
   */
  AMD_DBGAPI_WAVE_STOP_REASON_FP_INVALID_OPERATION = (1 << 8),
  /**
   * The wave stopped due to triggering an enabled integer divide by zero
   * exception.
   *
   * This stop reason would normally put the wave's queue into the queue error
   * state and include the ::AMD_DBGAPI_EXCEPTION_WAVE_TRAP queue error reason.
   */
  AMD_DBGAPI_WAVE_STOP_REASON_INT_DIVIDE_BY_0 = (1 << 9),
  /**
   * The wave stopped due to executing a debug trap instruction.  The program
   * counter is left positioned after the trap instruction.  The wave can be
   * resumed using ::amd_dbgapi_wave_resume.
   *
   * The debug trap instruction can be generated using the \p llvm.debugtrap
   * compiler intrinsic.  See [User Guide for AMDGPU Backend - Code Conventions
   * - AMDHSA - Trap Handler ABI]
   * (https://llvm.org/docs/AMDGPUUsage.html#trap-handler-abi).
   *
   * A debug trap can be used to explicitly insert stop points in a program to
   * help debugging.  They behave as no operations if a debugger is not
   * connected and stop the wave if executed with the debugger attached.
   */
  AMD_DBGAPI_WAVE_STOP_REASON_DEBUG_TRAP = (1 << 10),
  /**
   * The wave stopped due to executing an assert trap instruction.  The program
   * counter is left positioned at the assert trap instruction.
   *
   * The trap instruction can be generated using the \p llvm.trap
   * compiler intrinsic.  See [User Guide for AMDGPU Backend - Code Conventions
   * - AMDHSA - Trap Handler ABI]
   * (https://llvm.org/docs/AMDGPUUsage.html#trap-handler-abi).
   *
   * An assert trap can be used to abort the execution of the dispatches
   * executing on a queue.
   *
   * This stop reason would normally put the wave's queue into the queue error
   * state and include the ::AMD_DBGAPI_EXCEPTION_WAVE_TRAP queue error reason.
   */
  AMD_DBGAPI_WAVE_STOP_REASON_ASSERT_TRAP = (1 << 11),
  /**
   * The wave stopped due to executing a trap instruction other than the
   * ::AMD_DBGAPI_WAVE_STOP_REASON_DEBUG_TRAP or
   * ::AMD_DBGAPI_WAVE_STOP_REASON_ASSERT_TRAP trap instruction.  The program
   * counter is left positioned at the trap instruction.
   *
   * This stop reason would normally put the wave's queue into the queue error
   * state and include the ::AMD_DBGAPI_EXCEPTION_WAVE_TRAP queue error reason.
   */
  AMD_DBGAPI_WAVE_STOP_REASON_TRAP = (1 << 12),
  /**
   * The wave stopped due to a memory violation.  It indicates a non-existent
   * page was accessed or a page without the necessary permission (such as
   * writing to a readonly page or executing a non-execute page).
   *
   * The program counter may not be positioned at the instruction that caused
   * the memory violation as the AMD GPU can continue executing instructions
   * after initiating a memory operation.  If the architecture supports it, the
   * ::amd_dbgapi_set_memory_precision can be used to control the memory
   * exception reporting precision, but may significantly reduce performance.
   *
   * This stop reason would normally put the wave's queue into the queue error
   * state and include the ::AMD_DBGAPI_EXCEPTION_WAVE_MEMORY_VIOLATION queue
   * error reason.
   */
  AMD_DBGAPI_WAVE_STOP_REASON_MEMORY_VIOLATION = (1 << 13),
  /**
   * The wave stopped due to an aperture violation.  It indicates the memory
   * address is outside the virtual address range.
   *
   * The program counter may not be positioned at the instruction that caused
   * the aperture violation as the AMD GPU can continue executing instructions
   * after initiating a memory operation.  If the architecture supports it, the
   * ::amd_dbgapi_set_memory_precision can be used to control the precision,
   * but may significantly reduce performance.
   *
   * This stop reason would normally put the wave's queue into the queue error
   * state and include the ::AMD_DBGAPI_EXCEPTION_WAVE_ADDRESS_ERROR
   * queue error reason.
   */
  AMD_DBGAPI_WAVE_STOP_REASON_ADDRESS_ERROR = (1 << 14),
  /** Old deprecated name kept for backward compatibility.  Will be removed in
    * a future release.
    */
  AMD_DBGAPI_WAVE_STOP_REASON_APERTURE_VIOLATION DEPRECATED
    = AMD_DBGAPI_WAVE_STOP_REASON_ADDRESS_ERROR,
  /**
   * The wave stopped due to executing an illegal instruction.  The program
   * counter is left positioned at the illegal instruction.
   *
   * This stop reason would normally put the wave's queue into the queue error
   * state and include the ::AMD_DBGAPI_EXCEPTION_WAVE_ILLEGAL_INSTRUCTION
   * queue error reason.
   */
  AMD_DBGAPI_WAVE_STOP_REASON_ILLEGAL_INSTRUCTION = (1 << 15),
  /**
   * The wave stopped due to detecting an unrecoverable ECC error.
   *
   * The program counter may not be positioned at the instruction that caused
   * the memory violation as the AMD GPU can continue executing instructions
   * after initiating a memory operation.  If the architecture supports it, the
   * ::amd_dbgapi_set_memory_precision can be used to control the precision,
   * but may significantly reduce performance.
   *
   * This stop reason would normally put the wave's queue into the queue error
   * state and include the ::AMD_DBGAPI_EXCEPTION_WAVE_TRAP queue error reason.
   */
  AMD_DBGAPI_WAVE_STOP_REASON_ECC_ERROR = (1 << 16),
  /**
   * The wave stopped after causing a hardware fatal halt.
   *
   * This stop reason would normally put the wave's queue into the queue error
   * state and include the ::AMD_DBGAPI_EXCEPTION_WAVE_TRAP queue error reason.
   */
  AMD_DBGAPI_WAVE_STOP_REASON_FATAL_HALT = (1 << 17)
} amd_dbgapi_wave_stop_reasons_t;

/**
 * Return the list of existing waves.
 *
 * The order of the wave handles in the list is unspecified and can vary
 * between calls.
 *
 * \param[in] process_id If ::AMD_DBGAPI_PROCESS_NONE then the wave list for
 * all processes is requested.  Otherwise, the wave list of process \p
 * process_id is requested.
 *
 * \param[out] wave_count The number of waves executing in the process.
 *
 * \param[out] waves If \p changed is not NULL and the wave list of all of the
 * processes requested have not changed since the last call(s) to
 * ::amd_dbgapi_process_wave_list for each of them, then return NULL.
 * Otherwise, return a pointer to an array of ::amd_dbgapi_wave_id_t with \p
 * wave_count elements.  It is allocated by the
 * amd_dbgapi_callbacks_s::allocate_memory callback and is owned by the client.
 *
 * \param[in,out] changed If NULL then left unaltered.  If non-NULL, set to
 * ::AMD_DBGAPI_CHANGED_NO if the list of waves for each requested process is
 * the same as when ::amd_dbgapi_process_wave_list was last called for them.
 * Otherwise, set to ::AMD_DBGAPI_CHANGED_YES.
 *
 * \retval ::AMD_DBGAPI_STATUS_SUCCESS The function has been executed
 * successfully and the result is stored in \p changed, \p wave_count, and \p
 * waves.
 *
 * \retval ::AMD_DBGAPI_STATUS_FATAL A fatal error occurred.  The library is
 * left uninitialized; and \p changed, \p wave_count, and \p waves are
 * unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_NOT_INITIALIZED The library is not
 * initialized.  The library is left uninitialized; and \p wave_count, \p
 * waves, and \p changed are unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_PROCESS_ID \p process_id is
 * invalid.  \p wave_count, \p waves, and \p unchanged are unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ARGUMENT \p wave_count or \p waves
 * are NULL, or \p changed is invalid.  \p wave_count, \p waves, and \p changed
 * are unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_CLIENT_CALLBACK This will be reported if
 * the amd_dbgapi_callbacks_s::allocate_memory callback used to allocate \p
 * waves returns NULL.  \p wave_count, \p waves, and \p changed are unaltered.
 */
amd_dbgapi_status_t AMD_DBGAPI amd_dbgapi_process_wave_list (
    amd_dbgapi_process_id_t process_id, size_t *wave_count,
    amd_dbgapi_wave_id_t **waves,
    amd_dbgapi_changed_t *changed) AMD_DBGAPI_VERSION_0_54;

/**
 * Request a wave to stop executing.
 *
 * The wave may or may not immediately stop.  If the wave does not immediately
 * stop, the stop request is termed outstanding until the wave does stop or the
 * wave terminates before stopping.  When the wave does stop it will create a
 * ::AMD_DBGAPI_EVENT_KIND_WAVE_STOP event.  If the wave terminates before
 * stopping it will create a ::AMD_DBGAPI_EVENT_KIND_WAVE_COMMAND_TERMINATED
 * event.
 *
 * A process in the ::AMD_DBGAPI_PROGRESS_NO_FORWARD progress mode will report
 * the ::AMD_DBGAPI_EVENT_KIND_WAVE_STOP or
 * ::AMD_DBGAPI_EVENT_KIND_WAVE_COMMAND_TERMINATED event.  It is not necessary
 * to change the progress mode to ::AMD_DBGAPI_PROGRESS_NORMAL for these events
 * to be reported.
 *
 * It is an error to request a wave to stop that has terminated.  The wave
 * handle will be reported as invalid.  It is up to the client to use
 * ::amd_dbgapi_process_wave_list to determine what waves have been created and
 * terminated.  No event is reported when a wave is created or terminates.
 *
 * It is an error to request a wave to stop that is already in the
 * ::AMD_DBGAPI_WAVE_STATE_STOP state.
 *
 * It is an error to request a wave to stop for which there is an outstanding
 * ::amd_dbgapi_wave_stop request.
 *
 * Sending a stop request to a wave that has already stopped, but whose
 * ::AMD_DBGAPI_EVENT_KIND_WAVE_STOP event has not yet been returned by
 * ::amd_dbgapi_process_next_pending_event, is allowed since the wave is still
 * in the ::AMD_DBGAPI_WAVE_STATE_RUN state.  In this case the wave is not
 * affected and the already existing ::AMD_DBGAPI_EVENT_KIND_WAVE_STOP will
 * notify the client that the stop request has completed.  The client must be
 * prepared that a wave may stop for other reasons in response to a stop
 * request.  It can use the ::AMD_DBGAPI_WAVE_INFO_STOP_REASON query to
 * determine if there are other reason(s).  See ::AMD_DBGAPI_WAVE_STATE_STOP
 * for more information.
 *
 * Sending a stop request to a wave that is in the
 * ::AMD_DBGAPI_WAVE_STATE_SINGLE_STEP state will attempt to stop the wave and
 * either report a ::AMD_DBGAPI_EVENT_KIND_WAVE_STOP or
 * ::AMD_DBGAPI_EVENT_KIND_WAVE_COMMAND_TERMINATED event.  If the wave did
 * stop, the setting of the ::AMD_DBGAPI_WAVE_STOP_REASON_SINGLE_STEP stop
 * reason will indicate whether the wave completed the single step.  If the
 * single step does complete, but terminates the wave, then
 * ::AMD_DBGAPI_EVENT_KIND_WAVE_COMMAND_TERMINATED will be reported.
 *
 * Sending a stop request to a wave that is present at the time of the request,
 * and does stop, will result in a ::AMD_DBGAPI_EVENT_KIND_WAVE_STOP event.
 *
 * Sending a stop request to a wave that is present at the time of the request,
 * but terminates before completing the stop request, will result in a
 * ::AMD_DBGAPI_EVENT_KIND_WAVE_COMMAND_TERMINATED event.
 *
 * \param[in] wave_id The wave being requested to stop.
 *
 * \retval ::AMD_DBGAPI_STATUS_SUCCESS The function has been executed
 * successfully and the wave will either report a
 * ::AMD_DBGAPI_EVENT_KIND_WAVE_STOP or
 * ::AMD_DBGAPI_EVENT_KIND_WAVE_COMMAND_TERMINATED event.
 *
 * \retval ::AMD_DBGAPI_STATUS_FATAL A fatal error occurred.  The library
 * is left uninitialized.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_NOT_INITIALIZED The library is not
 * initialized.  The library is left uninitialized and no wave is stopped.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_WAVE_ID \p wave_id is invalid.  No
 * wave is stopped.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_WAVE_STOPPED \p wave_id is already
 * stopped.  The wave remains stopped.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_WAVE_OUTSTANDING_STOP The wave already
 * has an outstanding stop request.  This stop request is ignored and the
 * previous stop request continues to stop the wave.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_PROCESS_FROZEN The process the wave
 * belongs to is frozen.  The wave is already stopped.  The wave remains
 * stopped.
 */
amd_dbgapi_status_t AMD_DBGAPI
amd_dbgapi_wave_stop (amd_dbgapi_wave_id_t wave_id) AMD_DBGAPI_VERSION_0_76;

/**
 * The mode in which to resuming the execution of a wave.
 */
typedef enum
{
  /**
   * Resume normal execution.
   */
  AMD_DBGAPI_RESUME_MODE_NORMAL = 0,
  /**
   * Resume execution in in single step mode.
   */
  AMD_DBGAPI_RESUME_MODE_SINGLE_STEP = 1
} amd_dbgapi_resume_mode_t;

/**
 * Resume execution of a stopped wave.
 *
 * The wave can be resumed normally in which case it will be in the
 * ::AMD_DBGAPI_WAVE_STATE_RUN state and be available for the hardware to
 * execute instructions.  Just because it is in the run state does not mean the
 * hardware will start executing instructions immediately as that depends on
 * the AMD GPU hardware scheduler.
 *
 * If while in the ::AMD_DBGAPI_WAVE_STATE_RUN state, the wave encounters
 * something that stops its execution, or ::amd_dbgapi_wave_stop is used to
 * stop the wave execution, then a ::AMD_DBGAPI_EVENT_KIND_WAVE_STOP event will
 * be created.
 *
 * If while in the ::AMD_DBGAPI_WAVE_STATE_RUN state the wave terminates, no
 * event is created.
 *
 * The wave can be resumed in single step mode in which case it will be in the
 * ::AMD_DBGAPI_WAVE_STATE_SINGLE_STEP state.  It is available for the hardware
 * to execute one instruction.  After completing execution of a regular
 * instruction, a ::AMD_DBGAPI_EVENT_KIND_WAVE_STOP event will be created that
 * indicates the wave has stopped.  The stop reason of the wave will include
 * ::AMD_DBGAPI_WAVE_STOP_REASON_SINGLE_STEP.  After completing execution of a
 * wave termination instruction, a
 * ::AMD_DBGAPI_EVENT_KIND_WAVE_COMMAND_TERMINATED event will be created that
 * indicates that the wave has terminated.
 *
 * Resuming a wave in single step mode does not necessarily cause it to execute
 * any instructions as it is up to the AMD GPU hardware scheduler to decide
 * what waves to execute.  For example, the AMD GPU hardware scheduler may not
 * execute any instructions of a wave until other waves have terminated.  If
 * the client has stopped other waves this can prevent a wave from ever
 * performing a single step.  The client should handle this gracefully and not
 * rely on a single step request always resulting in a
 * ::AMD_DBGAPI_EVENT_KIND_WAVE_STOP event.  If necessary, the client should
 * respond to the stop events of other waves to allow them to make forward
 * progress, and handle the single step stop request when it finally arrives.
 * If necessary, the client can cancel the single step request by using
 * ::amd_dbgapi_wave_stop and allow the user to attempt it again later when
 * other waves have terminated.
 *
 * It is an error to resume a wave that has terminated.  The wave handle will
 * be reported as invalid.  It is up to the client to use
 * ::amd_dbgapi_process_wave_list to determine what waves have been created and
 * terminated.  No event is reported when a wave is created or terminates.
 *
 * It is an error to request a wave to resume that is not in the
 * ::AMD_DBGAPI_WAVE_STATE_STOP state, or is in the
 * ::AMD_DBGAPI_WAVE_STATE_STOP state but the ::AMD_DBGAPI_EVENT_KIND_WAVE_STOP
 * event that put it in the stop state has not yet been completed using the
 * ::amd_dbgapi_event_processed operation.  Therefore, it is not allowed to
 * execute multiple resume requests as all but the first one will give an
 * error.
 *
 * It also means it is an error to resume a wave that has already stopped, but
 * whose ::AMD_DBGAPI_EVENT_KIND_WAVE_STOP event has not yet been returned by
 * ::amd_dbgapi_process_next_pending_event, since the wave is still in the
 * ::AMD_DBGAPI_WAVE_STATE_RUN state.  The ::AMD_DBGAPI_EVENT_KIND_WAVE_STOP
 * must be processed first.
 *
 * Since a resume request can only be sent to a wave that has stopped, there is
 * no issue of the wave terminating while making the request.  However, the
 * wave may terminate after being resumed.  Except for single stepping the wave
 * termination instruction described above, no event is reported when the wave
 * terminates.
 *
 * Resuming a wave that is in the halt state or belongs to a queue that is in
 * the queue error state will not result in it executing any further
 * instructions.  Resuming a wave in single step mode that does not belong to a
 * queue that is in the queue error state will therefore not report a
 * ::AMD_DBGAPI_EVENT_KIND_WAVE_STOP event that includes the
 * ::AMD_DBGAPI_WAVE_STOP_REASON_SINGLE_STEP until the wave is no longer in the
 * halt state.
 *
 * Resuming a wave in single step mode that does belong to a queue that is in
 * the queue error state, or if the queue enters the queue error state after
 * the wave has been resumed in single step mode but before it actually
 * executes an instruction, will report a
 * ::AMD_DBGAPI_EVENT_KIND_WAVE_COMMAND_TERMINATED event to indicate that the
 * single step request has been cancelled.  Waves in such queues are inhibited
 * from executing any further instructions.  The application can delete the
 * queue, which will result in all the associated waves to also be deleted, and
 * then create a new queue.
 *
 * A wave may stop with stop reasons that would normally cause the inferior's
 * runtime to put the queue into the queue error state (see
 * ::amd_dbgapi_wave_stop_reasons_t).  However, when the
 * ::AMD_DBGAPI_EVENT_KIND_WAVE_STOP event is reported, the inferior's runtime
 * will not have been notified, and so the exception will not have caused the
 * queue to enter the queue error state.  This allows the user to inspect the
 * wave state before the inferior's runtime may cause the queue and all its
 * waves to be deleted.
 *
 * In order to deliver the stop reason exceptions to the inferior's runtime,
 * the client can resume the wave and specify the exceptions using the \p
 * exceptions argument.  The client may use ::AMD_DBGAPI_EXCEPTION_NONE so no
 * exceptions are delivered, effectively ignoring the exceptions, or the client
 * may pass different exceptions.  The client may also pass exceptions to any
 * wave even if it did not stop with a stop reason that includes any
 * exceptions.  Note that resuming a wave and ignoring exceptions may result in
 * unpredictable behavior.  For example, the
 * ::AMD_DBGAPI_WAVE_STOP_REASON_ASSERT_TRAP stop reason assumes that execution
 * will not be continued, and so the following bytes may not be legal
 * instructions, or may be unrelated instructions.
 *
 * \param[in] wave_id The wave being requested to resume.
 *
 * \param[in] resume_mode If ::AMD_DBGAPI_RESUME_MODE_NORMAL, then resume
 * normal execution of the wave.  If ::AMD_DBGAPI_RESUME_MODE_SINGLE_STEP, then
 * resume the wave in single step mode.
 *
 * \param[in] exceptions If ::AMD_DBGAPI_EXCEPTION_NONE, indicates the wave
 * execution is resumed without delivering any exceptions.  Any other value of
 * ::amd_dbgapi_exceptions_t causes the wave to be put in the halt state and
 * the inferior's runtime notified of the specified exceptions.  The inferior's
 * runtime will put the wave's queue into the queue error state such that the
 * queue's ::AMD_DBGAPI_QUEUE_INFO_ERROR_REASON query will include the
 * exceptions specified by \p exceptions.  See ::AMD_DBGAPI_QUEUE_STATE_ERROR
 * for information in the events created when a queue is put in the queue error
 * state.
 *
 * \retval ::AMD_DBGAPI_STATUS_SUCCESS The function has been executed
 * successfully and the wave will either terminate or be stopped.  In either
 * case a ::AMD_DBGAPI_EVENT_KIND_WAVE_STOP event will be reported.
 *
 * \retval ::AMD_DBGAPI_STATUS_FATAL A fatal error occurred.  The library is
 * left uninitialized.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_NOT_INITIALIZED The library is not
 * initialized.  The library is left uninitialized and no wave is resumed.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_WAVE_ID \p wave_id is invalid.  No
 * wave is resumed.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ARGUMENT \p resume_mode is invalid
 * or \p exceptions does not contain only wave exceptions.  No wave is resumed.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_WAVE_NOT_STOPPED \p wave_id is not
 * stopped.  The wave remains running.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_WAVE_NOT_RESUMABLE The event that put \p
 * wave_id in the stop state has not yet been completed using the
 * ::amd_dbgapi_event_processed operation.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_RESUME_DISPLACED_STEPPING \p wave_id is
 * stopped and has an associated displaced stepping buffer. The \p resume_mode
 * is either not ::AMD_DBGAPI_RESUME_MODE_SINGLE_STEP, or the \p wave_id has
 * already been single stepped by one instruction and so
 * ::amd_dbgapi_displaced_stepping_complete must be used before the wave can be
 * resumed.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_PROCESS_FROZEN The process the wave
 * belongs to is frozen.  No wave is resumed.
 */
amd_dbgapi_status_t AMD_DBGAPI amd_dbgapi_wave_resume (
  amd_dbgapi_wave_id_t wave_id, amd_dbgapi_resume_mode_t resume_mode,
  amd_dbgapi_exceptions_t exceptions) AMD_DBGAPI_VERSION_0_76;

/** @} */

/** \defgroup displaced_stepping_group Displaced Stepping
 *
 * Operations related to AMD GPU breakpoint displaced stepping.
 *
 * The library supports displaced stepping buffers.  These allow an instruction
 * that is overwritten by a breakpoint instruction to be copied to a buffer and
 * single stepped in that buffer.  This avoids needing to remove the breakpoint
 * instruction by replacing it with the original instruction bytes, single
 * stepping the original instruction, and finally restoring the breakpoint
 * instruction.
 *
 * This allows a client to support non-stop debugging where waves are left
 * executing while others are halted after hitting a breakpoint.  If resuming
 * from a breakpoint involved removing the breakpoint, it could result in the
 * running waves missing the removed breakpoint.
 *
 * When an instruction is copied into a displaced stepping buffer, it may be
 * necessary to modify the instruction, or its register inputs to account for
 * the fact that it is executing at a different address.  Similarly, after
 * single stepping it, registers and program counter may need adjusting.  It
 * may also be possible to know the effect of an instruction and avoid single
 * stepping it at all and simply update the wave state directly.  For example,
 * branches can be trivial to emulate this way.
 *
 * The operations in this section allow displaced stepping buffers to be
 * allocated and used.  They will take care of all the architecture specific
 * details described above.
 *
 * The number of displaced stepping buffers supported by the library is
 * unspecified, but there is always at least one.  It may be possible for the
 * library to share the same displaced stepping buffer with multiple waves. For
 * example, if the waves are at the same breakpoint.  The library will
 * determine when this is possible, but the client should not rely on this.
 * Some waves at the same breakpoint may be able to share while others may not.
 * In general, it is best for the client to single step as many waves as
 * possible to minimize the time to get all waves stepped over the breakpoints.
 *
 * The client may be able to maximize the number of waves it can single step at
 * once by requesting displaced stepping buffers for all waves at the same
 * breakpoint.  Just because there is no displaced stepping buffer for one
 * wave, does not mean another wave cannot be assigned to a displaced stepping
 * buffer through sharing, or through buffers being associated with specific
 * agents or queues.
 *
 * If allocating a displaced stepping buffer
 * (::amd_dbgapi_displaced_stepping_start) is successful, then the client must
 * resume the wave (::amd_dbgapi_wave_resume) in single step mode.  When the
 * single step is reported as completed (::AMD_DBGAPI_EVENT_KIND_WAVE_STOP),
 * the buffer can be released (::amd_dbgapi_displaced_stepping_complete), and
 * the wave resumed normally (::amd_dbgapi_wave_resume).
 *
 * If the single step is reported as terminated
 * (::AMD_DBGAPI_EVENT_KIND_WAVE_COMMAND_TERMINATED), then that indicates that
 * the wave has exited.  When a wave exits, any associated displaced stepping
 * buffer is automatically released.
 *
 * If the wave does not report the single step as complete
 * (::AMD_DBGAPI_EVENT_KIND_WAVE_STOP) or terminated
 * (::AMD_DBGAPI_EVENT_KIND_WAVE_COMMAND_TERMINATED), then the wave can be
 * stopped (::amd_dbgapi_wave_stop), and the buffer released
 * (::amd_dbgapi_displaced_stepping_complete).  This will leave the wave still
 * at the breakpoint, and the client can retry stepping over the breakpoint
 * later (::amd_dbgapi_displaced_stepping_start).
 *
 * If allocating a displaced stepping buffer indicates no more are available,
 * the client must complete ongoing single steppings and release the associated
 * buffers.  It can do that by ensuring the waves with allocated stepping
 * buffers are resumed in single step mode, ensure that the waves will make
 * forward progress, and process any reported pending events.  This allows
 * waves to perform the single step, report the single step has completed by an
 * event, and the client's processing of the event will release the displaced
 * stepping buffer (::amd_dbgapi_displaced_stepping_complete).  That may free
 * up a displaced stepping buffer for use by the client for other waves.  Since
 * there is always at least one displaced stepping buffer, in general, the
 * worst case is that one wave at a time can be single stepped over a
 * breakpoint using a displaced stepping buffer.
 *
 * However, the weak forward progress of AMD GPU execution can result in no
 * waves that have successfully been allocated a displaced stepping buffer from
 * actually reporting completion of the single step.  For example, this can
 * happen if the waves being single stepped are prevented from becoming
 * resident on the hardware due to other waves that are halted.  The waves
 * being single stepped can be stopped before completing the single step to
 * release the displaced stepping buffer for use by a different set of waves.
 * In the worst case, the user may have to continue halted waves and allow them
 * to terminate before other waves can make forward progress to complete the
 * single step using a displaced stepping buffer.
 *
 * \sa ::amd_dbgapi_wave_resume, ::amd_dbgapi_wave_stop,
 * ::amd_dbgapi_process_set_progress, ::amd_dbgapi_process_next_pending_event
 *
 * @{
 */

/**
 * Opaque displaced stepping handle.
 *
 * Globally unique for a single library instance.
 */
typedef struct
{
  uint64_t handle;
} amd_dbgapi_displaced_stepping_id_t;

/**
 * The NULL displaced stepping handle.
 */
#define AMD_DBGAPI_DISPLACED_STEPPING_NONE                                    \
  (amd_dbgapi_displaced_stepping_id_t{ 0 })

/**
 * Displaced stepping queries that are supported by
 * ::amd_dbgapi_displaced_stepping_id_t.
 *
 * Each query specifies the type of data returned in the \p value argument to
 * ::amd_dbgapi_displaced_stepping_id_t.
 */
typedef enum
{
  /**
   * Return the process to which this displaced stepping buffer belongs.  The
   * type of this attribute is ::amd_dbgapi_process_id_t.
   */
  AMD_DBGAPI_DISPLACED_STEPPING_INFO_PROCESS = 1
} amd_dbgapi_displaced_stepping_info_t;

/**
 * Query information about a displaced stepping buffer.
 *
 * ::amd_dbgapi_displaced_stepping_info_t specifies the queries supported and
 * the type returned using the \p value argument.
 *
 * \param[in] displaced_stepping_id The handle of the displaced stepping buffer
 * being queried.
 *
 * \param[in] query The query being requested.
 *
 * \param[in] value_size Size of the memory pointed to by \p value.  Must be
 * equal to the byte size of the query result.
 *
 * \param[out] value Pointer to memory where the query result is stored.
 *
 * \retval ::AMD_DBGAPI_STATUS_SUCCESS The function has been executed
 * successfully and the result is stored in \p value.
 *
 * \retval ::AMD_DBGAPI_STATUS_FATAL A fatal error occurred.  The library is
 * left uninitialized and \p value is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_NOT_INITIALIZED The library is not
 * initialized.  The library is left uninitialized and \p value is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_DISPLACED_STEPPING_ID \p
 * displaced_stepping_id is invalid.  \p value is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ARGUMENT \p value is NULL or
 * \p query is invalid.  \p value is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ARGUMENT_COMPATIBILITY \p
 * value_size does not match the size of the \p query result.  \p value is
 * unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_CLIENT_CALLBACK This will be
 * reported if the amd_dbgapi_callbacks_s::allocate_memory callback used to
 * allocate \p value returns NULL.  \p value is unaltered.
 */
amd_dbgapi_status_t AMD_DBGAPI amd_dbgapi_displaced_stepping_get_info (
    amd_dbgapi_displaced_stepping_id_t displaced_stepping_id,
    amd_dbgapi_displaced_stepping_info_t query, size_t value_size,
    void *value) AMD_DBGAPI_VERSION_0_54;

/**
 * Associate an active displaced stepping buffer with a wave.
 *
 * The wave must be stopped and not already have an active displaced stepping
 * buffer.
 *
 * Displaced stepping buffers are intended to be used to step over breakpoints.
 * In that case, the wave will be stopped with a program counter set to a
 * breakpoint instruction that was placed by the client overwriting all or part
 * of the original instruction where the breakpoint was placed.  The client
 * must provide the overwritten bytes of the original instruction.
 *
 * The wave program counter and other registers may be read and written as part
 * of creating a displaced stepping buffer.  Therefore, the client should flush
 * any dirty cached register values before creating a displaced stepping
 * buffer.
 *
 * If a displaced stepping handle is returned successfully, the wave is still
 * stopped.  The client should resume the wave in single step mode using
 * ::amd_dbgapi_wave_resume.  Once the single step is complete as indicated by
 * the ::AMD_DBGAPI_EVENT_KIND_WAVE_STOP event with a stop reason that includes
 * ::AMD_DBGAPI_WAVE_STOP_REASON_SINGLE_STEP, the client should use
 * ::amd_dbgapi_displaced_stepping_complete to release the displaced stepping
 * buffer.  The wave can then be resumed normally using
 * ::amd_dbgapi_wave_resume.
 *
 * If the single step is cancelled by stopping the wave, the client must
 * determine if the wave completed the single step to determine if the wave can
 * be resumed or must retry the displaced stepping later.  See
 * ::amd_dbgapi_wave_stop.
 *
 * \param[in] wave_id The wave for which to create a displaced stepping buffer.
 *
 * \param[in] saved_instruction_bytes The original instruction bytes that the
 * breakpoint instruction replaced.  The number of bytes must be
 * ::AMD_DBGAPI_ARCHITECTURE_INFO_BREAKPOINT_INSTRUCTION_SIZE.
 *
 * \param[out] displaced_stepping The displaced stepping handle.
 *
 * \retval ::AMD_DBGAPI_STATUS_SUCCESS The function has been executed
 * successfully and \p displaced_stepping is set to a valid displaced stepping
 * handle.
 *
 * \retval ::AMD_DBGAPI_STATUS_FATAL A fatal error occurred.  The library is
 * left uninitialized, no displaced stepping buffer is allocated, and \p
 * displaced_stepping is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_NOT_INITIALIZED The library is not
 * initialized.  The library is left uninitialized, no displaced stepping
 * buffer is allocated, and \p displaced_stepping is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_WAVE_ID \p wave_id is invalid.  No
 * displaced stepping buffer is allocated and \p displaced_stepping is
 * unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_WAVE_NOT_STOPPED \p wave_id is not
 * stopped.  No displaced stepping buffer is allocated and \p
 * displaced_stepping is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_DISPLACED_STEPPING_ACTIVE \p wave_id
 * already has an active displaced stepping buffer.  No displaced stepping
 * buffer is allocated and \p displaced_stepping is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_DISPLACED_STEPPING_BUFFER_NOT_AVAILABLE No
 * more displaced stepping buffers are available that are suitable for use by
 * \p wave_id.  No displaced stepping buffer is allocated and \p
 * displaced_stepping is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ARGUMENT \p original_instruction
 * or \p displaced_stepping are NULL.  No displaced stepping buffer is
 * allocated and \p displaced_stepping is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_MEMORY_ACCESS The memory at the wave's
 * program counter could not be successfully read.  No displaced stepping
 * buffer is allocated and \p displaced_stepping is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_ILLEGAL_INSTRUCTION The instruction at the
 * wave's program counter is not a legal instruction for the architecture.  No
 * displaced stepping buffer is allocated and \p displaced_stepping is
 * unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_PROCESS_FROZEN The process is frozen.  No
 * displaced stepping buffer is allocated and \p displaced_stepping is
 * unaltered.
 */
amd_dbgapi_status_t AMD_DBGAPI amd_dbgapi_displaced_stepping_start (
    amd_dbgapi_wave_id_t wave_id, const void *saved_instruction_bytes,
    amd_dbgapi_displaced_stepping_id_t *displaced_stepping)
    AMD_DBGAPI_VERSION_0_76;

/**
 * Complete a displaced stepping buffer for a wave.
 *
 * The wave must be stopped and have an associated displaced stepping buffer by
 * using ::amd_dbgapi_displaced_stepping_start.
 *
 * If the wave single step has not completed, the wave state is reset to what
 * it was before ::amd_dbgapi_displaced_stepping_start.  The wave is left
 * stopped and the client can retry stepping over the breakpoint again later.
 *
 * If the single step has completed, then the wave state is updated to be after
 * the instruction at which the breakpoint instruction is placed.
 *
 * Completing a displaced stepping buffer may read and write the wave program
 * counter and other registers so the client should invalidate any cached
 * register values after completing a displaced stepping buffer.  The wave is
 * left stopped and can be resumed normally by the client.
 *
 * If the wave is the last one using the displaced stepping buffer, the buffer
 * is freed and the handle invalidated.
 *
 * \param[in] wave_id The wave using the displaced stepping buffer.
 *
 * \param[in] displaced_stepping The displaced stepping buffer to complete.
 *
 * \retval ::AMD_DBGAPI_STATUS_SUCCESS The function has been executed
 * successfully.  The displaced stepping buffer is completed, and the wave is
 * either stepped over the breakpoint, or still at the breakpoint.
 *
 * \retval ::AMD_DBGAPI_STATUS_FATAL A fatal error occurred.  The library is
 * left uninitialized, and no displaced stepping buffer is completed.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_NOT_INITIALIZED The library is not
 * initialized.  The library is left uninitialized, no displaced stepping
 * buffer completed.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_WAVE_ID \p wave_id is invalid.  No
 * displaced stepping buffer is completed.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_DISPLACED_STEPPING_ID \p
 * displaced_stepping is invalid.  No displaced stepping buffer is completed.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_WAVE_NOT_STOPPED \p wave_id is not
 * stopped.  No displaced stepping buffer is completed.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ARGUMENT_COMPATIBILITY \p
 * displaced_stepping is not in use by \p wave_id (which includes that the wave
 * has already completed the displaced stepping buffer).  No displaced stepping
 * buffer is completed.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_PROCESS_FROZEN The process is frozen.  No
 * displaced stepping buffer is completed.
 */
amd_dbgapi_status_t AMD_DBGAPI amd_dbgapi_displaced_stepping_complete (
    amd_dbgapi_wave_id_t wave_id,
    amd_dbgapi_displaced_stepping_id_t displaced_stepping)
    AMD_DBGAPI_VERSION_0_76;

/** @} */

/** \defgroup watchpoint_group Watchpoints
 *
 * Operations related to AMD GPU hardware data watchpoints.
 *
 * A data watchpoint is a hardware supported mechanism to generate wave stop
 * events after a wave accesses memory in a certain way in a certain address
 * range.  The memory access will have been completed before the event is
 * reported.
 *
 * The number of watchpoints, the granularity of base address, and the address
 * range is process specific.  If a process has multiple agents, then the
 * values are the lowest common denominator of the capabilities of the
 * architectures of all the agents of a process.
 *
 * The number of watchpoints supported by a process is available using the
 * ::AMD_DBGAPI_PROCESS_INFO_WATCHPOINT_COUNT query and may be 0.  The
 * ::AMD_DBGAPI_PROCESS_INFO_WATCHPOINT_SHARE query can be used to determine if
 * watchpoints are shared between processes.
 *
 * When a wave stops due to a data watchpoint the stop reason will include
 * ::AMD_DBGAPI_WAVE_STOP_REASON_WATCHPOINT.  The set of watchpoints triggered
 * can be queried using ::AMD_DBGAPI_WAVE_INFO_WATCHPOINTS.
 *
 * @{
 */

/**
 * Opaque hardware data watchpoint handle.
 *
 * Globally unique for a single library instance.
 */
typedef struct
{
  uint64_t handle;
} amd_dbgapi_watchpoint_id_t;

/**
 * The NULL hardware data watchpoint handle.
 */
#define AMD_DBGAPI_WATCHPOINT_NONE                                            \
    AMD_DBGAPI_HANDLE_LITERAL (amd_dbgapi_watchpoint_id_t, 0)

/**
 * Watchpoint queries that are supported by ::amd_dbgapi_watchpoint_get_info.
 *
 * Each query specifies the type of data returned in the \p value argument to
 * ::amd_dbgapi_watchpoint_get_info.
 */
typedef enum
{
  /**
   * Return the process to which this watchpoint belongs.  The type of this
   * attribute is ::amd_dbgapi_process_id_t.
   */
  AMD_DBGAPI_WATCHPOINT_INFO_PROCESS = 1,
  /**
   * The base address of the created watchpoint.  The type of this attribute is
   * ::amd_dbgapi_global_address_t.
   */
   AMD_DBGAPI_WATCHPOINT_INFO_ADDRESS = 2,
  /**
   * The byte size of the created watchpoint.  The type of this attribute is
   * ::amd_dbgapi_size_t.
   */
   AMD_DBGAPI_WATCHPOINT_INFO_SIZE = 3
} amd_dbgapi_watchpoint_info_t;

/**
 * Query information about a watchpoint.
 *
 * ::amd_dbgapi_watchpoint_info_t specifies the queries supported and the
 * type returned using the \p value argument.
 *
 * \param[in] watchpoint_id The handle of the watchpoint being queried.
 *
 * \param[in] query The query being requested.
 *
 * \param[in] value_size Size of the memory pointed to by \p value.  Must be
 * equal to the byte size of the query result.
 *
 * \param[out] value Pointer to memory where the query result is stored.
 *
 * \retval ::AMD_DBGAPI_STATUS_SUCCESS The function has been executed
 * successfully and the result is stored in \p value.
 *
 * \retval ::AMD_DBGAPI_STATUS_FATAL A fatal error occurred.  The library is
 * left uninitialized and \p value is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_NOT_INITIALIZED The library is not
 * initialized.  The library is left uninitialized and \p value is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_WATCHPOINT_ID \p watchpoint_id
 * is invalid.  \p value is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ARGUMENT \p value is NULL or
 * \p query is invalid.  \p value is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ARGUMENT_COMPATIBILITY \p
 * value_size does not match the size of the \p query result.  \p value is
 * unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_CLIENT_CALLBACK This will be
 * reported if the amd_dbgapi_callbacks_s::allocate_memory callback used to
 * allocate \p value returns NULL.  \p value is unaltered.
 */
amd_dbgapi_status_t AMD_DBGAPI amd_dbgapi_watchpoint_get_info (
    amd_dbgapi_watchpoint_id_t watchpoint_id,
    amd_dbgapi_watchpoint_info_t query, size_t value_size,
    void *value) AMD_DBGAPI_VERSION_0_54;

/**
 * The way watchpoints are shared between processes.
 *
 * The ::AMD_DBGAPI_PROCESS_INFO_WATCHPOINT_SHARE query can be used to
 * determine the watchpoint sharing for an architecture.
 */
typedef enum
{
  /**
   * Watchpoints are not supported.
   */
  AMD_DBGAPI_WATCHPOINT_SHARE_KIND_UNSUPPORTED = 0,
  /**
   * The watchpoints are not shared across processes.  Every process can use
   * all
   * ::AMD_DBGAPI_PROCESS_INFO_WATCHPOINT_COUNT watchpoints.
   */
  AMD_DBGAPI_WATCHPOINT_SHARE_KIND_UNSHARED = 1,
  /**
   * The watchpoints of a process are shared between all processes.  The number
   * of watchpoints available to a process may be reduced if watchpoints are
   * used by another process.
   */
  AMD_DBGAPI_WATCHPOINT_SHARE_KIND_SHARED = 2
} amd_dbgapi_watchpoint_share_kind_t;

/**
 * Watchpoint memory access kinds.
 *
 * The watchpoint is triggered only when the memory instruction is of the
 * specified kind.
 */
typedef enum
{
  /**
   * Read access by load instructions.
   */
  AMD_DBGAPI_WATCHPOINT_KIND_LOAD = 1,
  /**
   * Write access by store instructions or read-modify-write access by atomic
   * instructions.
   */
  AMD_DBGAPI_WATCHPOINT_KIND_STORE_AND_RMW = 2,
  /**
   * Read-modify-write access by atomic instructions.
   */
  AMD_DBGAPI_WATCHPOINT_KIND_RMW = 3,
  /**
   * Read, write, or read-modify-write access by load, store, or atomic
   * instructions.
   */
  AMD_DBGAPI_WATCHPOINT_KIND_ALL = 4
} amd_dbgapi_watchpoint_kind_t;

/**
 * A set of watchpoints.
 *
 * Used by the ::AMD_DBGAPI_WAVE_INFO_WATCHPOINTS query to report the
 * watchpoint(s) triggered by a wave.
 */
typedef struct
{
  size_t count;
  amd_dbgapi_watchpoint_id_t *watchpoint_ids;
} amd_dbgapi_watchpoint_list_t;

/**
 * Set a hardware data watchpoint.
 *
 * The AMD GPU has limitations on the base address and size of hardware data
 * watchpoints that can be set, and the limitations may vary by architecture. A
 * watchpoint is created with the smallest range, supported by the
 * architectures of all the agents of a process, that covers the requested
 * range specified by \p address and \p size.
 *
 * If the requested range is larger than is supported by the architectures of
 * all the agents of a process, then a watchpoint is created with the smallest
 * range that includes \p address and covers as much of the requested range as
 * possible.
 *
 * The range of the created watchpoint can be queried using
 * ::AMD_DBGAPI_WATCHPOINT_INFO_PROCESS and ::AMD_DBGAPI_WATCHPOINT_INFO_SIZE.
 * The client is responsible for determining if the created watchpoint
 * completely covers the requested range.  If it does not, the client can
 * attempt to create additional watchpoints for the uncovered portion of the
 * requested range.
 *
 * When a watchpoint is triggered, the client is responsible for determining if
 * the access was to the requested range.  For example, for writes the client
 * can compare the original value with the current value to determine if it
 * changed.
 *
 * Each process has its own set of watchpoints.  Only waves executing on the
 * agents of a process will trigger the watchpoints set on that process.
 *
 * \param[in] process_id The process on which to set the watchpoint.
 *
 * \param[in] address The base address of memory area to set a watchpoint.
 *
 * \param[in] size The non-zero number of bytes that the watchpoint should
 * cover.
 *
 * \param[in] kind The kind of memory access that should trigger the
 * watchpoint.
 *
 * \param[out] watchpoint_id The watchpoint created.
 *
 * \retval ::AMD_DBGAPI_STATUS_SUCCESS The function has been executed
 * successfully and the watchpoint has been created with handle \p
 * watchpoint_id.
 *
 * \retval ::AMD_DBGAPI_STATUS_FATAL A fatal error occurred.  The library is
 * left uninitialized; and \p watchpoint_id is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_NOT_INITIALIZED The library is not
 * initialized.  The library is left uninitialized; and \p watchpoint_id is
 * unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_PROCESS_ID \p process_id is
 * invalid.  No watchpoint is set and \p watchpoint_id is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_NO_WATCHPOINT_AVAILABLE No more
 * watchpoints are available.  No watchpoint is set and \p watchpoint_id is
 * unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_NOT_SUPPORTED Watchpoints are not
 * supported for the architectures of all the agents.  No watchpoint is set and
 * \p watchpoint_id is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ARGUMENT \p kind is invalid; \p
 * size is 0 or \p watchpoint_id is NULL.  No watchpoint is set and \p
 * watchpoint_id is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_PROCESS_FROZEN This operation is not
 * permitted when the process frozen.  No watchpoint is set and
 * \p watchpoint_id is unaltered.
 */
amd_dbgapi_status_t AMD_DBGAPI amd_dbgapi_set_watchpoint (
    amd_dbgapi_process_id_t process_id, amd_dbgapi_global_address_t address,
    amd_dbgapi_size_t size, amd_dbgapi_watchpoint_kind_t kind,
    amd_dbgapi_watchpoint_id_t *watchpoint_id) AMD_DBGAPI_VERSION_0_76;

/**
 * Remove a hardware data watchpoint previously set by
 * ::amd_dbgapi_set_watchpoint.
 *
 * \param[in] watchpoint_id The watchpoint to remove.
 *
 * \retval ::AMD_DBGAPI_STATUS_SUCCESS The function has been executed
 * successfully and the watchpoint has been removed.
 *
 * \retval ::AMD_DBGAPI_STATUS_FATAL A fatal error occurred.  The library is
 * left uninitialized.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_NOT_INITIALIZED The library is not
 * initialized.  The library is left uninitialized and no watchpoint is
 * removed.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_WATCHPOINT_ID \p watchpoint_id is
 * invalid.  No watchpoint is removed.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_PROCESS_FROZEN This operation is not
 * allowed when the process is frozen.  No watchpoint is removed.
 */
amd_dbgapi_status_t AMD_DBGAPI amd_dbgapi_remove_watchpoint (
    amd_dbgapi_watchpoint_id_t watchpoint_id) AMD_DBGAPI_VERSION_0_76;

/** @} */

/** \defgroup register_group Registers
 *
 * Operations related to AMD GPU register access.
 *
 * @{
 */

/**
 * Opaque register class handle.
 *
 * A handle that denotes the set of classes of hardware registers supported by
 * an architecture.  The registers of the architecture all belong to one or
 * more register classes.  The register classes are a convenience for grouping
 * registers that have similar uses and properties.  They can be useful when
 * presenting register lists to a user.  For example, there could be a register
 * class for \e system, \e general, and \e vector.
 *
 * Globally unique for a single library instance.
 */
typedef struct
{
  uint64_t handle;
} amd_dbgapi_register_class_id_t;

/**
 * The NULL register class handle.
 */
#define AMD_DBGAPI_REGISTER_CLASS_NONE                                        \
    AMD_DBGAPI_HANDLE_LITERAL (amd_dbgapi_register_class_id_t, 0)

/**
 * Register class queries that are supported by
 * ::amd_dbgapi_architecture_register_class_get_info.
 *
 * Each query specifies the type of data returned in the \p value argument to
 * ::amd_dbgapi_architecture_register_class_get_info.
 */
typedef enum
{
  /**
   * Return the architecture to which this register class belongs.  The type of
   * this attribute is ::amd_dbgapi_architecture_id_t.
   */
  AMD_DBGAPI_REGISTER_CLASS_INFO_ARCHITECTURE = 1,
  /**
   * Return the register class name.  The type of this attribute is a pointer
   * to a NUL terminated \p char.  It is allocated by the
   * amd_dbgapi_callbacks_s::allocate_memory callback and is owned by the
   * client.
   */
  AMD_DBGAPI_REGISTER_CLASS_INFO_NAME = 2
} amd_dbgapi_register_class_info_t;

/**
 * Query information about a register class of an architecture.
 *
 * ::amd_dbgapi_register_class_info_t specifies the queries supported and the
 * type returned using the \p value argument.
 *
 * \param[in] register_class_id The handle of the register class being queried.
 *
 * \param[in] query The query being requested.
 *
 * \param[in] value_size Size of the memory pointed to by \p value.  Must be
 * equal to the byte size of the query result.
 *
 * \param[out] value Pointer to memory where the query result is stored.
 *
 * \retval ::AMD_DBGAPI_STATUS_SUCCESS The function has been executed
 * successfully and the result is stored in \p value.
 *
 * \retval ::AMD_DBGAPI_STATUS_FATAL A fatal error occurred.  The library is
 * left uninitialized and \p value is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_NOT_INITIALIZED The library is not
 * initialized.  The library is left uninitialized and \p value is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_REGISTER_CLASS_ID \p
 * register_class_id is invalid.  \p value is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ARGUMENT \p value is NULL or
 * \p query is invalid.  \p value is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ARGUMENT_COMPATIBILITY \p
 * value_size does not match the size of the \p query result.  \p value is
 * unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_CLIENT_CALLBACK This will be
 * reported if the amd_dbgapi_callbacks_s::allocate_memory callback used to
 * allocate \p value returns NULL.  \p value is unaltered.
 */
amd_dbgapi_status_t AMD_DBGAPI
amd_dbgapi_architecture_register_class_get_info (
    amd_dbgapi_register_class_id_t register_class_id,
    amd_dbgapi_register_class_info_t query, size_t value_size,
    void *value) AMD_DBGAPI_VERSION_0_54;

/**
 * Report the list of register classes supported by the architecture.
 *
 * The order of the register handles in the list is stable between calls.
 *
 * \param[in] architecture_id The architecture being queried.
 *
 * \param[out] register_class_count The number of architecture register
 * classes.
 *
 * \param[out] register_classes A pointer to an array of
 * ::amd_dbgapi_register_class_id_t with \p register_class_count elements.  It
 * is allocated by the amd_dbgapi_callbacks_s::allocate_memory callback and
 * is owned by the client.
 *
 * \retval ::AMD_DBGAPI_STATUS_SUCCESS The function has been executed
 * successfully and the result is stored in \p register_class_count and
 * \p register_classes.
 *
 * \retval ::AMD_DBGAPI_STATUS_FATAL A fatal error occurred.  The library is
 * left uninitialized; and \p register_class_count and \p register_classes are
 * unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_NOT_INITIALIZED The library is not
 * initialized.  The library is left uninitialized; and \p register_class_count
 * and \p register_classes are unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ARCHITECTURE_ID \p
 * architecture_id is invalid.  \p register_class_count and \p register_classes
 * are unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ARGUMENT \p register_class_count
 * or \p register_classes are NULL.  \p register_class_count and
 * \p register_classes are unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_CLIENT_CALLBACK This will be
 * reported if the amd_dbgapi_callbacks_s::allocate_memory callback used to
 * allocate \p register_classes returns NULL.  \p register_class_count and
 * \p register_classes are unaltered.
 */
amd_dbgapi_status_t AMD_DBGAPI amd_dbgapi_architecture_register_class_list (
    amd_dbgapi_architecture_id_t architecture_id, size_t *register_class_count,
    amd_dbgapi_register_class_id_t **register_classes) AMD_DBGAPI_VERSION_0_54;

/**
 * Opaque register handle.
 *
 * A handle that denotes the set of hardware registers supported by an
 * architecture.
 *
 * Globally unique for a single library instance.
 */
typedef struct
{
  uint64_t handle;
} amd_dbgapi_register_id_t;

/**
 * The NULL register handle.
 */
#define AMD_DBGAPI_REGISTER_NONE                                              \
    AMD_DBGAPI_HANDLE_LITERAL (amd_dbgapi_register_id_t, 0)

/**
 * A bit mask on register properties.
 *
 * The properties of a register are available using the
 * ::AMD_DBGAPI_REGISTER_INFO_PROPERTIES query.
 */
typedef enum
{
  /**
   * There are no properties.
   */
  AMD_DBGAPI_REGISTER_PROPERTY_NONE = 0,
  /**
   * At least one bit of the register value is readonly.  It is advisable for
   * the client to read the register after writing it to determine the value of
   * the readonly bits.
   */
  AMD_DBGAPI_REGISTER_PROPERTY_READONLY_BITS = (1 << 0),
  /**
   * The register value may change as a consequence of changing a register of
   * the same wavefront with the
   * ::AMD_DBGAPI_REGISTER_PROPERTY_INVALIDATE_VOLATILE property.  It is
   * advisable for the client to not cache the value of the register.
   */
  AMD_DBGAPI_REGISTER_PROPERTY_VOLATILE = (1 << 1),
  /**
   * Changing the value of the register may change a register of the same
   * wavefront with the ::AMD_DBGAPI_REGISTER_PROPERTY_VOLATILE property.  It
   * is advisable to invalidate any cached registers with the
   * ::AMD_DBGAPI_REGISTER_PROPERTY_VOLATILE property.
   */
  AMD_DBGAPI_REGISTER_PROPERTY_INVALIDATE_VOLATILE = (1 << 2)
} amd_dbgapi_register_properties_t;

/**
 * Register queries that are supported by ::amd_dbgapi_register_get_info.
 *
 * Each query specifies the type of data returned in the \p value argument to
 * ::amd_dbgapi_register_get_info.
 */
typedef enum
{
  /**
   * Return the architecture to which this register belongs.  The type of this
   * attribute is ::amd_dbgapi_architecture_id_t.
   */
  AMD_DBGAPI_REGISTER_INFO_ARCHITECTURE = 1,
  /**
   * Return the register name.  The type of this attribute is a pointer to a
   * NUL terminated \p char.  It is allocated by the
   * amd_dbgapi_callbacks_s::allocate_memory callback and is owned by the
   * client.
   */
  AMD_DBGAPI_REGISTER_INFO_NAME = 2,
  /**
   * Return the size of the register in bytes.  The type of this attribute
   * is ::amd_dbgapi_size_t.
   */
  AMD_DBGAPI_REGISTER_INFO_SIZE = 3,
  /**
   * Return the register type as a C style type string.  This can be used as
   * the default type to use when displaying values of the register.  The type
   * string syntax is defined by the following BNF syntax:
   *
   *     type          ::= integer_type
   *                     | float_type
   *                     | function_type
   *                     | flag_type
   *                     | array_type
   *     array_type    ::= ( integer_type
   *                       | float_type
   *                       | function_type
   *                       | flag_type
   *                       ) "[" element_count "]"
   *     element_count ::= DECIMAL_NUMBER
   *     integer_type  ::= "uint32_t"
   *                     | "uint64_t"
   *     float_type    ::= "float"
   *                     | "double"
   *     function_type ::= "void(void)"
   *     flag_type     ::= ( "flags32_t"
   *                       | "flags64_t"
   *                       )
   *                       type_name
   *                       [ "{" [ fields ] "}" ]
   *     fields        ::= field ";" [ fields ]
   *     field         ::= "bool" field_name
   *                       "@" bit_position
   *                     | ( integer_type
   *                       | enum_type
   *                       )
   *                       field_name
   *                       "@" bit_position
   *                       "-" bit_position
   *     field_name    ::= IDENTIFIER
   *     enum_type     ::= "enum" type_name
   *                       [ "{" [ enum_values ] "}" ]
   *     enum_values   ::= enum_value [ "," enum_values ]
   *     enum_value    ::= enum_name "=" enum_ordinal
   *     enum_name     ::= IDENTIFIER
   *     enum_ordinal  ::= DECIMAL_NUMBER
   *     type_name     ::= IDENTIFIER
   *     bit_position  ::= DECIMAL_NUMBER
   *
   * ``IDENTIFIER`` is string starting with an alphabetic character followed by
   * zero or more alphabetic, numeric, "_", or "." characters.
   *
   * ``DECIMAL_NUMBER`` is a decimal C integral literal.
   *
   * Whitespace is allowed between lexical tokens.
   *
   * The type size matches the size of the register.  \p uint32, \p float, and
   * \p flag32 types are 4 bytes.  \p unit64, \p double, and \p flag64 types
   * are 8 bytes.  \p void(void) is the size of a global address.
   *
   * The type of this attribute is a pointer to a NUL terminated \p char.  It
   * is allocated by the amd_dbgapi_callbacks_s::allocate_memory callback and
   * is owned by the client.
   */
  AMD_DBGAPI_REGISTER_INFO_TYPE = 4,
  /**
   * Return the AMD GPU DWARF register number for the register's architecture.
   * The type of this attribute is \p uint64_t.
   *
   * If the requested register has no associated DWARF register number, then
   * ::amd_dbgapi_register_get_info returns the
   * ::AMD_DBGAPI_STATUS_ERROR_NOT_AVAILABLE error.
   */
  AMD_DBGAPI_REGISTER_INFO_DWARF = 5,
  /**
   * Return the register's properties.  The type of this attribute is \p
   * uint32_t with values defined by ::amd_dbgapi_register_properties_t.
   */
  AMD_DBGAPI_REGISTER_INFO_PROPERTIES = 6
} amd_dbgapi_register_info_t;

/**
 * Query information about a register.
 *
 * ::amd_dbgapi_register_info_t specifies the queries supported and the type
 * returned using the \p value argument.
 *
 * \param[in] register_id The handle of the register being queried.
 *
 * \param[in] query The query being requested.
 *
 * \param[in] value_size Size of the memory pointed to by \p value.  Must be
 * equal to the byte size of the query result.
 *
 * \param[out] value Pointer to memory where the query result is stored.
 *
 * \retval ::AMD_DBGAPI_STATUS_SUCCESS The function has been executed
 * successfully and the result is stored in \p value.
 *
 * \retval ::AMD_DBGAPI_STATUS_FATAL A fatal error occurred.  The library is
 * left uninitialized and \p value is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_NOT_INITIALIZED The library is not
 * initialized.  The library is left uninitialized and \p value is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_REGISTER_ID \p register_id is
 * invalid for \p architecture_id.  \p value is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ARGUMENT \p value is NULL, or \p
 * query is invalid or not supported for an architecture.  \p value is
 * unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ARGUMENT_COMPATIBILITY \p
 * value_size does not match the size of the \p query result.  \p value is
 * unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_NOT_AVAILABLE \p The requested information
 * is not available.  See ::amd_dbgapi_register_info_t for queries that can
 * produce this error.  \p value is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_CLIENT_CALLBACK This will be reported if
 * the amd_dbgapi_callbacks_s::allocate_memory callback used to allocate \p
 * value returns NULL.  \p value is unaltered.
 */
amd_dbgapi_status_t AMD_DBGAPI amd_dbgapi_register_get_info (
    amd_dbgapi_register_id_t register_id, amd_dbgapi_register_info_t query,
    size_t value_size, void *value) AMD_DBGAPI_VERSION_0_70;

/**
 * Indication of if a wave has a register.
 */
typedef enum
{
  /**
   * The wave does not have the register.
   */
  AMD_DBGAPI_REGISTER_ABSENT = 0,
  /**
   * The wave has the register.
   */
  AMD_DBGAPI_REGISTER_PRESENT = 1
} amd_dbgapi_register_exists_t;

/**
 * Query if a register exists for a wave.
 *
 * The register and wave must both belong to the same architecture.
 *
 * \param[in] wave_id The wave being queried.
 *
 * \param[in] register_id The register being queried.
 *
 * \param[out] exists Indication of whether \p wave_id has \p register_id.
 *
 * \retval ::AMD_DBGAPI_STATUS_SUCCESS The function has been executed
 * successfully and the result is stored in \p exists.
 *
 * \retval ::AMD_DBGAPI_STATUS_FATAL A fatal error occurred.  The library is
 * left uninitialized and \p exists is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_NOT_INITIALIZED The library is not
 * initialized.  The library is left uninitialized and \p exists is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_WAVE_ID \p wave_id is invalid.
 * \p exists is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_REGISTER_ID \p register_id is
 * invalid.  \p exists is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ARGUMENT \p exists is NULL.  \p
 * exists is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ARGUMENT_COMPATIBILITY The
 * architectures of \p wave_id and \p register_id are not the same.  \p exists
 * is unaltered.
 */
amd_dbgapi_status_t AMD_DBGAPI amd_dbgapi_wave_register_exists (
    amd_dbgapi_wave_id_t wave_id, amd_dbgapi_register_id_t register_id,
    amd_dbgapi_register_exists_t *exists) AMD_DBGAPI_VERSION_0_54;

/**
 * Report the list of registers supported by the architecture.
 *
 * This list is all the registers the architecture can support, but a specific
 * wave may not have all these registers.  For example, AMD GPU architectures
 * can specify the number of vector and scalar registers when a wave is
 * created.  Use the ::amd_dbgapi_wave_register_list operation to determine the
 * registers supported by a specific wave.
 *
 * The order of the register handles in the list is stable between calls and
 * registers on the same major class are contiguous in ascending hardware
 * number order.
 *
 * \param[in] architecture_id The architecture being queried.
 *
 * \param[out] register_count The number of architecture registers.
 *
 * \param[out] registers A pointer to an array of ::amd_dbgapi_register_id_t
 * with \p register_count elements.  It is allocated by the
 * amd_dbgapi_callbacks_s::allocate_memory callback and is owned by the
 * client.
 *
 * \retval ::AMD_DBGAPI_STATUS_SUCCESS The function has been executed
 * successfully and the result is stored in \p register_count and \p registers.
 *
 * \retval ::AMD_DBGAPI_STATUS_FATAL A fatal error occurred.  The library
 * is left uninitialized; and \p register_count and \p registers are unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_NOT_INITIALIZED The library is not
 * initialized.  The library is left uninitialized; and \p register_count and
 * \p registers are unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ARCHITECTURE_ID \p
 * architecture_id is invalid.  \p register_count and \p registers are
 * unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ARGUMENT \p register_count or
 * \p registers are NULL.  \p register_count and \p registers are unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_CLIENT_CALLBACK This will be
 * reported if the amd_dbgapi_callbacks_s::allocate_memory callback used to
 * allocate \p registers returns NULL.  \p register_count and \p registers are
 * unaltered.
 */
amd_dbgapi_status_t AMD_DBGAPI amd_dbgapi_architecture_register_list (
    amd_dbgapi_architecture_id_t architecture_id, size_t *register_count,
    amd_dbgapi_register_id_t **registers) AMD_DBGAPI_VERSION_0_54;

/**
 * Report the list of registers supported by a wave.
 *
 * This list is the registers allocated for a specific wave and may not be all
 * the registers supported by the architecture.  For example, AMD GPU
 * architectures can specify the number of vector and scalar registers when a
 * wave is created.  Use the ::amd_dbgapi_architecture_register_list operation
 * to determine the full set of registers supported by the architecture.
 *
 * The order of the register handles in the list is stable between calls.
 * It is equal to, or a subset of, those returned by
 * ::amd_dbgapi_architecture_register_list and in the same order.
 *
 * \param[in] wave_id The wave being queried.
 *
 * \param[out] register_count The number of wave registers.
 *
 * \param[out] registers A pointer to an array of ::amd_dbgapi_register_id_t
 * with \p register_count elements.  It is allocated by the
 * amd_dbgapi_callbacks_s::allocate_memory callback and is owned by the
 * client.
 *
 * \retval ::AMD_DBGAPI_STATUS_SUCCESS The function has been executed
 * successfully and the result is stored in \p register_count and \p registers.
 *
 * \retval ::AMD_DBGAPI_STATUS_FATAL A fatal error occurred.  The library is
 * left uninitialized; and \p register_count and \p registers are unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_NOT_INITIALIZED The library is not
 * initialized.  The library is left uninitialized; and \p register_count and
 * \p registers are unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_WAVE_ID \p wave_id is invalid.  \p
 * register_count and \p registers are unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ARGUMENT \p register_count or
 * \p registers are NULL.  \p register_count and \p registers are unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_CLIENT_CALLBACK This will be
 * reported if the amd_dbgapi_callbacks_s::allocate_memory callback used to
 * allocate \p registers returns NULL.  \p register_count and \p registers are
 * unaltered.
 */
amd_dbgapi_status_t AMD_DBGAPI amd_dbgapi_wave_register_list (
    amd_dbgapi_wave_id_t wave_id, size_t *register_count,
    amd_dbgapi_register_id_t **registers) AMD_DBGAPI_VERSION_0_54;

/**
 * Return a register handle from an AMD GPU DWARF register number for an
 * architecture.
 *
 * The AMD GPU DWARF register number must be valid for the architecture.
 *
 * See [User Guide for AMDGPU Backend - Code Object - DWARF - Register
 * Identifier] (https://llvm.org/docs/AMDGPUUsage.html#register-identifier).
 *
 * \param[in] architecture_id The architecture of the DWARF register.
 *
 * \param[in] dwarf_register The AMD GPU DWARF register number.
 *
 * \param[out] register_id The register handle that corresponds to the DWARF
 * register ID.
 *
 * \retval ::AMD_DBGAPI_STATUS_SUCCESS The function has been executed
 * successfully and the result is stored in \p register_id.
 *
 * \retval ::AMD_DBGAPI_STATUS_FATAL A fatal error occurred.  The library is
 * left uninitialized and \p register_id is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_NOT_INITIALIZED The library is not
 * initialized.  The library is left uninitialized and \p register_id is
 * unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ARCHITECTURE_ID \p
 * architecture_id is invalid.  \p register_id is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ARGUMENT \p register_id is NULL.
 * \p register_id is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ARGUMENT_COMPATIBILITY \p
 * dwarf_register is not valid for the \p architecture_id.  \p register_id is
 * unaltered.
 */
amd_dbgapi_status_t AMD_DBGAPI amd_dbgapi_dwarf_register_to_register (
    amd_dbgapi_architecture_id_t architecture_id, uint64_t dwarf_register,
    amd_dbgapi_register_id_t *register_id) AMD_DBGAPI_VERSION_0_54;

/**
 * Indication of whether a register is a member of a register class.
 */
typedef enum
{
  /**
   * The register is not a member of the register class.
   */
  AMD_DBGAPI_REGISTER_CLASS_STATE_NOT_MEMBER = 0,
  /**
   * The register is a member of the register class.
   */
  AMD_DBGAPI_REGISTER_CLASS_STATE_MEMBER = 1
} amd_dbgapi_register_class_state_t;

/**
 * Determine if a register is a member of a register class.
 *
 * The register and register class must both belong to the same architecture.
 *
 * \param[in] register_class_id The handle of the register class being queried.
 *
 * \param[in] register_id The handle of the register being queried.
 *
 * \param[out] register_class_state
 * ::AMD_DBGAPI_REGISTER_CLASS_STATE_NOT_MEMBER if the register is not in the
 * register class.  ::AMD_DBGAPI_REGISTER_CLASS_STATE_MEMBER if the register is
 * in the register class.
 *
 * \retval ::AMD_DBGAPI_STATUS_SUCCESS The function has been executed
 * successfully and the result is stored in \p register_class_state.
 *
 * \retval ::AMD_DBGAPI_STATUS_FATAL A fatal error occurred.  The library is
 * left uninitialized and \p register_class_state is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_NOT_INITIALIZED The library is not
 * initialized.  The library is left uninitialized and \p register_class_state
 * is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_REGISTER_ID \p register_id is
 * invalid.  \p register_class_state is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_REGISTER_CLASS_ID \p
 * register_class_id is invalid.  \p register_class_state is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ARGUMENT \p register_class_state
 * is NULL.  \p register_class_state is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ARGUMENT_COMPATIBILITY The
 * architectures of \p register_class_id and \p register_id are not the same.
 * \p register_class_state is unaltered.
 */
amd_dbgapi_status_t AMD_DBGAPI amd_dbgapi_register_is_in_register_class (
    amd_dbgapi_register_class_id_t register_class_id,
    amd_dbgapi_register_id_t register_id,
    amd_dbgapi_register_class_state_t *register_class_state)
    AMD_DBGAPI_VERSION_0_54;

/**
 * Read a register.
 *
 * \p value_size bytes are read from the register starting at \p offset into \p
 * value.
 *
 * The wave must be stopped.  The register and wave must both belong to the
 * same architecture, and the wave must have allocated that register.
 *
 * The register size can be obtained using ::amd_dbgapi_register_get_info with
 * the ::AMD_DBGAPI_REGISTER_INFO_SIZE query.
 *
 * \param[in] wave_id The wave to being queried for the register.
 *
 * \param[in] register_id The register being requested.
 *
 * \param[in] offset The first byte to start reading the register.  The offset
 * is zero based starting from the least significant byte of the register.
 *
 * \param[in] value_size The number of bytes to read from the register which
 * must be greater than 0 and less than the size of the register minus \p
 * offset.
 *
 * \param[out] value The bytes read from the register.  Must point to an array
 * of at least \p value_size bytes.
 *
 * \retval ::AMD_DBGAPI_STATUS_SUCCESS The function has been executed
 * successfully and \p value is set to \p value_size bytes starting at \p
 * offset from the contents of the register.
 *
 * \retval ::AMD_DBGAPI_STATUS_FATAL A fatal error occurred.  The library is
 * left uninitialized and \p value is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_NOT_INITIALIZED The library is not
 * initialized.  The library is left uninitialized and \p value is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_WAVE_ID \p wave_id is invalid.  \p
 * value is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_REGISTER_ID \p register_id is
 * invalid.  \p value is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_WAVE_NOT_STOPPED \p wave_id is not
 * stopped.  \p value is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ARGUMENT \p value is NULL or \p
 * value_size is 0.  \p value is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ARGUMENT_COMPATIBILITY \p
 * value_size is greater than the size of the \p register_id minus \p offset
 * or the architectures of \p wave_id and \p register_id are not the same.  \p
 * value is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_REGISTER_NOT_AVAILABLE \p register_id is
 * not allocated for \p wave_id.  \p value is unaltered.
 */
amd_dbgapi_status_t AMD_DBGAPI amd_dbgapi_read_register (
    amd_dbgapi_wave_id_t wave_id, amd_dbgapi_register_id_t register_id,
    amd_dbgapi_size_t offset, amd_dbgapi_size_t value_size,
    void *value) AMD_DBGAPI_VERSION_0_62;

/**
 * Write a register.
 *
 * \p value_size bytes are written into the register starting at \p offset.
 *
 * The wave must be stopped.  The register and wave must both belong to the
 * same architecture, and the wave must have allocated that register.  The wave
 * must not have an active displaced stepping buffer (see
 * ::amd_dbgapi_displaced_stepping_start) as the program counter and other
 * registers may be changed as part of creating the displaced stepping buffer.
 *
 * The register size can be obtained using ::amd_dbgapi_register_get_info with
 * the ::AMD_DBGAPI_REGISTER_INFO_SIZE query.
 *
 * \param[in] wave_id The wave to being queried for the register.
 *
 * \param[in] register_id The register being requested.
 *
 * \param[in] offset The first byte to start writing the register.  The offset
 * is zero based starting from the least significant byte of the register.
 *
 * \param[in] value_size The number of bytes to write to the register which
 * must be greater than 0 and less than the size of the register minus \p
 * offset.
 *
 * \param[in] value The bytes to write to the register.  Must point to an array
 * of at least \p value_size bytes.
 *
 * \retval ::AMD_DBGAPI_STATUS_SUCCESS The function has been executed
 * successfully and \p value_size bytes have been written to the contents of
 * the register starting at \p offset.
 *
 * \retval ::AMD_DBGAPI_STATUS_FATAL A fatal error occurred.  The library is
 * left uninitialized and the register is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_NOT_INITIALIZED The library is not
 * initialized.  The library is left uninitialized.  The register is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_WAVE_ID \p wave_id is invalid.
 * The register is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_REGISTER_ID \p register_id is
 * invalid.  \p value is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_WAVE_NOT_STOPPED \p wave_id is not
 * stopped.  The register is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_DISPLACED_STEPPING_ACTIVE \p wave_id has
 * an active displaced stepping buffer.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ARGUMENT \p value is NULL or \p
 * value_size is 0.  \p value is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ARGUMENT_COMPATIBILITY \p
 * value_size is greater than the size of the \p register_id minus \p offset
 * or the architectures of \p wave_id and \p register_id are not the same.  \p
 * value is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_REGISTER_NOT_AVAILABLE \p register_id is
 * not allocated for \p wave_id.  \p value is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_PROCESS_FROZEN \p the process the wave
 * belongs to is frozen.  \p value is unaltered.
 */
amd_dbgapi_status_t AMD_DBGAPI amd_dbgapi_write_register (
    amd_dbgapi_wave_id_t wave_id, amd_dbgapi_register_id_t register_id,
    amd_dbgapi_size_t offset, amd_dbgapi_size_t value_size,
    const void *value) AMD_DBGAPI_VERSION_0_76;

/**
 * Prefetch register values.
 *
 * A hint to indicate that a range of registers may be read using
 * ::amd_dbgapi_read_register in the future.  This can improve the performance
 * of reading registers as the library may be able to batch the prefetch
 * requests into one request.
 *
 * The wave must be stopped.  The register and wave must both belong to the
 * same architecture, and the wave must have allocated that register.
 *
 * If the wave is resumed, then any prefetch requests for registers that were
 * not subsequently read may be discarded and so provide no performance
 * benefit.  Prefetch requests for registers that are never subsequently read
 * may in fact reduce performance.
 *
 * The registers to prefetch are specified as the first register and the number
 * of registers.  The first register can be any register supported by the wave.
 * The number of registers is in terms of the wave register order returned by
 * ::amd_dbgapi_wave_register_list.  If the number exceeds the number of wave
 * registers, then only up to the last wave register is prefetched.
 *
 * \param[in] wave_id The wave being queried for the register.
 *
 * \param[in] register_id The first register being requested.
 *
 * \param[in] register_count The number of registers being requested.
 *
 * \retval ::AMD_DBGAPI_STATUS_SUCCESS The function has been executed
 * successfully.  Registers may be prefetched.
 *
 * \retval ::AMD_DBGAPI_STATUS_FATAL A fatal error occurred.  The library is
 * left uninitialized.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_NOT_INITIALIZED The library is not
 * initialized.  The library is left uninitialized.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_WAVE_ID \p wave_id is invalid.  No
 * registers are prefetched.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_REGISTER_ID \p register_id is
 * invalid.  No registers are prefetched.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_WAVE_NOT_STOPPED \p wave_id is not
 * stopped.  No registers are prefetched.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ARGUMENT_COMPATIBILITY The
 * architectures of \p wave_id and \p register_id are not the same.  No
 * registers are prefetched.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_REGISTER_NOT_AVAILABLE \p register_id is
 * not allocated for \p wave_id.  No registers are prefetched.
 */
amd_dbgapi_status_t AMD_DBGAPI amd_dbgapi_prefetch_register (
    amd_dbgapi_wave_id_t wave_id, amd_dbgapi_register_id_t register_id,
    amd_dbgapi_size_t register_count) AMD_DBGAPI_VERSION_0_62;

/**
 * Instruction information for direct call instructions.
 *
 * Used by ::amd_dbgapi_classify_instruction to report the target address and
 * saved return registers IDs information for instructions of the
 * ::AMD_DBGAPI_INSTRUCTION_KIND_DIRECT_CALL_REGISTER_PAIR kind.
 */
typedef struct
{
  amd_dbgapi_global_address_t target_address;
  amd_dbgapi_register_id_t saved_return_address_register[2];
} amd_dbgapi_direct_call_register_pair_information_t;

/** @} */

/** \defgroup memory_group Memory
 *
 * Operations related to AMD GPU memory access.
 *
 * The AMD GPU supports allocating memory in different address spaces.  See
 * [User Guide for AMDGPU Backend - LLVM - Address Spaces]
 * (https://llvm.org/docs/AMDGPUUsage.html#address-spaces).
 *
 * @{
 */

/**
 * A wave lane handle.
 *
 * A wave can have one or more lanes controlled by an execution mask.  Vector
 * instructions will be performed for each lane of the wave that the execution
 * mask has enabled.  Vector instructions can access registers that are vector
 * registers.  A vector register has a separate value for each lane, and vector
 * instructions will access the corresponding component for each lane's
 * evaluation of the instruction.
 *
 * The number of lanes of a wave can be obtained with the
 * ::AMD_DBGAPI_WAVE_INFO_LANE_COUNT query.  Different waves of the same
 * architecture can have different lane counts.
 *
 * The AMD GPU compiler may map source language threads of execution to lanes
 * of a wave.  The DWARF debug information which maps such source languages to
 * the generated architecture specific code must include information about the
 * lane mapping.
 *
 * The \p DW_ASPACE_AMDGPU_private_lane DWARF address space supports memory
 * allocated independently for each lane of a wave.
 *
 * Lanes are numbered from 0 to ::AMD_DBGAPI_WAVE_INFO_LANE_COUNT minus 1.
 *
 * Only unique for a single wave of a single process.
 */
typedef uint32_t amd_dbgapi_lane_id_t;

/**
 * The NULL lane handle.
 */
#define AMD_DBGAPI_LANE_NONE ((amd_dbgapi_lane_id_t) (-1))

/**
 * Opaque source language address class handle.
 *
 * A source language address class describes the source language address
 * spaces.  It is used to define source language pointer and reference types.
 * Each architecture has its own mapping of them to the architecture specific
 * address spaces.
 *
 * Globally unique for a single library instance.
 *
 * See [User Guide for AMDGPU Backend - Code Object - DWARF - Address Class
 * Mapping] (https://llvm.org/docs/AMDGPUUsage.html#address-class-mapping).
 */
typedef struct
{
  uint64_t handle;
} amd_dbgapi_address_class_id_t;

/**
 * The NULL address class handle.
 */
#define AMD_DBGAPI_ADDRESS_CLASS_NONE                                         \
    AMD_DBGAPI_HANDLE_LITERAL (amd_dbgapi_address_class_id_t, 0)

/**
 * Source language address class queries that are supported by
 * ::amd_dbgapi_address_class_get_info.
 *
 * Each query specifies the type of data returned in the \p value argument to
 * ::amd_dbgapi_address_class_get_info.
 */
typedef enum
{
  /**
   * Return the source language address class name.  The type of this attribute
   * is a pointer to a NUL terminated \p char.  It is allocated by the
   * amd_dbgapi_callbacks_s::allocate_memory callback and is owned by the
   * client.
   */
  AMD_DBGAPI_ADDRESS_CLASS_INFO_NAME = 1,
  /**
   * Return the architecture specific address space that is used to implement a
   * pointer or reference to the source language address class.  The type of
   * this attribute is ::amd_dbgapi_address_class_id_t.
   *
   * See [User Guide for AMDGPU Backend - Code Object - DWARF - Address Class
   * Mapping] (https://llvm.org/docs/AMDGPUUsage.html#address-class-mapping).
   */
  AMD_DBGAPI_ADDRESS_CLASS_INFO_ADDRESS_SPACE = 2,
  /**
   * Return the AMD GPU DWARF address class number for the address class'
   * architecture.  The type of this attribute is \p uint64_t.
   */
  AMD_DBGAPI_ADDRESS_CLASS_INFO_DWARF = 3
} amd_dbgapi_address_class_info_t;

/**
 * Query information about a source language address class of an architecture.
 *
 * ::amd_dbgapi_address_class_info_t specifies the queries supported and the
 * type returned using the \p value argument.
 *
 * \param[in] address_class_id The handle of the source language address class
 * being queried.
 *
 * \param[in] query The query being requested.
 *
 * \param[in] value_size Size of the memory pointed to by \p value.  Must be
 * equal to the byte size of the query result.
 *
 * \param[out] value Pointer to memory where the query result is stored.
 *
 * \retval ::AMD_DBGAPI_STATUS_SUCCESS The function has been executed
 * successfully and the result is stored in \p value.
 *
 * \retval ::AMD_DBGAPI_STATUS_FATAL A fatal error occurred.  The library is
 * left uninitialized and \p value is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_NOT_INITIALIZED The library is not
 * initialized.  The library is left uninitialized and \p value is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ADDRESS_CLASS_ID \p
 * address_class_id is invalid.  \p value is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ARGUMENT \p value is NULL or \p
 * query is invalid.  \p value is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ARGUMENT_COMPATIBILITY \p
 * value_size does not match the size of the \p query result.  \p value is
 * unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_CLIENT_CALLBACK This will be reported if
 * the amd_dbgapi_callbacks_s::allocate_memory callback used to allocate \p
 * value returns NULL.  \p value is unaltered.
 */
amd_dbgapi_status_t AMD_DBGAPI amd_dbgapi_address_class_get_info (
    amd_dbgapi_address_class_id_t address_class_id,
    amd_dbgapi_address_class_info_t query, size_t value_size,
    void *value) AMD_DBGAPI_VERSION_0_62;

/**
 * Report the list of source language address classes supported by the
 * architecture.
 *
 * The order of the source language address class handles in the list is stable
 * between calls.
 *
 * \param[in] architecture_id The architecture being queried.
 *
 * \param[out] address_class_count The number of architecture source language
 * address classes.
 *
 * \param[out] address_classes A pointer to an array of
 * ::amd_dbgapi_address_class_id_t with \p address_class_count elements.  It is
 * allocated by the amd_dbgapi_callbacks_s::allocate_memory callback and is
 * owned by the client.
 *
 * \retval ::AMD_DBGAPI_STATUS_SUCCESS The function has been executed
 * successfully and the result is stored in \p address_class_count and \p
 * address_classes.
 *
 * \retval ::AMD_DBGAPI_STATUS_FATAL A fatal error occurred.  The library is
 * left uninitialized; and \p address_class_count and \p address_classes are
 * unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_NOT_INITIALIZED The library is not
 * initialized.  The library is left uninitialized; and \p address_class_count
 * and \p address_classes are unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ARCHITECTURE_ID \p architecture_id
 * is invalid.  \p address_class_count and \p address_classes are unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ARGUMENT \p address_class_count or
 * \p address_classes are NULL.  \p address_class_count and \p address_classes
 * are unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_CLIENT_CALLBACK This will be reported if
 * the amd_dbgapi_callbacks_s::allocate_memory callback used to allocate \p
 * address_classes returns NULL.  \p address_class_count and \p address_classes
 * are unaltered.
 */
amd_dbgapi_status_t AMD_DBGAPI amd_dbgapi_architecture_address_class_list (
    amd_dbgapi_architecture_id_t architecture_id, size_t *address_class_count,
    amd_dbgapi_address_class_id_t **address_classes) AMD_DBGAPI_VERSION_0_54;

/**
 * Return the architecture source language address class from a DWARF address
 * class number for an architecture.
 *
 * The AMD GPU DWARF address class number must be valid for the architecture.
 *
 * See [User Guide for AMDGPU Backend - Code Object - DWARF - Address Class
 * Mapping] (https://llvm.org/docs/AMDGPUUsage.html#address-class-mapping).
 *
 * \param[in] architecture_id The architecture of the source language address
 * class.
 *
 * \param[in] dwarf_address_class The DWARF source language address class.
 *
 * \param[out] address_class_id The source language address class that
 * corresponds to the DWARF address class for the architecture.
 *
 * \retval ::AMD_DBGAPI_STATUS_SUCCESS The function has been executed
 * successfully and the result is stored in \p address_class_id.
 *
 * \retval ::AMD_DBGAPI_STATUS_FATAL A fatal error occurred.  The library is
 * left uninitialized and \p address_class_id is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_NOT_INITIALIZED The library is not
 * initialized.  The library is left uninitialized and \p address_class_id is
 * unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ARCHITECTURE_ID \p architecture_id
 * is invalid.  \p address_class_id is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ARGUMENT \p address_class_id is
 * NULL.  \p address_class_id is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ARGUMENT_COMPATIBILITY \p
 * dwarf_address_class is not valid for the \p architecture_id.  \p
 * address_class_id is unaltered.
 */
amd_dbgapi_status_t AMD_DBGAPI
amd_dbgapi_dwarf_address_class_to_address_class (
    amd_dbgapi_architecture_id_t architecture_id, uint64_t dwarf_address_class,
    amd_dbgapi_address_class_id_t *address_class_id) AMD_DBGAPI_VERSION_0_54;

/**
 * Opaque address space handle.
 *
 * A handle that denotes the set of address spaces supported by an
 * architecture.
 *
 * Globally unique for a single library instance.
 *
 * See [User Guide for AMDGPU Backend - LLVM - Address Spaces]
 * (https://llvm.org/docs/AMDGPUUsage.html#address-spaces).
 */
typedef struct
{
  uint64_t handle;
} amd_dbgapi_address_space_id_t;

/**
 * The NULL address space handle.
 */
#define AMD_DBGAPI_ADDRESS_SPACE_NONE                                         \
    AMD_DBGAPI_HANDLE_LITERAL (amd_dbgapi_address_space_id_t, 0)

/**
 * The global address space handle.
 *
 * Every architecture supports a global address space that uses the same
 * address space ID.
 */
#define AMD_DBGAPI_ADDRESS_SPACE_GLOBAL                                       \
    AMD_DBGAPI_HANDLE_LITERAL (amd_dbgapi_address_space_id_t, 1)

/**
 * Indication of how the address space is accessed.
 */
typedef enum
{
  /**
   * The address space supports all accesses.  Values accessed can change
   * during the lifetime of the program.
   */
  AMD_DBGAPI_ADDRESS_SPACE_ACCESS_ALL = 1,
  /**
   * The address space is read only.  Values accessed are always the same
   * value for the lifetime of the program execution.
   */
  AMD_DBGAPI_ADDRESS_SPACE_ACCESS_PROGRAM_CONSTANT = 2,
  /**
   * The address space is only read the waves of a kernel dispatch.  Values
   * accessed are always the same value for the lifetime of the dispatch.
   */
  AMD_DBGAPI_ADDRESS_SPACE_ACCESS_DISPATCH_CONSTANT = 3
} amd_dbgapi_address_space_access_t;

/**
 * Address space queries that are supported by
 * ::amd_dbgapi_address_space_get_info.
 *
 * Each query specifies the type of data returned in the \p value argument to
 * ::amd_dbgapi_address_space_get_info.
 */
typedef enum
{
  /**
   * Return the address space name.  The type of this attribute is a pointer to
   * a NUL terminated \p char*.  It is allocated by the
   * amd_dbgapi_callbacks_s::allocate_memory callback and is owned by the
   * client.
   */
  AMD_DBGAPI_ADDRESS_SPACE_INFO_NAME = 1,
  /**
   * Return the byte size of an address in the address space.  The type of this
   * attribute is ::amd_dbgapi_size_t.
   */
  AMD_DBGAPI_ADDRESS_SPACE_INFO_ADDRESS_SIZE = 2,
  /**
   * Return the NULL segment address value in the address space.  The type of
   * this attribute is \p amd_dbgapi_segment_address_t.
   */
  AMD_DBGAPI_ADDRESS_SPACE_INFO_NULL_ADDRESS = 3,
  /**
   * Return the address space access.  The type of this attribute is \p
   * uint32_t with values defined by ::amd_dbgapi_address_space_access_t.
   */
  AMD_DBGAPI_ADDRESS_SPACE_INFO_ACCESS = 4,
  /**
   * Return the AMD GPU DWARF address space number for the address space's
   * architecture. The type of this attribute is \p uint64_t.
   */
  AMD_DBGAPI_ADDRESS_SPACE_INFO_DWARF = 5
} amd_dbgapi_address_space_info_t;

/**
 * Query information about an address space.
 *
 * ::amd_dbgapi_address_space_info_t specifies the queries supported and the
 * type returned using the \p value argument.
 *
 * \param[in] address_space_id The address space.
 *
 * \param[in] query The query being requested.
 *
 * \param[in] value_size Size of the memory pointed to by \p value.  Must be
 * equal to the byte size of the query result.
 *
 * \param[out] value Pointer to memory where the query result is stored.
 *
 * \retval ::AMD_DBGAPI_STATUS_SUCCESS The function has been executed
 * successfully and the result is stored in \p value.
 *
 * \retval ::AMD_DBGAPI_STATUS_FATAL A fatal error occurred.  The library is
 * left uninitialized and \p value is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_NOT_INITIALIZED The library is not
 * initialized.  The library is left uninitialized and \p value is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ADDRESS_SPACE_ID \p
 * address_space_id is invalid.  \p value is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ARGUMENT \p query is invalid or \p
 * value is NULL.  \p value is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ARGUMENT_COMPATIBILITY \p
 * value_size does not match the size of the \p query result.  \p value is
 * unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_CLIENT_CALLBACK This will be reported if
 * the amd_dbgapi_callbacks_s::allocate_memory callback used to allocate \p
 * value returns NULL.  \p value is unaltered.
 */
amd_dbgapi_status_t AMD_DBGAPI amd_dbgapi_address_space_get_info (
    amd_dbgapi_address_space_id_t address_space_id,
    amd_dbgapi_address_space_info_t query, size_t value_size,
    void *value) AMD_DBGAPI_VERSION_0_62;

/**
 * Report the list of address spaces supported by the architecture.
 *
 * The order of the address space handles in the list is stable between calls.
 *
 * \param[in] architecture_id The architecture being queried.
 *
 * \param[out] address_space_count The number of architecture address spaces.
 *
 * \param[out] address_spaces A pointer to an array of
 * ::amd_dbgapi_address_space_id_t with \p address_space_count elements.  It is
 * allocated by the amd_dbgapi_callbacks_s::allocate_memory callback and is
 * owned by the client.
 *
 * \retval ::AMD_DBGAPI_STATUS_SUCCESS The function has been executed
 * successfully and the result is stored in \p address_space_count and \p
 * address_spaces.
 *
 * \retval ::AMD_DBGAPI_STATUS_FATAL A fatal error occurred.  The library is
 * left uninitialized; and \p address_space_count and \p address_spaces are
 * unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_NOT_INITIALIZED The library is not
 * initialized.  The library is left uninitialized; and \p address_space_count
 * and \p address_spaces are unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ARCHITECTURE_ID \p architecture_id
 * is invalid.  \p address_space_count and \p address_spaces are unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ARGUMENT \p address_space_count
 * and \p address_spaces are NULL.  \p address_space_count and \p
 * address_spaces are unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_CLIENT_CALLBACK This will be reported if
 * the amd_dbgapi_callbacks_s::allocate_memory callback used to allocate \p
 * address_spaces returns NULL.  \p address_space_count and \p address_spaces
 * are unaltered.
 */
amd_dbgapi_status_t AMD_DBGAPI amd_dbgapi_architecture_address_space_list (
    amd_dbgapi_architecture_id_t architecture_id, size_t *address_space_count,
    amd_dbgapi_address_space_id_t **address_spaces) AMD_DBGAPI_VERSION_0_54;

/**
 * Return the address space from an AMD GPU DWARF address space number for an
 * architecture.
 *
 * A DWARF address space describes the architecture specific address spaces.
 * It is used in DWARF location expressions that calculate addresses.  See
 * [User Guide for AMDGPU Backend - Code Object - DWARF - Address Space
 * Mapping] (https://llvm.org/docs/AMDGPUUsage.html#address-space-mapping).
 *
 * The AMD GPU DWARF address space number must be valid for the architecture.
 *
 * \param[in] architecture_id The architecture of the address space.
 *
 * \param[in] dwarf_address_space The AMD GPU DWARF address space.
 *
 * \param[out] address_space_id The address space that corresponds to the DWARF
 * address space for the architecture \p architecture_id.
 *
 * \retval ::AMD_DBGAPI_STATUS_SUCCESS The function has been executed
 * successfully and the result is stored in \p address_space_id.
 *
 * \retval ::AMD_DBGAPI_STATUS_FATAL A fatal error occurred.  The library is
 * left uninitialized and \p address_space_id is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_NOT_INITIALIZED The library is not
 * initialized.  The library is left uninitialized and \p address_space_id is
 * unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ARCHITECTURE_ID \p architecture_id
 * is invalid.  \p address_space_id is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ARGUMENT \p address_space_id is
 * NULL.  \p address_space_id is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ARGUMENT_COMPATIBILITY \p
 * dwarf_address_space is not valid for \p architecture_id.  \p
 * address_class_id is unaltered.
 */
amd_dbgapi_status_t AMD_DBGAPI
amd_dbgapi_dwarf_address_space_to_address_space (
    amd_dbgapi_architecture_id_t architecture_id, uint64_t dwarf_address_space,
    amd_dbgapi_address_space_id_t *address_space_id) AMD_DBGAPI_VERSION_0_54;

/**
 * Each address space has its own linear address to access it termed a segment
 * address.
 *
 * Different address spaces may have memory locations that alias each other,
 * but the segment address for such memory locations may be different in each
 * address space.  Consequently a segment address is specific to an address
 * space.
 *
 * Some address spaces may access memory that is allocated independently for
 * each workgroup, for each wave, or for each lane of of a wave.  Consequently
 * a segment address may be specific to a wave or lane of a wave.
 *
 * See [User Guide for AMDGPU Backend - LLVM - Address Spaces]
 * (https://llvm.org/docs/AMDGPUUsage.html#address-spaces).
 */
typedef uint64_t amd_dbgapi_segment_address_t;

/**
 * Convert a source segment address in the source address space into a
 * destination segment address in the destination address space.
 *
 * If the source segment address is the NULL value in the source address space
 * then it is converted to the NULL value in the destination address space.
 * The NULL address is provided by the
 * ::AMD_DBGAPI_ADDRESS_SPACE_INFO_NULL_ADDRESS query.
 *
 * An error is returned if the source segment address has no corresponding
 * segment address in the destination address space.
 *
 * The source and destination address spaces do not have to have the same
 * linear ordering.  For example, for AMD GPU the \p private_swizzled address
 * space is implemented as \p global address space memory that interleaves the
 * dwords of the wave's lanes.  So converting a \p private_swizzled address to
 * a \p global address will result in the corresponding scratch backing memory
 * address.  The \p destination_contiguous_bytes will indicate how many bytes,
 * starting at the \p destination_segment_address, before the scratch backing
 * memory corresponds to a dword of the adjacent lane.  To get the scratch
 * backing memory address of the byte after \p destination_contiguous_bytes
 * bytes requires ::amd_dbgapi_convert_address_space to be called again with
 * the address \p source_segment_address plus \p destination_contiguous_bytes.
 *
 * A client can use this operation to help manage caching the bytes of
 * different address spaces.  An address in an address space that is being
 * accessed can attempt to be converted to the various address spaces being
 * cached to see if it aliases with bytes being cached under a different
 * address space.  For example, an address in the AMD GPU \p generic address
 * space may alias with an address in the \p global, \p private_swizzled, or \p
 * local address spaces.
 *
 * \param[in] wave_id The wave that is using the address.  If the \p
 * address_space is ::AMD_DBGAPI_ADDRESS_SPACE_GLOBAL then \p wave_id may be
 * ::AMD_DBGAPI_WAVE_NONE, as the address space does not depend on the active
 * wave, in which case \p process_id is used.
 *
 * \param[in] lane_id The lane of the \p wave_id that is using the address.  If
 * the \p address_space does not depend on the active lane then this may be
 * ::AMD_DBGAPI_LANE_NONE.  For example, the ::AMD_DBGAPI_ADDRESS_SPACE_GLOBAL
 * address space does not depend on the lane.
 *
 * \param[in] source_address_space_id The address space of the \p
 * source_segment_address.
 *
 * \param[in] source_segment_address The integral value of the source segment
 * address.  Only the bits corresponding to the address size for the \p
 * source_address_space_id requested are used.  The address size is provided by
 * the ::AMD_DBGAPI_ADDRESS_SPACE_INFO_ADDRESS_SIZE query.
 *
 * \param[in] destination_address_space_id The address space to which to
 * convert \p source_segment_address that is in \p source_address_space_id.
 *
 * \param[out] destination_segment_address The integral value of the segment
 * address in \p destination_address_space_id that corresponds to \p
 * source_segment_address in \p source_address_space_id.  The bits
 * corresponding to the address size for the \p destination_address_space_id
 * are updated, and any remaining bits are set to zero.  The address size is
 * provided by the ::AMD_DBGAPI_ADDRESS_SPACE_INFO_ADDRESS_SIZE query.
 *
 * \param[out] destination_contiguous_bytes The number of contiguous bytes for
 * which the converted \p destination_segment_address continues to correspond
 * to the \p source_segment_address.
 *
 * \retval ::AMD_DBGAPI_STATUS_SUCCESS The function has been executed
 * successfully and the result is stored in \p destination_segment_address and
 * \p destination_contiguous_bytes.
 *
 * \retval ::AMD_DBGAPI_STATUS_FATAL A fatal error occurred.  The library is
 * left uninitialized and \p destination_segment_address and \p
 * destination_contiguous_bytes are unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_NOT_INITIALIZED The library is not
 * initialized.  The library is left uninitialized and \p
 * destination_segment_address and \p destination_contiguous_bytes are
 * unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_WAVE_ID \p wave_id is invalid, or
 * \p wave_id is ::AMD_DBGAPI_WAVE_NONE and \p source_address_space_id or \p
 * destination_address_space_id depends on the active wave.
 * \p destination_segment_address and \p destination_contiguous_bytes are
 * unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_LANE_ID \p wave_id is
 * ::AMD_DBGAPI_WAVE_NONE and \p lane_id is not ::AMD_DBGAPI_LANE_NONE.  \p
 * wave_id is not ::AMD_DBGAPI_WAVE_NONE and \p lane_id is not
 * ::AMD_DBGAPI_LANE_NONE and is not valid for \p wave_id.  \p lane_id is
 * ::AMD_DBGAPI_LANE_NONE and \p source_address_space_id or \p
 * destination_address_space_id depends on the active lane.  \p
 * destination_segment_address and \p destination_contiguous_bytes are
 * unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ADDRESS_SPACE_ID \p
 * source_address_space_id or \p destination_address_space_id are invalid.  \p
 * destination_segment_address and \p destination_contiguous_bytes are
 * unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ADDRESS_SPACE_CONVERSION The \p
 * source_segment_address in the \p source_address_space_id is not an address
 * that can be represented in the \p destination_address_space_id.
 * \p destination_segment_address and \p destination_contiguous_bytes are
 * unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ARGUMENT \p
 * destination_segment_address or \p destination_contiguous_bytes are NULL.
 * \p destination_segment_address and \p destination_contiguous_bytes are
 * unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ARGUMENT_COMPATIBILITY \p
 * source_address_space_id or \p destination_address_space_id address spaces
 * are not supported by the architecture of \p wave_id (if not
 * ::AMD_DBGAPI_WAVE_NONE).  \p destination_segment_address and \p
 * destination_contiguous_bytes are unaltered.
 */
amd_dbgapi_status_t AMD_DBGAPI amd_dbgapi_convert_address_space (
    amd_dbgapi_wave_id_t wave_id, amd_dbgapi_lane_id_t lane_id,
    amd_dbgapi_address_space_id_t source_address_space_id,
    amd_dbgapi_segment_address_t source_segment_address,
    amd_dbgapi_address_space_id_t destination_address_space_id,
    amd_dbgapi_segment_address_t *destination_segment_address,
    amd_dbgapi_size_t *destination_contiguous_bytes)
    AMD_DBGAPI_VERSION_0_62;

/**
 * The dependency when reading or writing a specific segment address of an
 * address space using the ::amd_dbgapi_read_memory and
 * ::amd_dbgapi_write_memory operations.
 */
typedef enum
{
  /**
   * No dependence is available.
   */
  AMD_DBGAPI_SEGMENT_ADDRESS_DEPENDENCE_NONE = 0,
  /**
   * Reading or writing the segment address depends on the lane.
   */
  AMD_DBGAPI_SEGMENT_ADDRESS_DEPENDENCE_LANE = 1,
  /**
   * Reading or writing the segment address depends on the wavefront.
   */
  AMD_DBGAPI_SEGMENT_ADDRESS_DEPENDENCE_WAVE = 2,
  /**
   * Reading or writing the segment address depends on the workgroup.
   */
  AMD_DBGAPI_SEGMENT_ADDRESS_DEPENDENCE_WORKGROUP = 3,
  /**
   * Reading or writing the segment address depends on the agent.
   */
  AMD_DBGAPI_SEGMENT_ADDRESS_DEPENDENCE_AGENT = 4,
  /**
   * Reading or writing the segment address depends on the process.
   */
  AMD_DBGAPI_SEGMENT_ADDRESS_DEPENDENCE_PROCESS = 5
} amd_dbgapi_segment_address_dependency_t;

/**
 * Determine the dependency of a segment address value in a particular address
 * space.
 *
 * This indicates which arguments ::amd_dbgapi_read_memory and
 * ::amd_dbgapi_write_memory require when reading and writing memory when given
 * a specific segment address in an address space.
 *
 * \param[in] address_space_id The address space of the \p segment_address.
 *
 * \param[in] segment_address The integral value of the segment address.  Only
 * the bits corresponding to the address size for the \p address_space_id
 * requested are used.  The address size is provided by the
 * ::AMD_DBGAPI_ADDRESS_SPACE_INFO_ADDRESS_SIZE query.
 *
 * \param[out] segment_address_dependency The dependency of the \p
 * segment_address value in \p address_space_id.  Will be a value of
 * ::amd_dbgapi_segment_address_dependency_t other than
 * ::AMD_DBGAPI_SEGMENT_ADDRESS_DEPENDENCE_NONE.
 *
 * \retval ::AMD_DBGAPI_STATUS_SUCCESS The function has been executed
 * successfully and the result is stored in \p segment_address_dependencies.
 *
 * \retval ::AMD_DBGAPI_STATUS_FATAL A fatal error occurred.  The library is
 * left uninitialized and \p segment_address_dependencies is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_NOT_INITIALIZED The library is not
 * initialized.  The library is left uninitialized and \p
 * destination_segment_address and \p segment_address_dependencies is
 * unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ADDRESS_SPACE_ID \p
 * address_space_id is invalid.  \p segment_address_dependencies is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ARGUMENT \p
 * segment_address_dependencies is NULL. \p segment_address_dependencies is
 * unaltered.
 */
amd_dbgapi_status_t AMD_DBGAPI amd_dbgapi_address_dependency (
  amd_dbgapi_address_space_id_t address_space_id,
  amd_dbgapi_segment_address_t segment_address,
  amd_dbgapi_segment_address_dependency_t *segment_address_dependency)
  AMD_DBGAPI_VERSION_0_64;

/**
 * Indication of whether a segment address in an address space is a member of
 * an source language address class.
 */
typedef enum
{
  /**
   * The segment address in the address space is not a member of the source
   * language address class.
   */
  AMD_DBGAPI_ADDRESS_CLASS_STATE_NOT_MEMBER = 0,
  /**
   * The segment address in the address space is a member of the source
   * language address class.
   */
  AMD_DBGAPI_ADDRESS_CLASS_STATE_MEMBER = 1
} amd_dbgapi_address_class_state_t;

/**
 * Determine if a segment address in an address space is a member of a source
 * language address class.
 *
 * The address space and source language address class must both belong to the
 * same architecture.
 *
 * The address space, source language address class, and wave must all belong
 * to the same architecture.
 *
 * \param[in] wave_id The wave that is using the address.  If the \p
 * address_space is ::AMD_DBGAPI_ADDRESS_SPACE_GLOBAL then \p wave_id may be
 * ::AMD_DBGAPI_WAVE_NONE, as the address space does not depend on the active
 * wave, in which case \p process_id is used.
 *
 * \param[in] lane_id The lane of the \p wave_id that is using the address.  If
 * the \p address_space does not depend on the active lane then this may be
 * ::AMD_DBGAPI_LANE_NONE.  For example, the ::AMD_DBGAPI_ADDRESS_SPACE_GLOBAL
 * address space does not depend on the lane.
 *
 * \param[in] address_space_id The address space of the \p segment_address.  If
 * the address space is dependent on: the active lane then the \p lane_id with
 * in the \p wave_id is used; the active workgroup then the workgroup of \p
 * wave_id is used; or the active wave then the \p wave_id is used.
 *
 * \param[in] segment_address The integral value of the segment address.  Only
 * the bits corresponding to the address size for the \p address_space
 * requested are used.  The address size is provided by the
 * ::AMD_DBGAPI_ADDRESS_SPACE_INFO_ADDRESS_SIZE query.
 *
 * \param[in] address_class_id The handle of the source language address class.
 *
 * \param[out] address_class_state ::AMD_DBGAPI_ADDRESS_CLASS_STATE_NOT_MEMBER
 * if the address is not in the address class.
 * ::AMD_DBGAPI_ADDRESS_CLASS_STATE_MEMBER if the address is in the address
 * class.
 *
 * \retval ::AMD_DBGAPI_STATUS_SUCCESS The function has been executed
 * successfully and the result is stored in \p address_class_state.
 *
 * \retval ::AMD_DBGAPI_STATUS_FATAL A fatal error occurred.  The library is
 * left uninitialized and \p address_class_state is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_NOT_INITIALIZED The library is not
 * initialized.  The library is left uninitialized and \p address_class_state
 * is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_WAVE_ID \p wave_id is invalid, or
 * \p wave_id is ::AMD_DBGAPI_WAVE_NONE and \p address_space is not
 * ::AMD_DBGAPI_ADDRESS_SPACE_GLOBAL.  \p address_class_state is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_LANE_ID \p wave_id is
 * ::AMD_DBGAPI_WAVE_NONE and \p lane_id is not ::AMD_DBGAPI_LANE_NONE.  \p
 * wave_id is not ::AMD_DBGAPI_WAVE_NONE and \p lane_id is not
 * ::AMD_DBGAPI_LANE_NONE and is not valid for \p wave_id.  \p lane_id is
 * ::AMD_DBGAPI_LANE_NONE and \p address_space_id depends on the active lane.
 * \p address_class_state is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ADDRESS_SPACE_ID \p
 * address_space_id is invalid.  \p address_class_state is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ADDRESS_CLASS_ID \p
 * address_class_id is invalid.  \p address_class_state is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ARGUMENT \p address_class_state is
 * NULL.  \p address_class_state is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ARGUMENT_COMPATIBILITY The
 * architectures of \p wave_id, \p address_space_id, and \p address_class_id
 * are not the same.  \p address_class_state is unaltered.
 */
amd_dbgapi_status_t AMD_DBGAPI amd_dbgapi_address_is_in_address_class (
    amd_dbgapi_wave_id_t wave_id, amd_dbgapi_lane_id_t lane_id,
    amd_dbgapi_address_space_id_t address_space_id,
    amd_dbgapi_segment_address_t segment_address,
    amd_dbgapi_address_class_id_t address_class_id,
    amd_dbgapi_address_class_state_t *address_class_state)
    AMD_DBGAPI_VERSION_0_54;

/**
 * Read memory.
 *
 * The memory bytes in \p address_space are read for \p lane_id of \p wave_id
 * starting at \p segment_address sequentially into \p value until \p
 * value_size bytes have been read or an invalid memory address is reached.  \p
 * value_size is set to the number of bytes read successfully.
 *
 * If \p wave_id is not ::AMD_DBGAPI_WAVE_NONE then it must be stopped,  must
 * belong to \p process_id, and its architecture must be the same as that of
 * the address space.
 *
 * The library performs all necessary hardware cache management so that the
 * memory values read are coherent with the \p wave_id if not
 * ::AMD_DBGAPI_WAVE_NONE.  In order for the memory values read to be coherent
 * with other waves, the waves must be stopped when invoking this operation.
 * Stopping wave creation, stopping all waves, performing this operation,
 * resuming any stopped waves, and then allowing wave creation can achieve this
 * requirement.  This requirement also applies if memory is read by other
 * operating system supported means.
 *
 * \param[in] process_id The process to read memory from if \p wave_id is
 * ::AMD_DBGAPI_WAVE_NONE the \p address_space is
 * ::AMD_DBGAPI_ADDRESS_SPACE_GLOBAL.
 *
 * \param[in] wave_id The wave that is accessing the memory.  If the \p
 * address_space is ::AMD_DBGAPI_ADDRESS_SPACE_GLOBAL then \p wave_id may be
 * ::AMD_DBGAPI_WAVE_NONE, as the address space does not depend on the active
 * wave, in which case \p process_id is used.
 *
 * \param[in] lane_id The lane of \p wave_id that is accessing the memory.  If
 * the \p address_space does not depend on the active lane then this may be
 * ::AMD_DBGAPI_LANE_NONE.  For example, the ::AMD_DBGAPI_ADDRESS_SPACE_GLOBAL
 * address space does not depend on the lane.
 *
 * \param[in] address_space_id The address space of the \p segment_address.  If
 * the address space is dependent on: the active lane then the \p lane_id with
 * in the \p wave_id is used; the active workgroup then the workgroup of \p
 * wave_id is used; or the active wave then the \p wave_id is used.
 *
 * \param[in] segment_address The integral value of the segment address.  Only
 * the bits corresponding to the address size for the \p address_space
 * requested are used.  The address size is provided by the
 * ::AMD_DBGAPI_ADDRESS_SPACE_INFO_ADDRESS_SIZE query.
 *
 * \param[in,out] value_size Pass in the number of bytes to read from memory.
 * Return the number of bytes successfully read from memory.
 *
 * \param[out] value Pointer to memory where the result is stored.  Must be an
 * array of at least input \p value_size bytes.
 *
 * \retval ::AMD_DBGAPI_STATUS_SUCCESS Either the input \p value_size was 0, or
 * the input \p value_size was greater than 0 and one or more bytes have been
 * read successfully.  The output \p value_size is set to the number of bytes
 * successfully read, which will be 0 if the input \p value_size was 0.  The
 * first output \p value_size bytes of \p value are set to the bytes
 * successfully read, all other bytes in \p value are unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_FATAL A fatal error occurred.  The library is
 * left uninitialized; and \p value_size and \p value are unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_NOT_INITIALIZED The library is not
 * initialized.  The library is left uninitialized; and \p value_size and \p
 * value are unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_PROCESS_ID \p process_id is
 * invalid.  \p value_size and \p value are unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_WAVE_ID \p wave_id is invalid, or
 * \p wave_id is ::AMD_DBGAPI_WAVE_NONE and \p address_space is not
 * ::AMD_DBGAPI_ADDRESS_SPACE_GLOBAL.  \p value_size and \p value are
 * unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_LANE_ID \p wave_id is
 * ::AMD_DBGAPI_WAVE_NONE and \p lane_id is not ::AMD_DBGAPI_LANE_NONE.  \p
 * wave_id is not ::AMD_DBGAPI_WAVE_NONE and \p lane_id is not
 * ::AMD_DBGAPI_LANE_NONE and is not valid for \p wave_id.  \p lane_id is
 * ::AMD_DBGAPI_LANE_NONE and \p address_space_id depends on the active lane.
 * \p value_size and \p value are unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ADDRESS_SPACE_ID \p
 * address_space_id is invalid.  \p value is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_WAVE_NOT_STOPPED \p wave_id is not
 * stopped.  \p value_size and \p value are unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ARGUMENT \p value or \p value_size
 * are NULL.  \p value_size and \p value are unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ARGUMENT_COMPATIBILITY \p wave_id
 * in not ::AMD_DBGAPI_WAVE_NONE and does not belong to \p process_id or have
 * the same the architecture as \p address_space_id.  \p value_size and \p
 * value are unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_MEMORY_ACCESS The input \p value_size was
 * greater than 0 and no bytes were successfully read.  The output \p
 * value_size is set to 0.  \p value is unaltered.
 */
amd_dbgapi_status_t AMD_DBGAPI amd_dbgapi_read_memory (
    amd_dbgapi_process_id_t process_id, amd_dbgapi_wave_id_t wave_id,
    amd_dbgapi_lane_id_t lane_id,
    amd_dbgapi_address_space_id_t address_space_id,
    amd_dbgapi_segment_address_t segment_address,
    amd_dbgapi_size_t *value_size, void *value) AMD_DBGAPI_VERSION_0_54;

/**
 * Write memory.
 *
 * The memory bytes in \p address_space are written for \p lane_id of \p
 * wave_id starting at \p segment_address sequentially from \p value until \p
 * value_size bytes have been written or an invalid memory address is reached.
 * \p value_size is set to the number of bytes written successfully.
 *
 * If \p wave_id is not ::AMD_DBGAPI_WAVE_NONE then it must be stopped,  must
 * belong to \p process_id, and its architecture must be the same as that of
 * the address space.
 *
 * The library performs all necessary hardware cache management so that the
 * memory values written are coherent with the \p wave_id if not
 * ::AMD_DBGAPI_WAVE_NONE.  In order for the memory values written to be
 * coherent with other waves, the waves must be stopped when invoking this
 * operation.  Stopping wave creation, stopping all waves, performing this
 * operation, resuming any stopped waves, and then allowing wave creation can
 * achieve this requirement.  This requirement also applies if memory is
 * written by other operating system supported means.
 *
 * \param[in] process_id The process to write memory to if \p wave_id is
 * ::AMD_DBGAPI_WAVE_NONE the \p address_space is
 * ::AMD_DBGAPI_ADDRESS_SPACE_GLOBAL.
 *
 * \param[in] wave_id The wave that is accessing the memory.  If the \p
 * address_space is ::AMD_DBGAPI_ADDRESS_SPACE_GLOBAL then \p wave_id may be
 * ::AMD_DBGAPI_WAVE_NONE, as the address space does not depend on the active
 * wave, in which case \p process_id is used.
 *
 * \param[in] lane_id The lane of \p wave_id that is accessing the memory.  If
 * the \p address_space does not depend on the active lane then this may be
 * ::AMD_DBGAPI_LANE_NONE.  For example, the ::AMD_DBGAPI_ADDRESS_SPACE_GLOBAL
 * address space does not depend on the lane.
 *
 * \param[in] address_space_id The address space of the \p segment_address.  If
 * the address space is dependent on: the active lane then the \p lane_id with
 * in the \p wave_id is used; the active workgroup then the workgroup of \p
 * wave_id is used; or the active wave then the \p wave_id is used.
 *
 * \param[in] segment_address The integral value of the segment address.  Only
 * the bits corresponding to the address size for the \p address_space
 * requested are used.  The address size is provided by the
 * ::AMD_DBGAPI_ADDRESS_SPACE_INFO_ADDRESS_SIZE query.
 *
 * \param[in,out] value_size Pass in the number of bytes to write to memory.
 * Return the number of bytes successfully written to memory.
 *
 * \param[in] value The bytes to write to memory.  Must point to an array of at
 * least input \p value_size bytes.
 *
 * \retval ::AMD_DBGAPI_STATUS_SUCCESS Either the input \p value_size was 0, or
 * the input \p value_size was greater than 0 and one or more bytes have been
 * written successfully.  The output \p value_size is set to the number of
 * bytes successfully written, which will be 0 if the input \p value_size was
 * 0.  The first output \p value_size bytes of memory starting at \p
 *     segment_address are updated, all other memory is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_FATAL A fatal error occurred.  The library is
 * left uninitialized; and the memory and \p value_size are unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_NOT_INITIALIZED The library is not
 * initialized.  The library is left uninitialized;  the memory and \p
 * value_size are unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_PROCESS_ID \p process_id is
 * invalid.  The memory and \p value_size are unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_WAVE_ID \p wave_id is invalid, or
 * \p wave_id is ::AMD_DBGAPI_WAVE_NONE and \p address_space is
 * ::AMD_DBGAPI_ADDRESS_SPACE_GLOBAL.  The memory and \p value_size are
 * unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_LANE_ID \p wave_id is
 * ::AMD_DBGAPI_WAVE_NONE and \p lane_id is not ::AMD_DBGAPI_LANE_NONE.  \p
 * wave_id is not ::AMD_DBGAPI_WAVE_NONE and \p lane_id is not
 * ::AMD_DBGAPI_LANE_NONE and is not valid for \p wave_id.  \p lane_id is
 * ::AMD_DBGAPI_LANE_NONE and \p address_space_id depends on the active lane.
 * The memory and \p value_size are unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ADDRESS_SPACE_ID \p
 * address_space_id is invalid.  The memory and \p value_size are unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_WAVE_NOT_STOPPED \p wave_id is not
 * stopped.  The memory and \p value_size are unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ARGUMENT \p value or \p value_size
 * are NULL.  The memory and \p value_size are unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ARGUMENT_COMPATIBILITY \p wave_id
 * in not ::AMD_DBGAPI_WAVE_NONE and does not belong to \p process_id or have
 * the same the architecture as \p address_space_id.  The memory and \p
 * value_size are unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_MEMORY_ACCESS The input \p value_size was
 * greater than 0 and no bytes were successfully written.  The output \p
 * value_size is set to 0.  The memory and \p value_size are unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_PROCESS_FROZEN This operation is not
 * permitted because the process is frozen.  The memory and \p value_size are
 * unaltered.
 */
amd_dbgapi_status_t AMD_DBGAPI amd_dbgapi_write_memory (
    amd_dbgapi_process_id_t process_id, amd_dbgapi_wave_id_t wave_id,
    amd_dbgapi_lane_id_t lane_id,
    amd_dbgapi_address_space_id_t address_space_id,
    amd_dbgapi_segment_address_t segment_address,
    amd_dbgapi_size_t *value_size, const void *value) AMD_DBGAPI_VERSION_0_76;

/**
 * Memory access precision.
 *
 * The AMD GPU can overlap the execution of memory instructions with other
 * instructions.  This can result in a wave stopping due to a memory violation
 * or hardware data watchpoint hit with a program counter beyond the
 * instruction that caused the wave to stop.
 *
 * Some architectures allow the hardware to be configured to always wait for
 * memory operations to complete before continuing.  This will result in the
 * wave stopping at the instruction immediately after the one that caused the
 * stop event.  Enabling this mode can make execution of waves significantly
 * slower.
 *
 * The ::AMD_DBGAPI_PROCESS_INFO_PRECISE_MEMORY_SUPPORTED query can be used to
 * determine if the architectures of all the agents of a process support
 * controlling precise memory accesses.
 */
typedef enum
{
  /**
   * Memory instructions execute normally and a wave does not wait for the
   * memory access to complete.
   */
  AMD_DBGAPI_MEMORY_PRECISION_NONE = 0,
  /**
   * A wave waits for memory instructions to complete before executing further
   * instructions.  This can cause a wave to execute significantly slower.
   */
  AMD_DBGAPI_MEMORY_PRECISION_PRECISE = 1
} amd_dbgapi_memory_precision_t;

/**
 * Control precision of memory access reporting.
 *
 * A process can be set to ::AMD_DBGAPI_MEMORY_PRECISION_NONE to disable
 * precise memory reporting.  Use the
 * ::AMD_DBGAPI_PROCESS_INFO_PRECISE_MEMORY_SUPPORTED query to determine if the
 * architectures of all the agents of a process support another memory
 * precision.
 *
 * The memory precision is set independently for each process, and only affects
 * the waves executing on the agents of that process.  The setting may be
 * changed at any time, including when waves are executing, and takes effect
 * immediately.
 *
 * \param[in] process_id The process being configured.
 *
 * \param[in] memory_precision The memory precision to set.
 *
 * \retval ::AMD_DBGAPI_STATUS_SUCCESS The function has been executed
 * successfully and the agents of the process have been configured.
 *
 * \retval ::AMD_DBGAPI_STATUS_FATAL A fatal error occurred.  The library is
 * left uninitialized.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_NOT_INITIALIZED The library is not
 * initialized.  The library is left uninitialized and no configuration is
 * changed.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_PROCESS_ID \p process_id is
 * invalid.  No configuration is changed.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ARGUMENT \p memory_precision is
 * an invalid value.  No configuration is changed.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_NOT_SUPPORTED The requested \p
 * memory_precision is not supported by the architecture of all the agents of
 * \p process_id.  No configuration is changed.
 */
amd_dbgapi_status_t AMD_DBGAPI amd_dbgapi_set_memory_precision (
    amd_dbgapi_process_id_t process_id,
    amd_dbgapi_memory_precision_t memory_precision) AMD_DBGAPI_VERSION_0_54;

/**
 * ALU exceptions reporting precision.
 *
 * Waves may issue multiple instructions and advance the program counter
 * before a previous ALU instruction has executed and reported exceptions.
 * This can result in a wave stopping due to an ALU exception beyond the
 * instruction that caused the wave to stop.
 *
 * Some architectures allow hardware to be configured to always wait for ALU
 * instructions to complete before issuing to the next instruction.  If an
 * exception is raised by the instruction, the wave will stop at the
 * instruction immediately following it.  Enabling this mode can make
 * execution of waves slower.
 *
 * The ::AMD_DBGAPI_PROCESS_INFO_PRECISE_ALU_EXCEPTIONS_SUPPORTED query can be
 * used to determine if the architectures of all the agents of a process support
 * controlling precise ALU exceptions reporting.
 */
typedef enum
{
  /**
   * ALU exceptions delivery might be reported at any time after the
   * instructions causing them have executed.
   */
  AMD_DBGAPI_ALU_EXCEPTIONS_PRECISION_NONE = 0,
  /**
   * When an ALU exception is delivered to the waves, the wave's PC is at the
   * instruction following the one causing the exception to be raised.  This
   * can cause waves to execute slower.
   */
  AMD_DBGAPI_ALU_EXCEPTIONS_PRECISION_PRECISE = 1
} amd_dbgapi_alu_exceptions_precision_t;

/**
 * Control precision of ALU exceptions reporting.
 *
 * A process can set ::AMD_DBGAPI_ALU_EXCEPTIONS_PRECISION_NONE to disable
 * precise ALU exception reporting.  Use the
 * ::AMD_DBGAPI_PROCESS_INFO_PRECISE_ALU_EXCEPTIONS_SUPPORTED query to determine
 * if the architecture of all agents of the process support another ALU
 * precision mode.
 *
 * The ALU exceptions precision is set independently for each process, and only
 * affects the waves executing on the agents of that process.  The setting may
 * be changed at any time, including when waves are executing, and takes effect
 * immediately.
 *
 * \param[in] process_id The process being configured.
 *
 * \param[in] alu_exceptions_precision The ALU exception precision mode to set.
 *
 * \retval ::AMD_DBGAPI_STATUS_SUCCESS The function has been executed
 * successfully and the agents of the process have been configured.
 *
 * \retval ::AMD_DBGAPI_STATUS_FATAL A fatal error occurred.  The library is
 * left uninitialized.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_NOT_INITIALIZED The library is not
 * initialized.  The library is left uninitialized and no configuration is
 * changed.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_PROCESS_ID \p process_id is
 * invalid.  No configuration is changed.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ARGUMENT
 * \p alu_exceptions_precision is an invalid value.  No configuration is
 * changed.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_NOT_SUPPORTED The requested \p
 * alu_exceptions_precision is not supported by the architecture of all the
 * agents of \p process_id.  No configuration is changed.
 */
amd_dbgapi_status_t AMD_DBGAPI amd_dbgapi_set_alu_exceptions_precision (
    amd_dbgapi_process_id_t process_id,
    amd_dbgapi_alu_exceptions_precision_t alu_exceptions_precision)
     AMD_DBGAPI_VERSION_0_77;

/** @} */

/** \defgroup event_group Events
 *
 * Asynchronous event management.
 *
 * Events can occur asynchronously.  The library maintains a list of pending
 * events that have happened but not yet been reported to the client.  Events
 * are maintained independently for each process.
 *
 * When ::amd_dbgapi_process_attach successfully attaches to a process a
 * ::amd_dbgapi_notifier_t notifier is created that is available using the
 * ::AMD_DBGAPI_PROCESS_INFO_NOTIFIER query.  When this indicates there may be
 * pending events for the process, ::amd_dbgapi_process_next_pending_event can
 * be used to retrieve the pending events.
 *
 * The notifier must be reset before retrieving pending events so that the
 * notifier will always conservatively indicate there may be pending events.
 * After the client has processed an event it must report completion using
 * ::amd_dbgapi_event_processed.
 *
 * \sa ::amd_dbgapi_notifier_t
 *
 * @{
 */

/**
 * Opaque event handle.
 *
 * Globally unique for a single library instance.
 */
typedef struct
{
  uint64_t handle;
} amd_dbgapi_event_id_t;

/**
 * The NULL event handle.
 */
#define AMD_DBGAPI_EVENT_NONE                                                 \
    AMD_DBGAPI_HANDLE_LITERAL (amd_dbgapi_event_id_t, 0)

/**
 * The event kinds.
 */
typedef enum
{
  /**
   * No event.
   */
  AMD_DBGAPI_EVENT_KIND_NONE = 0,
  /**
   * A wave has stopped.
   */
  AMD_DBGAPI_EVENT_KIND_WAVE_STOP = 1,
  /**
   * A command for a wave was not able to complete because the wave has
   * terminated.
   *
   * Commands that can result in this event are ::amd_dbgapi_wave_stop and
   * ::amd_dbgapi_wave_resume in single step mode.  Since the wave terminated
   * before stopping, this event will be reported instead of
   * ::AMD_DBGAPI_EVENT_KIND_WAVE_STOP.
   *
   * The wave that terminated is available by the ::AMD_DBGAPI_EVENT_INFO_WAVE
   * query.  However, the wave will be invalid since it has already terminated.
   * It is the client's responsibility to know what command was being performed
   * and was unable to complete due to the wave terminating.
   */
  AMD_DBGAPI_EVENT_KIND_WAVE_COMMAND_TERMINATED = 2,
  /**
   * The list of code objects has changed.
   *
   * This event is only reported when a thread is in the process of loading or
   * unloading a code object.  It is not reported when attaching to a process
   * even if there are loaded code objects.  It is the client's responsibility
   * to fetch the current code object list using
   * ::amd_dbgapi_process_code_object_list.
   *
   * The thread that caused the code object list to change will be stopped
   * until the event is reported as processed.  Before reporting the event has
   * been processed, the client must set any pending breakpoints for newly
   * loaded code objects so that breakpoints will be set before any code in the
   * code object is executed.
   *
   * When the event is reported as complete, a
   * ::AMD_DBGAPI_EVENT_KIND_BREAKPOINT_RESUME event may be created which must
   * be processed to resume the thread that caused the code object list to
   * change.  Leaving the thread stopped may prevent the inferior's runtime
   * from servicing requests from other threads.
   */
  AMD_DBGAPI_EVENT_KIND_CODE_OBJECT_LIST_UPDATED = 3,
  /**
   * Request to resume a host breakpoint.
   *
   * If ::amd_dbgapi_report_breakpoint_hit returns with \p resume as
   * false then it indicates that events must be processed before the thread
   * hitting the breakpoint can be resumed.  When the necessary event(s) are
   * reported as processed, this event will be added to the pending events.
   * The breakpoint and client thread can then be queried by
   * ::amd_dbgapi_event_get_info using ::AMD_DBGAPI_EVENT_INFO_BREAKPOINT
   * and ::AMD_DBGAPI_EVENT_INFO_CLIENT_THREAD respectively.  The client must
   * then resume execution of the thread.
   */
  AMD_DBGAPI_EVENT_KIND_BREAKPOINT_RESUME = 4,
  /**
   * The runtime support in the inferior is enabled or disabled.
   *
   * The client can use this event to determine when to activate and deactivate
   * AMD GPU debugging functionality.  The status of the inferior's runtime
   * support can be queried by ::amd_dbgapi_event_get_info using
   * ::AMD_DBGAPI_EVENT_INFO_RUNTIME_STATE.  If not enabled
   * (::AMD_DBGAPI_RUNTIME_STATE_UNLOADED), or enabled but not compatible
   * (::AMD_DBGAPI_RUNTIME_STATE_LOADED_ERROR_RESTRICTION), then no code
   * objects, queues, or waves will be reported to exist, and the only event
   * that will be reported is ::AMD_DBGAPI_EVENT_KIND_RUNTIME.  If enabled
   * successfully (::AMD_DBGAPI_RUNTIME_STATE_LOADED_SUCCESS) full debugging is
   * supported by the library.
   */
  AMD_DBGAPI_EVENT_KIND_RUNTIME = 5,
  /**
   * The inferior's runtime has put a queue into the queue error state due to
   * exceptions being reported for the queue.
   *
   * No further waves will be started on the queue.  All waves that belong to
   * the queue are inhibited from executing further instructions regardless of
   * whether they are in the halt state.  See ::AMD_DBGAPI_QUEUE_STATE_ERROR.
   *
   * The ::AMD_DBGAPI_QUEUE_INFO_ERROR_REASON query will include the union of
   * the exceptions that were reported.  Some waves may be stopped before they
   * were able to report a queue error condition.  The wave stop reason will
   * only include the exceptions that were reported.  For example, if many
   * waves encounter a memory violation at the same time, only some of the
   * waves may report it before execution of all the waves in the queue is
   * inhibited.  Only the waves that were able to report the memory violation
   * before all the waves were stopped will include the
   * ::AMD_DBGAPI_WAVE_STOP_REASON_MEMORY_VIOLATION stop reason.
   *
   * Any waves that have a pending single step will report a
   * ::AMD_DBGAPI_EVENT_KIND_WAVE_COMMAND_TERMINATED event to indicate that the
   * single step has been cancelled.  Waves in such queues are inhibited from
   * executing any further instructions.  The application can delete the queue,
   * which will result in all the waves to also be deleted, and then create a
   * new queue.
   *
   * The inferior's runtime will not notify the application of the queue error
   * until this event is reported as complete by calling
   * ::amd_dbgapi_event_processed.  Once the application is notified, it may
   * abort, or it may delete and re-create the queue in order to continue
   * submitting dispatches to the AMD GPU.  If the application deletes a queue
   * then all information about the waves executing on the queue will be lost,
   * preventing the client from determining if a wave caused the error.
   */
  AMD_DBGAPI_EVENT_KIND_QUEUE_ERROR = 6
} amd_dbgapi_event_kind_t;

/**
 * Obtain the next pending event.
 *
 * The order events are returned is unspecified.  If the client requires
 * fairness then it can retrieve all pending events and randomize the order of
 * processing.
 *
 * \param[in] process_id If ::AMD_DBGAPI_PROCESS_NONE then retrieve a pending
 * event from any processes.  Otherwise, retrieve a pending event from process
 * \p process_id.
 *
 * \param[out] event_id The event handle of the next pending event.  Each event
 * is only returned once.  If there are no pending events the
 * ::AMD_DBGAPI_EVENT_NONE handle is returned.
 *
 * \param[out] kind The kind of the returned event.  If there are no pending
 * events, then ::AMD_DBGAPI_EVENT_KIND_NONE is returned.
 *
 * \retval ::AMD_DBGAPI_STATUS_SUCCESS The function has been executed
 * successfully and an event or the NULL event has been returned.
 *
 * \retval ::AMD_DBGAPI_STATUS_FATAL A fatal error occurred.  The library is
 * left uninitialized; and \p event_id and \p kind are unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_NOT_INITIALIZED The library is not
 * initialized.  The library is left uninitialized; and \p event_id and \p kind
 * are unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_PROCESS_ID The \p process_id is
 * invalid.  No event is retrieved and \p event_id and \p kind are
 * unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ARGUMENT \p event_id or \p kind
 * are NULL.  No event is retrieved and \p event_id and \p kind are unaltered.
 */
amd_dbgapi_status_t AMD_DBGAPI amd_dbgapi_process_next_pending_event (
    amd_dbgapi_process_id_t process_id, amd_dbgapi_event_id_t *event_id,
    amd_dbgapi_event_kind_t *kind) AMD_DBGAPI_VERSION_0_54;

/**
 * Inferior's runtime state.
 */
typedef enum
{
  /**
   * The inferior's runtime has been loaded and debugging is supported by the
   * library.
   */
  AMD_DBGAPI_RUNTIME_STATE_LOADED_SUCCESS = 1,
  /**
   * The inferior's runtime has been unloaded.
   */
  AMD_DBGAPI_RUNTIME_STATE_UNLOADED = 2,
  /**
   * The inferior's runtime has been loaded but there is a restriction error
   * that prevents debugging the process.  See
   * ::AMD_DBGAPI_STATUS_ERROR_RESTRICTION for possible reasons.
   */
  AMD_DBGAPI_RUNTIME_STATE_LOADED_ERROR_RESTRICTION = 3
} amd_dbgapi_runtime_state_t;

/**
 * Event queries that are supported by ::amd_dbgapi_event_get_info.
 *
 * Each query specifies the type of data returned in the \p value argument to
 * ::amd_dbgapi_event_get_info.
 */
typedef enum
{
  /**
   * Return the process to which this event belongs.  The type of this
   * attribute is ::amd_dbgapi_process_id_t.
   */
  AMD_DBGAPI_EVENT_INFO_PROCESS = 1,
  /**
   * Return the event kind.  The type of this attribute is
   * ::amd_dbgapi_event_kind_t.
   */
  AMD_DBGAPI_EVENT_INFO_KIND = 2,
  /**
   * Return the wave of a ::AMD_DBGAPI_EVENT_KIND_WAVE_STOP or
   * ::AMD_DBGAPI_EVENT_KIND_WAVE_COMMAND_TERMINATED event.  The type of this
   * attribute is a ::amd_dbgapi_wave_id_t.
   */
  AMD_DBGAPI_EVENT_INFO_WAVE = 3,
  /**
   * Return the breakpoint of a ::AMD_DBGAPI_EVENT_KIND_BREAKPOINT_RESUME
   * event.  The type of this attribute is a ::amd_dbgapi_breakpoint_id_t.
   */
  AMD_DBGAPI_EVENT_INFO_BREAKPOINT = 4,
  /**
   * Return the client thread of a ::AMD_DBGAPI_EVENT_KIND_BREAKPOINT_RESUME
   * event.  The type of this attribute is a ::amd_dbgapi_client_thread_id_t.
   */
  AMD_DBGAPI_EVENT_INFO_CLIENT_THREAD = 5,
  /**
   * Return if the runtime loaded in the inferior is supported by the library
   * for a ::AMD_DBGAPI_EVENT_KIND_RUNTIME event.  The type of this
   * attribute is \p uint32_t with a value defined by
   * ::amd_dbgapi_runtime_state_t.
   */
  AMD_DBGAPI_EVENT_INFO_RUNTIME_STATE = 6,
  /**
   * Return the queue of a ::AMD_DBGAPI_EVENT_KIND_QUEUE_ERROR event.  The type
   * of this attribute is a ::amd_dbgapi_queue_id_t.
   */
  AMD_DBGAPI_EVENT_INFO_QUEUE = 7
} amd_dbgapi_event_info_t;

/**
 * Query information about an event.
 *
 * ::amd_dbgapi_event_info_t specifies the queries supported and the
 * type returned using the \p value argument.
 *
 * \param[in] event_id The event being queried.
 *
 * \param[in] query The query being requested.
 *
 * \param[in] value_size Size of the memory pointed to by \p value.  Must be
 * equal to the byte size of the query result.
 *
 * \param[out] value Pointer to memory where the query result is stored.
 *
 * \retval ::AMD_DBGAPI_STATUS_SUCCESS The function has been executed
 * successfully and the result is stored in \p value.
 *
 * \retval ::AMD_DBGAPI_STATUS_FATAL A fatal error occurred.  The library is
 * left uninitialized and \p value is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_NOT_INITIALIZED The library is not
 * initialized.  The library is left uninitialized and \p value is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_EVENT_ID \p event_id is invalid
 * or the NULL event.  \p value is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ARGUMENT \p value is NULL or
 * \p query is for an attribute not present for the kind of the event.
 * \p value is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ARGUMENT_COMPATIBILITY \p
 * value_size does not match the size of the \p query result.  \p value is
 * unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_CLIENT_CALLBACK This will be
 * reported if the amd_dbgapi_callbacks_s::allocate_memory callback used to
 * allocate \p value returns NULL.  \p value is unaltered.
 */
amd_dbgapi_status_t AMD_DBGAPI amd_dbgapi_event_get_info (
    amd_dbgapi_event_id_t event_id, amd_dbgapi_event_info_t query,
    size_t value_size, void *value) AMD_DBGAPI_VERSION_0_54;

/**
 * Report that an event has been processed.
 *
 * Every event returned by ::amd_dbgapi_process_next_pending_event must be
 * reported as processed exactly once.  Events do not have to be reported
 * completed in the same order they are retrieved.
 *
 * \param[in] event_id The event that has been processed.
 *
 * \retval ::AMD_DBGAPI_STATUS_SUCCESS The function has been executed
 * successfully and the event has been reported as processed.  The \p event_id
 * is invalidated.
 *
 * \retval ::AMD_DBGAPI_STATUS_FATAL A fatal error occurred.  The library is
 * left uninitialized.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_NOT_INITIALIZED The library is not
 * initialized.  The library is left uninitialized.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_EVENT_ID The \p event_id is
 * invalid or the NULL event.  No event is marked as processed.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ARGUMENT \p event_id or \p kind
 * are NULL.  No event is marked as processed.
 */
amd_dbgapi_status_t AMD_DBGAPI amd_dbgapi_event_processed (
    amd_dbgapi_event_id_t event_id) AMD_DBGAPI_VERSION_0_54;

/** @} */

/** \defgroup logging_group Logging
 *
 * Control logging.
 *
 * When the library is initially loaded the logging level is set to
 * ::AMD_DBGAPI_LOG_LEVEL_NONE.  The log level is not changed by
 * ::amd_dbgapi_initialize or ::amd_dbgapi_finalize.
 *
 * The log messages are delivered to the client using the
 * amd_dbgapi_callbacks_s::log_message call back.
 *
 * Note that logging can be helpful for debugging.
 *
 * @{
 */

/**
 * The logging levels supported.
 */
typedef enum
{
  /**
   * Print no messages.
   */
  AMD_DBGAPI_LOG_LEVEL_NONE = 0,
  /**
   * Print fatal error messages.  Any library function that returns the
   * ::AMD_DBGAPI_STATUS_FATAL status code also logs a message with this level.
   */
  AMD_DBGAPI_LOG_LEVEL_FATAL_ERROR = 1,
  /**
   * Print fatal error and warning messages.
   */
  AMD_DBGAPI_LOG_LEVEL_WARNING = 2,
  /**
   * Print fatal error, warning, and info messages.
   */
  AMD_DBGAPI_LOG_LEVEL_INFO = 3,
  /**
   * Print fatal error, warning, info, and API tracing messages.
   */
  AMD_DBGAPI_LOG_LEVEL_TRACE = 4,
  /**
   * Print fatal error, warning, info, API tracing, and verbose messages.
   */
  AMD_DBGAPI_LOG_LEVEL_VERBOSE = 5
} amd_dbgapi_log_level_t;

/**
 * Set the logging level.
 *
 * Internal logging messages less than the set logging level will not be
 * reported.  If ::AMD_DBGAPI_LOG_LEVEL_NONE then no messages will be reported.
 *
 * This function can be used even when the library is uninitialized.  However,
 * no messages will be reported until the library is initialized when the
 * callbacks are provided.
 *
 * \param[in] level The logging level to set.
 *
 * \retval ::AMD_DBGAPI_STATUS_SUCCESS The function has been executed
 * successfully.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ARGUMENT \p level is invalid.  The
 * logging level is ot changed.
 */
void AMD_DBGAPI amd_dbgapi_set_log_level (amd_dbgapi_log_level_t level)
    AMD_DBGAPI_VERSION_0_54;

/** @} */

/** \defgroup callbacks_group Callbacks
 *
 * The library requires the client to provide a number of services.  These
 * services are specified by providing callbacks when initializing
 * the library using ::amd_dbgapi_initialize.
 *
 * The callbacks defined in this section are invoked by the library and must
 * not themselves invoke any function provided by the library before returning.
 *
 * @{
 */

/**
 * Opaque breakpoint handle.
 *
 * The implementation of the library requests the client to insert breakpoints
 * in certain functions so that it can be notified when certain actions are
 * being performed, and to stop the thread performing the action.  This allows
 * the data to be retrieved and updated without conflicting with the thread.
 * The library will resume the thread when it has completed the access.
 *
 * Globally unique for a single library instance.
 */
typedef struct
{
  uint64_t handle;
} amd_dbgapi_breakpoint_id_t;

/**
 * The NULL breakpoint handle.
 */
#define AMD_DBGAPI_BREAKPOINT_NONE                                            \
    AMD_DBGAPI_HANDLE_LITERAL (amd_dbgapi_breakpoint_id_t, 0)

/**
 * Breakpoint queries that are supported by ::amd_dbgapi_breakpoint_get_info.
 *
 * Each query specifies the type of data returned in the \p value argument to
 * ::amd_dbgapi_breakpoint_get_info.
 */
typedef enum
{
  /**
   * Return the process to which this breakpoint belongs.  The type of this
   * attribute is ::amd_dbgapi_process_id_t.
   */
  AMD_DBGAPI_BREAKPOINT_INFO_PROCESS = 1
} amd_dbgapi_breakpoint_info_t;

/**
 * Query information about a breakpoint.
 *
 * ::amd_dbgapi_breakpoint_info_t specifies the queries supported and the
 * type returned using the \p value argument.
 *
 * \param[in] breakpoint_id The handle of the breakpoint being queried.
 *
 * \param[in] query The query being requested.
 *
 * \param[in] value_size Size of the memory pointed to by \p value.  Must be
 * equal to the byte size of the query result.
 *
 * \param[out] value Pointer to memory where the query result is stored.
 *
 * \retval ::AMD_DBGAPI_STATUS_SUCCESS The function has been executed
 * successfully and the result is stored in \p value.
 *
 * \retval ::AMD_DBGAPI_STATUS_FATAL A fatal error occurred.  The library is
 * left uninitialized and \p value is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_NOT_INITIALIZED The library is not
 * initialized.  The library is left uninitialized and \p value is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_BREAKPOINT_ID \p breakpoint_id
 * is invalid.  \p value is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ARGUMENT \p value is NULL or
 * \p query is invalid.  \p value is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ARGUMENT_COMPATIBILITY \p
 * value_size does not match the size of the \p query result.  \p value is
 * unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_CLIENT_CALLBACK This will be
 * reported if the amd_dbgapi_callbacks_s::allocate_memory callback used to
 * allocate \p value returns NULL.  \p value is unaltered.
 */
amd_dbgapi_status_t AMD_DBGAPI amd_dbgapi_breakpoint_get_info (
    amd_dbgapi_breakpoint_id_t breakpoint_id,
    amd_dbgapi_breakpoint_info_t query, size_t value_size,
    void *value) AMD_DBGAPI_VERSION_0_54;

/**
 * The action to perform after reporting a breakpoint has been hit.
 */
typedef enum
{
  /**
   * Resume execution.
   */
  AMD_DBGAPI_BREAKPOINT_ACTION_RESUME = 1,
  /**
   * Leave execution halted.
   */
  AMD_DBGAPI_BREAKPOINT_ACTION_HALT = 2
} amd_dbgapi_breakpoint_action_t;

/**
 * Opaque client thread handle.
 *
 * A pointer to client data associated with a thread.  This pointer is
 * passed in to the ::amd_dbgapi_report_breakpoint_hit so it can be
 * passed out by the ::AMD_DBGAPI_EVENT_KIND_BREAKPOINT_RESUME event to
 * allow the client of the library to identify the thread that must be
 * resumed.
 */
typedef struct amd_dbgapi_client_thread_s *amd_dbgapi_client_thread_id_t;

/**
 * Report that a breakpoint inserted by the
 * amd_dbgapi_callbacks_s::insert_breakpoint callback has been hit.
 *
 * The thread that hit the breakpoint must remain halted while this
 * function executes, at which point it must be resumed if
 * \p breakpoint_action is ::AMD_DBGAPI_BREAKPOINT_ACTION_RESUME.  If
 * \p breakpoint_action is :AMD_DBGAPI_BREAKPOINT_ACTION_HALT then the client
 * should process pending events which will cause a
 * ::AMD_DBGAPI_EVENT_KIND_BREAKPOINT_RESUME event to be added which specifies
 * that the thread should now be resumed.
 *
 * \param[in] breakpoint_id The breakpoint that has been hit.
 *
 * \param[in] client_thread_id The client identification of the thread that
 * hit the breakpoint.
 *
 * \param[out] breakpoint_action Indicate if the thread hitting the breakpoint
 * should be resumed or remain halted when this function returns.
 *
 * \retval ::AMD_DBGAPI_STATUS_SUCCESS The function has been executed
 * successfully and \p breakpoint_action indicates if the thread hitting the
 * breakpoint should be resumed.
 *
 * \retval ::AMD_DBGAPI_STATUS_FATAL A fatal error occurred.  The library is
 * left uninitialized and \p breakpoint_action is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_NOT_INITIALIZED The library is not
 * initialized.  The library is left uninitialized and \p breakpoint_action is
 * unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_BREAKPOINT_ID The
 * \p breakpoint_id is invalid.  \p breakpoint_action is unaltered.
 *
 * \retval ::AMD_DBGAPI_STATUS_ERROR_INVALID_ARGUMENT \p breakpoint_action is
 * NULL.  \p breakpoint_action is unaltered.
 */
amd_dbgapi_status_t AMD_DBGAPI amd_dbgapi_report_breakpoint_hit (
    amd_dbgapi_breakpoint_id_t breakpoint_id,
    amd_dbgapi_client_thread_id_t client_thread_id,
    amd_dbgapi_breakpoint_action_t *breakpoint_action) AMD_DBGAPI_VERSION_0_54;

/**
 * Client queries that are supported by the client_process_get_info callback.
 *
 * Each query specifies the type of data returned in the \p value argument to
 * amd_dbgapi_callbacks_s::client_process_get_info.
 */
typedef enum
{
  /**
   * Return the native operating system process handle.  This value is required
   * to not change during the lifetime of the process associated with the
   * client process handle.
   *
   * For Linux<sup>&reg;</sup> this is the \p pid_t from \p sys/types.h and the
   * corresponding process is required to have already been \p ptrace enabled.
   *
   * The type of this attribute is ::amd_dbgapi_os_process_id_t.
   */
  AMD_DBGAPI_CLIENT_PROCESS_INFO_OS_PID = 1,
  /**
   * If the current process is created from a core file, return the content of
   * the AMDGPU state note if present.  If the process image is not created
   * from a core dump or if such state note is not present in the core dump,
   * the client_process_get_info callback returns
   * ::AMD_DBGAPI_STATUS_ERROR_NOT_AVAILABLE.
   *
   * The type of this attribute is ::amd_dbgapi_core_state_data_t.
   */
  AMD_DBGAPI_CLIENT_PROCESS_INFO_CORE_STATE = 2,
} amd_dbgapi_client_process_info_t;

/**
 * Callbacks that the client of the library must provide.
 *
 * The client implementation of the callbacks must not invoke any operation of
 * the library.
 */
struct amd_dbgapi_callbacks_s
{

  /**
   * Allocate memory to be used to return a value from the library that is then
   * owned by the client.
   *
   * The memory should be suitably aligned for any type.  If \p byte_size is 0
   * or if unable to allocate memory of the byte size specified by \p byte_size
   * then return NULL and allocate no memory.  The client is responsible for
   * deallocating this memory, and so is responsible for tracking the size of
   * the allocation.  Note that these requirements can be met by implementing
   * using \p malloc.
   */
  void *(*allocate_memory) (size_t byte_size);

  /**
   * Deallocate memory that was allocated by
   * amd_dbgapi_callbacks_s::allocate_memory.
   *
   * \p data will be a pointer returned by
   * amd_dbgapi_callbacks_s::allocate_memory that will not be returned to the
   * client.  If \p data is NULL then it indicates the allocation failed or was
   * for 0 bytes: in either case the callback is required to take no action.
   * If \p data is not NULL then it will not have been deallocated by a
   * previous call to amd_dbgapi_callbacks_s::allocate_memory.  Note that these
   * requirements can be met by implementing using \p free.
   *
   * Note this callback may be used by the library implementation if it
   * encounters an error after using amd_dbgapi_callbacks_s::allocate_memory to
   * allocate memory.
   */
  void (*deallocate_memory) (void *data);

  /**
   * Query information about the client process.
   *
   * \p client_process_id is the client handle of the process for which the
   * operating system process handle is being queried.
   *
   * \p query identifies the client process information queried by the
   * library.
   *
   * \p value_size is the size in bytes of the buffer \p value points to.
   *
   * \p value points to a buffer of size  \p value_size where the client should
   * copy the value requested by the library.
   *
   * Return ::AMD_DBGAPI_STATUS_SUCCESS if successful and \p value is updated.
   *
   * Return ::AMD_DBGAPI_STATUS_ERROR_INVALID_CLIENT_PROCESS_ID if the \p
   * client_process_id handle is invalid.
   *
   * Return ::AMD_DBGAPI_STATUS_ERROR_PROCESS_EXITED if the \p
   * client_process_id handle is associated with a native operating system
   * process that has already exited.
   *
   * Return ::AMD_DBGAPI_STATUS_ERROR_INVALID_ARGUMENT if \p value is NULL.
   *
   * Return ::AMD_DBGAPI_STATUS_ERROR_NOT_AVAILABLE if the requested
   * information is not available for the process referenced by \p
   * client_process_id.
   *
   * Return ::AMD_DBGAPI_STATUS_ERROR_INVALID_ARGUMENT_COMPATIBILITY if \p
   * value_size does not match the size of the data requested by the library.
   *
   * Return ::AMD_DBGAPI_STATUS_ERROR if an error was encountered.
   */
  amd_dbgapi_status_t (*client_process_get_info) (
      amd_dbgapi_client_process_id_t client_process_id,
      amd_dbgapi_client_process_info_t query,
      size_t value_size, void *value);

  /**
   * Insert a breakpoint in a shared library using a global address.
   *
   * The library only inserts breakpoints in loaded shared libraries.  It will
   * request to be notified when the shared library is unloaded, and will
   * remove any breakpoints it has inserted when notified that the shared
   * library is unloaded.
   *
   * It is the client's responsibility to to actually insert the breakpoint.
   *
   * \p client_process_id is the client handle of the process in which the
   * breakpoint is to be added.
   *
   * \p address is the global address to add the breakpoint.
   *
   * \p breakpoint_id is the handle to identify this breakpoint.  Each
   * added breakpoint for a process will have a unique handle, multiple
   * breakpoints for the same process will not be added with the same handle.
   * It must be specified when ::amd_dbgapi_report_breakpoint_hit is used to
   * report a breakpoint hit, and in the
   * ::AMD_DBGAPI_EVENT_KIND_BREAKPOINT_RESUME event that may be used to resume
   * the thread.
   *
   * Return ::AMD_DBGAPI_STATUS_SUCCESS if successful.  The breakpoint is
   * added.
   *
   * Return ::AMD_DBGAPI_STATUS_ERROR_INVALID_CLIENT_PROCESS_ID if the \p
   * client_process_id handle is invalid.  No breakpoint is added.
   *
   * Return ::AMD_DBGAPI_STATUS_ERROR_INVALID_BREAKPOINT_ID if there is a
   * breakpoint already added with \p breakpoint_id.  No breakpoint is added.
   *
   * Return ::AMD_DBGAPI_STATUS_ERROR if another error was encountered.  No
   * breakpoint is inserted and the \p breakpoint_id handle is invalidated.
   */
  amd_dbgapi_status_t (*insert_breakpoint) (
      amd_dbgapi_client_process_id_t client_process_id,
      amd_dbgapi_global_address_t address,
      amd_dbgapi_breakpoint_id_t breakpoint_id);

  /**
   * Remove a breakpoint previously inserted by
   * amd_dbgapi_callbacks_s::insert_breakpoint.
   *
   * It is the client's responsibility to to actually remove the breakpoint.
   *
   * \p breakpoint_id is invalidated.
   *
   * \p client_process_id is the client handle of the process in which the
   * breakpoint is to be removed.
   *
   * \p breakpoint_id is the breakpoint handle of the breakpoint to remove.
   *
   * Return ::AMD_DBGAPI_STATUS_SUCCESS if successful.  The breakpoint is
   * removed.
   *
   * Return ::AMD_DBGAPI_STATUS_ERROR_INVALID_CLIENT_PROCESS_ID if the
   * \p client_process_id handle is invalid.  No breakpoint is removed.
   *
   * Return ::AMD_DBGAPI_STATUS_ERROR_INVALID_BREAKPOINT_ID if \p breakpoint_id
   * handle is invalid.  No breakpoint is removed.
   *
   * Return ::AMD_DBGAPI_STATUS_ERROR_LIBRARY_NOT_LOADED if the
   * shared library containing the breakpoint is not currently loaded.  The
   * breakpoint will already have been removed.
   *
   * Return ::AMD_DBGAPI_STATUS_ERROR if another error was encountered.  The
   * breakpoint is considered removed and the \p breakpoint_id
   * handle is invalidated.
   */
  amd_dbgapi_status_t (*remove_breakpoint) (
      amd_dbgapi_client_process_id_t client_process_id,
      amd_dbgapi_breakpoint_id_t breakpoint_id);

  /**
   * Uncached global memory transfer.
   *
   * \p client_process_id is the client handle of the process for which the
   * memory transfer is being requested.
   *
   * \p global_address is the global address space address of the start of
   * the memory transfer being requested.
   *
   * \p value_size is the number of bytes of the memory transfer being
   * requested.
   *
   * \p read_buffer if not NULL then a read transfer is being requested. On
   * return, contains the read bytes and \p value_size is set to the number of
   * bytes actually read.
   *
   * \p write_buffer if not NULL then a write transfer is being requested.
   * Contains the bytes to be written. On return \p value_size is set to the
   * number of bytes actually written.
   *
   * Return ::AMD_DBGAPI_STATUS_SUCCESS if successful.
   *
   * Return ::AMD_DBGAPI_STATUS_ERROR_INVALID_CLIENT_PROCESS_ID if the
   * \p client_process_id handle is invalid.
   *
   * Return ::AMD_DBGAPI_STATUS_ERROR_INVALID_ARGUMENT_COMPATIBILITY if not
   * only one of read_buffer and \p write_puffer are NULL.
   *
   * Return ::AMD_DBGAPI_STATUS_ERROR_PROCESS_EXITED if the \p
   * client_process_id handle is associated with a native operating system
   * process that has already exited.
   *
   * Return ::AMD_DBGAPI_STATUS_ERROR_MEMORY_ACCESS if the input \p value_size
   * was greater than 0 and no bytes were successfully transferred.  The output
   * \p value_size is set to 0.  \p read_buffer and write_buffer are unaltered.
   */
  amd_dbgapi_status_t (*xfer_global_memory) (
      amd_dbgapi_client_process_id_t client_process_id,
      amd_dbgapi_global_address_t global_address,
      amd_dbgapi_size_t *value_size, void *read_buffer,
      const void *write_buffer);

  /**
   * Report a log message.
   *
   * \p level is the log level.
   *
   * \p message is a NUL terminated string to print that is owned by the
   * library and is only valid while the callback executes.
   */
  void (*log_message) (amd_dbgapi_log_level_t level, const char *message);
};

/** @} */

#if defined(__cplusplus)
} /* extern "C" */
#endif /* defined (__cplusplus) */

#endif /* amd-dbgapi.h */
