# (c) Copyright 2009-2021. CodeWeavers, Inc.

import os

from gi.repository import GdkPixbuf
from gi.repository import Gdk
from gi.repository import Gtk

import bottlequery
import cxguitools
import cxproduct
import cxutils
import distversion
import ratingutils

import logindialog

from cxutils import cxgettext as _

class RatingController(object):

    def set_ask_for_ratings(self, value):
        try:
            bottlequery.set_config_value(self.bottle, 'Bottle', 'AskForRatings', str(value))
        except bottlequery.NotFoundError:
            pass

    def __init__(self, bottle, appid, parent, is_nag=False, ask_again=True):
        self.xml = Gtk.Builder()
        self.xml.set_translation_domain("crossover")
        self.xml.add_from_file(cxguitools.get_ui_path("ratingdialog"))

        self.bottle = bottle
        self.rating_utils = ratingutils.RatingUtils(self.bottle, appid)
        success = self.rating_utils.get_rating()

        # Don't ask for a rating if we can't connect to the API or if
        # the user has already submitted a rating.
        if (not success and self.rating_utils.connection_failed) or (is_nag and self.rating_utils.rating):
            # Quit the main loop even before the window is shown if we don't
            # have to ask the user for a rating through a standalone window.
            if not parent:
                self.xml.get_object("RatingDialog").connect('event', self.quit_requested)
                self.xml.get_object("RatingDialog").show()
            if is_nag and self.rating_utils.rating:
                self.set_ask_for_ratings(0)
            return

        if not parent:
            # This is a standalone window, so we have to quit the main loop when the
            # dialog is destroyed.
            self.xml.get_object("RatingDialog").connect('destroy', self.quit_requested)

        self.rating_widget = RatingBox(self.xml.get_object('RatingDialogBox'), False, self.rating_utils)

        self.xml.get_object("RatingDialog").set_title(
            _("Submit a Rating for '%s'") % self.rating_utils.name)

        product = cxproduct.this_product()
        self.xml.get_object("RatingLabel").set_text(
            _("Please leave a rating in order to help us better target our improvements to %(product)s and to inform other users on how well '%(application)s' works with %(product)s.") % {'product': product['name'], 'application': self.rating_utils.name})

        if is_nag and ask_again:
            self.xml.get_object("LaterRatingButton").show()
            self.xml.get_object("LaterRatingButton").connect(
                'clicked', self.on_cancel_clicked)

            self.xml.get_object("NeverRatingButton").show()
            self.xml.get_object("NeverRatingButton").connect(
                'clicked', self.on_never_clicked)
        else:
            self.xml.get_object("CancelRatingButton").show()
            self.xml.get_object("CancelRatingButton").connect(
                'clicked', self.on_cancel_clicked)

        if is_nag:
            self.xml.get_object("RatingCheckButton").show()
            self.xml.get_object("RatingCheckButton").connect(
                'toggled', self.on_rating_checkbox_toggled)

        self.xml.get_object("SubmitRatingButton").connect('clicked', self.on_submit_clicked)

        self.xml.get_object("RatingDialog").set_transient_for(parent)
        self.xml.get_object("RatingDialog").show()

    @staticmethod
    def on_rating_checkbox_toggled(caller):
        if caller.get_active():
            # Don't ask again
            cxproduct.save_setting('CrossOver', 'AskForRatings', '0')
        else:
            cxproduct.save_setting('CrossOver', 'AskForRatings', '1')

    def on_cancel_clicked(self, _button):
        dialog = self.xml.get_object('RatingDialog')
        dialog.destroy()

    def on_never_clicked(self, _button):
        self.set_ask_for_ratings(0)
        dialog = self.xml.get_object('RatingDialog')
        dialog.destroy()

    def on_submit_clicked(self, _button=None):
        dialog = self.xml.get_object('RatingDialog')
        success = self.rating_utils.set_rating(self.rating_widget.rating, self.rating_widget.get_text())
        if not success and self.rating_utils.login_required:
            logindialog.LoginController(dialog.get_toplevel(), self.on_submit_clicked)
            return

        if not success:
            cxguitools.CXMessageDlg(primary=_("Rating submission failed"), buttons=Gtk.ButtonsType.OK,
                                    parent=self.xml.get_object('RatingDialog').get_toplevel(),
                                    message_type=Gtk.MessageType.ERROR)
            return

        self.set_ask_for_ratings(0)
        dialog.destroy()

    @staticmethod
    def quit_requested(*_args):
        Gtk.main_quit()

class RatingBox(object):

    def __init__(self, parent, static=True, rating_utils=None):
        self.xml = Gtk.Builder()
        self.xml.set_translation_domain("crossover")
        self.xml.add_from_file(cxguitools.get_ui_path("ratingdialog"))
        self.xml.connect_signals(self)

        self.icons = []
        self._load_images()
        self.static = static
        self.rating = None
        self.rating_utils = rating_utils

        self.placeholder_text = ''
        self.has_placeholder_text = False

        if not static:
            self.placeholder_text = _('Additional notes (optional)')
            self.has_placeholder_text = True
            self.xml.get_object("RatingEntry").get_buffer().set_text(self.placeholder_text)
            self.xml.get_object("RatingEntry").get_style_context().add_class("dim-label")
            self.xml.get_object("RatingEntryBox").show()
            self.xml.get_object("CharactersLeft").set_text(_('Max 140 characters, %d left') % 140)
            self.xml.get_object("CharactersLeft").get_style_context().add_class("dim-label")
            self.xml.get_object("CharactersLeft").show()

            self.xml.get_object("RatingEntry").get_buffer().connect('changed', self.on_text_changed)
            self.xml.get_object("RatingEntry").get_buffer().connect_after('insert-text', self.on_text_insert)

        if self.rating_utils:
            self.update_rating(self.rating_utils.rating)
            if self.rating_utils.notes:
                self.xml.get_object("RatingEntry").get_style_context().remove_class("dim-label")
                self.xml.get_object("RatingEntry").get_buffer().set_text(self.rating_utils.notes)
                self.has_placeholder_text = False

        widget = self.xml.get_object('RatingBox')
        parent.add(widget)

    def update_rating(self, rating=None, version_text=None, version_tooltip=None, temp_rating=False):
        if rating is None:
            self.icons = ['star-empty'] * 5
        elif 1 <= rating <= 5:
            self.icons = ['star'] * rating + ['star-empty'] * (5 - rating)
        elif rating == 0:
            self.icons = ['question'] * 5
        else:
            self.icons = ['untrusted'] * 5

        for i, icon in enumerate(self.icons):
            if icon in self.images:
                self.xml.get_object("star %d" % i).set_from_pixbuf(self.images[icon])

        text = self.rating_descriptions.get(rating, '')
        tooltip = self.rating_tooltips.get(rating, '')

        self.xml.get_object("PackageRatingView").show()
        self.xml.get_object("PackageRatingLabel").set_text(text)
        self.xml.get_object("PackageRatingLabel").show()

        if version_text:
            self.xml.get_object("PackageLastTested").set_text(version_text)
            self.xml.get_object("PackageLastTested").set_tooltip_text(version_tooltip)
            self.xml.get_object("PackageLastTested").show()
        else:
            self.xml.get_object("PackageLastTested").hide()

        self.xml.get_object("PackageRatingView").set_tooltip_text(tooltip)
        self.xml.get_object("PackageRatingLabel").set_tooltip_text(tooltip)

        if not temp_rating:
            self.rating = rating

    def hide(self):
        self.xml.get_object("PackageRatingView").hide()
        self.xml.get_object("PackageRatingLabel").hide()
        self.xml.get_object("PackageLastTested").hide()

    def _load_images(self):
        fg = self.xml.get_object("PackageRatingView").get_style_context().get_color(Gtk.StateType.NORMAL)
        brightness = (max(fg.red, fg.green, fg.blue) + 0.15) / 1.3

        self.images = {}

        star_icon = GdkPixbuf.Pixbuf.new_from_file(os.path.join(cxutils.CX_ROOT, 'share/images/rating_star.png'))
        if star_icon:
            star_icon = cxguitools.recolor_pixbuf(star_icon, Gdk.RGBA(brightness, brightness, brightness/3, 1.0))
            self.images['star'] = star_icon

        untrusted_icon = GdkPixbuf.Pixbuf.new_from_file(os.path.join(cxutils.CX_ROOT, 'share/images/rating_untrusted.png'))
        if untrusted_icon:
            untrusted_icon = cxguitools.recolor_pixbuf(untrusted_icon, fg)
            self.images['untrusted'] = untrusted_icon

        question_icon = GdkPixbuf.Pixbuf.new_from_file(os.path.join(cxutils.CX_ROOT, 'share/images/rating_question.png'))
        if question_icon:
            question_icon = cxguitools.recolor_pixbuf(question_icon, fg)
            self.images['question'] = question_icon

        fg.alpha *= 0.5
        star_empty_icon = GdkPixbuf.Pixbuf.new_from_file(os.path.join(cxutils.CX_ROOT, 'share/images/rating_star_empty.png'))
        if star_empty_icon:
            star_empty_icon = cxguitools.recolor_pixbuf(star_empty_icon, fg)
            self.images['star-empty'] = star_empty_icon

    def on_PackageRatingView_style_set(self, _widget, _previous_style):
        self._load_images()

        for i, icon in enumerate(self.icons):
            if icon in self.images:
                self.xml.get_object("star %d" % i).set_from_pixbuf(self.images[icon])

    def on_rating_set(self, widget, _event):
        if self.static:
            return

        rating = int(Gtk.Buildable.get_name(widget)[-1]) + 1
        self.update_rating(rating)

    def on_rating_enter(self, widget, _event):
        if self.static:
            return

        rating = int(Gtk.Buildable.get_name(widget)[-1]) + 1
        self.update_rating(rating, temp_rating=True)

    def on_rating_leave(self, _widget, _event):
        if self.static:
            return

        self.update_rating(self.rating)

    def on_text_focus_in(self, widget, _event):
        if self.has_placeholder_text:
            widget.get_style_context().remove_class("dim-label")
            widget.get_buffer().set_text('')
            self.has_placeholder_text = False

    def on_text_focus_out(self, widget, _event):
        textbuf = widget.get_buffer()
        if textbuf.get_text(textbuf.get_start_iter(), textbuf.get_end_iter(), True) == '':
            widget.get_style_context().add_class("dim-label")
            textbuf.set_text(self.placeholder_text)
            self.has_placeholder_text = True

    def on_text_insert(self, textbuf, location, _text, _length):
        length = textbuf.get_char_count()
        if length > 140:
            offset = textbuf.get_iter_at_offset(140)
            end = textbuf.get_end_iter()
            textbuf.delete(offset, end)
            location.assign(offset)
            length = 140
        self.xml.get_object("CharactersLeft").set_text(_('Max 140 characters, %d left') % (140 - length))

    def on_text_changed(self, textbuf):
        length = textbuf.get_char_count()
        self.xml.get_object("CharactersLeft").set_text(_('Max 140 characters, %d left') % (140 - length))

    def get_text(self):
        if self.has_placeholder_text:
            return ''

        textbuf = self.xml.get_object("RatingEntry").get_buffer()
        start, end = textbuf.get_bounds()
        return textbuf.get_text(start, end, False)

    def destroy_parent(self, _data):
        self.xml.get_object('RatingBox').get_toplevel().destroy()

    rating_descriptions = {
        5: _("Runs Great"),
        4: _("Runs Well"),
        3: _("Limited Functionality"),
        2: _("Installs, Will Not Run"),
        1: _("Will Not Install"),
        0: _("Untested"),
        -1: _("Untrusted"),
    }

    rating_tooltips = {
        5: _(u"This application installs and runs great, the same as it would in Windows.\n\nPractically all of the application is functional. Using this application in CrossOver should be the same as running the application on native Windows OS."),
        4: _(u"This application installs and runs quite well, with only minor glitches found.\n\nThe core functionality of this Windows application works fine under CrossOver. There still may be some extra functions that don\u2019t work right, but overall this application should be mostly usable. We don\u2019t maintain a list of what specific functions work/don\u2019t work per-application in our compatibility database, but you may want to check this application\u2019s Forums at the CodeWeavers website and see if other users have reported any particular details about running this Windows application in CrossOver."),
        3: _(u"This application installs and runs, but has problems.\n\nThis application runs, but there are significant functional gaps. CrossOver translates Windows commands to their corresponding %(platform)s commands and back. When a Windows program uses commands that CrossOver doesn\u2019t know how to translate the Windows program may not respond when you click things, have garbled output, or crash. You might be able to add different dependencies or try workarounds to get this application to run better, so please check this application\u2019s page at the CodeWeavers website. It\u2019s also possible that programming added in a future release of CrossOver might help this Windows application run better, so check back later.") % {'platform': distversion.PLATFORM},
        2: _(u"This application installs, but fails to run.\n\nThe Windows application immediately closes or freezes when you try to run it. You may be able to add various dependencies or try different workarounds to get this application to start, so please check the Support section of the CodeWeavers website for more information and tutorials. If the program still won\u2019t launch then whatever programming this Windows application needs to work doesn\u2019t exist yet in CrossOver."),
        1: _(u"This application won\u2019t get through installation.\n\nMost installers are standardized and CrossOver will be able to run them without a problem. However some companies customize the installer, or write their own, and the custom programming fails in CrossOver. The Windows application itself might be functional, but there\u2019s no way to get it out of the installer."),
        0: _(u"No ratings have been reported.\n\nAll application entries start at this rating. Over time staff and users can visit the What Runs page of the CodeWeavers website and rate how far they got in running this Windows application in CrossOver. If you try this application please submit a rating for how this application worked, or didn\u2019t, in CrossOver."),
        -1: _(u"This installation recipe is from an unknown source. Installing may pose a security risk to your system."),
    }
