"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.create = create;
const language_core_1 = require("@vue/language-core");
const shared_1 = require("@vue/shared");
const path_browserify_1 = require("path-browserify");
const getUserPreferences_1 = require("volar-service-typescript/lib/configs/getUserPreferences");
const vscode_uri_1 = require("vscode-uri");
const nameCasing_1 = require("../nameCasing");
const vue_extract_file_1 = require("../plugins/vue-extract-file");
const utils_1 = require("../utils");
function create(ts, { getImportPathForFile }) {
    return {
        name: 'vue-document-drop',
        capabilities: {
            documentDropEditsProvider: true,
        },
        create(context) {
            return {
                async provideDocumentDropEdits(document, _position, dataTransfer) {
                    const info = (0, utils_1.getEmbeddedInfo)(context, document, 'template', 'html');
                    if (!info) {
                        return;
                    }
                    const { sourceScript, root } = info;
                    let importUri;
                    for (const [mimeType, item] of dataTransfer) {
                        if (mimeType === 'text/uri-list') {
                            importUri = item.value;
                        }
                    }
                    if (!importUri || !root.vueCompilerOptions.extensions.some(ext => importUri.endsWith(ext))) {
                        return;
                    }
                    const { sfc } = root;
                    const script = sfc.scriptSetup ?? sfc.script;
                    if (!script) {
                        return;
                    }
                    const casing = await (0, nameCasing_1.checkCasing)(context, sourceScript.id);
                    const baseName = path_browserify_1.posix.basename(importUri);
                    const newName = (0, shared_1.capitalize)((0, shared_1.camelize)(baseName.slice(0, baseName.lastIndexOf('.'))));
                    const additionalEdit = {};
                    const code = [...(0, language_core_1.forEachEmbeddedCode)(root)].find(code => code.id === (sfc.scriptSetup ? 'scriptsetup_raw' : 'script_raw'));
                    const lastImportNode = (0, vue_extract_file_1.getLastImportNode)(ts, script.ast);
                    const incomingFileName = vscode_uri_1.URI.parse(importUri).fsPath.replace(/\\/g, '/');
                    let importPath;
                    const serviceScript = sourceScript.generated.languagePlugin.typescript?.getServiceScript(root);
                    if (serviceScript) {
                        const tsDocumentUri = context.encodeEmbeddedDocumentUri(sourceScript.id, serviceScript.code.id);
                        const tsDocument = context.documents.get(tsDocumentUri, serviceScript.code.languageId, serviceScript.code.snapshot);
                        const preferences = await (0, getUserPreferences_1.getUserPreferences)(context, tsDocument);
                        importPath = (await getImportPathForFile(root.fileName, incomingFileName, preferences) ?? {}).path;
                    }
                    if (!importPath) {
                        importPath = path_browserify_1.posix.relative(path_browserify_1.posix.dirname(root.fileName), incomingFileName)
                            || importUri.slice(importUri.lastIndexOf('/') + 1);
                        if (!importPath.startsWith('./') && !importPath.startsWith('../')) {
                            importPath = './' + importPath;
                        }
                    }
                    const embeddedDocumentUriStr = context.encodeEmbeddedDocumentUri(sourceScript.id, code.id).toString();
                    additionalEdit.changes ??= {};
                    additionalEdit.changes[embeddedDocumentUriStr] = [];
                    additionalEdit.changes[embeddedDocumentUriStr].push({
                        range: lastImportNode
                            ? {
                                start: script.ast.getLineAndCharacterOfPosition(lastImportNode.end),
                                end: script.ast.getLineAndCharacterOfPosition(lastImportNode.end),
                            }
                            : {
                                start: script.ast.getLineAndCharacterOfPosition(0),
                                end: script.ast.getLineAndCharacterOfPosition(0),
                            },
                        newText: `\nimport ${newName} from '${importPath}'`
                            + (lastImportNode ? '' : '\n'),
                    });
                    if (sfc.script) {
                        const edit = (0, vue_extract_file_1.createAddComponentToOptionEdit)(ts, sfc, sfc.script.ast, newName);
                        if (edit) {
                            additionalEdit.changes[embeddedDocumentUriStr].push({
                                range: {
                                    start: document.positionAt(edit.range.start),
                                    end: document.positionAt(edit.range.end),
                                },
                                newText: edit.newText,
                            });
                        }
                    }
                    return {
                        insertText: `<${casing.tag === nameCasing_1.TagNameCasing.Kebab ? (0, shared_1.hyphenate)(newName) : newName}$0 />`,
                        insertTextFormat: 2,
                        additionalEdit,
                    };
                },
            };
        },
    };
}
//# sourceMappingURL=vue-document-drop.js.map