// This file is part of libigl, a simple c++ geometry processing library.
//
// Copyright (C) 2023 Alec Jacobson
//
// This Source Code Form is subject to the terms of the Mozilla Public License
// v. 2.0. If a copy of the MPL was not distributed with this file, You can
// obtain one at http://mozilla.org/MPL/2.0/.
#include "lscm_hessian.h"
#include "vector_area_matrix.h"
#include "cotmatrix.h"
#include "repdiag.h"
template <typename DerivedV, typename DerivedF, typename QScalar>
void igl::lscm_hessian(
  const Eigen::MatrixBase<DerivedV> & V,
  const Eigen::MatrixBase<DerivedF> & F,
  Eigen::SparseMatrix<QScalar> & Q)
{
  // Assemble the area matrix (note that A is #Vx2 by #Vx2)
  Eigen::SparseMatrix<QScalar> A;
  igl::vector_area_matrix(F,A);
  // Assemble the cotan laplacian matrix
  Eigen::SparseMatrix<QScalar> L;
  igl::cotmatrix(V,F,L);
  Eigen::SparseMatrix<QScalar> L_flat;
  igl::repdiag(L,2,L_flat);
  Q = -L_flat - 2.*A;
}

#ifdef IGL_STATIC_LIBRARY
// Explicit template instantiation
// generated by autoexplicit.sh
template void igl::lscm_hessian<Eigen::Matrix<double, -1, -1, 0, -1, -1>, Eigen::Matrix<int, -1, -1, 0, -1, -1>, double>(Eigen::MatrixBase<Eigen::Matrix<double, -1, -1, 0, -1, -1> > const&, Eigen::MatrixBase<Eigen::Matrix<int, -1, -1, 0, -1, -1> > const&, Eigen::SparseMatrix<double, 0, int>&);
#endif
