# switch_row.py
#
# Copyright 2024 Hari Rana (TheEvilSkeleton)
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# SPDX-License-Identifier: GPL-3.0-or-later


from typing import Any

from gi.repository import GObject, Gtk, Gio

from refine.widgets.action_row import RefineActionRow
from refine.compositions import RefineCompositions

import logging


@Gtk.Template(resource_path="/page/tesk/Refine/widgets/switch-row.ui")
class RefineSwitchRow(RefineActionRow):
    """A `RefineActionRow` with on/off switches."""

    __gtype_name__ = "RefineSwitchRow"

    revert_button = Gtk.Template.Child()
    revealer = Gtk.Template.Child()

    is_default = GObject.Property(type=bool, default=True)
    active = GObject.Property(type=bool, default=False)

    def __init__(self, **kwargs: Any) -> None:
        super().__init__(**kwargs)

        self.connect("notify::key", self.__key_set)

    def __key_set(self, *args: Any) -> None:
        self.settings.bind(self.key, self, "active", Gio.SettingsBindFlags.DEFAULT)

        self.__active_set()
        self.connect("notify::active", self.__active_set_callback)
        self.revert_button.connect("clicked", self.__revert_button_clicked)
        RefineCompositions.set_up_separator_revealer(
            self.revealer, self.revert_button, "can-target"
        )

    def __active_set(self, *args: Any) -> None:
        self.is_default = self.settings.get_default_value(
            self.key
        ) != self.settings.get_value(self.key)
        to_float = float(self.is_default)

        self.revert_button.set_opacity(to_float)
        self.revert_button.set_can_focus(self.is_default)
        self.revert_button.set_can_target(self.is_default)

    def __active_set_callback(self, *args: Any) -> None:
        self.__active_set()
        self.value_set(self.settings.get_value(self.key))

    def __revert_button_clicked(self, *args: Any) -> None:
        logging.debug("Revert button pressed")
        self.settings.reset(self.key)
