# (c) Copyright 2022. CodeWeavers, Inc.

from gi.repository import Gdk
from gi.repository import Gtk

import os
import re

import cxguitools
import cxproduct
import cxutils

from cxutils import cxgettext as _


class LoggingOptionsController:

    def __init__(self, filename, parent_window, installer=False):

        #  Setup the GUI
        self.xml = Gtk.Builder()
        self.xml.set_translation_domain('crossover')
        self.xml.add_from_file(cxguitools.get_ui_path('loggingoptions'))
        self.xml.connect_signals(self)

        self.menu_open = False
        self.parent_window = parent_window

        self.enable_logging = False
        self.debug_channels = ''
        self.environment_variables = ''

        self.default_filename = True
        self.log_file = None

        self.set_default_filename(filename)

        self.recent = []
        self.load_recent()

        # Add indents to debug channel menu items
        for item in self.xml.get_object('DebugChannelMenu').get_children()[3:6]:
            item.set_label('  ' + item.get_label())

        for item in self.xml.get_object('DebugChannelMenu').get_children()[8:]:
            item.set_label('  ' + item.get_label())

        if installer:
            self.xml.get_object('InstallDebugChannelMenuItem').show()

    def get_view(self):
        return self.xml.get_object('LoggingOptions')

    def set_default_filename(self, filename):
        if self.default_filename:
            if 'CX_LOG' in os.environ:
                self.log_file = os.environ['CX_LOG']
            else:
                directory = os.path.join(cxproduct.get_user_dir(), 'logs')
                self.log_file = self.get_unique_filename(directory, filename)

            self.xml.get_object('LogFileButtonLabel').set_label(self.log_file)

    @staticmethod
    def get_unique_filename(directory, filename):
        if not filename:
            filename = 'log.cxlog'

        path = os.path.join(directory, filename)
        exists = os.path.exists(path)
        if not exists:
            return path

        filename, extension = os.path.splitext(filename)

        count = 1
        while exists:
            path = os.path.join(directory, filename + '_' + str(count) + extension)
            exists = os.path.exists(path)
            count += 1

        return path

    def load_recent(self):
        '''Load the 3 most recent logging options'''
        self.recent = []

        config = cxproduct.get_config()
        recent = config['OfficeSetup'].get('RecentLoggingOptions')
        if recent:
            self.recent = cxutils.string_to_utf8(recent).decode('unicode_escape').split('\0')
            self.recent = self.recent[:3]

        if self.recent:
            self.xml.get_object('RecentChannels').show()
            self.xml.get_object('RecentSeparator').show()
            for i in range(len(self.recent)):
                name = self.recent[i]
                if len(name) > 40:
                    name = name[:40] + '…'
                self.xml.get_object('Recent' + str(i + 1)).set_label(name)
                self.xml.get_object('Recent' + str(i + 1)).set_tooltip_text(self.recent[i])
                self.xml.get_object('Recent' + str(i + 1)).show()
        else:
            self.xml.get_object('RecentChannels').hide()
            self.xml.get_object('RecentSeparator').hide()

        for i in range(len(self.recent), 3):
            self.xml.get_object('Recent' + str(i + 1)).hide()

    def save_recent(self):
        '''Store the 3 most recent logging options'''

        # Escape the environment variables so they don't include "
        last = self.debug_channels + ' ' + self.environment_variables
        last = last.replace('"', "'").strip()

        # Prepend the last used item, even if it was already in the list
        if last in self.recent:
            self.recent.remove(last)
        else:
            self.recent = self.recent[:2]

        self.recent.insert(0, last)

        cxproduct.set_config_value('OfficeSetup', 'RecentLoggingOptions',
                                   '\0'.join(self.recent).encode('unicode_escape'))

        self.load_recent()

    def on_DebugChannelButton_button_press_event(self, widget, event):
        if self.menu_open or event.button != 1:
            return False # propagate

        menu = self.xml.get_object('DebugChannelMenu')
        cxguitools.popup_at_widget(
            menu,
            self.xml.get_object('DebugChannelButton'),
            Gdk.Gravity.SOUTH_WEST,
            Gdk.Gravity.NORTH_WEST,
            event)

        self.menu_open = True

        widget.set_active(True)

        return True

    def on_DebugChannelButton_clicked(self, widget):
        if self.menu_open or not widget.get_active():
            return

        menu = self.xml.get_object('DebugChannelMenu')
        cxguitools.popup_at_widget(
            menu,
            self.xml.get_object('DebugChannelButton'),
            Gdk.Gravity.SOUTH_WEST,
            Gdk.Gravity.NORTH_WEST,
            None)

        self.menu_open = True

    def on_DebugChannelEntry_changed(self, widget):
        self.debug_channels = widget.get_text()

    @staticmethod
    def on_DebugChannelEntry_insert_text(widget, text, _length, _position):
        filtered_text = re.sub(r'[^\w\+\-,\.: ]', '', text)
        if text != filtered_text:
            position = widget.get_position()
            widget.insert_text(filtered_text, position)
            widget.stop_emission_by_name('insert_text')

    def sanitize_channels(self):
        widget = self.xml.get_object('DebugChannelEntry')
        channels = widget.get_text()
        channels = re.sub(r' +', ' ', channels)  # Collapse sequential spaces
        channels = re.sub(r'^ ', '', channels)  # Leading spaces
        channels = re.sub(r' $', '', channels)  # Trailing spaces
        channels = re.sub(r' ?([,:]) ?', r'\1', channels)  # Spaces around commas or colons
        channels = re.sub(r',+', ',', channels)  # Collapse sequential commas
        channels = re.sub(r'^,', '', channels)  # Leading commas
        channels = re.sub(r',$', '', channels)  # Trailing commas
        self.debug_channels = channels
        widget.set_text(channels)

    def on_DebugChannelMenu_deactivate(self, _menushell):
        self.menu_open = False
        button = self.xml.get_object('DebugChannelButton')
        button.set_active(False)

    def on_DebugChannelMenuClear_activate(self, _menu):
        self.xml.get_object('DebugChannelEntry').set_text('')

    def on_DebugChannelMenuItem_activate(self, menu):
        channels = self.xml.get_object('DebugChannelEntry').get_text().split(',')
        parts = menu.get_tooltip_text().split(' ')
        for channel in parts[0].split(','):
            if channel not in channels:
                channels.append(channel)

        self.environment_variables = ''
        if len(parts) > 1:
            self.environment_variables = ' '.join(x for x in parts[1:] if x)

        self.xml.get_object('DebugChannelEntry').set_text(','.join(x for x in channels if x))

    def on_LogFileButton_clicked(self, _widget):
        file_chooser = Gtk.FileChooserDialog(
            title=_('Specify where to save the log file'),
            transient_for=self.parent_window,
            action=Gtk.FileChooserAction.SAVE)

        file_filter = Gtk.FileFilter()
        file_filter.add_pattern('*.cxlog')
        file_filter.set_name(_('CrossOver Log Files'))
        file_chooser.add_filter(file_filter)

        file_chooser.add_button(Gtk.STOCK_CANCEL, Gtk.ResponseType.CANCEL)
        button = file_chooser.add_button(Gtk.STOCK_SAVE, Gtk.ResponseType.OK)
        button.get_style_context().add_class('suggested-action')
        file_chooser.set_do_overwrite_confirmation(True)
        file_chooser.set_filename(self.log_file)
        file_chooser.set_current_name(os.path.basename(self.log_file))

        cxguitools.set_default_extension(file_chooser, 'cxlog')

        if file_chooser.run() == Gtk.ResponseType.OK:
            self.default_filename = False
            self.log_file = file_chooser.get_filename()
            self.xml.get_object('LogFileButtonLabel').set_label(self.log_file)
            file_chooser.destroy()
        else:
            file_chooser.destroy()
            return

    def on_LogFileCheckBox_toggled(self, widget):
        self.xml.get_object('LoggingChannels').set_sensitive(widget.get_active())
        self.xml.get_object('LogFileButtonLabel').set_label(self.log_file)
        self.enable_logging = widget.get_active()

        # Populate the default channels if the user hasn't changed anything.
        if self.enable_logging and not self.xml.get_object('DebugChannelEntry').get_text():
            self.xml.get_object('StandardDebugChannelMenuItem').activate()

    def on_LoggingOptions_realize(self, _widget):
        self.xml.get_object('DebugChannelEntry').set_text(self.debug_channels)
        self.xml.get_object('LogFileCheckBox').set_active(self.enable_logging)
