# (c) Copyright 2010. CodeWeavers, Inc.

"""Provides a framework for classes that have observable properties."""

ANY_EVENT = None


class Object:
    """Provides a framework for a class that has observable properties.

    The derived classes should have an observable_events class attribute that
    defines the set of events this class can emit. Events can be strings or
    any other type of object.
    """

    # This is the list of observable events
    observable_events = frozenset()

    def __init__(self):
        self._observers = {}

    def add_observer(self, events, observer, observer_data=None):
        """Adds an event observer for the current object.

        events can either be a single event or a sequence of events the
          observer is interested in. To subscribe to all the events from
          this object, specify the ANY_EVENT value.

        observer_data is any data the observer would like to be given back
          when it gets called.

        observer is a callable. Its signature must be of the form:
            observer(obj, event, observer_data, *args)
          Where obj is the object the event happens on, event is the event that
          was emitted, observer_data is the data specified above, and *args is
          an optional set of extra data specific to the event. Observers must
          not raise exceptions.
        """
        if isinstance(events, str) or not hasattr(events, '__iter__'):
            events = (events,)

        for event in events:
            if event and event not in self.observable_events:
                raise TypeError("class %s has no %s observable event" %
                                (self.__class__, repr(event)))
            self._observers.setdefault(event, {})

            observer_id = id(observer)
            self._observers[event][observer_id] = observer, observer_data
            return observer_id

    def remove_observer(self, events, observer_id):
        """Removes the specified observer for the specified events.

        No exception is raised if the observer has already been removed.
        """
        if not isinstance(observer_id, int):
            raise TypeError('Observers should be removed by id')

        if isinstance(events, str) or not hasattr(events, '__iter__'):
            events = (events,)

        for event in events:
            if event and event not in self.observable_events:
                raise TypeError("class %s has no %s observable event" %
                                (self.__class__, repr(event)))
            try:
                del self._observers[event][observer_id]
            except KeyError:
                # Just assume this observer has been removed already to
                # simplify error handling
                pass

    def emit_event_for_observer(self, event, observer, observer_data, *args):
        """Emits the specified event for a specific observer. Override this
        method if you want the observers to be called in a different
        way.

        """
        observer(self, event, observer_data, *args)

    def emit_event(self, event, *args):
        """Emits the specified event.

        Note that observers will be called in the same thread as emit_event().
        """
        if event not in self.observable_events:
            raise TypeError("class %s has no %s observable event" %
                            (self.__class__, repr(event)))

        for observer_event in (event, ANY_EVENT):
            if observer_event in self._observers:
                for (observer, observer_data) in list(self._observers[observer_event].values()):
                    self.emit_event_for_observer(event, observer, observer_data, *args)

    def all_observers(self):
        """Return a dictionary containing all observers"""
        ret = {}
        for event, observers in self._observers.items():
            ret[event] = []
            for observer, _data in observers.values():
                ret[event].append(observer)

        return ret
