# (c) Copyright 2010, 2015. CodeWeavers, Inc.

import os
import platform
import time

import distversion

import bottlequery
import bottlemanagement
import c4profilesmanager
import cxconfig
import cxdiag
import cxfsnotifier
import cxlog
import cxmenu
import cxobjc
import cxutils
import pyop

# for localization
from cxutils import cxgettext as _


menu_observer_queue = pyop.PythonOperationQueue(2)


class BottleWrapper(cxobjc.Proxy):

    STATUS_INIT = _("Scanning…")
    STATUS_UPGRADE = _("Needs upgrade")
    STATUS_READY = _("Ready")
    STATUS_ARCHIVING = _("Archiving…")
    STATUS_DEFAULTING = _("Making default…")
    STATUS_RENAMING = _("Renaming…")
    STATUS_DELETING = _("Deleting…")
    STATUS_DOWNING = _("Shutting down…")
    STATUS_FORCE_DOWNING = _("Forcing shutdown…")
    STATUS_UPGRADING = _("Upgrading…")
    STATUS_REPAIRING = _("Repairing…")
    STATUS_INSTALLING = _("Installing…")
    STATUS_DISABLED = _("Disabled")
    STATUS_SUSPENDED = _("Suspended")

    STATUS_GRAPHICS_BACKEND_UNKNOWN = "Graphics: Unknown"
    STATUS_GRAPHICS_BACKEND_AUTO = "Graphics: Auto"
    STATUS_GRAPHICS_BACKEND_D3DMETAL = "Graphics: D3DMetal"
    STATUS_GRAPHICS_BACKEND_DXMT = "Graphics: DXMT"
    STATUS_GRAPHICS_BACKEND_DXVK = "Graphics: DXVK"
    STATUS_GRAPHICS_BACKEND_WINE = "Graphics: Wine"

    STATUS_ESYNC_UNKNOWN = "ESync: Unknown"
    STATUS_ESYNC_ENABLED = "ESync: Enabled"
    STATUS_ESYNC_DISABLED = "ESync: Disabled"

    STATUS_HIGHRES_UNKNOWN = "High resolution mode: Unknown"
    STATUS_HIGHRES_ENABLED = "High resolution mode: Enabled"
    STATUS_HIGHRES_DISABLED = "High resolution mode: Disabled"
    STATUS_HIGHRES_UNAVAILABLE = "High resolution mode: Unavailable"

    STATUS_MSYNC_UNKNOWN = "MSync: Unknown"
    STATUS_MSYNC_ENABLED = "MSync: Enabled"
    STATUS_MSYNC_DISABLED = "MSync: Disabled"
    STATUS_MSYNC_UNAVAILABLE = "MSync: Unavailable"

    STATUS_PREVIEW_UNKNOWN = "Preview: Unknown"
    STATUS_PREVIEW_ENABLED = "Preview: Enabled"
    STATUS_PREVIEW_DISABLED = "Preview: Disabled"
    STATUS_PREVIEW_UNAVAILABLE = "Preview: Unavailable"

    name = cxobjc.object_property()
    appid = cxobjc.object_property()
    winePrefix = cxobjc.object_property(python_name="wine_prefix")
    systemDrive = cxobjc.object_property(python_name="system_drive")

    changeableName = cxobjc.object_property(python_name="changeablename")
    currentDescription = cxobjc.object_property(python_name="current_description")

    lastQuitFailed = cxobjc.object_property(python_name="last_quit_failed", typestr=cxobjc.BOOL)

    isDefault = cxobjc.object_property(python_name="is_default", typestr=cxobjc.BOOL)
    isManaged = cxobjc.object_property(python_name="is_managed", typestr=cxobjc.BOOL)
    isUpToDate = cxobjc.object_property(python_name="up_to_date", typestr=cxobjc.BOOL)

    isD3DMetalAvailable = cxobjc.object_property(python_name="is_d3dmetal_available", typestr=cxobjc.BOOL)
    isDXMTAvailable = cxobjc.object_property(python_name="is_dxmt_available", typestr=cxobjc.BOOL)

    controlPanelTable = cxobjc.array_property(python_name="control_panel_table")
    isControlPanelReady = cxobjc.object_property(
        python_name="control_panel_ready", typestr=cxobjc.BOOL)

    isHighResolutionEnabledReady = cxobjc.object_property(
        python_name="is_high_resolution_enabled_ready", typestr=cxobjc.BOOL)

    installedPackages = cxobjc.dict_property(python_name="installed_packages")
    isInstalledPackagesReady = cxobjc.object_property(
        python_name="installed_packages_ready", typestr=cxobjc.BOOL)

    menuChangedTime = cxobjc.object_property()

    def __init__(self, name):
        cxobjc.Proxy.__init__(self)

        self.name = name
        self.changeablename = name

        self.config = None
        self._change_delegates = []
        self.current_description = ""
        self.template = ""
        self.windows_version = ""
        self.arch = ""
        self.last_quit_failed = False
        self._config_changed_observer = None
        self._windows_version_observer = {}

        self.operations = []

        self.control_panel_loading = False
        self.control_panel_ready = False
        self.control_panel_table = []

        self.graphics_backend_state = BottleWrapper.STATUS_GRAPHICS_BACKEND_UNKNOWN
        self.is_d3dmetal_available = False
        self.is_dxmt_available = False

        self.is_esync_enabled_state = BottleWrapper.STATUS_ESYNC_UNKNOWN

        self.is_high_resolution_enabled_ready = False
        self.is_high_resolution_enabled_loading = False
        self.is_high_resolution_enabled_state = BottleWrapper.STATUS_HIGHRES_UNKNOWN

        self.is_msync_enabled_state = BottleWrapper.STATUS_MSYNC_UNKNOWN

        self.is_preview_enabled_state = BottleWrapper.STATUS_PREVIEW_UNKNOWN

        self.installed_packages = {}
        self.installed_packages_ready = False
        self.installed_packages_loading = False

        self.status = BottleWrapper.STATUS_INIT
        self.status_overrides = []

        # Let's learn a bit about ourselves. Everything here
        #  should be very quick.
        self.initialize()

    # This is a special initializer for objc. It must always be called
    #  explicitly on the mac.
    def initWithName_(self, inName):
        self = cxobjc.Proxy.nsobject_init(self)
        if self is not None:
            self.__init__(inName)
        return self

    @cxobjc.python_method
    def bottle_changed(self):
        self.callback('bottleChanged', self)

    @cxobjc.python_method
    def config_changed(self, *_args):
        '''Called when the bottle config has changed.
        This means that all bottle information has to be
        reloaded from the main thread.'''

        if self.STATUS_DELETING in self.status_overrides or \
           self.STATUS_RENAMING in self.status_overrides:
            return

        pyop.perform_on_main_thread(self.reload_info)

    @cxobjc.python_method
    def set_menu_changed_time(self):
        self.menuChangedTime = time.time()

    @cxobjc.python_method
    def menu_changed(self):
        '''Called when the bottle menu has changed.
        This means that all bottle information and launchers have to
        be reloaded from the main thread.'''

        if self.STATUS_DELETING in self.status_overrides or \
           self.STATUS_RENAMING in self.status_overrides:
            return

        pyop.perform_on_main_thread(self.set_menu_changed_time)
        pyop.perform_on_main_thread(self.reload_info)

    @cxobjc.namedSelector(b'addMenuObservers')
    def add_menu_observers(self):
        op = AddMenuObserversOperation(self)
        menu_observer_queue.enqueue(op)

    def add_menu_observers_main(self):
        '''This can be really slow so this should not be performed on the main thread.'''
        self.menu.add_observers(self.menu_changed)

    @cxobjc.namedSelector(b'removeMenuObservers')
    def remove_menu_observers(self):
        self.menu.remove_observers(self.menu_changed)

    @cxobjc.python_method
    def get_config_value(self, section, name, default):
        return self.config[section].get(name, default)

    @cxobjc.python_method
    def set_config_value(self, section, name, value):
        wconfig = self.config.get_save_config()
        if not wconfig:
            cxlog.warn("Unable to set a config value in bottle %s" % cxlog.to_str(self.name))
            return

        wconfig.lock_file()
        wconfig[section][name] = value
        wconfig.save_and_unlock_file()

    def suspend(self):
        self.add_status_override(BottleWrapper.STATUS_SUSPENDED)

        for operation in self.operations:
            operation.suspend()

    def resume(self):
        for operation in self.operations:
            operation.resume()

        self.remove_status_override(BottleWrapper.STATUS_SUSPENDED)

    @cxobjc.python_method
    def initialize(self):
        # Reset the bottle query cache before querying anything
        bottlequery.reset_cache(self.name)

        self.remove_config_observer()
        self.wine_prefix = bottlequery.get_prefix_for_bottle(self.name)
        self.config = bottlequery.get_config(self.name)
        self.add_config_observer()

        self.load_basic_info()

        self.system_drive = os.path.join(self.wine_prefix, 'drive_c')
        self.is_default = (self.name == bottlequery.get_default_bottle())
        self.menu = cxmenu.MenuPrefs(self.name, self.is_managed)

        self.add_windows_version_observer()

    @cxobjc.namedSelector(b'removeObservers')
    def remove_observers(self):
        self.remove_windows_version_observer()
        self.remove_config_observer()
        self.remove_menu_observers()

    @cxobjc.namedSelector(b'loadBasicInfo')
    def load_basic_info(self):
        """This function loads the bottle's basic properties.
        On the Mac, this function MUST be called in the main thread because
        it modifies properties which are bound to the UI.
        """
        if not self.wine_prefix:
            return

        self.arch = self.get_config_value("Bottle", "WineArch", "win32")
        self.current_description = self.get_config_value("Bottle", "Description", "")
        self.is_managed = self.get_config_value("Bottle", "Updater", "") != ""
        self.template = self.get_config_value("Bottle", "Template", "win98")
        self.windows_version = self.get_config_value("Bottle", "WindowsVersion", self.template.split("_")[0])
        self.appid = self.get_appid()

        self.refresh_up_to_date()

        self.load_graphics_backend_info()
        self.load_is_esync_enabled()
        self.load_is_msync_enabled()
        self.load_is_preview_enabled()

        self.bottle_changed()

    @cxobjc.namedSelector(b'reloadInfo')
    def reload_info(self):
        self.load_basic_info()

        if self.installed_packages_ready:
            self.installed_packages_ready = False
            self.load_application_info()

        if self.control_panel_ready:
            self.control_panel_ready = False
            self.load_control_panel_info()

        if self.is_high_resolution_enabled_ready:
            self.is_high_resolution_enabled_ready = False
            self.load_high_resolution_info()

    @cxobjc.namedSelector(b'refreshUpToDate')
    def refresh_up_to_date(self):
        """On the Mac, this function MUST be called in the main thread because
        it modifies properties which are bound to the UI.
        """
        if self.is_managed:
            self.up_to_date = bottlemanagement.get_up_to_date(self.name, "managed")
        else:
            self.up_to_date = bottlemanagement.get_up_to_date(self.name)

        if not self.up_to_date:
            self.add_status_override(BottleWrapper.STATUS_UPGRADE)
        else:
            self.remove_all_status_overrides(BottleWrapper.STATUS_UPGRADE)

    @cxobjc.namedSelector(b'setIsDefault:completionHandler:')
    def set_is_default(self, state, callback=None):
        op = SetIsDefaultOperation(self, state, callback)
        pyop.sharedOperationQueue.enqueue(op)

    @cxobjc.python_method
    def add_config_observer(self):
        self._config_changed_observer = self.config.add_observer(cxconfig.RELOADED, self.config_changed)

    @cxobjc.python_method
    def remove_config_observer(self):
        if self.config is None:
            return

        self.config.remove_observer(cxconfig.RELOADED, self._config_changed_observer)

    @cxobjc.python_method
    def add_windows_version_observer(self):
        self.remove_windows_version_observer()

        version_file = os.path.join(self.wine_prefix, '.version')
        self._windows_version_observer[version_file] = cxfsnotifier.add_observer(
            version_file, self.on_windows_version_changed)

        if not os.path.exists(version_file):
            with open(version_file, 'w', encoding='ascii'):
                pass

    @cxobjc.python_method
    def remove_windows_version_observer(self):
        for version_file, observer_id in self._windows_version_observer.items():
            cxfsnotifier.remove_observer(version_file, observer_id)

        self._windows_version_observer = {}

    @cxobjc.python_method
    def on_windows_version_changed(self, _event=None, _path=None, _data=None):
        if self.STATUS_DELETING in self.status_overrides or \
           self.STATUS_RENAMING in self.status_overrides:
            return

        try:
            version = bottlequery.get_windows_version(self.name)
            if version == self.windows_version:
                return

            self.set_config_value('Bottle', 'WindowsVersion', version)

            # Reload the bottle info from the main thread
            self.config_changed()
        except IOError as error:
            cxlog.log("Error while updating windows version in bottle " + self.name + ": " + str(error))

    #####
    #
    # Graphics backends
    #
    #####

    @cxobjc.python_method
    def is_macos14(self):
        mac_ver = platform.mac_ver()[0]
        mac_ver = mac_ver.split('.')[0]
        return distversion.IS_MACOSX and mac_ver and int(mac_ver) >= 14

    @cxobjc.python_method
    def is_apple_silicon(self):
        is_apple_silicon = False
        if 'arm' in platform.machine():
            is_apple_silicon = True
        else:
            import ctypes
            import ctypes.util

            libc = ctypes.CDLL(ctypes.util.find_library("c"))
            ret = ctypes.c_int()
            if libc.sysctlbyname(b"sysctl.proc_translated", ctypes.byref(ret), ctypes.byref(ctypes.c_size_t(4)), None, ctypes.c_size_t(0)) == 0:
                if (ret.value == 1):
                    is_apple_silicon = True

        return is_apple_silicon

    @cxobjc.python_method
    def load_graphics_backend_info(self):
        if self.is_macos14() and self.is_apple_silicon():
            self.is_d3dmetal_available = True
            self.is_dxmt_available = True

        backend = self.get_config_value("EnvironmentVariables", "CX_GRAPHICS_BACKEND", "")
        if self.is_d3dmetal_available and backend == "d3dmetal":
            self.graphics_backend_state = BottleWrapper.STATUS_GRAPHICS_BACKEND_D3DMETAL
        elif self.is_dxmt_available and backend == "dxmt":
            self.graphics_backend_state = BottleWrapper.STATUS_GRAPHICS_BACKEND_DXMT
        elif backend == "dxvk":
            self.graphics_backend_state = BottleWrapper.STATUS_GRAPHICS_BACKEND_DXVK
        elif backend == "wined3d":
            self.graphics_backend_state = BottleWrapper.STATUS_GRAPHICS_BACKEND_WINE
        else:
            self.graphics_backend_state = BottleWrapper.STATUS_GRAPHICS_BACKEND_AUTO

    @cxobjc.namedSelector(b'setGraphicsBackendAuto')
    def set_graphics_backend_auto(self):
        self.set_config_value("EnvironmentVariables", "CX_GRAPHICS_BACKEND", "")
        self.load_graphics_backend_info()
        self.bottle_changed()

    @cxobjc.namedSelector(b'setGraphicsBackendD3DMetal')
    def set_graphics_backend_d3dmetal(self):
        self.set_config_value("EnvironmentVariables", "CX_GRAPHICS_BACKEND", "d3dmetal")
        self.load_graphics_backend_info()
        self.bottle_changed()

    @cxobjc.namedSelector(b'setGraphicsBackendDXMT')
    def set_graphics_backend_dxmt(self):
        self.set_config_value("EnvironmentVariables", "CX_GRAPHICS_BACKEND", "dxmt")
        self.load_graphics_backend_info()
        self.bottle_changed()

    @cxobjc.namedSelector(b'setGraphicsBackendDXVK')
    def set_graphics_backend_dxvk(self):
        self.set_config_value("EnvironmentVariables", "CX_GRAPHICS_BACKEND", "dxvk")
        self.load_graphics_backend_info()
        self.bottle_changed()

    @cxobjc.namedSelector(b'setGraphicsBackendWine')
    def set_graphics_backend_wine(self):
        self.set_config_value("EnvironmentVariables", "CX_GRAPHICS_BACKEND", "wined3d")
        self.load_graphics_backend_info()
        self.bottle_changed()

    @cxobjc.python_method
    def is_d3dmetal_enabled(self):
        return self.graphics_backend_state == BottleWrapper.STATUS_GRAPHICS_BACKEND_D3DMETAL

    isD3DMetalEnabled = cxobjc.object_property(read_only=True, typestr=cxobjc.BOOL,
                                               getter=is_d3dmetal_enabled,
                                               depends_on=["graphics_backend_state"])

    @cxobjc.python_method
    def is_dxmt_enabled(self):
        return self.graphics_backend_state == BottleWrapper.STATUS_GRAPHICS_BACKEND_DXMT

    isDXMTEnabled = cxobjc.object_property(read_only=True, typestr=cxobjc.BOOL,
                                           getter=is_dxmt_enabled,
                                           depends_on=["graphics_backend_state"])

    @cxobjc.python_method
    def is_dxvk_enabled(self):
        return self.graphics_backend_state == BottleWrapper.STATUS_GRAPHICS_BACKEND_DXVK

    isDXVKEnabled = cxobjc.object_property(read_only=True, typestr=cxobjc.BOOL,
                                           getter=is_dxvk_enabled,
                                           depends_on=["graphics_backend_state"])

    @cxobjc.python_method
    def is_wined3d_enabled(self):
        return self.graphics_backend_state == BottleWrapper.STATUS_GRAPHICS_BACKEND_WINE

    isWineD3DEnabled = cxobjc.object_property(read_only=True, typestr=cxobjc.BOOL,
                                           getter=is_wined3d_enabled,
                                           depends_on=["graphics_backend_state"])

    #####
    #
    # load esync preferences
    #
    #####

    @cxobjc.python_method
    def load_is_esync_enabled(self):
        self.is_esync_enabled_state = BottleWrapper.STATUS_ESYNC_UNKNOWN

        enabled = self.get_config_value("EnvironmentVariables", "WINEESYNC", "0") != "0"
        if enabled:
            self.is_esync_enabled_state = BottleWrapper.STATUS_ESYNC_ENABLED
        else:
            self.is_esync_enabled_state = BottleWrapper.STATUS_ESYNC_DISABLED

    @cxobjc.namedSelector(b'enableESync')
    def enable_esync(self):
        if self.is_msync_enabled():
            self.disable_msync()

        self.set_config_value("EnvironmentVariables", "WINEESYNC", "1")
        self.is_esync_enabled_state = BottleWrapper.STATUS_ESYNC_ENABLED

        self.bottle_changed()

    @cxobjc.namedSelector(b'disableESync')
    def disable_esync(self):
        self.set_config_value("EnvironmentVariables", "WINEESYNC", "0")
        self.is_esync_enabled_state = BottleWrapper.STATUS_ESYNC_DISABLED

        self.bottle_changed()

    @cxobjc.python_method
    def is_esync_enabled(self):
        return self.is_esync_enabled_state == BottleWrapper.STATUS_ESYNC_ENABLED

    isESyncEnabled = cxobjc.object_property(read_only=True, typestr=cxobjc.BOOL,
                                            getter=is_esync_enabled,
                                            depends_on=["is_esync_enabled_state"])

    #####
    #
    # load high resolution mode preferences
    #
    #####

    @cxobjc.python_method
    def get_logpixels(self):
        log_pixels_key = "HKEY_CURRENT_USER\\Control Panel\\Desktop"
        old_log_pixels_key = "HKEY_LOCAL_MACHINE\\System\\CurrentControlSet\\Hardware Profiles\\Current\\Software\\Fonts"

        try:
            _subkeys, values = bottlequery.get_registry_key(self.name, log_pixels_key)

            if 'logpixels' not in values:
                _subkeys, values = bottlequery.get_registry_key(self.name, old_log_pixels_key)

            return values.get('logpixels', 96)
        except bottlequery.NotFoundError:
            return 96

    @cxobjc.namedSelector(b'loadHighResolutionInfo')
    def load_high_resolution_info(self):
        if self.is_high_resolution_enabled_ready or self.is_high_resolution_enabled_loading:
            return

        op = LoadHighResolutionInfoOperation(self)
        pyop.sharedOperationQueue.enqueue(op)

    def enableHighResolution(self):
        # macOS wrapper
        self.enable_high_resolution(96 * 2)

    @cxobjc.python_method
    def enable_high_resolution(self, logpixels):
        op = SetHighResolutionEnabledStateOperation(self, True, logpixels)
        pyop.sharedOperationQueue.enqueue(op)

    @cxobjc.namedSelector(b'disableHighResolution')
    def disable_high_resolution(self):
        op = SetHighResolutionEnabledStateOperation(self, False)
        pyop.sharedOperationQueue.enqueue(op)

    @cxobjc.python_method
    def is_high_resolution_enabled(self):
        return self.is_high_resolution_enabled_state == BottleWrapper.STATUS_HIGHRES_ENABLED

    isHighResolutionEnabled = cxobjc.object_property(read_only=True, typestr=cxobjc.BOOL,
                                                     getter=is_high_resolution_enabled,
                                                     depends_on=["is_high_resolution_enabled_state"])

    #####
    #
    # load msync preferences
    #
    #####

    @cxobjc.python_method
    def load_is_msync_enabled(self):
        self.is_msync_enabled_state = BottleWrapper.STATUS_MSYNC_UNKNOWN

        if not distversion.IS_MACOSX:
            self.is_msync_enabled_state = BottleWrapper.STATUS_MSYNC_UNAVAILABLE
            return

        enabled = self.get_config_value("EnvironmentVariables", "WINEMSYNC", "0") != "0"
        if enabled:
            self.is_msync_enabled_state = BottleWrapper.STATUS_MSYNC_ENABLED
        else:
            self.is_msync_enabled_state = BottleWrapper.STATUS_MSYNC_DISABLED

    @cxobjc.namedSelector(b'enableMSync')
    def enable_msync(self):
        if self.is_esync_enabled():
            self.disable_esync()

        self.set_config_value("EnvironmentVariables", "WINEMSYNC", "1")
        self.is_msync_enabled_state = BottleWrapper.STATUS_MSYNC_ENABLED

        self.bottle_changed()

    @cxobjc.namedSelector(b'disableMSync')
    def disable_msync(self):
        self.set_config_value("EnvironmentVariables", "WINEMSYNC", "0")
        self.is_msync_enabled_state = BottleWrapper.STATUS_MSYNC_DISABLED

        self.bottle_changed()

    @cxobjc.python_method
    def is_msync_enabled(self):
        return self.is_msync_enabled_state == BottleWrapper.STATUS_MSYNC_ENABLED

    isMSyncEnabled = cxobjc.object_property(read_only=True, typestr=cxobjc.BOOL,
                                            getter=is_msync_enabled,
                                            depends_on=["is_msync_enabled_state"])

    # Preview preferences

    @cxobjc.python_method
    def load_is_preview_enabled(self):
        self.is_preview_enabled_state = BottleWrapper.STATUS_PREVIEW_UNKNOWN

        if not distversion.IS_PREVIEW:
            self.is_preview_enabled_state = BottleWrapper.STATUS_PREVIEW_UNAVAILABLE
            return

        enabled = self.get_config_value("Bottle", "Preview", "0") != "0"
        if enabled:
            self.is_preview_enabled_state = BottleWrapper.STATUS_PREVIEW_ENABLED
            self.remove_all_status_overrides(BottleWrapper.STATUS_DISABLED)
        else:
            self.is_preview_enabled_state = BottleWrapper.STATUS_PREVIEW_DISABLED
            self.add_status_override(BottleWrapper.STATUS_DISABLED)

    @cxobjc.namedSelector(b'enablePreview')
    def enable_preview(self):
        self.set_config_value("Bottle", "Preview", "1")
        self.is_preview_enabled_state = BottleWrapper.STATUS_PREVIEW_ENABLED
        self.remove_all_status_overrides(BottleWrapper.STATUS_DISABLED)

        self.config_changed()

    @cxobjc.namedSelector(b'disablePreview')
    def disable_preview(self):
        self.set_config_value("Bottle", "Preview", "0")
        self.is_preview_enabled_state = BottleWrapper.STATUS_PREVIEW_DISABLED
        self.add_status_override(BottleWrapper.STATUS_DISABLED)

        self.config_changed()

    @cxobjc.python_method
    def is_preview_enabled(self):
        return self.is_preview_enabled_state == BottleWrapper.STATUS_PREVIEW_ENABLED

    isPreviewEnabled = cxobjc.object_property(read_only=True, typestr=cxobjc.BOOL,
                                              getter=is_preview_enabled,
                                              depends_on=["is_preview_enabled_state"])

    # Ratings preferences

    @cxobjc.namedSelector(b'disableRatingNag')
    def disable_rating_nag(self):
        self.set_config_value("Bottle", "AskForRatings", "0")
        self.bottle_changed()

    @cxobjc.python_method
    def should_nag_for_ratings(self):
        if self.get_config_value("Bottle", "AskForRatings", "1") != "1":
            return False

        if distversion.IS_PREVIEW:
            return False

        try:
            next_hook = int(self.get_config_value("Bottle", "RatingHookDate", "0"))
        except ValueError:
            # The macOS UI used a different value in CrossOver 24 and earlier.
            next_hook = 0

        now = time.time()
        if now < next_hook:
            return False

        self.set_config_value("Bottle", "RatingHookDate", "%d" % int(now + 7 * 60 * 60 * 24))
        if next_hook == 0:
            return False

        return True

    shouldNagForRatings = cxobjc.object_property(read_only=True, typestr=cxobjc.BOOL,
                                                 getter=should_nag_for_ratings,
                                                 depends_on=["config"])

    @cxobjc.namedSelector(b'loadControlPanelInfo')
    def load_control_panel_info(self):
        if self.control_panel_ready or self.control_panel_loading:
            return

        op = LoadControlPanelInfoOperation(self)
        pyop.sharedOperationQueue.enqueue(op)

    @cxobjc.namedSelector(b'launchControlPanelApplet:')
    def launch_control_panel_applet(self, applet):
        wine = os.path.join(cxutils.CX_ROOT, "bin", "wine")
        if applet.lower().endswith(".exe"):
            args = [wine, "--bottle", self.name, "--wl-app", applet]
            if applet.lower() == "reboot.exe":
                args.append("--show-gui")
        elif applet.lower().endswith(".cpl"):
            args = [wine, "--bottle", self.name, "--wl-app", "rundll32", "shell32.dll,Control_RunDLL", applet]
        else:
            __import__(applet).start(self)
            return

        cxutils.run(args, background=True)

    @cxobjc.namedSelector(b'loadApplicationInfo')
    def load_application_info(self):
        if self.installed_packages_ready or self.installed_packages_loading:
            return

        op = LoadApplicationInfoOperation(self)
        pyop.sharedOperationQueue.enqueue(op)

    @cxobjc.namedSelector(b'fileIsInBottle:')
    def file_is_in_bottle(self, filename):
        drivepath = os.path.abspath(os.path.realpath(self.system_drive))
        filepath = os.path.abspath(os.path.realpath(filename))
        return filepath.startswith(drivepath)

    @cxobjc.namedSelector(b'writeDescription')
    def write_description(self):
        #  Note:  Expensive!  Should only be done within an operation.
        if self.get_config_value("Bottle", "Description", "") != self.current_description:
            bottlemanagement.set_bottle_description(self.name, self.current_description)

    @cxobjc.python_method
    def wait_for_shutdown(self):
        '''Wait a bit for for the bottle to truly shut down after calling quit'''
        for _i in range(50):
            if self.can_force_quit or not bottlemanagement.is_running(self.name):
                break
            time.sleep(0.1)

    @cxobjc.namedSelector(b'delete:')
    def delete(self, callback=None):
        def quit_callback():
            op = DeleteBottleOperation(self, callback)
            pyop.sharedOperationQueue.enqueue(op)

        self.suspend()

        self.quit(quit_callback)

    @cxobjc.namedSelector(b'rename:completionHandler:')
    def rename(self, new_name, callback=None):
        def quit_callback():
            op = RenameBottleOperation(self, new_name, callback)
            pyop.sharedOperationQueue.enqueue(op)

        self.suspend()

        self.quit(quit_callback)

    @cxobjc.namedSelector(b'addChangeDelegate:')
    def add_change_delegate(self, delegate):
        self._change_delegates.append(delegate)

    @cxobjc.namedSelector(b'removeChangeDelegate:')
    def remove_change_delegate(self, delegate):
        if delegate in self._change_delegates:
            self._change_delegates.remove(delegate)

    @cxobjc.namedSelector(b'addStatusOverride:')
    def add_status_override(self, status):
        self.status_overrides.append(status)
        self.status = status
        self.bottle_changed()

    @cxobjc.namedSelector(b'removeAllStatusOverrides:')
    def remove_all_status_overrides(self, status):
        while status in self.status_overrides:
            self.remove_status_override(status)

    @cxobjc.namedSelector(b'removeStatusOverride:')
    def remove_status_override(self, status):
        if status in self.status_overrides:
            self.status_overrides.remove(status)
        if self.status_overrides:
            self.status = self.status_overrides[len(self.status_overrides) - 1]
        elif not self.installed_packages_ready or not self.control_panel_ready:
            self.status = BottleWrapper.STATUS_INIT
        else:
            self.status = BottleWrapper.STATUS_READY
        self.bottle_changed()

    def get_can_run_commands(self):
        return (self.up_to_date or not self.is_managed) and not self.is_busy and not self.is_disabled

    can_run_commands = property(get_can_run_commands)

    canRunCommands = cxobjc.object_property(read_only=True, typestr=cxobjc.BOOL,
                                            getter=get_can_run_commands,
                                            depends_on=["up_to_date", "is_managed", "is_busy"])

    def get_can_edit(self):
        return self.up_to_date and not self.is_busy and not self.is_managed

    can_edit = property(get_can_edit)

    canEdit = cxobjc.object_property(read_only=True, typestr=cxobjc.BOOL,
                                     getter=get_can_edit,
                                     depends_on=["up_to_date", "is_managed", "is_busy"])

    def get_is_usable(self):
        for status in self.status_overrides:
            if status not in (self.STATUS_INIT, self.STATUS_UPGRADE,
                              self.STATUS_READY, self.STATUS_DEFAULTING):
                return False

        return True

    is_usable = property(get_is_usable)

    isUsable = cxobjc.object_property(read_only=True, typestr=cxobjc.BOOL,
                                      getter=get_is_usable,
                                      depends_on=["status"])

    def get_can_install(self):
        return not self.is_managed and self.is_usable

    can_install = property(get_can_install)

    canInstall = cxobjc.object_property(read_only=True, typestr=cxobjc.BOOL,
                                        getter=get_can_install,
                                        depends_on=["is_managed", "is_usable"])

    def get_is_active(self):
        return not self.status_overrides and self.up_to_date

    is_active = property(get_is_active)

    isActive = cxobjc.object_property(read_only=True, typestr=cxobjc.BOOL,
                                      getter=get_is_active,
                                      depends_on=["status", "up_to_date"])

    def get_can_force_quit(self):
        """Be careful of race conditions with this property.  If all you've done
        is initiate an asynchronous operation to quit the bottle, these statuses
        won't necessarily be set yet."""
        return (self.last_quit_failed or BottleWrapper.STATUS_DOWNING in self.status_overrides) and \
            BottleWrapper.STATUS_FORCE_DOWNING not in self.status_overrides

    can_force_quit = property(get_can_force_quit)

    canForceQuit = cxobjc.object_property(read_only=True, typestr=cxobjc.BOOL,
                                          getter=get_can_force_quit,
                                          depends_on=["status", "last_quit_failed"])

    def get_is_busy(self):
        return BottleWrapper.STATUS_RENAMING in self.status_overrides or \
            BottleWrapper.STATUS_DELETING in self.status_overrides or \
            BottleWrapper.STATUS_REPAIRING in self.status_overrides or \
            BottleWrapper.STATUS_ARCHIVING in self.status_overrides or \
            BottleWrapper.STATUS_INSTALLING in self.status_overrides or \
            BottleWrapper.STATUS_DOWNING in self.status_overrides or \
            BottleWrapper.STATUS_FORCE_DOWNING in self.status_overrides

    is_busy = property(get_is_busy)

    isBusy = cxobjc.object_property(read_only=True, typestr=cxobjc.BOOL,
                                    getter=get_is_busy,
                                    depends_on=["status"])

    def get_is_disabled(self):
        return BottleWrapper.STATUS_DISABLED in self.status_overrides

    is_disabled = property(get_is_disabled)

    isDisabled = cxobjc.object_property(read_only=True, typestr=cxobjc.BOOL,
                                        getter=get_is_disabled,
                                        depends_on=["status"])

    def get_is_deleted(self):
        return not os.path.isdir(self.wine_prefix)

    is_deleted = property(get_is_deleted)

    isDeleted = cxobjc.object_property(read_only=True, typestr=cxobjc.BOOL,
                                       getter=get_is_deleted)

    def get_is_renaming(self):
        return BottleWrapper.STATUS_RENAMING in self.status_overrides

    is_renaming = property(get_is_renaming)

    isRenaming = cxobjc.object_property(read_only=True, typestr=cxobjc.BOOL,
                                        getter=get_is_renaming,
                                        depends_on=["status"])

    def get_is_suspended(self):
        return BottleWrapper.STATUS_SUSPENDED in self.status_overrides

    is_suspended = property(get_is_suspended)

    def get_can_rename(self):
        return not self.status_overrides and not self.is_managed

    can_rename = property(get_can_rename)

    canRename = cxobjc.object_property(read_only=True, typestr=cxobjc.BOOL,
                                       getter=get_can_rename,
                                       depends_on=["status", "is_managed"])

    def get_needs_upgrade(self):
        return self.is_managed and not self.up_to_date

    needs_upgrade = property(get_needs_upgrade)

    needsUpgrade = cxobjc.object_property(read_only=True, typestr=cxobjc.BOOL,
                                          getter=get_needs_upgrade,
                                          depends_on=["up_to_date", "is_managed"])

    @cxobjc.namedSelector(b'quit:')
    def quit(self, callback=None):
        op = QuitBottleOperation(self, callback)
        pyop.sharedOperationQueue.enqueue(op)

    @cxobjc.namedSelector(b'forceQuit:')
    def force_quit(self, callback=None):
        op = ForceQuitBottleOperation(self, callback)
        pyop.sharedOperationQueue.enqueue(op)

    def get_display_name(self):
        if self.is_default:
            return _("%s (default)") % self.name
        return self.name

    display_name = property(get_display_name)

    displayName = cxobjc.object_property(read_only=True, getter=get_display_name,
                                         depends_on=["name", "is_default"])

    @cxobjc.python_method
    def get_windows_version_display_name(self):
        version_name = bottlequery.get_windows_version_name(self.windows_version)
        bitness = '32' if self.arch == 'win32' else '64'
        return _('%(version)s %(bitness)s-bit') % {'version': version_name, 'bitness': bitness}

    windows_version_display_name = property(get_windows_version_display_name)

    windowsVersionDisplayName = cxobjc.object_property(read_only=True, getter=get_windows_version_display_name,
                                                       depends_on=["windows_version", "arch"])

    def get_appid(self):
        appid = self.get_config_value('EnvironmentVariables', 'CX_BOTTLE_CREATOR_APPID', None)
        if appid and appid.startswith('com.codeweavers.c4.'):
            c4_id = appid[19:]
            if c4_id.isdigit():
                return c4_id
        return None

    def get_profile(self):
        if not self.appid:
            return None

        appid = 'com.codeweavers.c4.%s' % self.appid
        profiles = c4profilesmanager.C4ProfilesSet.all_profiles()
        return profiles.get(appid, None)

    profile = property(get_profile)

    @cxobjc.namedSelector(b'getMissingDependenciesProfile')
    def get_missing_dependencies_profile(self):
        profile = self.profile
        if profile:
            profile = profile.copy()
            profile.app_profile.flags.add('virtual')
            profile.app_profile.steamid = None

        return profile

    # A couple of crutches for reverse-compatibility with the old BottleWrapper.m
    def bottleName(self):
        return self.name

    @cxobjc.python_method
    def callback(self, name, *args):
        '''Perform a callback in the main UI'''
        def do_nothing(*_args, **_kwargs):
            pass

        for delegate in self._change_delegates:
            getattr(delegate, name, do_nothing)(*args)


class BottleWrapperOperation(pyop.PythonOperation):
    def __init__(self, bottle):
        pyop.PythonOperation.__init__(self)
        self.bottle = bottle

    def __repr__(self):
        return self.__class__.__name__ + " for " + self.bottle.name

    def cancel(self):
        pyop.PythonOperation.cancel(self)

        if self in self.bottle.operations:
            self.bottle.operations.remove(self)

    def enqueued(self):
        pyop.PythonOperation.enqueued(self)
        self.bottle.operations.append(self)

        if self.bottle.is_suspended:
            self.suspend()

    def main(self):
        raise NotImplementedError

    def finish(self):
        if self in self.bottle.operations:
            self.bottle.operations.remove(self)

        pyop.PythonOperation.finish(self)


class AddMenuObserversOperation(BottleWrapperOperation):
    def __init__(self, bottle):
        BottleWrapperOperation.__init__(self, bottle)

    def main(self):
        self.bottle.add_menu_observers_main()


class LoadApplicationInfoOperation(BottleWrapperOperation):
    def __init__(self, bottle):
        BottleWrapperOperation.__init__(self, bottle)

        self._installed_packages_off_thread = {}

    def enqueued(self):
        BottleWrapperOperation.enqueued(self)

        self.bottle.installed_packages_ready = False
        self.bottle.installed_packages_loading = True
        self._installed_packages_off_thread = {}

        self.bottle.add_status_override(BottleWrapper.STATUS_INIT)

    def main(self):
        import appdetector

        profiles = c4profilesmanager.C4ProfilesSet.all_profiles()
        self._installed_packages_off_thread = appdetector.fast_get_installed_applications(self.bottle.name, profiles)

    def finish(self):
        self.bottle.installed_packages = self._installed_packages_off_thread
        self.bottle.installed_packages_ready = True
        self.bottle.installed_packages_loading = False

        self.bottle.remove_status_override(BottleWrapper.STATUS_INIT)

        BottleWrapperOperation.finish(self)


class LoadControlPanelInfoOperation(BottleWrapperOperation):
    def __init__(self, bottle):
        BottleWrapperOperation.__init__(self, bottle)
        self.priority = -1

        self._control_panel_off_thread_table = []

    def enqueued(self):
        BottleWrapperOperation.enqueued(self)

        self.bottle.control_panel_loading = True
        self.bottle.control_panel_ready = False

        self.bottle.add_status_override(BottleWrapper.STATUS_INIT)

    def main(self):
        self._control_panel_off_thread_table = bottlequery.get_control_panel_info(self.bottle.name)

        if not distversion.IS_MACOSX:
            self._control_panel_off_thread_table.append(["cxassoceditui", _("Edit Associations"),
                                                         _("Manage the Windows programs used to open files in the native environment"), "cxassocedit"])

            self._control_panel_off_thread_table.append(["cxmenueditui", _("Edit Menus"),
                                                         _("Manage the menus and desktop icons in this bottle"), "cxmenuedit"])

    def finish(self):
        cptable = []
        for panel in self._control_panel_off_thread_table:
            cpdict = {}
            cpdict["exe"] = panel[0]
            cpdict["name"] = panel[1]
            cpdict["description"] = panel[2]
            cptable.append(cpdict)

        self.bottle.control_panel_table = cptable
        self.bottle.control_panel_ready = True
        self.bottle.control_panel_loading = False

        self.bottle.remove_status_override(BottleWrapper.STATUS_INIT)

        BottleWrapperOperation.finish(self)


class LoadHighResolutionInfoOperation(BottleWrapperOperation):
    def __init__(self, bottle):
        BottleWrapperOperation.__init__(self, bottle)

    def enqueued(self):
        BottleWrapperOperation.enqueued(self)

        self.bottle.is_high_resolution_enabled_ready = False
        self.bottle.is_high_resolution_enabled_loading = True

        self.bottle.add_status_override(BottleWrapper.STATUS_INIT)

    def main(self):
        mac_driver_key = "HKEY_CURRENT_USER\\Software\\Wine\\Mac Driver"

        self.bottle.is_high_resolution_enabled_state = BottleWrapper.STATUS_HIGHRES_UNKNOWN

        if not distversion.IS_MACOSX:
            dpi = int(float(cxdiag.get(None).properties.get('display.dpi', 0)))
            if dpi < 110:
                self.bottle.is_high_resolution_enabled_state = BottleWrapper.STATUS_HIGHRES_UNAVAILABLE
                return

        try:
            if distversion.IS_MACOSX:
                _subkeys, values = bottlequery.get_registry_key(self.bottle.name, mac_driver_key)

                if 'retinamode' in values and values['retinamode'][0] in set("yYtT1"):
                    self.bottle.is_high_resolution_enabled_state = BottleWrapper.STATUS_HIGHRES_ENABLED
            elif self.bottle.get_logpixels() >= 110:
                self.bottle.is_high_resolution_enabled_state = BottleWrapper.STATUS_HIGHRES_ENABLED
        except bottlequery.NotFoundError:
            pass

        if self.bottle.is_high_resolution_enabled_state == BottleWrapper.STATUS_HIGHRES_UNKNOWN:
            self.bottle.is_high_resolution_enabled_state = BottleWrapper.STATUS_HIGHRES_DISABLED

    def finish(self):
        self.bottle.is_high_resolution_enabled_ready = True
        self.bottle.is_high_resolution_enabled_loading = False

        self.bottle.remove_status_override(BottleWrapper.STATUS_INIT)

        BottleWrapperOperation.finish(self)


class SetHighResolutionEnabledStateOperation(BottleWrapperOperation):
    def __init__(self, bottle, state, logpixels=96):
        BottleWrapperOperation.__init__(self, bottle)
        self.state = state
        self.logpixels = logpixels

    def enqueued(self):
        BottleWrapperOperation.enqueued(self)

        self.bottle.is_high_resolution_enabled_loading = True
        self.bottle.is_high_resolution_enabled_ready = False

    def main(self):
        mac_driver_key = "HKEY_CURRENT_USER\\Software\\Wine\\Mac Driver"
        log_pixels_key = "HKEY_CURRENT_USER\\Control Panel\\Desktop"

        logpixels = self.logpixels
        if self.state:
            if distversion.IS_MACOSX and bottlequery.set_registry_key(self.bottle.name, mac_driver_key, "RetinaMode", "y"):
                logpixels = self.bottle.get_logpixels() * 2
        else:
            # Force the DPI to 96 or the previous value so winewrapper won't overwrite it.
            if distversion.IS_MACOSX and bottlequery.unset_registry_value(self.bottle.name, mac_driver_key, "RetinaMode"):
                logpixels = max(self.bottle.get_logpixels() // 2, logpixels)

        bottlequery.set_registry_key(self.bottle.name, log_pixels_key, "LogPixels", "dword:" + hex(logpixels)[2:])

        if self.state:
            self.bottle.is_high_resolution_enabled_state = BottleWrapper.STATUS_HIGHRES_ENABLED
        else:
            self.bottle.is_high_resolution_enabled_state = BottleWrapper.STATUS_HIGHRES_DISABLED

    def finish(self):
        self.bottle.is_high_resolution_enabled_ready = True
        self.bottle.is_high_resolution_enabled_loading = False

        self.bottle.quit()

        BottleWrapperOperation.finish(self)


class SetIsDefaultOperation(BottleWrapperOperation):
    def __init__(self, bottle, state, callback):
        BottleWrapperOperation.__init__(self, bottle)
        self.state = state
        self.callback = callback
        self.success = False

    def enqueued(self):
        BottleWrapperOperation.enqueued(self)
        self.bottle.add_status_override(self.bottle.STATUS_DEFAULTING)

    def main(self):
        self.success, _err = bottlemanagement.set_default_bottle(self.bottle.name, self.state)

    def finish(self):
        self.bottle.remove_status_override(self.bottle.STATUS_DEFAULTING)

        if self.success and self.callback:
            self.callback()

        BottleWrapperOperation.finish(self)


class DeleteBottleOperation(BottleWrapperOperation):
    def __init__(self, bottle, callback):
        BottleWrapperOperation.__init__(self, bottle)
        self.callback = callback
        self.success = False

    def suspend(self):
        return

    def enqueued(self):
        BottleWrapperOperation.enqueued(self)

        self.bottle.add_status_override(self.bottle.STATUS_DELETING)

    def main(self):
        self.bottle.wait_for_shutdown()
        self.success, _err = bottlemanagement.delete_bottle(self.bottle.name, self.bottle.is_managed)

    def finish(self):
        if self.callback:
            self.callback()

        if not self.success and not self.bottle.is_deleted:
            self.bottle.remove_status_override(self.bottle.STATUS_DELETING)

        # Cancel waiting operations so they can be cleaned up
        for operation in self.bottle.operations:
            operation.cancel()

        BottleWrapperOperation.finish(self)


class RenameBottleOperation(BottleWrapperOperation):
    def __init__(self, bottle, new_name, callback):
        BottleWrapperOperation.__init__(self, bottle)
        self.new_name = new_name
        self.callback = callback
        self.success = True

    def suspend(self):
        return

    def enqueued(self):
        BottleWrapperOperation.enqueued(self)

        self.bottle.add_status_override(BottleWrapper.STATUS_RENAMING)
        self.bottle.changeablename = self.new_name

        if self.new_name == self.bottle.name:
            self.success = False
        elif not bottlequery.is_valid_new_bottle_name(self.new_name):
            self.success = False

    def main(self):
        if not self.success:
            return

        self.bottle.wait_for_shutdown()
        self.success, err = bottlemanagement.rename_bottle(self.bottle.name, self.bottle.changeablename)
        if not self.success:
            cxlog.warn("Rename from %s to %s failed: %s" % (self.bottle.name, self.bottle.changeablename, err))

    def finish(self):
        if not self.success:
            self.bottle.load_basic_info()
            self.bottle.changeablename = self.bottle.name
        elif self.bottle.menu.config_observer:
            self.bottle.remove_menu_observers()
            self.bottle.name = self.bottle.changeablename
            self.bottle.initialize()
            self.bottle.add_menu_observers()
        else:
            self.bottle.name = self.bottle.changeablename
            self.bottle.initialize()

        self.bottle.resume()

        self.bottle.remove_status_override(BottleWrapper.STATUS_RENAMING)

        if self.success and self.callback:
            self.callback()

        BottleWrapperOperation.finish(self)


class QuitBottleOperation(BottleWrapperOperation):
    def __init__(self, bottle, callback):
        BottleWrapperOperation.__init__(self, bottle)
        self.callback = callback
        self.success = None

    def suspend(self):
        return

    def enqueued(self):
        BottleWrapperOperation.enqueued(self)

        self.bottle.last_quit_failed = False
        self.bottle.add_status_override(BottleWrapper.STATUS_DOWNING)

    def main(self):
        self.success = bottlemanagement.quit_bottle(self.bottle.name)

    def finish(self):
        if not self.success:
            self.bottle.last_quit_failed = True

        self.bottle.remove_status_override(BottleWrapper.STATUS_DOWNING)

        if self.callback:
            self.callback()

        BottleWrapperOperation.finish(self)


class ForceQuitBottleOperation(BottleWrapperOperation):
    def __init__(self, bottle, callback):
        BottleWrapperOperation.__init__(self, bottle)
        self.callback = callback
        self.success = None

    def suspend(self):
        return

    def enqueued(self):
        BottleWrapperOperation.enqueued(self)

        self.bottle.add_status_override(BottleWrapper.STATUS_FORCE_DOWNING)

    def main(self):
        self.success = bottlemanagement.kill_bottle(self.bottle.name)

    def finish(self):
        if self.success:
            self.bottle.last_quit_failed = False

        self.bottle.remove_status_override(BottleWrapper.STATUS_FORCE_DOWNING)

        if self.callback:
            self.callback()

        BottleWrapperOperation.finish(self)
