"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;
exports.registerADRoutes = registerADRoutes;
var _lodash = require("lodash");
var _constants = require("../utils/constants");
var _helpers = require("../utils/helpers");
var _adHelpers = require("./utils/adHelpers");
function _defineProperty(e, r, t) { return (r = _toPropertyKey(r)) in e ? Object.defineProperty(e, r, { value: t, enumerable: !0, configurable: !0, writable: !0 }) : e[r] = t, e; }
function _toPropertyKey(t) { var i = _toPrimitive(t, "string"); return "symbol" == typeof i ? i : i + ""; }
function _toPrimitive(t, r) { if ("object" != typeof t || !t) return t; var e = t[Symbol.toPrimitive]; if (void 0 !== e) { var i = e.call(t, r || "default"); if ("object" != typeof i) return i; throw new TypeError("@@toPrimitive must return a primitive value."); } return ("string" === r ? String : Number)(t); } /*
                                                                                                                                                                                                                                                                                                                          * SPDX-License-Identifier: Apache-2.0
                                                                                                                                                                                                                                                                                                                          *
                                                                                                                                                                                                                                                                                                                          * The OpenSearch Contributors require contributions made to
                                                                                                                                                                                                                                                                                                                          * this file be licensed under the Apache-2.0 license or a
                                                                                                                                                                                                                                                                                                                          * compatible open source license.
                                                                                                                                                                                                                                                                                                                          *
                                                                                                                                                                                                                                                                                                                          * Modifications Copyright OpenSearch Contributors. See
                                                                                                                                                                                                                                                                                                                          * GitHub history for details.
                                                                                                                                                                                                                                                                                                                          */
function registerADRoutes(apiRouter, adService) {
  // create detector
  apiRouter.post('/detectors', adService.putDetector);
  apiRouter.post('/detectors/{dataSourceId}', adService.putDetector);

  // put detector
  apiRouter.put('/detectors/{detectorId}', adService.putDetector);
  apiRouter.put('/detectors/{detectorId}/{dataSourceId}', adService.putDetector);

  // routes not used in the UI, therefore no data source id
  apiRouter.post('/detectors/_search', adService.searchDetector);

  // post search anomaly results
  apiRouter.post('/detectors/results/_search', adService.searchResults);
  apiRouter.post('/detectors/results/_search/{dataSourceId}', adService.searchResults);
  apiRouter.post('/detectors/results/_search/{resultIndex}/{onlyQueryCustomResultIndex}', adService.searchResults);
  apiRouter.post('/detectors/results/_search/{resultIndex}/{onlyQueryCustomResultIndex}/{dataSourceId}', adService.searchResults);

  // list detectors
  apiRouter.get('/detectors/_list', adService.getDetectors);
  apiRouter.get('/detectors/_list/{dataSourceId}', adService.getDetectors);

  // preview detector
  apiRouter.post('/detectors/preview', adService.previewDetector);
  apiRouter.post('/detectors/preview/{dataSourceId}', adService.previewDetector);

  // get detector anomaly results
  apiRouter.get('/detectors/{id}/results/{isHistorical}/{resultIndex}/{onlyQueryCustomResultIndex}', adService.getAnomalyResults);
  apiRouter.get('/detectors/{id}/results/{isHistorical}/{resultIndex}/{onlyQueryCustomResultIndex}/{dataSourceId}', adService.getAnomalyResults);
  apiRouter.get('/detectors/{id}/results/{isHistorical}', adService.getAnomalyResults);
  apiRouter.get('/detectors/{id}/results/{isHistorical}/{dataSourceId}', adService.getAnomalyResults);

  // delete detector
  apiRouter.delete('/detectors/{detectorId}', adService.deleteDetector);
  apiRouter.delete('/detectors/{detectorId}/{dataSourceId}', adService.deleteDetector);

  // start detector
  apiRouter.post('/detectors/{detectorId}/start', adService.startDetector);
  apiRouter.post('/detectors/{detectorId}/start/{dataSourceId}', adService.startDetector);

  // stop detector
  apiRouter.post('/detectors/{detectorId}/stop/{isHistorical}', adService.stopDetector);
  apiRouter.post('/detectors/{detectorId}/stop/{isHistorical}/{dataSourceId}', adService.stopDetector);
  apiRouter.get('/detectors/{detectorId}/_profile', adService.getDetectorProfile);

  // get detector
  apiRouter.get('/detectors/{detectorId}', adService.getDetector);
  apiRouter.get('/detectors/{detectorId}/{dataSourceId}', adService.getDetector);

  // match detector
  apiRouter.get('/detectors/{detectorName}/_match', adService.matchDetector);
  apiRouter.get('/detectors/{detectorName}/_match/{dataSourceId}', adService.matchDetector);

  // get detector count
  apiRouter.get('/detectors/_count', adService.getDetectorCount);
  apiRouter.get('/detectors/_count/{dataSourceId}', adService.getDetectorCount);

  // post get top anomaly results
  apiRouter.post('/detectors/{detectorId}/_topAnomalies/{isHistorical}', adService.getTopAnomalyResults);
  apiRouter.post('/detectors/{detectorId}/_topAnomalies/{isHistorical}/{dataSourceId}', adService.getTopAnomalyResults);

  // validate detector
  apiRouter.post('/detectors/_validate/{validationType}', adService.validateDetector);
  apiRouter.post('/detectors/_validate/{validationType}/{dataSourceId}', adService.validateDetector);
}
class AdService {
  constructor(client, dataSourceEnabled) {
    _defineProperty(this, "client", void 0);
    _defineProperty(this, "dataSourceEnabled", void 0);
    _defineProperty(this, "deleteDetector", async (context, request, opensearchDashboardsResponse) => {
      try {
        const {
          detectorId
        } = request.params;
        const {
          dataSourceId = ''
        } = request.params;
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        const response = await callWithRequest('ad.deleteDetector', {
          detectorId
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: response
          }
        });
      } catch (err) {
        console.log('Anomaly detector - deleteDetector', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "previewDetector", async (context, request, opensearchDashboardsResponse) => {
      try {
        const {
          dataSourceId = ''
        } = request.params;
        const requestBody = JSON.stringify((0, _adHelpers.convertPreviewInputKeysToSnakeCase)(request.body));
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        const response = await callWithRequest('ad.previewDetector', {
          body: requestBody
        });
        const transformedKeys = (0, _helpers.mapKeysDeep)(response, _helpers.toCamel);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            //@ts-ignore
            response: (0, _adHelpers.anomalyResultMapper)(transformedKeys.anomalyResult)
          }
        });
      } catch (err) {
        console.log('Anomaly detector - previewDetector', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "putDetector", async (context, request, opensearchDashboardsResponse) => {
      try {
        const {
          detectorId
        } = request.params;
        const {
          dataSourceId = ''
        } = request.params;

        //@ts-ignore
        const ifSeqNo = request.body.seqNo;
        //@ts-ignore
        const ifPrimaryTerm = request.body.primaryTerm;
        const requestBody = JSON.stringify((0, _adHelpers.convertDetectorKeysToSnakeCase)(request.body));
        let params = {
          detectorId: detectorId,
          ifSeqNo: ifSeqNo,
          ifPrimaryTerm: ifPrimaryTerm,
          body: requestBody
        };
        let response;
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        if ((0, _lodash.isNumber)(ifSeqNo) && (0, _lodash.isNumber)(ifPrimaryTerm)) {
          response = await callWithRequest('ad.updateDetector', params);
        } else {
          response = await callWithRequest('ad.createDetector', {
            body: params.body
          });
        }
        const resp = {
          ...response.anomaly_detector,
          id: response._id,
          primaryTerm: response._primary_term,
          seqNo: response._seq_no
        };
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: (0, _adHelpers.convertDetectorKeysToCamelCase)(resp)
          }
        });
      } catch (err) {
        console.log('Anomaly detector - PutDetector', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "validateDetector", async (context, request, opensearchDashboardsResponse) => {
      try {
        let {
          validationType
        } = request.params;
        const {
          dataSourceId = ''
        } = request.params;
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        const requestBody = JSON.stringify((0, _adHelpers.convertPreviewInputKeysToSnakeCase)(request.body));
        const response = await callWithRequest('ad.validateDetector', {
          body: requestBody,
          validationType: validationType
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: response
          }
        });
      } catch (err) {
        console.log('Anomaly detector - validateDetector', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "getDetector", async (context, request, opensearchDashboardsResponse) => {
      try {
        const {
          detectorId
        } = request.params;
        const {
          dataSourceId = ''
        } = request.params;
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        const detectorResponse = await callWithRequest('ad.getDetector', {
          detectorId
        });

        // Populating static detector fields
        const staticFields = {
          id: detectorResponse._id,
          primaryTerm: detectorResponse._primary_term,
          seqNo: detectorResponse._seq_no,
          ...(0, _adHelpers.convertStaticFieldsToCamelCase)(detectorResponse.anomaly_detector)
        };

        // Get real-time and historical task info to populate the
        // task and job-related fields
        // We wrap these calls in a try/catch, and suppress any index_not_found_exceptions
        // which can occur if no detector jobs have been ran on a new cluster.
        let realtimeTasksResponse = {};
        let historicalTasksResponse = {};
        try {
          const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
          realtimeTasksResponse = await callWithRequest('ad.searchTasks', {
            body: (0, _adHelpers.getLatestTaskForDetectorQuery)(detectorId, true)
          });
          historicalTasksResponse = await callWithRequest('ad.searchTasks', {
            body: (0, _adHelpers.getLatestTaskForDetectorQuery)(detectorId, false)
          });
        } catch (err) {
          if (!(0, _adHelpers.isIndexNotFoundError)(err)) {
            throw err;
          }
        }
        const realtimeTask = (0, _lodash.get)((0, _lodash.get)(realtimeTasksResponse, 'hits.hits', []).map(taskResponse => {
          return {
            id: (0, _lodash.get)(taskResponse, '_id'),
            ...(0, _lodash.get)(taskResponse, '_source')
          };
        }), 0);
        const historicalTask = (0, _lodash.get)((0, _lodash.get)(historicalTasksResponse, 'hits.hits', []).map(taskResponse => {
          return {
            id: (0, _lodash.get)(taskResponse, '_id'),
            ...(0, _lodash.get)(taskResponse, '_source')
          };
        }), 0);
        const taskAndJobFields = (0, _adHelpers.convertTaskAndJobFieldsToCamelCase)(realtimeTask, historicalTask, detectorResponse.anomaly_detector_job);

        // Combine the static and task-and-job-related fields into
        // a final response
        const finalResponse = {
          ...staticFields,
          ...taskAndJobFields
        };
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: finalResponse
          }
        });
      } catch (err) {
        console.log('Anomaly detector - Unable to get detector', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "startDetector", async (context, request, opensearchDashboardsResponse) => {
      try {
        var _request$body, _request$body2;
        const {
          detectorId
        } = request.params;
        const {
          dataSourceId = ''
        } = request.params;
        //@ts-ignore
        const startTime = (_request$body = request.body) === null || _request$body === void 0 ? void 0 : _request$body.startTime;
        //@ts-ignore
        const endTime = (_request$body2 = request.body) === null || _request$body2 === void 0 ? void 0 : _request$body2.endTime;
        let requestParams = {
          detectorId: detectorId
        };
        let requestPath = 'ad.startDetector';
        // If a start and end time are passed: we want to start a historical analysis
        if ((0, _lodash.isNumber)(startTime) && (0, _lodash.isNumber)(endTime)) {
          requestParams = {
            ...requestParams,
            body: {
              start_time: startTime,
              end_time: endTime
            }
          };
          requestPath = 'ad.startHistoricalDetector';
        }
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        const response = await callWithRequest(requestPath, requestParams);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: response
          }
        });
      } catch (err) {
        console.log('Anomaly detector - startDetector', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "stopDetector", async (context, request, opensearchDashboardsResponse) => {
      try {
        let {
          detectorId,
          isHistorical
        } = request.params;
        const {
          dataSourceId = ''
        } = request.params;
        isHistorical = JSON.parse(isHistorical);
        const requestPath = isHistorical ? 'ad.stopHistoricalDetector' : 'ad.stopDetector';
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        const response = await callWithRequest(requestPath, {
          detectorId
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: response
          }
        });
      } catch (err) {
        console.log('Anomaly detector - stopDetector', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "getDetectorProfile", async (context, request, opensearchDashboardsResponse) => {
      try {
        const {
          detectorId
        } = request.params;
        const response = await this.client.asScoped(request).callAsCurrentUser('ad.detectorProfile', {
          detectorId
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response
          }
        });
      } catch (err) {
        console.log('Anomaly detector - detectorProfile', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "searchDetector", async (context, request, opensearchDashboardsResponse) => {
      try {
        const requestBody = JSON.stringify(request.body);
        const response = await this.client.asScoped(request).callAsCurrentUser('ad.searchDetector', {
          body: requestBody
        });
        const totalDetectors = (0, _lodash.get)(response, 'hits.total.value', 0);
        const detectors = (0, _lodash.get)(response, 'hits.hits', []).map(detector => ({
          ...(0, _adHelpers.convertDetectorKeysToCamelCase)(detector._source),
          id: detector._id,
          seqNo: detector._seq_no,
          primaryTerm: detector._primary_term
        }));
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: {
              totalDetectors,
              detectors
            }
          }
        });
      } catch (err) {
        console.log('Anomaly detector - Unable to search detectors', err);
        if ((0, _adHelpers.isIndexNotFoundError)(err)) {
          return opensearchDashboardsResponse.ok({
            body: {
              ok: true,
              response: {
                totalDetectors: 0,
                detectors: []
              }
            }
          });
        }
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "searchResults", async (context, request, opensearchDashboardsResponse) => {
      try {
        var {
          resultIndex,
          onlyQueryCustomResultIndex
        } = request.params;
        const {
          dataSourceId = ''
        } = request.params;
        if (!resultIndex || !resultIndex.startsWith(_constants.CUSTOM_AD_RESULT_INDEX_PREFIX)) {
          // Set resultIndex as '' means no custom result index specified, will only search anomaly result from default index.
          resultIndex = '';
        }
        let requestParams = {
          resultIndex: resultIndex,
          onlyQueryCustomResultIndex: onlyQueryCustomResultIndex
        };
        const requestBody = JSON.stringify(request.body);
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        const response = !resultIndex ? await callWithRequest('ad.searchResults', {
          body: requestBody
        }) : await callWithRequest('ad.searchResultsFromCustomResultIndex', {
          ...requestParams,
          body: requestBody
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response
          }
        });
      } catch (err) {
        console.log('Anomaly detector - Unable to search anomaly result', err);
        if ((0, _adHelpers.isIndexNotFoundError)(err)) {
          return opensearchDashboardsResponse.ok({
            body: {
              ok: true,
              response: {
                totalDetectors: 0,
                detectors: []
              }
            }
          });
        }
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "getDetectors", async (context, request, opensearchDashboardsResponse) => {
      console.log('getDetectors request', request);
      try {
        const {
          from = 0,
          size = 20,
          search = '',
          indices = '',
          sortDirection = _constants.SORT_DIRECTION.DESC,
          sortField = 'name',
          dataSourceId = ''
        } = request.query;
        const mustQueries = [];
        if (search.trim()) {
          mustQueries.push({
            query_string: {
              fields: ['name', 'description'],
              default_operator: 'AND',
              query: `*${search.trim().split('-').join('* *')}*`
            }
          });
        }
        if (indices.trim()) {
          mustQueries.push({
            query_string: {
              fields: ['indices'],
              default_operator: 'AND',
              query: `*${indices.trim().split('-').join('* *')}*`
            }
          });
        }
        //Allowed sorting columns
        const sortQueryMap = {
          name: {
            'name.keyword': sortDirection
          },
          indices: {
            'indices.keyword': sortDirection
          },
          lastUpdateTime: {
            last_update_time: sortDirection
          }
        };
        let sort = {};
        const sortQuery = sortQueryMap[sortField];
        if (sortQuery) {
          sort = sortQuery;
        }
        //Preparing search request
        const requestBody = {
          sort,
          size,
          from,
          query: {
            bool: {
              must: mustQueries
            }
          }
        };
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        console.log('requestBody', requestBody);
        const response = await callWithRequest('ad.searchDetector', {
          body: requestBody
        });
        const totalDetectors = (0, _lodash.get)(response, 'hits.total.value', 0);

        //Get all detectors from search detector API
        const allDetectors = (0, _lodash.get)(response, 'hits.hits', []).reduce((acc, detectorResponse) => ({
          ...acc,
          [detectorResponse._id]: {
            id: detectorResponse._id,
            primaryTerm: detectorResponse._primary_term,
            seqNo: detectorResponse._seq_no,
            ...(0, _adHelpers.convertStaticFieldsToCamelCase)(detectorResponse._source)
          }
        }), {});
        const allDetectorsMap = Object.values(allDetectors).reduce((acc, detector) => ({
          ...acc,
          [detector.id]: detector
        }), {});

        //Given each detector from previous result, get aggregation to power list
        const allDetectorIds = Object.keys(allDetectorsMap);
        let requestParams = {
          // If specifying result index, will query anomaly result from both default and custom result indices.
          // If no valid result index specified, just query anomaly result from default result index.
          // Here we specify custom AD result index prefix pattern to query all custom result indices.
          resultIndex: _constants.CUSTOM_AD_RESULT_INDEX_PREFIX + '*',
          onlyQueryCustomResultIndex: 'false'
        };
        const aggregationResult = await callWithRequest('ad.searchResultsFromCustomResultIndex', {
          ...requestParams,
          body: (0, _adHelpers.getResultAggregationQuery)(allDetectorIds, {
            from,
            size,
            sortField,
            sortDirection,
            search,
            indices
          })
        });
        const aggsDetectors = (0, _lodash.get)(aggregationResult, 'aggregations.unique_detectors.buckets', []).reduce((acc, agg) => {
          return {
            ...acc,
            [agg.key]: {
              ...allDetectorsMap[agg.key],
              totalAnomalies: agg.total_anomalies_in_24hr.doc_count,
              lastActiveAnomaly: agg.latest_anomaly_time.value
            }
          };
        }, {});

        // Aggregation will not return values where anomalies for detectors are not generated, loop through it and fill values with 0
        const unUsedDetectors = (0, _lodash.pullAll)(allDetectorIds, Object.keys(aggsDetectors)).reduce((acc, unusedDetector) => {
          return {
            ...acc,
            [unusedDetector]: {
              ...allDetectorsMap[unusedDetector],
              totalAnomalies: 0,
              lastActiveAnomaly: 0
            }
          };
        }, {});

        // If sorting criteria is from the aggregation manage pagination in memory.
        let finalDetectors = (0, _lodash.orderBy)({
          ...aggsDetectors,
          ...unUsedDetectors
        }, [sortField], [sortDirection]);
        if (!sortQueryMap[sortField]) {
          finalDetectors = Object.values(finalDetectors).slice(from, from + size).reduce((acc, detector) => ({
            ...acc,
            [detector.id]: detector
          }), {});
        }

        // Fetch the latest realtime and historical tasks for all detectors
        // using terms aggregations
        // We wrap these calls in a try/catch, and suppress any index_not_found_exceptions
        // which can occur if no detector jobs have been ran on a new cluster.
        let realtimeTasksResponse = {};
        let historicalTasksResponse = {};
        try {
          realtimeTasksResponse = await callWithRequest('ad.searchTasks', {
            body: (0, _adHelpers.getLatestDetectorTasksQuery)(true)
          });
          historicalTasksResponse = await callWithRequest('ad.searchTasks', {
            body: (0, _adHelpers.getLatestDetectorTasksQuery)(false)
          });
        } catch (err) {
          if (!(0, _adHelpers.isIndexNotFoundError)(err)) {
            throw err;
          }
        }
        const realtimeTasks = (0, _lodash.get)(realtimeTasksResponse, 'aggregations.detectors.buckets', []).reduce((acc, bucket) => {
          return {
            ...acc,
            [bucket.key]: {
              realtimeTask: (0, _lodash.get)(bucket, 'latest_tasks.hits.hits.0', undefined)
            }
          };
        }, {});
        const historicalTasks = (0, _lodash.get)(historicalTasksResponse, 'aggregations.detectors.buckets', []).reduce((acc, bucket) => {
          return {
            ...acc,
            [bucket.key]: {
              historicalTask: (0, _lodash.get)(bucket, 'latest_tasks.hits.hits.0', undefined)
            }
          };
        }, {});

        // Get real-time and historical task info by looping through each detector & retrieving
        //    - curState by getting real-time task state
        //    - enabledTime by getting real-time task's execution_start time
        //    - taskId by getting historical task's _id
        finalDetectors.forEach(detector => {
          const realtimeTask = (0, _lodash.get)(realtimeTasks[detector.id], 'realtimeTask._source');
          detector.curState = (0, _adHelpers.getTaskState)(realtimeTask);
          detector.enabledTime = (0, _lodash.get)(realtimeTask, 'execution_start_time');
          detector.taskId = (0, _lodash.get)(historicalTasks[detector.id], 'historicalTask._id');
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: {
              totalDetectors,
              detectorList: Object.values(finalDetectors)
            }
          }
        });
      } catch (err) {
        console.log('Anomaly detector - Unable to search detectors', err);
        if ((0, _adHelpers.isIndexNotFoundError)(err)) {
          return opensearchDashboardsResponse.ok({
            body: {
              ok: true,
              response: {
                totalDetectors: 0,
                detectorList: []
              }
            }
          });
        }
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "getAnomalyResults", async (context, request, opensearchDashboardsResponse) => {
      let {
        id,
        isHistorical,
        resultIndex,
        onlyQueryCustomResultIndex
      } = request.params;
      const {
        dataSourceId = ''
      } = request.params;
      if (!resultIndex || !resultIndex.startsWith(_constants.CUSTOM_AD_RESULT_INDEX_PREFIX)) {
        // Set resultIndex as '' means no custom result index specified, will only search anomaly result from default index.
        resultIndex = '';
      }
      isHistorical = JSON.parse(isHistorical);

      // Search by task id if historical, or by detector id if realtime
      const searchTerm = isHistorical ? {
        task_id: id
      } : {
        detector_id: id
      };
      try {
        const {
          from = 0,
          size = 20,
          sortDirection = _constants.SORT_DIRECTION.DESC,
          sortField = _constants.AD_DOC_FIELDS.DATA_START_TIME,
          startTime = 0,
          endTime = 0,
          fieldName = '',
          anomalyThreshold = -1,
          entityList = ''
        } = request.query;
        const entityListAsObj = entityList.length === 0 ? {} : JSON.parse(entityList);
        const entityFilters = (0, _lodash.isEmpty)(entityListAsObj) ? [] : (0, _adHelpers.getFiltersFromEntityList)(entityListAsObj);

        //Allowed sorting columns
        const sortQueryMap = {
          anomalyGrade: {
            anomaly_grade: sortDirection
          },
          confidence: {
            confidence: sortDirection
          },
          [_constants.AD_DOC_FIELDS.DATA_START_TIME]: {
            [_constants.AD_DOC_FIELDS.DATA_START_TIME]: sortDirection
          },
          [_constants.AD_DOC_FIELDS.DATA_END_TIME]: {
            [_constants.AD_DOC_FIELDS.DATA_END_TIME]: sortDirection
          }
        };
        let sort = {};
        const sortQuery = sortQueryMap[sortField];
        if (sortQuery) {
          sort = sortQuery;
        }

        //Preparing search request
        const requestBody = {
          sort,
          size,
          from,
          query: {
            bool: {
              filter: [{
                term: searchTerm
              }, {
                range: {
                  anomaly_grade: {
                    gt: anomalyThreshold
                  }
                }
              }, ...entityFilters]
            }
          }
        };

        // If querying RT results: remove any results that include a task_id, as this indicates
        // a historical result from a historical task.
        if (!isHistorical) {
          requestBody.query.bool = {
            ...requestBody.query.bool,
            ...{
              must_not: {
                exists: {
                  field: 'task_id'
                }
              }
            }
          };
        }
        try {
          const filterSize = requestBody.query.bool.filter.length;
          if (fieldName) {
            (startTime || endTime) && (0, _lodash.set)(requestBody.query.bool.filter, `${filterSize}.range.${fieldName}.format`, 'epoch_millis');
            startTime && (0, _lodash.set)(requestBody.query.bool.filter, `${filterSize}.range.${fieldName}.gte`, startTime);
            endTime && (0, _lodash.set)(requestBody.query.bool.filter, `${filterSize}.range.${fieldName}.lte`, endTime);
          }
        } catch (error) {
          console.log('wrong date range filter', error);
        }
        let requestParams = {
          resultIndex: resultIndex,
          onlyQueryCustomResultIndex: onlyQueryCustomResultIndex
        };
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        const response = !resultIndex ? await callWithRequest('ad.searchResults', {
          body: requestBody
        }) : await callWithRequest('ad.searchResultsFromCustomResultIndex', {
          ...requestParams,
          body: requestBody
        });
        const totalResults = (0, _lodash.get)(response, 'hits.total.value', 0);
        const detectorResult = [];
        const featureResult = {};
        (0, _lodash.get)(response, 'hits.hits', []).forEach(result => {
          detectorResult.push({
            startTime: result._source.data_start_time,
            endTime: result._source.data_end_time,
            plotTime: result._source.data_end_time,
            contributions: result._source.anomaly_grade > 0 ? result._source.relevant_attribution : {},
            confidence: result._source.confidence != null && result._source.confidence !== 'NaN' && result._source.confidence > 0 ? (0, _helpers.toFixedNumberForAnomaly)(Number.parseFloat(result._source.confidence)) : 0,
            anomalyGrade: result._source.anomaly_grade != null && result._source.anomaly_grade !== 'NaN' && result._source.anomaly_grade > 0 ? (0, _helpers.toFixedNumberForAnomaly)(Number.parseFloat(result._source.anomaly_grade)) : 0,
            ...(result._source.entity != null ? {
              entity: result._source.entity
            } : {}),
            // TODO: we should refactor other places to read feature data from
            // AnomalyResult, instead of having separate FeatureData which is hard
            // to know feature data belongs to which anomaly result
            features: this.getFeatureData(result)
          });
          result._source.feature_data.forEach(featureData => {
            if (!featureResult[featureData.feature_id]) {
              featureResult[featureData.feature_id] = [];
            }
            featureResult[featureData.feature_id].push({
              startTime: result._source.data_start_time,
              endTime: result._source.data_end_time,
              plotTime: result._source.data_end_time,
              data: featureData.data != null && featureData.data !== 'NaN' ? (0, _helpers.toFixedNumberForAnomaly)(Number.parseFloat(featureData.data)) : 0,
              name: featureData.feature_name,
              expectedValue: this.getExpectedValue(result, featureData)
            });
          });
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: {
              totalAnomalies: totalResults,
              results: detectorResult,
              featureResults: featureResult
            }
          }
        });
      } catch (err) {
        console.log('Anomaly detector - Unable to get results', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "getTopAnomalyResults", async (context, request, opensearchDashboardsResponse) => {
      try {
        let {
          detectorId,
          isHistorical
        } = request.params;
        const {
          dataSourceId = ''
        } = request.params;
        isHistorical = JSON.parse(isHistorical);
        const requestPath = isHistorical ? 'ad.topHistoricalAnomalyResults' : 'ad.topAnomalyResults';
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        const response = await callWithRequest(requestPath, {
          detectorId: detectorId,
          body: request.body
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: response
          }
        });
      } catch (err) {
        console.log('Anomaly detector - getTopAnomalyResults', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "matchDetector", async (context, request, opensearchDashboardsResponse) => {
      try {
        const {
          detectorName
        } = request.params;
        const {
          dataSourceId = ''
        } = request.params;
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        const response = await callWithRequest('ad.matchDetector', {
          detectorName
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: response
          }
        });
      } catch (err) {
        console.log('Anomaly detector - matchDetector', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "getDetectorCount", async (context, request, opensearchDashboardsResponse) => {
      try {
        const {
          dataSourceId = ''
        } = request.params;
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        const response = await callWithRequest('ad.detectorCount');
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: response
          }
        });
      } catch (err) {
        console.log('Anomaly detector - getDetectorCount', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "getFeatureData", rawResult => {
      const featureResult = {};
      rawResult._source.feature_data.forEach(featureData => {
        featureResult[featureData.feature_id] = {
          startTime: rawResult._source.data_start_time,
          endTime: rawResult._source.data_end_time,
          plotTime: rawResult._source.data_end_time,
          data: featureData.data != null && featureData.data !== 'NaN' ? (0, _helpers.toFixedNumberForAnomaly)(Number.parseFloat(featureData.data)) : 0,
          name: featureData.feature_name,
          expectedValue: this.getExpectedValue(rawResult, featureData)
        };
      });
      return featureResult;
    });
    _defineProperty(this, "getExpectedValue", (rawResult, featureData) => {
      let expectedValue = featureData.data != null && featureData.data !== 'NaN' ? (0, _helpers.toFixedNumberForAnomaly)(Number.parseFloat(featureData.data)) : 0;
      if (rawResult._source.anomaly_grade > 0) {
        const expectedValueList = rawResult._source.expected_values;
        if ((expectedValueList === null || expectedValueList === void 0 ? void 0 : expectedValueList.length) > 0) {
          expectedValueList[0].value_list.forEach(expect => {
            if (expect.feature_id === featureData.feature_id) {
              expectedValue = expect.data;
            }
          });
        }
      }
      return expectedValue;
    });
    this.client = client;
    this.dataSourceEnabled = dataSourceEnabled;
  }
}
exports.default = AdService;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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