///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
//
// geo tetgen inputs / outputs
//
// author:
//      Pierre.Saramito@imag.fr
//
// date: 23 nov 2009
//

// ============================================================================
//  includes
// ============================================================================

#include "rheolef/georep.h"
#include "rheolef/iorheo.h"
#include "rheolef/rheostream.h" // i/o utility
#include "rheolef/field.h"
#include "rheolef/form.h"
#include "rheolef/tiny_element.h"
using namespace std;
namespace rheolef { 


// ============================================================================
// tetgen input (3d)
// use as input the concatenation of 3 files : .node .ele .face
// example:
// cat toto.node toto.elt toto.face | sed -e 's/#.*$/' | 
//		geo -input-tetgen -upgrade -geo - > toto.geo
// Note: strip comments lines as "#comments"
// ============================================================================

static
void
get_nodes_tetgen (
	istream& s, 
    	georep::iterator_node p, 
	size_t np,
  	size_t n_vert_attr,
	size_t n_bdr_mark,
    	point& xmin, 
	point& xmax)
{
  typedef georep::size_type size_type;
  xmin[0] = xmin[1] = xmin[2] =  numeric_limits<Float>::max();
  xmax[0] = xmax[1] = xmax[2] = -numeric_limits<Float>::max();
  size_t vertice_number = 0;
  for (size_t i = 0; i < np; i++) {
    // <point #> <x> <y> <z> [attributes] [boundary marker]
    size_t idx;
    s >> idx; idx--;
    check_macro (idx < np, "node index "<<idx+1<<" is out of range 1:"<<np);
    s >> p[idx][0] >> p[idx][1] >> p[idx][2];
    // skip 0d domains
    size_t dummy;
    for (size_t j = 0; j < n_vert_attr + n_bdr_mark; j++) {
      s >> dummy;
    }
  }
}
//
// get boundary triangles
//
static
void
get_boundary_triangles_tetgen (istream& is,
    size_t n_bdr_triangles, bool has_face_attr, size_t n_vert,
    vector<domain>& domains,
    vector<size_t>& triangle_domain_tetgen_number)
{
  map<size_t, vector<tiny_element> > domain_map;
  tiny_element S;
  S.set_variant (reference_element::t);
  for (size_t j = 0; is.good() && j < n_bdr_triangles; j++) {
    size_t S_idx;
    is >> S_idx; S_idx--;
    for (size_t i = 0 ; i < 3; i++) {
      is >> S[i]; S[i]--;
      check_macro (S[i] < n_vert, "tetgen input: face " << S_idx+1 << ": "
	<< i+1 << "-th vertice index "
        << S[i]+1 << " out of range 1:" << n_vert);
    }
    size_t r = 0;
    if (has_face_attr) is >> r;
    domain_map[r].push_back (S);
  }
  check_macro (is, "a problem occurs while loading a tetgen mesh");

  // copy domain_map into domains (table)
  triangle_domain_tetgen_number.resize (domain_map.size()); 
  domains.resize (domain_map.size());
  vector<domain>::iterator iter_domain = domains.begin();
  size_t i_dom = 0;
  for (map<size_t, vector<tiny_element> >::const_iterator iter_domain_map = domain_map.begin();
      iter_domain_map != domain_map.end(); iter_domain_map++, iter_domain++, i_dom++) {
      triangle_domain_tetgen_number[i_dom] = (*iter_domain_map).first;
      const vector<tiny_element>& dom = (*iter_domain_map).second;
      if (dom.size() == 0) continue;
      (*iter_domain).set(dom.begin(), dom.size(), 2, "unnamed"+itos(i_dom));
  }
}
istream&
georep::get_tetgen (istream& s, bool check_optional_domain_name) 
{
  // tetgen has only boundary edges that are numbered
  // and have domain numbers 
  // => this is only a version 1 mesh file format type
  //    not consistent for P2 elements
  _version = 1;

  typedef georep::size_type size_type;
  check_macro (s.good(), "bad input stream for tetgen.");
  bool verbose = iorheo::getverbose(s);
  //
  // get coordinates
  // header: <# of points> <dimension (must be 3)> <# of attributes> <# of boundary markers (0 or 1)>
  //
  size_type n_vert, n_vert_attr, n_bdr_mark;
  s >> n_vert >>_dim >> n_vert_attr >> n_bdr_mark;
  check_macro (s, "bad input stream for tetgen.");
  check_macro (_dim == 3, "bad dimension "<<_dim<<" for tetgen: expect 3");
  _x.resize(n_vert);
  get_nodes_tetgen (s, begin_node(), n_vert, n_vert_attr, n_bdr_mark, _xmin, _xmax);
  check_macro (s, "bad input stream for tetgen.");
  _count_geo [0] = _count_element [0] = n_vert;
  //
  // get elements
  // header: <# of tetrahedra> <nodes per tetrahedron> <# of attributes> 
  // lines : <tetrahedron #> <node> <node> <node> <node> ... [attributes]
  size_type n_elt, n_vert_per_elt, n_tetra_attr;
  s >> n_elt >> n_vert_per_elt >> n_tetra_attr;
  check_macro (n_vert_per_elt == 4, "bad #node per tetra "<<n_vert_per_elt<<" for tetgen: expect 4");
  resize(n_elt);
  iterator elt = begin();
  for (size_t j = 0; j < n_elt; j++) {
    size_t K_idx;
    s >> K_idx; K_idx--;
    check_macro (K_idx < n_elt, "element index "<<K_idx+1<<" is out of range 1:"<<n_elt);
    geo_element& K = elt[K_idx];
    K.set_variant (n_vert_per_elt, _dim);
    K.set_index(K_idx) ;
    for (size_type i = 0 ; i < n_vert_per_elt; i++) {
      s >> K[i]; K[i]--;
      check_macro (K[i] < n_vert, "tetgen input: element " << K_idx+1 << ": "
	<< i+1 << "-th vertice index "
        << K[i]+1 << " out of range 1:" << n_vert);
    }
    // skip 3d domains
    size_t dummy;
    for (size_t j = 0; j < n_tetra_attr; j++) {
      s >> dummy;
    }
  }
  // read triangular faces:
  // First line: <# of faces> <boundary marker (0 or 1)>
  // Remaining lines list of # of faces:
  // <face #> <node> <node> <node> [boundary marker]
  size_t n_face, bdry_mark;
  s >> n_face >> bdry_mark;
  bool has_face_attr = (bdry_mark == 1);
  vector<domain> triangle_domain;
  vector<size_t> triangle_domain_tetgen_number;
  get_boundary_triangles_tetgen (s, n_face, has_face_attr, n_vert,
	triangle_domain, triangle_domain_tetgen_number);

  // ---------------------------------------------------------------
  // check extension to optional domain names
  // ---------------------------------------------------------------
  if (!check_optional_domain_name) return s;
  char c;
  s >> ws >> c; // skip white and grab a char
  // have "FaceDomainNames", "EdgeDomainNames" or "VerticeDomainNames" ?
  // tetgen mesh may be followed by field data and such, so be carrefull...
  while (c == 'E' || c == 'F' || c == 'V') {
      s.unget(); // put char back
      if (c == 'F') {
        if (!scatch(s,"FaceDomainNames")) return s;
        // ---------------------------------------------------------------
        // get triangle domain names
        // ---------------------------------------------------------------
        size_type n_dom_triangle;
        s >> n_dom_triangle;
        if (n_dom_triangle != triangle_domain.size()) {
           string numbers;
           for (size_t j = 0; j < triangle_domain_tetgen_number.size(); j++) {
              numbers = numbers + itos(triangle_domain_tetgen_number[j]) + " ";
           }
           warning_macro("geo input: "
	    << n_dom_triangle << " domain names founded while "
            << triangle_domain.size() << " tetgen triangle domains numbers are defined: "
            << numbers);
           cerr << endl;
           error_macro("HINT: check domain name file (.dmn)");
        }
        size_t start_triangle_domain = _domlist.size();
        _domlist.resize(_domlist.size() + n_dom_triangle);
        for (size_type k = 0; k < n_dom_triangle; k++) {
          string name;
          s >> name;
          triangle_domain[k].set_name(name);
          _domlist.at(start_triangle_domain+k) = triangle_domain[k];
        }
      } else if (c == 'E') {
        if (!scatch(s,"EdgeDomainNames")) return s;
        // ---------------------------------------------------------------
        // get edge domain names
        // ---------------------------------------------------------------
	warning_macro ("EdgeDomainNames skiped");
        size_type n_dom_edge;
        s >> n_dom_edge;
        for (size_type k = 0; k < n_dom_edge; k++) {
          string name;
          s >> name;
        }
      } else if (c == 'V') {
        if (!scatch(s,"VerticeDomainNames")) return s;
        // ---------------------------------------------------------------
        // get vertice domain names
        // ---------------------------------------------------------------
	warning_macro ("VerticeDomainNames skiped");
        size_type n_dom_vertice;
        s >> n_dom_vertice;
        for (size_type k = 0; k < n_dom_vertice; k++) {
          string name;
          s >> name;
        }
      }
      s >> ws >> c; // skip white and grab a char
  }
  return s;
}
// =======================================================================
// tetgen output
// =======================================================================
ostream&
georep::put_tetgen (ostream& os, bool only_boundary) const
{
  fatal_macro ("put_tetgen: not yet!");
  return os;
}
}// namespace rheolef
