/* NVTV Chrontel TV-I2C access -- Dirk Thierbach <dthierbach@gmx.de>
 *
 * This file is part of nvtv, a tool for tv-output on NVidia cards.
 * 
 * nvtv is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * nvtv is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA
 *
 * $Id: tv_ch1_7007.c,v 1.9 2004/01/30 08:47:02 dthierbach Exp $
 *
 * Contents:
 *
 * Routines to access the Chrontel chip registers via the I2C bus.
 *
 */

#include "local.h" /* before everything else */
#include "xf86_ansic.h" 

#include "xf86i2c.h"
#include "bitmask.h"
#include "tv_i2c.h"
#include "tv_ch1_7007.h"

/* -------- Chrontel -------- */

static I2CByte nvChDefMacro [0x40] = {
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
};

static I2CByte nvChPalMacro00APS1 [0x40] = {
  0x00,0x80,0x6C,0x00,0x00,0x2E,0x00,0x00,
  0x18,0x00,0x00,0x00,0x33,0x80,0x40,0x22,
  0xE0,0x00,0x00,0x40,0x00,0x00,0x38,0x00,
  0x20,0xA0,0x20,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x40,0x10,0x9F,0x9F,0xBA,0x9B,
  0x64,0x0A,0xEE,0x26,0x2A,0x7E,0x9A,0x3F,
  0xE0,0x15,0x40,0xFE,0x7E,0x05,0x00,0x00,
};
static I2CByte nvChPalMacro08APS1 [0x40] = {
  0x00,0x80,0x6C,0x00,0x00,0x37,0x00,0x00,
  0x18,0x00,0x00,0x00,0x33,0x80,0x40,0x22,
  0xE0,0x00,0x00,0x40,0x00,0x00,0x38,0x00,
  0x20,0xA0,0x20,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x4C,0x30,0xBD,0xBD,0xDE,0x9B,
  0x64,0x3C,0x1B,0x2E,0x32,0x96,0xB7,0x3F,
  0xE0,0x15,0x40,0xFE,0x7E,0x05,0x00,0x00,
};
static I2CByte nvChPalMacro14APS1 [0x40] = {
  0x00,0x80,0x6C,0x00,0x00,0x3A,0x00,0x00,
  0x18,0x00,0x00,0x00,0x33,0x80,0x40,0x22,
  0xE0,0x00,0x00,0x40,0x00,0x00,0x38,0x00,
  0x20,0xA0,0x20,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x50,0x30,0xC8,0xC8,0xEA,0x9B,
  0x64,0x4C,0x29,0x30,0x34,0x9D,0xC0,0x3F,
  0xE0,0x15,0x40,0xFE,0x7E,0x05,0x00,0x00,
};
static I2CByte nvChPalMacro20APS1 [0x40] = {
  0x00,0x80,0x6C,0x00,0x00,0x47,0x00,0x00,
  0x18,0x00,0x00,0x00,0x33,0x80,0x40,0x22,
  0xE0,0x00,0x00,0x40,0x00,0x00,0x38,0x00,
  0x20,0xA0,0x20,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x61,0x38,0xF3,0xF3,0x1C,0x9B,
  0x64,0x94,0x6A,0x3B,0x40,0xC0,0xEA,0x3F,
  0xE0,0x15,0x40,0xFE,0x7E,0x05,0x00,0x00,
};

static I2CByte nvChNtscMacro03APS1 [0x40] = {
  0x00,0x80,0x0B,0x00,0x00,0x30,0x00,0x00,
  0x30,0x00,0x00,0x00,0x05,0x00,0x40,0x00,
  0xE0,0x00,0x00,0x00,0x00,0x00,0x0D,0x00,
  0x50,0x50,0x50,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x4A,0x00,0x00,0x00,0x00,0x00,
  0x00,0xBD,0xBD,0x37,0x37,0xBD,0xBD,0x0F,
  0xF0,0x00,0x00,0x0F,0x0F,0x05,0x00,0x00,
};
static I2CByte nvChNtscMacro03APS2 [0x40] = {
  0x00,0x80,0x0B,0x00,0x00,0x30,0x00,0x00,
  0x30,0x00,0x00,0x00,0x05,0xD0,0x40,0x11,
  0xF0,0xC8,0xFF,0x40,0x00,0x00,0x0D,0x00,
  0x10,0x50,0x10,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x4A,0x01,0x98,0x98,0xBB,0x9E,
  0x2D,0xBD,0xBD,0x37,0x37,0xBD,0xBD,0x0F,
  0xF0,0x00,0x00,0x0F,0x0F,0x05,0x00,0x00,
};
static I2CByte nvChNtscMacro03APS3 [0x40] = {
  0x00,0x80,0x0B,0x00,0x00,0x30,0x00,0x00,
  0x30,0x00,0x00,0x00,0x05,0xB0,0x40,0x15,
  0xF0,0xC8,0xFF,0x80,0x00,0x00,0x0D,0x00,
  0x50,0x50,0x50,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x4A,0x01,0x98,0x98,0xBB,0x98,
  0x29,0xBD,0xBD,0x37,0x37,0xBD,0xBD,0x0F,
  0xF0,0x00,0x00,0x0F,0x0F,0x05,0x00,0x00,
};

static I2CByte nvChNtscMacro11APS1 [0x40] = {
  0x00,0x80,0x0B,0x00,0x00,0x3A,0x00,0x00,
  0x30,0x00,0x00,0x00,0x05,0x00,0x40,0x00,
  0xE0,0x00,0x00,0x00,0x00,0x00,0x0D,0x00,
  0x50,0x50,0x50,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x59,0xF0,0x00,0x00,0x00,0x00,
  0x00,0x08,0x08,0x42,0x42,0x08,0x08,0x0F,
  0xF0,0x00,0x00,0x0F,0x0F,0x05,0x00,0x00,
};
static I2CByte nvChNtscMacro11APS2 [0x40] = {
  0x00,0x80,0x0B,0x00,0x00,0x3A,0x00,0x00,
  0x30,0x00,0x00,0x00,0x05,0xD0,0x40,0x11,
  0xF0,0xC8,0xFF,0x40,0x00,0x00,0x0D,0x00,
  0x10,0x50,0x10,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x59,0xF1,0x67,0x67,0xE2,0x9E,
  0x2D,0x08,0x08,0x42,0x42,0x08,0x08,0x0F,
  0xF0,0x00,0x00,0x0F,0x0F,0x05,0x00,0x00,
};
static I2CByte nvChNtscMacro11APS3 [0x40] = {
  0x00,0x80,0x0B,0x00,0x00,0x3A,0x00,0x00,
  0x30,0x00,0x00,0x00,0x05,0xB0,0x40,0x15,
  0xF0,0xC8,0xFF,0x80,0x00,0x00,0x0D,0x00,
  0x50,0x50,0x50,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x59,0xF1,0x67,0x67,0xE2,0x98,
  0x29,0x08,0x08,0x42,0x42,0x08,0x08,0x0F,
  0xF0,0x00,0x00,0x0F,0x0F,0x05,0x00,0x00,
};

static I2CByte nvChNtscMacro17APS1 [0x40] = {
  0x00,0x80,0x0B,0x00,0x40,0x37,0x80,0x00,
  0x30,0x00,0x00,0x00,0x05,0x00,0x40,0x00,
  0xE0,0x00,0x00,0x00,0x00,0x00,0x0D,0x00,
  0x50,0x50,0x50,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x54,0x00,0x00,0x00,0x00,0x00,
  0x00,0xFA,0xFA,0x3F,0x3F,0xFA,0xFA,0x0F,
  0xF0,0x00,0x00,0x0F,0x0F,0x05,0x00,0x00,
};
static I2CByte nvChNtscMacro17APS2 [0x40] = {
  0x00,0x80,0x0B,0x00,0x40,0x37,0x80,0x00,
  0x30,0x00,0x00,0x00,0x05,0xD0,0x40,0x11,
  0xF0,0xC8,0xFF,0x40,0x00,0x00,0x0D,0x00,
  0x10,0x50,0x10,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x54,0x01,0xAE,0xAE,0xD6,0x9E,
  0x2D,0xFA,0xFA,0x3F,0x3F,0xFA,0xFA,0x0F,
  0xF0,0x00,0x00,0x0F,0x0F,0x05,0x00,0x00,
};
static I2CByte nvChNtscMacro17APS3 [0x40] = {
  0x00,0x80,0x0B,0x00,0x40,0x37,0x80,0x00,
  0x30,0x00,0x00,0x00,0x05,0xB0,0x40,0x15,
  0xF0,0xC8,0xFF,0x80,0x00,0x00,0x0D,0x00,
  0x50,0x50,0x50,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x54,0x01,0xAE,0xAE,0xD6,0x98,
  0x29,0xFA,0xFA,0x3F,0x3F,0xFA,0xFA,0x0F,
  0xF0,0x00,0x00,0x0F,0x0F,0x05,0x00,0x00,
};

static I2CByte nvChNtscMacro24APS1 [0x40] = {
  0x00,0x80,0x0B,0x00,0x40,0x40,0x80,0x00,
  0x30,0x00,0x00,0x00,0x05,0x00,0x40,0x00,
  0xE0,0x00,0x00,0x00,0x00,0x00,0x0D,0x00,
  0x50,0x50,0x50,0x0B,0x04,0x01,0x06,0x05,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x6B,0xF0,0x00,0x00,0x00,0x00,
  0x00,0x3F,0x3F,0x50,0x50,0x3F,0x3F,0x0F,
  0xF0,0x00,0x00,0x0F,0x0F,0x05,0x00,0x00,
};
static I2CByte nvChNtscMacro24APS2 [0x40] = {
  0x00,0x80,0x0B,0x00,0x40,0x40,0x80,0x00,
  0x30,0x00,0x00,0x00,0x05,0xD0,0x40,0x11,
  0xF0,0xC8,0xFF,0x40,0x00,0x00,0x0D,0x00,
  0x10,0x50,0x10,0x0B,0x04,0x01,0x06,0x05,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x6B,0xF9,0xDD,0xDD,0x11,0x9E,
  0x2D,0x3F,0x3F,0x50,0x50,0x3F,0x3F,0x0F,
  0xF0,0x00,0x00,0x0F,0x0F,0x05,0x00,0x00,
};
static I2CByte nvChNtscMacro24APS3 [0x40] = {
  0x00,0x80,0x0B,0x00,0x40,0x40,0x80,0x00,
  0x30,0x00,0x00,0x00,0x05,0xB0,0x40,0x15,
  0xF0,0xC8,0xFF,0x80,0x00,0x00,0x0D,0x00,
  0x50,0x50,0x50,0x0B,0x04,0x01,0x06,0x05,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x6B,0xF9,0xDD,0xDD,0x11,0x98,
  0x29,0x3F,0x3F,0x50,0x50,0x3F,0x3F,0x0F,
  0xF0,0x00,0x00,0x0F,0x0F,0x05,0x00,0x00,
};

/* 00: ---, 7:6, 7:0, ---, 7+4, 7:0, ---, ---, */
/* 08: 7:3, ---, ---, ---, 7:0, 7:4, 7:5, 7:0, */
/* 10: 7:4, 7:3, 7:0, 7:5, ---, ---, 7:0, 7:6, */ 
/* 18: 7:4, 7:4, 7:4, ---, ---, 7:4, 7:4, 7:4, */
/* 20: 7:6, 7:5, ---, ---, ---, ---, ---, 7:6  */ 
/* 28: ---, ---, 7:0, 7:0, 7:0, 7:0, 7:0, 7:0  */ 
/* 30: 7:0, 7:0, 7:0, 7:0, 7:0, 7:0, 7:0, 7:0  */ 
/* 38: 7:0, 7:0, 7:0, 7:0, 7:0, 7:0, 7:0, ---  */ 

#if 0 /* not needed yet; ignore to prevent warning */
static I2CByte nvChMaskMacro [0x40] = {
  0xFF,0x3F,0x00,0xFF,0x7F,0x00,0x7F,0xFF,
  0x07,0xFF,0xFF,0xFF,0x00,0x0F,0x3F,0xC0,
  0x0F,0x07,0x00,0x1F,0xFF,0xFF,0x00,0xFF,
  0x0F,0x0F,0x0F,0xF0,0xF0,0xF0,0xF0,0xF0,
  0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,
  0xFF,0xFF,0x80,0xFF,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x80,0x80,0x00,0x00,0x80,
  0x00,0x80,0x00,0x00,0x00,0xF8,0xFF,0xFF
};
#endif

typedef struct {
  int mode;
  I2CByte *regs[4];
} TVCh1MacroTable;

static TVCh1MacroTable nvChMacroTable [] = {
  { 0, {nvChDefMacro, 
	nvChPalMacro00APS1, nvChPalMacro00APS1, nvChPalMacro00APS1}},
  { 3, {nvChDefMacro, 
	nvChNtscMacro03APS1, nvChNtscMacro03APS2, nvChNtscMacro03APS3}},
  { 8, {nvChDefMacro, 
	nvChPalMacro08APS1, nvChPalMacro08APS1, nvChPalMacro08APS1}},
  {11, {nvChDefMacro, 
	nvChNtscMacro11APS1, nvChNtscMacro11APS2, nvChNtscMacro11APS3}},
  {14, {nvChDefMacro, 
	nvChPalMacro14APS1, nvChPalMacro14APS1, nvChPalMacro14APS1}},
  {17, {nvChDefMacro, 
	nvChNtscMacro17APS1, nvChNtscMacro17APS2, nvChNtscMacro17APS3}},
  {20, {nvChDefMacro, 
	nvChPalMacro20APS1, nvChPalMacro20APS1, nvChPalMacro20APS1}},
  {24, {nvChDefMacro, 
	nvChNtscMacro24APS1, nvChNtscMacro24APS2, nvChNtscMacro24APS3}},
  {-1, {nvChDefMacro, nvChDefMacro, nvChDefMacro, nvChDefMacro}}
};

I2CByte *TVCh1MacroRegs (TVEncoderRegs *r)
{
  TVCh1MacroTable *t;

  if (!r || r->ch.macro < 0 || r->ch.macro > 3) 
    return nvChDefMacro;
  for (t = nvChMacroTable; t->mode != -1; t++) {
    if (r->ch.mode == t->mode) {
      return t->regs [r->ch.macro];
    }
  }
  return nvChDefMacro;
}

/* -------- */

/* Read hardware config on object creation.
 *
 * 0: 20-0  mem5v   (should be the same as pll5d)
 * 1: 20-1  pll5va  AVDD 5V / ?V
 * 2: 20-2  pll5vd  DVDD 5V / 3.3V
 * 3: 20-3  plls    (should be the same as pll5va)
 * 5: 1b-5  dvdd2   DVDD2 3.3V / 1.8V
 * 6: 1b-6  gpioin0 
 * 7: 1b-7  gpioin1
 * 8: 1c-6  goenb0
 * 9: 1c-7  goenb1
 */

void TVCh1Create (TVEncoderObj *this, TVChip chip_type, void* ctrl)
{
  register I2CDevPtr dev = (I2CDevPtr) ctrl;
  I2CByte result;

  this->type = chip_type;
  this->ctrl = ctrl;
  this->hwstate = 0x07;
  tvBusOk = TRUE;
  TVReadBus (dev, 0x80|0x20, &result);
  this->hwconfig = SetBitField(result, 3:0, 3:0);
  TVReadBus (dev, 0x80|0x1b, &result);
  this->hwconfig |= SetBitField(result, 7:5, 7:5);
  TVReadBus (dev, 0x80|0x1c, &result);
  this->hwconfig |= SetBitField(result, 7:6, 9:8);
#ifdef FAKE_I2C
  this->hwconfig = 0x32a; /* power on value of my card */
#endif
  if (!tvBusOk) {
    RAISE (MSG_ABORT, "Critical: Error reading Chrontel hardware config\n");
  }
  RAISE (MSG_DEBUG, "create ch hw=%08X", this->hwconfig);
}

/* Init chrontel register values (all done in SetChRegs)
 *
 * -  04-3:0 idf   = 5  (12bit mux "I" RBG)
 * -  06-7   cfrb  = 0  (free running clock in master mode)
 * -  06-6   m/s*  = 1  (master mode)
 * -  06-4   mcp   = 0  (latch on negative edge)
 * -  06-3:2 xcm   = 00 (XCLK 1x) 
 * -  06-1:0 pcm   = 00 (P-Out 1x)
 * -  	     aciv  = 0  (auto civ off, since it has bad quality)
 * -  0D-3   des   = 0  (no embedded sync)
 * 2  0D-2   syo   = 1! (default 0) (output horizontal and vertical sync)
 * 1  0D-1   vsp   = 1! (default 0) (vertical sync active high)
 * 0  0D-0   hsp   = 1! (default 0) (horizontal sync active high)
 * -  1B-5   dvdd2 = 1! (default 0) (dvdd2 at 3.3V)
 * -  1B-4   poutp = 0? (default 0) (P-OUT polarity) (NV=1 ?!)
 * -  1C-5   dsm   = 0  (default 1) (should be 0 if dsen=0)
 * -  1C-4   dsen  = 0  (default 1) (Use hsync for start of active video) 
 */

void TVCh1SetPort (TVEncoderObj *this, int port)
{
  RAISE (MSG_DEBUG, "tv port ch %08X", port);
  if ((port & PORT_FORMAT_MASK) != PORT_FORMAT_RGB) {
    RAISE (MSG_ABORT, "TVCh1SetPort: Only RGB multiplex format supported.");
  }
  if ((port & PORT_BLANK_POLARITY) != PORT_BLANK_LOW) {
    RAISE (MSG_ABORT, "TVCh1SetPort: Only active low blank supported.");
  }
  if ((port & PORT_BLANK_DIR) != PORT_BLANK_OUT) {
    RAISE (MSG_ABORT, "TVCh1SetPort: Only blank out supported.");
  }
  if ((port & PORT_BLANK_MODE) != PORT_BLANK_REGION) {
    RAISE (MSG_ABORT, "TVCh1SetPort: Only blank region supported.");
  }
  this->hwstate = 0x00
    | ((port & PORT_PCLK_POLARITY)  ? 0 : SetBit(4))
    | ((port & PORT_PCLK_MODE)      ? 0 : SetBit(3))
    | ((port & PORT_SYNC_DIR)       ? 0 : SetBit(2))
    | ((port & PORT_VSYNC_POLARITY) ? SetBit(1) : 0)
    | ((port & PORT_HSYNC_POLARITY) ? SetBit(0) : 0);
#if 0 /* FIXME TODO complete */  
  1c-4 dsen   PORT_BLANK_DIR
  1c-5 dsm    PORT_BLANK_DIR
  and setup BCO etc. properly in init!
#endif
}

void TVCh1GetPort (TVEncoderObj *this, int *port)
{
  register I2CDevPtr dev = (I2CDevPtr) this->ctrl;
  I2CByte res;

  RAISE (MSG_DEBUG, "tv get port ch");
  *port = PORT_FORMAT_RGB | PORT_BLANK_LOW | PORT_BLANK_OUT | 
          PORT_BLANK_REGION;
  /* FIXME handle at least some of those */
  TVReadBus (dev, 0x80|0x06, &res);
  *port |= (GetBit(res,6) ? PORT_PCLK_MASTER : PORT_PCLK_SLAVE)
        |  (GetBit(res,4) ? PORT_PCLK_LOW    : PORT_PCLK_HIGH);
  TVReadBus (dev, 0x80|0x0d, &res);
  *port |= (GetBit(res,2) ? PORT_SYNC_OUT   : PORT_SYNC_IN) 
        |  (GetBit(res,1) ? PORT_VSYNC_HIGH : PORT_VSYNC_LOW) 
        |  (GetBit(res,0) ? PORT_HSYNC_HIGH : PORT_HSYNC_LOW);
}

void TVCh1InitRegs (TVEncoderObj *this, int port)
{
  register I2CDevPtr dev = (I2CDevPtr) this->ctrl;

  TVCh1SetPort (this, port);

  /* 0x0d will be written again in SetRegs with macro info */
  TVWriteBus (dev, 0x80|0x0d, SetBitField(this->hwstate,2:0,2:0));

  /* To avoid overclocking (just in case, shouldn't happen anyway).
     Will be set in TVSetChRegs as well */
  TVWriteBus (dev, 0x80|0x06, 0x40); /* master mode clock */
  /* FIXME write 0x1B */
  /* FIXME compare PCLK_ACTIVE, PCLK_VDD etc with hwconfig */
}

/*
 * Note:
 *
 * 0e-3 reset = 1  (must be 1 -- doc is unclear about that)
 *
 */

void TVCh1SetRegs (TVEncoderObj *this, TVEncoderRegs *r, TVState state)
{
  register I2CDevPtr dev = (I2CDevPtr) this->ctrl;
  I2CByte *m;
  int i;

  RAISE (MSG_DEBUG, "tv regs ch");
#ifdef FAKE_I2C
  FPRINTF ("\ni2c [");
#endif
  m = TVCh1MacroRegs (r);
  TVWriteBus (dev, 0x80|0x00, SetBitField(r->ch.dmr_ir,2:0,7:5)
		            | SetBitField(r->ch.dmr_vs,1:0,4:3)
		            | SetBitField(r->ch.dmr_sr,2:0,2:0));
  TVWriteBus (dev, 0x80|0x01, m[0x01] 
	                    | SetBitField(r->ch.ffr_fc,1:0,5:4)
		            | SetBitField(r->ch.ffr_fy,1:0,3:2)
 	                    | SetBitField(r->ch.ffr_ft,1:0,1:0));
  TVWriteBus (dev, 0x80|0x02, m[0x02]);
  TVWriteBus (dev, 0x80|0x03, m[0x03] 
	                    | SetBitField(r->ch.vbw_flff, 0:0,7:7)
	                    | SetBitFlag(r->ch.flags,CH_FLAG_CVBW,6)
	                    | SetBitField(r->ch.vbw_cbw,  1:0,5:4)
	                    | SetBitField(r->ch.vbw_ypeak,0:0,3:3)
	                    | SetBitField(r->ch.vbw_ysv,  1:0,2:1)
			    | SetBitField(r->ch.vbw_ycv,  0:0,0:0));
  TVWriteBus (dev, 0x80|0x04, m[0x04] 
	                    | SetBitField(r->ch.dacg,0:0,6:6)
                            | 0x05 /* idf=5 */ );
  TVWriteBus (dev, 0x80|0x05, m[0x05]);
  TVWriteBus (dev, 0x80|0x06, m[0x06] 
		            | SetBitField(this->hwstate,4:4,4:4)
		            | SetBitField(this->hwstate,3:3,6:6)
	                    | SetBitFlag(r->ch.flags,CH_FLAG_CFRB,7));
  TVWriteBus (dev, 0x80|0x07, Set8Bits(r->ch.sav));
  TVWriteBus (dev, 0x80|0x08, m[0x08] 
	                    | SetBitField(r->ch.sav,8:8,2:2)
		            | SetBitField(r->ch.hpr,8:8,1:1)
		            | SetBitField(r->ch.vpr,8:8,0:0));
  TVWriteBus (dev, 0x80|0x09, Set8Bits(r->ch.blr));
  TVWriteBus (dev, 0x80|0x0a, Set8Bits(r->ch.hpr));
  TVWriteBus (dev, 0x80|0x0b, Set8Bits(r->ch.vpr));
  TVWriteBus (dev, 0x80|0x0c, m[0x0c]);
  TVWriteBus (dev, 0x80|0x0d, m[0x0d]
	      		    | SetBitField(this->hwstate,2:0,2:0));
  TVWriteBus (dev, 0x80|0x0e, m[0x0e] | 0x08 
 	                    | SetBitFlag(r->ch.flags, CH_FLAG_SCART, 4) 
                          | ((state != TV_ON) ? 0x05 
			    : SetBitField(r->ch.flags,1:0,CH_FLAG_DAC)));
  TVWriteBus (dev, 0x80|0x0f, m[0x0f]);
  TVWriteBus (dev, 0x80|0x10, 0x00 | m[0x10]); /* sense off */
  TVWriteBus (dev, 0x80|0x11, m[0x11] 
	                    | SetBitField(r->ch.ce,2:0,2:0));
  TVWriteBus (dev, 0x80|0x12, m[0x12]);
  TVWriteBus (dev, 0x80|0x13, m[0x13] 
	                    | SetBitField(r->ch.pll_n,9:8,2:1)
		            | SetBitField(r->ch.pll_m,8:8,0:0));
  TVWriteBus (dev, 0x80|0x14, Set8Bits(r->ch.pll_m));
  TVWriteBus (dev, 0x80|0x15, Set8Bits(r->ch.pll_n));
  TVWriteBus (dev, 0x80|0x16, m[0x16]);
  TVWriteBus (dev, 0x80|0x17, 0x00 | m[0x17]); /* BCO= 14MHz crystal */
  TVWriteBus (dev, 0x80|0x18, m[0x18] 
	                    | SetBitField(r->ch.fsci,31:28,3:0));
  TVWriteBus (dev, 0x80|0x19, m[0x19] 
	                    | SetBitField(r->ch.fsci,27:24,3:0));
  TVWriteBus (dev, 0x80|0x1a, m[0x1a] 
	                    | SetBitField(r->ch.fsci,23:20,3:0));
  TVWriteBus (dev, 0x80|0x1b, SetBitField(r->ch.fsci,19:16,3:0)
		            | SetBitField(this->hwconfig,7:5,7:5)
		            | SetBitFlag(r->ch.flags,CH_FLAG_POUTP,4));
  TVWriteBus (dev, 0x80|0x1c, SetBitField(r->ch.fsci,15:12,3:0)
		            | SetBitField(this->hwconfig,9:8,7:6));
  TVWriteBus (dev, 0x80|0x1d, m[0x1d] 
	                    | SetBitField(r->ch.fsci,11:8,3:0));
  TVWriteBus (dev, 0x80|0x1e, m[0x1e] 
	                    | SetBitField(r->ch.fsci,7:4,3:0));
  TVWriteBus (dev, 0x80|0x1f, m[0x1f] 
	                    | SetBitField(r->ch.fsci,3:0,3:0));
  TVWriteBus (dev, 0x80|0x20, m[0x20] 
	                    | SetBitField(r->ch.pllcap,0:0,4:4)
		            | SetBitField(this->hwconfig,3:0,3:0));
  TVWriteBus (dev, 0x80|0x21, m[0x21]
	                    | SetBitField(r->ch.civh,1:0,2:1)
	                    | SetBitFlag(r->ch.flags,CH_FLAG_ACIV,0));
  for (i = 0x2a; i <= 0x3d; i++) {
    TVWriteBus (dev, 0x80|i, m[i]);
  }
#if 0 /* Register 0x26-0x29 reserved for test */
  /* mtd, ms, rsa, bst, nst, de, ts */
  TVWriteBus (dev, 0x80|0x27, m[0x27] 
	                    | 0x00 /* FIXME */
	                    | SetBitField(r->ch.ylm,8:8,1:1)
		            | SetBitField(r->ch.clm,8:8,0:0));
  TVWriteBus (dev, 0x80|0x28, Set8Bits(r->ch.ylm));
  TVWriteBus (dev, 0x80|0x29, Set8Bits(r->ch.clm));
#endif
#ifdef FAKE_I2C
  FPRINTF ("]\n");
#endif
}

void TVCh1GetRegs (TVEncoderObj *this, TVEncoderRegs *r)
{
  register I2CDevPtr dev = (I2CDevPtr) this->ctrl;
  I2CByte regs[0x40];
  int i;

  for (i = 0x00; i < 0x40; i++) TVReadBus (dev, 0x80|i, &regs[i]);
  r->ch.dmr_ir    = SetBitField(regs[0x00],7:5,2:0);
  r->ch.dmr_vs    = SetBitField(regs[0x00],4:3,1:0);
  r->ch.dmr_sr    = SetBitField(regs[0x00],2:0,2:0);
  r->ch.ffr_fc    = SetBitField(regs[0x01],5:4,1:0);
  r->ch.ffr_fy    = SetBitField(regs[0x01],3:2,1:0);
  r->ch.ffr_ft    = SetBitField(regs[0x01],1:0,1:0);
  r->ch.vbw_flff  = SetBitField(regs[0x03],7:7,0:0);
  r->ch.vbw_cbw   = SetBitField(regs[0x03],5:4,1:0);
  r->ch.vbw_ypeak = SetBitField(regs[0x03],3:3,0:0);
  r->ch.vbw_ysv   = SetBitField(regs[0x03],2:1,1:0);
  r->ch.vbw_ycv   = SetBitField(regs[0x03],0:0,0:0);
  r->ch.dacg      = SetBitField(regs[0x04],6:6,0:0);
  r->ch.sav     = Set8Bits(regs[0x07])
                | SetBitField(regs[0x08],2:2,8:8);
  r->ch.blr     = Set8Bits(regs[0x09]);
  r->ch.hpr     = Set8Bits(regs[0x0a]) 
                | SetBitField(regs[0x08],1:1,8:8);
  r->ch.vpr     = Set8Bits(regs[0x0b]) 
                | SetBitField(regs[0x08],0:0,8:8);
  r->ch.ce      = SetBitField(regs[0x11],2:0,2:0);
  r->ch.te      = 0;
  r->ch.pll_n   = Set8Bits(regs[0x15]) 
                | SetBitField(regs[0x13],2:1,9:8);
  r->ch.pll_m   = Set8Bits(regs[0x14]) 
                | SetBitField(regs[0x13],0:0,8:8);
  r->ch.fsci    = SetBitField(regs[0x18],3:0,31:28)
                | SetBitField(regs[0x19],3:0,27:24)
                | SetBitField(regs[0x1a],3:0,23:20)
                | SetBitField(regs[0x1b],3:0,19:16)
                | SetBitField(regs[0x1c],3:0,15:12)
                | SetBitField(regs[0x1d],3:0,11: 8)
                | SetBitField(regs[0x1e],3:0, 7: 4)
                | SetBitField(regs[0x1f],3:0, 3: 0);
  r->ch.pllcap  = SetBitField(regs[0x20],4:4,0:0);
  r->ch.civh    = SetBitField(regs[0x21],2:1,1:0);
  r->ch.flags   = CH_FLAG_BOTH 
                | GetBitFlag(regs[0x03],6,CH_FLAG_CVBW)
	        | GetBitFlag(regs[0x06],7,CH_FLAG_CFRB)
 	        | GetBitFlag(regs[0x0e],4,CH_FLAG_SCART) 
	        | GetBitFlag(regs[0x1b],4,CH_FLAG_POUTP)
	        | GetBitFlag(regs[0x21],0,CH_FLAG_ACIV);
  r->ch.macro = 0;
  r->ch.mode = 0;
}
 
void TVCh1SetState (TVEncoderObj *this, TVEncoderRegs *r, TVState state)
{
  register I2CDevPtr dev = (I2CDevPtr) this->ctrl;
  I2CByte *m;

  RAISE (MSG_DEBUG, "tv state ch %i", state);
  tvState = state;
  m = TVCh1MacroRegs (r);
  switch (state)
  {
    case TV_OFF: 
    case TV_BARS: 
      TVWriteBus (dev, 0x80|0x0e, 0x0d); 
      break;
    case TV_ON: 
      TVWriteBus (dev, 0x80|0x0e,  m[0x0e] | 0x08 | 
		  SetBitField(r->ch.flags,1:0,CH_FLAG_DAC)); 
      /* FIXME CHECK (former val: 0x0b) */
      break;
    default:
      break;
  }
}

TVConnect TVCh1GetConnect (TVEncoderObj *this)
{
  register I2CDevPtr dev = (I2CDevPtr) this->ctrl;
  I2CByte power, connect;

  TVReadBus  (dev, 0x80|0x0e, &power);
  TVWriteBus (dev, 0x80|0x0e, 0x0b); 
  TVWriteBus (dev, 0x80|0x10, 0x01); 
  xf86usleep (10);  /* tested */
  TVWriteBus (dev, 0x80|0x10, 0x00); 
  xf86usleep (10);  /* tested */
  TVReadBus  (dev, 0x80|0x10, &connect);
  TVWriteBus (dev, 0x80|0x0e, power); 
  if (tvBusOk) {
    switch (connect & 0x0e) {
      case 0x0c: return CONNECT_COMPOSITE; break;
      case 0x02: return CONNECT_SVIDEO; break;
      case 0x0a: return CONNECT_CONVERT; break;
      case 0x0e: return CONNECT_NONE; break;
      default: 
	RAISE (MSG_WARNING, "Strange Chrontel connection status %02x", 
	       connect & 0x0e); 
      case 0x00: return CONNECT_BOTH; break;
    }
  } else {
    return CONNECT_NONE;
  }
}

long TVCh1GetStatus (TVEncoderObj *this, int index)
{ 
  register I2CDevPtr dev = (I2CDevPtr) this->ctrl;
  long civ;
  I2CByte result;

  tvBusOk = TRUE;  
  TVReadBus (dev, 0x80|0x21, &result);
  civ = SetBitField (result,4:3,0:1);
  TVReadBus (dev, 0x80|0x22, &result);
  civ = (civ << 8) | result;
  TVReadBus (dev, 0x80|0x23, &result);
  civ = (civ << 8) | result;
  TVReadBus (dev, 0x80|0x24, &result);
  civ = (civ << 8) | result;
  if (!tvBusOk) 
    return -1;
  else 
    return civ;
}

/*
 * Check for Chrontel chip on device dev. Return version string if found, 
 * NULL otherwise.
 */

char *TVDetectChrontel1 (I2CDevPtr dev, TVChip *encoder)
{
  I2CByte result;
  char *chip;
  static char version [50];

#ifdef FAKE_PROBE_CHRONTEL
  *encoder = TV_CHRONTEL_MODEL1;
  snprintf (version, 50, "Chrontel-1 Fake (%1s:%02X)", I2C_ID(dev));
  return version;
#else
#ifdef FAKE_I2C
  return NULL;
#endif
#endif  
  tvBusOk = TRUE;  
  TVReadBus (dev, 0x80|0x25, &result);
  RAISE (MSG_DEBUG, "ch version %02x (%s)", result, tvBusOk ? "ok" : "err");
  if (!tvBusOk) return NULL;
  *encoder = TV_CHRONTEL_MODEL1;
  switch (result) 
  {
    case 0x02: chip = "7003B"; break; /* 00000010 */ 
    case 0x30: chip = "7004A"; break; /* ?? */
    case 0x31: chip = "7004B"; break; /* ?? */
    case 0x32: chip = "7004C"; break; /* 00110010 */
    case 0x38: chip = "7005A"; break; /* ?? */
    case 0x39: chip = "7005B"; break; /* ?? */
    case 0x3a: chip = "7005C"; break; /* 00111010 */
    case 0x28: chip = "7006A"; break; /* ?? */
    case 0x29: chip = "7006B"; break; /* ?? */
    case 0x2a: chip = "7006C"; break; /* 00101010 */
    case 0x50: chip = "7007A"; break; /* 01010000 */
    case 0x40: chip = "7008A"; break; /* 01000000 */
    default:   chip = NULL; break;
  }
  if (chip) {
    snprintf (version, 50, "Chrontel %s (%1s:%02X)", chip, I2C_ID(dev));
  } else {
    snprintf (version, 50, "Chrontel-1 (id = %02X) (%1s:%02X)", 
	      result, I2C_ID(dev));
  }
  return version;
}

/* Clock range is a safe guess based on available modes */

TVEncoderObj tvCh1Template = {
  type: TV_CHRONTEL_MODEL1, ctrl: NULL, minClock: 20000, maxClock: 40000, 
  Create:     TVCh1Create,
  InitRegs:   TVCh1InitRegs, 
  SetRegs:    TVCh1SetRegs, 
  GetRegs:    TVCh1GetRegs, 
  SetPort:    TVCh1SetPort,
  GetPort:    TVCh1GetPort,
  SetState:   TVCh1SetState,
  GetConnect: TVCh1GetConnect, 
  GetStatus:  TVCh1GetStatus
};
