/*
 * Copyright (c) Members of the EGEE Collaboration. 2004-2010. 
 * See http://www.eu-egee.org/partners/ for details on the copyright
 * holders.  
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); 
 * you may not use this file except in compliance with the License. 
 * You may obtain a copy of the License at 
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0 
 * 
 * Unless required by applicable law or agreed to in writing, software 
 * distributed under the License is distributed on an "AS IS" BASIS, 
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
 * See the License for the specific language governing permissions and 
 * limitations under the License.
 *
 */


/**
   \defgroup LcmapsBasicInterface LCMAPS basic interfaces
   
    \brief  This interface returns the uid, gids and poolindex, i.e. leaseid,
            using LCMAPS and takes the user credential in PEM format as input

    The interface function is declared as a type, since the function is not
    pre-defined when using dlopen().
    -# lcmaps_return_account_from_pem_t: interface function type

    \ingroup LcmapsInterface
*/

/**
    \file   lcmaps_basic.h
    \brief  Basic interface of the LCMAPS library.
    \author Martijn Steenbakkers for the EU DataGrid.
    \author Oscar Koeroo for the EGEE project.


    This header contains the declarations of the LCMAPS library functions:
    -# lcmaps_init():
        To initialize the LCMAPS module
    -# lcmaps_init_and_log():
        To initialize the LCMAPS module and select logging type
    -# lcmaps_init_and_logfile():
        To initialize the LCMAPS module, select logging type and set logfile
    -# lcmaps_run_without_credentials_and_return_username():
        To do the user mapping, without credentials and return the user name
    -# lcmaps_run_with_fqans_mapcounter_and_return_account()
        To do the user mapping, based on DN+FQANs+mapcounter and return the account information
    -# lcmaps_run_with_fqans_and_return_account()
        To do the user mapping, based on DN+FQANs and return the account information
    -# lcmaps_term():
        To cleanly terminate the module

    \ingroup LcmapsBasicInterface
*/

#ifndef LCMAPS_BASIC_H
#define LCMAPS_BASIC_H


/******************************************************************************
                             Include header files
******************************************************************************/
#include <stdio.h>
#include <sys/types.h>
#include "lcmaps_version.h"
#include "lcmaps_types.h"
#include "lcmaps_if.h"

/******************************************************************************
 *                 Module definition
 *****************************************************************************/


/*!
    \brief Initialize the LCMAPS module.

    The function does the following:
    - initialize LCMAPS module.
    - setup logging, error handling (not yet).
    - start PluginManager

    \param fp file handle for logging (from gatekeeper or other previously opened file handle)
              If the file handle is zero, assume that only syslogging is requested
    \retval 0 initialization succeeded.
    \retval 1 initialization failed.
*/
typedef int lcmaps_init_t(
        FILE* fp
);

/*!
    \brief Initialize the LCMAPS module and select logging type

    The function does the following:
    - initialize LCMAPS module.
    - setup logging, error handling (not yet).
    - start PluginManager

    \param fp file handle for logging (from gatekeeper or other previously opened file handle)
              If the file handle is zero, assume that only syslogging is requested
    \param logtype type of logging (usrlog and/or syslog)
    \retval 0 initialization succeeded.
    \retval 1 initialization failed.
*/
typedef int lcmaps_init_and_log_t(
        FILE* fp,
        unsigned short logtype
);

/*!
    \brief Initialize the LCMAPS module, select logging type and set logfile

    The function does the following:
    - initialize LCMAPS module.
    - Setup logging by providing a file handle or file name, error handling (not yet).
    - start PluginManager

    \param logfile name of logfile 
    \param fp file handle for logging (from gatekeeper or other previously opened file handle)
              If the file handle is zero, assume that only syslogging is requested
    \param logtype type of logging (usrlog and/or syslog)
    \retval 0 initialization succeeded.
    \retval 1 initialization failed.
*/
typedef int lcmaps_init_and_logfile_t(
        char * logfile,
        FILE* fp,
        unsigned short logtype
);

/*!
    \brief Terminate the LCMAPS module.

    The function does the following:
    - terminate the LCMAPS module
    - terminate the plugins

    \retval 0 termination succeeded.
    \retval 1 termination failed.
*/
typedef int lcmaps_term_t(void);

/*!
    \brief do the user mapping using only the user DN and return the username

    Based on the only the user DN do the following:
    Do the user mapping based on the provided list of policies (first successful
    policy found in the lcmaps policy file (lcmaps.db) will result in the user
    mapping) and return user name.
    This interface can be used to provide the legacy
        \b globus_gss_assist_gridmap()
    interface.

    \param user_dn_tmp user DN
    \param request     authorization request as an RSL string
    \param usernamep   pointer to user name (to be freed by calling application).
                       Note: usernamep should be non-NULL at the start !
    \param npols       number of policies to be considered for evaluation
    \param policynames the names of the policies to be considered for evaluation

    \retval 0 mapping succeeded.
    \retval 1 mapping failed.
*/
typedef int lcmaps_run_without_credentials_and_return_username_t(
        char * user_dn_tmp,
        lcmaps_request_t request,
        char ** usernamep,
        int npols,
        char ** policynames
);

/*!
    \brief let LCMAPS handle the user mapping based on fqans and
           return the account information

    LCMAPS will run with a list of FQANs and the DN as an input. In addition a
    list of policies may be provided.
    The allocated uid, gids and the poolindex will be returned to the calling
    application.
    This interface is intended to be used by a wrapper function for the 
        \b WorkSpace \b Service (WSS, f.q.a. Dynamic Account Service - DAS).

    \param user_dn     the DN of the user
    \param fqan_list   the list of (VOMS) FQANs that have been asserted to the user
    \param nfqan       the number of FQANs in fqan_list
    \param mapcounter  the counter which will be added to the poolindex, effectively enabling
                       multiple account mappings
    \param request     RSL string
    \param npols       number of policies to be considered for evaluation
    \param policynames the names of the policies to be considered for evaluation
    \param puid        pointer to the uid found (output parameter)
    \param ppgid_list  pointer to the list of primary gids found (output parameter)
    \param pnpgid      pointer to the number of primary gids found (output parameter)
    \param psgid_list  pointer to the list of secondary gids found (output parameter)
    \param pnsgid      pointer to the number of secondary gids found (output parameter)
    \param poolindexp  pointer to poolindex string (output parameter)

    \retval 0 mapping succeeded.
    \retval 1 mapping failed.
*/
typedef int lcmaps_run_with_fqans_mapcounter_and_return_account_t(
        char *            user_dn,
        char **           fqan_list,
        int               nfqan,
        int               mapcounter,
        lcmaps_request_t  request,
        int               npols,
        char **           policynames,
        uid_t *           puid,
        gid_t **          ppgid_list,
        int *             pnpgid,
        gid_t **          psgid_list,
        int *             pnsgid,
        char **           poolindexp
);

/*!
    \brief let LCMAPS handle the user mapping based on fqans and
           return the account information

    LCMAPS will run with a list of FQANs and the DN as an input. In addition a
    list of policies may be provided.
    The allocated uid, gids and the poolindex will be returned to the calling
    application.
    This interface is intended to be used by a wrapper function for the 
        \b WorkSpace \b Service (WSS, f.q.a. Dynamic Account Service - DAS).

    \param user_dn     the DN of the user
    \param fqan_list   the list of (VOMS) FQANs that have been asserted to the user
    \param nfqan       the number of FQANs in fqan_list
    \param request     RSL string
    \param npols       number of policies to be considered for evaluation
    \param policynames the names of the policies to be considered for evaluation
    \param puid        pointer to the uid found (output parameter)
    \param ppgid_list  pointer to the list of primary gids found (output parameter)
    \param pnpgid      pointer to the number of primary gids found (output parameter)
    \param psgid_list  pointer to the list of secondary gids found (output parameter)
    \param pnsgid      pointer to the number of secondary gids found (output parameter)
    \param poolindexp  pointer to poolindex string (output parameter)

    \retval 0 mapping succeeded.
    \retval 1 mapping failed.
*/
typedef int lcmaps_run_with_fqans_and_return_account_t(
        char *            user_dn,
        char **           fqan_list,
        int               nfqan,
        lcmaps_request_t  request,
        int               npols,
        char **           policynames,
        uid_t *           puid,
        gid_t **          ppgid_list,
        int *             pnpgid,
        gid_t **          psgid_list,
        int *             pnsgid,
        char **           poolindexp
);

#ifndef LCMAPS_USE_DLOPEN
lcmaps_init_and_logfile_t lcmaps_init_and_logfile;
lcmaps_init_t lcmaps_init;
lcmaps_init_and_log_t lcmaps_init_and_log;
lcmaps_term_t lcmaps_term;
lcmaps_run_without_credentials_and_return_username_t lcmaps_run_without_credentials_and_return_username;
lcmaps_run_with_fqans_mapcounter_and_return_account_t lcmaps_run_with_fqans_mapcounter_and_return_account;
lcmaps_run_with_fqans_and_return_account_t lcmaps_run_with_fqans_and_return_account;
#endif

/*!
    \brief Do the mapping based on the user's credential in PEM-string format

    LCMAPS runs receiving a proxy credential in a PEM formatted string.
    A list of policies may be provided.
    The allocated uid, gids and the poolindex will be returned to the calling
    application.
    This interface is intended to be used by the modified suexec wrapper.

    \param user_dn     The DN of the user
    \param pem_string  The PEM-encoded string containing the user proxy
    \param mapcounter  The counter which will be added to the poolindex, effectively enabling
                       multiple account mappings
    \param request     RSL string
    \param npols       Number of policies to be considered for evaluation
    \param policynames The names of the policies to be considered for evaluation
    \param puid        Pointer to the uid found (output parameter)
    \param ppgid_list  Pointer to the list of primary gids found (output parameter)
    \param pnpgid      Pointer to the number of primary gids found (output parameter)
    \param psgid_list  Pointer to the list of secondary gids found (output parameter)
    \param pnsgid      Pointer to the number of secondary gids found (output parameter)
    \param poolindexp  Pointer to poolindex string (output parameter)

    \retval 0 mapping succeeded
    \retval 1 mapping failed
*/
typedef int lcmaps_run_with_pem_and_return_account_t(
        char *            user_dn,
        char *            pem_string,
        int               mapcounter,
        lcmaps_request_t  request,
        int               npols,
        char **           policynames,
        uid_t *           puid,
        gid_t **          ppgid_list,
        int *             pnpgid,
        gid_t **          psgid_list,
        int *             pnsgid,
        char **           poolindexp
);

/*!
    \brief Verify the account mapping

    LCMAPS verifies the account mapping based on the proxy credential
    in a PEM formatted string.
    A list of policies may be provided.
    This interface is intended to be used by the modified suexec wrapper.

    \param user_dn      the user DN (input)
    \param pem_string   the PEM-encoded string containing the user proxy (input)
    \param uid          the uid of the account that should be verified (input)
    \param pgid_list    the list of primary gids of the account that should be verified (input)
    \param npgid        the number of primary gids of the account that should be verified (input)
    \param sgid_list    the list of secondary gids of the account that should be verified (input)
    \param nsgid        the number of secondary gids of the account that should be verified (input)
    \param poolindex    poolindex string of the account that should be verified (input)
    \param request      RSL string (input)
    \param npols        number of policies to be considered for evaluation (input)
    \param policynames  the names of the policies to be considered for evaluation (input)

    \retval 0 verification succeeded
    \retval 1 verification failed
*/
typedef int lcmaps_run_and_verify_account_from_pem_t(
        char *            user_dn,
        char *            pem_string,
        uid_t             uid,
        gid_t *           pgid_list,
        int               npgid,
        gid_t *           sgid_list,
        int               nsgid,
        char *            poolindex,
        lcmaps_request_t  request,
        int               npols,
        char **           policynames
);

#ifndef LCMAPS_USE_DLOPEN
lcmaps_run_with_pem_and_return_account_t lcmaps_run_with_pem_and_return_account;
lcmaps_run_and_verify_account_from_pem_t lcmaps_run_and_verify_account_from_pem;

#endif /* LCMAPS_USE_DLOPEN */

/**
   \fn lcmaps_disable_voms_attributes_verification
   \brief disable all VOMS attribute verification by LCMAPS
   \since LCMAPS v1.4.21
   \ingroup LcmapsBasicInterface
   \see lcmaps_set_voms_attributes_verification
 */
typedef void lcmaps_disable_voms_attributes_verification_t(void);

/**
   \fn lcmaps_enable_voms_attributes_verification
   \brief enable full VOMS attribute verification by LCMAPS (on by default)
   \since LCMAPS v1.4.21
   \ingroup LcmapsBasicInterface
   \see lcmaps_set_voms_attributes_verification
 */
typedef void lcmaps_enable_voms_attributes_verification_t(void);

/**
 * \fn lcmaps_is_set_to_verify_voms_attributes
   \brief test if the VOMS attribute verification by LCMAPS is set.
   \since LCMAPS v1.4.21
   \ingroup LcmapsBasicInterface
   \retval 0 if not set
   \retval >0 if set
 */
typedef int lcmaps_is_set_to_verify_voms_attributes_t(void);

/**
   \fn lcmaps_set_voms_attributes_verification
   \brief sets set of to-be-verified VOMS attributes to verify_flags
   \since LCMAPS v1.6.5
   \ingroup LcmapsBasicInterface
   \see lcmaps_get_voms_attributes_verification_t
 */
typedef void lcmaps_set_voms_attributes_verification_t (unsigned int verify_flags);
/**
   \fn lcmaps_get_voms_attributes_verification
   \brief gets set of to-be-verified VOMS attributes
   \ingroup LcmapsBasicInterface
   \retval combination of flags, see voms_apic.h for valid flags
   \see lcmaps_set_voms_attributes_verification_t
 */
typedef unsigned int lcmaps_get_voms_attributes_verification_t (void);

#ifndef LCMAPS_USE_DLOPEN
lcmaps_disable_voms_attributes_verification_t lcmaps_disable_voms_attributes_verification;
lcmaps_enable_voms_attributes_verification_t lcmaps_enable_voms_attributes_verification;
lcmaps_is_set_to_verify_voms_attributes_t lcmaps_is_set_to_verify_voms_attributes;

lcmaps_set_voms_attributes_verification_t lcmaps_set_voms_attributes_verification;
lcmaps_get_voms_attributes_verification_t lcmaps_get_voms_attributes_verification;
#endif

/**
 * \fn lcmaps_set_voms_verification_time
 * \brief sets VOMS verification time to verify_time, either UNIX time
 * (flags==0), relative to notBefore of leaf proxy (flags==1) or relative to
 * notAfter (flags==2)
 * \ingroup LcmapsBasicInterface
 * \param verify_time (input)
 * \param flags (input)
 */
typedef void lcmaps_set_voms_verification_time_t(time_t verify_time, int flags);
/**
 * \fn lcmaps_get_voms_verification_time
 * \brief gets VOMS verification time
 * \ingroup LcmapsBasicInterface
 * \param verify_time (output)
 * \param flags (output)
 * \retval 0 success, -1 error
 */
typedef int lcmaps_get_voms_verification_time_t(time_t *verify_time, int *flags);

#ifndef LCMAPS_USE_DLOPEN
lcmaps_set_voms_verification_time_t lcmaps_set_voms_verification_time;
lcmaps_get_voms_verification_time_t lcmaps_get_voms_verification_time;
#endif
#endif /* LCMAPS_BASIC_H */


