// --------------------------------------------------------------------
// CanvasView
// --------------------------------------------------------------------
/*

    This file is part of the extensible drawing editor Ipe.
    Copyright (c) 1993-2019 Otfried Cheong

    Ipe is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, you have permission to link Ipe with the
    CGAL library and distribute executables, as long as you follow the
    requirements of the Gnu General Public License in regard to all of
    the software in the executable aside from CGAL.

    Ipe is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
    or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
    License for more details.

    You should have received a copy of the GNU General Public License
    along with Ipe; if not, you can find it at
    "http://www.gnu.org/copyleft/gpl.html", or write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include "canvasview_qt.h"
#include "ipetool.h"

using namespace ipe;

// --------------------------------------------------------------------

IpeAction::IpeAction(int cmd, const QString &text, const char *shortcut,
                     QObject *parent):
  QAction(text, parent)
{
  iCommand = cmd;
  connect(this, SIGNAL(triggered()), SLOT(forwardTrigger()));
  connect(this, SIGNAL(triggered(int)), parent, SLOT(cmd(int)));
  if (shortcut)
    setShortcut(QKeySequence(shortcut));
}

IpeAction::IpeAction(int cmd, const QString &text, const std::vector<std::string>& shortcuts,
                     QObject *parent):
  QAction(text, parent)
{
  iCommand = cmd;
  connect(this, SIGNAL(triggered()), SLOT(forwardTrigger()));
  connect(this, SIGNAL(triggered(int)), parent, SLOT(cmd(int)));
  if (!shortcuts.empty()) {
    QList<QKeySequence> shortcut_keys;
    for (unsigned i = 0; i < shortcuts.size(); ++i)
      shortcut_keys.push_back(QKeySequence(shortcuts[i].c_str()));
    setShortcuts(shortcut_keys);
  }
}

void IpeAction::forwardTrigger()
{
  emit triggered(iCommand);
}

// --------------------------------------------------------------------

CanvasView::CanvasView(Qt::WindowFlags f):
  QMainWindow(0, f)
{
  iDoc = 0;

  iCanvas = new Canvas(this);
  setCentralWidget(iCanvas);

  Canvas::Style style = iCanvas->canvasStyle();
  style.pretty = true;
  style.paperClip = true;
  iCanvas->setCanvasStyle(style);

  iSnap.iSnap = 0;
  iSnap.iGridVisible = false;
  iSnap.iGridSize = 8;
  iSnap.iAngleSize = M_PI / 6.0;
  iSnap.iSnapDistance = 10;
  iSnap.iWithAxes = false;
  iSnap.iOrigin = Vector::ZERO;
  iSnap.iDir = 0;
  iCanvas->setSnap(iSnap);
}

bool CanvasView::load(Document* doc, int iPageNo, int iViewNo)
{
  iDoc = doc;

  iCanvas->setResources(iDoc->resources());
  iCanvas->setPage(iDoc->page(iPageNo), iPageNo, iViewNo, iDoc->cascade());
  iCanvas->setPan(Vector(300, 400));
  static_cast<CanvasBase*>(iCanvas)->update();

  return true;
}

void CanvasView::fitBox(const Rect &box)
{
  if (box.isEmpty())
    return;

  ipeDebug("canvas: %d x %d", iCanvas->width(), iCanvas->height());
  double xfactor = box.width() > 0.0  ? (iCanvas->width() / box.width()) : 20.0;
  double yfactor = box.height() > 0.0 ? (iCanvas->height() / box.height()) : 20.0;
  double zoom = (xfactor > yfactor) ? yfactor : xfactor;
  iCanvas->setPan(0.5 * (box.bottomLeft() + box.topRight()));
  iCanvas->setZoom(zoom);
  static_cast<CanvasBase*>(iCanvas)->update();
}

void CanvasView::setView(int pageNo, int viewNo)
{
  iCanvas->setPage(iDoc->page(pageNo), pageNo, viewNo, iDoc->cascade());
  static_cast<CanvasBase*>(iCanvas)->update();
}

void CanvasView::resizeEvent(QResizeEvent *event)
{
  fitBox(iDoc->cascade()->findLayout()->paper());
}

// --------------------------------------------------------------------
