/* Copyright (C) 2009 Chris Vine

The library comprised in this file or of which this file is part is
distributed by Chris Vine under the GNU Lesser General Public
License as follows:

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public License
   as published by the Free Software Foundation; either version 2.1 of
   the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License, version 2.1, for more details.

   You should have received a copy of the GNU Lesser General Public
   License, version 2.1, along with this library (see the file LGPL.TXT
   which came with this source code package in the c++-gtk-utils
   sub-directory); if not, write to the Free Software Foundation, Inc.,
   59 Temple Place - Suite 330, Boston, MA, 02111-1307, USA.

*/

#include <c++-gtk-utils/lib_defs.h>

#include <c++-gtk-utils/emitter.h>

namespace Cgu {

Releaser::~Releaser(void) {

  Thread::Mutex::Lock lock(mutex);
  // we don't have lambdas in C++03, so use a struct in function scope
  struct DisconnectItem {
    static void exec(const Callback::SafeFunctor& f) {f();}
  };
  std::for_each(disconnect_list.begin(), disconnect_list.end(),
		DisconnectItem::exec);

}

Releaser& Releaser::operator=(const Releaser&) {

  // first disconnect ourselves from any existing emitters protected
  // by this object and then vacate disconnect_list - effectively, on
  // assignment we drop our old identity and become a blank sheet,
  // since our parent cannot be assigned any emitter connections in
  // its new identity - the assignee retains them - and it may not be
  // safe for it to keep its pre-assignment emitter connections.
  // (Note, EmitterArg<> and SafeEmitterArg<> objects can't be copied)

  Thread::Mutex::Lock lock(mutex);
  // we don't have lambdas in C++03, so use a struct in function scope
  struct DisconnectItem {
    static void exec(const Callback::SafeFunctor& f) {f();}
  };
  std::for_each(disconnect_list.begin(), disconnect_list.end(),
		DisconnectItem::exec);

  disconnect_list.clear();
  return *this;
}

void Releaser::add(const Callback::SafeFunctor& f) {
  Thread::Mutex::Lock lock(mutex);
  disconnect_list.push_back(f);
}

void Releaser::remove(Callback::SafeFunctor f) {
  Thread::Mutex::Lock lock(mutex);
  std::list<Callback::SafeFunctor>::iterator iter =
    std::find(disconnect_list.begin(), disconnect_list.end(), f);

  if (iter != disconnect_list.end()) {
    disconnect_list.erase(iter);
  }
}

// this method is called instead of Releaser::remove() when called by a
// SafeEmitterArg<> object which has locked its mutex, so as to avoid
// out of order locking deadlocks.  It is primarily intended to cover a
// case where both the Releaser and SafeEmitterArg<> destructors are
// operating and will cause the SafeEmitterArg<> destructor to spin
// until the Releaser destructor has done its business, although it is
// also used in SafeEmitterArg<>::disconnect()
void Releaser::try_remove(Callback::SafeFunctor f, int* result_p) {

  if (!(*result_p = mutex.trylock())) {
    Thread::Mutex::Lock lock(mutex, Thread::locked);
    std::list<Callback::SafeFunctor>::iterator iter =
      std::find(disconnect_list.begin(), disconnect_list.end(), f);

    if (iter != disconnect_list.end()) {
      disconnect_list.erase(iter);
    }
  }
}

EmitterArg<void>::~EmitterArg() {

  struct DisconnectReleaserItem {
    static void exec(const ListItem& l) {(l.f2)();}
  };
  std::for_each(emission_list.begin(), emission_list.end(),
		DisconnectReleaserItem::exec);
}

void EmitterArg<void>::emit() const {

  // create a local copy of emission_list, to enable a connected
  // function (i) to delete the EmitterArg<> object to which it is
  // connected, even if there are other functors still to execute in
  // the same emission (which will execute normally provided they do
  // not try to call any of the emitter's functions), (ii) to call
  // 'delete this' nothwithstanding that the connected function is
  // protected by a Releaser object (assuming all the other restraints
  // on calling 'delete this' are met), provided that no other access
  // would be made to the deleted object in a function call connected
  // to the same emitter which is due to execute subsequently in the
  // same emission, and (iii) to disconnect itself from the
  // EmitterArg<> object.  This design approach has a trade-off: if a
  // connected function tries to block, unblock or disconnect another
  // function connected to the same EmitterArg<> object which is due
  // to execute subsequently in the same emission (or to block,
  // unblock or disconnect itself when it is due to execute again
  // subsequently in the same emission), the attempted block, unblock
  // or disconnection will not have any effect on that emission (it
  // will only have effect on a subsequent emission).  In addition, a
  // connected function may not destroy an object whose non-static
  // method is connected to the same emitter and which would execute
  // subsequently in the same emission, even if that object is
  // protected by a Releaser object (the non-static method will
  // unsuccessfully attempt to execute notwithstanding the destruction
  // of the object it would be operating on).

  std::list<ListItem> local_list = emission_list;
  // we don't have lambdas in C++03, so use a struct in function scope
  struct EmitItem {
    static void exec(const ListItem& l) {if (!l.blocked) (l.f1)();}
  };
  std::for_each(local_list.begin(), local_list.end(),
		EmitItem::exec);
}

bool EmitterArg<void>::test_emit() const {
  if (emission_list.empty()) return false;
  emit();
  return true;
}

Callback::Functor EmitterArg<void>::connect(const Callback::Functor& f1) {
  emission_list.push_back(ListItem(f1, Callback::Functor()));
  return f1;
}

Callback::Functor EmitterArg<void>::connect(const Callback::Functor& f1, Releaser& r) {
  // In this method:
  // f1 is the functor we execute when we emit()
  // f2 is the functor we execute in our destructor if we are destroyed before the
  // remote object is
  // f3 is the functor the remote object executes in its Releaser if it is destroyed
  // before we are, or if Releaser::operator=() is called

  Callback::SafeFunctor f3(Callback::make_val(*this, &EmitterArg<void>::tracking_disconnect, f1));
  Callback::Functor f2(Callback::make_val(r, &Releaser::remove, f3));
  r.add(f3);
  try {
    emission_list.push_back(ListItem(f1, f2));
  }
  catch (...) {
    r.remove(f3);
    throw;
  }
  return f1;
}

void EmitterArg<void>::disconnect(const Callback::Functor& arg) {
  // we don't have lambdas in C++03, so use a struct in function scope
  struct Pred {
    // we can't pass const reference types as we bind with std::bind2nd below
    static bool pred(ListItem p, Callback::Functor f) {
      return (p.f1 == f);
    }
  };

  // in theory, we could have connected the same functor object
  // more than once, so cater for that
  std::list<ListItem>::iterator iter = emission_list.begin();
  for (;;) {
    iter = std::find_if(iter, emission_list.end(),
			std::bind2nd(std::ptr_fun(Pred::pred), arg));
    if (iter != emission_list.end()) {
      // remove ourselves from the remote Releaser object
      (iter->f2)();

      // remove this item from emission_list
      iter = emission_list.erase(iter);
    }
    else break;
  }
}

// tracking disconnect() is the same as disconnect(), except that we do not
// execute f2 as the remote Releaser object will destroy its own functors
// in that case
void EmitterArg<void>::tracking_disconnect(Callback::Functor arg) {
  // we don't have lambdas in C++03, so use a struct in function scope
  struct Pred {
    // we can't pass const reference types as we bind with std::bind2nd below
    static bool pred(ListItem p, Callback::Functor f) {
      return (p.f1 == f);
    }
  };

  std::list<ListItem>::iterator iter = emission_list.begin();
  for (;;) {
    iter = std::find_if(iter, emission_list.end(),
			std::bind2nd(std::ptr_fun(Pred::pred), arg));
    if (iter != emission_list.end()) {
      // remove this item from emission_list
      iter = emission_list.erase(iter);
    }
    else break;
  }
}

void EmitterArg<void>::block(const Callback::Functor& arg) {
  // we don't have lambdas in C++03, so use a struct in function scope
  struct Pred {
    // we can't pass const reference types as we bind with std::bind2nd below
    static bool pred(ListItem p, Callback::Functor f) {
      return (p.f1 == f);
    }
  };

  // in theory, we could have connected the same functor object
  // more than once, so cater for that
  std::list<ListItem>::iterator iter = emission_list.begin();
  for (;;) {
    iter = std::find_if(iter, emission_list.end(),
			std::bind2nd(std::ptr_fun(Pred::pred), arg));
    if (iter != emission_list.end()) {
      iter->blocked = true;
      ++iter;
    }
    else break;
  }
}

void EmitterArg<void>::unblock(const Callback::Functor& arg) {
  // we don't have lambdas in C++03, so use a struct in function scope
  struct Pred {
    // we can't pass const reference types as we bind with std::bind2nd below
    static bool pred(ListItem p, Callback::Functor f) {
      return (p.f1 == f);
    }
  };

  // in theory, we could have connected the same functor object
  // more than once, so cater for that
  std::list<ListItem>::iterator iter = emission_list.begin();
  for (;;) {
    iter = std::find_if(iter, emission_list.end(),
			std::bind2nd(std::ptr_fun(Pred::pred), arg));
    if (iter != emission_list.end()) {
      iter->blocked = false;
      ++iter;
    }
    else break;
  }
}

SafeEmitterArg<void>::~SafeEmitterArg() {

  // go through emission_list() item by item, popping off the front and erasing
  // as we go in case Releaser::try_remove() fails to acquire the lock on one
  // of the iterations
  Thread::Mutex::Lock lock(mutex);
  while (!emission_list.empty()) {
    std::list<ListItem>::iterator iter = emission_list.begin();
    int result = 0; // f2 might be a no-op
    // remove ourselves from the remote Releaser object
    (iter->f2)(&result);
    if (!result) { // we got the Releaser mutex lock or no-op
      // now remove this item from emission_list
      emission_list.erase(iter);
    }
    else {
      mutex.unlock();
      // spin nicely
#ifdef CGU_USE_SCHED_YIELD
      sched_yield();
#else
      usleep(10);
#endif
      mutex.lock();
    }
  }
}

void SafeEmitterArg<void>::emit() const {

  // create a local copy of emission_list, to enable a connected
  // function (i) to delete the EmitterArg<> object to which it is
  // connected, even if there are other functors still to execute in
  // the same emission (which will execute normally provided they do
  // not try to call any of the emitter's functions), (ii) to call
  // 'delete this' nothwithstanding that the connected function is
  // protected by a Releaser object (assuming all the other restraints
  // on calling 'delete this' are met), provided that no other access
  // would be made to the deleted object in a function call connected
  // to the same emitter which is due to execute subsequently in the
  // same emission, and (iii) to disconnect itself from the
  // EmitterArg<> object.  This design approach has a trade-off: if a
  // connected function tries to block, unblock or disconnect another
  // function connected to the same EmitterArg<> object which is due
  // to execute subsequently in the same emission (or to block,
  // unblock or disconnect itself when it is due to execute again
  // subsequently in the same emission), the attempted block, unblock
  // or disconnection will not have any effect on that emission (it
  // will only have effect on a subsequent emission).  In addition, a
  // connected function may not destroy an object whose non-static
  // method is connected to the same emitter and which would execute
  // subsequently in the same emission, even if that object is
  // protected by a Releaser object (the non-static method will
  // unsuccessfully attempt to execute notwithstanding the destruction
  // of the object it would be operating on).

  // SafeFunctorArg<> usage has the additional point that while an
  // emission is in course, another thread should not try to do any of
  // those things, or the same outcome will result.  Another thread
  // should leave alone objects connected to a SafeEmitterArg<> object
  // from the time of operator()() or emit() beginning to the time of
  // it ending, and not try to interfere.

  // a side effect of having a local list is that, as required, we
  // will not be holding our mutex when executing the functors it
  // contains.  It is OK having the functors in two different lists
  // which are potentially (when our mutex is released) in two
  // different threads, because the functors hold their
  // Callback::Callback objects by SharedLockPtr so their reference
  // count is protected (they are SafeFunctorArg<> functors).

  std::list<ListItem> local_list;
  { // scope block for mutex lock
    Thread::Mutex::Lock lock(mutex);
    local_list = emission_list;
  }

  // we don't have lambdas in C++03, so use a struct in function scope
  struct EmitItem {
    static void exec(const ListItem& l) {if (!l.blocked) (l.f1)();}
  };
  std::for_each(local_list.begin(), local_list.end(),
		EmitItem::exec);
}

bool SafeEmitterArg<void>::test_emit() const {

  std::list<ListItem> local_list;
  { // scope block for mutex lock
    Thread::Mutex::Lock lock(mutex);
    if (emission_list.empty()) return false;
    local_list = emission_list;
  }

  // we don't have lambdas in C++03, so use a struct in function scope
  struct EmitItem {
    static void exec(const ListItem& l) {if (!l.blocked) (l.f1)();}
  };
  std::for_each(local_list.begin(), local_list.end(),
		EmitItem::exec);

  return true;
}

Callback::SafeFunctor SafeEmitterArg<void>::connect(const Callback::SafeFunctor& f1) {
  Thread::Mutex::Lock lock(mutex);
  emission_list.push_back(ListItem(f1, Callback::SafeFunctorArg<int*>()));
  return f1;
}

Callback::SafeFunctor SafeEmitterArg<void>::connect(const Callback::SafeFunctor& f1, Releaser& r) {
  // In this method:
  // f1 is the functor we execute when we emit()
  // f2 is the functor we execute in our destructor if we are destroyed before the
  // remote object is, or in SafeEmitterArg<void>::disconnect()
  // f3 is the functor the remote object executes in its Releaser if it is destroyed
  // before we are, or if Releaser::operator=() is called

  Callback::SafeFunctor f3(Callback::make_val(*this, &SafeEmitterArg<void>::tracking_disconnect, f1));
  Callback::SafeFunctorArg<int*> f2(Callback::make_val(r, &Releaser::try_remove, f3));
  // we can't call Releaser::add() when holding our mutex or we will
  // get out of order locking, as Releaser's mutex is held when it
  // calls SafeEmitterArg<FreeArg>::tracking_disconnect() via f3, and
  // we don't need to do so
  r.add(f3);
  Thread::Mutex::Lock lock(mutex);
  try {
    emission_list.push_back(ListItem(f1, f2));
  }
  catch (...) {
    mutex.unlock();
    r.remove(f3);
    mutex.lock();
    throw;
  }
  return f1;
}

void SafeEmitterArg<void>::disconnect(const Callback::SafeFunctor& arg) {
  // we don't have lambdas in C++03, so use a struct in function scope
  struct Pred {
    // we can't pass const reference types as we bind with std::bind2nd below
    static bool pred(ListItem p, Callback::SafeFunctor f) {
      return (p.f1 == f);
    }
  };

  // in theory, we could have connected the same functor object more than
  // once, so cater for that as well as Releaser::try_remove() failing
  Thread::Mutex::Lock lock(mutex);
  std::list<ListItem>::iterator iter = emission_list.begin();
  for(;;) {
    iter = std::find_if(iter, emission_list.end(),
                        std::bind2nd(std::ptr_fun(Pred::pred), arg));
    if (iter != emission_list.end()) {
      int result = 0; // f2 might be a no-op
      // remove ourselves from the remote Releaser object
      (iter->f2)(&result);
      if (!result) { // we got the Releaser mutex lock or no-op
	// now remove this item from emission_list
        iter = emission_list.erase(iter);
      }
      else {
        mutex.unlock();
        // spin nicely
#ifdef CGU_USE_SCHED_YIELD
        sched_yield();
#else
        usleep(10);
#endif
	mutex.lock();
	// start again at the beginning - we have released the mutex
	// so our iterator may have become invalid
	iter = emission_list.begin();
      }
    }
    else break;
  }
}

// tracking disconnect() is the same as disconnect(), except that we do not
// execute f2 as the remote Releaser object will destroy its own functors
// in that case
void SafeEmitterArg<void>::tracking_disconnect(Callback::SafeFunctor arg) {
  // we don't have lambdas in C++03, so use a struct in function scope
  struct Pred {
    // we can't pass const reference types as we bind with std::bind2nd below
    static bool pred(ListItem p, Callback::SafeFunctor f) {
      return (p.f1 == f);
    }
  };

  Thread::Mutex::Lock lock(mutex);
  std::list<ListItem>::iterator iter = emission_list.begin();
  for (;;) {
    iter = std::find_if(iter, emission_list.end(),
			std::bind2nd(std::ptr_fun(Pred::pred), arg));
    if (iter != emission_list.end()) {
      // remove this item from emission_list
      iter = emission_list.erase(iter);
    }
    else break;
  }
}

void SafeEmitterArg<void>::block(const Callback::SafeFunctor& arg) {
  // we don't have lambdas in C++03, so use a struct in function scope
  struct Pred {
    // we can't pass const reference types as we bind with std::bind2nd below
    static bool pred(ListItem p, Callback::SafeFunctor f) {
      return (p.f1 == f);
    }
  };

  // in theory, we could have connected the same functor object
  // more than once, so cater for that
  Thread::Mutex::Lock lock(mutex);
  std::list<ListItem>::iterator iter = emission_list.begin();
  for (;;) {
    iter = std::find_if(iter, emission_list.end(),
			std::bind2nd(std::ptr_fun(Pred::pred), arg));
    if (iter != emission_list.end()) {
      iter->blocked = true;
      ++iter;
    }
    else break;
  }
}

void SafeEmitterArg<void>::unblock(const Callback::SafeFunctor& arg) {
  // we don't have lambdas in C++03, so use a struct in function scope
  struct Pred {
    // we can't pass const reference types as we bind with std::bind2nd below
    static bool pred(ListItem p, Callback::SafeFunctor f) {
      return (p.f1 == f);
    }
  };

  // in theory, we could have connected the same functor object
  // more than once, so cater for that
  Thread::Mutex::Lock lock(mutex);
  std::list<ListItem>::iterator iter = emission_list.begin();
  for (;;) {
    iter = std::find_if(iter, emission_list.end(),
			std::bind2nd(std::ptr_fun(Pred::pred), arg));
    if (iter != emission_list.end()) {
      iter->blocked = false;
      ++iter;
    }
    else break;
  }
}

} // namespace Cgu
