//
// Created by uos on 2022/3/18.
//

#include "Utils.h"
#include "Process.h"
#include "global.h"
#include <QFile>
#include <QStandardPaths>
#include <QtDBus/QtDBus>
#include <QtDBus/QDBusInterface>
#include <DSysInfo>
#include <sys/types.h>
#include <grp.h>
#include <pwd.h>
#include <QProcess>

DCORE_USE_NAMESPACE

namespace Utils {

    QPixmap renderSVG(const QString &path, const QSize &size)
    {
        QImageReader reader;
        QPixmap pixmap;
        reader.setFileName(path);
        if (reader.canRead()) {
            const qreal ratio = qApp->devicePixelRatio();
            reader.setScaledSize(size * ratio);
            pixmap = QPixmap::fromImage(reader.read());
            pixmap.setDevicePixelRatio(ratio);
        } else {
            pixmap.load(path);
        }
        return pixmap;
    }


    QJsonObject QStringToJson(const QString &jsonString)
    {
        QJsonObject jsonObject;
        QJsonDocument jsonDocument = QJsonDocument::fromJson(jsonString.toLocal8Bit().data());
        if (jsonDocument.isNull()) {
            return jsonObject;
        }
        jsonObject = jsonDocument.object();
        return jsonObject;

    }

    QString JsonToQString(QJsonObject &jsonObject)
    {
        return QString(QJsonDocument(jsonObject).toJson());
    }

    const QPixmap hidpiPixmap(const QString &path, const QSize &sz)
    {
        const auto ratio = qApp->devicePixelRatio();
        QPixmap iconPix = Utils::renderSVG(path, sz);
        iconPix.setDevicePixelRatio(ratio);

        return iconPix;
    }

    bool calculateDirSize(const QString &dirPath, const QStringList &excludeDir, quint64 &totalSizeBytes,
                          QString &error, bool samePartition)
    {
        totalSizeBytes = 0;
        QStringList args;
        if (samePartition) {
            args <<"du"<<"-sb" <<dirPath;
        } else {
            args <<"du"<<"-sbl" <<dirPath; // 跨分区需要计算硬链接大小
        }

        if (!excludeDir.isEmpty()) {
            args <<excludeDir;
        }

        QString out;
        if (!Process::spawnCmd("sudo", args, out, error)) {
            return false;
        }

        const int outColSize = 2; // size, dirPath
        foreach(QString line, out.split("\n")) {
            QStringList col = line.split("\t");
            if (outColSize != col.size()) {
                continue;
            }
            totalSizeBytes = col[0].trimmed().toULongLong();
            return true;
        }

        return false;
    }

    QString getUserName()
    {
        QString userPath = QStandardPaths::writableLocation(QStandardPaths::HomeLocation);
        return userPath.section("/", -1, -1);
    }

    bool authorization()
    {
        Authority::Result result;
        // 第一个参数是需要验证的action，和规则文件写的保持一致
        result = Authority::instance()->checkAuthorizationSync(
                "com.deepin.uosrecovery.checkAuthentication",
                UnixProcessSubject(getpid()),
                Authority::AllowUserInteraction);
        return result == PolkitQt1::Authority::Yes;
    }

    bool checkCommonUserAuthentication()
    {
        Authority::Result result;
        // 第一个参数是需要验证的action，和规则文件写的保持一致
        result = Authority::instance()->checkAuthorizationSync(
                "com.deepin.uosrecovery.checkCommonUserAuthentication",
                UnixProcessSubject(getpid()),
                Authority::AllowUserInteraction);
        return result == PolkitQt1::Authority::Yes;
    }

    bool readJsonFile(const QString &filepath, QJsonDocument &destdoc)
    {
        QFile file(filepath);
        if (!file.open(QIODevice::Text | QIODevice::ReadOnly)) {
            return false;
        }
        QByteArray data = file.readAll();
        file.close();
        QJsonParseError jsonError;
        destdoc = QJsonDocument::fromJson(data, &jsonError);
        if (jsonError.error != QJsonParseError::NoError) {
            return false;
        }

        return true;
    }


    QString byte2DisplaySize(quint64 size)
    {
        if (size >= KiB && size < MiB) {
            return QString().asprintf("%.1f KB", size * 1.0 / KiB);
        } else if (size >= MiB && size < GiB) {
            return QString().asprintf("%.1f MB", size * 1.0 / MiB);
        } else if (size >= GiB) {
            return QString().asprintf("%.1f GB", size * 1.0 / GiB);
        } else {
            return QString().asprintf("%llu Byte", size);
        }
    }

    quint64 parsePartitionSize(const QString& jsonString)
    {
        quint64 totalBytes = 0;
        if (jsonString.isEmpty()) {
            return totalBytes;
        }

        QString partitionString = jsonString;
        if (jsonString.contains("GB")) {
            totalBytes = partitionString.replace("GB", "").trimmed().toFloat() * GiB;
        } else if (jsonString.contains("MB")) {
            totalBytes = partitionString.replace("MB", "").trimmed().toFloat() * MiB;
        } else if (jsonString.contains("KB")) {
            totalBytes = partitionString.replace("KB", "").trimmed().toFloat() * KiB;
        } else if (jsonString.contains("Byte")) {
            totalBytes = partitionString.replace("Byte", "").trimmed().toUInt();
        }

        return totalBytes;
    }

    bool isAdminUserByDbus(const QString &userName, const QString &servicePath, const QString &accountPath,
                           const QString &accountInterfacePath, const QString &userInterfacePath)
    {
        QDBusInterface accountsInterface(servicePath, accountPath, accountInterfacePath,
                                         QDBusConnection::systemBus());
        if (!accountsInterface.isValid()) {
            return false;
        }

        QDBusReply<QString> userPathReply = accountsInterface.call("FindUserByName", userName);
        if (!userPathReply.isValid()) {
            return false;
        }

        QString userPath = userPathReply.value();
        QDBusInterface userInterface(servicePath, userPath, userInterfacePath,
                                     QDBusConnection::systemBus());
        if (!userInterface.isValid()) {
            return false;
        }

        QVariant userGroupProperty = userInterface.property("Groups");
        if (userGroupProperty.type() != QVariant::Type::StringList) {
            return false;
        }

        QStringList groupList = userGroupProperty.toStringList();
        if (groupList.contains("sudo")) {
            return true;
        } else {
            // 域管账户Groups有时获取到的是空，此时AccountType发现是1，否则域管账户这里值是2
            const int userTypeAdmin = 1; // 参考控制中心里的取值
            auto accountTypeReply = userInterface.property("AccountType");
            int accountType = accountTypeReply.toInt();
            if (userTypeAdmin == accountType) {
                return true;
            }
        }

        return false;
    }

    bool isAdminUser(const QString &userName)
    {
//        const QString servicePath = "com.deepin.daemon.Accounts";
//        const QString v20AccountPath = "/com/deepin/daemon/Accounts";
//        const QString v20AccountInterfacePath = "com.deepin.daemon.Accounts";
//        const QString v20UserInterfacePath = "com.deepin.daemon.Accounts.User";
//
//        bool isAdmin = isAdminUserByDbus(userName, servicePath, v20AccountPath, v20AccountInterfacePath,
//                                         v20UserInterfacePath);
//        if (!isAdmin) {
//            const QString v23AccountPath = "/org/deepin/daemon/Accounts1";
//            const QString v23AccountInterfacePath = "org.deepin.daemon.Accounts1";
//            const QString v23UserInterfacePath = "org.deepin.daemon.Accounts1.User";
//            isAdmin = isAdminUserByDbus(userName, servicePath, v23AccountPath, v23AccountInterfacePath,
//                                        v23UserInterfacePath);
//        }
        // 减少对dde dbus接口的依赖，改用系统调用的方式实现
        return isAdminUserBySysCall(userName);
    }

    bool isAdminUserBySysCall(const QString &userName)
    {
        bool isAdmin = false;
        struct passwd *pw = getpwnam(userName.toLocal8Bit().data());
        if (NULL == pw) {
            // invalid user
            return isAdmin;
        }

        int groupNum = 0;
        int retCode = getgrouplist(userName.toLocal8Bit().data(), pw->pw_gid, NULL, &groupNum);
        __gid_t *groups = static_cast<__gid_t *> (malloc (groupNum * sizeof(__gid_t)));
        if (NULL == groups) {
            qWarning()<<Q_FUNC_INFO<<"malloc failed, retCode = "<<retCode<<", userName = "<<userName;
            return false;
        }

        retCode = getgrouplist(userName.toLocal8Bit().data(), pw->pw_gid, groups, &groupNum);
        if (-1 == retCode) {
            qWarning()<<Q_FUNC_INFO<<"getgrouplist failed, groupNum = "<<groupNum<<", userName = "<<userName;
            if (NULL != groups) {
                free(groups);
                groups = NULL;
            }
            return false;
        }

        for (int i = 0; i < groupNum; ++i) {
            struct group *curGroup = getgrgid(groups[i]);
            if (curGroup != NULL) {
                QString groupName = curGroup->gr_name;
                if ("sudo" == groupName || "root" == groupName) {
                    isAdmin = true;
                    break;
                }
            }
        }

        if (NULL != groups) {
            free(groups);
            groups = NULL;
        }

        return isAdmin;
    }

    bool isOStree()
    {
        QString errMsg;
        QDBusInterface atomicInterface("org.deepin.AtomicUpgrade1",
                                       "/org/deepin/AtomicUpgrade1",
                                       "org.deepin.AtomicUpgrade1",
                                       QDBusConnection::systemBus());
        if (!atomicInterface.isValid()) {
            QDBusError dbusErr = atomicInterface.lastError();
            if (QDBusError::ErrorType::NoError != dbusErr.type()) {
                errMsg = dbusErr.message();
                qCritical() << Q_FUNC_INFO << "error: atomicInterface invalid, errMsg = " << errMsg;
                return false;
            }
        }

        QDBusReply<QStringList> listVersionReply = atomicInterface.call("ListVersion");
        if (!listVersionReply.isValid()) {
            errMsg = listVersionReply.error().message();
            qCritical() << Q_FUNC_INFO << "error: listVersionReply invalid, errMsg = "<<errMsg;
            return false;
        }

        QStringList versionList = listVersionReply.value();
        if (versionList.isEmpty()) {
            qCritical() << Q_FUNC_INFO << "error: ostree versionList isEmpty";
            return false;
        }

        // check SubmissionType
        QDBusReply<QStringList> subjectsReply = atomicInterface.call("QuerySubject", versionList);
        if (!subjectsReply.isValid()) {
            errMsg = subjectsReply.error().message();
            qCritical() << Q_FUNC_INFO << "error: subjectsReply invalid, errMsg = "<<errMsg;
            return false;
        }
        QStringList subjectList = subjectsReply.value();
        for (auto subject : subjectList) {
            QJsonObject jsonObj = Utils::QStringToJson(subject);
            if (jsonObj.contains("SubmissionType")) {
                CommitType type = static_cast<CommitType>(jsonObj.value("SubmissionType").toInt(-1));
                if (CommitType::InstallerCommit == type) {
                    return true;
                }
            }
        }

        return false;
    }

    quint64 getRemainSecond(const int &progress, QTime &baseTime)
    {
        if (progress <= 0) {
            return 1;
        }
        QTime currTime = QTime::currentTime();
        int elapsed = baseTime.msecsTo(currTime);
        double speed = (progress * 1000.0) / elapsed;
        return (100 - progress) / speed;
    }

    QString getOSVersion()
    {
        QString uosSysName = DSysInfo::uosSystemName(QLocale::English);
        DSysInfo::UosEdition edition = DSysInfo::uosEditionType();
        QString backupFlag = "UOS";
        QString version;
        if (DSysInfo::uosType() == DSysInfo::UosServer || DSysInfo::uosEditionType() == DSysInfo::UosEuler) {
            version = QString("%1-%2").arg(DSysInfo::majorVersion())
                    .arg(DSysInfo::minorVersion());
        } else if (DSysInfo::isDeepin()) {
            if (DSysInfo::UosEdition::UosCommunity == edition) {
                backupFlag = uosSysName;
            }
            version = QString("%1-V%2-%3").arg(backupFlag).arg(DSysInfo::majorVersion())
                    .arg(DSysInfo::minorVersion());
            QString buildVer = DSysInfo::buildVersion();
            if (!buildVer.isEmpty()) {
                version = QString("%1-%2").arg(version).arg(buildVer);
            }
        } else {
            version = QString("%1-%2").arg(DSysInfo::productVersion())
                    .arg(DSysInfo::productTypeString());
        }

        return version;
    }

    QString getOSVersionDisplay(bool localeEnglish)
    {
        QString versionDisplay;
        if (DSysInfo::uosType() == DSysInfo::UosServer || DSysInfo::uosEditionType() == DSysInfo::UosEuler) {
            versionDisplay = QString("%1 %2").arg(DSysInfo::majorVersion())
                    .arg(DSysInfo::minorVersion());
        } else if (DSysInfo::isDeepin()) {
            QLocale locale = localeEnglish ? QLocale::English : QLocale::system();
            QString uosSysName = DSysInfo::uosSystemName(locale);
            QString editionName = DSysInfo::uosEditionName(locale);
            versionDisplay = QString("%1 %2 %3").arg(uosSysName).arg(editionName).arg(DSysInfo::minorVersion());
        } else {
            versionDisplay = QString("%1 %2").arg(DSysInfo::productVersion())
                    .arg(DSysInfo::productTypeString());
        }

        return versionDisplay;
    }

    bool getDestPartInfoByDir(const QString &selectDir, QJsonObject &destInfo, QString &err)
    {
        // 找到目录所在分区
        destInfo = QJsonObject();
        QString destDir = selectDir;
        if (destDir.contains(" ")) {
            destDir = QString("\"%1\"").arg(selectDir);
        }

        QString cmd = QString("df %1 | awk 'END {print $1}'").arg(destDir);
        QString cmdLog = "";
        if (!Process::spawnCmd("/bin/bash", {"-c", cmd}, cmdLog, err)) {
            err = "get " + selectDir + " device error cmd : " + cmd + " err : " + err;
            return false;
        }

        // 找到目录所在分区的UUID
        cmdLog.replace(" ", "");
        cmdLog.replace("\n", "");
        if (cmdLog.isEmpty()) {
            qInfo()<<"getDestPartInfoByDir destDir = "<<destDir<<" ,err = "<<err;
            return false;
        }

        QJsonArray destInfoArray;
        err = "";
        if (!getLsblkJsonReturn(QString("lsblk %1 -pbJO").arg(cmdLog), destInfoArray, err)){
            return false;
        }

        if (destInfoArray.size() > 0) {
            destInfo = destInfoArray.first().toObject();
        } else {
            err = "no info get";
            return false;
        }

        return true;
    }

    QStringList getCmdListReturn(const QString &cmd)
    {
        QString cmdLog = "";
        QString cmdErr = "";
        if (!Process::spawnCmd("/bin/bash", {"-c", cmd}, cmdLog, cmdErr)) {
            //qWarning() << "error cmd : " << cmd << "err : " << cmdErr;
            return {};
        }

        cmdLog.replace("0B", "0");
        cmdLog.replace("none", "0");
        cmdLog.replace("null", "0");

#if QT_VERSION >= QT_VERSION_CHECK(5,15,0)
        return cmdLog.split("\n", Qt::SkipEmptyParts);
#else
        return cmdLog.split("\n", QString::SkipEmptyParts);
#endif
    }

    int currentPlatformByString(const QString &platform)
    {
        QMap<QString, PlatformType> platformMap = {
            { "x86_64", PlatformType::X86_64 },
            { "i386", PlatformType::I386 },
            { "i686", PlatformType::I686 },
            { "amd64", PlatformType::AMD64 },
            { "x86", PlatformType::X86 },
            { "sw_64", PlatformType::SW_64 },
            { "mips64", PlatformType::MIPS64 },
            { "loongarch64", PlatformType::LOONGARCH64 },
            { "aarch64", PlatformType::AARCH64 }
        };

        if (!platformMap.keys().contains(platform)) {
            return PlatformType::UNKNOW;
        }

        return platformMap[platform];
    }

    bool isEFIMode()
    {
        if (QFile::exists("/sys/firmware/efi")) {
            return true;
        }

        QStringList infoList = getCmdListReturn("uname -m");
        if (infoList.size() > 0) {
            if (infoList.first().contains("SW")) {
                return true;
            }
        }

        return false;
    }

    bool isSystemUpgrade()
    {
        // 根据grub.cfg文件判断是否使用了ab-recovery升级了系统
        QString cmd = "cat /boot/grub/grub.cfg | grep deepin-ab-recovery";
        QString cmdLog = "";
        QString err = "";
        if (!Process::spawnCmd("/bin/bash", {"-c", cmd}, cmdLog, err)) {
            // err = "get " + selectDir + " device error cmd : " + cmd + " err : " + err;
            return false;
        }

        if ((!cmdLog.contains("linux")) && (!cmdLog.contains("initrd"))) {
            return false;
        }
        return true;
    }

    bool getLsblkJsonReturn(const QString &cmd, QJsonArray &jsonOut, QString &err)
    {
        jsonOut = QJsonArray();

        QString cmdLog = "";
        if (!Process::spawnCmd("/bin/bash", {"-c", cmd}, cmdLog, err)) {
            err = "getLsblkJsonReturn error cmd : " + cmd + " err : " + err;
            return false;
        }

        cmdLog.replace("0B", "0");
        cmdLog.replace("none", "0");
        cmdLog.replace("null", "0");

        QJsonParseError jsonError;
        QJsonObject cmdDestInfo = QJsonDocument::fromJson(cmdLog.toStdString().data(), &jsonError).object();
        if (jsonError.error != QJsonParseError::NoError) {
            err = "getLsblkJsonReturn read info json failed " + jsonError.errorString();
            return false;
        }

        jsonOut = cmdDestInfo.value("blockdevices").toArray();

        return true;
    }


    bool getDimFileJsonInfo(const QString &fileName, QJsonObject &fileInfo, QString &err)
    {
        fileInfo = QJsonObject();

        QString cmd = QString("deepin-clone -i %1").arg(fileName);
        QString cmdLog = "";
        err = "";
        if (!Process::spawnCmd("/bin/bash", {"-c", cmd}, cmdLog, err)) {
            err = "getDimFileTotalSize error cmd : " + cmd + "err : " + err;
            return false;
        }

        // 解析总大小的值
        QJsonParseError jsonError;
        QJsonDocument sizeDoc = QJsonDocument::fromJson(cmdLog.toLocal8Bit(), &jsonError);
        if (jsonError.error != QJsonParseError::NoError) {
            err = "getDimFileTotalSize read json error";
            return false;
        }

        fileInfo = sizeDoc.object();
        return true;
    }

    // 根据配置的设备类型过滤设备，返回过滤后剩余设备的数量
    int filterDevice(QJsonArray &deviceArray)
    {
        QStringList deviceTypes = {"disk"};
        QJsonArray::iterator iter = deviceArray.begin();
        for (;iter != deviceArray.end();) {
            QJsonObject deviceObj = iter->toObject();
            qint64 deviceSize = deviceObj.value("size").toVariant().toLongLong() / MiB;
            QString deviceType = deviceObj.value("type").toString();

            if ((!deviceTypes.contains(deviceType))// 过滤掉不在期望类型中的设备
                || (deviceSize <= 1)) {// 过滤掉于1M的设备
                iter = deviceArray.erase(iter);
                continue;
            } else {
                ++iter;
            }
        }

        return deviceArray.size();
    }

    bool getFormatPartitionCmd(const QString &label, const QString &filesystem, const QString &path, QString &cmd, QString &err)
    {
        cmd = "";

        if (filesystem.isEmpty()) {
            return false;
        }

        QMap<QString, QString> cmdMap;
        cmdMap.insert("btrfs", QString("mkfs.btrfs -f [-L %1] %2"));
        cmdMap.insert("efi", QString("mkfs.vfat -F32 -v -I [-n %1] %2"));
        cmdMap.insert("ext2", QString("mkfs.ext2 -F [-L %1] %2"));
        cmdMap.insert("ext3", QString("mkfs.ext3 -F [-L %1] %2"));
        cmdMap.insert("ext4", QString("mkfs.ext4 -F [-L %1] %2"));
        cmdMap.insert("f2fs", QString("mkfs.f2fs -f [-l %1] %2"));
        cmdMap.insert("fat16", QString("mkfs.fat -F16 -v -I [-n %1] %2"));
        cmdMap.insert("fat32", QString("mkfs.fat -F32 -v -I [-n %1] %2"));
        cmdMap.insert("hfs", QString("/usr/bin/hformat -f [-l %1] %2"));
        cmdMap.insert("hfsplus", QString("/usr/bin/hpfsck -f [-v %1] %2"));
        cmdMap.insert("jfs", QString("mkfs.jfs -q [-L %1] %2"));
        cmdMap.insert("nilfs2", QString("mkfs.nilfs2 -f [-L %1] %2"));
        cmdMap.insert("ntfs", QString("mkfs.ntfs -Q -v -F [-L %1] %2"));
        cmdMap.insert("reiser4", QString("mkfs.reiser4 --force --yes [--label %1] %2"));
        cmdMap.insert("reiserfs", QString("mkfs.reiserfs -f [--label %1] %2"));
        cmdMap.insert("xfs", QString("mkfs.xfs -f [-L %1] %2"));
        cmdMap.insert("recovery", QString("mkfs.ext4 -F [-L %1] %2"));
        cmdMap.insert("linux-swap", QString("mkswap -f [-L %1] %2"));
        cmdMap.insert("swap", QString("mkswap -f [-L %1] %2"));
        cmdMap.insert("vfat", QString("mkfs.vfat -F32 [-n %1] %2"));
        cmdMap.insert("ext4_ls", QString("mkfs.ext4 -O ^64bit -F [-L %1] %2"));
        cmdMap.insert("default", QString("mkfs -t %1 ").arg(filesystem) + "[-L %1] %2");

        QString key = filesystem;
        if (!cmdMap.contains(filesystem)) {
            key = "default";
        } else {
            // 针对ext4文件系统，申威，龙芯这些平台需要特殊处理
            if (filesystem == "ext4") {
                QStringList infoList = Utils::getCmdListReturn("uname -m");
                if (infoList.size() > 0) {
                    if (infoList.first().contains(QRegExp("sw|loongson|loongarch64"))) {
                        key = "ext4_ls";
                    }
                }
            }
        }

        cmd = cmdMap.value(key).arg(label, path);

        if (label.isEmpty()) {
            cmd.replace(QRegExp("\\[[^\\[\\]]*\\]"), "");
        } else {
            cmd.replace(QRegExp("[\\[|\\]]"), "");
        }

        return true;
    }

    int getCPUCores()
    {
        CpuInfo cpuInfo;
        bool retCode = getCpuInfo(cpuInfo);
        if (!retCode) {
            return 1;
        }

        return cpuInfo.cpuNum;
    }

    bool getProcessNameByPid(pid_t pid, QString &name)
    {
        QString cmd = QString("cat /proc/%1/stat | awk -F\")\" '{print $1}' | awk -F\"(\" '{print $2}'").arg(pid);
        QStringList args;
        args <<"-c"<< cmd;
        QString err;
        // process name (may be truncated by kernel if it's too long)
        if (!Process::spawnCmd("/bin/bash", args, name, err)) {
            return false;
        }

        name.replace(" ", "");
        name.replace("\n", "");

        return true;
    }

    QString getKernelArch()
    {
        QString arch = "";
        QString cmd = QString("dpkg --print-architecture");
        QStringList args;
        args <<"-c"<< cmd;
        QString out;
        QString err;
        if (!Process::spawnCmd("/bin/bash", args, out, err)) {
            return arch;
        }

        arch = out.trimmed();
        return arch;
    }

    QString getOSEditionType()
    {
        QString editionType = "";
        DSysInfo::UosEdition edition = DSysInfo::uosEditionType();
        switch (edition) {
            case DSysInfo::UosEdition::UosProfessional: {
                editionType = "Professional";
                break;
            }
            case DSysInfo::UosEdition::UosHome: {
                editionType = "Home";
                break;
            }
            case DSysInfo::UosEdition::UosCommunity: {
                editionType = "Community";
                break;
            }
            case DSysInfo::UosEdition::UosMilitary: {
                editionType = "Military";
                break;
            }
            case DSysInfo::UosEdition::UosEnterprise: {
                editionType = "Enterprise";
                break;
            }
            case DSysInfo::UosEdition::UosEnterpriseC: {
                editionType = "EnterpriseC";
                break;
            }
            case DSysInfo::UosEdition::UosEuler: {
                editionType = "Euler";
                break;
            }
            case DSysInfo::UosEdition::UosMilitaryS: {
                editionType = "MilitaryS";
                break;
            }
            case DSysInfo::UosEdition::UosDeviceEdition: {
                editionType = "DeviceEdition";
                break;
            }
            case DSysInfo::UosEdition::UosEducation: {
                editionType = "Education";
                break;
            }
            default: {
                editionType = "";
                break;
            }
        }

        return editionType;
    }

    QString getGhostFileName(const QString &curTime)
    {
        // ghost_20_Professional_1050_109_amd64_efi_20221016_101734.uimg
        static QString osEditionType = Utils::getOSEditionType();
        static QString majorVer = DSysInfo::majorVersion();
        static QString minorVer = DSysInfo::minorVersion();
        static QString buildVer = DSysInfo::buildVersion();
        static QString arch = Utils::getKernelArch();
        static bool isEFI = Utils::isEFIMode();
        QString bootMode = isEFI ? "efi" : "legacy";

        QString ghostName = QString("ghost_%1_%2_%3").arg(majorVer).arg(osEditionType).arg(minorVer);
        if (!buildVer.isEmpty()) {
            ghostName += "_" + buildVer;
        }

        if (!arch.isEmpty()) {
            ghostName += "_" + arch;
        }

        ghostName += "_" + bootMode;

        if (!curTime.isEmpty()) {
            ghostName += "_" + curTime;
        }

        ghostName += ".uimg";

        return ghostName;
    }

    bool isLVM(bool &isEnrypted)
    {
        isEnrypted = false;
        QString cmd = QString("lsblk --output TYPE,UUID | grep -E 'crypt|lvm'");
        QStringList args;
        args <<"-c"<< cmd;
        QString out;
        QString err;
        if (!Process::spawnCmd("/bin/bash", args, out, err)) {
            return false;
        }

        if (!out.contains("lvm")) {
            return false;
        }

        if (out.contains("crypt")) {
            isEnrypted = true;
        }

#if QT_VERSION >= QT_VERSION_CHECK(5,15,0)
        QStringList outList = out.split("\n", Qt::SkipEmptyParts);
#else
        QStringList outList = out.split("\n", QString::SkipEmptyParts);
#endif

        for (QString &line : outList) {
            QString fsType = line.left(line.indexOf(" "));
            if ("lvm" == fsType) {
                QString uuid = line.right(line.length() - line.lastIndexOf(" "));
                // check uuid in fstab
                args.clear();
                QString uuidOut;
                args <<"-c"<< "cat /etc/fstab | grep " +  uuid + "| awk '{print $1}' ";
                if (Process::spawnCmd("/bin/bash", args, uuidOut, err)) {
                    if (!uuidOut.isEmpty() && !uuidOut.startsWith("#")) {
                        return true;
                    }
                }
            }
        }

        isEnrypted = false;
        return false;
    }

    bool getCpuInfo(CpuInfo &cpu)
    {
        QString cmd = QString("lscpu");
        QStringList args;
        args <<"-c"<< cmd;
        QString cpuOut;
        QString err;
        if (!Process::spawnCmd("/bin/bash", args, cpuOut, err)) {
            qInfo()<<"CpuInfo::init call lscpu failed, err = "<<err;
            return false;
        }

        QStringList cpuList;
#if QT_VERSION >= QT_VERSION_CHECK(5,15,0)
        cpuList = cpuOut.split("\n", Qt::SkipEmptyParts);
#else
        cpuList = cpuOut.split("\n", QString::SkipEmptyParts);
#endif

        for (auto line : cpuList) {
            line = line.trimmed();
            if (line.startsWith("Architecture:")) {
                cpu.arch = line.right(line.length() - line.indexOf(":") - 1);
                cpu.arch = cpu.arch.trimmed();
            } else if (line.startsWith("Byte Order:")) {
                cpu.byteOrder = line.right(line.length() - line.indexOf(":") - 1);
                cpu.byteOrder = cpu.byteOrder.trimmed();
            } else if (line.startsWith("CPU(s):")) {
                QString cpus = line.right(line.length() - line.indexOf(":") - 1);
                cpu.cpuNum = cpus.trimmed().toInt();
            } else if (line.startsWith("Model name:")) {
                cpu.modelName = line.right(line.length() - line.indexOf(":") - 1);
                cpu.modelName = cpu.modelName.trimmed();
            }
        }
        return true;
    }

    static void getMapInfoFromLshw(const QString &info, QMap<QString, QString> &mapInfo, const QString &ch)
    {
#if QT_VERSION >= QT_VERSION_CHECK(5,15,0)
        QStringList infoList = info.split(ch, Qt::SkipEmptyParts);
#else
        QStringList infoList = info.split(ch, QString::SkipEmptyParts);
#endif
        if (infoList.size() == 2) {
            QString key = infoList[0].trimmed();
            QString val = infoList[1].trimmed();
            mapInfo.insert(key, val);
        }
    }

     bool getLshwInfo(QMap<QString, QMap<QString, QString>> &lshwMap)
     {
        QString cmd = QString("lshw");
        QStringList args;
        args <<"-c"<< cmd;
        QString out;
        QString err;
        if (!Process::spawnCmd("/bin/bash", args, out, err)) {
            qInfo()<<"getLshwInfo call lshw failed, err = "<<err;
            return false;
        }

        QStringList itemList = out.split("*-");
        for (auto &item : itemList) {
            if (item.startsWith("display")) {
#if QT_VERSION >= QT_VERSION_CHECK(5,15,0)
                QStringList displayList = item.split("\n", Qt::SkipEmptyParts);
#else
                QStringList displayList = item.split("\n", QString::SkipEmptyParts);
#endif
                for (auto &display : displayList) {
                    display = display.trimmed();
                    QMap<QString, QString> &displayMap = lshwMap["display"];
                    getMapInfoFromLshw(display, displayMap, ":");
//                    qInfo()<<"display = "<<display;
                }
            }
        }

        return true;
     }

    bool getVGAFromLshw(QString &display)
    {
        display = "";
        QMap<QString, QMap<QString, QString>> lshwMap;
        if (!getLshwInfo(lshwMap)) {
            return false;
        }

        if (lshwMap.contains("display")) {
            const QMap<QString, QString> &displayValMap = lshwMap.value("display");
            if(displayValMap.contains("product")) {
                display = displayValMap.value("product");
                return true;
            }
        }

        return false;
    }

    QStringList getAllSystemBackupUUID(const QString &recoveryConf)
    {
        QStringList uuidList;
        if (!recoveryConf.endsWith(".ini")) {
            return uuidList;
        }

        // get all system backup partition uuid
        const QString uosRecoveryConf = "/" + UOS_RECOVERY_INI;
        QSettings settings(uosRecoveryConf, QSettings::IniFormat);
        settings.beginGroup(BACKUP_GROUP);
        QString deviceUUID = settings.value(BACKUP_DEVICE_UUID_KEY).toString();
        QString historyDeviceUUID = settings.value(BACKUP_HISTORY_DEVICE_UUID_KEY).toString();
        settings.endGroup();

#if QT_VERSION >= QT_VERSION_CHECK(5,15,0)
        uuidList = historyDeviceUUID.split(",", Qt::SkipEmptyParts);
#else
        uuidList = historyDeviceUUID.split(",", QString::SkipEmptyParts);
#endif
        uuidList.append(deviceUUID);
        uuidList.removeDuplicates();

        return uuidList;
    }

    bool getFsTypeAndDevicePathByDir(const QString &dir, QString &fsType, QString &devicePath)
    {
        fsType = "";
        QString dirPath = dir;
        if (dirPath.contains(" ")) {
            dirPath = QString("\"%1\"").arg(dirPath);
        }

        QString cmd = QString("df -T %1 | awk 'END {print $1\":\"$2}'").arg(dirPath);
        QString output = "";
        QString err;
        if (!Process::spawnCmd("/bin/bash", {"-c", cmd}, output, err)) {
            qInfo()<<"getFsTypeByDir failed, err = "<<err<<", dir = "<<dir;
            return false;
        }

        if (!err.isEmpty()) {
            qInfo()<<"getFsTypeByDir error, err = "<<err;
            return false;
        }

        output = output.trimmed();
        int pos = output.indexOf(":");
        int len = output.length();
        devicePath = output.left(pos);
        fsType = output.right(len - pos - 1);

        return true;
    }

    bool isDeviceRemoveble(const QString &devicePath)
    {
        QString cmd = QString("lsblk --output HOTPLUG,PATH | grep %1").arg(devicePath);
        QString output = "";
        QString err;
        if (!Process::spawnCmd("/bin/bash", {"-c", cmd}, output, err)) {
            qInfo()<<"isDeviceRemoveble failed, err = "<<err;
            return false;
        }

        output = output.left(output.indexOf("/")).trimmed();
        return output == "1";
    }

    int getGhostPolicy(const QString &recoveryConf)
    {
        bool ok = false;
        QSettings settings(recoveryConf, QSettings::IniFormat);
        settings.beginGroup(GHOST_GROUP);
        int ghostPolicyVal = settings.value(GHOST_POLICY).toInt(&ok);
        settings.endGroup();

        if (ok) {
            return ghostPolicyVal;
        }

        return GhostPolicy::GHOST_POLICY_UNKOWN;
    }

    int getUILayoutType(int module)
    {
        int defaultLayout = -1; // invalid
        const QMap<int, QPair<QString, QVector<int>>> moduleLayoutMap = {
                // QVector 字段里第一个值要求都是VERTICAL的
                { UI::UI_BACKUP_MODULE, {
                        UI_LAYOUT_BACKUP, {UI::UI_LAYOUT_BACKUP_VERTICAL, UI::UI_LAYOUT_BACKUP_HORIZONTAL}
                    }
                },
                { UI::UI_RESTORE_MODULE, {
                        UI_LAYOUT_RESTORE, {UI::UI_LAYOUT_RESTORE_VERTICAL, UI::UI_LAYOUT_RESTORE_HORIZONTAL}
                    }
                },
                { UI::UI_ADVANCE_MODULE, {
                        UI_LAYOUT_ADVANCE, {UI::UI_LAYOUT_ADVANCE_VERTICAL, UI::UI_LAYOUT_ADVANCE_HORIZONTAL}
                    }
                }
        };

        if (!moduleLayoutMap.contains(module)) {
            return defaultLayout;
        }

        const QPair<QString, QVector<int>> &layoutPair = moduleLayoutMap[module];
        QString rootPath = "/";
        QString recoveryConf = rootPath + UOS_RECOVERY_INI;
        QSettings settings(recoveryConf, QSettings::IniFormat);
        settings.beginGroup(LAYOUT_GROUP);
        int layoutVal = settings.value(layoutPair.first).toInt();
        settings.endGroup();

        if (layoutPair.second.contains(layoutVal)) {
            defaultLayout = layoutVal;
        } else {
            // UI调整后，默认采用VERTICAL布局
            defaultLayout = layoutPair.second.first();
        }

        return defaultLayout;
    }

    bool isDirectionRTL()
    {
        static QString locale = QLocale::system().name();
        if ("ug_CN" == locale) {
            return true;
        }

        return false;
    }
}
