r"""
S-Boxes used in cryptographic schemes

This module provides the following SBoxes:

constructions
    - BrackenLeander ([BraLea2008]_)
    - CarletTangTangLiao ([CTTL2014]_)
    - Gold ([Gol1968]_)
    - Kasami ([Kas1971]_)
    - Niho ([Dob1999a]_)
    - Welch ([Dob1999b]_)

8 bit to 8 bit
    - AES ([DR2002]_)
    - Anubis ([BR2000a]_)
    - ARIA_s2 ([KKPSSSYYLLCHH2004]_)
    - BelT ([Bel2011]_)
    - Camellia ([AIKMMNT2001]_)
    - CMEA ([WSK1997]_)
    - Chiasmus ([STW2013]_)
    - CLEFIA_S0, CLEFIA_S1 ([SSAMI2007]_)
    - Crypton_0_5 ([Lim]_)
    - Crypton_1_0_S0, ..., Crypton_1_0_S3 ([Lim2001]_)
    - CS_cipher ([SV2000]_)
    - CSA ([WW2005]_)
    - CSS ([BD2004]_)
    - DBlock ([WZY2015]_)
    - E2 ([KMAUTOM2000]_)
    - Enocoro ([WFYTP2008]_)
    - Fantomas ([GLSV2014]_)
    - FLY ([KG2016]_)
    - Fox ([VJ2004]_)
    - Iceberg ([SPRQL2004]_)
    - Iraqi (:wikipedia:`Iraqi_block_cipher`)
    - iScream ([GLSVJGK2014]_)
    - Kalyna_pi0, ..., Kalyna_pi3 ([OGKRKGBDDP2015]_)
    - Khazad ([BR2000b]_)
    - Kuznyechik (Kuznechik, Streebog, Stribog) ([Fed2015]_)
    - MD2 ([Kal1992]_)
    - newDES ([Sco1985]_)
    - Picaro ([PRC2012]_)
    - Safer ([Mas1994]_)
    - Scream ([CDL2015]_,[GLSVJGK2014]_)
    - SEED_S0, SEED_S1 ([LLYCL2005]_)
    - SKINNY_8 ([BJKLMPSSS2016]_)
    - Skipjack ([U.S1998]_)
    - SNOW_3G_sq ([ETS2006a]_)
    - SMS4 ([Ltd06]_)
    - Turing ([RH2003b]_)
    - Twofish_p0, Twofish_p1 ([SKWWHF1998]_)
    - Whirlpool ([BR2000c]_)
    - Zorro ([GGNS2013]_)
    - ZUC_S0, ZUC_S1 ([ETS2011]_)

6 bit to 6 bit
    - Fides_6 ([BBKMW2013]_)
    - APN_6 ([BDMW2010]_)
    - SC2000_6 ([SYYTIYTT2002]_)

5 bit to 5 bit
    - Ascon ([DEMS2016]_)
    - Fides_5 ([BBKMW2013]_)
    - SC2000_5 ([SYYTIYTT2002]_)

4 bit to 4 bit
    - DES_S1_1, ..., DES_S1_4, ..., DES_S8_4 ([U.S1999]_)
    - Lucifer_S0, Lucifer_S1 ([Sor1984]_)
    - GOST_1, ..., GOST_8 (http://www.cypherpunks.ru/pygost/)
    - GOST2_1, GOST2_2 (http://www.cypherpunks.ru/pygost/)
    - Magma_1, ..., Magma_8 ([Fed2015]_)
    - GOST_IETF_1, ..., GOST_IETF_8 (http://www.cypherpunks.ru/pygost/)
    - Hummingbird_2_S1, ..., Hummingbird_2_S4 ([ESSS2011]_)
    - LBlock_0, ..., LBlock_9 ([WZ2011]_)
    - SERPENT_S0, ..., SERPENT_S7 ([BAK1998]_)
    - KLEIN ([GNL2011]_)
    - MIBS ([ISSK2009)]
    - Midori_Sb0 (MANTIS), Midori_Sb1 ([BBISHAR2015]_)
    - Noekeon ([DPVAR2000]_)
    - Piccolo ([SIHMAS2011]_)
    - Panda ([YWHWXSW2014]_)
    - PRESENT ([BKLPPRSV2007]_)
    - GIFT ([BPPSST2017]_)
    - Pride ([ADKLPY2014]_)
    - PRINCE ([BCGKKKLNPRRTY2012]_)
    - Prost ([KLLRSY2014]_)
    - Qarma_sigma0, Qarma_sigma1, Qarma_sigma2 ([Ava2017]_)
    - REC_0 (earlier version of [ZBLRYV2015]_)
    - Rectangle ([ZBLRYV2015]_)
    - SC2000_4 ([SYYTIYTT2002]_)
    - SKINNY_4 ([BJKLMPSSS2016]_)
    - TWINE ([SMMK2013]_)

    - Luffa_v1 ([DCSW2008]_)
    - Luffa ([DCSW2008]_)
    - BLAKE_1, ..., BLAKE_9 ([AHMP2008]_)
    - JH_S0, JH_S1 ([Wu2009]_)
    - SMASH_256_S1, ..., SMASH_256_S3 ([Knu2005]_)

    - Anubis_S0, Anubis_S1 ([BR2000a]_)
    - CLEFIA_SS0, ..., CLEFIA_SS3 ([SSAMI2007]_)
    - Enocoro_S4 ([WFYTP2008]_)
    - Iceberg_S0, Iceberg_S1 ([SPRQL2004]_)
    - Khazad_P, Khazad_Q ([BR2000b]_)
    - Whirlpool_E, Whirlpool_R ([BR2000c]_)
    - CS_cipher_F, CS_cipher_G ([SV2000]_)
    - Fox_S1, ..., Fox_S3 ([VJ2004]_)
    - Twofish_Q0_T0, ..., Twofish_Q0_T3, Twofish_Q1_T0, ..., Twofish_Q1_T3 ([SKWWHF1998]_)
    - Kuznyechik_nu0, Kuznyechik_nu1, Kuznyechik_sigma, Kuznyechik_phi ([BPU2016]_)

    - UDCIKMP11 ([UDCIKMP2011]_)
    - Optimal_S0, ..., Optimal_S15 ([LP2007]_)
    - Serpent_type_S0, ..., Serpent_type_S19 ([LP2007]_)
    - Golden_S0, ..., Golden_S3 ([Saa2011]_)

    - representatives for all 302 affine equivalence classes ([dCa2007]_)

3 bit to 3 bit
    - SEA ([SPGQ2006]_)
    - PRINTcipher ([KLPR2010]_)

Additionally this modules offers a dictionary `sboxes` of all implemented above S-boxes
for the purpose of easy iteration over all available S-boxes.

EXAMPLES:

We can print the S-Boxes with differential uniformity 2::

    sage: from sage.crypto.sboxes import sboxes
    sage: sorted(name for name, s in sboxes.items()
    ....:     if s.differential_uniformity() == 2)
    ['APN_6', 'Fides_5', 'Fides_6', 'PRINTcipher', 'SC2000_5', 'SEA']

AUTHOR:

- Leo Perrin: initial collection of sboxes
- Friedrich Wiemer (2017-05-12): refactored list for inclusion in SAGE
"""
import sys

from sage.crypto.sbox import SBox
from sage.misc.functional import is_odd, is_even


def bracken_leander(n):
    r"""
    Return the Bracken-Leander construction.

    For n = 4*k and odd k, the construction is `x \mapsto x^{2^{2k} + 2^k + 1}`
    over `\GF{2^n}`

    INPUT:

    - ``n`` -- size of the S-Box

    EXAMPLES::

        sage: from sage.crypto.sboxes import bracken_leander
        sage: sbox = bracken_leander(12); [sbox(i) for i in range(8)]
        [0, 1, 2742, 4035, 1264, 408, 1473, 1327]
    """
    if n % 4 == 1 or is_even(n / 4):
        raise TypeError("Bracken-Leander functions are only defined for n = 4k with k odd")

    k = n / 4
    e = 2**(2*k) + 2**k + 1
    return monomial_function(n, e)


def carlet_tang_tang_liao(n, c=None, bf=None):
    r"""
    Return the Carlet-Tang-Tang-Liao construction.

    See [CTTL2014]_ for its definition.

    INPUT:

    - ``n`` -- integer, the bit length of inputs and outputs, has to be even and >= 6
    - ``c`` -- element of `\GF{2^{n-1}}` used in the construction
        (default: random element)
    - ``f`` -- Function from `\GF{2^n} \to \GF{2}` or BooleanFunction on `n-1` bits
        (default: ``x -> (1/(x+1)).trace())``

    EXAMPLES::

        sage: from sage.crypto.sboxes import carlet_tang_tang_liao as cttl
        sage: cttl(6).differential_uniformity()
        4
    """
    from sage.crypto.boolean_function import BooleanFunction
    from sage.rings.finite_rings.finite_field_constructor import GF

    if n < 6 or n % 2:
        raise TypeError("n >= 6 has to be even")
    K = GF(2**(n-1))
    L = GF(2**n)

    if c is None:
        c = K.random_element()
        while c.trace() == 0 or (1/c).trace() == 0:
            c = K.random_element()
    elif c.trace() == 0 or (1/c).trace() == 0:
        raise TypeError("c.trace() and (1/c).trace() have to be 1")

    if bf is None:
        def bf(x):
            if x == 1:
                return 0
            return (1/(x+1)).trace()

    elif isinstance(bf, (BooleanFunction,)):
        bf_f2 = bf
        def bf(x):
            xprime = map(int, x.polynomial().list())
            xprime += [0]*(n-1 - len(xprime))
            return int(bf_f2(xprime))

    def f(x):
        xs = x.polynomial().list()
        xs += [0]*(n - len(xs))
        xprime = K(xs[:n-1])

        if xprime == 0:
            res = [0]*(n-1), bf(xprime/c) + xs[-1]
        elif xs[-1] == 0:
            res = (1/xprime).polynomial().list(), bf(xprime)
        else:
            res = (c/xprime).polynomial().list(), bf(xprime/c) + 1

        res = res[0] + [0]*(n-1-len(res[0])) + [res[1]]

        return L(res)

    return SBox([f(L(x)) for x in GF(2)**n])


def gold(n, i):
    r"""
    Return the Gold function defined by `x \mapsto x^{2^i + 1}` over `\GF{2^n}`.

    INPUT:

    - ``n`` -- size of the S-Box
    - ``i`` -- a positive integer

    EXAMPLES::

        sage: from sage.crypto.sboxes import gold
        sage: gold(3, 1)
        (0, 1, 3, 4, 5, 6, 7, 2)
        sage: gold(3, 1).differential_uniformity()
        2
        sage: gold(4, 2)
        (0, 1, 6, 6, 7, 7, 7, 6, 1, 7, 1, 6, 1, 6, 7, 1)
    """
    e = 2**i + 1
    return monomial_function(n, e)


def kasami(n, i):
    r"""
    Return the Kasami function defined by `x \mapsto x^{2^{2i} - 2^i + 1}` over `\GF{2^n}`.

    INPUT:

    - ``n`` -- size of the S-Box
    - ``i`` -- a positive integer

    EXAMPLES::

        sage: from sage.crypto.sboxes import kasami
        sage: kasami(3, 1)
        (0, 1, 3, 4, 5, 6, 7, 2)
        sage: from sage.crypto.sboxes import gold
        sage: kasami(3, 1) == gold(3, 1)
        True
        sage: kasami(4, 2)
        (0, 1, 13, 11, 14, 9, 6, 7, 10, 4, 15, 2, 8, 3, 5, 12)
        sage: kasami(4, 2) != gold(4, 2)
        True
    """
    e = 2**(2*i) - 2**i + 1
    return monomial_function(n, e)


def niho(n):
    r"""
    Return the Niho function over `\GF{2^n}`.

    It is defined by `x \mapsto x^{2^t + 2^s - 1}` with `s = t/2` if t is even
    or `s = (3t+1)/2` if t is odd.

    INPUT:

    - ``n`` -- size of the S-Box

    EXAMPLES::

        sage: from sage.crypto.sboxes import niho
        sage: niho(3)
        (0, 1, 7, 2, 3, 4, 5, 6)

        sage: niho(3).differential_uniformity()
        2
    """
    if not is_odd(n):
        raise TypeError("Niho functions are only defined for n odd")

    t = (n - 1) / 2

    if is_even(t):
        e = 2**t + 2**(t/2) - 1
    else:
        e = 2**t + 2**((3*t+1)/2) - 1

    return monomial_function(n, e)


def welch(n):
    r"""
    Return the Welch function defined by `x \mapsto x^{2^{(n-1)/2} + 3}` over `\GF{2^n}`.

    INPUT:

    - ``n`` -- size of the S-Box

    EXAMPLES::

        sage: from sage.crypto.sboxes import welch
        sage: welch(3)
        (0, 1, 7, 2, 3, 4, 5, 6)
        sage: welch(3).differential_uniformity()
        2
    """
    if not is_odd(n):
        raise TypeError("Welch functions are only defined for odd n")

    t = (n - 1) / 2
    e = 2**t + 3
    return monomial_function(n, e)


def monomial_function(n, e):
    r"""
    Return an S-Box as a function `x^e` defined over `\GF{2^n}`.

    INPUT:

    - ``n`` -- size of the S-Box (i.e. the degree of the finite field extension)
    - ``e`` -- exponent of the monomial function

    EXAMPLES::

        sage: from sage.crypto.sboxes import monomial_function
        sage: S = monomial_function(7, 3)
        sage: S.differential_uniformity()
        2
        sage: S.input_size()
        7
        sage: S.is_permutation()
        True
    """
    from sage.rings.polynomial.polynomial_ring_constructor import PolynomialRing
    from sage.rings.finite_rings.finite_field_constructor import GF

    base_ring = GF(2**n, name='x')
    R = PolynomialRing(base_ring, name='X')
    X = R.gen()
    return SBox(X**e)


AES = SBox([
    0x63,0x7c,0x77,0x7b,0xf2,0x6b,0x6f,0xc5,0x30,0x01,0x67,0x2b,0xfe,0xd7,0xab,0x76,
    0xca,0x82,0xc9,0x7d,0xfa,0x59,0x47,0xf0,0xad,0xd4,0xa2,0xaf,0x9c,0xa4,0x72,0xc0,
    0xb7,0xfd,0x93,0x26,0x36,0x3f,0xf7,0xcc,0x34,0xa5,0xe5,0xf1,0x71,0xd8,0x31,0x15,
    0x04,0xc7,0x23,0xc3,0x18,0x96,0x05,0x9a,0x07,0x12,0x80,0xe2,0xeb,0x27,0xb2,0x75,
    0x09,0x83,0x2c,0x1a,0x1b,0x6e,0x5a,0xa0,0x52,0x3b,0xd6,0xb3,0x29,0xe3,0x2f,0x84,
    0x53,0xd1,0x00,0xed,0x20,0xfc,0xb1,0x5b,0x6a,0xcb,0xbe,0x39,0x4a,0x4c,0x58,0xcf,
    0xd0,0xef,0xaa,0xfb,0x43,0x4d,0x33,0x85,0x45,0xf9,0x02,0x7f,0x50,0x3c,0x9f,0xa8,
    0x51,0xa3,0x40,0x8f,0x92,0x9d,0x38,0xf5,0xbc,0xb6,0xda,0x21,0x10,0xff,0xf3,0xd2,
    0xcd,0x0c,0x13,0xec,0x5f,0x97,0x44,0x17,0xc4,0xa7,0x7e,0x3d,0x64,0x5d,0x19,0x73,
    0x60,0x81,0x4f,0xdc,0x22,0x2a,0x90,0x88,0x46,0xee,0xb8,0x14,0xde,0x5e,0x0b,0xdb,
    0xe0,0x32,0x3a,0x0a,0x49,0x06,0x24,0x5c,0xc2,0xd3,0xac,0x62,0x91,0x95,0xe4,0x79,
    0xe7,0xc8,0x37,0x6d,0x8d,0xd5,0x4e,0xa9,0x6c,0x56,0xf4,0xea,0x65,0x7a,0xae,0x08,
    0xba,0x78,0x25,0x2e,0x1c,0xa6,0xb4,0xc6,0xe8,0xdd,0x74,0x1f,0x4b,0xbd,0x8b,0x8a,
    0x70,0x3e,0xb5,0x66,0x48,0x03,0xf6,0x0e,0x61,0x35,0x57,0xb9,0x86,0xc1,0x1d,0x9e,
    0xe1,0xf8,0x98,0x11,0x69,0xd9,0x8e,0x94,0x9b,0x1e,0x87,0xe9,0xce,0x55,0x28,0xdf,
    0x8c,0xa1,0x89,0x0d,0xbf,0xe6,0x42,0x68,0x41,0x99,0x2d,0x0f,0xb0,0x54,0xbb,0x16])

Anubis = SBox([
    0xa7,0xd3,0xe6,0x71,0xd0,0xac,0x4d,0x79,0x3a,0xc9,0x91,0xfc,0x1e,0x47,0x54,0xbd,
    0x8c,0xa5,0x7a,0xfb,0x63,0xb8,0xdd,0xd4,0xe5,0xb3,0xc5,0xbe,0xa9,0x88,0x0c,0xa2,
    0x39,0xdf,0x29,0xda,0x2b,0xa8,0xcb,0x4c,0x4b,0x22,0xaa,0x24,0x41,0x70,0xa6,0xf9,
    0x5a,0xe2,0xb0,0x36,0x7d,0xe4,0x33,0xff,0x60,0x20,0x08,0x8b,0x5e,0xab,0x7f,0x78,
    0x7c,0x2c,0x57,0xd2,0xdc,0x6d,0x7e,0x0d,0x53,0x94,0xc3,0x28,0x27,0x06,0x5f,0xad,
    0x67,0x5c,0x55,0x48,0x0e,0x52,0xea,0x42,0x5b,0x5d,0x30,0x58,0x51,0x59,0x3c,0x4e,
    0x38,0x8a,0x72,0x14,0xe7,0xc6,0xde,0x50,0x8e,0x92,0xd1,0x77,0x93,0x45,0x9a,0xce,
    0x2d,0x03,0x62,0xb6,0xb9,0xbf,0x96,0x6b,0x3f,0x07,0x12,0xae,0x40,0x34,0x46,0x3e,
    0xdb,0xcf,0xec,0xcc,0xc1,0xa1,0xc0,0xd6,0x1d,0xf4,0x61,0x3b,0x10,0xd8,0x68,0xa0,
    0xb1,0x0a,0x69,0x6c,0x49,0xfa,0x76,0xc4,0x9e,0x9b,0x6e,0x99,0xc2,0xb7,0x98,0xbc,
    0x8f,0x85,0x1f,0xb4,0xf8,0x11,0x2e,0x00,0x25,0x1c,0x2a,0x3d,0x05,0x4f,0x7b,0xb2,
    0x32,0x90,0xaf,0x19,0xa3,0xf7,0x73,0x9d,0x15,0x74,0xee,0xca,0x9f,0x0f,0x1b,0x75,
    0x86,0x84,0x9c,0x4a,0x97,0x1a,0x65,0xf6,0xed,0x09,0xbb,0x26,0x83,0xeb,0x6f,0x81,
    0x04,0x6a,0x43,0x01,0x17,0xe1,0x87,0xf5,0x8d,0xe3,0x23,0x80,0x44,0x16,0x66,0x21,
    0xfe,0xd5,0x31,0xd9,0x35,0x18,0x02,0x64,0xf2,0xf1,0x56,0xcd,0x82,0xc8,0xba,0xf0,
    0xef,0xe9,0xe8,0xfd,0x89,0xd7,0xc7,0xb5,0xa4,0x2f,0x95,0x13,0x0b,0xf3,0xe0,0x37])

ARIA_s2 = SBox([
    0xe2,0x4e,0x54,0xfc,0x94,0xc2,0x4a,0xcc,0x62,0x0d,0x6a,0x46,0x3c,0x4d,0x8b,0xd1,
    0x5e,0xfa,0x64,0xcb,0xb4,0x97,0xbe,0x2b,0xbc,0x77,0x2e,0x03,0xd3,0x19,0x59,0xc1,
    0x1d,0x06,0x41,0x6b,0x55,0xf0,0x99,0x69,0xea,0x9c,0x18,0xae,0x63,0xdf,0xe7,0xbb,
    0x00,0x73,0x66,0xfb,0x96,0x4c,0x85,0xe4,0x3a,0x09,0x45,0xaa,0x0f,0xee,0x10,0xeb,
    0x2d,0x7f,0xf4,0x29,0xac,0xcf,0xad,0x91,0x8d,0x78,0xc8,0x95,0xf9,0x2f,0xce,0xcd,
    0x08,0x7a,0x88,0x38,0x5c,0x83,0x2a,0x28,0x47,0xdb,0xb8,0xc7,0x93,0xa4,0x12,0x53,
    0xff,0x87,0x0e,0x31,0x36,0x21,0x58,0x48,0x01,0x8e,0x37,0x74,0x32,0xca,0xe9,0xb1,
    0xb7,0xab,0x0c,0xd7,0xc4,0x56,0x42,0x26,0x07,0x98,0x60,0xd9,0xb6,0xb9,0x11,0x40,
    0xec,0x20,0x8c,0xbd,0xa0,0xc9,0x84,0x04,0x49,0x23,0xf1,0x4f,0x50,0x1f,0x13,0xdc,
    0xd8,0xc0,0x9e,0x57,0xe3,0xc3,0x7b,0x65,0x3b,0x02,0x8f,0x3e,0xe8,0x25,0x92,0xe5,
    0x15,0xdd,0xfd,0x17,0xa9,0xbf,0xd4,0x9a,0x7e,0xc5,0x39,0x67,0xfe,0x76,0x9d,0x43,
    0xa7,0xe1,0xd0,0xf5,0x68,0xf2,0x1b,0x34,0x70,0x05,0xa3,0x8a,0xd5,0x79,0x86,0xa8,
    0x30,0xc6,0x51,0x4b,0x1e,0xa6,0x27,0xf6,0x35,0xd2,0x6e,0x24,0x16,0x82,0x5f,0xda,
    0xe6,0x75,0xa2,0xef,0x2c,0xb2,0x1c,0x9f,0x5d,0x6f,0x80,0x0a,0x72,0x44,0x9b,0x6c,
    0x90,0x0b,0x5b,0x33,0x7d,0x5a,0x52,0xf3,0x61,0xa1,0xf7,0xb0,0xd6,0x3f,0x7c,0x6d,
    0xed,0x14,0xe0,0xa5,0x3d,0x22,0xb3,0xf8,0x89,0xde,0x71,0x1a,0xaf,0xba,0xb5,0x81])

BelT = SBox([
    0xb1,0x94,0xba,0xc8,0x0a,0x08,0xf5,0x3b,0x36,0x6d,0x00,0x8e,0x58,0x4a,0x5d,0xe4,
    0x85,0x04,0xfa,0x9d,0x1b,0xb6,0xc7,0xac,0x25,0x2e,0x72,0xc2,0x02,0xfd,0xce,0x0d,
    0x5b,0xe3,0xd6,0x12,0x17,0xb9,0x61,0x81,0xfe,0x67,0x86,0xad,0x71,0x6b,0x89,0x0b,
    0x5c,0xb0,0xc0,0xff,0x33,0xc3,0x56,0xb8,0x35,0xc4,0x05,0xae,0xd8,0xe0,0x7f,0x99,
    0xe1,0x2b,0xdc,0x1a,0xe2,0x82,0x57,0xec,0x70,0x3f,0xcc,0xf0,0x95,0xee,0x8d,0xf1,
    0xc1,0xab,0x76,0x38,0x9f,0xe6,0x78,0xca,0xf7,0xc6,0xf8,0x60,0xd5,0xbb,0x9c,0x4f,
    0xf3,0x3c,0x65,0x7b,0x63,0x7c,0x30,0x6a,0xdd,0x4e,0xa7,0x79,0x9e,0xb2,0x3d,0x31,
    0x3e,0x98,0xb5,0x6e,0x27,0xd3,0xbc,0xcf,0x59,0x1e,0x18,0x1f,0x4c,0x5a,0xb7,0x93,
    0xe9,0xde,0xe7,0x2c,0x8f,0x0c,0x0f,0xa6,0x2d,0xdb,0x49,0xf4,0x6f,0x73,0x96,0x47,
    0x06,0x07,0x53,0x16,0xed,0x24,0x7a,0x37,0x39,0xcb,0xa3,0x83,0x03,0xa9,0x8b,0xf6,
    0x92,0xbd,0x9b,0x1c,0xe5,0xd1,0x41,0x01,0x54,0x45,0xfb,0xc9,0x5e,0x4d,0x0e,0xf2,
    0x68,0x20,0x80,0xaa,0x22,0x7d,0x64,0x2f,0x26,0x87,0xf9,0x34,0x90,0x40,0x55,0x11,
    0xbe,0x32,0x97,0x13,0x43,0xfc,0x9a,0x48,0xa0,0x2a,0x88,0x5f,0x19,0x4b,0x09,0xa1,
    0x7e,0xcd,0xa4,0xd0,0x15,0x44,0xaf,0x8c,0xa5,0x84,0x50,0xbf,0x66,0xd2,0xe8,0x8a,
    0xa2,0xd7,0x46,0x52,0x42,0xa8,0xdf,0xb3,0x69,0x74,0xc5,0x51,0xeb,0x23,0x29,0x21,
    0xd4,0xef,0xd9,0xb4,0x3a,0x62,0x28,0x75,0x91,0x14,0x10,0xea,0x77,0x6c,0xda,0x1d])

Camellia = SBox([
    112,130, 44,236,179, 39,192,229,228,133, 87, 53,234, 12,174, 65,
    35,239,107,147, 69, 25,165, 33,237, 14, 79, 78, 29,101,146,189,
    134,184,175,143,124,235, 31,206, 62, 48,220, 95, 94,197, 11, 26,
    166,225, 57,202,213, 71, 93, 61,217,  1, 90,214, 81, 86,108, 77,
    139, 13,154,102,251,204,176, 45,116, 18, 43, 32,240,177,132,153,
    223, 76,203,194, 52,126,118,  5,109,183,169, 49,209, 23,  4,215,
    20, 88, 58, 97,222, 27, 17, 28, 50, 15,156, 22, 83, 24,242, 34,
    254, 68,207,178,195,181,122,145, 36,  8,232,168, 96,252,105, 80,
    170,208,160,125,161,137, 98,151, 84, 91, 30,149,224,255,100,210,
    16,196,  0, 72,163,247,117,219,138,  3,230,218,  9, 63,221,148,
    135, 92,131,  2,205, 74,144, 51,115,103,246,243,157,127,191,226,
    82,155,216, 38,200, 55,198, 59,129,150,111, 75, 19,190, 99, 46,
    233,121,167,140,159,110,188,142, 41,245,249,182, 47,253,180, 89,
    120,152,  6,106,231, 70,113,186,212, 37,171, 66,136,162,141,250,
    114,  7,185, 85,248,238,172, 10, 54, 73, 42,104, 60, 56,241,164,
    64, 40,211,123,187,201, 67,193, 21,227,173,244,119,199,128,158])

# source: https://www.schneier.com/academic/paperfiles/paper-cmea.pdf
CMEA = SBox([
    0xd9,0x23,0x5f,0xe6,0xca,0x68,0x97,0xb0,0x7b,0xf2,0x0c,0x34,0x11,0xa5,0x8d,0x4e,
    0x0a,0x46,0x77,0x8d,0x10,0x9f,0x5e,0x62,0xf1,0x34,0xec,0xa5,0xc9,0xb3,0xd8,0x2b,
    0x59,0x47,0xe3,0xd2,0xff,0xae,0x64,0xca,0x15,0x8b,0x7d,0x38,0x21,0xbc,0x96,0x00,
    0x49,0x56,0x23,0x15,0x97,0xe4,0xcb,0x6f,0xf2,0x70,0x3c,0x88,0xba,0xd1,0x0d,0xae,
    0xe2,0x38,0xba,0x44,0x9f,0x83,0x5d,0x1c,0xde,0xab,0xc7,0x65,0xf1,0x76,0x09,0x20,
    0x86,0xbd,0x0a,0xf1,0x3c,0xa7,0x29,0x93,0xcb,0x45,0x5f,0xe8,0x10,0x74,0x62,0xde,
    0xb8,0x77,0x80,0xd1,0x12,0x26,0xac,0x6d,0xe9,0xcf,0xf3,0x54,0x3a,0x0b,0x95,0x4e,
    0xb1,0x30,0xa4,0x96,0xf8,0x57,0x49,0x8e,0x05,0x1f,0x62,0x7c,0xc3,0x2b,0xda,0xed,
    0xbb,0x86,0x0d,0x7a,0x97,0x13,0x6c,0x4e,0x51,0x30,0xe5,0xf2,0x2f,0xd8,0xc4,0xa9,
    0x91,0x76,0xf0,0x17,0x43,0x38,0x29,0x84,0xa2,0xdb,0xef,0x65,0x5e,0xca,0x0d,0xbc,
    0xe7,0xfa,0xd8,0x81,0x6f,0x00,0x14,0x42,0x25,0x7c,0x5d,0xc9,0x9e,0xb6,0x33,0xab,
    0x5a,0x6f,0x9b,0xd9,0xfe,0x71,0x44,0xc5,0x37,0xa2,0x88,0x2d,0x00,0xb6,0x13,0xec,
    0x4e,0x96,0xa8,0x5a,0xb5,0xd7,0xc3,0x8d,0x3f,0xf2,0xec,0x04,0x60,0x71,0x1b,0x29,
    0x04,0x79,0xe3,0xc7,0x1b,0x66,0x81,0x4a,0x25,0x9d,0xdc,0x5f,0x3e,0xb0,0xf8,0xa2,
    0x91,0x34,0xf6,0x5c,0x67,0x89,0x73,0x05,0x22,0xaa,0xcb,0xee,0xbf,0x18,0xd0,0x4d,
    0xf5,0x36,0xae,0x01,0x2f,0x94,0xc3,0x49,0x8b,0xbd,0x58,0x12,0xe0,0x77,0x6c,0xda])

Chiasmus = SBox([
    0x65,0x33,0xcf,0xb9,0x37,0x64,0xcd,0xf3,0x26,0x3a,0xc1,0xa2,0x72,0x8a,0x8f,0xe3,
    0xfd,0x56,0xb3,0x0f,0x10,0x2b,0x3e,0xa0,0xbd,0x1e,0xab,0x1d,0x9c,0xe2,0x87,0x98,
    0xa8,0xd3,0xb4,0xdf,0x92,0x75,0x3b,0x39,0x20,0xa5,0xfa,0x1b,0xbe,0x90,0xf6,0x09,
    0xe5,0x61,0xc4,0xc9,0x06,0xc2,0xa6,0x1c,0xf9,0x94,0x7b,0x53,0x73,0x01,0x25,0x9a,
    0x1a,0xff,0xe9,0x5a,0x76,0x13,0x4b,0x95,0xac,0x0b,0xc7,0xb2,0xb8,0xd6,0x17,0xa9,
    0x27,0xeb,0xd1,0x5C,0xc3,0x9b,0x22,0x15,0x8e,0x40,0x11,0x5e,0x57,0x16,0xd0,0xb0,
    0x5d,0x79,0x31,0xbb,0xea,0x4f,0xd9,0xde,0x00,0x0a,0xd7,0xad,0x3f,0x99,0x68,0x34,
    0x66,0xf0,0x44,0x35,0x89,0x54,0x81,0xb1,0x84,0x2a,0x8b,0x6f,0xc0,0x43,0xfe,0x96,
    0x48,0x82,0x0c,0xda,0x74,0xbc,0x21,0xf1,0x67,0x2e,0xdb,0x49,0xe4,0xd5,0x71,0x59,
    0x29,0xe0,0xa1,0x30,0xdd,0x91,0x6b,0xb7,0xb6,0x69,0xc5,0x80,0xaa,0x6d,0xa3,0x2c,
    0x05,0x78,0xba,0x51,0x14,0x07,0xd4,0xec,0x7e,0xcc,0x24,0x62,0x9e,0xdc,0x8c,0xd8,
    0x1f,0x46,0xe8,0x9f,0x4e,0xa4,0x85,0x32,0xce,0xa7,0xfc,0xe1,0x97,0xae,0x2d,0x52,
    0x7d,0x0e,0x6c,0x83,0x5f,0xbf,0x18,0x7c,0x36,0x63,0x0d,0xef,0xc8,0x5b,0x55,0x12,
    0x4a,0xf2,0x70,0x38,0xf8,0xaf,0x86,0x77,0x47,0x04,0x23,0x02,0x6e,0x4c,0x58,0x03,
    0x50,0x7a,0x3d,0x28,0xf5,0xe7,0x41,0xf4,0x45,0x60,0x6a,0x08,0x88,0x7f,0x9d,0x93,
    0x4d,0xd2,0x2f,0xee,0xe6,0xcb,0xed,0xfb,0xca,0xf7,0x19,0xb5,0x42,0x8d,0xc6,0x3c])

CLEFIA_S0 = SBox([
    0x57,0x49,0xd1,0xc6,0x2f,0x33,0x74,0xfb,0x95,0x6d,0x82,0xea,0x0e,0xb0,0xa8,0x1c,
    0x28,0xd0,0x4b,0x92,0x5c,0xee,0x85,0xb1,0xc4,0x0a,0x76,0x3d,0x63,0xf9,0x17,0xaf,
    0xbf,0xa1,0x19,0x65,0xf7,0x7a,0x32,0x20,0x06,0xce,0xe4,0x83,0x9d,0x5b,0x4c,0xd8,
    0x42,0x5d,0x2e,0xe8,0xd4,0x9b,0x0f,0x13,0x3c,0x89,0x67,0xc0,0x71,0xaa,0xb6,0xf5,
    0xa4,0xbe,0xfd,0x8c,0x12,0x00,0x97,0xda,0x78,0xe1,0xcf,0x6b,0x39,0x43,0x55,0x26,
    0x30,0x98,0xcc,0xdd,0xeb,0x54,0xb3,0x8f,0x4e,0x16,0xfa,0x22,0xa5,0x77,0x09,0x61,
    0xd6,0x2a,0x53,0x37,0x45,0xc1,0x6c,0xae,0xef,0x70,0x08,0x99,0x8b,0x1d,0xf2,0xb4,
    0xe9,0xc7,0x9f,0x4a,0x31,0x25,0xfe,0x7c,0xd3,0xa2,0xbd,0x56,0x14,0x88,0x60,0x0b,
    0xcd,0xe2,0x34,0x50,0x9e,0xdc,0x11,0x05,0x2b,0xb7,0xa9,0x48,0xff,0x66,0x8a,0x73,
    0x03,0x75,0x86,0xf1,0x6a,0xa7,0x40,0xc2,0xb9,0x2c,0xdb,0x1f,0x58,0x94,0x3e,0xed,
    0xfc,0x1b,0xa0,0x04,0xb8,0x8d,0xe6,0x59,0x62,0x93,0x35,0x7e,0xca,0x21,0xdf,0x47,
    0x15,0xf3,0xba,0x7f,0xa6,0x69,0xc8,0x4d,0x87,0x3b,0x9c,0x01,0xe0,0xde,0x24,0x52,
    0x7b,0x0c,0x68,0x1e,0x80,0xb2,0x5a,0xe7,0xad,0xd5,0x23,0xf4,0x46,0x3f,0x91,0xc9,
    0x6e,0x84,0x72,0xbb,0x0d,0x18,0xd9,0x96,0xf0,0x5f,0x41,0xac,0x27,0xc5,0xe3,0x3a,
    0x81,0x6f,0x07,0xa3,0x79,0xf6,0x2d,0x38,0x1a,0x44,0x5e,0xb5,0xd2,0xec,0xcb,0x90,
    0x9a,0x36,0xe5,0x29,0xc3,0x4f,0xab,0x64,0x51,0xf8,0x10,0xd7,0xbc,0x02,0x7d,0x8e])

CLEFIA_S1 = SBox([
    0x6c,0xda,0xc3,0xe9,0x4e,0x9d,0x0a,0x3d,0xb8,0x36,0xb4,0x38,0x13,0x34,0x0c,0xd9,
    0xbf,0x74,0x94,0x8f,0xb7,0x9c,0xe5,0xdc,0x9e,0x07,0x49,0x4f,0x98,0x2c,0xb0,0x93,
    0x12,0xeb,0xcd,0xb3,0x92,0xe7,0x41,0x60,0xe3,0x21,0x27,0x3b,0xe6,0x19,0xd2,0x0e,
    0x91,0x11,0xc7,0x3f,0x2a,0x8e,0xa1,0xbc,0x2b,0xc8,0xc5,0x0f,0x5b,0xf3,0x87,0x8b,
    0xfb,0xf5,0xde,0x20,0xc6,0xa7,0x84,0xce,0xd8,0x65,0x51,0xc9,0xa4,0xef,0x43,0x53,
    0x25,0x5d,0x9b,0x31,0xe8,0x3e,0x0d,0xd7,0x80,0xff,0x69,0x8a,0xba,0x0b,0x73,0x5c,
    0x6e,0x54,0x15,0x62,0xf6,0x35,0x30,0x52,0xa3,0x16,0xd3,0x28,0x32,0xfa,0xaa,0x5e,
    0xcf,0xea,0xed,0x78,0x33,0x58,0x09,0x7b,0x63,0xc0,0xc1,0x46,0x1e,0xdf,0xa9,0x99,
    0x55,0x04,0xc4,0x86,0x39,0x77,0x82,0xec,0x40,0x18,0x90,0x97,0x59,0xdd,0x83,0x1f,
    0x9a,0x37,0x06,0x24,0x64,0x7c,0xa5,0x56,0x48,0x08,0x85,0xd0,0x61,0x26,0xca,0x6f,
    0x7e,0x6a,0xb6,0x71,0xa0,0x70,0x05,0xd1,0x45,0x8c,0x23,0x1c,0xf0,0xee,0x89,0xad,
    0x7a,0x4b,0xc2,0x2f,0xdb,0x5a,0x4d,0x76,0x67,0x17,0x2d,0xf4,0xcb,0xb1,0x4a,0xa8,
    0xb5,0x22,0x47,0x3a,0xd5,0x10,0x4c,0x72,0xcc,0x00,0xf9,0xe0,0xfd,0xe2,0xfe,0xae,
    0xf8,0x5f,0xab,0xf1,0x1b,0x42,0x81,0xd6,0xbe,0x44,0x29,0xa6,0x57,0xb9,0xaf,0xf2,
    0xd4,0x75,0x66,0xbb,0x68,0x9f,0x50,0x02,0x01,0x3c,0x7f,0x8d,0x1a,0x88,0xbd,0xac,
    0xf7,0xe4,0x79,0x96,0xa2,0xfc,0x6d,0xb2,0x6b,0x03,0xe1,0x2e,0x7d,0x14,0x95,0x1d])

Crypton_0_5 = SBox([
    0xf0,0x12,0x4c,0x7a,0x47,0x16,0x03,0x3a,0xe6,0x9d,0x44,0x77,0x53,0xca,0x3b,0x0f,
    0x9b,0x98,0x54,0x90,0x3d,0xac,0x74,0x56,0x9e,0xde,0x5c,0xf3,0x86,0x39,0x7c,0xc4,
    0x91,0xa9,0x97,0x5f,0x9c,0x0d,0x78,0xcc,0xfd,0x43,0xbf,0x02,0x4b,0x92,0x60,0x3e,
    0x7d,0x1d,0x50,0xcb,0xb8,0xb9,0x70,0x27,0xaa,0x96,0x48,0x88,0x38,0xd7,0x68,0x42,
    0xa8,0xd0,0xa6,0x2e,0x25,0xf4,0x2c,0x6e,0x0c,0xb7,0xce,0xe0,0xbe,0x0b,0x24,0x67,
    0x8c,0xec,0xc5,0x52,0xd9,0xd8,0x09,0xb4,0xcf,0x8f,0x8d,0x8b,0x59,0x23,0x51,0xe3,
    0xd3,0xb1,0x18,0xf8,0xd4,0x05,0xa2,0xdb,0x82,0x6c,0x00,0x46,0x8a,0xaf,0xda,0xbc,
    0x99,0x1a,0xad,0xb3,0x1f,0x0e,0x71,0x4f,0xc7,0x2b,0xe5,0x2a,0xe2,0x58,0x29,0x06,
    0xf6,0xfe,0xf9,0x19,0x6b,0xea,0xbb,0xc2,0xa3,0x55,0xa1,0xdf,0x6f,0x45,0x83,0x69,
    0x8e,0x7b,0x72,0x3c,0xee,0xff,0x07,0xa5,0xe8,0xf1,0x0a,0x1c,0x75,0xe1,0x2f,0x21,
    0xd2,0xb6,0x3f,0xf7,0x73,0xb2,0x5d,0x79,0x35,0x80,0x17,0x41,0x94,0x7e,0x15,0xed,
    0xb5,0xd5,0x93,0x14,0x20,0x61,0x76,0x31,0xc9,0x6a,0xab,0x34,0xa0,0xa4,0x1e,0xba,
    0xe7,0x13,0x4e,0xc6,0xd6,0x87,0x7f,0xbd,0x84,0x62,0x26,0x95,0x6d,0x4d,0x57,0x28,
    0x04,0x64,0x4a,0x11,0x01,0x40,0x65,0x08,0xb0,0xe9,0x32,0xcd,0x81,0x66,0x2d,0x5b,
    0xef,0xa7,0xfb,0xdd,0xf2,0x33,0x5a,0x63,0xc1,0xe4,0xc3,0xae,0xdc,0xfc,0x22,0x10,
    0xfa,0x9f,0xd1,0x85,0x9a,0x1b,0x5e,0x30,0xeb,0xc8,0x89,0x49,0x37,0xc0,0x36,0xf5])

Crypton_1_0_S0 = SBox([
    0x63,0xec,0x59,0xaa,0xdb,0x8e,0x66,0xc0,0x37,0x3c,0x14,0xff,0x13,0x44,0xa9,0x91,
    0x3b,0x78,0x8d,0xef,0xc2,0x2a,0xf0,0xd7,0x61,0x9e,0xa5,0xbc,0x48,0x15,0x12,0x47,
    0xed,0x42,0x1a,0x33,0x38,0xc8,0x17,0x90,0xa6,0xd5,0x5d,0x65,0x6a,0xfe,0x8f,0xa1,
    0x93,0xca,0x2f,0x0c,0x68,0x58,0xdf,0xf4,0x45,0x11,0xa0,0xa7,0x22,0x96,0xfb,0x7d,
    0x1d,0xb4,0x84,0xe0,0xbf,0x57,0xe9,0x0a,0x4e,0x83,0xcc,0x7a,0x71,0x39,0xc7,0x32,
    0x74,0x3d,0xde,0x50,0x85,0x06,0x6f,0x53,0xe8,0xad,0x82,0x19,0xe1,0xba,0x36,0xcb,
    0x0e,0x28,0xf3,0x9b,0x4a,0x62,0x94,0x1f,0xbd,0xf6,0x67,0x41,0xd8,0xd1,0x2d,0xa4,
    0x86,0xb7,0x01,0xc5,0xb0,0x75,0x02,0xf9,0x2c,0x29,0x6e,0xd2,0x5f,0x8b,0xfc,0x5a,
    0xe4,0x7f,0xdd,0x07,0x55,0xb1,0x2b,0x89,0x72,0x18,0x3a,0x4c,0xb6,0xe3,0x80,0xce,
    0x49,0xcf,0x6b,0xb9,0xf2,0x0d,0xdc,0x64,0x95,0x46,0xf7,0x10,0x9a,0x20,0xa2,0x3f,
    0xd6,0x87,0x70,0x3e,0x21,0xfd,0x4d,0x7b,0xc3,0xae,0x09,0x8a,0x04,0xb3,0x54,0xf8,
    0x30,0x00,0x56,0xd4,0xe7,0x25,0xbb,0xac,0x98,0x73,0xea,0xc9,0x9d,0x4f,0x7e,0x03,
    0xab,0x92,0xa8,0x43,0x0f,0xfa,0x24,0x5c,0x1e,0x60,0x31,0x97,0xcd,0xc6,0x79,0xf5,
    0x5e,0xe5,0x34,0x76,0x1c,0x81,0xb2,0xaf,0x0b,0x5b,0xd9,0xe2,0x27,0x6d,0xd0,0x88,
    0xc1,0x51,0xe6,0x9c,0x77,0xbe,0x99,0x23,0xda,0xeb,0x52,0x2e,0xb5,0x08,0x05,0x6c,
    0xb8,0x1b,0xa3,0x69,0x8c,0xd3,0x40,0x26,0xf1,0xc4,0x9f,0x35,0xee,0x7c,0x4b,0x16])

Crypton_1_0_S1 = SBox([
    0x8d,0xb3,0x65,0xaa,0x6f,0x3a,0x99,0x03,0xdc,0xf0,0x50,0xff,0x4c,0x11,0xa6,0x46,
    0xec,0xe1,0x36,0xbf,0x0b,0xa8,0xc3,0x5f,0x85,0x7a,0x96,0xf2,0x21,0x54,0x48,0x1d,
    0xb7,0x09,0x68,0xcc,0xe0,0x23,0x5c,0x42,0x9a,0x57,0x75,0x95,0xa9,0xfb,0x3e,0x86,
    0x4e,0x2b,0xbc,0x30,0xa1,0x61,0x7f,0xd3,0x15,0x44,0x82,0x9e,0x88,0x5a,0xef,0xf5,
    0x74,0xd2,0x12,0x83,0xfe,0x5d,0xa7,0x28,0x39,0x0e,0x33,0xe9,0xc5,0xe4,0x1f,0xc8,
    0xd1,0xf4,0x7b,0x41,0x16,0x18,0xbd,0x4d,0xa3,0xb6,0x0a,0x64,0x87,0xea,0xd8,0x2f,
    0x38,0xa0,0xcf,0x6e,0x29,0x89,0x52,0x7c,0xf6,0xdb,0x9d,0x05,0x63,0x47,0xb4,0x92,
    0x1a,0xde,0x04,0x17,0xc2,0xd5,0x08,0xe7,0xb0,0xa4,0xb9,0x4b,0x7d,0x2e,0xf3,0x69,
    0x93,0xfd,0x77,0x1c,0x55,0xc6,0xac,0x26,0xc9,0x60,0xe8,0x31,0xda,0x8f,0x02,0x3b,
    0x25,0x3f,0xad,0xe6,0xcb,0x34,0x73,0x91,0x56,0x19,0xdf,0x40,0x6a,0x80,0x8a,0xfc,
    0x5b,0x1e,0xc1,0xf8,0x84,0xf7,0x35,0xed,0x0f,0xba,0x24,0x2a,0x10,0xce,0x51,0xe3,
    0xc0,0x00,0x59,0x53,0x9f,0x94,0xee,0xb2,0x62,0xcd,0xab,0x27,0x76,0x3d,0xf9,0x0c,
    0xae,0x4a,0xa2,0x0d,0x3c,0xeb,0x90,0x71,0x78,0x81,0xc4,0x5e,0x37,0x1b,0xe5,0xd7,
    0x79,0x97,0xd0,0xd9,0x70,0x06,0xca,0xbe,0x2c,0x6d,0x67,0x8b,0x9c,0xb5,0x43,0x22,
    0x07,0x45,0x9b,0x72,0xdd,0xfa,0x66,0x8c,0x6b,0xaf,0x49,0xb8,0xd6,0x20,0x14,0xb1,
    0xe2,0x6c,0x8e,0xa5,0x32,0x4f,0x01,0x98,0xc7,0x13,0x7e,0xd4,0xbb,0xf1,0x2d,0x58])

Crypton_1_0_S2 = SBox([
    0xb1,0x72,0x76,0xbf,0xac,0xee,0x55,0x83,0xed,0xaa,0x47,0xd8,0x33,0x95,0x60,0xc4,
    0x9b,0x39,0x1e,0x0c,0x0a,0x1d,0xff,0x26,0x89,0x5b,0x22,0xf1,0xd4,0x40,0xc8,0x67,
    0x9d,0xa4,0x3c,0xe7,0xc6,0xb5,0xf7,0xdc,0x61,0x79,0x15,0x86,0x78,0x6e,0xeb,0x32,
    0xb0,0xca,0x4f,0x23,0xd2,0xfb,0x5e,0x08,0x24,0x4d,0x8a,0x10,0x09,0x51,0xa3,0x9f,
    0xf6,0x6b,0x21,0xc3,0x0d,0x38,0x99,0x1f,0x1c,0x90,0x64,0xfe,0x8b,0xa6,0x48,0xbd,
    0x53,0xe1,0xea,0x57,0xae,0x84,0xb2,0x45,0x35,0x02,0x7f,0xd9,0xc7,0x2a,0xd0,0x7c,
    0xc9,0x18,0x65,0x00,0x97,0x2b,0x06,0x6a,0x34,0xf3,0x2c,0x92,0xef,0xdd,0x7a,0x56,
    0xa2,0x4c,0x88,0xb9,0x50,0x75,0xd3,0xe4,0x11,0xce,0x4b,0xa7,0xfd,0x3f,0xbe,0x81,
    0x8e,0xd5,0x5a,0x49,0x42,0x54,0x70,0xa1,0xdf,0x87,0xab,0x7d,0xf4,0x12,0x05,0x2e,
    0x27,0x0f,0xc1,0x30,0x66,0x98,0x3d,0xcb,0xb8,0xe6,0x9c,0x63,0xe3,0xbc,0x19,0xfa,
    0x3a,0x2f,0x9e,0xf2,0x6f,0x1a,0x28,0x3b,0xc2,0x0e,0x03,0xc0,0xb7,0x59,0xa9,0xd7,
    0x74,0x85,0xd6,0xad,0x41,0xec,0x8c,0x71,0xf0,0x93,0x5d,0xb6,0x1b,0x68,0xe5,0x44,
    0x07,0xe0,0x14,0xa8,0xf9,0x73,0xcd,0x4e,0x25,0xbb,0x31,0x5f,0x4a,0xcc,0x8f,0x91,
    0xde,0x6d,0x7b,0xf5,0xb3,0x29,0xa0,0x17,0x6c,0xda,0xe8,0x04,0x96,0x82,0x52,0x36,
    0x43,0x5c,0xdb,0x8d,0x80,0xd1,0xe2,0xb4,0x58,0x46,0xba,0xe9,0x01,0x20,0xfc,0x13,
    0x16,0xf8,0x94,0x62,0x37,0xcf,0x69,0x9a,0xaf,0x77,0xc5,0x3e,0x7e,0xa5,0x2d,0x0b])

Crypton_1_0_S3 = SBox([
    0xb1,0xf6,0x8e,0x07,0x72,0x6b,0xd5,0xe0,0x76,0x21,0x5a,0x14,0xbf,0xc3,0x49,0xa8,
    0xac,0x0d,0x42,0xf9,0xee,0x38,0x54,0x73,0x55,0x99,0x70,0xcd,0x83,0x1f,0xa1,0x4e,
    0xed,0x1c,0xdf,0x25,0xaa,0x90,0x87,0xbb,0x47,0x64,0xab,0x31,0xd8,0xfe,0x7d,0x5f,
    0x33,0x8b,0xf4,0x4a,0x95,0xa6,0x12,0xcc,0x60,0x48,0x05,0x8f,0xc4,0xbd,0x2e,0x91,
    0x9b,0x53,0x27,0xde,0x39,0xe1,0x0f,0x6d,0x1e,0xea,0xc1,0x7b,0x0c,0x57,0x30,0xf5,
    0x0a,0xae,0x66,0xb3,0x1d,0x84,0x98,0x29,0xff,0xb2,0x3d,0xa0,0x26,0x45,0xcb,0x17,
    0x89,0x35,0xb8,0x6c,0x5b,0x02,0xe6,0xda,0x22,0x7f,0x9c,0xe8,0xf1,0xd9,0x63,0x04,
    0xd4,0xc7,0xe3,0x96,0x40,0x2a,0xbc,0x82,0xc8,0xd0,0x19,0x52,0x67,0x7c,0xfa,0x36,
    0x9d,0xc9,0x3a,0x43,0xa4,0x18,0x2f,0x5c,0x3c,0x65,0x9e,0xdb,0xe7,0x00,0xf2,0x8d,
    0xc6,0x97,0x6f,0x80,0xb5,0x2b,0x1a,0xd1,0xf7,0x06,0x28,0xe2,0xdc,0x6a,0x3b,0xb4,
    0x61,0x34,0xc2,0x58,0x79,0xf3,0x0e,0x46,0x15,0x2c,0x03,0xba,0x86,0x92,0xc0,0xe9,
    0x78,0xef,0xb7,0x01,0x6e,0xdd,0x59,0x20,0xeb,0x7a,0xa9,0xfc,0x32,0x56,0xd7,0x13,
    0xb0,0xa2,0x74,0x16,0xca,0x4c,0x85,0xf8,0x4f,0x88,0xd6,0x94,0x23,0xb9,0xad,0x62,
    0xd2,0x50,0x41,0x37,0xfb,0x75,0xec,0xcf,0x5e,0xd3,0x8c,0x69,0x08,0xe4,0x71,0x9a,
    0x24,0x11,0xf0,0xaf,0x4d,0xce,0x93,0x77,0x8a,0x4b,0x5d,0xc5,0x10,0xa7,0xb6,0x3e,
    0x09,0xfd,0x1b,0x7e,0x51,0x3f,0x68,0xa5,0xa3,0xbe,0xe5,0x2d,0x9f,0x81,0x44,0x0b])

CS_cipher = SBox([
    0x29,0xd,0x61,0x40,0x9c,0xeb,0x9e,0x8f,0x1f,0x85,0x5f,0x58,0x5b,0x1,0x39,0x86,
    0x97,0x2e,0xd7,0xd6,0x35,0xae,0x17,0x16,0x21,0xb6,0x69,0x4e,0xa5,0x72,0x87,0x8,
    0x3c,0x18,0xe6,0xe7,0xfa,0xad,0xb8,0x89,0xb7,0x0,0xf7,0x6f,0x73,0x84,0x11,0x63,
    0x3f,0x96,0x7f,0x6e,0xbf,0x14,0x9d,0xac,0xa4,0xe,0x7e,0xf6,0x20,0x4a,0x62,0x30,
    0x3,0xc5,0x4b,0x5a,0x46,0xa3,0x44,0x65,0x7d,0x4d,0x3d,0x42,0x79,0x49,0x1b,0x5c,
    0xf5,0x6c,0xb5,0x94,0x54,0xff,0x56,0x57,0xb,0xf4,0x43,0xc,0x4f,0x70,0x6d,0xa,
    0xe4,0x2,0x3e,0x2f,0xa2,0x47,0xe0,0xc1,0xd5,0x1a,0x95,0xa7,0x51,0x5e,0x33,0x2b,
    0x5d,0xd4,0x1d,0x2c,0xee,0x75,0xec,0xdd,0x7c,0x4c,0xa6,0xb4,0x78,0x48,0x3a,0x32,
    0x98,0xaf,0xc0,0xe1,0x2d,0x9,0xf,0x1e,0xb9,0x27,0x8a,0xe9,0xbd,0xe3,0x9f,0x7,
    0xb1,0xea,0x92,0x93,0x53,0x6a,0x31,0x10,0x80,0xf2,0xd8,0x9b,0x4,0x36,0x6,0x8e,
    0xbe,0xa9,0x64,0x45,0x38,0x1c,0x7a,0x6b,0xf3,0xa1,0xf0,0xcd,0x37,0x25,0x15,0x81,
    0xfb,0x90,0xe8,0xd9,0x7b,0x52,0x19,0x28,0x26,0x88,0xfc,0xd1,0xe2,0x8c,0xa0,0x34,
    0x82,0x67,0xda,0xcb,0xc7,0x41,0xe5,0xc4,0xc8,0xef,0xdb,0xc3,0xcc,0xab,0xce,0xed,
    0xd0,0xbb,0xd3,0xd2,0x71,0x68,0x13,0x12,0x9a,0xb3,0xc2,0xca,0xde,0x77,0xdc,0xdf,
    0x66,0x83,0xbc,0x8d,0x60,0xc6,0x22,0x23,0xb2,0x8b,0x91,0x5,0x76,0xcf,0x74,0xc9,
    0xaa,0xf1,0x99,0xa8,0x59,0x50,0x3b,0x2a,0xfe,0xf9,0x24,0xb0,0xba,0xfd,0xf8,0x55])


CSA = SBox([
    0x3a,0xea,0x68,0xfe,0x33,0xe9,0x88,0x1a,0x83,0xcf,0xe1,0x7f,0xba,0xe2,0x38,0x12,
    0xe8,0x27,0x61,0x95,0x0c,0x36,0xe5,0x70,0xa2,0x06,0x82,0x7c,0x17,0xa3,0x26,0x49,
    0xbe,0x7a,0x6d,0x47,0xc1,0x51,0x8f,0xf3,0xcc,0x5b,0x67,0xbd,0xcd,0x18,0x08,0xc9,
    0xff,0x69,0xef,0x03,0x4e,0x48,0x4a,0x84,0x3f,0xb4,0x10,0x04,0xdc,0xf5,0x5c,0xc6,
    0x16,0xab,0xac,0x4c,0xf1,0x6a,0x2f,0x3c,0x3b,0xd4,0xd5,0x94,0xd0,0xc4,0x63,0x62,
    0x71,0xa1,0xf9,0x4f,0x2e,0xaa,0xc5,0x56,0xe3,0x39,0x93,0xce,0x65,0x64,0xe4,0x58,
    0x6c,0x19,0x42,0x79,0xdd,0xee,0x96,0xf6,0x8a,0xec,0x1e,0x85,0x53,0x45,0xde,0xbb,
    0x7e,0x0a,0x9a,0x13,0x2a,0x9d,0xc2,0x5e,0x5a,0x1f,0x32,0x35,0x9c,0xa8,0x73,0x30,
    0x29,0x3d,0xe7,0x92,0x87,0x1b,0x2b,0x4b,0xa5,0x57,0x97,0x40,0x15,0xe6,0xbc,0x0e,
    0xeb,0xc3,0x34,0x2d,0xb8,0x44,0x25,0xa4,0x1c,0xc7,0x23,0xed,0x90,0x6e,0x50,0x00,
    0x99,0x9e,0x4d,0xd9,0xda,0x8d,0x6f,0x5f,0x3e,0xd7,0x21,0x74,0x86,0xdf,0x6b,0x05,
    0x8e,0x5d,0x37,0x11,0xd2,0x28,0x75,0xd6,0xa7,0x77,0x24,0xbf,0xf0,0xb0,0x02,0xb7,
    0xf8,0xfc,0x81,0x09,0xb1,0x01,0x76,0x91,0x7d,0x0f,0xc8,0xa0,0xf2,0xcb,0x78,0x60,
    0xd1,0xf7,0xe0,0xb5,0x98,0x22,0xb3,0x20,0x1d,0xa6,0xdb,0x7b,0x59,0x9f,0xae,0x31,
    0xfb,0xd3,0xb6,0xca,0x43,0x72,0x07,0xf4,0xd8,0x41,0x14,0x55,0x0d,0x54,0x8b,0xb9,
    0xad,0x46,0x0b,0xaf,0x80,0x52,0x2c,0xfa,0x8c,0x89,0x66,0xfd,0xb2,0xa9,0x9b,0xc0])

# source: http://citeseerx.ist.psu.edu/viewdoc/download;jsessionid=3672D97255B2446765DA47DA97960CDF?doi=10.1.1.118.6103&rep=rep1&type=pdf
CSS = SBox([
    0x33,0x73,0x3b,0x26,0x63,0x23,0x6b,0x76,0x3e,0x7e,0x36,0x2b,0x6e,0x2e,0x66,0x7b,
    0xd3,0x93,0xdb,0x06,0x43,0x03,0x4b,0x96,0xde,0x9e,0xd6,0x0b,0x4e,0x0e,0x46,0x9b,
    0x57,0x17,0x5f,0x82,0xc7,0x87,0xcf,0x12,0x5a,0x1a,0x52,0x8f,0xca,0x8a,0xc2,0x1f,
    0xd9,0x99,0xd1,0x00,0x49,0x09,0x41,0x90,0xd8,0x98,0xd0,0x01,0x48,0x08,0x40,0x91,
    0x3d,0x7d,0x35,0x24,0x6d,0x2d,0x65,0x74,0x3c,0x7c,0x34,0x25,0x6c,0x2c,0x64,0x75,
    0xdd,0x9d,0xd5,0x04,0x4d,0x0d,0x45,0x94,0xdc,0x9c,0xd4,0x05,0x4c,0x0c,0x44,0x95,
    0x59,0x19,0x51,0x80,0xc9,0x89,0xc1,0x10,0x58,0x18,0x50,0x81,0xc8,0x88,0xc0,0x11,
    0xd7,0x97,0xdf,0x02,0x47,0x07,0x4f,0x92,0xda,0x9a,0xd2,0x0f,0x4a,0x0a,0x42,0x9f,
    0x53,0x13,0x5b,0x86,0xc3,0x83,0xcb,0x16,0x5e,0x1e,0x56,0x8b,0xce,0x8e,0xc6,0x1b,
    0xb3,0xf3,0xbb,0xa6,0xe3,0xa3,0xeb,0xf6,0xbe,0xfe,0xb6,0xab,0xee,0xae,0xe6,0xfb,
    0x37,0x77,0x3f,0x22,0x67,0x27,0x6f,0x72,0x3a,0x7a,0x32,0x2f,0x6a,0x2a,0x62,0x7f,
    0xb9,0xf9,0xb1,0xa0,0xe9,0xa9,0xe1,0xf0,0xb8,0xf8,0xb0,0xa1,0xe8,0xa8,0xe0,0xf1,
    0x5d,0x1d,0x55,0x84,0xcd,0x8d,0xc5,0x14,0x5c,0x1c,0x54,0x85,0xcc,0x8c,0xc4,0x15,
    0xbd,0xfd,0xb5,0xa4,0xed,0xad,0xe5,0xf4,0xbc,0xfc,0xb4,0xa5,0xec,0xac,0xe4,0xf5,
    0x39,0x79,0x31,0x20,0x69,0x29,0x61,0x70,0x38,0x78,0x30,0x21,0x68,0x28,0x60,0x71,
    0xb7,0xf7,0xbf,0xa2,0xe7,0xa7,0xef,0xf2,0xba,0xfa,0xb2,0xaf,0xea,0xaa,0xe2,0xff])

DBlock = SBox([
    0x51,0x36,0x93,0x53,0xd9,0x4a,0xfc,0x58,0xe4,0x2e,0x0d,0x14,0xda,0x9d,0x91,0x69,
    0xef,0x72,0x03,0xc6,0x15,0x8d,0x5c,0x62,0x3f,0xb9,0x45,0x70,0x13,0xa3,0x95,0x6f,
    0x84,0xdb,0xb8,0x89,0x8a,0x6e,0xd4,0x7b,0x40,0xdc,0x9b,0x0c,0x50,0x8e,0xee,0x6a,
    0x88,0x3b,0x0f,0x6b,0x85,0xd3,0x54,0xa8,0x20,0xdf,0xb5,0x1b,0x32,0x7c,0x56,0x64,
    0x74,0xfa,0xc7,0x2d,0x96,0x17,0xae,0xcd,0xb4,0xf5,0x57,0x8c,0xf1,0xbc,0xd8,0xfe,
    0x27,0x06,0xe1,0xa9,0x1a,0x0e,0x5b,0x08,0xf4,0x9f,0x4b,0xed,0x73,0xb7,0xac,0x76,
    0x23,0xca,0x16,0xba,0xa7,0x00,0x8b,0x46,0x41,0xd5,0x7e,0xf2,0x05,0xf6,0x63,0x67,
    0x61,0x8f,0x3d,0xc8,0x1c,0x5a,0xb0,0x79,0x38,0x81,0xaa,0x33,0x97,0xe6,0x2c,0x01,
    0x22,0x87,0x4f,0xbe,0x24,0x71,0x35,0x9c,0xb1,0xad,0xc5,0x1d,0x80,0x3e,0x75,0xb3,
    0x28,0x68,0x2a,0xa0,0xbf,0x2f,0xb2,0xc4,0xce,0x19,0xd7,0xcf,0xaf,0x02,0xa4,0xa5,
    0x7a,0x39,0xd2,0x04,0xab,0xf7,0x60,0x2b,0x4c,0xec,0x4d,0x10,0x90,0x12,0xfb,0x78,
    0x82,0x4e,0x37,0x47,0xd6,0xa2,0xd1,0x86,0xb6,0xc1,0xe9,0xdd,0xa1,0xf8,0x55,0xde,
    0x98,0x7d,0xe5,0x30,0xfd,0xe2,0xcc,0x3a,0xea,0xd0,0x0a,0x29,0xe8,0xe3,0xeb,0xf0,
    0x9a,0x5d,0x3c,0x21,0xc0,0x48,0x6d,0x1e,0xe7,0x1f,0xc9,0x44,0x34,0x18,0x83,0xf9,
    0x59,0x5f,0x42,0x92,0x6c,0x11,0xa6,0x52,0xff,0x9e,0x49,0x26,0x07,0x43,0xbd,0xc3,
    0x99,0xf3,0x77,0x0b,0x5e,0xcb,0x09,0x31,0xe0,0xc2,0x65,0x7f,0x25,0x94,0xbb,0x66])

E2 = SBox([
    0xe1,0x42,0x3e,0x81,0x4e,0x17,0x9e,0xfd,0xb4,0x3f,0x2c,0xda,0x31,0x1e,0xe0,0x41,
    0xcc,0xf3,0x82,0x7d,0x7c,0x12,0x8e,0xbb,0xe4,0x58,0x15,0xd5,0x6f,0xe9,0x4c,0x4b,
    0x35,0x7b,0x5a,0x9a,0x90,0x45,0xbc,0xf8,0x79,0xd6,0x1b,0x88,0x02,0xab,0xcf,0x64,
    0x09,0x0c,0xf0,0x01,0xa4,0xb0,0xf6,0x93,0x43,0x63,0x86,0xdc,0x11,0xa5,0x83,0x8b,
    0xc9,0xd0,0x19,0x95,0x6a,0xa1,0x5c,0x24,0x6e,0x50,0x21,0x80,0x2f,0xe7,0x53,0x0f,
    0x91,0x22,0x04,0xed,0xa6,0x48,0x49,0x67,0xec,0xf7,0xc0,0x39,0xce,0xf2,0x2d,0xbe,
    0x5d,0x1c,0xe3,0x87,0x07,0x0d,0x7a,0xf4,0xfb,0x32,0xf5,0x8c,0xdb,0x8f,0x25,0x96,
    0xa8,0xea,0xcd,0x33,0x65,0x54,0x06,0x8d,0x89,0x0a,0x5e,0xd9,0x16,0x0e,0x71,0x6c,
    0x0b,0xff,0x60,0xd2,0x2e,0xd3,0xc8,0x55,0xc2,0x23,0xb7,0x74,0xe2,0x9b,0xdf,0x77,
    0x2b,0xb9,0x3c,0x62,0x13,0xe5,0x94,0x34,0xb1,0x27,0x84,0x9f,0xd7,0x51,0x00,0x61,
    0xad,0x85,0x73,0x03,0x08,0x40,0xef,0x68,0xfe,0x97,0x1f,0xde,0xaf,0x66,0xe8,0xb8,
    0xae,0xbd,0xb3,0xeb,0xc6,0x6b,0x47,0xa9,0xd8,0xa7,0x72,0xee,0x1d,0x7e,0xaa,0xb6,
    0x75,0xcb,0xd4,0x30,0x69,0x20,0x7f,0x37,0x5b,0x9d,0x78,0xa3,0xf1,0x76,0xfa,0x05,
    0x3d,0x3a,0x44,0x57,0x3b,0xca,0xc7,0x8a,0x18,0x46,0x9c,0xbf,0xba,0x38,0x56,0x1a,
    0x92,0x4d,0x26,0x29,0xa2,0x98,0x10,0x99,0x70,0xa0,0xc5,0x28,0xc1,0x6d,0x14,0xac,
    0xf9,0x5f,0x4f,0xc4,0xc3,0xd1,0xfc,0xdd,0xb2,0x59,0xe6,0xb5,0x36,0x52,0x4a,0x2a])

Enocoro = SBox([
    99,82,26,223,138,246,174,85,137,231,208,45,189,1,36,120,
    27,217,227,84,200,164,236,126,171,0,156,46,145,103,55,83,
    78,107,108,17,178,192,130,253,57,69,254,155,52,215,167,8,
    184,154,51,198,76,29,105,161,110,62,197,10,87,244,241,131,
    245,71,31,122,165,41,60,66,214,115,141,240,142,24,170,193,
    32,191,230,147,81,14,247,152,221,186,106,5,72,35,109,212,
    30,96,117,67,151,42,49,219,132,25,175,188,204,243,232,70,
    136,172,139,228,123,213,88,54,2,177,7,114,225,220,95,47,
    93,229,209,12,38,153,181,111,224,74,59,222,162,104,146,23,
    202,238,169,182,3,94,211,37,251,157,97,89,6,144,116,44,
    39,149,160,185,124,237,4,210,80,226,73,119,203,58,15,158,
    112,22,92,239,33,179,159,13,166,201,34,148,250,75,216,101,
    133,61,150,40,20,91,102,234,127,206,249,64,19,173,195,176,
    242,194,56,128,207,113,11,135,77,53,86,233,100,190,28,187,
    183,48,196,43,255,98,65,168,21,140,18,199,121,143,90,252,
    205,9,79,125,248,134,218,16,50,118,180,163,63,68,129,235])

Fantomas = SBox([
    0x1e,0x75,0x5f,0xe1,0x99,0xfc,0x89,0x2f,0x86,0xee,0xf1,0x7b,0x23,0x52,0x10,0x94,
    0x0c,0xb7,0x4d,0x67,0xd8,0x42,0xc8,0xd6,0xc4,0x6b,0xaa,0xba,0x3d,0xa5,0x00,0x33,
    0x53,0x2d,0x0b,0xb8,0xda,0xa8,0xc5,0x6c,0xca,0xb6,0xa4,0x22,0x60,0x07,0x5d,0xd7,
    0x4f,0xf4,0x15,0x32,0x81,0x1b,0x9c,0x8e,0x91,0x3f,0xe6,0xf9,0x70,0xe9,0x43,0x7e,
    0x8d,0xf3,0xcc,0x65,0x08,0x7a,0x18,0xab,0x16,0x6a,0x77,0xfd,0xa7,0xc0,0x82,0x04,
    0x9f,0x31,0xde,0xe3,0x49,0xd0,0x59,0x46,0x54,0xef,0x2e,0x3c,0xbb,0x21,0x92,0xb5,
    0x55,0x3e,0x0f,0xa9,0xdc,0xb9,0xc1,0x7f,0xce,0xa6,0xb4,0x30,0x72,0x03,0x5b,0xd1,
    0x4b,0xe4,0x13,0x20,0x85,0x1d,0x9a,0x8a,0x97,0x2c,0xf6,0xe8,0x62,0xf8,0x47,0x6d,
    0x29,0x41,0x68,0xd5,0xac,0xcb,0xbe,0x1a,0xb0,0xdb,0xc7,0x4e,0x17,0x64,0x26,0xa0,
    0x39,0x83,0x78,0x51,0xed,0x76,0xff,0xe2,0xf2,0x5c,0x9d,0x8f,0x0a,0x93,0x34,0x05,
    0x25,0x58,0x7c,0xcd,0xaf,0xdf,0xb3,0x19,0xbd,0xc2,0xd2,0x56,0x14,0x71,0x2a,0xa3,
    0x3a,0x80,0x61,0x44,0xf5,0x6e,0xeb,0xfb,0xe7,0x48,0x90,0x8c,0x06,0x9e,0x37,0x09,
    0x98,0xe5,0xd9,0x73,0x1f,0x6f,0x0d,0xbc,0x02,0x7d,0x63,0xea,0xb1,0xd4,0x96,0x12,
    0x88,0x27,0xc9,0xf7,0x5e,0xc6,0x4c,0x50,0x40,0xfa,0x3b,0x2b,0xae,0x35,0x84,0xa1,
    0x01,0x69,0x5a,0xfe,0x8b,0xec,0x95,0x28,0x9b,0xf0,0xe0,0x66,0x24,0x57,0x0e,0x87,
    0x1c,0xb2,0x45,0x74,0xd3,0x4a,0xcf,0xdd,0xc3,0x79,0xa2,0xbf,0x36,0xad,0x11,0x38])


FLY = SBox([
    0x00,0x9b,0xc2,0x15,0x5d,0x84,0x4c,0xd1,0x67,0x38,0xef,0xb0,0x7e,0x2b,0xf6,0xa3,
    0xb9,0xaa,0x36,0x78,0x2f,0x6e,0xe3,0xf7,0x12,0x5c,0x9a,0xd4,0x89,0xcd,0x01,0x45,
    0x2c,0x63,0x44,0xde,0x02,0x96,0x39,0x70,0xba,0xe4,0x18,0x57,0xa1,0xf5,0x8b,0xce,
    0x51,0x87,0xed,0xff,0xb5,0xa8,0xca,0x1b,0xdf,0x90,0x6c,0x32,0x46,0x03,0x7d,0x29,
    0xd5,0xf2,0x20,0x5b,0xcc,0x31,0x04,0xbd,0xa6,0x41,0x8e,0x79,0xea,0x9f,0x68,0x1c,
    0x48,0xe6,0x69,0x8a,0x13,0x77,0x9e,0xaf,0xf3,0x05,0xcb,0x2d,0xb4,0xd0,0x37,0x52,
    0xc4,0x3e,0x93,0xac,0x40,0xe9,0x22,0x56,0x7b,0x8d,0xf1,0x06,0x17,0x62,0xbf,0xda,
    0x1d,0x7f,0x07,0xb1,0xdb,0xfa,0x65,0x88,0x2e,0xc9,0xa5,0x43,0x58,0x3c,0xe0,0x94,
    0x76,0x21,0xab,0xfd,0x6a,0x3f,0xb7,0xe2,0xdd,0x4f,0x53,0x8c,0xc0,0x19,0x95,0x08,
    0x83,0xc5,0x4e,0x09,0x14,0x50,0xd8,0x9c,0xf4,0xee,0x27,0x61,0x3b,0x7a,0xa2,0xb6,
    0xfe,0xa9,0x81,0xc6,0xe8,0xbc,0x1f,0x5a,0x35,0x72,0x99,0x0a,0xd3,0x47,0x24,0x6d,
    0x0b,0x4d,0x75,0x23,0x97,0xd2,0x60,0x34,0xc8,0x16,0xa0,0xbb,0xfc,0xe1,0x5e,0x8f,
    0xe7,0x98,0x1a,0x64,0xae,0x4b,0x71,0x85,0x0c,0xb3,0x3d,0xcf,0x55,0x28,0xd9,0xf0,
    0xb2,0xdc,0x5f,0x30,0xf9,0x0d,0x26,0xc3,0x91,0xa7,0x74,0x1e,0x82,0x66,0x4a,0xeb,
    0x6f,0x10,0xb8,0xd7,0x86,0x73,0xfb,0x0e,0x59,0x2a,0x42,0xe5,0x9d,0xa4,0x33,0xc7,
    0x3a,0x54,0xec,0x92,0xc1,0x25,0xad,0x49,0x80,0x6b,0xd6,0xf8,0x0f,0xbe,0x7c,0x11])

Fox = SBox([
    0x5D,0xDE,0x00,0xB7,0xD3,0xCA,0x3C,0x0D,0xC3,0xF8,0xCB,0x8D,0x76,0x89,0xAA,0x12,
    0x88,0x22,0x4F,0xDB,0x6D,0x47,0xE4,0x4C,0x78,0x9A,0x49,0x93,0xC4,0xC0,0x86,0x13,
    0xA9,0x20,0x53,0x1C,0x4E,0xCF,0x35,0x39,0xB4,0xA1,0x54,0x64,0x03,0xC7,0x85,0x5C,
    0x5B,0xCD,0xD8,0x72,0x96,0x42,0xB8,0xE1,0xA2,0x60,0xEF,0xBD,0x02,0xAF,0x8C,0x73,
    0x7C,0x7F,0x5E,0xF9,0x65,0xE6,0xEB,0xAD,0x5A,0xA5,0x79,0x8E,0x15,0x30,0xEC,0xA4,
    0xC2,0x3E,0xE0,0x74,0x51,0xFB,0x2D,0x6E,0x94,0x4D,0x55,0x34,0xAE,0x52,0x7E,0x9D,
    0x4A,0xF7,0x80,0xF0,0xD0,0x90,0xA7,0xE8,0x9F,0x50,0xD5,0xD1,0x98,0xCC,0xA0,0x17,
    0xF4,0xB6,0xC1,0x28,0x5F,0x26,0x01,0xAB,0x25,0x38,0x82,0x7D,0x48,0xFC,0x1B,0xCE,
    0x3F,0x6B,0xE2,0x67,0x66,0x43,0x59,0x19,0x84,0x3D,0xF5,0x2F,0xC9,0xBC,0xD9,0x95,
    0x29,0x41,0xDA,0x1A,0xB0,0xE9,0x69,0xD2,0x7B,0xD7,0x11,0x9B,0x33,0x8A,0x23,0x09,
    0xD4,0x71,0x44,0x68,0x6F,0xF2,0x0E,0xDF,0x87,0xDC,0x83,0x18,0x6A,0xEE,0x99,0x81,
    0x62,0x36,0x2E,0x7A,0xFE,0x45,0x9C,0x75,0x91,0x0C,0x0F,0xE7,0xF6,0x14,0x63,0x1D,
    0x0B,0x8B,0xB3,0xF3,0xB2,0x3B,0x08,0x4B,0x10,0xA6,0x32,0xB9,0xA8,0x92,0xF1,0x56,
    0xDD,0x21,0xBF,0x04,0xBE,0xD6,0xFD,0x77,0xEA,0x3A,0xC8,0x8F,0x57,0x1E,0xFA,0x2B,
    0x58,0xC5,0x27,0xAC,0xE3,0xED,0x97,0xBB,0x46,0x05,0x40,0x31,0xE5,0x37,0x2C,0x9E,
    0x0A,0xB1,0xB5,0x06,0x6C,0x1F,0xA3,0x2A,0x70,0xFF,0xBA,0x07,0x24,0x16,0xC6,0x61])

Iceberg = SBox([
    0x24,0xc1,0x38,0x30,0xe7,0x57,0xdf,0x20,0x3e,0x99,0x1a,0x34,0xca,0xd6,0x52,0xfd,
    0x40,0x6c,0xd3,0x3d,0x4a,0x59,0xf8,0x77,0xfb,0x61,0x0a,0x56,0xb9,0xd2,0xfc,0xf1,
    0x07,0xf5,0x93,0xcd,0x00,0xb6,0x62,0xa7,0x63,0xfe,0x44,0xbd,0x5f,0x92,0x6b,0x68,
    0x03,0x4e,0xa2,0x97,0x0b,0x60,0x83,0xa3,0x02,0xe5,0x45,0x67,0xf4,0x13,0x08,0x8b,
    0x10,0xce,0xbe,0xb4,0x2a,0x3a,0x96,0x84,0xc8,0x9f,0x14,0xc0,0xc4,0x6f,0x31,0xd9,
    0xab,0xae,0x0e,0x64,0x7c,0xda,0x1b,0x05,0xa8,0x15,0xa5,0x90,0x94,0x85,0x71,0x2c,
    0x35,0x19,0x26,0x28,0x53,0xe2,0x7f,0x3b,0x2f,0xa9,0xcc,0x2e,0x11,0x76,0xed,0x4d,
    0x87,0x5e,0xc2,0xc7,0x80,0xb0,0x6d,0x17,0xb2,0xff,0xe4,0xb7,0x54,0x9d,0xb8,0x66,
    0x74,0x9c,0xdb,0x36,0x47,0x5d,0xde,0x70,0xd5,0x91,0xaa,0x3f,0xc9,0xd8,0xf3,0xf2,
    0x5b,0x89,0x2d,0x22,0x5c,0xe1,0x46,0x33,0xe6,0x09,0xbc,0xe8,0x81,0x7d,0xe9,0x49,
    0xe0,0xb1,0x32,0x37,0xea,0x5a,0xf6,0x27,0x58,0x69,0x8a,0x50,0xba,0xdd,0x51,0xf9,
    0x75,0xa1,0x78,0xd0,0x43,0xf7,0x25,0x7b,0x7e,0x1c,0xac,0xd4,0x9a,0x2b,0x42,0xe3,
    0x4b,0x01,0x72,0xd7,0x4c,0xfa,0xeb,0x73,0x48,0x8c,0x0c,0xf0,0x6a,0x23,0x41,0xec,
    0xb3,0xef,0x1d,0x12,0xbb,0x88,0x0d,0xc3,0x8d,0x4f,0x55,0x82,0xee,0xad,0x86,0x06,
    0xa0,0x95,0x65,0xbf,0x7a,0x39,0x98,0x04,0x9b,0x9e,0xa4,0xc6,0xcf,0x6e,0xdc,0xd1,
    0xcb,0x1f,0x8f,0x8e,0x3c,0x21,0xa6,0xb5,0x16,0xaf,0xc5,0x18,0x1e,0x0f,0x29,0x79])

Iraqi = SBox([
    173,84,240,67,1,53,254,36,41,172,115,109,223,199,152,189,90,46,
    149,193,218,130,250,40,203,4,35,237,236,246,213,143,169,176,48,
    23,61,206,69,34,97,155,4,109,183,220,42,64,21,123,29,233,253,
    105,183,209,1,191,113,12,46,7,8,183,166,199,166,7,78,37,135,
    252,174,84,140,164,152,94,22,185,59,68,181,60,176,67,51,25,28,
    190,138,198,44,90,92,221,149,175,186,25,49,210,50,237,41,207,
    31,226,114,121,230,15,58,25,142,58,98,232,59,3,189,28,8,116,
    131,185,78,250,239,33,116,173,94,45,104,62,122,179,18,150,246,
    250,17,8,79,157,225,238,47,10,133,58,8,126,82,68,153,141,2,158,
    204,50,130,53,59,32,243,160,172,35,24,107,35,115,228,143,28,
    224,77,55,25,28,120,89,186,152,49,84,117,180,30,138,134,77,182,
    157,61,230,22,149,54,15,110,32,213,155,106,78,16,23,89,140,158,
    169,96,136,186,104,30,199,67,35,218,159,210,109,28,238,33,150,
    173,180,247,201,83,150,105,164,228,59,207,101,221,99,52,120,
    199,31,6,144,202,215,209,49,42,195])

iScream = SBox([
    0x00,0x85,0x65,0xD2,0x5B,0xFF,0x7A,0xCE,0x4D,0xE2,0x2C,0x36,0x92,0x15,0xBD,0xAD,
    0x57,0xF3,0x37,0x2D,0x88,0x0D,0xAC,0xBC,0x18,0x9F,0x7E,0xCA,0x41,0xEE,0x61,0xD6,
    0x59,0xEC,0x78,0xD4,0x47,0xF9,0x26,0xA3,0x90,0x8B,0xBF,0x30,0x0A,0x13,0x6F,0xC0,
    0x2B,0xAE,0x91,0x8A,0xD8,0x74,0x0B,0x12,0xCC,0x63,0xFD,0x43,0xB2,0x3D,0xE8,0x5D,
    0xB6,0x1C,0x83,0x3B,0xC8,0x45,0x9D,0x24,0x52,0xDD,0xE4,0xF4,0xAB,0x08,0x77,0x6D,
    0xF5,0xE5,0x48,0xC5,0x6C,0x76,0xBA,0x10,0x99,0x20,0xA7,0x04,0x87,0x3F,0xD0,0x5F,
    0xA5,0x1E,0x9B,0x39,0xB0,0x02,0xEA,0x67,0xC6,0xDF,0x71,0xF6,0x54,0x4F,0x8D,0x2E,
    0xE7,0x6A,0xC7,0xDE,0x35,0x97,0x55,0x4E,0x22,0x81,0x06,0xB4,0x7C,0xFB,0x1A,0xA1,
    0xD5,0x79,0xFC,0x42,0x84,0x01,0xE9,0x5C,0x14,0x93,0x33,0x29,0xC1,0x6E,0xA8,0xB8,
    0x28,0x32,0x0C,0x89,0xB9,0xA9,0xD9,0x75,0xED,0x58,0xCD,0x62,0xF8,0x46,0x9E,0x19,
    0xCB,0x7F,0xA2,0x27,0xD7,0x60,0xFE,0x5A,0x8E,0x95,0xE3,0x4C,0x16,0x0F,0x31,0xBE,
    0x64,0xD3,0x3C,0xB3,0x7B,0xCF,0x40,0xEF,0x8F,0x94,0x56,0xF2,0x17,0x0E,0xAF,0x2A,
    0x2F,0x8C,0xF1,0xE1,0xDC,0x53,0x68,0x72,0x44,0xC9,0x1B,0xA0,0x38,0x9A,0x07,0xB5,
    0x5E,0xD1,0x03,0xB1,0x23,0x80,0x1F,0xA4,0x34,0x96,0xE0,0xF0,0xC4,0x49,0x73,0x69,
    0xDA,0xC3,0x09,0xAA,0x4A,0x51,0xF7,0x70,0x3E,0x86,0x66,0xEB,0x21,0x98,0x1D,0xB7,
    0xDB,0xC2,0xBB,0x11,0x4B,0x50,0x6B,0xE6,0x9C,0x25,0xFA,0x7D,0x82,0x3A,0xA6,0x05])

Kalyna_pi0 = SBox([
    0xa8,0x43,0x5f,0x6,0x6b,0x75,0x6c,0x59,0x71,0xdf,0x87,0x95,0x17,0xf0,0xd8,0x9,
    0x6d,0xf3,0x1d,0xcb,0xc9,0x4d,0x2c,0xaf,0x79,0xe0,0x97,0xfd,0x6f,0x4b,0x45,0x39,
    0x3e,0xdd,0xa3,0x4f,0xb4,0xb6,0x9a,0xe,0x1f,0xbf,0x15,0xe1,0x49,0xd2,0x93,0xc6,
    0x92,0x72,0x9e,0x61,0xd1,0x63,0xfa,0xee,0xf4,0x19,0xd5,0xad,0x58,0xa4,0xbb,0xa1,
    0xdc,0xf2,0x83,0x37,0x42,0xe4,0x7a,0x32,0x9c,0xcc,0xab,0x4a,0x8f,0x6e,0x4,0x27,
    0x2e,0xe7,0xe2,0x5a,0x96,0x16,0x23,0x2b,0xc2,0x65,0x66,0xf,0xbc,0xa9,0x47,0x41,
    0x34,0x48,0xfc,0xb7,0x6a,0x88,0xa5,0x53,0x86,0xf9,0x5b,0xdb,0x38,0x7b,0xc3,0x1e,
    0x22,0x33,0x24,0x28,0x36,0xc7,0xb2,0x3b,0x8e,0x77,0xba,0xf5,0x14,0x9f,0x8,0x55,
    0x9b,0x4c,0xfe,0x60,0x5c,0xda,0x18,0x46,0xcd,0x7d,0x21,0xb0,0x3f,0x1b,0x89,0xff,
    0xeb,0x84,0x69,0x3a,0x9d,0xd7,0xd3,0x70,0x67,0x40,0xb5,0xde,0x5d,0x30,0x91,0xb1,
    0x78,0x11,0x1,0xe5,0x0,0x68,0x98,0xa0,0xc5,0x2,0xa6,0x74,0x2d,0xb,0xa2,0x76,
    0xb3,0xbe,0xce,0xbd,0xae,0xe9,0x8a,0x31,0x1c,0xec,0xf1,0x99,0x94,0xaa,0xf6,0x26,
    0x2f,0xef,0xe8,0x8c,0x35,0x3,0xd4,0x7f,0xfb,0x5,0xc1,0x5e,0x90,0x20,0x3d,0x82,
    0xf7,0xea,0xa,0xd,0x7e,0xf8,0x50,0x1a,0xc4,0x7,0x57,0xb8,0x3c,0x62,0xe3,0xc8,
    0xac,0x52,0x64,0x10,0xd0,0xd9,0x13,0xc,0x12,0x29,0x51,0xb9,0xcf,0xd6,0x73,0x8d,
    0x81,0x54,0xc0,0xed,0x4e,0x44,0xa7,0x2a,0x85,0x25,0xe6,0xca,0x7c,0x8b,0x56,0x80])

Kalyna_pi1 = SBox([
    0xce,0xbb,0xeb,0x92,0xea,0xcb,0x13,0xc1,0xe9,0x3a,0xd6,0xb2,0xd2,0x90,0x17,0xf8,
    0x42,0x15,0x56,0xb4,0x65,0x1c,0x88,0x43,0xc5,0x5c,0x36,0xba,0xf5,0x57,0x67,0x8d,
    0x31,0xf6,0x64,0x58,0x9e,0xf4,0x22,0xaa,0x75,0xf,0x2,0xb1,0xdf,0x6d,0x73,0x4d,
    0x7c,0x26,0x2e,0xf7,0x8,0x5d,0x44,0x3e,0x9f,0x14,0xc8,0xae,0x54,0x10,0xd8,0xbc,
    0x1a,0x6b,0x69,0xf3,0xbd,0x33,0xab,0xfa,0xd1,0x9b,0x68,0x4e,0x16,0x95,0x91,0xee,
    0x4c,0x63,0x8e,0x5b,0xcc,0x3c,0x19,0xa1,0x81,0x49,0x7b,0xd9,0x6f,0x37,0x60,0xca,
    0xe7,0x2b,0x48,0xfd,0x96,0x45,0xfc,0x41,0x12,0xd,0x79,0xe5,0x89,0x8c,0xe3,0x20,
    0x30,0xdc,0xb7,0x6c,0x4a,0xb5,0x3f,0x97,0xd4,0x62,0x2d,0x6,0xa4,0xa5,0x83,0x5f,
    0x2a,0xda,0xc9,0x0,0x7e,0xa2,0x55,0xbf,0x11,0xd5,0x9c,0xcf,0xe,0xa,0x3d,0x51,
    0x7d,0x93,0x1b,0xfe,0xc4,0x47,0x9,0x86,0xb,0x8f,0x9d,0x6a,0x7,0xb9,0xb0,0x98,
    0x18,0x32,0x71,0x4b,0xef,0x3b,0x70,0xa0,0xe4,0x40,0xff,0xc3,0xa9,0xe6,0x78,0xf9,
    0x8b,0x46,0x80,0x1e,0x38,0xe1,0xb8,0xa8,0xe0,0xc,0x23,0x76,0x1d,0x25,0x24,0x5,
    0xf1,0x6e,0x94,0x28,0x9a,0x84,0xe8,0xa3,0x4f,0x77,0xd3,0x85,0xe2,0x52,0xf2,0x82,
    0x50,0x7a,0x2f,0x74,0x53,0xb3,0x61,0xaf,0x39,0x35,0xde,0xcd,0x1f,0x99,0xac,0xad,
    0x72,0x2c,0xdd,0xd0,0x87,0xbe,0x5e,0xa6,0xec,0x4,0xc6,0x3,0x34,0xfb,0xdb,0x59,
    0xb6,0xc2,0x1,0xf0,0x5a,0xed,0xa7,0x66,0x21,0x7f,0x8a,0x27,0xc7,0xc0,0x29,0xd7])

Kalyna_pi2 = SBox([
    0x93,0xd9,0x9a,0xb5,0x98,0x22,0x45,0xfc,0xba,0x6a,0xdf,0x2,0x9f,0xdc,0x51,0x59,
    0x4a,0x17,0x2b,0xc2,0x94,0xf4,0xbb,0xa3,0x62,0xe4,0x71,0xd4,0xcd,0x70,0x16,0xe1,
    0x49,0x3c,0xc0,0xd8,0x5c,0x9b,0xad,0x85,0x53,0xa1,0x7a,0xc8,0x2d,0xe0,0xd1,0x72,
    0xa6,0x2c,0xc4,0xe3,0x76,0x78,0xb7,0xb4,0x9,0x3b,0xe,0x41,0x4c,0xde,0xb2,0x90,
    0x25,0xa5,0xd7,0x3,0x11,0x0,0xc3,0x2e,0x92,0xef,0x4e,0x12,0x9d,0x7d,0xcb,0x35,
    0x10,0xd5,0x4f,0x9e,0x4d,0xa9,0x55,0xc6,0xd0,0x7b,0x18,0x97,0xd3,0x36,0xe6,0x48,
    0x56,0x81,0x8f,0x77,0xcc,0x9c,0xb9,0xe2,0xac,0xb8,0x2f,0x15,0xa4,0x7c,0xda,0x38,
    0x1e,0xb,0x5,0xd6,0x14,0x6e,0x6c,0x7e,0x66,0xfd,0xb1,0xe5,0x60,0xaf,0x5e,0x33,
    0x87,0xc9,0xf0,0x5d,0x6d,0x3f,0x88,0x8d,0xc7,0xf7,0x1d,0xe9,0xec,0xed,0x80,0x29,
    0x27,0xcf,0x99,0xa8,0x50,0xf,0x37,0x24,0x28,0x30,0x95,0xd2,0x3e,0x5b,0x40,0x83,
    0xb3,0x69,0x57,0x1f,0x7,0x1c,0x8a,0xbc,0x20,0xeb,0xce,0x8e,0xab,0xee,0x31,0xa2,
    0x73,0xf9,0xca,0x3a,0x1a,0xfb,0xd,0xc1,0xfe,0xfa,0xf2,0x6f,0xbd,0x96,0xdd,0x43,
    0x52,0xb6,0x8,0xf3,0xae,0xbe,0x19,0x89,0x32,0x26,0xb0,0xea,0x4b,0x64,0x84,0x82,
    0x6b,0xf5,0x79,0xbf,0x1,0x5f,0x75,0x63,0x1b,0x23,0x3d,0x68,0x2a,0x65,0xe8,0x91,
    0xf6,0xff,0x13,0x58,0xf1,0x47,0xa,0x7f,0xc5,0xa7,0xe7,0x61,0x5a,0x6,0x46,0x44,
    0x42,0x4,0xa0,0xdb,0x39,0x86,0x54,0xaa,0x8c,0x34,0x21,0x8b,0xf8,0xc,0x74,0x67])

Kalyna_pi3 = SBox([
    0x68,0x8d,0xca,0x4d,0x73,0x4b,0x4e,0x2a,0xd4,0x52,0x26,0xb3,0x54,0x1e,0x19,0x1f,
    0x22,0x3,0x46,0x3d,0x2d,0x4a,0x53,0x83,0x13,0x8a,0xb7,0xd5,0x25,0x79,0xf5,0xbd,
    0x58,0x2f,0xd,0x2,0xed,0x51,0x9e,0x11,0xf2,0x3e,0x55,0x5e,0xd1,0x16,0x3c,0x66,
    0x70,0x5d,0xf3,0x45,0x40,0xcc,0xe8,0x94,0x56,0x8,0xce,0x1a,0x3a,0xd2,0xe1,0xdf,
    0xb5,0x38,0x6e,0xe,0xe5,0xf4,0xf9,0x86,0xe9,0x4f,0xd6,0x85,0x23,0xcf,0x32,0x99,
    0x31,0x14,0xae,0xee,0xc8,0x48,0xd3,0x30,0xa1,0x92,0x41,0xb1,0x18,0xc4,0x2c,0x71,
    0x72,0x44,0x15,0xfd,0x37,0xbe,0x5f,0xaa,0x9b,0x88,0xd8,0xab,0x89,0x9c,0xfa,0x60,
    0xea,0xbc,0x62,0xc,0x24,0xa6,0xa8,0xec,0x67,0x20,0xdb,0x7c,0x28,0xdd,0xac,0x5b,
    0x34,0x7e,0x10,0xf1,0x7b,0x8f,0x63,0xa0,0x5,0x9a,0x43,0x77,0x21,0xbf,0x27,0x9,
    0xc3,0x9f,0xb6,0xd7,0x29,0xc2,0xeb,0xc0,0xa4,0x8b,0x8c,0x1d,0xfb,0xff,0xc1,0xb2,
    0x97,0x2e,0xf8,0x65,0xf6,0x75,0x7,0x4,0x49,0x33,0xe4,0xd9,0xb9,0xd0,0x42,0xc7,
    0x6c,0x90,0x0,0x8e,0x6f,0x50,0x1,0xc5,0xda,0x47,0x3f,0xcd,0x69,0xa2,0xe2,0x7a,
    0xa7,0xc6,0x93,0xf,0xa,0x6,0xe6,0x2b,0x96,0xa3,0x1c,0xaf,0x6a,0x12,0x84,0x39,
    0xe7,0xb0,0x82,0xf7,0xfe,0x9d,0x87,0x5c,0x81,0x35,0xde,0xb4,0xa5,0xfc,0x80,0xef,
    0xcb,0xbb,0x6b,0x76,0xba,0x5a,0x7d,0x78,0xb,0x95,0xe3,0xad,0x74,0x98,0x3b,0x36,
    0x64,0x6d,0xdc,0xf0,0x59,0xa9,0x4c,0x17,0x7f,0x91,0xb8,0xc9,0x57,0x1b,0xe0,0x61])

Khazad = SBox([
    0xba,0x54,0x2f,0x74,0x53,0xd3,0xd2,0x4d,0x50,0xac,0x8d,0xbf,0x70,0x52,0x9a,0x4c,
    0xea,0xd5,0x97,0xd1,0x33,0x51,0x5b,0xa6,0xde,0x48,0xa8,0x99,0xdb,0x32,0xb7,0xfc,
    0xe3,0x9e,0x91,0x9b,0xe2,0xbb,0x41,0x6e,0xa5,0xcb,0x6b,0x95,0xa1,0xf3,0xb1,0x02,
    0xcc,0xc4,0x1d,0x14,0xc3,0x63,0xda,0x5d,0x5f,0xdc,0x7d,0xcd,0x7f,0x5a,0x6c,0x5c,
    0xf7,0x26,0xff,0xed,0xe8,0x9d,0x6f,0x8e,0x19,0xa0,0xf0,0x89,0x0f,0x07,0xaf,0xfb,
    0x08,0x15,0x0d,0x04,0x01,0x64,0xdf,0x76,0x79,0xdd,0x3d,0x16,0x3f,0x37,0x6d,0x38,
    0xb9,0x73,0xe9,0x35,0x55,0x71,0x7b,0x8c,0x72,0x88,0xf6,0x2a,0x3e,0x5e,0x27,0x46,
    0x0c,0x65,0x68,0x61,0x03,0xc1,0x57,0xd6,0xd9,0x58,0xd8,0x66,0xd7,0x3a,0xc8,0x3c,
    0xfa,0x96,0xa7,0x98,0xec,0xb8,0xc7,0xae,0x69,0x4b,0xab,0xa9,0x67,0x0a,0x47,0xf2,
    0xb5,0x22,0xe5,0xee,0xbe,0x2b,0x81,0x12,0x83,0x1b,0x0e,0x23,0xf5,0x45,0x21,0xce,
    0x49,0x2c,0xf9,0xe6,0xb6,0x28,0x17,0x82,0x1a,0x8b,0xfe,0x8a,0x09,0xc9,0x87,0x4e,
    0xe1,0x2e,0xe4,0xe0,0xeb,0x90,0xa4,0x1e,0x85,0x60,0x00,0x25,0xf4,0xf1,0x94,0x0b,
    0xe7,0x75,0xef,0x34,0x31,0xd4,0xd0,0x86,0x7e,0xad,0xfd,0x29,0x30,0x3b,0x9f,0xf8,
    0xc6,0x13,0x06,0x05,0xc5,0x11,0x77,0x7c,0x7a,0x78,0x36,0x1c,0x39,0x59,0x18,0x56,
    0xb3,0xb0,0x24,0x20,0xb2,0x92,0xa3,0xc0,0x44,0x62,0x10,0xb4,0x84,0x43,0x93,0xc2,
    0x4a,0xbd,0x8f,0x2d,0xbc,0x9c,0x6a,0x40,0xcf,0xa2,0x80,0x4f,0x1f,0xca,0xaa,0x42])

Kuznyechik = SBox([
    0xFC,0xEE,0xDD,0x11,0xCF,0x6E,0x31,0x16,0xFB,0xC4,0xFA,0xDA,0x23,0xC5,0x04,0x4D,
    0xE9,0x77,0xF0,0xDB,0x93,0x2E,0x99,0xBA,0x17,0x36,0xF1,0xBB,0x14,0xCD,0x5F,0xC1,
    0xF9,0x18,0x65,0x5A,0xE2,0x5C,0xEF,0x21,0x81,0x1C,0x3C,0x42,0x8B,0x01,0x8E,0x4F,
    0x05,0x84,0x02,0xAE,0xE3,0x6A,0x8F,0xA0,0x06,0x0B,0xED,0x98,0x7F,0xD4,0xD3,0x1F,
    0xEB,0x34,0x2C,0x51,0xEA,0xC8,0x48,0xAB,0xF2,0x2A,0x68,0xA2,0xFD,0x3A,0xCE,0xCC,
    0xB5,0x70,0x0E,0x56,0x08,0x0C,0x76,0x12,0xBF,0x72,0x13,0x47,0x9C,0xB7,0x5D,0x87,
    0x15,0xA1,0x96,0x29,0x10,0x7B,0x9A,0xC7,0xF3,0x91,0x78,0x6F,0x9D,0x9E,0xB2,0xB1,
    0x32,0x75,0x19,0x3D,0xFF,0x35,0x8A,0x7E,0x6D,0x54,0xC6,0x80,0xC3,0xBD,0x0D,0x57,
    0xDF,0xF5,0x24,0xA9,0x3E,0xA8,0x43,0xC9,0xD7,0x79,0xD6,0xF6,0x7C,0x22,0xB9,0x03,
    0xE0,0x0F,0xEC,0xDE,0x7A,0x94,0xB0,0xBC,0xDC,0xE8,0x28,0x50,0x4E,0x33,0x0A,0x4A,
    0xA7,0x97,0x60,0x73,0x1E,0x00,0x62,0x44,0x1A,0xB8,0x38,0x82,0x64,0x9F,0x26,0x41,
    0xAD,0x45,0x46,0x92,0x27,0x5E,0x55,0x2F,0x8C,0xA3,0xA5,0x7D,0x69,0xD5,0x95,0x3B,
    0x07,0x58,0xB3,0x40,0x86,0xAC,0x1D,0xF7,0x30,0x37,0x6B,0xE4,0x88,0xD9,0xE7,0x89,
    0xE1,0x1B,0x83,0x49,0x4C,0x3F,0xF8,0xFE,0x8D,0x53,0xAA,0x90,0xCA,0xD8,0x85,0x61,
    0x20,0x71,0x67,0xA4,0x2D,0x2B,0x09,0x5B,0xCB,0x9B,0x25,0xD0,0xBE,0xE5,0x6C,0x52,
    0x59,0xA6,0x74,0xD2,0xE6,0xF4,0xB4,0xC0,0xD1,0x66,0xAF,0xC2,0x39,0x4B,0x63,0xB6])
Kuznechik = Kuznyechik
Streebog = Kuznyechik
Stribog = Kuznyechik

#source: https://crypto.stackexchange.com/questions/11935/how-is-the-md2-hash-function-s-table-constructed-from-pi
# structure: pseudo-random, derived from the digits of pi
MD2 = SBox([
    0x29,0x2E,0x43,0xC9,0xA2,0xD8,0x7C,0x01,0x3D,0x36,0x54,0xA1,0xEC,0xF0,0x06,0x13,
    0x62,0xA7,0x05,0xF3,0xC0,0xC7,0x73,0x8C,0x98,0x93,0x2B,0xD9,0xBC,0x4C,0x82,0xCA,
    0x1E,0x9B,0x57,0x3C,0xFD,0xD4,0xE0,0x16,0x67,0x42,0x6F,0x18,0x8A,0x17,0xE5,0x12,
    0xBE,0x4E,0xC4,0xD6,0xDA,0x9E,0xDE,0x49,0xA0,0xFB,0xF5,0x8E,0xBB,0x2F,0xEE,0x7A,
    0xA9,0x68,0x79,0x91,0x15,0xB2,0x07,0x3F,0x94,0xC2,0x10,0x89,0x0B,0x22,0x5F,0x21,
    0x80,0x7F,0x5D,0x9A,0x5A,0x90,0x32,0x27,0x35,0x3E,0xCC,0xE7,0xBF,0xF7,0x97,0x03,
    0xFF,0x19,0x30,0xB3,0x48,0xA5,0xB5,0xD1,0xD7,0x5E,0x92,0x2A,0xAC,0x56,0xAA,0xC6,
    0x4F,0xB8,0x38,0xD2,0x96,0xA4,0x7D,0xB6,0x76,0xFC,0x6B,0xE2,0x9C,0x74,0x04,0xF1,
    0x45,0x9D,0x70,0x59,0x64,0x71,0x87,0x20,0x86,0x5B,0xCF,0x65,0xE6,0x2D,0xA8,0x02,
    0x1B,0x60,0x25,0xAD,0xAE,0xB0,0xB9,0xF6,0x1C,0x46,0x61,0x69,0x34,0x40,0x7E,0x0F,
    0x55,0x47,0xA3,0x23,0xDD,0x51,0xAF,0x3A,0xC3,0x5C,0xF9,0xCE,0xBA,0xC5,0xEA,0x26,
    0x2C,0x53,0x0D,0x6E,0x85,0x28,0x84,0x09,0xD3,0xDF,0xCD,0xF4,0x41,0x81,0x4D,0x52,
    0x6A,0xDC,0x37,0xC8,0x6C,0xC1,0xAB,0xFA,0x24,0xE1,0x7B,0x08,0x0C,0xBD,0xB1,0x4A,
    0x78,0x88,0x95,0x8B,0xE3,0x63,0xE8,0x6D,0xE9,0xCB,0xD5,0xFE,0x3B,0x00,0x1D,0x39,
    0xF2,0xEF,0xB7,0x0E,0x66,0x58,0xD0,0xE4,0xA6,0x77,0x72,0xF8,0xEB,0x75,0x4B,0x0A,
    0x31,0x44,0x50,0xB4,0x8F,0xED,0x1F,0x1A,0xDB,0x99,0x8D,0x33,0x9F,0x11,0x83,0x14])

newDES = SBox([
    32,137,239,188,102,125,221, 72,212, 68, 81, 37, 86,237,147,149,
    70,229, 17,124,115,207, 33, 20,122,143, 25,215, 51,183,138,142,
    146,211,110,173,  1,228,189, 14,103, 78,162, 36,253,167,116,255,
    158, 45,185, 50, 98,168,250,235, 54,141,195,247,240, 63,148,  2,
    224,169,214,180, 62, 22,117,108, 19,172,161,159,160, 47, 43,171,
    194,175,178, 56,196,112, 23,220, 89, 21,164,130,157,  8, 85,251,
    216, 44, 94,179,226, 38, 90,119, 40,202, 34,206, 35, 69,231,246,
    29,109, 74, 71,176,  6, 60,145, 65, 13, 77,151, 12,127, 95,199,
    57,101,  5,232,150,210,129, 24,181, 10,121,187, 48,193,139,252,
    219, 64, 88,233, 96,128, 80, 53,191,144,218, 11,106,132,155,104,
    91,136, 31, 42,243, 66,126,135, 30, 26, 87,186,182,154,242,123,
    82,166,208, 39,152,190,113,205,114,105,225, 84, 73,163, 99,111,
    204, 61,200,217,170, 15,198, 28,192,254,134,234,222,  7,236,248,
    201, 41,177,156, 92,131, 67,249,245,184,203,  9,241,  0, 27, 46,
    133,174, 75, 18, 93,209,100,120, 76,213, 16, 83,  4,107,140, 52,
    58, 55,  3,244, 97,197,238,227,118, 49, 79,230,223,165,153, 59])

Picaro = SBox([
    0x08,0x0c,0x03,0x06,0x06,0x04,0x05,0x06,0x05,0x04,0x0c,0x0c,0x04,0x03,0x05,0x03,
    0x0a,0x1f,0x29,0x3b,0x4b,0x55,0x62,0x7b,0x82,0x95,0xaf,0xbf,0xc5,0xd9,0xe2,0xf9,
    0x01,0x2d,0x45,0x6a,0x8a,0xac,0xcf,0xea,0x9f,0xbc,0xdd,0xfd,0x1c,0x35,0x5f,0x75,
    0x0f,0x34,0x61,0x52,0xc2,0xfb,0xa3,0x92,0x13,0x2b,0x74,0x44,0xdb,0xe1,0xb3,0x81,
    0x0f,0x44,0x81,0xc2,0x92,0xdb,0x13,0x52,0xb3,0xfb,0x34,0x74,0x2b,0x61,0xa3,0xe1,
    0x0e,0x59,0xa4,0xf8,0xd8,0x87,0x7c,0x28,0x3c,0x67,0x99,0xc9,0xe7,0xb4,0x4c,0x14,
    0x02,0x63,0xca,0xad,0x1d,0x71,0xd7,0xbd,0x27,0x41,0xe3,0x83,0x31,0x5a,0xf7,0x9a,
    0x0f,0x74,0xe1,0x92,0x52,0x2b,0xb3,0xc2,0xa3,0xdb,0x44,0x34,0xfb,0x81,0x13,0x61,
    0x02,0x83,0x9a,0x1d,0xbd,0x31,0x27,0xad,0xf7,0x71,0x63,0xe3,0x41,0xca,0xd7,0x5a,
    0x0e,0x99,0xb4,0x28,0xf8,0x67,0x4c,0xd8,0x7c,0xe7,0xc9,0x59,0x87,0x14,0x3c,0xa4,
    0x0a,0xaf,0xd9,0x7b,0x3b,0x95,0xe2,0x4b,0x62,0xc5,0xbf,0x1f,0x55,0xf9,0x82,0x29,
    0x0a,0xbf,0xf9,0x4b,0x7b,0xc5,0x82,0x3b,0xe2,0x55,0x1f,0xaf,0x95,0x29,0x62,0xd9,
    0x0e,0xc9,0x14,0xd8,0x28,0xe7,0x3c,0xf8,0x4c,0x87,0x59,0x99,0x67,0xa4,0x7c,0xb4,
    0x01,0xdd,0x35,0xea,0x6a,0xbc,0x5f,0x8a,0xcf,0x1c,0xfd,0x2d,0xac,0x75,0x9f,0x45,
    0x02,0xe3,0x5a,0xbd,0xad,0x41,0xf7,0x1d,0xd7,0x31,0x83,0x63,0x71,0x9a,0x27,0xca,
    0x01,0xfd,0x75,0x8a,0xea,0x1c,0x9f,0x6a,0x5f,0xac,0x2d,0xdd,0xbc,0x45,0xcf,0x35])

Safer = SBox([
    1,  45, 226, 147, 190,  69,  21, 174, 120,   3, 135, 164, 184,  56, 207,  63,
    8, 103,   9, 148, 235,  38, 168, 107, 189,  24,  52,  27, 187, 191, 114, 247,
    64,  53,  72, 156,  81,  47,  59,  85, 227, 192, 159, 216, 211, 243, 141, 177,
    255, 167,  62, 220, 134, 119, 215, 166,  17, 251, 244, 186, 146, 145, 100, 131,
    241,  51, 239, 218,  44, 181, 178,  43, 136, 209, 153, 203, 140, 132,  29,  20,
    129, 151, 113, 202,  95, 163, 139,  87,  60, 130, 196,  82,  92,  28, 232, 160,
    4, 180, 133,  74, 246,  19,  84, 182, 223,  12,  26, 142, 222, 224,  57, 252,
    32, 155,  36,  78, 169, 152, 158, 171, 242,  96, 208, 108, 234, 250, 199, 217,
    0, 212,  31, 110,  67, 188, 236,  83, 137, 254, 122,  93,  73, 201,  50, 194,
    249, 154, 248, 109,  22, 219,  89, 150,  68, 233, 205, 230,  70,  66, 143,  10,
    193, 204, 185, 101, 176, 210, 198, 172,  30,  65,  98,  41,  46,  14, 116,  80,
    2,  90, 195,  37, 123, 138,  42,  91, 240,   6,  13,  71, 111, 112, 157, 126,
    16, 206,  18,  39, 213,  76,  79, 214, 121,  48, 104,  54, 117, 125, 228, 237,
    128, 106, 144,  55, 162,  94, 118, 170, 197, 127,  61, 175, 165, 229,  25,  97,
    253,  77, 124, 183,  11, 238, 173,  75,  34, 245, 231, 115,  35,  33, 200,   5,
    225, 102, 221, 179,  88, 105,  99,  86,  15, 161,  49, 149,  23,   7,  58,  40])

Scream = SBox([
    0x20,0x8D,0xB2,0xDA,0x33,0x35,0xA6,0xFF,0x7A,0x52,0x6A,0xC6,0xA4,0xA8,0x51,0x23,
    0xA2,0x96,0x30,0xAB,0xC8,0x17,0x14,0x9E,0xE8,0xF3,0xF8,0xDD,0x85,0xE2,0x4B,0xD8,
    0x6C,0x01,0x0E,0x3D,0xB6,0x39,0x4A,0x83,0x6F,0xAA,0x86,0x6E,0x68,0x40,0x98,0x5F,
    0x37,0x13,0x05,0x87,0x04,0x82,0x31,0x89,0x24,0x38,0x9D,0x54,0x22,0x7B,0x63,0xBD,
    0x75,0x2C,0x47,0xE9,0xC2,0x60,0x43,0xAC,0x57,0xA1,0x1F,0x27,0xE7,0xAD,0x5C,0xD2,
    0x0F,0x77,0xFD,0x08,0x79,0x3A,0x49,0x5D,0xED,0x90,0x65,0x7C,0x56,0x4F,0x2E,0x69,
    0xCD,0x44,0x3F,0x62,0x5B,0x88,0x6B,0xC4,0x5E,0x2D,0x67,0x0B,0x9F,0x21,0x29,0x2A,
    0xD6,0x7E,0x74,0xE0,0x41,0x73,0x50,0x76,0x55,0x97,0x3C,0x09,0x7D,0x5A,0x92,0x70,
    0x84,0xB9,0x26,0x34,0x1D,0x81,0x32,0x2B,0x36,0x64,0xAE,0xC0,0x00,0xEE,0x8F,0xA7,
    0xBE,0x58,0xDC,0x7F,0xEC,0x9B,0x78,0x10,0xCC,0x2F,0x94,0xF1,0x3B,0x9C,0x6D,0x16,
    0x48,0xB5,0xCA,0x11,0xFA,0x0D,0x8E,0x07,0xB1,0x0C,0x12,0x28,0x4C,0x46,0xF4,0x8B,
    0xA9,0xCF,0xBB,0x03,0xA0,0xFC,0xEF,0x25,0x80,0xF6,0xB3,0xBA,0x3E,0xF7,0xD5,0x91,
    0xC3,0x8A,0xC1,0x45,0xDE,0x66,0xF5,0x0A,0xC9,0x15,0xD9,0xA3,0x61,0x99,0xB0,0xE4,
    0xD1,0xFB,0xD3,0x4E,0xBF,0xD4,0xD7,0x71,0xCB,0x1E,0xDB,0x02,0x1A,0x93,0xEA,0xC5,
    0xEB,0x72,0xF9,0x1C,0xE5,0xCE,0x4D,0xF2,0x42,0x19,0xE1,0xDF,0x59,0x95,0xB7,0x8C,
    0x9A,0xF0,0x18,0xE6,0xC7,0xAF,0xBC,0xB8,0xE3,0x1B,0xD0,0xA5,0x53,0xB4,0x06,0xFE])

# Source: https://tools.ietf.org/html/rfc4269
SEED_S0 = SBox([
    0xA9,0x85,0xD6,0xD3,0x54,0x1D,0xAC,0x25,0x5D,0x43,0x18,0x1E,0x51,0xFC,0xCA,0x63,
    0x28,0x44,0x20,0x9D,0xE0,0xE2,0xC8,0x17,0xA5,0x8F,0x03,0x7B,0xBB,0x13,0xD2,0xEE,
    0x70,0x8C,0x3F,0xA8,0x32,0xDD,0xF6,0x74,0xEC,0x95,0x0B,0x57,0x5C,0x5B,0xBD,0x01,
    0x24,0x1C,0x73,0x98,0x10,0xCC,0xF2,0xD9,0x2C,0xE7,0x72,0x83,0x9B,0xD1,0x86,0xC9,
    0x60,0x50,0xA3,0xEB,0x0D,0xB6,0x9E,0x4F,0xB7,0x5A,0xC6,0x78,0xA6,0x12,0xAF,0xD5,
    0x61,0xC3,0xB4,0x41,0x52,0x7D,0x8D,0x08,0x1F,0x99,0x00,0x19,0x04,0x53,0xF7,0xE1,
    0xFD,0x76,0x2F,0x27,0xB0,0x8B,0x0E,0xAB,0xA2,0x6E,0x93,0x4D,0x69,0x7C,0x09,0x0A,
    0xBF,0xEF,0xF3,0xC5,0x87,0x14,0xFE,0x64,0xDE,0x2E,0x4B,0x1A,0x06,0x21,0x6B,0x66,
    0x02,0xF5,0x92,0x8A,0x0C,0xB3,0x7E,0xD0,0x7A,0x47,0x96,0xE5,0x26,0x80,0xAD,0xDF,
    0xA1,0x30,0x37,0xAE,0x36,0x15,0x22,0x38,0xF4,0xA7,0x45,0x4C,0x81,0xE9,0x84,0x97,
    0x35,0xCB,0xCE,0x3C,0x71,0x11,0xC7,0x89,0x75,0xFB,0xDA,0xF8,0x94,0x59,0x82,0xC4,
    0xFF,0x49,0x39,0x67,0xC0,0xCF,0xD7,0xB8,0x0F,0x8E,0x42,0x23,0x91,0x6C,0xDB,0xA4,
    0x34,0xF1,0x48,0xC2,0x6F,0x3D,0x2D,0x40,0xBE,0x3E,0xBC,0xC1,0xAA,0xBA,0x4E,0x55,
    0x3B,0xDC,0x68,0x7F,0x9C,0xD8,0x4A,0x56,0x77,0xA0,0xED,0x46,0xB5,0x2B,0x65,0xFA,
    0xE3,0xB9,0xB1,0x9F,0x5E,0xF9,0xE6,0xB2,0x31,0xEA,0x6D,0x5F,0xE4,0xF0,0xCD,0x88,
    0x16,0x3A,0x58,0xD4,0x62,0x29,0x07,0x33,0xE8,0x1B,0x05,0x79,0x90,0x6A,0x2A,0x9A])

SEED_S1 = SBox([
    0x38,0xE8,0x2D,0xA6,0xCF,0xDE,0xB3,0xB8,0xAF,0x60,0x55,0xC7,0x44,0x6F,0x6B,0x5B,
    0xC3,0x62,0x33,0xB5,0x29,0xA0,0xE2,0xA7,0xD3,0x91,0x11,0x06,0x1C,0xBC,0x36,0x4B,
    0xEF,0x88,0x6C,0xA8,0x17,0xC4,0x16,0xF4,0xC2,0x45,0xE1,0xD6,0x3F,0x3D,0x8E,0x98,
    0x28,0x4E,0xF6,0x3E,0xA5,0xF9,0x0D,0xDF,0xD8,0x2B,0x66,0x7A,0x27,0x2F,0xF1,0x72,
    0x42,0xD4,0x41,0xC0,0x73,0x67,0xAC,0x8B,0xF7,0xAD,0x80,0x1F,0xCA,0x2C,0xAA,0x34,
    0xD2,0x0B,0xEE,0xE9,0x5D,0x94,0x18,0xF8,0x57,0xAE,0x08,0xC5,0x13,0xCD,0x86,0xB9,
    0xFF,0x7D,0xC1,0x31,0xF5,0x8A,0x6A,0xB1,0xD1,0x20,0xD7,0x02,0x22,0x04,0x68,0x71,
    0x07,0xDB,0x9D,0x99,0x61,0xBE,0xE6,0x59,0xDD,0x51,0x90,0xDC,0x9A,0xA3,0xAB,0xD0,
    0x81,0x0F,0x47,0x1A,0xE3,0xEC,0x8D,0xBF,0x96,0x7B,0x5C,0xA2,0xA1,0x63,0x23,0x4D,
    0xC8,0x9E,0x9C,0x3A,0x0C,0x2E,0xBA,0x6E,0x9F,0x5A,0xF2,0x92,0xF3,0x49,0x78,0xCC,
    0x15,0xFB,0x70,0x75,0x7F,0x35,0x10,0x03,0x64,0x6D,0xC6,0x74,0xD5,0xB4,0xEA,0x09,
    0x76,0x19,0xFE,0x40,0x12,0xE0,0xBD,0x05,0xFA,0x01,0xF0,0x2A,0x5E,0xA9,0x56,0x43,
    0x85,0x14,0x89,0x9B,0xB0,0xE5,0x48,0x79,0x97,0xFC,0x1E,0x82,0x21,0x8C,0x1B,0x5F,
    0x77,0x54,0xB2,0x1D,0x25,0x4F,0x00,0x46,0xED,0x58,0x52,0xEB,0x7E,0xDA,0xC9,0xFD,
    0x30,0x95,0x65,0x3C,0xB6,0xE4,0xBB,0x7C,0x0E,0x50,0x39,0x26,0x32,0x84,0x69,0x93,
    0x37,0xE7,0x24,0xA4,0xCB,0x53,0x0A,0x87,0xD9,0x4C,0x83,0x8F,0xCE,0x3B,0x4A,0xB7])

SKINNY_8 = SBox([
    0x65,0x4c,0x6a,0x42,0x4b,0x63,0x43,0x6b,0x55,0x75,0x5a,0x7a,0x53,0x73,0x5b,0x7b,
    0x35,0x8c,0x3a,0x81,0x89,0x33,0x80,0x3b,0x95,0x25,0x98,0x2a,0x90,0x23,0x99,0x2b,
    0xe5,0xcc,0xe8,0xc1,0xc9,0xe0,0xc0,0xe9,0xd5,0xf5,0xd8,0xf8,0xd0,0xf0,0xd9,0xf9,
    0xa5,0x1c,0xa8,0x12,0x1b,0xa0,0x13,0xa9,0x05,0xb5,0x0a,0xb8,0x03,0xb0,0x0b,0xb9,
    0x32,0x88,0x3c,0x85,0x8d,0x34,0x84,0x3d,0x91,0x22,0x9c,0x2c,0x94,0x24,0x9d,0x2d,
    0x62,0x4a,0x6c,0x45,0x4d,0x64,0x44,0x6d,0x52,0x72,0x5c,0x7c,0x54,0x74,0x5d,0x7d,
    0xa1,0x1a,0xac,0x15,0x1d,0xa4,0x14,0xad,0x02,0xb1,0x0c,0xbc,0x04,0xb4,0x0d,0xbd,
    0xe1,0xc8,0xec,0xc5,0xcd,0xe4,0xc4,0xed,0xd1,0xf1,0xdc,0xfc,0xd4,0xf4,0xdd,0xfd,
    0x36,0x8e,0x38,0x82,0x8b,0x30,0x83,0x39,0x96,0x26,0x9a,0x28,0x93,0x20,0x9b,0x29,
    0x66,0x4e,0x68,0x41,0x49,0x60,0x40,0x69,0x56,0x76,0x58,0x78,0x50,0x70,0x59,0x79,
    0xa6,0x1e,0xaa,0x11,0x19,0xa3,0x10,0xab,0x06,0xb6,0x08,0xba,0x00,0xb3,0x09,0xbb,
    0xe6,0xce,0xea,0xc2,0xcb,0xe3,0xc3,0xeb,0xd6,0xf6,0xda,0xfa,0xd3,0xf3,0xdb,0xfb,
    0x31,0x8a,0x3e,0x86,0x8f,0x37,0x87,0x3f,0x92,0x21,0x9e,0x2e,0x97,0x27,0x9f,0x2f,
    0x61,0x48,0x6e,0x46,0x4f,0x67,0x47,0x6f,0x51,0x71,0x5e,0x7e,0x57,0x77,0x5f,0x7f,
    0xa2,0x18,0xae,0x16,0x1f,0xa7,0x17,0xaf,0x01,0xb2,0x0e,0xbe,0x07,0xb7,0x0f,0xbf,
    0xe2,0xca,0xee,0xc6,0xcf,0xe7,0xc7,0xef,0xd2,0xf2,0xde,0xfe,0xd7,0xf7,0xdf,0xff])

Skipjack = SBox([
    0xa3,0xd7,0x09,0x83,0xf8,0x48,0xf6,0xf4,0xb3,0x21,0x15,0x78,0x99,0xb1,0xaf,0xf9,
    0xe7,0x2d,0x4d,0x8a,0xce,0x4c,0xca,0x2e,0x52,0x95,0xd9,0x1e,0x4e,0x38,0x44,0x28,
    0x0a,0xdf,0x02,0xa0,0x17,0xf1,0x60,0x68,0x12,0xb7,0x7a,0xc3,0xe9,0xfa,0x3d,0x53,
    0x96,0x84,0x6b,0xba,0xf2,0x63,0x9a,0x19,0x7c,0xae,0xe5,0xf5,0xf7,0x16,0x6a,0xa2,
    0x39,0xb6,0x7b,0x0f,0xc1,0x93,0x81,0x1b,0xee,0xb4,0x1a,0xea,0xd0,0x91,0x2f,0xb8,
    0x55,0xb9,0xda,0x85,0x3f,0x41,0xbf,0xe0,0x5a,0x58,0x80,0x5f,0x66,0x0b,0xd8,0x90,
    0x35,0xd5,0xc0,0xa7,0x33,0x06,0x65,0x69,0x45,0x00,0x94,0x56,0x6d,0x98,0x9b,0x76,
    0x97,0xfc,0xb2,0xc2,0xb0,0xfe,0xdb,0x20,0xe1,0xeb,0xd6,0xe4,0xdd,0x47,0x4a,0x1d,
    0x42,0xed,0x9e,0x6e,0x49,0x3c,0xcd,0x43,0x27,0xd2,0x07,0xd4,0xde,0xc7,0x67,0x18,
    0x89,0xcb,0x30,0x1f,0x8d,0xc6,0x8f,0xaa,0xc8,0x74,0xdc,0xc9,0x5d,0x5c,0x31,0xa4,
    0x70,0x88,0x61,0x2c,0x9f,0x0d,0x2b,0x87,0x50,0x82,0x54,0x64,0x26,0x7d,0x03,0x40,
    0x34,0x4b,0x1c,0x73,0xd1,0xc4,0xfd,0x3b,0xcc,0xfb,0x7f,0xab,0xe6,0x3e,0x5b,0xa5,
    0xad,0x04,0x23,0x9c,0x14,0x51,0x22,0xf0,0x29,0x79,0x71,0x7e,0xff,0x8c,0x0e,0xe2,
    0x0c,0xef,0xbc,0x72,0x75,0x6f,0x37,0xa1,0xec,0xd3,0x8e,0x62,0x8b,0x86,0x10,0xe8,
    0x08,0x77,0x11,0xbe,0x92,0x4f,0x24,0xc5,0x32,0x36,0x9d,0xcf,0xf3,0xa6,0xbb,0xac,
    0x5e,0x6c,0xa9,0x13,0x57,0x25,0xb5,0xe3,0xbd,0xa8,0x3a,0x01,0x05,0x59,0x2a,0x46])

# source: www.gsma.com/aboutus/wp-content/uploads/2014/12/snow3gspec.doc
SNOW_3G_sq = SBox([
    0x25,0x24,0x73,0x67,0xD7,0xAE,0x5C,0x30,0xA4,0xEE,0x6E,0xCB,0x7D,0xB5,0x82,0xDB,
    0xE4,0x8E,0x48,0x49,0x4F,0x5D,0x6A,0x78,0x70,0x88,0xE8,0x5F,0x5E,0x84,0x65,0xE2,
    0xD8,0xE9,0xCC,0xED,0x40,0x2F,0x11,0x28,0x57,0xD2,0xAC,0xE3,0x4A,0x15,0x1B,0xB9,
    0xB2,0x80,0x85,0xA6,0x2E,0x02,0x47,0x29,0x07,0x4B,0x0E,0xC1,0x51,0xAA,0x89,0xD4,
    0xCA,0x01,0x46,0xB3,0xEF,0xDD,0x44,0x7B,0xC2,0x7F,0xBE,0xC3,0x9F,0x20,0x4C,0x64,
    0x83,0xA2,0x68,0x42,0x13,0xB4,0x41,0xCD,0xBA,0xC6,0xBB,0x6D,0x4D,0x71,0x21,0xF4,
    0x8D,0xB0,0xE5,0x93,0xFE,0x8F,0xE6,0xCF,0x43,0x45,0x31,0x22,0x37,0x36,0x96,0xFA,
    0xBC,0x0F,0x08,0x52,0x1D,0x55,0x1A,0xC5,0x4E,0x23,0x69,0x7A,0x92,0xFF,0x5B,0x5A,
    0xEB,0x9A,0x1C,0xA9,0xD1,0x7E,0x0D,0xFC,0x50,0x8A,0xB6,0x62,0xF5,0x0A,0xF8,0xDC,
    0x03,0x3C,0x0C,0x39,0xF1,0xB8,0xF3,0x3D,0xF2,0xD5,0x97,0x66,0x81,0x32,0xA0,0x00,
    0x06,0xCE,0xF6,0xEA,0xB7,0x17,0xF7,0x8C,0x79,0xD6,0xA7,0xBF,0x8B,0x3F,0x1F,0x53,
    0x63,0x75,0x35,0x2C,0x60,0xFD,0x27,0xD3,0x94,0xA5,0x7C,0xA1,0x05,0x58,0x2D,0xBD,
    0xD9,0xC7,0xAF,0x6B,0x54,0x0B,0xE0,0x38,0x04,0xC8,0x9D,0xE7,0x14,0xB1,0x87,0x9C,
    0xDF,0x6F,0xF9,0xDA,0x2A,0xC4,0x59,0x16,0x74,0x91,0xAB,0x26,0x61,0x76,0x34,0x2B,
    0xAD,0x99,0xFB,0x72,0xEC,0x33,0x12,0xDE,0x98,0x3B,0xC0,0x9B,0x3E,0x18,0x10,0x3A,
    0x56,0xE1,0x77,0xC9,0x1E,0x9E,0x95,0xA3,0x90,0x19,0xA8,0x6C,0x09,0xD0,0xF0,0x86])

SMS4 = SBox([
    0xd6,0x90,0xe9,0xfe,0xcc,0xe1,0x3d,0xb7,0x16,0xb6,0x14,0xc2,0x28,0xfb,0x2c,0x05,
    0x2b,0x67,0x9a,0x76,0x2a,0xbe,0x04,0xc3,0xaa,0x44,0x13,0x26,0x49,0x86,0x06,0x99,
    0x9c,0x42,0x50,0xf4,0x91,0xef,0x98,0x7a,0x33,0x54,0x0b,0x43,0xed,0xcf,0xac,0x62,
    0xe4,0xb3,0x1c,0xa9,0xc9,0x08,0xe8,0x95,0x80,0xdf,0x94,0xfa,0x75,0x8f,0x3f,0xa6,
    0x47,0x07,0xa7,0xfc,0xf3,0x73,0x17,0xba,0x83,0x59,0x3c,0x19,0xe6,0x85,0x4f,0xa8,
    0x68,0x6b,0x81,0xb2,0x71,0x64,0xda,0x8b,0xf8,0xeb,0x0f,0x4b,0x70,0x56,0x9d,0x35,
    0x1e,0x24,0x0e,0x5e,0x63,0x58,0xd1,0xa2,0x25,0x22,0x7c,0x3b,0x01,0x21,0x78,0x87,
    0xd4,0x00,0x46,0x57,0x9f,0xd3,0x27,0x52,0x4c,0x36,0x02,0xe7,0xa0,0xc4,0xc8,0x9e,
    0xea,0xbf,0x8a,0xd2,0x40,0xc7,0x38,0xb5,0xa3,0xf7,0xf2,0xce,0xf9,0x61,0x15,0xa1,
    0xe0,0xae,0x5d,0xa4,0x9b,0x34,0x1a,0x55,0xad,0x93,0x32,0x30,0xf5,0x8c,0xb1,0xe3,
    0x1d,0xf6,0xe2,0x2e,0x82,0x66,0xca,0x60,0xc0,0x29,0x23,0xab,0x0d,0x53,0x4e,0x6f,
    0xd5,0xdb,0x37,0x45,0xde,0xfd,0x8e,0x2f,0x03,0xff,0x6a,0x72,0x6d,0x6c,0x5b,0x51,
    0x8d,0x1b,0xaf,0x92,0xbb,0xdd,0xbc,0x7f,0x11,0xd9,0x5c,0x41,0x1f,0x10,0x5a,0xd8,
    0x0a,0xc1,0x31,0x88,0xa5,0xcd,0x7b,0xbd,0x2d,0x74,0xd0,0x12,0xb8,0xe5,0xb4,0xb0,
    0x89,0x69,0x97,0x4a,0x0c,0x96,0x77,0x7e,0x65,0xb9,0xf1,0x09,0xc5,0x6e,0xc6,0x84,
    0x18,0xf0,0x7d,0xec,0x3a,0xdc,0x4d,0x20,0x79,0xee,0x5f,0x3e,0xd7,0xcb,0x39,0x48])

# source: https://www.iacr.org/archive/fse2003/28870306/28870306.pdf
# structure: random, obtained from RC4
Turing = SBox([
    0x61,0x51,0xeb,0x19,0xb9,0x5d,0x60,0x38,0x7c,0xb2,0x06,0x12,0xc4,0x5b,0x16,0x3b,
    0x2b,0x18,0x83,0xb0,0x7f,0x75,0xfa,0xa0,0xe9,0xdd,0x6d,0x7a,0x6b,0x68,0x2d,0x49,
    0xb5,0x1c,0x90,0xf7,0xed,0x9f,0xe8,0xce,0xae,0x77,0xc2,0x13,0xfd,0xcd,0x3e,0xcf,
    0x37,0x6a,0xd4,0xdb,0x8e,0x65,0x1f,0x1a,0x87,0xcb,0x40,0x15,0x88,0x0d,0x35,0xb3,
    0x11,0x0f,0xd0,0x30,0x48,0xf9,0xa8,0xac,0x85,0x27,0x0e,0x8a,0xe0,0x50,0x64,0xa7,
    0xcc,0xe4,0xf1,0x98,0xff,0xa1,0x04,0xda,0xd5,0xbc,0x1b,0xbb,0xd1,0xfe,0x31,0xca,
    0xba,0xd9,0x2e,0xf3,0x1d,0x47,0x4a,0x3d,0x71,0x4c,0xab,0x7d,0x8d,0xc7,0x59,0xb8,
    0xc1,0x96,0x1e,0xfc,0x44,0xc8,0x7b,0xdc,0x5c,0x78,0x2a,0x9d,0xa5,0xf0,0x73,0x22,
    0x89,0x05,0xf4,0x07,0x21,0x52,0xa6,0x28,0x9a,0x92,0x69,0x8f,0xc5,0xc3,0xf5,0xe1,
    0xde,0xec,0x09,0xf2,0xd3,0xaf,0x34,0x23,0xaa,0xdf,0x7e,0x82,0x29,0xc0,0x24,0x14,
    0x03,0x32,0x4e,0x39,0x6f,0xc6,0xb1,0x9b,0xea,0x72,0x79,0x41,0xd8,0x26,0x6c,0x5e,
    0x2c,0xb4,0xa2,0x53,0x57,0xe2,0x9c,0x86,0x54,0x95,0xb6,0x80,0x8c,0x36,0x67,0xbd,
    0x08,0x93,0x2f,0x99,0x5a,0xf8,0x3a,0xd7,0x56,0x84,0xd2,0x01,0xf6,0x66,0x4d,0x55,
    0x8b,0x0c,0x0b,0x46,0xb7,0x3c,0x45,0x91,0xa4,0xe3,0x70,0xd6,0xfb,0xe6,0x10,0xa9,
    0xc9,0x00,0x9e,0xe7,0x4f,0x76,0x25,0x3f,0x5f,0xa3,0x33,0x20,0x02,0xef,0x62,0x74,
    0xee,0x17,0x81,0x42,0x58,0x0a,0x4b,0x63,0xe5,0xbe,0x6e,0xad,0xbf,0x43,0x94,0x97])

# source: https://www.schneier.com/cryptography/paperfiles/paper-twofish-paper.pdf
# structure: ASAS, the 4 bit S-Boxes q0 ti for i=0..3 are provided below
Twofish_p0 = SBox([
    0xA9,0x67,0xB3,0xE8,0x04,0xFD,0xA3,0x76,0x9A,0x92,0x80,0x78,0xE4,0xDD,0xD1,0x38,
    0x0D,0xC6,0x35,0x98,0x18,0xF7,0xEC,0x6C,0x43,0x75,0x37,0x26,0xFA,0x13,0x94,0x48,
    0xF2,0xD0,0x8B,0x30,0x84,0x54,0xDF,0x23,0x19,0x5B,0x3D,0x59,0xF3,0xAE,0xA2,0x82,
    0x63,0x01,0x83,0x2E,0xD9,0x51,0x9B,0x7C,0xA6,0xEB,0xA5,0xBE,0x16,0x0C,0xE3,0x61,
    0xC0,0x8C,0x3A,0xF5,0x73,0x2C,0x25,0x0B,0xBB,0x4E,0x89,0x6B,0x53,0x6A,0xB4,0xF1,
    0xE1,0xE6,0xBD,0x45,0xE2,0xF4,0xB6,0x66,0xCC,0x95,0x03,0x56,0xD4,0x1C,0x1E,0xD7,
    0xFB,0xC3,0x8E,0xB5,0xE9,0xCF,0xBF,0xBA,0xEA,0x77,0x39,0xAF,0x33,0xC9,0x62,0x71,
    0x81,0x79,0x09,0xAD,0x24,0xCD,0xF9,0xD8,0xE5,0xC5,0xB9,0x4D,0x44,0x08,0x86,0xE7,
    0xA1,0x1D,0xAA,0xED,0x06,0x70,0xB2,0xD2,0x41,0x7B,0xA0,0x11,0x31,0xC2,0x27,0x90,
    0x20,0xF6,0x60,0xFF,0x96,0x5C,0xB1,0xAB,0x9E,0x9C,0x52,0x1B,0x5F,0x93,0x0A,0xEF,
    0x91,0x85,0x49,0xEE,0x2D,0x4F,0x8F,0x3B,0x47,0x87,0x6D,0x46,0xD6,0x3E,0x69,0x64,
    0x2A,0xCE,0xCB,0x2F,0xFC,0x97,0x05,0x7A,0xAC,0x7F,0xD5,0x1A,0x4B,0x0E,0xA7,0x5A,
    0x28,0x14,0x3F,0x29,0x88,0x3C,0x4C,0x02,0xB8,0xDA,0xB0,0x17,0x55,0x1F,0x8A,0x7D,
    0x57,0xC7,0x8D,0x74,0xB7,0xC4,0x9F,0x72,0x7E,0x15,0x22,0x12,0x58,0x07,0x99,0x34,
    0x6E,0x50,0xDE,0x68,0x65,0xBC,0xDB,0xF8,0xC8,0xA8,0x2B,0x40,0xDC,0xFE,0x32,0xA4,
    0xCA,0x10,0x21,0xF0,0xD3,0x5D,0x0F,0x00,0x6F,0x9D,0x36,0x42,0x4A,0x5E,0xC1,0xE0])

# source: https://www.schneier.com/cryptography/paperfiles/paper-twofish-paper.pdf
# structure: ASAS, the 4 bit S-Boxes q1 ti for i=0..3 are provided below
Twofish_p1 = SBox([
    0x75,0xF3,0xC6,0xF4,0xDB,0x7B,0xFB,0xC8,0x4A,0xD3,0xE6,0x6B,0x45,0x7D,0xE8,0x4B,
    0xD6,0x32,0xD8,0xFD,0x37,0x71,0xF1,0xE1,0x30,0x0F,0xF8,0x1B,0x87,0xFA,0x06,0x3F,
    0x5E,0xBA,0xAE,0x5B,0x8A,0x00,0xBC,0x9D,0x6D,0xC1,0xB1,0x0E,0x80,0x5D,0xD2,0xD5,
    0xA0,0x84,0x07,0x14,0xB5,0x90,0x2C,0xA3,0xB2,0x73,0x4C,0x54,0x92,0x74,0x36,0x51,
    0x38,0xB0,0xBD,0x5A,0xFC,0x60,0x62,0x96,0x6C,0x42,0xF7,0x10,0x7C,0x28,0x27,0x8C,
    0x13,0x95,0x9C,0xC7,0x24,0x46,0x3B,0x70,0xCA,0xE3,0x85,0xCB,0x11,0xD0,0x93,0xB8,
    0xA6,0x83,0x20,0xFF,0x9F,0x77,0xC3,0xCC,0x03,0x6F,0x08,0xBF,0x40,0xE7,0x2B,0xE2,
    0x79,0x0C,0xAA,0x82,0x41,0x3A,0xEA,0xB9,0xE4,0x9A,0xA4,0x97,0x7E,0xDA,0x7A,0x17,
    0x66,0x94,0xA1,0x1D,0x3D,0xF0,0xDE,0xB3,0x0B,0x72,0xA7,0x1C,0xEF,0xD1,0x53,0x3E,
    0x8F,0x33,0x26,0x5F,0xEC,0x76,0x2A,0x49,0x81,0x88,0xEE,0x21,0xC4,0x1A,0xEB,0xD9,
    0xC5,0x39,0x99,0xCD,0xAD,0x31,0x8B,0x01,0x18,0x23,0xDD,0x1F,0x4E,0x2D,0xF9,0x48,
    0x4F,0xF2,0x65,0x8E,0x78,0x5C,0x58,0x19,0x8D,0xE5,0x98,0x57,0x67,0x7F,0x05,0x64,
    0xAF,0x63,0xB6,0xFE,0xF5,0xB7,0x3C,0xA5,0xCE,0xE9,0x68,0x44,0xE0,0x4D,0x43,0x69,
    0x29,0x2E,0xAC,0x15,0x59,0xA8,0x0A,0x9E,0x6E,0x47,0xDF,0x34,0x35,0x6A,0xCF,0xDC,
    0x22,0xC9,0xC0,0x9B,0x89,0xD4,0xED,0xAB,0x12,0xA2,0x0D,0x52,0xBB,0x02,0x2F,0xA9,
    0xD7,0x61,0x1E,0xB4,0x50,0x04,0xF6,0xC2,0x16,0x25,0x86,0x56,0x55,0x09,0xBE,0x91])

Whirlpool = SBox([
    0x18,0x23,0xc6,0xE8,0x87,0xB8,0x01,0x4F,0x36,0xA6,0xd2,0xF5,0x79,0x6F,0x91,0x52,
    0x60,0xBc,0x9B,0x8E,0xA3,0x0c,0x7B,0x35,0x1d,0xE0,0xd7,0xc2,0x2E,0x4B,0xFE,0x57,
    0x15,0x77,0x37,0xE5,0x9F,0xF0,0x4A,0xdA,0x58,0xc9,0x29,0x0A,0xB1,0xA0,0x6B,0x85,
    0xBd,0x5d,0x10,0xF4,0xcB,0x3E,0x05,0x67,0xE4,0x27,0x41,0x8B,0xA7,0x7d,0x95,0xd8,
    0xFB,0xEE,0x7c,0x66,0xdd,0x17,0x47,0x9E,0xcA,0x2d,0xBF,0x07,0xAd,0x5A,0x83,0x33,
    0x63,0x02,0xAA,0x71,0xc8,0x19,0x49,0xd9,0xF2,0xE3,0x5B,0x88,0x9A,0x26,0x32,0xB0,
    0xE9,0x0F,0xd5,0x80,0xBE,0xcd,0x34,0x48,0xFF,0x7A,0x90,0x5F,0x20,0x68,0x1A,0xAE,
    0xB4,0x54,0x93,0x22,0x64,0xF1,0x73,0x12,0x40,0x08,0xc3,0xEc,0xdB,0xA1,0x8d,0x3d,
    0x97,0x00,0xcF,0x2B,0x76,0x82,0xd6,0x1B,0xB5,0xAF,0x6A,0x50,0x45,0xF3,0x30,0xEF,
    0x3F,0x55,0xA2,0xEA,0x65,0xBA,0x2F,0xc0,0xdE,0x1c,0xFd,0x4d,0x92,0x75,0x06,0x8A,
    0xB2,0xE6,0x0E,0x1F,0x62,0xd4,0xA8,0x96,0xF9,0xc5,0x25,0x59,0x84,0x72,0x39,0x4c,
    0x5E,0x78,0x38,0x8c,0xd1,0xA5,0xE2,0x61,0xB3,0x21,0x9c,0x1E,0x43,0xc7,0xFc,0x04,
    0x51,0x99,0x6d,0x0d,0xFA,0xdF,0x7E,0x24,0x3B,0xAB,0xcE,0x11,0x8F,0x4E,0xB7,0xEB,
    0x3c,0x81,0x94,0xF7,0xB9,0x13,0x2c,0xd3,0xE7,0x6E,0xc4,0x03,0x56,0x44,0x7F,0xA9,
    0x2A,0xBB,0xc1,0x53,0xdc,0x0B,0x9d,0x6c,0x31,0x74,0xF6,0x46,0xAc,0x89,0x14,0xE1,
    0x16,0x3A,0x69,0x09,0x70,0xB6,0xd0,0xEd,0xcc,0x42,0x98,0xA4,0x28,0x5c,0xF8,0x86])

Zorro = SBox([
    0xB2,0xE5,0x5E,0xFD,0x5F,0xC5,0x50,0xBC,0xDC,0x4A,0xFA,0x88,0x28,0xD8,0xE0,0xD1,
    0xB5,0xD0,0x3C,0xB0,0x99,0xC1,0xE8,0xE2,0x13,0x59,0xA7,0xFB,0x71,0x34,0x31,0xF1,
    0x9F,0x3A,0xCE,0x6E,0xA8,0xA4,0xB4,0x7E,0x1F,0xB7,0x51,0x1D,0x38,0x9D,0x46,0x69,
    0x53,0xE,0x42,0x1B,0xF,0x11,0x68,0xCA,0xAA,0x6,0xF0,0xBD,0x26,0x6F,0x0,0xD9,
    0x62,0xF3,0x15,0x60,0xF2,0x3D,0x7F,0x35,0x63,0x2D,0x67,0x93,0x1C,0x91,0xF9,0x9C,
    0x66,0x2A,0x81,0x20,0x95,0xF8,0xE3,0x4D,0x5A,0x6D,0x24,0x7B,0xB9,0xEF,0xDF,0xDA,
    0x58,0xA9,0x92,0x76,0x2E,0xB3,0x39,0xC,0x29,0xCD,0x43,0xFE,0xAB,0xF5,0x94,0x23,
    0x16,0x80,0xC0,0x12,0x4C,0xE9,0x48,0x19,0x8,0xAE,0x41,0x70,0x84,0x14,0xA2,0xD5,
    0xB8,0x33,0x65,0xBA,0xED,0x17,0xCF,0x96,0x1E,0x3B,0xB,0xC2,0xC8,0xB6,0xBB,0x8B,
    0xA1,0x54,0x75,0xC4,0x10,0x5D,0xD6,0x25,0x97,0xE6,0xFC,0x49,0xF7,0x52,0x18,0x86,
    0x8D,0xCB,0xE1,0xBF,0xD7,0x8E,0x37,0xBE,0x82,0xCC,0x64,0x90,0x7C,0x32,0x8F,0x4B,
    0xAC,0x1A,0xEA,0xD3,0xF4,0x6B,0x2C,0xFF,0x55,0xA,0x45,0x9,0x89,0x1,0x30,0x2B,
    0xD2,0x77,0x87,0x72,0xEB,0x36,0xDE,0x9E,0x8C,0xDB,0x6C,0x9B,0x5,0x2,0x4E,0xAF,
    0x4,0xAD,0x74,0xC3,0xEE,0xA6,0xF6,0xC7,0x7D,0x40,0xD4,0xD,0x3E,0x5B,0xEC,0x78,
    0xA0,0xB1,0x44,0x73,0x47,0x5C,0x98,0x21,0x22,0x61,0x3F,0xC6,0x7A,0x56,0xDD,0xE7,
    0x85,0xC9,0x8A,0x57,0x27,0x7,0x9A,0x3,0xA3,0x83,0xE4,0x6A,0xA5,0x2F,0x79,0x4F])

ZUC_S0 = SBox([
    0x3e,0x72,0x5b,0x47,0xca,0xe0,0x00,0x33,0x04,0xd1,0x54,0x98,0x09,0xb9,0x6d,0xcb,
    0x7b,0x1b,0xf9,0x32,0xaf,0x9d,0x6a,0xa5,0xb8,0x2d,0xfc,0x1d,0x08,0x53,0x03,0x90,
    0x4d,0x4e,0x84,0x99,0xe4,0xce,0xd9,0x91,0xdd,0xb6,0x85,0x48,0x8b,0x29,0x6e,0xac,
    0xcd,0xc1,0xf8,0x1e,0x73,0x43,0x69,0xc6,0xb5,0xbd,0xfd,0x39,0x63,0x20,0xd4,0x38,
    0x76,0x7d,0xb2,0xa7,0xcf,0xed,0x57,0xc5,0xf3,0x2c,0xbb,0x14,0x21,0x06,0x55,0x9b,
    0xe3,0xef,0x5e,0x31,0x4f,0x7f,0x5a,0xa4,0x0d,0x82,0x51,0x49,0x5f,0xba,0x58,0x1c,
    0x4a,0x16,0xd5,0x17,0xa8,0x92,0x24,0x1f,0x8c,0xff,0xd8,0xae,0x2e,0x01,0xd3,0xad,
    0x3b,0x4b,0xda,0x46,0xeb,0xc9,0xde,0x9a,0x8f,0x87,0xd7,0x3a,0x80,0x6f,0x2f,0xc8,
    0xb1,0xb4,0x37,0xf7,0x0a,0x22,0x13,0x28,0x7c,0xcc,0x3c,0x89,0xc7,0xc3,0x96,0x56,
    0x07,0xbf,0x7e,0xf0,0x0b,0x2b,0x97,0x52,0x35,0x41,0x79,0x61,0xa6,0x4c,0x10,0xfe,
    0xbc,0x26,0x95,0x88,0x8a,0xb0,0xa3,0xfb,0xc0,0x18,0x94,0xf2,0xe1,0xe5,0xe9,0x5d,
    0xd0,0xdc,0x11,0x66,0x64,0x5c,0xec,0x59,0x42,0x75,0x12,0xf5,0x74,0x9c,0xaa,0x23,
    0x0e,0x86,0xab,0xbe,0x2a,0x02,0xe7,0x67,0xe6,0x44,0xa2,0x6c,0xc2,0x93,0x9f,0xf1,
    0xf6,0xfa,0x36,0xd2,0x50,0x68,0x9e,0x62,0x71,0x15,0x3d,0xd6,0x40,0xc4,0xe2,0x0f,
    0x8e,0x83,0x77,0x6b,0x25,0x05,0x3f,0x0c,0x30,0xea,0x70,0xb7,0xa1,0xe8,0xa9,0x65,
    0x8d,0x27,0x1a,0xdb,0x81,0xb3,0xa0,0xf4,0x45,0x7a,0x19,0xdf,0xee,0x78,0x34,0x60])

ZUC_S1 = SBox([
    0x55,0xc2,0x63,0x71,0x3b,0xc8,0x47,0x86,0x9f,0x3c,0xda,0x5b,0x29,0xaa,0xfd,0x77,
    0x8c,0xc5,0x94,0x0c,0xa6,0x1a,0x13,0x00,0xe3,0xa8,0x16,0x72,0x40,0xf9,0xf8,0x42,
    0x44,0x26,0x68,0x96,0x81,0xd9,0x45,0x3e,0x10,0x76,0xc6,0xa7,0x8b,0x39,0x43,0xe1,
    0x3a,0xb5,0x56,0x2a,0xc0,0x6d,0xb3,0x05,0x22,0x66,0xbf,0xdc,0x0b,0xfa,0x62,0x48,
    0xdd,0x20,0x11,0x06,0x36,0xc9,0xc1,0xcf,0xf6,0x27,0x52,0xbb,0x69,0xf5,0xd4,0x87,
    0x7f,0x84,0x4c,0xd2,0x9c,0x57,0xa4,0xbc,0x4f,0x9a,0xdf,0xfe,0xd6,0x8d,0x7a,0xeb,
    0x2b,0x53,0xd8,0x5c,0xa1,0x14,0x17,0xfb,0x23,0xd5,0x7d,0x30,0x67,0x73,0x08,0x09,
    0xee,0xb7,0x70,0x3f,0x61,0xb2,0x19,0x8e,0x4e,0xe5,0x4b,0x93,0x8f,0x5d,0xdb,0xa9,
    0xad,0xf1,0xae,0x2e,0xcb,0x0d,0xfc,0xf4,0x2d,0x46,0x6e,0x1d,0x97,0xe8,0xd1,0xe9,
    0x4d,0x37,0xa5,0x75,0x5e,0x83,0x9e,0xab,0x82,0x9d,0xb9,0x1c,0xe0,0xcd,0x49,0x89,
    0x01,0xb6,0xbd,0x58,0x24,0xa2,0x5f,0x38,0x78,0x99,0x15,0x90,0x50,0xb8,0x95,0xe4,
    0xd0,0x91,0xc7,0xce,0xed,0x0f,0xb4,0x6f,0xa0,0xcc,0xf0,0x02,0x4a,0x79,0xc3,0xde,
    0xa3,0xef,0xea,0x51,0xe6,0x6b,0x18,0xec,0x1b,0x2c,0x80,0xf7,0x74,0xe7,0xff,0x21,
    0x5a,0x6a,0x54,0x1e,0x41,0x31,0x92,0x35,0xc4,0x33,0x07,0x0a,0xba,0x7e,0x0e,0x34,
    0x88,0xb1,0x98,0x7c,0xf3,0x3d,0x60,0x6c,0x7b,0xca,0xd3,0x1f,0x32,0x65,0x04,0x28,
    0x64,0xbe,0x85,0x9b,0x2f,0x59,0x8a,0xd7,0xb0,0x25,0xac,0xaf,0x12,0x03,0xe2,0xf2])

# Bijective S-Boxes mapping 6 bits to 6
# =====================================

Fides_6 = SBox([
    0x36,0x00,0x30,0x0d,0x0f,0x12,0x23,0x35,0x3f,0x19,0x2d,0x34,0x03,0x14,0x21,0x29,
    0x08,0x0a,0x39,0x25,0x3b,0x24,0x22,0x02,0x1a,0x32,0x3a,0x18,0x3c,0x13,0x0e,0x2a,
    0x2e,0x3d,0x05,0x31,0x1f,0x0b,0x1c,0x04,0x0c,0x1e,0x37,0x16,0x09,0x06,0x20,0x17,
    0x1b,0x27,0x15,0x11,0x10,0x1d,0x3e,0x01,0x28,0x2f,0x33,0x38,0x07,0x2b,0x26,0x2c])

APN_6 = SBox([
    0x0,0x36,0x30,0xd,0xf,0x12,0x35,0x23,0x19,0x3f,0x2d,0x34,0x3,0x14,0x29,0x21,
    0x3b,0x24,0x2,0x22,0xa,0x8,0x39,0x25,0x3c,0x13,0x2a,0xe,0x32,0x1a,0x3a,0x18,
    0x27,0x1b,0x15,0x11,0x10,0x1d,0x1,0x3e,0x2f,0x28,0x33,0x38,0x7,0x2b,0x2c,0x26,
    0x1f,0xb,0x4,0x1c,0x3d,0x2e,0x5,0x31,0x9,0x6,0x17,0x20,0x1e,0xc,0x37,0x16])

SC2000_6 = SBox([
    47,59,25,42,15,23,28,39,26,38,36,19,60,24,29,56,
    37,63,20,61,55,2,30,44,9,10,6,22,53,48,51,11,
    62,52,35,18,14,46,0,54,17,40,27,4,31,8,5,12,
    3,16,41,34,33,7,45,49,50,58,1,21,43,57,32,13])

# Bijective S-Boxes mapping 5 bits to 5
# =====================================

Ascon = SBox([
    0x04,0x0b,0x1f,0x14,0x1a,0x15,0x09,0x02,0x1b,0x05,0x08,0x12,0x1d,0x03,0x06,0x1c,
    0x1e,0x13,0x07,0x0e,0x00,0x0d,0x11,0x18,0x10,0x0c,0x01,0x19,0x16,0x0a,0x0f,0x17])

Fides_5 = SBox([
    0x01,0x00,0x19,0x1a,0x11,0x1d,0x15,0x1b,0x14,0x05,0x04,0x17,0x0e,0x12,0x02,0x1c,
    0x0f,0x08,0x06,0x03,0x0d,0x07,0x18,0x10,0x1e,0x09,0x1f,0x0a,0x16,0x0c,0x0b,0x13])

SC2000_5 = SBox([
    20,26,7,31,19,12,10,15,22,30,13,14,4,24,9,
    18,27,11,1,21,6,16,2,28,23,5,8,3,0,17,29,25])

# Bijective S-Boxes mapping 4 bits to 4
# =====================================

# DES

DES_S1_1 = SBox([14,4,13,1,2,15,11,8,3,10,6,12,5,9,0,7])
DES_S1_2 = SBox([0,15,7,4,14,2,13,1,10,6,12,11,9,5,3,8])
DES_S1_3 = SBox([4,1,14,8,13,6,2,11,15,12,9,7,3,10,5,0])
DES_S1_4 = SBox([15,12,8,2,4,9,1,7,5,11,3,14,10,0,6,13])

DES_S2_1 = SBox([15,1,8,14,6,11,3,4,9,7,2,13,12,0,5,10])
DES_S2_2 = SBox([3,13,4,7,15,2,8,14,12,0,1,10,6,9,11,5])
DES_S2_3 = SBox([0,14,7,11,10,4,13,1,5,8,12,6,9,3,2,15])
DES_S2_4 = SBox([13,8,10,1,3,15,4,2,11,6,7,12,0,5,14,9])

DES_S3_1 = SBox([10,0,9,14,6,3,15,5,1,13,12,7,11,4,2,8])
DES_S3_2 = SBox([13,7,0,9,3,4,6,10,2,8,5,14,12,11,15,1])
DES_S3_3 = SBox([13,6,4,9,8,15,3,0,11,1,2,12,5,10,14,7])
DES_S3_4 = SBox([1,10,13,0,6,9,8,7,4,15,14,3,11,5,2,12])

DES_S4_1 = SBox([7,13,14,3,0,6,9,10,1,2,8,5,11,12,4,15])
DES_S4_2 = SBox([13,8,11,5,6,15,0,3,4,7,2,12,1,10,14,9])
DES_S4_3 = SBox([10,6,9,0,12,11,7,13,15,1,3,14,5,2,8,4])
DES_S4_4 = SBox([3,15,0,6,10,1,13,8,9,4,5,11,12,7,2,14])

DES_S5_1 = SBox([2,12,4,1,7,10,11,6,8,5,3,15,13,0,14,9])
DES_S5_2 = SBox([14,11,2,12,4,7,13,1,5,0,15,10,3,9,8,6])
DES_S5_3 = SBox([4,2,1,11,10,13,7,8,15,9,12,5,6,3,0,14])
DES_S5_4 = SBox([11,8,12,7,1,14,2,13,6,15,0,9,10,4,5,3])

DES_S6_1 = SBox([12,1,10,15,9,2,6,8,0,13,3,4,14,7,5,11])
DES_S6_2 = SBox([10,15,4,2,7,12,9,5,6,1,13,14,0,11,3,8])
DES_S6_3 = SBox([9,14,15,5,2,8,12,3,7,0,4,10,1,13,11,6])
DES_S6_4 = SBox([4,3,2,12,9,5,15,10,11,14,1,7,6,0,8,13])

DES_S7_1 = SBox([4,11,2,14,15,0,8,13,3,12,9,7,5,10,6,1])
DES_S7_2 = SBox([13,0,11,7,4,9,1,10,14,3,5,12,2,15,8,6])
DES_S7_3 = SBox([1,4,11,13,12,3,7,14,10,15,6,8,0,5,9,2])
DES_S7_4 = SBox([6,11,13,8,1,4,10,7,9,5,0,15,14,2,3,12])

DES_S8_1 = SBox([13,2,8,4,6,15,11,1,10,9,3,14,5,0,12,7])
DES_S8_2 = SBox([1,15,13,8,10,3,7,4,12,5,6,11,0,14,9,2])
DES_S8_3 = SBox([7,11,4,1,9,12,14,2,0,6,10,13,15,3,5,8])
DES_S8_4 = SBox([2,1,14,7,4,10,8,13,15,12,9,0,3,5,6,11])

# source: http://www.quadibloc.com/crypto/co0401.htm
Lucifer_S0 = SBox([12,15,7,10,14,13,11,0,2,6,3,1,9,4,5,8])
Lucifer_S1 = SBox([7,2,14,9,3,11,0,4,12,13,1,10,6,15,8,5])

# First GOST block cipher and its modification.
# ref: https://en.wikipedia.org/wiki/GOST_28147-89
GOST_1 = SBox([0x4,0xa,0x9,0x2,0xd,0x8,0x0,0xe,0x6,0xb,0x1,0xc,0x7,0xf,0x5,0x3])
GOST_2 = SBox([0xe,0xb,0x4,0xc,0x6,0xd,0xf,0xa,0x2,0x3,0x8,0x1,0x0,0x7,0x5,0x9])
GOST_3 = SBox([0x5,0x8,0x1,0xd,0xa,0x3,0x4,0x2,0xe,0xf,0xc,0x7,0x6,0x0,0x9,0xb])
GOST_4 = SBox([0x7,0xd,0xa,0x1,0x0,0x8,0x9,0xf,0xe,0x4,0x6,0xc,0xb,0x2,0x5,0x3])
GOST_5 = SBox([0x6,0xc,0x7,0x1,0x5,0xf,0xd,0x8,0x4,0xa,0x9,0xe,0x0,0x3,0xb,0x2])
GOST_6 = SBox([0x4,0xb,0xa,0x0,0x7,0x2,0x1,0xd,0x3,0x6,0x8,0x5,0x9,0xc,0xf,0xe])
GOST_7 = SBox([0xd,0xb,0x4,0x1,0x3,0xf,0x5,0x9,0x0,0xa,0xe,0x7,0x6,0x8,0x2,0xc])
GOST_8 = SBox([0x1,0xf,0xd,0x0,0x5,0x7,0xa,0x4,0x9,0x2,0x3,0xe,0x6,0xb,0x8,0xc])

# ref: https://eprint.iacr.org/2015/065.pdf
GOST2_1 = SBox([0x6,0xA,0xF,0x4,0x3,0x8,0x5,0x0,0xD,0xE,0x7,0x1,0x2,0xB,0xC,0x9])
GOST2_2 = SBox([0xE,0x0,0x8,0x1,0x7,0xA,0x5,0x6,0xD,0x2,0x4,0x9,0x3,0xF,0xC,0xB])

Magma_1 = SBox([0xC,0x4,0x6,0x2,0xA,0x5,0xB,0x9,0xE,0x8,0xD,0x7,0x0,0x3,0xF,0x1])
Magma_2 = SBox([0x6,0x8,0x2,0x3,0x9,0xA,0x5,0xC,0x1,0xE,0x4,0x7,0xB,0xD,0x0,0xF])
Magma_3 = SBox([0xB,0x3,0x5,0x8,0x2,0xF,0xA,0xD,0xE,0x1,0x7,0x4,0xC,0x9,0x6,0x0])
Magma_4 = SBox([0xC,0x8,0x2,0x1,0xD,0x4,0xF,0x6,0x7,0x0,0xA,0x5,0x3,0xE,0x9,0xB])
Magma_5 = SBox([0x7,0xF,0x5,0xA,0x8,0x1,0x6,0xD,0x0,0x9,0x3,0xE,0xB,0x4,0x2,0xC])
Magma_6 = SBox([0x5,0xD,0xF,0x6,0x9,0x2,0xC,0xA,0xB,0x7,0x8,0x1,0x4,0x3,0xE,0x0])
Magma_7 = SBox([0x8,0xE,0x2,0x5,0x6,0x9,0x1,0xC,0xF,0x4,0xB,0x0,0xD,0xA,0x3,0x7])
Magma_8 = SBox([0x1,0x7,0xE,0xD,0x0,0x5,0x8,0x3,0x4,0xF,0xA,0x6,0x9,0xC,0xB,0x2])

GOST_IETF_1 = SBox([0x9,0x6,0x3,0x2,0x8,0xb,0x1,0x7,0xa,0x4,0xe,0xf,0xc,0x0,0xd,0x5])
GOST_IETF_2 = SBox([0x3,0x7,0xe,0x9,0x8,0xa,0xf,0x0,0x5,0x2,0x6,0xc,0xb,0x4,0xd,0x1])
GOST_IETF_3 = SBox([0xe,0x4,0x6,0x2,0xb,0x3,0xd,0x8,0xc,0xf,0x5,0xa,0x0,0x7,0x1,0x9])
GOST_IETF_4 = SBox([0xe,0x7,0xa,0xc,0xd,0x1,0x3,0x9,0x0,0x2,0xb,0x4,0xf,0x8,0x5,0x6])
GOST_IETF_5 = SBox([0xb,0x5,0x1,0x9,0x8,0xd,0xf,0x0,0xe,0x4,0x2,0x3,0xc,0x7,0xa,0x6])
GOST_IETF_6 = SBox([0x3,0xa,0xd,0xc,0x1,0x2,0x0,0xb,0x7,0x5,0x9,0x4,0x8,0xf,0xe,0x6])
GOST_IETF_7 = SBox([0x1,0xd,0x2,0x9,0x7,0xa,0x6,0x0,0x8,0xc,0x4,0x5,0xf,0x3,0xb,0xe])
GOST_IETF_8 = SBox([0xb,0xa,0xf,0x5,0x0,0xc,0xe,0x8,0x6,0x2,0x3,0x9,0x1,0x7,0xd,0x4])

# Hummingbird-2
Hummingbird_2_S1 = SBox([7,12,14,9,2,1,5,15,11,6,13,0,4,8,10,3])
Hummingbird_2_S2 = SBox([4,10,1,6,8,15,7,12,3,0,14,13,5,9,11,2])
Hummingbird_2_S3 = SBox([2,15,12,1,5,6,10,13,14,8,3,4,0,11,9,7])
Hummingbird_2_S4 = SBox([15,4,5,8,9,7,2,1,10,3,0,14,6,12,13,11])

# LBlock
LBlock_0 = SBox([14, 9, 15, 0, 13, 4, 10, 11, 1, 2, 8, 3, 7, 6, 12, 5])
LBlock_1 = SBox([4, 11, 14, 9, 15, 13, 0, 10, 7, 12, 5, 6, 2, 8, 1, 3])
LBlock_2 = SBox([1, 14, 7, 12, 15, 13, 0, 6, 11, 5, 9, 3, 2, 4, 8, 10])
LBlock_3 = SBox([7, 6, 8, 11, 0, 15, 3, 14, 9, 10, 12, 13, 5, 2, 4, 1])
LBlock_4 = SBox([14, 5, 15, 0, 7, 2, 12, 13, 1, 8, 4, 9, 11, 10, 6, 3])
LBlock_5 = SBox([2, 13, 11, 12, 15, 14, 0, 9, 7, 10, 6, 3, 1, 8, 4, 5])
LBlock_6 = SBox([11, 9, 4, 14, 0, 15, 10, 13, 6, 12, 5, 7, 3, 8, 1, 2])
LBlock_7 = SBox([13, 10, 15, 0, 14, 4, 9, 11, 2, 1, 8, 3, 7, 5, 12, 6])
LBlock_8 = SBox([8, 7, 14, 5, 15, 13, 0, 6, 11, 12, 9, 10, 2, 4, 1, 3])
LBlock_9 = SBox([11, 5, 15, 0, 7, 2, 9, 13, 4, 8, 1, 12, 14, 10, 3, 6])

# SERPENT
SERPENT_S0 = SBox([3,8,15,1,10,6,5,11,14,13,4,2,7,0,9,12])
SERPENT_S1 = SBox([15,12,2,7,9,0,5,10,1,11,14,8,6,13,3,4])
SERPENT_S2 = SBox([8,6,7,9,3,12,10,15,13,1,14,4,0,11,5,2])
SERPENT_S3 = SBox([0,15,11,8,12,9,6,3,13,1,2,4,10,7,5,14])
SERPENT_S4 = SBox([1,15,8,3,12,0,11,6,2,5,4,10,9,14,7,13])
SERPENT_S5 = SBox([15,5,2,11,4,10,9,12,0,3,14,8,13,6,7,1])
SERPENT_S6 = SBox([7,2,12,5,8,4,6,11,14,9,1,15,13,3,10,0])
SERPENT_S7 = SBox([1,13,15,0,14,8,2,11,7,4,12,10,9,3,5,6])

# Other Block ciphers
KLEIN        = SBox([0x7,0x4,0xA,0x9,0x1,0xF,0xB,0x0,0xC,0x3,0x2,0x6,0x8,0xE,0xD,0x5])
MIBS         = SBox([4,15,3,8,13,10,12,0,11,5,7,14,2,6,1,9])
Midori_Sb0   = SBox([0xc,0xa,0xd,0x3,0xe,0xb,0xf,0x7,0x8,0x9,0x1,0x5,0x0,0x2,0x4,0x6])
MANTIS = Midori_Sb0
Midori_Sb1   = SBox([0x1,0x0,0x5,0x3,0xe,0x2,0xf,0x7,0xd,0xa,0x9,0xb,0xc,0x8,0x4,0x6])
Noekeon      = SBox([0x7,0xA,0x2,0xC,0x4,0x8,0xF,0x0,0x5,0x9,0x1,0xE,0x3,0xD,0xB,0x6])
Piccolo      = SBox([0xe,0x4,0xb,0x2,0x3,0x8,0x0,0x9,0x1,0xa,0x7,0xf,0x6,0xc,0x5,0xd])
Panda        = SBox([0x0,0x1,0x3,0x2,0xf,0xc,0x9,0xb,0xa,0x6,0x8,0x7,0x5,0xe,0xd,0x4])
PRESENT      = SBox([0xC,0x5,0x6,0xB,0x9,0x0,0xA,0xD,0x3,0xE,0xF,0x8,0x4,0x7,0x1,0x2])
GIFT         = SBox([0x1,0xa,0x4,0xc,0x6,0xf,0x3,0x9,0x2,0xd,0xb,0x7,0x5,0x0,0x8,0xe])
Pride        = SBox([0x0,0x4,0x8,0xf,0x1,0x5,0xe,0x9,0x2,0x7,0xa,0xc,0xb,0xd,0x6,0x3])
PRINCE       = SBox([0xB,0xF,0x3,0x2,0xA,0xC,0x9,0x1,0x6,0x7,0x8,0x0,0xE,0x5,0xD,0x4])
Prost        = SBox([0x0,0x4,0x8,0xf,0x1,0x5,0xe,0x9,0x2,0x7,0xa,0xc,0xb,0xd,0x6,0x3])
Qarma_sigma0 = SBox([0, 14, 2, 10, 9, 15, 8, 11, 6, 4, 3, 7, 13, 12, 1, 5])
Qarma_sigma1 = SBox([10, 13, 14, 6, 15, 7, 3, 5, 9, 8, 0, 12, 11, 1, 2, 4])
Qarma_sigma2 = SBox([11, 6, 8, 15, 12, 0, 9, 14, 3, 7, 4, 5, 13, 2, 1, 10])
REC_0        = SBox([0x9,0x4,0xF,0xA,0xE,0x1,0x0,0x6,0xC,0x7,0x3,0x8,0x2,0xB,0x5,0xD])
Rectangle    = SBox([0x6,0x5,0xC,0xA,0x1,0xE,0x7,0x9,0xB,0x0,0x3,0xD,0x8,0xF,0x4,0x2])
SC2000_4     = SBox([2,5,10,12,7,15,1,11,13,6,0,9,4,8,3,14])
SKINNY_4     = SBox([0xc,0x6,0x9,0x0,0x1,0xa,0x2,0xb,0x3,0x8,0x5,0xd,0x4,0xe,0x7,0xf])
TWINE        = SBox([0xC,0x0,0xF,0xA,0x2,0xB,0x9,0x5,0x8,0x3,0xD,0x7,0x1,0xE,0x6,0x4])

# Sub-components of hash functions
Luffa_v1 = SBox([0x7,0xd,0xb,0xa,0xc,0x4,0x8,0x3,0x5,0xf,0x6,0x0,0x9,0x1,0x2,0xe])
Luffa = SBox([0xd,0xe,0x0,0x1,0x5,0xa,0x7,0x6,0xb,0x3,0x9,0xc,0xf,0x8,0x2,0x4])
BLAKE_1 = SBox([14,10,4,8,9,15,13,6,1,12,0,2,11,7,5,3])
BLAKE_2 = SBox([11,8,12,0,5,2,15,13,10,14,3,6,7,1,9,4])
BLAKE_3 = SBox([7,9,3,1,13,12,11,14,2,6,5,10,4,0,15,8])
BLAKE_4 = SBox([9,0,5,7,2,4,10,15,14,1,11,12,6,8,3,13])
BLAKE_5 = SBox([2,12,6,10,0,11,8,3,4,13,7,5,15,14,1,9])
BLAKE_6 = SBox([12,5,1,15,14,13,4,10,0,7,6,3,9,2,8,11])
BLAKE_7 = SBox([13,11,7,14,12,1,3,9,5,0,15,4,8,6,2,10])
BLAKE_8 = SBox([6,15,14,9,11,3,0,8,12,2,13,7,1,4,10,5])
BLAKE_9 = SBox([10,2,8,4,7,6,1,5,15,11,9,14,3,12,13,0])
JH_S0 = SBox([9,0,4,11,13,12,3,15,1,10,2,6,7,5,8,14])
JH_S1 = SBox([3,12,6,13,5,7,1,9,15,2,0,4,11,10,14,8])
SMASH_256_S1 = SBox([6,13,12,7,15,1,3,10,8,11,5,0,2,4,14,9])
SMASH_256_S2 = SBox([1,11,6,0,14,13,5,10,12,2,9,7,3,8,15,4])
SMASH_256_S3 = SBox([4,2,9,12,8,1,14,7,15,5,0,11,6,10,3,13])

# Sub-components of larger S-Boxes
Anubis_S0 = SBox([0xd,0x7,0x3,0x2,0x9,0xa,0xc,0x1,0xf,0x4,0x5,0xe,0x6,0x0,0xb,0x8])
Anubis_S1 = SBox([0x4,0xa,0xf,0xc,0x0,0xd,0x9,0xb,0xe,0x6,0x1,0x7,0x3,0x5,0x8,0x2])
CLEFIA_SS0 = SBox([0xe,0x6,0xc,0xa,0x8,0x7,0x2,0xf,0xb,0x1,0x4,0x0,0x5,0x9,0xd,0x3])
CLEFIA_SS1 = SBox([0x6,0x4,0x0,0xd,0x2,0xb,0xa,0x3,0x9,0xc,0xe,0xf,0x8,0x7,0x5,0x1])
CLEFIA_SS2 = SBox([0xb,0x8,0x5,0xe,0xa,0x6,0x4,0xc,0xf,0x7,0x2,0x3,0x1,0x0,0xd,0x9])
CLEFIA_SS3 = SBox([0xa,0x2,0x6,0xd,0x3,0x4,0x5,0xe,0x0,0x7,0x8,0x9,0xb,0xf,0xc,0x1])
Enocoro_S4 = SBox([1,3,9,10,5,14,7,2,13,0,12,15,4,8,6,11])
Iceberg_S0 = SBox([0xd,0x7,0x3,0x2,0x9,0xa,0xc,0x1,0xf,0x4,0x5,0xe,0x6,0x0,0xb,0x8])
Iceberg_S1 = SBox([0x4,0xa,0xf,0xc,0x0,0xd,0x9,0xb,0xe,0x6,0x1,0x7,0x3,0x5,0x8,0x2])
Khazad_P = SBox([0x3,0xF,0xE,0x0,0x5,0x4,0xB,0xC,0xD,0xA,0x9,0x6,0x7,0x8,0x2,0x1])
Khazad_Q = SBox([0x9,0xE,0x5,0x6,0xA,0x2,0x3,0xC,0xF,0x0,0x4,0xD,0x7,0xB,0x1,0x8])
Whirlpool_E = SBox([0x1,0xB,0x9,0xC,0xD,0x6,0xF,0x3,0xE,0x8,0x7,0x4,0xA,0x2,0x5,0x0])
Whirlpool_R = SBox([0x7,0xC,0xB,0xD,0xE,0x4,0x9,0xF,0x6,0x3,0x8,0xA,0x2,0x5,0x1,0x0])
CS_cipher_F = SBox([0xf,0xd,0xb,0xb,0x7,0x5,0x7,0x7,0xe,0xd,0xa,0xb,0xe,0xd,0xe,0xf])
CS_cipher_G = SBox([0xa,0x6,0x0,0x2,0xb,0xe,0x1,0x8,0xd,0x4,0x5,0x3,0xf,0xc,0x7,0x9])
Fox_S1 = SBox([0x2,0x5,0x1,0x9,0xE,0xA,0xC,0x8,0x6,0x4,0x7,0xF,0xD,0xB,0x0,0x3])
Fox_S2 = SBox([0xB,0x4,0x1,0xF,0x0,0x3,0xE,0xD,0xA,0x8,0x7,0x5,0xC,0x2,0x9,0x6])
Fox_S3 = SBox([0xD,0xA,0xB,0x1,0x4,0x3,0x8,0x9,0x5,0x7,0x2,0xC,0xF,0x0,0x6,0xE])
Twofish_Q0_T0 = SBox([0x8,0x1,0x7,0xD,0x6,0xF,0x3,0x2,0x0,0xB,0x5,0x9,0xE,0xC,0xA,0x4])
Twofish_Q0_T1 = SBox([0xE,0xC,0xB,0x8,0x1,0x2,0x3,0x5,0xF,0x4,0xA,0x6,0x7,0x0,0x9,0xD])
Twofish_Q0_T2 = SBox([0xB,0xA,0x5,0xE,0x6,0xD,0x9,0x0,0xC,0x8,0xF,0x3,0x2,0x4,0x7,0x1])
Twofish_Q0_T3 = SBox([0xD,0x7,0xF,0x4,0x1,0x2,0x6,0xE,0x9,0xB,0x3,0x0,0x8,0x5,0xC,0xA])
Twofish_Q1_T0 = SBox([0x2,0x8,0xB,0xD,0xF,0x7,0x6,0xE,0x3,0x1,0x9,0x4,0x0,0xA,0xC,0x5])
Twofish_Q1_T1 = SBox([0x1,0xE,0x2,0xB,0x4,0xC,0x3,0x7,0x6,0xD,0xA,0x5,0xF,0x9,0x0,0x8])
Twofish_Q1_T2 = SBox([0x4,0xC,0x7,0x5,0x1,0x6,0x9,0xA,0x0,0xE,0xD,0x8,0x2,0xB,0x3,0xF])
Twofish_Q1_T3 = SBox([0xB,0x9,0x5,0x1,0xC,0x3,0xD,0xE,0x6,0x4,0x7,0xF,0x2,0x0,0x8,0xA])
Kuznyechik_nu0  = SBox([0x2,0x5,0x3,0xb,0x6,0x9,0xe,0xa,0x0,0x4,0xf,0x1,0x8,0xd,0xc,0x7])
Kuznyechik_nu1  = SBox([0x7,0x6,0xc,0x9,0x0,0xf,0x8,0x1,0x4,0x5,0xb,0xe,0xd,0x2,0x3,0xa])
Kuznyechik_sigma = SBox([0xc,0xd,0x0,0x4,0x8,0xb,0xa,0xe,0x3,0x9,0x5,0x2,0xf,0x1,0x6,0x7])
Kuznyechik_phi   = SBox([0xb,0x2,0xb,0x8,0xc,0x4,0x1,0xc,0x6,0x3,0x5,0x8,0xe,0x3,0x6,0xb])
Optimal_S0 = SBox([0, 1, 2, 13, 4, 7, 15, 6, 8, 11, 12, 9, 3, 14, 10, 5])
Optimal_S1 = SBox([0, 1, 2, 13, 4, 7, 15, 6, 8, 11, 14, 3, 5, 9, 10, 12])
Optimal_S2 = SBox([0, 1, 2, 13, 4, 7, 15, 6, 8, 11, 14, 3, 10, 12, 5, 9])
Optimal_S3 = SBox([0, 1, 2, 13, 4, 7, 15, 6, 8, 12, 5, 3, 10, 14, 11, 9])
Optimal_S4 = SBox([0, 1, 2, 13, 4, 7, 15, 6, 8, 12, 9, 11, 10, 14, 5, 3])
Optimal_S5 = SBox([0, 1, 2, 13, 4, 7, 15, 6, 8, 12, 11, 9, 10, 14, 3, 5])
Optimal_S6 = SBox([0, 1, 2, 13, 4, 7, 15, 6, 8, 12, 11, 9, 10, 14, 5, 3])
Optimal_S7 = SBox([0, 1, 2, 13, 4, 7, 15, 6, 8, 12, 14, 11, 10, 9, 3, 5])
Optimal_S8 = SBox([0, 1, 2, 13, 4, 7, 15, 6, 8, 14, 9, 5, 10, 11, 3, 12])
Optimal_S9 = SBox([0, 1, 2, 13, 4, 7, 15, 6, 8, 14, 11, 3, 5, 9, 10, 12])
Optimal_S10 = SBox([0, 1, 2, 13, 4, 7, 15, 6, 8, 14, 11, 5, 10, 9, 3, 12])
Optimal_S11 = SBox([0, 1, 2, 13, 4, 7, 15, 6, 8, 14, 11, 10, 5, 9, 12, 3])
Optimal_S12 = SBox([0, 1, 2, 13, 4, 7, 15, 6, 8, 14, 11, 10, 9, 3, 12, 5])
Optimal_S13 = SBox([0, 1, 2, 13, 4, 7, 15, 6, 8, 14, 12, 9, 5, 11, 10, 3])
Optimal_S14 = SBox([0, 1, 2, 13, 4, 7, 15, 6, 8, 14, 12, 11, 3, 9, 5, 10])
Optimal_S15 = SBox([0, 1, 2, 13, 4, 7, 15, 6, 8, 14, 12, 11, 9, 3, 10, 5])
Serpent_type_S0 = SBox([0, 3, 5, 6, 7, 10, 11, 12, 13, 4, 14, 9, 8, 1, 2, 15])
Serpent_type_S1 = SBox([0, 3, 5, 8, 6, 9, 10, 7, 11, 12, 14, 2, 1, 15, 13, 4])
Serpent_type_S2 = SBox([0, 3, 5, 8, 6, 9, 11, 2, 13, 4, 14, 1, 10, 15, 7, 12])
Serpent_type_S3 = SBox([0, 3, 5, 8, 6, 10, 15, 4, 14, 13, 9, 2, 1, 7, 12, 11])
Serpent_type_S4 = SBox([0, 3, 5, 8, 6, 12, 11, 7, 9, 14, 10, 13, 15, 2, 1, 4])
Serpent_type_S5 = SBox([0, 3, 5, 8, 6, 12, 11, 7, 10, 4, 9, 14, 15, 1, 2, 13])
Serpent_type_S6 = SBox([0, 3, 5, 8, 6, 12, 11, 7, 10, 13, 9, 14, 15, 1, 2, 4])
Serpent_type_S7 = SBox([0, 3, 5, 8, 6, 12, 11, 7, 13, 10, 14, 4, 1, 15, 2, 9])
Serpent_type_S8 = SBox([0, 3, 5, 8, 6, 12, 15, 1, 10, 4, 9, 14, 13, 11, 2, 7])
Serpent_type_S9 = SBox([0, 3, 5, 8, 6, 12, 15, 2, 14, 9, 11, 7, 13, 10, 4, 1])
Serpent_type_S10 = SBox([0, 3, 5, 8, 6, 13, 15, 1, 9, 12, 2, 11, 10, 7, 4, 14])
Serpent_type_S11 = SBox([0, 3, 5, 8, 6, 13, 15, 2, 7, 4, 14, 11, 10, 1, 9, 12])
Serpent_type_S12 = SBox([0, 3, 5, 8, 6, 13, 15, 2, 12, 9, 10, 4, 11, 14, 1, 7])
Serpent_type_S13 = SBox([0, 3, 5, 8, 6, 15, 10, 1, 7, 9, 14, 4, 11, 12, 13, 2])
Serpent_type_S14 = SBox([0, 3, 5, 8, 7, 4, 9, 14, 15, 6, 2, 11, 10, 13, 12, 1])
Serpent_type_S15 = SBox([0, 3, 5, 8, 7, 9, 11, 14, 10, 13, 15, 4, 12, 2, 6, 1])
Serpent_type_S16 = SBox([0, 3, 5, 8, 9, 12, 14, 7, 10, 13, 15, 4, 6, 11, 1, 2])
Serpent_type_S17 = SBox([0, 3, 5, 8, 10, 13, 9, 4, 15, 6, 2, 1, 12, 11, 7, 14])
Serpent_type_S18 = SBox([0, 3, 5, 8, 11, 12, 6, 15, 14, 9, 2, 7, 4, 10, 13, 1])
Serpent_type_S19 = SBox([0, 3, 5, 10, 7, 12, 11, 6, 13, 4, 2, 9, 14, 1, 8, 15])
Golden_S0 = SBox([0, 3, 5, 8, 6, 9, 12, 7, 13, 10, 14, 4, 1, 15, 11, 2])
Golden_S1 = SBox([0, 3, 5, 8, 6, 12, 11, 7, 9, 14, 10, 13, 15, 2, 1, 4])
Golden_S2 = SBox([0, 3, 5, 8, 6, 10, 15, 4, 14, 13, 9, 2, 1, 7, 12, 11])
Golden_S3 = SBox([0, 3, 5, 8, 6, 12, 11, 7, 10, 4, 9, 14, 15, 1, 2, 13])

# S-Boxes from the literature on Boolean functions

# Ullrich, Markus, et al. "Finding optimal bitsliced implementations
# of 4x4-bit S-boxes." SKEW 2011 Symmetric Key Encryption Workshop,
# Copenhagen, Denmark. 2011.
UDCIKMP11 = SBox([0x0,0x8,0x6,0xD,0x5,0xF,0x7,0xC,0x4,0xE,0x2,0x3,0x9,0x1,0xB,0xA])

# Bijective S-Boxes mapping 3 bits to 3
# =====================================

SEA = SBox([0x0,0x5,0x6,0x7,0x4,0x3,0x1,0x2])
PRINTcipher = SBox([0x0, 0x1, 0x3, 0x6, 0x7, 0x4, 0x5, 0x2])

affine_equiv_classes = {}
# affine equivalent classes for 4 bit
affine_equiv_classes[4] = [
    SBox([0x4, 0x0, 0x1, 0xF, 0x2, 0xB, 0x6, 0x7, 0x3, 0x9, 0xA, 0x5, 0xC, 0xD, 0xE, 0x8]),
    SBox([0x8, 0x0, 0x1, 0xC, 0x2, 0x5, 0x6, 0x9, 0x4, 0x3, 0xA, 0xB, 0x7, 0xD, 0xE, 0xF]),
    SBox([0x8, 0x0, 0x1, 0xC, 0xF, 0x5, 0x6, 0x7, 0x4, 0x3, 0xA, 0xB, 0x9, 0xD, 0xE, 0x2]),
    SBox([0x2, 0x0, 0x1, 0x8, 0x3, 0xD, 0x6, 0x7, 0x4, 0x9, 0xA, 0x5, 0xC, 0xB, 0xE, 0xF]),
    SBox([0x2, 0x0, 0x1, 0x8, 0x3, 0xF, 0x6, 0x7, 0x4, 0x9, 0x5, 0xB, 0xC, 0xD, 0xE, 0xA]),
    SBox([0x2, 0x0, 0x1, 0x8, 0x3, 0xB, 0x6, 0x7, 0x4, 0x9, 0xA, 0xF, 0xC, 0xD, 0xE, 0x5]),
    SBox([0x4, 0x8, 0x1, 0x2, 0x3, 0xB, 0x6, 0x7, 0x0, 0x9, 0xA, 0xE, 0xC, 0xD, 0x5, 0xF]),
    SBox([0x8, 0x0, 0x1, 0x9, 0x2, 0x5, 0xD, 0x7, 0x4, 0x6, 0xA, 0xB, 0xC, 0x3, 0xE, 0xF]),
    SBox([0x8, 0xE, 0x1, 0x2, 0x3, 0x5, 0x6, 0x7, 0x4, 0xC, 0xA, 0xB, 0x9, 0xD, 0x0, 0xF]),
    SBox([0x8, 0xE, 0x1, 0x2, 0x3, 0x5, 0x6, 0x7, 0x4, 0x9, 0xF, 0xB, 0xC, 0xD, 0x0, 0xA]),
    SBox([0x8, 0xF, 0x1, 0x2, 0x3, 0x5, 0xC, 0x7, 0x4, 0x9, 0xA, 0xB, 0x6, 0xD, 0xE, 0x0]),
    SBox([0x8, 0xF, 0x1, 0x2, 0x3, 0x5, 0x6, 0xD, 0x4, 0x9, 0xA, 0xB, 0xC, 0x7, 0xE, 0x0]),
    SBox([0xC, 0x0, 0x1, 0x9, 0x3, 0x5, 0x4, 0x7, 0x6, 0x2, 0xA, 0xB, 0x8, 0xD, 0xE, 0xF]),
    SBox([0xC, 0xB, 0x1, 0x2, 0x3, 0x5, 0x4, 0x7, 0x6, 0x9, 0xA, 0x0, 0x8, 0xD, 0xE, 0xF]),
    SBox([0xC, 0x9, 0x1, 0x2, 0x3, 0x5, 0x4, 0x7, 0x6, 0x0, 0xA, 0xB, 0x8, 0xD, 0xE, 0xF]),
    SBox([0x8, 0xE, 0x1, 0x2, 0x3, 0x5, 0x4, 0x7, 0x6, 0x9, 0xA, 0x0, 0xC, 0xD, 0xB, 0xF]),
    SBox([0x8, 0xA, 0x1, 0x2, 0x3, 0xB, 0x6, 0x7, 0x4, 0x9, 0x0, 0x5, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x4, 0x0, 0x1, 0x8, 0x2, 0xC, 0x3, 0x7, 0x6, 0x9, 0xA, 0xB, 0x5, 0xD, 0xE, 0xF]),
    SBox([0x4, 0x0, 0x1, 0x8, 0x2, 0x5, 0x3, 0xB, 0x6, 0x9, 0xA, 0x7, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x4, 0x0, 0x1, 0x8, 0x2, 0x5, 0x3, 0xE, 0x6, 0x9, 0xA, 0xB, 0xC, 0xD, 0x7, 0xF]),
    SBox([0x6, 0x8, 0x1, 0x2, 0x3, 0x5, 0x4, 0xB, 0x0, 0x9, 0xA, 0x7, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x4, 0x0, 0x1, 0x8, 0x2, 0x9, 0x3, 0x7, 0x6, 0x5, 0xA, 0xB, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x8, 0xF, 0x1, 0x2, 0x3, 0x5, 0x6, 0x7, 0x4, 0x9, 0x0, 0xB, 0xC, 0xD, 0xE, 0xA]),
    SBox([0x4, 0x0, 0x1, 0x8, 0x2, 0x5, 0xF, 0x7, 0x6, 0x9, 0xA, 0xB, 0xC, 0xD, 0xE, 0x3]),
    SBox([0xD, 0x0, 0x1, 0x2, 0x3, 0x5, 0xE, 0x7, 0x6, 0x9, 0xA, 0xB, 0xC, 0x8, 0x4, 0xF]),
    SBox([0xC, 0xA, 0x1, 0x2, 0x3, 0x5, 0x4, 0x7, 0x6, 0x9, 0x0, 0xB, 0x8, 0xD, 0xE, 0xF]),
    SBox([0xC, 0x0, 0x1, 0x2, 0x3, 0x5, 0xF, 0x7, 0x6, 0x9, 0xA, 0xB, 0x8, 0xD, 0xE, 0x4]),
    SBox([0x8, 0xA, 0x1, 0x2, 0x3, 0x0, 0x4, 0x7, 0x6, 0x9, 0x5, 0xB, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x8, 0xD, 0x1, 0x2, 0x3, 0x0, 0x4, 0x7, 0x6, 0x9, 0xA, 0xB, 0xC, 0x5, 0xE, 0xF]),
    SBox([0x6, 0x8, 0x1, 0xE, 0x3, 0x5, 0x4, 0x7, 0x0, 0x9, 0xA, 0xB, 0xC, 0xD, 0x2, 0xF]),
    SBox([0x4, 0x0, 0x1, 0x8, 0x2, 0x5, 0xA, 0x7, 0x6, 0x9, 0x3, 0xB, 0xC, 0xD, 0xE, 0xF]),
    SBox([0xC, 0x0, 0xA, 0x2, 0x3, 0x5, 0x4, 0x7, 0x6, 0x9, 0x1, 0xB, 0x8, 0xD, 0xE, 0xF]),
    SBox([0xD, 0x0, 0xA, 0x2, 0x3, 0x5, 0x4, 0x7, 0x6, 0x9, 0x1, 0xB, 0xC, 0x8, 0xE, 0xF]),
    SBox([0x8, 0xF, 0x1, 0x2, 0x3, 0xC, 0x6, 0x7, 0x4, 0x9, 0xA, 0xB, 0x5, 0xD, 0xE, 0x0]),
    SBox([0x8, 0x0, 0x1, 0xC, 0x2, 0x5, 0xB, 0x7, 0x4, 0x9, 0xA, 0x6, 0x3, 0xD, 0xE, 0xF]),
    SBox([0x8, 0x0, 0x1, 0xC, 0x2, 0x5, 0xD, 0x7, 0x4, 0x9, 0xA, 0xB, 0x3, 0x6, 0xE, 0xF]),
    SBox([0xC, 0x0, 0x1, 0x2, 0x3, 0xF, 0x6, 0x7, 0x4, 0x9, 0xA, 0xB, 0x8, 0xD, 0xE, 0x5]),
    SBox([0xC, 0x0, 0x1, 0x2, 0x3, 0x5, 0x6, 0xD, 0x4, 0x9, 0xA, 0xB, 0x8, 0x7, 0xE, 0xF]),
    SBox([0x8, 0xA, 0x1, 0x2, 0x3, 0x5, 0x6, 0x9, 0x4, 0x7, 0x0, 0xB, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x8, 0x0, 0x1, 0xC, 0x2, 0xE, 0x6, 0x7, 0x4, 0x9, 0xA, 0xB, 0x3, 0xD, 0x5, 0xF]),
    SBox([0x4, 0x0, 0x1, 0x8, 0x2, 0xD, 0x3, 0x7, 0x6, 0x9, 0xA, 0xB, 0xC, 0x5, 0xE, 0xF]),
    SBox([0x4, 0x0, 0x1, 0x8, 0x2, 0xA, 0x3, 0x7, 0x6, 0x9, 0x5, 0xB, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x6, 0x8, 0x1, 0x2, 0x3, 0x5, 0x4, 0xC, 0x0, 0x9, 0xA, 0xB, 0x7, 0xD, 0xE, 0xF]),
    SBox([0x6, 0x8, 0x1, 0x2, 0x3, 0x5, 0x4, 0xD, 0x0, 0x9, 0xA, 0xB, 0xC, 0x7, 0xE, 0xF]),
    SBox([0x6, 0x8, 0x1, 0x2, 0x3, 0x5, 0x4, 0x9, 0x0, 0x7, 0xA, 0xB, 0xC, 0xD, 0xE, 0xF]),
    SBox([0xD, 0xA, 0x1, 0x2, 0x3, 0x5, 0x6, 0x7, 0x4, 0x9, 0x0, 0xB, 0xC, 0x8, 0xE, 0xF]),
    SBox([0xE, 0x0, 0x1, 0xA, 0x3, 0x5, 0x6, 0x7, 0x4, 0x9, 0x2, 0xB, 0xC, 0xD, 0x8, 0xF]),
    SBox([0x8, 0xA, 0x1, 0x2, 0x3, 0x5, 0x9, 0x7, 0x4, 0x6, 0x0, 0xB, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x8, 0x0, 0x1, 0xD, 0x2, 0x5, 0x6, 0x7, 0x4, 0x9, 0x3, 0xB, 0xC, 0xA, 0xE, 0xF]),
    SBox([0x8, 0x0, 0x1, 0x6, 0x2, 0x5, 0xD, 0x7, 0x4, 0x9, 0xA, 0xB, 0xC, 0x3, 0xE, 0xF]),
    SBox([0x8, 0xC, 0x1, 0x2, 0x3, 0x5, 0x4, 0x7, 0x6, 0x9, 0xA, 0xB, 0x0, 0xD, 0xE, 0xF]),
    SBox([0x6, 0x8, 0xD, 0x2, 0x3, 0x5, 0x4, 0x7, 0x0, 0x9, 0xA, 0xB, 0xC, 0x1, 0xE, 0xF]),
    SBox([0x6, 0x8, 0xE, 0x2, 0x3, 0x5, 0x4, 0x7, 0x0, 0x9, 0xA, 0xB, 0xC, 0xD, 0x1, 0xF]),
    SBox([0x4, 0x0, 0x1, 0xF, 0x2, 0x5, 0x3, 0x7, 0x6, 0x9, 0xA, 0xB, 0xC, 0xD, 0xE, 0x8]),
    SBox([0x4, 0x0, 0x1, 0xA, 0x2, 0x8, 0x6, 0x7, 0x3, 0x9, 0x5, 0xB, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x4, 0x0, 0x1, 0xA, 0x2, 0x5, 0x6, 0x8, 0x3, 0x9, 0x7, 0xB, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x6, 0x0, 0x8, 0xF, 0x3, 0x5, 0x4, 0x7, 0x1, 0x9, 0xA, 0xB, 0xC, 0xD, 0xE, 0x2]),
    SBox([0x6, 0x0, 0x8, 0xC, 0x3, 0x5, 0x4, 0x7, 0x1, 0x9, 0xA, 0xB, 0x2, 0xD, 0xE, 0xF]),
    SBox([0x6, 0x8, 0x1, 0xC, 0x3, 0x5, 0x4, 0x7, 0x0, 0x9, 0xA, 0xB, 0x2, 0xD, 0xE, 0xF]),
    SBox([0x6, 0x8, 0x1, 0xD, 0x3, 0x5, 0x4, 0x7, 0x0, 0x9, 0xA, 0xB, 0xC, 0x2, 0xE, 0xF]),
    SBox([0x8, 0x0, 0x1, 0xC, 0x2, 0x5, 0x6, 0x7, 0x4, 0x3, 0xA, 0xB, 0x9, 0xD, 0xE, 0xF]),
    SBox([0x8, 0x0, 0x1, 0xA, 0x2, 0x5, 0x6, 0x3, 0x4, 0x9, 0x7, 0xB, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x8, 0x0, 0x1, 0xD, 0x2, 0x5, 0x3, 0x7, 0x4, 0x9, 0xA, 0xB, 0xC, 0x6, 0xE, 0xF]),
    SBox([0x8, 0x0, 0x1, 0xA, 0x2, 0x5, 0x6, 0x7, 0x4, 0x9, 0xD, 0xB, 0xC, 0x3, 0xE, 0xF]),
    SBox([0x8, 0x0, 0x1, 0xE, 0x3, 0x5, 0x4, 0x7, 0x6, 0x9, 0xA, 0xB, 0xC, 0xD, 0x2, 0xF]),
    SBox([0x6, 0x0, 0x1, 0x8, 0xD, 0x5, 0x4, 0x7, 0x2, 0x9, 0xA, 0xB, 0xC, 0x3, 0xE, 0xF]),
    SBox([0x6, 0x8, 0x1, 0x2, 0xF, 0x5, 0x4, 0x7, 0x0, 0x9, 0xA, 0xB, 0xC, 0xD, 0xE, 0x3]),
    SBox([0x6, 0x8, 0x1, 0x2, 0x3, 0xA, 0x4, 0x7, 0x0, 0x9, 0x5, 0xB, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x8, 0x0, 0x1, 0x3, 0x2, 0xA, 0x6, 0x7, 0x4, 0x9, 0xF, 0xB, 0xC, 0xD, 0xE, 0x5]),
    SBox([0x4, 0x0, 0x1, 0x8, 0x2, 0x5, 0xB, 0x7, 0x6, 0x9, 0xA, 0x3, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x8, 0x0, 0x1, 0x6, 0x2, 0x5, 0xC, 0x7, 0x4, 0x9, 0xA, 0xB, 0x3, 0xD, 0xE, 0xF]),
    SBox([0x8, 0x0, 0x1, 0x7, 0x2, 0x5, 0x6, 0xC, 0x4, 0x9, 0xA, 0xB, 0x3, 0xD, 0xE, 0xF]),
    SBox([0x4, 0x0, 0x1, 0x8, 0x2, 0x5, 0x9, 0x7, 0x6, 0x3, 0xA, 0xB, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x6, 0x8, 0x1, 0x2, 0x3, 0x9, 0x4, 0x7, 0x0, 0x5, 0xA, 0xB, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x6, 0x8, 0x1, 0x9, 0x3, 0x5, 0x4, 0x7, 0x0, 0x2, 0xA, 0xB, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x8, 0x0, 0x1, 0x9, 0x2, 0x5, 0x3, 0x7, 0x4, 0x6, 0xA, 0xB, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x8, 0x0, 0x1, 0xC, 0x2, 0x5, 0x3, 0x7, 0x4, 0x9, 0xA, 0xB, 0x6, 0xD, 0xE, 0xF]),
    SBox([0x8, 0x0, 0x1, 0x3, 0x2, 0xF, 0x6, 0x7, 0x4, 0x9, 0x5, 0xB, 0xC, 0xD, 0xE, 0xA]),
    SBox([0x6, 0xD, 0x1, 0x2, 0x3, 0x5, 0x4, 0x7, 0x0, 0x9, 0xA, 0xB, 0xC, 0x8, 0xE, 0xF]),
    SBox([0x8, 0xB, 0x1, 0x2, 0x3, 0x5, 0x4, 0x7, 0x6, 0x9, 0xA, 0x0, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x8, 0xD, 0x1, 0x2, 0x3, 0x5, 0x4, 0x7, 0x6, 0x9, 0xA, 0xB, 0xC, 0x0, 0xE, 0xF]),
    SBox([0x6, 0x8, 0xF, 0x2, 0x3, 0x5, 0x4, 0x7, 0x0, 0x9, 0xA, 0xB, 0xC, 0xD, 0xE, 0x1]),
    SBox([0x6, 0x8, 0xC, 0x2, 0x3, 0x5, 0x4, 0x7, 0x0, 0x9, 0xA, 0xB, 0x1, 0xD, 0xE, 0xF]),
    SBox([0x8, 0x5, 0x1, 0x2, 0x3, 0x0, 0x4, 0x7, 0x6, 0x9, 0xA, 0xB, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x8, 0xA, 0x1, 0x2, 0x3, 0x5, 0x4, 0x7, 0x6, 0x9, 0x0, 0xB, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x6, 0x8, 0x1, 0x2, 0x3, 0x5, 0xA, 0x7, 0x0, 0x9, 0x4, 0xB, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x8, 0xA, 0x1, 0xC, 0x3, 0x5, 0x6, 0x7, 0x4, 0x9, 0x0, 0xB, 0x2, 0xD, 0xE, 0xF]),
    SBox([0x8, 0x0, 0x1, 0xC, 0x2, 0x5, 0x6, 0x3, 0x4, 0x9, 0xA, 0xB, 0x7, 0xD, 0xE, 0xF]),
    SBox([0x2, 0x0, 0x1, 0x8, 0x3, 0xB, 0x6, 0x7, 0x4, 0x9, 0xA, 0x5, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x4, 0x8, 0x1, 0x2, 0x3, 0xB, 0x6, 0x7, 0x0, 0x9, 0xA, 0x5, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x4, 0x8, 0x1, 0x2, 0x3, 0x5, 0x6, 0x9, 0x0, 0x7, 0xA, 0xB, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x8, 0x6, 0x1, 0x2, 0x9, 0x5, 0x0, 0x7, 0x4, 0x3, 0xA, 0xB, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x6, 0x8, 0x1, 0x2, 0x3, 0xB, 0x4, 0x7, 0x0, 0x9, 0xA, 0x5, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x6, 0x8, 0x1, 0x2, 0x3, 0xE, 0x4, 0x7, 0x0, 0x9, 0xA, 0xB, 0xC, 0xD, 0x5, 0xF]),
    SBox([0x6, 0x8, 0x1, 0x2, 0x3, 0xF, 0x4, 0x7, 0x0, 0x9, 0xA, 0xB, 0xC, 0xD, 0xE, 0x5]),
    SBox([0xF, 0x0, 0x1, 0x3, 0x2, 0xC, 0x6, 0x7, 0x4, 0x9, 0xA, 0xB, 0x5, 0xD, 0xE, 0x8]),
    SBox([0x8, 0xD, 0x1, 0x2, 0x3, 0x5, 0x0, 0x7, 0x4, 0x9, 0xA, 0xB, 0xC, 0x6, 0xE, 0xF]),
    SBox([0x4, 0x0, 0x1, 0xE, 0x2, 0x5, 0x6, 0x7, 0x3, 0x9, 0x8, 0xB, 0xC, 0xD, 0xA, 0xF]),
    SBox([0x8, 0x0, 0x1, 0xF, 0x2, 0x5, 0x3, 0x7, 0x4, 0x9, 0xA, 0xB, 0xC, 0xD, 0xE, 0x6]),
    SBox([0x2, 0x0, 0x1, 0x3, 0xF, 0xA, 0x6, 0x7, 0x4, 0x9, 0x5, 0xB, 0xC, 0xD, 0xE, 0x8]),
    SBox([0x2, 0x0, 0x1, 0x8, 0x3, 0xA, 0x6, 0x7, 0x4, 0x9, 0x5, 0xB, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x2, 0x0, 0x1, 0x8, 0x3, 0xD, 0x6, 0x7, 0x4, 0x9, 0xA, 0xB, 0xC, 0x5, 0xE, 0xF]),
    SBox([0x4, 0x0, 0x1, 0x8, 0x2, 0xB, 0x6, 0x7, 0x3, 0x9, 0xA, 0x5, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x4, 0x0, 0x1, 0x8, 0x2, 0xC, 0x6, 0x7, 0x3, 0x9, 0xA, 0xB, 0x5, 0xD, 0xE, 0xF]),
    SBox([0x2, 0x0, 0x1, 0x8, 0x3, 0x5, 0x6, 0x7, 0x4, 0xC, 0xA, 0xB, 0x9, 0xD, 0xE, 0xF]),
    SBox([0x8, 0xF, 0x1, 0x2, 0x3, 0x5, 0x4, 0x7, 0x6, 0x9, 0xA, 0xB, 0xC, 0xD, 0xE, 0x0]),
    SBox([0x8, 0x0, 0x1, 0xD, 0x3, 0x5, 0x4, 0x7, 0x6, 0x9, 0xA, 0xB, 0xC, 0x2, 0xE, 0xF]),
    SBox([0x6, 0x8, 0x1, 0x2, 0xC, 0x5, 0x4, 0x7, 0x0, 0x9, 0xA, 0xB, 0x3, 0xD, 0xE, 0xF]),
    SBox([0x4, 0x8, 0x1, 0x2, 0x3, 0xE, 0x6, 0x7, 0x0, 0x9, 0xA, 0xB, 0xC, 0xD, 0x5, 0xF]),
    SBox([0x8, 0x0, 0x1, 0x2, 0x3, 0xB, 0x6, 0x7, 0x4, 0x9, 0xA, 0x5, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x6, 0x8, 0x1, 0x2, 0x3, 0x5, 0xE, 0x7, 0x0, 0x9, 0xA, 0xB, 0xC, 0xD, 0x4, 0xF]),
    SBox([0x4, 0x8, 0x1, 0x2, 0x3, 0x5, 0x6, 0xD, 0x0, 0x9, 0xA, 0xB, 0xC, 0x7, 0xE, 0xF]),
    SBox([0x4, 0x8, 0x1, 0x2, 0x3, 0x5, 0x6, 0x7, 0x0, 0xF, 0xA, 0xB, 0xC, 0xD, 0xE, 0x9]),
    SBox([0x4, 0x8, 0x1, 0x2, 0x3, 0x5, 0x6, 0x7, 0x0, 0x9, 0xA, 0xE, 0xC, 0xD, 0xB, 0xF]),
    SBox([0x8, 0x0, 0x1, 0x2, 0x3, 0x5, 0x6, 0x9, 0x4, 0x7, 0xA, 0xB, 0xC, 0xD, 0xE, 0xF]),
    SBox([0xF, 0x0, 0x1, 0xA, 0x2, 0x5, 0x6, 0x7, 0x4, 0x9, 0x3, 0xB, 0xC, 0xD, 0xE, 0x8]),
    SBox([0x2, 0x0, 0x1, 0x8, 0x3, 0xF, 0x6, 0x7, 0x4, 0x9, 0xA, 0xB, 0xC, 0xD, 0xE, 0x5]),
    SBox([0x4, 0x0, 0x1, 0x8, 0x2, 0x5, 0xF, 0x7, 0x3, 0x9, 0xA, 0xB, 0xC, 0xD, 0xE, 0x6]),
    SBox([0x6, 0x8, 0x1, 0xB, 0x3, 0x5, 0x4, 0x7, 0x0, 0x9, 0xA, 0x2, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x4, 0x8, 0x1, 0x2, 0x3, 0xA, 0x6, 0x7, 0x0, 0x9, 0x5, 0xB, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x8, 0x0, 0x1, 0x2, 0x3, 0xF, 0x6, 0x7, 0x4, 0x9, 0xA, 0xB, 0xC, 0xD, 0xE, 0x5]),
    SBox([0x4, 0x8, 0x1, 0x2, 0x3, 0x5, 0xB, 0x7, 0x0, 0x9, 0xA, 0x6, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x4, 0x8, 0x1, 0x2, 0x3, 0x5, 0xD, 0x7, 0x0, 0x9, 0xA, 0xB, 0xC, 0x6, 0xE, 0xF]),
    SBox([0x4, 0x8, 0x1, 0x2, 0x3, 0x5, 0x6, 0xA, 0x0, 0x9, 0x7, 0xB, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x8, 0x0, 0x1, 0x2, 0x3, 0x5, 0x6, 0xD, 0x4, 0x9, 0xA, 0xB, 0xC, 0x7, 0xE, 0xF]),
    SBox([0x8, 0x0, 0x1, 0x2, 0x3, 0x5, 0x6, 0x7, 0x4, 0xC, 0xA, 0xB, 0x9, 0xD, 0xE, 0xF]),
    SBox([0x4, 0x8, 0x1, 0x2, 0x3, 0x5, 0x6, 0x7, 0x0, 0x9, 0xF, 0xB, 0xC, 0xD, 0xE, 0xA]),
    SBox([0x4, 0x8, 0x1, 0x2, 0x3, 0x5, 0x9, 0x7, 0x0, 0x6, 0xA, 0xB, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x8, 0x0, 0x1, 0x2, 0x3, 0x5, 0x6, 0x7, 0x4, 0x9, 0xA, 0xE, 0xC, 0xD, 0xB, 0xF]),
    SBox([0x4, 0x0, 0x1, 0xA, 0x2, 0x5, 0x3, 0x7, 0x6, 0x9, 0x8, 0xB, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x4, 0x8, 0x1, 0x2, 0x3, 0x5, 0xA, 0x7, 0x0, 0x9, 0x6, 0xB, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x8, 0x0, 0xB, 0x2, 0x3, 0x5, 0x4, 0x7, 0x6, 0x9, 0xA, 0x1, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x8, 0x0, 0x1, 0x2, 0x3, 0x5, 0xB, 0x7, 0x6, 0x9, 0xA, 0x4, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x8, 0x0, 0xC, 0x2, 0x3, 0x5, 0x4, 0x7, 0x6, 0x9, 0xA, 0xB, 0x1, 0xD, 0xE, 0xF]),
    SBox([0x8, 0x0, 0xD, 0x2, 0x3, 0x5, 0x4, 0x7, 0x6, 0x9, 0xA, 0xB, 0xC, 0x1, 0xE, 0xF]),
    SBox([0x8, 0x0, 0x1, 0x2, 0xE, 0x5, 0x4, 0x7, 0x6, 0x9, 0xA, 0xB, 0xC, 0xD, 0x3, 0xF]),
    SBox([0x8, 0x0, 0x1, 0x2, 0xF, 0x5, 0x4, 0x7, 0x6, 0x9, 0xA, 0xB, 0xC, 0xD, 0xE, 0x3]),
    SBox([0x6, 0x0, 0x8, 0x2, 0x3, 0x5, 0xB, 0x7, 0x1, 0x9, 0xA, 0x4, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x8, 0x0, 0x1, 0x2, 0x3, 0x5, 0xC, 0x7, 0x4, 0x9, 0xA, 0xB, 0x6, 0xD, 0xE, 0xF]),
    SBox([0x4, 0x0, 0x1, 0x8, 0x2, 0x9, 0x6, 0x7, 0x3, 0x5, 0xA, 0xB, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x8, 0x0, 0x1, 0x2, 0x3, 0x5, 0x6, 0x7, 0x4, 0x9, 0xF, 0xB, 0xC, 0xD, 0xE, 0xA]),
    SBox([0xC, 0x0, 0x1, 0x2, 0x3, 0x5, 0x4, 0x7, 0x6, 0x9, 0xA, 0xB, 0x8, 0xD, 0xE, 0xF]),
    SBox([0xD, 0x0, 0x1, 0x2, 0x3, 0x5, 0x4, 0x7, 0x6, 0x9, 0xA, 0xB, 0xC, 0x8, 0xE, 0xF]),
    SBox([0x8, 0xE, 0x1, 0x2, 0x3, 0x5, 0x4, 0x7, 0x6, 0x9, 0xA, 0xB, 0xC, 0xD, 0x0, 0xF]),
    SBox([0x8, 0x0, 0x1, 0xF, 0x2, 0x5, 0x6, 0x7, 0x4, 0x9, 0x3, 0xB, 0xC, 0xD, 0xE, 0xA]),
    SBox([0x8, 0x0, 0x1, 0xB, 0x3, 0x5, 0x4, 0x7, 0x6, 0x9, 0xA, 0x2, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x8, 0x0, 0x1, 0xC, 0x3, 0x5, 0x4, 0x7, 0x6, 0x9, 0xA, 0xB, 0x2, 0xD, 0xE, 0xF]),
    SBox([0x6, 0x8, 0x1, 0x2, 0xB, 0x5, 0x4, 0x7, 0x0, 0x9, 0xA, 0x3, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x6, 0x8, 0x1, 0x2, 0xD, 0x5, 0x4, 0x7, 0x0, 0x9, 0xA, 0xB, 0xC, 0x3, 0xE, 0xF]),
    SBox([0x8, 0x0, 0x1, 0x2, 0x3, 0x5, 0xF, 0x7, 0x6, 0x9, 0xA, 0xB, 0xC, 0xD, 0xE, 0x4]),
    SBox([0x8, 0x0, 0x1, 0x6, 0x2, 0x5, 0xF, 0x7, 0x4, 0x9, 0xA, 0xB, 0xC, 0xD, 0xE, 0x3]),
    SBox([0x4, 0x0, 0x8, 0x2, 0x3, 0x5, 0x6, 0xB, 0x1, 0x9, 0xA, 0x7, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x4, 0x8, 0x1, 0x2, 0x3, 0x5, 0x6, 0x7, 0x0, 0x9, 0xA, 0xB, 0xE, 0xD, 0xC, 0xF]),
    SBox([0x4, 0x8, 0x1, 0x2, 0x3, 0x5, 0x6, 0x7, 0x0, 0x9, 0xA, 0xB, 0xC, 0xF, 0xE, 0xD]),
    SBox([0x8, 0x0, 0xF, 0x2, 0x3, 0x5, 0x4, 0x7, 0x6, 0x9, 0xA, 0xB, 0xC, 0xD, 0xE, 0x1]),
    SBox([0x8, 0x0, 0x1, 0x9, 0x2, 0x5, 0x6, 0x7, 0x4, 0xF, 0xA, 0xB, 0xC, 0xD, 0xE, 0x3]),
    SBox([0x4, 0x0, 0x1, 0x8, 0x2, 0x5, 0x6, 0x7, 0x3, 0x9, 0xD, 0xB, 0xC, 0xA, 0xE, 0xF]),
    SBox([0xB, 0x0, 0x1, 0x2, 0x3, 0x5, 0x6, 0x7, 0x4, 0x9, 0xA, 0xD, 0xC, 0x8, 0xE, 0xF]),
    SBox([0x2, 0x0, 0x1, 0x8, 0x3, 0xC, 0x6, 0x7, 0x4, 0x9, 0xA, 0xB, 0x5, 0xD, 0xE, 0xF]),
    SBox([0x4, 0x0, 0x1, 0x8, 0x2, 0xA, 0x6, 0x7, 0x3, 0x9, 0x5, 0xB, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x2, 0x0, 0x1, 0x8, 0x3, 0x5, 0x6, 0x7, 0x4, 0xD, 0xA, 0xB, 0xC, 0x9, 0xE, 0xF]),
    SBox([0x4, 0x0, 0x1, 0x8, 0x2, 0x5, 0x9, 0x7, 0x3, 0x6, 0xA, 0xB, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x4, 0x0, 0x8, 0x2, 0x3, 0xB, 0x6, 0x7, 0x1, 0x9, 0xA, 0x5, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x8, 0x0, 0x1, 0x2, 0x3, 0xE, 0x6, 0x7, 0x4, 0x9, 0xA, 0xB, 0xC, 0xD, 0x5, 0xF]),
    SBox([0x2, 0x0, 0x1, 0x3, 0x8, 0x4, 0x6, 0xE, 0x5, 0x9, 0xA, 0xB, 0xC, 0xD, 0x7, 0xF]),
    SBox([0x8, 0x0, 0x1, 0x2, 0x3, 0x5, 0x6, 0xC, 0x4, 0x9, 0xA, 0xB, 0x7, 0xD, 0xE, 0xF]),
    SBox([0x4, 0x0, 0x1, 0x8, 0x2, 0x5, 0x6, 0x7, 0x3, 0xC, 0xA, 0xB, 0x9, 0xD, 0xE, 0xF]),
    SBox([0x8, 0x0, 0x1, 0x2, 0x3, 0x5, 0x6, 0x7, 0x4, 0xD, 0xA, 0xB, 0xC, 0x9, 0xE, 0xF]),
    SBox([0x4, 0x0, 0x1, 0x8, 0x2, 0x5, 0x6, 0x7, 0x3, 0x9, 0xA, 0xE, 0xC, 0xD, 0xB, 0xF]),
    SBox([0x8, 0x0, 0x1, 0x2, 0x3, 0x5, 0x6, 0x7, 0x4, 0x9, 0xA, 0xF, 0xC, 0xD, 0xE, 0xB]),
    SBox([0x4, 0x0, 0x1, 0xF, 0x2, 0x5, 0x6, 0x7, 0x3, 0x9, 0xA, 0xB, 0xC, 0xD, 0xE, 0x8]),
    SBox([0x8, 0xE, 0x1, 0x2, 0x3, 0x5, 0x6, 0x7, 0x4, 0x9, 0xA, 0xB, 0xC, 0xD, 0x0, 0xF]),
    SBox([0x8, 0xF, 0x1, 0x2, 0x3, 0x5, 0x6, 0x7, 0x4, 0x9, 0xA, 0xB, 0xC, 0xD, 0xE, 0x0]),
    SBox([0x4, 0x0, 0x1, 0xD, 0x2, 0x5, 0x6, 0x7, 0x3, 0x9, 0xA, 0xB, 0xC, 0x8, 0xE, 0xF]),
    SBox([0x8, 0x0, 0x1, 0xD, 0x3, 0x5, 0x6, 0x7, 0x4, 0x9, 0xA, 0xB, 0xC, 0x2, 0xE, 0xF]),
    SBox([0x8, 0x0, 0x1, 0x2, 0x3, 0xC, 0x6, 0x7, 0x4, 0x9, 0xA, 0xB, 0x5, 0xD, 0xE, 0xF]),
    SBox([0x8, 0x0, 0x1, 0x2, 0x3, 0x5, 0x6, 0xE, 0x4, 0x9, 0xA, 0xB, 0xC, 0xD, 0x7, 0xF]),
    SBox([0x4, 0x8, 0xE, 0x2, 0x3, 0x5, 0x6, 0x7, 0x0, 0x9, 0xA, 0xB, 0xC, 0xD, 0x1, 0xF]),
    SBox([0x8, 0x0, 0x1, 0x2, 0x3, 0xA, 0x6, 0x7, 0x4, 0x9, 0x5, 0xB, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x4, 0x0, 0x8, 0x2, 0x3, 0x5, 0xB, 0x7, 0x1, 0x9, 0xA, 0x6, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x8, 0x0, 0x1, 0x2, 0x3, 0x5, 0x9, 0x7, 0x4, 0x6, 0xA, 0xB, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x8, 0x0, 0x1, 0x2, 0x3, 0x5, 0xB, 0x7, 0x4, 0x9, 0xA, 0x6, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x8, 0x6, 0x1, 0xD, 0x3, 0x5, 0x0, 0x7, 0x4, 0x9, 0xA, 0xB, 0xC, 0x2, 0xE, 0xF]),
    SBox([0x2, 0x0, 0x1, 0x3, 0x8, 0x4, 0xD, 0x7, 0x5, 0x9, 0xA, 0xB, 0xC, 0x6, 0xE, 0xF]),
    SBox([0x4, 0x0, 0x1, 0x8, 0x2, 0x5, 0xA, 0x7, 0x3, 0x9, 0x6, 0xB, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x8, 0x0, 0x1, 0x2, 0x3, 0x5, 0xD, 0x7, 0x4, 0x9, 0xA, 0xB, 0xC, 0x6, 0xE, 0xF]),
    SBox([0x4, 0x0, 0x1, 0x8, 0x2, 0x5, 0x6, 0x7, 0x3, 0x9, 0xA, 0xB, 0xE, 0xD, 0xC, 0xF]),
    SBox([0x8, 0x0, 0x1, 0x2, 0x3, 0x5, 0x6, 0x7, 0x4, 0x9, 0xE, 0xB, 0xC, 0xD, 0xA, 0xF]),
    SBox([0x8, 0xA, 0x1, 0x2, 0x3, 0x5, 0x6, 0x7, 0x4, 0x9, 0x0, 0xB, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x4, 0x0, 0x1, 0xA, 0x2, 0x5, 0x6, 0x7, 0x3, 0x9, 0x8, 0xB, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x4, 0x0, 0x1, 0x8, 0x2, 0x5, 0x3, 0x7, 0x6, 0x9, 0xA, 0xB, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x6, 0x0, 0x1, 0x8, 0x3, 0x5, 0x4, 0x7, 0x2, 0x9, 0xA, 0xB, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x6, 0x8, 0x1, 0x2, 0x3, 0x5, 0x4, 0x7, 0x0, 0x9, 0xA, 0xB, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x2, 0x0, 0x1, 0x3, 0xF, 0xB, 0x6, 0x7, 0x4, 0x9, 0xA, 0x5, 0xC, 0xD, 0xE, 0x8]),
    SBox([0x1, 0x0, 0x4, 0x8, 0x2, 0x5, 0xF, 0x7, 0x3, 0x9, 0xA, 0xB, 0xC, 0xD, 0xE, 0x6]),
    SBox([0x4, 0x0, 0x1, 0x8, 0x2, 0x5, 0x6, 0x7, 0x3, 0xE, 0xA, 0xB, 0xC, 0xD, 0x9, 0xF]),
    SBox([0x8, 0x0, 0x1, 0x3, 0x2, 0xA, 0x6, 0x7, 0x4, 0x9, 0x5, 0xB, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x8, 0x0, 0x1, 0xC, 0x2, 0x5, 0x6, 0x7, 0x4, 0x9, 0xA, 0xB, 0x3, 0xD, 0xE, 0xF]),
    SBox([0x8, 0x0, 0x1, 0xD, 0x2, 0x5, 0x6, 0x7, 0x4, 0x9, 0xA, 0xB, 0xC, 0x3, 0xE, 0xF]),
    SBox([0x1, 0x0, 0x4, 0x8, 0x2, 0xA, 0x6, 0x7, 0x3, 0x9, 0x5, 0xB, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x3, 0x0, 0x1, 0x2, 0x8, 0xC, 0x6, 0x7, 0x4, 0x9, 0xA, 0xB, 0x5, 0xD, 0xE, 0xF]),
    SBox([0x2, 0x0, 0x1, 0x3, 0x8, 0x4, 0xE, 0x7, 0x5, 0x9, 0xA, 0xB, 0xC, 0xD, 0x6, 0xF]),
    SBox([0x2, 0x0, 0x1, 0x3, 0x8, 0x4, 0xF, 0x7, 0x5, 0x9, 0xA, 0xB, 0xC, 0xD, 0xE, 0x6]),
    SBox([0x4, 0x0, 0x1, 0x8, 0x2, 0x5, 0x6, 0x7, 0x3, 0x9, 0xA, 0xB, 0xC, 0xF, 0xE, 0xD]),
    SBox([0xD, 0x0, 0x1, 0x2, 0x3, 0x5, 0x6, 0x7, 0x4, 0x9, 0xA, 0xB, 0xC, 0x8, 0xE, 0xF]),
    SBox([0xE, 0x0, 0x1, 0x2, 0x3, 0x5, 0x6, 0x7, 0x4, 0x9, 0xA, 0xB, 0xC, 0xD, 0x8, 0xF]),
    SBox([0x8, 0xD, 0x1, 0x2, 0x3, 0x5, 0x6, 0x7, 0x4, 0x9, 0xA, 0xB, 0xC, 0x0, 0xE, 0xF]),
    SBox([0x8, 0x0, 0xC, 0x2, 0x3, 0x5, 0x6, 0x7, 0x4, 0x9, 0xA, 0xB, 0x1, 0xD, 0xE, 0xF]),
    SBox([0x4, 0x0, 0x1, 0xC, 0x2, 0x5, 0x6, 0x7, 0x3, 0x9, 0xA, 0xB, 0x8, 0xD, 0xE, 0xF]),
    SBox([0x4, 0x0, 0x1, 0xE, 0x2, 0x5, 0x6, 0x7, 0x3, 0x9, 0xA, 0xB, 0xC, 0xD, 0x8, 0xF]),
    SBox([0x8, 0x0, 0x1, 0x3, 0x2, 0xF, 0x6, 0x7, 0x4, 0x9, 0xA, 0xB, 0xC, 0xD, 0xE, 0x5]),
    SBox([0x8, 0x6, 0x1, 0x2, 0x3, 0x5, 0x0, 0x7, 0x4, 0x9, 0xA, 0xB, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x8, 0x0, 0x1, 0xA, 0x2, 0x5, 0x6, 0x7, 0x4, 0x9, 0x3, 0xB, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x8, 0x0, 0x1, 0x6, 0x2, 0x5, 0x3, 0x7, 0x4, 0x9, 0xA, 0xB, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x8, 0x0, 0x1, 0x2, 0xC, 0x5, 0x4, 0x7, 0x6, 0x9, 0xA, 0xB, 0x3, 0xD, 0xE, 0xF]),
    SBox([0x8, 0x0, 0x1, 0x2, 0xD, 0x5, 0x4, 0x7, 0x6, 0x9, 0xA, 0xB, 0xC, 0x3, 0xE, 0xF]),
    SBox([0x2, 0x0, 0x1, 0x3, 0x8, 0xE, 0x6, 0x7, 0x4, 0x9, 0xA, 0xB, 0xC, 0xD, 0x5, 0xF]),
    SBox([0xC, 0x0, 0x1, 0x2, 0x3, 0x5, 0x6, 0x7, 0x4, 0x9, 0xA, 0xB, 0x8, 0xD, 0xE, 0xF]),
    SBox([0x8, 0x0, 0xD, 0x2, 0x3, 0x5, 0x6, 0x7, 0x4, 0x9, 0xA, 0xB, 0xC, 0x1, 0xE, 0xF]),
    SBox([0x2, 0x0, 0x1, 0x4, 0x3, 0x8, 0x6, 0x7, 0x5, 0x9, 0xA, 0xB, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x4, 0x0, 0x1, 0x2, 0x3, 0x5, 0x6, 0x8, 0x7, 0x9, 0xA, 0xB, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x4, 0x0, 0x1, 0x2, 0x3, 0x8, 0x6, 0x7, 0x5, 0x9, 0xA, 0xB, 0xC, 0xD, 0xE, 0xF]),
    SBox([0xF, 0x0, 0x1, 0x3, 0x2, 0x5, 0x6, 0x7, 0x4, 0x9, 0xA, 0xB, 0xC, 0xD, 0xE, 0x8]),
    SBox([0x2, 0x0, 0x1, 0x3, 0xF, 0x5, 0x6, 0x7, 0x4, 0x9, 0xA, 0xB, 0xC, 0xD, 0xE, 0x8]),
    SBox([0x2, 0x0, 0x1, 0x3, 0x8, 0xC, 0x6, 0x7, 0x4, 0x9, 0xA, 0xB, 0x5, 0xD, 0xE, 0xF]),
    SBox([0x7, 0x0, 0x1, 0x2, 0x3, 0x5, 0x6, 0x8, 0x4, 0x9, 0xA, 0xB, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x8, 0x0, 0x1, 0xF, 0x2, 0x5, 0x6, 0x7, 0x4, 0x9, 0xA, 0xB, 0xC, 0xD, 0xE, 0x3]),
    SBox([0x2, 0x0, 0x1, 0x8, 0x3, 0x5, 0x6, 0x7, 0x4, 0x9, 0xA, 0xB, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x4, 0x8, 0x1, 0x2, 0x3, 0x5, 0x6, 0x7, 0x0, 0x9, 0xA, 0xB, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x4, 0x0, 0x1, 0x8, 0x2, 0x5, 0x6, 0x7, 0x3, 0x9, 0xA, 0xB, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x8, 0x0, 0x1, 0x3, 0x2, 0x5, 0x6, 0x7, 0x4, 0x9, 0xA, 0xB, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x4, 0x0, 0x8, 0x2, 0x3, 0x5, 0xA, 0x7, 0x1, 0x9, 0x6, 0xB, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x4, 0x0, 0x8, 0x2, 0x3, 0x5, 0x6, 0x7, 0x1, 0x9, 0xA, 0xB, 0xE, 0xD, 0xC, 0xF]),
    SBox([0x6, 0x0, 0x8, 0x2, 0x3, 0x5, 0xA, 0x7, 0x1, 0x9, 0x4, 0xB, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x6, 0x0, 0x8, 0x2, 0x3, 0x5, 0x4, 0x7, 0x1, 0x9, 0xA, 0xB, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x8, 0x0, 0x1, 0x2, 0x3, 0x5, 0x4, 0x7, 0x6, 0x9, 0xA, 0xB, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x2, 0x0, 0x1, 0x3, 0x8, 0x4, 0xC, 0x7, 0x5, 0x9, 0xA, 0xB, 0x6, 0xD, 0xE, 0xF]),
    SBox([0x3, 0x0, 0x1, 0x2, 0x8, 0x5, 0xC, 0x7, 0x4, 0x9, 0xA, 0xB, 0x6, 0xD, 0xE, 0xF]),
    SBox([0x3, 0x0, 0x1, 0x2, 0xC, 0x5, 0x6, 0x7, 0x4, 0x9, 0xA, 0xB, 0x8, 0xD, 0xE, 0xF]),
    SBox([0x4, 0x0, 0x1, 0x2, 0x3, 0x5, 0x8, 0x7, 0x6, 0x9, 0xA, 0xB, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x4, 0x0, 0x8, 0x2, 0x3, 0x5, 0x6, 0x7, 0x1, 0x9, 0xA, 0xB, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x8, 0x0, 0x1, 0x2, 0x3, 0x5, 0x6, 0x7, 0x4, 0x9, 0xA, 0xB, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x8, 0x0, 0xA, 0x2, 0x3, 0x5, 0x4, 0x7, 0x6, 0x9, 0x1, 0xB, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x8, 0x0, 0x6, 0x2, 0x3, 0x5, 0x1, 0x7, 0x4, 0x9, 0xA, 0xB, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x4, 0x0, 0x1, 0x3, 0x2, 0x5, 0x6, 0x7, 0xE, 0x8, 0xA, 0x9, 0xC, 0xD, 0xB, 0xF]),
    SBox([0x4, 0x0, 0x1, 0x3, 0x2, 0x5, 0x6, 0x7, 0xD, 0x9, 0x8, 0xB, 0xC, 0xA, 0xE, 0xF]),
    SBox([0x4, 0x0, 0x1, 0x3, 0x2, 0x5, 0x6, 0x7, 0xA, 0xC, 0x8, 0xB, 0x9, 0xD, 0xE, 0xF]),
    SBox([0x3, 0x0, 0x1, 0x2, 0x8, 0xB, 0x6, 0x7, 0x4, 0x9, 0xA, 0x5, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x1, 0x0, 0x4, 0x8, 0x2, 0x5, 0xA, 0x7, 0x3, 0x9, 0x6, 0xB, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x4, 0x0, 0x1, 0x2, 0x3, 0x5, 0x6, 0x7, 0xE, 0x9, 0x8, 0xB, 0xC, 0xD, 0xA, 0xF]),
    SBox([0x4, 0x0, 0x1, 0x2, 0x3, 0x5, 0x6, 0x7, 0xF, 0x9, 0x8, 0xB, 0xC, 0xD, 0xE, 0xA]),
    SBox([0x4, 0x0, 0x1, 0x3, 0x2, 0x5, 0x6, 0x7, 0xA, 0xF, 0x9, 0xB, 0xC, 0xD, 0xE, 0x8]),
    SBox([0x4, 0x0, 0x1, 0x2, 0x3, 0x5, 0x6, 0x7, 0xC, 0x9, 0x8, 0xB, 0xA, 0xD, 0xE, 0xF]),
    SBox([0x4, 0x0, 0x1, 0x2, 0x3, 0x5, 0x6, 0x7, 0xD, 0x9, 0x8, 0xB, 0xC, 0xA, 0xE, 0xF]),
    SBox([0x4, 0x0, 0x1, 0x3, 0x2, 0x5, 0x6, 0x7, 0xA, 0xD, 0x9, 0xB, 0xC, 0x8, 0xE, 0xF]),
    SBox([0x6, 0x0, 0x1, 0x2, 0x3, 0x5, 0x4, 0x7, 0xC, 0x9, 0xA, 0xB, 0x8, 0xD, 0xE, 0xF]),
    SBox([0x6, 0x0, 0x1, 0x2, 0x3, 0x5, 0x4, 0x7, 0xD, 0x9, 0xA, 0xB, 0xC, 0x8, 0xE, 0xF]),
    SBox([0x6, 0x5, 0x1, 0x2, 0x3, 0x0, 0x4, 0x7, 0x8, 0x9, 0xA, 0xB, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x2, 0x0, 0x1, 0x3, 0x8, 0xA, 0x6, 0x7, 0x4, 0x9, 0x5, 0xB, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x2, 0x0, 0x1, 0x4, 0x3, 0x5, 0x6, 0x7, 0xC, 0x9, 0xA, 0xB, 0x8, 0xD, 0xE, 0xF]),
    SBox([0x4, 0x0, 0x1, 0x2, 0x3, 0x5, 0x6, 0x7, 0xD, 0x9, 0xA, 0xB, 0xC, 0x8, 0xE, 0xF]),
    SBox([0x4, 0x0, 0x1, 0x2, 0x3, 0x5, 0x6, 0x7, 0xE, 0x9, 0xA, 0xB, 0xC, 0xD, 0x8, 0xF]),
    SBox([0x4, 0x0, 0x1, 0x3, 0x2, 0x5, 0x6, 0x7, 0xE, 0x8, 0xA, 0xB, 0xC, 0xD, 0x9, 0xF]),
    SBox([0x4, 0x0, 0x1, 0x3, 0x2, 0x5, 0x6, 0x7, 0xB, 0x8, 0xA, 0x9, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x4, 0x0, 0x1, 0x3, 0x2, 0x5, 0x6, 0x7, 0xF, 0x9, 0xA, 0xB, 0xC, 0xD, 0xE, 0x8]),
    SBox([0x2, 0x0, 0x1, 0x3, 0x5, 0x4, 0x6, 0x7, 0xE, 0x9, 0xA, 0xB, 0xC, 0xD, 0x8, 0xF]),
    SBox([0x4, 0x0, 0x1, 0x2, 0x3, 0x5, 0x6, 0x7, 0xC, 0x9, 0xA, 0xB, 0x8, 0xD, 0xE, 0xF]),
    SBox([0x4, 0x0, 0x1, 0x2, 0x3, 0x5, 0x6, 0x7, 0xF, 0x9, 0xA, 0xB, 0xC, 0xD, 0xE, 0x8]),
    SBox([0x1, 0x0, 0x4, 0x6, 0x2, 0x5, 0x3, 0x7, 0xA, 0x9, 0x8, 0xB, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x4, 0x0, 0x1, 0x3, 0x2, 0x5, 0x6, 0x7, 0xD, 0x8, 0x9, 0xB, 0xC, 0xA, 0xE, 0xF]),
    SBox([0x3, 0x0, 0x1, 0x2, 0x8, 0xA, 0x6, 0x7, 0x4, 0x9, 0x5, 0xB, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x4, 0x0, 0x1, 0x3, 0x2, 0x5, 0x6, 0x7, 0xA, 0x9, 0x8, 0xB, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x2, 0x0, 0x1, 0x3, 0x8, 0x4, 0x6, 0x7, 0x5, 0x9, 0xA, 0xB, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x1, 0x0, 0x4, 0x8, 0x2, 0x5, 0x6, 0x7, 0x3, 0x9, 0xA, 0xB, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x2, 0x0, 0x1, 0x3, 0x8, 0x5, 0x6, 0x7, 0x4, 0x9, 0xA, 0xB, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x4, 0x6, 0x1, 0x2, 0x3, 0x5, 0x0, 0x7, 0x8, 0x9, 0xA, 0xB, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x4, 0x0, 0x1, 0x6, 0x2, 0x5, 0x3, 0x7, 0x8, 0x9, 0xA, 0xB, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x4, 0x0, 0x1, 0x3, 0x2, 0x5, 0x6, 0x7, 0x9, 0x8, 0xA, 0xB, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x1, 0x0, 0x4, 0x6, 0x2, 0x5, 0x3, 0x7, 0x8, 0x9, 0xA, 0xB, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x2, 0x0, 0x1, 0x4, 0x3, 0x5, 0x6, 0x7, 0x8, 0x9, 0xA, 0xB, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x4, 0x0, 0x1, 0x3, 0x2, 0x5, 0x6, 0x7, 0x8, 0x9, 0xA, 0xB, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x4, 0x0, 0x1, 0x2, 0x3, 0x5, 0x6, 0x7, 0xA, 0x9, 0x8, 0xB, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x4, 0x0, 0x1, 0x3, 0x2, 0x5, 0x6, 0x7, 0xA, 0x8, 0x9, 0xB, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x6, 0x0, 0x1, 0x2, 0x3, 0x5, 0x4, 0x7, 0x8, 0x9, 0xA, 0xB, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x3, 0x0, 0x1, 0x2, 0x8, 0x5, 0x6, 0x7, 0x4, 0x9, 0xA, 0xB, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x4, 0x0, 0x1, 0x2, 0x3, 0x5, 0x6, 0x7, 0x8, 0x9, 0xA, 0xB, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x1, 0x0, 0x4, 0x3, 0x2, 0x5, 0x6, 0x7, 0xF, 0x9, 0xA, 0xB, 0xC, 0xD, 0xE, 0x8]),
    SBox([0x2, 0x0, 0x1, 0x3, 0x5, 0x4, 0x6, 0x7, 0xA, 0x9, 0x8, 0xB, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x2, 0x0, 0x1, 0x3, 0x7, 0x4, 0x6, 0x5, 0x8, 0x9, 0xA, 0xB, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x1, 0x0, 0x4, 0x3, 0x2, 0x5, 0x6, 0x7, 0x8, 0x9, 0xA, 0xB, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x2, 0x0, 0x1, 0x3, 0x5, 0x4, 0x6, 0x7, 0x8, 0x9, 0xA, 0xB, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x2, 0x0, 0x1, 0x3, 0x4, 0x5, 0x6, 0x7, 0x8, 0x9, 0xA, 0xB, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x8, 0x0, 0xA, 0x2, 0x3, 0x5, 0x1, 0x7, 0x4, 0x9, 0x6, 0xB, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x4, 0x0, 0x1, 0x3, 0x2, 0x5, 0x6, 0x7, 0xC, 0x8, 0x9, 0xB, 0xA, 0xD, 0xE, 0xF]),
    SBox([0x4, 0x0, 0x6, 0x2, 0x3, 0x5, 0x1, 0x7, 0xA, 0x9, 0x8, 0xB, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x4, 0x0, 0x6, 0x2, 0x3, 0x5, 0x1, 0x7, 0x8, 0x9, 0xA, 0xB, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x2, 0x0, 0x1, 0x3, 0x6, 0x4, 0x5, 0x7, 0x8, 0x9, 0xA, 0xB, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x3, 0x0, 0x1, 0x2, 0x6, 0x5, 0x4, 0x7, 0x8, 0x9, 0xA, 0xB, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x3, 0x0, 0x1, 0x2, 0x4, 0x5, 0x6, 0x7, 0x8, 0x9, 0xA, 0xB, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x1, 0x0, 0x2, 0x3, 0x4, 0x5, 0x6, 0x7, 0x8, 0x9, 0xA, 0xB, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x1, 0x0, 0x3, 0x2, 0x4, 0x5, 0x6, 0x7, 0x8, 0x9, 0xA, 0xB, 0xC, 0xD, 0xE, 0xF]),
    SBox([0x0, 0x1, 0x2, 0x3, 0x4, 0x5, 0x6, 0x7, 0x8, 0x9, 0xA, 0xB, 0xC, 0xD, 0xE, 0xF])
]

# Dictionary of all available SBoxes
sboxes = {}
for k in dir(sys.modules[__name__]):
    v = getattr(sys.modules[__name__], k)
    if isinstance(v, SBox):
        sboxes[k] = v
