//roarfilt.c:

/*
 *      Copyright (C) Philipp 'ph3-der-loewe' Schafft - 2008-2019
 *
 *  This file is part of roarclients a part of RoarAudio,
 *  a cross-platform sound system for both, home and professional use.
 *  See README for details.
 *
 *  This file is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 3
 *  as published by the Free Software Foundation.
 *
 *  RoarAudio is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this software; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 *
 */

#include <roaraudio.h>
#include <libroardsp/libroardsp.h>

#ifdef ROAR_HAVE_LIBM
#include <math.h>
#endif

#define BUFFERSIZE 1024

void usage (void) {
 printf("roarfilt [OPTIONS]...\n");

 printf("\nOptions:\n\n");

 printf("  --server SERVER     - Set server hostname\n"
        "  --rate  -R RATE     - Set sample rate\n"
        "  --bits  -B BITS     - Set bits per sample\n"
        "  --chans -C CHANNELS - Set number of channels\n"
        "  --aiprofile PROFILE - Set audio profile\n"
        "  --help              - Show this help\n"
        "\n"
        "  --half              - half the volume\n"
        "  --double            - double the volume\n"
        "  --amp VAL           - Set amplification\n"
        "  --mul VAL           - Set mul\n"
        "  --div VAL           - Set div\n"
        "  --filter  name      - add filter name\n"
        "  --ffreq   freq      - set filter freq\n"
        "  --fmul    mult      - set filter multiplier\n"
        "  --fdiv    div       - set filter divider\n"
        "  --fn      N         - set filter N parameter\n"
        "  --flimit  limit     - set filter limit parameter\n"
        "  --fmode   mode      - set filter mode parameter\n"
        "  --fq      Q         - set filter quality\n"
       );

}

#define _volX(bits,twobits,sf) \
void vol##bits (void * data, int32_t mul, int32_t div, size_t len) { \
 int##bits##_t * samples = (int##bits##_t *) data; \
 size_t i; \
\
 len /= sf; \
\
 for (i = 0; i < len; i++) \
  samples[i] = ((int##twobits##_t) samples[i] * mul) / div; \
}

_volX( 8,16,1)
_volX(16,32,2)
_volX(32,64,4)

int main (int argc, char * argv[]) {
 struct roar_audio_info info;
 const char  * server   = NULL;
 const char  * k;
 int     i;
 int32_t mul = 1, div = 1;
 int     filter_id;
 int32_t tmp;
 float   tmpfp;
 char    buf[BUFFERSIZE];
 struct roardsp_filterchain fc;
 struct roardsp_filter      filter_real[8];
 struct roardsp_filter    * filter = filter_real - 1;
 struct roar_stream         stream;
 struct roar_vio_calls      svio;

 _LIBROAR_IGNORE_RET(roar_profile2info(&info, "default-server"));
 info.codec = ROAR_CODEC_DEFAULT;

 roardsp_fchain_init(&fc);

 for (i = 1; i < argc; i++) {
  k = argv[i];

  if ( strcmp(k, "--server") == 0 || strcmp(k, "-s") == 0 ) {
   ROAR_CKHAVEARGS(1);
   server = argv[++i];
  } else if ( strcmp(k, "--rate") == 0 || strcmp(k, "-R") == 0 || strcmp(k, "-r") == 0 ) {
   ROAR_CKHAVEARGS(1);
   info.rate = roar_str2rate(argv[++i]);
  } else if ( strcmp(k, "--bits") == 0 || strcmp(k, "-B") == 0 ) {
   ROAR_CKHAVEARGS(1);
   info.bits = roar_str2bits(argv[++i]);
  } else if ( strcmp(k, "--channels") == 0 || strcmp(k, "--chans") == 0 || strcmp(k, "-C") == 0 ) {
   ROAR_CKHAVEARGS(1);
   info.channels = roar_str2channels(argv[++i]);
  } else if ( strcmp(k, "-b") == 0 ) {
   info.bits = 8;
  } else if ( strcmp(k, "-m") == 0 ) {
   info.channels = 1;
  } else if ( !strcmp(k, "--aiprofile") ) {
   ROAR_CKHAVEARGS(1);
   if ( roar_profile2info(&info, argv[++i]) == -1 ) {
    fprintf(stderr, "Error: Can not load audio profile: %s: %s\n", argv[i], roar_error2str(roar_error));
    return 1;
   }
   info.codec = ROAR_CODEC_DEFAULT;
  } else if ( strcmp(k, "--half") == 0 || strcmp(k, "-half") == 0 ) {
   div *= 2;
  } else if ( strcmp(k, "--double") == 0 || strcmp(k, "-double") == 0 ) {
   mul *= 2;
  } else if ( strcmp(k, "--amp") == 0 ) {
   ROAR_CKHAVEARGS(1);
   mul *= atoi(argv[++i]);
  } else if ( strcmp(k, "--mul") == 0 ) {
   ROAR_CKHAVEARGS(1);
   mul  = atoi(argv[++i]);
  } else if ( strcmp(k, "--div") == 0 ) {
   ROAR_CKHAVEARGS(1);
   div  = atoi(argv[++i]);
  } else if ( strcmp(k, "--filter") == 0 ) {
   ROAR_CKHAVEARGS(1);
   stream.info = info;
   filter_id = roardsp_filter_str2id(argv[++i]);
   if ( filter_id == -1 ) {
    ROAR_WARN("Can not add filter as filter ID is unknown: %s: %s", argv[i], roar_error2str(roar_error));
   } else {
    filter++;
    if ( roardsp_filter_init(filter, &stream, filter_id) == -1 ) {
     ROAR_WARN("Can not add filter: %s: %s", argv[i], roar_error2str(roar_error));
     filter--;
    } else {
     roardsp_fchain_add(&fc, filter);
    }
   }
  } else if ( strcmp(k, "--ffreq") == 0 ) {
   ROAR_CKHAVEARGS(1);
   tmpfp = atof(argv[++i]);
   roardsp_filter_ctl(filter, ROARDSP_FCTL_FREQ, &tmpfp);
  } else if ( strcmp(k, "--fmul") == 0 ) {
   ROAR_CKHAVEARGS(1);
   tmp = atoi(argv[++i]);
   roardsp_filter_ctl(filter, ROARDSP_FCTL_MUL, &tmp);
  } else if ( strcmp(k, "--fdiv") == 0 ) {
   ROAR_CKHAVEARGS(1);
   tmp = atoi(argv[++i]);
   roardsp_filter_ctl(filter, ROARDSP_FCTL_DIV, &tmp);
  } else if ( strcmp(k, "--fn") == 0 ) {
   ROAR_CKHAVEARGS(1);
   tmp = atoi(argv[++i]);
   roardsp_filter_ctl(filter, ROARDSP_FCTL_N, &tmp);
  } else if ( strcmp(k, "--fq") == 0 ) {
   ROAR_CKHAVEARGS(1);
   tmp = atoi(argv[++i]);
   roardsp_filter_ctl(filter, ROARDSP_FCTL_Q, &tmp);
  } else if ( strcmp(k, "--flimit") == 0 ) {
   ROAR_CKHAVEARGS(1);
   tmp = atoi(argv[++i]);
   roardsp_filter_ctl(filter, ROARDSP_FCTL_LIMIT, &tmp);
  } else if ( strcmp(k, "--fmode") == 0 ) {
   ROAR_CKHAVEARGS(1);
   tmp = atoi(argv[++i]);
   roardsp_filter_ctl(filter, ROARDSP_FCTL_MODE, &tmp);
  } else if ( strcmp(k, "--help") == 0 || strcmp(k, "-h") == 0 ) {
   usage();
   return 0;
  } else {
   fprintf(stderr, "Error: unknown argument: %s\n", k);
   usage();
   return 1;
  }
 }

 if ( roar_vio_simple_stream(&svio,
                             info.rate, info.channels, info.bits, info.codec,
                             server, ROAR_DIR_FILTER, "roarfilt", -1) == -1 ) {
  fprintf(stderr, "Error: can not start playback\n");
  return 1;
 }

 if ( mul == div &&
      roardsp_fchain_num(&fc) == 0 ) {
  fprintf(stderr, "Error: filter is useless!\n");
  return 0;
 }

 switch (info.bits) {
  case 8:
    while((i = roar_vio_read(&svio, buf, sizeof(buf)))) {
     vol8((void*)buf, mul, div, i);
     roardsp_fchain_calc(&fc, (void*)buf, (8*i)/info.bits);
     if (roar_vio_write(&svio, buf, i) != i)
      break;
    }
   break;
  case 16:
    while((i = roar_vio_read(&svio, buf, sizeof(buf)))) {
     if ( mul != div )
      vol16((void*)buf, mul, div, i);
     roardsp_fchain_calc(&fc, (void*)buf, (8*i)/info.bits);
     if (roar_vio_write(&svio, buf, i) != i)
      break;
    }
   break;
  case 32:
    while((i = roar_vio_read(&svio, buf, sizeof(buf)))) {
     vol32((void*)buf, mul, div, i);
     roardsp_fchain_calc(&fc, (void*)buf, (8*i)/info.bits);
     if (roar_vio_write(&svio, buf, i) != i)
      break;
    }
   break;
  default:
    fprintf(stderr, "Error: %i bits per sample is not supported!\n", (int)info.bits);
    return 1;
 }

 roar_vio_close(&svio);

 roardsp_fchain_uninit(&fc);

 return 0;
}

//ll
