# ifndef _SKIT_CSR_ALGO_AMUD_H
# define _SKIT_CSR_ALGO_AMUD_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================
//
// CSR: Compressed Sparse Row format
//
// csr*diag
//
// author: Pierre.Saramito@imag.fr
//
// date: 24 september 1997
//
//@!\vfill\listofalgorithms

/*@! 
 \vfill \pagebreak \mbox{} \vfill \begin{algorithm}[h]
  \caption{{\tt amuld}: csr*diag sparse product}
  \begin{algorithmic}
    \INPUT {the diagonal and the matrix in CSR format}
      ia(0:nrow), ja(0:nnz-1), a(0:nnz-1),
      d(0:ncol-1)
    \ENDINPUT
    \OUTPUT {the result in CSR format}
      ib(0:nrow), jb(0:nnzl-1), b(0:nnzl-1)
    \ENDOUTPUT
    \NOTE {the algorithm is in-place}
    \ENDNOTE
    \BEGIN 
      {\em first: copy ia into ib} \\
      \FORTO {i := 0} {nrow}
	  ib(i) := ia(i)
      \ENDFOR
      
      {\em second: copy ja into jb} \\
      \FORTO {p := 0} {nnz-1}
	  jb(p) := ja(p)
      \ENDFOR
	
      {\em third: compute b from a and d} \\
      \FORTO {i := 0} {nrow-1}
        \FORTO {p := ia(i)} {ia(i+1)-1}
	    b (p) := a(p) * d(ja(p))
        \ENDFOR
      \ENDFOR
    \END
 \end{algorithmic} \end{algorithm}
 \vfill \pagebreak \mbox{} \vfill
*/

namespace rheolef { 

// third pass: compute b from a and d
template <
    class InputIterator1, 
    class InputIterator2, 
    class InputIterator3, 
    class RandomAcessIterator,
    class OutputIterator>
void
amuld (
    InputIterator1      ia,
    InputIterator1      last_ia,
    InputIterator2      ja,
    InputIterator3      first_a,
    RandomAcessIterator d,
    OutputIterator      b)
{
    InputIterator3 a = first_a + (*ia++);
    while (ia != last_ia) {
        InputIterator3 last_a = first_a + (*ia++);
	while (a != last_a)
	    *b++ = (*a++) * d [*ja++];
    }
}
//@!\vfill
}// namespace rheolef
# endif // _SKIT_CSR_ALGO_AMUD_H
