#ifndef _RHEO_BASIS_OPTION_H
#define _RHEO_BASIS_OPTION_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================
//
// author: Pierre.Saramito@imag.fr
//
// date: 2 september 2017
//
#include "rheolef/smart_pointer.h"
#include "rheolef/reference_element.h"

namespace rheolef { 

/*Class:quadrature
NAME: @code{basis_option} - options of the finite element space
@cindex  finite element
@clindex space
DESCRIPTION:
  @noindent
  The @code{basis_option} class is used to set options for
  the @code{space} constructor of the finite element space (@pxref{space class}).
  These options are directly transmitted to the @code{basis}
  class when computing the polynomial basis (@pxref{basis class} and @ref{basis command}).
  There are two main options: Lagrange nodes and raw polynomial basis.

LAGRANGE NODES:
  @noindent
  There are two possible node sets.
  The @code{equispaced}, refers to an equispaced
  set of Lagrange node during interpolation.
  The @code{warburton} one, refers to a non-equispaced
  optimized set of Lagrange nodes, suitable for high order
  polynomials (see Warburton, 2006, J. Eng. Math.).
  With this choice, the interpolation error is dramatically
  decreased for high order polynomials.
  The default is the @code{equispaced} node set,
  for backward compatibility purpose.

CONTINUOUS FEATURE:
  @noindent
  Some elements are possibly continuous.
  For instance, @code{Pk} elements family, with k >= 1,
  possibly have continuous interelements reconnections
  and thus, could define a piecewise polynomial
  and globaly continuous function.
  These basis families can be used both as continuous or discontinuous elements.
  Nevertheless, this choice should be done when building the basis:
  the information is transmitted during basis construction
  by sending the @code{basis_option} class
  and setting appropriately this feature
  with the @code{set_continuous(c)} member function.

RAW POLYNOMIAL BASIS:
  @noindent
  The raw (or initial) basis is used for building the 
  Lagrange basis, via the Vandermonde matrix and its inverse.
  Its choice do not change the definition of the FEM basis,
  but only the way it is builded.
  There are three possible raw basis.
  The @code{monomial} basis is the simplest choice,
  suitable for low order polynomials (less than five).
  For higher order polynomial, the 
  Vandermonde matrix becomes ill-conditionned and
  its inverse leads to errors in double precision.
  The @code{dubiner} basis (see Dubiner, 1991 J. Sci. Comput.)
  leads to better condition number.
  The @code{bernstein} basis could also be used as an alternative raw basis.
  The default is the @code{dubiner} raw basis.

STRING OPTION REPRESENTATION:
  The @code{is_option(string)} and @code{set(string)} members
  leads to easy setting of combined options at run time.
  By this way, options can be specified, together with basis basename,
  on the command line or from a file.

  @noindent
  The @code{stamp()} member returns an unique string.
  This string is used for specifying basis options,
  e.g. on command line or in files.
  This string is empty when all options are set to default values.
  Otherwise, it returns a comma separated list of options,
  enclosed by braces, specifying only non-default options.
  For instance, combining Warburton node set and Dubiner raw
  polynomials leads to @code{"@{warburton@}"}.
  Also, combining Warburton node set and Bernstein raw
  polynomials leads to @code{"@{warburton,bernstein@}"}.

  @noindent
  Note that the continuous or discontinuous feature
  is not specified by the @code{stamp()} string:
  it will be specified into the basis basename, by
  appending a @code{"d"} letter, as in @code{"P6d@{warburton@}"}.

NOTES:
  There are two distinct kind of polynomial basis:
  the raw basis and the finite element one.
  (@pxref{basis class} and @ref{basis command}).
  When using the @code{Pk} Lagrange finite element basis,
  these options are used to transform from one raw (initial)
  polynomial basis to the Lagrange one, based on a node set.
  When using an alternative finite element basis, e.g.
  the spectral @code{Sk} or the Bernstein @code{Bk},
  these options do not have any effect.

End:
*/

//<doc:
class basis_option {
public:
// typedefs:

  typedef size_t size_type;

  typedef enum {
    equispaced = 0,
    warburton  = 1,
    fekete     = 2,
    max_node   = 3
  } node_type; // update also node_name[]

  typedef enum {
    monomial	       = 0,
    bernstein	       = 1,
    dubiner 	       = 2,
    max_raw_polynomial = 3
  } raw_polynomial_type; // update also raw_polynomial_name[]

  static const node_type           default_node           = basis_option::equispaced;
  static const raw_polynomial_type default_raw_polynomial = basis_option::dubiner;

// allocators:

  basis_option(
    node_type           nt = default_node,
    raw_polynomial_type pt = default_raw_polynomial);

  basis_option (const basis_option& sopt);
  basis_option& operator= (const basis_option& sopt);

// accessors & modifiers:

  node_type           get_node() const;
  raw_polynomial_type get_raw_polynomial() const;
  std::string         get_node_name() const;
  std::string         get_raw_polynomial_name() const;
  bool                is_continuous() const;
  bool	  	      is_discontinuous() const;

  void set_node           (node_type type);
  void set_raw_polynomial (raw_polynomial_type type);
  void set                (std::string option_name);
  void set_node           (std::string node_name);
  void set_raw_polynomial (std::string raw_polynomial_name);
  void set_continuous     (bool c = true);
  void set_discontinuous  (bool c = false);

  bool is_node_name           (std::string name) const;
  bool is_raw_polynomial_name (std::string name) const;
  bool is_option_name         (std::string name) const;

  std::string stamp() const;

// data:
protected:
  node_type            _node;
  raw_polynomial_type  _poly;
  bool                 _is_continuous;
};
//>doc:
// ------------------------------------------------------------
// inlined
// ------------------------------------------------------------
inline 
basis_option::basis_option(
        node_type           nt,
        raw_polynomial_type pt)
 : _node(nt),
   _poly(pt),
  _is_continuous(true)
{
}
inline
basis_option::basis_option (const basis_option& sopt)
 : _node(sopt._node),
   _poly(sopt._poly),
   _is_continuous(sopt._is_continuous)
{
}
inline
basis_option&
basis_option::operator= (const basis_option& sopt)
{
  _node          = sopt._node;
  _poly          = sopt._poly;
  _is_continuous = sopt._is_continuous;
  return *this;
}
inline
basis_option::node_type
basis_option::get_node () const
{
    return _node;
}
inline
basis_option::raw_polynomial_type
basis_option::get_raw_polynomial () const
{
    return _poly;
}
inline
bool
basis_option::is_continuous() const
{
  return  _is_continuous;
}
inline
bool
basis_option::is_discontinuous() const
{
  return !_is_continuous;
}
inline
void
basis_option::set_node (node_type nt)
{
    _node = nt;
}
inline
void
basis_option::set_raw_polynomial (raw_polynomial_type pt)
{
    _poly = pt;
}
inline
void
basis_option::set_continuous (bool c)
{
  _is_continuous = c;
}
inline
void
basis_option::set_discontinuous (bool c)
{
  _is_continuous = c;
}

}// namespace rheolef
#endif // _RHEO_BASIS_OPTION_H
