/*
 * Maildir folder access
 *
 * Copyright (C) 2004--2013  Enrico Zini <enrico@debian.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */

#include "config.h"
#include <buffy/mailfolder/maildir.h>
#include <wibble/exception.h>
#include <wibble/sys/fs.h>
#include <wibble/string.h>
#include <buffy/utils/system.h>

#include <sys/types.h>  // stat, opendir, readdir, utimes
#include <sys/stat.h>   // stat
#include <dirent.h>		// opendir, readdir
#include <cstring>
#include <sys/time.h>	// utimes

#include <cerrno>
#include <cstring>

using namespace std;
using namespace wibble;

namespace buffy {
namespace mailfolder {

static string concat(const std::string& str1, char sep, const std::string& str2)
{
	// Avoid adding a separator if str1 or str are empty, if str1 ends with the
	// separator or if str2 starts with the separator
	if (!str1.empty() && str1[str1.size() - 1] != sep && ! str2.empty() && str2[0] != sep)
		return str1 + sep + str2;
	else
		return str1 + str2;
}

Maildir::Maildir(const std::string& path) throw ()
	: _path(path), _stat_total(-1), _stat_unread(-1), _stat_new(-1), _stat_flagged(-1),
	  _deleted(false), _new_mtime(0), _cur_mtime(0)
{
	_name = _path;

	/// Normalize the folder name

	// Remove trailing '/'
	while (_name[_name.size() - 1] == '/')
		_name.resize(_name.size() - 1);

	// Remove leading path
	size_t lastslash = _name.find_last_of('/');
	if (lastslash != string::npos)
		_name = _name.substr(lastslash + 1);

	// Remove leading dot
	if (_name[0] == '.')
		_name = _name.substr(1);
}

Maildir::Maildir(const std::string& name, const std::string& path) throw ()
	: _name(name), _path(path), _stat_total(-1), _stat_unread(-1), _stat_new(-1), _stat_flagged(-1),
	  _deleted(false), _new_mtime(0), _cur_mtime(0) {}


bool Maildir::changed()
{
	// Compute 'new' and 'cur' directory names
	string path_new = _path + "/new";
	string path_cur = _path + "/cur";

    std::auto_ptr<struct stat> st_new = sys::fs::stat(path_new);
    if (!st_new.get())
        if (! _deleted)
        {
            _deleted = true;
            return true;
        }

    std::auto_ptr<struct stat> st_cur = sys::fs::stat(path_cur);
    if (!st_cur.get())
        if (! _deleted)
        {
            _deleted = true;
            return true;
        }

	if (_deleted)
	{
		_deleted = false;
		return true;
	}

    return st_new->st_mtime > _new_mtime || st_cur->st_mtime > _cur_mtime;
}

void Maildir::updateStatistics()
{
	int res_total = 0;
	int res_unread = 0;
	int res_new = 0;
	int res_flagged = 0;

	// Compute 'new' and 'cur' directory names
	string path_new = _path + "/new";
	string path_cur = _path + "/cur";

    // Perform consistency checks on the 'new' directory
    std::auto_ptr<struct stat> st_new = sys::fs::stat(path_new);
    if (!st_new.get())
    {
		_stat_total = 0;
		_stat_unread = 0;
		_stat_new = 0;
		_stat_flagged = 0;
		_deleted = true;
		return;
    }
    if (S_ISDIR(st_new->st_mode) == 0)
        throw wibble::exception::Consistency(path_new + " is not a directory");

    // Perform consistency checks on the 'cur' directory
    std::auto_ptr<struct stat> st_cur = sys::fs::stat(path_cur);
    if (!st_cur.get())
    {
		_stat_total = 0;
		_stat_unread = 0;
		_stat_new = 0;
		_stat_flagged = 0;
		_deleted = true;
		return;
    }
    if (S_ISDIR(st_cur->st_mode) == 0)
        throw wibble::exception::Consistency(path_cur + " is not a directory");

	if (_deleted)
		_deleted = false;

    _new_mtime = st_new->st_mtime;
    _cur_mtime = st_cur->st_mtime;

    /// Count messages in the 'new' directory

    // Count the files in the 'new' directory
    {
        sys::fs::Directory dir(path_new);
        for (sys::fs::Directory::const_iterator d = dir.begin(); d != dir.end(); ++d)
        {
            if ((*d)[0] == '.')
                continue;
            res_total++;
            res_new++;
        }
    }

    // Restore the access time of the mailbox for other checking programs
    struct timeval t[2];
    t[0].tv_sec = st_new->st_atime;
    t[0].tv_usec = 0;
    t[1].tv_sec = st_new->st_mtime;
    t[1].tv_usec = 0;
    utimes(path_new.c_str(), t);


    /// Count messages in the 'cur' directory

    // Count the files in the 'cur' directory
    {
        sys::fs::Directory dir(path_cur);
        for (sys::fs::Directory::const_iterator d = dir.begin(); d != dir.end(); ++d)
        {
            string name = *d;
            if (name[0] == '.') continue;

            res_total++;

            // Look for an `info' block in the name
            size_t p_info = name.rfind(':');
            if (p_info == string::npos) continue;

            // Ensure that the info block is in the right format
            if (name.compare(p_info, 3, ":2,") != 0) continue;

            // Look for the 'S' flag (it should not be there)
            p_info += 3;
            if (name.find('S', p_info) == string::npos)
                res_unread++;
            if (name.find('F', p_info) != string::npos)
                res_flagged++;
        }
    }

    // Restore the access time of the mailbox for other checking programs
    t[0].tv_sec = st_cur->st_atime;
    t[0].tv_usec = 0;
    t[1].tv_sec = st_cur->st_mtime;
    t[1].tv_usec = 0;
    utimes(path_cur.c_str(), t);


	// Return the values
	_stat_total = res_total;
	_stat_unread = res_unread + res_new;
	_stat_new = res_new;
	_stat_flagged = res_flagged;
}


static bool isMaildir(const std::string& pathname)
{
    try {
        // It must be a directory
        std::auto_ptr<struct stat> st = sys::fs::stat(pathname);
        if (!st.get())
            return false;
        if (S_ISDIR(st->st_mode) == 0)
            return false;

        // It must contain cur, new and tmp subdirectories
        const char* subdirs[3] = { "cur", "new", "tmp" };
        for (int i = 0; i < 3; i++)
            if (!sys::fs::isdir(str::joinpath(pathname, subdirs[i])))
                return false;

        // It appears to be a maildir directory
        return true;
    } catch (wibble::exception::System& e) {
        // If we can't even stat() it, then it's not a maildir
        return false;
    }
}

MailFolder Maildir::accessFolder(const std::string& path)
{
	try {
		if (isMaildir(path))
			return MailFolder(new Maildir(path));
	} catch (wibble::exception::System& e) {
		// FIXME: cerr << e.type() << ": " << e.fullInfo() << endl;
	}
	return MailFolder();
}

static void enumerateSubfolders(
		const std::string& parent,
		const std::string& name,
		MailFolderConsumer& cons,
		InodeSet seen = InodeSet())
{
    try {
        std::auto_ptr<struct stat> st;
        try {
            st = sys::fs::stat(parent);
        } catch (wibble::exception::File& e) {
            // If we can't even stat() it, then we don't try to enumerate its subfolders
            // FIXME: cerr << e.type() << ": " << e.fullInfo() << endl;
            return;
        }
        if (!st.get()) return;

        // It must be a directory
        if (S_ISDIR(st->st_mode) == 0)
            return;

        // Check that we aren't looping
        if (seen.has(st->st_ino))
            return;

		if (isMaildir(parent))
		{
			MailFolder f(new Maildir(name, parent));
			cons.consume(f);
		}

        // Recursively enumerate the Maildirs in the directory
        sys::fs::Directory dir(parent);
        for (sys::fs::Directory::const_iterator d = dir.begin(); d != dir.end(); ++d)
        {
            string dname = *d;
            if (dname == ".") continue;
            if (dname == "..") continue;
            if (dname == "tmp") continue;
            if (dname == "cur") continue;
            if (dname == "new") continue;

            enumerateSubfolders(
                    concat(parent, '/', dname),
                    concat(name, '.', dname),
                    cons, seen + st->st_ino);
        }
    } catch (wibble::exception::Generic& e) {
        // FIXME: cerr << e.type() << ": " << e.fullInfo() << endl;
    }
}

void Maildir::enumerateFolders(const std::string& parent, MailFolderConsumer& cons)
{
	// Remove trailing slash from the parent directory
	// The root name is empty if parent is not a maildir
	//   else, it is the last component of parent's path
	string root;
	string rootName;

	size_t pos = parent.rfind('/');
	if (pos == string::npos)
		root = rootName = parent;
	else if (pos == parent.size() - 1)
	{
		pos = parent.rfind('/', pos - 1);
		root = parent.substr(0, parent.size() - 1);
		rootName = parent.substr(pos+1, parent.size() - pos - 2);
	}
	else
	{
		root = parent;
		rootName = parent.substr(pos + 1);
	}

	if (!isMaildir(parent))
		rootName = string();

	enumerateSubfolders(root, rootName, cons);
}

}
}

// vim:set ts=4 sw=4:
