// GTK_SIMPLE_DIALOGS.CPP

// Copyright (C) 1999 Tommi Hassinen.

// This package is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.

// This package is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with this package; if not, write to the Free Software
// Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA

/*################################################################################################*/

#include "gtk_simple_dialogs.h"

#include <ghemical/atom.h>
#include <ghemical/bond.h>

#include "gtk_project.h"

#include "gtk_trajview_dialog.h"

#include "local_i18n.h"

#include <cstring>
using namespace std;

/*################################################################################################*/

gtk_element_dialog * gtk_element_dialog::current_object = NULL;

GtkWidget * gtk_element_dialog::dialog = NULL;
GtkWidget * gtk_element_dialog::current_element_widget = NULL;

gtk_element_dialog::gtk_element_dialog(void)
{
	current_object = this;
	dialog = gtk_dialog_new();
	
	gtk_window_set_modal(GTK_WINDOW(dialog), TRUE);		// request a modal window.
	
	gtk_window_set_title(GTK_WINDOW(dialog), _("Set Current Element"));
	gtk_window_set_position(GTK_WINDOW(dialog), GTK_WIN_POS_CENTER);
	
	GtkWidget * table = gtk_table_new(21, 38, TRUE);
	gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dialog)->vbox), table);
	
	int attachdata[ELEMENT_SYMBOLS][2] =
	{
		{ 1, 1 }, { 35, 1 }, { 1, 3 }, { 3, 3 }, { 25, 3 },		// B
		{ 27, 3 }, { 29, 3 }, { 31, 3 }, { 33, 3 }, { 35, 3 },		// Ne
		{ 1, 5 }, { 3, 5 }, { 25, 5 }, { 27, 5 }, { 29, 5 },		// P
		{ 31, 5 }, { 33, 5 }, { 35, 5 }, { 1, 7 }, { 3, 7 },		// Ca
		{ 5, 7 }, { 7, 7 }, { 9, 7 }, { 11, 7 }, { 13, 7 },		// Mn
		{ 15, 7 }, { 17, 7 }, { 19, 7 }, { 21, 7 }, { 23, 7 },		// Zn
		{ 25, 7 }, { 27, 7 }, { 29, 7 }, { 31, 7 }, { 33, 7 },		// Br
		{ 35, 7 }, { 1, 9 }, { 3, 9 }, { 5, 9 }, { 7, 9 },		// Zr
		{ 9, 9 }, { 11, 9 }, { 13, 9 }, { 15, 9 }, { 17, 9 },		// Rh
		{ 19, 9 }, { 21, 9 }, { 23, 9 }, { 25, 9 }, { 27, 9 },		// Sn
		{ 29, 9 }, { 31, 9 }, { 33, 9 }, { 35, 9 }, { 1, 11 },		// Cs
		{ 3, 11 }, { 5, 11 }, { 9, 16 }, { 11, 16 }, { 13, 16 },	// Nd
		{ 15, 16 }, { 17, 16 }, { 19, 16 }, { 21, 16 }, { 23, 16 },	// Tb
		{ 25, 16 }, { 27, 16 }, { 29, 16 }, { 31, 16 }, { 33, 16 },	// Yb
		{ 35, 16 }, { 7, 11 }, { 9, 11 }, { 11, 11 }, { 13, 11 },	// Re
		{ 15, 11 }, { 17, 11 }, { 19, 11 }, { 21, 11 }, { 23, 11 },	// Hg
		{ 25, 11 }, { 27, 11 }, { 29, 11 }, { 31, 11 }, { 33, 11 },	// At
		{ 35, 11 }, { 1, 13 }, { 3, 13 }, { 5, 13 }, { 9, 18 },		// Th
		{ 11, 18 }, { 13, 18 }, { 15, 18 }, { 17, 18 }, { 19, 18 },	// Am
		{ 21, 18 }, { 23, 18 }, { 25, 18 }, { 27, 18 }, { 29, 18 },	// Fm
		{ 31, 18 }, { 33, 18 }, { 35, 18 }, { 7, 13 }, { 9, 13 },	// Db
		{ 11, 13 }, { 13, 13 }, { 15, 13 }, { 17, 13 }, { 19, 13 }
	};
	
	GtkSignalFunc sf = (GtkSignalFunc) gtk_element_dialog::SignalHandler;
	GtkWidget * button;
	
	for (int n1 = 0; n1 < ELEMENT_SYMBOLS; n1++)
	{
		element tmp_element(n1 + 1);
		button = gtk_toggle_button_new_with_label(tmp_element.GetSymbol());
		
		if (n1 + 1 == element::current_element.GetAtomicNumber())
		{
			gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), true);
			current_element_widget = button;	// do we need this???
		}
		
		gtk_signal_connect(GTK_OBJECT(button), "clicked", sf, (void *) (n1 + 1));
		gtk_widget_show(button);

		int x = attachdata[n1][0]; int y = attachdata[n1][1];
		gtk_table_attach_defaults(GTK_TABLE(table), button, x, x + 2, y, y + 2);
	}
	
	gtk_widget_show(table);
	gtk_widget_show(dialog);
}

gtk_element_dialog::~gtk_element_dialog(void)
{
}

void gtk_element_dialog::SignalHandler(GtkWidget *, int signal)
{
	element::current_element = element(signal);
	cout << _("Current element is now ") << element::current_element.GetSymbol() << "." << endl;	// which prj???
	
	gtk_widget_destroy(dialog);
	delete current_object;		// the object was created with new!!!
}

/*################################################################################################*/

gtk_bondtype_dialog * gtk_bondtype_dialog::current_object = NULL;

GtkWidget * gtk_bondtype_dialog::dialog = NULL;
GtkWidget * gtk_bondtype_dialog::current_bond_widget = NULL;

gtk_bondtype_dialog::gtk_bondtype_dialog(void)
{
	current_object = this;
	dialog = gtk_dialog_new();
	
	gtk_window_set_modal(GTK_WINDOW(dialog), TRUE);		// request a modal window.
	
	gtk_window_set_title(GTK_WINDOW(dialog), _("Set Current Bondtype"));
	gtk_window_set_position(GTK_WINDOW(dialog), GTK_WIN_POS_CENTER);
	
	GtkWidget * table = gtk_table_new((2 + 4 + 1), 6, TRUE);
	gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dialog)->vbox), table);
	
	char bt_symbols[4] = { 'S', 'D', 'T', 'C' };
	
	GtkSignalFunc sf = (GtkSignalFunc) gtk_bondtype_dialog::SignalHandler;
	GtkWidget * button;
	
	int x = 1; int y = 0;
	for (int n1 = 0;n1 < 4;n1++)
	{
		bondtype tmp_bondtype(bt_symbols[n1]);
		button = gtk_toggle_button_new_with_label(tmp_bondtype.GetString());
		
		if (bt_symbols[n1] == bondtype::current_bondtype.GetSymbol1())
		{
			gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), true);
			current_bond_widget = button;	// do we need this??
		}

		i32s long_bt_symbol = bt_symbols[n1];	// convert 8bit -> 32bit
		gtk_signal_connect(GTK_OBJECT(button), "clicked", sf, (void *) long_bt_symbol);
		
		y += 1;
		gtk_table_attach_defaults(GTK_TABLE(table), button, x, x + 4, y, y + 1);
		gtk_widget_show(button);
		
		if (n1 == 2)	// add a separator ; 20050415
		{
			button = gtk_hseparator_new();
			
			y += 1;
			gtk_table_attach_defaults(GTK_TABLE(table), button, x, x + 4, y, y + 1);
			gtk_widget_show(button);
		}
	}
	
	gtk_widget_show(table);
	gtk_widget_show(dialog);
}

gtk_bondtype_dialog::~gtk_bondtype_dialog(void)
{
}

void gtk_bondtype_dialog::SignalHandler(GtkWidget *, int signal)
{
	bondtype::current_bondtype = bondtype((char) signal);
	cout << _("Current bondtype is now ") << bondtype::current_bondtype.GetString() << "." << endl;		// which prj???
	
	gtk_widget_destroy(dialog);
	delete current_object;		// the object was created with new!!!
}

/*################################################################################################*/

const char * make_tmp_filename(gtk_project * prj)
{
	static char tmp_filename[256];
	prj->GetFullProjectFileName(tmp_filename, 256);
	return tmp_filename;
}

GtkWidget * gtk_file_dialog::dialog;

gtk_file_dialog::gtk_file_dialog(const char * title, const char * filter, const char * def_fn)
{
	dialog = gtk_file_selection_new(title);
	gtk_window_set_modal(GTK_WINDOW(dialog), TRUE);		// request a modal window.
	
	if (filter != NULL) gtk_file_selection_complete(GTK_FILE_SELECTION(dialog), filter);
	if (def_fn != NULL) gtk_file_selection_set_filename(GTK_FILE_SELECTION(dialog), def_fn);
	
	gtk_signal_connect(GTK_OBJECT(dialog),
		"destroy", GTK_SIGNAL_FUNC(DestroyHandler), this);
		
	gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(dialog)->ok_button),
		"clicked", GTK_SIGNAL_FUNC(OkButtonHandler), this);
		
	gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(dialog)->cancel_button),
		"clicked", GTK_SIGNAL_FUNC(CancelButtonHandler), this);

	gtk_widget_show(dialog);
}

gtk_file_dialog::~gtk_file_dialog(void)
{
}

void gtk_file_dialog::DestroyHandler(GtkWidget *, gpointer data)
{
	gtk_file_dialog * ref = (gtk_file_dialog *) data;
	
	delete ref;	// the object was created using new!!!
}

void gtk_file_dialog::OkButtonHandler(GtkWidget *, gpointer data)
{
	gtk_file_dialog * ref = (gtk_file_dialog *) data;
	
	// do not close the dialog if we had for example wrong file type...
	
	if (ref->OkEvent(gtk_file_selection_get_filename(GTK_FILE_SELECTION(dialog))))
	{
		gtk_widget_destroy(dialog);
	}
}

void gtk_file_dialog::CancelButtonHandler(GtkWidget *, gpointer data)
{
	gtk_file_dialog * ref = (gtk_file_dialog *) data;
	ref->CancelEvent();
	
	gtk_widget_destroy(dialog);
}

/*################################################################################################*/

const char gtk_file_open_dialog::title[] = "Open Project File";
gtk_project * gtk_file_open_dialog::prj = NULL;

gtk_file_open_dialog::gtk_file_open_dialog(gtk_project * p1) :
	gtk_file_dialog(title, NULL, NULL)
{
	prj = p1;
}

gtk_file_open_dialog::~gtk_file_open_dialog(void)
{
}

bool gtk_file_open_dialog::OkEvent(const char * filename)
{
	if (prj == NULL)
	{
		gtk_app::GetAppX()->SetNewProject();
	}
	
	bool open = prj->IsEmpty();	// open, not insert
	
	ifstream ifile;
	ifile.open(filename, ios::in);
	ReadGPR(* prj, ifile, !open);
	ifile.close();
	
	if (open) 	// we're "switching" to this file
	{
		prj->ParseProjectFileNameAndPath(filename);
		
		char buffer[256];
		prj->GetProjectFileName(buffer, 256, true);
		
	// the file name change here is not yet properly working.
	// the project notebook widget should be stored (if not already) in gtk_project and
	// a call of UpdateAllWindowTitles() should change the labels.
		
		custom_app::GetAppC()->UpdateAllWindowTitles();
	}
	else		// we append this file to the current project
	{
		if (prj->selected_object != NULL)		// if an object was selected, remove
		{						// that selection and leave the newly added
			prj->selected_object = NULL;		// atoms as selection; now the mouse tools
		}						// also affect the newly added atoms!
	}
	
	prj->UpdateAllGraphicsViews();
	return true;
}

void gtk_file_open_dialog::CancelEvent(void)
{
}

/*################################################################################################*/

const char gtk_file_save_dialog::title[] = "Save Project File";
gtk_project * gtk_file_save_dialog::prj = NULL;

gtk_file_save_dialog::gtk_file_save_dialog(gtk_project * p1) :
	gtk_file_dialog(title, FILENAME_FILTER, make_tmp_filename(p1))
{
	prj = p1;
}

gtk_file_save_dialog::~gtk_file_save_dialog(void)
{
}

bool gtk_file_save_dialog::OkEvent(const char * filename)
{
	if (prj == NULL)
	{
		assertion_failed(__FILE__, __LINE__, "prj == NULL");
	}
	
	// should check to see if file already exists...
	
	prj->ParseProjectFileNameAndPath(filename);
	
	char buffer[256];
	prj->GetProjectFileName(buffer, 256, true);
	
	custom_app::GetAppC()->UpdateAllWindowTitles();
	
	char tmp_filename[256];
	prj->GetFullProjectFileName(tmp_filename, 256);
	
	ofstream ofile;
	ofile.open(tmp_filename, ios::out);
	WriteGPR(* prj, ofile);
	ofile.close();

	return true;
}

void gtk_file_save_dialog::CancelEvent(void)
{
}

/*################################################################################################*/

const char gtk_file_save_graphics_dialog::title[] = "Save Graphics File - ps";
gtk_project * gtk_file_save_graphics_dialog::prj = NULL;

gtk_file_save_graphics_dialog::gtk_file_save_graphics_dialog(gtk_project * p1) :
	gtk_file_dialog(title, "ps", make_tmp_filename(p1))
{
	prj = p1;
}

gtk_file_save_graphics_dialog::~gtk_file_save_graphics_dialog(void)
{
}

bool gtk_file_save_graphics_dialog::OkEvent(const char * filename)
{
	if (prj == NULL)
	{
		assertion_failed(__FILE__, __LINE__, "prj == NULL");
	}
	
	// write graphics event
	// write graphics event
	// write graphics event
	
	return true;
}

void gtk_file_save_graphics_dialog::CancelEvent(void)
{
}

/*################################################################################################*/

const char gtk_trajfile_dialog::title[] = "Open Trajectory File - traj";
gtk_project * gtk_trajfile_dialog::prj = NULL;

gtk_trajfile_dialog::gtk_trajfile_dialog(gtk_project * p1) :
	gtk_file_dialog(title, NULL, NULL)
{
	prj = p1;
}

gtk_trajfile_dialog::~gtk_trajfile_dialog(void)
{
}

bool gtk_trajfile_dialog::OkEvent(const char * filename)
{
	cout << "DEBUG : trying to open \"" << filename << "\"." << endl;
	
	prj->OpenTrajectory(filename);
// check if there were problems with OpenTrajectory()?!?!?!
// check if there were problems with OpenTrajectory()?!?!?!
// check if there were problems with OpenTrajectory()?!?!?!
	
	static gtk_trajview_dialog * tvd = NULL;
	
	if (tvd != NULL) delete tvd;		// how to safely release the memory...
	tvd = new gtk_trajview_dialog(prj);	// ...right after the dialog is closed?
	
	// the dialog will call prj->CloseTrajectory() itself when closed!!!
	// the dialog will call prj->CloseTrajectory() itself when closed!!!
	// the dialog will call prj->CloseTrajectory() itself when closed!!!
	
	gtk_widget_destroy(dialog);	// close the dialog -> must return false...
	return false;
}

void gtk_trajfile_dialog::CancelEvent(void)
{
}

/*################################################################################################*/

const char gtk_importpdb_dialog::title[] = "Import File - Brookhaven PDB/ENT";
gtk_project * gtk_importpdb_dialog::prj = NULL;

gtk_importpdb_dialog::gtk_importpdb_dialog(gtk_project * p1) :
	gtk_file_dialog(title, NULL, NULL)
{
	prj = p1;
}

gtk_importpdb_dialog::~gtk_importpdb_dialog(void)
{
}

bool gtk_importpdb_dialog::OkEvent(const char * filename)
{
	if (prj != NULL)
	{
		prj->WarningMessage(_("This alternative PDB reader is designed to remove any gaps that might\nexist in the structure data. Therefore the sequence might be altered!"));
		
		prj->ParseProjectFileNameAndPath(filename);
		
		prj->importpdb_mdata = prj->readpdb_ReadMData(filename);
		prj->readpdb_ReadData(filename, prj->importpdb_mdata, -1);
		
		prj->UpdateAllGraphicsViews();
		custom_app::GetAppC()->UpdateAllWindowTitles();
	}
	
	return true;
}

void gtk_importpdb_dialog::CancelEvent(void)
{
}

/*################################################################################################*/

gtk_project * gtk_command_dialog::prj;
oglview_wcl * gtk_command_dialog::wcl;

GtkWidget * gtk_command_dialog::dialog;
GtkWidget * gtk_command_dialog::entry;

gtk_command_dialog::gtk_command_dialog(gtk_project * p1, oglview_wcl * p2, const char * def_str)
{
	prj = p1; wcl = p2;
	
	dialog = gtk_dialog_new();
	gtk_window_set_modal(GTK_WINDOW(dialog), TRUE);		// request a modal window.
	
	gtk_window_set_title(GTK_WINDOW(dialog), _("Command Interpreter"));
	gtk_window_set_position(GTK_WINDOW(dialog), GTK_WIN_POS_CENTER);
	gtk_window_set_default_size(GTK_WINDOW(dialog), 400, 50);
	
	gtk_container_set_border_width(GTK_CONTAINER(dialog), 10);
	
	gtk_signal_connect(GTK_OBJECT(dialog), "destroy", GTK_SIGNAL_FUNC(DestroyHandler), this);
	
	GtkWidget * vbox = gtk_vbox_new(TRUE, 5);
	gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dialog)->vbox), vbox);
	
	entry = gtk_entry_new();
	if (def_str != NULL) gtk_entry_set_text(GTK_ENTRY(entry), def_str);
	
	GtkWidget * hbox = gtk_hbox_new(TRUE, 5);
	
	gtk_box_pack_start(GTK_BOX(vbox), entry, FALSE, TRUE, 0);
	gtk_widget_show(entry);
	
	gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, TRUE, 0);
	
	GtkWidget * button_ok = gtk_button_new_with_label(_("Execute"));
	gtk_signal_connect(GTK_OBJECT(button_ok), "clicked", GTK_SIGNAL_FUNC(OkButtonHandler), this);
	
	GtkWidget * button_cancel = gtk_button_new_with_label(_("Cancel"));
	gtk_signal_connect(GTK_OBJECT(button_cancel), "clicked", GTK_SIGNAL_FUNC(CancelButtonHandler), this);
	
	gtk_box_pack_start(GTK_BOX(hbox), button_ok, FALSE, TRUE, 0);
	gtk_widget_show(button_ok);
	
	gtk_box_pack_start(GTK_BOX(hbox), button_cancel, FALSE, TRUE, 0);
	gtk_widget_show(button_cancel);
	
	gtk_widget_show(hbox);
	gtk_widget_show(vbox);
	
	gtk_widget_show(dialog);
}

gtk_command_dialog::~gtk_command_dialog(void)
{
	prj = NULL;
	wcl = NULL;
}

void gtk_command_dialog::DestroyHandler(GtkWidget *, gpointer data)
{
	gtk_command_dialog * ref = (gtk_command_dialog *) data;
	
	delete ref;	// the object was created using new!!!
}

void gtk_command_dialog::OkButtonHandler(GtkWidget *, gpointer data)
{
	gtk_command_dialog * ref = (gtk_command_dialog *) data;
	
	const char * command = gtk_entry_get_text(GTK_ENTRY(ref->entry));
	
	// first get rid of the dialog...
	// ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
	
	gtk_widget_hide(dialog);
	
	// ...and then process the command string (if given).
	// ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
	
	if (strlen(command) > 0) prj->ProcessCommandString(wcl, command);
	
	gtk_widget_destroy(dialog);
}

void gtk_command_dialog::CancelButtonHandler(GtkWidget *, gpointer data)
{
	gtk_command_dialog * ref = (gtk_command_dialog *) data;
	
	gtk_widget_destroy(dialog);
}

/*################################################################################################*/

// eof
