/* caldav_db.h -- abstract interface for per-user CalDAV database
 *
 * Copyright (c) 1994-2012 Carnegie Mellon University.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The name "Carnegie Mellon University" must not be used to
 *    endorse or promote products derived from this software without
 *    prior written permission. For permission or any legal
 *    details, please contact
 *      Carnegie Mellon University
 *      Center for Technology Transfer and Enterprise Creation
 *      4615 Forbes Avenue
 *      Suite 302
 *      Pittsburgh, PA  15213
 *      (412) 268-7393, fax: (412) 268-7395
 *      innovation@andrew.cmu.edu
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Computing Services
 *     at Carnegie Mellon University (http://www.cmu.edu/computing/)."
 *
 * CARNEGIE MELLON UNIVERSITY DISCLAIMS ALL WARRANTIES WITH REGARD TO
 * THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS, IN NO EVENT SHALL CARNEGIE MELLON UNIVERSITY BE LIABLE
 * FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING
 * OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#ifndef CALDAV_DB_H
#define CALDAV_DB_H

#include <config.h>

extern time_t caldav_epoch;
extern time_t caldav_eternity;

#include <libical/ical.h>

#include "dav_db.h"

#ifndef HAVE_VAVAILABILITY
/* Allow us to compile without #ifdef HAVE_VAVAILABILITY everywhere */
#define ICAL_VAVAILABILITY_COMPONENT  ICAL_X_COMPONENT
#define ICAL_XAVAILABLE_COMPONENT     ICAL_X_COMPONENT
#endif

#ifndef HAVE_VPOLL
/* Allow us to compile without #ifdef HAVE_VPOLL everywhere */
#define ICAL_VPOLL_COMPONENT          ICAL_NO_COMPONENT
#define ICAL_VVOTER_COMPONENT         ICAL_X_COMPONENT
#define ICAL_METHOD_POLLSTATUS        ICAL_METHOD_NONE
#define ICAL_VOTER_PROPERTY           ICAL_NO_PROPERTY
#define icalproperty_get_voter        icalproperty_get_attendee
#endif

/* Bitmask of calendar components */
enum {
    /* "Real" components - MUST remain in this order (values used in DAV DB) */
    CAL_COMP_VEVENT =           (1<<0),
    CAL_COMP_VTODO =            (1<<1),
    CAL_COMP_VJOURNAL =         (1<<2),
    CAL_COMP_VFREEBUSY =        (1<<3),
    CAL_COMP_VAVAILABILITY =    (1<<4),
    CAL_COMP_VPOLL =            (1<<5),
    /* Append additional "real" components here */

    /* Other components - values don't matter - prepend here */
    CAL_COMP_VALARM =           (1<<13),
    CAL_COMP_VTIMEZONE =        (1<<14),
    CAL_COMP_VCALENDAR =        (1<<15)
};

#define CAL_COMP_REAL            0xff   /* All "real" components */

struct caldav_db;

struct comp_flags {
    unsigned recurring    : 1;          /* Has RRULE property */
    unsigned transp       : 1;          /* Is TRANSParent */
    unsigned status       : 2;          /* STATUS property value (see below) */
    unsigned tzbyref      : 1;          /* VTIMEZONEs by reference */
    unsigned mattach      : 1;          /* Has managed ATTACHment(s) */
};

/* Status values */
enum {
    CAL_STATUS_BUSY = 0,
    CAL_STATUS_CANCELED,
    CAL_STATUS_TENTATIVE,
    CAL_STATUS_UNAVAILABLE
};

struct caldav_data {
    struct dav_data dav;  /* MUST be first so we can typecast */
    unsigned comp_type;
    const char *ical_uid;
    const char *organizer;
    const char *dtstart;
    const char *dtend;
    struct comp_flags comp_flags;
    const char *sched_tag;
};

typedef int caldav_cb_t(void *rock, struct caldav_data *cdata);

/* prepare for caldav operations in this process */
int caldav_init(void);

/* done with all caldav operations for this process */
int caldav_done(void);

/* get a database handle corresponding to mailbox */
struct caldav_db *caldav_open_mailbox(struct mailbox *mailbox);
struct caldav_db *caldav_open_userid(const char *userid);

/* close this handle */
int caldav_close(struct caldav_db *caldavdb);

/* lookup an entry from 'caldavdb' by resource
   (optionally inside a transaction for updates) */
int caldav_lookup_resource(struct caldav_db *caldavdb,
                           const char *mailbox, const char *resource,
                           struct caldav_data **result,
                           int tombstones);

/* lookup an entry from 'caldavdb' by mailbox and IMAP uid
   (optionally inside a transaction for updates) */
int caldav_lookup_imapuid(struct caldav_db *caldavdb,
                          const char *mailbox, int uid,
                          struct caldav_data **result,
                          int tombstones);

/* lookup an entry from 'caldavdb' by iCal UID
   (optionally inside a transaction for updates) */
int caldav_lookup_uid(struct caldav_db *caldavdb, const char *ical_uid,
                      struct caldav_data **result);

/* process each entry for 'mailbox' in 'caldavdb' with cb() */
int caldav_foreach(struct caldav_db *caldavdb, const char *mailbox,
                   caldav_cb_t *cb, void *rock);

/* process each entry for 'mailbox' in 'caldavdb' with cb()
 * which last recurrence ends after 'after' and first
 * recurrence starts before 'before'. The largest possible
 * timerange spans from caldav_epoch to caldav_eternity. */
int caldav_foreach_timerange(struct caldav_db *caldavdb, const char *mailbox,
                             time_t after, time_t before,
                             caldav_cb_t *cb, void *rock);

/* write an entry to 'caldavdb' */
int caldav_write(struct caldav_db *caldavdb, struct caldav_data *cdata);
int caldav_writeentry(struct caldav_db *caldavdb, struct caldav_data *cdata,
                      icalcomponent *ical);

/* delete an entry from 'caldavdb' */
int caldav_delete(struct caldav_db *caldavdb, unsigned rowid);

/* delete all entries for 'mailbox' from 'caldavdb' */
int caldav_delmbox(struct caldav_db *caldavdb, const char *mailbox);

/* begin transaction */
int caldav_begin(struct caldav_db *caldavdb);

/* commit transaction */
int caldav_commit(struct caldav_db *caldavdb);

/* abort transaction */
int caldav_abort(struct caldav_db *caldavdb);

char *caldav_mboxname(const char *userid, const char *name);

int caldav_get_events(struct caldav_db *caldavdb,
                      const char *mailbox, const char *ical_uid,
                      caldav_cb_t *cb, void *rock);

/* Process each entry for 'caldavdb' with a modseq higher than oldmodseq,
 * in ascending order of modseq.
 * If mailbox is not NULL, only process entries of this mailbox.
 * If max_records is positive, only call cb for at most this entries. */
int caldav_get_updates(struct caldav_db *caldavdb,
                       modseq_t oldmodseq, const char *mailbox, int kind,
                       int max_records, caldav_cb_t *cb, void *rock);

#endif /* CALDAV_DB_H */
