# -.- coding: utf-8 -.-
#
# Activity Log Manager
#
# Copyright © 2011 Collabora Ltd.
#             By Siegfried-Angel Gevatter Pujals <siegfried@gevatter.com>
#             By Seif Lotfy <seif@lotfy.com>
# Copyright © 2011 Manish Sinha <manishsinha@ubuntu.com>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

from itertools import imap

from zeitgeist.client import ZeitgeistClient
from zeitgeist.datamodel import Event, Subject

class ZeitgeistInterface:

	INCOGNITO = Event.new_for_values()
	INCOGNITO_TEMPLATE_ID = "block-all"

	def __init__(self):
		self._client = ZeitgeistClient()
		if self._client.get_version() < [0, 7, 99]:
			raise SystemExit, _("Zeitgeist version 0.8 or later required.")
		
		self._blacklist = self._client._iface.get_extension('Blacklist',
			'blacklist')
		self._all = None

	@property
	def all_templates(self):
		if self._all is None:
			self._all = {}
			# Connect to signals
			self._blacklist.connect('TemplateAdded', self._template_added)
			self._blacklist.connect('TemplateRemoved', self._template_removed)
			# And fetch the current templates
			for blacklist_id, template in self._blacklist.GetTemplates().iteritems():
				self._all[blacklist_id] = Event(template)
		return self._all

	def _template_added(self, blacklist_id, template):
		if blacklist_id not in self._all:
			self._all[blacklist_id] = template
	
	def _template_removed(self, blacklist_id, template):
		if blacklist_id in self._all:
			del self._all[blacklist_id]
	
	def add_blacklist_template(self, blacklist_id, template):
		"""
		Add a new blacklist template.
		"""
		self._blacklist.AddTemplate(blacklist_id, template)
		self._all[blacklist_id] = template

	def remove_blacklist_template(self, blacklist_id):
		"""
		Remove the blacklist template for the provided template id.
		"""
		self._blacklist.RemoveTemplate(blacklist_id)
		del self._all[blacklist_id]

	def get_incognito(self):
		"""
		The blacklist template for incognito is matched against all the 
		blacklist items.
		"""
		return any(imap(self.INCOGNITO.matches_template, self.all_templates.itervalues()))

	def set_incognito(self, enabled):
		if enabled:
			self.add_blacklist_template(self.INCOGNITO_TEMPLATE_ID, self.INCOGNITO)
		else:
			self.remove_blacklist_template(self.INCOGNITO_TEMPLATE_ID)
			
	def remove_history(self, timerange, callback):
		last_ids = []
		def get_ids_callback(ids):
			last_ids = ids
			def remove_callback(timerange):
				callback(last_ids)
			self._client.delete_events(ids, remove_callback)
		self._client.find_event_ids_for_templates([], get_ids_callback, timerange, 2, 0)
		
