/*
    ACfax - Fax reception with X11-interface for amateur radio
    Copyright (C) 1995-1998 Andreas Czechanowski, DL4SDC

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software Foundation,
    Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

    andreas.czechanowski@ins.uni-stuttgart.de
*/
    
/*
 * x_image.c - XImage and colormap allocation functions
 */

#include <stdio.h>
#include <stdlib.h>
/*
#include <X11/X.h>
#include <X11/Xlib.h>
#include <X11/Intrinsic.h>
*/
#include "x_image.h"

unsigned long	coltab[256];	/* mapping table grayscale->pixel-value */
char	  cused[256];	/* table indicating used pixel-values */
Display	  *dpy = NULL;		/* X-display used */
Screen	  *scrn = NULL;		/* Screen of display */
Visual	  *visl = NULL;		/* Visual of screen to use */
int	  depth = -1;		/* depth of screen (# of planes) */
Colormap  icmap;		/* the colormap to use */
XImage	  *horimag = NULL;	/* intermediate-storage for horizontal scan */
XImage	  *verimag = NULL;	/* intermediate-storage for vertical scan */
XVisualInfo visinfo;		/* Visual-info-structure (from Xutil.h) */
unsigned long r_mask;		/* bit mask bits (truecolor) for red */
unsigned long g_mask;		/* bit mask bits (truecolor) for green */
unsigned long b_mask;		/* bit mask bits (truecolor) for blue */
int	  r_shift;		/* position of LSB in r_mask */
int	  g_shift;		/* position of LSB in g_mask */
int	  b_shift;		/* position of LSB in b_mask */
int	  r_bits;		/* count of significant bits in r_mask */
int	  g_bits;		/* count of significant bits in g_mask */
int	  b_bits;		/* count of significant bits in b_mask */

/*
 * get some global X11-variables and resources from the passed Widget
 */
void get_globals(Widget toplevel)
{
  if (!dpy)
    dpy = XtDisplay(toplevel);
  if (!scrn)
    scrn = XtScreen(toplevel);
  if (depth < 0)
    depth = DefaultDepthOfScreen(scrn);
  if (!visl) {
    visl = DefaultVisualOfScreen(scrn);
#if 0
    if (!(XMatchVisualInfo(dpy, XScreenNumberOfScreen(scrn), depth,
	 PseudoColor, &visinfo))) {
      fprintf(stderr, "cannot get PseudoColor-visual with depth %d !\n", depth);
      exit(1);
    }
    visl = visinfo.visual;
#endif
  }
}

/*
 * create two XImages as "transportation storage" between the raw-resulution
 * picture and the canvas-widget's pixmap (one horizontal stripe with
 * DEFHEIGHT lines, and one vertical stripe with DEFWIDTH columns).
 * If called more than once, the old
 * XImages are Destroyes and new ones are created with the given size.
 */
void create_images(Widget toplevel, Dimension wid, Dimension hei)
{
  get_globals(toplevel);
  fprintf(stderr, "creating XImage for %d x %d pixels\n", wid, hei);
  /* delete old images and their data (when image is resized) */
  if (horimag)
    XDestroyImage(horimag);
  if (verimag)
    XDestroyImage(verimag);
  horimag = XCreateImage(dpy, visl, depth, ZPixmap, 0, NULL,
        wid, DEFHEIGHT, 32, 0);
  verimag = XCreateImage(dpy, visl, depth, ZPixmap, 0, NULL,
        DEFWIDTH, hei, 32, 0);
  if (!(horimag) || !(verimag)) {
    fprintf(stderr,"cannot allocate Ximages !\n");
    exit(1);
  }
  /* allocate data for the XImages (not done by XCreateImage !) */
  horimag->data = (char *)XtMalloc(DEFHEIGHT * horimag->bytes_per_line);  
  verimag->data = (char *)XtMalloc(hei * verimag->bytes_per_line);
  if (!(horimag->data) || !(verimag->data)) {
    fprintf(stderr,"cannot allocate space for Ximage !\n");
    exit(1);
  }
}

/*
 * free the space allocated by the two XImages
 */
void destroy_images(void)
{
  if (horimag)
    XDestroyImage(horimag);
  if (verimag)
    XDestroyImage(verimag);
}

/*
 * allocate a new colormap. This is needed for PseudoColor visuals
 * of depth 8. For truecolor displays, nothing is to be done here.
 */
void alloc_cmap(Widget toplevel, Pixel *respix, unsigned nrespix)
{
  Colormap rootcmap;
  XColor col;
  static int cmap_created = 0;
  int i;
  XVisualInfo vTemplate;
  XVisualInfo *visuals;
  int nvisuals;
  unsigned long tmpmask;

  get_globals(toplevel);
  switch(visl->class)
  {
    case PseudoColor:
    case StaticColor:
      rootcmap = DefaultColormapOfScreen(scrn);
      if (!(cmap_created))
	icmap = XCreateColormap(dpy, XtWindow(toplevel), visl, AllocAll);
      cmap_created = 1;
      /* XAllocColorCells(dpy, icmap, True, planes, 0, pixels, 216); */
      /* copy the first 16 colors from the root-colormap, and also copy
	 the colors needed for our widgets */
      for (i=0; i<(1 << depth); i++)
	cused[i] = 0;
      for (i=0; i<nrespix; i++)
      {
/*
	fprintf(stderr, "respix[%d] = %u\n", i, (unsigned) respix[i]);
*/
	if (respix[i] > ((1 << depth) - 1)) respix[i] = 0;
	col.pixel = respix[i];
	cused[respix[i]] = 1;
	XQueryColor(dpy, rootcmap, &col);
	XStoreColor(dpy, icmap, &col);
      }
     /* now, we can use the rest of the colors for the
        grayscale or 6x6x6 colormap, but this is done
        in fill_cmap_col and fill_cmap_gray */
      break;
    case TrueColor:
    case DirectColor:
      /* we want a visual with the depth of our screen : */
      vTemplate.screen = XScreenNumberOfScreen(scrn);
      vTemplate.depth = DefaultDepth(dpy, vTemplate.screen);
      nvisuals = 0;
      visuals = XGetVisualInfo(dpy, VisualScreenMask | VisualDepthMask,
			&vTemplate, &nvisuals);
      if (nvisuals < 1)
      {
	fprintf(stderr, "cannot find visual matching default depth %d\n",
		vTemplate.depth);
	exit(1);
      }
      /* copy first visual info to own location */
      visinfo = visuals[0];
      /* discard the array from XGetVisualInfo() */
      XFree(visuals);

      /* determine the bit positions and count of significant bits
       * for red, green and blue, store it in *_shift and *_bits */
      r_mask = visinfo.red_mask;
      g_mask = visinfo.green_mask;
      b_mask = visinfo.blue_mask;
      if (r_mask == 0 || g_mask == 0 || b_mask == 0)
      {
	fprintf(stderr, "one of the color masks is zero : %08lx %08lx %08lx\n",
		r_mask, g_mask, b_mask);
	exit(1);
      }

      /* determine number of bits and position of mask for every color */
      r_shift = g_shift = b_shift = 0;
      r_bits = g_bits = b_bits = 0;

      tmpmask = r_mask;
      while ((tmpmask & 1) == 0)
      {
	tmpmask >>= 1;
	r_shift++;
      }
      while ((tmpmask & 1) == 1)
      {
	tmpmask >>= 1;
	r_bits++;
      }

      tmpmask = g_mask;
      while ((tmpmask & 1) == 0)
      {
	tmpmask >>= 1;
	g_shift++;
      }
      while ((tmpmask & 1) == 1)
      {
	tmpmask >>= 1;
	g_bits++;
      }

      tmpmask = b_mask;
      while ((tmpmask & 1) == 0)
      {
	tmpmask >>= 1;
	b_shift++;
      }
      while ((tmpmask & 1) == 1)
      {
	tmpmask >>= 1;
	b_bits++;
      }
      break;
    case StaticGray:
      fprintf (stderr, "Cannot handle StaticGray visual - abort\n");
      exit(1);
      /*NOTREACHED*/
    case GrayScale:
      fprintf (stderr, "Cannot handle GrayScale visual - abort\n");
      exit(1);
      /*NOTREACHED*/
    default:
      fprintf (stderr, "unknown visual type %x - abort\n", visl->class);
      exit(1);
      /*NOTREACHED*/
  }
}

/*
 * fill the colormap with a 6x6x6 color-cube.
 * when flip is set, the green and blue indexing is exchanged, and rotate
 * defines the order in which the colors are indexed.
 * The array coltab is filled with a lookup-table color_value -> pixel_index.
 */
void fill_cmap_col(int invert, int flip, int rotate)
{
  int cm[3];
  unsigned ixr, ixg, ixb;
  XColor col;
  int i = 0;

  fprintf(stderr, "filling colormap with 6x6x6 cube\n");
  col.flags = DoRed | DoGreen | DoBlue;
  switch(rotate) {
    case 1:
      ixr = 1;
      ixg = (flip) ? 0 : 2;
      ixb = (flip) ? 2 : 0;
      break;
    case 2:
      ixr = 2;
      ixg = (flip) ? 1 : 0;
      ixb = (flip) ? 0 : 1;
      break;
    case 0:
    default:
      ixr = 0;
      ixg = (flip) ? 2 : 1;
      ixb = (flip) ? 1 : 2;
      break;
  }
  for (cm[0]=0; cm[0]<6; cm[0]++)
    for (cm[1]=0; cm[1]<6; cm[1]++)
      for (cm[2]=0; cm[2]<6; cm[2]++) {
        while ((cused[i]) && i < 256) i++;
        if (i == 256) {
          fprintf(stderr,"cannot assign all required colors !\n");
          break;
        }
        col.pixel = i;
        if (invert) {
          col.red =   65535 - 9362 * (cm[ixr] + 1);
          col.green = 65535 - 9362 * (cm[ixg] + 1);
          col.blue =  65535 - 9362 * (cm[ixb] + 1);
        } else {
          col.red =   9362 * (cm[ixr] + 1);
          col.green = 9362 * (cm[ixg] + 1);
          col.blue =  9362 * (cm[ixb] + 1);
        }
        XStoreColor(dpy, icmap, &col);
        coltab[cm[0]+6*cm[1]+36*cm[2]] = i;
        i++;
  }
}

/*
 * fill the colormap with a grayscale of 64 steps (should be good enough).
 * The array coltab is filled with a lookup-table gray_value -> pixel_index.
 */
void fill_cmap_gray(int invert)
{
  int g, yv;
  XColor col;
  int i = 0;
  unsigned long cval;
 
  fprintf(stderr, "filling colormap with grayscale\n");
  switch (visl->class)
  {
    case StaticColor:
    case PseudoColor:
      col.flags = DoRed | DoGreen | DoBlue;
      for (g=0; g<64; g++)
      {
	while ((cused[i]) && i < 256) i++;
	if (i == 256)
	{
          fprintf(stderr,"cannot assign all required grayscales !\n");
          break;
	}
	col.pixel = i;
	yv = (invert) ? 63 - g : g;
	col.red = col.green = col.blue = 1040.2 * yv;
	XStoreColor(dpy, icmap, &col);
	coltab[g] = i;
	i++;
      }
      break;
    case TrueColor:
    case DirectColor:
      /* allocate a translation table grayscale -> truecolor_value */
      for (g=0; g<64; g++)
      {
	yv = (invert) ? 63 - g : g;
	cval = (((1 << r_bits) - 1) * yv / 63) << r_shift;
	cval += (((1 << g_bits) - 1) * yv / 63) << g_shift;
	cval += (((1 << b_bits) - 1) * yv / 63) << b_shift;
	coltab[g] = cval;
      }
      break;
  }
}
