// (C) Copyright 2011-2012 Hewlett-Packard Development Company, L.P.
/**
 * @type {UserSettingsEditView}
 */
define(['hp/core/Localizer',
        'fs/presenter/users/UsersPresenter',
        'hp/model/Session',
        'hp/core/Notifications',
        'text!fsPages/users/user_settings_edit.html',
        'hp/view/DialogView',
        'hp/view/FormStateView',
        'jquery',
        'hp/lib/jquery.hpEllipsis'],
function(localizer, presenter, session, Notifications, userSelfEdit, DialogView, FormStateView) {"use strict";

    var UserSettingsEditView = (function() {

        var FORM = '#fs-user-settingsedit-form',
            CANCEL = "#fs-usersettings-edit-cancel",
            ERROR = '#fs-usersettings-edit-error',
            NAME = '#fs-usersettings-loginname',
            FULLNAME = '#fs-usersettings-fullname',
            CURRENT_PASSWD = '#fs-usersettings-currentpasswd',
            NEW_PASSWD = '#fs-usersettings-newpasswd',
            CONFIRM_PASSWD = '#fs-usersettings-confirmpasswd',
            EMAIL = '#fs-usersettings-email',
            OFFICE_PHONE = '#fs-usersettings-officephone',
            MOBILE_PHONE = '#fs-usersettings-mobilephone',
            ROLES_LIST = '#fs-usersettings-roles-list',
            HEADER = '#fs-usersettings-edit-header',
            MIN_PASSWORD_LENGTH = 8,
            MAX_NAME_LENGTH = 39;

        /**
         * Constructor
         * Show a dialog
         * @param {args} Object with properties as follows:
         *
         *    contents: contents of dialog
         *    ok: the function called when the user clicks ok.
         *    cancel: the function called when the user clicks cancel.
         */
        function UserSettingsEditView() {

            var initialized = false;
            var validator;
            // prevent double click
            var isSubmitted;
            var firstRoleTemplate = null, roleTemplate = null;
            var dialog;
            var formStateView;
            var userSelfEditHtml = $(userSelfEdit);
            var taskName;
            var numberOfClicks = 0;

            /**
             * @private
             */
            function reset() {
                $(NEW_PASSWD).val("");
                $(FULLNAME).val("");
                $(CURRENT_PASSWD).val("");
                $(CONFIRM_PASSWD).val("");
                $(EMAIL).val("");
                $(OFFICE_PHONE).val("");
                $(MOBILE_PHONE).val("");
                validator.resetForm();
                formStateView.reset();
                numberOfClicks = 0;
                isSubmitted = false;
            }

            function onUserModifiedSuccess(user) {
                var alertMsg = localizer.getString('fs.users.edit.success', [ user.userName ]);
                reset();
                var notif = {status: 'ok',
                    uri : taskName,
                    changing : false,
                    summary : alertMsg
                };
                //If we are on the users show page, and we are looking at details of
                //the user we are editing - when the dialog is closed, the details must
                //refresh. Happens only for administrator.
                if (window.location.hash.indexOf('#/user') != -1 &&
                    presenter.getSelectedUser().userName.toLowerCase() === session.getUser().toLowerCase()){
                    //window.location = '#/user/show';
                    presenter.getUser(user.userName);
                }
                //removeEvents();
                Notifications.add(notif, true);
                $(CANCEL).trigger('click');
            }

            /**
             * @private
             */
            function onUserModifiedError(errorMessage) {
              var modifyMsg = localizer.getString('fs.users.edit.error',[$(NAME).val()]);
              var errorMsg = localizer.formatRestError(modifyMsg,
                      errorMessage);
                formStateView.setMessage({summary: modifyMsg, status: 'error',
                    errorMessage: errorMessage});
                var notif = {status: 'error',
                    uri : taskName,
                    changing : false,
                    summary : modifyMsg,
                    details : errorMsg
                };
                Notifications.add(notif, true);
                isSubmitted = false;
            }

            /**
             * @private
             */
            function onEditUserClick() {
                if ($(FORM).valid() && isSubmitted == false) {
                  numberOfClicks = numberOfClicks + 1;
                    isSubmitted = true;
                    var alertMsg = localizer.getString('fs.users.edit.editMsg', [ $(NAME).val() ]);
                    formStateView.setMessage({summary:alertMsg, status:'info', changing : true});
                    if(numberOfClicks == 1){
                        taskName = $(NAME).val() + '-editUserTask-' + (new Date()).getTime();
                    }
                    var userEditProgress = {
                        summary : alertMsg,
                        uri : taskName,
                        changing : true,
                        status : "info"
                    };
                    Notifications.add(userEditProgress, false);
                    var roles = [];
                    presenter.modifyUser($(NAME).val(), $(FULLNAME).val(),
                            $(CURRENT_PASSWD).val(), $(NEW_PASSWD).val(), roles,
                            $(EMAIL).val(), $(OFFICE_PHONE).val(),
                            $(MOBILE_PHONE).val(), false, "self");
                }
                return false;
            }

            function userChanged(user) {
                reset();
                $(ERROR).hide();
                $(NAME).val(user.userName);
                $(FULLNAME).val(user.fullName);
                $(EMAIL).val(user.emailAddress);
                $(OFFICE_PHONE).val(user.officePhone);
                $(MOBILE_PHONE).val(user.mobilePhone);
            }


            /**
             * @private
             */
            function userRolesError(errorMessage) {
                var roleMsg = localizer.getString('fs.users.show.roleserror');
                formStateView.setMessage({summary: roleMsg, status: 'error',
                    errorMessage: errorMessage});
            }

            /**
             * @private
             */
             function updateRoles(retrievedRoles) {

                 var retrievedRolesList = (retrievedRoles.hasOwnProperty('members') ?
                         retrievedRoles.members : retrievedRoles);
                 var rolesList = $(ROLES_LIST);
                 $(rolesList).empty();
                 $.each(retrievedRolesList, function(index, role) {
                     var correctCaseRole = '';
                     var splitRole = (role.roleName ? role.roleName.replace(/_/, ' ') : 'noname').split(' ');
                     $.each(splitRole, function(key, word) {
                       if(key !=0){
                           correctCaseRole = correctCaseRole +' '+ word.toLowerCase();
                       }else{
                           correctCaseRole = correctCaseRole + word;
                       }
                     });
                   var label = $('<label/>');
                   label.html(correctCaseRole);
                   $(ROLES_LIST).append(label);
                   if (index !== retrievedRolesList.length - 1){
                       $(ROLES_LIST).append($('<br/>'));
                   }
                 });


             }

             /**
             * @private
             */
             function myUserChangedSuccess(user) {
               userChanged(user);
             }

             /**
             * @private
             */
             function myUserChangedError(errorMessage) {
                var userMsg = localizer.getString('fs.users.show.error');
                formStateView.setMessage({summary: userMsg, status: 'error',
                    errorMessage: errorMessage});
             }


            function initForm() {
                isSubmitted = false;

                validator = $(FORM, userSelfEditHtml).validate({
                    rules : {
                       fullname : {
                             cicUserSettingsFullNameCheck : FULLNAME,
                             maxlength : MAX_NAME_LENGTH
                         },
                        email : {
                            email : EMAIL
                        },
                        currentpasswd : {
                            cicUserSettingsEditCheckPasswdString : CURRENT_PASSWD,
                            cicUserSettingsEditCheckCurrPass : CURRENT_PASSWD
                        },
                        newpasswd : {
                            cicUserSettingsEditCheckPasswdString : NEW_PASSWD,
                            cicUserSettingsEditCheckPwdLength : MIN_PASSWORD_LENGTH,
                            cicUserSettingsEditCheckReqd : NEW_PASSWD
                        },
                        confirmpasswd : {
                            cicUserSettingsEditMatchPasswd : NEW_PASSWD
                        },
                        officephone : {
                            cicUserSettingsEditPhoneNumCheck : OFFICE_PHONE
                        },
                        mobilephone : {
                            cicUserSettingsEditPhoneNumCheck : MOBILE_PHONE
                        }
                    }
                });
            }

            function registerEvents() {
                presenter.on("userSelfChangedSuccess", myUserChangedSuccess);
                presenter.on("userChangedError", myUserChangedError);
                presenter.on("userSelfRoleChanged", updateRoles);
                presenter.on("userRolesError", userRolesError);
                presenter.on("userSelfModifiedSuccess", onUserModifiedSuccess);
                presenter.on("userModifiedError", onUserModifiedError);
            }

            function removeEvents() {
                presenter.off("userSelfChangedSuccess", myUserChangedSuccess);
                presenter.off("userChangedError", myUserChangedError);
                presenter.off("userSelfRoleChanged", updateRoles);
                presenter.off("userRolesError", userRolesError);
                presenter.off("userSelfModifiedSuccess", onUserModifiedSuccess);
                presenter.off("userModifiedError", onUserModifiedError);
            }

            function initUserEdit(){
                var currentUser = session.getUser();
                firstRoleTemplate = $('#fs-usersettings-first-role-template').detach();
                roleTemplate = $('#fs-usersettings-role-template').detach();
                initForm();
                $(HEADER).html(localizer.getString('fs.users.edit.link')+' '+currentUser);
                $(NAME).text(currentUser);
                presenter.getUser(currentUser);
                presenter.getAllUserRoles(currentUser);
            }

            function init() {
               //because usually only first time init is called and other times load is called.
               //But in this case, the dialog is always brought up using initialize().
                if (!initialized) {
                    initialized = true;
                    formStateView = new FormStateView();
                    formStateView.init({form: $(FORM, userSelfEditHtml), autoTrack: true});
                    //customized help link implemented for this particular div, as default help page is improper
                    $('.hp-help-current').attr('href','/doc#/atc/user/edit/');
                    var validMatchPasswd = localizer.getString('fs.users.edit.matchpasswd');
                    var validCurrentPasswd = localizer.getString('fs.users.validations.matchcurrpasswd');
                    var validPasswdLength = localizer.getString('fs.users.validations.checkLength');
                    var validPhoneNum = localizer.getString('fs.users.validations.phonenumValidation');
                    var validPassword = localizer.getString('fs.users.validations.passwordValidation');
                    var validFullname = localizer.getString('fs.users.validations.fullnameValidation');
                    var fieldRequired = localizer.getString('fs.users.validations.fieldRequired');

                    // check for fullname contain letters ' . - and space
                    $.validator.addMethod("cicUserSettingsFullNameCheck",
                        function(value, element, param) {
                            var isValidFullname = /^[a-zA-Z0-9 ._'\-]+$/.test(value);
                            return (isValidFullname || value.length == 0);
                        },
                        validFullname);

                    $.validator.addMethod("cicUserSettingsEditCheckCurrPass",
                        function(value, element, param) {
                            var result = true;
                            if (value.length == 0 &&
                                ((($(NEW_PASSWD).val() && $(NEW_PASSWD).val().length > 0)) ||
                                    ($(CONFIRM_PASSWD).val() && ($(CONFIRM_PASSWD).val().length >0)))){
                                result = false;
                            }
                            return result;
                        },
                        validCurrentPasswd);

                    $.validator.addMethod("cicUserSettingsEditCheckReqd",
                        function(value, element, param) {
                            var result = true;
                            if (value.length == 0 &&
                                ((($(CURRENT_PASSWD).val() && $(CURRENT_PASSWD).val().length > 0)) &&
                                    ($(CONFIRM_PASSWD).val() && ($(CONFIRM_PASSWD).val().length >0)))){
                                result = false;
                            }
                            return result;
                        },
                        fieldRequired);

                    $.validator.addMethod("cicUserSettingsEditMatchPasswd",
                        function(value, element, param) {
                            // bind to the blur event of the target in order to revalidate whenever the target field is updated
                            var target = $(param).unbind(".validate-equalTo")
                                .bind("blur.validate-equalTo", function() {
                                    $(element).valid();
                                });
                            return (($(NEW_PASSWD).val() && ($(NEW_PASSWD).val().length == 0)) || (value == target.val()));
                        },
                        validMatchPasswd);

                    //function to check length of password. Since it is optional field,
                    //user can leave the dummy value there and modify other details.
                    //If he removes the dummy characters somehow, it will still not prompt for pwd
                    $.validator.addMethod("cicUserSettingsEditCheckPwdLength",
                        function(value, element, param) {
                            return ((value.length >= 8) || (value.length == 0));
                        },
                        validPasswdLength);

                    // check for  special characters
                    $.validator.addMethod("cicUserSettingsEditPhoneNumCheck",
                        function(value, element, param) {
                            var isValidPh = /^[^"'&\\\/|=!;<>,]+$/.test(value);
                            return (value.length == 0 || isValidPh);
                        },
                        validPhoneNum);

                    //No special characters allowed <>;,"'&\/|+.:
                    $.validator.addMethod("cicUserSettingsEditCheckPasswdString",
                        function(value, element, param) {
                            var isValidPwd = /^[^<>;,"'&\\\/|+:= ]+$/.test(value);
                            return (value.length == 0 || isValidPwd);
                        },
                        validPassword);
                }


            }

            //to clear the validators
            function functionCheck(){
              if (initialized){
                validator.resetForm();
                formStateView.reset();
              }
            }

            //initialise the dialog
            this.initialize = function(){
                registerEvents();
                functionCheck();
                dialog = new DialogView({
                    contents:userSelfEditHtml,
                    ok : onEditUserClick,
                    cancel : function() {
                        isSubmitted = true;
                        formStateView.reset();
                        removeEvents();
                    }
                });
                init();
                initUserEdit();
            };
        }

        return  new UserSettingsEditView();
    }());

    return UserSettingsEditView;
});
