// (C) Copyright 2012 Hewlett-Packard Development Company, L.P.

define(['jquery'], function() {
    (function($) {
        // jQuery plugin definition
        
        /**
         * Provides a searchable combo box control.
         *
         * Initialize with $(...).hpSearchCombo(args);
         *
         * @param {args} Object with properties as follows:
         *
         *  REQUIRED
         *    getResults: function that will be called when the drop down
         *      is made visible and when the user searches.
         *      function (string, handlers) {
         *          // generate results, can be asynchronous
         *          handlers.success({count: N, members: [{id: X, name: Y}, ...]});
         *      }
         *      You can also add "help" and "error" properties to the returned
         *      results to provide additional annotations to the choices.
         *    OR
         *    results: same object signature as returned by getResults() above.
         *      Use this for smaller, fixed data sets.
         *  OPTIONAL
         *    nameProperty: results property name for the visible label
         *    valueProperty: results property name for the invisible value
         *    maxResults: how many results to make available
         *      Don't set this too high to avoid ungainly long scrolling lists.
         *      Instead, rely on the user to search to narrow the choices.
         *    searchingMessage: text to show in the footer while searching
         *    noMatchesMessage: text to show when no results were returned
         *    matchesMessage: text to show when we've searched and matched
         *    size: how wide to make the text input, similar to <input size=""/>
         *    width: how wide to make the text input, similar to <input style="width:"/>
         *
         * Subsequently set value with $(...).hpSearchCombo('set', {...});
         *    @param {action} 'set'
         *    @param {args} can be:
         *      an object mimicing a single result returned by getResults,
         *      a string value,
         *      omitted to clear
         */
        $.fn.hpSearchCombo = function(action, args) {
          
            var SELECT = '.hp-search-combo-select';
            var SELECT_OPTION = SELECT + ' option';
            var INPUT = '.hp-search-combo-input';
            var MENU = '.hp-search-combo-menu';
            var OPTIONS = MENU + ' .hp-options';
            var OPTION = OPTIONS + ' li';
            var MESSAGE = MENU + ' .hp-message';
            var CLEAR = '.hp-close';
            var CONTROL = '.hp-search-combo-control';
            var BODY = '#hp-body-div';
            var ACTIVE = 'hp-active';
            var SELECTED = 'hp-selected';
            var DISABLED = 'hp-disabled';
            var ENTER = 13;
            var ESCAPE = 27;
            var TAB = 9;
            var UP_ARROW = 38;
            var DOWN_ARROW = 40;
            var instance; // keep Sonar happy
            
            function hpSearchCombo(elem) {
                
                var container;
                var maxResults = 6;
                var valueProperty = 'id';
                var nameProperty = 'name';
                var getResults = null;
                var fixedResults = null;
                var doSearching = false;
                var selectedOptionIndex = -1;
                var searchText;
                var searchingMessage = "Searching ...";
                var noMatchesMessage = "No matches";
                var matchesMessage = "matches";
                var inputChanged = false;   // track user input since last setSelection call
                var controlClicked = false; // track control button click
                
                function getSelection() {
                    var selectOption = $(SELECT_OPTION, container);
                    var result = null;
                    if (selectOption.length > 0) {
                        result = {name: selectOption.text()};
                        if (selectOption.attr('value')) {
                            result.value = selectOption.attr('value');
                        }
                    }
                    return result;
                }
                
                function setSelection(option) {
                    var prior = getSelection();
                    inputChanged = false;
                  
                    $(INPUT, container).val(option.name);
                    if (option.name && option.name.length > 0) {
                        $(CLEAR, container).show();
                    } else {
                        $(CLEAR, container).hide();
                    }
                    
                    $(SELECT, container).empty();
                    
                    if (option.value) {
                        $(SELECT, container).
                            append('<option value="' + option.value +
                                '" selected>' + option.name + '</option>');
                        $(SELECT, container).val(option.value);
                        if (! prior || prior.value !== option.value) {
                            $(SELECT, container).trigger('change', option.value);
                        }
                    } else {
                        $(SELECT, container).
                            append('<option selected>' + option.name + '</option>');
                        $(SELECT, container).val(option.name);
                        if (! prior || prior.name !== option.name) {
                            $(SELECT, container).trigger('change', option.name);
                        }
                    }
                }
                
                function hideMenu(ev) {
                    if (! ev || ev.target !== $(INPUT, container)[0]) {
                        $(BODY).unbind('click', hideMenu);
                        $(MENU, container).removeClass(ACTIVE).
                            css('min-width', '');
                    }
                }
                
                function showMenu() {
                    if (! $(MENU, container).hasClass(ACTIVE)) {
                        $(MENU, container).addClass(ACTIVE).
                            css('min-width', $(INPUT, container).outerWidth());
                        // avoid bouncing
                        setTimeout(function () {
                            $(BODY).bind('click', hideMenu);
                        }, 50);
                    }
                }
                
                function onOptionClick(ev) {
                    var option = $(this);
                    setSelection({name: $('.hp-name', option).text(),
                        value: option.attr('data-id')});
                    hideMenu();
                    ev.preventDefault();
                }
                
                function showMessage(text) {
                    $(MESSAGE, container).text(text).show().
                        css('min-width', $(INPUT, container).outerWidth());
                    showMenu();
                }
                
                function searchResults(string, results) {
                    var searchedResults = results;
                    if (string && string.length > 0) {
                        var newResults = 
                            $.grep(results.members, function(result, index) {
                                return (result[nameProperty].match(
                                    new RegExp(string, 'i')));
                            });
                        searchedResults = {count: newResults.length,
                            members: newResults};
                    }
                    return searchedResults;
                }
                
                function onResults(string, results) {
                    // are these are the latest results?
                    if (string === searchText) {
                        
                        // convert array results to REST collection style
                        if ($.isArray(results)) {
                            results = {count: results.length, members: results};
                        }
                        
                        if (doSearching) {
                            results = searchResults(string, results);
                        }
                        
                        $(OPTIONS, container).empty();
                        if (results.count > 0) {
                            $.each(results.members, function (index, result) {
                                var option = $('<li data-id="' +
                                    result[valueProperty] + '"></li>');
                                option.append('<span class="hp-name">' +
                                    result[nameProperty] + '</span>');
                                if (result.help) {
                                    option.append('<span class="hp-help">' +
                                        result.help + '</span>');
                                }
                                if (result.error) {
                                    option.append('<span class="hp-error">' +
                                        result.error + '</span>');
                                }
                                $(OPTIONS, container).append(option);
                                if (index >= (maxResults - 1)) {
                                    return false;
                                }
                            });
                            $(OPTION, container).click(onOptionClick);
                            
                            if (results.count > maxResults) {
                                showMessage(results.count + ' ' +
                                    matchesMessage);
                            } else {
                                $(MESSAGE, container).hide();
                            }
                        } else {
                            showMessage(noMatchesMessage);
                        }
                    }
                }
                
                function showOptions(string) {
                    selectedOptionIndex = -1;
                    $(INPUT, container).focus();
                    showMessage(searchingMessage);
                    searchText = string;
                    
                    if (getResults) {
                        getResults(string, {success: function(results) {
                            onResults(string, results);
                        }});
                    } else if (fixedResults) {
                        onResults(string, fixedResults);
                    }
                }
                
                function onControlClick() {
                    controlClicked = true;
                    $(INPUT, container).focus();
                    
                    if ($(MENU, container).hasClass(ACTIVE)) {
                        hideMenu();
                    } else if (!container.hasClass(DISABLED)) {
                        showOptions('.');
                    }
                }
                
                function onClearClick() {
                    setSelection({name: ''});
                    if ($(OPTIONS, container).hasClass(ACTIVE) &&
                        !container.hasClass(DISABLED)) {
                        showOptions('');
                    }
                }
                
                function updateSelectedOption() {
                    $(OPTION, container).removeClass(SELECTED);
                    if (selectedOptionIndex >= 0) {
                        var option =
                            $('.hp-options li:eq(' + selectedOptionIndex + ')',
                                container);
                        $(INPUT, container).val($('.hp-name', option).text());
                        option.addClass(SELECTED);
                    }
                }
                
                function onKeyDown(ev) {
                    var keyCode = (ev.which ? ev.which : ev.keyCode);
                    if (keyCode == ESCAPE || keyCode == TAB) {
                        hideMenu();
                    } else if (keyCode == ENTER) {
                        if (selectedOptionIndex >= 0) {
                            $('ol li:eq(' + selectedOptionIndex + ')', container).
                                trigger('click');
                        }
                        hideMenu();
                        ev.preventDefault();
                    } else if (keyCode == DOWN_ARROW || keyCode == UP_ARROW) {
                        ev.preventDefault();
                    }
                }
                
                function onKeyUp(ev) {
                    var keyCode = (ev.which ? ev.which : ev.keyCode);
                    if (keyCode == ENTER || keyCode == ESCAPE || keyCode == TAB) {
                        // handled by keydown
                    } else if (keyCode == DOWN_ARROW) {
                        if (! $(MENU, container).hasClass(ACTIVE)) {
                            showOptions($(INPUT, container).val());
                        } else {
                            selectedOptionIndex += 1;
                            selectedOptionIndex = Math.min($(OPTION, container).length,
                                selectedOptionIndex);
                            updateSelectedOption();
                        }
                    } else if (keyCode == UP_ARROW) {
                        selectedOptionIndex -= 1;
                        selectedOptionIndex = Math.max(0, selectedOptionIndex);
                        updateSelectedOption();
                    } else {
                        if ($(INPUT, container).val().length > 0) {
                            $(CLEAR, container).show();
                        } else {
                            $(CLEAR, container).hide();
                        }
                        showOptions($(INPUT, container).val());
                    }
                }
                
                function onInput(ev) {
                    inputChanged = true;
                    controlClicked = false;
                    if ($(INPUT, container).val().length > 0) {
                        $(CLEAR, container).show();
                    } else {
                        $(CLEAR, container).hide();
                    }
                    showOptions($(INPUT, container).val());
                }
                
                /*
                 * Evaluate if the value in the input field should be processed
                 * when the input lost focus.
                 * 
                 * The value should be processed if the input field has changed since the last
                 * setSelection() call AND focus is not lost to the user clicking the control.
                 */
                function onBlur(ev) {
                    var name = $(INPUT, container).val();
                    var selection = getSelection();
                    if (inputChanged) {
                        if (controlClicked) {
                            controlClicked = false;
                        } else if (! selection || name !== selection.name) {
                            selection = {name: name};
                            // see if we have a value we can use from an option
                            $(OPTION + ' .hp-name', container).
                                each(function (index, elem) {
                                    if (name === $(elem).text()) {
                                        selection.value =
                                            $(elem).parent().attr('data-id');
                                        return false;
                                    }
                                });
                            setSelection(selection);                      
                        }
                    }
                }
                
                function build() {
                    var selector = $(elem);
                    
                    if (selector.hasClass('hp-search-combo-select')) {
                        container = selector.parent();
                    } else {
                        selector.addClass('hp-search-combo-select');
                        selector.wrap('<div class="hp-search-combo"/>');
                        container = selector.parent();
                        selector.hide();

                        container.append('<input id="' + selector.attr('id') +
                            '-input" class="hp-search-combo-input"/>');
                        container.append('<div class="hp-close"></div>');
                        container.append('<div class="hp-search-combo-control"/>');
                        
                        // IE workaround for not handling input padding
                        if ($('html').hasClass('ie8') ||
                            $('html').hasClass('ie9')) {
                            container.append('<div class="hp-search-combo-input-mask"></div>');
                        }
                        
                        // Turn off Firefox autocomplete
                        $('input', container).attr('autocomplete', 'off');
                        
                        container.append('<div class="hp-search-combo-menu">' +
                            '<ol class="hp-options"/>' +
                            '<div class="hp-message"/></div>');
                    }
                }
                // This was pulled out of Initialize, because sonar was unhappy w/ Cyclomatic complexity.
                function initializeAction(){
                        build();
                    
                        getResults = args.getResults;
                        fixedResults = args.results;
                        if (! getResults && fixedResults) {
                            doSearching = true;
                        }
                    
                        if (args.valueProperty) {
                            valueProperty = args.valueProperty;
                        }
                        if (args.nameProperty) {
                            nameProperty = args.nameProperty;
                        }
                        if (args.maxResults) {
                            maxResults = args.maxResults;
                        }
                        if (args.searchingMessage) {
                            searchingMessage = args.searchingMessage;
                        }
                        if (args.noMatchesMessage) {
                            noMatchesMessage = args.noMatchesMessage;
                        }
                        if (args.matchesMessage) {
                            matchesMessage = args.matchesMessage;
                        }
                        if (args.size) {
                            $(INPUT, container).attr('size', args.size);
                        }
                        if (args.width) {
                            $(INPUT, container).css('width', args.width);
                        }
                    
                        $(INPUT, container).bind('keyup', onKeyUp);
                        $(INPUT, container).bind('keydown', onKeyDown);
                        $(INPUT, container).bind('input', onInput);
                        // Modernizr can't detect oninput
                        if ($('html').hasClass('ie8')) {
                            $(INPUT, container).bind('paste', function () {
                                setTimeout(onInput, 10);
                            });
                        }
                        
                        // give onControlClick() a chance to process first before onBlur.
                        $(INPUT, container).bind('blur', function() {
                            setTimeout(onBlur, 200);
                        });
                        $(CONTROL, container).bind('click', onControlClick);
                        $(CLEAR, container).bind('click', onClearClick).hide();
                }

                function initialize() {
                  
                    if (! action || 'object' === typeof(action)) {
                        args = action;
                        action = 'initialize';
                    }
                    
                    if ('initialize' === action) {
                        initializeAction();
                    } else if ('disable' === action) {
                        container = $(elem).parent();
                        container.addClass(DISABLED);
                        $('input', container).attr('disabled', 'disabled');
                    } else if ('enable' === action) {
                        container = $(elem).parent();
                        container.removeClass(DISABLED);
                        $('input', container).removeAttr('disabled');
                    } else if ('set' === action) {
                    
                        container = $(elem).parent();
                        var option = {name: ''};
                        if (args) {
                            if ('string' === typeof(args)) {
                                option.name = args;
                            } else {
                                option.name = args[nameProperty];
                                option.value = args[valueProperty];
                            }
                        }
                        setSelection(option);
                        // close menu
                        if ($(MENU, container).hasClass(ACTIVE)) {
                            $(CONTROL, container).trigger('click');
                        }
                    }
                }
                
                initialize();
            }
            
            // pluginify
            var ret;
            this.each(function() {
                var $elem = $(this);
                instance = new hpSearchCombo($elem[0]);
                ret = ret ? ret.add($elem) : $elem;
            });
            return ret;
        };
    }(jQuery));
});
