//
// Syd: rock-solid application kernel
// src/kernel/unlink.rs: rmdir(2), unlink(2) and unlinkat(2) handlers
//
// Copyright (c) 2023, 2024, 2025 Ali Polatel <alip@chesswob.org>
//
// SPDX-License-Identifier: GPL-3.0

use std::os::fd::AsFd;

use libseccomp::ScmpNotifResp;
use nix::{
    errno::Errno,
    unistd::{unlinkat, UnlinkatFlags},
};

use crate::{
    fs::FsFlags,
    hook::{PathArgs, SysArg, UNotifyEventRequest},
    kernel::syscall_path_handler,
};

pub(crate) fn sys_rmdir(request: UNotifyEventRequest) -> ScmpNotifResp {
    // rmdir() does not work on fds!
    // Hence, we have to use WANT_BASE to split base.
    let argv = &[SysArg {
        path: Some(0),
        dotlast: Some(Errno::EINVAL),
        fsflags: FsFlags::MUST_PATH | FsFlags::WANT_BASE,
        ..Default::default()
    }];
    syscall_path_handler(
        request,
        "rmdir",
        argv,
        |path_args: PathArgs, request, sandbox| {
            drop(sandbox); // release the read-lock.

            // SAFETY: SysArg has one element.
            #[allow(clippy::disallowed_methods)]
            let path = path_args.0.as_ref().unwrap();

            unlinkat(
                path.dir.as_ref().map(|fd| fd.as_fd()).ok_or(Errno::EBADF)?,
                path.base,
                UnlinkatFlags::RemoveDir,
            )
            .map(|_| request.return_syscall(0))
        },
    )
}

pub(crate) fn sys_unlink(request: UNotifyEventRequest) -> ScmpNotifResp {
    // unlink() does not work on fds!
    // Hence, we have to use WANT_BASE to split base.
    let argv = &[SysArg {
        path: Some(0),
        dotlast: Some(Errno::EINVAL),
        fsflags: FsFlags::NO_FOLLOW_LAST | FsFlags::MUST_PATH | FsFlags::WANT_BASE,
        ..Default::default()
    }];
    syscall_path_handler(
        request,
        "unlink",
        argv,
        |path_args: PathArgs, request, sandbox| {
            drop(sandbox); // release the read-lock.

            // SAFETY: SysArg has one element.
            #[allow(clippy::disallowed_methods)]
            let path = path_args.0.as_ref().unwrap();

            unlinkat(
                path.dir.as_ref().map(|fd| fd.as_fd()).ok_or(Errno::EBADF)?,
                path.base,
                UnlinkatFlags::NoRemoveDir,
            )
            .map(|_| request.return_syscall(0))
        },
    )
}

pub(crate) fn sys_unlinkat(request: UNotifyEventRequest) -> ScmpNotifResp {
    let req = request.scmpreq;

    // SAFETY: Reject undefined/invalid flags.
    let flags: libc::c_int = match req.data.args[2].try_into() {
        Ok(flags) if flags & !libc::AT_REMOVEDIR != 0 => {
            return request.fail_syscall(Errno::EINVAL)
        }
        Ok(flags) => flags,
        Err(_) => return request.fail_syscall(Errno::EINVAL),
    };

    let flags = if flags & libc::AT_REMOVEDIR != 0 {
        UnlinkatFlags::RemoveDir
    } else {
        UnlinkatFlags::NoRemoveDir
    };

    // unlinkat() does not work on fds!
    // Hence, we have to use WANT_BASE to split base.
    let argv = &[SysArg {
        dirfd: Some(0),
        path: Some(1),
        dotlast: Some(Errno::EINVAL),
        fsflags: FsFlags::NO_FOLLOW_LAST | FsFlags::MUST_PATH | FsFlags::WANT_BASE,
        ..Default::default()
    }];
    syscall_path_handler(
        request,
        "unlinkat",
        argv,
        |path_args: PathArgs, request, sandbox| {
            drop(sandbox); // release the read-lock.

            // SAFETY: SysArg has one element.
            #[allow(clippy::disallowed_methods)]
            let path = path_args.0.as_ref().unwrap();

            unlinkat(
                path.dir.as_ref().map(|fd| fd.as_fd()).ok_or(Errno::EBADF)?,
                path.base,
                flags,
            )
            .map(|_| request.return_syscall(0))
        },
    )
}
