//
// Syd: rock-solid application kernel
// src/kernel/rename.rs: rename(2), renameat(2) and renameat2(2) handlers
//
// Copyright (c) 2023, 2024, 2025 Ali Polatel <alip@chesswob.org>
//
// SPDX-License-Identifier: GPL-3.0

use std::os::fd::{AsFd, AsRawFd};

use libseccomp::{ScmpNotifResp, ScmpSyscall};
use nix::{errno::Errno, fcntl::renameat, NixPath};
use once_cell::sync::Lazy;

use crate::{
    fs::FsFlags,
    hook::{PathArgs, SysArg, UNotifyEventRequest},
    kernel::syscall_path_handler,
};

// Note renameat2 may not be available,
// and libc::SYS_renameat2 may not be defined.
// Therefore we query the number using libseccomp.
static SYS_RENAMEAT2: Lazy<libc::c_long> = Lazy::new(|| {
    ScmpSyscall::from_name("renameat2")
        .map(i32::from)
        .map(libc::c_long::from)
        .unwrap_or(-1) // Invalid system call.
});

pub(crate) fn sys_rename(request: UNotifyEventRequest) -> ScmpNotifResp {
    let argv = &[
        SysArg {
            path: Some(0),
            dotlast: Some(Errno::EINVAL),
            fsflags: FsFlags::MUST_PATH | FsFlags::NO_FOLLOW_LAST | FsFlags::WANT_BASE,
            ..Default::default()
        },
        SysArg {
            path: Some(1),
            dotlast: Some(Errno::EINVAL),
            fsflags: FsFlags::NO_FOLLOW_LAST | FsFlags::WANT_BASE,
            ..Default::default()
        },
    ];

    syscall_path_handler(
        request,
        "rename",
        argv,
        |path_args: PathArgs, request, sandbox| {
            drop(sandbox); // release the read-lock.

            syscall_rename_handler(request, path_args)
        },
    )
}

pub(crate) fn sys_renameat(request: UNotifyEventRequest) -> ScmpNotifResp {
    let argv = &[
        SysArg {
            dirfd: Some(0),
            path: Some(1),
            dotlast: Some(Errno::EINVAL),
            fsflags: FsFlags::MUST_PATH | FsFlags::NO_FOLLOW_LAST | FsFlags::WANT_BASE,
            ..Default::default()
        },
        SysArg {
            dirfd: Some(2),
            path: Some(3),
            dotlast: Some(Errno::EINVAL),
            fsflags: FsFlags::NO_FOLLOW_LAST | FsFlags::WANT_BASE,
            ..Default::default()
        },
    ];

    syscall_path_handler(
        request,
        "renameat",
        argv,
        |path_args: PathArgs, request, sandbox| {
            drop(sandbox); // release the read-lock.

            syscall_rename_handler(request, path_args)
        },
    )
}

pub(crate) fn sys_renameat2(request: UNotifyEventRequest) -> ScmpNotifResp {
    if *SYS_RENAMEAT2 < 0 {
        // renameat(2) is not implemented by host kernel.
        return request.fail_syscall(Errno::ENOSYS);
    }

    let req = request.scmpreq;
    #[allow(clippy::cast_possible_truncation)]
    let flags = req.data.args[4] as u32;
    let noreplace = flags & libc::RENAME_NOREPLACE != 0;

    let argv = &[
        SysArg {
            dirfd: Some(0),
            path: Some(1),
            dotlast: Some(Errno::EINVAL),
            fsflags: FsFlags::MUST_PATH | FsFlags::NO_FOLLOW_LAST | FsFlags::WANT_BASE,
            ..Default::default()
        },
        SysArg {
            dirfd: Some(2),
            path: Some(3),
            dotlast: Some(Errno::EINVAL),
            fsflags: if noreplace {
                FsFlags::MISS_LAST | FsFlags::NO_FOLLOW_LAST | FsFlags::WANT_BASE
            } else {
                FsFlags::NO_FOLLOW_LAST | FsFlags::WANT_BASE
            },
            ..Default::default()
        },
    ];

    syscall_path_handler(
        request,
        "renameat2",
        argv,
        |path_args: PathArgs, request, sandbox| {
            drop(sandbox); // release the read-lock.

            // SAFETY: SysArg has two elements.
            #[allow(clippy::disallowed_methods)]
            let old_path = path_args.0.as_ref().unwrap();
            #[allow(clippy::disallowed_methods)]
            let new_path = path_args.1.as_ref().unwrap();

            let old_dirfd = old_path
                .dir
                .as_ref()
                .map(|fd| fd.as_raw_fd())
                .ok_or(Errno::EBADF)?;
            let new_dirfd = new_path
                .dir
                .as_ref()
                .map(|fd| fd.as_raw_fd())
                .ok_or(Errno::EBADF)?;

            old_path
                .base
                .with_nix_path(|old_cstr| {
                    new_path.base.with_nix_path(|new_cstr| {
                        // SAFETY: musl does not define renameat2!
                        Errno::result(unsafe {
                            libc::syscall(
                                *SYS_RENAMEAT2,
                                old_dirfd,
                                old_cstr.as_ptr(),
                                new_dirfd,
                                new_cstr.as_ptr(),
                                flags,
                            )
                        })
                    })
                })??
                .map(|_| request.return_syscall(0))
        },
    )
}

/// A helper function to handle rename and renameat syscalls.
fn syscall_rename_handler(
    request: &UNotifyEventRequest,
    args: PathArgs,
) -> Result<ScmpNotifResp, Errno> {
    // SAFETY: SysArg has two elements.
    #[allow(clippy::disallowed_methods)]
    let old_path = args.0.as_ref().unwrap();
    #[allow(clippy::disallowed_methods)]
    let new_path = args.1.as_ref().unwrap();

    renameat(
        old_path
            .dir
            .as_ref()
            .map(|fd| fd.as_fd())
            .ok_or(Errno::EBADF)?,
        old_path.base,
        new_path
            .dir
            .as_ref()
            .map(|fd| fd.as_fd())
            .ok_or(Errno::EBADF)?,
        new_path.base,
    )
    .map(|_| request.return_syscall(0))
}
