// Syd: rock-solid application kernel
// src/kernel/net/accept.rs: accept(2) and accept4(2) handler
//
// Copyright (c) 2023, 2024, 2025 Ali Polatel <alip@chesswob.org>
//
// SPDX-License-Identifier: GPL-3.0

use std::os::fd::{AsRawFd, FromRawFd, OwnedFd};

use libseccomp::ScmpNotifResp;
use nix::{
    errno::Errno,
    sys::socket::{SockFlag, SockaddrLike, SockaddrStorage},
};

use crate::{
    compat::getsockdomain,
    fs::{get_nonblock, has_recv_timeout},
    hook::UNotifyEventRequest,
    kernel::net::{sandbox_addr_inet, sandbox_addr_inet6},
    op2errno,
    sandbox::Capability,
};

pub(crate) fn handle_accept(
    fd: OwnedFd,
    request: &UNotifyEventRequest,
    args: &[u64; 6],
    op: u8,
) -> Result<ScmpNotifResp, Errno> {
    // Determine the socket family.
    let fml = getsockdomain(&fd).or(Err(op2errno(op)))?;
    let ipv6 = match fml {
        libc::AF_INET6 => true,
        libc::AF_INET => false,
        _ => {
            // Not an IPv{4,6} socket, continue.
            // SAFETY: No pointer-dereference in access check.
            return unsafe { Ok(request.continue_syscall()) };
        }
    };

    // Determine address length if specified.
    let addrlen = if args[2] != 0 {
        const SIZEOF_SOCKLEN_T: usize = std::mem::size_of::<libc::socklen_t>();
        let mut buf = [0u8; SIZEOF_SOCKLEN_T];
        if request.read_mem(&mut buf, args[2])? == SIZEOF_SOCKLEN_T {
            // libc defines socklen_t as u32,
            // however we should check for negative values
            // and return EINVAL as necessary.
            let len = i32::from_ne_bytes(buf);
            let len = libc::socklen_t::try_from(len).or(Err(Errno::EINVAL))?;
            if args[1] == 0 {
                // address length is positive however address is NULL,
                // return EFAULT.
                return Err(Errno::EFAULT);
            }
            Some(len)
        } else {
            // Invalid/short read, assume invalid address length.
            return Err(Errno::EINVAL);
        }
    } else {
        None
    };

    let mut flags = if op == 0x12 {
        // accept4
        SockFlag::from_bits(args[3].try_into().or(Err(Errno::EINVAL))?).ok_or(Errno::EINVAL)?
    } else {
        // accept
        SockFlag::empty()
    };
    let cloexec = flags.contains(SockFlag::SOCK_CLOEXEC);
    flags.insert(SockFlag::SOCK_CLOEXEC);

    // SAFETY: Record blocking call so it can get invalidated.
    let req = request.scmpreq;
    let is_blocking = if !get_nonblock(&fd)? {
        let ignore_restart = has_recv_timeout(&fd)?;

        // Record the blocking call.
        request.cache.add_sys_block(req, ignore_restart)?;

        true
    } else {
        false
    };

    // Do the accept call.
    let result = do_accept4(fd, flags);

    // Remove invalidation record unless interrupted.
    if is_blocking && !matches!(result, Err(Errno::EINTR)) {
        request.cache.del_sys_block(req.id);
    }

    // Only now, bail if accept failed.
    let (fd, addr, addrlen_out) = result?;

    // Check the returned address for access.
    let sandbox = request.get_sandbox();
    if ipv6 {
        sandbox_addr_inet6(request, &sandbox, &addr, op, Capability::CAP_NET_CONNECT)?;
    } else {
        sandbox_addr_inet(request, &sandbox, &addr, op, Capability::CAP_NET_CONNECT)?;
    }
    drop(sandbox); // release the read lock.

    // Write address buffer as necessary.
    if let Some(addrlen) = addrlen {
        // Create a byte slice from the socket address pointer.
        // SAFETY:
        // 1. `addrlen_out` value is returned by the host Linux kernel
        //    and is therefore trusted.
        // 2. `ptr` is a valid pointer to memory of at least
        //    `addrlen_out` bytes, as it is provided by the
        //    `SockaddrStorage` instance.
        // 3. The `SockaddrStorage` type ensures that the memory pointed
        //    to by `ptr` is valid and properly aligned.
        let buf = unsafe { std::slice::from_raw_parts(addr.as_ptr().cast(), addrlen_out as usize) };

        // Write the truncated socket address into memory.
        // SAFETY: We truncate late to avoid potential UB in
        // std::slice::slice_from_raw_parts().
        let len = addrlen_out.min(addrlen) as usize;
        request.write_mem(&buf[..len], args[1])?;

        // Convert `addrlen_out` into a vector of bytes.
        // SAFETY: This must be socklen_t and _not_ usize!
        let buf = addrlen_out.to_ne_bytes();

        // Write `addrlen_out` into memory.
        request.write_mem(&buf, args[2])?;
    }

    // Send the fd and return.
    request.send_fd(fd, cloexec)
}

fn do_accept4(
    fd: OwnedFd,
    flags: SockFlag,
) -> Result<(OwnedFd, SockaddrStorage, libc::socklen_t), Errno> {
    // Allocate storage for the address.
    let mut addr: [u8; std::mem::size_of::<SockaddrStorage>()] =
        [0u8; std::mem::size_of::<SockaddrStorage>()];
    #[allow(clippy::cast_possible_truncation)]
    let mut len = std::mem::size_of::<SockaddrStorage>() as libc::socklen_t;

    // Cast the storage buffer to a sockaddr pointer.
    #[allow(clippy::cast_ptr_alignment)]
    let ptr = addr.as_mut_ptr() as *mut libc::sockaddr;

    // SAFETY: In libc we trust.
    let fd = Errno::result(unsafe { libc::accept4(fd.as_raw_fd(), ptr, &mut len, flags.bits()) })
        .map(|fd| {
        // SAFETY: accept4 returns a valid FD.
        unsafe { OwnedFd::from_raw_fd(fd) }
    })?;

    // SAFETY:
    // Convert the raw address into a SockaddrStorage structure.
    // accept4 returned success so the pointer is valid.
    let addr = unsafe { SockaddrStorage::from_raw(ptr, Some(len)) }.ok_or(Errno::EINVAL)?;

    Ok((fd, addr, len))
}
