//
// Syd: rock-solid application kernel
// src/kernel/mod.rs: Secure computing hooks
//
// Copyright (c) 2025 Ali Polatel <alip@chesswob.org>
//
// SPDX-License-Identifier: GPL-3.0

macro_rules! syscall_handler {
    ($request:ident, $body:expr) => {{
        let request_id = $request.scmpreq.id;

        #[allow(clippy::arithmetic_side_effects)]
        match $body($request) {
            Ok(result) => result,
            // SAFETY: Harden against UnknownErrno so as not to
            // confuse the Linux API from returning no-op.
            Err(Errno::UnknownErrno) => ScmpNotifResp::new(request_id, 0, -libc::ENOSYS, 0),
            // SAFETY: ECANCELED is used by Syd internally to denote
            // requests that should be turned into no-op.
            Err(Errno::ECANCELED) => ScmpNotifResp::new(request_id, 0, 0, 0),
            Err(errno) => ScmpNotifResp::new(request_id, 0, -(errno as i32), 0),
        }
    }};
}

/// access(2), faccessat(2) and faccessat2(2) handlers
pub(crate) mod access;

/// chdir(2) and fchdir(2) handlers
pub(crate) mod chdir;

/// chmod(2), fchmod(2), fchmodat(2), and fchmodat2(2) handlers
pub(crate) mod chmod;

/// chown(2), lchown(2), fchown(2), and fchownat(2) handlers
pub(crate) mod chown;

/// chroot(2) handler
pub(crate) mod chroot;

/// exec(3) handlers
pub(crate) mod exec;

/// fanotify_mark(2) handler
pub(crate) mod fanotify;

/// fcntl{,64}(2) handlers
pub(crate) mod fcntl;

/// getdents64(2) handler
pub(crate) mod getdents;

/// inotify_add_watch(2) handler
pub(crate) mod inotify;

/// ioctl(2) handlers
pub(crate) mod ioctl;

/// link(2) and linkat(2) handlers
pub(crate) mod link;

/// Memory syscall handlers
pub(crate) mod mem;

/// memfd_create(2) handler
pub(crate) mod memfd;

/// mkdir(2) and mkdirat(2) handlers
pub(crate) mod mkdir;

/// mknod(2) and mknodat(2) handlers
pub(crate) mod mknod;

/// Network syscall handlers
pub(crate) mod net;

/// creat(2), open(2), openat(2), and openat2(2) handlers
pub(crate) mod open;

/// prctl(2) handler
pub(crate) mod prctl;

/// rename(2), renameat(2) and renameat2(2) handlers
pub(crate) mod rename;

/// Set UID/GID syscall handlers
pub(crate) mod setid;

/// {,rt_}sigaction(2) handler
pub(crate) mod sigaction;

/// {,rt_}sigreturn(2) handler
pub(crate) mod sigreturn;

/// Signal syscall handlers
pub(crate) mod signal;

/// stat syscall handlers
pub(crate) mod stat;

/// statfs syscall handlers
pub(crate) mod statfs;

/// symlink(2) and symlinkat(2) handlers
pub(crate) mod symlink;

/// sysinfo(2) handler
pub(crate) mod sysinfo;

/// syslog(2) handler
pub(crate) mod syslog;

/// truncate and allocate handlers
pub(crate) mod truncate;

/// uname(2) handler
pub(crate) mod uname;

/// utime handlers
pub(crate) mod utime;

/// rmdir(2), unlink(2) and unlinkat(2) handlers
pub(crate) mod unlink;

/// xattr handlers
pub(crate) mod xattr;

use std::{borrow::Cow, os::fd::RawFd};

use libseccomp::ScmpNotifResp;
use memchr::memmem;
use nix::{errno::Errno, fcntl::AtFlags, sys::stat::Mode};

use crate::{
    fs::{CanonicalPath, FileInfo, FileType},
    hook::{PathArgs, RemoteProcess, SysArg, SysFlags, UNotifyEventRequest},
    notice,
    path::{XPath, XPathBuf},
    sandbox::{Action, Capability, SandboxGuard},
    warn,
};

/// Process the given path argument.
#[allow(clippy::cognitive_complexity)]
#[allow(clippy::too_many_arguments)]
pub(crate) fn sandbox_path(
    request: Option<&UNotifyEventRequest>,
    sandbox: &SandboxGuard,
    process: &RemoteProcess,
    path: &XPath,
    caps: Capability,
    hide: bool,
    syscall_name: &str,
) -> Result<(), Errno> {
    // Check for chroot.
    if sandbox.is_chroot() {
        return Err(Errno::ENOENT);
    }

    // Check enabled capabilities.
    let caps_old = caps;
    let mut caps = sandbox.getcaps(caps);
    let stat = sandbox.enabled(Capability::CAP_STAT);
    if caps.is_empty() && (!hide || !stat) {
        return if caps_old.intersects(Capability::CAP_WRSET) && sandbox.is_append(path) {
            // SAFETY: Protect append-only paths against writes.
            // We use ECANCELED which will result in a no-op.
            Err(Errno::ECANCELED)
        } else {
            Ok(())
        };
    }

    // Convert /proc/${pid} to /proc/self as necessary.
    let path = if let Some(p) = path.split_prefix(b"/proc") {
        let mut buf = itoa::Buffer::new();
        let pid = buf.format(process.pid.as_raw());
        if let Some(p) = p.split_prefix(pid.as_bytes()) {
            let mut pdir = XPathBuf::from("/proc/self");
            pdir.push(p.as_bytes());
            Cow::Owned(pdir)
        } else {
            Cow::Borrowed(path)
        }
    } else {
        Cow::Borrowed(path)
    };

    let mut action = Action::Allow;
    let mut filter = false;
    let mut deny_errno = Errno::EACCES;

    // Sandboxing.
    for cap in caps & Capability::CAP_PATH {
        let (new_action, new_filter) = sandbox.check_path(cap, &path);

        if new_action >= action {
            action = new_action;
        }
        if !filter && new_filter {
            filter = true;
        }
    }

    // SAFETY: Do an additional stat check to correct errno to ENOENT,
    // for sandboxing types other than Stat.
    let check_hidden = stat && hide && (caps.is_empty() || action.is_denying());
    if check_hidden || caps.contains(Capability::CAP_STAT) {
        let (new_action, new_filter) = sandbox.check_path(Capability::CAP_STAT, &path);

        if !check_hidden {
            deny_errno = Errno::ENOENT;
            action = new_action;
            filter = new_filter;
        } else if new_action.is_denying() {
            deny_errno = Errno::ENOENT;
            if caps.is_empty() {
                action = new_action;
                filter = new_filter;
                caps.insert(Capability::CAP_STAT);
            }
        }

        if path.is_rootfs() && deny_errno == Errno::ENOENT {
            // SAFETY: No point in hiding `/`.
            deny_errno = Errno::EACCES;
        }
    }

    if !filter && action >= Action::Warn {
        // Log warn for normal cases.
        // Log info for path hiding unless explicitly specified to warn.
        let is_warn = if caps != Capability::CAP_STAT {
            true
        } else {
            !matches!(
                sandbox.default_action(Capability::CAP_STAT),
                Action::Filter | Action::Deny
            )
        };

        if let Some(request) = request {
            let args = request.scmpreq.data.args;
            if sandbox.verbose {
                if is_warn {
                    warn!("ctx": "access", "cap": caps, "act": action,
                        "sys": syscall_name, "path": &path, "args": args,
                        "tip": format!("configure `allow/{}+{}'",
                            caps.to_string().to_ascii_lowercase(),
                            path),
                        "req": request);
                } else {
                    notice!("ctx": "access", "cap": caps, "act": action,
                        "sys": syscall_name, "path": &path, "args": args,
                        "tip": format!("configure `allow/{}+{}'",
                            caps.to_string().to_ascii_lowercase(),
                            path),
                        "req": request);
                }
            } else if is_warn {
                warn!("ctx": "access", "cap": caps, "act": action,
                    "sys": syscall_name, "path": &path, "args": args,
                    "tip": format!("configure `allow/{}+{}'",
                        caps.to_string().to_ascii_lowercase(),
                        path),
                    "pid": request.scmpreq.pid);
            } else {
                notice!("ctx": "access", "cap": caps, "act": action,
                    "sys": syscall_name, "path": &path, "args": args,
                    "tip": format!("configure `allow/{}+{}'",
                        caps.to_string().to_ascii_lowercase(),
                        path),
                    "pid": request.scmpreq.pid);
            }
        } else if is_warn {
            warn!("ctx": "access", "cap": caps, "act": action,
                "sys": syscall_name, "path": &path,
                "tip": format!("configure `allow/{}+{}'",
                    caps.to_string().to_ascii_lowercase(),
                    path),
                "pid": process.pid.as_raw());
        } else {
            notice!("ctx": "access", "cap": caps, "act": action,
                "sys": syscall_name, "path": &path,
                "tip": format!("configure `allow/{}+{}'",
                    caps.to_string().to_ascii_lowercase(),
                    path),
                "pid": process.pid.as_raw());
        }
    }

    match action {
        Action::Allow | Action::Warn => {
            if caps.intersects(Capability::CAP_WRSET) && sandbox.is_append(&path) {
                // SAFETY: Protect append-only paths against writes.
                // We use ECANCELED which will result in a no-op.
                Err(Errno::ECANCELED)
            } else {
                Ok(())
            }
        }
        Action::Deny | Action::Filter => Err(deny_errno),
        Action::Panic => panic!(),
        Action::Exit => std::process::exit(deny_errno as i32),
        Action::Stop => {
            if let Some(request) = request {
                let _ = request.pidfd_kill(libc::SIGSTOP);
            } else {
                let _ = process.pidfd_kill(libc::SIGSTOP);
            }
            Err(deny_errno)
        }
        Action::Abort => {
            if let Some(request) = request {
                let _ = request.pidfd_kill(libc::SIGABRT);
            } else {
                let _ = process.pidfd_kill(libc::SIGABRT);
            }
            Err(deny_errno)
        }
        Action::Kill => {
            if let Some(request) = request {
                let _ = request.pidfd_kill(libc::SIGKILL);
            } else {
                let _ = process.pidfd_kill(libc::SIGKILL);
            }
            Err(deny_errno)
        }
    }
}

///
/// Handles syscalls related to paths, reducing code redundancy and ensuring a uniform way of dealing with paths.
///
/// # Parameters
///
/// - `request`: User notification request from seccomp.
/// - `syscall_name`: The name of the syscall being handled, used for logging and error reporting.
/// - `arg_mappings`: Non-empty list of argument mappings containing dirfd and path indexes, if applicable.
/// - `handler`: Closure that processes the constructed canonical paths and performs additional syscall-specific operations.
///
/// # Returns
///
/// - `ScmpNotifResp`: Response indicating the result of the syscall handling.
#[allow(clippy::cognitive_complexity)]
pub(crate) fn syscall_path_handler<H>(
    request: UNotifyEventRequest,
    syscall_name: &str,
    path_argv: &[SysArg],
    handler: H,
) -> ScmpNotifResp
where
    H: Fn(PathArgs, &UNotifyEventRequest, SandboxGuard) -> Result<ScmpNotifResp, Errno>,
{
    syscall_handler!(request, |request: UNotifyEventRequest| {
        let req = request.scmpreq;

        // Determine system call capabilities.
        let mut caps = Capability::try_from((req, syscall_name))?;

        // Check for chroot:
        //
        // Delay Chdir to allow the common `cd /`. use case
        // right after chroot.
        let sandbox = request.get_sandbox();
        if sandbox.is_chroot() && !caps.contains(Capability::CAP_CHDIR) {
            return Err(Errno::ENOENT);
        }

        // If sandboxing for all the selected capabilities is off, return immediately.
        let crypt = sandbox.enabled(Capability::CAP_CRYPT);
        let hide = sandbox.enabled(Capability::CAP_STAT);

        // EXCEPTION: We do want to return success
        // to _access_(2) calls to magic paths in
        // case the sandbox lock allows it.
        let mut magic = !sandbox.locked_for(req.pid())
            && memmem::find_iter(syscall_name.as_bytes(), b"access")
                .next()
                .is_some();

        let mut paths: [Option<CanonicalPath>; 2] = [None, None];
        for (idx, arg) in path_argv.iter().enumerate() {
            // Handle system calls that take a FD only,
            // such as fchmod, fchown, falllocate, ftruncate,
            // fgetxattr, fsetxattr safely and efficiently.
            if arg.path.is_some() {
                let (path, is_magic) = request.read_path(&sandbox, *arg, magic)?;
                magic = is_magic;

                if sandbox.is_chroot() {
                    return if caps.contains(Capability::CAP_CHDIR) && path.abs().is_rootfs() {
                        // SAFETY: Allow `cd /` after chroot.
                        Ok(unsafe { request.continue_syscall() })
                    } else {
                        Err(Errno::ENOENT)
                    };
                }

                paths[idx] = Some(path);
            } else if let Some(arg_dirfd) = arg.dirfd {
                #[allow(clippy::cast_possible_truncation)]
                let dirfd = req.data.args[arg_dirfd] as RawFd;

                if sandbox.is_chroot() {
                    return if caps.contains(Capability::CAP_CHDIR) {
                        // SAFETY: Do not allow fchdir after chroot.
                        Err(Errno::EACCES)
                    } else {
                        Err(Errno::ENOENT)
                    };
                }

                if dirfd != libc::AT_FDCWD {
                    // SAFETY: Get the file descriptor before access check
                    // as it may change after which is a TOCTOU vector.
                    let fd = request.get_fd(dirfd)?;

                    // Handle ftruncate etc. for files with encryption in progress.
                    let crypt_path = if crypt {
                        if let Ok(info) = FileInfo::from_fd(&fd) {
                            let mut found = None;
                            #[allow(clippy::disallowed_methods)]
                            let files = request.crypt_map.as_ref().unwrap();
                            for (path, map) in
                                &files.read().unwrap_or_else(|err| err.into_inner()).0
                            {
                                if info == map.info {
                                    found = Some(path.clone());
                                    break;
                                }
                            }
                            found
                        } else {
                            None
                        }
                    } else {
                        None
                    };

                    let mut path = if let Some(path) = crypt_path {
                        // SAFETY: Only regular files are encrypted.
                        CanonicalPath::new(path, FileType::Reg, arg.fsflags)?
                    } else {
                        CanonicalPath::new_fd(fd.into(), req.pid(), dirfd)?
                    };

                    if arg.flags.contains(SysFlags::UNSAFE_CONT) {
                        // FD not required if we're continuing...
                        path.dir = None;
                    }

                    paths[idx] = Some(path);
                } else {
                    let mut path =
                        CanonicalPath::new_fd(libc::AT_FDCWD.into(), req.pid(), libc::AT_FDCWD)?;

                    if arg.flags.contains(SysFlags::UNSAFE_CONT) {
                        // FD not required if we're continuing...
                        path.dir = None;
                    }

                    paths[idx] = Some(path);
                }
            } else {
                unreachable!("BUG: Both dirfd and path are None in SysArg!");
            }
        }

        if !magic {
            // Unused when request.is_some()
            let process = RemoteProcess::new(request.scmpreq.pid());

            // Call sandbox access checker, skip magic paths.
            match (&paths[0], &paths[1]) {
                (Some(path), None) => {
                    // Adjust capabilities.
                    if caps.contains(Capability::CAP_CREATE) && path.typ.is_some() {
                        caps.remove(Capability::CAP_CREATE);
                    }
                    if caps.contains(Capability::CAP_DELETE) && path.typ.is_none() {
                        caps.remove(Capability::CAP_DELETE);
                    }
                    if caps.contains(Capability::CAP_CHDIR) && path.typ != Some(FileType::Dir) {
                        caps.remove(Capability::CAP_CHDIR);
                    }
                    if caps.contains(Capability::CAP_MKDIR) && path.typ.is_some() {
                        caps.remove(Capability::CAP_MKDIR);
                    }

                    sandbox_path(
                        Some(&request),
                        &sandbox,
                        &process,
                        path.abs(),
                        caps,
                        hide,
                        syscall_name,
                    )?
                }
                (Some(path_0), Some(path_1)) => {
                    // link, linkat, rename, renameat, renameat2.
                    // All of which have RENAME capability.
                    // It's the second argument that is being
                    // created.
                    sandbox_path(
                        Some(&request),
                        &sandbox,
                        &process,
                        path_0.abs(),
                        Capability::CAP_RENAME,
                        hide,
                        syscall_name,
                    )?;

                    // Careful, rename* may overwrite, link* must create.
                    if path_1.typ.is_none() || !path_argv[1].fsflags.missing() {
                        sandbox_path(
                            Some(&request),
                            &sandbox,
                            &process,
                            path_1.abs(),
                            Capability::CAP_CREATE,
                            hide,
                            syscall_name,
                        )?;
                    }
                }
                _ => unreachable!("BUG: number of path arguments is not 1 or 2!"),
            }
        }

        // SAFETY: Path hiding is done, now it is safe to:
        //
        // 1. Return EEXIST if options had MISS_LAST.
        // 2. Return ENOTDIR for non-directories with trailing slash.
        for (idx, path) in paths.iter_mut().enumerate() {
            if let Some(path) = path {
                let arg = if let Some(arg) = path_argv.get(idx) {
                    arg
                } else {
                    break;
                };

                if arg.fsflags.missing() && path.typ.is_some() {
                    return Err(Errno::EEXIST);
                }

                if let Some(file_type) = &path.typ {
                    if !matches!(file_type, FileType::Dir | FileType::MagicLnk(_, _))
                        && path.abs().last() == Some(b'/')
                    {
                        return Err(Errno::ENOTDIR);
                    }
                }
            }
        }

        // Call the system call handler.
        handler(
            PathArgs(paths[0].take(), paths[1].take()),
            &request,
            sandbox,
        )
    })
}

// Convert system call argument to AtFlags safely.
// Use `valid` to limit set of valid AtFlags.
#[inline]
pub(crate) fn to_atflags(arg: u64, valid: AtFlags) -> Result<AtFlags, Errno> {
    // SAFETY: Reject undefined flags.
    let flags = arg.try_into().or(Err(Errno::EINVAL))?;

    // SAFETY: Keep invalid flags for future compat!
    let flags = AtFlags::from_bits_retain(flags);

    // SAFETY: Reject unused flags.
    if !flags.difference(valid).is_empty() {
        return Err(Errno::EINVAL);
    }

    Ok(flags)
}

#[inline]
fn to_mode(arg: u64) -> Result<Mode, Errno> {
    let mode = arg.try_into().or(Err(Errno::EINVAL))?;
    Mode::from_bits(mode).ok_or(Errno::EINVAL)
}

#[cfg(test)]
mod tests {
    use super::*;
    use crate::fs::AT_EXECVE_CHECK;

    #[test]
    fn test_to_atflags() {
        let valid = AtFlags::AT_SYMLINK_NOFOLLOW | AtFlags::AT_EMPTY_PATH | AT_EXECVE_CHECK;
        assert_eq!(to_atflags(valid.bits() as u64, valid), Ok(valid));

        let invalid = AtFlags::AT_REMOVEDIR;
        assert_eq!(to_atflags(invalid.bits() as u64, valid), Err(Errno::EINVAL));
        assert_eq!(
            to_atflags((valid | invalid).bits() as u64, valid),
            Err(Errno::EINVAL)
        );
        assert_eq!(
            to_atflags((valid | invalid).bits() as u64, valid | invalid),
            Ok(valid | invalid)
        );
    }
}
