//
// Syd: rock-solid application kernel
// src/kernel/mknod.rs: mknod(2) and mknodat(2) handlers
//
// Copyright (c) 2023, 2024, 2025 Ali Polatel <alip@chesswob.org>
//
// SPDX-License-Identifier: GPL-3.0

use std::os::fd::AsFd;

use libseccomp::ScmpNotifResp;
use nix::{
    errno::Errno,
    sys::stat::{mknodat, umask, Mode, SFlag},
};

use crate::{
    fs::FsFlags,
    hook::{PathArgs, SysArg, UNotifyEventRequest},
    kernel::syscall_path_handler,
    proc::proc_umask,
};

pub(crate) fn sys_mknod(request: UNotifyEventRequest) -> ScmpNotifResp {
    let req = request.scmpreq;

    // SAFETY: Reject undefined/invalid kind.
    let kind = match to_sflag(req.data.args[1]) {
        Ok(kind) => kind,
        Err(errno) => return request.fail_syscall(errno),
    };

    // SAFETY: Reject undefined/invalid perm.
    let perm = match to_mode(req.data.args[1]) {
        Ok(perm) => perm,
        Err(errno) => return request.fail_syscall(errno),
    };

    // SAFETY: Reject invalid dev.
    #[allow(clippy::useless_conversion)]
    let dev: libc::dev_t = match req.data.args[2].try_into() {
        Ok(dev) => dev,
        Err(_) => return request.fail_syscall(Errno::EINVAL),
    };

    // We want NO_FOLLOW_LAST because creating an entry
    // through a dangling symbolic link should return EEXIST!
    let argv = &[SysArg {
        path: Some(0),
        fsflags: FsFlags::MISS_LAST | FsFlags::NO_FOLLOW_LAST,
        ..Default::default()
    }];
    syscall_path_handler(
        request,
        "mknod",
        argv,
        |path_args: PathArgs, request, sandbox| {
            let umask = sandbox.umask;
            drop(sandbox); // release the read-lock.
            syscall_mknod_handler(request, path_args, kind, perm, dev, umask)
        },
    )
}

pub(crate) fn sys_mknodat(request: UNotifyEventRequest) -> ScmpNotifResp {
    let req = request.scmpreq;

    // SAFETY: Reject undefined/invalid kind.
    let kind = match to_sflag(req.data.args[2]) {
        Ok(kind) => kind,
        Err(errno) => return request.fail_syscall(errno),
    };

    // SAFETY: Reject undefined/invalid perm.
    let perm = match to_mode(req.data.args[2]) {
        Ok(perm) => perm,
        Err(errno) => return request.fail_syscall(errno),
    };

    // SAFETY: Reject invalid dev.
    #[allow(clippy::useless_conversion)]
    let dev: libc::dev_t = match req.data.args[3].try_into() {
        Ok(dev) => dev,
        Err(_) => return request.fail_syscall(Errno::EINVAL),
    };

    // We want NO_FOLLOW_LAST because creating an entry
    // through a dangling symbolic link should return EEXIST!
    let argv = &[SysArg {
        dirfd: Some(0),
        path: Some(1),
        fsflags: FsFlags::MISS_LAST | FsFlags::NO_FOLLOW_LAST,
        ..Default::default()
    }];
    syscall_path_handler(
        request,
        "mknodat",
        argv,
        |path_args: PathArgs, request, sandbox| {
            let umask = sandbox.umask;
            drop(sandbox); // release the read-lock.
            syscall_mknod_handler(request, path_args, kind, perm, dev, umask)
        },
    )
}

/// A helper function to handle mknod* syscalls.
fn syscall_mknod_handler(
    request: &UNotifyEventRequest,
    args: PathArgs,
    kind: SFlag,
    mut perm: Mode,
    dev: libc::dev_t,
    force_umask: Option<Mode>,
) -> Result<ScmpNotifResp, Errno> {
    // SAFETY: SysArg has one element.
    #[allow(clippy::disallowed_methods)]
    let path = args.0.as_ref().unwrap();

    // SAFETY:
    // 1. force_umask is only applied to regular files.
    // 2. force_umask overrides POSIX ACLs.
    if kind == SFlag::S_IFREG {
        if let Some(mask) = force_umask {
            perm &= !mask;
        }
    }

    let req = request.scmpreq;
    let mask = proc_umask(req.pid())?;

    // SAFETY: Honour process' umask.
    // Note, the umask is per-thread here.
    // Note, POSIX ACLs may override this.
    umask(mask);

    mknodat(
        path.dir.as_ref().map(|fd| fd.as_fd()).ok_or(Errno::EBADF)?,
        path.base,
        kind,
        perm,
        dev,
    )
    .map(|_| request.return_syscall(0))
}

#[inline]
fn to_sflag(arg: u64) -> Result<SFlag, Errno> {
    let kind = arg
        .try_into()
        .map(|kind: libc::mode_t| kind & SFlag::S_IFMT.bits())
        .or(Err(Errno::EINVAL))?;

    // Careful here, zero file type is equivalent to S_IFREG.
    if kind == 0 {
        Ok(SFlag::S_IFREG)
    } else {
        SFlag::from_bits(kind).ok_or(Errno::EINVAL)
    }
}

#[inline]
fn to_mode(arg: u64) -> Result<Mode, Errno> {
    let mode = arg
        .try_into()
        .map(|mode: libc::mode_t| mode & !SFlag::S_IFMT.bits())
        .or(Err(Errno::EINVAL))?;
    Mode::from_bits(mode).ok_or(Errno::EINVAL)
}
