//
// Syd: rock-solid application kernel
// src/kernel/inotify.rs: inotify_add_watch(2) handler
//
// Copyright (c) 2023, 2024, 2025 Ali Polatel <alip@chesswob.org>
//
// SPDX-License-Identifier: GPL-3.0

use std::os::fd::{AsFd, AsRawFd, RawFd};

use libseccomp::ScmpNotifResp;
use nix::{errno::Errno, sys::inotify::AddWatchFlags, NixPath};

use crate::{
    fs::{fd_mode, inotify_add_watch, is_sidechannel_device, FsFlags},
    hook::{PathArgs, SysArg, UNotifyEventRequest},
    kernel::syscall_path_handler,
    path::XPathBuf,
};

pub(crate) fn sys_inotify_add_watch(request: UNotifyEventRequest) -> ScmpNotifResp {
    let req = request.scmpreq;

    // SAFETY: Reject undefined masks.
    let mask: u32 = match req.data.args[2].try_into() {
        Ok(mask) => mask,
        Err(_) => return request.fail_syscall(Errno::EINVAL),
    };

    // SAFETY: Reject invalid masks.
    let mask = match AddWatchFlags::from_bits(mask) {
        Some(mask) => mask,
        None => return request.fail_syscall(Errno::EINVAL),
    };

    // inotify(7) requires read access to the file or directory,
    // with the exception of symbolic links. However, due to
    // the `/proc' indirection in the handler we must not pass
    // WANT_READ here. See LTP inotify tests.
    let mut fsflags = FsFlags::MUST_PATH;
    if mask.contains(AddWatchFlags::IN_DONT_FOLLOW) {
        fsflags |= FsFlags::NO_FOLLOW_LAST;
    }

    let argv = &[SysArg {
        dirfd: None,
        path: Some(1),
        fsflags,
        ..Default::default()
    }];
    syscall_path_handler(
        request,
        "inotify_add_watch",
        argv,
        |path_args: PathArgs, request, sandbox| {
            drop(sandbox); // release the read-lock.

            // SAFETY: SysArg has one element.
            #[allow(clippy::disallowed_methods)]
            let path = path_args.0.as_ref().unwrap();

            if !path.base.is_empty() {
                // SAFETY: Ensure we have a direct FD.
                unreachable!("BUG: inotify_add_watch with base path!");
            }

            // SAFETY: ^^ empty base asserts dir is Some.
            #[allow(clippy::disallowed_methods)]
            let dfd = path.dir.as_ref().map(|fd| fd.as_fd()).unwrap();

            // SAFETY:
            // 1. Strip IN_{ACCESS,MODIFY} if we're marking a sidechannel device.
            // 2. Strip IN_DONT_FOLLOW which has already been handled during canonicalization.
            let st_mode = fd_mode(dfd)?;
            let mut mask = mask & !AddWatchFlags::IN_DONT_FOLLOW;
            if is_sidechannel_device(st_mode) {
                mask.remove(AddWatchFlags::IN_ACCESS);
                mask.remove(AddWatchFlags::IN_MODIFY);
            }

            // SAFETY: We open a FD to the path and then use the
            // proc path /proc/self/fd/$fd in address' path argument
            // to avoid symlink TOCTOU.
            let mut pfd = XPathBuf::from("/proc/self/fd");
            pfd.push_fd(dfd.as_raw_fd());

            // Get the INotify FD.
            #[allow(clippy::cast_possible_truncation)]
            let fd = request.get_fd(req.data.args[0] as RawFd)?;

            inotify_add_watch(&fd, &pfd, mask)
                .map(|retval| request.return_syscall(i64::from(retval)))
        },
    )
}
