//
// Syd: rock-solid application kernel
// src/kernel/chdir.rs: chdir(2) and fchdir(2) handlers
//
// Copyright (c) 2023, 2024, 2025 Ali Polatel <alip@chesswob.org>
//
// SPDX-License-Identifier: GPL-3.0

use libseccomp::ScmpNotifResp;
use nix::errno::Errno;

use crate::{
    compat::{fstatx, statx, STATX_INO, STATX_MNT_ID, STATX_MNT_ID_UNIQUE},
    config::{HAVE_STATX_MNT_ID_UNIQUE, PROC_FILE},
    debug, error,
    fs::{readlinkat, CanonicalPath, FsFlags},
    hook::{PathArgs, RemoteProcess, SysArg, SysFlags, UNotifyEventRequest},
    kernel::{sandbox_path, syscall_path_handler},
    path::XPathBuf,
    ptrace::{ptrace_get_error, ptrace_syscall_info, ptrace_syscall_info_seccomp},
    sandbox::{Capability, SandboxGuard},
};

// Note, chdir is a ptrace(2) hook, not a seccomp hook!
// The seccomp hook is only used with trace/allow_unsafe_ptrace:1.
pub(crate) fn sysenter_chdir<'a>(
    process: &RemoteProcess,
    sandbox: &SandboxGuard,
    data: ptrace_syscall_info_seccomp,
) -> Result<CanonicalPath<'a>, Errno> {
    let mut arg = SysArg {
        path: Some(0),
        ..Default::default()
    };

    // SAFETY: Apply deny_dotdot as necessary for chdir.
    if sandbox.deny_dotdot() {
        arg.fsflags.insert(FsFlags::NO_RESOLVE_DOTDOT);
    }

    // Read remote path.
    let (path, _, _) =
        // SAFETY: PidFd is validated.
        unsafe { process.read_path(sandbox, data.args, arg, false, None) }?;
    if !process.is_alive() {
        return Err(Errno::ESRCH);
    }

    // Check for chroot, allow for the
    // common `cd /` use case.
    if sandbox.is_chroot() {
        return if path.abs().is_rootfs() {
            Ok(CanonicalPath::new_root())
        } else {
            Err(Errno::ENOENT)
        };
    }

    let mut caps = Capability::empty();
    if let Some(typ) = path.typ.as_ref() {
        if typ.is_dir() {
            caps.insert(Capability::CAP_CHDIR);
        }
    } else {
        return Err(Errno::ENOENT);
    }

    sandbox_path(None, sandbox, process, path.abs(), caps, true, "chdir")?;

    if !caps.contains(Capability::CAP_CHDIR) {
        // SAFETY: Return this after sandboxing
        // to honour hidden paths.
        return Err(Errno::ENOTDIR);
    }

    Ok(path)
}

#[allow(clippy::cognitive_complexity)]
pub(crate) fn sysexit_chdir(
    process: RemoteProcess,
    info: ptrace_syscall_info,
    path: CanonicalPath,
) -> Result<(), Errno> {
    // Check for successful sigaction exit.
    match ptrace_get_error(process.pid, info.arch) {
        Ok(None) => {
            // Successful chdir call, validate CWD magiclink.
        }
        Ok(Some(_)) => {
            // Unsuccessful chdir call, continue process.
            return Ok(());
        }
        Err(_) => {
            // SAFETY: Failed to get return value,
            // terminate the process.
            let _ = process.pidfd_kill(libc::SIGKILL);
            return Err(Errno::ESRCH);
        }
    };

    // SAFETY: Validate /proc/$pid/cwd against TOCTTOU!
    let mut pfd = XPathBuf::from_pid(process.pid);
    pfd.push(b"cwd");

    let mut mask = STATX_INO;
    mask |= if *HAVE_STATX_MNT_ID_UNIQUE {
        STATX_MNT_ID_UNIQUE
    } else {
        STATX_MNT_ID
    };

    #[allow(clippy::disallowed_methods)]
    let fd = path.dir.as_ref().unwrap();

    let stx_fd = match fstatx(fd, mask) {
        Ok(stx) => stx,
        Err(errno) => {
            // SAFETY: Failed to stat FD,
            // assume TOCTTOU: terminate the process.
            error!("ctx": "chdir", "op": "fstat_dir_fd",
                "err": format!("failed to fstat dir-fd for `{path}': {errno}"),
                "pid": process.pid.as_raw(),
                "path": &path,
                "errno": errno as i32);
            let _ = process.pidfd_kill(libc::SIGKILL);
            return Err(Errno::ESRCH);
        }
    };

    let stx_cwd = match statx(PROC_FILE(), &pfd, 0, mask) {
        Ok(stx) => stx,
        Err(errno) => {
            // SAFETY: Failed to stat CWD,
            // assume TOCTTOU: terminate the process.
            error!("ctx": "chdir", "op": "stat_cwd_symlink",
                "err": format!("failed to stat cwd-symlink for `{path}': {errno}"),
                "pid": process.pid.as_raw(),
                "path": &path,
                "errno": errno as i32);
            let _ = process.pidfd_kill(libc::SIGKILL);
            return Err(Errno::ESRCH);
        }
    };

    // SAFETY: Validate CWD stat information.
    let mut is_match = true;

    // Step 1: Check inodes.
    if stx_fd.stx_ino != stx_cwd.stx_ino {
        is_match = false;
    }

    // Step 2: Compare mount ids.
    if stx_fd.stx_mnt_id != stx_cwd.stx_mnt_id {
        is_match = false;
    }

    if !is_match {
        // SAFETY: CWD changed, which indicates
        // successful TOCTTOU attempt: terminate the process.
        let cwd = readlinkat(PROC_FILE(), &pfd)
            .ok()
            .unwrap_or_else(|| XPathBuf::from("?"));
        error!("ctx": "chdir", "op": "dir_mismatch",
            "err": format!("dir mismatch detected for directory `{path}' -> `{cwd}': assume TOCTTOU!"),
            "pid": process.pid.as_raw(),
            "path": &path,
            "real": cwd,
            "cwd_mount_id": stx_cwd.stx_mnt_id,
            "dir_mount_id": stx_fd.stx_mnt_id,
            "cwd_inode": stx_cwd.stx_ino,
            "dir_inode": stx_fd.stx_ino);
        let _ = process.pidfd_kill(libc::SIGKILL);
        return Err(Errno::ESRCH);
    } else {
        debug!("ctx": "chdir", "op": "verify_chdir",
            "msg": format!("dir change to `{path}' approved"),
            "pid": process.pid.as_raw(),
            "path": &path,
            "cwd_mount_id": stx_cwd.stx_mnt_id,
            "dir_mount_id": stx_fd.stx_mnt_id,
            "cwd_inode": stx_cwd.stx_ino,
            "dir_inode": stx_fd.stx_ino);
    }

    // Continue process.
    Ok(())
}
pub(crate) fn sys_chdir(request: UNotifyEventRequest) -> ScmpNotifResp {
    let argv = &[SysArg {
        path: Some(0),
        flags: SysFlags::UNSAFE_CONT,
        ..Default::default()
    }];

    syscall_path_handler(
        request,
        "chdir",
        argv,
        |path_args: PathArgs, request, sandbox| {
            drop(sandbox); // release the read-lock.

            // SAFETY: SysArg has one element.
            #[allow(clippy::disallowed_methods)]
            if let Some(typ) = path_args.0.as_ref().unwrap().typ.as_ref() {
                if !typ.is_dir() {
                    return Ok(request.fail_syscall(Errno::ENOTDIR));
                }
            } else {
                return Ok(request.fail_syscall(Errno::ENOENT));
            }

            // SAFETY: This is vulnerable to TOCTTOU.
            // We only use this hook with trace/allow_unsafe_ptrace:1
            // hence the user is aware of the consequences.
            Ok(unsafe { request.continue_syscall() })
        },
    )
}

pub(crate) fn sys_fchdir(request: UNotifyEventRequest) -> ScmpNotifResp {
    // SAFETY: fchdir is fd-only, so UNSAFE_CONT is ok.
    let argv = &[SysArg {
        dirfd: Some(0),
        flags: SysFlags::UNSAFE_CONT,
        ..Default::default()
    }];

    syscall_path_handler(
        request,
        "fchdir",
        argv,
        |path_args: PathArgs, request, sandbox| {
            drop(sandbox); // release the read-lock.

            // SAFETY: SysArg has one element.
            #[allow(clippy::disallowed_methods)]
            if let Some(typ) = path_args.0.as_ref().unwrap().typ.as_ref() {
                if !typ.is_dir() {
                    return Ok(request.fail_syscall(Errno::ENOTDIR));
                }
            } else {
                return Ok(request.fail_syscall(Errno::ENOENT));
            }

            // SAFETY: fchdir is fd-only.
            Ok(unsafe { request.continue_syscall() })
        },
    )
}
