//LICENSE Portions Copyright 2019-2021 ZomboDB, LLC.
//LICENSE
//LICENSE Portions Copyright 2021-2023 Technology Concepts & Design, Inc.
//LICENSE
//LICENSE Portions Copyright 2023-2023 PgCentral Foundation, Inc. <contact@pgcentral.org>
//LICENSE
//LICENSE All rights reserved.
//LICENSE
//LICENSE Use of this source code is governed by the MIT license that can be found in the LICENSE file.
//! Submodule of `build.rs`

/// Hardcoded list of that are known to be from system libraries and other
/// non-postgres sources, which we exclude from wrappers to avoid symbol
/// versioning pain. This is quite hacky, and is done as a stopgap to avoid to
/// forcing pgrx extensions to require a newer version of glibc than they might
/// otherwise. (As a result, this only covers exported functions, and not types,
/// constants, and other things we may not want in the `pgrx-pg-sys` wrappers)
///
/// The list was generated with something like the following `fish-shell`
/// script, run on a Fedora-family Linux system, and followed with editing (to
/// write this comment, for example):
///
/// ```fish
/// begin
///     printf "pub(super) const EXCLUDED_SYMS: &[&str] = &["
///     for lib in libc.so.6 libm.so.6 libresolv.so.2 libpthread.so.0 librt.so.1 libdl.so.2
///         nm -D --defined-only "/usr/lib64/$lib"
///     end | string replace -fr '^[\da-fA-F]+\s+[^AU]\s+([^\s@]+)?(@\S+)?$' '$1' \
///         | sort | uniq | xargs printf '\"%s\",' | string join ''
///     printf "]"
/// end | rustfmt
/// ```
///
/// We use it on non-linux platforms as well, since it does a decent job at
/// approximating the set of symbols we'd like to exclude, but it's not perfect
/// anywhere (but especially not on other targets).
///
/// A future version of pgrx will achieve this effect in a more complete and less
/// hacky manner. Effort should be spent there rather than updating this list.
pub const SYMBOLS: &[&str] = &[
    "a64l",
    "abort",
    "__abort_msg",
    "abs",
    "accept",
    "accept4",
    "access",
    "acct",
    "acos",
    "acosf",
    "acosf128",
    "__acosf128_finite",
    "acosf32",
    "acosf32x",
    "acosf64",
    "acosf64x",
    "__acosf_finite",
    "__acos_finite",
    "acosh",
    "acoshf",
    "acoshf128",
    "__acoshf128_finite",
    "acoshf32",
    "acoshf32x",
    "acoshf64",
    "acoshf64x",
    "__acoshf_finite",
    "__acosh_finite",
    "acoshl",
    "__acoshl_finite",
    "acosl",
    "__acosl_finite",
    "addmntent",
    "addseverity",
    "adjtime",
    "__adjtimex",
    "adjtimex",
    "advance",
    "__after_morecore_hook",
    "aio_cancel",
    "aio_cancel64",
    "aio_error",
    "aio_error64",
    "aio_fsync",
    "aio_fsync64",
    "aio_init",
    "aio_read",
    "aio_read64",
    "aio_return",
    "aio_return64",
    "aio_suspend",
    "aio_suspend64",
    "aio_write",
    "aio_write64",
    "alarm",
    "aligned_alloc",
    "alphasort",
    "alphasort64",
    "__arch_prctl",
    "arch_prctl",
    "argp_err_exit_status",
    "argp_error",
    "argp_failure",
    "argp_help",
    "argp_parse",
    "argp_program_bug_address",
    "argp_program_version",
    "argp_program_version_hook",
    "argp_state_help",
    "argp_usage",
    "argz_add",
    "argz_add_sep",
    "argz_append",
    "__argz_count",
    "argz_count",
    "argz_create",
    "argz_create_sep",
    "argz_delete",
    "argz_extract",
    "argz_insert",
    "__argz_next",
    "argz_next",
    "argz_replace",
    "__argz_stringify",
    "argz_stringify",
    "asctime",
    "asctime_r",
    "asin",
    "asinf",
    "asinf128",
    "__asinf128_finite",
    "asinf32",
    "asinf32x",
    "asinf64",
    "asinf64x",
    "__asinf_finite",
    "__asin_finite",
    "asinh",
    "asinhf",
    "asinhf128",
    "asinhf32",
    "asinhf32x",
    "asinhf64",
    "asinhf64x",
    "asinhl",
    "asinl",
    "__asinl_finite",
    "__asprintf",
    "asprintf",
    "__asprintf_chk",
    "__assert",
    "__assert_fail",
    "__assert_perror_fail",
    "atan",
    "atan2",
    "atan2f",
    "atan2f128",
    "__atan2f128_finite",
    "atan2f32",
    "atan2f32x",
    "atan2f64",
    "atan2f64x",
    "__atan2f_finite",
    "__atan2_finite",
    "atan2l",
    "__atan2l_finite",
    "atanf",
    "atanf128",
    "atanf32",
    "atanf32x",
    "atanf64",
    "atanf64x",
    "atanh",
    "atanhf",
    "atanhf128",
    "__atanhf128_finite",
    "atanhf32",
    "atanhf32x",
    "atanhf64",
    "atanhf64x",
    "__atanhf_finite",
    "__atanh_finite",
    "atanhl",
    "__atanhl_finite",
    "atanl",
    "atof",
    "atoi",
    "atol",
    "atoll",
    "authdes_create",
    "authdes_getucred",
    "authdes_pk_create",
    "_authenticate",
    "authnone_create",
    "authunix_create",
    "authunix_create_default",
    "__b64_ntop",
    "__b64_pton",
    "__backtrace",
    "backtrace",
    "__backtrace_symbols",
    "backtrace_symbols",
    "__backtrace_symbols_fd",
    "backtrace_symbols_fd",
    "basename",
    "bcmp",
    "bcopy",
    "bdflush",
    "bind",
    "bindresvport",
    "bindtextdomain",
    "bind_textdomain_codeset",
    "brk",
    "__bsd_getpgrp",
    "bsd_signal",
    "bsearch",
    "btowc",
    "__bzero",
    "bzero",
    "c16rtomb",
    "c32rtomb",
    "cabs",
    "cabsf",
    "cabsf128",
    "cabsf32",
    "cabsf32x",
    "cabsf64",
    "cabsf64x",
    "cabsl",
    "cacos",
    "cacosf",
    "cacosf128",
    "cacosf32",
    "cacosf32x",
    "cacosf64",
    "cacosf64x",
    "cacosh",
    "cacoshf",
    "cacoshf128",
    "cacoshf32",
    "cacoshf32x",
    "cacoshf64",
    "cacoshf64x",
    "cacoshl",
    "cacosl",
    "calloc",
    "call_once",
    "callrpc",
    "__call_tls_dtors",
    "canonicalize",
    "canonicalizef",
    "canonicalizef128",
    "canonicalizef32",
    "canonicalizef32x",
    "canonicalizef64",
    "canonicalizef64x",
    "canonicalize_file_name",
    "canonicalizel",
    "capget",
    "capset",
    "carg",
    "cargf",
    "cargf128",
    "cargf32",
    "cargf32x",
    "cargf64",
    "cargf64x",
    "cargl",
    "casin",
    "casinf",
    "casinf128",
    "casinf32",
    "casinf32x",
    "casinf64",
    "casinf64x",
    "casinh",
    "casinhf",
    "casinhf128",
    "casinhf32",
    "casinhf32x",
    "casinhf64",
    "casinhf64x",
    "casinhl",
    "casinl",
    "catan",
    "catanf",
    "catanf128",
    "catanf32",
    "catanf32x",
    "catanf64",
    "catanf64x",
    "catanh",
    "catanhf",
    "catanhf128",
    "catanhf32",
    "catanhf32x",
    "catanhf64",
    "catanhf64x",
    "catanhl",
    "catanl",
    "catclose",
    "catgets",
    "catopen",
    "cbc_crypt",
    "cbrt",
    "cbrtf",
    "cbrtf128",
    "cbrtf32",
    "cbrtf32x",
    "cbrtf64",
    "cbrtf64x",
    "cbrtl",
    "ccos",
    "ccosf",
    "ccosf128",
    "ccosf32",
    "ccosf32x",
    "ccosf64",
    "ccosf64x",
    "ccosh",
    "ccoshf",
    "ccoshf128",
    "ccoshf32",
    "ccoshf32x",
    "ccoshf64",
    "ccoshf64x",
    "ccoshl",
    "ccosl",
    "ceil",
    "ceilf",
    "ceilf128",
    "ceilf32",
    "ceilf32x",
    "ceilf64",
    "ceilf64x",
    "ceill",
    "cexp",
    "cexpf",
    "cexpf128",
    "cexpf32",
    "cexpf32x",
    "cexpf64",
    "cexpf64x",
    "cexpl",
    "cfgetispeed",
    "cfgetospeed",
    "cfmakeraw",
    "cfree",
    "cfsetispeed",
    "cfsetospeed",
    "cfsetspeed",
    "chdir",
    "__check_rhosts_file",
    "chflags",
    "__chk_fail",
    "chmod",
    "chown",
    "chroot",
    "cimag",
    "cimagf",
    "cimagf128",
    "cimagf32",
    "cimagf32x",
    "cimagf64",
    "cimagf64x",
    "cimagl",
    "clearenv",
    "clearerr",
    "clearerr_unlocked",
    "clnt_broadcast",
    "clnt_create",
    "clnt_pcreateerror",
    "clnt_perrno",
    "clnt_perror",
    "clntraw_create",
    "clnt_spcreateerror",
    "clnt_sperrno",
    "clnt_sperror",
    "clnttcp_create",
    "clntudp_bufcreate",
    "clntudp_create",
    "clntunix_create",
    "clock",
    "clock_adjtime",
    "clock_getcpuclockid",
    "clock_getres",
    "__clock_gettime",
    "clock_gettime",
    "clock_nanosleep",
    "clock_settime",
    "clog",
    "__clog10",
    "clog10",
    "__clog10f",
    "clog10f",
    "clog10f128",
    "clog10f32",
    "clog10f32x",
    "clog10f64",
    "clog10f64x",
    "__clog10l",
    "clog10l",
    "clogf",
    "clogf128",
    "clogf32",
    "clogf32x",
    "clogf64",
    "clogf64x",
    "clogl",
    "__clone",
    "clone",
    "__close",
    "close",
    "closedir",
    "closefrom",
    "closelog",
    "__close_nocancel",
    "close_range",
    "__cmsg_nxthdr",
    "cnd_broadcast",
    "cnd_destroy",
    "cnd_init",
    "cnd_signal",
    "cnd_timedwait",
    "cnd_wait",
    "confstr",
    "__confstr_chk",
    "conj",
    "conjf",
    "conjf128",
    "conjf32",
    "conjf32x",
    "conjf64",
    "conjf64x",
    "conjl",
    "__connect",
    "connect",
    "copy_file_range",
    "__copy_grp",
    "copysign",
    "copysignf",
    "copysignf128",
    "copysignf32",
    "copysignf32x",
    "copysignf64",
    "copysignf64x",
    "copysignl",
    "cos",
    "cosf",
    "cosf128",
    "cosf32",
    "cosf32x",
    "cosf64",
    "cosf64x",
    "cosh",
    "coshf",
    "coshf128",
    "__coshf128_finite",
    "coshf32",
    "coshf32x",
    "coshf64",
    "coshf64x",
    "__coshf_finite",
    "__cosh_finite",
    "coshl",
    "__coshl_finite",
    "cosl",
    "cpow",
    "cpowf",
    "cpowf128",
    "cpowf32",
    "cpowf32x",
    "cpowf64",
    "cpowf64x",
    "cpowl",
    "cproj",
    "cprojf",
    "cprojf128",
    "cprojf32",
    "cprojf32x",
    "cprojf64",
    "cprojf64x",
    "cprojl",
    "creal",
    "crealf",
    "crealf128",
    "crealf32",
    "crealf32x",
    "crealf64",
    "crealf64x",
    "creall",
    "creat",
    "creat64",
    "create_module",
    "csin",
    "csinf",
    "csinf128",
    "csinf32",
    "csinf32x",
    "csinf64",
    "csinf64x",
    "csinh",
    "csinhf",
    "csinhf128",
    "csinhf32",
    "csinhf32x",
    "csinhf64",
    "csinhf64x",
    "csinhl",
    "csinl",
    "csqrt",
    "csqrtf",
    "csqrtf128",
    "csqrtf32",
    "csqrtf32x",
    "csqrtf64",
    "csqrtf64x",
    "csqrtl",
    "ctan",
    "ctanf",
    "ctanf128",
    "ctanf32",
    "ctanf32x",
    "ctanf64",
    "ctanf64x",
    "ctanh",
    "ctanhf",
    "ctanhf128",
    "ctanhf32",
    "ctanhf32x",
    "ctanhf64",
    "ctanhf64x",
    "ctanhl",
    "ctanl",
    "ctermid",
    "ctime",
    "ctime_r",
    "__ctype32_b",
    "__ctype32_tolower",
    "__ctype32_toupper",
    "__ctype_b",
    "__ctype_b_loc",
    "__ctype_get_mb_cur_max",
    "__ctype_init",
    "__ctype_tolower",
    "__ctype_tolower_loc",
    "__ctype_toupper",
    "__ctype_toupper_loc",
    "__curbrk",
    "cuserid",
    "__cxa_atexit",
    "__cxa_at_quick_exit",
    "__cxa_finalize",
    "__cxa_thread_atexit_impl",
    "__cyg_profile_func_enter",
    "__cyg_profile_func_exit",
    "daddl",
    "daemon",
    "__daylight",
    "daylight",
    "__dcgettext",
    "dcgettext",
    "dcngettext",
    "ddivl",
    "__default_morecore",
    "delete_module",
    "des_setparity",
    "__dgettext",
    "dgettext",
    "difftime",
    "dirfd",
    "dirname",
    "div",
    "dladdr",
    "dladdr1",
    "_dl_catch_error",
    "_dl_catch_exception",
    "dlclose",
    "dlerror",
    "dlinfo",
    "dl_iterate_phdr",
    "_dl_mcount_wrapper",
    "_dl_mcount_wrapper_check",
    "dlmopen",
    "dlopen",
    "_dl_signal_error",
    "_dl_signal_exception",
    "dlsym",
    "dlvsym",
    "dmull",
    "__dn_comp",
    "dn_comp",
    "__dn_count_labels",
    "__dn_expand",
    "dn_expand",
    "dngettext",
    "__dn_skipname",
    "dn_skipname",
    "dprintf",
    "__dprintf_chk",
    "drand48",
    "drand48_r",
    "drem",
    "dremf",
    "dreml",
    "dsubl",
    "dup",
    "__dup2",
    "dup2",
    "dup3",
    "__duplocale",
    "duplocale",
    "dysize",
    "eaccess",
    "ecb_crypt",
    "ecvt",
    "ecvt_r",
    "endaliasent",
    "endfsent",
    "endgrent",
    "endhostent",
    "__endmntent",
    "endmntent",
    "endnetent",
    "endnetgrent",
    "endprotoent",
    "endpwent",
    "endrpcent",
    "endservent",
    "endsgent",
    "endspent",
    "endttyent",
    "endusershell",
    "endutent",
    "endutxent",
    "__environ",
    "_environ",
    "environ",
    "envz_add",
    "envz_entry",
    "envz_get",
    "envz_merge",
    "envz_remove",
    "envz_strip",
    "epoll_create",
    "epoll_create1",
    "epoll_ctl",
    "epoll_pwait",
    "epoll_wait",
    "erand48",
    "erand48_r",
    "erf",
    "erfc",
    "erfcf",
    "erfcf128",
    "erfcf32",
    "erfcf32x",
    "erfcf64",
    "erfcf64x",
    "erfcl",
    "erff",
    "erff128",
    "erff32",
    "erff32x",
    "erff64",
    "erff64x",
    "erfl",
    "err",
    "errno",
    "__errno_location",
    "error",
    "error_at_line",
    "error_message_count",
    "error_one_per_line",
    "error_print_progname",
    "errx",
    "ether_aton",
    "ether_aton_r",
    "ether_hostton",
    "ether_line",
    "ether_ntoa",
    "ether_ntoa_r",
    "ether_ntohost",
    "euidaccess",
    "eventfd",
    "eventfd_read",
    "eventfd_write",
    "execl",
    "execle",
    "execlp",
    "execv",
    "execve",
    "execveat",
    "execvp",
    "execvpe",
    "_exit",
    "exit",
    "_Exit",
    "exp",
    "exp10",
    "exp10f",
    "exp10f128",
    "__exp10f128_finite",
    "exp10f32",
    "exp10f32x",
    "exp10f64",
    "exp10f64x",
    "__exp10f_finite",
    "__exp10_finite",
    "exp10l",
    "__exp10l_finite",
    "exp2",
    "exp2f",
    "exp2f128",
    "__exp2f128_finite",
    "exp2f32",
    "exp2f32x",
    "exp2f64",
    "exp2f64x",
    "__exp2f_finite",
    "__exp2_finite",
    "exp2l",
    "__exp2l_finite",
    "expf",
    "expf128",
    "__expf128_finite",
    "expf32",
    "expf32x",
    "expf64",
    "expf64x",
    "__expf_finite",
    "__exp_finite",
    "expl",
    "__expl_finite",
    "explicit_bzero",
    "__explicit_bzero_chk",
    "expm1",
    "expm1f",
    "expm1f128",
    "expm1f32",
    "expm1f32x",
    "expm1f64",
    "expm1f64x",
    "expm1l",
    "f32addf128",
    "f32addf32x",
    "f32addf64",
    "f32addf64x",
    "f32divf128",
    "f32divf32x",
    "f32divf64",
    "f32divf64x",
    "f32mulf128",
    "f32mulf32x",
    "f32mulf64",
    "f32mulf64x",
    "f32subf128",
    "f32subf32x",
    "f32subf64",
    "f32subf64x",
    "f32xaddf128",
    "f32xaddf64",
    "f32xaddf64x",
    "f32xdivf128",
    "f32xdivf64",
    "f32xdivf64x",
    "f32xmulf128",
    "f32xmulf64",
    "f32xmulf64x",
    "f32xsubf128",
    "f32xsubf64",
    "f32xsubf64x",
    "f64addf128",
    "f64addf64x",
    "f64divf128",
    "f64divf64x",
    "f64mulf128",
    "f64mulf64x",
    "f64subf128",
    "f64subf64x",
    "f64xaddf128",
    "f64xdivf128",
    "f64xmulf128",
    "f64xsubf128",
    "fabs",
    "fabsf",
    "fabsf128",
    "fabsf32",
    "fabsf32x",
    "fabsf64",
    "fabsf64x",
    "fabsl",
    "faccessat",
    "fadd",
    "faddl",
    "fallocate",
    "fallocate64",
    "fanotify_init",
    "fanotify_mark",
    "fattach",
    "__fbufsize",
    "fchdir",
    "fchflags",
    "fchmod",
    "fchmodat",
    "fchown",
    "fchownat",
    "fclose",
    "fcloseall",
    "__fcntl",
    "fcntl",
    "fcntl64",
    "fcvt",
    "fcvt_r",
    "fdatasync",
    "__fdelt_chk",
    "__fdelt_warn",
    "fdetach",
    "fdim",
    "fdimf",
    "fdimf128",
    "fdimf32",
    "fdimf32x",
    "fdimf64",
    "fdimf64x",
    "fdiml",
    "fdiv",
    "fdivl",
    "fdopen",
    "fdopendir",
    "feclearexcept",
    "fedisableexcept",
    "feenableexcept",
    "fegetenv",
    "fegetexcept",
    "fegetexceptflag",
    "fegetmode",
    "fegetround",
    "feholdexcept",
    "__fentry__",
    "feof",
    "feof_unlocked",
    "feraiseexcept",
    "ferror",
    "ferror_unlocked",
    "fesetenv",
    "fesetexcept",
    "fesetexceptflag",
    "fesetmode",
    "fesetround",
    "fetestexcept",
    "fetestexceptflag",
    "feupdateenv",
    "fexecve",
    "fflush",
    "fflush_unlocked",
    "__ffs",
    "ffs",
    "ffsl",
    "ffsll",
    "fgetc",
    "fgetc_unlocked",
    "fgetgrent",
    "fgetgrent_r",
    "fgetpos",
    "fgetpos64",
    "fgetpwent",
    "fgetpwent_r",
    "fgets",
    "__fgets_chk",
    "fgetsgent",
    "fgetsgent_r",
    "fgetspent",
    "fgetspent_r",
    "fgets_unlocked",
    "__fgets_unlocked_chk",
    "fgetwc",
    "fgetwc_unlocked",
    "fgetws",
    "__fgetws_chk",
    "fgetws_unlocked",
    "__fgetws_unlocked_chk",
    "fgetxattr",
    "__file_change_detection_for_fp",
    "__file_change_detection_for_path",
    "__file_change_detection_for_stat",
    "__file_is_unchanged",
    "fileno",
    "fileno_unlocked",
    "__finite",
    "finite",
    "__finitef",
    "finitef",
    "__finitef128",
    "__finitel",
    "finitel",
    "__flbf",
    "flistxattr",
    "flock",
    "flockfile",
    "floor",
    "floorf",
    "floorf128",
    "floorf32",
    "floorf32x",
    "floorf64",
    "floorf64x",
    "floorl",
    "_flushlbf",
    "fma",
    "fmaf",
    "fmaf128",
    "fmaf32",
    "fmaf32x",
    "fmaf64",
    "fmaf64x",
    "fmal",
    "fmax",
    "fmaxf",
    "fmaxf128",
    "fmaxf32",
    "fmaxf32x",
    "fmaxf64",
    "fmaxf64x",
    "fmaxl",
    "fmaxmag",
    "fmaxmagf",
    "fmaxmagf128",
    "fmaxmagf32",
    "fmaxmagf32x",
    "fmaxmagf64",
    "fmaxmagf64x",
    "fmaxmagl",
    "fmemopen",
    "fmin",
    "fminf",
    "fminf128",
    "fminf32",
    "fminf32x",
    "fminf64",
    "fminf64x",
    "fminl",
    "fminmag",
    "fminmagf",
    "fminmagf128",
    "fminmagf32",
    "fminmagf32x",
    "fminmagf64",
    "fminmagf64x",
    "fminmagl",
    "fmod",
    "fmodf",
    "fmodf128",
    "__fmodf128_finite",
    "fmodf32",
    "fmodf32x",
    "fmodf64",
    "fmodf64x",
    "__fmodf_finite",
    "__fmod_finite",
    "fmodl",
    "__fmodl_finite",
    "fmtmsg",
    "fmul",
    "fmull",
    "fnmatch",
    "fopen",
    "fopen64",
    "fopencookie",
    "__fork",
    "fork",
    "_Fork",
    "forkpty",
    "__fortify_fail",
    "fpathconf",
    "__fpclassify",
    "__fpclassifyf",
    "__fpclassifyf128",
    "__fpclassifyl",
    "__fpending",
    "__fp_nquery",
    "__fp_query",
    "__fp_resstat",
    "fprintf",
    "__fprintf_chk",
    "__fpu_control",
    "__fpurge",
    "fputc",
    "fputc_unlocked",
    "fputs",
    "fputs_unlocked",
    "fputwc",
    "fputwc_unlocked",
    "fputws",
    "fputws_unlocked",
    "fread",
    "__freadable",
    "__fread_chk",
    "__freading",
    "fread_unlocked",
    "__fread_unlocked_chk",
    "free",
    "freeaddrinfo",
    "__free_hook",
    "freeifaddrs",
    "__freelocale",
    "freelocale",
    "fremovexattr",
    "freopen",
    "freopen64",
    "frexp",
    "frexpf",
    "frexpf128",
    "frexpf32",
    "frexpf32x",
    "frexpf64",
    "frexpf64x",
    "frexpl",
    "fromfp",
    "fromfpf",
    "fromfpf128",
    "fromfpf32",
    "fromfpf32x",
    "fromfpf64",
    "fromfpf64x",
    "fromfpl",
    "fromfpx",
    "fromfpxf",
    "fromfpxf128",
    "fromfpxf32",
    "fromfpxf32x",
    "fromfpxf64",
    "fromfpxf64x",
    "fromfpxl",
    "fscanf",
    "fseek",
    "fseeko",
    "__fseeko64",
    "fseeko64",
    "__fsetlocking",
    "fsetpos",
    "fsetpos64",
    "fsetxattr",
    "fstat",
    "__fstat64",
    "fstat64",
    "fstatat",
    "fstatat64",
    "fstatfs",
    "fstatfs64",
    "fstatvfs",
    "fstatvfs64",
    "fsub",
    "fsubl",
    "fsync",
    "ftell",
    "ftello",
    "__ftello64",
    "ftello64",
    "ftime",
    "ftok",
    "ftruncate",
    "ftruncate64",
    "ftrylockfile",
    "fts64_children",
    "fts64_close",
    "fts64_open",
    "fts64_read",
    "fts64_set",
    "fts_children",
    "fts_close",
    "fts_open",
    "fts_read",
    "fts_set",
    "ftw",
    "ftw64",
    "funlockfile",
    "futimens",
    "futimes",
    "futimesat",
    "fwide",
    "fwprintf",
    "__fwprintf_chk",
    "__fwritable",
    "fwrite",
    "fwrite_unlocked",
    "__fwriting",
    "fwscanf",
    "__fxstat",
    "__fxstat64",
    "__fxstatat",
    "__fxstatat64",
    "gai_cancel",
    "gai_error",
    "gai_strerror",
    "gai_suspend",
    "gamma",
    "gammaf",
    "__gammaf128_r_finite",
    "__gammaf_r_finite",
    "gammal",
    "__gammal_r_finite",
    "__gamma_r_finite",
    "__gconv_create_spec",
    "__gconv_destroy_spec",
    "__gconv_get_alias_db",
    "__gconv_get_cache",
    "__gconv_get_modules_db",
    "__gconv_open",
    "__gconv_transliterate",
    "gcvt",
    "getaddrinfo",
    "getaddrinfo_a",
    "getaliasbyname",
    "getaliasbyname_r",
    "getaliasent",
    "getaliasent_r",
    "__getauxval",
    "getauxval",
    "get_avphys_pages",
    "getc",
    "getchar",
    "getchar_unlocked",
    "getcontext",
    "getcpu",
    "getc_unlocked",
    "get_current_dir_name",
    "getcwd",
    "__getcwd_chk",
    "getdate",
    "getdate_err",
    "getdate_r",
    "__getdelim",
    "getdelim",
    "getdents64",
    "getdirentries",
    "getdirentries64",
    "getdomainname",
    "__getdomainname_chk",
    "getdtablesize",
    "getegid",
    "getentropy",
    "getenv",
    "geteuid",
    "getfsent",
    "getfsfile",
    "getfsspec",
    "getgid",
    "getgrent",
    "getgrent_r",
    "getgrgid",
    "getgrgid_r",
    "getgrnam",
    "getgrnam_r",
    "getgrouplist",
    "getgroups",
    "__getgroups_chk",
    "gethostbyaddr",
    "gethostbyaddr_r",
    "gethostbyname",
    "gethostbyname2",
    "gethostbyname2_r",
    "gethostbyname_r",
    "gethostent",
    "gethostent_r",
    "gethostid",
    "gethostname",
    "__gethostname_chk",
    "_gethtbyaddr",
    "_gethtbyname",
    "_gethtbyname2",
    "_gethtent",
    "getifaddrs",
    "getipv4sourcefilter",
    "getitimer",
    "get_kernel_syms",
    "getline",
    "getloadavg",
    "getlogin",
    "getlogin_r",
    "__getlogin_r_chk",
    "_getlong",
    "getmntent",
    "__getmntent_r",
    "getmntent_r",
    "getmsg",
    "get_myaddress",
    "getnameinfo",
    "getnetbyaddr",
    "getnetbyaddr_r",
    "getnetbyname",
    "getnetbyname_r",
    "getnetent",
    "getnetent_r",
    "getnetgrent",
    "getnetgrent_r",
    "getnetname",
    "get_nprocs",
    "get_nprocs_conf",
    "getopt",
    "getopt_long",
    "getopt_long_only",
    "__getpagesize",
    "getpagesize",
    "getpass",
    "getpayload",
    "getpayloadf",
    "getpayloadf128",
    "getpayloadf32",
    "getpayloadf32x",
    "getpayloadf64",
    "getpayloadf64x",
    "getpayloadl",
    "getpeername",
    "__getpgid",
    "getpgid",
    "getpgrp",
    "get_phys_pages",
    "__getpid",
    "getpid",
    "getpmsg",
    "getppid",
    "getpriority",
    "getprotobyname",
    "getprotobyname_r",
    "getprotobynumber",
    "getprotobynumber_r",
    "getprotoent",
    "getprotoent_r",
    "getpt",
    "getpublickey",
    "getpw",
    "getpwent",
    "getpwent_r",
    "getpwnam",
    "getpwnam_r",
    "getpwuid",
    "getpwuid_r",
    "getrandom",
    "getresgid",
    "getresuid",
    "__getrlimit",
    "getrlimit",
    "getrlimit64",
    "getrpcbyname",
    "getrpcbyname_r",
    "getrpcbynumber",
    "getrpcbynumber_r",
    "getrpcent",
    "getrpcent_r",
    "getrpcport",
    "getrusage",
    "gets",
    "__gets_chk",
    "getsecretkey",
    "getservbyname",
    "getservbyname_r",
    "getservbyport",
    "getservbyport_r",
    "getservent",
    "getservent_r",
    "getsgent",
    "getsgent_r",
    "getsgnam",
    "getsgnam_r",
    "_getshort",
    "getsid",
    "getsockname",
    "getsockopt",
    "getsourcefilter",
    "getspent",
    "getspent_r",
    "getspnam",
    "getspnam_r",
    "getsubopt",
    "gettext",
    "gettid",
    "__gettimeofday",
    "gettimeofday",
    "getttyent",
    "getttynam",
    "getuid",
    "getusershell",
    "getutent",
    "getutent_r",
    "getutid",
    "getutid_r",
    "getutline",
    "getutline_r",
    "getutmp",
    "getutmpx",
    "getutxent",
    "getutxid",
    "getutxline",
    "getw",
    "getwc",
    "getwchar",
    "getwchar_unlocked",
    "getwc_unlocked",
    "getwd",
    "__getwd_chk",
    "getxattr",
    "glob",
    "glob64",
    "globfree",
    "globfree64",
    "glob_pattern_p",
    "gmtime",
    "__gmtime_r",
    "gmtime_r",
    "gnu_dev_major",
    "gnu_dev_makedev",
    "gnu_dev_minor",
    "gnu_get_libc_release",
    "gnu_get_libc_version",
    "grantpt",
    "group_member",
    "gsignal",
    "gtty",
    "hasmntopt",
    "hcreate",
    "hcreate_r",
    "hdestroy",
    "hdestroy_r",
    "h_errlist",
    "__h_errno",
    "__h_errno_location",
    "herror",
    "h_nerr",
    "host2netname",
    "__hostalias",
    "hsearch",
    "hsearch_r",
    "hstrerror",
    "htonl",
    "htons",
    "hypot",
    "hypotf",
    "hypotf128",
    "__hypotf128_finite",
    "hypotf32",
    "hypotf32x",
    "hypotf64",
    "hypotf64x",
    "__hypotf_finite",
    "__hypot_finite",
    "hypotl",
    "__hypotl_finite",
    "iconv",
    "iconv_close",
    "iconv_open",
    "__idna_from_dns_encoding",
    "__idna_to_dns_encoding",
    "if_freenameindex",
    "if_indextoname",
    "if_nameindex",
    "if_nametoindex",
    "ilogb",
    "ilogbf",
    "ilogbf128",
    "ilogbf32",
    "ilogbf32x",
    "ilogbf64",
    "ilogbf64x",
    "ilogbl",
    "imaxabs",
    "imaxdiv",
    "in6addr_any",
    "in6addr_loopback",
    "index",
    "inet6_opt_append",
    "inet6_opt_find",
    "inet6_opt_finish",
    "inet6_opt_get_val",
    "inet6_opt_init",
    "inet6_option_alloc",
    "inet6_option_append",
    "inet6_option_find",
    "inet6_option_init",
    "inet6_option_next",
    "inet6_option_space",
    "inet6_opt_next",
    "inet6_opt_set_val",
    "inet6_rth_add",
    "inet6_rth_getaddr",
    "inet6_rth_init",
    "inet6_rth_reverse",
    "inet6_rth_segments",
    "inet6_rth_space",
    "__inet6_scopeid_pton",
    "inet_addr",
    "inet_aton",
    "__inet_aton_exact",
    "inet_lnaof",
    "inet_makeaddr",
    "inet_neta",
    "inet_net_ntop",
    "inet_netof",
    "inet_net_pton",
    "inet_network",
    "inet_nsap_addr",
    "inet_nsap_ntoa",
    "inet_ntoa",
    "inet_ntop",
    "inet_pton",
    "__inet_pton_length",
    "initgroups",
    "init_module",
    "initstate",
    "initstate_r",
    "innetgr",
    "inotify_add_watch",
    "inotify_init",
    "inotify_init1",
    "inotify_rm_watch",
    "insque",
    "__internal_endnetgrent",
    "__internal_getnetgrent_r",
    "__internal_setnetgrent",
    "_IO_2_1_stderr_",
    "_IO_2_1_stdin_",
    "_IO_2_1_stdout_",
    "_IO_adjust_column",
    "_IO_adjust_wcolumn",
    "ioctl",
    "_IO_default_doallocate",
    "_IO_default_finish",
    "_IO_default_pbackfail",
    "_IO_default_uflow",
    "_IO_default_xsgetn",
    "_IO_default_xsputn",
    "_IO_doallocbuf",
    "_IO_do_write",
    "_IO_enable_locks",
    "_IO_fclose",
    "_IO_fdopen",
    "_IO_feof",
    "_IO_ferror",
    "_IO_fflush",
    "_IO_fgetpos",
    "_IO_fgetpos64",
    "_IO_fgets",
    "_IO_file_attach",
    "_IO_file_close",
    "_IO_file_close_it",
    "_IO_file_doallocate",
    "_IO_file_finish",
    "_IO_file_fopen",
    "_IO_file_init",
    "_IO_file_jumps",
    "_IO_file_open",
    "_IO_file_overflow",
    "_IO_file_read",
    "_IO_file_seek",
    "_IO_file_seekoff",
    "_IO_file_setbuf",
    "_IO_file_stat",
    "_IO_file_sync",
    "_IO_file_underflow",
    "_IO_file_write",
    "_IO_file_xsputn",
    "_IO_flockfile",
    "_IO_flush_all",
    "_IO_flush_all_linebuffered",
    "_IO_fopen",
    "_IO_fprintf",
    "_IO_fputs",
    "_IO_fread",
    "_IO_free_backup_area",
    "_IO_free_wbackup_area",
    "_IO_fsetpos",
    "_IO_fsetpos64",
    "_IO_ftell",
    "_IO_ftrylockfile",
    "_IO_funlockfile",
    "_IO_fwrite",
    "_IO_getc",
    "_IO_getline",
    "_IO_getline_info",
    "_IO_gets",
    "_IO_init",
    "_IO_init_marker",
    "_IO_init_wmarker",
    "_IO_iter_begin",
    "_IO_iter_end",
    "_IO_iter_file",
    "_IO_iter_next",
    "_IO_least_wmarker",
    "_IO_link_in",
    "_IO_list_all",
    "_IO_list_lock",
    "_IO_list_resetlock",
    "_IO_list_unlock",
    "_IO_marker_delta",
    "_IO_marker_difference",
    "_IO_padn",
    "_IO_peekc_locked",
    "ioperm",
    "iopl",
    "_IO_popen",
    "_IO_printf",
    "_IO_proc_close",
    "_IO_proc_open",
    "_IO_putc",
    "_IO_puts",
    "_IO_remove_marker",
    "_IO_seekmark",
    "_IO_seekoff",
    "_IO_seekpos",
    "_IO_seekwmark",
    "_IO_setb",
    "_IO_setbuffer",
    "_IO_setvbuf",
    "_IO_sgetn",
    "_IO_sprintf",
    "_IO_sputbackc",
    "_IO_sputbackwc",
    "_IO_sscanf",
    "_IO_str_init_readonly",
    "_IO_str_init_static",
    "_IO_str_overflow",
    "_IO_str_pbackfail",
    "_IO_str_seekoff",
    "_IO_str_underflow",
    "_IO_sungetc",
    "_IO_sungetwc",
    "_IO_switch_to_get_mode",
    "_IO_switch_to_main_wget_area",
    "_IO_switch_to_wbackup_area",
    "_IO_switch_to_wget_mode",
    "_IO_ungetc",
    "_IO_un_link",
    "_IO_unsave_markers",
    "_IO_unsave_wmarkers",
    "_IO_vfprintf",
    "_IO_vfscanf",
    "_IO_vsprintf",
    "_IO_wdefault_doallocate",
    "_IO_wdefault_finish",
    "_IO_wdefault_pbackfail",
    "_IO_wdefault_uflow",
    "_IO_wdefault_xsgetn",
    "_IO_wdefault_xsputn",
    "_IO_wdoallocbuf",
    "_IO_wdo_write",
    "_IO_wfile_jumps",
    "_IO_wfile_overflow",
    "_IO_wfile_seekoff",
    "_IO_wfile_sync",
    "_IO_wfile_underflow",
    "_IO_wfile_xsputn",
    "_IO_wmarker_delta",
    "_IO_wsetb",
    "iruserok",
    "iruserok_af",
    "isalnum",
    "__isalnum_l",
    "isalnum_l",
    "isalpha",
    "__isalpha_l",
    "isalpha_l",
    "isascii",
    "__isascii_l",
    "isastream",
    "isatty",
    "isblank",
    "__isblank_l",
    "isblank_l",
    "__iscanonicall",
    "iscntrl",
    "__iscntrl_l",
    "iscntrl_l",
    "__isctype",
    "isctype",
    "isdigit",
    "__isdigit_l",
    "isdigit_l",
    "__iseqsig",
    "__iseqsigf",
    "__iseqsigf128",
    "__iseqsigl",
    "isfdtype",
    "isgraph",
    "__isgraph_l",
    "isgraph_l",
    "__isinf",
    "isinf",
    "__isinff",
    "isinff",
    "__isinff128",
    "__isinfl",
    "isinfl",
    "islower",
    "__islower_l",
    "islower_l",
    "__isnan",
    "isnan",
    "__isnanf",
    "isnanf",
    "__isnanf128",
    "__isnanl",
    "isnanl",
    "__isoc99_fscanf",
    "__isoc99_fwscanf",
    "__isoc99_scanf",
    "__isoc99_sscanf",
    "__isoc99_swscanf",
    "__isoc99_vfscanf",
    "__isoc99_vfwscanf",
    "__isoc99_vscanf",
    "__isoc99_vsscanf",
    "__isoc99_vswscanf",
    "__isoc99_vwscanf",
    "__isoc99_wscanf",
    "isprint",
    "__isprint_l",
    "isprint_l",
    "ispunct",
    "__ispunct_l",
    "ispunct_l",
    "__issignaling",
    "__issignalingf",
    "__issignalingf128",
    "__issignalingl",
    "isspace",
    "__isspace_l",
    "isspace_l",
    "isupper",
    "__isupper_l",
    "isupper_l",
    "iswalnum",
    "__iswalnum_l",
    "iswalnum_l",
    "iswalpha",
    "__iswalpha_l",
    "iswalpha_l",
    "iswblank",
    "__iswblank_l",
    "iswblank_l",
    "iswcntrl",
    "__iswcntrl_l",
    "iswcntrl_l",
    "__iswctype",
    "iswctype",
    "__iswctype_l",
    "iswctype_l",
    "iswdigit",
    "__iswdigit_l",
    "iswdigit_l",
    "iswgraph",
    "__iswgraph_l",
    "iswgraph_l",
    "iswlower",
    "__iswlower_l",
    "iswlower_l",
    "iswprint",
    "__iswprint_l",
    "iswprint_l",
    "iswpunct",
    "__iswpunct_l",
    "iswpunct_l",
    "iswspace",
    "__iswspace_l",
    "iswspace_l",
    "iswupper",
    "__iswupper_l",
    "iswupper_l",
    "iswxdigit",
    "__iswxdigit_l",
    "iswxdigit_l",
    "isxdigit",
    "__isxdigit_l",
    "isxdigit_l",
    "_itoa_lower_digits",
    "__ivaliduser",
    "j0",
    "j0f",
    "j0f128",
    "__j0f128_finite",
    "j0f32",
    "j0f32x",
    "j0f64",
    "j0f64x",
    "__j0f_finite",
    "__j0_finite",
    "j0l",
    "__j0l_finite",
    "j1",
    "j1f",
    "j1f128",
    "__j1f128_finite",
    "j1f32",
    "j1f32x",
    "j1f64",
    "j1f64x",
    "__j1f_finite",
    "__j1_finite",
    "j1l",
    "__j1l_finite",
    "jn",
    "jnf",
    "jnf128",
    "__jnf128_finite",
    "jnf32",
    "jnf32x",
    "jnf64",
    "jnf64x",
    "__jnf_finite",
    "__jn_finite",
    "jnl",
    "__jnl_finite",
    "jrand48",
    "jrand48_r",
    "key_decryptsession",
    "key_decryptsession_pk",
    "__key_decryptsession_pk_LOCAL",
    "key_encryptsession",
    "key_encryptsession_pk",
    "__key_encryptsession_pk_LOCAL",
    "key_gendes",
    "__key_gendes_LOCAL",
    "key_get_conv",
    "key_secretkey_is_set",
    "key_setnet",
    "key_setsecret",
    "kill",
    "killpg",
    "klogctl",
    "l64a",
    "labs",
    "lchmod",
    "lchown",
    "lckpwdf",
    "lcong48",
    "lcong48_r",
    "ldexp",
    "ldexpf",
    "ldexpf128",
    "ldexpf32",
    "ldexpf32x",
    "ldexpf64",
    "ldexpf64x",
    "ldexpl",
    "ldiv",
    "lfind",
    "lgamma",
    "lgammaf",
    "lgammaf128",
    "lgammaf128_r",
    "__lgammaf128_r_finite",
    "lgammaf32",
    "lgammaf32_r",
    "lgammaf32x",
    "lgammaf32x_r",
    "lgammaf64",
    "lgammaf64_r",
    "lgammaf64x",
    "lgammaf64x_r",
    "lgammaf_r",
    "__lgammaf_r_finite",
    "lgammal",
    "lgammal_r",
    "__lgammal_r_finite",
    "lgamma_r",
    "__lgamma_r_finite",
    "lgetxattr",
    "__libc_alloca_cutoff",
    "__libc_allocate_once_slow",
    "__libc_allocate_rtsig",
    "__libc_alloc_buffer_alloc_array",
    "__libc_alloc_buffer_allocate",
    "__libc_alloc_buffer_copy_bytes",
    "__libc_alloc_buffer_copy_string",
    "__libc_alloc_buffer_create_failure",
    "__libc_calloc",
    "__libc_clntudp_bufcreate",
    "__libc_current_sigrtmax",
    "__libc_current_sigrtmin",
    "__libc_dlerror_result",
    "__libc_dn_expand",
    "__libc_dn_skipname",
    "__libc_dynarray_at_failure",
    "__libc_dynarray_emplace_enlarge",
    "__libc_dynarray_finalize",
    "__libc_dynarray_resize",
    "__libc_dynarray_resize_clear",
    "__libc_early_init",
    "__libc_fatal",
    "__libc_fcntl64",
    "__libc_fork",
    "__libc_free",
    "__libc_freeres",
    "__libc_ifunc_impl_list",
    "__libc_init_first",
    "_libc_intl_domainname",
    "__libc_mallinfo",
    "__libc_malloc",
    "__libc_mallopt",
    "__libc_memalign",
    "__libc_msgrcv",
    "__libc_msgsnd",
    "__libc_ns_makecanon",
    "__libc_ns_samename",
    "__libc_pread",
    "__libc_pvalloc",
    "__libc_pwrite",
    "__libc_realloc",
    "__libc_reallocarray",
    "__libc_res_dnok",
    "__libc_res_hnok",
    "__libc_res_nameinquery",
    "__libc_res_queriesmatch",
    "__libc_rpc_getport",
    "__libc_sa_len",
    "__libc_scratch_buffer_dupfree",
    "__libc_scratch_buffer_grow",
    "__libc_scratch_buffer_grow_preserve",
    "__libc_scratch_buffer_set_array_size",
    "__libc_secure_getenv",
    "__libc_sigaction",
    "__libc_single_threaded",
    "__libc_start_main",
    "__libc_system",
    "__libc_unwind_link_get",
    "__libc_valloc",
    "__libdl_version_placeholder",
    "__libpthread_version_placeholder",
    "__librt_version_placeholder",
    "_LIB_VERSION",
    "link",
    "linkat",
    "lio_listio",
    "lio_listio64",
    "listen",
    "listxattr",
    "llabs",
    "lldiv",
    "llistxattr",
    "__lll_lock_wait_private",
    "__lll_lock_wake_private",
    "llogb",
    "llogbf",
    "llogbf128",
    "llogbf32",
    "llogbf32x",
    "llogbf64",
    "llogbf64x",
    "llogbl",
    "llrint",
    "llrintf",
    "llrintf128",
    "llrintf32",
    "llrintf32x",
    "llrintf64",
    "llrintf64x",
    "llrintl",
    "llround",
    "llroundf",
    "llroundf128",
    "llroundf32",
    "llroundf32x",
    "llroundf64",
    "llroundf64x",
    "llroundl",
    "llseek",
    "loc1",
    "loc2",
    "localeconv",
    "localtime",
    "localtime_r",
    "__loc_aton",
    "lockf",
    "lockf64",
    "__loc_ntoa",
    "locs",
    "log",
    "log10",
    "log10f",
    "log10f128",
    "__log10f128_finite",
    "log10f32",
    "log10f32x",
    "log10f64",
    "log10f64x",
    "__log10f_finite",
    "__log10_finite",
    "log10l",
    "__log10l_finite",
    "log1p",
    "log1pf",
    "log1pf128",
    "log1pf32",
    "log1pf32x",
    "log1pf64",
    "log1pf64x",
    "log1pl",
    "log2",
    "log2f",
    "log2f128",
    "__log2f128_finite",
    "log2f32",
    "log2f32x",
    "log2f64",
    "log2f64x",
    "__log2f_finite",
    "__log2_finite",
    "log2l",
    "__log2l_finite",
    "logb",
    "logbf",
    "logbf128",
    "logbf32",
    "logbf32x",
    "logbf64",
    "logbf64x",
    "logbl",
    "logf",
    "logf128",
    "__logf128_finite",
    "logf32",
    "logf32x",
    "logf64",
    "logf64x",
    "__logf_finite",
    "__log_finite",
    "login",
    "login_tty",
    "logl",
    "__logl_finite",
    "logout",
    "logwtmp",
    "_longjmp",
    "longjmp",
    "__longjmp_chk",
    "lrand48",
    "lrand48_r",
    "lremovexattr",
    "lrint",
    "lrintf",
    "lrintf128",
    "lrintf32",
    "lrintf32x",
    "lrintf64",
    "lrintf64x",
    "lrintl",
    "lround",
    "lroundf",
    "lroundf128",
    "lroundf32",
    "lroundf32x",
    "lroundf64",
    "lroundf64x",
    "lroundl",
    "lsearch",
    "__lseek",
    "lseek",
    "lseek64",
    "lsetxattr",
    "lstat",
    "lstat64",
    "lutimes",
    "__lxstat",
    "__lxstat64",
    "__madvise",
    "madvise",
    "makecontext",
    "mallinfo",
    "mallinfo2",
    "malloc",
    "__malloc_hook",
    "malloc_info",
    "__malloc_initialize_hook",
    "malloc_stats",
    "malloc_trim",
    "malloc_usable_size",
    "mallopt",
    "mallwatch",
    "matherr",
    "mblen",
    "__mbrlen",
    "mbrlen",
    "mbrtoc16",
    "mbrtoc32",
    "__mbrtowc",
    "mbrtowc",
    "mbsinit",
    "mbsnrtowcs",
    "__mbsnrtowcs_chk",
    "mbsrtowcs",
    "__mbsrtowcs_chk",
    "mbstowcs",
    "__mbstowcs_chk",
    "mbtowc",
    "mcheck",
    "mcheck_check_all",
    "mcheck_pedantic",
    "_mcleanup",
    "_mcount",
    "mcount",
    "memalign",
    "__memalign_hook",
    "memccpy",
    "memchr",
    "memcmp",
    "memcpy",
    "__memcpy_chk",
    "memfd_create",
    "memfrob",
    "memmem",
    "memmove",
    "__memmove_chk",
    "__mempcpy",
    "mempcpy",
    "__mempcpy_chk",
    "__mempcpy_small",
    "memrchr",
    "memset",
    "__memset_chk",
    "__merge_grp",
    "mincore",
    "mkdir",
    "mkdirat",
    "mkdtemp",
    "mkfifo",
    "mkfifoat",
    "mknod",
    "mknodat",
    "mkostemp",
    "mkostemp64",
    "mkostemps",
    "mkostemps64",
    "mkstemp",
    "mkstemp64",
    "mkstemps",
    "mkstemps64",
    "__mktemp",
    "mktemp",
    "mktime",
    "mlock",
    "mlock2",
    "mlockall",
    "__mmap",
    "mmap",
    "mmap64",
    "modf",
    "modff",
    "modff128",
    "modff32",
    "modff32x",
    "modff64",
    "modff64x",
    "modfl",
    "modify_ldt",
    "moncontrol",
    "__monstartup",
    "monstartup",
    "__morecore",
    "mount",
    "mprobe",
    "__mprotect",
    "mprotect",
    "mq_close",
    "mq_getattr",
    "mq_notify",
    "mq_open",
    "__mq_open_2",
    "mq_receive",
    "mq_send",
    "mq_setattr",
    "mq_timedreceive",
    "mq_timedsend",
    "mq_unlink",
    "mrand48",
    "mrand48_r",
    "mremap",
    "msgctl",
    "msgget",
    "msgrcv",
    "msgsnd",
    "msync",
    "mtrace",
    "mtx_destroy",
    "mtx_init",
    "mtx_lock",
    "mtx_timedlock",
    "mtx_trylock",
    "mtx_unlock",
    "munlock",
    "munlockall",
    "__munmap",
    "munmap",
    "muntrace",
    "name_to_handle_at",
    "nan",
    "nanf",
    "nanf128",
    "nanf32",
    "nanf32x",
    "nanf64",
    "nanf64x",
    "nanl",
    "__nanosleep",
    "nanosleep",
    "nearbyint",
    "nearbyintf",
    "nearbyintf128",
    "nearbyintf32",
    "nearbyintf32x",
    "nearbyintf64",
    "nearbyintf64x",
    "nearbyintl",
    "__netlink_assert_response",
    "netname2host",
    "netname2user",
    "__newlocale",
    "newlocale",
    "nextafter",
    "nextafterf",
    "nextafterf128",
    "nextafterf32",
    "nextafterf32x",
    "nextafterf64",
    "nextafterf64x",
    "nextafterl",
    "nextdown",
    "nextdownf",
    "nextdownf128",
    "nextdownf32",
    "nextdownf32x",
    "nextdownf64",
    "nextdownf64x",
    "nextdownl",
    "nexttoward",
    "nexttowardf",
    "nexttowardl",
    "nextup",
    "nextupf",
    "nextupf128",
    "nextupf32",
    "nextupf32x",
    "nextupf64",
    "nextupf64x",
    "nextupl",
    "nfsservctl",
    "nftw",
    "nftw64",
    "ngettext",
    "nice",
    "_nl_default_dirname",
    "_nl_domain_bindings",
    "nl_langinfo",
    "__nl_langinfo_l",
    "nl_langinfo_l",
    "_nl_msg_cat_cntr",
    "__nptl_create_event",
    "__nptl_death_event",
    "__nptl_last_event",
    "__nptl_nthreads",
    "__nptl_rtld_global",
    "__nptl_threads_events",
    "__nptl_version",
    "nrand48",
    "nrand48_r",
    "ns_datetosecs",
    "ns_format_ttl",
    "__ns_get16",
    "ns_get16",
    "__ns_get32",
    "ns_get32",
    "ns_initparse",
    "ns_makecanon",
    "ns_msg_getflag",
    "__ns_name_compress",
    "ns_name_compress",
    "ns_name_ntol",
    "__ns_name_ntop",
    "ns_name_ntop",
    "__ns_name_pack",
    "ns_name_pack",
    "__ns_name_pton",
    "ns_name_pton",
    "ns_name_rollback",
    "__ns_name_skip",
    "ns_name_skip",
    "__ns_name_uncompress",
    "ns_name_uncompress",
    "__ns_name_unpack",
    "ns_name_unpack",
    "ns_parserr",
    "ns_parse_ttl",
    "ns_put16",
    "ns_put32",
    "ns_samedomain",
    "ns_samename",
    "__nss_configure_lookup",
    "__nss_database_get",
    "__nss_database_lookup",
    "__nss_disable_nscd",
    "_nss_dns_getcanonname_r",
    "_nss_dns_gethostbyaddr2_r",
    "_nss_dns_gethostbyaddr_r",
    "_nss_dns_gethostbyname2_r",
    "_nss_dns_gethostbyname3_r",
    "_nss_dns_gethostbyname4_r",
    "_nss_dns_gethostbyname_r",
    "_nss_dns_getnetbyaddr_r",
    "_nss_dns_getnetbyname_r",
    "__nss_files_data_endent",
    "__nss_files_data_open",
    "__nss_files_data_put",
    "__nss_files_data_setent",
    "_nss_files_endaliasent",
    "_nss_files_endetherent",
    "_nss_files_endgrent",
    "_nss_files_endhostent",
    "_nss_files_endnetent",
    "_nss_files_endnetgrent",
    "_nss_files_endprotoent",
    "_nss_files_endpwent",
    "_nss_files_endrpcent",
    "_nss_files_endservent",
    "_nss_files_endsgent",
    "_nss_files_endspent",
    "__nss_files_fopen",
    "_nss_files_getaliasbyname_r",
    "_nss_files_getaliasent_r",
    "_nss_files_getetherent_r",
    "_nss_files_getgrent_r",
    "_nss_files_getgrgid_r",
    "_nss_files_getgrnam_r",
    "_nss_files_gethostbyaddr_r",
    "_nss_files_gethostbyname2_r",
    "_nss_files_gethostbyname3_r",
    "_nss_files_gethostbyname4_r",
    "_nss_files_gethostbyname_r",
    "_nss_files_gethostent_r",
    "_nss_files_gethostton_r",
    "_nss_files_getnetbyaddr_r",
    "_nss_files_getnetbyname_r",
    "_nss_files_getnetent_r",
    "_nss_files_getnetgrent_r",
    "_nss_files_getntohost_r",
    "_nss_files_getprotobyname_r",
    "_nss_files_getprotobynumber_r",
    "_nss_files_getprotoent_r",
    "_nss_files_getpwent_r",
    "_nss_files_getpwnam_r",
    "_nss_files_getpwuid_r",
    "_nss_files_getrpcbyname_r",
    "_nss_files_getrpcbynumber_r",
    "_nss_files_getrpcent_r",
    "_nss_files_getservbyname_r",
    "_nss_files_getservbyport_r",
    "_nss_files_getservent_r",
    "_nss_files_getsgent_r",
    "_nss_files_getsgnam_r",
    "_nss_files_getspent_r",
    "_nss_files_getspnam_r",
    "_nss_files_init",
    "_nss_files_initgroups_dyn",
    "_nss_files_parse_etherent",
    "_nss_files_parse_grent",
    "_nss_files_parse_netent",
    "_nss_files_parse_protoent",
    "_nss_files_parse_pwent",
    "_nss_files_parse_rpcent",
    "_nss_files_parse_servent",
    "_nss_files_parse_sgent",
    "_nss_files_parse_spent",
    "_nss_files_setaliasent",
    "_nss_files_setetherent",
    "_nss_files_setgrent",
    "_nss_files_sethostent",
    "_nss_files_setnetent",
    "_nss_files_setnetgrent",
    "_nss_files_setprotoent",
    "_nss_files_setpwent",
    "_nss_files_setrpcent",
    "_nss_files_setservent",
    "_nss_files_setsgent",
    "_nss_files_setspent",
    "__nss_group_lookup",
    "__nss_group_lookup2",
    "__nss_hash",
    "__nss_hostname_digits_dots",
    "__nss_hosts_lookup",
    "__nss_hosts_lookup2",
    "ns_skiprr",
    "__nss_lookup",
    "__nss_lookup_function",
    "_nss_netgroup_parseline",
    "__nss_next",
    "__nss_next2",
    "__nss_parse_line_result",
    "__nss_passwd_lookup",
    "__nss_passwd_lookup2",
    "ns_sprintrr",
    "ns_sprintrrf",
    "__nss_readline",
    "__nss_services_lookup2",
    "ns_subdomain",
    "ntohl",
    "ntohs",
    "ntp_adjtime",
    "ntp_gettime",
    "ntp_gettimex",
    "_null_auth",
    "_obstack",
    "_obstack_allocated_p",
    "obstack_alloc_failed_handler",
    "_obstack_begin",
    "_obstack_begin_1",
    "obstack_exit_failure",
    "_obstack_free",
    "obstack_free",
    "_obstack_memory_used",
    "_obstack_newchunk",
    "obstack_printf",
    "__obstack_printf_chk",
    "obstack_vprintf",
    "__obstack_vprintf_chk",
    "on_exit",
    "__open",
    "open",
    "__open_2",
    "__open64",
    "open64",
    "__open64_2",
    "__open64_nocancel",
    "openat",
    "__openat_2",
    "openat64",
    "__openat64_2",
    "open_by_handle_at",
    "__open_catalog",
    "opendir",
    "openlog",
    "open_memstream",
    "__open_nocancel",
    "openpty",
    "open_wmemstream",
    "optarg",
    "opterr",
    "optind",
    "optopt",
    "__overflow",
    "parse_printf_format",
    "passwd2des",
    "pathconf",
    "pause",
    "__p_cdname",
    "__p_cdnname",
    "__p_class",
    "__p_class_syms",
    "pclose",
    "perror",
    "personality",
    "__p_fqname",
    "__p_fqnname",
    "__pipe",
    "pipe",
    "pipe2",
    "pivot_root",
    "pkey_alloc",
    "pkey_free",
    "pkey_get",
    "pkey_mprotect",
    "pkey_set",
    "pmap_getmaps",
    "pmap_getport",
    "pmap_rmtcall",
    "pmap_set",
    "pmap_unset",
    "__poll",
    "poll",
    "__poll_chk",
    "popen",
    "__p_option",
    "posix_fadvise",
    "posix_fadvise64",
    "posix_fallocate",
    "posix_fallocate64",
    "__posix_getopt",
    "posix_madvise",
    "posix_memalign",
    "posix_openpt",
    "posix_spawn",
    "posix_spawnattr_destroy",
    "posix_spawnattr_getflags",
    "posix_spawnattr_getpgroup",
    "posix_spawnattr_getschedparam",
    "posix_spawnattr_getschedpolicy",
    "posix_spawnattr_getsigdefault",
    "posix_spawnattr_getsigmask",
    "posix_spawnattr_init",
    "posix_spawnattr_setflags",
    "posix_spawnattr_setpgroup",
    "posix_spawnattr_setschedparam",
    "posix_spawnattr_setschedpolicy",
    "posix_spawnattr_setsigdefault",
    "posix_spawnattr_setsigmask",
    "posix_spawn_file_actions_addchdir_np",
    "posix_spawn_file_actions_addclose",
    "posix_spawn_file_actions_addclosefrom_np",
    "posix_spawn_file_actions_adddup2",
    "posix_spawn_file_actions_addfchdir_np",
    "posix_spawn_file_actions_addopen",
    "posix_spawn_file_actions_destroy",
    "posix_spawn_file_actions_init",
    "posix_spawnp",
    "pow",
    "pow10",
    "pow10f",
    "pow10l",
    "powf",
    "powf128",
    "__powf128_finite",
    "powf32",
    "powf32x",
    "powf64",
    "powf64x",
    "__powf_finite",
    "__pow_finite",
    "powl",
    "__powl_finite",
    "ppoll",
    "__ppoll_chk",
    "__p_query",
    "__p_rcode",
    "prctl",
    "pread",
    "__pread64",
    "pread64",
    "__pread64_chk",
    "__pread64_nocancel",
    "__pread_chk",
    "preadv",
    "preadv2",
    "preadv64",
    "preadv64v2",
    "printf",
    "__printf_chk",
    "__printf_fp",
    "printf_size",
    "printf_size_info",
    "prlimit",
    "prlimit64",
    "process_vm_readv",
    "process_vm_writev",
    "profil",
    "__profile_frequency",
    "__progname",
    "__progname_full",
    "program_invocation_name",
    "program_invocation_short_name",
    "__p_secstodate",
    "pselect",
    "psiginfo",
    "psignal",
    "pthread_atfork",
    "pthread_attr_destroy",
    "pthread_attr_getaffinity_np",
    "pthread_attr_getdetachstate",
    "pthread_attr_getguardsize",
    "pthread_attr_getinheritsched",
    "pthread_attr_getschedparam",
    "pthread_attr_getschedpolicy",
    "pthread_attr_getscope",
    "pthread_attr_getsigmask_np",
    "pthread_attr_getstack",
    "pthread_attr_getstackaddr",
    "pthread_attr_getstacksize",
    "pthread_attr_init",
    "pthread_attr_setaffinity_np",
    "pthread_attr_setdetachstate",
    "pthread_attr_setguardsize",
    "pthread_attr_setinheritsched",
    "pthread_attr_setschedparam",
    "pthread_attr_setschedpolicy",
    "pthread_attr_setscope",
    "pthread_attr_setsigmask_np",
    "pthread_attr_setstack",
    "pthread_attr_setstackaddr",
    "pthread_attr_setstacksize",
    "pthread_barrierattr_destroy",
    "pthread_barrierattr_getpshared",
    "pthread_barrierattr_init",
    "pthread_barrierattr_setpshared",
    "pthread_barrier_destroy",
    "pthread_barrier_init",
    "pthread_barrier_wait",
    "pthread_cancel",
    "_pthread_cleanup_pop",
    "_pthread_cleanup_pop_restore",
    "_pthread_cleanup_push",
    "_pthread_cleanup_push_defer",
    "__pthread_cleanup_routine",
    "pthread_clockjoin_np",
    "pthread_condattr_destroy",
    "pthread_condattr_getclock",
    "pthread_condattr_getpshared",
    "pthread_condattr_init",
    "pthread_condattr_setclock",
    "pthread_condattr_setpshared",
    "pthread_cond_broadcast",
    "pthread_cond_clockwait",
    "pthread_cond_destroy",
    "pthread_cond_init",
    "pthread_cond_signal",
    "pthread_cond_timedwait",
    "pthread_cond_wait",
    "pthread_create",
    "pthread_detach",
    "pthread_equal",
    "pthread_exit",
    "pthread_getaffinity_np",
    "pthread_getattr_default_np",
    "pthread_getattr_np",
    "pthread_getconcurrency",
    "pthread_getcpuclockid",
    "__pthread_get_minstack",
    "pthread_getname_np",
    "pthread_getschedparam",
    "__pthread_getspecific",
    "pthread_getspecific",
    "pthread_join",
    "__pthread_key_create",
    "pthread_key_create",
    "pthread_key_delete",
    "__pthread_keys",
    "pthread_kill",
    "pthread_kill_other_threads_np",
    "__pthread_mutexattr_destroy",
    "pthread_mutexattr_destroy",
    "pthread_mutexattr_getkind_np",
    "pthread_mutexattr_getprioceiling",
    "pthread_mutexattr_getprotocol",
    "pthread_mutexattr_getpshared",
    "pthread_mutexattr_getrobust",
    "pthread_mutexattr_getrobust_np",
    "pthread_mutexattr_gettype",
    "__pthread_mutexattr_init",
    "pthread_mutexattr_init",
    "pthread_mutexattr_setkind_np",
    "pthread_mutexattr_setprioceiling",
    "pthread_mutexattr_setprotocol",
    "pthread_mutexattr_setpshared",
    "pthread_mutexattr_setrobust",
    "pthread_mutexattr_setrobust_np",
    "__pthread_mutexattr_settype",
    "pthread_mutexattr_settype",
    "pthread_mutex_clocklock",
    "pthread_mutex_consistent",
    "pthread_mutex_consistent_np",
    "__pthread_mutex_destroy",
    "pthread_mutex_destroy",
    "pthread_mutex_getprioceiling",
    "__pthread_mutex_init",
    "pthread_mutex_init",
    "__pthread_mutex_lock",
    "pthread_mutex_lock",
    "pthread_mutex_setprioceiling",
    "pthread_mutex_timedlock",
    "__pthread_mutex_trylock",
    "pthread_mutex_trylock",
    "__pthread_mutex_unlock",
    "pthread_mutex_unlock",
    "__pthread_once",
    "pthread_once",
    "__pthread_register_cancel",
    "__pthread_register_cancel_defer",
    "pthread_rwlockattr_destroy",
    "pthread_rwlockattr_getkind_np",
    "pthread_rwlockattr_getpshared",
    "pthread_rwlockattr_init",
    "pthread_rwlockattr_setkind_np",
    "pthread_rwlockattr_setpshared",
    "pthread_rwlock_clockrdlock",
    "pthread_rwlock_clockwrlock",
    "__pthread_rwlock_destroy",
    "pthread_rwlock_destroy",
    "__pthread_rwlock_init",
    "pthread_rwlock_init",
    "__pthread_rwlock_rdlock",
    "pthread_rwlock_rdlock",
    "pthread_rwlock_timedrdlock",
    "pthread_rwlock_timedwrlock",
    "__pthread_rwlock_tryrdlock",
    "pthread_rwlock_tryrdlock",
    "__pthread_rwlock_trywrlock",
    "pthread_rwlock_trywrlock",
    "__pthread_rwlock_unlock",
    "pthread_rwlock_unlock",
    "__pthread_rwlock_wrlock",
    "pthread_rwlock_wrlock",
    "pthread_self",
    "pthread_setaffinity_np",
    "pthread_setattr_default_np",
    "pthread_setcancelstate",
    "pthread_setcanceltype",
    "pthread_setconcurrency",
    "pthread_setname_np",
    "pthread_setschedparam",
    "pthread_setschedprio",
    "__pthread_setspecific",
    "pthread_setspecific",
    "pthread_sigmask",
    "pthread_sigqueue",
    "pthread_spin_destroy",
    "pthread_spin_init",
    "pthread_spin_lock",
    "pthread_spin_trylock",
    "pthread_spin_unlock",
    "pthread_testcancel",
    "pthread_timedjoin_np",
    "pthread_tryjoin_np",
    "__pthread_unregister_cancel",
    "__pthread_unregister_cancel_restore",
    "__pthread_unwind_next",
    "pthread_yield",
    "__p_time",
    "ptrace",
    "ptsname",
    "ptsname_r",
    "__ptsname_r_chk",
    "__p_type",
    "__p_type_syms",
    "putc",
    "putchar",
    "putchar_unlocked",
    "putc_unlocked",
    "putenv",
    "putgrent",
    "__putlong",
    "putmsg",
    "putpmsg",
    "putpwent",
    "puts",
    "putsgent",
    "__putshort",
    "putspent",
    "pututline",
    "pututxline",
    "putw",
    "putwc",
    "putwchar",
    "putwchar_unlocked",
    "putwc_unlocked",
    "pvalloc",
    "pwrite",
    "__pwrite64",
    "pwrite64",
    "pwritev",
    "pwritev2",
    "pwritev64",
    "pwritev64v2",
    "qecvt",
    "qecvt_r",
    "qfcvt",
    "qfcvt_r",
    "qgcvt",
    "qsort",
    "qsort_r",
    "query_module",
    "quick_exit",
    "quotactl",
    "raise",
    "rand",
    "random",
    "random_r",
    "rand_r",
    "__rawmemchr",
    "rawmemchr",
    "rcmd",
    "rcmd_af",
    "__rcmd_errstr",
    "__read",
    "read",
    "readahead",
    "__read_chk",
    "readdir",
    "readdir64",
    "readdir64_r",
    "readdir_r",
    "readlink",
    "readlinkat",
    "__readlinkat_chk",
    "__readlink_chk",
    "__read_nocancel",
    "readv",
    "realloc",
    "reallocarray",
    "__realloc_hook",
    "realpath",
    "__realpath_chk",
    "reboot",
    "re_comp",
    "re_compile_fastmap",
    "re_compile_pattern",
    "__recv",
    "recv",
    "__recv_chk",
    "recvfrom",
    "__recvfrom_chk",
    "recvmmsg",
    "recvmsg",
    "re_exec",
    "regcomp",
    "regerror",
    "regexec",
    "regfree",
    "__register_atfork",
    "register_printf_function",
    "register_printf_modifier",
    "register_printf_specifier",
    "register_printf_type",
    "registerrpc",
    "remainder",
    "remainderf",
    "remainderf128",
    "__remainderf128_finite",
    "remainderf32",
    "remainderf32x",
    "remainderf64",
    "remainderf64x",
    "__remainderf_finite",
    "__remainder_finite",
    "remainderl",
    "__remainderl_finite",
    "remap_file_pages",
    "re_match",
    "re_match_2",
    "re_max_failures",
    "remove",
    "removexattr",
    "remque",
    "remquo",
    "remquof",
    "remquof128",
    "remquof32",
    "remquof32x",
    "remquof64",
    "remquof64x",
    "remquol",
    "rename",
    "renameat",
    "renameat2",
    "_res",
    "__res_close",
    "__res_context_hostalias",
    "__res_context_mkquery",
    "__res_context_query",
    "__res_context_search",
    "__res_context_send",
    "__res_dnok",
    "res_dnok",
    "re_search",
    "re_search_2",
    "re_set_registers",
    "re_set_syntax",
    "res_gethostbyaddr",
    "res_gethostbyname",
    "res_gethostbyname2",
    "__res_get_nsaddr",
    "_res_hconf",
    "__res_hnok",
    "res_hnok",
    "__res_hostalias",
    "__res_iclose",
    "__res_init",
    "__res_isourserver",
    "__res_mailok",
    "res_mailok",
    "__res_mkquery",
    "res_mkquery",
    "__res_nameinquery",
    "__res_nclose",
    "__res_ninit",
    "__res_nmkquery",
    "res_nmkquery",
    "__res_nopt",
    "__res_nquery",
    "res_nquery",
    "__res_nquerydomain",
    "res_nquerydomain",
    "__res_nsearch",
    "res_nsearch",
    "__res_nsend",
    "res_nsend",
    "__resolv_context_get",
    "__resolv_context_get_override",
    "__resolv_context_get_preinit",
    "__resolv_context_put",
    "_res_opcodes",
    "__res_ownok",
    "res_ownok",
    "__resp",
    "__res_queriesmatch",
    "__res_query",
    "res_query",
    "__res_querydomain",
    "res_querydomain",
    "__res_randomid",
    "__res_search",
    "res_search",
    "__res_send",
    "res_send",
    "res_send_setqhook",
    "res_send_setrhook",
    "__res_state",
    "re_syntax_options",
    "revoke",
    "rewind",
    "rewinddir",
    "rexec",
    "rexec_af",
    "rexecoptions",
    "rindex",
    "rint",
    "rintf",
    "rintf128",
    "rintf32",
    "rintf32x",
    "rintf64",
    "rintf64x",
    "rintl",
    "rmdir",
    "round",
    "roundeven",
    "roundevenf",
    "roundevenf128",
    "roundevenf32",
    "roundevenf32x",
    "roundevenf64",
    "roundevenf64x",
    "roundevenl",
    "roundf",
    "roundf128",
    "roundf32",
    "roundf32x",
    "roundf64",
    "roundf64x",
    "roundl",
    "rpc_createerr",
    "_rpc_dtablesize",
    "__rpc_thread_createerr",
    "__rpc_thread_svc_fdset",
    "__rpc_thread_svc_max_pollfd",
    "__rpc_thread_svc_pollfd",
    "rpmatch",
    "rresvport",
    "rresvport_af",
    "rtime",
    "ruserok",
    "ruserok_af",
    "ruserpass",
    "__sbrk",
    "sbrk",
    "scalb",
    "scalbf",
    "__scalbf_finite",
    "__scalb_finite",
    "scalbl",
    "__scalbl_finite",
    "scalbln",
    "scalblnf",
    "scalblnf128",
    "scalblnf32",
    "scalblnf32x",
    "scalblnf64",
    "scalblnf64x",
    "scalblnl",
    "scalbn",
    "scalbnf",
    "scalbnf128",
    "scalbnf32",
    "scalbnf32x",
    "scalbnf64",
    "scalbnf64x",
    "scalbnl",
    "scandir",
    "scandir64",
    "scandirat",
    "scandirat64",
    "scanf",
    "__sched_cpualloc",
    "__sched_cpucount",
    "__sched_cpufree",
    "sched_getaffinity",
    "sched_getcpu",
    "__sched_getparam",
    "sched_getparam",
    "__sched_get_priority_max",
    "sched_get_priority_max",
    "__sched_get_priority_min",
    "sched_get_priority_min",
    "__sched_getscheduler",
    "sched_getscheduler",
    "sched_rr_get_interval",
    "sched_setaffinity",
    "sched_setparam",
    "__sched_setscheduler",
    "sched_setscheduler",
    "__sched_yield",
    "sched_yield",
    "__secure_getenv",
    "secure_getenv",
    "seed48",
    "seed48_r",
    "seekdir",
    "__select",
    "select",
    "sem_clockwait",
    "sem_close",
    "semctl",
    "sem_destroy",
    "semget",
    "sem_getvalue",
    "sem_init",
    "semop",
    "sem_open",
    "sem_post",
    "semtimedop",
    "sem_timedwait",
    "sem_trywait",
    "sem_unlink",
    "sem_wait",
    "__send",
    "send",
    "sendfile",
    "sendfile64",
    "__sendmmsg",
    "sendmmsg",
    "sendmsg",
    "sendto",
    "setaliasent",
    "setbuf",
    "setbuffer",
    "setcontext",
    "setdomainname",
    "setegid",
    "setenv",
    "_seterr_reply",
    "seteuid",
    "setfsent",
    "setfsgid",
    "setfsuid",
    "setgid",
    "setgrent",
    "setgroups",
    "sethostent",
    "sethostid",
    "sethostname",
    "_sethtent",
    "setipv4sourcefilter",
    "setitimer",
    "_setjmp",
    "setjmp",
    "setlinebuf",
    "setlocale",
    "setlogin",
    "setlogmask",
    "__setmntent",
    "setmntent",
    "setnetent",
    "setnetgrent",
    "setns",
    "setpayload",
    "setpayloadf",
    "setpayloadf128",
    "setpayloadf32",
    "setpayloadf32x",
    "setpayloadf64",
    "setpayloadf64x",
    "setpayloadl",
    "setpayloadsig",
    "setpayloadsigf",
    "setpayloadsigf128",
    "setpayloadsigf32",
    "setpayloadsigf32x",
    "setpayloadsigf64",
    "setpayloadsigf64x",
    "setpayloadsigl",
    "__setpgid",
    "setpgid",
    "setpgrp",
    "setpriority",
    "setprotoent",
    "setpwent",
    "setregid",
    "setresgid",
    "setresuid",
    "setreuid",
    "setrlimit",
    "setrlimit64",
    "setrpcent",
    "setservent",
    "setsgent",
    "setsid",
    "setsockopt",
    "setsourcefilter",
    "setspent",
    "setstate",
    "setstate_r",
    "settimeofday",
    "setttyent",
    "setuid",
    "setusershell",
    "setutent",
    "setutxent",
    "setvbuf",
    "setxattr",
    "sgetsgent",
    "sgetsgent_r",
    "sgetspent",
    "sgetspent_r",
    "shmat",
    "shmctl",
    "shmdt",
    "shmget",
    "__shm_get_name",
    "shm_open",
    "shm_unlink",
    "shutdown",
    "sigabbrev_np",
    "__sigaction",
    "sigaction",
    "__sigaddset",
    "sigaddset",
    "sigaltstack",
    "sigandset",
    "sigblock",
    "__sigdelset",
    "sigdelset",
    "sigdescr_np",
    "sigemptyset",
    "sigfillset",
    "siggetmask",
    "sighold",
    "sigignore",
    "siginterrupt",
    "sigisemptyset",
    "__sigismember",
    "sigismember",
    "siglongjmp",
    "signal",
    "signalfd",
    "__signbit",
    "__signbitf",
    "__signbitf128",
    "__signbitl",
    "__signgam",
    "signgam",
    "significand",
    "significandf",
    "significandl",
    "sigorset",
    "__sigpause",
    "sigpause",
    "sigpending",
    "sigprocmask",
    "sigqueue",
    "sigrelse",
    "sigreturn",
    "sigset",
    "__sigsetjmp",
    "sigsetmask",
    "sigstack",
    "__sigsuspend",
    "sigsuspend",
    "__sigtimedwait",
    "sigtimedwait",
    "sigvec",
    "sigwait",
    "sigwaitinfo",
    "sin",
    "sincos",
    "sincosf",
    "sincosf128",
    "sincosf32",
    "sincosf32x",
    "sincosf64",
    "sincosf64x",
    "sincosl",
    "sinf",
    "sinf128",
    "sinf32",
    "sinf32x",
    "sinf64",
    "sinf64x",
    "sinh",
    "sinhf",
    "sinhf128",
    "__sinhf128_finite",
    "sinhf32",
    "sinhf32x",
    "sinhf64",
    "sinhf64x",
    "__sinhf_finite",
    "__sinh_finite",
    "sinhl",
    "__sinhl_finite",
    "sinl",
    "sleep",
    "__snprintf",
    "snprintf",
    "__snprintf_chk",
    "sockatmark",
    "__socket",
    "socket",
    "socketpair",
    "splice",
    "sprintf",
    "__sprintf_chk",
    "sprofil",
    "sqrt",
    "sqrtf",
    "sqrtf128",
    "__sqrtf128_finite",
    "sqrtf32",
    "sqrtf32x",
    "sqrtf64",
    "sqrtf64x",
    "__sqrtf_finite",
    "__sqrt_finite",
    "sqrtl",
    "__sqrtl_finite",
    "srand",
    "srand48",
    "srand48_r",
    "srandom",
    "srandom_r",
    "sscanf",
    "ssignal",
    "sstk",
    "__stack_chk_fail",
    "stat",
    "stat64",
    "__statfs",
    "statfs",
    "statfs64",
    "statvfs",
    "statvfs64",
    "statx",
    "stderr",
    "stdin",
    "stdout",
    "step",
    "stime",
    "__stpcpy",
    "stpcpy",
    "__stpcpy_chk",
    "__stpcpy_small",
    "__stpncpy",
    "stpncpy",
    "__stpncpy_chk",
    "__strcasecmp",
    "strcasecmp",
    "__strcasecmp_l",
    "strcasecmp_l",
    "__strcasestr",
    "strcasestr",
    "strcat",
    "__strcat_chk",
    "strchr",
    "strchrnul",
    "strcmp",
    "strcoll",
    "__strcoll_l",
    "strcoll_l",
    "strcpy",
    "__strcpy_chk",
    "__strcpy_small",
    "strcspn",
    "__strcspn_c1",
    "__strcspn_c2",
    "__strcspn_c3",
    "__strdup",
    "strdup",
    "strerror",
    "strerrordesc_np",
    "strerror_l",
    "strerrorname_np",
    "__strerror_r",
    "strerror_r",
    "strfmon",
    "__strfmon_l",
    "strfmon_l",
    "strfromd",
    "strfromf",
    "strfromf128",
    "strfromf32",
    "strfromf32x",
    "strfromf64",
    "strfromf64x",
    "strfroml",
    "strfry",
    "strftime",
    "__strftime_l",
    "strftime_l",
    "strlen",
    "strncasecmp",
    "__strncasecmp_l",
    "strncasecmp_l",
    "strncat",
    "__strncat_chk",
    "strncmp",
    "strncpy",
    "__strncpy_chk",
    "__strndup",
    "strndup",
    "strnlen",
    "strpbrk",
    "__strpbrk_c2",
    "__strpbrk_c3",
    "strptime",
    "strptime_l",
    "strrchr",
    "strsep",
    "__strsep_1c",
    "__strsep_2c",
    "__strsep_3c",
    "__strsep_g",
    "strsignal",
    "strspn",
    "__strspn_c1",
    "__strspn_c2",
    "__strspn_c3",
    "strstr",
    "strtod",
    "__strtod_internal",
    "__strtod_l",
    "strtod_l",
    "__strtod_nan",
    "strtof",
    "strtof128",
    "__strtof128_internal",
    "strtof128_l",
    "__strtof128_nan",
    "strtof32",
    "strtof32_l",
    "strtof32x",
    "strtof32x_l",
    "strtof64",
    "strtof64_l",
    "strtof64x",
    "strtof64x_l",
    "__strtof_internal",
    "__strtof_l",
    "strtof_l",
    "__strtof_nan",
    "strtoimax",
    "strtok",
    "__strtok_r",
    "strtok_r",
    "__strtok_r_1c",
    "strtol",
    "strtold",
    "__strtold_internal",
    "__strtold_l",
    "strtold_l",
    "__strtold_nan",
    "__strtol_internal",
    "__strtol_l",
    "strtol_l",
    "strtoll",
    "__strtoll_internal",
    "__strtoll_l",
    "strtoll_l",
    "strtoq",
    "strtoul",
    "__strtoul_internal",
    "__strtoul_l",
    "strtoul_l",
    "strtoull",
    "__strtoull_internal",
    "__strtoull_l",
    "strtoull_l",
    "strtoumax",
    "strtouq",
    "__strverscmp",
    "strverscmp",
    "strxfrm",
    "__strxfrm_l",
    "strxfrm_l",
    "stty",
    "svcauthdes_stats",
    "svcerr_auth",
    "svcerr_decode",
    "svcerr_noproc",
    "svcerr_noprog",
    "svcerr_progvers",
    "svcerr_systemerr",
    "svcerr_weakauth",
    "svc_exit",
    "svcfd_create",
    "svc_fdset",
    "svc_getreq",
    "svc_getreq_common",
    "svc_getreq_poll",
    "svc_getreqset",
    "svc_max_pollfd",
    "svc_pollfd",
    "svcraw_create",
    "svc_register",
    "svc_run",
    "svc_sendreply",
    "svctcp_create",
    "svcudp_bufcreate",
    "svcudp_create",
    "svcudp_enablecache",
    "svcunix_create",
    "svcunixfd_create",
    "svc_unregister",
    "swab",
    "swapcontext",
    "swapoff",
    "swapon",
    "swprintf",
    "__swprintf_chk",
    "swscanf",
    "symlink",
    "symlinkat",
    "__sym_ntop",
    "__sym_ntos",
    "__sym_ston",
    "sync",
    "sync_file_range",
    "syncfs",
    "syscall",
    "__sysconf",
    "sysconf",
    "__sysctl",
    "sysctl",
    "_sys_errlist",
    "sys_errlist",
    "sysinfo",
    "syslog",
    "__syslog_chk",
    "_sys_nerr",
    "sys_nerr",
    "sys_sigabbrev",
    "_sys_siglist",
    "sys_siglist",
    "system",
    "__sysv_signal",
    "sysv_signal",
    "tan",
    "tanf",
    "tanf128",
    "tanf32",
    "tanf32x",
    "tanf64",
    "tanf64x",
    "tanh",
    "tanhf",
    "tanhf128",
    "tanhf32",
    "tanhf32x",
    "tanhf64",
    "tanhf64x",
    "tanhl",
    "tanl",
    "tcdrain",
    "tcflow",
    "tcflush",
    "tcgetattr",
    "tcgetpgrp",
    "tcgetsid",
    "tcsendbreak",
    "tcsetattr",
    "tcsetpgrp",
    "__tdelete",
    "tdelete",
    "tdestroy",
    "tee",
    "telldir",
    "tempnam",
    "textdomain",
    "__tfind",
    "tfind",
    "tgamma",
    "tgammaf",
    "tgammaf128",
    "tgammaf32",
    "tgammaf32x",
    "tgammaf64",
    "tgammaf64x",
    "tgammal",
    "tgkill",
    "thrd_create",
    "thrd_current",
    "thrd_detach",
    "thrd_equal",
    "thrd_exit",
    "thrd_join",
    "thrd_sleep",
    "thrd_yield",
    "_thread_db_const_thread_area",
    "_thread_db_dtv_dtv",
    "_thread_db_dtv_slotinfo_gen",
    "_thread_db_dtv_slotinfo_list_len",
    "_thread_db_dtv_slotinfo_list_next",
    "_thread_db_dtv_slotinfo_list_slotinfo",
    "_thread_db_dtv_slotinfo_map",
    "_thread_db_dtv_t_counter",
    "_thread_db_dtv_t_pointer_val",
    "_thread_db_link_map_l_tls_modid",
    "_thread_db_link_map_l_tls_offset",
    "_thread_db_list_t_next",
    "_thread_db_list_t_prev",
    "_thread_db___nptl_last_event",
    "_thread_db___nptl_nthreads",
    "_thread_db___nptl_rtld_global",
    "_thread_db_pthread_cancelhandling",
    "_thread_db_pthread_dtvp",
    "_thread_db_pthread_eventbuf",
    "_thread_db_pthread_eventbuf_eventmask",
    "_thread_db_pthread_eventbuf_eventmask_event_bits",
    "_thread_db_pthread_key_data_data",
    "_thread_db_pthread_key_data_level2_data",
    "_thread_db_pthread_key_data_seq",
    "_thread_db___pthread_keys",
    "_thread_db_pthread_key_struct_destr",
    "_thread_db_pthread_key_struct_seq",
    "_thread_db_pthread_list",
    "_thread_db_pthread_nextevent",
    "_thread_db_pthread_report_events",
    "_thread_db_pthread_schedparam_sched_priority",
    "_thread_db_pthread_schedpolicy",
    "_thread_db_pthread_specific",
    "_thread_db_pthread_start_routine",
    "_thread_db_pthread_tid",
    "_thread_db_rtld_global__dl_stack_used",
    "_thread_db_rtld_global__dl_stack_user",
    "_thread_db_rtld_global__dl_tls_dtv_slotinfo_list",
    "_thread_db_sizeof_dtv_slotinfo",
    "_thread_db_sizeof_dtv_slotinfo_list",
    "_thread_db_sizeof_list_t",
    "_thread_db_sizeof_pthread",
    "_thread_db_sizeof_pthread_key_data",
    "_thread_db_sizeof_pthread_key_data_level2",
    "_thread_db_sizeof_pthread_key_struct",
    "_thread_db_sizeof_td_eventbuf_t",
    "_thread_db_sizeof_td_thr_events_t",
    "_thread_db_td_eventbuf_t_eventdata",
    "_thread_db_td_eventbuf_t_eventnum",
    "_thread_db_td_thr_events_t_event_bits",
    "time",
    "timegm",
    "timelocal",
    "timer_create",
    "timer_delete",
    "timerfd_create",
    "timerfd_gettime",
    "timerfd_settime",
    "timer_getoverrun",
    "timer_gettime",
    "timer_settime",
    "times",
    "timespec_get",
    "timespec_getres",
    "__timezone",
    "timezone",
    "tmpfile",
    "tmpfile64",
    "tmpnam",
    "tmpnam_r",
    "toascii",
    "__toascii_l",
    "_tolower",
    "tolower",
    "__tolower_l",
    "tolower_l",
    "totalorder",
    "totalorderf",
    "totalorderf128",
    "totalorderf32",
    "totalorderf32x",
    "totalorderf64",
    "totalorderf64x",
    "totalorderl",
    "totalordermag",
    "totalordermagf",
    "totalordermagf128",
    "totalordermagf32",
    "totalordermagf32x",
    "totalordermagf64",
    "totalordermagf64x",
    "totalordermagl",
    "_toupper",
    "toupper",
    "__toupper_l",
    "toupper_l",
    "__towctrans",
    "towctrans",
    "__towctrans_l",
    "towctrans_l",
    "towlower",
    "__towlower_l",
    "towlower_l",
    "towupper",
    "__towupper_l",
    "towupper_l",
    "tr_break",
    "trunc",
    "truncate",
    "truncate64",
    "truncf",
    "truncf128",
    "truncf32",
    "truncf32x",
    "truncf64",
    "truncf64x",
    "truncl",
    "__tsearch",
    "tsearch",
    "tss_create",
    "tss_delete",
    "tss_get",
    "tss_set",
    "ttyname",
    "ttyname_r",
    "__ttyname_r_chk",
    "ttyslot",
    "__twalk",
    "twalk",
    "__twalk_r",
    "twalk_r",
    "__tzname",
    "tzname",
    "tzset",
    "ualarm",
    "__uflow",
    "ufromfp",
    "ufromfpf",
    "ufromfpf128",
    "ufromfpf32",
    "ufromfpf32x",
    "ufromfpf64",
    "ufromfpf64x",
    "ufromfpl",
    "ufromfpx",
    "ufromfpxf",
    "ufromfpxf128",
    "ufromfpxf32",
    "ufromfpxf32x",
    "ufromfpxf64",
    "ufromfpxf64x",
    "ufromfpxl",
    "ulckpwdf",
    "ulimit",
    "umask",
    "umount",
    "umount2",
    "uname",
    "__underflow",
    "ungetc",
    "ungetwc",
    "unlink",
    "unlinkat",
    "unlockpt",
    "unsetenv",
    "unshare",
    "updwtmp",
    "updwtmpx",
    "uselib",
    "__uselocale",
    "uselocale",
    "user2netname",
    "usleep",
    "ustat",
    "utime",
    "utimensat",
    "utimes",
    "utmpname",
    "utmpxname",
    "valloc",
    "vasprintf",
    "__vasprintf_chk",
    "vdprintf",
    "__vdprintf_chk",
    "verr",
    "verrx",
    "versionsort",
    "versionsort64",
    "__vfork",
    "vfork",
    "vfprintf",
    "__vfprintf_chk",
    "__vfscanf",
    "vfscanf",
    "vfwprintf",
    "__vfwprintf_chk",
    "vfwscanf",
    "vhangup",
    "vlimit",
    "vmsplice",
    "vprintf",
    "__vprintf_chk",
    "vscanf",
    "__vsnprintf",
    "vsnprintf",
    "__vsnprintf_chk",
    "vsprintf",
    "__vsprintf_chk",
    "__vsscanf",
    "vsscanf",
    "vswprintf",
    "__vswprintf_chk",
    "vswscanf",
    "vsyslog",
    "__vsyslog_chk",
    "vtimes",
    "vwarn",
    "vwarnx",
    "vwprintf",
    "__vwprintf_chk",
    "vwscanf",
    "__wait",
    "wait",
    "wait3",
    "wait4",
    "waitid",
    "__waitpid",
    "waitpid",
    "warn",
    "warnx",
    "wcpcpy",
    "__wcpcpy_chk",
    "wcpncpy",
    "__wcpncpy_chk",
    "wcrtomb",
    "__wcrtomb_chk",
    "wcscasecmp",
    "__wcscasecmp_l",
    "wcscasecmp_l",
    "wcscat",
    "__wcscat_chk",
    "wcschr",
    "wcschrnul",
    "wcscmp",
    "wcscoll",
    "__wcscoll_l",
    "wcscoll_l",
    "wcscpy",
    "__wcscpy_chk",
    "wcscspn",
    "wcsdup",
    "wcsftime",
    "__wcsftime_l",
    "wcsftime_l",
    "wcslen",
    "wcsncasecmp",
    "__wcsncasecmp_l",
    "wcsncasecmp_l",
    "wcsncat",
    "__wcsncat_chk",
    "wcsncmp",
    "wcsncpy",
    "__wcsncpy_chk",
    "wcsnlen",
    "wcsnrtombs",
    "__wcsnrtombs_chk",
    "wcspbrk",
    "wcsrchr",
    "wcsrtombs",
    "__wcsrtombs_chk",
    "wcsspn",
    "wcsstr",
    "wcstod",
    "__wcstod_internal",
    "__wcstod_l",
    "wcstod_l",
    "wcstof",
    "wcstof128",
    "__wcstof128_internal",
    "wcstof128_l",
    "wcstof32",
    "wcstof32_l",
    "wcstof32x",
    "wcstof32x_l",
    "wcstof64",
    "wcstof64_l",
    "wcstof64x",
    "wcstof64x_l",
    "__wcstof_internal",
    "__wcstof_l",
    "wcstof_l",
    "wcstoimax",
    "wcstok",
    "wcstol",
    "wcstold",
    "__wcstold_internal",
    "__wcstold_l",
    "wcstold_l",
    "__wcstol_internal",
    "__wcstol_l",
    "wcstol_l",
    "wcstoll",
    "__wcstoll_internal",
    "__wcstoll_l",
    "wcstoll_l",
    "wcstombs",
    "__wcstombs_chk",
    "wcstoq",
    "wcstoul",
    "__wcstoul_internal",
    "__wcstoul_l",
    "wcstoul_l",
    "wcstoull",
    "__wcstoull_internal",
    "__wcstoull_l",
    "wcstoull_l",
    "wcstoumax",
    "wcstouq",
    "wcswcs",
    "wcswidth",
    "wcsxfrm",
    "__wcsxfrm_l",
    "wcsxfrm_l",
    "wctob",
    "wctomb",
    "__wctomb_chk",
    "wctrans",
    "__wctrans_l",
    "wctrans_l",
    "wctype",
    "__wctype_l",
    "wctype_l",
    "wcwidth",
    "wmemchr",
    "wmemcmp",
    "wmemcpy",
    "__wmemcpy_chk",
    "wmemmove",
    "__wmemmove_chk",
    "wmempcpy",
    "__wmempcpy_chk",
    "wmemset",
    "__wmemset_chk",
    "wordexp",
    "wordfree",
    "__woverflow",
    "wprintf",
    "__wprintf_chk",
    "__write",
    "write",
    "__write_nocancel",
    "writev",
    "wscanf",
    "__wuflow",
    "__wunderflow",
    "__x86_get_cpuid_feature_leaf",
    "xdecrypt",
    "xdr_accepted_reply",
    "xdr_array",
    "xdr_authdes_cred",
    "xdr_authdes_verf",
    "xdr_authunix_parms",
    "xdr_bool",
    "xdr_bytes",
    "xdr_callhdr",
    "xdr_callmsg",
    "xdr_char",
    "xdr_cryptkeyarg",
    "xdr_cryptkeyarg2",
    "xdr_cryptkeyres",
    "xdr_des_block",
    "xdr_double",
    "xdr_enum",
    "xdr_float",
    "xdr_free",
    "xdr_getcredres",
    "xdr_hyper",
    "xdr_int",
    "xdr_int16_t",
    "xdr_int32_t",
    "xdr_int64_t",
    "xdr_int8_t",
    "xdr_keybuf",
    "xdr_key_netstarg",
    "xdr_key_netstres",
    "xdr_keystatus",
    "xdr_long",
    "xdr_longlong_t",
    "xdrmem_create",
    "xdr_netnamestr",
    "xdr_netobj",
    "xdr_opaque",
    "xdr_opaque_auth",
    "xdr_pmap",
    "xdr_pmaplist",
    "xdr_pointer",
    "xdr_quad_t",
    "xdrrec_create",
    "xdrrec_endofrecord",
    "xdrrec_eof",
    "xdrrec_skiprecord",
    "xdr_reference",
    "xdr_rejected_reply",
    "xdr_replymsg",
    "xdr_rmtcall_args",
    "xdr_rmtcallres",
    "xdr_short",
    "xdr_sizeof",
    "xdrstdio_create",
    "xdr_string",
    "xdr_u_char",
    "xdr_u_hyper",
    "xdr_u_int",
    "xdr_uint16_t",
    "xdr_uint32_t",
    "xdr_uint64_t",
    "xdr_uint8_t",
    "xdr_u_long",
    "xdr_u_longlong_t",
    "xdr_union",
    "xdr_unixcred",
    "xdr_u_quad_t",
    "xdr_u_short",
    "xdr_vector",
    "xdr_void",
    "xdr_wrapstring",
    "xencrypt",
    "__xmknod",
    "__xmknodat",
    "__xpg_basename",
    "__xpg_sigpause",
    "__xpg_strerror_r",
    "xprt_register",
    "xprt_unregister",
    "__xstat",
    "__xstat64",
    "y0",
    "y0f",
    "y0f128",
    "__y0f128_finite",
    "y0f32",
    "y0f32x",
    "y0f64",
    "y0f64x",
    "__y0f_finite",
    "__y0_finite",
    "y0l",
    "__y0l_finite",
    "y1",
    "y1f",
    "y1f128",
    "__y1f128_finite",
    "y1f32",
    "y1f32x",
    "y1f64",
    "y1f64x",
    "__y1f_finite",
    "__y1_finite",
    "y1l",
    "__y1l_finite",
    "yn",
    "ynf",
    "ynf128",
    "__ynf128_finite",
    "ynf32",
    "ynf32x",
    "ynf64",
    "ynf64x",
    "__ynf_finite",
    "__yn_finite",
    "ynl",
    "__ynl_finite",
];
