SYD(2)

# NAME

*/dev/syd* virtual system call interface

# SYNOPSIS

```
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

/* Execute sandbox commands */
int stat("/dev/syd/[command]", NULL);         // "syd -m command"
int stat("/dev/syd/[config]?", NULL);
int stat("/dev/syd/[config]:[value]", NULL);  // "syd -m config:value"
int stat("/dev/syd/[list]+[value]", NULL);    // "syd -m list+value"
int stat("/dev/syd/[list]-[value]", NULL);    // "syd -m list-value"
int stat("/dev/syd/[list]^[value]", NULL);    // "syd -m list^value"
int stat("/dev/syd/[command]![value]", NULL);

/* Read sandbox state as JSON */
int open("/dev/syd", O_RDONLY);

/* Read syd.el which is the Emacs Lisp implementation of the API */
int open("/dev/syd.el", O_RDONLY);

/* Read syd.sh which exports esyd shell function */
int open("/dev/syd.sh", O_RDONLY);
```

# DESCRIPTION

The */dev/syd* virtual system call interface is a unique mechanism designed for
runtime configuration of the Syd sandbox environment. It enables sandboxed
processes to interact with the Syd process to dynamically adjust sandbox
settings or query its state. This interaction is facilitated through the use of
virtual system calls, specifically via the _stat_(2) system call, applied to
specially constructed paths under */dev/syd*.

This interface allows for a range of operations, including enabling or disabling
sandbox features, appending or removing elements from lists, querying the
sandbox state, and executing special Syd commands. Operations are specified
through paths constructed with */dev/syd* as the prefix, followed by a sandbox
command and an operation character that denotes the desired action:

- *:* for setting a value (boolean, integer, string),
- *?* for querying a value,
- *+* for appending to a string array,
- *-* for removing an element from a string array,
- *^* for removing all matching elements from a string array, and
- *!* for executing a special Syd command.

This interface supports a flexible and powerful method for managing sandbox
policies dynamically, allowing for real-time adjustments to the security and
operational behavior of sandboxed processes. *libsyd* is a comprehensive C
library designed for interfacing with the Syd stat interface. It offers
functionalities for managing sandbox states, and facilitating runtime
configuration and interaction with the Syd sandboxing environment. *gosyd* is
a Go module that uses _cgo_ to use *libsyd*. *plsyd* is a Perl module that uses
_FFI::Platypus_ to use *libsyd*. *pysyd* is a Python module that uses _ctypes_
to use *libsyd*. *rbsyd* is a Ruby module that uses _ffi_ gem to use *libsyd*.
*syd.el* is an _Emacs Lisp_ implementation of the Syd stat interface.

# COMMANDS

The */dev/syd* interface supports the following commands for runtime
configuration of the sandbox. Each command can be invoked through the _stat_(2)
system call on special paths under */dev/syd*. Note, Syd provides similar
interfaces for the _stat_(2) interface, *-m* command-line option and the
configuration file. Some sandbox commands only take affect when they're
submitted on startup, such as *unshare/user*, and *sandbox/lock*. Such commands
are noted as *static* in the descriptions below.

## stat

This command causes Syd to output sandbox state on standard error.

## reset

This command causes Syd to reset sandboxing to the default state.
Allowlists, denylists and filters are going to be cleared. The state of
the sandbox lock is not affected by reset. This ensures an unintended
reset cannot open window for a sandbox bypass. In addition, the state of
Crypt sandboxing is not affected by reset too. This ensures concurrent
or near-concurrent encryption operations continue uninterrupted.

## panic

This command causes Syd to exit immediately with code 127.

Due to security reasons, this command is only available via the virtual stat
call, it may not be used with the *-m* command line switch or in a configuration
file.

Due to safety reasons, panic may not be called when Crypt sandboxing is
on. In this case the virtual _stat_(2) returns -1 and sets errno to
"EBUSY". This ensures concurrent or near-concurrent encryption
operations continue uninterrupted.

## ghost

This command initiates Ghost mode. Ghost mode is irreversible so you can
call this command only once during Syd runtime. See Ghost mode section
in _syd_(7) manual page for more information. In case of successful initiation,
the virtual _stat_(2) call returns -1 and sets the errno to "EOWNERDEAD".

Due to security reasons, this command implies "reset", ie. the sandbox
state is reset before Ghost mode initiation to ensure there're no
run-away exec processes after the invocation of the "ghost" command.

Due to security reasons, this command is only available via the virtual stat
call, it may not be used with the *-m* command line switch or in a configuration
file.

## config/expand

[[ type
:[ *integer (u64)*
|  default
:  *0*
|  static
:  *yes*

Given zero as timeout in seconds, which is the default, enables
environment variable and tilde expansion using the "shellexpand" crate.
This runs much faster as it does not require confinement, however it
does not support command substitution and recursive environment variable
expansion like _wordexp_(3) does. Notably, unset environment variables
are not expanded to empty strings. On environment variable lookup errors
and UTF-8 decoding errors Syd stops parsing and exits with error. This
is done for safety as an unintended empty-string expansion can
potentially cause the resulting sandboxing rule to allowlist unintended
paths without the user easily noticing it. The user is recommended to
set default values for environment variables using the familiar
"${HOME:-/var/empty}" notation.  If you really want empty-string
expansion on unset environment variables, you can get this effect using
the notation "${HOME:-}" but this is not recommended and should be used
with care.

Given a positive integer as timeout in seconds, enables environment
variable expansion and command substitutiton for configuration using
_wordexp_(3). The fork process which calls "/bin/sh" for expansion is
executed in a confined environment and it is terminated if its runtime
exceeds the given timeout. Confinement is done using Landlock,
namespaces and seccomp.

Note, this is a static, ie startup-only, setting: For safety reasons, no
expansion is performed for runtime configuration.

## lock

[[ type
:[ *string*

Set the state of the sandbox lock. Possible values are *on*, *off* and *exec*.
If the sandbox lock is *on* no sandbox commands are allowed. If *exec* is
specified, the sandbox lock is set to *on* for all processes except the initial
process, aka Syd exec child. Note, the sandbox lock used to default to *exec*
but as a hardening measure and to ensure security by default, as of version
3.17.0, this has been changed such that the default is _unset_ and if no lock
clause has been specified by the time Syd executes the initial sandbox process,
then the sandbox lock is automatically set to *on*. This means if no "lock"
clause is specified in any of the profiles, configuration files or "-m" CLI
arguments, the lock will be *on* by default. Setting lock to *on* at any point
during configuration parsing prevents further commands from being emitted by the
sandbox. This feature may be used to lock site-wide defaults for a Syd login
shell by adding a "lock:on" clause at the end of the site-wide configuration
file which prevents Syd from subsequently parsing the user configuration file,
practically enforcing the site-wide defaults.

Note, setting lock to *off* or *exec* at startup makes Syd skip preventing
_execve_(2) and _execveat_(2) system calls as part of the "Execution Control
(EEC)" feature. This is done to allow "cmd/exec" command to execute commands
outside the sandbox.

## log/level

[[ type
:[ *string*
|  default
:  *warn*

Set the log level. Available log levels are "emerg", "alert", "crit",
"error", "warn", "notice", "info", and "debug". Defaults to "warn"
unless "SYD_LOG" environment variable is set at startup.  Notably, Syd
logs access violations with the "warn" log level. You may also use an
integer in the closed range _[0,7]_ as an argument to set the level
where "0" corresponds to "emerg" and "7" corresponds to "debug".

## log/syslog

[[ type
:[ *boolean*
|  default
:  *on*

Set host syslogging. If this variable is set to on, Syd sends log
messages with log level "warn" and higher to _syslog_(3). Defaults to on
unless "SYD_NO_SYSLOG" environment variable is set at startup.

## log/verbose

[[ type
:[ *boolean*
|  default
:  *on*

Set verbose logging. In verbose mode, Syd
acquires various details about the current _seccomp_(2) request and adds
this information to the JSON payload under the "req" key. When disabled,
Syd only logs the process ID of the current _seccomp_(2) request. This
option is on by default, turning it off can help reduce log size.

## sandbox/stat

[[ type
:[ *boolean*
|  default
:  *on*
|  query
:  *yes*

Turn Stat sandboxing *on* or *off*.

## sandbox/read

[[ type
:[ *boolean*
|  default
:  *on*
|  query
:  *yes*

Turn Read sandboxing *on* or *off*.

## sandbox/write

[[ type
:[ *boolean*
|  default
:  *on*
|  query
:  *yes*

Turn Write sandboxing *on* or *off*.

## sandbox/exec

[[ type
:[ *boolean*
|  default
:  *on*
|  query
:  *yes*

Turn Exec sandboxing *on* or *off*.

## sandbox/ioctl

[[ type
:[ *boolean*
|  default
:  *off*
|  query
:  *yes*
|  static
:  *yes*

Turn Ioctl sandboxing *on* or *off*.

For performance reasons, this only works at startup. If not given at
startup, Syd will just allow _ioctl_(2) system calls at seccomp-bpf
level, and all _ioctl_(2) requests _except those in the denylist_ will be
allowed regardless of path. Turning this sandboxing off during runtime
is still possible, in this case the _ioctl_(2) system call handler will
only check the _ioctl_(2) request against the allowlist and the denylist
but will not check the path to the _ioctl_(2) file descriptor for
access.

## sandbox/create

[[ type
:[ *boolean*
|  default
:  *on*
|  query
:  *yes*

Turn Create sandboxing *on* or *off*.

## sandbox/delete

[[ type
:[ *boolean*
|  default
:  *on*
|  query
:  *yes*

Turn Delete sandboxing *on* or *off*.

## sandbox/rename

[[ type
:[ *boolean*
|  default
:  *on*
|  query
:  *yes*

Turn Rename sandboxing *on* or *off*.

## sandbox/symlink

[[ type
:[ *boolean*
|  default
:  *on*
|  query
:  *yes*

Turn Symlink sandboxing *on* or *off*.

## sandbox/truncate

[[ type
:[ *boolean*
|  default
:  *on*
|  query
:  *yes*

Turn Truncate sandboxing *on* or *off*.

## sandbox/chdir

[[ type
:[ *boolean*
|  default
:  *on*
|  query
:  *yes*

Turn Chdir sandboxing *on* or *off*.

## sandbox/readdir

[[ type
:[ *boolean*
|  default
:  *on*
|  query
:  *yes*

Turn Readdir sandboxing *on* or *off*.

## sandbox/mkdir

[[ type
:[ *boolean*
|  default
:  *on*
|  query
:  *yes*

Turn Mkdir sandboxing *on* or *off*.

## sandbox/rmdir

[[ type
:[ *boolean*
|  default
:  *on*
|  query
:  *yes*

Turn Rmdir sandboxing *on* or *off*.

## sandbox/chown

[[ type
:[ *boolean*
|  default
:  *on*
|  query
:  *yes*

Turn Chown sandboxing *on* or *off*.

## sandbox/chgrp

[[ type
:[ *boolean*
|  default
:  *on*
|  query
:  *yes*

Turn Chgrp sandboxing *on* or *off*.

## sandbox/chmod

[[ type
:[ *boolean*
|  default
:  *on*
|  query
:  *yes*

Turn Chmod sandboxing *on* or *off*.

## sandbox/chattr

[[ type
:[ *boolean*
|  default
:  *on*
|  query
:  *yes*

Turn Chattr sandboxing *on* or *off*.

## sandbox/chroot

[[ type
:[ *boolean*
|  default
:  *on*
|  query
:  *yes*

Turn Chroot sandboxing *on* or *off*.

## sandbox/utime

[[ type
:[ *boolean*
|  default
:  *on*
|  query
:  *yes*

Turn Utime sandboxing *on* or *off*.

## sandbox/mkdev

[[ type
:[ *boolean*
|  default
:  *on*
|  query
:  *yes*

Turn Mkdev sandboxing *on* or *off*.

## sandbox/mkfifo

[[ type
:[ *boolean*
|  default
:  *on*
|  query
:  *yes*

Turn Mkfifo sandboxing *on* or *off*.

## sandbox/mktemp

[[ type
:[ *boolean*
|  default
:  *on*
|  query
:  *yes*

Turn Mktemp sandboxing *on* or *off*.

## sandbox/net

[[ type
:[ *boolean*
|  default
:  *on*
|  query
:  *yes*

Turn Network sandboxing *on* or *off*.

## sandbox/lock

[[ type
:[ *boolean*
|  query
:  *yes*
|  static
:  *yes*

Turn Landlock sandboxing *on* or *off*.

## sandbox/force

[[ type
:[ *boolean*
|  query
:  *yes*

Turn Force sandboxing *on* or *off*.

## sandbox/tpe

[[ type
:[ *boolean*
|  default
:  *on*
|  query
:  *yes*

Turn Trusted Path Execution (TPE) sandboxing *on* or *off*.

## sandbox/crypt

[[ type
:[ *boolean*
|  query
:  *yes*

Turn Crypt sandboxing *on* or *off*.

To set this option *on*, a key must have already been specified with
"crypt/key".

Note, setting this sandboxing type to *on* implies
"trace/allow_safe_kcapi:1" to allow cryptographic operations using the
Kernel Cryptography API (KCAPI).

Note, setting this sandboxing type to *on* implies
"trace/exit_wait_all:1" so as not to leave any ongoing encryption
processes behind on sandbox process exit.

## sandbox/proxy

[[ type
:[ *boolean*
|  query
:  *yes*
|  static
:  *yes*

Turn Proxy sandboxing *on* or *off*.

Defaults to proxying through TOR. See the options "proxy/addr",
"proxy/port", "proxy/ext/host", and "proxy/ext/port" to configure a
different proxy.

Implies *unshare/net:1*.

## sandbox/mem

[[ type
:[ *boolean*
|  query
:  *yes*
|  static
:  *yes*

Turn Memory sandboxing *on* or *off*.

For performance reasons, this only works at startup. If not given at startup,
Syd will just allow _brk_(2), _mmap_(2), _mmap2_(2), and _mremap_(2) system
calls at seccomp-bpf level. Turning this sandboxing off during runtime is still
possible, in this case the respective system calls handlers will do nothing and
just continue the calls.

## sandbox/pid

[[ type
:[ *boolean*
|  query
:  *yes*
|  static
:  *yes*

Turn PID sandboxing *on* or *off*.

## default/stat

[[ type
:[ *string*
|  default
:  *deny*

Specify the default action for Stat sandboxing access violations.

The value must be exactly one of "allow", "warn", "filter", "deny",
"panic", "stop", "abort", "kill", or "exit", where the default is "deny".

## default/read

[[ type
:[ *string*
|  default
:  *deny*

Specify the default action for Read sandboxing access violations.

The value must be exactly one of "allow", "warn", "filter", "deny",
"panic", "stop", "abort", "kill", or "exit", where the default is "deny".

## default/write

[[ type
:[ *string*
|  default
:  *deny*

Specify the default action for Write sandboxing access violations.

The value must be exactly one of "allow", "warn", "filter", "deny",
"panic", "stop", "abort", "kill", or "exit", where the default is "deny".

## default/exec

[[ type
:[ *string*
|  default
:  *deny*

Specify the default action for Exec sandboxing access violations.

The value must be exactly one of "allow", "warn", "filter", "deny",
"panic", "stop", "abort", "kill", or "exit", where the default is "deny".

## default/ioctl

[[ type
:[ *string*
|  default
:  *deny*

Specify the default action for Ioctl sandboxing access violations.

The value must be exactly one of "allow", "warn", "filter", "deny",
"panic", "stop", "abort", "kill", or "exit", where the default is "deny".

## default/create

[[ type
:[ *string*
|  default
:  *deny*

Specify the default action for Create sandboxing access violations.

The value must be exactly one of "allow", "warn", "filter", "deny",
"panic", "stop", "abort", "kill", or "exit", where the default is "deny".

## default/delete

[[ type
:[ *string*
|  default
:  *deny*

Specify the default action for Delete sandboxing access violations.

The value must be exactly one of "allow", "warn", "filter", "deny",
"stop", "abort", "kill", "panic", or "exit", where the default is "deny".

## default/rename

[[ type
:[ *string*
|  default
:  *deny*

Specify the default action for Rename sandboxing access violations.

The value must be exactly one of "allow", "warn", "filter", "deny",
"stop", "abort", "kill", "panic", or "exit", where the default is "deny".

## default/symlink

[[ type
:[ *string*
|  default
:  *deny*

Specify the default action for Symlink sandboxing access violations.

The value must be exactly one of "allow", "warn", "filter", "deny",
"stop", "abort", "kill", "panic", or "exit", where the default is "deny".

## default/truncate

[[ type
:[ *string*
|  default
:  *deny*

Specify the default action for Truncate sandboxing access violations.

The value must be exactly one of "allow", "warn", "filter", "deny",
"panic", "stop", "abort", "kill", or "exit", where the default is "deny".

## default/chdir

[[ type
:[ *string*
|  default
:  *deny*

Specify the default action for Chdir sandboxing access violations.

The value must be exactly one of "allow", "warn", "filter", "deny",
"panic", "stop", "abort", "kill", or "exit", where the default is "deny".

## default/readdir

[[ type
:[ *string*
|  default
:  *deny*

Specify the default action for Readdir sandboxing access violations.

The value must be exactly one of "allow", "warn", "filter", "deny",
"panic", "stop", "abort", "kill", or "exit", where the default is "deny".

## default/mkdir

[[ type
:[ *string*
|  default
:  *deny*

Specify the default action for Mkdir sandboxing access violations.

The value must be exactly one of "allow", "warn", "filter", "deny",
"panic", "stop", "abort", "kill", or "exit", where the default is "deny".

## default/rmdir

[[ type
:[ *string*
|  default
:  *deny*

Specify the default action for Rmdir sandboxing access violations.

The value must be exactly one of "allow", "warn", "filter", "deny",
"panic", "stop", "abort", "kill", or "exit", where the default is "deny".

## default/chown

[[ type
:[ *string*
|  default
:  *deny*

Specify the default action for Chown sandboxing access violations.

The value must be exactly one of "allow", "warn", "filter", "deny",
"panic", "stop", "abort", "kill", or "exit", where the default is "deny".

## default/chgrp

[[ type
:[ *string*
|  default
:  *deny*

Specify the default action for Chgrp sandboxing access violations.

The value must be exactly one of "allow", "warn", "filter", "deny",
"panic", "stop", "abort", "kill", or "exit", where the default is "deny".

## default/chmod

[[ type
:[ *string*
|  default
:  *deny*

Specify the default action for Chmod sandboxing access violations.

The value must be exactly one of "allow", "warn", "filter", "deny",
"panic", "stop", "abort", "kill", or "exit", where the default is "deny".

## default/chattr

[[ type
:[ *string*
|  default
:  *deny*

Specify the default action for Chattr sandboxing access violations.

The value must be exactly one of "allow", "warn", "filter", "deny",
"panic", "stop", "abort", "kill", or "exit", where the default is "deny".

## default/chroot

[[ type
:[ *string*
|  default
:  *deny*

Specify the default action for Chattr sandboxing access violations.

The value must be exactly one of "allow", "warn", "filter", "deny",
"panic", "stop", "abort", "kill", or "exit", where the default is "deny".

## default/utime

[[ type
:[ *string*
|  default
:  *deny*

Specify the default action for Utime sandboxing access violations.

The value must be exactly one of "allow", "warn", "filter", "deny",
"panic", "stop", "abort", "kill", or "exit", where the default is "deny".

## default/mkdev

[[ type
:[ *string*
|  default
:  *deny*

Specify the default action for Mkdev sandboxing access violations.

The value must be exactly one of "allow", "warn", "filter", "deny",
"panic", "stop", "abort", "kill", or "exit", where the default is "deny".

## default/mkfifo

[[ type
:[ *string*
|  default
:  *deny*

Specify the default action for Mkfifo sandboxing access violations.

The value must be exactly one of "allow", "warn", "filter", "deny",
"panic", "stop", "abort", "kill", or "exit", where the default is "deny".

## default/mktemp

[[ type
:[ *string*
|  default
:  *deny*

Specify the default action for Mktemp sandboxing access violations.

The value must be exactly one of "allow", "warn", "filter", "deny",
"panic", "stop", "abort", "kill", or "exit", where the default is "deny".

## default/net

[[ type
:[ *string*
|  default
:  *deny*

Specify the default action for Network sandboxing access violations.

The value must be exactly one of "allow", "warn", "filter", "deny",
"panic", "stop", "abort", "kill", or "exit", where the default is "deny".

## default/block

[[ type
:[ *string*
|  default
:  *deny*

Specify the action for IP blocklist violations.

The value must be exactly one of "warn", "filter", "deny", "panic",
"stop", "abort", "kill", or "exit", where the default is "deny".

## default/force

[[ type
:[ *string*
|  default
:  *deny*

For force sandboxing, define the default action to take when the path of
a binary is not in the Integrity Force map.

The value must be either one of "warn", "filter", "deny", "panic",
"stop", "abort", "kill", "exit", where the default is "deny" for
_seccomp_(2) stops and "kill" for _ptrace_(2) stops.

## default/segvguard

[[ type
:[ *string*
|  default
:  *deny*

Specify the action for SegvGuard access violations.

The value must be exactly one of "warn", "filter", "deny", "panic",
"stop", "abort", "kill", or "exit", where the default is "deny" for
_seccomp_(2) stops and "kill" for _ptrace_(2) stops.

## default/tpe

[[ type
:[ *string*
|  default
:  *deny*

Specify the action for TPE sandboxing access violations.

The value must be exactly one of "warn", "filter", "deny", "panic",
"stop", "abort", "kill", or "exit", where the default is "deny" for
_seccomp_(2) stops and "kill" for _ptrace_(2) stops.

## default/mem

[[ type
:[ *string*
|  default
:  *deny*

Specify the action for Memory sandboxing access violations.

The value must be exactly one of "allow", "warn", "filter", "deny",
"panic", "stop", "abort", "kill", or "exit", where the default is "deny".

## default/pid

[[ type
:[ *string*
|  default
:  *kill*

Specify the action for PID sandboxing access violations.

The value must be either one of "warn", "filter", "stop", "abort",
"kill", "exit", where the default is "kill".

## unshare/mount

[[ type
:[ *boolean*
|  query
:  *yes*
|  static
:  *yes*

Create Mount namespace on startup.

## unshare/uts

[[ type
:[ *boolean*
|  query
:  *yes*
|  static
:  *yes*

Create UTS namespace on startup.

## unshare/ipc

[[ type
:[ *boolean*
|  query
:  *yes*
|  static
:  *yes*

Create IPC namespace on startup.

## unshare/user

[[ type
:[ *boolean*
|  query
:  *yes*
|  static
:  *yes*

Create User namespace on startup.

## unshare/pid

[[ type
:[ *boolean*
|  query
:  *yes*
|  static
:  *yes*

Create Pid namespace on startup.

## unshare/net

[[ type
:[ *boolean*
|  query
:  *yes*
|  static
:  *yes*

Create Net namespace on startup.

## unshare/cgroup

[[ type
:[ *boolean*
|  query
:  *yes*
|  static
:  *yes*

Create CGroup namespace on startup.

## unshare/time

[[ type
:[ *boolean*
|  query
:  *yes*
|  static
:  *yes*

Create Time namespace on startup. Syd resets the boot-time clock
such that _uptime_(1) will report container uptime rather than
host uptime.

## root

[[ type
:[ *string*
|  static
:  *yes*

Change directory and chroot to root before starting the process. The
directory *root/proc* _must_ exist to mount private proc filesystem.

As of version 3.23.14, symbolic links are not followed in any part of
the root directory and path traversal using ".." is not permitted. In
addition, root directory must be an absolute path, relative paths are
not permitted.

## root/map

[[ type
:[ *boolean*
|  static
:  *yes*

Map current user to root in the sandbox on startup.

This option does nothing without "unshare/user:1".

## root/fake

[[ type
:[ *boolean*
|  static
:  *yes*

In *fakeroot* mode, the system will return a user/group id of *0*,
mimicking the *root* user. This allows users to execute commands with
apparent root privileges, without actual superuser rights. It's useful
for tasks like package building where root-like environment is needed,
but not actual root permissions.

## name/host

[[ type
:[ *string*
|  default
:  *<auto>*
|  static
:  *yes*

Set host name in the sandbox. Only useful when combined with *unshare/uts:1*.

## name/domain

[[ type
:[ *string*
|  default
:  *syd-3*
|  static
:  *yes*

Set NIS/YP domain name in the sandbox. Only useful when combined with *unshare/uts:1*.

## time

[[ type
:[ *time_t*
|  static
:  *yes*

Set clock monotonic and boottime offset (seconds) in Time Namespace.

## ioctl/allow

[[ type
:[ *positive integer-array (u64)*
|  default
:  *[...]*

Add to or remove a request from the _ioctl_(2) request allowlist.
Accepts an unsigned 64-bit integer as argument. Prefix with "0x" for
hexadecimal and "0o" for octal input. Use "ioctl/allow+<request>" to add
to, and "ioctl/allow-<request>" to remove from the allowlist. By default
the list contains the _ioctl_(2) requests FIOCLEX, FIONCLEX, FIONBIO,
FIOASYNC, FIOQSIZE, FIFREEZE, FITHAW, FS_IOC_FIEMAP, FIGETBSZ, FICLONE,
FICLONERANGE, FIDEDUPERANGE, FS_IOC_GETFSUUID, and
FS_IOC_GETFSSYSFSPATH.

Note, for rules added at startup deny rules have precedence over allow
rules because the denylist is checked at kernel-space, whereas the
allowlist is checked at user-space. For rules added after startup, the
last matching rule wins.

## ioctl/deny

[[ type
:[ *positive integer-array (u64)*
|  default
:  *[...]*
|  static
:  *add is dynamic, remove is partly static*

Add to or remove a request from the _ioctl_(2) request denylist. Accepts
an unsigned 64-bit integer as argument. Prefix with "0x" for hexadecimal
and "0o" for octal input. Use "ioctl/deny+<request>" to add to, and
"ioctl/deny-<request>" to remove from the allowlist. By default the list
of denylisted _ioctl_(2) requests are FIBMAP, FS_IOC_FSGETXATTR,
FS_IOC_FSSETXATTR, FS_IOC_SETFLAGS, KDSETKEYCODE, KDSIGACCEPT, TIOCCONS,
TIOCLINUX, TIOCSETD, and TIOCSTI.

Note, for security reasons, the _ioctl_(2) denylist is applied at the
parent seccomp-bpf filter at startup. This means the Syd process is
included in this restriction as well. This also means, removing elements
from this list after startup has no effect. However, if Ioctl sandboxing
was enabled at startup, adding new elements to the _ioctl_(2) denylist
will further restrict the _ioctl_(2) request space.

Note, for rules added at startup, deny rules have precedence over allow
rules because the denylist is checked at kernel-space, whereas the
allowlist is checked at user-space. For rules added after startup, the
last matching rule wins.

Further reading about denylisted _ioctl_(2) requests:

. https://cve.mitre.org/cgi-bin/cvename.cgi?name=CVE-2023-1523
. https://a13xp0p0v.github.io/2017/03/24/CVE-2017-2636.html
. http://phrack.org/issues/52/6.html#article
. https://git.kernel.org/pub/scm/linux/kernel/git/torvalds/linux.git/commit/?id=83efeeeb3d04b22aaed1df99bc70a48fe9d22c4d
. https://git.kernel.org/pub/scm/linux/kernel/git/torvalds/linux.git/commit/?id=8d1b43f6a6df7bcea20982ad376a000d90906b42
. https://seclists.org/oss-sec/2024/q1/13
. https://seclists.org/oss-sec/2024/q1/14
. https://forums.grsecurity.net/viewtopic.php?f=7&t=2522
. http://lkml.indiana.edu/hypermail/linux/kernel/9907.0/0132.html
. http://linux.derkeiler.com/Mailing-Lists/Kernel/2007-11/msg07723.html

## mem/max

[[ type
:[ *positive integer (u64)*
|  default
:  *128MB*

This setting specifies the limit on per-process memory usage. Setting this value
to *0* disables testing for this type of memory usage. Note, the value is parsed
using the *parse-size* crate. Refer to their documentation for information on
formatting.

## mem/vm_max

[[ type
:[ *positive integer (u64)*
|  default
:  *0*

This setting specifies the limit on per-process virtual memory usage. Setting
this value to *0* disables testing for this type of memory usage. Note, the
value is parsed using the *parse-size* crate. Refer to their documentation for
information on formatting.

## pid/max

[[ type
:[ *positive integer (u64)*
|  default
:  *1*

This setting specifies the limit on the number of running tasks for pid
sandboxing. Setting this value to *0* is functionally equivalent to setting
*sandbox/pid* to *off*.

## bind

[[ type
:[ *string-array*
|  static
:  *yes*

This command causes Syd to bind mount a directory on startup. The
format is *source-dir:target-dir:mount-options,...* where _the source and
target directories may be equal_. Mount options are a comma-separated list
of a combination of the following options:

- "ro" to mount the filesystem read-only.
- "nodev" to not interpret character or block special devices on the filesystem.
- "noexec" to not permit direct execution of any binaries on the mounted \
filesystem.
- "nosuid" to not honour set-user-ID and set-group-ID bits or file capabilities \
when executing programs from this filesystem. In addition, SELinux domain \
transitions require permission "nosuid_transition", which in turn needs also \
policy capability "nnp_nosuid_transition".
- "nosymfollow" to not follow symbolic links when resolving paths. \
Symbolic links can still be created, and _readlink_(1), \
_readlink_(2), _realpath_(1), and _realpath_(3) all still work \
properly.
- "noatime" to not update inode access times on this filesystem (e.g. for \
faster access on the news spool to speed up news servers). This works for all \
inode types (directories too), so it implies "nodiratime".
- "nodiratime" to not update directory inode access times on this filesystem. \
(This option is implied when "noatime" is set.)
- "relatime" to update inode access times relative to modify or change time.

Mount options may be omitted. If the source directory does not start
with a slash, it is interpreted as the filesystem type rather than the
source directory. This may be used to mount special filesystems such as
cgroupfs, overlayfs or _tmpfs_(5) into the mount namespace. In this
case, any mount options supported by this filesystem type may be
submitted in options argument not just the ones listed above. You may
find some examples below:

- bind+/:/:ro
- bind+tmpfs:/tmp:noexec,size=16M
- bind+cgroup2:/sys/fs/cgroup:nodev,noexec,nosuid
- bind+overlay:/tmp/target:lowerdir=/tmp/lower,upperdir=/tmp/upper,workdir=/tmp/work,nosuid

This command may be used to create immutable containers. For example, the
command "bind+/:/:ro" is functionally equivalent to "deny/write+/\*\*\*" except
the restriction happens at _kernel VFS layer_ rather than at user level using
_seccomp-notify_. Alternatively this can also be achieved at the kernel level
using _Landlock_.

If the source or target directory does not exist, Syd logs an "info"
level entry and continues processing the rest of the mounts. Any other
error during bind mounting is fatal.

As of version 3.23.14, symbolic links are not followed in any part of
the source or target directory paths and path traversal using ".." is
not permitted. In addition, target directory must be an absolute path,
relative paths are not permitted.

As of version 3.23.14, mounting the special "proc" filesystem under a
custom path is not permitted. Syd handles this mount itself specially
after all bind mounts are processed.

## crypt

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to encrypt for *Crypt sandboxing*.

## crypt/kdf/salt

[[ type
:[ *string*
|  default
:  *<empty>*
|  static
:  *yes*

Specify salt used in key derivation function for *Crypt sandboxing*.

There are two usage options for the salt:

- Empty or static for domain separation in a private setting
- Guaranteed to be uniformly-distributed and unique in a public setting

## crypt/kdf/info/enc

[[ type
:[ *string*
|  default
:  *SYD-ENC*
|  static
:  *yes*

Specify informational context string for encryption key used in key
derivation function for *Crypt sandboxing*.

This information acts as a label and is not private.

## crypt/kdf/info/mac

[[ type
:[ *string*
|  default
:  *SYD-MAC*
|  static
:  *yes*

Specify informational context string for authentication key used in key
derivation function for *Crypt sandboxing*.

This information acts as a label and is not private.

## crypt/key

[[ type
:[ *string*
|  static
:  *yes*

Specify 256-bit AES-CTR key for *Crypt sandboxing*. The key must be
encoded in hexadecimal and be exactly 64 characters.

## crypt/tmp

[[ type
:[ *string*
|  default
:  mem
|  static
:  *yes*

Specify temporary backing directory for transparent file decryption. The
argument must be an absolute path or the special value "mem". The user
must ensure this directory is secure as decrypted contents will be
written to temporary files under this directory. Specify the special
value "mem" to use anonymous files which live in RAM with a volatile
backing storage created with _memfd_create_(2). This is the default. The
user is encouraged to specify this option for efficient handling of
large files for *Crypt sandboxing*.

## force

[[ type
:[ *string-array*

Add or remove an integrity force rule for Force Sandboxing. The format
is _force+/path:hashhex:action_ for addition and _force-/path_ for
removal. Use _force^_ to clear the Integrity Force map. Available
actions are "warn", "filter", "deny", "panic", "stop", "abort", "kill"
and "exit" where the default is "deny". hashhex is either a 8-character
CRC32 checksum, 16-character CRC64 checksum, 32-character MD5 checksum,
a 40-character SHA1 checksum, a 64-character SHA3-256 checksum, a
96-character SHA3-384 checksum or a 128-character SHA3-512 checksum.

- _syd-sha_(1) is a helper tool to calculate checksums of files.
- _syd-path_(1) is a helper tool to write integrity force rules for binaries under PATH.

## proxy/addr

[[ type
:[ *IP address*
|  default
:  *127.0.0.1*
|  static
:  *yes*

Set internal address for Proxy sandboxing. This must be an IPv4 or an
IPv6 address. Defaults to 127.0.0.1.

## proxy/port

[[ type
:[ *integer*
|  default
:  *9050*
|  static
:  *yes*

Set internal port for Proxy sandboxing. Defaults to 9050.

## proxy/ext/host

[[ type
:[ *Hostname or IP*
|  default
:  *127.0.0.1*
|  static
:  *yes*

Set external address for Proxy sandboxing. This must either be an IPv4
address or an IPv6 address or a hostname. If the argument does not parse
as an IP address, Syd resolves the name using the system DNS resolver
and selects a response IP randomly.

Defaults to "127.0.0.1", which may be be overriden with the environment variable
*SYD_PROXY_HOST* at startup.

## proxy/ext/port

[[ type
:[ *integer*
|  default
:  *9050*
|  static
:  *yes*

Set external port for Proxy sandboxing.

Defaults to 9050, which may be overriden with the environment variable
*SYD_PROXY_PORT* at startup.

## segvguard/expiry

[[ type
:[ *integer (u64)*
|  default
:  120

Specify SegvGuard expiry timeout in seconds. Set to 0 to disable SegvGuard.

## segvguard/suspension

[[ type
:[ *integer (u64)*
|  default
:  600

Specify SegvGuard suspension timeout in seconds.

## segvguard/maxcrashes

[[ type
:[ *integer (u8)*
|  default
:  5

Specify SegvGuard max crashes.

## tpe/gid

[[ type
:[ *integer (uid_t)*

Specify untrusted GID for Trusted Path Execution (TPE). By default, TPE
is applied to users of all groups including root and this setting can be
used to limit it to a certain group. To unset a previously set GID and
return to the default state set "none" as the value.

## tpe/negate

[[ type
:[ *boolean*

Negate GID logic for Trusted Path Execution (TPE). This turns "tpe/gid"
from untrusted into trusted such that users belonging to this group will
be exempt from TPE.

## tpe/root_owned

[[ type
:[ *boolean*

Ensure directory is root-owned for Trusted Path Execution (TPE).

Note, this option will misbehave with "unshare/user:1" if the real root
user is not mapped inside the container.

## tpe/user_owned

[[ type
:[ *boolean*

Ensure directory is user-owned or root for Trusted Path Execution (TPE).

Note, this option will misbehave with "unshare/user:1" if the real root
is not mapped inside the container.

## allow/stat

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to allow for *Stat sandboxing*.

## allow/read

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to allow for *Read sandboxing*.

## allow/write

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to allow for *Write sandboxing*.

## allow/exec

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to allow for *Exec sandboxing*.

## allow/ioctl

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to allow for *Ioctl sandboxing*.

## allow/create

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to allow for *Create sandboxing*.

## allow/delete

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to allow for *Delete sandboxing*.

## allow/rename

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to allow for *Rename sandboxing*.

## allow/symlink

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to allow for *Symlink sandboxing*.

## allow/truncate

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to allow for *Truncate sandboxing*.

## allow/chdir

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to allow for *Chdir sandboxing*.

## allow/readdir

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to allow for *Readdir sandboxing*.

## allow/mkdir

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to allow for *Mkdir sandboxing*.

## allow/rmdir

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to allow for *Rmdir sandboxing*.

## allow/chown

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to allow for *Chown sandboxing*.

## allow/chgrp

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to allow for *Chgrp sandboxing*.

## allow/chmod

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to allow for *Chmod sandboxing*.

## allow/chattr

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to allow for *Chattr sandboxing*.

## allow/chroot

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to allow for *Chroot sandboxing*.

## allow/utime

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to allow for *Utime sandboxing*.

## allow/mkdev

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to allow for *Mkdev sandboxing*.

## allow/mkfifo

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to allow for *Mkfifo sandboxing*.

## allow/mktemp

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to allow for *Mktemp sandboxing*.

## allow/net/bind

[[ type
:[ *string-array*

Specifies a list of network address patterns to allow for *Bind network sandboxing*.

## allow/net/connect

[[ type
:[ *string-array*

Specifies a list of network address patterns to allow for *Connect network sandboxing*.

## allow/net/sendfd

[[ type
:[ *string-array*

Specifies a list of network address patterns to allow for *SendFd network sandboxing*.

## allow/net/link

[[ type
:[ *string-array*
|  static
:  *yes*

Specifies a list of netlink families to allow for *Link network sandboxing*.

Accepts a comma-delimited list of the following items: "route",
"usersock", "firewall", "sock_diag", "nflog", "xfrm", "selinux",
"iscsi", "audit", "fib_lookup", "connector", "netfilter", "ip6_fw",
"dnrtmsg", "kobject_uevent", "generic", "scsitransport", "ecryptfs",
"rdma", "crypto", and "smc".

## allow/lock/read

[[ type
:[ *string-set*
|  static
:  *yes*
|  default
:  _("/dev/null", "/proc")_

Specifies a set of beneath paths to grant file read access for *Lock
sandboxing*. This category corresponds to the Landlock access right
"LANDLOCK_ACCESS_FS_READ_FILE" and only applies to the content of the
directory not the directory itself. As of version 3.21.0, this set
includes the paths "/dev/null" and "/proc" by default as Syd is included
in the Landlock sandbox and Syd requires read access to these paths to
function correctly.

## allow/lock/write

[[ type
:[ *string-set*
|  static
:  *yes*
|  default
:  _("/dev/null")_

Specifies a set of beneath paths to grant file write access for *Lock
sandboxing*. This category corresponds to the Landlock access right
"LANDLOCK_ACCESS_FS_WRITE_FILE" and only applies to the content of the
directory not the directory itself. As of version 3.21.0, this set
includes the path "/dev/null" by default as Syd is included in the
Landlock sandbox and Syd requires write access to this file to function
correctly.

## allow/lock/exec

[[ type
:[ *string-set*
|  static
:  *yes*

Specifies a set of beneath paths to grant file execute access for *Lock
sandboxing*. This category corresponds to the Landlock access right
"LANDLOCK_ACCESS_FS_EXECUTE" and only applies to the content of the
directory not the directory itself.

## allow/lock/ioctl

[[ type
:[ *string-set*
|  static
:  *yes*

Specifies a set of beneath paths to grant _ioctl_(2) access for *Lock
sandboxing*. This category corresponds to the Landlock access right
"LANDLOCK_ACCESS_FS_IOCTL_DEV" and only applies to the content of the
directory not the directory itself. Landlock _ioctl_(2) support requires
ABI 5 or later. Fifth Landlock ABI was introduced with Linux 6.10. On
older kernels, this command is a no-op and is not going to confine
_ioctl_(2) operations.

## allow/lock/create

[[ type
:[ *string-set*
|  static
:  *yes*

Specifies a set of beneath paths to grant file creation, rename and link
access for *Lock sandboxing*. This category corresponds to the Landlock
access right "LANDLOCK_ACCESS_FS_MAKE_REG" and only applies to the
content of the directory not the directory itself.

## allow/lock/delete

[[ type
:[ *string-set*
|  static
:  *yes*

Specifies a set of beneath paths to grant file unlink, rename and link
access for *Lock sandboxing*. This category corresponds to the Landlock
access right "LANDLOCK_ACCESS_FS_REMOVE_FILE" and only applies to the
content of the directory not the directory itself.

## allow/lock/rename

[[ type
:[ *string-set*
|  static
:  *yes*

Specifies a set of beneath paths to grant access to link or rename a
file from or to a different directory (i.e. reparent a file hierarchy)
for *Lock sandboxing*. This category corresponds to the Landlock access
right "LANDLOCK_ACCESS_FS_REFER" and only applies to the content of the
directory not the directory itself. Landlock rename support requires ABI
2 or later. Second Landlock ABI was introduced with Linux 5.19. On older
kernels, this type of access is always denied with Landlock.

## allow/lock/symlink

[[ type
:[ *string-set*
|  static
:  *yes*

Specifies a set of beneath paths to grant symbolic link creation, rename
and link access for *Lock sandboxing*. This category corresponds to the
Landlock access right "LANDLOCK_ACCESS_FS_MAKE_SYM" and only applies
to the content of the directory not the directory itself.

## allow/lock/truncate

[[ type
:[ *string-set*
|  static
:  *yes*
|  default
:  _("/dev/null")_

Specifies a set of beneath paths to grant file truncation access for
*Lock sandboxing*. This category corresponds to the Landlock access
right "LANDLOCK_ACCESS_FS_TRUNCATE" and only applies to the content of
the directory not the directory itself. Landlock file truncation support
requires ABI 3 or later. Third Landlock ABI was introduced with Linux
6.2. On older kernels, this command is a no-op and is not going to
confine file truncation operations. As of version 3.21.0, this set
includes the path "/dev/null" by default as Syd is included in the
Landlock sandbox and Syd requires truncation access to this file to
function correctly.

## allow/lock/readdir

[[ type
:[ *string-set*
|  static
:  *yes*
|  default
:  _("/proc")_

Specifies a set of beneath paths to grant directory list access for
*Lock sandboxing*. This category corresponds to the Landlock access
right "LANDLOCK_ACCESS_FS_READ_DIR" and applies to the directory
and the directories beneath it. As of version 3.21.0, this set includes
the directory "/proc" by default as Syd is included in the Landlock
sandbox and Syd requires readdir access to this directory to function
correctly.

## allow/lock/mkdir

[[ type
:[ *string-set*
|  static
:  *yes*

Specifies a set of beneath paths to grant directory creation and rename
access for *Lock sandboxing*. This category corresponds to the Landlock
access right "LANDLOCK_ACCESS_FS_MAKE_DIR" and only applies to the
content of the directory not the directory itself.

## allow/lock/rmdir

[[ type
:[ *string-set*
|  static
:  *yes*

Specifies a set of beneath paths to grant directory deletion and rename
access for *Lock sandboxing*. This category corresponds to the Landlock
access right "LANDLOCK_ACCESS_FS_REMOVE_DIR" and only applies to the
content of the directory not the directory itself.

## allow/lock/mkdev

[[ type
:[ *string-set*
|  static
:  *yes*

Specifies a set of beneath paths to grant character device creation
access for *Lock sandboxing*. This category corresponds to the Landlock
access right "LANDLOCK_ACCESS_FS_MAKE_CHAR" and only applies to the
content of the directory not the directory itself.

## allow/lock/mkfifo

[[ type
:[ *string-set*
|  static
:  *yes*

Specifies a set of beneath paths to grant named pipe (FIFO) creation
access for *Lock sandboxing*. This category corresponds to the Landlock
access right "LANDLOCK_ACCESS_FS_MAKE_FIFO" and only applies to the
content of the directory not the directory itself.

## allow/lock/bind

[[ type
:[ *(u16-set, string-set)*
|  static
:  *yes*

Specifies a list of allowed _bind_(2) ports and UNIX domain socket paths
for *Lock sandboxing*. This category corresponds to the Landlock access
rights "LANDLOCK_ACCESS_NET_BIND_TCP" and "LANDLOCK_ACCESS_FS_MAKE_SOCK"
and only applies to the content of the directory not the directory
itself. Argument is either a single port or a closed range in format
*port1-port2*, or an absolute UNIX domain socket path. Landlock network
support requires ABI 4 or later. Fourth Landlock ABI was introduced with
Linux 6.7. On older kernels, this command is a no-op when specified with
port arguments and does not do any network confinement.

## allow/lock/connect

[[ type
:[ *u16-set*
|  static
:  *yes*

Specifies a list of allowed _connect_(2) ports for *Lock sandboxing*.
This category corresponds to the Landlock access right
"LANDLOCK_ACCESS_NET_BIND_CONNECT". Argument is either a single port or
a closed range in format *port1-port2*. Landlock network support
requires ABI 4 or later. Fourth Landlock ABI was introduced with Linux
6.7. On older kernels, this command is a no-op and does not do any
network confinement.

## warn/stat

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to warn for *Stat sandboxing*.

## warn/read

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to warn for *Read sandboxing*.

## warn/write

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to warn for *Write sandboxing*.

## warn/exec

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to warn for *Exec sandboxing*.

## warn/ioctl

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to warn for *Ioctl sandboxing*.

## warn/create

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to warn for *Create sandboxing*.

## warn/delete

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to warn for *Delete sandboxing*.

## warn/rename

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to warn for *Rename sandboxing*.

## warn/symlink

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to warn for *Symlink sandboxing*.

## warn/truncate

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to warn for *Truncate sandboxing*.

## warn/chdir

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to warn for *Chdir sandboxing*.

## warn/readdir

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to warn for *Readdir sandboxing*.

## warn/mkdir

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to warn for *Mkdir sandboxing*.

## warn/rmdir

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to warn for *Rmdir sandboxing*.

## warn/chown

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to warn for *Chown sandboxing*.

## warn/chgrp

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to warn for *Chgrp sandboxing*.

## warn/chmod

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to warn for *Chmod sandboxing*.

## warn/chattr

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to warn for *Chattr sandboxing*.

## warn/chroot

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to warn for *Chroot sandboxing*.

## warn/utime

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to warn for *Utime sandboxing*.

## warn/mkdev

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to warn for *Mkdev sandboxing*.

## warn/mkfifo

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to warn for *Mkfifo sandboxing*.

## warn/mktemp

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to warn for *Mktemp sandboxing*.

## warn/net/bind

[[ type
:[ *string-array*

Specifies a list of network address patterns to warn for *Bind network sandboxing*.

## warn/net/connect

[[ type
:[ *string-array*

Specifies a list of network address patterns to warn for *Connect network sandboxing*.

## warn/net/sendfd

[[ type
:[ *string-array*

Specifies a list of network address patterns to warn for *SendFd network sandboxing*.

## deny/stat

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to deny for *Stat sandboxing*.

## deny/read

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to deny for *Read sandboxing*.

## deny/write

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to deny for *Write sandboxing*.

## deny/exec

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to deny for *Exec sandboxing*.

## deny/ioctl

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to deny for *Ioctl sandboxing*.

## deny/create

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to deny for *Create sandboxing*.

## deny/delete

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to deny for *Delete sandboxing*.

## deny/rename

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to deny for *Rename sandboxing*.

## deny/symlink

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to deny for *Symlink sandboxing*.

## deny/truncate

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to deny for *Truncate sandboxing*.

## deny/chdir

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to deny for *Chdir sandboxing*.

## deny/readdir

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to deny for *Readdir sandboxing*.

## deny/mkdir

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to deny for *Mkdir sandboxing*.

## deny/rmdir

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to deny for *Rmdir sandboxing*.

## deny/chown

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to deny for *Chown sandboxing*.

## deny/chgrp

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to deny for *Chgrp sandboxing*.

## deny/chmod

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to deny for *Chmod sandboxing*.

## deny/chattr

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to deny for *Chattr sandboxing*.

## deny/chroot

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to deny for *Chroot sandboxing*.

## deny/utime

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to deny for *Utime sandboxing*.

## deny/mkdev

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to deny for *Mkdev sandboxing*.

## deny/mkfifo

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to deny for *Mkfifo sandboxing*.

## deny/mktemp

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to deny for *Mktemp sandboxing*.

## deny/net/bind

[[ type
:[ *string-array*

Specifies a list of network address patterns to deny for *Bind network sandboxing*.

## deny/net/connect

[[ type
:[ *string-array*

Specifies a list of network address patterns to deny for *Connect network sandboxing*.

## deny/net/sendfd

[[ type
:[ *string-array*

Specifies a list of network address patterns to deny for *SendFd network sandboxing*.

## panic/stat

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to panic for *Stat sandboxing*.

## panic/read

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to panic for *Read sandboxing*.

## panic/write

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to panic for *Write sandboxing*.

## panic/exec

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to panic for *Exec sandboxing*.

## panic/ioctl

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to panic for *Ioctl sandboxing*.

## panic/create

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to panic for *Create sandboxing*.

## panic/delete

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to panic for *Delete sandboxing*.

## panic/rename

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to panic for *Rename sandboxing*.

## panic/symlink

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to panic for *Symlink sandboxing*.

## panic/truncate

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to panic for *Truncate sandboxing*.

## panic/chdir

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to panic for *Chdir sandboxing*.

## panic/readdir

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to panic for *Readdir sandboxing*.

## panic/mkdir

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to panic for *Mkdir sandboxing*.

## panic/rmdir

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to panic for *Rmdir sandboxing*.

## panic/chown

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to panic for *Chown sandboxing*.

## panic/chgrp

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to panic for *Chgrp sandboxing*.

## panic/chmod

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to panic for *Chmod sandboxing*.

## panic/chattr

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to panic for *Chattr sandboxing*.

## panic/chroot

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to panic for *Chroot sandboxing*.

## panic/utime

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to panic for *Utime sandboxing*.

## panic/mkdev

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to panic for *Mkdev sandboxing*.

## panic/mkfifo

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to panic for *Mkfifo sandboxing*.

## panic/mktemp

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to panic for *Mktemp sandboxing*.

## panic/net/bind

[[ type
:[ *string-array*

Specifies a list of network address patterns to panic for *Bind network sandboxing*.

## panic/net/connect

[[ type
:[ *string-array*

Specifies a list of network address patterns to panic for *Connect network sandboxing*.

## panic/net/sendfd

[[ type
:[ *string-array*

Specifies a list of network address patterns to panic for *SendFd network sandboxing*.

## stop/stat

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to stop for *Stat sandboxing*.

## stop/read

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to stop for *Read sandboxing*.

## stop/write

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to stop for *Write sandboxing*.

## stop/exec

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to stop for *Exec sandboxing*.

## stop/ioctl

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to stop for *Ioctl sandboxing*.

## stop/create

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to stop for *Create sandboxing*.

## stop/delete

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to stop for *Delete sandboxing*.

## stop/rename

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to stop for *Rename sandboxing*.

## stop/symlink

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to stop for *Symlink sandboxing*.

## stop/truncate

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to stop for *Truncate sandboxing*.

## stop/chdir

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to stop for *Chdir sandboxing*.

## stop/readdir

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to stop for *Readdir sandboxing*.

## stop/mkdir

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to stop for *Mkdir sandboxing*.

## stop/rmdir

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to stop for *Rmdir sandboxing*.

## stop/chown

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to stop for *Chown sandboxing*.

## stop/chgrp

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to stop for *Chgrp sandboxing*.

## stop/chmod

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to stop for *Chmod sandboxing*.

## stop/chattr

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to stop for *Chattr sandboxing*.

## stop/chroot

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to stop for *Chroot sandboxing*.

## stop/utime

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to stop for *Utime sandboxing*.

## stop/mkdev

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to stop for *Mkdev sandboxing*.

## stop/mkfifo

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to stop for *Mkfifo sandboxing*.

## stop/mktemp

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to stop for *Mktemp sandboxing*.

## stop/net/bind

[[ type
:[ *string-array*

Specifies a list of network address patterns to stop for *Bind network sandboxing*.

## stop/net/connect

[[ type
:[ *string-array*

Specifies a list of network address patterns to stop for *Connect network sandboxing*.

## stop/net/sendfd

[[ type
:[ *string-array*

Specifies a list of network address patterns to stop for *SendFd network sandboxing*.

## abort/stat

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to abort for *Stat sandboxing*.

## abort/read

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to abort for *Read sandboxing*.

## abort/write

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to abort for *Write sandboxing*.

## abort/exec

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to abort for *Exec sandboxing*.

## abort/ioctl

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to abort for *Ioctl sandboxing*.

## abort/create

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to abort for *Create sandboxing*.

## abort/delete

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to abort for *Delete sandboxing*.

## abort/rename

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to abort for *Rename sandboxing*.

## abort/symlink

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to abort for *Symlink sandboxing*.

## abort/truncate

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to abort for *Truncate sandboxing*.

## abort/chdir

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to abort for *Chdir sandboxing*.

## abort/readdir

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to abort for *Readdir sandboxing*.

## abort/mkdir

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to abort for *Mkdir sandboxing*.

## abort/rmdir

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to abort for *Rmdir sandboxing*.

## abort/chown

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to abort for *Chown sandboxing*.

## abort/chgrp

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to abort for *Chgrp sandboxing*.

## abort/chmod

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to abort for *Chmod sandboxing*.

## abort/chattr

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to abort for *Chattr sandboxing*.

## abort/chroot

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to abort for *Chroot sandboxing*.

## abort/utime

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to abort for *Utime sandboxing*.

## abort/mkdev

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to abort for *Mkdev sandboxing*.

## abort/mkfifo

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to abort for *Mkfifo sandboxing*.

## abort/mktemp

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to abort for *Mktemp sandboxing*.

## abort/net/bind

[[ type
:[ *string-array*

Specifies a list of network address patterns to abort for *Bind network sandboxing*.

## abort/net/connect

[[ type
:[ *string-array*

Specifies a list of network address patterns to abort for *Connect network sandboxing*.

## abort/net/sendfd

[[ type
:[ *string-array*

Specifies a list of network address patterns to abort for *SendFd network sandboxing*.

## kill/stat

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to kill for *Stat sandboxing*.

## kill/read

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to kill for *Read sandboxing*.

## kill/write

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to kill for *Write sandboxing*.

## kill/exec

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to kill for *Exec sandboxing*.

## kill/ioctl

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to kill for *Ioctl sandboxing*.

## kill/create

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to kill for *Create sandboxing*.

## kill/delete

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to kill for *Delete sandboxing*.

## kill/rename

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to kill for *Rename sandboxing*.

## kill/symlink

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to kill for *Symlink sandboxing*.

## kill/truncate

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to kill for *Truncate sandboxing*.

## kill/chdir

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to kill for *Chdir sandboxing*.

## kill/readdir

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to kill for *Readdir sandboxing*.

## kill/mkdir

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to kill for *Mkdir sandboxing*.

## kill/rmdir

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to kill for *Rmdir sandboxing*.

## kill/chown

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to kill for *Chown sandboxing*.

## kill/chgrp

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to kill for *Chgrp sandboxing*.

## kill/chmod

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to kill for *Chmod sandboxing*.

## kill/chattr

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to kill for *Chattr sandboxing*.

## kill/chroot

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to kill for *Chroot sandboxing*.

## kill/utime

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to kill for *Utime sandboxing*.

## kill/mkdev

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to kill for *Mkdev sandboxing*.

## kill/mkfifo

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to kill for *Mkfifo sandboxing*.

## kill/mktemp

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to kill for *Mktemp sandboxing*.

## kill/net/bind

[[ type
:[ *string-array*

Specifies a list of network address patterns to kill for *Bind network sandboxing*.

## kill/net/connect

[[ type
:[ *string-array*

Specifies a list of network address patterns to kill for *Connect network sandboxing*.

## kill/net/sendfd

[[ type
:[ *string-array*

Specifies a list of network address patterns to kill for *SendFd network sandboxing*.

## exit/stat

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to exit for *Stat sandboxing*.

## exit/read

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to exit for *Read sandboxing*.

## exit/write

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to exit for *Write sandboxing*.

## exit/exec

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to exit for *Exec sandboxing*.

## exit/ioctl

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to exit for *Ioctl sandboxing*.

## exit/create

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to exit for *Create sandboxing*.

## exit/delete

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to exit for *Delete sandboxing*.

## exit/rename

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to exit for *Rename sandboxing*.

## exit/symlink

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to exit for *Symlink sandboxing*.

## exit/truncate

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to exit for *Truncate sandboxing*.

## exit/chdir

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to exit for *Chdir sandboxing*.

## exit/readdir

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to exit for *Readdir sandboxing*.

## exit/mkdir

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to exit for *Mkdir sandboxing*.

## exit/rmdir

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to exit for *Rmdir sandboxing*.

## exit/chown

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to exit for *Chown sandboxing*.

## exit/chgrp

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to exit for *Chgrp sandboxing*.

## exit/chmod

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to exit for *Chmod sandboxing*.

## exit/chattr

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to exit for *Chattr sandboxing*.

## exit/chroot

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to exit for *Chroot sandboxing*.

## exit/utime

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to exit for *Utime sandboxing*.

## exit/mkdev

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to exit for *Mkdev sandboxing*.

## exit/mkfifo

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to exit for *Mkfifo sandboxing*.

## exit/mktemp

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to exit for *Mktemp sandboxing*.

## exit/net/bind

[[ type
:[ *string-array*

Specifies a list of network address patterns to exit for *Bind network sandboxing*.

## exit/net/connect

[[ type
:[ *string-array*

Specifies a list of network address patterns to exit for *Connect network sandboxing*.

## exit/net/sendfd

[[ type
:[ *string-array*

Specifies a list of network address patterns to exit for *SendFd network sandboxing*.

## append

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to files that should be made
append-only for *Write sandboxing*.

If a path is append-only, Syd adds "O_APPEND" and removes "O_TRUNC" from
flags on any sandbox granted attempt to _open_(2) this path. Unsetting
the "O_APPEND" flag using _fcntl_(2) "F_SETFL" command is prevented.
Similarly, any attempt to _rename_(2), _truncate_(2) and _unlink_(2) the
file is prevented. This is typically useful for history and log files.

## mask

[[ type
:[ *string-array*

Specifies a list of _glob_(3p) patterns to mask for *Read & Write sandboxing*.

If a path is masked, Syd returns a file descriptor to "/dev/null" on any
sandbox granted attempt to _open_(2) this path. Masking can effectively
be used to hide the contents of a file in a more relaxed and compatible
way than denying read/write access to it. _stat_(2) calls on a masked
file returns the original file metadata and a masked file may be
executed.

## block

[[ type
:[ *ip-range*

Specifies a range of IP networks to be blocked when specified as the
target address of "connect" group system calls which are _connect_(2),
_sendto_(2), _sendmsg_(2), _sendmmsg_(2) and when received as the source
address in return from _accept_(2) and _accept4_(2) system calls for
IPv4 and IPv6 family sockets. Use "block+<net>" and "block-<net>" to add
and remove ip networks from the range. Alternatively the range can also
be populated by including "ipset" and "netset" files from within Syd
configuration.  Use "block^" to clear the list and "block!" to simplify
the ip range by aggregating networks together. "block!" is useful to
call after importing big IP blocklists, it helps reduce memory
consumption and improve matching performance. Below is a configuration
snippet that imports Feodo and DShield blocklists:

```
# Enable IP blocklists
# Source: https://github.com/firehol/blocklist-ipsets.git
include /usr/src/blocklist-ipsets/feodo.ipset
include /usr/src/blocklist-ipsets/feodo_badips.ipset
include /usr/src/blocklist-ipsets/dshield.netset
include /usr/src/blocklist-ipsets/dshield_1d.netset
include /usr/src/blocklist-ipsets/dshield_30d.netset
include /usr/src/blocklist-ipsets/dshield_7d.netset
include /usr/src/blocklist-ipsets/dshield_top_1000.ipset
block!
```

## cmd/exec

[[ type
:[ *command*

Makes Syd execute an external command without sandboxing. The process is
executed in a new process group with its standard input attached to
"/dev/null". Standard output and standard error file descriptors are
inherited. Syd also ensures no non-standard file descriptors leak into
the new process utilizing the _close_range_(2) system call. Current
working directory is changed to the root directory, aka "/". The
_umask_(2) is set to 077. The program name and arguments must be
separated with the *US* (unit separator, hex: 0x1f, octal: 037)
character. To ease usage, the _syd-exec_(1) helper utility is provided
to construct a sandbox command of this type:

```
; syd -puser -mlock:exec -- sh -c 'test -c $(syd-exec echo hello world)'
hello world
;
```

## load

[[ type
:[ *integer (fd) or string (profile-name)*

Read configuration from the given file descriptor, the file must be open for
reading. Syd uses _pidfd_getfd_(2) to acquire the file descriptor and reads
sandbox configuration from it. This command is useful to load a set of sandbox
commands into Syd in a single step and is typically used with *reset*, e.g:

```
int fd = open("/tmp", O_RDWR | O_TMPFILE | O_CLOEXEC, 0);
if (fd == -1) errx(1, "Failed to open temporary file");

const char *syd = "sandbox:stat/on\\nallow/stat+/***\\ndeny/stat+/\\nlock:on\\n";
errx(write(fd, syd, strlen(syd)) == -1, "Failed to write config");
errx(lseek(fd, 0, SEEK_SET) == -1, "Failed to seek in file");

char load[64];
sprintf(load, "/dev/syd/load/%d", fd);
errx(stat("/dev/syd/reset", NULL) == -1, "Failed to reset syd");
errx(stat(load, NULL) == -1, "Failed to load syd profile");

errx(execvp("/bin/sh", (char *[]){"/bin/sh", "-l", NULL}) == -1, "execvp failed");
```

Due to security reasons, this command is only available via the virtual
_stat_(2) call, it may not be used with the *-m* command line switch or
in a configuration file.

As of version 3.30.0, this command may be used to load builtin profiles,
when Syd falls back to parsing the "load" argument as a profile name if
parsing the argument as a file descriptor fails.

## trace/allow_safe_setuid

[[ type
:[ *boolean*
|  static
:  *yes*

Specify whether the Linux capability "CAP_SETUID" should be retained.
This option in combination with *SafeSetID* allows the sandbox process
to change UID. Note, Syd will change its UID with the sandbox process.

Note, because NPTL uses real-time signals to ensure all threads share
the same UID/GID, setting this option disables the SROP mitigator. See
the "Enhanced Execution Control (EEC)" section of the _syd_(7) manual
page for more information.

## trace/allow_safe_setgid

[[ type
:[ *boolean*
|  static
:  *yes*

Specify whether the Linux capability "CAP_SETGID" should be retained.
This option in combination with *SafeSetID* allows the sandbox process
to change GID. Note, Syd will change its GID with the sandbox process.

Note, because NPTL uses real-time signals to ensure all threads share
the same UID/GID, setting this option disables the SROP mitigator. See
the "Enhanced Execution Control (EEC)" section of the _syd_(7) manual
page for more information.

## setuid

[[ type
:[ *[(uid, uid)]*

Add, remove a UID transition or reset UID transitions. Only a single transition
from a source UID can be defined. Transitions to root are not allowed.

Usage:

```
setuid+0:65534      # Define a UID transition from root to nobody.
setuid+root:nobody  # Same as above but using user names.
setuid-0:65534      # Remove a previously defined UID transition.
setuid^0            # Remove all UID transitions matching source UID.
setuid^             # Remove all UID transitions.
```

## setgid

[[ type
:[ *[(gid, gid)]*

Add, remove a GID transition or reset GID transitions. Only a single transition
from a source GID can be defined. Transitions to root are not allowed.

Usage:

```
setgid+0:65534      # Define a GID transition from root to nogroup.
setgid+root:nogroup # Same as above but using group names.
setgid-0:65534      # Remove a previously defined GID transition.
setgid^0            # Remove all GID transitions matching source GID.
setgid^             # Remove all GID transitions.
```

## trace/allow_unsafe_cbpf

[[ type
:[ *boolean*
|  static
:  *yes*

A boolean specifying whether Syd should allow additional _seccomp_(2)
cbpf filters to be installed by sandbox processes. By default, this is
denied to mitigate confused deputy problems and _errno_(3) is set
to "EINVAL", aka "Invalid argument" for compatibility reasons. On the one
hand, stacked _seccomp_(2) cbpf filters allow for incremental
confinement and therefore added hardening, on the other hand they may be
abused to install system call filters with more precedent actions than
user-notify thereby bypassing Syd's own _seccomp_(2) cbpf filters. To
quote the _seccomp_unotify_(2): "... a user-space notifier can be
bypassed if the existing filters allow the use of _seccomp_(2) or
_prctl_(2) to install a filter that returns an action value with a
higher precedence than "SECCOMP_RET_USER_NOTIF" (see _seccomp_(2))."
Setting the option "trace/allow_unsafe_prctl:1" overrides this option
and allows the "PR_SET_SECCOMP" _prctl_(2) operation inside the sandbox.
This may be changed in the future for clearer separation of mitigations.

## trace/allow_unsafe_ebpf

[[ type
:[ *boolean*
|  static
:  *yes*

Allows direct eBPF use inside the Syd sandbox using the _bpf_(2) system
call, whose unprivileged use is permitted since Linux-4.4. On the one
hand, eBPF programs can be used for additional hardening, on the other
hand eBPF is a frequent source of vulnerabilities due to churn,
complexity, improper validation and complexity of validation. eBPF may
also be abused to implement efficient and portable rootkits.

Note, as of version 3.32.6, Syd does not drop the capability "CAP_BPF"
when this option is set at startup. Therefore, this option allows the
privileged use of eBPF inside the Syd sandbox in case the host Linux
kernel allows the sandbox process, e.g. when running Syd as root.
Consult the _bpf_(2) and _capabilities_(7) manual pages for more
information about this Linux capability which is implemented in
Linux-5.8 or newer.

Note, as of version 3.32.8, Syd includes the _uretprobe_(2) system call
into this mitigation. _uretprobe_(2) system call is implemented in
Linux-6.11 or newer and is used by the kernel to execute pending return
uprobes.

## trace/allow_unsafe_dumpable

[[ type
:[ *boolean*
|  static
:  *yes*

A boolean specifying whether Syd should skip from setting its process
dumpable attribute to false. This allows core dumps for the Syd process,
and allows debugging/profiling/tracing the Syd process. You should not
set this option unless you're developing Syd.

## trace/allow_unsafe_exec

[[ type
:[ *boolean*
|  static
:  *yes*

A boolean specifying whether exec calls with NULL argument and environment
pointers should be allowed.

## trace/allow_unsafe_ptrace

[[ type
:[ *boolean*
|  static
:  *yes*

A boolean specifying whether _ptrace_(2) should be used to secure the
exec handler. Setting this option to true effectively removes the
_ptrace_(2) dependency from the sandbox. This is necessary to trace syd
together with its children, e.g. with "strace -f". *Warning*, this
option makes _syd_(1) keep the "CAP_SYS_PTRACE" capability and disables
Force Sandboxing, SegvGuard and the exec-TOCTOU mitigator. It allows the
sandbox process to trivially break out of the sandbox by e.g. attaching
to the _syd_(1) main thread with _ptrace_(2) and getting a handle to the
_seccomp_(2) notify file descriptor. Therefore, *this option should
only be used in trusted environments.*

## trace/allow_unsafe_perf

[[ type
:[ *boolean*
|  static
:  *yes*

A boolean specifying whether perf calls should be allowed within the sandbox.

## trace/allow_unsafe_filename

[[ type
:[ *boolean*

A boolean specifying whether the restrictions on file names should be
lifted. By default, file names with control characters, forbidden
characters or invalid UTF-8 are denied with "EINVAL" as necessary.
Read "Enhanced Path Integrity Measures" of the _syd_(7) manual page for
more information.

## trace/allow_unsafe_libc

[[ type
:[ *boolean*
|  static
:  *yes*

A boolean specifying whether turning on secure mode for libc should be
skipped. Read "AT_SECURE" section of the _getauxval_(3) manual page
about the implications of secure mode.

## trace/allow_unsafe_magiclinks

[[ type
:[ *boolean*

A boolean specifying whether /proc magic links should be followed even when
per-process directory id differs from the caller process id.  Magic links are
symbolic link-like objects that are most notably found in _proc_(5); examples
include "/proc/pid/exe" and "/proc/pid/fd/\*".  See _symlink_(7) for more
details. Unknowingly opening magic links can be risky for some applications.
Examples of such risks include the following:

- If  the  process  opening a pathname is a controlling process that
  currently has no controlling terminal (see _credentials_(7)), then
  opening a magic link inside "/proc/pid/fd" that happens to refer to a
  terminal would cause the process to acquire a controlling terminal.
- In a containerized environment, a magic link inside "/proc" may refer to
  an object outside the container, and thus may provide a means to escape
  from the container.

Because of such risks, Syd denies access to magic links which do not
belong to the current process by default.

## trace/allow_unsafe_nice

[[ type
:[ *boolean*
|  static
:  *yes*

A boolean specifying whether process and I/O priority changes are
allowed for the sandbox. See the "Process Priority and Resource
Management" section of the _syd_(7) manual page for more information.

## trace/allow_unsafe_nopie

[[ type
:[ *boolean*

A boolean specifying whether execution of non-PIE binaries should be
allowed. This is generally not recommended but may be necessary on some
systems.

## trace/allow_unsafe_chown

[[ type
:[ *boolean*
|  static
:  *yes*

Makes Syd keep the capability "CAP_CHOWN" and sandbox process will
inherit the capability from Syd.

## trace/allow_unsafe_chroot

[[ type
:[ *boolean*
|  static
:  *yes*

Disable Chroot sandboxing and turn _chroot_(2) system call into a no-op
like the _pivot_root_(2) system call. See the explanation in "chroot"
category in "SANDBOXING" section of the _syd_(7) manual page for more
information.

## trace/allow_unsafe_open_path

[[ type
:[ *boolean*

A boolean specifying whether the mitigation to turn "O_PATH" file
descriptors into "O_RDONLY" file descriptors for safe emulation should
be disabled. With this option, syd continues the _open_(2) system calls
with the "O_PATH" in the sandbox process which opens a TOCTOU vector.

## trace/allow_unsafe_open_cdev

[[ type
:[ *boolean*

A boolean specifying whether _open_(2) calls to character devices should
be continued in the sandbox process rather than opening them in the Syd
emulator thread and sending the file descriptor. Some character devices,
e.g. "/dev/kfd" for AMD GPUs, require per-application access to the GPU
device, therefore opening the device in the Syd emulator thread and then
continuing the subsequent _ioctl_(2) system calls in the sandbox process
is going to return "EBADF", aka "Bad file number". Until Syd has a way
to fully emulate the _ioctl_(2) request space and is able to call the
_ioctl_(2) system call directly from Syd emulator threads, this option
may be used to access such character devices. *Note, setting this option
opens a TOCTOU attack vector, whereby the sandbox process can open an
arbitrary file instead of the character device in question!* Syd applies
the following mitigations to limit the scope of the attack vector:

- Syd *continues* the system call if and only if "O_RDWR" is set in the flags
  argument.
- Syd does not *continue* the system call if at least one of the flags
  "O_CREAT", "O_TRUNC" or "O_TMPFILE" is set in the flags argument.
- Syd returns "ENOSYS", aka "Function not implemented", for the
  _openat2_(2) system call rather than *continuing* it in the sandbox
  process to prevent the "struct open_how" pointer indirection to bypass
  the restrictions applied to the flags argument. Refer to the
  _openat2_(2) manual page for more information.
- This option may be changed at runtime, and it is highly recommended to
  unset this option using the _syd_(2) virtual system call API right after
  the character device is opened.

## trace/allow_unsafe_cpu

[[ type
:[ *boolean*
|  static
:  *yes*

Specify whether CPU emulation system calls should be allowed. By
default, as of version 3.22.1, Syd denies the _modify_ldt_(2),
_subpage_prot_(2), _switch_endian_(2), _vm86_(2), and _vm86old_(2)
system calls, which are associated with CPU emulation functionalities.
Enabling this option (trace/allow_unsafe_cpu:1) permits these calls,
thus relaxing the restriction. This option should be used with caution,
as allowing these system calls can introduce potential vulnerabilities
by enabling processes to modify CPU state or memory protections. Use
this setting only in trusted environments where the execution of these
system calls is necessary.

## trace/allow_unsafe_keyring

[[ type
:[ *boolean*
|  static
:  *yes*

Specify whether the _add_key_(2), _keyctl_(2), and _request_key_(2)
system calls should be allowed. Enabling this setting permits key
management within the sandbox, which can introduce security risks by
allowing keyring manipulations. Use only in trusted environments.

## trace/allow_unsafe_pkey

[[ type
:[ *boolean*
|  static
:  *yes*

Specifies whether the _pkey_alloc_(2), _pkey_free_(2), and
_pkey_mprotect_(2) system calls should be allowed. By default, these
calls are denied to enhance security. Setting this option to true
enables these system calls, allowing the use of memory protection keys.
This option should be used with caution and only in trusted environments
where the use of these system calls is necessary.

## trace/allow_unsafe_msgsnd

[[ type
:[ *boolean*
|  static
:  *yes*

Specifies whether the _msgsnd_(2) system call should be allowed. By
default, this call is denied to enhance security as the ability of this
system call to allocate large, contiguous blocks of memory in the kernel
heap is often used to orchestrate kernel heap spraying attacks. See the
"Mitigation Against Heap Spraying" section of the _syd_(7) manual page
for more information.

## trace/allow_unsafe_time

[[ type
:[ *boolean*
|  static
:  *yes*

A boolean specifying whether system calls which adjust the system time are
allowed. Note, this also causes Syd to keep the *CAP_SYS_TIME* capability.
Use *syd-ls time* to see the list of system calls allowed by this setting.

## trace/allow_unsafe_uring

[[ type
:[ *boolean*
|  static
:  *yes*

A boolean specifying whether system calls of the _io_uring_(7) interface are
allowed. Normally, these are denied because they may be used to bypass path
sandboxing. Use *syd-ls uring* to see the list of system calls allowed by
this setting.

## trace/allow_unsafe_caps

[[ type
:[ boolean
|  static
:  yes

A boolean specifying whether Syd should skip dropping Linux capabilities
at startup. This setting can be used to construct _privileged
containers_ and should be used with extreme care.

Note, _syd-oci_(1) sets this option to honour the list of capabilities
specified by the container engine. You may unset it using the container
configuration file. See CONFIGURATION section in _syd-oci_(1) manual
page for more information.

## trace/allow_unsafe_env

[[ type
:[ *boolean*
|  static
:  *yes*

Specify whether unsafe environment variables should be allowed into the
environment of the sandbox process. See *syd-ls env* for the list of unsafe
environment variables.

## trace/allow_safe_kcapi

[[ type
:[ *boolean*

Specify whether access to the Linux kernel cryptography API (aka:
"KCAPI") should be allowed when network sandboxing is on. This option
has no effect when network sandboxing is off.

As most things in life, cryptography has good and evil uses: KCAPI is
convenient as it may be used to implement cryptography without depending
on user-space libraries such as OpenSSL but it may also enable malicious
code to efficiently turn itself into ransomware. Adhering to the goal to
be secure by default Syd disallows this access by default.

Note, Syd does not hook into _setsockopt_(2) and the "ALG_SET_KEY"
operation to set the encryption key is directly handled by the host
kernel therefore the encryption key is not copied into Syd's address
space.

Note again, Syd hooks into _bind_(2), _sendto_(2), _sendmsg_(2), and
_sendmmsg_(2) but *not* _read_(2), _write_(2), _recv_(2), or
_splice_(2). To reduce syscall overhead, user is recommended to use the
unhooked system calls when they can to interact with KCAPI.

## trace/allow_safe_syslog

[[ type
:[ *boolean*
|  static
:  *yes*

Specify whether unprivileged sandbox processes can access Syd's
_syslog_(2) emulation using _dmesg_(8). Unprivileged processes include
the set of _all_ sandbox processes with the sandbox lock "off", and _all
but the initial_ sandbox process with the sandbox lock set to "exec".
Note, this option has nothing to do with access to the host syslog
which is never allowed.

## trace/allow_safe_bind

[[ type
:[ *boolean*
|  static
:  *yes*

Specify whether the socket address arguments of successful _bind_(2)
calls should be allowed for _connect_(2), _sendto_(2), _sendmsg_(2), and
_sendmmsg_(2) system calls.

*Note*, these addresses are allowed globally and not per-process for
usability reasons. Thus, for example, a process which forks to call
_bind_(2) will have its address allowed for their parent as well.

## trace/allow_unsafe_bind

[[ type
:[ *boolean*
|  static
:  *yes*

Specify whether the Linux capability "CAP_NET_BIND_SERVICE", which allows a
process to _bind_(2) to ports lower than 1024, should be retained.

## trace/allow_unsafe_socket

[[ type
:[ *boolean*
|  static
:  *yes*

Specify whether unsafe socket families should be allowed. When set, Syd does
not drop the capability _CAP_NET_RAW_ on startup. This allows:

. use of RAW and PACKET sockets;
. bind to any address for transparent proxying.

## trace/allow_unsupp_socket

[[ type
:[ *boolean*

Specify whether unsupported socket families such as netlink sockets
should be allowed access when network sandboxing is on. By default Syd
allows sandboxed access to unix, ipv4 and ipv6 sockets. This option has
no effect when network sandboxing is off.

As of version 3.16.6 Syd allows access to alg sockets with the
"trace/allow_safe_kcapi" option rather than with this option. Alg
sockets are used to interact with the Linux kernel cryptography API.

Note, on architectures with a multiplexed _socketcall_(2) system call,
enabling this option is insecure because it is vulnerable to TOCTOU.
You may use _syd-sys_(1) utility to check if this system call is
available on your architecture using e.g: "syd-sys socketcall && echo
vulnerable".

## trace/allow_unsafe_prctl

[[ type
:[ *boolean*
|  static
:  *yes*

Specify whether _prctl_(2) restrictions should be lifted. See *syd-ls prctl* for
the list of allowed prctl requests.

## trace/allow_unsafe_prlimit

[[ type
:[ *boolean*
|  static
:  *yes*

Specify whether _prlimit_(2) restrictions should be lifted.

## trace/allow_unsafe_sysinfo

[[ type
:[ *boolean*
|  static
:  *yes*

Specify whether the _sysinfo_(2) randomizer should be disabled at startup.
If this option is set at startup the _sysinfo_(2) system call becomes
allowed and provides identical info to the files "/proc/loadavg" and
"/proc/meminfo" which are disabled by default by common profiles such
as the "linux" and "user" profiles. Notably this mitigation is unset
for the "paludis" profile because leaking this side-channel is
irrelevant for package builds.

## trace/allow_unsafe_syslog

[[ type
:[ *boolean*
|  static
:  *yes*

Specify whether the Linux capability "CAP_SYSLOG" should be retained. This
allows the process to perform privileged _syslog_(2) operations. This is
useful when sandboxing a service such as syslogd.

## trace/allow_unsafe_sync

[[ type
:[ *boolean*
|  static
:  *yes*

Specify whether the _sync_(2) and _syncfs_(2) system calls should be
allowed inside the sandbox. By default these system calls are turned
into no-ops to prevent potential local DoS, however it may be useful to
disable this restriction in scenarios where sync is actually expected to
work such as when sandboxing databases.

## trace/allow_unsafe_memfd

[[ type
:[ *boolean*

A boolean specifying whether secret memory file descriptors and
executable memory file descriptors should be enabled. By default Syd
strips the "MFD_EXEC" and adds the "MFD_NOEXEC_SEAL" flag to
_memfd_create_(2) flags argument. This ensures the memory file
descriptor can never be made executable. The "MFD_NOEXEC_SEAL" flag
requires Linux-6.3 or newer therefore on older kernels this option must
be enabled to make memory file descriptors work. However, the user
should be aware that allowing encrypted memory file descriptors does
allow an attacker to bypass Exec, Force and TPE sandboxing and execute
denylisted code.

## trace/allow_unsafe_memory

[[ type
:[ *boolean*
|  static
:  *yes*

Specify whether the Memory-Deny-Write-Execute (MDWE) protections should be
bypassed. See *Memory-Deny-Write-Execute Protections* section of the _syd_(7)
manual page for more information.

## trace/deny_dotdot

[[ type
:[ *boolean*

Specify whether ".." components should be denied during path resolution for
_chdir_(2) and _open_(2) family system calls. This is useful in mitigating path
traversal attacks. See "Path Resolution Restriction For Chdir and Open Calls" of
the _syd_(7) manual page for more information.

## trace/deny_elf32

[[ type
:[ *boolean*

Deny the execution of 32-bit ELF binaries.

## trace/deny_elf_dynamic

[[ type
:[ *boolean*

Deny the execution of dynamically linked ELF binaries.

## trace/deny_elf_static

[[ type
:[ *boolean*

Deny the execution of statically linked ELF binaries.

## trace/deny_script

[[ type
:[ *boolean*

Deny the execution of scripts (files with #!<interpreter> on first line).

Note, the _execve_(2) TOCTOU mitigations do not cover this option which
means the functionality is vulnerable to TOCTOU. This allows an attacker
to execute a script whose path is denylisted. This TOCTOU is limited to
scripts and requires the interpreter binary to be allowlisted for exec.
Hence this vulnerability does not allow an attacker to execute
denylisted binaries. This is why the user is recommended to deny the
respective interpreter binaries for execution instead for a safe and
secure approach.

## trace/deny_tsc

[[ type
:[ *boolean*
|  static
:  *yes*

Specify whether reading the timestamp counter should be denied. Without an
accurate timer, many timing attacks are going to be harder to perform.

- This works on *x86 only*.
- This breaks time related calls in the *vDSO*, which can be trivially worked \
around by writing a *LD_PRELOAD* library to call the respective system calls \
directly. See *libsydtime*, https://lib.rs/libsydtime, for a reference \
implementation.
- This has a negative performance impact on programs that rely on \
_gettimeofday_(2) being a *vDSO* call.

## trace/lock_personality

[[ type
:[ *boolean*
|  static
:  *yes*

If set, locks down the _personality_(2) system call so that the kernel execution
domain may not be changed from the default. This may be useful to improve
security, because odd personality emulations may be poorly tested and source of
vulnerabilities.

## trace/exit_wait_all

[[ type
:[ *boolean*
|  static
:  *yes*

Specify whether Syd should wait for all processes to exit before exiting. By
default, Syd exits with the eldest process and any leftover processes in the
background are automatically killed.

## trace/force_umask

[[ type
:[ *octal*
|  static
:  *yes*

Specify an umask mode to force. To unset a previously configured force
umask use -1 as the value. As of version 3.15.6, _chmod_(2) family
system calls also honour force umask for added hardening. As of version
3.22.1, this setting does not apply to directory creation for _mkdir_(2)
and _mkdirat_(2) system calls. As of version 3.26.2, this setting does
not apply to UNIX domain socket creation for _bind_(2) system calls, and
non-regular file creation for _mknod_(2) and _mknodat_(2) system calls.

## trace/memory_access

[[ type
:[ *integer*
|  default
:  *2*
|  static
:  *yes*

Set mode on cross memory attach and _proc_pid_mem_(5) usage. Cross
memory attach is done using the system calls _process_vm_readv_(2) and
_process_vm_writev_(2) which requires a Linux kernel configured with the
*CONFIG_CROSS_MEMORY_ATTACH* option enabled. Supported modes are:

- *0*: Use cross memory attach if available, use _proc_pid_mem_(5) otherwise.
- *1*: Use _/proc/pid/mem_(5) unconditionally.
- *2*: Use cross memory attach unconditionally.

From a security point of view, these two modes of access have an
important distinction where cross memory attach honours page protections
of the target process, however using _/proc/pid/mem_(5) does not. This
makes direct _proc_pid_mem_(5) access dangerous in that a Syd deputy
process may be confused into corrupting or even controlling memory
regions the sandbox process otherwise does not have direct access to.
This is the main reason why mode *2* has been added as of version 3.32.6
as a secure default alternative to the previous default mode *0* whose
fallback behaviour can be unpredictable and is against the idea of
secure defaults. Therefore as of version 3.32.6, the user is asked
to change the memory access mode explicitly if their Linux kernel
is not configured with the *CONFIG_CROSS_MEMORY_ATTACH* option.
You may also use the environment variables *SYD_NO_CROSS_MEMORY_ATTACH*
and *SYD_PROC_PID_MEM_FALLBACK*, see the "ENVIRONMENT" section of the
_syd_(1) manual page for more information. For further information
about the security impact of _proc_pid_mem_(5) writes refer to the
following links:

- https://lore.kernel.org/lkml/202403011451.C236A38@keescook/T/
- https://lwn.net/Articles/476947/
- https://issues.chromium.org/issues/40089045

```
; strace -q -eprocess_vm_readv -fc -- syd -poff -pD -mtrace/memory_access:0 true
% time     seconds  usecs/call     calls    errors syscall
------ ----------- ----------- --------- --------- -----------------
100.00    0.000031          10         3           process_vm_readv
------ ----------- ----------- --------- --------- -----------------
100.00    0.000031          10         3           total
; strace -q -eprocess_vm_readv -fc -- syd -poff -pD -mtrace/memory_access:1 true
; strace -q -eprocess_vm_readv -fc -- syd -poff -pD -mtrace/memory_access:2 true
% time     seconds  usecs/call     calls    errors syscall
------ ----------- ----------- --------- --------- -----------------
100.00    0.000008           2         3           process_vm_readv
------ ----------- ----------- --------- --------- -----------------
100.00    0.000008           2         3           total
```

## trace/sync_seccomp

[[ type
:[ *boolean*
|  static
:  *yes*

Use synchronous mode for seccomp-notify so each Syd syscall handler thread wakes
up on the same CPU as the respective sandbox thread that executed the system
call. This option makes no functional difference and it may or may not help with
performance; it defaults to false unless *SYD_SYNC_SCMP* environment variable is
set at startup. This is because in our benchmarks setting this option typically
hurts performance which may be attributed to Syd's multithreaded nature. Use
_perf_(1) to benchmark seccomp synchronous mode:

```
; perf bench sched seccomp-notify
# Running 'sched/seccomp-notify' benchmark:
# Executed 1000000 system calls
Total time: 6.736 [sec]
6.736395 usecs/op
148447 ops/sec
; perf bench sched seccomp-notify --sync-mode
# Running 'sched/seccomp-notify' benchmark:
# Executed 1000000 system calls
Total time: 4.188 [sec]
4.188846 usecs/op
238729 ops/sec
```

## cache/capacity/path

[[ type
:[ *usize*
|  default
:  *4096*
|  static
:  *yes*

Estimated items capacity of the security policy cache for paths.

## cache/capacity/addr

[[ type
:[ *usize*
|  default
:  *1024*
|  static
:  *yes*

Estimated items capacity of the security policy cache for IP addresses.

# SHORTCUTS

*read*, *stat*, *write*, and *exec* may be passed to *allow*, *deny*, and
*filter* either as a single unit or as a comma-delimited list, e.g:

```
allow/read,write,stat,exec+/***
allow/read,write,stat-/***
deny/read,write,stat+/***
deny/read,write-/***
filter/read,write,stat+/dev/mem
filter/read,write-/dev/mem
```

Since 3.18.14, sandboxing modes may be specified as a single unit or as a
comma-delimited list, e.g:

```
sandbox/read,write,stat,exec:on
sandbox/net,lock:off
```

Since 3.19.0, namespace types may be specified as a single unit or as a
comma-delimited list, e.g.:

```
unshare/user,pid,mount:on
unshare/net,cgroup:off
```

# PATTERN MATCHING

Syd uses shell-style pattern matching for allowlists and filters. The
matching code uses the *globset* crate. Check their documentation for more
information about patterns. Note, patterns are _case sensitive_, the *empty
alternates* option is set when building patterns, i.e. *foo{,.txt}* in a pattern
will match both *foo* and *foo.txt* and Syd applies the *triple star*
extension to patterns, i.e. */dev/\*\*\** matches both */dev* and any file
recursively under */dev*. Note also, Syd gets patterns from multiple sources:
a configuration file, a profile, the *-m* command line switch, or a *stat* call
with */dev/syd* prefix. There is no precedence between different sources. All
patterns in a list are compiled together in a *GlobSet* and pattern matching
during access control happens in a single step where _the last matching pattern
decides the outcome._

# ADDRESS MATCHING

Syd has a simple address scheme to match network addresses. The addresses can
either be a *glob* pattern to match *UNIX* and *abstract UNIX* socket addresses,
or *IP CIDR* followed by a port range to match *IPv4* and *IPv6* addresses. Port
range can either be a single port or a closed range in format *port1-port2*. The
address and the port range must be split by the character *!*.  The precedence
logic is same as *Pattern Matching* where _the last matching pattern decides the
outcome._

In addition there are some *aliases*, you may use instead of specifying an address:

- *any*: Expanded to *any4* + *any6*.
- *any4*: Expanded to *0.0.0.0/0* which matches the whole Ipv4 address space.
- *any6*: Expanded to *::/0* which matches the whole Ipv6 address space.
- *loopback*: Expanded to *loopback4* + *loopback6*.
- *loopback4*: Expanded to *127.0.0.0/8*
- *loopback6*: Expanded to *::1/128*
- *linklocal*: Expanded to *linklocal4* + *linklocal6*.
- *linklocal4*: Expanded to *169.254.0.0/16*
- *linklocal6*: Expanded to *fe80::/10*
- *local*: Expanded to *local4* + *local6*.
- *local4*: Expanded to four addresses as defined in *RFC1918*:
	- *127.0.0.0/8*
	- *10.0.0.0/8*
	- *172.16.0.0/12*
	- *192.168.0.0/16*
- *local6*: Expanded to four addresses:
	- *::1/128*
	- *fe80::/7*
	- *fc00::/7*
	- *fec0::/7*

# SECURITY

The interface is only available if the sandbox lock is not set for the calling
process. Similarly, command-line option parsing and configuration file parsing
stops once a *lock:on* clause is executed.

# RETURN VALUE

For _stat_(2) calls, on success, zero is returned. On error, -1 is returned, and
errno is set to indicate the error.

For _open_(2) calls, on success the new file descriptor (a nonnegative integer)
is returned. On error, -1 is returned and errno is set to indicate the error.

# ERRORS

|[ *EBUSY*
:< Attempted to edit a setting at runtime that must be configured at startup

|[ *EEXIST*
:< Attempted to add an UID/GID transition for SafeSetID but a transition with the same source UID/GID exists.

|[ *EINVAL*
:< Sandbox command is syntactically incorrect.

|[ *ENOENT*
:< Result of the given sandbox query is false (e.g. "test -c /dev/syd/sandbox/stat?").

|[ *ENOENT*
:< Sandbox lock is on, no commands are allowed.

|[ *ENOSYS*
:< elf feature is disabled and command is one of "trace/deny_elf32", "trace/deny\_dynamic", or "trace/deny\_static".

# SEE ALSO

_syd_(1), _syd_(5), _syd_(7), _syd-ls_(1) _open_(2), _stat_(2),
_perf_(1), _pledge_(2), _ptrace_(2), _seccomp_(2), _strace_(1),
_glob_(3p), _io_uring_(7), _gdb_(1), _valgrind_(1), _wordexp_(3)

- *syd* homepage: https://sydbox.exherbolinux.org
- *libsyd* homepage: https://libsyd.exherbolinux.org
- *gosyd* homepage: https://gosyd.exherbolinux.org
- *plsyd* homepage:  https://plsyd.exherbolinux.org
- *pysyd* homepage: https://pysyd.exherbolinux.org
- *rbsyd* homepage: https://rbsyd.exherbolinux.org
- *syd.el* homepage: https://sydel.exherbolinux.org
- *libsydtime* homepage: https://lib.rs/libsydtime
- *LandLock* homepage: https://landlock.io/
- *vDSO* wiki: https://en.wikipedia.org/wiki/VDSO
- *parse-size* documentation: https://docs.rs/parse-size/1.0.0/parse_size/
- *globset* documentation: https://docs.rs/globset
- *GlobSet* documentation: https://docs.rs/globset/latest/globset/struct.GlobSet.html
- GlobBuilder *empty_alternates* method: https://docs.rs/globset/latest/globset/struct.GlobBuilder.html#method.empty_alternates
- *ipnetwork* documentation: https://docs.rs/ipnetwork

# AUTHORS

Maintained by Ali Polatel. Up-to-date sources can be found at
https://gitlab.exherbo.org/sydbox/sydbox.git and bugs/patches can be
submitted to https://gitlab.exherbo.org/groups/sydbox/-/issues. Discuss
in #sydbox on Libera Chat or in #sydbox:mailstation.de on Matrix.
