//
// libsyd: Rust-based C library for syd interaction via /dev/syd
// lib/src/lib.rs: syd API C Library
//
// Copyright (c) 2023, 2024, 2025 Ali Polatel <alip@chesswob.org>
//
// SPDX-License-Identifier: LGPL-3.0

//! # libsyd - syd API Rust Library
//!
//! `libsyd` is a C library written in Rust that implements the syd
//! stat API, providing an interface to the `/dev/syd` of syd. It
//! allows for runtime configuration and interaction with the syd
//! sandboxing environment.
//!
//! ## Overview
//! The library is designed to interact with the syd sandboxing
//! environment, offering functionalities to check and modify the state
//! of the sandbox lock, and perform system calls to `/dev/syd`.
//!
//! For more detailed information and usage instructions, refer to the syd
//! manual, available at [syd Manual](http://man.exherbolinux.org/syd.2.html).
//!
//! ## Author
//! Ali Polatel <alip@chesswob.org>

// We like safe, clean and simple code with documentation.
#![deny(missing_docs)]
#![deny(clippy::allow_attributes_without_reason)]
#![deny(clippy::arithmetic_side_effects)]
#![deny(clippy::as_ptr_cast_mut)]
#![deny(clippy::as_underscore)]
#![deny(clippy::assertions_on_result_states)]
#![deny(clippy::borrow_as_ptr)]
#![deny(clippy::branches_sharing_code)]
#![deny(clippy::case_sensitive_file_extension_comparisons)]
#![deny(clippy::cast_lossless)]
#![deny(clippy::cast_possible_truncation)]
#![deny(clippy::cast_possible_wrap)]
#![deny(clippy::cast_precision_loss)]
#![deny(clippy::cast_ptr_alignment)]
#![deny(clippy::cast_sign_loss)]
#![deny(clippy::checked_conversions)]
#![deny(clippy::clear_with_drain)]
#![deny(clippy::clone_on_ref_ptr)]
#![deny(clippy::cloned_instead_of_copied)]
#![deny(clippy::cognitive_complexity)]
#![deny(clippy::collection_is_never_read)]
#![deny(clippy::copy_iterator)]
#![deny(clippy::create_dir)]
#![deny(clippy::dbg_macro)]
#![deny(clippy::debug_assert_with_mut_call)]
#![deny(clippy::decimal_literal_representation)]
#![deny(clippy::default_trait_access)]
#![deny(clippy::default_union_representation)]
#![deny(clippy::derive_partial_eq_without_eq)]
#![deny(clippy::doc_link_with_quotes)]
#![deny(clippy::doc_markdown)]
#![deny(clippy::explicit_into_iter_loop)]
#![deny(clippy::explicit_iter_loop)]
#![deny(clippy::fallible_impl_from)]
#![deny(clippy::missing_safety_doc)]
#![deny(clippy::undocumented_unsafe_blocks)]

use std::{
    ffi::{CStr, OsStr, OsString},
    fmt,
    fs::{symlink_metadata, Metadata},
    os::{
        fd::RawFd,
        raw::{c_char, c_int},
        unix::{
            ffi::OsStrExt,
            fs::{FileTypeExt, MetadataExt},
        },
    },
    path::{Path, PathBuf},
};

/// `lock_state_t_t` type represents possible states for the sandbox lock.
#[allow(non_camel_case_types)]
pub type lock_state_t = u8;

/// The sandbox lock is off, allowing all sandbox commands.
pub const LOCK_OFF: lock_state_t = 0;
/// The sandbox lock is set to on for all processes except the initial
/// process (syd exec child).
pub const LOCK_EXEC: lock_state_t = 1;
/// The sandbox lock is on, disallowing all sandbox commands.
pub const LOCK_ON: lock_state_t = 2;

// An enumeration of the possible states for the sandbox lock.
#[repr(u8)]
#[derive(Copy, Clone, Debug)]
enum LockState {
    // The sandbox lock is off, allowing all sandbox commands.
    Off = LOCK_OFF,
    // The sandbox lock is set to on for all processes except the initial
    // process (syd exec child).
    Exec = LOCK_EXEC,
    // The sandbox lock is on, disallowing all sandbox commands.
    On = LOCK_ON,
}

impl TryFrom<lock_state_t> for LockState {
    type Error = ();

    fn try_from(value: lock_state_t) -> Result<Self, Self::Error> {
        match value {
            LOCK_OFF => Ok(LockState::Off),
            LOCK_EXEC => Ok(LockState::Exec),
            LOCK_ON => Ok(LockState::On),
            _ => Err(()),
        }
    }
}

impl fmt::Display for LockState {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        let state_str = match self {
            LockState::Off => "off",
            LockState::Exec => "exec",
            LockState::On => "on",
        };
        write!(f, "{state_str}")
    }
}

/// `action_t` type represents possible sandboxing action values.
#[allow(non_camel_case_types)]
pub type action_t = u8;

/// Allow system call.
pub const ACTION_ALLOW: action_t = 0;
/// Allow system call and warn.
pub const ACTION_WARN: action_t = 1;
/// Deny system call silently.
pub const ACTION_FILTER: action_t = 2;
/// Deny system call and warn.
pub const ACTION_DENY: action_t = 3;
/// Deny system call, warn and panic the current Syd thread.
pub const ACTION_PANIC: action_t = 4;
/// Deny system call, warn and stop the offending process.
pub const ACTION_STOP: action_t = 5;
/// Deny system call, warn and abort the offending process.
pub const ACTION_ABORT: action_t = 6;
/// Deny system call, warn and kill the offending process.
pub const ACTION_KILL: action_t = 7;
/// Warn, and exit Syd immediately with deny errno as exit value.
pub const ACTION_EXIT: action_t = 8;

// An enumeration of the possible actions for sandboxing.
#[repr(u8)]
#[derive(Copy, Clone, Debug)]
enum Action {
    // Allow system call.
    Allow = ACTION_ALLOW,
    // Allow system call and warn.
    Warn = ACTION_WARN,
    // Deny system call silently.
    Filter = ACTION_FILTER,
    // Deny system call and warn.
    Deny = ACTION_DENY,
    // Deny system call, warn and panic the current Syd thread.
    Panic = ACTION_PANIC,
    // Deny system call, warn and stop the offending process.
    Stop = ACTION_STOP,
    // Deny system call, warn and abort offending process.
    Abort = ACTION_ABORT,
    // Deny system call, warn and kill the offending process.
    Kill = ACTION_KILL,
    // Warn, and exit Syd immediately with deny errno as exit value.
    Exit = ACTION_EXIT,
}

impl TryFrom<action_t> for Action {
    type Error = ();

    fn try_from(value: action_t) -> Result<Self, Self::Error> {
        match value {
            ACTION_ALLOW => Ok(Action::Allow),
            ACTION_WARN => Ok(Action::Warn),
            ACTION_FILTER => Ok(Action::Filter),
            ACTION_DENY => Ok(Action::Deny),
            ACTION_PANIC => Ok(Action::Panic),
            ACTION_STOP => Ok(Action::Stop),
            ACTION_ABORT => Ok(Action::Abort),
            ACTION_KILL => Ok(Action::Kill),
            ACTION_EXIT => Ok(Action::Exit),
            _ => Err(()),
        }
    }
}

impl fmt::Display for Action {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        let action_str = match self {
            Action::Allow => "allow",
            Action::Warn => "warn",
            Action::Filter => "filter",
            Action::Deny => "deny",
            Action::Panic => "panic",
            Action::Stop => "stop",
            Action::Abort => "abort",
            Action::Kill => "kill",
            Action::Exit => "exit",
        };
        write!(f, "{action_str}")
    }
}

const EFAULT: i32 = 14;
const EINVAL: i32 = 22;

#[inline(always)]
fn check_stat(stat: &Metadata) -> bool {
    if !stat.file_type().is_char_device() {
        return false;
    }

    let rdev = stat.rdev();

    let major = (rdev >> 8) & 0xff;
    let minor = rdev & 0xff;

    // dev/null
    major == 1 && minor == 3
}

fn stat<P: AsRef<Path>>(path: P) -> c_int {
    match symlink_metadata(path) {
        Ok(stat) if check_stat(&stat) => 0,
        Ok(_) => -EINVAL,
        Err(error) => match error.raw_os_error() {
            Some(e) => e.checked_neg().unwrap_or(-EINVAL),
            None => -EINVAL,
        },
    }
}

fn esyd<P: AsRef<Path>>(rule: P, elem: *const c_char, op: u8) -> c_int {
    if !matches!(op, b'+' | b'-' | b'^' | b':') {
        return -EINVAL;
    }

    if elem.is_null() {
        return -EFAULT;
    }

    // SAFETY: Trust that `elem` is a null-terminated string.
    let elem = unsafe { CStr::from_ptr(elem) };
    let elem = OsStr::from_bytes(elem.to_bytes());

    // Manually concatenate the path segments
    let mut path = OsString::from("/dev/syd/");
    path.push(rule.as_ref());
    path.push(OsStr::from_bytes(&[op]));
    path.push(elem);

    // Convert the OsString to PathBuf
    let path = PathBuf::from(path);

    stat(path)
}

/// Performs a syd API check
///
/// The caller is advised to perform this check before
/// calling any other syd API calls.
///
/// Returns API number on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_api() -> c_int {
    match stat("/dev/syd/3") {
        0 => 3,
        n => n,
    }
}

/// Performs an lstat system call on the file "/dev/syd".
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_check() -> c_int {
    stat("/dev/syd")
}

/// Causes syd to exit immediately with code 127
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_panic() -> c_int {
    stat("/dev/syd/panic")
}

/// Causes syd to reset sandboxing to the default state.
/// Allowlists, denylists and filters are going to be cleared.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_reset() -> c_int {
    stat("/dev/syd/reset")
}

/// Causes syd to read configuration from the given file descriptor.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_load(fd: c_int) -> c_int {
    let fd = match RawFd::try_from(fd) {
        Ok(fd) if fd < 0 => return -EINVAL,
        Ok(fd) => fd,
        Err(_) => return -EINVAL,
    };
    stat(&format!("/dev/syd/load/{fd}"))
}

/// Sets the state of the sandbox lock.
///
/// state: The desired state of the sandbox lock.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_lock(state: lock_state_t) -> c_int {
    // Convert lock_state_t enum to corresponding lock state string.
    let state = match LockState::try_from(state) {
        Ok(state) => state,
        Err(_) => return -EINVAL,
    };

    stat(&format!("/dev/syd/lock:{state}"))
}

/// Checks if stat sandboxing is enabled.
///
/// Returns true if stat sandboxing is enabled, false otherwise.
#[no_mangle]
pub extern "C" fn syd_enabled_stat() -> bool {
    stat("/dev/syd/sandbox/stat?") == 0
}

/// Enable stat sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_enable_stat() -> c_int {
    stat("/dev/syd/sandbox/stat:on")
}

/// Disable stat sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_disable_stat() -> c_int {
    stat("/dev/syd/sandbox/stat:off")
}

/// Checks if read sandboxing is enabled.
///
/// Returns true if read sandboxing is enabled, false otherwise.
#[no_mangle]
pub extern "C" fn syd_enabled_read() -> bool {
    stat("/dev/syd/sandbox/read?") == 0
}

/// Enable read sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_enable_read() -> c_int {
    stat("/dev/syd/sandbox/read:on")
}

/// Disable read sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_disable_read() -> c_int {
    stat("/dev/syd/sandbox/read:off")
}

/// Checks if write sandboxing is enabled.
///
/// Returns true if write sandboxing is enabled, false otherwise.
#[no_mangle]
pub extern "C" fn syd_enabled_write() -> bool {
    stat("/dev/syd/sandbox/write?") == 0
}

/// Enable write sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_enable_write() -> c_int {
    stat("/dev/syd/sandbox/write:on")
}

/// Disable write sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_disable_write() -> c_int {
    stat("/dev/syd/sandbox/write:off")
}

/// Checks if exec sandboxing is enabled.
///
/// Returns true if exec sandboxing is enabled, false otherwise.
#[no_mangle]
pub extern "C" fn syd_enabled_exec() -> bool {
    stat("/dev/syd/sandbox/exec?") == 0
}

/// Enable exec sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_enable_exec() -> c_int {
    stat("/dev/syd/sandbox/exec:on")
}

/// Disable exec sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_disable_exec() -> c_int {
    stat("/dev/syd/sandbox/exec:off")
}

/// Checks if ioctl sandboxing is enabled.
///
/// Returns true if ioctl sandboxing is enabled, false otherwise.
#[no_mangle]
pub extern "C" fn syd_enabled_ioctl() -> bool {
    stat("/dev/syd/sandbox/ioctl?") == 0
}

/// Enable ioctl sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_enable_ioctl() -> c_int {
    stat("/dev/syd/sandbox/ioctl:on")
}

/// Disable ioctl sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_disable_ioctl() -> c_int {
    stat("/dev/syd/sandbox/ioctl:off")
}

/// Checks if create sandboxing is enabled.
///
/// Returns true if create sandboxing is enabled, false otherwise.
#[no_mangle]
pub extern "C" fn syd_enabled_create() -> bool {
    stat("/dev/syd/sandbox/create?") == 0
}

/// Enable create sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_enable_create() -> c_int {
    stat("/dev/syd/sandbox/create:on")
}

/// Disable create sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_disable_create() -> c_int {
    stat("/dev/syd/sandbox/create:off")
}

/// Checks if delete sandboxing is enabled.
///
/// Returns true if delete sandboxing is enabled, false otherwise.
#[no_mangle]
pub extern "C" fn syd_enabled_delete() -> bool {
    stat("/dev/syd/sandbox/delete?") == 0
}

/// Enable delete sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_enable_delete() -> c_int {
    stat("/dev/syd/sandbox/delete:on")
}

/// Disable delete sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_disable_delete() -> c_int {
    stat("/dev/syd/sandbox/delete:off")
}

/// Checks if rename sandboxing is enabled.
///
/// Returns true if rename sandboxing is enabled, false otherwise.
#[no_mangle]
pub extern "C" fn syd_enabled_rename() -> bool {
    stat("/dev/syd/sandbox/rename?") == 0
}

/// Enable rename sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_enable_rename() -> c_int {
    stat("/dev/syd/sandbox/rename:on")
}

/// Disable rename sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_disable_rename() -> c_int {
    stat("/dev/syd/sandbox/rename:off")
}

/// Checks if symlink sandboxing is enabled.
///
/// Returns true if symlink sandboxing is enabled, false otherwise.
#[no_mangle]
pub extern "C" fn syd_enabled_symlink() -> bool {
    stat("/dev/syd/sandbox/symlink?") == 0
}

/// Enable symlink sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_enable_symlink() -> c_int {
    stat("/dev/syd/sandbox/symlink:on")
}

/// Disable symlink sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_disable_symlink() -> c_int {
    stat("/dev/syd/sandbox/symlink:off")
}

/// Checks if truncate sandboxing is enabled.
///
/// Returns true if truncate sandboxing is enabled, false otherwise.
#[no_mangle]
pub extern "C" fn syd_enabled_truncate() -> bool {
    stat("/dev/syd/sandbox/truncate?") == 0
}

/// Enable truncate sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_enable_truncate() -> c_int {
    stat("/dev/syd/sandbox/truncate:on")
}

/// Disable truncate sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_disable_truncate() -> c_int {
    stat("/dev/syd/sandbox/truncate:off")
}

/// Checks if chdir sandboxing is enabled.
///
/// Returns true if chdir sandboxing is enabled, false otherwise.
#[no_mangle]
pub extern "C" fn syd_enabled_chdir() -> bool {
    stat("/dev/syd/sandbox/chdir?") == 0
}

/// Enable chdir sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_enable_chdir() -> c_int {
    stat("/dev/syd/sandbox/chdir:on")
}

/// Disable chdir sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_disable_chdir() -> c_int {
    stat("/dev/syd/sandbox/chdir:off")
}

/// Checks if readdir sandboxing is enabled.
///
/// Returns true if readdir sandboxing is enabled, false otherwise.
#[no_mangle]
pub extern "C" fn syd_enabled_readdir() -> bool {
    stat("/dev/syd/sandbox/readdir?") == 0
}

/// Enable readdir sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_enable_readdir() -> c_int {
    stat("/dev/syd/sandbox/readdir:on")
}

/// Disable readdir sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_disable_readdir() -> c_int {
    stat("/dev/syd/sandbox/readdir:off")
}

/// Checks if mkdir sandboxing is enabled.
///
/// Returns true if mkdir sandboxing is enabled, false otherwise.
#[no_mangle]
pub extern "C" fn syd_enabled_mkdir() -> bool {
    stat("/dev/syd/sandbox/mkdir?") == 0
}

/// Enable mkdir sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_enable_mkdir() -> c_int {
    stat("/dev/syd/sandbox/mkdir:on")
}

/// Disable mkdir sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_disable_mkdir() -> c_int {
    stat("/dev/syd/sandbox/mkdir:off")
}

/// Checks if rmdir sandboxing is enabled.
///
/// Returns true if rmdir sandboxing is enabled, false otherwise.
#[no_mangle]
pub extern "C" fn syd_enabled_rmdir() -> bool {
    stat("/dev/syd/sandbox/rmdir?") == 0
}

/// Enable rmdir sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_enable_rmdir() -> c_int {
    stat("/dev/syd/sandbox/rmdir:on")
}

/// Disable rmdir sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_disable_rmdir() -> c_int {
    stat("/dev/syd/sandbox/rmdir:off")
}

/// Checks if chown sandboxing is enabled.
///
/// Returns true if chown sandboxing is enabled, false otherwise.
#[no_mangle]
pub extern "C" fn syd_enabled_chown() -> bool {
    stat("/dev/syd/sandbox/chown?") == 0
}

/// Enable chown sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_enable_chown() -> c_int {
    stat("/dev/syd/sandbox/chown:on")
}

/// Disable chown sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_disable_chown() -> c_int {
    stat("/dev/syd/sandbox/chown:off")
}

/// Checks if chgrp sandboxing is enabled.
///
/// Returns true if chgrp sandboxing is enabled, false otherwise.
#[no_mangle]
pub extern "C" fn syd_enabled_chgrp() -> bool {
    stat("/dev/syd/sandbox/chgrp?") == 0
}

/// Enable chgrp sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_enable_chgrp() -> c_int {
    stat("/dev/syd/sandbox/chgrp:on")
}

/// Disable chgrp sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_disable_chgrp() -> c_int {
    stat("/dev/syd/sandbox/chgrp:off")
}

/// Checks if chmod sandboxing is enabled.
///
/// Returns true if chmod sandboxing is enabled, false otherwise.
#[no_mangle]
pub extern "C" fn syd_enabled_chmod() -> bool {
    stat("/dev/syd/sandbox/chmod?") == 0
}

/// Enable chmod sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_enable_chmod() -> c_int {
    stat("/dev/syd/sandbox/chmod:on")
}

/// Disable chmod sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_disable_chmod() -> c_int {
    stat("/dev/syd/sandbox/chmod:off")
}

/// Checks if chattr sandboxing is enabled.
///
/// Returns true if chattr sandboxing is enabled, false otherwise.
#[no_mangle]
pub extern "C" fn syd_enabled_chattr() -> bool {
    stat("/dev/syd/sandbox/chattr?") == 0
}

/// Enable chattr sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_enable_chattr() -> c_int {
    stat("/dev/syd/sandbox/chattr:on")
}

/// Disable chattr sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_disable_chattr() -> c_int {
    stat("/dev/syd/sandbox/chattr:off")
}

/// Checks if chroot sandboxing is enabled.
///
/// Returns true if chroot sandboxing is enabled, false otherwise.
#[no_mangle]
pub extern "C" fn syd_enabled_chroot() -> bool {
    stat("/dev/syd/sandbox/chroot?") == 0
}

/// Enable chroot sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_enable_chroot() -> c_int {
    stat("/dev/syd/sandbox/chroot:on")
}

/// Disable chroot sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_disable_chroot() -> c_int {
    stat("/dev/syd/sandbox/chroot:off")
}

/// Checks if utime sandboxing is enabled.
///
/// Returns true if utime sandboxing is enabled, false otherwise.
#[no_mangle]
pub extern "C" fn syd_enabled_utime() -> bool {
    stat("/dev/syd/sandbox/utime?") == 0
}

/// Enable utime sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_enable_utime() -> c_int {
    stat("/dev/syd/sandbox/utime:on")
}

/// Disable utime sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_disable_utime() -> c_int {
    stat("/dev/syd/sandbox/utime:off")
}

/// Checks if mkdev sandboxing is enabled.
///
/// Returns true if mkdev sandboxing is enabled, false otherwise.
#[no_mangle]
pub extern "C" fn syd_enabled_mkdev() -> bool {
    stat("/dev/syd/sandbox/mkdev?") == 0
}

/// Enable mkdev sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_enable_mkdev() -> c_int {
    stat("/dev/syd/sandbox/mkdev:on")
}

/// Disable mkdev sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_disable_mkdev() -> c_int {
    stat("/dev/syd/sandbox/mkdev:off")
}

/// Checks if mkfifo sandboxing is enabled.
///
/// Returns true if mkfifo sandboxing is enabled, false otherwise.
#[no_mangle]
pub extern "C" fn syd_enabled_mkfifo() -> bool {
    stat("/dev/syd/sandbox/mkfifo?") == 0
}

/// Enable mkfifo sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_enable_mkfifo() -> c_int {
    stat("/dev/syd/sandbox/mkfifo:on")
}

/// Disable mkfifo sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_disable_mkfifo() -> c_int {
    stat("/dev/syd/sandbox/mkfifo:off")
}

/// Checks if mktemp sandboxing is enabled.
///
/// Returns true if mktemp sandboxing is enabled, false otherwise.
#[no_mangle]
pub extern "C" fn syd_enabled_mktemp() -> bool {
    stat("/dev/syd/sandbox/mktemp?") == 0
}

/// Enable mktemp sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_enable_mktemp() -> c_int {
    stat("/dev/syd/sandbox/mktemp:on")
}

/// Disable mktemp sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_disable_mktemp() -> c_int {
    stat("/dev/syd/sandbox/mktemp:off")
}

/// Checks if net sandboxing is enabled.
///
/// Returns true if net sandboxing is enabled, false otherwise.
#[no_mangle]
pub extern "C" fn syd_enabled_net() -> bool {
    stat("/dev/syd/sandbox/net?") == 0
}

/// Enable net sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_enable_net() -> c_int {
    stat("/dev/syd/sandbox/net:on")
}

/// Disable net sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_disable_net() -> c_int {
    stat("/dev/syd/sandbox/net:off")
}

/// Checks if memory sandboxing is enabled.
///
/// Returns true if memory sandboxing is enabled, false otherwise.
#[no_mangle]
pub extern "C" fn syd_enabled_mem() -> bool {
    stat("/dev/syd/sandbox/mem?") == 0
}

/// Enable memory sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_enable_mem() -> c_int {
    stat("/dev/syd/sandbox/mem:on")
}

/// Disable memory sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_disable_mem() -> c_int {
    stat("/dev/syd/sandbox/mem:off")
}

/// Checks if PID sandboxing is enabled.
///
/// Returns true if PID sandboxing is enabled, false otherwise.
#[no_mangle]
pub extern "C" fn syd_enabled_pid() -> bool {
    stat("/dev/syd/sandbox/pid?") == 0
}

/// Enable PID sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_enable_pid() -> c_int {
    stat("/dev/syd/sandbox/pid:on")
}

/// Disable PID sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_disable_pid() -> c_int {
    stat("/dev/syd/sandbox/pid:off")
}

/// Checks if lock sandboxing is enabled.
///
/// Returns true if lock sandboxing is enabled, false otherwise.
#[no_mangle]
pub extern "C" fn syd_enabled_lock() -> bool {
    stat("/dev/syd/sandbox/lock?") == 0
}

/// Checks if crypt sandboxing is enabled.
///
/// Returns true if crypt sandboxing is enabled, false otherwise.
#[no_mangle]
pub extern "C" fn syd_enabled_crypt() -> bool {
    stat("/dev/syd/sandbox/crypt?") == 0
}

/// Checks if proxy sandboxing is enabled.
///
/// Returns true if proxy sandboxing is enabled, false otherwise.
#[no_mangle]
pub extern "C" fn syd_enabled_proxy() -> bool {
    stat("/dev/syd/sandbox/proxy?") == 0
}

/// Checks if force sandboxing is enabled.
///
/// Returns true if force sandboxing is enabled, false otherwise.
#[no_mangle]
pub extern "C" fn syd_enabled_force() -> bool {
    stat("/dev/syd/sandbox/force?") == 0
}

/// Enable force sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_enable_force() -> c_int {
    stat("/dev/syd/sandbox/force:on")
}

/// Disable force sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_disable_force() -> c_int {
    stat("/dev/syd/sandbox/force:off")
}

/// Checks if TPE sandboxing is enabled.
///
/// Returns true if TPE sandboxing is enabled, false otherwise.
#[no_mangle]
pub extern "C" fn syd_enabled_tpe() -> bool {
    stat("/dev/syd/sandbox/tpe?") == 0
}

/// Enable TPE sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_enable_tpe() -> c_int {
    stat("/dev/syd/sandbox/tpe:on")
}

/// Disable TPE sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_disable_tpe() -> c_int {
    stat("/dev/syd/sandbox/tpe:off")
}

/// Set the default action for Stat Sandboxing.
#[no_mangle]
pub extern "C" fn syd_default_stat(action: action_t) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };
    stat(&format!("/dev/syd/default/stat:{action}"))
}

/// Set the default action for Read Sandboxing.
#[no_mangle]
pub extern "C" fn syd_default_read(action: action_t) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };
    stat(&format!("/dev/syd/default/read:{action}"))
}

/// Set the default action for Write Sandboxing.
#[no_mangle]
pub extern "C" fn syd_default_write(action: action_t) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };
    stat(&format!("/dev/syd/default/write:{action}"))
}

/// Set the default action for Exec Sandboxing.
#[no_mangle]
pub extern "C" fn syd_default_exec(action: action_t) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };
    stat(&format!("/dev/syd/default/exec:{action}"))
}

/// Set the default action for Ioctl Sandboxing.
#[no_mangle]
pub extern "C" fn syd_default_ioctl(action: action_t) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };
    stat(&format!("/dev/syd/default/ioctl:{action}"))
}

/// Set the default action for Create Sandboxing.
#[no_mangle]
pub extern "C" fn syd_default_create(action: action_t) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };
    stat(&format!("/dev/syd/default/create:{action}"))
}

/// Set the default action for Delete Sandboxing.
#[no_mangle]
pub extern "C" fn syd_default_delete(action: action_t) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };
    stat(&format!("/dev/syd/default/delete:{action}"))
}

/// Set the default action for Rename Sandboxing.
#[no_mangle]
pub extern "C" fn syd_default_rename(action: action_t) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };
    stat(&format!("/dev/syd/default/rename:{action}"))
}

/// Set the default action for Symlink Sandboxing.
#[no_mangle]
pub extern "C" fn syd_default_symlink(action: action_t) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };
    stat(&format!("/dev/syd/default/symlink:{action}"))
}

/// Set the default action for Truncate Sandboxing.
#[no_mangle]
pub extern "C" fn syd_default_truncate(action: action_t) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };
    stat(&format!("/dev/syd/default/truncate:{action}"))
}

/// Set the default action for Chdir Sandboxing.
#[no_mangle]
pub extern "C" fn syd_default_chdir(action: action_t) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };
    stat(&format!("/dev/syd/default/chdir:{action}"))
}

/// Set the default action for Readdir Sandboxing.
#[no_mangle]
pub extern "C" fn syd_default_readdir(action: action_t) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };
    stat(&format!("/dev/syd/default/readdir:{action}"))
}

/// Set the default action for Mkdir Sandboxing.
#[no_mangle]
pub extern "C" fn syd_default_mkdir(action: action_t) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };
    stat(&format!("/dev/syd/default/mkdir:{action}"))
}

/// Set the default action for Rmdir Sandboxing.
#[no_mangle]
pub extern "C" fn syd_default_rmdir(action: action_t) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };
    stat(&format!("/dev/syd/default/rmdir:{action}"))
}

/// Set the default action for Chown Sandboxing.
#[no_mangle]
pub extern "C" fn syd_default_chown(action: action_t) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };
    stat(&format!("/dev/syd/default/chown:{action}"))
}

/// Set the default action for Chgrp Sandboxing.
#[no_mangle]
pub extern "C" fn syd_default_chgrp(action: action_t) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };
    stat(&format!("/dev/syd/default/chgrp:{action}"))
}

/// Set the default action for Chmod Sandboxing.
#[no_mangle]
pub extern "C" fn syd_default_chmod(action: action_t) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };
    stat(&format!("/dev/syd/default/chmod:{action}"))
}

/// Set the default action for Chattr Sandboxing.
#[no_mangle]
pub extern "C" fn syd_default_chattr(action: action_t) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };
    stat(&format!("/dev/syd/default/chattr:{action}"))
}

/// Set the default action for Chroot Sandboxing.
#[no_mangle]
pub extern "C" fn syd_default_chroot(action: action_t) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };
    stat(&format!("/dev/syd/default/chroot:{action}"))
}

/// Set the default action for Utime Sandboxing.
#[no_mangle]
pub extern "C" fn syd_default_utime(action: action_t) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };
    stat(&format!("/dev/syd/default/utime:{action}"))
}

/// Set the default action for Mkdev Sandboxing.
#[no_mangle]
pub extern "C" fn syd_default_mkdev(action: action_t) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };
    stat(&format!("/dev/syd/default/mkdev:{action}"))
}

/// Set the default action for Mkfifo Sandboxing.
#[no_mangle]
pub extern "C" fn syd_default_mkfifo(action: action_t) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };
    stat(&format!("/dev/syd/default/mkfifo:{action}"))
}

/// Set the default action for Mktemp Sandboxing.
#[no_mangle]
pub extern "C" fn syd_default_mktemp(action: action_t) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };
    stat(&format!("/dev/syd/default/mktemp:{action}"))
}

/// Set the default action for Network Sandboxing.
#[no_mangle]
pub extern "C" fn syd_default_net(action: action_t) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };
    stat(&format!("/dev/syd/default/net:{action}"))
}

/// Set the default action for IP blocklist violations.
#[no_mangle]
pub extern "C" fn syd_default_block(action: action_t) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };
    stat(&format!("/dev/syd/default/block:{action}"))
}

/// Set the default action for Memory Sandboxing.
#[no_mangle]
pub extern "C" fn syd_default_mem(action: action_t) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };
    stat(&format!("/dev/syd/default/mem:{action}"))
}

/// Set the default action for PID Sandboxing.
#[no_mangle]
pub extern "C" fn syd_default_pid(action: action_t) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };
    stat(&format!("/dev/syd/default/pid:{action}"))
}

/// Set the default action for Force Sandboxing.
#[no_mangle]
pub extern "C" fn syd_default_force(action: action_t) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };
    stat(&format!("/dev/syd/default/force:{action}"))
}

/// Set the default action for SegvGuard
#[no_mangle]
pub extern "C" fn syd_default_segvguard(action: action_t) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };
    stat(&format!("/dev/syd/default/segvguard:{action}"))
}

/// Set the default action for TPE Sandboxing.
#[no_mangle]
pub extern "C" fn syd_default_tpe(action: action_t) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };
    stat(&format!("/dev/syd/default/tpe:{action}"))
}

/// Adds a request to the _ioctl_(2) denylist.
#[no_mangle]
pub extern "C" fn syd_ioctl_deny(request: u64) -> c_int {
    stat(&format!("/dev/syd/ioctl/deny+{request}"))
}

/// Adds an entry to the Integrity Force map for Force Sandboxing.
///
/// # Safety
///
/// This function is marked `unsafe` because it dereferences raw
/// pointers, which is inherently unsafe in Rust.
///
/// The caller must ensure the following conditions are met to safely
/// use this function:
///
/// 1. The `path` pointer must point to a valid, null-terminated C-style
///    string.
/// 2. The `hash` pointer must point to a valid, null-terminated C-style
///    string.
#[no_mangle]
pub unsafe extern "C" fn syd_force_add(
    path: *const c_char,
    hash: *const c_char,
    action: action_t,
) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };

    if path.is_null() || hash.is_null() {
        return -EFAULT;
    }

    // SAFETY: Trust that `path` and `hash` are a null-terminated strings.
    let path = unsafe { CStr::from_ptr(path) };
    // SAFETY: ditto
    let hash = unsafe { CStr::from_ptr(hash) };
    let path = match path.to_str() {
        Ok(s) => s,
        Err(_) => return -EINVAL,
    };
    let hash = match hash.to_str() {
        Ok(s) => s,
        Err(_) => return -EINVAL,
    };

    // Call the stat function with the formatted string.
    stat(format!("/dev/syd/force+{path}:{hash}:{action}"))
}

/// Removes an entry from the Integrity Force map for Force Sandboxing.
/// # Safety
///
/// This function is marked `unsafe` because it dereferences raw
/// pointers, which is inherently unsafe in Rust.
///
/// The caller must ensure the following conditions are met to safely
/// use this function:
///
/// 1. The `path` pointer must point to a valid, null-terminated C-style
///    string.
#[no_mangle]
pub unsafe extern "C" fn syd_force_del(path: *const c_char) -> c_int {
    if path.is_null() {
        return -EFAULT;
    }

    // SAFETY: Trust that `path` is a null-terminated string.
    let path = unsafe { CStr::from_ptr(path) };
    let path = match path.to_str() {
        Ok(s) => s,
        Err(_) => return -EINVAL,
    };

    // Call the stat function with the formatted string.
    stat(format!("/dev/syd/force-{path}"))
}

/// Clears the Integrity Force map for Force Sandboxing.
#[no_mangle]
pub extern "C" fn syd_force_clr() -> c_int {
    stat("/dev/syd/force^")
}

/// Adds to the given actionlist of stat sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_stat_add(action: action_t, glob: *const c_char) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };

    // Call magic function with add operator.
    esyd(&format!("{action}/stat"), glob, b'+')
}

/// Removes the first instance from the end of the given actionlist of
/// stat sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_stat_del(action: action_t, glob: *const c_char) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };

    // Call magic function with del operator.
    esyd(&format!("{action}/stat"), glob, b'-')
}

/// Removes all matching patterns from the given actionlist of stat sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_stat_rem(action: action_t, glob: *const c_char) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };

    // Call magic function with rem operator.
    esyd(&format!("{action}/stat"), glob, b'^')
}

/// Adds to the given actionlist of read sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_read_add(action: action_t, glob: *const c_char) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };

    // Call magic function with add operator.
    esyd(&format!("{action}/read"), glob, b'+')
}

/// Removes the first instance from the end of the given actionlist of
/// read sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_read_del(action: action_t, glob: *const c_char) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };

    // Call magic function with del operator.
    esyd(&format!("{action}/read"), glob, b'-')
}

/// Removes all matching patterns from the given actionlist of read sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_read_rem(action: action_t, glob: *const c_char) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };

    // Call magic function with rem operator.
    esyd(&format!("{action}/read"), glob, b'^')
}

/// Adds to the given actionlist of write sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_write_add(action: action_t, glob: *const c_char) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };

    // Call magic function with add operator.
    esyd(&format!("{action}/write"), glob, b'+')
}

/// Removes the first instance from the end of the given actionlist of
/// write sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_write_del(action: action_t, glob: *const c_char) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };

    // Call magic function with del operator.
    esyd(&format!("{action}/write"), glob, b'-')
}

/// Removes all matching patterns from the given actionlist of write sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_write_rem(action: action_t, glob: *const c_char) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };

    // Call magic function with rem operator.
    esyd(&format!("{action}/write"), glob, b'^')
}

/// Adds to the given actionlist of exec sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_exec_add(action: action_t, glob: *const c_char) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };

    // Call magic function with add operator.
    esyd(&format!("{action}/exec"), glob, b'+')
}

/// Removes the first instance from the end of the given actionlist of
/// exec sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_exec_del(action: action_t, glob: *const c_char) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };

    // Call magic function with del operator.
    esyd(&format!("{action}/exec"), glob, b'-')
}

/// Removes all matching patterns from the given actionlist of exec sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_exec_rem(action: action_t, glob: *const c_char) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };

    // Call magic function with rem operator.
    esyd(&format!("{action}/exec"), glob, b'^')
}

/// Adds to the given actionlist of ioctl sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_ioctl_add(action: action_t, glob: *const c_char) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };

    // Call magic function with add operator.
    esyd(&format!("{action}/ioctl"), glob, b'+')
}

/// Removes the first instance from the end of the given actionlist of
/// ioctl sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_ioctl_del(action: action_t, glob: *const c_char) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };

    // Call magic function with del operator.
    esyd(&format!("{action}/ioctl"), glob, b'-')
}

/// Removes all matching patterns from the given actionlist of ioctl sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_ioctl_rem(action: action_t, glob: *const c_char) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };

    // Call magic function with rem operator.
    esyd(&format!("{action}/ioctl"), glob, b'^')
}

/// Adds to the given actionlist of create sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_create_add(action: action_t, glob: *const c_char) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };

    // Call magic function with add operator.
    esyd(&format!("{action}/create"), glob, b'+')
}

/// Removes the first instance from the end of the given actionlist of
/// create sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_create_del(action: action_t, glob: *const c_char) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };

    // Call magic function with del operator.
    esyd(&format!("{action}/create"), glob, b'-')
}

/// Removes all matching patterns from the given actionlist of create sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_create_rem(action: action_t, glob: *const c_char) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };

    // Call magic function with rem operator.
    esyd(&format!("{action}/create"), glob, b'^')
}

/// Adds to the given actionlist of delete sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_delete_add(action: action_t, glob: *const c_char) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };

    // Call magic function with add operator.
    esyd(&format!("{action}/delete"), glob, b'+')
}

/// Removes the first instance from the end of the given actionlist of
/// delete sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_delete_del(action: action_t, glob: *const c_char) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };

    // Call magic function with del operator.
    esyd(&format!("{action}/delete"), glob, b'-')
}

/// Removes all matching patterns from the given actionlist of delete sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_delete_rem(action: action_t, glob: *const c_char) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };

    // Call magic function with rem operator.
    esyd(&format!("{action}/delete"), glob, b'^')
}

/// Adds to the given actionlist of rename sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_rename_add(action: action_t, glob: *const c_char) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };

    // Call magic function with add operator.
    esyd(&format!("{action}/rename"), glob, b'+')
}

/// Removes the first instance from the end of the given actionlist of
/// rename sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_rename_del(action: action_t, glob: *const c_char) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };

    // Call magic function with del operator.
    esyd(&format!("{action}/rename"), glob, b'-')
}

/// Removes all matching patterns from the given actionlist of rename sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_rename_rem(action: action_t, glob: *const c_char) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };

    // Call magic function with rem operator.
    esyd(&format!("{action}/rename"), glob, b'^')
}

/// Adds to the given actionlist of symlink sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_symlink_add(action: action_t, glob: *const c_char) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };

    // Call magic function with add operator.
    esyd(&format!("{action}/symlink"), glob, b'+')
}

/// Removes the first instance from the end of the given actionlist of
/// symlink sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_symlink_del(action: action_t, glob: *const c_char) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };

    // Call magic function with del operator.
    esyd(&format!("{action}/symlink"), glob, b'-')
}

/// Removes all matching patterns from the given actionlist of symlink sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_symlink_rem(action: action_t, glob: *const c_char) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };

    // Call magic function with rem operator.
    esyd(&format!("{action}/symlink"), glob, b'^')
}

/// Adds to the given actionlist of truncate sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_truncate_add(action: action_t, glob: *const c_char) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };

    // Call magic function with add operator.
    esyd(&format!("{action}/truncate"), glob, b'+')
}

/// Removes the first instance from the end of the given actionlist of
/// truncate sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_truncate_del(action: action_t, glob: *const c_char) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };

    // Call magic function with del operator.
    esyd(&format!("{action}/truncate"), glob, b'-')
}

/// Removes all matching patterns from the given actionlist of truncate sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_truncate_rem(action: action_t, glob: *const c_char) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };

    // Call magic function with rem operator.
    esyd(&format!("{action}/truncate"), glob, b'^')
}

/// Adds to the given actionlist of chdir sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_chdir_add(action: action_t, glob: *const c_char) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };

    // Call magic function with add operator.
    esyd(&format!("{action}/chdir"), glob, b'+')
}

/// Removes the first instance from the end of the given actionlist of
/// chdir sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_chdir_del(action: action_t, glob: *const c_char) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };

    // Call magic function with del operator.
    esyd(&format!("{action}/chdir"), glob, b'-')
}

/// Removes all matching patterns from the given actionlist of chdir sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_chdir_rem(action: action_t, glob: *const c_char) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };

    // Call magic function with rem operator.
    esyd(&format!("{action}/chdir"), glob, b'^')
}

/// Adds to the given actionlist of readdir sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_readdir_add(action: action_t, glob: *const c_char) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };

    // Call magic function with add operator.
    esyd(&format!("{action}/readdir"), glob, b'+')
}

/// Removes the first instance from the end of the given actionlist of
/// readdir sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_readdir_del(action: action_t, glob: *const c_char) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };

    // Call magic function with del operator.
    esyd(&format!("{action}/readdir"), glob, b'-')
}

/// Removes all matching patterns from the given actionlist of readdir sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_readdir_rem(action: action_t, glob: *const c_char) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };

    // Call magic function with del operator.
    esyd(&format!("{action}/readdir"), glob, b'^')
}

/// Adds to the given actionlist of mkdir sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_mkdir_add(action: action_t, glob: *const c_char) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };

    // Call magic function with add operator.
    esyd(&format!("{action}/mkdir"), glob, b'+')
}

/// Removes the first instance from the end of the given actionlist of
/// mkdir sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_mkdir_del(action: action_t, glob: *const c_char) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };

    // Call magic function with del operator.
    esyd(&format!("{action}/mkdir"), glob, b'-')
}

/// Removes all matching patterns from the given actionlist of mkdir sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_mkdir_rem(action: action_t, glob: *const c_char) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };

    // Call magic function with del operator.
    esyd(&format!("{action}/mkdir"), glob, b'^')
}

/// Adds to the given actionlist of rmdir sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_rmdir_add(action: action_t, glob: *const c_char) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };

    // Call magic function with add operator.
    esyd(&format!("{action}/rmdir"), glob, b'+')
}

/// Removes the first instance from the end of the given actionlist of
/// rmdir sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_rmdir_del(action: action_t, glob: *const c_char) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };

    // Call magic function with del operator.
    esyd(&format!("{action}/rmdir"), glob, b'-')
}

/// Removes all matching patterns from the given actionlist of rmdir sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_rmdir_rem(action: action_t, glob: *const c_char) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };

    // Call magic function with del operator.
    esyd(&format!("{action}/rmdir"), glob, b'^')
}

/// Adds to the given actionlist of chown sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_chown_add(action: action_t, glob: *const c_char) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };

    // Call magic function with add operator.
    esyd(&format!("{action}/chown"), glob, b'+')
}

/// Removes the first instance from the end of the given actionlist of
/// chown sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_chown_del(action: action_t, glob: *const c_char) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };

    // Call magic function with del operator.
    esyd(&format!("{action}/chown"), glob, b'-')
}

/// Removes all matching patterns from the given actionlist of chown sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_chown_rem(action: action_t, glob: *const c_char) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };

    // Call magic function with rem operator.
    esyd(&format!("{action}/chown"), glob, b'^')
}

/// Adds to the given actionlist of chgrp sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_chgrp_add(action: action_t, glob: *const c_char) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };

    // Call magic function with add operator.
    esyd(&format!("{action}/chgrp"), glob, b'+')
}

/// Removes the first instance from the end of the given actionlist of
/// chgrp sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_chgrp_del(action: action_t, glob: *const c_char) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };

    // Call magic function with del operator.
    esyd(&format!("{action}/chgrp"), glob, b'-')
}

/// Removes all matching patterns from the given actionlist of chgrp sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_chgrp_rem(action: action_t, glob: *const c_char) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };

    // Call magic function with rem operator.
    esyd(&format!("{action}/chgrp"), glob, b'^')
}

/// Adds to the given actionlist of chmod sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_chmod_add(action: action_t, glob: *const c_char) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };

    // Call magic function with add operator.
    esyd(&format!("{action}/chmod"), glob, b'+')
}

/// Removes the first instance from the end of the given actionlist of
/// chmod sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_chmod_del(action: action_t, glob: *const c_char) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };

    // Call magic function with del operator.
    esyd(&format!("{action}/chmod"), glob, b'-')
}

/// Removes all matching patterns from the given actionlist of chmod sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_chmod_rem(action: action_t, glob: *const c_char) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };

    // Call magic function with rem operator.
    esyd(&format!("{action}/chmod"), glob, b'^')
}

/// Adds to the given actionlist of chattr sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_chattr_add(action: action_t, glob: *const c_char) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };

    // Call magic function with add operator.
    esyd(&format!("{action}/chattr"), glob, b'+')
}

/// Removes the first instance from the end of the given actionlist of
/// chattr sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_chattr_del(action: action_t, glob: *const c_char) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };

    // Call magic function with del operator.
    esyd(&format!("{action}/chattr"), glob, b'-')
}

/// Removes all matching patterns from the given actionlist of chattr sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_chattr_rem(action: action_t, glob: *const c_char) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };

    // Call magic function with rem operator.
    esyd(&format!("{action}/chattr"), glob, b'^')
}

/// Adds to the given actionlist of chroot sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_chroot_add(action: action_t, glob: *const c_char) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };

    // Call magic function with add operator.
    esyd(&format!("{action}/chroot"), glob, b'+')
}

/// Removes the first instance from the end of the given actionlist of
/// chroot sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_chroot_del(action: action_t, glob: *const c_char) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };

    // Call magic function with del operator.
    esyd(&format!("{action}/chroot"), glob, b'-')
}

/// Removes all matching patterns from the given actionlist of chroot sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_chroot_rem(action: action_t, glob: *const c_char) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };

    // Call magic function with rem operator.
    esyd(&format!("{action}/chroot"), glob, b'^')
}

/// Adds to the given actionlist of utime sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_utime_add(action: action_t, glob: *const c_char) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };

    // Call magic function with add operator.
    esyd(&format!("{action}/utime"), glob, b'+')
}

/// Removes the first instance from the end of the given actionlist of
/// utime sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_utime_del(action: action_t, glob: *const c_char) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };

    // Call magic function with del operator.
    esyd(&format!("{action}/utime"), glob, b'-')
}

/// Removes all matching patterns from the given actionlist of utime sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_utime_rem(action: action_t, glob: *const c_char) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };

    // Call magic function with rem operator.
    esyd(&format!("{action}/utime"), glob, b'^')
}

/// Adds to the given actionlist of mkdev sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_mkdev_add(action: action_t, glob: *const c_char) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };

    // Call magic function with add operator.
    esyd(&format!("{action}/mkdev"), glob, b'+')
}

/// Removes the first instance from the end of the given actionlist of
/// mkdev sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_mkdev_del(action: action_t, glob: *const c_char) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };

    // Call magic function with del operator.
    esyd(&format!("{action}/mkdev"), glob, b'-')
}

/// Removes all matching patterns from the given actionlist of mkdev sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_mkdev_rem(action: action_t, glob: *const c_char) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };

    // Call magic function with rem operator.
    esyd(&format!("{action}/mkdev"), glob, b'^')
}

/// Adds to the given actionlist of mkfifo sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_mkfifo_add(action: action_t, glob: *const c_char) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };

    // Call magic function with add operator.
    esyd(&format!("{action}/mkfifo"), glob, b'+')
}

/// Removes the first instance from the end of the given actionlist of
/// mkfifo sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_mkfifo_del(action: action_t, glob: *const c_char) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };

    // Call magic function with del operator.
    esyd(&format!("{action}/mkfifo"), glob, b'-')
}

/// Removes all matching patterns from the given actionlist of mkfifo sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_mkfifo_rem(action: action_t, glob: *const c_char) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };

    // Call magic function with rem operator.
    esyd(&format!("{action}/mkfifo"), glob, b'^')
}

/// Adds to the given actionlist of mktemp sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_mktemp_add(action: action_t, glob: *const c_char) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };

    // Call magic function with add operator.
    esyd(&format!("{action}/mktemp"), glob, b'+')
}

/// Removes the first instance from the end of the given actionlist of
/// mktemp sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_mktemp_del(action: action_t, glob: *const c_char) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };

    // Call magic function with del operator.
    esyd(&format!("{action}/mktemp"), glob, b'-')
}

/// Removes all matching patterns from the given actionlist of mktemp sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_mktemp_rem(action: action_t, glob: *const c_char) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };

    // Call magic function with rem operator.
    esyd(&format!("{action}/mktemp"), glob, b'^')
}

/// Adds to the given actionlist of net/bind sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_net_bind_add(action: action_t, glob: *const c_char) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };

    // Call magic function with add operator.
    esyd(&format!("{action}/net/bind"), glob, b'+')
}

/// Removes the first instance from the end of the given actionlist of
/// net/bind sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_net_bind_del(action: action_t, glob: *const c_char) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };

    // Call magic function with del operator.
    esyd(&format!("{action}/net/bind"), glob, b'-')
}

/// Removes all matching patterns from the given actionlist of net/bind sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_net_bind_rem(action: action_t, glob: *const c_char) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };

    // Call magic function with rem operator.
    esyd(&format!("{action}/net/bind"), glob, b'^')
}

/// Adds to the given actionlist of net/connect sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_net_connect_add(action: action_t, glob: *const c_char) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };

    // Call magic function with add operator.
    esyd(&format!("{action}/net/connect"), glob, b'+')
}

/// Removes the first instance from the end of the given actionlist of
/// net/connect sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_net_connect_del(action: action_t, glob: *const c_char) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };

    // Call magic function with del operator.
    esyd(&format!("{action}/net/connect"), glob, b'-')
}

/// Removes all matching patterns from the given actionlist of net/connect sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_net_connect_rem(action: action_t, glob: *const c_char) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };

    // Call magic function with rem operator.
    esyd(&format!("{action}/net/connect"), glob, b'^')
}

/// Adds to the given actionlist of net/sendfd sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_net_sendfd_add(action: action_t, glob: *const c_char) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };

    // Call magic function with add operator.
    esyd(&format!("{action}/net/sendfd"), glob, b'+')
}

/// Removes the first instance from the end of the given actionlist of
/// net/sendfd sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_net_sendfd_del(action: action_t, glob: *const c_char) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };

    // Call magic function with del operator.
    esyd(&format!("{action}/net/sendfd"), glob, b'-')
}

/// Removes all matching patterns from the given actionlist of net/sendfd sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_net_sendfd_rem(action: action_t, glob: *const c_char) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };

    // Call magic function with rem operator.
    esyd(&format!("{action}/net/sendfd"), glob, b'^')
}

/// Adds to the given actionlist of net/link sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_net_link_add(action: action_t, family: *const c_char) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };

    // Call magic function with add operator.
    esyd(&format!("{action}/net/link"), family, b'+')
}

/// Removes the first instance from the end of the given actionlist of
/// net/link sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_net_link_del(action: action_t, family: *const c_char) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };

    // Call magic function with del operator.
    esyd(&format!("{action}/net/link"), family, b'-')
}

/// Removes all matching patterns from the given actionlist of net/link sandboxing.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_net_link_rem(action: action_t, family: *const c_char) -> c_int {
    // Convert action_t enum to corresponding action string.
    let action = match Action::try_from(action) {
        Ok(action) => action,
        Err(_) => return -EINVAL,
    };

    // Call magic function with rem operator.
    esyd(&format!("{action}/net/link"), family, b'^')
}

/// Set syd maximum per-process memory usage limit for memory sandboxing.
///
/// parse-size crate is used to parse the value so formatted strings are OK.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_mem_max(size: *const c_char) -> c_int {
    esyd("mem/max", size, b':')
}

/// Set syd maximum per-process virtual memory usage limit for memory sandboxing.
///
/// parse-size crate is used to parse the value so formatted strings are OK.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_mem_vm_max(size: *const c_char) -> c_int {
    esyd("mem/vm_max", size, b':')
}

/// Set syd maximum process id limit for PID sandboxing
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_pid_max(size: usize) -> c_int {
    stat(&format!("/dev/syd/pid/max:{size}"))
}

/// Specify SegvGuard entry expiry timeout in seconds.
/// Setting this timeout to 0 effectively disables SegvGuard.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_segvguard_expiry(timeout: u64) -> c_int {
    stat(&format!("/dev/syd/segvguard/expiry:{timeout}"))
}

/// Specify SegvGuard entry suspension timeout in seconds.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_segvguard_suspension(timeout: u64) -> c_int {
    stat(&format!("/dev/syd/segvguard/suspension:{timeout}"))
}

/// Specify SegvGuard max number of crashes before suspension.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub extern "C" fn syd_segvguard_maxcrashes(max: u8) -> c_int {
    stat(&format!("/dev/syd/segvguard/maxcrashes:{max}"))
}

/// Execute a command outside the sandbox without sandboxing
///
/// # Safety
///
/// This function is marked `unsafe` because it dereferences raw
/// pointers, which is inherently unsafe in Rust.
///
/// The caller must ensure the following conditions are met to safely
/// use this function:
///
/// 1. The `file` pointer must point to a valid, null-terminated C-style
///    string.
///
/// 2. The `argv` pointer must point to an array of pointers, where each
///    pointer refers to a valid, null-terminated C-style string. The
///    last pointer in the array must be null, indicating the end of the
///    array.
///
/// 3. The memory pointed to by `file` and `argv` must remain valid for
///    the duration of the call.
///
/// Failing to uphold these guarantees can lead to undefined behavior,
/// including memory corruption and data races.
///
/// Returns 0 on success, negated errno on failure.
#[no_mangle]
pub unsafe extern "C" fn syd_exec(file: *const c_char, argv: *const *const c_char) -> c_int {
    if file.is_null() || argv.is_null() {
        return -EFAULT;
    }

    // SAFETY: Trust that `file` is a null-terminated string.
    let file = CStr::from_ptr(file);
    let file = OsStr::from_bytes(file.to_bytes());

    let mut path = OsString::from("/dev/syd/cmd/exec!");
    path.push(file);

    let mut idx: isize = 0;
    while !(*argv.offset(idx)).is_null() {
        // SAFETY: Trust that each `argv` element is a null-terminated string.
        let arg = CStr::from_ptr(*argv.offset(idx));
        let arg = OsStr::from_bytes(arg.to_bytes());

        path.push(OsStr::from_bytes(&[b'\x1F'])); // ASCII Unit Separator
        path.push(arg);

        idx = idx.saturating_add(1);
    }

    let path = PathBuf::from(path);
    stat(path)
}
