//
// Syd: rock-solid application kernel
// src/syd-lock.rs: Run a command under Landlock
//
// Copyright (c) 2024, 2025 Ali Polatel <alip@chesswob.org>
//
// SPDX-License-Identifier: GPL-3.0

use std::{
    collections::HashSet,
    ops::RangeInclusive,
    os::unix::{ffi::OsStrExt, process::CommandExt},
    process::{Command, ExitCode},
};

use memchr::arch::all::is_equal;
use nix::errno::Errno;
use syd::{
    err::SydResult,
    hash::SydRandomState,
    landlock::{AccessFs, AccessNet, CompatLevel, RulesetStatus, ABI},
    landlock_policy::LandlockPolicy,
    lock_enabled,
    parsers::sandbox::{parse_landlock_cmd, LandlockCmd, LandlockFilter},
    path::{XPath, XPathBuf},
};

fn main() -> SydResult<ExitCode> {
    use lexopt::prelude::*;

    syd::set_sigpipe_dfl()?;

    // Parse CLI options.
    //
    // Note, option parsing is POSIXly correct:
    // POSIX recommends that no more options are parsed after the first
    // positional argument. The other arguments are then all treated as
    // positional arguments.
    // See: https://pubs.opengroup.org/onlinepubs/9699919799/basedefs/V1_chap12.html#tag_12_02
    let mut opt_abick = false;
    let mut opt_check = false;
    let mut opt_verbose = false;
    let mut opt_cmd = None;
    let mut opt_arg = Vec::new();
    let mut policy = LandlockPolicy {
        compat_level: Some(CompatLevel::HardRequirement),
        scoped_abs: true,
        scoped_sig: true,
        ..Default::default()
    };

    let mut parser = lexopt::Parser::from_env();
    while let Some(arg) = parser.next()? {
        match arg {
            Short('h') => {
                help();
                return Ok(ExitCode::SUCCESS);
            }
            Short('A') => opt_abick = true,
            Short('V') => opt_check = true,
            Short('v') => opt_verbose = true,
            // Interface to Landlock compatibility levels.
            Short('C') => {
                let level = parser.value()?;
                let level = level.as_bytes();
                if is_equal(level, b"h") || is_equal(level, b"hard") {
                    policy.compat_level = Some(CompatLevel::HardRequirement);
                } else if is_equal(level, b"s") || is_equal(level, b"soft") {
                    policy.compat_level = Some(CompatLevel::SoftRequirement);
                } else if is_equal(level, b"b") || is_equal(level, b"best") {
                    policy.compat_level = Some(CompatLevel::BestEffort);
                } else {
                    return Err(Errno::EINVAL.into());
                }
            }
            // New interface with refined categories.
            Short('l') => {
                let command = parser.value().map(XPathBuf::from)?;
                let command = parse_landlock_cmd(&format!("allow/lock/{command}"))?;
                handle_lock_command(&mut policy, command)?;
            }
            // Old interface with practical read/write generalization.
            Short('r') => {
                let path = parser.value().map(XPathBuf::from)?;
                let command =
                    parse_landlock_cmd(&format!("allow/lock/read,readdir,exec,ioctl+{path}"))?;
                handle_lock_command(&mut policy, command)?;
            }
            Short('w') => {
                let path = parser.value().map(XPathBuf::from)?;
                let command = parse_landlock_cmd(&format!("allow/lock/all+{path}"))?;
                handle_lock_command(&mut policy, command)?;
            }
            Short('b') => {
                let port = parser.value().map(XPathBuf::from)?;
                let command = parse_landlock_cmd(&format!("allow/lock/bind+{port}"))?;
                handle_lock_command(&mut policy, command)?;
            }
            Short('c') => {
                let port = parser.value().map(XPathBuf::from)?;
                let command = parse_landlock_cmd(&format!("allow/lock/connect+{port}"))?;
                handle_lock_command(&mut policy, command)?;
            }
            Value(prog) => {
                opt_cmd = Some(prog);
                opt_arg.extend(parser.raw_args()?);
            }
            _ => return Err(arg.unexpected().into()),
        }
    }

    if opt_abick && opt_check {
        eprintln!("-A and -V are mutually exclusive!");
        return Err(Errno::EINVAL.into());
    }

    let abi = ABI::new_current();
    if opt_abick {
        let abi = abi as i32 as u8;
        print!("{abi}");
        return Ok(ExitCode::from(abi));
    } else if opt_check {
        if abi == ABI::Unsupported {
            println!("Landlock is not supported.");
            return Ok(ExitCode::from(127));
        }

        let state = lock_enabled(abi);
        let state_verb = match state {
            0 => "fully enforced",
            1 => "partially enforced",
            2 => "not enforced",
            _ => "unsupported",
        };
        println!("Landlock ABI {} is {state_verb}.", abi as i32);
        return Ok(ExitCode::from(state));
    }

    // Prepare command or bail if not passed.
    let cmd = if let Some(cmd) = opt_cmd {
        cmd
    } else {
        help();
        return Ok(ExitCode::SUCCESS);
    };

    // Set up Landlock sandbox.
    macro_rules! vprintln {
        ($($arg:tt)*) => {
            if opt_verbose {
                eprintln!($($arg)*);
            }
        };
    }

    match policy.restrict_self(abi) {
        Ok(status) => match status.ruleset {
            RulesetStatus::FullyEnforced => {
                vprintln!("syd-lock: Landlock ABI {} is fully enforced.", abi as i32)
            }
            RulesetStatus::PartiallyEnforced => {
                vprintln!(
                    "syd-lock: Landlock ABI {} is partially enforced.",
                    abi as i32
                )
            }
            RulesetStatus::NotEnforced => {
                eprintln!("syd-lock: Landlock ABI {} is not enforced!", abi as i32);
                return Ok(ExitCode::FAILURE);
            }
        },
        Err(error) => {
            eprintln!("syd-lock: Landlock ABI {} unsupported: {error}", abi as i32);
            return Ok(ExitCode::FAILURE);
        }
    };

    // Execute command, /bin/sh by default.
    Ok(ExitCode::from(
        127 + Command::new(cmd)
            .args(opt_arg)
            .exec()
            .raw_os_error()
            .unwrap_or(0) as u8,
    ))
}

fn help() {
    println!("Usage: syd-lock [-hvAV] [-l category[,category...]{{+|-}}path|port[-port]]... {{command [args...]}}");
    println!("Run a command under Landlock.");
    println!("Use -v to increase verbosity.");
    println!("Use -A to exit with Landlock ABI version, rather than running a command.");
    println!("Use -V to check for Landlock support, rather than running a command.");
    println!("Use -l cat[,cat...]{{+|-}}path|port[-port] to specify sandbox categories with path or closed port range.");
    println!();
    println!("Supported sandbox categories are read, write, exec, ioctl,");
    println!("create, delete, rename, symlink, truncate, readdir, mkdir,");
    println!("rmdir, mkdev, mkfifo, bind, connect, and all.");
    println!("Categories other than bind and connect must specify file or directory paths.");
    println!("Categories bind and connect must specify network ports or closed port ranges;");
    println!("the bind category also supports absolute UNIX domain socket paths.");
    println!("For full details and specific behavior of each sandbox category,");
    println!(
        "refer to the \"Sandboxing\" and \"Lock Sandboxing\" sections of the syd(7) manual page."
    );
}

fn handle_lock_command(policy: &mut LandlockPolicy, command: LandlockCmd) -> Result<(), Errno> {
    let pat = XPathBuf::from(command.arg);
    let mut access_fs = AccessFs::EMPTY;
    let mut access_net = AccessNet::EMPTY;

    if command.filter == LandlockFilter::All {
        // nice-to-have: allow/lock/all+/trusted
        //
        // SAFETY: Leave out AccessFs::MakeBlock:
        // Block device creation is never allowed.
        access_fs = AccessFs::Execute |
            AccessFs::WriteFile |
            AccessFs::ReadFile |
            AccessFs::ReadDir |
            AccessFs::RemoveDir |
            AccessFs::RemoveFile |
            AccessFs::MakeChar |
            AccessFs::MakeDir |
            AccessFs::MakeReg |
            AccessFs::MakeSock |
            AccessFs::MakeFifo |
            // AccessFs::MakeBlock |
            AccessFs::MakeSym |
            AccessFs::Refer |
            AccessFs::Truncate |
            AccessFs::IoctlDev;
    } else if let LandlockFilter::Many(access) = command.filter {
        let access: HashSet<String, SydRandomState> = HashSet::from_iter(access);

        // Determine between AccessFs and AccessNet.
        // For simplicity we require absolute path names
        // for the only colliding category `lock/bind`
        // and otherwise we assume a port-range if access
        // rights include only bind and/or connect.
        let has_bind = access.contains("bind");
        let has_conn = access.contains("connect");

        let n = access.len();
        if has_conn && ((has_bind && n != 2) || (!has_bind && n != 1)) {
            // connect specified with irrelevant category.
            return Err(Errno::EINVAL);
        }

        if pat.as_bytes()[0] != b'/' {
            if has_conn {
                access_net |= AccessNet::ConnectTcp;
                if has_bind {
                    access_net |= AccessNet::BindTcp;
                }
            } else if has_bind {
                // If any non-net category is specified with bind, assume fs.
                if n == 1 {
                    access_net |= AccessNet::BindTcp;
                }
            } // No bind or connect in categories, assume fs.
        }

        if access_net.is_empty() {
            // FS access, populate rights.
            for access in access {
                access_fs |= match access.as_str() {
                    "read" => AccessFs::ReadFile,
                    "write" => AccessFs::WriteFile,
                    "exec" => AccessFs::Execute,
                    "ioctl" => AccessFs::IoctlDev,
                    "create" => AccessFs::MakeReg,
                    "delete" => AccessFs::RemoveFile,
                    "rename" => AccessFs::Refer,
                    "symlink" => AccessFs::MakeSym,
                    "truncate" => AccessFs::Truncate,
                    "readdir" => AccessFs::ReadDir,
                    "mkdir" => AccessFs::MakeDir,
                    "rmdir" => AccessFs::RemoveDir,
                    "mkdev" => AccessFs::MakeChar,
                    "mkfifo" => AccessFs::MakeFifo,
                    "bind" => AccessFs::MakeSock,
                    _ => unreachable!("Invalid lock rule regex!"),
                };
            }
        }
    }

    let op = command.op;
    if !access_fs.is_empty() {
        // For ease of use the `-' and `^' operations are functionally
        // equivalent for sets.
        match op {
            '+' => {
                // add rule
                rule_add_lock_fs(policy, access_fs, &pat)
            }
            '-' | '^' => {
                // remove all matching rules
                rule_del_lock_fs(policy, access_fs, &pat)
            }
            _ => Err(Errno::EINVAL),
        }
    } else if !access_net.is_empty() {
        // For ease of use the `-' and `^' operations are functionally
        // equivalent for sets.
        match op {
            '+' => {
                // add rule
                rule_add_lock_net(policy, access_net, &pat.to_string())
            }
            '-' | '^' => {
                // remove all matching rules
                rule_del_lock_net(policy, access_net, &pat.to_string())
            }
            _ => Err(Errno::EINVAL),
        }
    } else {
        Err(Errno::EINVAL)
    }
}

fn rule_add_lock_fs(
    policy: &mut LandlockPolicy,
    access: AccessFs,
    pat: &XPath,
) -> Result<(), Errno> {
    if access.is_empty() {
        return Err(Errno::EINVAL);
    } else if access.contains(AccessFs::MakeBlock) {
        // SAFETY: Block device creation is never allowed.
        return Err(Errno::EACCES);
    }

    for access in access.iter() {
        let set = get_pathset_mut(policy, access);
        if let Some(ref mut set) = set {
            set.insert(pat.to_owned());
        } else {
            let mut new_set = HashSet::default();
            new_set.insert(pat.to_owned());
            *set = Some(new_set);
        }
    }

    Ok(())
}

fn rule_del_lock_fs(
    policy: &mut LandlockPolicy,
    access: AccessFs,
    pat: &XPath,
) -> Result<(), Errno> {
    if access.is_empty() {
        return Err(Errno::EINVAL);
    } else if access.contains(AccessFs::MakeBlock) {
        // SAFETY: Block device creation is never allowed.
        return Err(Errno::EACCES);
    }

    for access in access.iter() {
        let set = get_pathset_mut(policy, access);
        if let Some(ref mut set_ref) = set {
            set_ref.remove(pat);
            if set_ref.is_empty() {
                *set = None;
            }
        }
    }

    Ok(())
}

fn rule_add_lock_net(
    policy: &mut LandlockPolicy,
    access: AccessNet,
    pat: &str,
) -> Result<(), Errno> {
    if access.is_empty() {
        return Err(Errno::EINVAL);
    }

    // Argument is either a single port or a closed range in format "port1-port2".
    let pat = {
        let parts: Vec<&str> = pat.splitn(2, '-').collect();
        if parts.len() == 2 {
            let start = parts[0].parse::<u16>().or(Err(Errno::EINVAL))?;
            let end = parts[1].parse::<u16>().or(Err(Errno::EINVAL))?;
            start..=end
        } else {
            let port = parts[0].parse::<u16>().or(Err(Errno::EINVAL))?;
            port..=port
        }
    };

    for access in access.iter() {
        let set = get_portset_mut(policy, access);
        if let Some(ref mut set) = set {
            set.insert(pat.clone());
        } else {
            let mut new_set = HashSet::default();
            new_set.insert(pat.clone());
            *set = Some(new_set);
        }
    }

    Ok(())
}

fn rule_del_lock_net(
    policy: &mut LandlockPolicy,
    access: AccessNet,
    pat: &str,
) -> Result<(), Errno> {
    if access.is_empty() {
        return Err(Errno::EINVAL);
    }

    // Argument is either a single port or a closed range in format "port1-port2".
    let pat = {
        let parts: Vec<&str> = pat.splitn(2, '-').collect();
        if parts.len() == 2 {
            let start = parts[0].parse::<u16>().or(Err(Errno::EINVAL))?;
            let end = parts[1].parse::<u16>().or(Err(Errno::EINVAL))?;
            start..=end
        } else {
            let port = parts[0].parse::<u16>().or(Err(Errno::EINVAL))?;
            port..=port
        }
    };

    for access in access.iter() {
        let set = get_portset_mut(policy, access);
        if let Some(ref mut set_ref) = set {
            set_ref.remove(&pat);
            if set_ref.is_empty() {
                *set = None;
            }
        }
    }

    Ok(())
}

#[inline]
fn get_pathset_mut(
    policy: &mut LandlockPolicy,
    access: AccessFs,
) -> &mut Option<HashSet<XPathBuf, SydRandomState>> {
    match access {
        AccessFs::ReadFile => &mut policy.read_pathset,
        AccessFs::WriteFile => &mut policy.write_pathset,
        AccessFs::Execute => &mut policy.exec_pathset,
        AccessFs::IoctlDev => &mut policy.ioctl_pathset,
        AccessFs::MakeReg => &mut policy.create_pathset,
        AccessFs::RemoveFile => &mut policy.delete_pathset,
        AccessFs::Refer => &mut policy.rename_pathset,
        AccessFs::MakeSym => &mut policy.symlink_pathset,
        AccessFs::Truncate => &mut policy.truncate_pathset,
        AccessFs::ReadDir => &mut policy.readdir_pathset,
        AccessFs::MakeDir => &mut policy.mkdir_pathset,
        AccessFs::RemoveDir => &mut policy.rmdir_pathset,
        AccessFs::MakeChar => &mut policy.mkdev_pathset,
        AccessFs::MakeFifo => &mut policy.mkfifo_pathset,
        AccessFs::MakeSock => &mut policy.bind_pathset,
        AccessFs::MakeBlock => {
            panic!("BUG: requested pathset for forbidden access right MakeBlock!")
        }
        _ => unreachable!(),
    }
}

#[inline]
fn get_portset_mut(
    policy: &mut LandlockPolicy,
    access: AccessNet,
) -> &mut Option<HashSet<RangeInclusive<u16>, SydRandomState>> {
    match access {
        AccessNet::BindTcp => &mut policy.bind_portset,
        AccessNet::ConnectTcp => &mut policy.conn_portset,
        _ => unreachable!(),
    }
}
