# ========================== begin_copyright_notice ============================
#
# Copyright (C) 2022-2024 Intel Corporation
#
# SPDX-License-Identifier: MIT
#
# =========================== end_copyright_notice =============================

#===----------------------------------------------------------------------===//
#
# This file defines all of the Internal-specific intrinsics, which correspond to
# vISA instructions.
#
# Comment lines with a triple slash ### introduction are extracted and
# appended to docs/Targets/Internal/InternalLangRef.rst to give the Internal backend
# language reference in docs/autogenerated/Targets/Internal/InternalLangRef.rst.
#
#===------------------------------------------------------------------------===#

#------------ Currently Supported Types ----------------------
#PointerTypes = ["ptr_private", "ptr_global", "ptr_constant", "ptr_local", "ptr_generic"]
#FloatingPointTypes = ["half", "float", "double"]
#IntegerTypes = ["bool", "char", "short", "int", "long"]
#AdditionalTypes = ["vararg"]
#AttributeEntries = ["None", "NoReturn", "NoDuplicate", "Convergent", "SideEffects"]
#FIXME: It should be possible to specify multiple attribute entries as a comma separated list
#       (e.g., "Convergent,NoDuplicate")
#
# EX. "blah": {"result" : {return_type}, "arguments" : [arg1_type, arg2_type.....], "attributes" : Property }
#
# The "any" type can be followed by a default type if a type is not explicitly specified : Ex. "any:int"
#
# 0 - LLVMMatchType<0>
# 1 - LLVMMatchType<1>
# {int} - LLVMMatchType<{int}>

Imported_Intrinsics = {
## ``llvm.vc.internal.jump.table`` : CMC internal, no VISA
## ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
##
## * arg0: integer BasicBlock index in the full set of destinations
## * arg1-N: the full set of switch labels
##
## * Return value: selected label
##
## The intrinsic is a helper for switch jump tables generation. Arg0
## will be used by visa switchjmp as index. Return value and arg1-N are
## used to make ir semantically legal.
##
    "jump_table" : { "result" : "anyptr",
                     "arguments" :  ["anyint", "vararg"],
                     "attributes":  "None",
                     "memory_effects":
                         { "access": "NoModRef" } },

## ``llvm.vc.internal.read.variable.region`` : read a vISA variable region
## ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
##
## * arg0: ptr pointer to a global variable that corresponds to a vISA variable
##         (overloaded)
## * arg1: i32 vstride in elements, constant
## * arg2: i32 width in elements, constant
## * arg3: i32 stride in elements, constant
## * arg4: i32 offset in elements, constant
##
## * Return value: iN, fN, vXiN, vXfN the read value (overloaded)
##
## This corresponds to MOV instruction or a general source operand in visa.
## Utilizes technique of using global variable in LLVM IR for predefined
## vISA variables.
##
    "read_variable_region" : { "result": "any",
                               "arguments" : ["anyptr", "int", "int", "int",
                                              "int"],
                               "attributes" : "None",
                               "memory_effects":
                                   { "access": "Ref" }, },

## ``llvm.vc.internal.write.variable.region`` : write a vISA variable region
## ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
##
## * arg0: ptr pointer to a global variable that corresponds to a vISA variable
##         (overloaded)
## * arg1: iN, fN, vXiN, vXfN value to write (overloaded)
## * arg2: i32 stride in elements, constant
## * arg3: i32 offset in elements, constant
## * arg4: i1 or vXi1 mask (overloaded)
##
## This corresponds to MOV instruction or a general destination operand in visa.
## Utilizes technique of using global variable in LLVM IR for predefined
## vISA variables.
##
    "write_variable_region" : { "result": "void",
                                "arguments" : ["anyptr", "any", "int",
                                               "int", "anyint"],
                                "attributes" : "None",
                                "memory_effects":
                                    { "access": "Mod" }, },

## ``llvm.vc.internal.cast.to.ptr.explicit`` : convert ptr_generic to
## private/local/global ptr.
## ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
##
## * arg0: generic pointer
##
## * Return value: private/local/global pointer
##
## This intrisic attempts to explicitly convert a generic ptr to a
##  private/local/global ptr. If the cast fails the intrisic returns null pointer.
    "cast_to_ptr_explicit" : { "result": "anyptr",
                               "arguments": ["ptr_generic"],
                               "attributes": "None",
                               "memory_effects":
                                   { "access": "NoModRef" }, },

## ``llvm.vc.internal.optimization.fence`` : optimization fence
## ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
##
## * arg0: input value
##
## * Return value: the same as arg0
##
## The purpose of the intrinsic is to prevent the compiler from
## optimizing the value.
##
    "optimization_fence" : { "result": "any",
                             "arguments": [0],
                             "attributes": "None",
                             "memory_effects": { "access": "NoModRef" }, },

### --------------
### ALU intrinsics
### --------------

## ``llvm.vc.internal.cast.to.bf16`` : convert float into bfloat16
## ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
##
## * arg0: input data, f32 scalar or vector (overloaded)
##
## * Return value: i16 scalar or vector (overloaded)
##
## This intrinsic represents float -> bfloat16 conversion operation
    "cast_to_bf16" : { "result": "anyint",
                       "arguments": ["anyfloat"],
                       "attributes": "None",
                       "memory_effects":
                           { "access": "NoModRef" }, },
## ``llvm.vc.internal.cast.from.bf16`` : convert bfloat16 into float
## ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
##
## * arg0: bfloat16 input data, i16 scalar or vector (overloaded)
##
## * Return value: f32 scalar or vector (overloaded)
##
## This intrinsic represents float -> bfloat16 conversion operation
    "cast_from_bf16" : { "result": "anyfloat",
                         "arguments": ["anyint"],
                         "attributes": "None",
                         "memory_effects":
                             { "access": "NoModRef" }, },

## ``llvm.vc.internal.round.to.tf32`` : round float into tfloat32
## ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
##
## * arg0: input data, f32 scalar or vector (overloaded)
##
## * Return value: i32 scalar or vector (overloaded)
##
## This intrinsic represents float -> tfloat32 conversion operation
    "round_to_tf32" : { "result": "anyfloat",
                        "arguments": ["anyint"],
                        "attributes": "None",
                        "memory_effects":
                            { "access": "NoModRef" }, },

## ``llvm.vc.internal.stochastic.round.to.f16`` : half stochastic rounding operation
## ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
##
## * arg0: input data, f32 scalar or vector (overloaded)
## * arg1: random number, i16 scalar or vector of the same width as arg0
##
## * Return value: f16 scalar or vector of the same width as arg0
##
## This intrinsic represents float -> half stochastic rounding operation
    "stochastic_round_to_f16" : { "result": "anyfloat",
                                  "arguments": ["anyfloat", "anyint"],
                                  "attributes": "None",
                                  "memory_effects":
                                      { "access": "NoModRef" }, },
## ``llvm.vc.internal.stochastic.round.to.bf8`` : bf8 stochastic rounding operation
## ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
##
## * arg0: input data, f16 scalar or vector (overloaded)
## * arg1: random number, i8 scalar or vector of the same width as arg0
##
## * Return value: i8 scalar or vector of the same width as arg0
##
## This intrinsic represents half->bf8 stochastic rounding operations
    "stochastic_round_to_bf8" : { "result": "anyint",
                                  "arguments": ["anyfloat", "anyint"],
                                  "attributes": "None",
                                  "memory_effects":
                                      { "access": "NoModRef" }, },


## ``llvm.vc.internal.atomic.`` : intrinsics to represent SPIR-V atomic instructions
## ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
##
## * arg0: ptr to global/local/generic memory (overloaded)
## * arg1: i32 constant scope
## * arg2: i32 constant semantics
## * arg3: f16/f32/f64 scalar value (overloaded)
##
## * Return value: atomic result whose type matches a type of the input value (overloaded)
##
    "atomic_fmin" : { "result": "anyfloat",
                      "arguments": [ "anyptr", "int", "int",
                                     "anyfloat"],
                      "attributes": "SideEffects", },
    "atomic_fmax" : { "result": "anyfloat",
                      "arguments": [ "anyptr", "int", "int",
                                     "anyfloat"],
                      "attributes": "SideEffects", },

## ``llvm.vc.internal.rsqrtm`` : computes component-wise reciprocal square root
## ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
##
## * arg0: input data, floating-point scalar or vector, same as return value
##
## * Return structure:
## * result: vNfloat - rsqrtm result (overloaded)
## * pred:   vNxi1   - result predicate, indicates special number (NAN/INF/ZERO)
##
## Intrinsic converts to RSQTM visa-instruction
##
    "rsqrtm": { "result": ["anyfloat", "anyint"],
                "arguments": [0],
                "target" : ["hasIEEEDivSqrt"],
                "attributes": "None",
                "memory_effects":
                    { "access": "NoModRef" }, },

## ``llvm.vc.internal.invm`` : compute component-wise divide
## ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
##
## * arg0: input data, floating-point scalar or vector, same as return value
##
## * Return structure:
## * result: vNfloat - rsqrtm result (overloaded)
## * pred:   vNxi1   - result predicate, indicates special number (NAN/INF/ZERO)
##
## Intrinsic converts to INVM visa-instruction
##
    "invm": { "result": ["anyfloat", "anyint"],
              "arguments": [0, 0],
              "target" : ["hasIEEEDivSqrt"],
              "attributes": "None",
              "memory_effects":
                  { "access": "NoModRef" }, },


### ---------------------------
### Low-level memory intrinsics
### ---------------------------

## ``llvm.vc.internal.lsc.atomic.*``: LSC atomic intrinsics
## ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
##
## * arg0: vNxi1 Predicate (overloaded)
## * arg1: i8 Atomic opcode [MBC]
## * arg2: i8 Address size [MBC]
## * arg3: i8 Element size [MBC]
## * arg4: vNi8 Cache controls, where N is a number of supported cache levels [MBC]
## * arg5: i64 Address base (for stateless)
##         i32 Surface BTI (for stateful)
## * arg6: vNxi32 or vNxi64 Address indices (overloaded)
## * arg7: i16 Address scale [MBC]
## * arg8: i32 Address immediate offset [MBC]
## * arg9: 1st source vector for the atomic operation,
##          must be undef for unary operations
## * arg10: 2nd source vector for the atomic operation,
##          must be undef for unary and binary operations
## * arg11: vector to take values for masked simd lanes from
##
## * Return value: the value read from memory, merged with arg11 by predicate
##
    "lsc_atomic_bti": { "result": "anyvector",
                        "arguments": [
                            "anyint", # vNxi1, predicate
                            "char",   # atomic opcode
                            "char",   # address size
                            "char",   # element size
                            "anyint", # cache controls
                            "int",    # i32 BTI
                            "anyint", # vNi32 address offsets
                            "short",  # address scale
                            "int",    # address immediate offset
                            0,        # src1
                            0,        # src2
                            0,        # passthru value
                        ],
                        "target" : [
                            "hasLSCMessages",
                        ],
                        "attributes": "SideEffects", },
    "lsc_atomic_bss": { "result": "anyvector",
                        "arguments": [
                            "anyint", # vNxi1, predicate
                            "char",   # atomic opcode
                            "char",   # address size
                            "char",   # element size
                            "anyint", # cache controls
                            "int",    # i32 BSS
                            "anyint", # vNi32 address offsets
                            "short",  # address scale
                            "int",    # address immediate offset
                            0,        # src1
                            0,        # src2
                            0,        # passthru value
                        ],
                        "target" : [
                            "hasLSCMessages",
                        ],
                        "attributes": "SideEffects", },
    "lsc_atomic_slm": { "result": "anyvector",
                        "arguments": [
                            "anyint", # vNxi1, predicate
                            "char",   # atomic opcode
                            "char",   # address size
                            "char",   # element size
                            "anyint", # cache controls
                            "int",    # i32 address base
                            "anyint", # vNi32 address offsets
                            "short",  # address scale
                            "int",    # address immediate offset
                            0,        # src1
                            0,        # src2
                            0,        # passthru value
                        ],
                        "target" : [
                            "hasLSCMessages",
                        ],
                        "attributes": "SideEffects", },
    "lsc_atomic_ugm": { "result": "anyvector",
                        "arguments": [
                            "anyint", # vNxi1, predicate
                            "char",   # atomic opcode
                            "char",   # address size
                            "char",   # element size
                            "anyint", # cache controls
                            "long",   # i64 address base
                            "anyint", # vNi32 or vNi64 address offsets
                            "short",  # address scale
                            "int",    # address immediate offset
                            0,        # src1
                            0,        # src2
                            0,        # passthru value
                        ],
                        "target" : [
                            "hasLSCMessages",
                        ],
                        "attributes": "SideEffects", },

## ``llvm.vc.internal.lsc.load.*`` : LSC load intrinsics
## ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
##
## * arg0: vNxi1 Predicate (overloaded)
## * arg1: i8 Address size [MBC]
## * arg2: i8 Element size [MBC]
## * arg3: i8 Vector size (number of elements per SIMD lane) [MBC]
##         i8 Channel mask (for quad intrinsics) [MBC]
## * arg4: vNi8 Cache controls, where N is a number of supported cache levels [MBC]
## * arg5: i64 Address base (for stateless)
##         i32 Surface BTI (for stateful)
## * arg6: vNxi32 or vNxi64 Address indices (overloaded)
## * arg7: i16 Address scale [MBC]
## * arg8: i32 Address immediate offset [MBC]
## * arg9: vector to take values for masked simd lanes from
##
## * Return value: the value read from memory, merged with arg9 by predicate
##
    "lsc_load_bti": { "result": "anyvector",
                      "arguments": [
                          "anyint", # vNxi1, predicate
                          "char",   # address size
                          "char",   # element size
                          "char",   # vector size
                          "anyint", # cache controls
                          "int",    # i32 BTI
                          "anyint", # vNi32 address offsets
                          "short",  # address scale
                          "int",    # address immediate offset
                          0,        # passthru value
                      ],
                      "target" : [
                          "hasLSCMessages",
                      ],
                      "attributes": "None",
                      "memory_effects":
                          { "access": "Ref" }, },
    "lsc_load_bss": { "result": "anyvector",
                      "arguments": [
                          "anyint", # vNxi1, predicate
                          "char",   # address size
                          "char",   # element size
                          "char",   # vector size
                          "anyint", # cache controls
                          "int",    # i32 BSS
                          "anyint", # vNi32 address offsets
                          "short",  # address scale
                          "int",    # address immediate offset
                          0,        # passthru value
                      ],
                      "target" : [
                          "hasLSCMessages",
                      ],
                      "attributes": "None",
                      "memory_effects":
                          { "access": "Ref" }, },
    "lsc_load_slm": { "result": "anyvector",
                      "arguments": [
                          "anyint", # vNxi1, predicate
                          "char",   # address size
                          "char",   # element size
                          "char",   # vector size
                          "anyint", # cache controls
                          "int",    # i32 address base
                          "anyint", # vNi32 address offsets
                          "short",  # address scale
                          "int",    # address immediate offset
                          0,        # passthru value
                      ],
                      "target" : [
                          "hasLSCMessages",
                      ],
                      "attributes": "None",
                      "memory_effects":
                          { "access": "Ref" }, },
    "lsc_load_ugm": { "result": "anyvector",
                      "arguments": [
                          "anyint", # vNxi1, predicate
                          "char",   # address size
                          "char",   # element size
                          "char",   # vector size
                          "anyint", # cache controls
                          "long",   # i64 address base
                          "anyint", # vNi32 or vNi64 address offsets
                          "short",  # address scale
                          "int",    # address immediate offset
                          0,        # passthru value
                      ],
                      "target" : [
                          "hasLSCMessages",
                      ],
                      "attributes": "None",
                      "memory_effects":
                          { "access": "Ref" }, },

    "lsc_load_quad_bti": { "result": "anyvector",
                           "arguments": [
                               "anyint", # vNxi1, predicate
                               "char",   # address size
                               "char",   # element size
                               "char",   # channel mask
                               "anyint", # cache controls
                               "int",    # i32 BTI
                               "anyint", # vNi32 address offsets
                               "short",  # address scale
                               "int",    # address immediate offset
                               0,        # passthru value
                           ],
                           "target" : [
                               "hasLSCMessages",
                           ],
                           "attributes": "None",
                           "memory_effects":
                               { "access": "Ref" }, },
    "lsc_load_quad_bss": { "result": "anyvector",
                           "arguments": [
                               "anyint", # vNxi1, predicate
                               "char",   # address size
                               "char",   # element size
                               "char",   # channel mask
                               "anyint", # cache controls
                               "int",    # i32 BSS
                               "anyint", # vNi32 address offsets
                               "short",  # address scale
                               "int",    # address immediate offset
                               0,        # passthru value
                           ],
                           "target" : [
                               "hasLSCMessages",
                           ],
                           "attributes": "None",
                           "memory_effects":
                               { "access": "Ref" }, },
    "lsc_load_quad_slm": { "result": "anyvector",
                           "arguments": [
                               "anyint", # vNxi1, predicate
                               "char",   # address size
                               "char",   # element size
                               "char",   # channel mask
                               "anyint", # cache controls
                               "int",    # i32 address base
                               "anyint", # vNi32 address offsets
                               "short",  # address scale
                               "int",    # address immediate offset
                               0,        # passthru value
                           ],
                           "target" : [
                               "hasLSCMessages",
                           ],
                           "attributes": "None",
                           "memory_effects":
                               { "access": "Ref" }, },
    "lsc_load_quad_ugm": { "result": "anyvector",
                           "arguments": [
                               "anyint", # vNxi1, predicate
                               "char",   # address size
                               "char",   # element size
                               "char",   # channel mask
                               "anyint", # cache controls
                               "long",   # i64 address base
                               "anyint", # vNi32 or vNi64 address offsets
                               "short",  # address scale
                               "int",    # address immediate offset
                               0,        # passthru value
                           ],
                           "target" : [
                               "hasLSCMessages",
                           ],
                           "attributes": "None",
                           "memory_effects":
                               { "access": "Ref" }, },

## ``llvm.vc.internal.lsc.prefetch.*`` : LSC prefetch intrinsics
## ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
##
## * arg0: vNxi1 Predicate (overloaded)
## * arg1: i8 Address size [MBC]
## * arg2: i8 Element size [MBC]
## * arg3: i8 Vector size (number of elements per SIMD lane) [MBC]
##         i8 Channel mask (for quad intrinsics) [MBC]
## * arg4: vNi8 Cache controls, where N is a number of supported cache levels [MBC]
## * arg5: i64 Address base (for stateless)
##         i32 Surface BTI (for stateful)
## * arg6: vNxi32 or vNxi64 Address indices (overloaded)
## * arg7: i16 Address scale [MBC]
## * arg8: i32 Address immediate offset [MBC]
##
## * Return value: void
##
    "lsc_prefetch_bti": { "result": "void",
                          "arguments": [
                              "anyint", # vNxi1, predicate
                              "char",   # address size
                              "char",   # element size
                              "char",   # vector size
                              "anyint", # cache controls
                              "int",    # i32 BTI
                              "anyint", # vNi32 address offsets
                              "short",  # address scale
                              "int",    # address immediate offset
                          ],
                          "target" : [
                              "hasLSCMessages",
                          ],
                          "attributes": "SideEffects", },
    "lsc_prefetch_bss": { "result": "void",
                          "arguments": [
                              "anyint", # vNxi1, predicate
                              "char",   # address size
                              "char",   # element size
                              "char",   # vector size
                              "anyint", # cache controls
                              "int",    # i32 BSS
                              "anyint", # vNi32 address offsets
                              "short",  # address scale
                              "int",    # address immediate offset
                          ],
                          "target" : [
                              "hasLSCMessages",
                          ],
                          "attributes": "SideEffects", },
    "lsc_prefetch_ugm": { "result": "void",
                          "arguments": [
                              "anyint", # vNxi1, predicate
                              "char",   # address size
                              "char",   # element size
                              "char",   # vector size
                              "anyint", # cache controls
                              "long",   # i64 address base
                              "anyint", # vNi32 or vNi64 address offsets
                              "short",  # address scale
                              "int",    # address immediate offset
                          ],
                          "target" : [
                              "hasLSCMessages",
                          ],
                          "attributes": "SideEffects", },

    "lsc_prefetch_quad_bti": { "result": "void",
                               "arguments": [
                                   "anyint", # vNxi1, predicate
                                   "char",   # address size
                                   "char",   # element size
                                   "char",   # channel mask
                                   "anyint", # cache controls
                                   "int",    # i32 BTI
                                   "anyint", # vNi32 address offsets
                                   "short",  # address scale
                                   "int",    # address immediate offset
                               ],
                               "target" : [
                                   "hasLSCMessages",
                               ],
                               "attributes": "SideEffects", },
    "lsc_prefetch_quad_bss": { "result": "void",
                               "arguments": [
                                   "anyint", # vNxi1, predicate
                                   "char",   # address size
                                   "char",   # element size
                                   "char",   # channel mask
                                   "anyint", # cache controls
                                   "int",    # i32 BSS
                                   "anyint", # vNi32 address offsets
                                   "short",  # address scale
                                   "int",    # address immediate offset
                               ],
                               "target" : [
                                   "hasLSCMessages",
                               ],
                               "attributes": "SideEffects", },
    "lsc_prefetch_quad_ugm": { "result": "void",
                               "arguments": [
                                   "anyint", # vNxi1, predicate
                                   "char",   # address size
                                   "char",   # element size
                                   "char",   # channel mask
                                   "anyint", # cache controls
                                   "long",   # i64 address base
                                   "anyint", # vNi32 or vNi64 address offsets
                                   "short",  # address scale
                                   "int",    # address immediate offset
                               ],
                               "target" : [
                                   "hasLSCMessages",
                               ],
                               "attributes": "SideEffects", },

## ``llvm.vc.internal.lsc.store.*`` : LSC store intrinsics
## ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
##
## * arg0: vNxi1 Predicate (overloaded)
## * arg1: i8 Address size [MBC]
## * arg2: i8 Element size [MBC]
## * arg3: i8 Vector size (number of elements per SIMD lane) [MBC]
##         i8 Channel mask (for quad intrinsics) [MBC]
## * arg4: vNi8 Cache controls, where N is a number of supported cache levels [MBC]
## * arg5: i64 Address base (for stateless)
##         i32 Surface BTI (for stateful)
## * arg6: vNxi32 or vNxi64 Address indices (overloaded)
## * arg7: i16 Address scale [MBC]
## * arg8: i32 Address immediate offset [MBC]
## * arg9: Data to write (overloaded)
##
## * Return value: void
##
    "lsc_store_bti": { "result": "void",
                       "arguments": [
                           "anyint", # vNxi1, predicate
                           "char",   # address size
                           "char",   # element size
                           "char",   # vector size
                           "anyint", # cache controls
                           "int",    # i32 BTI
                           "anyint", # vNi32 address offsets
                           "short",  # address scale
                           "int",    # address immediate offset
                           "anyvector", # Data to write
                       ],
                       "target" : [
                           "hasLSCMessages",
                       ],
                       "attributes": "None",
                       "memory_effects":
                           { "access": "Mod" }, },
    "lsc_store_bss": { "result": "void",
                       "arguments": [
                           "anyint", # vNxi1, predicate
                           "char",   # address size
                           "char",   # element size
                           "char",   # vector size
                           "anyint", # cache controls
                           "int",    # i32 BSS
                           "anyint", # vNi32 address offsets
                           "short",  # address scale
                           "int",    # address immediate offset
                           "anyvector", # Data to write
                       ],
                       "target" : [
                           "hasLSCMessages",
                       ],
                       "attributes": "None",
                       "memory_effects":
                           { "access": "Mod" }, },
    "lsc_store_slm": { "result": "void",
                       "arguments": [
                           "anyint", # vNxi1, predicate
                           "char",   # address size
                           "char",   # element size
                           "char",   # vector size
                           "anyint", # cache controls
                           "int",    # i32 address base
                           "anyint", # vNi32 address offsets
                           "short",  # address scale
                           "int",    # address immediate offset
                           "anyvector", # Data to write
                       ],
                       "target" : [
                           "hasLSCMessages",
                       ],
                       "attributes": "None",
                       "memory_effects":
                           { "access": "Mod" }, },
    "lsc_store_ugm": { "result": "void",
                       "arguments": [
                           "anyint", # vNxi1, predicate
                           "char",   # address size
                           "char",   # element size
                           "char",   # vector size
                           "anyint", # cache controls
                           "long",   # i64 address base
                           "anyint", # vNi32 or vNi64 address offsets
                           "short",  # address scale
                           "int",    # address immediate offset
                           "anyvector", # Data to write
                       ],
                       "target" : [
                           "hasLSCMessages",
                       ],
                       "attributes": "None",
                       "memory_effects":
                           { "access": "Mod" }, },

    "lsc_store_quad_bti": { "result": "void",
                            "arguments": [
                                "anyint", # vNxi1, predicate
                                "char",   # address size
                                "char",   # element size
                                "char",   # channel mask
                                "anyint", # cache controls
                                "int",    # i32 BTI
                                "anyint", # vNi32 address offsets
                                "short",  # address scale
                                "int",    # address immediate offset
                                "anyvector", # Data to write
                            ],
                            "target" : [
                                "hasLSCMessages",
                            ],
                            "attributes": "None",
                            "memory_effects":
                                { "access": "Mod" }, },
    "lsc_store_quad_bss": { "result": "void",
                            "arguments": [
                                "anyint", # vNxi1, predicate
                                "char",   # address size
                                "char",   # element size
                                "char",   # channel mask
                                "anyint", # cache controls
                                "int",    # i32 BSS
                                "anyint", # vNi32 address offsets
                                "short",  # address scale
                                "int",    # address immediate offset
                                "anyvector", # Data to write
                            ],
                            "target" : [
                                "hasLSCMessages",
                            ],
                            "attributes": "None",
                            "memory_effects":
                                { "access": "Mod" }, },
    "lsc_store_quad_slm": { "result": "void",
                            "arguments": [
                                "anyint", # vNxi1, predicate
                                "char",   # address size
                                "char",   # element size
                                "char",   # channel mask
                                "anyint", # cache controls
                                "int",    # i32 address base
                                "anyint", # vNi32 address offsets
                                "short",  # address scale
                                "int",    # address immediate offset
                                "anyvector", # Data to write
                            ],
                            "target" : [
                                "hasLSCMessages",
                            ],
                            "attributes": "None",
                            "memory_effects":
                                { "access": "Mod" }, },
    "lsc_store_quad_ugm": { "result": "void",
                            "arguments": [
                                "anyint", # vNxi1, predicate
                                "char",   # address size
                                "char",   # element size
                                "char",   # channel mask
                                "anyint", # cache controls
                                "long",   # i64 address base
                                "anyint", # vNi32 or vNi64 address offsets
                                "short",  # address scale
                                "int",    # address immediate offset
                                "anyvector", # Data to write
                            ],
                            "target" : [
                                "hasLSCMessages",
                            ],
                            "attributes": "None",
                            "memory_effects":
                                { "access": "Mod" }, },

## ``llvm.vc.internal.lsc.*.block.2d.ugm.*`` : LSC untyped 2d block intrinsics
## ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
##
## * Execution size is always 1.
## * arg0: i1, Predicate
## * arg1: i8, Element size [MBC]
## * arg2: vNi8 Cache controls, where N is a number of supported cache levels [MBC]
## * arg3: i8, Number of blocks [MBC]
## * arg4: i8, Block width (in elements) [MBC]
## * arg5: i8, Block height [MBC]
## * arg6: i64, Memory base address
## * arg7: i32, Memory matrix width (minus 1, in bytes)
## * arg8: i32, Memory matrix height (minus 1)
## * arg9: i32, Memory matrix row pitch (minus 1, in bytes)
## * arg10: i32, Memory block X position (in elements)
## * arg11: i32, Memory block Y position
## * arg12: i32, Memory block X position immediate offset (in elements) [MBC]
## * arg13: i32, Memory block Y position immediate offset [MBC]
## * arg14: value to passthru when predicate is false on load, data to store
##
## * Return value: loaded data
##
    "lsc_load_block_2d_ugm": { "result": "anyvector",
                               "arguments": [
                                   "bool",   # i1, predicate
                                   "char",   # element size
                                   "anyint", # cache controls
                                   "char",   # number of blocks
                                   "short",  # block width
                                   "short",  # block heigth
                                   "long",   # memory base address
                                   "int",    # memory matrix width (minus 1)
                                   "int",    # memory matrix height (minus 1)
                                   "int",    # memory matrix row pitch (minus 1)
                                   "int",    # memory block X position
                                   "int",    # memory block Y position
                                   "int",    # X offset
                                   "int",    # Y offset
                                   0,        # passthru
                               ],
                               "target" : [
                                   "hasLSCMessages",
                               ],
                               "attributes": "None",
                               "memory_effects":
                                   { "access": "Ref" }, },
    "lsc_load_block_2d_ugm_transposed": { "result": "anyvector",
                                          "arguments": [
                                              "bool",   # i1, predicate
                                              "char",   # element size
                                              "anyint", # cache controls
                                              "char",   # number of blocks
                                              "short",  # block width
                                              "short",  # block heigth
                                              "long",   # memory base address
                                              "int",    # memory matrix width (minus 1)
                                              "int",    # memory matrix height (minus 1)
                                              "int",    # memory matrix row pitch (minus 1)
                                              "int",    # memory block X position
                                              "int",    # memory block Y position
                                              "int",    # X offset
                                              "int",    # Y offset
                                              0,        # passthru
                                          ],
                                          "target" : [
                                              "hasLSCMessages",
                                          ],
                                          "attributes": "None",
                                          "memory_effects":
                                              { "access": "Ref" }, },
    "lsc_load_block_2d_ugm_vnni": { "result": "anyvector",
                                    "arguments": [
                                        "bool",   # i1, predicate
                                        "char",   # element size
                                        "anyint", # cache controls
                                        "char",   # number of blocks
                                        "short",  # block width
                                        "short",  # block heigth
                                        "long",   # memory base address
                                        "int",    # memory matrix width (minus 1)
                                        "int",    # memory matrix height (minus 1)
                                        "int",    # memory matrix row pitch (minus 1)
                                        "int",    # memory block X position
                                        "int",    # memory block Y position
                                        "int",    # X offset
                                        "int",    # Y offset
                                        0,        # passthru
                                    ],
                                    "target" : [
                                       "hasLSCMessages",
                                    ],
                                    "attributes": "None",
                                    "memory_effects":
                                        { "access": "Ref" }, },
    "lsc_prefetch_block_2d_ugm": { "result": "void",
                                   "arguments": [
                                       "bool",   # i1, predicate
                                       "char",   # element size
                                       "anyint", # cache controls
                                       "char",   # number of blocks
                                       "short",  # block width
                                       "short",  # block heigth
                                       "long",   # memory base address
                                       "int",    # memory matrix width (minus 1)
                                       "int",    # memory matrix height (minus 1)
                                       "int",    # memory matrix row pitch (minus 1)
                                       "int",    # memory block X position
                                       "int",    # memory block Y position
                                       "int",    # X offset
                                       "int",    # Y offset
                                   ],
                                   "target" : [
                                       "hasLSCMessages",
                                   ],
                                   "attributes": "SideEffects", },
    "lsc_store_block_2d_ugm": { "result": "void",
                                "arguments": [
                                    "bool",      # i1, predicate
                                    "char",      # element size
                                    "anyint",    # cache controls
                                    "char",      # number of blocks
                                    "short",     # block width
                                    "short",     # block heigth
                                    "long",      # memory base address
                                    "int",       # memory matrix width (minus 1)
                                    "int",       # memory matrix height (minus 1)
                                    "int",       # memory matrix row pitch (minus 1)
                                    "int",       # memory block X position
                                    "int",       # memory block Y position
                                    "int",       # X offset
                                    "int",       # Y offset
                                    "anyvector", # data to store
                                ],
                                "target" : [
                                    "hasLSCMessages",
                                ],
                                "attributes": "None",
                                "memory_effects":
                                    { "access": "Mod" }, },

## ``llvm.vc.internal.lsc.*.2d.ugm.desc.*`` : 2d block load/store/prefetch instructions
## ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
##
## * arg0: i1, Predicate
## * arg1: vNi8, Cache controls, where N is the number of supported cache levels [MBC]
## * arg2: i8, Number of blocks [MBC]
## * arg3: i8, Block width (in elements) [MBC]
## * arg4: i8, Block height [MBC]
## * arg5: v16i32 Matrix descriptor [MBC]
## * arg6: i32, Memory block X immediate offset (in elements) [MBC]
## * arg7: i32, Memory block Y immediate offset [MBC]
## * arg8: value to passthru when predicate is false on load,
##         or value to write on store,
##         or dummy value for prefetch to deduce the matrix element type
##
## * Return value: the value read or void
##
## The matrix descriptor is a 16-element vector that describes the 2D block layout in memory.
## The descriptor layout is as follows:
## desc[0]: low 32 bits of the base address
## desc[1]: high 32 bits of the base address
## desc[2]: matrix width in bytes, minus 1
## desc[3]: matrix height, minus 1
## desc[4]: matrix pitch in bytes, minus 1
## desc[5]: block start X in elements, signed
## desc[6]: block start Y in rows, signed
## desc[7]: block size encoded as follows:
##          (block_width - 1) | ((block_height - 1) << 8) | ((number_of_blocks - 1) << 16)
## desc[8-15]: reserved
##
    "lsc_load_2d_ugm_desc" : { "result" : "anyvector",
                               "arguments" : [
                                   "bool",      # i1, predicate
                                   "anyvector", # cache controls
                                   "char",      # number of blocks
                                   "short",     # block width
                                   "short",     # block height
                                   "int16",     # matrix descriptor
                                   "int",       # X offset
                                   "int",       # Y offset
                                   0,           # value to passthru when predicate is false
                                ],
                                "target" : [
                                   "hasLSCMessages",
                                ],
                                "attributes" : "None",
                                "memory_effects":
                                    { "access": "Ref" }, },
    "lsc_load_2d_ugm_desc_transpose" : { "result" : "anyvector",
                                         "arguments" : [
                                             "bool",      # i1, predicate
                                             "anyvector", # cache controls
                                             "char",      # number of blocks
                                             "short",     # block width
                                             "short",     # block height
                                             "int16",     # matrix descriptor
                                             "int",       # X offset
                                             "int",       # Y offset
                                             0,           # value to passthru when predicate is false
                                          ],
                                          "target" : [
                                             "hasLSCMessages",
                                          ],
                                          "attributes" : "None",
                                          "memory_effects":
                                              { "access": "Ref" }, },
    "lsc_load_2d_ugm_desc_vnni" : { "result" : "anyvector",
                                    "arguments" : [
                                        "bool",      # i1, predicate
                                        "anyvector", # cache controls
                                        "char",      # number of blocks
                                        "short",     # block width
                                        "short",     # block height
                                        "int16",     # matrix descriptor
                                        "int",       # X offset
                                        "int",       # Y offset
                                        0,           # value to passthru when predicate is false
                                     ],
                                     "target" : [
                                         "hasLSCMessages",
                                     ],
                                     "attributes" : "None",
                                     "memory_effects":
                                         { "access": "Ref" }, },
    "lsc_prefetch_2d_ugm_desc" : { "result" : "void",
                                   "arguments" : [
                                       "bool",      # i1, predicate
                                       "anyvector", # cache controls
                                       "char",      # number of blocks
                                       "short",     # block width
                                       "short",     # block height
                                       "int16",     # matrix descriptor
                                       "int",       # X offset
                                       "int",       # Y offset
                                       "anyvector", # dummy value, only element type is used
                                    ],
                                    "target" : [
                                        "hasLSCMessages",
                                    ],
                                    "attributes" : "SideEffects", },
    "lsc_store_2d_ugm_desc" : { "result" : "void",
                                "arguments" : [
                                    "bool",      # i1, predicate
                                    "anyvector", # cache controls
                                    "char",      # number of blocks
                                    "short",     # block width
                                    "short",     # block height
                                    "int16",     # matrix descriptor
                                    "int",       # X offset
                                    "int",       # Y offset
                                    "anyvector", # value to store
                                 ],
                                 "target" : [
                                     "hasLSCMessages",
                                 ],
                                 "attributes" : "None",
                                 "memory_effects":
                                     { "access": "Mod" }, },

## ``llvm.vc.internal.lsc.*2d.typed.bti.*`` : LSC typed 2d block BTI intrinsics
## ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
## * arg0: vNi8, Cache controls, where N is the number of supported cache levels [MBC]
## * arg1: i32, Surface BTI
## * arg2: i32, Block height [MBC]
## * arg3: i32, Block width (in elements) [MBC]
## * arg4: i32, Memory block X position (in bytes)
## * arg5: i32, Memory block Y position
## * arg6: data to write (store only)
##
## * Return value: the value read or void
##
    "lsc_load_2d_tgm_bti" : { "result" : "anyvector",
                              "arguments" : [
                                  "anyvector", # cache controls
                                  "int",       # i32 BTI
                                  "int",       # block height
                                  "int",       # block width
                                  "int",       # X offset
                                  "int"        # Y offset
                              ],
                              "target" : [
                                  "hasLSCMessages",
                                  "hasLSCTypedMessages",
                              ],
                              "attributes" : "None",
                              "memory_effects":
                                  { "access": "Ref" }, },
    "lsc_store_2d_tgm_bti" : { "result" : "void",
                               "arguments" : [
                                  "anyvector", # cache controls
                                  "int",       # i32 BTI
                                  "int",       # block height
                                  "int",       # block width
                                  "int",       # X offset
                                  "int",       # Y offset
                                  "anyvector"
                               ],
                               "target" : [
                                   "hasLSCMessages",
                                   "hasLSCTypedMessages",
                               ],
                               "attributes" : "None",
                               "memory_effects":
                                   { "access": "Mod" }, },

## ``llvm.vc.internal.lsc.*2d.typed.bss.*`` : LSC typed 2d block bindless intrinsics
## ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
## * arg0: vNi8, Cache controls, where N is the number of supported cache levels [MBC]
## * arg1: i32, Surface BSS
## * arg2: i32, Block height [MBC]
## * arg3: i32, Block width (in elements) [MBC]
## * arg4: i32, Memory block X position (in bytes)
## * arg5: i32, Memory block Y position
## * arg6: data to write (store only)
##
## * Return value: the value read or void
##
    "lsc_load_2d_tgm_bss" : { "result" : "anyvector",
                              "arguments" : [
                                  "anyvector", # cache controls
                                  "int",       # i32 BSS
                                  "int",       # block height
                                  "int",       # block width
                                  "int",       # X offset
                                  "int"        # Y offset
                              ],
                              "target" : [
                                  "hasLSCMessages",
                                  "hasLSCTypedMessages",
                              ],
                              "attributes" : "None",
                              "memory_effects":
                                  { "access": "Ref" }, },
    "lsc_store_2d_tgm_bss" : { "result" : "void",
                               "arguments" : [
                                  "anyvector", # cache controls
                                  "int",       # i32 BSS
                                  "int",       # block height
                                  "int",       # block width
                                  "int",       # X offset
                                  "int",       # Y offset
                                  "anyvector"
                               ],
                               "target" : [
                                   "hasLSCMessages",
                                   "hasLSCTypedMessages",
                               ],
                               "attributes" : "None",
                               "memory_effects":
                                   { "access": "Mod" }, },



## ``llvm.vc.internal.lsc.*.quad.tgm`` : Typed LSC load BTI intrinsic
## ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
## * arg0: vNi1, Predicate (overloaded)
## * arg1: vNi8, Cache controls, where N is the number of supported cache levels [MBC]
## * arg2: i8, Channel mask [MBC]
## * arg3: i32, Surface BTI
## * arg4: vNi32, U pixel indices (overloaded)
## * arg5: vNi32, V pixel indices
## * arg6: vNi32, R pixel indices
## * arg7: vNi32, LOD pixel indices
## * arg8: vector to take values for masked simd lanes from (load)
##         vector to take values to write (store)
##
## * Return value: the value read from memory (load) or void (store, prefetch)
##
    "lsc_load_quad_tgm": { "result": "anyvector",
                           "arguments": [
                               "anyint",    # vNxi1, predicate
                               "anyvector", # cache controls
                               "char",      # channel mask
                               "int",       # i32 BTI
                               "anyint",    # vNi32 U pixel index
                               3,           # vNi32 V pixel index
                               3,           # vNi32 R pixel index
                               3,           # vNi32 LOD pixel index
                               0,           # passthru value
                           ],
                           "target" : [
                               "hasLSCMessages",
                               "hasLSCTypedMessages",
                           ],
                           "attributes": "None",
                           "memory_effects":
                               { "access": "Ref" }, },
    "lsc_store_quad_tgm": { "result": "void",
                            "arguments": [
                                "anyint",    # vNxi1, predicate
                                "anyvector", # cache controls
                                "char",      # channel mask
                                "int",       # i32 BTI
                                "anyint",    # vNi32 U pixel index
                                2,           # vNi32 V pixel index
                                2,           # vNi32 R pixel index
                                2,           # vNi32 LOD pixel index
                                "anyvector", # data to write
                            ],
                            "target" : [
                                "hasLSCMessages",
                                "hasLSCTypedMessages",
                            ],
                            "attributes": "None",
                            "memory_effects":
                                { "access": "Mod" }, },
    "lsc_prefetch_quad_tgm": { "result": "void",
                               "arguments": [
                                   "anyint",    # vNxi1, predicate
                                   "anyvector", # cache controls
                                   "char",      # channel mask
                                   "int",       # i32 BTI
                                   "anyint",    # vNi32 U pixel index
                                   2,           # vNi32 V pixel index
                                   2,           # vNi32 R pixel index
                                   2,           # vNi32 LOD pixel index
                               ],
                               "target" : [
                                   "hasLSCMessages",
                                   "hasLSCTypedMessages",
                               ],
                               "attributes": "SideEffects", },

## ``llvm.vc.internal.lsc.*.quad.tgm.bss`` : Typed LSC load bindless intrinsic
## ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
## * arg0: vNi1, Predicate (overloaded)
## * arg1: vNi8, Cache controls, where N is the number of supported cache levels [MBC]
## * arg2: i8, Channel mask [MBC]
## * arg3: i32, Surface BSS
## * arg4: vNi32, U pixel indices (overloaded)
## * arg5: vNi32, V pixel indices
## * arg6: vNi32, R pixel indices
## * arg7: vNi32, LOD pixel indices
## * arg8: vector to take values for masked simd lanes from (load)
##         vector to take values to write (store)
##
## * Return value: the value read from memory (load) or void (store, prefetch)
##
    "lsc_load_quad_tgm_bss": { "result": "anyvector",
                               "arguments": [
                                   "anyint",    # vNxi1, predicate
                                   "anyvector", # cache controls
                                   "char",      # channel mask
                                   "int",       # i32 BSS
                                   "anyint",    # vNi32 U pixel index
                                   3,           # vNi32 V pixel index
                                   3,           # vNi32 R pixel index
                                   3,           # vNi32 LOD pixel index
                                   0,           # passthru value
                               ],
                               "target" : [
                                   "hasLSCMessages",
                                   "hasLSCTypedMessages",
                               ],
                               "attributes": "None",
                               "memory_effects":
                                   { "access": "Ref" }, },
    "lsc_store_quad_tgm_bss": { "result": "void",
                                "arguments": [
                                    "anyint",    # vNxi1, predicate
                                    "anyvector", # cache controls
                                    "char",      # channel mask
                                    "int",       # i32 BSS
                                    "anyint",    # vNi32 U pixel index
                                    2,           # vNi32 V pixel index
                                    2,           # vNi32 R pixel index
                                    2,           # vNi32 LOD pixel index
                                    "anyvector", # data to write
                                ],
                                "target" : [
                                    "hasLSCMessages",
                                    "hasLSCTypedMessages",
                                ],
                                "attributes": "None",
                                "memory_effects":
                                    { "access": "Mod" }, },
    "lsc_prefetch_quad_tgm_bss": { "result": "void",
                                   "arguments": [
                                       "anyint",    # vNxi1, predicate
                                       "anyvector", # cache controls
                                       "char",      # channel mask
                                       "int",       # i32 BSS
                                       "anyint",    # vNi32 U pixel index
                                       2,           # vNi32 V pixel index
                                       2,           # vNi32 R pixel index
                                       2,           # vNi32 LOD pixel index
                                   ],
                                   "target" : [
                                       "hasLSCMessages",
                                       "hasLSCTypedMessages",
                                   ],
                                   "attributes": "SideEffects", },


### ----------------------------
### Low-level sampler intrinsics
### ----------------------------

## ``llvm.vc.internal.sampler.load.bti.*`` : Sampler load intrinsic
## ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
##
## * arg0: vNi1 Predicate (overloaded)
## * arg1: i16, Opcode [MBC]
## * arg2: i8, Channel mask [MBC]
## * arg3: i16, Address offset packed immediates (aoffimmi) [MBC]
## * arg4: i32, Surface BTI
## * arg5: vector to take values for masked simd lanes from
## * arg6: vNi32 or vNi16, first sampler message parameter (overloaded)
## * arg7-arg14: sampler message parameters, same type as arg6
##
## * Return value: the value read from image (overloaded)
##
    "sampler_load_bti" : { "result" : "anyvector",
                           "arguments" : [
                               "anyint", # vNxi1, predicate
                               "short",  # opcode
                               "char",   # channel mask
                               "short",  # aoffimmi
                               "int",    # BTI
                               0,        # passthru
                               "anyint", # first sampler message parameter
                               2, 2, 2, 2, 2, 2, 2, 2, # sampler message parameters
                           ],
                           "target" : [
                               "hasSampler",
                           ],
                           "attributes" : "None",
                           "memory_effects":
                               { "access": "Ref" }, },

## ``llvm.vc.internal.sampler.load.predef.surface.*`` : Sampler load predefined surface intrinsic
## ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
##
## * arg0: vNi1 Predicate (overloaded)
## * arg1: i16, Opcode [MBC]
## * arg2: i8, Channel mask [MBC]
## * arg3: i16, Address offset packed immediates (aoffimmi) [MBC]
## * arg4: ptr predefined surface (overloaded)
## * arg5: vector to take values for masked simd lanes from
## * arg6: vNi32 or vNi16, first sampler message parameter (overloaded)
## * arg7-arg14: sampler message parameters, same type as arg6
##
## * Return value: the value read from image (overloaded)
##

    "sampler_load_predef_surface" : { "result" : "anyvector",
                                      "arguments" : [
                                          "anyint", # vNxi1, predicate
                                          "short",  # opcode
                                          "char",   # channel mask
                                          "short",  # aoffimmi
                                          "anyptr", # ptr predefined surface
                                          0,        # passthru
                                          "anyint", # first sampler message parameter
                                          3, 3, 3, 3, 3, 3, 3, 3, # sampler message parameters
                                      ],
                                      "target" : [
                                          "hasSampler",
                                      ],
                                      "attributes" : "None",
                                      "memory_effects":
                                          { "access": "Ref" }, },

## ``llvm.vc.internal.sample.bti.*`` : Sampler load intrinsic
## ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
##
## * arg0: vNi1 Predicate (overloaded)
## * arg1: i16, Opcode [MBC]
## * arg2: i8, Channel mask [MBC]
## * arg3: i16, Address offset packed immediates (aoffimmi) [MBC]
## * arg4: i32, Surface BTI
## * arg5: i32, Sampler Index
## * arg6: vector to take values for masked simd lanes from
## * arg7: vNi32 or vNi16, first sampler message parameter (overloaded)
## * arg8-arg18: sampler message parameters, same type as arg7
##
## * Return value: the value read from image (overloaded)
##
    "sample_bti" : { "result" : "anyvector",
                     "arguments" : [
                         "anyint", # vNxi1, predicate
                         "short",  # opcode
                         "char",   # channel mask
                         "short",  # aoffimmi
                         "int",    # BTI
                         "int",    # Sampler index
                         0,        # passthru
                         "anyint", # first sampler message parameter
                         2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, # sampler message parameters
                     ],
                     "target" : [
                         "hasSampler",
                     ],
                     "attributes" : "None",
                     "memory_effects":
                         { "access": "Ref" }, },

## ``llvm.vc.internal.sample.predef.surface.*`` : Sampler load predefined surface intrinsic
## ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
##
## * arg0: vNi1 Predicate (overloaded)
## * arg1: i16, Opcode [MBC]
## * arg2: i8, Channel mask [MBC]
## * arg3: i16, Address offset packed immediates (aoffimmi) [MBC]
## * arg4: ptr predefined surface (overloaded)
## * arg5: ptr predefined sampler (overloaded)
## * arg6: vector to take values for masked simd lanes from
## * arg7: vNi32 or vNi16, first sampler message parameter (overloaded)
## * arg8-arg18: sampler message parameters, same type as arg7
##
## * Return value: the value read from image (overloaded)
##
    "sample_predef_surface" : { "result" : "anyvector",
                                "arguments" : [
                                    "anyint", # vNxi1, predicate
                                    "short",  # opcode
                                    "char",   # channel mask
                                    "short",  # aoffimmi
                                    "anyptr", # ptr predefined surface
                                    "anyptr", # ptr predefined sampler
                                    0,        # passthru
                                    "anyint", # first sampler message parameter
                                    4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, # sampler message parameters
                                ],
                                "target" : [
                                    "hasSampler",
                                ],
                                "attributes" : "None",
                                "memory_effects":
                                    { "access": "Ref" }, },


### --------------------
### Thread ID intrinsics
### --------------------

## ``llvm.vc.internal.sync.buffer`` : read stateless pointer to sync buffer
## ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
## ``llvm.vc.internal.sync.buffer`` : read implicit arg sync buffer ptr
##
## * Return value: i64 address of sync buffer
##
    "sync_buffer" : { "result" : "long",
                      "arguments": [],
                      "attributes": "None",
                      "memory_effects":
                          { "access": "Ref" }, },

## ``llvm.vc.internal.logical.thread.id`` : logical global thread ID
## ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
##
## * Return value: i32 logical global thread ID within a gpu tile
##
    "logical_thread_id" : { "result": "int",
                            "arguments": [],
                            "attributes": "None",
                            "memory_effects":
                                { "access": "NoModRef" }, },

### ---------------------------
### Print and assert intrinsics
### ---------------------------

## ``llvm.vc.internal.assert.buffer`` : read stateless pointer to assert buffer
## ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
## ``llvm.vc.internal.assert.buffer`` : read implicit arg print assert ptr
##
## * return value: i64 address of assert buffer
##
    "assert_buffer" : { "result" : "long",
                        "arguments" : [],
                        "attributes" : "None",
                        "memory_effects":
                            { "access": "Ref" }, },

## ``llvm.vc.internal.print.buffer`` : read stateless pointer to print buffer
## ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
## ``llvm.vc.internal.print.buffer`` : read implicit arg print buffer ptr
##
## * return value: i64 address of print buffer
##
    "print_buffer" : { "result" : "long",
                       "arguments" : [],
                       "attributes" : "None",
                       "memory_effects":
                           { "access": "Ref" }, },

## ``llvm.vc.internal.print.format.index`` : add printf format string to collection
## ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
## ``llvm.vc.internal.print.format.index`` :  return index of printf format string
##
## * arg0: pointer for printf format string
##
## * Return value: the vector value read
##
    "print_format_index" : { "result" : "int",
                             "arguments" : ["anyptr"],
                             "attributes": "None",
                             "memory_effects":
                                 { "access": "NoModRef" }, },

## ``llvm.vc.internal.media.ld.predef.surface.*`` : legacy media load predefined surface
## ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
##
## * arg0: i32 modifiers, constant
## * arg1: ptr predefined surface (overloaded)
## * arg2: i32 plane, constant
## * arg3: i32 block width in bytes, constant
## * (block height inferred from return type size and block width)
## * arg4: i32 x byte offset
## * arg5: i32 y byte offset
##
## * Return value: the data read.
##
    "media_ld_predef_surface": { "result" : "anyvector",
                                 "arguments" : [
                                   "int",       # modifiers, constant
                                   "anyptr",    # ptr predefined surface
                                   "int",       # plane, constant
                                   "int",       # block width in bytes, constant
                                   "int",       # x byte offset
                                   "int"        # y byte offset
                                 ],
                                 "target" : [
                                     "!noLegacyDataport"
                                 ],
                                 "attributes" : "None",
                                 "memory_effects":
                                     { "access": "Ref" }, },
## ``llvm.vc.internal.media.st.predef.surface.*`` : legacy media store predefined surface
## ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
##
## * arg0: i32 modifiers, constant
## * arg1: ptr predefined surface (overloaded)
## * arg2: i32 plane, constant
## * arg3: i32 block width in bytes, constant
## * (block height inferred from data type size and block width)
## * arg4: i32 x byte offset
## * arg5: i32 y byte offset
## * arg6: data to write (overloaded)
##
    "media_st_predef_surface" : { "result" : "void",
                                  "arguments" : [
                                      "int",       # modifiers, constant
                                      "anyptr",    # ptr predefined surface
                                      "int",       # plane, constant
                                      "int",       # block width in bytes, constant
                                      "int",       # x byte offset
                                      "int",       # y byte offset
                                      "anyvector"  # data to write
                                  ],
                                  "target" : [
                                     "!noLegacyDataport"
                                  ],
                                  "attributes" : "None",
                                  "memory_effects":
                                      { "access": "Mod" }, },

## ``llvm.vc.internal.typed.atomic.*.predef.surface.*`` : legacy atomic typed predefined surface
## ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
## * (Exec_size inferred from element offset type)
## * arg0: vXi1 predicate (overloaded)
## * arg1: ptr predefined surface (overloaded)
## * arg2: vXT src
## * arg3: vXi32 u (overloaded)
## * arg4: vXi32 v - can be a constant 0 and becomes undef in lowering
## * arg5: vXi32 r - can be a constant 0 and becomes undef in lowering
## * arg6: vXi32 LOD - can be constant 0 and becomes undef in lowering
##
## * Return value: vXi32 the old value read
##
## Predicate, element offset, src, and the return value must all have the
## same vector width (which in reality must be 8)
##
    "typed_atomic_add_predef_surface" : { "result" : "anyvector",
                                          "arguments" : [
                                              "anyvector",  # predicate
                                              "anyptr",     # ptr predefined surface
                                              0,            # source
                                              "anyint",     # u coordinate
                                              3,            # v coordinate
                                              3,            # r coordinate
                                              3             # lod coordinate
                                            ],
                                          "target" : [
                                             "!noLegacyDataport"
                                          ],
                                          "attributes" : "SideEffects" },
    "typed_atomic_sub_predef_surface" : { "result" : "anyvector",
                                          "arguments" : [
                                              "anyvector",
                                              "anyptr",
                                              0,
                                              "anyint",
                                              3,
                                              3,
                                              3
                                            ],
                                          "target" : [
                                             "!noLegacyDataport"
                                          ],
                                          "attributes" : "SideEffects" },
    "typed_atomic_min_predef_surface" : { "result" : "anyvector",
                                          "arguments" : [
                                              "anyvector",
                                              "anyptr",
                                              0,
                                              "anyint",
                                              3,
                                              3,
                                              3
                                            ],
                                          "target" : [
                                             "!noLegacyDataport"
                                          ],
                                          "attributes" : "SideEffects" },
    "typed_atomic_max_predef_surface" : { "result" : "anyvector",
                                          "arguments" : [
                                              "anyvector",
                                              "anyptr",
                                              0,
                                              "anyint",
                                              3,
                                              3,
                                              3
                                            ],
                                          "target" : [
                                             "!noLegacyDataport"
                                          ],
                                          "attributes" : "SideEffects" },

    "typed_atomic_xchg_predef_surface" : { "result" : "anyvector",
                                          "arguments" : [
                                              "anyvector",
                                              "anyptr",
                                              0,
                                              "anyint",
                                              3,
                                              3,
                                              3
                                            ],
                                          "target" : [
                                             "!noLegacyDataport"
                                          ],
                                          "attributes" : "SideEffects" },
    "typed_atomic_and_predef_surface" : { "result" : "anyvector",
                                          "arguments" : [
                                              "anyvector",
                                              "anyptr",
                                              0,
                                              "anyint",
                                              3,
                                              3,
                                              3
                                            ],
                                          "target" : [
                                             "!noLegacyDataport"
                                          ],
                                          "attributes" : "SideEffects" },
    "typed_atomic_or_predef_surface" : { "result" : "anyvector",
                                          "arguments" : [
                                              "anyvector",
                                              "anyptr",
                                              0,
                                              "anyint",
                                              3,
                                              3,
                                              3
                                            ],
                                          "target" : [
                                             "!noLegacyDataport"
                                          ],
                                          "attributes" : "SideEffects" },


    "typed_atomic_xor_predef_surface" : { "result" : "anyvector",
                                          "arguments" : [
                                              "anyvector",
                                              "anyptr",
                                              0,
                                              "anyint",
                                              3,
                                              3,
                                              3
                                            ],
                                          "target" : [
                                             "!noLegacyDataport"
                                          ],
                                          "attributes" : "SideEffects" },
    "typed_atomic_imin_predef_surface" : { "result" : "anyvector",
                                          "arguments" : [
                                              "anyvector",
                                              "anyptr",
                                              0,
                                              "anyint",
                                              3,
                                              3,
                                              3
                                            ],
                                          "target" : [
                                             "!noLegacyDataport"
                                          ],
                                          "attributes" : "SideEffects" },
    "typed_atomic_imax_predef_surface" : { "result" : "anyvector",
                                          "arguments" : [
                                              "anyvector",
                                              "anyptr",
                                              0,
                                              "anyint",
                                              3,
                                              3,
                                              3
                                            ],
                                          "target" : [
                                             "!noLegacyDataport"
                                          ],
                                          "attributes" : "SideEffects" },

## ``llvm.vc.internal.typed.atomic.*.predef.surface.*`` : legacy atomic typed predefined surface
## ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
##
## * (Exec_size inferred from element offset type)
## * arg0: vXi1 predicate (overloaded)
## * arg1: ptr predefined surface (overloaded)
## * arg2: vXi32 u (overloaded)
## * arg3: vXi32 v - can be a constant 0 and becomes undef in lowering
## * arg4: vXi32 r - can be a constant 0 and becomes undef in lowering
## * arg5: vXi32 LOD - can be a constant 0 and becomes undef in lowering
##
## * Return value: vXi32 the old value read
##
## Predicate, element offset, src, and the return value must all have the
## same vector width (which in reality must be 8)
##
    "typed_atomic_inc_predef_surface" : { "result" : "anyvector",
                                          "arguments" : [
                                              "anyvector",  # predicate
                                              "anyptr",     # ptr predefined surface
                                              "anyint",     # u coordinate
                                              3,            # v coordinate
                                              3,            # r coordinate
                                              3             # lod coordinate
                                            ],
                                          "target" : [
                                              "!noLegacyDataport"
                                          ],
                                          "attributes" : "SideEffects" },
    "typed_atomic_dec_predef_surface" : { "result" : "anyvector",
                                          "arguments" : [
                                              "anyvector",  # predicate
                                              "anyptr",     # ptr predefined surface
                                              "anyint",     # u coordinate
                                              3,            # v coordinate
                                              3,            # r coordinate
                                              3             # lod coordinate
                                            ],
                                          "target" : [
                                              "!noLegacyDataport"
                                          ],
                                          "attributes" : "SideEffects" },

## ``llvm.vc.internal.typed.atomic.*.predef.surface.cmpxchg.*`` : legacy atomic typed CMPXCHG predefined surface
## ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
##
## * (Exec_size inferred from element offset type)
## * arg0: vXi1 predicate (overloaded)
## * arg1: ptr predefined surface (overloaded)
## * arg2: vXT src0
## * arg3: vXT src1
## * arg4: vXi32 u (overloaded)
## * arg5: vXi32 v - can be a constant 0 and becomes undef in lowering
## * arg6: vXi32 r - can be a constant 0 and becomes undef in lowering
## * arg7: vXi32 LOD - can be a constant 0 and becomes undef in lowering
##
## * Return value: vXi32 the old value read
##
## Predicate, element offset, src, and the return value must all have the
## same vector width (which in reality must be 8)
##
    "typed_atomic_cmpxchg_predef_surface" : { "result" : "anyvector",
                                              "arguments" : [
                                                  "anyvector",  # predicate
                                                  "anyptr",     # ptr predefined surface
                                                  0,            # src1
                                                  0,            # src2
                                                  "anyint",     # u coordinate
                                                  3,            # v coordinate
                                                  3,            # r coordinate
                                                  3             # lod coordinate
                                                ],
                                              "target" : [
                                                  "!noLegacyDataport"
                                              ],
                                              "attributes" : "SideEffects" },

## ``llvm.vc.internal.gather4.typed.predef.surface.*`` : legacy cmask typed load predefined surface
## ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
##
## * arg0: i32 channel mask, constant
## * arg1: vXi1 predicate (Num_elts inferred from element offset type) (overloaded)
## * arg2: ptr predefined surface (overloaded)
## * arg3: vXi32 U pixel address (overloaded)
## * arg4: vXi32 V pixel address
## * arg5: vXi32 R pixel address
## * arg6: old value of the data read
##
## * Return value: the data read
##

    "gather4_typed_predef_surface" : { "result" : "anyvector",
                                       "arguments" : [
                                           "int",       # cmask
                                           "anyvector", # predicate
                                           "anyptr",    # ptr predefined surface
                                           "anyvector", # u pixel address
                                           3,           # v pixel address
                                           3,           # r pixel address
                                           0            # passthru value
                                        ],
                                        "target" : [
                                            "!noLegacyDataport"
                                        ],
                                        "attributes" : "None",
                                        "memory_effects":
                                            { "access": "Ref" }, },
## ``llvm.vc.internal.scatter4.typed.predef.surface.*`` : legacy cmask typed store predefined surface
## ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
##
## * arg0: i32 channel mask, constant
## * arg1: vXi1 predicate (Num_elts inferred from U pixel address type) (overloaded)
## * arg2: ptr predefined surface (overloaded)
## * arg3: v8Xi32 U pixel address (overloaded)
## * arg4: v8Xi32 V pixel address
## * arg5: v8Xi32 R pixel address
## * arg6: data to write (overloaded)
##
    "scatter4_typed_predef_surface" : { "result" : "void",
                                        "arguments" : [
                                            "int",       # cmask
                                            "anyvector", # predicate
                                            "anyptr",    # ptr predefined surface
                                            "anyvector", # u pixel address
                                            2,           # v pixel address
                                            2,           # r pixel address
                                            "anyvector"  # data
                                        ],
                                        "target" : [
                                            "!noLegacyDataport"
                                        ],
                                        "attributes" : "None",
                                        "memory_effects":
                                            { "access": "Mod" }, },

## ``llvm.vc.internal.write.predef.sampler`` : write predefined sampler variable
## ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
##
## * arg0: ptr predefined surface variable
## * arg1: i32 value to write
##
## This corresponds to MOVS visa instruction and utilizes technique of using
## global variable in LLVM IR for predefined surfaces.
##
    "write_predef_sampler" : { "result": "void",
                               "arguments" : [
                                   "anyptr",
                                   "int"
                                ],
                                "target" : [
                                    "hasSampler",
                                ],
                                "attributes" : "None",
                                "memory_effects":
                                    { "access": "Mod" }, },

}
