#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: bsd-spec-analyze.py $
# pylint: disable=invalid-name,too-many-lines

"""
ARM BSD specification analyser.
"""

from __future__ import print_function;

__copyright__ = \
"""
Copyright (C) 2025 Oracle and/or its affiliates.

This file is part of VirtualBox base platform packages, as
available from https://www.virtualbox.org.

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, in version 3 of the
License.

This program is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, see <https://www.gnu.org/licenses>.

SPDX-License-Identifier: GPL-3.0-only
"""
__version__ = "$Revision: 169348 $"

# Standard python imports.
import argparse;
import collections;
import datetime;
import functools;
import hashlib;
import io;
import json;
import operator;
import os;
import re;
import sys;
import tarfile;
import time;
import traceback;
# profiling:
import cProfile;
import pstats


## Program start time for logging.
g_nsProgStart = int(time.time_ns())

def numToStr1000Sep(iNumber):
    """ Formats iNumber with spaces as thousand separators. """
    sStr   = '%d' % (iNumber,)
    off    = len(sStr);
    offEnd = sStr[0] == '-';
    sRet   = '';
    while True:
        if off - offEnd <= 3:
            return sStr[:off] + sRet;
        sRet = ' ' + sStr[off - 3 : off] + sRet;
        off -= 3;

def nsElapsedAsStr(nsStart):
    """ Given a start timestamp, calculates and formats the number of nanoseconds elapsed. """
    return numToStr1000Sep(time.time_ns() - nsStart);


## Mapping from ARM FEAT_xxxx to CPUMFEATURESARMV8 member.
#
# Sed script for extracting this stuff from cpum.h (-n option, sort output):
#
# # Match all comment lines with a (FEAT_XXX) in them.
# /[ (]FEAT_[A-Zn].*\*\//!d
#
# # Extract the feature string, quote it for dict key and pad to the value column.
# s,^.*[ (]\(FEAT_[A-Zn][^ )]*\)[ )].*\*\/.*$,'\1':,
# :morepadding
# s/$/ /
# /^................................/!b morepadding
#
# # Load the next line with the value, extract the member name and quote it for dict value.
# N
# s/\n *uint32_t  *\(f[a-zA-Z0-9][a-zA-Z0-9_]*\) * :.*$/'\1',/
# p
#
g_dSpecFeatToCpumFeat = {
    # Generated by sed + sort:
    'FEAT_AA32':                    'fAa32',
    'FEAT_AA32BF16':                'fAa32Bf16',
    'FEAT_AA32EL0':                 'fAa32El0',
    'FEAT_AA32EL1':                 'fAa32El1',
    'FEAT_AA32EL2':                 'fAa32El2',
    'FEAT_AA32EL3':                 'fAa32El3',
    'FEAT_AA32HPD':                 'fAa32Hpd',
    'FEAT_AA32I8MM':                'fAa32I8mm',
    'FEAT_AA64':                    'fAa64',
    'FEAT_AA64EL0':                 'fAa64El0',
    'FEAT_AA64EL1':                 'fAa64El1',
    'FEAT_AA64EL2':                 'fAa64El2',
    'FEAT_AA64EL3':                 'fAa64El3',
    'FEAT_ABLE':                    'fAble',
    'FEAT_ADERR':                   'fAderr',
    'FEAT_AdvSIMD':                 'fAdvSimd',
    'FEAT_AES':                     'fAes',
    'FEAT_AFP':                     'fAfp',
    'FEAT_AIE':                     'fAie',
    'FEAT_AMUv1':                   'fAmuV1',
    'FEAT_AMUv1p1':                 'fAmuV1p1',
    'FEAT_ANERR':                   'fAnerr',
    'FEAT_ASID16':                  'fAsid16',
    'FEAT_ASID2':                   'fAsid2',
    'FEAT_ATS1A':                   'fATs1a',
    'FEAT_BBM':                     'fBbm',
    'FEAT_BF16':                    'fBf16',
    'FEAT_BRBE':                    'fBrbe',
    'FEAT_BRBEv1p1':                'fBrbeV1p1',
    'FEAT_BTI':                     'fBti',
    'FEAT_BWE':                     'fBwe',
    'FEAT_BWE2':                    'fBwe2',
    'FEAT_CCIDX':                   'fCcidx',
    'FEAT_CHK':                     'fChk',
    'FEAT_CLRBHB':                  'fClrBhb',
    'FEAT_CMOW':                    'fCmow',
    'FEAT_CMPBR':                   'fCmpBr',
    #'FEAT_CNTSC':                   'fCntsc',              # needs external registers for detection
    'FEAT_CONSTPACFIELD':           'fConstPacField',
    #'FEAT_CP15SDISABLE2':           'fCp15SDisable2', - can't be detected?
    'FEAT_CPA':                     'fCpa',
    'FEAT_CPA2':                    'fCpa2',
    'FEAT_CRC32':                   'fCrc32',
    'FEAT_CSSC':                    'fCssc',
    'FEAT_CSV2':                    'fCsv2',
    'FEAT_CSV2_1p1':                'fCsv21p1',
    'FEAT_CSV2_1p2':                'fCsv21p2',
    'FEAT_CSV2_2':                  'fCvs2_2',
    'FEAT_CSV2_3':                  'fCsv2v3',
    'FEAT_CSV3':                    'fCsv3',
    'FEAT_D128':                    'fD128',
    'FEAT_Debugv8p1':               'fDebugV8p1',
    'FEAT_Debugv8p2':               'fDebugV8p2',
    'FEAT_Debugv8p4':               'fDebugV8p4',
    'FEAT_Debugv8p8':               'fDebugV8p8',
    'FEAT_Debugv8p9':               'fDebugV8p9',
    'FEAT_DGH':                     'fDgh',
    'FEAT_DIT':                     'fDit',
    #'FEAT_DoPD':                    'fDopd',               # needs external registers for detection
    'FEAT_DotProd':                 'fDotProd',
    'FEAT_DoubleFault':             'fDoubleFault',
    'FEAT_DoubleFault2':            'fDoubleFault2',
    'FEAT_DoubleLock':              'fDoubleLock',
    'FEAT_DPB':                     'fDpb',
    'FEAT_DPB2':                    'fDpb2',
    'FEAT_E0PD':                    'fE0Pd',
    'FEAT_E2H0':                    'fE2H0',
    'FEAT_E3DSE':                   'fE3Dse',
    'FEAT_EBEP':                    'fEbep',
    'FEAT_EBF16':                   'fEbf16',
    'FEAT_ECBHB':                   'fEcBhb',
    'FEAT_ECV':                     'fEcv',
    'FEAT_ECV_POFF':                'fEcvPOff',
    #'FEAT_EDHSR':                   'fEdhsr',              # needs external registers for detection
    'FEAT_EPAC':                    'fEpac',
    'FEAT_ETE':                     'fEte',
    'FEAT_ETEv1p1':                 'fEteV1p1',
    'FEAT_ETEv1p2':                 'fEteV1p2',
    'FEAT_ETEv1p3':                 'fEteV1p3',
    'FEAT_ETMv4':                   'fEtmV4',
    'FEAT_ETMv4p1':                 'fEtmV4p1',
    'FEAT_ETMv4p2':                 'fEtmV4p2',
    'FEAT_ETMv4p3':                 'fEtmV4p3',
    'FEAT_ETMv4p4':                 'fEtmV4p4',
    'FEAT_ETMv4p5':                 'fEtmV4p5',
    'FEAT_ETMv4p6':                 'fEtmV4p6',
    'FEAT_ETS2':                    'fEts2',
    'FEAT_ETS3':                    'fEts3',
    'FEAT_EVT':                     'fEvt',
    'FEAT_ExS':                     'fExs',
    'FEAT_F32MM':                   'fF32mm',
    'FEAT_F64MM':                   'fF64mm',
    'FEAT_F8F16MM':                 'fF8F16mm',
    'FEAT_F8F32MM':                 'fF8F32mm',
    'FEAT_FAMINMAX':                'fFaMinMax',
    'FEAT_FCMA':                    'fFcma',
    'FEAT_FGT':                     'fFgt',
    'FEAT_FGT2':                    'fFgt2',
    'FEAT_FGWTE3':                  'fFgwtE3',
    'FEAT_FHM':                     'fFhm',
    'FEAT_FlagM':                   'fFlagM',
    'FEAT_FlagM2':                  'fFlagM2',
    'FEAT_FP':                      'fFp',
    'FEAT_FP16':                    'fFp16',
    'FEAT_FP8':                     'fFp8',
    'FEAT_FP8DOT2':                 'fFp8Dot2',
    'FEAT_FP8DOT4':                 'fFp8Dot4',
    'FEAT_FP8FMA':                  'fFp8Fma',
    'FEAT_FPAC':                    'fFpac',
    'FEAT_FPACC_SPEC':              'fFpaccSpec',
    'FEAT_FPACCOMBINE':             'fFpacCombine',
    'FEAT_FPMR':                    'fFpmr',
    'FEAT_FPRCVT':                  'fFpRcvt',
    'FEAT_FRINTTS':                 'fFrintts',
    'FEAT_GCS':                     'fGcs',
    'FEAT_GICv3':                   'fGicV3',
    'FEAT_GICv3_NMI':               'fGicV3Nmi',
    'FEAT_GICv3_TDIR':              'fGicV3Tdir',
    'FEAT_GICv3p1':                 'fGicV3p1',
    'FEAT_GICv4':                   'fGicV4',
    'FEAT_GICv4p1':                 'fGicV4p1',
    'FEAT_GTG':                     'fGtg',
    'FEAT_HACDBS':                  'fHacdbs',
    'FEAT_HAFDBS':                  'fHafdbs',
    'FEAT_HAFT':                    'fHaft',
    'FEAT_HBC':                     'fHbc',
    'FEAT_HCX':                     'fHcx',
    'FEAT_HDBSS':                   'fHdbss',
    'FEAT_HPDS':                    'fHpds',
    'FEAT_HPDS2':                   'fHpds2',
    'FEAT_HPMN0':                   'fHpmn0',
    'FEAT_I8MM':                    'fI8mm',
    'FEAT_IDST':                    'fIdst',
    'FEAT_IDTE3':                   'fIdte3',
    'FEAT_IESB':                    'fIesb',
    'FEAT_ITE':                     'fIte',
    'FEAT_IVIPT':                   'fIvipt',
    'FEAT_JSCVT':                   'fJscvt',
    'FEAT_LOR':                     'fLor',
    'FEAT_LPA':                     'fLpa',
    'FEAT_LPA2':                    'fLpa2',
    'FEAT_LRCPC':                   'fLrcpc',
    'FEAT_LRCPC2':                  'fLrcpc2',
    'FEAT_LRCPC3':                  'fLrcpc3',
    'FEAT_LS64':                    'fLs64',
    'FEAT_LS64_ACCDATA':            'fLs64Accdata',
    'FEAT_LS64_V':                  'fLs64V',
    'FEAT_LS64WB':                  'fLs64WB',
    'FEAT_LSE':                     'fLse',
    'FEAT_LSE128':                  'fLse128',
    'FEAT_LSE2':                    'fLse2',
    'FEAT_LSFE':                    'fLsfe',
    'FEAT_LSMAOC':                  'fLsmaoc',
    'FEAT_LSUI':                    'fLsui',
    'FEAT_LUT':                     'fLut',
    'FEAT_LVA':                     'fLva',
    'FEAT_LVA3':                    'fLva3',
    'FEAT_MEC':                     'fMec',
    'FEAT_MixedEnd':                'fMixedEnd',
    'FEAT_MixedEndEL0':             'fMixedEndEl0',
    'FEAT_MOPS':                    'fMops',
    'FEAT_MPAM':                    'fMpam',
    #'FEAT_MPAM_MSC_DCTRL':          'fMpamMscDCtrl',       # needs external registers for detection
    #'FEAT_MPAM_MSC_DOMAINS':        'fMpamMscDomains',     # needs external registers for detection
    'FEAT_MPAM_PE_BW_CTRL':         'fMpamPeBwCtrl',
    'FEAT_MPAMv0p1':                'fMpamV0p1',
    'FEAT_MPAMv1p1':                'fMpamV1p1',
    'FEAT_MTE':                     'fMte',
    'FEAT_MTE_ASYM_FAULT':          'fMteAsymFault',
    'FEAT_MTE_ASYNC':               'fMteAsync',
    'FEAT_MTE_CANONICAL_TAGS':      'fMteCanonicalTags',
    'FEAT_MTE_NO_ADDRESS_TAGS':     'fMteNoAddressTags',
    'FEAT_MTE_PERM':                'fMtePerm',
    #'FEAT_MTE_PERM_S1':             'fMtePermS1', # removed?
    'FEAT_MTE_STORE_ONLY':          'fMteStoreOnly',
    'FEAT_MTE_TAGGED_FAR':          'fMteTaggedFar',
    'FEAT_MTE2':                    'fMte2',
    'FEAT_MTE3':                    'fMte3',
    'FEAT_MTE4':                    'fMte4',
    'FEAT_MTPMU':                   'fMtPmu',
    'FEAT_NMI':                     'fNmi',
    'FEAT_nTLBPA':                  'fNTlbpa',
    'FEAT_NV':                      'fNv',
    'FEAT_NV2':                     'fNv2',
    'FEAT_NV2p1':                   'fNV2p1',
    'FEAT_OCCMO':                   'fOccmo',
    'FEAT_PACIMP':                  'fPacImp',
    'FEAT_PACQARMA3':               'fPacQarma3',
    'FEAT_PACQARMA5':               'fPacQarma5',
    'FEAT_PAN':                     'fPan',
    'FEAT_PAN2':                    'fPan2',
    'FEAT_PAN3':                    'fPan3',
    'FEAT_PAuth':                   'fPAuth',
    'FEAT_PAuth_LR':                'fPAuthLR',
    'FEAT_PAuth2':                  'fPAuth2',
    'FEAT_PCDPHINT':                'fPCDPHint',
    #'FEAT_PCSRv8':                  'fPcsrV8',             # needs external registers for detection
    #'FEAT_PCSRv8p2':                'fPcsrV8p2',           # needs external registers for detection
    #'FEAT_PCSRv8p9':                'fPcsrV8p9',           # needs external registers for detection
    'FEAT_PFAR':                    'fPfar',
    'FEAT_PMULL':                   'fPmull',
    'FEAT_PMUv3':                   'fPmuV3',
    'FEAT_PMUv3_EDGE':              'fPmuV3Edge',
    #'FEAT_PMUv3_EXT':               'fPmuV3Ext',           # difficult to detect
    #'FEAT_PMUv3_EXT32':             'fPmuV3Ext32',         # difficult to detect
    #'FEAT_PMUv3_EXT64':             'fPmuV3Ext64',         # difficult to detect
    'FEAT_PMUv3_ICNTR':             'fPmuV3Icntr',
    'FEAT_PMUv3_SME':               'fPmuV3Sme',
    'FEAT_PMUv3_SS':                'fPmuV3Ss',
    'FEAT_PMUv3_TH':                'fPmuV3Th',
    'FEAT_PMUv3_TH2':               'fPmuV3Th2',
    'FEAT_PMUv3p1':                 'fPmuV3p1',
    'FEAT_PMUv3p4':                 'fPmuV3p4',
    'FEAT_PMUv3p5':                 'fPmuV3p5',
    'FEAT_PMUv3p7':                 'fPmuV3p7',
    'FEAT_PMUv3p8':                 'fPmuV3p8',
    'FEAT_PMUv3p9':                 'fPmuV3p9',
    'FEAT_PRFMSLC':                 'fPrfmSlc',
    'FEAT_RAS':                     'fRas',
    #'FEAT_RASSAv1p1':               'fRassaV1p1',          # difficult to detect
    #'FEAT_RASSAv2':                 'fRasSaV2',            # difficult to detect
    'FEAT_RASv1p1':                 'fRasV1p1',
    'FEAT_RASv2':                   'fRasV2',
    'FEAT_RDM':                     'fRdm',
    'FEAT_RME':                     'fRme',
    'FEAT_RME_GDI':                 'fRmeGdi',
    'FEAT_RME_GPC2':                'fRmeGpc2',
    'FEAT_RME_GPC3':                'fRmeGpc3',
    'FEAT_RNG':                     'fRng',
    'FEAT_RNG_TRAP':                'fRngTrap',
    'FEAT_RPRES':                   'fRpres',
    'FEAT_RPRFM':                   'fRprfm',
    'FEAT_S1PIE':                   'fS1Pie',
    'FEAT_S1POE':                   'fS1Poe',
    'FEAT_S2FWB':                   'fS2Fwb',
    'FEAT_S2PIE':                   'fS2Pie',
    'FEAT_S2POE':                   'fS2Poe',
    'FEAT_S2TGran16K':              'fS2TGran16K',
    'FEAT_S2TGran4K':               'fS2TGran4K',
    'FEAT_S2TGran64K':              'fS2TGran64K',
    'FEAT_SB':                      'fSb',
    'FEAT_SCTLR2':                  'fSctlr2',
    'FEAT_SEBEP':                   'fSebep',
    'FEAT_SEL2':                    'fSecEl2',
    'FEAT_SHA1':                    'fSha1',
    'FEAT_SHA256':                  'fSha256',
    'FEAT_SHA3':                    'fSha3',
    'FEAT_SHA512':                  'fSha512',
    'FEAT_SM3':                     'fSm3',
    'FEAT_SM4':                     'fSm4',
    'FEAT_SME':                     'fSme',
    'FEAT_SME_B16B16':              'fSmeB16B16',
    'FEAT_SME_F16F16':              'fSmeF16F16',
    'FEAT_SME_F64F64':              'fSmeF64F64',
    'FEAT_SME_F8F16':               'fSmeF8F16',
    'FEAT_SME_F8F32':               'fSmeF8F32',
    'FEAT_SME_FA64':                'fSmeFA64',
    'FEAT_SME_I16I64':              'fSmeI16I64',
    'FEAT_SME_LUTv2':               'fSmeLutv2',
    'FEAT_SME_MOP4':                'fSmeMop4',
    'FEAT_SME_TMOP':                'fSmeTmop',
    'FEAT_SME2':                    'fSme2',
    'FEAT_SME2p1':                  'fSme2p1',
    'FEAT_SME2p2':                  'fSme2p2',
    'FEAT_SPE':                     'fSpe',
    'FEAT_SPE_ALTCLK':              'fSpeAltClk',
    'FEAT_SPE_CRR':                 'fSpeCrr',
    'FEAT_SPE_DPFZS':               'fSpeDpfzs',
    'FEAT_SPE_EFT':                 'fSpeEft',
    'FEAT_SPE_EXC':                 'fSpeExc',
    'FEAT_SPE_FDS':                 'fSpeFds',
    'FEAT_SPE_FnE':                 'fSpeFnE',
    'FEAT_SPE_FPF':                 'fSpeFpf',
    'FEAT_SPE_nVM':                 'fSpeNvm',
    'FEAT_SPE_PBT':                 'fSpePbt',
    'FEAT_SPE_SME':                 'fSpeSme',
    'FEAT_SPECRES':                 'fSpecres',
    'FEAT_SPECRES2':                'fSpecres2',
    'FEAT_SpecSEI':                 'fSpecSei',
    'FEAT_SPEv1p1':                 'fSpeV1p1',
    'FEAT_SPEv1p2':                 'fSpeV1p2',
    'FEAT_SPEv1p3':                 'fSpeV1p3',
    'FEAT_SPEv1p4':                 'fSpeV1p4',
    'FEAT_SPEv1p5':                 'fSpev1p5',
    'FEAT_SPMU':                    'fSpmu',
    'FEAT_SPMU2':                   'fSpmu2',
    'FEAT_SSBS':                    'fSsbs',
    'FEAT_SSBS2':                   'fSsbs2',
    'FEAT_SSVE_AES':                'fSsveAes',
    'FEAT_SSVE_BitPerm':            'fSsveBitPerm',
    'FEAT_SSVE_FEXPA':              'fSsveFexpa',
    'FEAT_SSVE_FP8DOT2':            'fSsveFp8Dot2',
    'FEAT_SSVE_FP8DOT4':            'fSsveFp8Dot4',
    'FEAT_SSVE_FP8FMA':             'fSsveFp8Fma',
    'FEAT_STEP2':                   'fStep2',
    'FEAT_SVE':                     'fSve',
    'FEAT_SVE_AES':                 'fSveAes',
    'FEAT_SVE_AES2':                'fSveAes2',
    'FEAT_SVE_B16B16':              'fSveB16B16',
    'FEAT_SVE_BFSCALE':             'fSveBfscale',
    'FEAT_SVE_BitPerm':             'fSveBitPerm',
    'FEAT_SVE_F16F32MM':            'fSveF16F32mm',
    'FEAT_SVE_PMULL128':            'fSvePmull128',
    'FEAT_SVE_SHA3':                'fSveSha3',
    'FEAT_SVE_SM4':                 'fSveSm4',
    'FEAT_SVE2':                    'fSve2',
    'FEAT_SVE2p1':                  'fSve2p1',
    'FEAT_SVE2p2':                  'fSve2p2',
    'FEAT_SYSINSTR128':             'fSysInstr128',
    'FEAT_SYSREG128':               'fSysReg128',
    'FEAT_TCR2':                    'fTcr2',
    'FEAT_TGran16K':                'fTGran16K',
    'FEAT_TGran4K':                 'fTGran4K',
    'FEAT_TGran64K':                'fTGran64K',
    'FEAT_THE':                     'fThe',
    'FEAT_TIDCP1':                  'fTidcp1',
    'FEAT_TLBIOS':                  'fTlbios',
    'FEAT_TLBIRANGE':               'fTlbirange',
    'FEAT_TLBIW':                   'fTlbiW',
    'FEAT_TME':                     'fTme',
    'FEAT_TRBE':                    'fTrbe',
    'FEAT_TRBE_EXC':                'fTrbeExc',
    'FEAT_TRBE_EXT':                'fTrbeExt',
    'FEAT_TRBE_MPAM':               'fTrbeMpam',
    'FEAT_TRBEv1p1':                'fTrbev1p1',
    'FEAT_TRC_SR':                  'fTrcSr',
    'FEAT_TRF':                     'fTrf',
    'FEAT_TTCNP':                   'fTtcnp',
    'FEAT_TTL':                     'fTtl',
    'FEAT_TTST':                    'fTtst',
    'FEAT_TWED':                    'fTwed',
    'FEAT_UAO':                     'fUao',
    'FEAT_UINJ':                    'fUinj',
    'FEAT_VHE':                     'fVhe',
    'FEAT_VMID16':                  'fVmid16',
    'FEAT_VPIPT':                   'fVpipt',
    'FEAT_WFxT':                    'fWfxt',
    'FEAT_XNX':                     'fXnx',
    'FEAT_XS':                      'fXs',

    ## @todo
    'FEAT_SRMASK':                  False,
};


#
# Misc Helpers for decoding ARM specs and JSON.
#
def assertJsonAttribsInSet(dJson, oAttribSet):
    """ Checks that the JSON element has all the attributes in the set and nothing else. """
    #assert set(dJson) == oAttribSet, '%s - %s' % (set(dJson) ^ oAttribSet, dJson,);
    assert len(dJson) == len(oAttribSet) and sum(sKey in oAttribSet for sKey in dJson), \
           '%s - %s' % (set(dJson) ^ oAttribSet, dJson,);


#
# The ARM instruction AST stuff.
#

class ArmAstBase(object):
    """
    ARM instruction AST base class.
    """

    ksTypeAssignment    = 'AST.Assignment';
    ksTypeBinaryOp      = 'AST.BinaryOp';
    ksTypeBool          = 'AST.Bool';
    ksTypeConcat        = 'AST.Concat';
    ksTypeDotAtom       = 'AST.DotAtom';
    ksTypeFunction      = 'AST.Function';
    ksTypeIdentifier    = 'AST.Identifier';
    ksTypeInteger       = 'AST.Integer';
    ksTypeReturn        = 'AST.Return';
    ksTypeSet           = 'AST.Set';
    ksTypeSlice         = 'AST.Slice';
    ksTypeSquareOp      = 'AST.SquareOp';
    ksTypeType          = 'AST.Type';
    ksTypeTypeAnnotation= 'AST.TypeAnnotation';
    ksTypeTuple         = 'AST.Tuple';
    ksTypeUnaryOp       = 'AST.UnaryOp';
    ksTypeValue         = 'Values.Value';
    ksTypeEquationValue = 'Values.EquationValue';
    ksTypeValuesGroup   = 'Values.Group';
    ksTypeField         = 'Types.Field';
    ksTypeString        = 'Types.String';
    ksTypeRegisterType  = 'Types.RegisterType';

    ksModeAccessor      = 'accessor';
    ksModeAccessorCond  = 'accessor-cond';
    ksModeConditions    = 'condition';
    ksModeConstraints   = 'contraints';
    ksModeValuesOnly    = 'values-only'; # Value and EquationValue

    def __init__(self, sType):
        self.sType = sType;

    kAttribSetBinaryOp = frozenset(['_type', 'left', 'op', 'right']);
    @staticmethod
    def fromJsonBinaryOp(oJson, sMode):
        assert sMode != ArmAstBase.ksModeValuesOnly;
        assertJsonAttribsInSet(oJson, ArmAstBase.kAttribSetBinaryOp);
        return ArmAstBinaryOp(ArmAstBase.fromJson(oJson['left'], sMode),
                              oJson['op'],
                              ArmAstBase.fromJson(oJson['right'], sMode),
                              sMode == ArmAstBase.ksModeConstraints);

    kAttribSetUnaryOp = frozenset(['_type', 'op', 'expr']);
    @staticmethod
    def fromJsonUnaryOp(oJson, sMode):
        assert sMode != ArmAstBase.ksModeValuesOnly;
        assertJsonAttribsInSet(oJson, ArmAstBase.kAttribSetUnaryOp);
        return ArmAstUnaryOp(oJson['op'], ArmAstBase.fromJson(oJson['expr'], sMode));

    kAttribSetSlice = frozenset(['_type', 'left', 'right']);
    @staticmethod
    def fromJsonSlice(oJson, sMode):
        assert sMode in (ArmAstBase.ksModeAccessor, ArmAstBase.ksModeAccessorCond);
        assertJsonAttribsInSet(oJson, ArmAstBase.kAttribSetSlice);
        return ArmAstSlice(ArmAstBase.fromJson(oJson['left'], sMode), ArmAstBase.fromJson(oJson['right'], sMode));

    kAttribSetSquareOp = frozenset(['_type', 'var', 'arguments']);
    @staticmethod
    def fromJsonSquareOp(oJson, sMode):
        assert sMode != ArmAstBase.ksModeValuesOnly;
        assertJsonAttribsInSet(oJson, ArmAstBase.kAttribSetSquareOp);
        return ArmAstSquareOp(ArmAstBase.fromJson(oJson['var'], sMode),
                              [ArmAstBase.fromJson(oArg, sMode) for oArg in oJson['arguments']]);

    kAttribSetTuple = frozenset(['_type', 'values']);
    @staticmethod
    def fromJsonTuple(oJson, sMode):
        assert sMode == ArmAstBase.ksModeAccessor;
        assertJsonAttribsInSet(oJson, ArmAstBase.kAttribSetTuple);
        return ArmAstTuple([ArmAstBase.fromJson(oArg, sMode) for oArg in oJson['values']]);

    kAttribSetDotAtom = frozenset(['_type', 'values']);
    @staticmethod
    def fromJsonDotAtom(oJson, sMode):
        assert sMode in (ArmAstBase.ksModeConstraints, ArmAstBase.ksModeAccessor, ArmAstBase.ksModeAccessorCond);
        assertJsonAttribsInSet(oJson, ArmAstBase.kAttribSetDotAtom);
        return ArmAstDotAtom([ArmAstBase.fromJson(oArg, sMode) for oArg in oJson['values']]);

    kAttribSetConcat = frozenset(['_type', 'values']);
    @staticmethod
    def fromJsonConcat(oJson, sMode):
        assert sMode != ArmAstBase.ksModeValuesOnly;
        assertJsonAttribsInSet(oJson, ArmAstBase.kAttribSetConcat);
        return ArmAstConcat([ArmAstBase.fromJson(oArg, sMode) for oArg in oJson['values']]);

    kAttribSetFunction = frozenset(['_type', 'name', 'arguments']);
    @staticmethod
    def fromJsonFunction(oJson, sMode):
        assert sMode != ArmAstBase.ksModeValuesOnly;
        assertJsonAttribsInSet(oJson, ArmAstBase.kAttribSetFunction);
        return ArmAstFunction(oJson['name'], [ArmAstBase.fromJson(oArg, sMode) for oArg in oJson['arguments']]);

    kAttribSetIdentifier = frozenset(['_type', 'value']);
    @staticmethod
    def fromJsonIdentifier(oJson, sMode):
        assert sMode != ArmAstBase.ksModeValuesOnly;
        assertJsonAttribsInSet(oJson, ArmAstBase.kAttribSetIdentifier);
        return ArmAstIdentifier(oJson['value'], sMode == ArmAstBase.ksModeConstraints);

    kAttribSetBool = frozenset(['_type', 'value']);
    @staticmethod
    def fromJsonBool(oJson, sMode):
        assert sMode != ArmAstBase.ksModeValuesOnly;
        assertJsonAttribsInSet(oJson, ArmAstBase.kAttribSetBool);
        return ArmAstBool(oJson['value']);

    kAttribSetInteger = frozenset(['_type', 'value']);
    @staticmethod
    def fromJsonInteger(oJson, sMode):
        assert sMode != ArmAstBase.ksModeValuesOnly;
        assertJsonAttribsInSet(oJson, ArmAstBase.kAttribSetInteger);
        return ArmAstInteger(oJson['value']);

    kAttribSetSet = frozenset(['_type', 'values']);
    @staticmethod
    def fromJsonSet(oJson, sMode):
        assert sMode != ArmAstBase.ksModeValuesOnly;
        assertJsonAttribsInSet(oJson, ArmAstBase.kAttribSetSet);
        return ArmAstSet([ArmAstBase.fromJson(oArg, sMode) for oArg in oJson['values']]);

    kAttribSetValue = frozenset(['_type', 'value', 'meaning']);
    @staticmethod
    def fromJsonValue(oJson, sMode):
        _ = sMode;
        assertJsonAttribsInSet(oJson, ArmAstBase.kAttribSetValue);
        return ArmAstValue(oJson['value']);

    kAttribSetEquationValue      = frozenset(['_type', 'value', 'meaning', 'slice']);
    kAttribSetEquationValueRange = frozenset(['_type', 'start', 'width']);
    @staticmethod
    def fromJsonEquationValue(dJson, sMode):
        assert sMode == ArmAstBase.ksModeValuesOnly;
        assertJsonAttribsInSet(dJson, ArmAstBase.kAttribSetEquationValue);
        assert len(dJson['slice']) == 1;
        dSlice = dJson['slice'][0];
        assert dSlice['_type'] == 'Range';
        assertJsonAttribsInSet(dSlice, ArmAstBase.kAttribSetEquationValueRange);
        return ArmAstEquationValue(dJson['value'], int(dSlice['start']), int(dSlice['width']));

    kAttribSetValuesGroup = frozenset(['_type', 'value', 'meaning', 'values']);
    @staticmethod
    def fromJsonValuesGroup(dJson, sMode):
        assert sMode == ArmAstBase.ksModeValuesOnly;
        assertJsonAttribsInSet(dJson, ArmAstBase.kAttribSetValuesGroup);
        assert dJson['values']['_type'] == 'Valuesets.Values';
        assert len(dJson['values']['values']) == 0;
        return ArmAstValuesGroup(dJson['value']);

    kAttribSetField = frozenset(['_type', 'value']);
    @staticmethod
    def fromJsonString(oJson, sMode):
        assert sMode in (ArmAstBase.ksModeConstraints, # Seen in register as 'input' to ImpDefBool.
                         ArmAstBase.ksModeAccessorCond);
        assertJsonAttribsInSet(oJson, ArmAstBase.kAttribSetField);
        return ArmAstString(oJson['value']);

    kAttribSetField = frozenset(['_type', 'value']);
    kAttribSetFieldValue = frozenset(['field', 'name', 'state', 'instance', 'slices']);
    @staticmethod
    def fromJsonField(oJson, sMode):
        assert sMode in (ArmAstBase.ksModeConstraints, ArmAstBase.ksModeAccessor, ArmAstBase.ksModeAccessorCond);
        assertJsonAttribsInSet(oJson, ArmAstBase.kAttribSetField);
        dJsonValue = oJson['value'];
        assertJsonAttribsInSet(dJsonValue, ArmAstBase.kAttribSetFieldValue);
        return ArmAstField(dJsonValue['field'], dJsonValue['name'], dJsonValue['state'],
                           dJsonValue['slices'], dJsonValue['instance']);

    kAttribSetRegisterType = frozenset(['_type', 'value']);
    kAttribSetRegisterTypeValue = frozenset(['name', 'state', 'instance', 'slices']);
    @staticmethod
    def fromJsonRegisterType(oJson, sMode):
        assert sMode in (ArmAstBase.ksModeConstraints, ArmAstBase.ksModeAccessorCond);
        assertJsonAttribsInSet(oJson, ArmAstBase.kAttribSetRegisterType);
        dJsonValue = oJson['value'];
        assertJsonAttribsInSet(dJsonValue, ArmAstBase.kAttribSetRegisterTypeValue);
        return ArmAstRegisterType(dJsonValue['name'], dJsonValue['state'], dJsonValue['slices'], dJsonValue['instance']);

    kAttribSetType = frozenset(['_type', 'name']);
    @staticmethod
    def fromJsonType(dJson, sMode):
        assert sMode == ArmAstBase.ksModeAccessor;
        assertJsonAttribsInSet(dJson, ArmAstBase.kAttribSetType);
        return ArmAstType(ArmAstBase.fromJson(dJson['name'], sMode));

    kAttribSetTypeAnnotation = frozenset(['_type', 'type', 'var']);
    @staticmethod
    def fromJsonTypeAnnotation(dJson, sMode):
        assert sMode == ArmAstBase.ksModeAccessor;
        assertJsonAttribsInSet(dJson, ArmAstBase.kAttribSetTypeAnnotation);
        return ArmAstTypeAnnotation(ArmAstBase.fromJson(dJson['var'], sMode), ArmAstBase.fromJson(dJson['type'], sMode));

    kAttribSetAssignment = frozenset(['_type', 'val', 'var']);
    @staticmethod
    def fromJsonAssignment(dJson, sMode):
        assert sMode == ArmAstBase.ksModeAccessor;
        assertJsonAttribsInSet(dJson, ArmAstBase.kAttribSetAssignment);
        return ArmAstAssignment(ArmAstBase.fromJson(dJson['var'], sMode), ArmAstBase.fromJson(dJson['val'], sMode));

    kAttribSetReturn = frozenset(['_type', 'val']);
    @staticmethod
    def fromJsonReturn(dJson, sMode):
        assert sMode == ArmAstBase.ksModeAccessor;
        assertJsonAttribsInSet(dJson, ArmAstBase.kAttribSetReturn);
        return ArmAstReturn(ArmAstBase.fromJson(dJson['val'], sMode) if dJson['val'] else None);

    kfnTypeMap = {
        ksTypeBinaryOp:         fromJsonBinaryOp,
        ksTypeUnaryOp:          fromJsonUnaryOp,
        ksTypeSlice:            fromJsonSlice,
        ksTypeSquareOp:         fromJsonSquareOp,
        ksTypeTuple:            fromJsonTuple,
        ksTypeDotAtom:          fromJsonDotAtom,
        ksTypeConcat:           fromJsonConcat,
        ksTypeFunction:         fromJsonFunction,
        ksTypeIdentifier:       fromJsonIdentifier,
        ksTypeBool:             fromJsonBool,
        ksTypeInteger:          fromJsonInteger,
        ksTypeSet:              fromJsonSet,
        ksTypeValue:            fromJsonValue,
        ksTypeEquationValue:    fromJsonEquationValue,
        ksTypeValuesGroup:      fromJsonValuesGroup,
        ksTypeString:           fromJsonString,
        ksTypeField:            fromJsonField,
        ksTypeRegisterType:     fromJsonRegisterType,
        ksTypeType:             fromJsonType,
        ksTypeTypeAnnotation:   fromJsonTypeAnnotation,
        ksTypeAssignment:       fromJsonAssignment,
        ksTypeReturn:           fromJsonReturn,
    };

    @staticmethod
    def fromJson(dJson, sMode = ksModeConditions):
        """ Decodes an AST/Values expression. """
        #print('debug ast: %s' % dJson['_type'])
        return ArmAstBase.kfnTypeMap[dJson['_type']](dJson, sMode);

    def toString(self):
        return 'todo<%s>' % (self.sType,);

    def __str__(self):
        return self.toString();

    def __repr__(self):
        return self.toString();

    #
    # Convenience matching routines, matching node type and type specific value.
    #

    def isBoolAndTrue(self):
        """ Checks if this is a boolean with the value True. """
        # This is overridden by ArmAstBool.
        return False;

    def isBoolAndFalse(self):
        """ Checks if this is a boolean with the value False. """
        # This is overridden by ArmAstBool.
        return False;

    def isMatchingIdentifier(self, sName):
        """ Checks if this is an identifier with the given name. """
        # This is overridden by ArmAstIdentifier.
        _ = sName;
        return False;

    def getIdentifierName(self):
        """ If this is an identifier, its name is return, otherwise None. """
        # This is overridden by ArmAstIdentifier.
        return None;

    def isMatchingDotAtom(self, *asElements):
        """ Checks if this is a dot atom node with the given list of string identifiers. """
        # This is overridden by ArmAstDotAtom.
        _ = asElements;
        return False;

    def isMatchingInteger(self, iValue):
        """ Checks if this is an integer node with the given value. """
        # This is overridden by ArmAstInteger.
        _ = iValue;
        return False;

    def isMatchingSquareOp(self, sVar, *aoValueMatches):
        """
        Checks if this is a square op node with the given variable and values.
        Values are mapped as following:
            - int value to ArmAstInteger.
            - str value to ArmAstIdentifier.
            - None matches anything..
        """
        # This is overridden by ArmAstSquareOp.
        _ = sVar; _ = aoValueMatches;
        return False;

    def isMatchingFunctionCall(self, sFunctionName, *aoArgMatches):
        """
        Checks if this is a function (call) node with the given name and arguments.
        Values are mapped as following:
            - int value to ArmAstInteger.
            - str value to the toString result.
            - None matches anything..
        """
        # This is overridden by ArmAstFunction.
        _ = sFunctionName; _ = aoArgMatches;
        return False;



class ArmAstBinaryOp(ArmAstBase):
    kOpTypeCompare      = 'cmp';
    kOpTypeLogical      = 'log';
    kOpTypeArithmetical = 'arit';
    kOpTypeSet          = 'set';
    kOpTypeConstraints  = 'constraints';
    kOpTypeBitwise      = 'bitwise';
    kdOps = {
        '||':  kOpTypeLogical,
        '&&':  kOpTypeLogical,
        '==':  kOpTypeCompare,
        '!=':  kOpTypeCompare,
        '>':   kOpTypeCompare,
        '<':   kOpTypeCompare,
        '>=':  kOpTypeCompare,
        '<=':  kOpTypeCompare,
        'IN':  kOpTypeSet,
        '+':   kOpTypeArithmetical,
        '-':   kOpTypeArithmetical,
        'MOD': kOpTypeArithmetical,
        '*':   kOpTypeArithmetical,
        'AND': kOpTypeBitwise,
        'OR':  kOpTypeBitwise,
        '-->': kOpTypeConstraints,    # implies that the right hand side is true when left hand side is.
        '<->': kOpTypeConstraints,    # bidirectional version of -->, i.e. it follows strictly in both directions.
    };
    kdOpsToC = {
        '||':  '||',
        '&&':  '&&',
        '==':  '==',
        '!=':  '!=',
        '>':   '>',
        '<':   '<',
        '>=':  '>=',
        '<=':  '<=',
        #'IN':  'IN',
        '+':   '+',
        '-':   '-',
        'MOD': '%',
        '*':   '*',
        'AND': '&',
        'OR':  '|',
        #'-->': kOpTypeConstraints,
        #'<->': kOpTypeConstraints,
    };

    def __init__(self, oLeft, sOp, oRight, fConstraints = False):
        ArmAstBase.__init__(self, ArmAstBase.ksTypeBinaryOp);
        assert sOp in ArmAstBinaryOp.kdOps and (fConstraints or ArmAstBinaryOp.kdOps[sOp] != ArmAstBinaryOp.kOpTypeConstraints),\
               'sOp="%s"' % (sOp,);
        self.oLeft  = oLeft;
        self.sOp    = sOp;
        self.oRight = oRight;

        # Switch value == field non-sense (simplifies transferConditionsToEncoding and such):
        if (    isinstance(oRight, ArmAstIdentifier)
            and isinstance(oLeft, (ArmAstValue, ArmAstInteger))
            and sOp in ['==', '!=']):
            self.oLeft  = oRight;
            self.oRight = oLeft;

    def clone(self):
        return ArmAstBinaryOp(self.oLeft.clone(), self.sOp, self.oRight.clone());

    def isSame(self, oOther):
        if isinstance(oOther, ArmAstBinaryOp):
            if self.sOp == oOther.sOp:
                if self.oLeft.isSame(oOther.oLeft):
                    if self.oRight.isSame(oOther.oRight):
                        return True;
            ## @todo switch sides and whatnot.
        return False;

    def transform(self, fnCallback, fEliminationAllowed, oCallbackArg):
        # Recurse first.
        fChildElimination = ArmAstBinaryOp.kdOps[self.sOp] in (ArmAstBinaryOp.kOpTypeLogical,);
        self.oLeft  = self.oLeft.transform(fnCallback, fChildElimination, oCallbackArg);
        self.oRight = self.oRight.transform(fnCallback, fChildElimination, oCallbackArg);

        if not fChildElimination:
            assert self.oLeft and self.oRight;
            return fnCallback(self, fEliminationAllowed, oCallbackArg);

        if self.oLeft and self.oRight:
            return fnCallback(self, fEliminationAllowed, oCallbackArg);

        if self.sOp == '||':
            ## @todo ensure boolean return?
            if self.oLeft:
                return self.oLeft;
            if self.oRight:
                return self.oRight;
        else:
            assert self.sOp == '&&';
        return fnCallback(ArmAstBool(False), fEliminationAllowed, oCallbackArg) if not fEliminationAllowed else None;

    @staticmethod
    def needParentheses(oNode, sOp = '&&'):
        if isinstance(oNode, ArmAstBinaryOp):
            if sOp != oNode.sOp or sOp not in ('||', '&&', '+', '-', '*'):
                return True;
        return False;

    def toString(self):
        sLeft = self.oLeft.toString();
        if ArmAstBinaryOp.needParentheses(self.oLeft, self.sOp):
            sLeft = '(%s)' % (sLeft);

        sRight = self.oRight.toString();
        if ArmAstBinaryOp.needParentheses(self.oRight, self.sOp):
            sRight = '(%s)' % (sRight);

        return '%s %s %s' % (sLeft, self.sOp, sRight);

    def toCExpr(self, oHelper):
        # Logical, compare, arithmetical & bitwise operations are straight forward.
        if ArmAstBinaryOp.kdOps[self.sOp] in (ArmAstBinaryOp.kOpTypeLogical,
                                              ArmAstBinaryOp.kOpTypeCompare,
                                              ArmAstBinaryOp.kOpTypeArithmetical,
                                              ArmAstBinaryOp.kOpTypeBitwise):
            sLeft = self.oLeft.toCExpr(oHelper);
            if ArmAstBinaryOp.needParentheses(self.oLeft, self.sOp):
                sLeft = '(%s)' % (sLeft);

            sRight = self.oRight.toCExpr(oHelper);
            if ArmAstBinaryOp.needParentheses(self.oRight, self.sOp):
                sRight = '(%s)' % (sRight);
            return '%s %s %s' % (sLeft, ArmAstBinaryOp.kdOpsToC[self.sOp], sRight);

        # 'x IN (y,z,...)' needs rewriting.
        if self.sOp == 'IN':
            if not isinstance(self.oRight, ArmAstSet):
                raise Exception('Unsupported right operand to IN operator: %s' % (self.toString(),));

            if isinstance(self.oLeft, ArmAstIdentifier):
                (sCName, cBitsWidth) = oHelper.getFieldInfo(self.oLeft.sName);
            elif isinstance(self.oLeft, ArmAstField):
                (sCName, cBitsWidth) = oHelper.getFieldInfo(self.oLeft.sField, self.oLeft.sName, self.oLeft.sState);
            else:
                raise Exception('Unsupported left operand to IN operator: %s' % (self.toString(),));

            asTests = [];
            for oValue in self.oRight.aoValues:
                if isinstance(oValue, ArmAstValue):
                    (fValue, fFixed, fWildcard, _) = ArmEncodesetField.parseValue(oValue.sValue, cBitsWidth);
                    fCombined = fValue | fFixed | fWildcard;
                    if fCombined < 0 or fCombined >= (1 << cBitsWidth):
                        raise Exception('Set value out of range: %s, width %u bits (expr: %s)'
                                        % (oValue.toString(), cBitsWidth, self.toString(),));
                    if fFixed == ((1 << cBitsWidth) - 1):
                        if fValue < 10:
                            asTests.append('%s == %u' % (sCName, fValue,));
                        elif fValue < (1 << 31):
                            asTests.append('%s == %#x' % (sCName, fValue,));
                        else:
                            asTests.append('%s == UINT32_C(%#010x)' % (sCName, fValue,));
                    else:
                        if fFixed < 10:
                            asTests.append('(%s & %u) == %u' % (sCName, fFixed, fValue,));
                        elif fFixed < (1 << 31):
                            asTests.append('(%s & %#x) == %#x' % (sCName, fFixed, fValue,));
                        else:
                            asTests.append('(%s & %#010x) == UINT32_C(%#010x)' % (sCName, fFixed, fValue,));
                elif isinstance(oValue, ArmAstInteger):
                    if oValue.iValue < 0 or oValue.iValue >= (1 << cBitsWidth):
                        raise Exception('Set value out of range: %s, width %u bits (expr: %s)'
                                        % (oValue.toString(), cBitsWidth, self.toString(),));
                    asTests.append('(%s == %s)' % (sCName, oValue.toCExpr(oHelper),));
                else:
                    raise Exception('Unsupported value in set: %s (expr: %s)' % (oValue.toString(), self.toString(),));

            if len(asTests) == 1:
                return asTests[0];
            return '(%s)' % (' || '.join(asTests),);

        raise Exception('Unsupported binary operator: %s (%s)' % (self.sOp, self.toString(),));

    def getWidth(self, oHelper):
        _ = oHelper;
        sOpType = self.kdOps[self.sOp];
        if sOpType in (self.kOpTypeCompare, self.kOpTypeLogical, self.kOpTypeSet):
            return 1; # boolean result.
        return -1;

    @staticmethod
    def andListToTree(aoAndConditions):
        """ Creates AST tree of AND binary checks from aoAndConditions. """
        if len(aoAndConditions) <= 1:
            return aoAndConditions[0].clone();
        return ArmAstBinaryOp(aoAndConditions[0].clone(), '&&', ArmAstBinaryOp.andListToTree(aoAndConditions[1:]));

    @staticmethod
    def orListToTree(aoOrConditions):
        """ Creates AST tree of OR binary checks from aoAndConditions. """
        if len(aoOrConditions) <= 1:
            return aoOrConditions[0].clone();
        return ArmAstBinaryOp(aoOrConditions[0].clone(), '||', ArmAstBinaryOp.orListToTree(aoOrConditions[1:]));


class ArmAstUnaryOp(ArmAstBase):
    kOpTypeLogical      = 'log';
    kOpTypeBitwise      = 'bitwise';
    kdOps = {
        '!':   kOpTypeLogical,
        'NOT': kOpTypeBitwise,
    };
    kdOpsToC = {
        '!':   '!',
        'NOT': '~',
    };

    def __init__(self, sOp, oExpr):
        ArmAstBase.__init__(self, ArmAstBase.ksTypeUnaryOp);
        assert sOp in ArmAstUnaryOp.kdOps, 'sOp=%s' % (sOp,);
        self.sOp   = sOp;
        self.oExpr = oExpr;

    def clone(self):
        return ArmAstUnaryOp(self.sOp, self.oExpr.clone());

    def isSame(self, oOther):
        if isinstance(oOther, ArmAstUnaryOp):
            if self.sOp == oOther.sOp:
                if self.oExpr.isSame(oOther.oExpr):
                    return True;
        return False;

    def transform(self, fnCallback, fEliminationAllowed, oCallbackArg):
        fChildElimination = ArmAstUnaryOp.kdOps[self.sOp] in (ArmAstUnaryOp.kOpTypeLogical,);
        self.oExpr = self.oExpr.transform(fnCallback, fChildElimination, oCallbackArg);
        if self.oExpr:
            return fnCallback(self, fEliminationAllowed, oCallbackArg);
        assert fChildElimination;
        if fEliminationAllowed:
            return None;
        assert self.sOp == '!';
        return fnCallback(ArmAstBool(True), fEliminationAllowed, oCallbackArg);

    @staticmethod
    def needParentheses(oNode):
        return isinstance(oNode, ArmAstBinaryOp)

    def toString(self):
        if ArmAstUnaryOp.needParentheses(self.oExpr):
            return '%s(%s)' % (self.sOp, self.oExpr.toString(),);
        return '%s%s' % (self.sOp, self.oExpr.toString(),);

    def toCExpr(self, oHelper):
        if ArmAstUnaryOp.needParentheses(self.oExpr):
            return '%s(%s)' % (ArmAstUnaryOp.kdOpsToC[self.sOp], self.oExpr.toCExpr(oHelper));
        return '%s%s' % (ArmAstUnaryOp.kdOpsToC[self.sOp], self.oExpr.toCExpr(oHelper));

    def getWidth(self, oHelper):
        if self.kdOps[self.sOp] == self.kOpTypeLogical:
            return 1; # boolean result.
        return self.oExpr.getWidth(oHelper);


class ArmAstSlice(ArmAstBase):
    def __init__(self, oFrom, oTo):
        ArmAstBase.__init__(self, ArmAstBase.ksTypeSlice);
        self.oFrom = oFrom; # left
        self.oTo   = oTo;   # right

    def clone(self):
        return ArmAstSlice(self.oFrom.clone(), self.oTo.clone());

    def isSame(self, oOther):
        if isinstance(oOther, ArmAstSlice):
            if self.oFrom.isSame(oOther.oFrom):
                if self.oTo.isSame(oOther.oTo):
                    return True;
        return False;

    def transform(self, fnCallback, fEliminationAllowed, oCallbackArg):
        self.oFrom = self.oFrom.transform(fnCallback, False, oCallbackArg);
        self.oTo   = self.oTo.transform(fnCallback, False, oCallbackArg);
        assert self.oFrom and self.oTo;
        return fnCallback(self, fEliminationAllowed, oCallbackArg);

    def toString(self):
        return '[%s:%s]' % (self.oFrom.toString(), self.oTo.toString());

    def toCExpr(self, oHelper):
        _ = oHelper;
        raise Exception('not implemented');

    def getWidth(self, oHelper):
        _ = oHelper;
        raise Exception('not implemented');


class ArmAstSquareOp(ArmAstBase):
    def __init__(self, oVar, aoValues):
        ArmAstBase.__init__(self, ArmAstBase.ksTypeSquareOp);
        assert isinstance(oVar, ArmAstBase);
        self.oVar     = oVar;
        self.aoValues = aoValues;

    def clone(self):
        return ArmAstSquareOp(self.oVar.clone(), [oValue.clone() for oValue in self.aoValues]);

    def isSame(self, oOther):
        if isinstance(oOther, ArmAstSquareOp):
            if self.oVar.isSame(oOther.oVar):
                if len(self.aoValues) == len(oOther.aoValues):
                    for idx, oMyValue in enumerate(self.aoValues):
                        if not oMyValue.isSame(oOther.aoValues[idx]):
                            return False;
                    return True;
        return False;

    def transform(self, fnCallback, fEliminationAllowed, oCallbackArg):
        self.oVar = self.oVar.transform(fnCallback, fEliminationAllowed, oCallbackArg);
        if self.oVar:
            self.aoValues = [oValue.transform(fnCallback, False, oCallbackArg) for oValue in self.aoValues];
            for oValue in self.aoValues:
                assert oValue;
            return fnCallback(self, fEliminationAllowed, oCallbackArg);
        assert fEliminationAllowed;
        return None;


    def toString(self):
        return '%s[%s]' % (self.oVar.toString(), ','.join([oValue.toString() for oValue in self.aoValues]),);

    def toCExpr(self, oHelper):
        _ = oHelper;
        raise Exception('ArmAstSquareOp does not support conversion to C expression: %s' % (self.toString()));

    def getWidth(self, oHelper):
        _ = oHelper;
        return -1;

    def isMatchingSquareOp(self, sVar, *aoValueMatches):
        if self.oVar.isMatchingIdentifier(sVar):
            if len(self.aoValues) == len(aoValueMatches):
                for i, oValue in enumerate(self.aoValues):
                    if isinstance(aoValueMatches[i], int):
                        if not oValue.isMatchingInteger(aoValueMatches[i]):
                            return False;
                    elif isinstance(aoValueMatches[i], str):
                        if not oValue.isMatchingIdentifier(aoValueMatches[i]):
                            return False;
                    elif aoValueMatches[i] is not None:
                        raise Exception('Unexpected #%u: %s' % (i, aoValueMatches[i],));
                return True;
        return False;


class ArmAstTuple(ArmAstBase):
    def __init__(self, aoValues):
        ArmAstBase.__init__(self, ArmAstBase.ksTypeTuple);
        self.aoValues = aoValues;

    def clone(self):
        return ArmAstTuple([oValue.clone() for oValue in self.aoValues]);

    def isSame(self, oOther):
        if isinstance(oOther, ArmAstTuple):
            if len(self.aoValues) == len(oOther.aoValues):
                for idx, oMyValue in enumerate(self.aoValues):
                    if not oMyValue.isSame(oOther.aoValues[idx]):
                        return False;
                return True;
        return False;

    def transform(self, fnCallback, fEliminationAllowed, oCallbackArg):
        self.aoValues = [oValue.transform(fnCallback, False, oCallbackArg) for oValue in self.aoValues];
        for oValue in self.aoValues:
            assert oValue;
        return fnCallback(self, fEliminationAllowed, oCallbackArg);

    def toString(self):
        return '(%s)' % (','.join([oValue.toString() for oValue in self.aoValues]),);

    def toCExpr(self, oHelper):
        _ = oHelper;
        raise Exception('ArmAstTuple does not support conversion to C expression: %s' % (self.toString()));

    def getWidth(self, oHelper):
        _ = oHelper;
        return -1;


class ArmAstDotAtom(ArmAstBase):
    def __init__(self, aoValues):
        ArmAstBase.__init__(self, ArmAstBase.ksTypeDotAtom);
        self.aoValues = aoValues;

    def clone(self):
        return ArmAstDotAtom([oValue.clone() for oValue in self.aoValues]);

    def isSame(self, oOther):
        if isinstance(oOther, ArmAstDotAtom):
            if len(self.aoValues) == len(oOther.aoValues):
                for idx, oMyValue in enumerate(self.aoValues):
                    if not oMyValue.isSame(oOther.aoValues[idx]):
                        return False;
                return True;
        return False;

    def transform(self, fnCallback, fEliminationAllowed, oCallbackArg):
        self.aoValues = [oValue.transform(fnCallback, False, oCallbackArg) for oValue in self.aoValues];
        for oValue in self.aoValues:
            assert oValue;
        return fnCallback(self, fEliminationAllowed, oCallbackArg);

    def toString(self):
        return '.'.join([oValue.toString() for oValue in self.aoValues]);

    def toCExpr(self, oHelper):
        _ = oHelper;
        raise Exception('ArmAstDotAtom does not support conversion to C expression: %s' % (self.toString()));

    #@staticmethod
    #def fromString(self, sExpr):
    #    """ Limited to identifiers separated byt dots """
    #    asValues = sExpr.split('.');

    def getWidth(self, oHelper):
        _ = oHelper;
        return -1;

    def isMatchingDotAtom(self, *asElements):
        if len(asElements) == len(self.aoValues):
            for idx, sName in enumerate(asElements):
                if not self.aoValues[idx].isMatchingIdentifier(sName):
                    return False;
            return True;
        return False;

class ArmAstConcat(ArmAstBase):
    def __init__(self, aoValues):
        ArmAstBase.__init__(self, ArmAstBase.ksTypeConcat);
        self.aoValues = aoValues;

    def clone(self):
        return ArmAstConcat([oValue.clone() for oValue in self.aoValues]);

    def isSame(self, oOther):
        if isinstance(oOther, ArmAstConcat):
            if len(self.aoValues) == len(oOther.aoValues):
                for idx, oMyValue in enumerate(self.aoValues):
                    if not oMyValue.isSame(oOther.aoValues[idx]):
                        return False;
                return True;
        return False;

    def transform(self, fnCallback, fEliminationAllowed, oCallbackArg):
        self.aoValues = [oValue.transform(fnCallback, False, oCallbackArg) for oValue in self.aoValues];
        for oValue in self.aoValues:
            assert oValue;
        return fnCallback(self, fEliminationAllowed, oCallbackArg);

    def toString(self):
        sRet = '';
        for oValue in self.aoValues:
            if sRet:
                sRet += ':'
            if isinstance(oValue, ArmAstIdentifier):
                sRet += oValue.sName;
            else:
                sRet += '(%s)' % (oValue.toString());
        return sRet;

    def toCExpr(self, oHelper):
        sConcat = '(';
        iBitPos = 0;
        for oPart in self.aoValues:
            if len(sConcat) > 1:
                sConcat += ' | ';
            if isinstance(oPart, ArmAstIdentifier):
                (sCName, cBitsWidth) = oHelper.getFieldInfo(oPart.sName);
                if iBitPos == 0:
                    sConcat += sCName;
                else:
                    sConcat += '(%s << %u)' % (sCName, iBitPos);
                iBitPos += cBitsWidth;
            else:
                raise Exception('Unexpected value type for concat(): %s' % (oPart.sType,));
        sConcat += ')';
        return sConcat;

    def getWidth(self, oHelper):
        _ = oHelper;
        cBitsWidth = 0;
        for oValue in self.aoValues:
            cBitsThis = oValue.getWidth(oHelper);
            if cBitsThis < 0:
                return -1;
            cBitsWidth += cBitsThis;
        return cBitsWidth;


class ArmAstFunction(ArmAstBase):
    s_oReValidName = re.compile('^[_A-Za-z][_A-Za-z0-9]+$');

    def __init__(self, sName, aoArgs):
        ArmAstBase.__init__(self, ArmAstBase.ksTypeFunction);
        assert self.s_oReValidName.match(sName), 'sName=%s' % (sName);
        self.sName  = sName;
        self.aoArgs = aoArgs;

    def clone(self):
        return ArmAstFunction(self.sName, [oArg.clone() for oArg in self.aoArgs]);

    def isSame(self, oOther):
        if isinstance(oOther, ArmAstFunction):
            if self.sName == oOther.sName:
                if len(self.aoArgs) == len(oOther.aoArgs):
                    for idx, oMyArg in enumerate(self.aoArgs):
                        if not oMyArg.isSame(oOther.aoArgs[idx]):
                            return False;
                    return True;
        return False;

    def transform(self, fnCallback, fEliminationAllowed, oCallbackArg):
        self.aoArgs = [oArgs.transform(fnCallback, False, oCallbackArg) for oArgs in self.aoArgs];
        for oArgs in self.aoArgs:
            assert oArgs;
        return fnCallback(self, fEliminationAllowed, oCallbackArg);

    def toString(self):
        return '%s(%s)' % (self.sName, ','.join([oArg.toString() for oArg in self.aoArgs]),);

    def toCExpr(self, oHelper):
        return oHelper.convertFunctionCall(self);

    def getWidth(self, oHelper):
        _ = oHelper;
        return -1;

    def isMatchingFunctionCall(self, sFunctionName, *aoArgMatches):
        if self.sName == sFunctionName:
            if len(self.aoArgs) == len(aoArgMatches):
                for i, oArg in enumerate(self.aoArgs):
                    if isinstance(aoArgMatches[i], int):
                        if not oArg.isMatchingInteger(aoArgMatches[i]):
                            return False;
                    elif isinstance(aoArgMatches[i], str):
                        if not oArg.toString() != aoArgMatches[i]:
                            return False;
                    elif aoArgMatches[i] is not None:
                        raise Exception('Unexpected #%u: %s' % (i, aoArgMatches[i],));
                return True;
        return False;


class ArmAstIdentifier(ArmAstBase):
    s_oReValidName        = re.compile('^[_A-Za-z][_A-Za-z0-9]*$');
    s_oReValidNameRelaxed = re.compile('^[_A-Za-z][_A-Za-z0-9<>]*$');
    s_aoReValidName = (s_oReValidName, s_oReValidNameRelaxed)

    def __init__(self, sName, fRelaxedName = False):
        ArmAstBase.__init__(self, ArmAstBase.ksTypeIdentifier);
        assert self.s_aoReValidName[fRelaxedName].match(sName), 'sName=%s' % (sName);
        self.sName = sName;

    def clone(self):
        return ArmAstIdentifier(self.sName);

    def isSame(self, oOther):
        if isinstance(oOther, ArmAstIdentifier):
            if self.sName == oOther.sName:
                return True;
        return False;

    def transform(self, fnCallback, fEliminationAllowed, oCallbackArg):
        return fnCallback(self, fEliminationAllowed, oCallbackArg);

    def toString(self):
        return self.sName;

    def toCExpr(self, oHelper):
        (sCName, _) = oHelper.getFieldInfo(self.sName);
        return sCName;

    def getWidth(self, oHelper):
        (_, cBitsWidth) = oHelper.getFieldInfo(self.sName);
        return cBitsWidth;

    def isMatchingIdentifier(self, sName):
        return self.sName == sName;

    def getIdentifierName(self):
        return self.sName;


class ArmAstBool(ArmAstBase):
    def __init__(self, fValue):
        ArmAstBase.__init__(self, ArmAstBase.ksTypeBool);
        assert fValue is True or fValue is False, '%s' % (fValue,);
        self.fValue = fValue;

    def clone(self):
        return ArmAstBool(self.fValue);

    def isSame(self, oOther):
        if isinstance(oOther, ArmAstBase):
            if self.fValue == oOther.fValue:
                return True;
        return False;

    def transform(self, fnCallback, fEliminationAllowed, oCallbackArg):
        return fnCallback(self, fEliminationAllowed, oCallbackArg);

    def toString(self):
        return 'true' if self.fValue is True else 'false';

    def toCExpr(self, oHelper):
        _ = oHelper;
        return 'true' if self.fValue is True else 'false';

    def getWidth(self, oHelper):
        _ = oHelper;
        return 1;

    def isBoolAndTrue(self):
        return self.fValue is True;

    def isBoolAndFalse(self):
        return self.fValue is False;


class ArmAstInteger(ArmAstBase):
    def __init__(self, iValue):
        ArmAstBase.__init__(self, ArmAstBase.ksTypeInteger);
        self.iValue = int(iValue);

    def clone(self):
        return ArmAstInteger(self.iValue);

    def isSame(self, oOther):
        if isinstance(oOther, ArmAstInteger):
            if self.iValue == oOther.iValue:
                return True;
        return False;

    def transform(self, fnCallback, fEliminationAllowed, oCallbackArg):
        return fnCallback(self, fEliminationAllowed, oCallbackArg);

    def toString(self):
        return '%#x' % (self.iValue,);

    def toCExpr(self, oHelper):
        _ = oHelper;
        if self.iValue < 10:
            return '%u' % (self.iValue,);
        if self.iValue & (1<<31):
            return 'UINT32_C(%#x)' % (self.iValue,);
        return '%#x' % (self.iValue,);

    def getWidth(self, oHelper):
        _ = oHelper;
        return self.iValue.bit_length() + (self.iValue < 0)

    def isMatchingInteger(self, iValue):
        return self.iValue == iValue;


class ArmAstSet(ArmAstBase):
    def __init__(self, aoValues):
        ArmAstBase.__init__(self, ArmAstBase.ksTypeSet);
        self.aoValues = aoValues;

    def clone(self):
        return ArmAstSet([oValue.clone() for oValue in self.aoValues]);

    def isSame(self, oOther):
        if isinstance(oOther, ArmAstSet):
            if len(self.aoValues) == len(oOther.aoValues):
                for idx, oMyValue in enumerate(self.aoValues):
                    if not oMyValue.isSame(oOther.aoValues[idx]):
                        return False;
                return True;
        return False;

    def transform(self, fnCallback, fEliminationAllowed, oCallbackArg):
        self.aoValues = [oValue.transform(fnCallback, False, oCallbackArg) for oValue in self.aoValues];
        for oValue in self.aoValues:
            assert oValue;
        return fnCallback(self, fEliminationAllowed, oCallbackArg);

    def toString(self):
        return '(%s)' % (', '.join([oValue.toString() for oValue in self.aoValues]),);

    def toCExpr(self, oHelper):
        _ = oHelper;
        raise Exception('ArmAstSet does not support conversion to C expression: %s' % (self.toString()));

    def getWidth(self, oHelper):
        _ = oHelper;
        if self.aoValues:
            return max(oValue.getWidth() for oValue in self.aoValues);
        return -1;


class ArmAstValue(ArmAstBase):
    def __init__(self, sValue):
        ArmAstBase.__init__(self, ArmAstBase.ksTypeValue);
        self.sValue = sValue;

    def clone(self):
        return ArmAstValue(self.sValue);

    def isSame(self, oOther):
        if isinstance(oOther, ArmAstValue):
            if self.sValue == oOther.sValue:
                return True;
        return False;

    def transform(self, fnCallback, fEliminationAllowed, oCallbackArg):
        return fnCallback(self, fEliminationAllowed, oCallbackArg);

    def toString(self):
        return self.sValue;

    def toCExpr(self, oHelper):
        _ = oHelper;
        (fValue, _, fWildcard, _) = ArmEncodesetField.parseValue(self.sValue, 0);
        if fWildcard:
            raise Exception('Value contains wildcard elements: %s' % (self.sValue,));
        if fValue < 10:
            return '%u' % (fValue,);
        if fValue & (1<<31):
            return 'UINT32_C(%#x)' % (fValue,);
        return '%#x' % (fValue,);

    def getWidth(self, oHelper):
        _ = oHelper;
        (_, _, _, cBitsWidth) = ArmEncodesetField.parseValue(self.sValue, 0);
        return cBitsWidth;


class ArmAstEquationValue(ArmAstBase):

    s_oSimpleName = re.compile('^[_A-Za-z][_A-Za-z0-9]+$');

    def __init__(self, sValue, iFirstBit, cBitsWidth):
        ArmAstBase.__init__(self, ArmAstBase.ksTypeValue);
        self.sValue     = sValue;
        self.iFirstBit  = iFirstBit;
        self.cBitsWidth = cBitsWidth;

    def clone(self):
        return ArmAstEquationValue(self.sValue, self.iFirstBit, self.cBitsWidth);

    def isSame(self, oOther):
        if isinstance(oOther, ArmAstEquationValue):
            if self.sValue == oOther.sValue:
                if self.iFirstBit == oOther.iFirstBit:
                    if self.cBitsWidth == oOther.cBitsWidth:
                        return True;
        return False;

    def transform(self, fnCallback, fEliminationAllowed, oCallbackArg):
        return fnCallback(self, fEliminationAllowed, oCallbackArg);

    def toString(self):
        if self.s_oSimpleName.match(self.sValue):
            return '%s[%u:%u]' % (self.sValue, self.iFirstBit, self.iFirstBit + self.cBitsWidth - 1,);
        return '(%s)[%u:%u]' % (self.sValue, self.iFirstBit, self.iFirstBit + self.cBitsWidth - 1,);

    def toCExpr(self, oHelper):
        _ = oHelper;
        raise Exception('todo');

    def getWidth(self, oHelper):
        _ = oHelper;
        return self.cBitsWidth;


class ArmAstValuesGroup(ArmAstBase):

    def __init__(self, sValue, aoValues = None):
        ArmAstBase.__init__(self, ArmAstBase.ksTypeValue);
        self.sValue     = sValue;
        self.aoValues   = aoValues;
        if aoValues is None:
            ## @todo split up the value string. Optionally handle a specific 'values' list
            ## when present (looks like it's suppressed in the spec generator).
            self.aoValues = [];
            off = 0;
            while off < len(sValue):
                offStart = off;
                fSlice = False;
                while off < len(sValue) and (sValue[off] != ':' or fSlice):
                    if sValue[off] == '[':
                        assert not fSlice;
                        assert off > offStart;
                        fSlice = True;
                    elif sValue[off] == ']':
                        assert fSlice;
                        fSlice = False;
                    off += 1;
                sSubValue = sValue[offStart:off]
                assert off > offStart;
                assert not fSlice;

                if sSubValue[0] == '\'':
                    self.aoValues.append(ArmAstValue(sSubValue));
                elif '[' in sSubValue[0]:
                    assert sSubValue[-1] == ']';
                    offSlice = sSubValue.find('[');
                    asSlice = sSubValue[offSlice + 1:-1].split(':');
                    assert len(asSlice) == 2;
                    self.aoValues.append(ArmAstEquationValue(sSubValue[:offSlice], int(asSlice[0]),
                                                             int(asSlice[1]) - int(asSlice[0]) + 1));
                off += 1;

    def clone(self):
        return ArmAstValuesGroup(self.sValue, [oValue.clone() for oValue in self.aoValues]);

    def isSame(self, oOther):
        if isinstance(oOther, ArmAstValuesGroup):
            if self.sValue == oOther.sValue:
                assert len(self.aoValues) == len(oOther.aoValues);
                for iValue, oValue in enumerate(self.aoValues):
                    assert oValue.isSame(oOther.aoValues[iValue]);
                return True;
        return False;

    def transform(self, fnCallback, fEliminationAllowed, oCallbackArg):
        self.aoValues = [oValue.transform(fnCallback, False, oCallbackArg) for oValue in self.aoValues];
        for oValue in self.aoValues:
            assert oValue;
        return fnCallback(self, fEliminationAllowed, oCallbackArg);

    def toString(self):
        return ':'.join([oValue.toString() for oValue in self.aoValues]);

    def toCExpr(self, oHelper):
        _ = oHelper;
        raise Exception('todo');

    def getWidth(self, oHelper):
        return sum(oValue.getWidth(oHelper) for oValue in self.aoValues);


class ArmAstString(ArmAstBase):
    def __init__(self, sValue):
        ArmAstBase.__init__(self, ArmAstBase.ksTypeValue);
        self.sValue = sValue;

    def clone(self):
        return ArmAstString(self.sValue);

    def isSame(self, oOther):
        if isinstance(oOther, ArmAstString):
            if self.sValue == oOther.sValue:
                return True;
        return False;

    def transform(self, fnCallback, fEliminationAllowed, oCallbackArg):
        return fnCallback(self, fEliminationAllowed, oCallbackArg);

    def toString(self):
        return '"' + self.sValue + '"';

    def toCExpr(self, oHelper):
        _ = oHelper;
        return '"' + self.sValue.replace('\\', '\\\\').replace('"', '\\"') + '"';

    def getWidth(self, oHelper):
        _ = oHelper;
        return -1;


class ArmAstField(ArmAstBase):
    def __init__(self, sField, sName, sState = 'AArch64', sSlices = None, sInstance = None):
        ArmAstBase.__init__(self, ArmAstBase.ksTypeField);
        self.sField    = sField;
        self.sName     = sName;
        self.sState    = sState;
        self.sSlices   = sSlices;
        self.sInstance = sInstance;
        assert sSlices is None;
        assert sInstance is None;

    def clone(self):
        return ArmAstField(self.sField, self.sName, self.sState, self.sSlices, self.sInstance);

    def isSame(self, oOther):
        if isinstance(oOther, ArmAstField):
            if self.sField == oOther.sField:
                if self.sName == oOther.sName:
                    if self.sState == oOther.sState:
                        if self.sSlices == oOther.sSlices:
                            if self.sInstance == oOther.sInstance:
                                return True;
        return False;

    def transform(self, fnCallback, fEliminationAllowed, oCallbackArg):
        return fnCallback(self, fEliminationAllowed, oCallbackArg);

    def toString(self):
        return '%s.%s.%s' % (self.sState, self.sName, self.sField,);

    def toCExpr(self, oHelper):
        (sCName, _) = oHelper.getFieldInfo(self.sField, self.sName, self.sState);
        return sCName;

    def getWidth(self, oHelper):
        (_, cBitsWidth) = oHelper.getFieldInfo(self.sField, self.sName, self.sState);
        return cBitsWidth;


class ArmAstRegisterType(ArmAstBase):
    def __init__(self, sName, sState = 'AArch64', sSlices = None, sInstance = None):
        ArmAstBase.__init__(self, ArmAstBase.ksTypeRegisterType);
        self.sName     = sName;
        self.sState    = sState;
        self.sSlices   = sSlices;
        self.sInstance = sInstance;
        assert sSlices is None;
        assert sInstance is None;

    def clone(self):
        return ArmAstRegisterType(self.sName, self.sState, self.sSlices, self.sInstance);

    def isSame(self, oOther):
        if isinstance(oOther, ArmAstRegisterType):
            if self.sName == oOther.sName:
                if self.sState == oOther.sState:
                    if self.sSlices == oOther.sSlices:
                        if self.sInstance == oOther.sInstance:
                            return True;
        return False;

    def toString(self):
        return '%s.%s' % (self.sState, self.sName,);

    def transform(self, fnCallback, fEliminationAllowed, oCallbackArg):
        return fnCallback(self, fEliminationAllowed, oCallbackArg);

    def toCExpr(self, oHelper):
        #(sCName, _) = oHelper.getFieldInfo(None, self.sName, self.sState);
        #return sCName;
        raise Exception('not implemented');

    def getWidth(self, oHelper):
        #(_, cBitsWidth) = oHelper.getFieldInfo(None, self.sName, self.sState);
        #return cBitsWidth;
        _ = oHelper;
        return -1;


class ArmAstType(ArmAstBase):
    def __init__(self, oName):
        ArmAstBase.__init__(self, ArmAstBase.ksTypeType);
        self.oName = oName;

    def clone(self):
        return ArmAstType(self.oName.clone());

    def isSame(self, oOther):
        if isinstance(oOther, ArmAstType):
            if self.oName.isSame(oOther.oName):
                return True;
        return False;

    def transform(self, fnCallback, fEliminationAllowed, oCallbackArg):
        self.oName = self.oName.transform(fnCallback, False, oCallbackArg);
        assert self.oName;
        return fnCallback(self, fEliminationAllowed, oCallbackArg);

    def toString(self):
        return self.oName.toString();

    def toCExpr(self, oHelper):
        _ = oHelper;
        raise Exception('not implemented');

    def getWidth(self, oHelper):
        _ = oHelper;
        raise Exception('not implemented');


class ArmAstTypeAnnotation(ArmAstBase):
    def __init__(self, oVar, oType):
        ArmAstBase.__init__(self, ArmAstBase.ksTypeTypeAnnotation);
        self.oVar  = oVar;
        self.oType = oType;

    def clone(self):
        return ArmAstTypeAnnotation(self.oVar.clone(), self.oType.clone());

    def isSame(self, oOther):
        if isinstance(oOther, ArmAstType):
            if self.oVar.isSame(oOther.oVar):
                if self.oType.isSame(oOther.oType):
                    return True;
        return False;

    def transform(self, fnCallback, fEliminationAllowed, oCallbackArg):
        self.oVar = self.oVar.transform(fnCallback, False, oCallbackArg);
        assert self.oVar;
        self.oType = self.oType.transform(fnCallback, False, oCallbackArg);
        assert self.oType;
        return fnCallback(self, fEliminationAllowed, oCallbackArg);

    def toString(self):
        return '(%s) %s' % (self.oType.toString(), self.oVar.toString(),);

    def toCExpr(self, oHelper):
        _ = oHelper;
        raise Exception('not implemented');

    def getWidth(self, oHelper):
        _ = oHelper;
        raise Exception('not implemented');



class ArmAstStatementBase(ArmAstBase):
    """
    Base class for statements.

    This adds the toStringList method and blocks the toCExpr and getWidth methods.
    """
    def __init__(self, sType):
        ArmAstBase.__init__(self, sType);

    def toString(self):
        return '\n'.join(self.toStringList());

    def toStringList(self, sIndent = '', sLang = None):
        _ = sIndent; _ = sLang;
        raise Exception('not implemented');

    def toCExpr(self, oHelper):
        _ = oHelper;
        raise Exception('not implemented');

    def getWidth(self, oHelper):
        _ = oHelper;
        raise Exception('not implemented');

    def isNop(self):
        """ Checks if this is a NOP statement. """
        return isinstance(self, ArmAstNop);


class ArmAstNop(ArmAstStatementBase):
    """
    NOP statement.
    Not part of ARM spec. We need it for transformations.
    """
    def __init__(self):
        ArmAstStatementBase.__init__(self, 'AST.Nop');

    def clone(self):
        return ArmAstNop();

    def isSame(self, oOther):
        return isinstance(oOther, ArmAstNop);

    def transform(self, fnCallback, fEliminationAllowed, oCallbackArg):
        return fnCallback(self, fEliminationAllowed, oCallbackArg);

    def toString(self):
        return 'NOP();';

    def toStringList(self, sIndent = '', sLang = None):
        return [ 'NOP();', ];



class ArmAstAssignment(ArmAstStatementBase):
    """ We classify assignments as statements. """

    def __init__(self, oVar, oValue):
        ArmAstStatementBase.__init__(self, ArmAstBase.ksTypeAssignment);
        self.oVar      = oVar;
        self.oValue    = oValue;

    def clone(self):
        return ArmAstAssignment(self.oVar.clone(), self.oValue.clone());

    def isSame(self, oOther):
        if isinstance(oOther, ArmAstAssignment):
            if self.oVar.isSame(oOther.oVar):
                if self.oValue.isSame(oOther.oValue):
                    return True;
        return False;

    def transform(self, fnCallback, fEliminationAllowed, oCallbackArg):
        self.oVar = self.oVar.transform(fnCallback, fEliminationAllowed, oCallbackArg);
        if self.oVar:
            self.oValue = self.oValue.transform(fnCallback, False, oCallbackArg);
            assert self.oValue;
            return fnCallback(self, fEliminationAllowed, oCallbackArg);
        assert fEliminationAllowed;
        return None;

    def toString(self):
        return '%s = %s;' % (self.oVar.toString(), self.oValue.toString());

    def toStringList(self, sIndent = '', sLang = None):
        _ = sLang;
        return [ sIndent + self.toString(), ];


class ArmAstReturn(ArmAstStatementBase):
    """ We classify assignments as statements. """

    def __init__(self, oValue):
        ArmAstStatementBase.__init__(self, ArmAstBase.ksTypeReturn);
        self.oValue = oValue;

    def clone(self):
        return ArmAstReturn(self.oValue.clone() if self.oValue else None);

    def isSame(self, oOther):
        if isinstance(oOther, ArmAstReturn):
            if self.oValue.isSame(oOther.oValue):
                return True;
        return False;

    def transform(self, fnCallback, fEliminationAllowed, oCallbackArg):
        if self.oValue:
            self.oValue = self.oValue.transform(fnCallback, False, oCallbackArg);
            assert self.oValue;
        return fnCallback(self, fEliminationAllowed, oCallbackArg);

    def toString(self):
        if self.oValue:
            return 'return %s;' % (self.oValue.toString(),);
        return 'return;';

    def toStringList(self, sIndent = '', sLang = None):
        _ = sLang;
        return [ sIndent + self.toString(), ];


#
# Some quick C++ AST nodes.
#

class ArmAstCppExpr(ArmAstBase):
    """ C++ AST node. """
    def __init__(self, sExpr, cBitsWidth = -1):
        ArmAstBase.__init__(self, 'C++');
        self.sExpr      = sExpr;
        self.cBitsWidth = cBitsWidth;

    def clone(self):
        return ArmAstCppExpr(self.sExpr, self.cBitsWidth);

    def isSame(self, oOther):
        if isinstance(oOther, ArmAstCppExpr):
            if self.sExpr == oOther.sExpr:
                if self.cBitsWidth == oOther.cBitsWidth:
                    return True;
        return False;

    def transform(self, fnCallback, fEliminationAllowed, oCallbackArg):
        return fnCallback(self, fEliminationAllowed, oCallbackArg);

    def toString(self):
        return self.sExpr;

    def toCExpr(self, oHelper):
        _ = oHelper;
        return self.sExpr;

    def getWidth(self, oHelper):
        _ = oHelper;
        return self.cBitsWidth;


class ArmAstCppStmt(ArmAstStatementBase):
    """ C++ AST statement node. """
    def __init__(self, *asStmts):
        ArmAstStatementBase.__init__(self, 'C++');
        self.asStmts = list(asStmts);

    def clone(self):
        return ArmAstCppStmt(*self.asStmts);

    def isSame(self, oOther):
        if isinstance(oOther, ArmAstCppStmt):
            if len(self.asStmts) == len(oOther.asStmts):
                for i, sMyStmt in enumerate(self.asStmts):
                    if sMyStmt != oOther.asStmts[i]:
                        return False;
                return True;
        return False;

    def transform(self, fnCallback, fEliminationAllowed, oCallbackArg):
        return fnCallback(self, fEliminationAllowed, oCallbackArg);

    def toString(self):
        return '\n'.join(self.asStmts);

    def toStringList(self, sIndent = '', sLang = None):
        _ = sLang;
        return [ sIndent + sStmt for sStmt in self.asStmts ];


#
# Instructions and their properties.
#

class ArmEncodesetField(object):
    """
    ARM Encodeset.Bits & Encodeset.Field.
    """
    def __init__(self, oJson, iFirstBit, cBitsWidth, fFixed, fValue, sName = None):
        self.oJson      = oJson;
        self.iFirstBit  = iFirstBit;
        self.cBitsWidth = cBitsWidth;
        self.fFixed     = fFixed;
        self.fValue     = fValue;
        self.sName      = sName; ##< None if Encodeset.Bits.

    def __str__(self):
        sRet = '[%2u:%-2u] = %#x/%#x/%#x' % (
            self.iFirstBit + self.cBitsWidth - 1, self.iFirstBit, self.fValue, self.fFixed, self.getMask()
        );
        if self.sName:
            sRet += ' # %s' % (self.sName,)
        return sRet;

    def __repr__(self):
        return self.__str__();

    def clone(self):
        return ArmEncodesetField(self.oJson, self.iFirstBit, self.cBitsWidth, self.fFixed, self.fValue, self.sName);

    def getMask(self):
        """ Field mask (unshifted). """
        return (1 << self.cBitsWidth) - 1;

    def getShiftedMask(self):
        """ Field mask, shifted. """
        return ((1 << self.cBitsWidth) - 1) << self.iFirstBit;

    @staticmethod
    def parseValue(sValue, cBitsWidth):
        """
        Returns (fValue, fFixed, fWildcard, cBitsWidth) tuple on success, raises AssertionError otherwise.
        """
        assert sValue[0] == '\'' and sValue[-1] == '\'', sValue;
        sValue = sValue[1:-1];
        assert not cBitsWidth or len(sValue) == cBitsWidth, 'cBitsWidth=%s sValue=%s' % (cBitsWidth, sValue,);
        cBitsWidth = len(sValue);
        fFixed     = 0;
        fWildcard  = 0;
        fValue     = 0;
        for ch in sValue:
            assert ch in 'x10', 'ch=%s' % ch;
            fFixed    <<= 1;
            fValue    <<= 1;
            fWildcard <<= 1;
            if ch != 'x':
                fFixed |= 1;
                if ch == '1':
                    fValue |= 1;
            else:
                fWildcard |= 1;
        return (fValue, fFixed, fWildcard, cBitsWidth);

    @staticmethod
    def fromJson(oJson):
        assert oJson['_type'] in ('Instruction.Encodeset.Field', 'Instruction.Encodeset.Bits'), oJson['_type'];

        oRange = oJson['range'];
        assert oRange['_type'] == 'Range';
        iFirstBit           = int(oRange['start']);
        cBitsWidth          = int(oRange['width']);
        sName               = oJson['name'] if oJson['_type'] == 'Instruction.Encodeset.Field' else None;
        (fValue, fFixed, _, _) = ArmEncodesetField.parseValue(oJson['value']['value'], cBitsWidth);
        return ArmEncodesetField(oJson, iFirstBit, cBitsWidth, fFixed, fValue, sName);

    @staticmethod
    def encodesetFromJson(oJson):
        assert oJson['_type'] == 'Instruction.Encodeset.Encodeset', oJson['_type'];
        aoSet   = [];
        fFields = 0;
        for oJsonValue in oJson['values']:
            oNewField = ArmEncodesetField.fromJson(oJsonValue);
            fNewMask  = oNewField.getShiftedMask();
            assert (fNewMask & fFields) == 0;
            aoSet.append(oNewField);
            fFields  |= fNewMask;
        return (aoSet, fFields);

    @staticmethod
    def encodesetAddParentFields(aoFields, fFields, aoParentFields):
        for oParentField in aoParentFields:
            fNewMask  = oParentField.getShiftedMask();
            if (fNewMask & fFields) != fNewMask:
                aoFields.append(oParentField.clone()); # (paranoid: clone)
                fFields |= fNewMask;
        return (aoFields, fFields);


class ArmInstructionBase(object):
    """
    Base class for ArmInstruction, ArmInstructionSet and ArmInstructionGroup

    Instances of ArmInstruction will have ArmInstructionGroup (or maybe
    ArmInstructionGroup) as parent.

    Instances of ArmInstructionGroup have ArmInstructionSet as parent.

    Instances of ArmInstructionSet doesn't have a parent, so it is None.
    """

    s_oReValidName = re.compile('^[_A-Za-z][_A-Za-z0-9]+$');

    def __init__(self, oJson, sName, aoFields, fFields, oCondition, oParent):
        self.oJson           = oJson;
        self.sName           = sName;
        self.oParent         = oParent;
        self.aoFields        = aoFields     # type: List[ArmEncodesetField]
        self.fFields         = fFields;
        self.oCondition      = oCondition;

        assert ArmInstructionBase.s_oReValidName.match(sName), '%s' % (sName);
        assert (oJson['_type'] == 'Instruction.InstructionSet') == (oParent is None);
        assert not oParent or isinstance(oParent, (ArmInstructionGroup, ArmInstructionSet));


    def getUpIterator(self):
        """ Get an iterator the starts with 'self' and goes up the parent chain. """
        class UpIterator(object):
            def __init__(self, oNode):
                self.oNext = oNode;

            def __iter__(self):
                return self;

            def __next__(self):
                oRet = self.oNext;
                if oRet:
                    self.oNext = oRet.oParent;
                    return oRet;
                raise StopIteration();
        return UpIterator(self);

    def getFieldByName(self, sName, fRaiseXpctIfNotFound = True):
        """ Looks up a named field in the aoFields. """
        for oField in self.aoFields:
            if oField.sName and oField.sName == sName:
                return oField;
        if fRaiseXpctIfNotFound:
            raise Exception('Could not find field %s in instruction %s' % (sName, self.sName,));
        return None;


class ArmInstructionOrganizerBase(ArmInstructionBase):
    """ Common base class for ArmInstructionSet and ArmInstructionGroup. """

    s_oReValidName = re.compile('^[_A-Za-z][_A-Za-z0-9]+$');

    def __init__(self, oJson, sName, aoFields, fFields, oCondition, oParent = None):
        ArmInstructionBase.__init__(self, oJson, sName, aoFields, fFields, oCondition, oParent);
        self.aoInstructions     = [];   ##< The instruction directly under this object (no in sub-set or groups).
        self.dInstructions      = {};   ##< The instructions in self.aoInstructions indexed by name.
        self.aoAllInstructions  = [];   ##< All the instructions in this set.
        self.dAllInstructions   = {};   ##< The instructions in self.aoAllInstructions indexed by name.
        self.aoGroups           = [];   ##< Groups under this object.

    def toString(self):
        return '%s-name=%s Fields=#%u/%#010x cond=%s parent=%s' \
             % ('set' if isinstance(self, ArmInstructionSet) else 'group', self.sName,
                len(self.aoFields), self.fFields, self.oCondition.toString(), self.oParent.sName if self.oParent else '<none>',);

    def addInstruction(self, oInstr):
        """ Recursively adds the instruction to the ALL collections."""
        self.aoAllInstructions.append(oInstr);
        assert oInstr.sName not in self.dAllInstructions;
        self.dAllInstructions[oInstr.sName] = oInstr;

        if self.oParent:
            self.oParent.addInstruction(oInstr);
        else:
            g_dAllArmInstructionsBySet[self.sName].append(oInstr);

    def addImmediateInstruction(self, oInstr):
        """ Adds an instruction immediately below this group/set. """
        assert oInstr.sName not in self.dInstructions;
        assert oInstr.oParent == self;

        self.aoInstructions.append(oInstr);
        self.dInstructions[oInstr.sName] = oInstr;

        if self.oParent:
            assert isinstance(self, ArmInstructionGroup);
            g_dAllArmInstructionsByGroup[self.sName].append(oInstr);

        self.addInstruction(oInstr);

class ArmInstructionSet(ArmInstructionOrganizerBase):
    """ Representation of a Instruction.InstructionSet object. """

    def __init__(self, oJson, sName, aoFields, fFields, oCondition, cBitsWidth):
        ArmInstructionOrganizerBase.__init__(self, oJson, sName, aoFields, fFields, oCondition);
        self.cBitsWidth = cBitsWidth;
        assert cBitsWidth == 32;

    def __str__(self):
        return self.toString();

    def __repr__(self):
        return self.toString();

    def toString(self):
        return ArmInstructionOrganizerBase.toString(self) + ' read_bits=%u' % (self.cBitsWidth,);

    @staticmethod
    def fromJson(oJson):
        assert oJson['_type'] == 'Instruction.InstructionSet';
        sName = oJson['name'];

        (aoFields, fFields) = ArmEncodesetField.encodesetFromJson(oJson['encoding']);
        oCondition          = ArmAstBase.fromJson(oJson['condition']);
        print('debug: Instruction set %s' % (sName,));
        return ArmInstructionSet(oJson, sName, aoFields, fFields, oCondition, int(oJson['read_width']));


class ArmInstructionGroup(ArmInstructionOrganizerBase):
    """ Representation of a Instruction.InstructionGroup object. """

    def __init__(self, oJson, sName, aoFields, fFields, oCondition, oParent):
        ArmInstructionOrganizerBase.__init__(self, oJson, sName, aoFields, fFields, oCondition, oParent);

    def __str__(self):
        return self.toString();

    def __repr__(self):
        return self.toString();

    def toString(self):
        return ArmInstructionOrganizerBase.toString(self);

    @staticmethod
    def fromJson(oJson, oParent):
        assert oJson['_type'] == 'Instruction.InstructionGroup';
        sName = oJson['name'];

        (aoFields, fFields) = ArmEncodesetField.encodesetFromJson(oJson['encoding']);
        oCondition          = ArmAstBase.fromJson(oJson['condition']);
        #print('debug: Instruction group %s' % (sName,));
        return ArmInstructionGroup(oJson, sName, aoFields, fFields, oCondition, oParent);



class ArmInstruction(ArmInstructionBase):
    """
    ARM instruction
    """
    s_oReValidName = re.compile('^[_A-Za-z][_A-Za-z0-9]+$');

    def __init__(self, oJson, sName, sMemonic, sAsmDisplay, aoFields, fFields, oCondition, oParent):
        ArmInstructionBase.__init__(self, oJson, sName, aoFields, fFields, oCondition, oParent);
        self.sMnemonic       = sMemonic;
        self.sAsmDisplay     = sAsmDisplay;
        self.fFixedMask      = 0;
        self.fFixedValue     = 0;
        for oField in aoFields:
            self.fFixedMask  |= oField.fFixed << oField.iFirstBit;
            self.fFixedValue |= oField.fValue << oField.iFirstBit;

        # State related to decoder.
        self.fDecoderLeafCheckNeeded = False;    ##< Whether we need to check fixed value/mask in leaf decoder functions.

        # Check input.
        assert self.s_oReValidName.match(sName), 'sName=%s' % (sName);

    def toString(self, cchName = 0, fEncoding = False):
        if self.sName == self.sMnemonic:
            sRet = 'sName=%-*s' % (cchName, self.sName,);
        else:
            sRet = 'sName=%-*s sMnemonic=%-*s' % (cchName, self.sName, cchName, self.sMnemonic);
        if not fEncoding:
            return '%s fFixedValue/Mask=%#x/%#x #encoding=%s' % (sRet, self.fFixedValue, self.fFixedMask, len(self.aoFields));
        return '%s fFixedValue/Mask=%#x/%#x encoding=\n    %s' \
             % (sRet, self.fFixedValue, self.fFixedMask, ',\n    '.join([str(s) for s in self.aoFields]),);

    def __str__(self):
        return self.toString();

    def __repr__(self):
        return self.toString();

    def getNamedNonFixedFieldsSortedByPosition(self):
        """
        Gets aoFields filtered by sName and fFixed != getMask(), sorted by iFirstBit.

        Note! This is used for generating argument lists.
        """
        return sorted([oField for oField in self.aoFields if oField.sName and oField.fFixed != oField.getMask()],
                      key = operator.attrgetter('iFirstBit'));

    def getCName(self):
        # Get rid of trailing underscore as it seems pointless.
        if self.sName[-1] != '_' or self.sName[:-1] in g_dAllArmInstructionsByName:
            return self.sName;
        return self.sName[:-1];

    def getInstrSetName(self):
        """ Returns the instruction set name. """
        oCur = self.oParent;
        while True:
            oParent = oCur.oParent;
            if oParent:
                oCur = oParent;
            else:
                return oCur.sName;

    def getSetAndGroupNames(self):
        asNames = [];
        oParent = self.oParent;
        while oParent:
            asNames.append(oParent.sName)
            oParent = oParent.oParent;
        return asNames;

    def getSetAndGroupNamesWithLabels(self):
        asNames = self.getSetAndGroupNames();
        if len(asNames) > 1:
            return 'Instruction Set: %s  Group%s: %s' % (asNames[-1], 's' if len(asNames) > 2 else '', ', '.join(asNames[:-1]),);
        return 'Instruction Set: %s' % (asNames[-1],);



#
# Features and their properties.
#

## To deal with bugs, things we get wrong and, 'missing' stuff.
g_dArmFeatureSupportExprOverrides = {
    'FEAT_AA32':    ArmAstBinaryOp(ArmAstFunction('UInt', [ArmAstField('EL0', 'ID_AA64PFR0_EL1'),]), '==', ArmAstInteger(2)),
    'FEAT_AA32EL0': ArmAstBinaryOp(ArmAstFunction('UInt', [ArmAstField('EL0', 'ID_AA64PFR0_EL1'),]), '==', ArmAstInteger(2)),
    'FEAT_AA64EL2': ArmAstBinaryOp(ArmAstFunction('UInt', [ArmAstField('EL2', 'ID_AA64PFR0_EL1'),]), '>=', ArmAstInteger(1)),
    'FEAT_AA64EL3': ArmAstBinaryOp(ArmAstFunction('UInt', [ArmAstField('EL2', 'ID_AA64PFR0_EL1'),]), '>=', ArmAstInteger(1)),

    # Obsolete/whatever:
    'FEAT_AA32EL1': ArmAstBinaryOp(ArmAstFunction('UInt', [ArmAstField('EL1', 'ID_AA64PFR0_EL1'),]), '==', ArmAstInteger(2)),
    'FEAT_AA32EL2': ArmAstBinaryOp(ArmAstFunction('UInt', [ArmAstField('EL2', 'ID_AA64PFR0_EL1'),]), '==', ArmAstInteger(2)),
    'FEAT_AA32EL3': ArmAstBinaryOp(ArmAstFunction('UInt', [ArmAstField('EL3', 'ID_AA64PFR0_EL1'),]), '==', ArmAstInteger(2)),
    'FEAT_AA64EL0': ArmAstBinaryOp(ArmAstFunction('UInt', [ArmAstField('EL0', 'ID_AA64PFR0_EL1'),]), '>=', ArmAstInteger(1)),
    'FEAT_AA64EL1': ArmAstBinaryOp(ArmAstFunction('UInt', [ArmAstField('EL1', 'ID_AA64PFR0_EL1'),]), '>=', ArmAstInteger(1)),
    'FEAT_AA64':    ArmAstBinaryOp(ArmAstBinaryOp(ArmAstIdentifier('FEAT_AA64EL0'), '||', ArmAstIdentifier('FEAT_AA64EL1')),
                                   '||',
                                   ArmAstBinaryOp(ArmAstIdentifier('FEAT_AA64EL2'), '||', ArmAstIdentifier('FEAT_AA64EL3'))),

    # Spec bugs:
    'FEAT_S2FWB':   ArmAstBinaryOp(ArmAstFunction('UInt', [ArmAstField('FWB', 'ID_AA64MMFR2_EL1'),]), '>=', ArmAstInteger(1)),
    'FEAT_UAO':     ArmAstBinaryOp(ArmAstFunction('UInt', [ArmAstField('UAO', 'ID_AA64MMFR2_EL1'),]), '>=', ArmAstInteger(1)),

    # Spec bugs in 2024-12:
    'FEAT_PMUv3_TH2': ArmAstBinaryOp(ArmAstFunction('UInt', [ArmAstField('EDGE', 'PMMIR_EL1'),]), '>=', ArmAstInteger(2)),
    'FEAT_PACIMP': ArmAstBinaryOp.andListToTree([
        ArmAstBinaryOp(ArmAstFunction('UInt', [ArmAstField('GPI', 'ID_AA64ISAR1_EL1'),]), '>=', ArmAstInteger(1)),
        ArmAstBinaryOp(ArmAstFunction('UInt', [ArmAstField('API', 'ID_AA64ISAR1_EL1'),]), '>=', ArmAstInteger(1)),
    ]),
    'FEAT_PACQARMA5': ArmAstBinaryOp.andListToTree([
        ArmAstBinaryOp(ArmAstFunction('UInt', [ArmAstField('GPA', 'ID_AA64ISAR1_EL1'),]), '>=', ArmAstInteger(1)),
        ArmAstBinaryOp(ArmAstFunction('UInt', [ArmAstField('APA', 'ID_AA64ISAR1_EL1'),]), '>=', ArmAstInteger(1)),
    ]),
    'FEAT_PACQARMA3': ArmAstBinaryOp.andListToTree([
        ArmAstBinaryOp(ArmAstFunction('UInt', [ArmAstField('GPA3', 'ID_AA64ISAR2_EL1'),]), '>=', ArmAstInteger(1)),
        ArmAstBinaryOp(ArmAstFunction('UInt', [ArmAstField('APA3', 'ID_AA64ISAR2_EL1'),]), '>=', ArmAstInteger(1)),
    ]),

    # Missing info:
    'FEAT_F8F16MM': ArmAstField('F8MM4', 'ID_AA64FPFR0_EL1'),
    'FEAT_F8F32MM': ArmAstField('F8MM8', 'ID_AA64FPFR0_EL1'),
    'FEAT_SVE_F16F32MM': ArmAstBinaryOp(ArmAstFunction('UInt', [ArmAstField('F16MM', 'ID_AA64ZFR0_EL1'),]),
                                        '>=', ArmAstInteger(1)),
    'FEAT_PAuth':   ArmAstBinaryOp.orListToTree([
        ArmAstBinaryOp(ArmAstFunction('UInt', [ArmAstField('API',  'ID_AA64ISAR1_EL1'),]), '>=', ArmAstInteger(1)),
        ArmAstBinaryOp(ArmAstFunction('UInt', [ArmAstField('APA',  'ID_AA64ISAR1_EL1'),]), '>=', ArmAstInteger(1)),
        ArmAstBinaryOp(ArmAstFunction('UInt', [ArmAstField('APA3', 'ID_AA64ISAR2_EL1'),]), '>=', ArmAstInteger(1)),
    ]),

    ## @todo FEAT_GICv3, FEAT_GICv3p, FEAT_GICv4 & FEAT_GICv4p1 detection is most probably incomplete or wrong.
    'FEAT_GICv3':   ArmAstBinaryOp(ArmAstFunction('UInt', [ArmAstField('GIC', 'ID_AA64PFR0_EL1'),]), '>=', ArmAstInteger(1)),
    'FEAT_GICv3p1': ArmAstBinaryOp(ArmAstFunction('UInt', [ArmAstField('GIC', 'ID_AA64PFR0_EL1'),]), '>=', ArmAstInteger(1)),
    'FEAT_GICv4':   ArmAstBinaryOp(ArmAstFunction('UInt', [ArmAstField('GIC', 'ID_AA64PFR0_EL1'),]), '>=', ArmAstInteger(1)),
    'FEAT_GICv4p1': ArmAstBinaryOp(ArmAstFunction('UInt', [ArmAstField('GIC', 'ID_AA64PFR0_EL1'),]), '>=', ArmAstInteger(3)),
    'FEAT_GICv3_NMI':  ArmAstBinaryOp(ArmAstIdentifier('FEAT_GICv3'), '&&', ArmAstIdentifier('FEAT_NMI')), # ?? from linux patch
    'FEAT_GICv3_TDIR': ArmAstBinaryOp(ArmAstIdentifier('FEAT_GICv3'), '&&', ArmAstField('TDS', 'ICH_VTR_EL2'),), # ??

    # Missing in 2024-12:
    'FEAT_SSVE_FEXPA': ArmAstField('SFEXPA', 'ID_AA64SMFR0_EL1'),

    # Odd ones:
    'FEAT_CHK':     ArmAstIdentifier('FEAT_GCS'),
    'FEAT_ETE':     ArmAstBinaryOp(ArmAstFunction('UInt', [ArmAstField('TraceVer', 'ID_AA64DFR0_EL1'),]),
                                   '>=', ArmAstInteger(1)),
    'FEAT_ETEv1p1': ArmAstBinaryOp(ArmAstIdentifier('FEAT_ETE'), '&&',
                                   ArmAstBinaryOp(ArmAstFunction('UInt', [ArmAstField('REVISION', 'TRCDEVARCH'),]),
                                                  '>=', ArmAstInteger(1))),
    'FEAT_ETEv1p2': ArmAstBinaryOp(ArmAstIdentifier('FEAT_ETE'), '&&',
                                   ArmAstBinaryOp(ArmAstFunction('UInt', [ArmAstField('REVISION', 'TRCDEVARCH'),]),
                                                  '>=', ArmAstInteger(2))),
    'FEAT_ETEv1p3': ArmAstBinaryOp(ArmAstIdentifier('FEAT_ETE'), '&&',
                                   ArmAstBinaryOp(ArmAstFunction('UInt', [ArmAstField('REVISION', 'TRCDEVARCH'),]),
                                                  '>=', ArmAstInteger(3))),

    'FEAT_ETMv4':   ArmAstBinaryOp.andListToTree([ # See 7.3.22 in IHI0064H_b_etm_v4_architecture_specification.pdf
            ArmAstIdentifier('FEAT_ETE'),
            ArmAstBinaryOp(ArmAstFunction('UInt', [ArmAstField('PRESENT',   'TRCDEVARCH'),]), '==', ArmAstInteger(1)),
            ArmAstBinaryOp(ArmAstFunction('UInt', [ArmAstField('ARCHITECT', 'TRCDEVARCH'),]), '==', ArmAstInteger(0x23b)),
            ArmAstBinaryOp(ArmAstFunction('UInt', [ArmAstField('ARCHVER',   'TRCDEVARCH'),]), '==', ArmAstInteger(4)),
            ArmAstBinaryOp(ArmAstFunction('UInt', [ArmAstField('ARCHPART',  'TRCDEVARCH'),]), '==', ArmAstInteger(0xa13)),
        ]),
    'FEAT_ETMv4p1': ArmAstBinaryOp.andListToTree([
        ArmAstIdentifier('FEAT_ETMv4'),
        ArmAstBinaryOp(ArmAstFunction('UInt', [ArmAstField('REVISION', 'TRCDEVARCH'),]), '>=', ArmAstInteger(1)),
    ]),
    'FEAT_ETMv4p2': ArmAstBinaryOp.andListToTree([
        ArmAstIdentifier('FEAT_ETMv4'),
        ArmAstBinaryOp(ArmAstFunction('UInt', [ArmAstField('REVISION', 'TRCDEVARCH'),]), '>=', ArmAstInteger(2)),
    ]),
    'FEAT_ETMv4p3': ArmAstBinaryOp.andListToTree([
        ArmAstIdentifier('FEAT_ETMv4'),
        ArmAstBinaryOp(ArmAstFunction('UInt', [ArmAstField('REVISION', 'TRCDEVARCH'),]), '>=', ArmAstInteger(3)),
    ]),
    'FEAT_ETMv4p4': ArmAstBinaryOp.andListToTree([
        ArmAstIdentifier('FEAT_ETMv4'),
        ArmAstBinaryOp(ArmAstFunction('UInt', [ArmAstField('REVISION', 'TRCDEVARCH'),]), '>=', ArmAstInteger(4)),
    ]),
    'FEAT_ETMv4p5': ArmAstBinaryOp.andListToTree([
        ArmAstIdentifier('FEAT_ETMv4'),
        ArmAstBinaryOp(ArmAstFunction('UInt', [ArmAstField('REVISION', 'TRCDEVARCH'),]), '>=', ArmAstInteger(5)),
    ]),
    'FEAT_ETMv4p6': ArmAstBinaryOp.andListToTree([
        ArmAstIdentifier('FEAT_ETMv4'),
        ArmAstBinaryOp(ArmAstFunction('UInt', [ArmAstField('REVISION', 'TRCDEVARCH'),]), '>=', ArmAstInteger(6)),
    ]),
    'FEAT_VPIPT': ArmAstBinaryOp(ArmAstFunction('UInt', [ArmAstField('L1Ip', 'CTR_EL0'),]), '==', ArmAstInteger(2)), # removed
    #'FEAT_RASSAv1p1': ArmAstIdentifier('FEAT_RASv1p1'), # Incomplete detection. ARMv8.4 implies this, but optional back to v8.2.

    # Too complex detection logic.
    'FEAT_LPA2': ArmAstBinaryOp.orListToTree([ # Possible that OR is slightly wrong here, but it's the simpler way out...
        ArmAstBinaryOp(ArmAstFunction('UInt', [ArmAstField('TGran4',    'ID_AA64MMFR0_EL1'),]), '>=', ArmAstInteger(1)),
        ArmAstBinaryOp(ArmAstFunction('UInt', [ArmAstField('TGran16',   'ID_AA64MMFR0_EL1'),]), '>=', ArmAstInteger(2)),
        ArmAstBinaryOp(ArmAstFunction('UInt', [ArmAstField('TGran4_2',  'ID_AA64MMFR0_EL1'),]), '>=', ArmAstInteger(3)),
        ArmAstBinaryOp(ArmAstFunction('UInt', [ArmAstField('TGran16_2', 'ID_AA64MMFR0_EL1'),]), '>=', ArmAstInteger(3)),
    ]),
    'FEAT_SME2p2': ArmAstBinaryOp.andListToTree([ ArmAstIdentifier('FEAT_SME'), ArmAstIdentifier('FEAT_SVE2p2'), ]),
};

class ArmFeature(object):
    """
    ARM instruction set feature.
    """

    def __init__(self, oJson, sName, aoConstraints, sType = 'Parameters.Boolean'):
        self.oJson          = oJson;
        self.sName          = sName;
        self.aoConstraints  = aoConstraints;
        self.sType          = sType;

        # Find the best is-supported expression, prioritizing registers in the
        # AArch64 set over AArch32 and ext.
        dSupportExpr = {};
        for oConstraint in aoConstraints:
            oExpr = self.extractFeatureIndicator(oConstraint, sName);
            if oExpr:
                dSupportExpr[oExpr.toString()] = oExpr;

        def CalcWeight(sStr):
            iRet = 10 if sStr.find('AArch64.') >= 0 else 0;
            if sStr.find('AArch32.') >= 0: iRet -= 1;
            if sStr.find('ext.') >= 0:     iRet -= 2;
            return iRet;

        def CmpExpr(sExpr1, sExpr2):
            iWeight1 = CalcWeight(sExpr1);
            iWeight2 = CalcWeight(sExpr2);
            if iWeight1 != iWeight2:
                return -1 if iWeight1 > iWeight2 else 1;
            if sExpr1 == sExpr2:
                return 0;
            return -1 if sExpr1 < sExpr2 else 1;

        asKeys = sorted(dSupportExpr.keys(), key = functools.cmp_to_key(CmpExpr));
        self.oSupportExpr = dSupportExpr[asKeys[0]] if asKeys else None;

        # Manual overrides and fixes.
        if sName in g_dArmFeatureSupportExprOverrides:
            self.oSupportExpr = g_dArmFeatureSupportExprOverrides[sName];
            #print('debug3: sName=%s override: %s' % (sName, self.oSupportExpr.toString()))

        # Find the variables in the expression (for optimizing explosion).
        self.asSupportExprVars = sorted(set(ArmFeature.extractVariables(self.oSupportExpr)));

    @staticmethod
    def extractFeatureIndicator(oConstraint, sFeature):
        """
        Returns the part of the constrains AST tree that (probably) detects the feature.
        Returns None if not found.

        Analyze the constraints and try figure out if there is a single system
        register that indicates this feature or not.  This is generally found
        as a FEAT_SELF <-> UInt(reg.field) >= x.  It may be nested and repeated.
        """
        if isinstance(oConstraint, ArmAstBinaryOp):
            # Is it a 'sFeature <-> sExpr'?
            if (    oConstraint.sOp == '<->'
                and isinstance(oConstraint.oLeft, ArmAstIdentifier)
                and oConstraint.oLeft.sName == sFeature):
                return oConstraint.oRight;

            # If this is an '-->' (implies) operator and the left side is our
            # feature, then the right side describing things implied by it.
            # If the operator is used with something else on the left side,
            # those are preconditions for the feature and the right hand side
            # may contain the <-> expression we're seeking.
            if (    oConstraint.sOp == '-->'
                and (   not isinstance(oConstraint.oLeft, ArmAstIdentifier)
                     or oConstraint.oLeft.sName != sFeature)):
                return ArmFeature.extractFeatureIndicator(oConstraint.oRight, sFeature);
        return None;

    @staticmethod
    def extractVariables(oConstraint):
        """ Returns a list of the identifiers as a list of strings. """
        if not oConstraint:
            return [];
        if isinstance(oConstraint, ArmAstIdentifier):
            return [ oConstraint.sName, ];
        if isinstance(oConstraint, ArmAstDotAtom):
            return [ oConstraint.toString(), ];
        if isinstance(oConstraint, ArmAstBinaryOp):
            return ArmFeature.extractVariables(oConstraint.oLeft) + ArmFeature.extractVariables(oConstraint.oRight);
        if isinstance(oConstraint, ArmAstField):
            return [oConstraint.sState + '.' + oConstraint.sName, ];

        aoRet = [];
        if isinstance(oConstraint, ArmAstFunction):
            for oArg in oConstraint.aoArgs:
                aoRet += ArmFeature.extractVariables(oArg);
        elif isinstance(oConstraint, (ArmAstSet, ArmAstConcat)):
            for oValue in oConstraint.aoValues:
                aoRet += ArmFeature.extractVariables(oValue);
        return aoRet;


    @staticmethod
    def fromJson(oJson):
        sType = oJson['_type'];
        assert sType in ('Parameters.Boolean', 'Parameters.Integer',), '_type=%s' % (oJson['_type'],);
        sName         = oJson['name'];
        aoConstraints = [ArmAstBase.fromJson(oItem, ArmAstBase.ksModeConstraints) for oItem in oJson['constraints']];
        return ArmFeature(oJson, sName, aoConstraints, sType);


#
# System Registers and their properties.
#

class ArmRange(object):
    """ Bit range in a field. """
    def __init__(self, iFirstBit, cBitsWidth):
        self.iFirstBit  = iFirstBit;
        self.cBitsWidth = cBitsWidth;


class ArmFieldsBase(object):
    """ Base class for all fields. """

    def __init__(self, oParent, aoRanges, sName):
        self.oParent  = oParent;
        self.aoRanges = aoRanges    # Type: List[ArmRange]
        self.sName    = sName;

    def toString(self):
        """ Approximate string representation of the field. """
        if len(self.aoRanges) == 1:
            return '%s@%u:%u' % (self.sName, self.aoRanges[0].iFirstBit, self.aoRanges[0].cBitsWidth,);
        return '%s@%s' % (self.sName, ','.join(['%u:%u' % (oRange.iFirstBit, oRange.cBitsWidth) for oRange in self.aoRanges]),);

    @staticmethod
    def addFieldListToLookupDict(daoFields, aoFields):
        """
        Adds the fields from aoFields to the lookup dictionary daoFields (returning it).

        The daoFields is indexed by field name and each entry is a list of fields
        by that name.
        """
        for oField in aoFields:
            if oField.sName:
                if oField.sName not in daoFields:
                    daoFields[oField.sName]  = [oField];
                else:
                    daoFields[oField.sName] += [oField];
            if isinstance(oField, ArmFieldsConditionalField):
                ArmFieldsBase.addFieldListToLookupDict(daoFields, [oCondField for _, oCondField in oField.atCondFields]);
        return daoFields;


    @staticmethod
    def rangesFromJson(adJson):
        """ Converts the rangeset array to a list of ranges. """
        aoRet = [];
        for dJson in adJson:
            assert dJson['_type'] == 'Range';
            aoRet.append(ArmRange(int(dJson['start']), int(dJson['width'])));
        return aoRet;


    kAttribSetReserved = frozenset(['_type', 'description', 'rangeset', 'value']);
    @staticmethod
    def fromJsonReserved(dJson, oParent):
        assertJsonAttribsInSet(dJson, ArmFieldsBase.kAttribSetReserved);
        return ArmFieldsReserved(oParent, ArmFieldsBase.rangesFromJson(dJson['rangeset']));


    kAttribSetImplementationDefined = frozenset(['_type', 'constraints', 'description', 'display', 'name',
                                                 'rangeset', 'resets', 'volatile']);
    @staticmethod
    def fromJsonImplementationDefined(dJson, oParent):
        assertJsonAttribsInSet(dJson, ArmFieldsBase.kAttribSetImplementationDefined);
        return ArmFieldsImplementationDefined(oParent, ArmFieldsBase.rangesFromJson(dJson['rangeset']), dJson['name']);


    kAttribSetField = frozenset(['_type', 'access', 'description', 'display', 'name',
                                 'rangeset', 'resets', 'values', 'volatile']);
    @staticmethod
    def fromJsonField(dJson, oParent):
        assertJsonAttribsInSet(dJson, ArmFieldsBase.kAttribSetField);
        return ArmFieldsField(oParent, ArmFieldsBase.rangesFromJson(dJson['rangeset']), dJson['name']);


    kAttribSetConditionalField = frozenset(['_type', 'description', 'display', 'fields', 'name',
                                            'rangeset', 'reservedtype', 'resets', 'volatile']);
    kAttribSetConditionalFieldEntry = frozenset(['condition', 'field']);
    @staticmethod
    def fromJsonConditionalField(dJson, oParent):
        assertJsonAttribsInSet(dJson, ArmFieldsBase.kAttribSetConditionalField);
        atCondFields = [];
        oNew = ArmFieldsConditionalField(oParent, ArmFieldsBase.rangesFromJson(dJson['rangeset']), dJson['name'], atCondFields);
        for dJsonField in dJson['fields']:
            assertJsonAttribsInSet(dJsonField, ArmFieldsBase.kAttribSetConditionalFieldEntry);
            atCondFields.append((ArmAstBase.fromJson(dJsonField['condition'], ArmAstBase.ksModeConstraints),
                                 ArmFieldsBase.fromJson(dJsonField['field'], oNew)));
        return oNew;


    kAttribSetConstantField = frozenset(['_type', 'access', 'description', 'name', 'rangeset', 'value']);
    @staticmethod
    def fromJsonConstantField(dJson, oParent):
        assertJsonAttribsInSet(dJson, ArmFieldsBase.kAttribSetConstantField);
        return ArmFieldsConstantField(oParent, ArmFieldsBase.rangesFromJson(dJson['rangeset']), dJson['name']);


    kAttribSetArray = frozenset(['_type', 'access', 'description', 'display', 'index_variable', 'indexes', 'name',
                                 'rangeset', 'resets' , 'values', 'volatile']);
    @staticmethod
    def fromJsonArray(dJson, oParent):
        assertJsonAttribsInSet(dJson, ArmFieldsBase.kAttribSetArray);
        aoIndexes = ArmFieldsArray.Index.fromJsonArray(dJson['indexes']);
        aoRanges  = ArmFieldsBase.rangesFromJson(dJson['rangeset']);
        assert len(aoIndexes) <= len(aoRanges), \
               '%s: aoIndexes=%s\naoRanges=%s' % (dJson['name'], dJson['indexes'], dJson['rangeset'],);

        cIndexSteps   = sum(oIndex.cSteps for oIndex in aoIndexes);
        cBitsRanges   = sum(oRange.cBitsWidth for oRange in aoRanges);
        cBitsPerEntry = cBitsRanges // cIndexSteps;
        assert cBitsPerEntry * cIndexSteps == cBitsRanges;

        return ArmFieldsArray(oParent, aoRanges, dJson['name'], dJson['index_variable'], aoIndexes, cIndexSteps, cBitsPerEntry);


    kAttribSetVector = frozenset(['reserved_type', 'size',]) | kAttribSetArray;
    @staticmethod
    def fromJsonVector(dJson, oParent):
        assertJsonAttribsInSet(dJson, ArmFieldsBase.kAttribSetVector);
        aoIndexes = ArmFieldsArray.Index.fromJsonArray(dJson['indexes']);
        aoRanges  = ArmFieldsBase.rangesFromJson(dJson['rangeset']);
        assert len(aoIndexes) <= len(aoRanges), \
               '%s: aoIndexes=%s\naoRanges=%s' % (dJson['name'], dJson['indexes'], dJson['rangeset'],);

        cIndexSteps   = sum(oIndex.cSteps for oIndex in aoIndexes);
        cBitsRanges   = sum(oRange.cBitsWidth for oRange in aoRanges);
        cBitsPerEntry = cBitsRanges // cIndexSteps;
        assert cBitsPerEntry * cIndexSteps == cBitsRanges;

        atCondSizes = [];
        for dJsonSize in dJson['size']:
            atCondSizes.append((ArmAstBase.fromJson(dJsonSize['condition']),
                                ArmAstBase.fromJson(dJsonSize['value'], ArmAstBase.ksModeConstraints))); ## @todo hackish mode

        return ArmFieldsVector(oParent, aoRanges, dJson['name'], dJson['index_variable'], aoIndexes, cIndexSteps,
                               cBitsPerEntry, atCondSizes);


    kAttribSetDynamic = frozenset(['_type', 'description', 'display', 'instances', 'name', 'rangeset', 'resets', 'volatile']);
    @staticmethod
    def fromJsonDynamic(dJson, oParent):
        assertJsonAttribsInSet(dJson, ArmFieldsBase.kAttribSetDynamic);
        return ArmFieldsDynamic(oParent, ArmFieldsBase.rangesFromJson(dJson['rangeset']), dJson['name'],
                                [ArmFieldset.fromJson(dJsonFieldSet) for dJsonFieldSet in dJson['instances']]);


    kfnTypeMap = {
        'Fields.Reserved':                  fromJsonReserved,
        'Fields.ImplementationDefined':     fromJsonImplementationDefined,
        'Fields.Field':                     fromJsonField,
        'Fields.ConditionalField':          fromJsonConditionalField,
        'Fields.ConstantField':             fromJsonConstantField,
        'Fields.Array':                     fromJsonArray,
        'Fields.Dynamic':                   fromJsonDynamic,
        'Fields.Vector':                    fromJsonVector,
    };

    @staticmethod
    def fromJson(dJson, oParent = None):
        """ Decodes a field. """
        return ArmFieldsBase.kfnTypeMap[dJson['_type']](dJson, oParent);


class ArmFieldsReserved(ArmFieldsBase):
    """ Fields.Reserved """
    def __init__(self, oParent, aoRanges):
        ArmFieldsBase.__init__(self, oParent, aoRanges, None);


class ArmFieldsField(ArmFieldsBase):
    """ Fields.Field """
    def __init__(self, oParent, aoRanges, sName):
        ArmFieldsBase.__init__(self, oParent, aoRanges, sName);


class ArmFieldsImplementationDefined(ArmFieldsBase):
    """ Fields.ImplementationDefined """
    def __init__(self, oParent, aoRanges, sName):
        ArmFieldsBase.__init__(self, oParent, aoRanges, sName);


class ArmFieldsConditionalField(ArmFieldsBase):
    """ Fields.ConditionalField """
    def __init__(self, oParent, aoRanges, sName, atCondFields):
        ArmFieldsBase.__init__(self, oParent, aoRanges, sName);
        self.atCondFields = atCondFields        # Type: List[Tuple(ArmAstBase,ArmFieldsBase)]


class ArmFieldsConstantField(ArmFieldsBase):
    """ Fields.ConstantField """
    def __init__(self, oParent, aoRanges, sName):
        ArmFieldsBase.__init__(self, oParent, aoRanges, sName);


class ArmFieldsArray(ArmFieldsBase):
    """ Fields.Array """

    class Index(object):
        """ Helper for ArmFieldsArray. """
        def __init__(self, idxStart, cSteps):
            self.idxStart = idxStart;
            self.idxEnd   = idxStart + cSteps;
            self.cSteps   = cSteps;

        @staticmethod
        def fromJson(dJson):
            assert dJson['_type'] == 'Range';
            return ArmFieldsArray.Index(int(dJson['start']), int(dJson['width']));

        @staticmethod
        def fromJsonArray(adJson):
            return [ArmFieldsArray.Index.fromJson(dJsonEntry) for dJsonEntry in adJson];

    def __init__(self, oParent, aoRanges, sName, sIdxVarNm, aoIndexes, cEntries, cBitsPerEntry):
        ArmFieldsBase.__init__(self, oParent, aoRanges, sName);
        self.sIdxVarNm     = sIdxVarNm;
        self.aoIndexes     = aoIndexes      # Type: List[ArmFieldsArray.Index]
        self.cEntries      = cEntries;
        self.cBitsPerEntry = cBitsPerEntry;


class ArmFieldsVector(ArmFieldsArray):
    """ Fields.Vector """

    def __init__(self, oParent, aoRanges, sName, sIdxVarNm, aoIndexes, cEntries, cBitsPerEntry, atCondSizes):
        ArmFieldsArray.__init__(self, oParent, aoRanges, sName, sIdxVarNm, aoIndexes, cEntries, cBitsPerEntry);
        self.atCondSizes   = atCondSizes    # Type: List[Tuple[ArmAstBase,ArmAstBase]] # tuple(condition, size-expr)


class ArmFieldsDynamic(ArmFieldsBase):
    """ Fields.Dynamic """
    def __init__(self, oParent, aoRanges, sName, aoFieldsets):
        ArmFieldsBase.__init__(self, oParent, aoRanges, sName);
        self.aoFieldsets = aoFieldsets      # Type: List[ArmFieldset]



class ArmFieldset(object):
    """
    A register field set.
    """

    def __init__(self, dJson, cBitsWidth, aoFields, sName = None):
        self.dJson      = dJson;
        self.cBitsWidth = cBitsWidth;
        self.aoFields   = aoFields          # type: List[ArmFieldsBase]
        self.sName      = sName;

    def toString(self):
        """ Gets display string. """
        return '%u bits%s: %s' \
             % (self.cBitsWidth, ', %s' % (self.sName,) if self.sName else '',
                ', '.join([oField.toString() for oField in sorted(self.aoFields, key = lambda o: o.aoRanges[0].iFirstBit)]),);

    @staticmethod
    def fromJson(dJson):
        assert dJson['_type'] == 'Fieldset', '_type=%s' % (dJson['_type'],);
        oNew = ArmFieldset(dJson, int(dJson['width']), [], dJson['name']);
        oNew.aoFields = [ArmFieldsBase.fromJson(dJsonField, oNew) for dJsonField in dJson['values']];
        return oNew;


class ArmRegEncoding(object):
    """ Register encoding. """

    kdSortOrder = {
        'op0': 1,
        'op1': 2,
        'CRn': 3,
        'CRm': 4,
        'op2': 5,
    };

    def __init__(self, sAsmValue, dNamedValues):
        self.sAsmValue    = sAsmValue;
        self.dNamedValues = collections.OrderedDict();
        self.fHasWildcard = False;
        self.fHasIndex    = False;
        for sKey in sorted(dNamedValues, key = lambda s: ArmRegEncoding.kdSortOrder.get(s, 9)):
            oValue = dNamedValues[sKey];
            self.dNamedValues[sKey] = oValue;
            self.fHasWildcard |= 'x' in oValue.sValue;
            self.fHasIndex    |= '[' in oValue.sValue;

    def toString(self):
        return '%s={%s}' \
             % (self.sAsmValue, ', '.join(['%s=%s' % (sKey, oValue.toString()) for sKey, oValue in self.dNamedValues.items()]),);

    def __str__(self):
        return self.toString();

    def __repr__(self):
        return self.toString();

    def getSysRegIdCreate(self):
        """ Returns the corresponding ARMV8_AARCH64_SYSREG_ID_CREATE invocation. """
        assert len(self.dNamedValues) == len(ArmRegEncoding.kdSortOrder), '%s: %s' % (self.sAsmValue, self.dNamedValues,);
        asArgs = [];
        for sKey in ArmRegEncoding.kdSortOrder:
            (fValue, _, fWildcard, _) = ArmEncodesetField.parseValue(self.dNamedValues[sKey].sValue, 0);
            if fWildcard == 0:
                asArgs.append('%s' % (fValue,));
            else:
                oXcpt = Exception('wildcard encoding for %s: %s:%s' % (self.sAsmValue, sKey, self.dNamedValues[sKey],));
                print('wtf: %s' % (oXcpt,))
                raise oXcpt;
        return 'ARMV8_AARCH64_SYSREG_ID_CREATE(' + ','.join(asArgs) + ')';

    kAttribSet = frozenset(['_type', 'asmvalue', 'encodings']);
    @staticmethod
    def fromJson(dJson):
        """ Decodes a register encoding object. """
        assert dJson['_type'] == 'Encoding';
        assertJsonAttribsInSet(dJson, ArmRegEncoding.kAttribSet);
        dNamedValues = collections.OrderedDict();
        for sName, dValue in dJson['encodings'].items():
            dNamedValues[sName] = ArmAstBase.fromJson(dValue, ArmAstBase.ksModeValuesOnly);
        return ArmRegEncoding(dJson['asmvalue'], dNamedValues);


## @todo remove this.
class ArmAccessorPermissionBase(object):
    """
    Register accessor permission base class (Accessors.Permission.*).
    """

    def __init__(self, oCondition):
        self.oCondition = oCondition;


    kAttribSetMemory = frozenset(['_type', 'access', 'condition',]);
    @staticmethod
    def fromJsonMemory(dJson, fNested):
        _ = fNested;
        assert dJson['_type'] == 'Accessors.Permission.MemoryAccess';
        assertJsonAttribsInSet(dJson, ArmAccessorPermissionBase.kAttribSetMemory);
        oCondition = ArmAstBase.fromJson(dJson['condition'], ArmAstBase.ksModeConstraints)
        # The 'access' attribute comes in three variations: Accessors.Permission.MemoryAccess list,
        # Accessors.Permission.AccessTypes.Memory.ReadWriteAccess and
        # Accessors.Permission.AccessTypes.Memory.ImplementationDefined.
        oJsonAccess = dJson['access'];
        if isinstance(oJsonAccess, list):
            aoAccesses = [ArmAccessorPermissionBase.fromJsonMemory(dJsonSub, True) for dJsonSub in oJsonAccess]
            return ArmAccessorPermissionMemoryAccessList(oCondition, aoAccesses);

        if oJsonAccess['_type'] == 'Accessors.Permission.AccessTypes.Memory.ReadWriteAccess':
            return ArmAccessorPermissionMemoryAccess(oCondition, ArmAccessorPermissionMemReadWriteAccess.fromJson(oJsonAccess));

        if oJsonAccess['_type'] == 'Accessors.Permission.AccessTypes.Memory.ImplementationDefined':
            aoConstraints = [ArmAccessorPermissionMemReadWriteAccess.fromJson(dJsonConstr)
                            for dJsonConstr in oJsonAccess['constraints']];
            return ArmAccessorPermissionMemoryAccessImplDef(oCondition, aoConstraints);
        raise Exception('Unexpected access attr type: %s' % (oJsonAccess['_type'],));


    kfnTypeMap = {
        'Accessors.Permission.MemoryAccess': fromJsonMemory,
    };

    @staticmethod
    def fromJson(dJson, fNested = False):
        """ Decodes a register accessor object. """
        return ArmAccessorPermissionBase.kfnTypeMap[dJson['_type']](dJson, fNested);


class ArmAccessorPermissionMemReadWriteAccess(object):
    """ Accessors.Permission.AccessTypes.Memory.ReadWriteAccess """
    def __init__(self, sRead, sWrite):
        self.sRead  = sRead;
        self.sWrite = sWrite;

    kAttribSet = frozenset(['_type', 'read', 'write',]);
    @staticmethod
    def fromJson(dJson):
        assert dJson['_type'] == 'Accessors.Permission.AccessTypes.Memory.ReadWriteAccess';
        assertJsonAttribsInSet(dJson, ArmAccessorPermissionMemReadWriteAccess.kAttribSet);
        return ArmAccessorPermissionMemReadWriteAccess(dJson['read'], dJson['write']);


class ArmAccessorPermissionMemoryAccess(ArmAccessorPermissionBase):
    """ Accessors.Permission.MemoryAccess """
    def __init__(self, oCondition, oReadWriteAccess):
        ArmAccessorPermissionBase.__init__(self, oCondition);
        self.oReadWrite = oReadWriteAccess;


class ArmAccessorPermissionMemoryAccessImplDef(ArmAccessorPermissionBase):
    """ Accessors.Permission.MemoryAccess """
    def __init__(self, oCondition, aoConstraints):
        ArmAccessorPermissionBase.__init__(self, oCondition);
        self.aoConstraints = aoConstraints;


class ArmAccessorPermissionMemoryAccessList(ArmAccessorPermissionBase):
    """ Accessors.Permission.MemoryAccess """
    def __init__(self, oCondition, aoAccesses):
        ArmAccessorPermissionBase.__init__(self, oCondition);
        self.aoAccesses = aoAccesses;


class ArmAstIfList(ArmAstStatementBase):
    """
    Accessors.Permission.SystemAccess

    We make this part of the AST.

    A series of conditional actions or nested conditional series:
        if cond1:
            action1;
        elif cond2:
            if cond2a:  # nested series
                action2a;
            else:
                action2b;
        else:
            action3;
    """

    def __init__(self, aoIfConditions, aoIfStatements, oElseStatement):
        ArmAstStatementBase.__init__(self, 'Accessors.Permission.MemoryAccess');
        # The if/elif condition expressions.
        self.aoIfConditions  = aoIfConditions   # type: List[ArmAstBase]
        # The if/elif statements, runs in parallel to aoIfConditions. ArmAstIfList allowed.
        self.aoIfStatements  = aoIfStatements   # type: List[ArmAstBase]
        # The else statement - optional.  ArmAstIfList allowed.
        self.oElseStatement  = oElseStatement   # type: ArmAstBase

        # Assert sanity.
        assert len(aoIfConditions) == len(aoIfStatements);
        assert aoIfStatements or oElseStatement;  # Pretty lax at the moment.
        for oStmt in list(aoIfStatements):
            assert isinstance(oStmt, (ArmAstStatementBase, ArmAstFunction));
        assert oElseStatement is None or isinstance(oElseStatement, (ArmAstStatementBase, ArmAstFunction));

    def clone(self):
        return ArmAstIfList([oIfCond.clone() for oIfCond in self.aoIfConditions],
                            [oIfStmt.clone() for oIfStmt in self.aoIfStatements],
                            self.oElseStatement.clone() if self.oElseStatement else None);

    def isSame(self, oOther):
        if isinstance(oOther, ArmAstIfList):
            if len(self.aoIfConditions) == len(oOther.aoIfConditions):
                for i, oIfCond in enumerate(self.aoIfConditions):
                    if not oIfCond.isSame(oOther.aoIfConditions[i]):
                        return False;
                    if not self.aoIfStatements[i].isSame(oOther.aoIfStatements[i]):
                        return False;
                if (self.oElseStatement is None) == (oOther.oElseStatement is None):
                    if self.oElseStatement and not self.oElseStatement.isSame(oOther.oElseStatement):
                        return False;
                    return True;
        return False;

    def transform(self, fnCallback, fEliminationAllowed, oCallbackArg):
        aoNewIfConditions = [];
        aoNewIfStatements = [];
        for idxIf, oIfCond in enumerate(self.aoIfConditions):
            oIfCond = oIfCond.transform(fnCallback, True, oCallbackArg);
            if oIfCond and not oIfCond.isBoolAndFalse():
                oIfStmt = self.aoIfStatements[idxIf].transform(fnCallback, True, oCallbackArg);
                assert oIfStmt;
                aoNewIfConditions.append(oIfCond);
                aoNewIfStatements.append(oIfStmt);
        oNewElseStatement = self.oElseStatement.transform(fnCallback, True, oCallbackArg) if self.oElseStatement else None;

        if aoNewIfConditions:
            self.aoIfConditions = aoNewIfConditions;
            self.aoIfStatements = aoNewIfStatements;
            self.oElseStatement = oNewElseStatement;
            return fnCallback(self, fEliminationAllowed, oCallbackArg);
        if oNewElseStatement:
            return oNewElseStatement;
        if fEliminationAllowed:
            return None;
        return fnCallback(ArmAstNop(), fEliminationAllowed, oCallbackArg);

    def toStringList(self, sIndent = '', sLang = None):
        asLines = [];
        sNextIndent = sIndent + '    ';
        for i, oIfCond in enumerate(self.aoIfConditions):
            asLines.append('%s%sif (%s)' % (sIndent, 'else ' if i > 0 else '', oIfCond.toString(),));
            oIfStmt = self.aoIfStatements[i];
            if isinstance(oIfStmt, ArmAstStatementBase):
                asStmts = oIfStmt.toStringList(sNextIndent, sLang);
                if sLang == 'C' and len(asStmts) != 1:
                    asLines.append(sIndent + '{');
                    asLines.extend(asStmts);
                    asLines.append(sIndent + '}');
                else:
                    asLines.extend(asStmts);
            else:
                asLines.append(sNextIndent + oIfStmt.toString());

        if self.oElseStatement:
            if self.aoIfConditions:
                asLines.append(sIndent + 'else');
            else:
                sNextIndent = sIndent; # Trick.
            if isinstance(self.oElseStatement, ArmAstStatementBase):
                asStmts = self.oElseStatement.toStringList(sNextIndent, sLang);
                if sLang == 'C' and len(asStmts) != 1:
                    asLines.append(sIndent + '{');
                    asLines.extend(asStmts);
                    asLines.append(sIndent + '}');
                else:
                    asLines.extend(asStmts);
            else:
                asLines.append(sNextIndent + self.oElseStatement.toString());
        return asLines;


    kAttribSet = frozenset(['_type', 'access', 'condition',]);
    @staticmethod
    def fromJson(dJson, uDepth = 0): # pylint: disable=arguments-renamed
        #
        # There are two variants of this object.
        #
        if dJson['_type'] != 'Accessors.Permission.SystemAccess': raise Exception('wrong type: %s' % (dJson['_type'],));
        assertJsonAttribsInSet(dJson, ArmAstIfList.kAttribSet);

        oCondition = ArmAstBase.fromJson(dJson['condition'], ArmAstBase.ksModeAccessorCond);

        #
        # 1. 'access' is an AST: This is one if/else + action without nesting.
        #
        if not isinstance(dJson['access'], list):
            oStmt = ArmAstBase.fromJson(dJson['access'], ArmAstBase.ksModeAccessor);
            if oCondition.isBoolAndTrue():
                assert isinstance(oStmt, (ArmAstStatementBase, ArmAstFunction)); ## @todo may need a wrapper.
                #print('debug/IfList/%u:%s 1a. oStmt=%s' % (uDepth, ' '*uDepth, oStmt,));
                return oStmt;
            oRet = ArmAstIfList([oCondition], [oStmt], None);
            #print('debug/IfList/%u:%s 1b. oRet=\n%s' % (uDepth, ' '*uDepth, oRet,));
            return oRet;

        #
        # 2. 'access' is a list of the same type: Typically nested if-list.
        #
        #    This is more complicated because of the nesting and different ways
        #    of expressing the same stuff.  We make it even more complicated by
        #    not mirroring the json representation 1:1.
        #
        aoChildren = [ArmAstIfList.fromJson(dJsonChild, uDepth + 1) for dJsonChild in dJson['access']];
        assert aoChildren;

        # Iff there is only one child, we need to check for some special cases.
        if len(aoChildren) == 1:
            oChild = aoChildren[0];
            if not isinstance(oChild, ArmAstIfList):
                if oCondition.isBoolAndTrue():
                    #print('debug/IfList/%u:%s 2a. oChild=%s' % (uDepth, ' '*uDepth, oChild,));
                    return oChild;
                #print('debug/IfList/%u:%s 2b. oCondition=%s oChild=%s' % (uDepth, ' '*uDepth, oCondition, oChild,));
                return ArmAstIfList([oCondition], aoChildren, None);

            # If our condition is a dummy one, return the child.
            if oCondition.isBoolAndTrue():
                #print('debug/IfList/%u:%s 2c. oChild=%s' % (uDepth, ' '*uDepth, oChild,));
                return oChild;
            assert oChild.aoIfConditions;

        # Generic.
        #print('debug/IfList/%u:%s 2d. #aoChildren=%s' % (uDepth, ' '*uDepth, len(aoChildren),));
        aoIfConds = [];
        aoIfStmts = [];
        oElseStmt = None;
        for i, oChild in enumerate(aoChildren):
            if isinstance(oChild, ArmAstIfList):
                assert not oChild.oElseStatement or i + 1 == len(aoChildren), \
                       'i=%s/%u\noIfConditions=%s\noElseStmts=%s\naoChildren=[\n%s]' \
                        % (i, len(aoChildren), oChild.aoIfConditions, oChild.oElseStatement,
                           '\n'.join(['  #%u: %s' % (j, o.toString()) for j, o in enumerate(aoChildren)])
                           );
                aoIfConds.extend(oChild.aoIfConditions);
                aoIfStmts.extend(oChild.aoIfStatements);
                if oChild.oElseStatement:
                    assert i == len(aoChildren) - 1 and not oElseStmt;
                    oElseStmt = oChild.oElseStatement;
                    #print('debug/IfList/%u:%s 2e. i=%u oChild=%s' % (uDepth, ' '*uDepth, i, oChild,));
                #else: print('debug/IfList/%u:%s 2f. i=%u oChild=%s' % (uDepth, ' '*uDepth, i, oChild,));
            else:
                #print('debug/IfList/%u:%s 2g. i=%u oChild=%s' % (uDepth, ' '*uDepth, i, oChild,));
                assert i == len(aoChildren) - 1 and not oElseStmt;
                oElseStmt = oChild;
        #print('debug/IfList/%u:%s 2x. oElseStmt=%s' % (uDepth, ' '*uDepth, oElseStmt));

        # Twist: Eliminate this level if the current condition is just 'true'.
        if oCondition.isBoolAndTrue():
            oRet = ArmAstIfList(aoIfConds, aoIfStmts, oElseStmt);
            #print('debug/IfList/%u:%s 2y. oRet=\n%s\n' % (uDepth, ' '*uDepth, oRet));
        else:
            oRet = ArmAstIfList([oCondition], [ArmAstIfList(aoIfConds, aoIfStmts, oElseStmt)], None);
            #print('debug/IfList/%u:%s 2z. oRet=\n%s\n' % (uDepth, ' '*uDepth, oRet));
        return oRet;



class ArmAccessorBase(object):
    """
    Register accessor base class.
    """
    def __init__(self, dJson, oCondition):
        self.dJson = dJson;
        self.oCondition = oCondition;

    kAttribSetBlockAccess = frozenset(['_type', 'access', 'condition']);
    @staticmethod
    def fromJsonBlockAccess(dJson):
        assertJsonAttribsInSet(dJson, ArmAccessorBase.kAttribSetBlockAccess);
        return ArmAccessorBlockAccess(dJson, ArmAstBase.fromJson(dJson['condition'], ArmAstBase.ksModeConstraints));


    kAttribSetBlockAccessArray = frozenset(['_type', 'access', 'condition', 'index_variables', 'indexes',
                                            'offset', 'references']);
    @staticmethod
    def fromJsonBlockAccessArray(dJson):
        assertJsonAttribsInSet(dJson, ArmAccessorBase.kAttribSetBlockAccessArray);
        return ArmAccessorBlockAccessArray(dJson, ArmAstBase.fromJson(dJson['condition'], ArmAstBase.ksModeConstraints));


    kAttribSetExternalDebug = frozenset(['_type', 'access', 'component', 'condition', 'instance', 'offset',
                                         'power_domain', 'range']);
    @staticmethod
    def fromJsonExternalDebug(dJson):
        assertJsonAttribsInSet(dJson, ArmAccessorBase.kAttribSetExternalDebug);
        return ArmAccessorExternalDebug(dJson, ArmAstBase.fromJson(dJson['condition'], ArmAstBase.ksModeConstraints));


    kAttribSetMemoryMapped = frozenset(['_type', 'access', 'component', 'condition', 'frame', 'instance', 'offset',
                                        'power_domain', 'range']);
    @staticmethod
    def fromJsonMemoryMapped(dJson):
        assertJsonAttribsInSet(dJson, ArmAccessorBase.kAttribSetMemoryMapped);
        return ArmAccessorExternalDebug(dJson, ArmAstBase.fromJson(dJson['condition'], ArmAstBase.ksModeConstraints));


    kAttribSetSystem = frozenset(['_type', 'access', 'condition', 'encoding', 'name']);
    @staticmethod
    def fromJsonSystem(dJson):
        assertJsonAttribsInSet(dJson, ArmAccessorBase.kAttribSetSystem);
        assert len(dJson['encoding']) == 1;
        sName     = dJson['name']; # For exception listing
        oEncoding = ArmRegEncoding.fromJson(dJson['encoding'][0]);
        oAccess   = ArmAstIfList.fromJson(dJson['access']) if dJson['access'] else None;
        return ArmAccessorSystem(dJson, ArmAstBase.fromJson(dJson['condition'], ArmAstBase.ksModeConstraints),
                                 sName, oEncoding, oAccess);


    kAttribSetSystemArray = frozenset(['_type', 'access', 'condition', 'encoding', 'index_variable', 'indexes', 'name']);
    @staticmethod
    def fromJsonSystemArray(dJson):
        assertJsonAttribsInSet(dJson, ArmAccessorBase.kAttribSetSystemArray);
        assert len(dJson['encoding']) == 1;
        oEncoding = ArmRegEncoding.fromJson(dJson['encoding'][0]);
        oAccess   = ArmAstIfList.fromJson(dJson['access']) if dJson['access'] else None;
        return ArmAccessorSystemArray(dJson, ArmAstBase.fromJson(dJson['condition'], ArmAstBase.ksModeConstraints),
                                      dJson['name'], oEncoding, oAccess);


    kfnTypeMap = {
        'Accessors.BlockAccess':            fromJsonBlockAccess,
        'Accessors.BlockAccessArray':       fromJsonBlockAccessArray,
        'Accessors.ExternalDebug':          fromJsonExternalDebug,
        'Accessors.MemoryMapped':           fromJsonMemoryMapped,
        'Accessors.SystemAccessor':         fromJsonSystem,
        'Accessors.SystemAccessorArray':    fromJsonSystemArray,
    };

    @staticmethod
    def fromJson(dJson):
        """ Decodes a register accessor object. """
        return ArmAccessorBase.kfnTypeMap[dJson['_type']](dJson);


class ArmAccessorBlockAccess(ArmAccessorBase):
    """ Accessors.BlockAccess """
    def __init__(self, dJson, oCondition):
        ArmAccessorBase.__init__(self, dJson, oCondition);


class ArmAccessorBlockAccessArray(ArmAccessorBase):
    """ Accessors.BlockAccessArray """
    def __init__(self, dJson, oCondition):
        ArmAccessorBase.__init__(self, dJson, oCondition);


class ArmAccessorExternalDebug(ArmAccessorBase):
    """ Accessors.ExternalDebug """
    def __init__(self, dJson, oCondition):
        ArmAccessorBase.__init__(self, dJson, oCondition);


class ArmAccessorMemoryMapped(ArmAccessorBase):
    """ Accessors.MemoryMapped """
    def __init__(self, dJson, oCondition):
        ArmAccessorBase.__init__(self, dJson, oCondition);


class ArmAccessorSystem(ArmAccessorBase):
    """ Accessors.SystemAccessor """
    def __init__(self, dJson, oCondition, sName, oEncoding, oAccess):
        ArmAccessorBase.__init__(self, dJson, oCondition);
        self.sName     = sName;
        self.oEncoding = oEncoding  # Type: ArmRegEncoding
        self.oAccess   = oAccess    # Type: ArmAstIfList # Can be None!


class ArmAccessorSystemArray(ArmAccessorSystem):
    """ Accessors.SystemAccessorArray """
    def __init__(self, dJson, oCondition, sName, oEncoding, oAccess):
        ArmAccessorSystem.__init__(self, dJson, oCondition, sName, oEncoding, oAccess);



class ArmRegister(object):
    """
    ARM system register.
    """

    def __init__(self, oJson, sName, sState, aoFieldsets, fRegArray, oCondition, aoAccessors):
        self.oJson          = oJson;
        self.sName          = sName;
        self.sState         = sState;
        self.aoFieldsets    = aoFieldsets       # Type: List[ArmFieldset]
        self.fRegArray      = fRegArray;
        self.oCondition     = oCondition        # Type: ArmAstBase
        self.aoAccessors    = aoAccessors       # Type: List[ArmAccessorBase]

        self.daoFields      = {}                # Type: Dict[str,List[ArmFieldsBase]]
        for oFieldset in aoFieldsets:
            ArmFieldsBase.addFieldListToLookupDict(self.daoFields, oFieldset.aoFields);

    def getVBoxConstant(self):
        """
        Returns the VBox constant for the register.

        For instance: ARMV8_AARCH64_SYSREG_ID_AA64ZFR0_EL1
        """
        return 'ARMV8_' + self.sState.upper() + '_SYSREG_' + self.sName.upper();

    @staticmethod
    def fromJson(dJson, sStatePrefix = ''):
        sType = dJson['_type'];
        assert sType in ('Register', 'RegisterArray'), '_type=%s' % (sType,);

        sName       = dJson['name'];
        sState      = dJson['state'];
        aoFieldsets = [ArmFieldset.fromJson(dJsonSet) for dJsonSet in dJson['fieldsets']];
        oCondition  = ArmAstBase.fromJson(dJson['condition'], ArmAstBase.ksModeConstraints); ## @todo hackish mode
        aoAccessors = [ArmAccessorBase.fromJson(dJsonAcc) for dJsonAcc in dJson['accessors']];

        return ArmRegister(dJson, sName, sStatePrefix + sState, aoFieldsets, sType == 'RegisterArray', oCondition, aoAccessors);



#
# AArch64 Specification Loader.
#

## @name System Registers
## @{

## The '_meta::version' dictionary from the first entry in the Registers.json file.
## @note Each register has a _meta::version attribute, but we ASSUME these are the
##       same within the file.
g_oArmRegistersVerInfo = None;

## All the system registers.
g_aoAllArmRegisters = []                                        # type: List[ArmRegister]

## All the system registers by 'state'.
g_daoAllArmRegistersByState = {}                                # type: Dict[str, List[ArmRegister]]

## All the system registers by 'state' and then name.
g_ddoAllArmRegistersByStateByName = {}                          # type: Dict[str, Dict[str, ArmRegister]]

## @}

## @name Features
## @{

## The '_meta::version' dictionary from the Features.json file.
g_oArmFeaturesVerInfo = None;

## All the features.
g_aoAllArmFeatures = []                                         # type: List[ArmFeature]

## All the features by name.
g_dAllArmFeaturesByName = {}                                    # type: Dict[str, ArmFeature]
## @}


## @name Instructions
## @{

## The '_meta::version' dictionary from the Instructions.json file.
g_oArmInstructionVerInfo = None;

## All the instructions.
g_aoAllArmInstructions = []                                     # type: List[ArmInstruction]

## All the instructions by name (not mnemonic).
g_dAllArmInstructionsByName = {}                                # type: Dict[str, ArmInstruction]

## All the instructions by instruction set name.
g_dAllArmInstructionsBySet = collections.defaultdict(list)      # type: Dict[str, List[ArmInstruction]]

## All the instructions by (immediate) instruction group name.
g_dAllArmInstructionsByGroup = collections.defaultdict(list)    # type: Dict[str, List[ArmInstruction]]

## The instruction sets.
g_aoArmInstructionSets = []                                     # type: List[ArmInstructionSet]

## The instruction sets by name.
g_dArmInstructionSets = {}                                      # type: Dict[str, ArmInstructionSet]

## The instruction groups.
g_aoArmInstructionGroups = []                                   # type: List[ArmInstructionGroup]

## The instruction groups.
g_dArmInstructionGroups = {}                                    # type: Dict[str, ArmInstructionGroup]

## @}

## Instruction corrections expressed as a list required conditionals.
#
# In addition to supplying missing conditionals (IsFeatureImplemented(FEAT_XXX)
# and missing fixed encoding values (field == <fixed-value>).
#
# The reason why this is a list and not an expression, is that it easier to skip
# stuff that's already present in newer specification and avoiding duplicate tests.
g_dArmEncodingCorrectionConditions = {
    # The sdot and udot vector instructions are missing the 'size' restrictions in the 2024-12 specs.
    'sdot_z_zzz_': (ArmAstBinaryOp(ArmAstIdentifier('size'), '!=', ArmAstValue("'00'")),
                    ArmAstBinaryOp(ArmAstIdentifier('size'), '!=', ArmAstValue("'01'")),),
    'udot_z_zzz_': (ArmAstBinaryOp(ArmAstIdentifier('size'), '!=', ArmAstValue("'00'")),
                    ArmAstBinaryOp(ArmAstIdentifier('size'), '!=', ArmAstValue("'01'")),),
    # These instructions are FEAT_MTE && FEAT_MOPS. The 2024-12 specs missed the former condition.
    'SETGEN_SET_memcms':  (ArmAstFunction('IsFeatureImplemented', [ArmAstIdentifier('FEAT_MTE'),]),),
    'SETGETN_SET_memcms': (ArmAstFunction('IsFeatureImplemented', [ArmAstIdentifier('FEAT_MTE'),]),),
    'SETGET_SET_memcms':  (ArmAstFunction('IsFeatureImplemented', [ArmAstIdentifier('FEAT_MTE'),]),),
    'SETGE_SET_memcms':   (ArmAstFunction('IsFeatureImplemented', [ArmAstIdentifier('FEAT_MTE'),]),),
    'SETGMN_SET_memcms':  (ArmAstFunction('IsFeatureImplemented', [ArmAstIdentifier('FEAT_MTE'),]),),
    'SETGMTN_SET_memcms': (ArmAstFunction('IsFeatureImplemented', [ArmAstIdentifier('FEAT_MTE'),]),),
    'SETGMT_SET_memcms':  (ArmAstFunction('IsFeatureImplemented', [ArmAstIdentifier('FEAT_MTE'),]),),
    'SETGM_SET_memcms':   (ArmAstFunction('IsFeatureImplemented', [ArmAstIdentifier('FEAT_MTE'),]),),
    'SETGPN_SET_memcms':  (ArmAstFunction('IsFeatureImplemented', [ArmAstIdentifier('FEAT_MTE'),]),),
    'SETGPTN_SET_memcms': (ArmAstFunction('IsFeatureImplemented', [ArmAstIdentifier('FEAT_MTE'),]),),
    'SETGPT_SET_memcms':  (ArmAstFunction('IsFeatureImplemented', [ArmAstIdentifier('FEAT_MTE'),]),),
    'SETGP_SET_memcms':   (ArmAstFunction('IsFeatureImplemented', [ArmAstIdentifier('FEAT_MTE'),]),),

    ## @todo fexpa_z_z: s/FEAT_SME2p2/FEAT_SSVE_FEXPA/ (2024-12 vs 2025-03); Not relevant since we don't support either.
};



def __asmChoicesFilterOutDefaultAndAbsent(adChoices, ddAsmRules):
    """
    Helper that __asmRuleIdToDisplayText uses to filter out any default choice
    that shouldn't be displayed.

    Returns choice list.
    """
    # There are sometime a 'none' tail entry.
    if adChoices[-1] is None:
        adChoices = adChoices[:-1];
    if len(adChoices) > 1:
        # Typically, one of the choices is 'absent' or 'default', eliminate it before we start...
        for iChoice, dChoice in enumerate(adChoices):
            fAllAbsentOrDefault = True;
            for dSymbol in dChoice['symbols']:
                if dSymbol['_type'] != 'Instruction.Symbols.RuleReference':
                    fAllAbsentOrDefault = False;
                    break;
                sRuleId = dSymbol['rule_id'];
                oRule = ddAsmRules[sRuleId];
                if (   ('display' in oRule and oRule['display'])
                    or ('symbols' in oRule and oRule['symbols'])):
                    fAllAbsentOrDefault = False;
                    break;
            if fAllAbsentOrDefault:
                return adChoices[:iChoice] + adChoices[iChoice + 1:];
    return adChoices;


def __asmRuleIdToDisplayText(sRuleId, ddAsmRules, sInstrNm):
    """
    Helper that asmSymbolsToDisplayText uses to process assembly rule references.
    """
    dRule = ddAsmRules[sRuleId];
    sRuleType = dRule['_type'];
    if sRuleType == 'Instruction.Rules.Token':
        assert dRule['default'], '%s: %s' % (sInstrNm, sRuleId);
        return dRule['default'];
    if sRuleType == 'Instruction.Rules.Rule':
        assert dRule['display'], '%s: %s' % (sInstrNm, sRuleId);
        return dRule['display'];
    if sRuleType == 'Instruction.Rules.Choice':
        # Some of these has display = None and we need to sort it out ourselves.
        if dRule['display']:
            return dRule['display'];
        sText = '{';
        assert len(dRule['choices']) > 1;
        for iChoice, dChoice in enumerate(__asmChoicesFilterOutDefaultAndAbsent(dRule['choices'], ddAsmRules)):
            if iChoice > 0:
                sText += ' | ';
            sText += asmSymbolsToDisplayText(dChoice['symbols'], ddAsmRules, sInstrNm);
        sText += '}';

        # Cache it.
        dRule['display'] = sText;
        return sText;

    raise Exception('%s: Unknown assembly rule type: %s for %s' % (sInstrNm, sRuleType, sRuleId));


def asmSymbolsToDisplayText(adSymbols, ddAsmRules, sInstrNm):
    """
    Translates the 'symbols' array of an instruction's 'assembly' property into
     a kind of assembly syntax outline.
    """
    sText = '';
    for dSym in adSymbols:
        sType = dSym['_type'];
        if sType == 'Instruction.Symbols.Literal':
            sText += dSym['value'];
        elif sType == 'Instruction.Symbols.RuleReference':
            sRuleId = dSym['rule_id'];
            sText += __asmRuleIdToDisplayText(sRuleId, ddAsmRules, sInstrNm);
        else:
            raise Exception('%s: Unknown assembly symbol type: %s' % (sInstrNm, sType,));
    return sText;


def parseInstructions(oInstrSet, oParent, aoJson, ddAsmRules):
    for oJson in aoJson:
        sType = oJson['_type'];
        if sType == 'Instruction.InstructionSet':
            if oParent: raise Exception("InstructionSet shouldn't have a parent!");
            assert not oInstrSet;
            oInstrSet = ArmInstructionSet.fromJson(oJson);
            assert oInstrSet.sName not in g_dArmInstructionSets;
            g_dArmInstructionSets[oInstrSet.sName] = oInstrSet;
            g_aoArmInstructionSets.append(oInstrSet);

            parseInstructions(oInstrSet, oInstrSet, oJson['children'], ddAsmRules);

        elif sType == 'Instruction.InstructionGroup':
            if not oParent: raise Exception("InstructionGroup should have a parent!");
            oInstrGroup = ArmInstructionGroup.fromJson(oJson, oParent);
            #if oInstrGroup.sName in g_dArmInstructionGroups: # happens with

            if oInstrGroup.sName in g_dArmInstructionGroups:
                if oInstrGroup.sName == oParent.sName: # sve_intx_clamp, sve_intx_dot2
                    oInstrGroup.sName += '_lvl2'
                else:
                    assert oInstrGroup.sName not in g_dArmInstructionGroups, '%s' % (oInstrGroup.sName,);

            g_dArmInstructionGroups[oInstrGroup.sName] = oInstrGroup;
            g_aoArmInstructionGroups.append(oInstrGroup);
            oParent.aoGroups.append(oInstrGroup);

            parseInstructions(oInstrSet, oInstrGroup, oJson['children'], ddAsmRules);

        elif sType == "Instruction.Instruction":
            if not oParent: raise Exception("Instruction should have a parent!");

            #
            # Start by getting the instruction attributes.
            #
            sInstrNm = oJson['name'];

            oCondition = ArmAstBase.fromJson(oJson['condition']);
            aoCorrectionConditions = g_dArmEncodingCorrectionConditions.get(sInstrNm)
            if aoCorrectionConditions:
                oCondition = addAndConditionsFromList(oCondition, aoCorrectionConditions);

            (aoFields, fFields) = ArmEncodesetField.encodesetFromJson(oJson['encoding']);
            for oUp in oParent.getUpIterator():
                if oUp.fFields & ~fFields:
                    (aoFields, fFields) = ArmEncodesetField.encodesetAddParentFields(aoFields, fFields, oUp.aoFields);
                if not oUp.oCondition.isBoolAndTrue():
                    oCondition = ArmAstBinaryOp(oCondition, '&&', oUp.oCondition.clone());
            if fFields != (1 << oInstrSet.cBitsWidth) - 1:
                raise Exception('Instruction %s has an incomplete encodingset: fFields=%#010x (missing %#010x)'
                                % (sInstrNm, fFields, fFields ^ ((1 << oInstrSet.cBitsWidth) - 1),))

            #sCondBefore = oCondition.toString();
            #print('debug transfer: %s: org:  %s' % (sInstrNm, sCondBefore));
            (oCondition, fMod) = transferConditionsToEncoding(oCondition, aoFields, collections.defaultdict(list), sInstrNm);
            #if fMod:
            #    print('debug transfer: %s: %s' % (sInstrNm, sCondBefore,));
            #    print('              %*s %s' % (len(sInstrNm) + 3, '--->', oCondition.toString(),));
            _ = fMod;

            # Come up with the assembly syntax (sAsmDisplay).
            if 'assembly' in oJson:
                oAsm = oJson['assembly'];
                assert oAsm['_type'] == 'Instruction.Assembly';
                assert 'symbols' in oAsm;
                sAsmDisplay = asmSymbolsToDisplayText(oAsm['symbols'], ddAsmRules, sInstrNm);
            else:
                sAsmDisplay = sInstrNm;

            # We derive the mnemonic from the assembly display string.
            sMnemonic = sAsmDisplay.split()[0];

            #
            # Instantiate it.
            #
            oInstr = ArmInstruction(oJson, sInstrNm, sMnemonic, sAsmDisplay, aoFields, fFields, oCondition, oParent);

            #
            # Add the instruction to the various lists and dictionaries.
            #
            g_aoAllArmInstructions.append(oInstr);
            assert oInstr.sName not in g_dAllArmInstructionsByName;
            g_dAllArmInstructionsByName[oInstr.sName] = oInstr;

            oParent.addImmediateInstruction(oInstr);

        else:
            raise Exception('Unexpected instruction object type: %s' % (sType,));

    return True;


def addAndConditionsFromList(oTree, aoAndConditions):
    """
    Adds the conditions in aoAndConditions that are not already present in
    oTree in an required (AND) form.

    This is used when we add corrections, so that we avoid duplicate feature
    checks and such.
    """
    if oTree.isBoolAndTrue():
        return andConditionListToTree(aoAndConditions);

    def isAndConditionPresent(oTree, oAndCondition):
        if oAndCondition.isSame(oTree):
            return True;
        if isinstance(oTree, ArmAstBinaryOp) and oTree.sOp == '&&':
            return isAndConditionPresent(oTree.oLeft, oAndCondition) or isAndConditionPresent(oTree.oRight, oAndCondition);
        return False;

    aoToAdd = [oTree,];
    for oAndCondition in aoAndConditions:
        if not isAndConditionPresent(oTree, oAndCondition):
            aoToAdd.append(oAndCondition);

    return andConditionListToTree(aoToAdd);


def andConditionListToTree(aoAndConditions):
    """ Creates AST tree of AND binary checks from aoAndConditions. """
    if len(aoAndConditions) <= 1:
        return aoAndConditions[0].clone();
    return ArmAstBinaryOp(aoAndConditions[0].clone(), '&&', andConditionListToTree(aoAndConditions[1:]));


def transferConditionsToEncoding(oCondition, aoFields, dPendingNotEq, sInstrNm, uDepth = 0, fMod = False):
    """
    This is for dealing with stuff like asr_z_p_zi_ and lsr_z_p_zi_ which has
    the same fixed encoding fields in the specs, but differs in the value of
    the named field 'U' as expressed in the conditions.

    This function will recursively take 'Field == value/integer' expression out
    of the condition tree and add them to the encodeset conditions when possible.

    The dPendingNotEq stuff is a hack to deal with stuff like this:
        sdot_z_zzz_:     U == '0' && size != '01' && size != '00'
                     && (IsFeatureImplemented(FEAT_SVE) || IsFeatureImplemented(FEAT_SME))
    The checks can be morphed into the 'size' field encoding criteria as '0b0x'.
    """
    if isinstance(oCondition, ArmAstBinaryOp):
        if oCondition.sOp == '&&':
            # Recurse into each side of an AND expression.
            #print('debug transfer: %s: recursion...' % (sInstrNm,));
            (oCondition.oLeft, fMod)  = transferConditionsToEncoding(oCondition.oLeft,  aoFields, dPendingNotEq,
                                                                     sInstrNm, uDepth + 1, fMod);
            (oCondition.oRight, fMod) = transferConditionsToEncoding(oCondition.oRight, aoFields, dPendingNotEq,
                                                                     sInstrNm, uDepth + 1, fMod);
            if oCondition.oLeft.isBoolAndTrue():
                return (oCondition.oRight, fMod);
            if oCondition.oRight.isBoolAndTrue():
                return (oCondition.oLeft, fMod);

        elif oCondition.sOp in ('==', '!='):
            # The pattern we're looking for is identifier (field) == fixed value.
            #print('debug transfer: %s: binaryop %s vs %s ...' % (sInstrNm, oCondition.oLeft.sType, oCondition.oRight.sType));
            if (    isinstance(oCondition.oLeft, ArmAstIdentifier)
                and isinstance(oCondition.oRight, (ArmAstValue, ArmAstInteger))):
                sFieldName = oCondition.oLeft.sName;
                oValue     = oCondition.oRight;
                #print('debug transfer: %s: binaryop step 2...' % (sInstrNm,));
                for oField in aoFields: # ArmEncodesetField
                    if oField.sName and oField.sName == sFieldName:
                        # ArmAstInteger - not used by spec, only corrections:
                        if isinstance(oValue, ArmAstInteger):
                            if oField.fFixed != 0:
                                raise Exception('%s: Condition checks fixed field value: %s (%#x/%#x) %s %s'
                                                % (sInstrNm, oField.sName, oField.fValue, oField.fFixed,
                                                   oCondition.sOp, oValue.iValue,));
                            assert oField.fValue == 0;
                            if oValue.iValue.bit_length() > oField.cBitsWidth:
                                raise Exception('%s: Condition field value check too wide: %s is %u bits, test value %s (%u bits)'
                                                % (sInstrNm, oField.sName, oField.cBitsWidth, oValue.iValue,
                                                   oValue.iValue.bit_count(),));
                            if oValue.iValue < 0:
                                raise Exception('%s: Condition field checks against negative value: %s, test value is %s'
                                                % (sInstrNm, oField.sName, oValue.iValue));
                            fFixed = (1 << oField.cBitsWidth) - 1;
                            if oCondition.sOp == '!=' and oField.cBitsWidth > 1:
                                dPendingNotEq[oField.sName] += [(oField, oValue.iValue, fFixed, oCondition)];
                                break;

                            print('debug transfer: %s: integer binaryop -> encoding: %s %s %#x/%#x'
                                  % (sInstrNm, oField.sName, oCondition.sOp, oValue.iValue, fFixed));
                            if oCondition.sOp == '==':
                                oField.fValue = oValue.iValue;
                            else:
                                oField.fValue = ~oValue.iValue & fFixed;
                            oField.fFixed = fFixed;
                            return (ArmAstBool(True), True);

                        # ArmAstValue.
                        assert isinstance(oValue, ArmAstValue);
                        (fValue, fFixed, _, _) = ArmEncodesetField.parseValue(oValue.sValue, oField.cBitsWidth);

                        if oCondition.sOp == '!=' and oField.cBitsWidth > 1 and (fFixed & (fFixed - 1)) != 0:
                            dPendingNotEq[oField.sName] += [(oField, fValue, fFixed, oCondition)];
                            break;
                        if fFixed & oField.fFixed:
                            raise Exception('%s: Condition checks fixed field value: %s (%#x/%#x) %s %s (%#x/%#x)'
                                            % (sInstrNm, oField.sName, oField.fValue, oField.fFixed, oCondition.sOp,
                                               oValue.sValue, fValue, fFixed));
                        #print('debug transfer: %s: value binaryop -> encoding: %s %s %#x (fFixed=%#x)'
                        #      % (sInstrNm, oField.sName, oCondition.sOp, fValue, fFixed,));
                        if oCondition.sOp == '==':
                            oField.fValue |= fValue;
                        else:
                            oField.fValue |= ~fValue & fFixed;
                        oField.fFixed |= fFixed;
                        return (ArmAstBool(True), True);

    #
    # Deal with pending '!=' optimizations for fields larger than a single bit.
    # Currently we only deal with two bit fields.
    #
    if uDepth == 0 and dPendingNotEq:
        def recursiveRemove(oCondition, aoToRemove):
            if isinstance(oCondition, ArmAstBinaryOp):
                if oCondition.sOp == '&&':
                    oCondition.oLeft  = recursiveRemove(oCondition.oLeft, aoToRemove);
                    oCondition.oRight = recursiveRemove(oCondition.oRight, aoToRemove);
                    if oCondition.oLeft.isBoolAndTrue():    return oCondition.oRight;
                    if oCondition.oRight.isBoolAndTrue():   return oCondition.oLeft;
                elif oCondition in aoToRemove:
                    assert isinstance(oCondition.oLeft, ArmAstIdentifier);
                    assert isinstance(oCondition.oRight, (ArmAstValue, ArmAstInteger));
                    assert oCondition.sOp == '!=';
                    return ArmAstBool(True);
            return oCondition;

        for sFieldNm, atOccurences in dPendingNotEq.items():
            # For a two bit field, we need at least two occurences to get any kind of fixed value.
            oField = atOccurences[0][0];
            if oField.cBitsWidth == 2 and len(atOccurences) >= 2:
                dValues = {};
                dFixed  = {};
                for oCurField, fValue, fFixed, _ in atOccurences:
                    assert oCurField is oField;
                    dValues[fValue] = 1;
                    dFixed[fFixed]  = 1;
                if len(dValues) in (2, 3) and len(dFixed) == 1 and 3 in dFixed:
                    afValues = list(dValues);
                    if len(dValues) == 2:
                        fFixed = 2 if (afValues[0] ^ afValues[1]) & 1 else 1; # One of the bits are fixed, the other ignored.
                    else:
                        fFixed = 3;                                           # Both bits are fixed.
                    fValue = afValues[0] & fFixed;
                    print('debug transfer: %s: %u binaryops -> encoding: %s == %#x/%#x'
                          % (sInstrNm, len(atOccurences), sFieldNm, ~fValue & fFixed, fFixed,));
                    oField.fValue |= ~fValue & fFixed;
                    oField.fFixed |= fFixed;

                    # Remove the associated conditions (they'll be leaves).
                    oCondition = recursiveRemove(oCondition, [oCondition for _, _, _, oCondition in atOccurences]);
                    fMod = True;
                else:
                    print('info: %s: transfer cond to enc failed for: %s dValues=%s dFixed=%s'
                          % (sInstrNm, sFieldNm, dValues, dFixed));
            elif oField.cBitsWidth == 3 and len(atOccurences) >= 7:
                print('info: %s: TODO: transfer cond to enc for 3 bit field: %s (%s)' % (sInstrNm, sFieldNm, atOccurences,));

    return (oCondition, fMod);


def parseFeatures(aoJson, fVerbose = False):
    """
    Parses the list of features (parameters).
    """
    global g_aoAllArmFeatures;
    g_aoAllArmFeatures = [ArmFeature.fromJson(oJson) for oJson in aoJson];

    #global g_dAllArmFeaturesByName;
    for oFeature in g_aoAllArmFeatures:
        if oFeature.sName in g_dAllArmFeaturesByName:
            raise Exception('Feature %s is listed twice!' % (oFeature.sName,))
        g_dAllArmFeaturesByName[oFeature.sName] = oFeature;

    for sFeature in ('FEAT_ETMv4p1', 'FEAT_ETMv4p2', 'FEAT_ETMv4p3', 'FEAT_ETMv4p4', 'FEAT_ETMv4p5', 'FEAT_ETMv4p6',
                     'FEAT_GICv3', 'FEAT_GICv3p1', 'FEAT_GICv4', 'FEAT_GICv4p1', 'FEAT_GICv3_NMI', 'FEAT_GICv3_TDIR',
                     'FEAT_VPIPT',
                     'FEAT_AA64', 'FEAT_AA32', 'FEAT_SSVE_FEXPA', # Missing in 2024-12:
                     ):
        if sFeature not in g_dAllArmFeaturesByName:
            oFeature = ArmFeature(None, sFeature, []);
            g_aoAllArmFeatures.append(oFeature);
            g_dAllArmFeaturesByName[sFeature] = oFeature;

    g_aoAllArmFeatures = sorted(g_aoAllArmFeatures, key = operator.attrgetter('sName'));

    if fVerbose:
        # print the features.
        cchMaxName = max(len(oFeature.sName) for oFeature in g_aoAllArmFeatures)
        dTypeNm = {
            'Parameters.Boolean': 'boolean',
            'Parameters.Integer': 'integer',
        };
        for iFeature, oFeature in enumerate(g_aoAllArmFeatures):
            if oFeature.oSupportExpr:
                print('%3u: %s  %-*s := %s'
                      % (iFeature, dTypeNm[oFeature.sType], cchMaxName, oFeature.sName, oFeature.oSupportExpr.toString()));
            else:
                print('%3u: %s  %s' % (iFeature, dTypeNm[oFeature.sType], oFeature.sName,));
            if not oFeature.oSupportExpr or oFeature.oSupportExpr.toString().find('AArch64') < 0:
                for iConstraint, oConstraint in enumerate(oFeature.aoConstraints):
                    print('        #%u: %s' % (iConstraint, oConstraint.toString(),));

    return True;


def parseRegisters(aoJson):
    """
    Parses the list of registers.
    """
    global g_aoAllArmRegisters;
    g_aoAllArmRegisters = [];
    for dJson in aoJson:
        if dJson['_type'] != 'RegisterBlock':
            g_aoAllArmRegisters.append(ArmRegister.fromJson(dJson));
        else:
            ## @todo proper handling of RegisterBlocks.
            sStatePrefix = dJson['name'] + '.';
            for dSubJson in dJson['blocks']:
                g_aoAllArmRegisters.append(ArmRegister.fromJson(dSubJson, sStatePrefix));

    g_aoAllArmRegisters = sorted(g_aoAllArmRegisters, key = operator.attrgetter('sState', 'sName'));

    #global g_daoAllArmRegistersByState;
    #global g_ddoAllArmRegistersByStateByName;
    for oRegister in g_aoAllArmRegisters:
        if oRegister.sState in g_daoAllArmRegistersByState:
            g_daoAllArmRegistersByState[oRegister.sState] += [oRegister,];
            if oRegister.sName in g_ddoAllArmRegistersByStateByName[oRegister.sState]:
                raise Exception('Register %s.%s is listed twice!' % (oRegister.sState, oRegister.sName,))
            g_ddoAllArmRegistersByStateByName[oRegister.sState][oRegister.sName] = oRegister;
        else:
            g_daoAllArmRegistersByState[oRegister.sState]  = [oRegister,];
            g_ddoAllArmRegistersByStateByName[oRegister.sState] = { oRegister.sName: oRegister, };

    ## print the features.
    #cchMaxName = max(len(oFeature.sName) for oFeature in g_aoAllArmFeatures)
    #dTypeNm = {
    #    'Parameters.Boolean': 'boolean',
    #    'Parameters.Integer': 'integer',
    #};
    #for iFeature, oFeature in enumerate(g_aoAllArmFeatures):
    #    if oFeature.oSupportExpr:
    #        print('%3u: %s  %-*s := %s'
    #              % (iFeature, dTypeNm[oFeature.sType], cchMaxName, oFeature.sName, oFeature.oSupportExpr.toString()));
    #    else:
    #        print('%3u: %s  %s' % (iFeature, dTypeNm[oFeature.sType], oFeature.sName,));
    #    if not oFeature.oSupportExpr or oFeature.oSupportExpr.toString().find('AArch64') < 0:
    #        for iConstraint, oConstraint in enumerate(oFeature.aoConstraints):
    #            print('        #%u: %s' % (iConstraint, oConstraint.toString(),));
    #
    return True;

def LoadArmOpenSourceSpecification(oOptions):
    """
    Loads the ARM specifications from a tar file, directory or individual files.

    Note! Currently only processes Instructions.json.

    @todo Need some reworking as it's taking oOptions as input. It should be
          generic and usable by code other than the decoder generator.
    """

    #
    # Load the files.
    #
    print("*** Loading specs ...");
    nsStart = time.time_ns();
    if oOptions.sTarFile:
        with tarfile.open(oOptions.sTarFile, 'r') as oTarFile:
            with oTarFile.extractfile(oOptions.sFileInstructions) as oFile:
                dRawInstructions = json.load(oFile);
            with oTarFile.extractfile(oOptions.sFileFeatures) as oFile:
                dRawFeatures     = json.load(oFile);
            with oTarFile.extractfile(oOptions.sFileRegisters) as oFile:
                dRawRegisters    = json.load(oFile);
    else:
        if oOptions.sSpecDir:
            if not os.path.isabs(oOptions.sFileInstructions):
                oOptions.sFileInstructions = os.path.normpath(os.path.join(oOptions.sSpecDir, oOptions.sFileInstructions));
            if not os.path.isabs(oOptions.sFileFeatures):
                oOptions.sFileFeatures     = os.path.normpath(os.path.join(oOptions.sSpecDir, oOptions.sFileFeatures));
            if not os.path.isabs(oOptions.sFileRegisters):
                oOptions.sFileRegisters    = os.path.normpath(os.path.join(oOptions.sSpecDir, oOptions.sFileRegisters));

        with open(oOptions.sFileInstructions, 'r', encoding = 'utf-8') as oFile:
            dRawInstructions = json.load(oFile);
        with open(oOptions.sFileFeatures, 'r', encoding = 'utf-8') as oFile:
            dRawFeatures     = json.load(oFile);
        with open(oOptions.sFileRegisters, 'r', encoding = 'utf-8') as oFile:
            dRawRegisters    = json.load(oFile);
    print("*** Done loading specs (%s ns)." % (nsElapsedAsStr(nsStart),));

    #
    # Parse the system registers.
    #
    print("*** Parsing registers ...");
    nsStart = time.time_ns()
    global g_oArmRegistersVerInfo;
    g_oArmRegistersVerInfo = dRawRegisters[0]['_meta']['version'];
    parseRegisters(dRawRegisters);
    print("Found %u registers in %u states (%s ns)"
          % (len(g_aoAllArmRegisters), len(g_daoAllArmRegistersByState), nsElapsedAsStr(nsStart),) );

    #
    # Parse the features.
    #
    print("*** Parsing features ...");
    nsStart = time.time_ns()
    global g_oArmFeaturesVerInfo;
    g_oArmFeaturesVerInfo = dRawFeatures['_meta']['version'];
    parseFeatures(dRawFeatures['parameters']);
    print("Found %u feature definitions (%s ns)" % (len(g_aoAllArmFeatures), nsElapsedAsStr(nsStart),) );

    #
    # Parse the Instructions.
    #
    print("*** Parsing instructions ...");
    global g_oArmInstructionVerInfo;
    g_oArmInstructionVerInfo = dRawInstructions['_meta']['version'];
    parseInstructions(None, None, dRawInstructions['instructions'], dRawInstructions['assembly_rules']);

    # Sort the instruction array by name.
    global g_aoAllArmInstructions;
    g_aoAllArmInstructions = sorted(g_aoAllArmInstructions, key = operator.attrgetter('sName', 'sAsmDisplay'));

    print("Found %u instructions (%s ns)" % (len(g_aoAllArmInstructions), nsElapsedAsStr(nsStart),));
    #oBrk = g_dAllArmInstructionsByName['BRK_EX_exception'];
    #print("oBrk=%s" % (oBrk,))
    return True;


def PrintSpecs(oOptions):
    """ Prints the specification if requested in the options. """

    if oOptions.fPrintInstructions or oOptions.fPrintInstructionsWithEncoding or oOptions.fPrintInstructionsWithConditions:
        for oInstr in g_aoAllArmInstructions:
            print('%08x/%08x %s %s' % (oInstr.fFixedMask, oInstr.fFixedValue, oInstr.getCName(), oInstr.sAsmDisplay));
            if oOptions.fPrintInstructionsWithEncoding:
                for oField in sorted(oInstr.aoFields, key = operator.attrgetter('iFirstBit')): # ArmEncodesetField
                    print('  %2u L %2u: %010x/%010x%s%s'
                          % (oField.iFirstBit, oField.cBitsWidth, oField.fFixed, oField.fValue,
                             ' ' if oField.sName else '', oField.sName if oField.sName else '',));
            if oOptions.fPrintInstructionsWithConditions and not oInstr.oCondition.isBoolAndTrue():
                print('  condition: %s' % (oInstr.oCondition.toString(),));

    # Print stats on fixed bits:
    if oOptions.fPrintFixedMaskStats:
        dCounts = collections.Counter();
        for oInstr in g_aoAllArmInstructions:
            cPopCount = bin(oInstr.fFixedMask).count('1');
            dCounts[cPopCount] += 1;

        print('');
        print('Fixed bit pop count distribution:');
        for i in range(33):
            if i in dCounts:
                print('  %2u: %u' % (i, dCounts[i]));

    # Top 10 fixed masks.
    if oOptions.fPrintFixedMaskTop10:
        dCounts = collections.Counter();
        for oInstr in g_aoAllArmInstructions:
            dCounts[oInstr.fFixedMask] += 1;

        print('');
        print('Top 20 fixed masks:');
        for fFixedMask, cHits in dCounts.most_common(20):
            print('  %#x: %u times' % (fFixedMask, cHits,));

    # System registers.
    if oOptions.fPrintSysRegs:
        print('');
        print('System registers:');
        for oReg in sorted(g_aoAllArmRegisters, key = operator.attrgetter('sState', 'sName')): # type: ArmRegister
            if oReg.sState != 'AArch64': continue; # temp
            print('   %s.%s' % (oReg.sState, oReg.sName, ));
            print('       Condition: %s' % (oReg.oCondition.toString(),));
            for oFieldset in oReg.aoFieldsets: # type: ArmFieldset
                print('       Fieldsset: %s' % (oFieldset.toString(),));
            for i, oAccessor in enumerate(oReg.aoAccessors): # type: int, ArmAccessorBase
                if isinstance(oAccessor, ArmAccessorSystem):
                    print('       Accessors[%u]: encoding=%s' % (i, oAccessor.oEncoding.toString(),));
                    print('                     name=%s' % (oAccessor.sName,));
                    if not ArmAstBool.isBoolAndTrue(oAccessor.oCondition):
                        print('                     condition=%s' % (oAccessor.oCondition.toString(),));
                    if oAccessor.oAccess: # ArmAstIfList
                        asLines = oAccessor.oAccess.toStringList('                         ');
                        print('\n'.join(asLines));
                else:
                    print('       Accessors[%u]: %s' % (i, oAccessor,));

    return True;


#
# Decoder structure helpers.
#

## Populated by --decoder-hint0
g_dDecoderFilterDepth0 = { };

## Populated by --decoder-hint1
g_ddDecoderFilterDepth1 = { };


class MaskZipper(object):
    """
    This is mainly a class for putting static methods relating to mask
    packing and unpack.
    """

    def __init__(self):
        pass;

    @staticmethod
    def compileAlgo(fMask):
        """
        Returns an with instructions for extracting the bits from the mask into
        a compacted form. Each array entry is an array/tuple of source bit [0],
        destination bit [1], and bit counts [2].
        """
        aaiAlgo   = [];
        iSrcBit   = 0;
        iDstBit   = 0;
        while fMask > 0:
            # Skip leading zeros.
            cSkip    = (fMask & -fMask).bit_length() - 1;
            #assert (fMask & ((1 << cSkip) - 1)) == 0 and ((fMask >> cSkip) & 1), 'fMask=%#x cSkip=%d' % (fMask, cSkip)
            iSrcBit += cSkip;
            fMask  >>= cSkip;

            # Calculate leading ones the same way.
            cCount = (~fMask & -~fMask).bit_length() - 1;
            #assert (fMask & ((1 << cCount) - 1)) == ((1 << cCount) - 1) and (fMask & (1 << cCount)) == 0

            # Append to algo list.
            aaiAlgo.append((iSrcBit, iDstBit, (1 << cCount) - 1));

            # Advance.
            iDstBit += cCount;
            iSrcBit += cCount;
            fMask  >>= cCount;
        return aaiAlgo;

    @staticmethod
    def compileAlgoLimited(fMask):
        """
        Version of compileAlgo that returns an empty list if there are
        more than three sections.
        """
        #assert fMask;

        #
        # Chunk 0:
        #

        # Skip leading zeros.
        iSrcBit0 = (fMask & -fMask).bit_length() - 1;
        fMask  >>= iSrcBit0;
        # Calculate leading ones the same way.
        cCount0  = (~fMask & -~fMask).bit_length() - 1;
        fMask  >>= cCount0;
        if not fMask:
            return [(iSrcBit0, 0, (1 << cCount0) - 1)];

        #
        # Chunk 1:
        #

        # Skip leading zeros.
        cSrcGap1 = (fMask & -fMask).bit_length() - 1;
        fMask  >>= cSrcGap1;
        # Calculate leading ones the same way.
        cCount1  = (~fMask & -~fMask).bit_length() - 1;
        fMask  >>= cCount1;
        if not fMask:
            return [ (iSrcBit0, 0, (1 << cCount0) - 1),
                     (iSrcBit0 + cCount0 + cSrcGap1, cCount0, (1 << cCount1) - 1)];

        #
        # Chunk 2:
        #

        # Skip leading zeros.
        cSrcGap2 = (fMask & -fMask).bit_length() - 1;
        fMask  >>= cSrcGap2;
        # Calculate leading ones the same way.
        cCount2  = (~fMask & -~fMask).bit_length() - 1;
        fMask  >>= cCount2;
        if not fMask:
            iSrcBit1 = iSrcBit0 + cCount0 + cSrcGap1;
            return [ (iSrcBit0, 0, (1 << cCount0) - 1),
                     (iSrcBit1, cCount0, (1 << cCount1) - 1),
                     (iSrcBit1 + cCount1 + cSrcGap2, cCount0 + cCount1, (1 << cCount2) - 1), ];

        # Too many fragments.
        return [];

    @staticmethod
    def compileAlgoFromList(aiOrderedBits):
        """
        Returns an with instructions for extracting the bits from the mask into
        a compacted form. Each array entry is an array/tuple of source bit [0],
        destination bit [1], and mask (shifted to pos 0) [2].
        """
        aaiAlgo = [];
        iDstBit = 0;
        i       = 0;
        while i < len(aiOrderedBits):
            iSrcBit = aiOrderedBits[i];
            cCount  = 1;
            i      += 1;
            while i < len(aiOrderedBits) and aiOrderedBits[i] == iSrcBit + cCount:
                cCount += 1;
                i      += 1;
            aaiAlgo.append([iSrcBit, iDstBit, (1 << cCount) - 1])
            iDstBit += cCount;
        return aaiAlgo;

    @staticmethod
    def algoToBitList(aaiAlgo):
        aiRet = [];
        for iSrcBit, _, fMask in aaiAlgo:
            cCount = fMask.bit_count();
            aiRet += [iSrcBit + i for i in range(cCount)];
        return aiRet;

    @staticmethod
    def zipMask(uValue, aaiAlgo):
        idxRet = 0;
        for iSrcBit, iDstBit, fMask in aaiAlgo:
            idxRet |= ((uValue >> iSrcBit) & fMask) << iDstBit;
        return idxRet;

    @staticmethod
    def __zipMask1(uValue, aaiAlgo):
        iSrcBit, _, fMask = aaiAlgo[0];
        return (uValue >> iSrcBit) & fMask;

    @staticmethod
    def __zipMask2(uValue, aaiAlgo):
        iSrcBit0, _,        fMask0 = aaiAlgo[0];
        iSrcBit1, iDstBit1, fMask1 = aaiAlgo[1];
        return ((uValue >> iSrcBit0) & fMask0) | (((uValue >> iSrcBit1) & fMask1) << iDstBit1);

    @staticmethod
    def __zipMask3(uValue, aaiAlgo):
        iSrcBit0, _,        fMask0 = aaiAlgo[0];
        iSrcBit1, iDstBit1, fMask1 = aaiAlgo[1];
        iSrcBit2, iDstBit2, fMask2 = aaiAlgo[2];
        return ((uValue >> iSrcBit0) & fMask0) \
             | (((uValue >> iSrcBit1) & fMask1) << iDstBit1) \
             | (((uValue >> iSrcBit2) & fMask2) << iDstBit2);

    @staticmethod
    def algoToZipLambda(aaiAlgo, fAlgoMask, fCompileIt = True):
        assert aaiAlgo;
        if not fCompileIt:
            if len(aaiAlgo) == 1: return MaskZipper.__zipMask1;
            if len(aaiAlgo) == 2: return MaskZipper.__zipMask2;
            if len(aaiAlgo) == 3: return MaskZipper.__zipMask3;
            return MaskZipper.zipMask;
        # Compile it:
        sBody = '';
        for iSrcBit, iDstBit, fMask in aaiAlgo:
            if sBody:
                sBody += ' | ';
            assert iSrcBit >= iDstBit;
            if iDstBit == 0:
                if iSrcBit == 0:
                    sBody += '(uValue & %#x)' % (fMask,);
                else:
                    sBody += '((uValue >> %u) & %#x)' % (iSrcBit, fMask);
            else:
                sBody += '((uValue >> %u) & %#x)' % (iSrcBit - iDstBit, fMask << iDstBit);
        _ = fAlgoMask
        #sFn = 'zipMaskCompiled_%#010x' % (fAlgoMask,);
        #sFn = 'zipMaskCompiled';
        #dTmp = {};
        #exec('def %s(uValue,_): return %s' % (sFn, sBody), globals(), dTmp);
        #return dTmp[sFn];
        return eval('lambda uValue,_: ' + sBody);

    @staticmethod
    def unzipMask(uValue, aaiAlgo):
        fRet = 0;
        for iSrcBit, iDstBit, fMask in aaiAlgo:
            fRet |= ((uValue >> iDstBit) & fMask) << iSrcBit;
        return fRet;

    @staticmethod
    def __unzipMask1(uValue, aaiAlgo):
        return uValue << aaiAlgo[0][0];

    @staticmethod
    def __unzipMask2(uValue, aaiAlgo):
        iSrcBit0, _,        fMask0 = aaiAlgo[0];
        iSrcBit1, iDstBit1, fMask1 = aaiAlgo[1];
        return ((uValue & fMask0) << iSrcBit0) | (((uValue >> iDstBit1) & fMask1) << iSrcBit1);

    @staticmethod
    def __unzipMask3(uValue, aaiAlgo):
        iSrcBit0, _,        fMask0 = aaiAlgo[0];
        iSrcBit1, iDstBit1, fMask1 = aaiAlgo[1];
        iSrcBit2, iDstBit2, fMask2 = aaiAlgo[2];
        return ((uValue & fMask0) << iSrcBit0) \
             | (((uValue >> iDstBit1) & fMask1) << iSrcBit1) \
             | (((uValue >> iDstBit2) & fMask2) << iSrcBit2);

    @staticmethod
    def algoToUnzipLambda(aaiAlgo, fAlgoMask, fCompileIt = True):
        assert aaiAlgo;
        if not fCompileIt:
            if len(aaiAlgo) == 1: return MaskZipper.__unzipMask1;
            if len(aaiAlgo) == 2: return MaskZipper.__unzipMask2;
            if len(aaiAlgo) == 3: return MaskZipper.__unzipMask3;
            return MaskZipper.unzipMask;
        # Compile it:
        sBody = '';
        for iSrcBit, iDstBit, fMask in aaiAlgo:
            if sBody:
                sBody += ' | ';
            if iDstBit == 0:
                if iSrcBit == 0:
                    sBody += '(uIdx & %#x)' % (fMask,);
                else:
                    sBody += '((uIdx & %#x) << %u)' % (fMask, iSrcBit);
            else:
                sBody += '((uIdx << %u) & %#x)' % (iSrcBit - iDstBit, fMask << iSrcBit);

        _ = fAlgoMask
        #dTmp = {};
        #sFn = 'unzipMaskCompiled';
        #sFn = 'unzipMaskCompiled_%#010x' % (fAlgoMask,);
        #exec('def %s(uIdx,_): return %s' % (sFn, sBody), globals(), dTmp);
        #return dTmp[sFn];
        return eval('lambda uIdx,_: ' + sBody);


class MaskIterator(object):
    """ Helper class for DecoderNode.constructNextLevel(). """

    ## Maximum number of mask sub-parts.
    # Lower number means fewer instructions required to convert it into an index.
    # This is implied by the code in MaskZipper.compileAlgoLimited.
    kcMaxMaskParts = 3

    def __init__(self, fMask, cMinTableSizeInBits, cMaxTableSizeInBits, fMaskNotDoneYet):
        self.fMask               = fMask;
        self.aaiAlgo             = MaskZipper.compileAlgo(fMask);
        self.fCompactMask        = MaskZipper.zipMask(fMask, self.aaiAlgo);
        self.fnExpandMask        = MaskZipper.algoToUnzipLambda(self.aaiAlgo, fMask,
                                                                self.fCompactMask.bit_count() >= 8);
        self.cMinTableSizeInBits = cMinTableSizeInBits;
        self.cMaxTableSizeInBits = cMaxTableSizeInBits;
        self.fCompactMaskNotDoneYet = MaskZipper.zipMask(fMaskNotDoneYet, self.aaiAlgo);
        #print('debug: fMask=%#x -> fCompactMask=%#x aaiAlgo=%s' % (fMask, self.fCompactMask, self.aaiAlgo));
        #self.cReturned           = 0;

    def __iter__(self):
        return self;

    def __next__(self):
        fCompactMask           = self.fCompactMask;
        fCompactMaskNotDoneYet = self.fCompactMaskNotDoneYet;
        cMinTableSizeInBits    = self.cMinTableSizeInBits
        cMaxTableSizeInBits    = self.cMaxTableSizeInBits
        while fCompactMask != 0:
            if fCompactMask & fCompactMaskNotDoneYet:
                cCurBits = fCompactMask.bit_count();
                if cMinTableSizeInBits <= cCurBits <= cMaxTableSizeInBits:
                    fMask = self.fnExpandMask(fCompactMask, self.aaiAlgo);
                    aaiMaskAlgo = MaskZipper.compileAlgoLimited(fMask);
                    if aaiMaskAlgo:
                        #assert aaiMaskAlgo == MaskZipper.compileAlgo(fMask), \
                        #    '%s vs %s' % (aaiMaskAlgo, MaskZipper.compileAlgo(fMask));
                        #self.cReturned += 1;
                        self.fCompactMask = fCompactMask - 1;
                        return (fMask, cCurBits, aaiMaskAlgo);
            fCompactMask -= 1;
        self.fCompactMask = 0;
        #print('MaskIterator: fMask=%#x -> %u items returned' % (self.fMask, self.cReturned));
        raise StopIteration;


class DecoderNode(object):

    ## The absolute maximum table size in bits index by the log2 of the instruction count.
    kacMaxTableSizesInBits = (
        2,      # [2^0 =     1] =>     4
        4,      # [2^1 =     2] =>    16
        5,      # [2^2 =     4] =>    32
        6,      # [2^3 =     8] =>    64
        7,      # [2^4 =    16] =>   128
        7,      # [2^5 =    32] =>   128
        8,      # [2^6 =    64] =>   256
        9,      # [2^7 =   128] =>   512
        10,     # [2^8 =   256] =>  1024
        11,     # [2^9 =   512] =>  2048
        12,     # [2^10 = 1024] =>  4096
        13,     # [2^11 = 2048] =>  8192
        14,     # [2^12 = 4096] => 16384
        14,     # [2^13 = 8192] => 16384
        15,     # [2^14 =16384] => 32768
    );

    kChildMaskOpcodeValueIf          = 0x7fffffff;
    kChildMaskMultipleOpcodeValueIfs = 0xffffffff;

    class TooExpensive(Exception):
        def __init__(self):
            Exception.__init__(self, None);

    def __init__(self, aoInstructions, fCheckedMask, fCheckedValue):
        #assert (~fCheckedMask & fCheckedValue) == 0;
        #for idxInstr, oInstr in enumerate(aoInstructions):
        #    assert ((oInstr.fFixedValue ^ fCheckedValue) & fCheckedMask & oInstr.fFixedMask) == 0, \
        #            '%s: fFixedValue=%#x fFixedMask=%#x fCheckedValue=%#x fCheckedMask=%#x -> %#x\naoInstructions: len=%s\n %s' \
        #            % (idxInstr, oInstr.fFixedValue, oInstr.fFixedMask, fCheckedValue, fCheckedMask,
        #               (oInstr.fFixedValue ^ fCheckedValue) & fCheckedMask & oInstr.fFixedMask,
        #               len(aoInstructions),
        #               '\n '.join(['%s%s: %#010x/%#010x %s' % ('*' if i == idxInstr else ' ', i,
        #                                                       oInstr2.fFixedValue, oInstr2.fFixedMask, oInstr2.sName)
        #                           for i, oInstr2 in enumerate(aoInstructions[:idxInstr+8])]));

        self.aoInstructions     = aoInstructions;   ##< The instructions at this level.
        self.fCheckedMask       = fCheckedMask;     ##< The opcode bit mask covered thus far.
        self.fCheckedValue      = fCheckedValue;    ##< The value that goes with fCheckedMask.
        self.fChildMask         = 0;                ##< The mask used to separate the children.
        self.dChildren          = {};               ##< Children, sparsely populated by constructNextLevel().

    @staticmethod
    def popCount(uValue):
        cBits = 0;
        while uValue:
            cBits += 1;
            uValue &= uValue - 1;
        return cBits;

    s_uLogLine = 0;
    @staticmethod
    def dprint(uDepth, sMsg):
        msNow = (time.time_ns() - g_nsProgStart) // 1000000;
        print('%u.%03u: %u: debug/%u: %s%s' % (msNow // 1000, msNow % 1000, DecoderNode.s_uLogLine, uDepth, '  ' * uDepth, sMsg));
        DecoderNode.s_uLogLine += 1;

    def constructNextLevel(self, uDepth, uMaxCost): # pylint: disable=too-many-locals,too-many-statements
        """
        Recursively constructs the
        """
        if uDepth == 0:
            for i, oInstr in enumerate(self.aoInstructions):
                self.dprint(uDepth, '%4u: %s' % (i, oInstr.toString(cchName = 32),));

        #
        # Special cases: 4 or fewer entries.
        #
        cInstructions = len(self.aoInstructions)
        if cInstructions <= 4:
            assert not self.dChildren;
            uCost = 0;
            # Special case: 1 instruction - leaf.
            if cInstructions <= 1:
                if self.aoInstructions[0].fFixedMask & ~self.fCheckedMask != 0:
                    self.fChildMask = DecoderNode.kChildMaskOpcodeValueIf;
                    uCost = 16;                                                         # 16 = kCostOpcodeValueIf
                else:
                    assert self.fChildMask == 0;

            # Special case: 2, 3 or 4 instructions - use a sequence of 'if ((uOpcode & fFixedMask) == fFixedValue)' checks.
            else:
                self.fChildMask = DecoderNode.kChildMaskMultipleOpcodeValueIfs;
                uCost = 32 * cInstructions * 2;                                         # 32 = kCostMultipleOpcodeValueIfs
            return uCost;

        #
        # The cost of one indirect call is 256, so just bail if we don't have
        # the budget for any of that.
        #
        if uMaxCost <= 256:                                                             # 256 = kCostIndirectCall
            raise DecoderNode.TooExpensive();
        if uDepth > 5:                                                                  #   5 = kMaxDepth
            raise DecoderNode.TooExpensive();

        #
        # Do an inventory of the fixed masks used by the instructions.
        #
        dMaskCounts  = collections.Counter();
        fCheckedMask = self.fCheckedMask;    # (Cache it as a local variable for speed.)
        for oInstr in self.aoInstructions:
            dMaskCounts[oInstr.fFixedMask & ~fCheckedMask] += 1;
        #assert 0 not in dMaskCounts or dMaskCounts[0] <= 1, \
        #        'dMaskCounts=%s cInstructions=%s\n%s' % (dMaskCounts, cInstructions, self.aoInstructions);
        # 0x00011c00 & 0xfffee000  = 0x0 (0)

        #
        # HACK ALERT! For level 0 and 1 we offer ways to insert hints to reduce
        #             the runtime, since it's tedious to wait for 30 min for
        #             each code tweak...  See --decoder-hint0 and --decoder-hint1.
        #
        dSpeedupFilter = None;
        if uDepth <= 1:
            if uDepth == 1:
                dSpeedupFilter = g_ddDecoderFilterDepth1.get('%x/%x' % (self.fCheckedMask, self.fCheckedValue), None);
            elif g_dDecoderFilterDepth0:
                dSpeedupFilter = g_dDecoderFilterDepth0;

        #
        # Whether to bother compiling the mask zip/unzip functions.
        #
        # The goal here is to keep the {built-in method builtins.eval} line far
        # away from top of the profiler stats, while at the same time keeping the
        # __zipMaskN and __unzipMaskN methods from taking up too much time.
        #
        fCompileMaskZip   = cInstructions >= 256;
        fCompileMaskUnzip = cInstructions >= 32; #?

        #
        # Work thru the possible masks and test out the variations (brute force style).
        #
        uCostBest        = uMaxCost;
        cChildrenBits    = 0;
        fChildrenBest    = 0;
        dChildrenBest    = {};

        fMaskNotDoneYet  = 0xffffffff;
        fCheckedValue    = self.fCheckedValue; # (Cache it as a local variable for speed.)
        iOuterLoop       = -1;
        for fOrgMask, cOccurences in dMaskCounts.most_common(3):
            iOuterLoop += 1;

            # Determin the max and min table sizes (in bits) based on the instructions using the mask.
            cMinTableSizeInBits = cOccurences.bit_length() - 1;
            if (1 << cMinTableSizeInBits) < cOccurences:
                cMinTableSizeInBits += 1;
            cMaxTableSizeInBits = self.kacMaxTableSizesInBits[cMinTableSizeInBits]; # Not quite sure about this...
            cMinTableSizeInBits -= 1;

            if uDepth <= 2:
                self.dprint(uDepth,
                            '%s Start/%u: %#010x (%u) - %u/%u instructions - tab size %u-%u; fChecked=%#x/%#x uCostBest=%#x'
                            % (('=' if iOuterLoop == 0 else '-') * 5, iOuterLoop, fOrgMask,
                               self.popCount(fOrgMask), cOccurences, cInstructions, cMinTableSizeInBits, cMaxTableSizeInBits,
                               fCheckedValue, fCheckedMask, uCostBest,));

            # Skip pointless stuff and things we've already covered.
            if cOccurences >= 2 and fOrgMask > 0 and fOrgMask != 0xffffffff and (fOrgMask & fMaskNotDoneYet) != 0:
                #
                # Step 1: Brute force relevant mask variations and pick a few masks.
                #
                # The MaskIterator skips masks that are too wide, too fragmented or
                # already covered.
                #
                # The cost calculation is mainly based on distribution vs table size,
                # trying to favor masks with more target slots.
                #
                dCandidates = {};
                for fMask, cMaskBits, aaiMaskToIdxAlgo in MaskIterator(fOrgMask, cMinTableSizeInBits, cMaxTableSizeInBits,
                                                                       fMaskNotDoneYet):
                    #if uDepth <= 2:
                    #    self.dprint(uDepth, '1>> fMask=%#010x cMaskBits=%s aaiMaskToIdxAlgo=%s...'
                    #                         % (fMask, cMaskBits, aaiMaskToIdxAlgo));
                    #assert cMaskBits <= cMaxTableSizeInBits;

                    # HACK ALERT! Skip the mask if we have a selection filter and it isn't in it.
                    if dSpeedupFilter and fMask not in dSpeedupFilter:
                        continue;

                    # Calculate base cost and check it against uCostBest before continuing.
                    uCostTmp    = 256;                                                  # 256 = kCostIndirectCall
                    uCostTmp   += (len(aaiMaskToIdxAlgo) - 1) * 2;                      #   2 = kCostPerExtraIndexStep
                    if uCostTmp >= uCostBest:
                        #if uDepth <= 2:
                        #    self.dprint(uDepth, '!!! %#010x too expensive #1: %#x vs %#x' % (fMask, uCostTmp, uCostBest));
                        continue;

                    # Compile the indexing/unindexing functions.
                    fnToIndex   = MaskZipper.algoToZipLambda(aaiMaskToIdxAlgo, fMask, fCompileMaskZip);

                    # Insert the instructions into the temporary table.
                    daoTmp = collections.defaultdict(list);
                    for oInstr in self.aoInstructions:
                        idx = fnToIndex(oInstr.fFixedValue, aaiMaskToIdxAlgo);
                        #self.dprint(uDepth, '%#010x -> %#05x %s' % (oInstr.fFixedValue, idx, oInstr.sName));
                        fNonFixedMatches = ~oInstr.fFixedMask & fMask;
                        if fNonFixedMatches == 0:
                            daoTmp[idx].append(oInstr);
                        else:
                            fIdxNonFixedMatches  = fnToIndex(fNonFixedMatches, aaiMaskToIdxAlgo);
                            cBitsNonFixedMatches = fNonFixedMatches.bit_count();
                            if cBitsNonFixedMatches < 8:
                                idxStep         = fIdxNonFixedMatches & (~fIdxNonFixedMatches + 1);
                                idxInstrMask    = fnToIndex(oInstr.fFixedMask, aaiMaskToIdxAlgo);
                                idxInstrLast    = idx | fIdxNonFixedMatches;
                                for idx2 in range(idx, idxInstrLast + idxStep, idxStep):
                                    if (idx2 & idxInstrMask) == idx:
                                        daoTmp[idx2].append(oInstr);
                            else:
                                aaiNonFixedAlgo = MaskZipper.compileAlgo(fIdxNonFixedMatches);
                                fnNonFixedUnzip = MaskZipper.algoToUnzipLambda(aaiNonFixedAlgo, fIdxNonFixedMatches);
                                for idx3 in range(MaskZipper.zipMask(fIdxNonFixedMatches, aaiNonFixedAlgo) + 1):
                                    idx2 = idx | fnNonFixedUnzip(idx3, aaiNonFixedAlgo);
                                    daoTmp[idx2].append(oInstr);

                    ## @todo Account for entires causing instruction duplication...
                    ##       Perhaps by summing up the number of instructions for the next level?
                    cEffTmpSize = len(daoTmp);

                    # Reject anything that ends up putting all the stuff in a single slot.
                    if cEffTmpSize <= 1:
                        #if uDepth <= 2: self.dprint(uDepth, '!!! bad distribution #1: fMask=%#x' % (fMask,));
                        continue;

                    # Add cost for poor average distribution.
                    rdAvgLen = float(cInstructions) / cEffTmpSize;
                    if rdAvgLen > 1.2:
                        uCostTmp += int(rdAvgLen * 8)
                        if uCostTmp >= uCostBest:
                            #if uDepth <= 2:
                            #    self.dprint(uDepth, '!!! %#010x too expensive #2: %#x vs %#x (rdAvgLen=%s)'
                            #                        % (fMask, uCostTmp, uCostBest, rdAvgLen));
                            continue;

                    # Add the cost for unused entries under reasonable table population.
                    cNominalFill = 1 << (cMaskBits - 1); # 50% full or better is great.
                    if cEffTmpSize < cNominalFill:
                        uCostTmp += ((cNominalFill - cEffTmpSize) * 2);                 # 2 = kCostUnusedTabEntry
                        if uCostTmp >= uCostBest:
                            #if uDepth <= 2:
                            #    self.dprint(uDepth, '!!! %#010x too expensive #3: %#x vs %#x' % (fMask, uCostTmp, uCostBest));
                            continue;

                    # Record it as a candidate.
                    dCandidates[uCostTmp] = (fMask, cMaskBits, aaiMaskToIdxAlgo, daoTmp);
                    if len(dCandidates) > 64:
                        dOld = dCandidates;
                        dCandidates = { uKey:dOld[uKey] for uKey in sorted(dCandidates.keys())[:4] };
                        del dOld;

                #
                # Step 2: Process the top 4 candidates.
                #
                for uCostTmp in sorted(dCandidates.keys())[:4]:
                    fMask, cMaskBits, aaiMaskToIdxAlgo, daoTmp = dCandidates[uCostTmp];

                    #if uDepth <= 2:
                    #    self.dprint(uDepth, '2>> fMask=%#010x cMaskBits=%s aaiMaskToIdxAlgo=%s #daoTmp=%s...'
                    #                         % (fMask, cMaskBits, aaiMaskToIdxAlgo, len(daoTmp),));
                    #assert cMaskBits <= cMaxTableSizeInBits;

                    # Construct decoder nodes from the aaoTmp lists, construct sub-levels and calculate costs.
                    fnFromIndex  = MaskZipper.algoToUnzipLambda(aaiMaskToIdxAlgo, fMask, fCompileMaskUnzip);
                    dChildrenTmp = {};
                    try:
                        for idx, aoInstrs in daoTmp.items():
                            oChild = DecoderNode(aoInstrs,
                                                 fCheckedMask  | fMask,
                                                 fCheckedValue | fnFromIndex(idx, aaiMaskToIdxAlgo));
                            dChildrenTmp[idx] = oChild;
                            uCostTmp += oChild.constructNextLevel(uDepth + 1, uCostBest - uCostTmp);
                            if uCostTmp >= uCostBest:
                                break;
                    except DecoderNode.TooExpensive:
                        #if uDepth <= 2:
                        #    self.dprint(uDepth, '!!! %#010x too expensive #4: %#x+child vs %#x' % (fMask, uCostTmp, uCostBest));
                        continue;

                    # Is this mask better than the previous?
                    if uCostTmp < uCostBest:
                        if uDepth <= 2:
                            self.dprint(uDepth,
                                        '+++ %s best!  %#010x (%u) uCost=%#x; %u ins in %u slots (previous %#010x / %#x) ...'
                                        % ('New' if cChildrenBits else '1st', fMask, cMaskBits, uCostTmp,
                                           cInstructions, len(dChildrenTmp), fChildrenBest, uCostBest, ));
                        uCostBest      = uCostTmp;
                        cChildrenBits  = cMaskBits;
                        fChildrenBest  = fMask;
                        dChildrenBest  = dChildrenTmp;
                    #elif uDepth <= 2:
                    #    self.dprint(uDepth, '!!! %#010x too expensive #5: %#x vs %#x' % (fMask, uCostTmp, uCostBest));

                # Note that we've covered all the permutations in the given mask.
                fMaskNotDoneYet &= ~fOrgMask;

        # Drop it if too expensive.
        if uCostBest >= uMaxCost:
            raise DecoderNode.TooExpensive();

        if dChildrenBest is None:
            print('warning! No solution! depth=%u #Instruction=%u' % (uDepth, cInstructions));
            raise Exception('fixme')

        #assert fChildrenBest.bit_count() == cChildrenBits;
        #assert len(dChildrenBest) <= (1 << cChildrenBits)
        if uDepth <= 2:
            self.dprint(uDepth,
                        '===== Final: fMask=%#010x (%u) uCost=%#x #Instructions=%u in %u slots over %u entries...'
                        % (fChildrenBest, cChildrenBits, uCostBest, cInstructions, len(dChildrenBest), 1 << cChildrenBits));

        # Done.
        self.fChildMask = fChildrenBest;
        self.dChildren  = dChildrenBest;

        return uCostBest;

    def setInstrProps(self, uDepth):
        """
        Sets the fDecoderLeafCheckNeeded instruction property.
        """
        if not self.dChildren:
            assert len(self.aoInstructions) != 1 or self.fChildMask in (0, DecoderNode.kChildMaskOpcodeValueIf);
            assert len(self.aoInstructions) == 1 or self.fChildMask == DecoderNode.kChildMaskMultipleOpcodeValueIfs;
            for oInstr in self.aoInstructions:
                oInstr.fDecoderLeafCheckNeeded = self.fChildMask == DecoderNode.kChildMaskOpcodeValueIf;
        else:
            for oChildNode in self.dChildren.values():
                oChildNode.setInstrProps(uDepth + 1);

    def getFuncName(self, sInstrSet, uDepth):
        """
        Returns the function name at the specific depth.
        """
        if not sInstrSet:
            sInstrSet = self.aoInstructions[0].getInstrSetName()
        if self.dChildren or len(self.aoInstructions) > 1:
            return 'iemDecode%s_%08x_%08x_%u' % (sInstrSet, self.fCheckedMask, self.fCheckedValue, uDepth,);
        return 'iemDecode%s_%s' % (sInstrSet, self.aoInstructions[0].getCName(),);

#
# Generators
#

class IEMArmGenerator(object):

    def __init__(self):
        self.oOptions        = None;
        self.dDecoderRoots   = {};
        self.dRootsIndexExpr = {};


    def constructDecoder(self):
        """
        Creates the decoder(s) to the best our abilities.
        """
        for oSet in g_aoArmInstructionSets:
            oRoot = DecoderNode(sorted(oSet.aoAllInstructions,
                                       key = operator.attrgetter('fFixedMask','fFixedValue', 'sName')),#[:384],
                                0, 0);
            self.dDecoderRoots[oSet.sName] = oRoot;
            oRoot.constructNextLevel(0, sys.maxsize);

            # Set the fDecoderLeafCheckNeeded property of the instructions.
            oRoot.setInstrProps(0);


    def generateLicenseHeader(self, oVerInfo):
        """
        Returns the lines for a license header.
        """
        sDashYear = '-%s' % datetime.date.today().year;
        if sDashYear == '-2025':
            sDashYear = '';
        return [
            '/*',
            ' * Autogenerated by $Id: bsd-spec-analyze.py $',
            ' * from the open source %s specs, build %s (%s)'
            % (oVerInfo['architecture'], oVerInfo['build'], oVerInfo['ref'],),
            ' * dated %s.' % (oVerInfo['timestamp'],),
            ' *',
            ' * Do not edit!',
            ' */',
            '',
            '/*',
            ' * Copyright (C) 2025' + sDashYear + ' Oracle and/or its affiliates.',
            ' *',
            ' * This file is part of VirtualBox base platform packages, as',
            ' * available from https://www.virtualbox.org.',
            ' *',
            ' * This program is free software; you can redistribute it and/or',
            ' * modify it under the terms of the GNU General Public License',
            ' * as published by the Free Software Foundation, in version 3 of the',
            ' * License.',
            ' *',
            ' * This program is distributed in the hope that it will be useful, but',
            ' * WITHOUT ANY WARRANTY; without even the implied warranty of',
            ' * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU',
            ' * General Public License for more details.',
            ' *',
            ' * You should have received a copy of the GNU General Public License',
            ' * along with this program; if not, see <https://www.gnu.org/licenses>.',
            ' *',
            ' * SPDX-License-Identifier: GPL-3.0-only',
            ' */',
            '',
            '',
        ];

    def generateImplementationStubHdr(self, sInstrSet):
        """
        Generate implementation stubs.
        """
        asLines = self.generateLicenseHeader(g_oArmInstructionVerInfo);

        # Organize this by instruction set, groups and instructions.
        sPrevCategory = '';
        for oInstr in sorted(g_dArmInstructionSets[sInstrSet].aoAllInstructions, key = ArmInstruction.getSetAndGroupNames):
            # New group/category?
            sCategory = ' / '.join(oInstr.getSetAndGroupNames(),);
            if sCategory != sPrevCategory:
                asLines += [
                    '',
                    '',
                    '/*',
                    ' *',
                    ' * Instruction Set & Groups: %s' % (sCategory,),
                    ' *',
                    ' */',
                ];
                sPrevCategory = sCategory;

            # Emit the instruction stub.
            asArgs  = [ oField.sName for oField in oInstr.getNamedNonFixedFieldsSortedByPosition() ];
            asLines += [
                '',
                '/* %s (%08x/%08x) */' % (oInstr.sAsmDisplay, oInstr.fFixedMask, oInstr.fFixedValue,),
                '//#define IEM_INSTR_IMPL_%s__%s(%s)' % (sInstrSet, oInstr.getCName(), ', '.join(asArgs)),
                '',
            ]

        return (True, asLines);

    def generateA64ImplementationStubHdr(self, sFilename, iPartNo):
        _ = sFilename; _ = iPartNo;
        return self.generateImplementationStubHdr('A64');


    def generateDecoderFunctions(self, sInstrSet):
        """
        Generates the leaf decoder functions.
        """

        class CExprHelper(object):
            def __init__(self, oInstr):
                self.oInstr = oInstr;

            def getFieldInfo(self, sName, sVar = '', sNamespace = ''):
                if not sVar and not sNamespace:
                    oInstr = self.oInstr;
                    oField = oInstr.getFieldByName(sName, False);
                    if oField:
                        return (sName, oField.cBitsWidth);
                    # Look for the field in groups and sets and generate a name that extracts it from uOpcode:
                    for oParent in oInstr.oParent.getUpIterator():
                        oField = oParent.getFieldByName(sName, False);
                        if oField:
                            fFieldOpMask = oField.getShiftedMask();
                            if (oInstr.fFixedMask & fFieldOpMask) == fFieldOpMask:
                                return ('%#x /*%s@%u*/'
                                        % ((oInstr.fFixedValue & fFieldOpMask) >> oField.iFirstBit, sName, oField.iFirstBit),
                                        oField.cBitsWidth);
                            return ('((uOpcode >> %u) & %#x)' % (oField.iFirstBit, oField.getMask()), oField.cBitsWidth);
                raise Exception('Field (%s.%s.)%s was not found for instruction %s'
                                % (sNamespace, sVar, sName, oInstr.sName,));

            def convertFunctionCall(self, oCall):
                if oCall.sName == 'IsFeatureImplemented':
                    if len(oCall.aoArgs) != 1:
                        raise Exception('Unexpected argument count for IsFeatureImplemented call: %s' % (oCall.aoArgs,));
                    if not isinstance(oCall.aoArgs[0], ArmAstIdentifier):
                        raise Exception('Argument to IsFeatureImplemented is not an identifier: %s (%s)'
                                        % (oCall.aoArgs[0].sType, oCall.aoArgs[0]));
                    sFeatureNm = oCall.aoArgs[0].sName;
                    sCpumFeature = g_dSpecFeatToCpumFeat.get(sFeatureNm, None);
                    if sCpumFeature is None:
                        raise Exception('Unknown IsFeatureImplemented parameter: %s (see g_dSpecFeatToCpumFeat)' % (sFeatureNm));
                    if not isinstance(sCpumFeature, str):
                        return 'false /** @todo IEM_GET_GUEST_CPU_FEATURES(pVCpu)->%s*/' % (sFeatureNm,);
                    return 'IEM_GET_GUEST_CPU_FEATURES(pVCpu)->%s /*%s*/' % (sCpumFeature, sFeatureNm,)
                raise Exception('Call to unsupported function: %s (%s)' % (oCall.sName, oCall.aoArgs,));

        asLines = [];
        for oInstr in sorted(g_dArmInstructionSets[sInstrSet].aoAllInstructions,
                             key = operator.attrgetter('sName', 'sAsmDisplay')):
            sCName = oInstr.getCName();
            asLines += [
                '',
                '/* %08x/%08x: %s' % (oInstr.fFixedMask, oInstr.fFixedValue, oInstr.sAsmDisplay,),
                '   %s */'% (oInstr.getSetAndGroupNamesWithLabels(),),
                'FNIEMOP_DEF_1(iemDecode%s_%s, uint32_t, uOpcode)' % (sInstrSet, sCName,),
                '{',
            ];

            # The final decoding step, if needed.
            sIndent = '';
            asTail  = [];
            if oInstr.fDecoderLeafCheckNeeded:
                asLines += [
                    '    if ((uOpcode & UINT32_C(%#010x)) == UINT32_C(%#010x))' % (oInstr.fFixedMask, oInstr.fFixedValue,),
                    '    {',
                ];
                asTail  = [
                    '    Log(("Invalid instruction %%#x at %%x\\n", uOpcode, pVCpu->cpum.GstCtx.Pc.u64));',
                    '    IEMOP_RAISE_INVALID_OPCODE_RET();',
                    '}',
                ];
                sIndent = '    ';


            # Decode the fields and prepare for passing them as arguments.
            asArgs  = [];
            sLogFmt = '';
            for oField in oInstr.getNamedNonFixedFieldsSortedByPosition(): # ArmEncodesetField
                assert oField.sName and oField.fFixed != oField.getMask();
                asArgs.append(oField.sName);
                if oField.cBitsWidth < 4:
                    sLogFmt += ' %s=%%u' % (oField.sName,)
                else:
                    sLogFmt += ' %s=%%#x' % (oField.sName,)
                asLines.append('%s    uint32_t const %-10s = (uOpcode >> %2u) & %#010x;'
                               % (sIndent, oField.sName, oField.iFirstBit, (1 << oField.cBitsWidth) - 1,));

            # Any additional conditions for the instructions.
            if not oInstr.oCondition.isBoolAndTrue():
                asLines += [
                    sIndent + '    if (' + oInstr.oCondition.toCExpr(CExprHelper(oInstr)) + ')',
                    sIndent + '    {',
                ];

                asTail = [
                    sIndent + '    Log(("Invalid instruction %%#x at %%x (cond)\\n", uOpcode, pVCpu->cpum.GstCtx.Pc.u64));',
                    sIndent + '    IEMOP_RAISE_INVALID_OPCODE_RET();',
                    sIndent + '}',
                ] + asTail;
                sIndent += '    ';

            # Log and call implementation.
            asLines += [
                '%s    LogFlow(("%%018x/%%010x: %s%s\\n", %s));'
                % (sIndent, sCName, sLogFmt, ', '.join(['pVCpu->cpum.GstCtx.Pc.u64', 'uOpcode',] + asArgs),),
                '#ifdef IEM_INSTR_IMPL_%s__%s' % (sInstrSet, sCName,),
                '%s    IEM_INSTR_IMPL_%s__%s(%s);' % (sIndent, sInstrSet, sCName, ', '.join(asArgs),),
                '#else',
                '%s    RT_NOREF(%s);' % (sIndent, ', '.join(asArgs + ['pVCpu', 'uOpcode',]),),
                '%s    return VERR_IEM_INSTR_NOT_IMPLEMENTED;' % (sIndent,),
                '#endif',
                '%s}' % (sIndent),
            ];

            asLines.extend(asTail);
        return asLines;


    class DecoderCodeBlock(object):
        """ A code block. """

        def __init__(self, sInstrSet, sName, sMatches, sStats = None):
            self.sHash     = sInstrSet;
            self.sName     = sName;
            self.asMatches = [sMatches,];
            self.sStats    = sStats;

        def getName(self):
            return self.sName;

        def getLines(self):
            return [
                '',
                '/* %s */' % ('\n   '.join(self.asMatches if not self.sStats else list(self.asMatches) + [self.sStats,]),),
            ];

        def getHash(self):
            return self.sHash;

        def _addLinesToHash(self, asLines):
            """ Adds the lines to the hash value. """
            oHash = hashlib.sha256();
            oHash.update(b'%u lines\n' % (len(asLines),));
            for sLine in asLines:
                oHash.update(sLine.encode('utf-8'));
            self.sHash += '-' + oHash.hexdigest();

    class DecoderCodeMultiIfFunc(DecoderCodeBlock):
        """ Helper class for deduplicating multiple-if functions. """
        def __init__(self, sInstrSet, oNode, uDepth):
            IEMArmGenerator.DecoderCodeBlock.__init__(self, sInstrSet, oNode.getFuncName(sInstrSet, uDepth),
                                                      '%08x/%08x level %u' % (oNode.fCheckedMask, oNode.fCheckedValue, uDepth,) );
            self.asBody = [];
            for oInstr in oNode.aoInstructions:
                self.asBody += [
                    '    if ((uOpcode & UINT32_C(%#010x)) == UINT32_C(%#010x))' % (oInstr.fFixedMask, oInstr.fFixedValue,),
                    '        return iemDecode%s_%s(pVCpu, uOpcode);' % (sInstrSet, oInstr.getCName(),),
                ];
            ## @todo check if the masks are restricted to a few bit differences at
            ## this point and we can skip the iemDecodeA64_Invalid call.
            self.asBody += [
                '    return iemDecode%s_Invalid(pVCpu, uOpcode);' % (sInstrSet,),
            ];
            self._addLinesToHash(self.asBody);

        def getLines(self):
            asLines  = IEMArmGenerator.DecoderCodeBlock.getLines(self);
            asLines += [
                'FNIEMOP_DEF_1(%s, uint32_t, uOpcode)' % (self.getName(),),
                '{',
            ];
            asLines += self.asBody;
            asLines += [
                '}',
            ];
            return asLines;

    class DecoderCodeTableLeafEntry(object):
        """ Special DecoderCodeTableFunc::dChildCode for leaf decoder functions. """
        def __init__(self, sName):
            self.sName = sName;

        def getHash(self):
            return self.sName;

        def getName(self):
            return self.sName;

    class DecoderCodeTableFunc(DecoderCodeBlock):
        """ Helper class for table based decoder function. """

        def __init__(self, sInstrSet, sName, sMatches, sStats, uDepth, cTabEntries, dChildCode, asIdx):
            IEMArmGenerator.DecoderCodeBlock.__init__(self, sInstrSet, sName, sMatches, sStats);
            self.sInstrSet   = sInstrSet;
            self.uDepth      = uDepth;
            self.cTabEntries = cTabEntries;
            self.dChildCode  = dChildCode;      # Note! DecoderCodeBlock or DecoderCodeTableLeafEntry instances.
            self.asIdx       = asIdx;

            self._addLinesToHash(asIdx);
            self._addLinesToHash(['%s-%s' % (idx, oCodeBlock.getHash(),) for idx, oCodeBlock in dChildCode.items()]);

        def getLines(self):
            # Generate the function.  For the top level we just do the table, as
            # the functions are static and we need the interpreter code to be able
            # to address the symbol and this is the speedier way.
            asLines = IEMArmGenerator.DecoderCodeBlock.getLines(self);
            if self.uDepth > 0:
                asLines += [
                    'FNIEMOP_DEF_1(%s, uint32_t, uOpcode)' % (self.getName(),),
                    '{',
                    '    static PFIEMOPU32 const s_apfn[] =',
                    '    {',
                ];
                sTabNm  = 's_apfn';
                sIndent = '    ';
            else:
                sTabNm  = 'g_apfnIemInterpretOnly' + self.sInstrSet;
                asLines += [
                    'PFIEMOPU32 const %s[] =' % (sTabNm,),
                    '{',
                ];
                sIndent = '';

            idxPrev = -1;
            for idx in sorted(self.dChildCode):
                idxPrev += 1;
                while idxPrev < idx:
                    asLines.append(sIndent + '    iemDecode%s_Invalid, // %s' % (self.sInstrSet, idxPrev,));
                    idxPrev += 1;
                asLines.append('%s    %s,' % (sIndent, self.dChildCode[idx].getName(),));

            while idxPrev + 1 < self.cTabEntries:
                idxPrev += 1;
                asLines.append(sIndent + '    iemDecode%s_Invalid, // %s' % (self.sInstrSet, idxPrev,));

            asLines += [
                '%s};' % (sIndent,),
                '%sAssertCompile(RT_ELEMENTS(%s) == %#x);' % (sIndent, sTabNm, self.cTabEntries,),
                '',
            ];

            if self.uDepth > 0:
                asLines += self.asIdx;
                asLines += [
                    '    return s_apfn[idx](pVCpu, uOpcode);',
                    '}'
                ];
            return asLines;


    def generateDecoderCode(self, sInstrSet, oNode, uDepth, dCodeCache):
        """
        Recursively generates the decoder code.
        """
        assert oNode.fChildMask != 0 and oNode.fChildMask not in (0x7fffffff, 0xffffffff, 0x4fffffff), \
            'fChildMask=%s #dChildren=%s aoInstr=%s' % (oNode.fChildMask, len(oNode.dChildren), oNode.aoInstructions,);
        assert oNode.dChildren;

        #
        # First recurse.
        #
        aoCodeBlocks    = [];
        dChildCode      = {};
        cLeafEntries    = 0;
        cMultiIfEntries = 0;
        cReusedCode     = 0;
        for idx in sorted(oNode.dChildren):
            oChildNode = oNode.dChildren[idx];
            if oChildNode.dChildren:
                aoSubCodeBlocks = self.generateDecoderCode(sInstrSet, oChildNode, uDepth + 1, dCodeCache);
            elif oChildNode.fChildMask == DecoderNode.kChildMaskMultipleOpcodeValueIfs:
                assert len(oChildNode.aoInstructions) > 1;
                aoSubCodeBlocks = [IEMArmGenerator.DecoderCodeMultiIfFunc(sInstrSet, oChildNode, uDepth + 1),];
                cMultiIfEntries += 1;
            else:
                assert len(oChildNode.aoInstructions) == 1;
                assert oChildNode.fChildMask in [DecoderNode.kChildMaskOpcodeValueIf, 0];
                cLeafEntries += 1;
                dChildCode[idx] = IEMArmGenerator.DecoderCodeTableLeafEntry(oChildNode.getFuncName(sInstrSet, -1));
                continue;

            oCodeBlock  = aoSubCodeBlocks[-1];
            oCachedCode = dCodeCache.get(oCodeBlock.getHash(), None);
            if len(aoSubCodeBlocks) != 1 or oCachedCode is None:
                assert oCachedCode is None; # Code shouldn't be in the hash if it has new dependencies!
                dChildCode[idx] = oCodeBlock;
                aoCodeBlocks.extend(aoSubCodeBlocks);
                dCodeCache[oCodeBlock.getHash()] = oCodeBlock; # comment out of disable code reuse.
            else:
                #print('debug: code cache hit');
                oCachedCode.asMatches += oCodeBlock.asMatches;
                dChildCode[idx] = oCachedCode;
                cReusedCode += 1;

        assert len(dChildCode) == len(oNode.dChildren);
        assert dChildCode;

        #
        # Match info w/ stats.
        #
        cTabEntries = 1 << oNode.fChildMask.bit_count();
        sStats   = 'mask=%#x entries=%#x children=%#x valid=%%%u (%#x) leaf=%%%u (%#x) multi-if=%%%u (%#x) reuse=%%%u (%#x)' \
                 % (oNode.fChildMask, cTabEntries, len(oNode.dChildren),
                    int(round(len(oNode.dChildren) * 100.0 / cTabEntries)), len(oNode.dChildren),
                    int(round(cLeafEntries         * 100.0 / cTabEntries)), cLeafEntries,
                    int(round(cMultiIfEntries      * 100.0 / cTabEntries)), cMultiIfEntries,
                    int(round(cReusedCode          * 100.0 / cTabEntries)), cReusedCode, );
        if uDepth == 0:
            sMatches = '--decoder-hint0 %#x' % (oNode.fChildMask,);
        elif uDepth == 1 and len(oNode.aoInstructions) >= self.oOptions.iDecoderL1Threshold:
            sMatches = '--decoder-hint1 %#x/%#x/%#x' % (oNode.fCheckedMask, oNode.fCheckedValue, oNode.fChildMask);
        else:
            sMatches = '%08x/%08x level %u' % (oNode.fCheckedMask, oNode.fCheckedValue, uDepth,);

        #
        # Code for extracting the index from uOpcode.
        #
        aaiAlgo = MaskZipper.compileAlgo(oNode.fChildMask);
        assert aaiAlgo, 'fChildMask=%s #children=%s instrs=%s' % (oNode.fChildMask, len(oNode.dChildren), oNode.aoInstructions,);
        asIdx = [
            '    /* fMask=%#010x -> %#010x */' % (oNode.fChildMask, cTabEntries - 1),
            '    uintptr_t const idx = ((uOpcode >> %2u) & UINT32_C(%#010x)) /* bit %2u L %u -> 0 */'
            % (aaiAlgo[0][0], aaiAlgo[0][2], aaiAlgo[0][0], aaiAlgo[0][2].bit_count(), ),
        ];
        for iSrcBit, iDstBit, fMask in aaiAlgo[1:]:
            asIdx.append('                        | ((uOpcode >> %2u) & UINT32_C(%#010x)) /* bit %2u L %u -> %u */'
                         % (iSrcBit - iDstBit, fMask << iDstBit, iSrcBit, fMask.bit_count(), iDstBit));
        asIdx[-1] += ';';

        # For the top level table, we save the expression so we can later put in a header file.
        if uDepth == 0:
            self.dRootsIndexExpr[sInstrSet] = asIdx;

        #
        # Create the code block for this table-based decoder function.
        #
        oCodeBlock = IEMArmGenerator.DecoderCodeTableFunc(sInstrSet, oNode.getFuncName(sInstrSet, uDepth), sMatches, sStats,
                                                          uDepth, cTabEntries, dChildCode, asIdx)
        return aoCodeBlocks + [oCodeBlock,];

    def generateDecoderCpp(self, sInstrSet):
        """ Generates the decoder data & code. """
        if sInstrSet not in self.dDecoderRoots:
            raise Exception('Instruction set not found: %s' % (sInstrSet,));

        asLines = self.generateLicenseHeader(g_oArmInstructionVerInfo);
        asLines += [
            '#define LOG_GROUP LOG_GROUP_IEM',
            '#define VMCPU_INCL_CPUM_GST_CTX',
            '#include "IEMInternal.h"',
            '#include <VBox/vmm/vm.h>',
            '#include "VBox/err.h"',
            '',
            '#include "iprt/armv8.h"',
            '',
            '#include "IEMMc.h"',
            '#include "IEMInline-armv8.h"',
            '',
            '#include "%s"' % (os.path.basename(self.oOptions.sFileDecoderHdr) if self.oOptions.sFileDecoderHdr
                               else 'IEMAllIntpr%sTables-armv8.h' % (sInstrSet),),
            '#include "%s"' % (os.path.basename(self.oOptions.sFileStubHdr) if self.oOptions.sFileStubHdr
                               else 'IEMAllInstr%sImpl.h' % (sInstrSet),),
            '',
            '',
            '/** Invalid instruction decoder function. */',
            'FNIEMOP_DEF_1(iemDecode%s_Invalid, uint32_t, uOpcode)' % (sInstrSet,),
            '{',
            '    Log(("Invalid instruction %#x at %x\\n", uOpcode, pVCpu->cpum.GstCtx.Pc.u64));',
            '    RT_NOREF_PV(uOpcode);',
            '    IEMOP_RAISE_INVALID_OPCODE_RET();',
            '}',
        ];

        asLines += self.generateDecoderFunctions(sInstrSet);

        dCodeCache   = {};
        assert self.dDecoderRoots[sInstrSet].dChildren;
        aoCodeBlocks = self.generateDecoderCode(sInstrSet, self.dDecoderRoots[sInstrSet], 0, dCodeCache);
        for oCodeBlock in aoCodeBlocks:
            asLines.extend(oCodeBlock.getLines());

        return (True, asLines);

    def generateA64DecoderCpp(self, sFilename, iPartNo):
        _ = sFilename; _ = iPartNo;
        return self.generateDecoderCpp('A64');


    def generateDecoderHdr(self, sFilename, iPartNo):
        """ Generates the decoder header file. """
        _ = iPartNo;

        asLines = self.generateLicenseHeader(g_oArmInstructionVerInfo);
        sBlockerName = re.sub('[.-]', '_', os.path.basename(sFilename));
        asLines += [
            '#ifndef VMM_INCLUDED_SRC_VMMAll_target_armv8_%s' % (sBlockerName,),
            '#define VMM_INCLUDED_SRC_VMMAll_target_armv8_%s' % (sBlockerName,),
            '#ifndef RT_WITHOUT_PRAGMA_ONCE',
            '# pragma once',
            '#endif',
            '',
        ];
        for sInstrSet in sorted(self.dDecoderRoots.keys()):
            asLines += [
                '/** The top-level %s decoder table for the IEM interpreter. */' % (sInstrSet,),
                'extern PFIEMOPU32 const g_apfnIemInterpretOnly%s[%#x];'
                % (sInstrSet, 1 << self.dDecoderRoots[sInstrSet].fChildMask.bit_count()),
                '',
                '/**',
                ' * Calculates the index for @a uOpcode in g_apfnIemInterpretOnly%s.' % (sInstrSet,),
                ' */',
                'DECL_FORCE_INLINE(uintptr_t) iemInterpretOnly%sCalcIndex(uint32_t uOpcode)' % (sInstrSet,),
                '{',
            ];
            assert sInstrSet in self.dRootsIndexExpr and len(self.dRootsIndexExpr[sInstrSet]); # Set by generateDecoderCpp!
            asLines += self.dRootsIndexExpr[sInstrSet];
            asLines += [
                '    return idx;',
                '}',
                '',
            ];
        asLines += [
            '#endif /* !VMM_INCLUDED_SRC_VMMAll_target_armv8_%s */' % (sBlockerName,),
            '',
        ];
        return (True, asLines);


    #
    # System registers
    #

    class SysRegAccessorInfo(object):
        """ Info about an accessor we're emitting code for. """
        def __init__(self, oAccessor, oReg, sEnc):
            self.oAccessor  = oAccessor     # type: ArmAccessorSystem
            self.oReg       = oReg;
            self.sEnc       = sEnc;
            self.oCode      = None;
            # Stats for the code generator.
            self.cCallsToIsFeatureImplemented = 0;


    class SysRegGeneratorBase(object):
        """ Base class for the system register access code generators. """
        def __init__(self, sInstr):
            self.sInstr     = sInstr;
            self.aoInfo     = []           # type: List[SysRegAccessorInfo]

        def generateOneHandler(self, oInfo):
            return [ '', '/// @todo %s / %s' % (self.sInstr, oInfo.sEnc,) ];

        def generateMainFunction(self):
            return [ '', '/// @todo %s main function' % (self.sInstr,) ];

        kdAstForFunctionsWithoutArguments = {
            'EL2Enabled': # HaveEL(EL2) && (!Have(EL3) || SCR_curr[].NS || IsSecureEL2Enabled())
            ArmAstBinaryOp(ArmAstFunction('HaveEL', [ArmAstIdentifier('EL2'),]),
                                         '&&',
                                         ArmAstBinaryOp.orListToTree([
                ArmAstUnaryOp('!', ArmAstFunction('HaveEL', [ArmAstIdentifier('EL3'),])),
                ArmAstDotAtom([ArmAstIdentifier('SCR_curr'), ArmAstIdentifier('NS')]), # SCR_curr is SCR or SCR_EL3.
                ArmAstFunction('IsSecureEL2Enabled', []),
            ])),
        };

        def transformCodePass1Callback(self, oNode, fEliminationAllowed, oInfo):
            """ Callback for pass 1: Code flow adjustments; Optimizations. """
            _ = fEliminationAllowed; _ = oInfo;
            if isinstance(oNode, ArmAstIfList):
                # If we have a complete series of PSTATE.EL == ELx checks,
                # turn the final one into an else case to help compilers make
                # better sense of the code flow.
                if len(oNode.aoIfConditions) >= 4 and not oNode.oElseStatement:
                    aidxEl = [-1, -1, -1, -1];
                    for idxCond, oCond in enumerate(oNode.aoIfConditions):
                        if isinstance(oCond, ArmAstBinaryOp):
                            if oCond.sOp == '==':
                                if oCond.oLeft.isMatchingDotAtom('PSTATE', 'EL'):
                                    idxEl = ('EL0', 'EL1', 'EL2', 'EL3').index(oCond.oRight.getIdentifierName());
                                    if idxEl >= 0:
                                        assert aidxEl[idxEl] == -1;
                                        aidxEl[idxEl] = idxCond;
                    if aidxEl[0] >= 0 and aidxEl[1] >= 0 and aidxEl[2] >= 0 and aidxEl[3] >= 0:
                        # We've found checks for each of the 4 EL levels.  Convert the last one into the else.
                        idxLast = max(aidxEl);
                        assert idxLast + 1 == len(oNode.aoIfStatements); # There shall not be anything after the final EL check.
                        oNode.oElseStatement = oNode.aoIfStatements[idxLast];
                        oNode.aoIfConditions = oNode.aoIfConditions[:idxLast];
                        oNode.aoIfStatements = oNode.aoIfStatements[:idxLast];

            elif isinstance(oNode, ArmAstFunction):
                # Since we don't implement any external debug state (no EDSCR.STATUS),
                # the Halted() function always returns False.  Eliminate it.
                if oNode.isMatchingFunctionCall('Halted'):
                    return None;

                # The EL3SDDUndefPriority() and EL3SDDUndef() can likewise be eliminated,
                # as they requires Halted() to be true and EDSCR.SDD to be set.
                if oNode.isMatchingFunctionCall('EL3SDDUndefPriority') or oNode.isMatchingFunctionCall('EL3SDDUndef'):
                    return None;

                # The HaveAArch64() must be true if we're in a A64 instruction handler.
                if oNode.isMatchingFunctionCall('HaveAArch64'):
                    if self.sInstr.startswith('A64'):
                        return ArmAstBool(True);
                    return ArmAstFunction('IsFeatureImplemented', [ArmAstIdentifier('FEAT_AA64')]);

                # Translate HaveEL(ELx) call into the corresponding feature checks.
                if oNode.sName == 'HaveEL' and len(oNode.aoArgs) == 1:
                    if oNode.aoArgs[0].isMatchingIdentifier('EL3'):
                        return ArmAstBinaryOp.orListToTree([
                            ArmAstFunction('IsFeatureImplemented', [ArmAstIdentifier('FEAT_AA64EL3')]),
                            ArmAstFunction('IsFeatureImplemented', [ArmAstIdentifier('FEAT_AA32EL3')]),
                        ]);
                    if oNode.aoArgs[0].isMatchingIdentifier('EL2'):
                        return ArmAstBinaryOp.orListToTree([
                            ArmAstFunction('IsFeatureImplemented', [ArmAstIdentifier('FEAT_AA64EL2')]),
                            ArmAstFunction('IsFeatureImplemented', [ArmAstIdentifier('FEAT_AA32EL2')]),
                        ]);
                    if oNode.aoArgs[0].isMatchingIdentifier('EL1') or oNode.aoArgs[0].isMatchingIdentifier('EL0'):
                        return ArmAstBool(True); # EL0 and EL1 are mandatory.
                    raise Exception('Unexpected HaveEL argument: %s' % (oNode.aoArgs[0],));

                # Generic mapping of functions without any arguments:
                if len(oNode.aoArgs) == 0 and oNode.sName in self.kdAstForFunctionsWithoutArguments:
                    oNode = self.kdAstForFunctionsWithoutArguments[oNode.sName].clone();
                    return self.transformCodePass1(oInfo, oNode);

            return oNode;

        def transformCodePass1(self, oInfo, oCode):
            """ Code transformation pass 1: Code flow adjustments; Optimizations. """
            return oCode.transform(self.transformCodePass1Callback, True, oInfo);

        def transformCodePass2Callback(self, oNode, fEliminationAllowed, oInfo):
            """ Callback for pass 2: C++ translation. """
            if isinstance(oNode, ArmAstFunction):
                # Undefined() -> return iemRaiseUndefined(pVCpu);
                if oNode.isMatchingFunctionCall('Undefined'):
                    return ArmAstCppStmt('return iemRaiseUndefined(pVCpu);');

                # IsFeatureImplemented(FEAT_xxxx) -> pGstFeat->fXxxx:
                if oNode.sName == 'IsFeatureImplemented':
                    if len(oNode.aoArgs) != 1 or not isinstance(oNode.aoArgs[0], ArmAstIdentifier):
                        raise Exception('Unexpected IsFeatureImplemented arguments: %s' % (oNode.aoArgs,));
                    sFeatureNm   = oNode.aoArgs[0].sName;
                    sCpumFeature = g_dSpecFeatToCpumFeat.get(sFeatureNm, None);
                    if sCpumFeature is None:
                        raise Exception('Unknown IsFeatureImplemented parameter: %s (see g_dSpecFeatToCpumFeat)' % (sFeatureNm));
                    if not isinstance(sCpumFeature, str):
                        return ArmAstCppExpr('false /** @todo pGstFeats->%s*/' % (sFeatureNm,));
                    oInfo.cCallsToIsFeatureImplemented += 1;
                    return ArmAstCppExpr('pGstFeats->%s' % (sCpumFeature,));

            elif isinstance(oNode, ArmAstBinaryOp):
                # PSTATE.EL == EL0 and similar:
                if oNode.oLeft.isMatchingDotAtom('PSTATE', 'EL'):
                    idxEl = ('EL0', 'EL1', 'EL2', 'EL3').index(oNode.oRight.getIdentifierName());
                    if idxEl >= 0:
                        oNode.oLeft  = ArmAstCppExpr('IEM_F_MODE_ARM_GET_EL(pVCpu->iem.s.fExec)');
                        oNode.oRight = ArmAstInteger(idxEl);
                    return oNode;

            _ = fEliminationAllowed;
            _ = oInfo;
            return oNode;

        def transformCodePass2(self, oInfo, oCode):
            """ Code transformation pass 2: C++ translation. """
            return oCode.transform(self.transformCodePass2Callback, True, oInfo);

        def morphCodeToC(self, oInfo):
            """ Morphs the accessor code and assigns the result to self.oCode """
            assert oInfo.oCode is None;
            oInfo.oCode = self.transformCodePass2(oInfo, self.transformCodePass1(oInfo, oInfo.oAccessor.oAccess.clone()));
            return True;


    class SysRegGeneratorA64Mrs(SysRegGeneratorBase):
        def __init__(self):
            IEMArmGenerator.SysRegGeneratorBase.__init__(self, 'A64.MRS');

        def generateOneHandler(self, oInfo):
            """ Generates one register access for A64.MRS. """
            asLines = [
                '',
                '/**',
                ' * %s - %s' % (oInfo.oAccessor.oEncoding.sAsmValue, oInfo.oAccessor.oEncoding.dNamedValues,),
                ' */',
                'static VBOXSTRICTRC iemCImplA64_mrs_%s(PVMCPU pVCpu, uint64_t *puDst)' % (oInfo.oAccessor.oEncoding.sAsmValue,),
                '{',
            ];
            if oInfo.cCallsToIsFeatureImplemented > 0:
                asLines.append('    const CPUMFEATURESARMV8 * const pGstFeats = IEM_GET_GUEST_CPU_FEATURES(pVCpu);');
            if oInfo.oCode:
                asLines.extend(oInfo.oCode.toStringList('    ', 'C'));

            asLines.append('    /* -------- Original code specification: -------- */');
            asLines.extend(oInfo.oAccessor.oAccess.toStringList('    // '));
            asLines += [
                '    return VERR_IEM_ASPECT_NOT_IMPLEMENTED;',
                '}',
            ];
            return asLines;

        def generateMainFunction(self):
            """ Generates the CIMPL function for A64.MRS. """
            asLines = [
                '',
                '/**',
                ' * Implements the MRS instruction.',
                ' *',
                ' * @returns Strict VBox status code.',
                ' * @param   pVCpu       The cross context virtual CPU structure of the',
                ' *                      calling thread.',
                ' * @param   idSysReg    The system register to read from (IPRT format).',
                ' * @param   puDst       Where to return the value.',
                ' * @todo    Place this in CPUM and return status codes rather than raising exceptions.',
                ' */',
                'DECLHIDDEN(VBOXSTRICT) iemCImplA64_mrs_generic(PVMCPU pVCpu, uint32_t idSysReg, uint64_t *puDst)',
                '{',
                '    uint64_t         uZeroDummy;',
                '    uint64_t * const puDst = idGprDst < ARMV8_A64_REG_XZR',
                '                           ? &pVCpu->cpum.GstCtx.aGRegs[idGprDst].x : &uZeroDummy;',
                '    switch (idSysReg)',
                '    {',
            ];
            for oInfo in self.aoInfo:
                if oInfo.sEnc[0] == 'A':
                    asLines.append('        case %s: %sreturn iemCImplA64_mrs_%s(pVCpu, puDst);'
                                   % (oInfo.sEnc, ' ' * (45 - len(oInfo.sEnc)), oInfo.oAccessor.oEncoding.sAsmValue));
            asLines += [
                '    }',
                '    /* Fall back on handcoded handler. */',
                '    return iemCImplA64_mrs_fallback(pVCpu, idGprDst, idSysReg);',
                '}',
                '',
                '/**',
                ' * Implements the MRS instruction.',
                ' *',
                ' * @param   idSysReg    The system register to read from.',
                ' * @param   idGprDst    The destination GPR register number (IPRT format).',
                ' */',
                'IEM_CIMPL_DEF_2(iemCImplA64_mrs, uint32_t, idSysReg, uint8_t, idGprDst)',
                '{',
                '    uint64_t         uZeroDummy;',
                '    uint64_t * const puDst = idGprDst < ARMV8_A64_REG_XZR',
                '                           ? &pVCpu->cpum.GstCtx.aGRegs[idGprDst].x : &uZeroDummy;',
                '    return iemCImplA64_mrs_generic(pVCpu, idSysReg, puDst);',
                '}',
            ];
            return asLines;


        def transformCodePass2Callback(self, oNode, fEliminationAllowed, oInfo):
            """ Callback used by the second pass."""
            if oNode.isMatchingSquareOp('X', 't', 64):
                return ArmAstCppExpr('*puDst', 64);

            return super().transformCodePass2Callback(oNode, fEliminationAllowed, oInfo);


    def generateCImplSysRegCpp(self, sInstrSet, sState):
        """ Worker for generateA64CImplSysRegCpp. """
        _ = sInstrSet;

        #
        # Gather the relevant system register access code.
        #
        dAccessors = {
            'A64.MRS':          IEMArmGenerator.SysRegGeneratorA64Mrs(),
            'A64.MSRregister':  IEMArmGenerator.SysRegGeneratorBase('A64.MSRregister'),
            'A64.MSRimmediate': IEMArmGenerator.SysRegGeneratorBase('A64.MSRimmediate'),
            'A64.MRRS':         IEMArmGenerator.SysRegGeneratorBase('A64.MRRS'),
            'A64.MSRRregister': IEMArmGenerator.SysRegGeneratorBase('A64.MSRRregister'),
        } # type: Dict[str,SysRegGeneratorBase]

        for oReg in g_daoAllArmRegistersByState[sState]: # type: ArmRegister
            assert oReg.sState == sState;
            for oAccessor in oReg.aoAccessors:  # type: ArmAccessorBase
                if isinstance(oAccessor, ArmAccessorSystem) and oAccessor.sName in dAccessors:
                    sEncSortKey = 'encoding=%s' % (oAccessor.oEncoding.dNamedValues,);
                    if (    not isinstance(oAccessor, ArmAccessorSystemArray)
                        and not oAccessor.oEncoding.fHasWildcard
                        and not oAccessor.oEncoding.fHasIndex
                        and oAccessor.sName in ('A64.MRS', 'A64.MSRregister')):
                        sEncSortKey = oAccessor.oEncoding.getSysRegIdCreate();
                    dAccessors[oAccessor.sName].aoInfo.append(self.SysRegAccessorInfo(oAccessor, oReg, sEncSortKey));

        for sKey, oGenerator in dAccessors.items():
            assert sKey == oGenerator.sInstr;
            oGenerator.aoInfo.sort(key = operator.attrgetter('sEnc'));

        #
        # File header.
        #
        asLines = self.generateLicenseHeader(g_oArmRegistersVerInfo);
        asLines += [
            '#define LOG_GROUP LOG_GROUP_IEM',
            '#define VMCPU_INCL_CPUM_GST_CTX',
            '#include "IEMInternal.h"',
            '#include <VBox/vmm/vm.h>',
            '#include "VBox/err.h"',
            '',
            '#include "iprt/armv8.h"',
            '',
            '#include "IEMMc.h"',
            '#include "IEMInline-armv8.h"',
            '',
            '',
        ];

        #
        # Generate the real code, accessor type by accessor type.
        #
        for oGenerator in dAccessors.values():
            asLines += [
                '',
                '',
                '',
                '/*',
                ' * %s - %u registers' % (oGenerator.sInstr, len(oGenerator.aoInfo),),
                ' */',
            ];

            # Individual handler functions.
            for oInfo in oGenerator.aoInfo:
                asLines.append('');
                if oInfo.sEnc[0] == 'A':
                    oGenerator.morphCodeToC(oInfo);
                    asLines += oGenerator.generateOneHandler(oInfo);
                else:
                    asLines += [
                        '// %s' % (oInfo.oAccessor.oEncoding.sAsmValue,),
                        '// %s' % (oInfo.sEnc,),
                    ];
                    if oInfo.oAccessor.oAccess:
                        asLines.extend(oInfo.oAccessor.oAccess.toStringList('//    '));
                    else:
                        asLines.append('// access is None!');

            # Main switch function.
            asLines += oGenerator.generateMainFunction();

        return (True, asLines);

    def generateA64CImplSysRegCpp(self, sFilename, iPartNo):
        """ Generates the IEMAllCImplA64SysReg-armv8.cpp file. """
        _ = sFilename; _ = iPartNo;
        return self.generateCImplSysRegCpp('A64', 'AArch64');


    #
    # Features
    #

    def generateFeaturesHdr(self, sFilename, iPartNo):
        _ = iPartNo;

        asLines = self.generateLicenseHeader(g_oArmFeaturesVerInfo);
        sBlockerName = re.sub('[.-]', '_', os.path.basename(sFilename));
        asLines += [
            '#ifndef VMM_INCLUDED_SRC_VMMR3_target_armv8_%s' % (sBlockerName,),
            '#define VMM_INCLUDED_SRC_VMMR3_target_armv8_%s' % (sBlockerName,),
            '#ifndef RT_WITHOUT_PRAGMA_ONCE',
            '# pragma once',
            '#endif',
            '',
        ];

        #
        # Gather the features we're listing and sort it so that we start with
        # expressions only requiring a single system register value and move
        # on to more complicated ones, ending with those including other
        # FEAT_XXXX references (relies on sorting order).
        #
        aoFeatures = [oFeature for oFeature in g_aoAllArmFeatures
                      if    oFeature.sName.startswith('FEAT_')
                        and oFeature.sName not in ('FEAT_EL0', 'FEAT_EL1', 'FEAT_EL2', 'FEAT_EL3') # removed in 2025
                        and oFeature.oSupportExpr
                        and oFeature.asSupportExprVars[0].split('.')[0] not in ('AArch32', 'PMU', 'AMU', 'ext', 'uext') ];
        aoFeatures = sorted(aoFeatures, key = lambda oFeature: (len(oFeature.asSupportExprVars), oFeature.asSupportExprVars));


        class CExprHelperFeatures(object):
            """ Helper class for creating C-expressions from the is-supported AST expressions. """
            def __init__(self):
                self.dVars        = {}      # Type: Dict[str, ArmRegister]
                self.idxFeatures  = 0;

            def _lookupSysRegValueExpr(self, oReg):
                return 'cpumCpuIdLookupSysReg(paSysRegs, cSysRegs, %s)' % (oReg.getVBoxConstant());

            def getFieldInfo(self, sName, sVar = '', sNamespace = ''):
                if sNamespace and sVar:
                    sCVarNm = 'u%s_%s' % (sNamespace, sVar)
                    oReg    = self.dVars.get(sCVarNm) # Type: ArmRegister
                    if not oReg:
                        oReg = g_ddoAllArmRegistersByStateByName[sNamespace][sVar];
                        self.dVars[sCVarNm] = oReg;
                        # Warning! We're appending lines the output here!
                        asLines.append('');
                        asLines.append('    uint64_t const %s = %s;' % (sCVarNm, self._lookupSysRegValueExpr(oReg),));

                    aoFields = oReg.daoFields.get(sName);
                    if not aoFields:
                        raise Exception('Field %s was not found in register %s.%s (known fields: %s)'
                                        % (sName, sNamespace, sVar, oReg.daoFields.keys(),));
                    if len(aoFields) > 1:
                        raise Exception('Ambigious field %s was in register %s.%s: %s' % (sName, sNamespace, sVar, aoFields,));
                    oField = aoFields[0]    # Type: ArmFieldsBase

                    if len(oField.aoRanges) != 1:
                        raise Exception('TODO: Using complicated field %s in register %s.%s: %s'
                                        % (sName, sNamespace, sVar, oField.aoRanges,));
                    iFirstBit  = oField.aoRanges[0].iFirstBit;
                    cBitsWidth = oField.aoRanges[0].cBitsWidth;
                    if oField.oParent and isinstance(oField.oParent, ArmFieldsConditionalField): # Relative to parent range.
                        if len(oField.oParent.aoRanges) != 1:
                            raise Exception('TODO: Using complicated conditional field %s in register %s.%s: %s'
                                            % (sName, sNamespace, sVar, oField.oParent.aoRanges,));
                        iFirstBit += oField.oParent.aoRanges[0].iFirstBit;
                        assert cBitsWidth <= oField.oParent.aoRanges[0].cBitsWidth;

                    return ('((%s >> %2u) & %#x /*%s*/)' % (sCVarNm, iFirstBit, (1 << cBitsWidth) - 1, sName,) , cBitsWidth);

                # We can deal with feature references, provided they're already initialized.
                # The sorting should take care of this, but we check. :)
                if sName.startswith('FEAT_'):
                    for i in range(self.idxFeatures):
                        if aoFeatures[i].sName == sName:
                            return ('pFeatures->%s' % (g_dSpecFeatToCpumFeat.get(sName, sName),), 1);
                    raise Exception('Internal error: Feature %s has not yet been initialized! (%s)'
                                    % (sName, ', '.join([aoFeatures[i].sName for i in range(self.idxFeatures)]),));

                # The v8Ap4 style stuff we just set to false for now.
                if sName in ('v8Ap4',):
                    return ('false', 1);

                ## @todo Need to load system registers.
                raise Exception('Field %s was not found ' % (sName,));

            def convertFunctionCall(self, oCall):
                if oCall.sName in ('UInt', 'SInt'):
                    if len(oCall.aoArgs) != 1:
                        raise Exception('Unexpected argument count for UInt call: %s' % (oCall.aoArgs,));
                    oArg = oCall.aoArgs[0] # ArmAstBase
                    if oCall.sName == 'UInt':
                        return '(uint32_t)(%s)' % (oArg.toCExpr(self),);
                    # Sign-extending a bit field requires us to know its size...
                    cBitsWidth = oArg.getWidth(self);
                    if cBitsWidth <= 0 or cBitsWidth >= 32:
                        raise Exception('Unexpected getWidth result for SInt argument: %s (%s)' % (cBitsWidth, oArg.toString()));
                    return '((int32_t)((uint32_t)(%s) << %u) >> %u)' % (oArg.toCExpr(self), 32 - cBitsWidth, 32 - cBitsWidth);

                raise Exception('Call to unsupported function: %s (%s)' % (oCall.sName, oCall.aoArgs,));

        asLines += [
            '',
            '',
            '/**',
            ' * Explodes ARMv8+ features from an array of system register values.',
            ' *',
            ' * @returns VBox status code',
            ' * @param   paSysRegs   The system registers and their values.',
            ' * @param   cSysRegs    Number of system register values.',
            ' * @param   pFeatures   The structure to explode the features into.',
            ' */',
            'VMMDECL(int) CPUMCpuIdExplodeFeaturesArmV8(PCSUPARMSYSREGVAL paSysRegs, uint32_t cSysRegs,',
            '                                           CPUMFEATURESARMV8 *pFeatures)',
            '{',
            '    RT_ZERO(*pFeatures);',
            '',
        ];
        asTodo  = []
        oHelper = CExprHelperFeatures();
        for oHelper.idxFeatures, oFeature in enumerate(aoFeatures):
            sFeatureMemberNm = g_dSpecFeatToCpumFeat.get(oFeature.sName, oFeature.sName);
            if sFeatureMemberNm is oFeature.sName:
                asTodo.append(oFeature.sName);

            print('debug: %s/%s <-> %s' % (oFeature.sName, sFeatureMemberNm, oFeature.oSupportExpr.toString()))
            sLine = '    pFeatures->%-22s = %s;' % (sFeatureMemberNm, oFeature.oSupportExpr.toCExpr(oHelper),)
            sLine = '%-116s /* %s */' % (sLine, oFeature.sName,);
            asLines.append(sLine);
        if asTodo:
            print('Error! Please add the features: %s' % (', '.join(asTodo),));
            for sFeature in asTodo:
                print('Error!   %-25s: %s -- %s'
                      % (sFeature, g_dAllArmFeaturesByName[sFeature].asSupportExprVars,
                         ';  '.join(['#%u: %s' % (iExpr, oExpr.toString())
                                              for iExpr, oExpr in enumerate(g_dAllArmFeaturesByName[sFeature].aoConstraints)]),));

        # Did we miss any features in CPUMFEATURESARMV8?
        asMissing = [];
        hsMissingFeatures = set(g_dSpecFeatToCpumFeat) - set({oFeature.sName for oFeature in aoFeatures});
        if hsMissingFeatures:
            print('Error! The following CPUMFEATURESARMV8 members have not been initialized: %s'
                  % (', '.join(hsMissingFeatures),));
            asMissing += [
                '',
                '    /* Initializing "missing" members: */',
            ];
            for sFeature in sorted(hsMissingFeatures):
                oFeature = g_dAllArmFeaturesByName.get(sFeature)
                if oFeature:
                    sExtra = ' (%s)' % ('; '.join(['#%u: %s' % (iExpr, oExpr.toString())
                                                   for iExpr, oExpr in enumerate(oFeature.aoConstraints)]),);
                else:
                    sExtra = '';
                asMissing.append('    pFeatures->%-22s = 0; /* %s%s */' % (g_dSpecFeatToCpumFeat[sFeature], sFeature, sExtra));

        asLines += asMissing;

        asLines += [
            '',
            '    return cpumCpuIdExplodeFeaturesArmV8Handcoded(paSysRegs, cSysRegs, pFeatures);',
            '}'
        ];

        #
        # Write a dumper function for the CPUMFEATURESARMV8 structure.
        #
        cchMaxFeatNm = max(len(sKey) for sKey in g_dSpecFeatToCpumFeat);
        asLines += [
            '',
            '#ifdef IN_RING3',
            '',
            '# include <VBox/vmm/dbgf.h> /* DBGFINFOHLP */',
            '',
            '/**',
            ' * Prints the ARMv8+ features in @a pFeatures.',
            ' *',
            ' * @param   pHlp            The output callback helper.',
            ' * @param   cchOutput       The width of the output (for multiple columns).',
            ' *                          Zero is taken to mean a single column.',
            ' * @param   pFeatures       The features to dump',
            ' * @param   pszLabel        The label.',
            ' * @param   pSecondary      Optional secondary feature set to compare with.',
            ' * @param   pszSecondary    The label for the secondary feature set.',
            ' */',
            'VMMR3DECL(void) CPUMR3CpuIdPrintArmV8Features(PCDBGFINFOHLP pHlp, uint32_t cchOutput,',
            '                                              CPUMFEATURESARMV8 const *pFeatures, const char *pszLabel,',
            '                                              CPUMFEATURESARMV8 const *pSecondary, const char *pszSecondary)',
            '{',
            '    unsigned const cchLabel  = (unsigned)strlen(pszLabel);',
            '    unsigned const cchLabel2 = pszSecondary ? (unsigned)strlen(pszSecondary) : 0;',
            '    unsigned const cchPad    = cchLabel - 1 + cchLabel2 - !!cchLabel2;',
            '    unsigned const cchColumn = 2 + %s + 3 + cchLabel + (pSecondary ? 3 + cchLabel2 : 0);' % (cchMaxFeatNm,),
            '    unsigned const cColumns  = cchOutput < cchColumn ? 1 : cchOutput / cchColumn;',
            '    if (pSecondary)',
            '        for (unsigned iColumn = 0; iColumn < cColumns; iColumn++)',
            '            pHlp->pfnPrintf(pHlp, "  %%%us = %%s (%%s)", "Features", pszLabel, pszSecondary);' % (cchMaxFeatNm,),
            '    else',
            '        for (unsigned iColumn = 0; iColumn < cColumns; iColumn++)',
            '            pHlp->pfnPrintf(pHlp, "  %%%us = %%s", "Features", pszLabel);' % (cchMaxFeatNm,),
            '    pHlp->pfnPrintf(pHlp, "\\n");',
            '',
            '    unsigned iColumn = 0;',
            '#define PRINT_FEATURE(a_Name, a_Member) do { \\',
            '            if (pSecondary) \\',
            '                pHlp->pfnPrintf(pHlp, "%%*s  %%%us = %%u (%%u)", \\' % (cchMaxFeatNm,),
            '                                iColumn ? cchPad : 0, "", #a_Name, pFeatures->a_Member, pSecondary->a_Member); \\',
            '            else \\',
            '                pHlp->pfnPrintf(pHlp, "%%*s  %%%us = %%u", \\' % (cchMaxFeatNm,),
            '                                iColumn ? cchPad : 0, "", #a_Name, pFeatures->a_Member); \\',
            '            iColumn += 1; \\',
            '            if (iColumn >= cColumns) \\',
            '            { \\',
            '                pHlp->pfnPrintf(pHlp, "\\n"); \\',
            '                iColumn = 0; \\',
            '            } \\',
            '        } while (0)',
        ];

        asLines.extend(['    PRINT_FEATURE(%-*s %s);' % (cchMaxFeatNm + 1, sFeature + ',', g_dSpecFeatToCpumFeat[sFeature])
                        for sFeature in sorted(g_dSpecFeatToCpumFeat.keys())]);
        asLines += [
            '}',
            '#endif /* IN_RING3 */',
        ];


        asLines += [
            '',
            '#endif /* !VMM_INCLUDED_SRC_VMMR3_target_armv8_%s */' % (sBlockerName,),
            '',
        ];
        return (not(asTodo), asLines);


    def main(self, asArgs):
        """ Main function. """

        #
        # Parse arguments.
        #
        class MyArgParser(argparse.ArgumentParser):
            def convert_arg_line_to_args(self, arg_line):
                return arg_line.split();

        oArgParser = MyArgParser(fromfile_prefix_chars = '@',
                                 formatter_class = argparse.RawDescriptionHelpFormatter,
                                 epilog = '''
Hints can be extracted from existing code with the following sed command:
    kmk_sed -e "/--decoder-hint/!d" -e "s/^...//" IEMAllIntprA64Tables-armv8.cpp > hints.rsp
Then add @hints.rsp to the command line to make use of them.''');
        oArgParser.add_argument('--tar',
                                metavar = 'AARCHMRS_BSD_A_profile-2024-12.tar.gz',
                                dest    = 'sTarFile',
                                action  = 'store',
                                default = None,
                                help    = 'Specification TAR file to get the files from.');
        oArgParser.add_argument('--instructions',
                                metavar = 'Instructions.json',
                                dest    = 'sFileInstructions',
                                action  = 'store',
                                default = 'Instructions.json',
                                help    = 'The path to the instruction specficiation file.');
        oArgParser.add_argument('--features',
                                metavar = 'Features.json',
                                dest    = 'sFileFeatures',
                                action  = 'store',
                                default = 'Features.json',
                                help    = 'The path to the features specficiation file.');
        oArgParser.add_argument('--registers',
                                metavar = 'Registers.json',
                                dest    = 'sFileRegisters',
                                action  = 'store',
                                default = 'Registers.json',
                                help    = 'The path to the registers specficiation file.');
        oArgParser.add_argument('--spec-dir',
                                metavar = 'dir',
                                dest    = 'sSpecDir',
                                action  = 'store',
                                default = '',
                                help    = 'Specification directory to prefix the specficiation files with.');
        oArgParser.add_argument('--out-decoder-cpp',
                                metavar = 'file-decoder.cpp',
                                dest    = 'sFileDecoderCpp',
                                action  = 'store',
                                default = None,
                                help    = 'The output C++ file for the decoder.');
        oArgParser.add_argument('--out-decoder-hdr',
                                metavar = 'file-decoder.h',
                                dest    = 'sFileDecoderHdr',
                                action  = 'store',
                                default = None,
                                help    = 'The output header file for the decoder.');
        oArgParser.add_argument('--out-stub-hdr',
                                metavar = 'file-stub.h',
                                dest    = 'sFileStubHdr',
                                action  = 'store',
                                default = None,
                                help    = 'The output header file for the implementation stubs.');
        oArgParser.add_argument('--out-cimpl-sysreg-cpp',
                                metavar = 'file-cimpl-sysreg.cpp',
                                dest    = 'sFileCImplSysRegCpp',
                                action  = 'store',
                                default = None,
                                help    = 'The output C++ file for system register handling (MRS, MSR, ++).');
        oArgParser.add_argument('--out-features-hdr',
                                metavar = 'file-features.h',
                                dest    = 'sFileFeaturesHdr',
                                action  = 'store',
                                default = None,
                                help    = 'The output header file for the feature extraction & dumping code.');

        # debug:
        oArgParser.add_argument('--print-instructions',
                                dest    = 'fPrintInstructions',
                                action  = 'store_true',
                                default = False,
                                help    = 'List the instructions after loading.');
        oArgParser.add_argument('--print-instructions-with-conditions',
                                dest    = 'fPrintInstructionsWithConditions',
                                action  = 'store_true',
                                default = False,
                                help    = 'List the instructions and conditions after loading.');
        oArgParser.add_argument('--print-instructions-with-encoding',
                                dest    = 'fPrintInstructionsWithEncoding',
                                action  = 'store_true',
                                default = False,
                                help    = 'List the instructions and encoding details after loading.');
        oArgParser.add_argument('--print-fixed-mask-stats',
                                dest    = 'fPrintFixedMaskStats',
                                action  = 'store_true',
                                default = False,
                                help    = 'List statistics on fixed bit masks.');
        oArgParser.add_argument('--print-fixed-mask-top-10',
                                dest    = 'fPrintFixedMaskTop10',
                                action  = 'store_true',
                                default = False,
                                help    = 'List the 10 top fixed bit masks.');
        oArgParser.add_argument('--print-sysregs',
                                dest    = 'fPrintSysRegs',
                                action  = 'store_true',
                                default = False,
                                help    = 'List system registers after loading.');
        # Hacks
        oArgParser.add_argument('--decoder-hint0',
                                metavar = 'mask-to-use',
                                dest    = 'asDecoderHintsL0',
                                action  = 'append',
                                default = [],
                                help    = 'Top level decoder hints used to shorten the runtime. Format: <mask-to-use>');
        oArgParser.add_argument('--decoder-hint1',
                                metavar = 'matched-mask/value/mask-to-use',
                                dest    = 'asDecoderHintsL1',
                                action  = 'append',
                                default = [],
                                help    = 'Level 1 decoder hints. Format: <matched-mask>/<matched-value/<mask-to-use>');
        oArgParser.add_argument('--decoder-hint1-threshold',
                                metavar = 'count',
                                dest    = 'iDecoderL1Threshold',
                                type    = int,
                                action  = 'store',
                                default = 20,
                                help    = 'The instruction threshold count for emitting --decoder-hint1 in the generated code.');

        # Do it!
        oOptions = oArgParser.parse_args(asArgs[1:]);
        self.oOptions = oOptions;

        # Process the decoder hints.
        for sValue in oOptions.asDecoderHintsL0:
            try:
                fTmp = int(sValue, 16);
                if fTmp <= 0 or fTmp >= 0x100000000 or fTmp.bit_count() <= 4 or fTmp.bit_count() >= 16:# pylint: disable=no-member
                    raise Exception();
            except:
                print('syntax error: Invalid --decoder-hint0 value: %s' % (sValue,))
                return 2;
            g_dDecoderFilterDepth0[fTmp] = True;

        for sValue in oOptions.asDecoderHintsL1:
            try:
                afVals = [int(sSub, 16) for sSub in sValue.split('/')];
                for iVal, fTmp in enumerate(afVals):
                    if fTmp < 0 or fTmp >= 0x100000000 or (fTmp == 0 and iVal != 1): raise Exception();
                if (afVals[0] & afVals[1]) != afVals[1] or (afVals[0] & afVals[2]) != 0: raise Exception();
            except:
                print('syntax error: Invalid --decoder-hint1 value: %s' % (sValue,))
                return 2;
            sKey = '%x/%x' % (afVals[0], afVals[1]);
            if sKey not in g_ddDecoderFilterDepth1:
                g_ddDecoderFilterDepth1[sKey] = {afVals[2]: True,};
            else:
                g_ddDecoderFilterDepth1[sKey][afVals[2]] = True;

        #
        # Load the specification.
        #
        if LoadArmOpenSourceSpecification(oOptions):
            PrintSpecs(oOptions)

            #
            # Check if we're generating any output before constructing the decoder.
            #
            aaoOutputFiles = [
                 ( oOptions.sFileDecoderCpp,      self.generateA64DecoderCpp,            0, 1, ),
                 ( oOptions.sFileDecoderHdr,      self.generateDecoderHdr,               0, 1, ), # after generateA64DecoderCpp!
                 ( oOptions.sFileStubHdr,         self.generateA64ImplementationStubHdr, 0, 1, ),
                 ( oOptions.sFileCImplSysRegCpp,  self.generateA64CImplSysRegCpp,        0, 0, ),
                 ( oOptions.sFileFeaturesHdr,     self.generateFeaturesHdr,              0, 0, ),
            ];

            cOutputFiles        = 0;
            cDecoderOutputFiles = 0;
            for sOutFile, _, _, fDecoder in aaoOutputFiles:
                cOutputFiles        += sOutFile is not None;
                cDecoderOutputFiles += sOutFile is not None and fDecoder;

            fRc = True;
            if cOutputFiles > 0:
                #
                # Sort out the decoding if needed.
                #
                if cDecoderOutputFiles > 0:
                    self.constructDecoder();

                #
                # Output.
                #
                for sOutFile, fnGenMethod, iPartNo, _ in aaoOutputFiles:
                    if not sOutFile:
                        continue;
                    (fRc2, asLines) = fnGenMethod(sOutFile, iPartNo);
                    fRc = fRc2 and fRc;

                    if sOutFile == '-':
                        sys.stdout.write('\n'.join(asLines));
                    else:
                        try:
                            oOut = open(sOutFile, 'w', encoding = 'utf-8', errors = 'strict');
                        except Exception as oXcpt:
                            print('error! Failed open "%s" for writing: %s' % (sOutFile, oXcpt,), file = sys.stderr);
                            return 1;
                        with oOut:
                            oOut.write('\n'.join(asLines));
            if fRc:
                return 0;

        return 1;

def printException(oXcpt):
    print('----- Exception Caught! -----', flush = True);
    cMaxLines = 1;
    try:    cchMaxLen = os.get_terminal_size()[0] * cMaxLines;
    except: cchMaxLen = 80 * cMaxLines;
    cchMaxLen -= len('     =  ...');

    oTB = traceback.TracebackException.from_exception(oXcpt, limit = None, capture_locals = True);
    # No locals for the outer frame.
    oTB.stack[0].locals = {};
    # Suppress insanely long variable values.
    for oFrameSummary in oTB.stack:
        if oFrameSummary.locals:
            #for sToDelete in ['ddAsmRules', 'aoInstructions',]:
            #    if sToDelete in oFrameSummary.locals:
            #        del oFrameSummary.locals[sToDelete];
            for sKey, sValue in oFrameSummary.locals.items():
                if len(sValue) > cchMaxLen - len(sKey):
                    sValue = sValue[:cchMaxLen - len(sKey)] + ' ...';
                if '\n' in sValue:
                    sValue = sValue.split('\n')[0] + ' ...';
                oFrameSummary.locals[sKey] = sValue;
    idxFrame = 0;
    asFormatted = [];
    oReFirstFrameLine = re.compile(r'^  File ".*", line \d+, in ')
    for sLine in oTB.format():
        if oReFirstFrameLine.match(sLine):
            idxFrame += 1;
        asFormatted.append(sLine);
    for sLine in asFormatted:
        if oReFirstFrameLine.match(sLine):
            idxFrame -= 1;
            sLine = '#%u %s' % (idxFrame, sLine.lstrip());
        print(sLine);
    print('----', flush = True);


if __name__ == '__main__':
    fProfileIt = 'VBOX_PROFILE_PYTHON' in os.environ;
    oProfiler = cProfile.Profile() if fProfileIt else None;
    try:
        if not oProfiler:
            rcExit = IEMArmGenerator().main(sys.argv);
        else:
            rcExit = oProfiler.runcall(IEMArmGenerator().main, sys.argv);
    except Exception as oXcptOuter:
        printException(oXcptOuter);
        rcExit = 2;
    except KeyboardInterrupt as oXcptOuter:
        printException(oXcptOuter);
        rcExit = 2;
    if oProfiler:
        sProfileSort = 'tottime'; iSortColumn = 1;
        #sProfileSort = 'cumtime'; iSortColumn = 3;
        if not oProfiler:
            oProfiler.print_stats(sort=sProfileSort);
        else:
            oStringStream = io.StringIO();
            pstats.Stats(oProfiler, stream = oStringStream).strip_dirs().sort_stats(sProfileSort).print_stats(64);
            for iStatLine, sStatLine in enumerate(oStringStream.getvalue().split('\n')):
                if iStatLine > 20:
                    asStatWords = sStatLine.split();
                    if (    len(asStatWords) > iSortColumn
                        and asStatWords[iSortColumn] in { '0.000', '0.001', '0.002', '0.003', '0.004', '0.005' }):
                        break;
                print(sStatLine);
    sys.exit(rcExit);

