/* $Id: Svga.h $ */
/** @file
 * VirtualBox Windows Guest Mesa3D - Gallium driver VMSVGA.
 */

/*
 * Copyright (C) 2016-2025 Oracle and/or its affiliates.
 *
 * This file is part of VirtualBox base platform packages, as
 * available from https://www.virtualbox.org.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation, in version 3 of the
 * License.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <https://www.gnu.org/licenses>.
 *
 * SPDX-License-Identifier: GPL-3.0-only
 */

#ifndef GA_INCLUDED_SRC_WINNT_Graphics_Video_mp_wddm_gallium_Svga_h
#define GA_INCLUDED_SRC_WINNT_Graphics_Video_mp_wddm_gallium_Svga_h
#ifndef RT_WITHOUT_PRAGMA_ONCE
# pragma once
#endif

#include "VBoxMPGaUtils.h"

#include <iprt/asm.h>
#include <iprt/assert.h>
#include <iprt/avl.h>
#include <iprt/list.h>

#include <VBoxGaTypes.h>
#include <VBoxGaHWSVGA.h>

 /* VMSVGA headers. */
#pragma pack(1) /* VMSVGA structures are '__packed'. */
#include <svga_reg.h>
#include <svga3d_reg.h>
#include <vbsvga3d_dx.h>
#pragma pack()

#define SVGA_SYNC_GENERIC         1
#define SVGA_SYNC_FIFOFULL        2

/* Make sure that correct svga3d_reg.h was included. */
AssertCompileSize(SVGA3dCmdHeader, 8);

typedef struct VMSVGAFIFO
{
    /** SVGA_FIFO_CAPABILITIES */
    uint32_t u32FifoCaps;

    /** How many bytes were reserved for a command. */
    uint32_t cbReserved;

    /** Temporary buffer for commands crossing the FIFO ring boundary. */
    void *pvBuffer;

    /** FIFO access. */
    FAST_MUTEX FifoMutex;

} VMSVGAFIFO;
typedef VMSVGAFIFO *PVMSVGAFIFO;

/*
 * Command buffers.
 */
typedef struct VMSVGACBPAGE
{
    RTR0MEMOBJ           hMemObjPages;
    RTR0MEMOBJ           hMemObjMapping;
    RTR0PTR              pvR0;
    RTHCPHYS             PhysAddr;
} VMSVGACBPAGE, *PVMSVGACBPAGE;

typedef int32_t VMSVGACBHEADERHANDLE;
#define VMSVGACBHEADER_NIL (-1)

struct VBOXWDDM_EXT_VMSVGA;
typedef DECLCALLBACKTYPE(void, FNCBCOMPLETION, (struct VBOXWDDM_EXT_VMSVGA *pSvga, void *pvData, uint32_t cbData));
typedef FNCBCOMPLETION *PFNCBCOMPLETION;

typedef struct VMSVGACBCOMPLETION
{
    RTLISTNODE                  nodeCompletion;             /* VMSVGACB::listCompletion */
    PFNCBCOMPLETION             pfn;                        /* Function to call. */
    uint32_t                    cb;                         /* Size of data in bytes. */
    /* cb bytes follow. */
} VMSVGACBCOMPLETION, *PVMSVGACBCOMPLETION;

typedef enum VMSVGACBTYPE
{
    VMSVGACB_INVALID = 0,
    VMSVGACB_CONTEXT_DEVICE = 1,                   /* SVGA_CB_CONTEXT_DEVICE commands. */
    VMSVGACB_MINIPORT = 2,                         /* Commands generated by the miniport. */
    VMSVGACB_UMD = 3                               /* WDDM DMA buffer. */
} VMSVGACBTYPE;

/* Information about a command buffer. */
typedef struct VMSVGACB
{
    RTLISTNODE           nodeQueue;                /* For a queue where the buffer is currently resides. */
    VMSVGACBTYPE         enmType;                  /* Type of the buffer. */
    uint32_t             idDXContext;              /* DX context of the buffer or SVGA3D_INVALID_ID. */
    uint32_t             cbBuffer;                 /* Total size. */
    uint32_t             cbCommand;                /* Size of commands. */
    uint32_t             cbReservedCmdHeader;      /* Reserved for the command header. */
    uint32_t             cbReservedCmd;            /* Reserved for the current command without the header. */
    uint32_t             u32ReservedCmd;           /* The current command. */
    VMSVGACBHEADERHANDLE hHeader;                  /* Handle of the header (index in the header pool array). */
    SVGACBHeader        *pCBHeader;                /* Pointer to the header. */
    PHYSICAL_ADDRESS     CBHeaderPhysAddr;
    union                                          /* Command data. */
    {
        VMSVGACBPAGE     page;                     /* VMSVGACB_CONTEXT_DEVICE and VMSVGACB_MINIPORT */
        PHYSICAL_ADDRESS DmaBufferPhysicalAddress; /* VMSVGACB_UMD */
    } commands;
    RTLISTANCHOR         listCompletion;           /* VMSVGACBCOMPLETION to be called on completion. */
#ifdef DEBUG
    bool fSubmitted : 1;
#endif
} VMSVGACB, *PVMSVGACB;

/* Buffer headers are allocated for submitted buffers, which sometimes are not immediately passed to the host.
 * Therefore the pool must be a bit larger than SVGA_CB_MAX_QUEUED_PER_CONTEXT * SVGA_CB_CONTEXT_MAX
 */
#define VMSVGA_CB_HEADER_POOL_NUM_PAGES         2
#define VMSVGA_CB_HEADER_POOL_HANDLES_PER_PAGE (PAGE_SIZE / sizeof(SVGACBHeader))
#define VMSVGA_CB_HEADER_POOL_NUM_HANDLES      (VMSVGA_CB_HEADER_POOL_NUM_PAGES * VMSVGA_CB_HEADER_POOL_HANDLES_PER_PAGE)

typedef struct VMSVGACBHEADERPOOL
{
    KSPIN_LOCK           SpinLock;
    VMSVGACBPAGE         aHeaderPoolPages[VMSVGA_CB_HEADER_POOL_NUM_PAGES];
    uint32_t             au32HeaderBits[VMSVGA_CB_HEADER_POOL_NUM_HANDLES / 32];
} VMSVGACBHEADERPOOL, *PVMSVGACBHEADERPOOL;

typedef struct VMSVGACBCONTEXT
{
    RTLISTANCHOR         QueuePending;             /* Buffers which will be submitted to the host. */
    RTLISTANCHOR         QueueSubmitted;           /* Buffers which are being processed by the host. */
    uint32_t             cSubmitted;               /* How many buffers were submitted to the host.
                                                    * Less than SVGA_CB_MAX_QUEUED_PER_CONTEXT */
} VMSVGACBCONTEXT, *PVMSVGACBCONTEXT;

typedef struct VMSVGACBSTATE
{
    VMSVGACBCONTEXT      aCBContexts[SVGA_CB_CONTEXT_MAX]; /* Command buffer contexts. */
    VMSVGACBHEADERPOOL   HeaderPool;               /* Array of buffer headers. */
    PVMSVGACB            pCBCurrent;               /* Current buffer for miniport commands. */
    FAST_MUTEX           CBCurrentMutex;           /* Access pCBCurrent. */
    KSPIN_LOCK           SpinLock;                 /* Lock for aCBContexts. */
} VMSVGACBSTATE, *PVMSVGACBSTATE;

/* Guest Backed Object: a set of locked pages and a page table for the host to access. */
typedef struct VMSVGAGBO
{
    int32_t volatile            cRefs;                      /* Reference count */
    struct
    {
        uint32_t                fMdl : 1;
        uint32_t                reserved : 31;
    }                           flags;
    uint32_t                    cbGbo;                      /* Size of gbo in bytes. */
    uint32_t                    cPTPages;                   /* How many pages are required to hold PPN64 page table. */
    SVGAMobFormat               enmMobFormat;               /* Page table format. */
    union                                                   /* Backing memory. */
    {
        PMDL                    pMdl;
        RTR0MEMOBJ              hMemObj;
    };
    PPN64                       base;                       /* Page which contains the page table. */
    RTR0MEMOBJ                  hMemObjPT;                  /* Page table pages. */
} VMSVGAGBO, *PVMSVGAGBO;

/* Contexts + One shaders mob per context + surfaces. */
#define SVGA3D_MAX_MOBS (SVGA3D_MAX_CONTEXT_IDS + SVGA3D_MAX_CONTEXT_IDS + SVGA3D_MAX_SURFACE_IDS)

typedef struct VMSVGAOT
{
    PVMSVGAGBO                  pGbo;
    uint32_t                    cEntries;                   /* How many objects can be stored in the OTable. */
} VMSVGAOT, *PVMSVGAOT;

/* VMSVGA specific part of Gallium device extension. */
typedef struct VBOXWDDM_EXT_VMSVGA
{
    union
    {
        struct
        {
            /** First IO port. SVGA_*_PORT are relative to it. */
            RTIOPORT ioportBase;
            /** Pointer to FIFO MMIO region. */
            volatile uint32_t *pu32FIFO;
        };
        struct
        {
            /** MMIO base address. SVGA_REG_* are relative to it. */
            volatile uint32_t *pu32MMIO;
        };
    } hw;

    /** Whether MMIO (hw.pu32MMIO) must be used instead of port IO. */
    bool fMMIO;

    /**
     * Hardware capabilities.
     */
    uint32_t u32Caps;         /** SVGA_REG_CAPABILITIES */
    uint32_t u32VramSize;     /** SVGA_REG_VRAM_SIZE */
    uint32_t u32FifoSize;     /** SVGA_REG_MEM_SIZE */
    uint32_t u32MaxWidth;     /** SVGA_REG_MAX_WIDTH */
    uint32_t u32MaxHeight;    /** SVGA_REG_MAX_HEIGHT */
    uint32_t u32GmrMaxIds;    /** SVGA_REG_GMR_MAX_IDS */
    uint32_t u32GmrMaxPages;  /** SVGA_REG_GMRS_MAX_PAGES */
    uint32_t u32MemorySize;   /** SVGA_REG_MEMORY_SIZE */
    uint32_t u32MaxMobSize;   /** SVGA_REG_MOB_MAX_SIZE */
    uint32_t u32MaxTextureWidth;  /** SVGA3D_DEVCAP_MAX_TEXTURE_WIDTH */
    uint32_t u32MaxTextureHeight; /** SVGA3D_DEVCAP_MAX_TEXTURE_HEIGHT */

    uint32_t u32MaxTextureLevels; /** 1 + floor(log2(max(u32MaxTextureWidth, u32MaxTextureHeight))) */

    /** Fifo state. */
    VMSVGAFIFO fifo;

    /** Command buffers state. */
    PVMSVGACBSTATE pCBState;
    /** Whether the host has generated an IRQ for buffer completion or error. */
    bool volatile fCommandBufferIrq;

    /** For atomic hardware access. */
    KSPIN_LOCK HwSpinLock;

    /** Maintaining the host objects lists. */
    KSPIN_LOCK HostObjectsSpinLock;

    /** For ids allocation. */
    KSPIN_LOCK IdSpinLock;

    /** AVL tree for mapping GMR id to the corresponding structure. */
    AVLU32TREE GMRTree;

    /** AVL tree for mapping sids to the surface objects. */
    AVLU32TREE SurfaceTree;

    /** AVL tree for mapping mobid to the corresponding structure. */
    AVLU32TREE MobTree;

    /** List of host objects, which must be deleted at PASSIVE_LEVEL. */
    RTLISTANCHOR DeletedHostObjectsList;

    /** SVGA data access. */
    FAST_MUTEX SvgaMutex;

    /** MOB access: MobTree. */
    KSPIN_LOCK MobSpinLock;

    struct
    {
        uint32_t u32Offset;
        uint32_t u32BytesPerLine;
    } lastGMRFB;

    VMSVGAOT aOT[SVGA_OTABLE_DX_MAX];

    /* Used by miniport to communicate with the device. */
    PVMSVGAGBO pMiniportGbo;
    SVGAMobId mobidMiniport;
    struct VMSVGAMINIPORTMOB volatile *pMiniportMobData; /* Pointer to the miniport mob content. */

    uint64_t volatile u64MobFence;
    RTLISTANCHOR listMobDeferredDestruction; /* Mob to be deleted after. */
    int32_t volatile cQueuedWorkItems;

#ifdef DEBUG
    /* Statistics. */
    uint32_t volatile cAllocatedGbos;
    uint32_t volatile cAllocatedMobs;
    uint32_t volatile cAllocatedGmrs;
    uint32_t volatile cAllocatedGmrPages;
#endif

    /** Bitmap of used GMR ids. Bit 0 - GMR id 0, etc. */
    uint32_t *pu32GMRBits; /* Number of GMRs is controlled by the host (u32GmrMaxIds), so allocate the bitmap. */
    uint32_t cbGMRBits;    /* Bytes allocated for pu32GMRBits */

    /** Bitmap of used context ids. Bit 0 - context id 0, etc. */
    uint32_t au32ContextBits[(SVGA3D_MAX_CONTEXT_IDS + 31) / 32];

    /** Bitmap of used surface ids. Bit 0 - surface id 0, etc. */
    uint32_t au32SurfaceBits[(SVGA3D_MAX_SURFACE_IDS + 31) / 32];

    /** Bitmap of used DX context ids. Bit 0 - context id 0, etc. */
    uint32_t au32DXContextBits[(SVGA3D_MAX_CONTEXT_IDS + 31) / 32];

    /** Bitmap of used MOB ids. Bit 0 - context id 0, etc. */
    uint32_t au32MobBits[(SVGA3D_MAX_MOBS + 31) / 32];

    /** Bitmap of used screen target ids. Bit 0 - stid 0, etc. */
    uint32_t au32ScreenTargetBits[(64 + 31) / 32]; /// @todo VBOX_VIDEO_MAX_SCREENS
} VBOXWDDM_EXT_VMSVGA;
typedef struct VBOXWDDM_EXT_VMSVGA *PVBOXWDDM_EXT_VMSVGA;

typedef struct VMSVGAMINIPORTMOB
{
    uint64_t u64MobFence; /* Host writes SVGA3dCmdDXMobFence64::value here. */
} VMSVGAMINIPORTMOB;

typedef struct VMSVGACOT
{
    SVGAMobId               mobid;                      /* COTable mob. */
    uint32_t                cEntries;                   /* How many objects can be stored in the COTable. */
} VMSVGACOT, *PVMSVGACOT;

typedef struct VMSVGACONTEXT
{
    uint32_t                    u32Cid;                     /* SVGA context id of this context. */
    bool                        fDXContext : 1;             /* Whether this context is a DX context or VGPU9. */
    bool                        fDebugVerifyCommands : 1;
    VMSVGACOT                   aCOT[VBSVGA_NUM_COTABLES];  /* Context Object Tables. */
} VMSVGACONTEXT, *PVMSVGACONTEXT;

typedef struct SVGAHOSTOBJECT SVGAHOSTOBJECT;
typedef DECLCALLBACKTYPE(NTSTATUS, FNHostObjectDestroy,(SVGAHOSTOBJECT *pThis));
typedef FNHostObjectDestroy *PFNHostObjectDestroy;

#define SVGA_HOST_OBJECT_UNDEFINED 0
#define SVGA_HOST_OBJECT_SURFACE   1

/* Information about an allocated surface. */
typedef struct SVGAHOSTOBJECT
{
    union {
        /* core.Key is the object id: surface id (sid), or (curently not required) context id (cid), gmr id (gid). */
        struct
        {
            AVLU32NODECORE core;
        } avl;

        struct {
            RTLISTNODE node;
            uint32_t u32Key;
        } list;
    } u;

    /* By UM driver, during submission of commands, by shared surface, etc. */
    uint32_t volatile cRefs;

    /* SVGA_HOST_OBJECT_ type. */
    uint32_t uType;

    /* Device the object is associated with. */
    VBOXWDDM_EXT_VMSVGA *pSvga;

    /* Destructor. */
    PFNHostObjectDestroy pfnHostObjectDestroy;
} SVGAHOSTOBJECT;

#define SVGAHOSTOBJECTID(pHO) ((pHO)->u.avl.core.Key)

/* Information about an allocated surface. */
typedef struct SURFACEOBJECT
{
    SVGAHOSTOBJECT ho;

    /* The actual sid, which must be used by the commands instead of the ho.core.Key.
     * Equal to the AVL node key for non-shared surfaces.
     */
    uint32_t u32SharedSid;

    uint32_t mobid;
} SURFACEOBJECT;
AssertCompile(RT_OFFSETOF(SURFACEOBJECT, ho) == 0);

typedef struct GAHWRENDERDATA
{
    RTLISTNODE node;
    uint32_t u32SubmissionFenceId;
    uint32_t u32Reserved;
    /* Private data follows. */
} GAHWRENDERDATA;

DECLINLINE(void) SvgaHostObjectsLock(VBOXWDDM_EXT_VMSVGA *pSvga, KIRQL *pOldIrql)
{
    KeAcquireSpinLock(&pSvga->HostObjectsSpinLock, pOldIrql);
}

DECLINLINE(void) SvgaHostObjectsUnlock(VBOXWDDM_EXT_VMSVGA *pSvga, KIRQL OldIrql)
{
    KeReleaseSpinLock(&pSvga->HostObjectsSpinLock, OldIrql);
}

NTSTATUS SvgaHostObjectsCleanup(VBOXWDDM_EXT_VMSVGA *pSvga);

NTSTATUS SvgaAdapterStart(PVBOXWDDM_EXT_VMSVGA *ppSvga,
                          DXGKRNL_INTERFACE *pDxgkInterface,
                          PHYSICAL_ADDRESS physFIFO,
                          ULONG cbFIFO,
                          PHYSICAL_ADDRESS physIO,
                          ULONG cbIO);
void SvgaAdapterStop(PVBOXWDDM_EXT_VMSVGA pSvga,
                     DXGKRNL_INTERFACE *pDxgkInterface);

NTSTATUS SvgaQueryInfo(PVBOXWDDM_EXT_VMSVGA pSvga,
                       VBOXGAHWINFOSVGA *pSvgaInfo);

NTSTATUS SvgaScreenDefine(PVBOXWDDM_EXT_VMSVGA pSvga,
                          uint32_t u32Offset,
                          uint32_t u32ScreenId,
                          int32_t xOrigin,
                          int32_t yOrigin,
                          uint32_t u32Width,
                          uint32_t u32Height,
                          bool fBlank);
NTSTATUS SvgaScreenDestroy(PVBOXWDDM_EXT_VMSVGA pSvga,
                           uint32_t u32ScreenId);

NTSTATUS Svga3dDefineGBScreenTarget(PVBOXWDDM_EXT_VMSVGA pSvga,
                                    uint32_t stid,
                                    uint32_t width,
                                    uint32_t height,
                                    int32_t xRoot,
                                    int32_t yRoot,
                                    SVGAScreenTargetFlags flags,
                                    uint32_t dpi);
NTSTATUS Svga3dDestroyGBScreenTarget(PVBOXWDDM_EXT_VMSVGA pSvga,
                                     uint32_t stid);
NTSTATUS Svga3dBindGBScreenTarget(PVBOXWDDM_EXT_VMSVGA pSvga,
                                  uint32_t stid,
                                  uint32_t sid);

NTSTATUS SvgaContextCreate(PVBOXWDDM_EXT_VMSVGA pSvga,
                           uint32_t u32Cid);
NTSTATUS SvgaContextDestroy(PVBOXWDDM_EXT_VMSVGA pSvga,
                            uint32_t u32Cid);

NTSTATUS SvgaSurfaceCreate(VBOXWDDM_EXT_VMSVGA *pSvga,
                           GASURFCREATE *pCreateParms,
                           GASURFSIZE *paSizes,
                           uint32_t cSizes,
                           uint32_t *pu32Sid);
NTSTATUS SvgaGBSurfaceCreate(VBOXWDDM_EXT_VMSVGA *pSvga,
                             void *pvOwner,
                             SVGAGBSURFCREATE *pCreateParms);
NTSTATUS SvgaSurfaceUnref(VBOXWDDM_EXT_VMSVGA *pSvga,
                          uint32_t u32Sid);
SURFACEOBJECT *SvgaSurfaceObjectQuery(VBOXWDDM_EXT_VMSVGA *pSvga,
                                      uint32_t u32Sid);
NTSTATUS SvgaSurfaceObjectRelease(SURFACEOBJECT *pSO);

NTSTATUS SvgaFence(PVBOXWDDM_EXT_VMSVGA pSvga,
                   uint32_t u32Fence);
NTSTATUS SvgaSurfaceDefine(PVBOXWDDM_EXT_VMSVGA pSvga,
                           GASURFCREATE const *pCreateParms,
                           GASURFSIZE const *paSizes,
                           uint32_t cSizes,
                           uint32_t u32Sid);
NTSTATUS SvgaSurfaceDestroy(PVBOXWDDM_EXT_VMSVGA pSvga,
                            uint32_t u32Sid);

NTSTATUS SvgaContextIdAlloc(PVBOXWDDM_EXT_VMSVGA pSvga,
                            uint32_t *pu32Cid);
NTSTATUS SvgaContextIdFree(PVBOXWDDM_EXT_VMSVGA pSvga,
                           uint32_t u32Cid);
NTSTATUS SvgaSurfaceIdAlloc(PVBOXWDDM_EXT_VMSVGA pSvga,
                            uint32_t *pu32Sid);
NTSTATUS SvgaSurfaceIdFree(PVBOXWDDM_EXT_VMSVGA pSvga,
                           uint32_t u32Sid);


NTSTATUS SvgaRenderCommands(PVBOXWDDM_EXT_VMSVGA pSvga,
                            struct VMSVGACONTEXT *pSvgaContext,
                            void *pvTarget,
                            uint32_t cbTarget,
                            const void *pvSource,
                            uint32_t cbSource,
                            uint32_t *pu32TargetLength,
                            uint32_t *pu32ProcessedLength,
                            GAHWRENDERDATA **ppHwRenderData);
NTSTATUS SvgaRenderComplete(PVBOXWDDM_EXT_VMSVGA pSvga,
                            GAHWRENDERDATA *pHwRenderData);

NTSTATUS SvgaSharedSidInsert(VBOXWDDM_EXT_VMSVGA *pSvga,
                             uint32_t u32Sid,
                             uint32_t u32SharedSid);
NTSTATUS SvgaSharedSidRemove(VBOXWDDM_EXT_VMSVGA *pSvga,
                             uint32_t u32Sid);
uint32_t SvgaSharedSidGet(VBOXWDDM_EXT_VMSVGA *pSvga,
                          uint32_t u32Sid);

NTSTATUS SvgaGenPresent(uint32_t u32Sid,
                        uint32_t u32Width,
                        uint32_t u32Height,
                        void *pvDst,
                        uint32_t cbDst,
                        uint32_t *pcbOut);
NTSTATUS SvgaPresent(PVBOXWDDM_EXT_VMSVGA pSvga,
                     uint32_t u32Sid,
                     uint32_t u32Width,
                     uint32_t u32Height);

NTSTATUS SvgaGenPresentVRAM(PVBOXWDDM_EXT_VMSVGA pSvga,
                            uint32_t u32Sid,
                            uint32_t u32Width,
                            uint32_t u32Height,
                            uint32_t u32VRAMOffset,
                            void *pvDst,
                            uint32_t cbDst,
                            uint32_t *pcbOut);
NTSTATUS SvgaPresentVRAM(PVBOXWDDM_EXT_VMSVGA pSvga,
                         uint32_t u32Sid,
                         uint32_t u32Width,
                         uint32_t u32Height,
                         uint32_t u32VRAMOffset);

NTSTATUS SvgaGenSurfaceDMA(PVBOXWDDM_EXT_VMSVGA pSvga,
                           SVGAGuestImage const *pGuestImage,
                           SVGA3dSurfaceImageId const *pSurfId,
                           SVGA3dTransferType enmTransferType, uint32_t xSrc, uint32_t ySrc,
                           uint32_t xDst, uint32_t yDst, uint32_t cWidth, uint32_t cHeight,
                           void *pvDst,
                           uint32_t cbDst,
                           uint32_t *pcbOut);

NTSTATUS SvgaGenBlitGMRFBToScreen(PVBOXWDDM_EXT_VMSVGA pSvga,
                                  uint32_t idDstScreen,
                                  int32_t xSrc,
                                  int32_t ySrc,
                                  RECT const *pDstRect,
                                  void *pvDst,
                                  uint32_t cbDst,
                                  uint32_t *pcbOut);
NTSTATUS SvgaGenBlitScreenToGMRFB(PVBOXWDDM_EXT_VMSVGA pSvga,
                                  uint32_t idSrcScreen,
                                  int32_t xSrc,
                                  int32_t ySrc,
                                  RECT const *pDstRect,
                                  void *pvDst,
                                  uint32_t cbDst,
                                  uint32_t *pcbOut);

NTSTATUS SvgaBlitGMRFBToScreen(PVBOXWDDM_EXT_VMSVGA pSvga,
                               uint32_t idDstScreen,
                               int32_t xSrc,
                               int32_t ySrc,
                               RECT const *pDstRect);

NTSTATUS SvgaGenBlitSurfaceToScreen(PVBOXWDDM_EXT_VMSVGA pSvga,
                                    uint32_t sid,
                                    RECT const *pSrcRect,
                                    uint32 idDstScreen,
                                    RECT const *pDstRect,
                                    uint32_t cDstClipRects,
                                    RECT const *paDstClipRects,
                                    void *pvDst,
                                    uint32_t cbDst,
                                    uint32_t *pcbOut,
                                    uint32_t *pcOutDstClipRects);

NTSTATUS SvgaUpdate(PVBOXWDDM_EXT_VMSVGA pSvga,
                    uint32_t u32X,
                    uint32_t u32Y,
                    uint32_t u32Width,
                    uint32_t u32Height);

NTSTATUS SvgaDefineCursor(PVBOXWDDM_EXT_VMSVGA pSvga,
                          uint32_t u32HotspotX,
                          uint32_t u32HotspotY,
                          uint32_t u32Width,
                          uint32_t u32Height,
                          uint32_t u32AndMaskDepth,
                          uint32_t u32XorMaskDepth,
                          void const *pvAndMask,
                          uint32_t cbAndMask,
                          void const *pvXorMask,
                          uint32_t cbXorMask);

NTSTATUS SvgaDefineAlphaCursor(PVBOXWDDM_EXT_VMSVGA pSvga,
                               uint32_t u32HotspotX,
                               uint32_t u32HotspotY,
                               uint32_t u32Width,
                               uint32_t u32Height,
                               void const *pvImage,
                               uint32_t cbImage);

NTSTATUS SvgaGenDefineGMRFB(PVBOXWDDM_EXT_VMSVGA pSvga,
                            uint32_t u32Offset,
                            uint32_t u32BytesPerLine,
                            void *pvDst,
                            uint32_t cbDst,
                            uint32_t *pcbOut);

NTSTATUS SvgaDefineGMRFB(PVBOXWDDM_EXT_VMSVGA pSvga,
                         uint32_t u32Offset,
                         uint32_t u32BytesPerLine,
                         bool fForce);

NTSTATUS SvgaRegionCreate(VBOXWDDM_EXT_VMSVGA *pSvga,
                          void *pvOwner,
                          uint32_t u32NumPages,
                          uint32_t *pu32GmrId,
                          uint64_t *pu64UserAddress);
NTSTATUS SvgaRegionUserAddressAndSize(VBOXWDDM_EXT_VMSVGA *pSvga,
                                      uint32_t u32GmrId,
                                      uint64_t *pu64UserAddress,
                                      uint32_t *pu32Size);
NTSTATUS SvgaRegionDestroy(VBOXWDDM_EXT_VMSVGA *pSvga,
                           uint32_t u32GmrId);
void SvgaRegionsDestroy(VBOXWDDM_EXT_VMSVGA *pSvga,
                        void *pvOwner);

NTSTATUS SvgaDXContextIdAlloc(PVBOXWDDM_EXT_VMSVGA pSvga,
                              uint32_t *pu32Cid);

NTSTATUS SvgaDXContextIdFree(PVBOXWDDM_EXT_VMSVGA pSvga,
                             uint32_t u32Cid);

NTSTATUS SvgaMobIdAlloc(PVBOXWDDM_EXT_VMSVGA pSvga,
                        uint32_t *pu32MobId);

NTSTATUS SvgaMobIdFree(PVBOXWDDM_EXT_VMSVGA pSvga,
                       uint32_t u32MobId);

NTSTATUS SvgaScreenTargetIdAlloc(PVBOXWDDM_EXT_VMSVGA pSvga,
                                 uint32_t *pu32ScreenTargetId);

NTSTATUS SvgaScreenTargetIdFree(PVBOXWDDM_EXT_VMSVGA pSvga,
                                uint32_t u32ScreenTargetId);

NTSTATUS SvgaDXContextCreate(PVBOXWDDM_EXT_VMSVGA pSvga,
                             uint32_t u32Cid);

NTSTATUS SvgaDXContextDestroy(PVBOXWDDM_EXT_VMSVGA pSvga,
                              uint32_t u32Cid);

NTSTATUS SvgaRenderCommandsD3D(PVBOXWDDM_EXT_VMSVGA pSvga,
                               PVMSVGACONTEXT pSvgaContext,
                               void *pvTarget,
                               uint32_t cbTarget,
                               const void *pvSource,
                               uint32_t cbSource,
                               uint32_t *pu32TargetLength,
                               uint32_t *pu32ProcessedLength);

#ifdef DEBUG
NTSTATUS SvgaDebugCommandsD3D(PVBOXWDDM_EXT_VMSVGA pSvga,
                              PVMSVGACONTEXT pSvgaContext,
                              const void *pvSource,
                              uint32_t cbSource);
#endif

NTSTATUS SvgaGboCreate(VBOXWDDM_EXT_VMSVGA *pSvga,
                       PVMSVGAGBO *ppGbo,
                       uint32_t cbGbo,
                       const char *pszTag);
NTSTATUS SvgaGboCreateForMdl(VBOXWDDM_EXT_VMSVGA *pSvga,
                             PVMSVGAGBO *ppGbo,
                             SIZE_T NumberOfPages,
                             PMDL pMdl,
                             ULONG MdlOffset);
void SvgaGboFree(VBOXWDDM_EXT_VMSVGA *pSvga,
                 VMSVGAGBO *pGbo);

DECLINLINE(void) SvgaGboReference(PVMSVGAGBO pGbo)
{
    if (pGbo)
    {
        int32_t const c = ASMAtomicIncS32(&pGbo->cRefs);
        Assert(c > 0); RT_NOREF(c);
    }
}

DECLINLINE(void) SvgaGboUnreference(VBOXWDDM_EXT_VMSVGA *pSvga,
                                    PVMSVGAGBO *ppGbo)
{
    if (*ppGbo)
    {
        int32_t const c = ASMAtomicDecS32(&(*ppGbo)->cRefs);
        Assert(c >= 0);
        if (c == 0)
            SvgaGboFree(pSvga, *ppGbo);
        *ppGbo = NULL;
    }
}

void SvgaMobFree(VBOXWDDM_EXT_VMSVGA *pSvga,
                 SVGAMobId *pMobid);
NTSTATUS SvgaMobAlloc(VBOXWDDM_EXT_VMSVGA *pSvga,
                      SVGAMobId *pMobid,
                      PVMSVGAGBO pGbo);
void *SvgaMobAddress(VBOXWDDM_EXT_VMSVGA *pSvga,
                     SVGAMobId mobid);
NTSTATUS SvgaMobDefine(VBOXWDDM_EXT_VMSVGA *pSvga,
                       SVGAMobId mobid,
                       void *pvCmd,
                       uint32_t cbReserved,
                       uint32_t *pcbCmd);
NTSTATUS SvgaMobDestroy(VBOXWDDM_EXT_VMSVGA *pSvga,
                        SVGAMobId mobid,
                        void *pvCmd,
                        uint32_t cbReserved,
                        uint32_t *pcbCmd);
void SvgaDeferredMobDestruction(PVBOXWDDM_EXT_VMSVGA pSvga);

NTSTATUS SvgaCOTNotifyId(VBOXWDDM_EXT_VMSVGA *pSvga,
                         PVMSVGACONTEXT pSvgaContext,
                         SVGACOTableType enmType,
                         uint32_t id);

#endif /* !GA_INCLUDED_SRC_WINNT_Graphics_Video_mp_wddm_gallium_Svga_h */
