"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.LEGACY_LAST_MODIFIED_VERSION = void 0;
exports.getMigrations = getMigrations;
exports.isSecuritySolutionRule = exports.isSecuritySolutionLegacyNotification = exports.isAnyActionSupportIncidents = void 0;
var _fp = require("lodash/fp");
var _migrations = require("./geo_containment/migrations");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const SIEM_APP_ID = 'securitySolution';
const SIEM_SERVER_APP_ID = 'siem';
const LEGACY_LAST_MODIFIED_VERSION = exports.LEGACY_LAST_MODIFIED_VERSION = 'pre-7.10.0';
function createEsoMigration(encryptedSavedObjects, isMigrationNeededPredicate, migrationFunc) {
  return encryptedSavedObjects.createMigration({
    isMigrationNeededPredicate,
    migration: migrationFunc,
    shouldMigrateIfDecryptionFails: true // shouldMigrateIfDecryptionFails flag that applies the migration to undecrypted document if decryption fails
  });
}
const SUPPORT_INCIDENTS_ACTION_TYPES = ['.servicenow', '.jira', '.resilient'];
const isAnyActionSupportIncidents = doc => doc.attributes.actions.some(action => SUPPORT_INCIDENTS_ACTION_TYPES.includes(action.actionTypeId));
exports.isAnyActionSupportIncidents = isAnyActionSupportIncidents;
const isSecuritySolutionRule = doc => doc.attributes.alertTypeId === 'siem.signals';

/**
 * Returns true if the alert type is that of "siem.notifications" which is a legacy notification system that was deprecated in 7.16.0
 * in favor of using the newer alerting notifications system.
 * @param doc The saved object alert type document
 * @returns true if this is a legacy "siem.notifications" rule, otherwise false
 * @deprecated Once we are confident all rules relying on side-car actions SO's have been migrated to SO references we should remove this function
 */
exports.isSecuritySolutionRule = isSecuritySolutionRule;
const isSecuritySolutionLegacyNotification = doc => doc.attributes.alertTypeId === 'siem.notifications';
exports.isSecuritySolutionLegacyNotification = isSecuritySolutionLegacyNotification;
function getMigrations(encryptedSavedObjects, isPreconfigured) {
  const migrationWhenRBACWasIntroduced = createEsoMigration(encryptedSavedObjects,
  // migrate all documents in 7.10 in order to add the "meta" RBAC field
  doc => true, pipeMigrations(markAsLegacyAndChangeConsumer, setAlertIdAsDefaultDedupkeyOnPagerDutyActions, initializeExecutionStatus));
  const migrationAlertUpdatedAtAndNotifyWhen = createEsoMigration(encryptedSavedObjects,
  // migrate all documents in 7.11 in order to add the "updatedAt" and "notifyWhen" fields
  doc => true, pipeMigrations(setAlertUpdatedAtDate, setNotifyWhen));
  const migrationActions7112 = createEsoMigration(encryptedSavedObjects, doc => isAnyActionSupportIncidents(doc), pipeMigrations(restructureConnectorsThatSupportIncident));
  const migrationSecurityRules713 = createEsoMigration(encryptedSavedObjects, doc => isSecuritySolutionRule(doc), pipeMigrations(removeNullsFromSecurityRules));
  const migrationSecurityRules714 = createEsoMigration(encryptedSavedObjects, doc => isSecuritySolutionRule(doc), pipeMigrations(removeNullAuthorFromSecurityRules));
  const migrationSecurityRules715 = createEsoMigration(encryptedSavedObjects, doc => isSecuritySolutionRule(doc), pipeMigrations(addExceptionListsToReferences));
  const migrateRules716 = createEsoMigration(encryptedSavedObjects, doc => true, pipeMigrations(setLegacyId, getRemovePreconfiguredConnectorsFromReferencesFn(isPreconfigured), addRuleIdsToLegacyNotificationReferences, _migrations.extractRefsFromGeoContainmentAlert));
  return {
    '7.10.0': executeMigrationWithErrorHandling(migrationWhenRBACWasIntroduced, '7.10.0'),
    '7.11.0': executeMigrationWithErrorHandling(migrationAlertUpdatedAtAndNotifyWhen, '7.11.0'),
    '7.11.2': executeMigrationWithErrorHandling(migrationActions7112, '7.11.2'),
    '7.13.0': executeMigrationWithErrorHandling(migrationSecurityRules713, '7.13.0'),
    '7.14.1': executeMigrationWithErrorHandling(migrationSecurityRules714, '7.14.1'),
    '7.15.0': executeMigrationWithErrorHandling(migrationSecurityRules715, '7.15.0'),
    '7.16.0': executeMigrationWithErrorHandling(migrateRules716, '7.16.0')
  };
}
function executeMigrationWithErrorHandling(migrationFunc, version) {
  return (doc, context) => {
    try {
      return migrationFunc(doc, context);
    } catch (ex) {
      context.log.error(`encryptedSavedObject ${version} migration failed for alert ${doc.id} with error: ${ex.message}`, {
        migrations: {
          alertDocument: doc
        }
      });
      throw ex;
    }
  };
}
const setAlertUpdatedAtDate = doc => {
  const updatedAt = doc.updated_at || doc.attributes.createdAt;
  return {
    ...doc,
    attributes: {
      ...doc.attributes,
      updatedAt
    }
  };
};
const setNotifyWhen = doc => {
  const notifyWhen = doc.attributes.throttle ? 'onThrottleInterval' : 'onActiveAlert';
  return {
    ...doc,
    attributes: {
      ...doc.attributes,
      notifyWhen
    }
  };
};
const consumersToChange = new Map(Object.entries({
  alerting: 'alerts',
  metrics: 'infrastructure',
  [SIEM_APP_ID]: SIEM_SERVER_APP_ID
}));
function markAsLegacyAndChangeConsumer(doc) {
  var _consumersToChange$ge;
  const {
    attributes: {
      consumer
    }
  } = doc;
  return {
    ...doc,
    attributes: {
      ...doc.attributes,
      consumer: (_consumersToChange$ge = consumersToChange.get(consumer)) !== null && _consumersToChange$ge !== void 0 ? _consumersToChange$ge : consumer,
      // mark any alert predating 7.10 as a legacy alert
      meta: {
        versionApiKeyLastmodified: LEGACY_LAST_MODIFIED_VERSION
      }
    }
  };
}
function setAlertIdAsDefaultDedupkeyOnPagerDutyActions(doc) {
  const {
    attributes
  } = doc;
  return {
    ...doc,
    attributes: {
      ...attributes,
      ...(attributes.actions ? {
        actions: attributes.actions.map(action => {
          var _action$params$dedupK;
          if (action.actionTypeId !== '.pagerduty' || action.params.eventAction === 'trigger') {
            return action;
          }
          return {
            ...action,
            params: {
              ...action.params,
              dedupKey: (_action$params$dedupK = action.params.dedupKey) !== null && _action$params$dedupK !== void 0 ? _action$params$dedupK : '{{alertId}}'
            }
          };
        })
      } : {})
    }
  };
}
function initializeExecutionStatus(doc) {
  const {
    attributes
  } = doc;
  return {
    ...doc,
    attributes: {
      ...attributes,
      executionStatus: {
        status: 'pending',
        lastExecutionDate: new Date().toISOString(),
        error: null
      }
    }
  };
}
function isEmptyObject(obj) {
  for (const attr in obj) {
    if (Object.prototype.hasOwnProperty.call(obj, attr)) {
      return false;
    }
  }
  return true;
}
function restructureConnectorsThatSupportIncident(doc) {
  const {
    actions
  } = doc.attributes;
  const newActions = actions.reduce((acc, action) => {
    if (['.servicenow', '.jira', '.resilient'].includes(action.actionTypeId) && action.params.subAction === 'pushToService') {
      var _incident, _action$params, _action$params$subAct;
      // Future developer, we needed to do that because when we created this migration
      // we forget to think about user already using 7.11.0 and having an incident attribute build the right way
      // IMPORTANT -> if you change this code please do the same inside of this file
      // x-pack/plugins/alerting/server/saved_objects/migrations.ts
      const subActionParamsIncident = (_incident = (_action$params = action.params) === null || _action$params === void 0 ? void 0 : (_action$params$subAct = _action$params.subActionParams) === null || _action$params$subAct === void 0 ? void 0 : _action$params$subAct.incident) !== null && _incident !== void 0 ? _incident : null;
      if (subActionParamsIncident != null && !isEmptyObject(subActionParamsIncident)) {
        return [...acc, action];
      }
      if (action.actionTypeId === '.servicenow') {
        const {
          title,
          comments,
          comment,
          description,
          severity,
          urgency,
          impact,
          short_description: shortDescription
        } = action.params.subActionParams;
        return [...acc, {
          ...action,
          params: {
            subAction: 'pushToService',
            subActionParams: {
              incident: {
                short_description: shortDescription !== null && shortDescription !== void 0 ? shortDescription : title,
                description,
                severity,
                urgency,
                impact
              },
              comments: [...(comments !== null && comments !== void 0 ? comments : []), ...(comment != null ? [{
                commentId: '1',
                comment
              }] : [])]
            }
          }
        }];
      } else if (action.actionTypeId === '.jira') {
        const {
          title,
          comments,
          description,
          issueType,
          priority,
          labels,
          parent,
          summary
        } = action.params.subActionParams;
        return [...acc, {
          ...action,
          params: {
            subAction: 'pushToService',
            subActionParams: {
              incident: {
                summary: summary !== null && summary !== void 0 ? summary : title,
                description,
                issueType,
                priority,
                labels,
                parent
              },
              comments
            }
          }
        }];
      } else if (action.actionTypeId === '.resilient') {
        const {
          title,
          comments,
          description,
          incidentTypes,
          severityCode,
          name
        } = action.params.subActionParams;
        return [...acc, {
          ...action,
          params: {
            subAction: 'pushToService',
            subActionParams: {
              incident: {
                name: name !== null && name !== void 0 ? name : title,
                description,
                incidentTypes,
                severityCode
              },
              comments
            }
          }
        }];
      }
    }
    return [...acc, action];
  }, []);
  return {
    ...doc,
    attributes: {
      ...doc.attributes,
      actions: newActions
    }
  };
}
function convertNullToUndefined(attribute) {
  return attribute != null ? attribute : undefined;
}
function removeNullsFromSecurityRules(doc) {
  const {
    attributes: {
      params
    }
  } = doc;
  return {
    ...doc,
    attributes: {
      ...doc.attributes,
      params: {
        ...params,
        buildingBlockType: convertNullToUndefined(params.buildingBlockType),
        note: convertNullToUndefined(params.note),
        index: convertNullToUndefined(params.index),
        language: convertNullToUndefined(params.language),
        license: convertNullToUndefined(params.license),
        outputIndex: convertNullToUndefined(params.outputIndex),
        savedId: convertNullToUndefined(params.savedId),
        timelineId: convertNullToUndefined(params.timelineId),
        timelineTitle: convertNullToUndefined(params.timelineTitle),
        meta: convertNullToUndefined(params.meta),
        query: convertNullToUndefined(params.query),
        filters: convertNullToUndefined(params.filters),
        riskScoreMapping: params.riskScoreMapping != null ? params.riskScoreMapping : [],
        ruleNameOverride: convertNullToUndefined(params.ruleNameOverride),
        severityMapping: params.severityMapping != null ? params.severityMapping : [],
        threat: params.threat != null ? params.threat : [],
        threshold: params.threshold != null && typeof params.threshold === 'object' && !Array.isArray(params.threshold) ? {
          field: Array.isArray(params.threshold.field) ? params.threshold.field : params.threshold.field === '' || params.threshold.field == null ? [] : [params.threshold.field],
          value: params.threshold.value,
          cardinality: params.threshold.cardinality != null ? params.threshold.cardinality : []
        } : undefined,
        timestampOverride: convertNullToUndefined(params.timestampOverride),
        exceptionsList: params.exceptionsList != null ? params.exceptionsList : params.exceptions_list != null ? params.exceptions_list : params.lists != null ? params.lists : [],
        threatFilters: convertNullToUndefined(params.threatFilters),
        machineLearningJobId: params.machineLearningJobId == null ? undefined : Array.isArray(params.machineLearningJobId) ? params.machineLearningJobId : [params.machineLearningJobId]
      }
    }
  };
}

/**
 * The author field was introduced later and was not part of the original rules. We overlooked
 * the filling in the author field as an empty array in an earlier upgrade routine from
 * 'removeNullsFromSecurityRules' during the 7.13.0 upgrade. Since we don't change earlier migrations,
 * but rather only move forward with the "arrow of time" we are going to upgrade and fix
 * it if it is missing for anyone in 7.14.0 and above release. Earlier releases if we want to fix them,
 * would have to be modified as a "7.13.1", etc... if we want to fix it there.
 * @param doc The document that is not migrated and contains a "null" or "undefined" author field
 * @returns The document with the author field fleshed in.
 */
function removeNullAuthorFromSecurityRules(doc) {
  const {
    attributes: {
      params
    }
  } = doc;
  return {
    ...doc,
    attributes: {
      ...doc.attributes,
      params: {
        ...params,
        author: params.author != null ? params.author : []
      }
    }
  };
}

/**
 * This migrates exception list containers to saved object references on an upgrade.
 * We only migrate if we find these conditions:
 *   - exceptionLists are an array and not null, undefined, or malformed data.
 *   - The exceptionList item is an object and id is a string and not null, undefined, or malformed data
 *   - The existing references do not already have an exceptionItem reference already found within it.
 * Some of these issues could crop up during either user manual errors of modifying things, earlier migration
 * issues, etc...
 * @param doc The document that might have exceptionListItems to migrate
 * @returns The document migrated with saved object references
 */
function addExceptionListsToReferences(doc) {
  const {
    attributes: {
      params: {
        exceptionsList
      }
    },
    references
  } = doc;
  if (!Array.isArray(exceptionsList)) {
    // early return if we are not an array such as being undefined or null or malformed.
    return doc;
  } else {
    const exceptionsToTransform = removeMalformedExceptionsList(exceptionsList);
    const newReferences = exceptionsToTransform.flatMap((exceptionItem, index) => {
      const existingReferenceFound = references === null || references === void 0 ? void 0 : references.find(reference => {
        return reference.id === exceptionItem.id && (reference.type === 'exception-list' && exceptionItem.namespace_type === 'single' || reference.type === 'exception-list-agnostic' && exceptionItem.namespace_type === 'agnostic');
      });
      if (existingReferenceFound) {
        // skip if the reference already exists for some uncommon reason so we do not add an additional one.
        // This enables us to be idempotent and you can run this migration multiple times and get the same output.
        return [];
      } else {
        return [{
          name: `param:exceptionsList_${index}`,
          id: String(exceptionItem.id),
          type: exceptionItem.namespace_type === 'agnostic' ? 'exception-list-agnostic' : 'exception-list'
        }];
      }
    });
    if (references == null && newReferences.length === 0) {
      // Avoid adding an empty references array if the existing saved object never had one to begin with
      return doc;
    } else {
      return {
        ...doc,
        references: [...(references !== null && references !== void 0 ? references : []), ...newReferences]
      };
    }
  }
}

/**
 * This will do a flatMap reduce where we only return exceptionsLists and their items if:
 *   - exceptionLists are an array and not null, undefined, or malformed data.
 *   - The exceptionList item is an object and id is a string and not null, undefined, or malformed data
 *
 * Some of these issues could crop up during either user manual errors of modifying things, earlier migration
 * issues, etc...
 * @param exceptionsList The list of exceptions
 * @returns The exception lists if they are a valid enough shape
 */
function removeMalformedExceptionsList(exceptionsList) {
  if (!Array.isArray(exceptionsList)) {
    // early return if we are not an array such as being undefined or null or malformed.
    return [];
  } else {
    return exceptionsList.flatMap(exceptionItem => {
      if (!(exceptionItem instanceof Object) || !(0, _fp.isString)(exceptionItem.id)) {
        // return early if we are not an object such as being undefined or null or malformed
        // or the exceptionItem.id is not a string from being malformed
        return [];
      } else {
        return [exceptionItem];
      }
    });
  }
}

/**
 * This migrates rule_id's within the legacy siem.notification to saved object references on an upgrade.
 * We only migrate if we find these conditions:
 *   - ruleAlertId is a string and not null, undefined, or malformed data.
 *   - The existing references do not already have a ruleAlertId found within it.
 * Some of these issues could crop up during either user manual errors of modifying things, earlier migration
 * issues, etc... so we are safer to check them as possibilities
 * @deprecated Once we are confident all rules relying on side-car actions SO's have been migrated to SO references we should remove this function
 * @param doc The document that might have "ruleAlertId" to migrate into the references
 * @returns The document migrated with saved object references
 */
function addRuleIdsToLegacyNotificationReferences(doc) {
  const {
    attributes: {
      params: {
        ruleAlertId
      }
    },
    references
  } = doc;
  if (!isSecuritySolutionLegacyNotification(doc) || !(0, _fp.isString)(ruleAlertId)) {
    // early return if we are not a string or if we are not a security solution notification saved object.
    return doc;
  } else {
    const existingReferences = references !== null && references !== void 0 ? references : [];
    const existingReferenceFound = existingReferences.find(reference => {
      return reference.id === ruleAlertId && reference.type === 'alert';
    });
    if (existingReferenceFound) {
      // skip this if the references already exists for some uncommon reason so we do not add an additional one.
      return doc;
    } else {
      const savedObjectReference = {
        id: ruleAlertId,
        name: 'param:alert_0',
        type: 'alert'
      };
      const newReferences = [...existingReferences, savedObjectReference];
      return {
        ...doc,
        references: newReferences
      };
    }
  }
}
function setLegacyId(doc) {
  const {
    id
  } = doc;
  return {
    ...doc,
    attributes: {
      ...doc.attributes,
      legacyId: id
    }
  };
}
function getRemovePreconfiguredConnectorsFromReferencesFn(isPreconfigured) {
  return doc => {
    return removePreconfiguredConnectorsFromReferences(doc, isPreconfigured);
  };
}
function removePreconfiguredConnectorsFromReferences(doc, isPreconfigured) {
  const {
    attributes: {
      actions
    },
    references
  } = doc;

  // Look for connector references
  const connectorReferences = (references !== null && references !== void 0 ? references : []).filter(ref => ref.name.startsWith('action_'));
  if (connectorReferences.length > 0) {
    const restReferences = (references !== null && references !== void 0 ? references : []).filter(ref => !ref.name.startsWith('action_'));
    const updatedConnectorReferences = [];
    const updatedActions = [];

    // For each connector reference, check if connector is preconfigured
    // If yes, we need to remove from the references array and update
    // the corresponding action so it directly references the preconfigured connector id
    connectorReferences.forEach(connectorRef => {
      // Look for the corresponding entry in the actions array
      const correspondingAction = getCorrespondingAction(actions, connectorRef.name);
      if (correspondingAction) {
        if (isPreconfigured(connectorRef.id)) {
          updatedActions.push({
            ...correspondingAction,
            actionRef: `preconfigured:${connectorRef.id}`
          });
        } else {
          updatedActions.push(correspondingAction);
          updatedConnectorReferences.push(connectorRef);
        }
      } else {
        // Couldn't find the matching action, leave as is
        updatedConnectorReferences.push(connectorRef);
      }
    });
    return {
      ...doc,
      attributes: {
        ...doc.attributes,
        actions: [...updatedActions]
      },
      references: [...updatedConnectorReferences, ...restReferences]
    };
  }
  return doc;
}
function getCorrespondingAction(actions, connectorRef) {
  if (!Array.isArray(actions)) {
    return null;
  } else {
    return actions.find(action => (action === null || action === void 0 ? void 0 : action.actionRef) === connectorRef);
  }
}
function pipeMigrations(...migrations) {
  return doc => migrations.reduce((migratedDoc, nextMigration) => nextMigration(migratedDoc), doc);
}