import _objectSpread from "@babel/runtime/helpers/objectSpread2";
import _slicedToArray from "@babel/runtime/helpers/slicedToArray";
import _defineProperty from "@babel/runtime/helpers/defineProperty";
import _toConsumableArray from "@babel/runtime/helpers/toConsumableArray";
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

import { chunk } from 'lodash/fp';
import { entriesExists, entriesMatch, entriesMatchAny, entriesNested } from '@kbn/securitysolution-io-ts-list-types';
import { hasLargeValueList } from '../has_large_value_list';
export var chunkExceptions = function chunkExceptions(exceptions, chunkSize) {
  return chunk(chunkSize, exceptions);
};

/**
 * Transforms the os_type into a regular filter as if the user had created it
 * from the fields for the next state of transforms which will create the elastic filters
 * from it.
 *
 * Note: We use two types of fields, the "host.os.type" and "host.os.name.caseless"
 * The endpoint/endgame agent has been using "host.os.name.caseless" as the same value as the ECS
 * value of "host.os.type" where the auditbeat, winlogbeat, etc... (other agents) are all using
 * "host.os.type". In order to be compatible with both, I create an "OR" between these two data types
 * where if either has a match then we will exclude it as part of the match. This should also be
 * forwards compatible for endpoints/endgame agents when/if they upgrade to using "host.os.type"
 * rather than using "host.os.name.caseless" values.
 *
 * Also we create another "OR" from the osType names so that if there are multiples such as ['windows', 'linux']
 * this will exclude anything with either 'windows' or with 'linux'
 * @param osTypes The os_type array from the REST interface that is an array such as ['windows', 'linux']
 * @param entries The entries to join the OR's with before the elastic filter change out
 */
export var transformOsType = function transformOsType(osTypes, entries) {
  var hostTypeTransformed = osTypes.map(function (osType) {
    return [{
      field: 'host.os.type',
      operator: 'included',
      type: 'match',
      value: osType
    }].concat(_toConsumableArray(entries));
  });
  var caseLessTransformed = osTypes.map(function (osType) {
    return [{
      field: 'host.os.name.caseless',
      operator: 'included',
      type: 'match',
      value: osType
    }].concat(_toConsumableArray(entries));
  });
  return [].concat(_toConsumableArray(hostTypeTransformed), _toConsumableArray(caseLessTransformed));
};

/**
 * This builds an exception item filter with the os type
 * @param osTypes The os_type array from the REST interface that is an array such as ['windows', 'linux']
 * @param entries The entries to join the OR's with before the elastic filter change out
 */
export var buildExceptionItemFilterWithOsType = function buildExceptionItemFilterWithOsType(osTypes, entries) {
  var entriesWithOsTypes = transformOsType(osTypes, entries);
  return entriesWithOsTypes.map(function (entryWithOsType) {
    return {
      bool: {
        filter: entryWithOsType.map(function (entry) {
          return createInnerAndClauses(entry);
        })
      }
    };
  });
};
export var buildExceptionItemFilter = function buildExceptionItemFilter(exceptionItem) {
  var entries = exceptionItem.entries,
    osTypes = exceptionItem.os_types;
  if (osTypes != null && osTypes.length > 0) {
    return buildExceptionItemFilterWithOsType(osTypes, entries);
  } else {
    if (entries.length === 1) {
      return [createInnerAndClauses(entries[0])];
    } else {
      return [{
        bool: {
          filter: entries.map(function (entry) {
            return createInnerAndClauses(entry);
          })
        }
      }];
    }
  }
};
export var createOrClauses = function createOrClauses(exceptionItems) {
  return exceptionItems.flatMap(function (exceptionItem) {
    return buildExceptionItemFilter(exceptionItem);
  });
};
export var buildExceptionFilter = function buildExceptionFilter(_ref) {
  var lists = _ref.lists,
    excludeExceptions = _ref.excludeExceptions,
    chunkSize = _ref.chunkSize;
  // Remove exception items with large value lists. These are evaluated
  // elsewhere for the moment being.
  var exceptionsWithoutLargeValueLists = lists.filter(function (item) {
    return !hasLargeValueList(item.entries);
  });
  var exceptionFilter = {
    meta: {
      alias: null,
      disabled: false,
      negate: excludeExceptions
    },
    query: {
      bool: {
        should: undefined
      }
    }
  };
  if (exceptionsWithoutLargeValueLists.length === 0) {
    return undefined;
  } else if (exceptionsWithoutLargeValueLists.length <= chunkSize) {
    var clause = createOrClauses(exceptionsWithoutLargeValueLists);
    exceptionFilter.query.bool.should = clause;
    return exceptionFilter;
  } else {
    var chunks = chunkExceptions(exceptionsWithoutLargeValueLists, chunkSize);
    var filters = chunks.map(function (exceptionsChunk) {
      var orClauses = createOrClauses(exceptionsChunk);
      return {
        meta: {
          alias: null,
          disabled: false,
          negate: false
        },
        query: {
          bool: {
            should: orClauses
          }
        }
      };
    });
    var clauses = filters.map(function (_ref2) {
      var query = _ref2.query;
      return query;
    });
    return {
      meta: {
        alias: null,
        disabled: false,
        negate: excludeExceptions
      },
      query: {
        bool: {
          should: clauses
        }
      }
    };
  }
};
export var buildExclusionClause = function buildExclusionClause(booleanFilter) {
  return {
    bool: {
      must_not: booleanFilter
    }
  };
};
export var buildMatchClause = function buildMatchClause(entry) {
  var field = entry.field,
    operator = entry.operator,
    value = entry.value;
  var matchClause = {
    bool: {
      minimum_should_match: 1,
      should: [{
        match_phrase: _defineProperty({}, field, value)
      }]
    }
  };
  if (operator === 'excluded') {
    return buildExclusionClause(matchClause);
  } else {
    return matchClause;
  }
};
export var getBaseMatchAnyClause = function getBaseMatchAnyClause(entry) {
  var field = entry.field,
    value = entry.value;
  if (value.length === 1) {
    return {
      bool: {
        minimum_should_match: 1,
        should: [{
          match_phrase: _defineProperty({}, field, value[0])
        }]
      }
    };
  }
  return {
    bool: {
      minimum_should_match: 1,
      should: value.map(function (val) {
        return {
          bool: {
            minimum_should_match: 1,
            should: [{
              match_phrase: _defineProperty({}, field, val)
            }]
          }
        };
      })
    }
  };
};
export var buildMatchAnyClause = function buildMatchAnyClause(entry) {
  var operator = entry.operator;
  var matchAnyClause = getBaseMatchAnyClause(entry);
  if (operator === 'excluded') {
    return buildExclusionClause(matchAnyClause);
  } else {
    return matchAnyClause;
  }
};
export var buildExistsClause = function buildExistsClause(entry) {
  var field = entry.field,
    operator = entry.operator;
  var existsClause = {
    bool: {
      minimum_should_match: 1,
      should: [{
        exists: {
          field: field
        }
      }]
    }
  };
  if (operator === 'excluded') {
    return buildExclusionClause(existsClause);
  } else {
    return existsClause;
  }
};
var isBooleanFilter = function isBooleanFilter(clause) {
  var keys = Object.keys(clause);
  return keys.includes('bool') != null;
};
export var getBaseNestedClause = function getBaseNestedClause(entries, parentField) {
  if (entries.length === 1) {
    var _entries = _slicedToArray(entries, 1),
      singleNestedEntry = _entries[0];
    var innerClause = createInnerAndClauses(singleNestedEntry, parentField);
    return isBooleanFilter(innerClause) ? innerClause : {
      bool: {}
    };
  }
  return {
    bool: {
      filter: entries.map(function (nestedEntry) {
        return createInnerAndClauses(nestedEntry, parentField);
      })
    }
  };
};
export var buildNestedClause = function buildNestedClause(entry) {
  var field = entry.field,
    entries = entry.entries;
  var baseNestedClause = getBaseNestedClause(entries, field);
  return {
    nested: {
      path: field,
      query: baseNestedClause,
      score_mode: 'none'
    }
  };
};
export var createInnerAndClauses = function createInnerAndClauses(entry, parent) {
  if (entriesExists.is(entry)) {
    var field = parent != null ? "".concat(parent, ".").concat(entry.field) : entry.field;
    return buildExistsClause(_objectSpread(_objectSpread({}, entry), {}, {
      field: field
    }));
  } else if (entriesMatch.is(entry)) {
    var _field = parent != null ? "".concat(parent, ".").concat(entry.field) : entry.field;
    return buildMatchClause(_objectSpread(_objectSpread({}, entry), {}, {
      field: _field
    }));
  } else if (entriesMatchAny.is(entry)) {
    var _field2 = parent != null ? "".concat(parent, ".").concat(entry.field) : entry.field;
    return buildMatchAnyClause(_objectSpread(_objectSpread({}, entry), {}, {
      field: _field2
    }));
  } else if (entriesNested.is(entry)) {
    return buildNestedClause(entry);
  } else {
    throw new TypeError("Unexpected exception entry: ".concat(entry));
  }
};