import _defineProperty from "@babel/runtime/helpers/defineProperty";
import _objectSpread from "@babel/runtime/helpers/objectSpread2";
import _toConsumableArray from "@babel/runtime/helpers/toConsumableArray";
import _asyncToGenerator from "@babel/runtime/helpers/asyncToGenerator";
import _regeneratorRuntime from "@babel/runtime/regenerator";
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

import * as fs from 'fs';
import * as path from 'path';
import { promisify } from 'util';
import { unique } from './core/helper';
var TRANSLATION_FILE_EXTENSION = '.json';

/**
 * Internal property for storing registered translations paths.
 * Key is locale, value is array of registered paths
 */
var translationsRegistry = {};

/**
 * Internal property for caching loaded translations files.
 * Key is path to translation file, value is object with translation messages
 */
var loadedFiles = {};

/**
 * Returns locale by the given translation file name
 * @param fullFileName
 * @returns locale
 * @example
 * getLocaleFromFileName('./path/to/translation/ru.json') // => 'ru'
 */
function getLocaleFromFileName(fullFileName) {
  if (!fullFileName) {
    throw new Error('Filename is empty');
  }
  var fileExt = path.extname(fullFileName);
  if (fileExt !== TRANSLATION_FILE_EXTENSION) {
    throw new Error("Translations must have 'json' extension. File being registered is ".concat(fullFileName));
  }
  return path.basename(fullFileName, TRANSLATION_FILE_EXTENSION);
}

/**
 * Loads file and parses it as JSON
 * @param pathToFile
 * @returns
 */
function loadFile(_x) {
  return _loadFile.apply(this, arguments);
}
/**
 * Loads translations files and adds them into "loadedFiles" cache
 * @param files
 * @returns
 */
function _loadFile() {
  _loadFile = _asyncToGenerator( /*#__PURE__*/_regeneratorRuntime.mark(function _callee(pathToFile) {
    return _regeneratorRuntime.wrap(function _callee$(_context) {
      while (1) switch (_context.prev = _context.next) {
        case 0:
          _context.t0 = JSON;
          _context.next = 3;
          return promisify(fs.readFile)(pathToFile, 'utf8');
        case 3:
          _context.t1 = _context.sent;
          return _context.abrupt("return", _context.t0.parse.call(_context.t0, _context.t1));
        case 5:
        case "end":
          return _context.stop();
      }
    }, _callee);
  }));
  return _loadFile.apply(this, arguments);
}
function loadAndCacheFiles(_x2) {
  return _loadAndCacheFiles.apply(this, arguments);
}
/**
 * Registers translation file with i18n loader
 * @param translationFilePath - Absolute path to the translation file to register.
 */
function _loadAndCacheFiles() {
  _loadAndCacheFiles = _asyncToGenerator( /*#__PURE__*/_regeneratorRuntime.mark(function _callee2(files) {
    var translations;
    return _regeneratorRuntime.wrap(function _callee2$(_context2) {
      while (1) switch (_context2.prev = _context2.next) {
        case 0:
          _context2.next = 2;
          return Promise.all(files.map(loadFile));
        case 2:
          translations = _context2.sent;
          files.forEach(function (file, index) {
            loadedFiles[file] = translations[index];
          });
        case 4:
        case "end":
          return _context2.stop();
      }
    }, _callee2);
  }));
  return _loadAndCacheFiles.apply(this, arguments);
}
export function registerTranslationFile(translationFilePath) {
  if (!path.isAbsolute(translationFilePath)) {
    throw new TypeError('Paths to translation files must be absolute. ' + "Got relative path: \"".concat(translationFilePath, "\""));
  }
  var locale = getLocaleFromFileName(translationFilePath);
  translationsRegistry[locale] = unique([].concat(_toConsumableArray(translationsRegistry[locale] || []), [translationFilePath]));
}

/**
 * Registers array of translation files with i18n loader
 * @param arrayOfPaths - Array of absolute paths to the translation files to register.
 */
export function registerTranslationFiles() {
  var arrayOfPaths = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : [];
  arrayOfPaths.forEach(registerTranslationFile);
}

/**
 * Returns an array of locales that have been registered with i18n loader
 * @returns registeredTranslations
 */
export function getRegisteredLocales() {
  return Object.keys(translationsRegistry);
}

/**
 * Returns translation messages by specified locale
 * @param locale
 * @returns translation messages
 */
export function getTranslationsByLocale(_x3) {
  return _getTranslationsByLocale.apply(this, arguments);
}

/**
 * Returns all translations for registered locales
 * @returns A Promise object
 * where keys are the locale and values are objects of translation messages
 */
function _getTranslationsByLocale() {
  _getTranslationsByLocale = _asyncToGenerator( /*#__PURE__*/_regeneratorRuntime.mark(function _callee3(locale) {
    var files, notLoadedFiles;
    return _regeneratorRuntime.wrap(function _callee3$(_context3) {
      while (1) switch (_context3.prev = _context3.next) {
        case 0:
          files = translationsRegistry[locale] || [];
          notLoadedFiles = files.filter(function (file) {
            return !loadedFiles[file];
          });
          if (!notLoadedFiles.length) {
            _context3.next = 5;
            break;
          }
          _context3.next = 5;
          return loadAndCacheFiles(notLoadedFiles);
        case 5:
          if (files.length) {
            _context3.next = 7;
            break;
          }
          return _context3.abrupt("return", {
            messages: {}
          });
        case 7:
          return _context3.abrupt("return", files.reduce(function (translation, file) {
            return {
              locale: loadedFiles[file].locale || translation.locale,
              formats: loadedFiles[file].formats || translation.formats,
              messages: _objectSpread(_objectSpread({}, loadedFiles[file].messages), translation.messages)
            };
          }, {
            locale: locale,
            messages: {}
          }));
        case 8:
        case "end":
          return _context3.stop();
      }
    }, _callee3);
  }));
  return _getTranslationsByLocale.apply(this, arguments);
}
export function getAllTranslations() {
  return _getAllTranslations.apply(this, arguments);
}

/**
 * Registers passed translations files, loads them and returns promise with
 * all translation messages
 * @param paths - Array of absolute paths to the translation files
 * @returns A Promise object where
 * keys are the locale and values are objects of translation messages
 */
function _getAllTranslations() {
  _getAllTranslations = _asyncToGenerator( /*#__PURE__*/_regeneratorRuntime.mark(function _callee4() {
    var locales, translations;
    return _regeneratorRuntime.wrap(function _callee4$(_context4) {
      while (1) switch (_context4.prev = _context4.next) {
        case 0:
          locales = getRegisteredLocales();
          _context4.next = 3;
          return Promise.all(locales.map(getTranslationsByLocale));
        case 3:
          translations = _context4.sent;
          return _context4.abrupt("return", locales.reduce(function (acc, locale, index) {
            return _objectSpread(_objectSpread({}, acc), {}, _defineProperty({}, locale, translations[index]));
          }, {}));
        case 5:
        case "end":
          return _context4.stop();
      }
    }, _callee4);
  }));
  return _getAllTranslations.apply(this, arguments);
}
export function getAllTranslationsFromPaths(_x4) {
  return _getAllTranslationsFromPaths.apply(this, arguments);
}
function _getAllTranslationsFromPaths() {
  _getAllTranslationsFromPaths = _asyncToGenerator( /*#__PURE__*/_regeneratorRuntime.mark(function _callee5(paths) {
    return _regeneratorRuntime.wrap(function _callee5$(_context5) {
      while (1) switch (_context5.prev = _context5.next) {
        case 0:
          registerTranslationFiles(paths);
          _context5.next = 3;
          return getAllTranslations();
        case 3:
          return _context5.abrupt("return", _context5.sent);
        case 4:
        case "end":
          return _context5.stop();
      }
    }, _callee5);
  }));
  return _getAllTranslationsFromPaths.apply(this, arguments);
}