//
// Syd: rock-solid application kernel
// src/workers/ipc.rs: `syd_ipc' remote-configuration thread
//
// Copyright (c) 2025 Ali Polatel <alip@chesswob.org>
//
// SPDX-License-Identifier: GPL-3.0

use std::{
    borrow::Cow,
    collections::HashMap,
    hash::{Hash, Hasher},
    io::{IoSlice, IoSliceMut},
    os::{
        fd::{AsFd, AsRawFd, FromRawFd, OwnedFd, RawFd},
        unix::net::{UnixListener, UnixStream},
    },
    sync::{Arc, RwLock},
    thread,
    thread::JoinHandle,
};

use libseccomp::{scmp_cmp, ScmpAction, ScmpFilterContext, ScmpSyscall};
use memchr::{arch::all::is_equal, memchr};
use nix::{
    cmsg_space,
    errno::Errno,
    fcntl::OFlag,
    poll::PollTimeout,
    sys::{
        epoll::{Epoll, EpollCreateFlags, EpollEvent, EpollFlags},
        socket::{
            accept4, bind, listen, recvmsg, sendmsg, setsockopt, socket, sockopt::PassCred,
            AddressFamily, Backlog, ControlMessageOwned, MsgFlags, SockFlag, SockType, UnixAddr,
            UnixCredentials,
        },
        stat::{umask, Mode},
    },
    unistd::{Gid, Uid},
};

use crate::{
    compat::{epoll_ctl_mod_safe, epoll_ctl_safe},
    config::*,
    err::{err2no, SydResult},
    fs::duprand,
    hash::SydRandomState,
    info,
    landlock_policy::LandlockPolicy,
    path::{XPath, XPathBuf},
    sandbox::{Flags, Sandbox, LINE_MAX},
};

const IPC_ACK: &[u8] = b"{\"err\":0,\"msg\":\"ACK\"}\n";
const IPC_AUTH: &[u8] = b"{\"err\":13,\"msg\":\"AUTH\"}\n";
const IPC_RATE: &[u8] = b"{\"err\":7,\"msg\":\"RATE\"}\n";
const IPC_PINK: &[u8] = b"{\"err\":0,\
\"msg\":\"Change return success. Going and coming without error. Action brings good fortune.\"}\n";
const IPC_PONG: &[u8] = b"{\"err\":0,\"msg\":\"PONG\"}\n";

// Epoll-based, single-threaded IPC server over a UNIX socket.
pub(crate) struct IpcWorker {
    // Path to the UNIX socket (possibly abstract if it starts with '@')
    //
    // This is set to `None` once the bind is successful.
    addr: Option<XPathBuf>,
    // Epoll file descriptor
    epoll: Option<Epoll>,
    // Listener socket
    sock: Option<UnixListener>,
    // Reference to the Sandbox to be configured
    sandbox: Option<Arc<RwLock<Sandbox>>>,
    // Sandbox flags specified at startup.
    flags: Flags,
    // Credentials for authentication.
    creds: (Option<Uid>, Option<Gid>),
}

// Prompt modes supported (similar to HAProxy)
#[derive(Clone, Copy, Debug, PartialEq, Eq)]
enum PromptMode {
    // Non-interactive: single command, then close.
    NonInteractive,
    // Interactive: accept multiple commands, no prompt.
    Interactive,
    // Prompt: accept multiple commands and send prompt "; " before each.
    Prompt,
}

/// Per-connection state: buffering incoming bytes,
/// pending outgoing bytes, and tracking mode.
struct Connection {
    /// Underlying UNIX-stream socket.
    stream: UnixStream,
    /// Buffered incoming bytes until a full line arrives.
    buf: Vec<u8>,
    /// Buffer of bytes to write (response + prompt).
    write_buf: Vec<u8>,
    /// Current position in `write_buf`.
    write_pos: usize,
    /// Prompt mode for this connection.
    mode: PromptMode,
    /// Whether to close after flushing the write buffer.
    should_close: bool,
}

// Type for Connection map.
// We use HashMap because HashSet does not have get_mut!
type ConnectionMap = HashMap<RawFd, Connection, SydRandomState>;

impl Hash for Connection {
    fn hash<H: Hasher>(&self, state: &mut H) {
        self.stream.as_fd().as_raw_fd().hash(state);
    }
}

impl PartialEq for Connection {
    fn eq(&self, other: &Self) -> bool {
        self.stream.as_fd().as_raw_fd() == other.stream.as_fd().as_raw_fd()
    }
}

impl Eq for Connection {}

impl Connection {
    fn new(stream: UnixStream) -> Self {
        Self {
            stream,
            buf: Vec::with_capacity(1024),
            write_buf: Vec::with_capacity(1024),
            write_pos: 0,
            mode: PromptMode::NonInteractive,
            should_close: false,
        }
    }

    // Change epoll(7) interests.
    fn ctl(&self, epoll: &Epoll, flags: EpollFlags) -> Result<(), Errno> {
        let fd = self.stream.as_raw_fd();

        #[allow(clippy::cast_sign_loss)]
        let event = libc::epoll_event {
            events: flags.bits() as u32,
            u64: fd as u64,
        };

        epoll_ctl_mod_safe(&epoll.0, fd, event)
    }

    // Append data to the write buffer.
    #[inline]
    fn enqueue_response(&mut self, data: &[u8]) {
        self.write_buf.extend_from_slice(data);
    }

    // Enqueue the prompt string ("; ") in `write_buf`.
    #[inline]
    fn enqueue_prompt(&mut self) {
        self.write_buf.extend_from_slice(b"; ");
    }
}

impl IpcWorker {
    // Create a new IpcWorker for the given path.
    //
    // If `addr` starts with '@', an abstract UNIX socket is used.
    //
    // This does not bind or listen yet; only stores the path.
    // Returns immediately without error.
    pub(crate) fn new(addr: &XPath, uid: Option<Uid>, gid: Option<Gid>, flags: Flags) -> Self {
        IpcWorker {
            flags,
            creds: (uid, gid),
            addr: Some(addr.to_owned()),
            epoll: None,
            sock: None,
            sandbox: None,
        }
    }

    // Create the listener socket, bind it, make it non-blocking, listen, and create epoll.
    //
    // This function sets umask(2) and therefore may not be thread-safe!
    pub(crate) fn setup(&mut self) -> Result<(), Errno> {
        // Prepare UNIX socket address, `@' prefix implies abstract socket.
        let addr = self.addr.as_ref().ok_or(Errno::EADDRNOTAVAIL)?;
        let addr = if matches!(addr.first(), Some(b'@')) {
            UnixAddr::new_abstract(&addr.as_bytes()[1..])?
        } else {
            UnixAddr::new(addr.as_bytes())?
        };
        self.addr = None; // No longer needed.

        // Create UNIX stream socket, set to non-blocking.
        let sock = socket(
            AddressFamily::Unix,
            SockType::Stream,
            SockFlag::SOCK_CLOEXEC | SockFlag::SOCK_NONBLOCK,
            None,
        )?;

        // SAFETY: Randomize to make fd reuse harder.
        let sock_fd = duprand(sock.as_raw_fd(), OFlag::O_CLOEXEC).map(|fd| {
            // SAFETY: Used to prevent leaks in sandbox process:
            let mut fd_str = itoa::Buffer::new();
            let fd_str = fd_str.format(fd);
            std::env::set_var(ENV_IPC_UNIX_FD, fd_str);

            // SAFETY: duprand returns a valid FD on success.
            unsafe { OwnedFd::from_raw_fd(fd) }
        })?;
        drop(sock);
        let sock = sock_fd;

        // Set SO_PASSCRED for authentication.
        setsockopt(&sock, PassCred, &true)?;

        // SAFETY: Ensure socket is created with sane permissions.
        let umask_orig = umask(Mode::from_bits_truncate(0o077));
        let result = bind(sock.as_raw_fd(), &addr);
        umask(umask_orig);
        result?;

        listen(&sock, Backlog::MAXCONN)?;

        // Create epoll instance.
        let epoll = Epoll::new(EpollCreateFlags::EPOLL_CLOEXEC)?;

        // SAFETY: Randomize the epoll fd to make fd reuse harder.
        let epoll_fd = duprand(epoll.0.as_raw_fd(), OFlag::O_CLOEXEC).map(|fd| {
            // SAFETY: Used to prevent leaks in sandbox process:
            let mut fd_str = itoa::Buffer::new();
            let fd_str = fd_str.format(fd);
            std::env::set_var(ENV_IPC_POLL_FD, fd_str);

            // SAFETY: duprand returns a valid FD on success.
            unsafe { OwnedFd::from_raw_fd(fd) }
        })?;
        drop(epoll);
        let epoll = Epoll(epoll_fd);

        // Register listener_fd for EPOLLIN.
        #[allow(clippy::cast_sign_loss)]
        let event = libc::epoll_event {
            events: EpollFlags::EPOLLIN.bits() as u32,
            u64: sock.as_fd().as_raw_fd() as u64,
        };
        epoll_ctl_safe(&epoll.0, sock.as_fd().as_raw_fd(), Some(event))?;

        self.epoll = Some(epoll);
        self.sock = Some(UnixListener::from(sock));

        Ok(())
    }

    // Set reference to the Sandbox to be configured.
    pub fn set_sandbox(&mut self, sandbox: Arc<RwLock<Sandbox>>) {
        self.sandbox = Some(sandbox);
    }

    // Confine IPC thread.
    #[allow(clippy::cognitive_complexity)]
    pub(crate) fn prepare_confine(
        epoll_fd: RawFd,
        flags: Flags,
        dry_run: bool,
    ) -> SydResult<ScmpFilterContext> {
        if !dry_run {
            // SAFETY: Set up a Landlock sandbox to disallow all access.
            // We do not enable scoped signals not to break inter-thread
            // signaling. This was fixed in recent versions of Landlock.
            // TODO: Make LandlockPolicy use scoped signals only if fixed.
            let abi = crate::landlock::ABI::new_current();
            let policy = LandlockPolicy {
                scoped_abs: true,
                ..Default::default()
            };
            let _ = policy.restrict_self(abi);
        }

        // Create seccomp filter with default action.
        let mut ctx = ScmpFilterContext::new(ScmpAction::KillProcess)?;

        // Enforce the NO_NEW_PRIVS functionality before
        // loading the seccomp filter into the kernel.
        ctx.set_ctl_nnp(true)?;

        // DO NOT synchronize filter to all threads.
        // Other threads will self-confine.
        ctx.set_ctl_tsync(false)?;

        // Disable Speculative Store Bypass mitigations
        // with trace/allow_unsafe_spec_exec:1
        ctx.set_ctl_ssb(flags.contains(Flags::FL_ALLOW_UNSAFE_SPEC_EXEC))?;

        // We kill for bad system call and bad arch.
        ctx.set_act_badarch(ScmpAction::KillProcess)?;

        // Use a binary tree sorted by syscall number if possible.
        let _ = ctx.set_ctl_optimize(2);

        // SAFETY: Do NOT add supported architectures to the filter.
        // This ensures Syd can never run a non-native system call,
        // which we do not need at all.
        // seccomp_add_architectures(&mut ctx)?;

        // Allow epoll(7) API to our single epoll fd only.
        #[allow(clippy::cast_sign_loss)]
        for sysname in EPOLL_SYSCALLS {
            match ScmpSyscall::from_name(sysname) {
                Ok(syscall) => {
                    ctx.add_rule_conditional(
                        ScmpAction::Allow,
                        syscall,
                        &[scmp_cmp!($arg0 == epoll_fd as u64)],
                    )?;
                }
                Err(_) => {
                    info!("ctx": "confine", "op": "allow_syscall",
                        "msg": format!("invalid or unsupported syscall {sysname}"));
                }
            }
        }

        // Deny open and stat family with ENOSYS rather than KillProcess.
        // We need this because Rust allocator has side effects such as
        // opening /proc/sys/vm/overcommit_memory on some architectures.
        for sysname in ["open", "openat", "stat", "lstat", "statx", "newfstatat"] {
            match ScmpSyscall::from_name(sysname) {
                Ok(syscall) => {
                    ctx.add_rule(ScmpAction::Errno(Errno::ENOSYS as i32), syscall)?;
                }
                Err(_) => {
                    info!("ctx": "confine", "op": "allow_syscall",
                        "msg": format!("invalid or unsupported syscall {sysname}"));
                }
            }
        }

        // Allow safe fcntl(2) utility calls.
        for sysname in ["fcntl", "fcntl64"] {
            let syscall = match ScmpSyscall::from_name(sysname) {
                Ok(syscall) => syscall,
                Err(_) => {
                    info!("ctx": "confine", "op": "allow_syscall",
                        "msg": format!("invalid or unsupported syscall {sysname}"));
                    continue;
                }
            };

            for op in IPC_FCNTL_OPS {
                ctx.add_rule_conditional(ScmpAction::Allow, syscall, &[scmp_cmp!($arg1 == *op)])?;
            }
        }

        // Allow safe system calls.
        for sysname in IPC_SYSCALLS {
            match ScmpSyscall::from_name(sysname) {
                Ok(syscall) => {
                    ctx.add_rule(ScmpAction::Allow, syscall)?;
                }
                Err(_) => {
                    info!("ctx": "confine", "op": "allow_syscall",
                        "msg": format!("invalid or unsupported syscall {sysname}"));
                }
            }
        }

        // Allow futex system calls.
        for sysname in FUTEX_SYSCALLS {
            match ScmpSyscall::from_name(sysname) {
                Ok(syscall) => {
                    ctx.add_rule(ScmpAction::Allow, syscall)?;
                }
                Err(_) => {
                    info!("ctx": "confine", "op": "allow_syscall",
                        "msg": format!("invalid or unsupported syscall {sysname}"));
                }
            }
        }

        // Allow getid system calls.
        for sysname in GET_ID_SYSCALLS {
            match ScmpSyscall::from_name(sysname) {
                Ok(syscall) => {
                    ctx.add_rule(ScmpAction::Allow, syscall)?;
                }
                Err(_) => {
                    info!("ctx": "confine", "op": "allow_syscall",
                        "msg": format!("invalid or unsupported syscall {sysname}"));
                }
            }
        }

        if flags.intersects(Flags::FL_ALLOW_SAFE_SETUID | Flags::FL_ALLOW_SAFE_SETGID) {
            // SAFETY: Main thread confines these further.
            // As these system calls as per-process,
            // the main thread's seccomp rules will apply
            // to us even without TSYNC.
            for sysname in SET_ID_SYSCALLS {
                match ScmpSyscall::from_name(sysname) {
                    Ok(syscall) => {
                        ctx.add_rule(ScmpAction::Allow, syscall)?;
                    }
                    Err(_) => {
                        info!("ctx": "confine", "op": "allow_syscall",
                            "msg": format!("invalid or unsupported syscall {sysname}"));
                    }
                }
            }

            // SAFETY:
            // Signal system calls are necessary to handle reserved signals.
            for sysname in ["sigreturn", "rt_sigreturn"] {
                match ScmpSyscall::from_name(sysname) {
                    Ok(syscall) => {
                        ctx.add_rule(ScmpAction::Allow, syscall)?;
                    }
                    Err(_) => {
                        info!("ctx": "confine", "op": "allow_syscall",
                            "msg": format!("invalid or unsupported syscall {sysname}"));
                    }
                }
            }
        }

        Ok(ctx)
    }

    // Spawn the IPC worker in a new thread named `syd_ipc'.
    pub fn try_spawn(self) -> Result<JoinHandle<()>, Errno> {
        thread::Builder::new()
            .name("syd_ipc".into())
            .stack_size(IPC_STACK_SIZE)
            .spawn(move || {
                let _ = self.main();
            })
            .map_err(|err| err2no(&err))
    }

    /// Main loop: waits on epoll, accepts new connections, reads commands, and writes responses.
    fn main(&self) -> Result<(), Errno> {
        let epoll = self.epoll.as_ref().ok_or(Errno::EADDRNOTAVAIL)?;
        let sock = self.sock.as_ref().ok_or(Errno::EADDRNOTAVAIL)?;
        let sock_fd = sock.as_fd().as_raw_fd();

        // Confine `syd_ipc' thread.
        // SAFETY: Panic if we cannot confine the thread.
        #[allow(clippy::disallowed_methods)]
        let ctx = Self::prepare_confine(epoll.0.as_raw_fd(), self.flags, false)
            .expect("prepare confine ipc thread");
        #[allow(clippy::disallowed_methods)]
        ctx.load().expect("confine ipc thread");

        // Set of active connections.
        let mut connections = ConnectionMap::default();

        // Buffer for epoll events.
        let mut events = [EpollEvent::empty(); 128];

        // Wait for events and handle EINTR.
        loop {
            // Wait for events, block indefinitely.
            let nfds = match epoll.wait(&mut events, PollTimeout::NONE) {
                Ok(n) => n,
                Err(Errno::EINTR) => continue,
                Err(errno) => return Err(errno),
            };

            for ev in events.iter().take(nfds) {
                #[allow(clippy::cast_possible_truncation)]
                let fd = ev.data() as RawFd;
                let flags = ev.events();

                if fd == sock_fd {
                    // New incoming connection(s).
                    self.accept_new_connections(&mut connections)?;
                } else {
                    // Existing connection.
                    if flags.contains(EpollFlags::EPOLLIN) {
                        self.handle_readable(&mut connections, fd)?;
                    }
                    if flags.contains(EpollFlags::EPOLLOUT) {
                        self.handle_writable(&mut connections, fd)?;
                    }
                    if flags.intersects(
                        EpollFlags::EPOLLERR | EpollFlags::EPOLLHUP | EpollFlags::EPOLLRDHUP,
                    ) {
                        // Error or hang-up: close connection.
                        self.close_connection(&mut connections, fd)?;
                    }
                }
            }
        }
    }

    /// Accept new connections, set them non-blocking, register with epoll.
    fn accept_new_connections(&self, connections: &mut ConnectionMap) -> Result<(), Errno> {
        let epoll = self.epoll.as_ref().ok_or(Errno::EADDRNOTAVAIL)?;
        let sock = self.sock.as_ref().ok_or(Errno::EADDRNOTAVAIL)?;

        loop {
            #[allow(unreachable_patterns)]
            match accept4(
                sock.as_raw_fd(),
                SockFlag::SOCK_CLOEXEC | SockFlag::SOCK_NONBLOCK,
            )
            .map(|fd| {
                // SAFETY: accept4 returns a valid FD on success.
                UnixStream::from(unsafe { OwnedFd::from_raw_fd(fd) })
            }) {
                Ok(stream) => {
                    // Register the new socket with epoll for reading.
                    let fd = stream.as_fd().as_raw_fd();
                    #[allow(clippy::cast_sign_loss)]
                    let event = libc::epoll_event {
                        events: EpollFlags::EPOLLIN.bits() as u32,
                        u64: fd as u64,
                    };
                    epoll_ctl_safe(&epoll.0, fd, Some(event))?;

                    // Create Connection state.
                    let mut conn = Connection::new(stream);

                    // In prompt mode, send initial prompt.
                    if conn.mode == PromptMode::Prompt {
                        conn.enqueue_prompt();
                    }
                    connections.insert(fd, conn);
                }
                Err(Errno::EAGAIN | Errno::EWOULDBLOCK) => {
                    // No more pending connections.
                    break;
                }
                Err(errno) => {
                    // Unexpected error.
                    return Err(errno);
                }
            }
        }

        Ok(())
    }

    /// Handle a readable event on connection `fd`.
    fn handle_readable(&self, connections: &mut ConnectionMap, fd: RawFd) -> Result<(), Errno> {
        // Lookup connection by fd.
        // We need a mutable reference; use get_mut.
        if let Some(conn) = connections.get_mut(&fd) {
            loop {
                let mut buf = [0u8; LINE_MAX];
                let mut iov = [IoSliceMut::new(&mut buf)];
                let mut cmsg = cmsg_space!(UnixCredentials);

                #[allow(clippy::arithmetic_side_effects)]
                #[allow(unreachable_patterns)]
                match recvmsg::<()>(
                    conn.stream.as_raw_fd(),
                    &mut iov,
                    Some(&mut cmsg),
                    MsgFlags::MSG_DONTWAIT,
                ) {
                    Ok(msg) if msg.bytes == 0 => {
                        // EOF on read side: if there's a pending write,
                        // defer closing until after flush.
                        if conn.write_buf.is_empty() {
                            // no response queued => close immediately.
                            self.close_connection(connections, fd)?;
                        } else {
                            // response pending => mark to close after writing.
                            conn.should_close = true;
                        }
                        return Ok(());
                    }
                    Ok(msg) if conn.buf.len() + msg.bytes >= LINE_MAX => {
                        // Input too large:
                        // 1. Add EPOLLOUT to interests.
                        // 2. Reject with error message.
                        // 3. Close connection after reply.
                        let epoll = self.epoll.as_ref().ok_or(Errno::EADDRNOTAVAIL)?;
                        conn.ctl(&epoll, EpollFlags::EPOLLOUT)?;
                        conn.enqueue_response(IPC_RATE);
                        conn.should_close = true;

                        return Ok(());
                    }
                    Ok(msg) => {
                        // Authenticate each and every message.
                        let mut auth = false;
                        if let Ok(cmsgs) = msg.cmsgs() {
                            for cmsg in cmsgs {
                                let creds = if let ControlMessageOwned::ScmCredentials(creds) = cmsg
                                {
                                    creds
                                } else {
                                    continue;
                                };

                                if creds.pid() == 0 {
                                    // Invalid credentials, skip.
                                    continue;
                                }

                                if creds.uid() == 0 && creds.gid() == 0 {
                                    // Matched UID and GID for ROOT: Authenticated.
                                    auth = true;
                                    break;
                                }

                                if let Some(uid) = self.creds.0 {
                                    if creds.uid() != uid.as_raw() {
                                        // UID mismatch.
                                        continue;
                                    }
                                }

                                if let Some(gid) = self.creds.1 {
                                    if creds.gid() != gid.as_raw() {
                                        // GID mismatch.
                                        continue;
                                    }
                                }

                                // Matched UID and GID: Authenticated.
                                auth = true;
                                break;
                            }
                        }

                        if !auth {
                            // Authentication failed:
                            // 1. Add EPOLLOUT to interests.
                            // 2. Reject with error message.
                            // 3. Close connection after reply.
                            let epoll = self.epoll.as_ref().ok_or(Errno::EADDRNOTAVAIL)?;
                            conn.ctl(&epoll, EpollFlags::EPOLLOUT)?;
                            conn.enqueue_response(IPC_AUTH);
                            conn.should_close = true;

                            return Ok(());
                        }

                        // Extend buffer with the message content.
                        for iov in msg.iovs() {
                            conn.buf.extend_from_slice(iov);
                        }

                        // Process any full lines in buffer.
                        while let Some(pos) = memchr(b'\n', &conn.buf) {
                            // Handle the line (no allocations needed here).
                            self.process_command(conn, pos)?;

                            // Remove the line *and* the '\n' from the buffer.
                            conn.buf.drain(..=pos);
                        }
                    }
                    Err(Errno::EINTR) => continue,
                    Err(Errno::EAGAIN | Errno::EWOULDBLOCK) => {
                        // No more data.
                        break;
                    }
                    Err(_) => {
                        // Other errors -> close conection.
                        self.close_connection(connections, fd)?;
                        return Ok(());
                    }
                }
            }
        }

        Ok(())
    }

    /// Handle a writable event on connection `fd` (flush pending writes).
    fn handle_writable(&self, connections: &mut ConnectionMap, fd: RawFd) -> Result<(), Errno> {
        let epoll = self.epoll.as_ref().ok_or(Errno::EADDRNOTAVAIL)?;

        if let Some(conn) = connections.get_mut(&fd) {
            while conn.write_pos < conn.write_buf.len() {
                let iov = [IoSlice::new(&conn.write_buf[conn.write_pos..])];

                #[allow(clippy::arithmetic_side_effects)]
                #[allow(unreachable_patterns)]
                match sendmsg::<()>(
                    conn.stream.as_raw_fd(),
                    &iov,
                    &[],
                    MsgFlags::MSG_DONTWAIT,
                    None,
                ) {
                    Ok(0) => {
                        // Would block or closed; stop for now.
                        break;
                    }
                    Ok(n) => {
                        conn.write_pos += n;
                    }
                    Err(Errno::EINTR) => continue,
                    Err(Errno::EAGAIN | Errno::EWOULDBLOCK) => {
                        // Can't write more right now.
                        break;
                    }
                    Err(_) => {
                        // Fatal write error; close.
                        self.close_connection(connections, fd)?;
                        return Ok(());
                    }
                }
            }

            if conn.write_pos >= conn.write_buf.len() {
                // All data written: clear buffer and disable EPOLLOUT.
                conn.ctl(&epoll, EpollFlags::EPOLLIN)?;
                conn.write_buf.clear();
                conn.write_pos = 0;

                // If mode is NonInteractive and the last command closed the connection,
                // we close the connection here.
                if conn.should_close {
                    self.close_connection(connections, fd)?;
                }
            }
        }

        Ok(())
    }

    // Process at `pos` boundary from connection `conn` on `fd`.
    fn process_command(&self, conn: &mut Connection, pos: usize) -> Result<(), Errno> {
        let epoll = self.epoll.as_ref().ok_or(Errno::EADDRNOTAVAIL)?;

        // Convert to UTF-8, errors are fatal.
        let cmd = String::from_utf8(conn.buf[..pos].to_vec()).or(Err(Errno::EINVAL))?;

        // Split on ASCII whitespace.
        let mut parts = cmd.split_ascii_whitespace();

        match parts.next() {
            Some("prompt") => {
                // Change mode.
                match parts.next() {
                    Some("n") => {
                        conn.mode = PromptMode::NonInteractive;
                        conn.should_close = true;
                    }
                    Some("i") => {
                        conn.mode = PromptMode::Interactive;
                    }
                    Some("p") => {
                        conn.mode = PromptMode::Prompt;
                        conn.enqueue_prompt();
                    }
                    None => {
                        // Toggle:
                        // 1. From NonInteractive → Prompt;
                        // 2. From Interactive → Prompt;
                        // 3. From Prompt → Interactive.
                        match conn.mode {
                            PromptMode::NonInteractive => {
                                conn.mode = PromptMode::Prompt;
                                conn.enqueue_prompt();
                            }
                            PromptMode::Interactive => {
                                conn.mode = PromptMode::Prompt;
                                conn.enqueue_prompt();
                            }
                            PromptMode::Prompt => {
                                conn.mode = PromptMode::Interactive;
                            }
                        }
                    }
                    _ => {
                        conn.enqueue_response(b"Unknown prompt mode!\n");
                        if conn.mode == PromptMode::Prompt {
                            conn.enqueue_prompt();
                        }
                    }
                }
            }
            Some("quit") | Some("exit") => conn.should_close = true,
            Some(part) => {
                let response = if part == "ping" {
                    // Alive check, send pong to ping.
                    Cow::Borrowed(IPC_PONG)
                } else if part == "pink" {
                    // Pink Floyd check, reply quote.
                    Cow::Borrowed(IPC_PINK)
                } else if part == "version" {
                    Cow::Owned(format!(
            "{{\"major\":{API_MAJOR_VERSION},\"minor\":{IPC_MINOR_VERSION},\"version\":\"{API_MAJOR_VERSION}.{IPC_MINOR_VERSION}\"}}\n").into_bytes())
                } else {
                    // Regular command: pass to handle_cmd.
                    Cow::Owned(self.handle_cmd(&cmd)?)
                };
                conn.enqueue_response(&response);
                if conn.mode == PromptMode::Prompt {
                    conn.enqueue_prompt();
                } else if conn.mode == PromptMode::NonInteractive {
                    conn.should_close = true;
                }
            }
            None => {
                // Empty line: just send prompt if in prompt mode.
                if conn.mode == PromptMode::Prompt {
                    conn.enqueue_prompt();
                }
            }
        }

        // After enqueueing data, ensure EPOLLOUT is enabled.
        if !conn.write_buf.is_empty() {
            conn.ctl(&epoll, EpollFlags::EPOLLIN | EpollFlags::EPOLLOUT)?;
        }

        Ok(())
    }

    // Close and clean up connection `fd`.
    fn close_connection(&self, connections: &mut ConnectionMap, fd: RawFd) -> Result<(), Errno> {
        let epoll = self.epoll.as_ref().ok_or(Errno::EADDRNOTAVAIL)?;

        // Unregister from epoll.
        epoll_ctl_safe(&epoll.0, fd, None)?;

        // Remove from set (drops the stream).
        connections.remove(&fd);

        Ok(())
    }

    // handle_cmd: process a single command string,
    // and return a response buffer.
    fn handle_cmd(&self, cmd: &str) -> Result<Vec<u8>, Errno> {
        // Lock sandbox for write.
        let sandbox = self.sandbox.as_ref().ok_or(Errno::EOWNERDEAD)?;
        let mut sandbox = sandbox.write().unwrap_or_else(|err| err.into_inner());

        // Close connection immediately if sandbox is locked.
        // This also results in thread-exit.
        if sandbox.locked() {
            return Err(Errno::EBUSY);
        }

        // Intercept display commands and send to socket
        // instead of Syd's standard error:
        // stat: Prints Sandbox in pretty-printed JSON.
        // stats: Prints Sandbox in human-readable format.
        if is_equal(cmd.as_bytes(), b"stat") {
            return Ok(serde_json::to_string(&*sandbox)
                .or(Err(Errno::EINVAL))?
                .into_bytes());
        } else if is_equal(cmd.as_bytes(), b"stats") {
            return Ok(format!("{sandbox}").into_bytes());
        }

        // We use config_unchecked to skip logging,
        // and environment initialization.
        let result = sandbox.config_unchecked(cmd);

        // Close connection immediately if sandbox got locked.
        // This also results in thread-exit.
        if sandbox.locked() {
            return Err(Errno::EBUSY);
        }

        match result {
            Ok(()) => Ok(IPC_ACK.to_vec()),
            Err(errno) => {
                Ok(format!("{{\"err\":{},\"msg\":\"{errno}\"}}\n", errno as i32).into_bytes())
            }
        }
    }
}
