//
// Syd: rock-solid application kernel
// src/kernel/prctl.rs: prctl(2) handler
//
// Copyright (c) 2023, 2024, 2025 Ali Polatel <alip@chesswob.org>
//
// SPDX-License-Identifier: GPL-3.0

use libseccomp::ScmpNotifResp;
use memchr::memchr;
use nix::errno::Errno;

use crate::{
    config::MMAP_MIN_ADDR, hook::UNotifyEventRequest, log::log_untrusted_buf, log_enabled,
    path::XPath, proc::proc_comm, sandbox::Capability, syslog::LogLevel, warn,
};

#[allow(clippy::cognitive_complexity)]
pub(crate) fn sys_prctl(request: UNotifyEventRequest) -> ScmpNotifResp {
    // Note, we only hook into the PR_SET_NAME request.
    let req = request.scmpreq;

    // SAFETY: Check pointer against mmap_min_addr.
    let ptr = req.data.args[1];
    if ptr < *MMAP_MIN_ADDR {
        return request.fail_syscall(Errno::EFAULT);
    }

    // Check if logging is enabled.
    if !log_enabled!(LogLevel::Warn) {
        return request.return_syscall(0);
    }

    // `!proc/name` is a dummy path we use
    // to disable logging, use e.g.
    // `filter/read+!proc/name'.
    let sandbox = request.get_sandbox();
    let verbose = sandbox.verbose;
    if sandbox.filter_path(Capability::CAP_READ, XPath::from_bytes(b"!proc/name")) {
        return request.return_syscall(0);
    }
    drop(sandbox); // release the read-lock.

    let mut buf = [0u8; 15];
    let name = match request.read_mem(&mut buf, ptr) {
        Ok(len) => {
            let nil = memchr(0, &buf[..len]).unwrap_or(len);
            &buf[..nil]
        }
        Err(err) => return request.fail_syscall(err),
    };

    // See if this is a request for change,
    // silently deny if no change was attempted.
    match proc_comm(req.pid()) {
        Ok(comm) if comm.is_equal(name) => {}
        Ok(comm) => {
            let (name, hex) = log_untrusted_buf(name);
            if verbose {
                warn!("ctx": "change_process_name",
                    "msg": format!("attempt to change process name from `{comm}' to `{name}' prevented"),
                    "tip": "use filter/read+!proc/name to silence, trace/allow_unsafe_prctl:1 to allow",
                    "sys": request.syscall, "name": name, "hex": hex, "comm": comm, "pid": req.pid,
                    "req": &request);
            } else {
                warn!("ctx": "change_process_name",
                    "msg": format!("attempt to change process name from `{comm}' to `{name}' prevented"),
                    "tip": "use filter/read+!proc/name to silence, trace/allow_unsafe_prctl:1 to allow",
                    "sys": request.syscall, "name": name, "hex": hex, "comm": comm, "pid": req.pid,
                    "pid": request.scmpreq.pid);
            }
        }
        Err(_) => {
            let (name, hex) = log_untrusted_buf(name);
            if verbose {
                warn!("ctx": "change_process_name",
                    "msg": format!("attempt to change process name to `{name}' prevented"),
                    "tip": "use filter/read+!proc/name to silence, trace/allow_unsafe_prctl:1 to allow",
                    "sys": request.syscall, "name": name, "hex": hex, "pid": req.pid,
                    "req": &request);
            } else {
                warn!("ctx": "change_process_name",
                    "msg": format!("attempt to change process name to `{name}' prevented"),
                    "tip": "use filter/read+!proc/name to silence, trace/allow_unsafe_prctl:1 to allow",
                    "sys": request.syscall, "name": name, "hex": hex, "pid": req.pid);
            }
        }
    }

    request.return_syscall(0)
}
