//
// Syd: rock-solid application kernel
// src/kernel/access.rs: access(2), faccessat(2) and faccessat2(2) handlers
//
// Copyright (c) 2023, 2024, 2025 Ali Polatel <alip@chesswob.org>
//
// SPDX-License-Identifier: GPL-3.0

use std::os::fd::AsRawFd;

use libseccomp::ScmpNotifResp;
use nix::{errno::Errno, fcntl::AtFlags, unistd::AccessFlags, NixPath};

use crate::{
    fs::FsFlags,
    hook::{PathArgs, SysArg, SysFlags, UNotifyEventRequest},
    kernel::{syscall_path_handler, to_atflags},
};

pub(crate) fn sys_access(request: UNotifyEventRequest) -> ScmpNotifResp {
    let req = request.scmpreq;

    // SAFETY: Reject undefined/invalid/unused mode.
    let mode = match to_accessflags(req.data.args[1]) {
        Ok(mode) => mode,
        Err(errno) => return request.fail_syscall(errno),
    };

    let argv = &[SysArg {
        path: Some(0),
        ..Default::default()
    }];
    syscall_path_handler(request, "access", argv, |path_args, request, sandbox| {
        drop(sandbox); // release the read-lock.
        syscall_access_handler(request, path_args, mode)
    })
}

pub(crate) fn sys_faccessat(request: UNotifyEventRequest) -> ScmpNotifResp {
    let req = request.scmpreq;

    // SAFETY: Reject undefined/invalid/unused mode.
    let mode = match to_accessflags(req.data.args[2]) {
        Ok(mode) => mode,
        Err(errno) => return request.fail_syscall(errno),
    };

    let argv = &[SysArg {
        dirfd: Some(0),
        path: Some(1),
        ..Default::default()
    }];
    syscall_path_handler(request, "faccessat", argv, |path_args, request, sandbox| {
        drop(sandbox); // release the read-lock.
        syscall_access_handler(request, path_args, mode)
    })
}

pub(crate) fn sys_faccessat2(request: UNotifyEventRequest) -> ScmpNotifResp {
    let req = request.scmpreq;

    // SAFETY: Reject undefined/invalid/unused flags.
    let flags = match to_atflags(
        req.data.args[3],
        AtFlags::AT_EACCESS | AtFlags::AT_EMPTY_PATH | AtFlags::AT_SYMLINK_NOFOLLOW,
    ) {
        Ok(flags) => flags,
        Err(errno) => return request.fail_syscall(errno),
    };

    // SAFETY: Reject undefined/invalid/unused mode.
    let mode = match to_accessflags(req.data.args[2]) {
        Ok(mode) => mode,
        Err(errno) => return request.fail_syscall(errno),
    };

    let fsflags = if flags.contains(AtFlags::AT_SYMLINK_NOFOLLOW) {
        FsFlags::NO_FOLLOW_LAST
    } else {
        FsFlags::empty()
    };
    let argv = &[SysArg {
        dirfd: Some(0),
        path: Some(1),
        flags: if flags.contains(AtFlags::AT_EMPTY_PATH) {
            SysFlags::EMPTY_PATH
        } else {
            SysFlags::empty()
        },
        fsflags,
        ..Default::default()
    }];
    syscall_path_handler(
        request,
        "faccessat2",
        argv,
        |path_args, request, sandbox| {
            drop(sandbox); // release the read-lock.
            syscall_access_handler(request, path_args, mode)
        },
    )
}

/// A helper function to handle access, faccessat, and faccessat2 syscalls.
fn syscall_access_handler(
    request: &UNotifyEventRequest,
    args: PathArgs,
    mode: AccessFlags,
) -> Result<ScmpNotifResp, Errno> {
    // SAFETY: SysArg has one element.
    #[allow(clippy::disallowed_methods)]
    let path = args.0.as_ref().unwrap();

    // SAFETY: Handle base path in a TOCTOU-free way.
    let flags = if path.base.is_empty() {
        libc::AT_EMPTY_PATH
    } else {
        libc::AT_SYMLINK_NOFOLLOW
    };

    let fd = path
        .dir
        .as_ref()
        .map(|fd| fd.as_raw_fd())
        .ok_or(Errno::EBADF)?;

    path.base
        .with_nix_path(|cstr| {
            // SAFETY: No libc wrapper for faccessat2 yet.
            Errno::result(unsafe {
                libc::syscall(
                    libc::SYS_faccessat2,
                    fd.as_raw_fd(),
                    cstr.as_ptr(),
                    mode.bits(),
                    flags,
                )
            })
        })?
        .map(|_| request.return_syscall(0))
}

#[inline]
fn to_accessflags(arg: u64) -> Result<AccessFlags, Errno> {
    let flags = arg.try_into().or(Err(Errno::EINVAL))?;
    AccessFlags::from_bits(flags).ok_or(Errno::EINVAL)
}
