/*
 * Decompiled with CFR 0.152.
 */
package org.openstreetmap.josm.gui.mappaint;

import java.awt.Dimension;
import java.awt.image.BufferedImage;
import java.awt.image.RenderedImage;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.NoSuchFileException;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Locale;
import java.util.Optional;
import java.util.function.DoubleSupplier;
import java.util.logging.Level;
import javax.imageio.ImageIO;
import org.openstreetmap.josm.cli.CLIModule;
import org.openstreetmap.josm.data.Bounds;
import org.openstreetmap.josm.data.ProjectionBounds;
import org.openstreetmap.josm.data.coor.EastNorth;
import org.openstreetmap.josm.data.coor.LatLon;
import org.openstreetmap.josm.data.coor.conversion.LatLonParser;
import org.openstreetmap.josm.data.osm.DataSet;
import org.openstreetmap.josm.data.preferences.JosmBaseDirectories;
import org.openstreetmap.josm.data.preferences.JosmUrls;
import org.openstreetmap.josm.data.projection.Projection;
import org.openstreetmap.josm.data.projection.ProjectionRegistry;
import org.openstreetmap.josm.data.projection.Projections;
import org.openstreetmap.josm.gui.mappaint.RenderingHelper;
import org.openstreetmap.josm.io.Compression;
import org.openstreetmap.josm.io.IllegalDataException;
import org.openstreetmap.josm.io.OsmReader;
import org.openstreetmap.josm.spi.lifecycle.Lifecycle;
import org.openstreetmap.josm.spi.preferences.Config;
import org.openstreetmap.josm.spi.preferences.MemoryPreferences;
import org.openstreetmap.josm.tools.Http1Client;
import org.openstreetmap.josm.tools.HttpClient;
import org.openstreetmap.josm.tools.I18n;
import org.openstreetmap.josm.tools.JosmDecimalFormatSymbolsProvider;
import org.openstreetmap.josm.tools.Logging;
import org.openstreetmap.josm.tools.OptionParser;
import org.openstreetmap.josm.tools.Stopwatch;
import org.openstreetmap.josm.tools.Territories;

public class RenderingCLI
implements CLIModule {
    public static final RenderingCLI INSTANCE = new RenderingCLI();
    private static final double PIXEL_PER_METER = 3779.5275590551178;
    private static final int DEFAULT_MAX_IMAGE_SIZE = 20000;
    private boolean argDebug;
    private boolean argTrace;
    private String argInput;
    private String argOutput;
    private List<RenderingHelper.StyleData> argStyles;
    private Integer argZoom;
    private Double argScale;
    private Bounds argBounds;
    private LatLon argAnchor;
    private Double argWidthM;
    private Double argHeightM;
    private Integer argWidthPx;
    private Integer argHeightPx;
    private String argProjection;
    private Integer argMaxImageSize;
    private RenderingHelper.StyleData argCurrentStyle;

    RenderingCLI() {
    }

    @Override
    public String getActionKeyword() {
        return "render";
    }

    @Override
    public void processArguments(String[] argArray) {
        try {
            this.parseArguments(argArray);
            this.initialize();
            Stopwatch stopwatch = Stopwatch.createStarted();
            String task = I18n.tr("Rendering {0} to {1}", this.argInput, this.argOutput);
            System.err.println(task);
            DataSet ds = this.loadDataset();
            RenderingArea area = this.determineRenderingArea(ds);
            RenderingHelper rh = new RenderingHelper(ds, area.bounds, area.scale, this.argStyles);
            this.checkPreconditions(rh);
            BufferedImage image = rh.render();
            this.writeImageToFile(image);
            System.err.println(stopwatch.toString(task));
        }
        catch (FileNotFoundException | NoSuchFileException e) {
            if (Logging.isDebugEnabled()) {
                e.printStackTrace();
            }
            System.err.println(I18n.tr("Error - file not found: ''{0}''", e.getMessage()));
            Lifecycle.exitJosm(true, 1);
        }
        catch (IOException | IllegalArgumentException | IllegalDataException e) {
            if (Logging.isDebugEnabled()) {
                e.printStackTrace();
            }
            if (e.getMessage() != null) {
                System.err.println(I18n.tr("Error: {0}", e.getMessage()));
            }
            Lifecycle.exitJosm(true, 1);
        }
        Lifecycle.exitJosm(true, 0);
    }

    void parseArguments(String[] argArray) {
        Logging.setLogLevel(Level.INFO);
        OptionParser parser = new OptionParser("JOSM rendering");
        for (Option o : Option.values()) {
            if (o.requiresArgument()) {
                parser.addArgumentParameter(o.getName(), o == Option.SETTING ? OptionParser.OptionCount.MULTIPLE : OptionParser.OptionCount.OPTIONAL, arg -> this.handleOption(o, (String)arg));
            } else {
                parser.addFlagParameter(o.getName(), () -> this.handleOption(o));
            }
            if (o.getShortOption() == '*') continue;
            parser.addShortAlias(o.getName(), Character.toString(o.getShortOption()));
        }
        this.argCurrentStyle = new RenderingHelper.StyleData();
        this.argStyles = new ArrayList<RenderingHelper.StyleData>();
        parser.parseOptionsOrExit(Arrays.asList(argArray));
        if (this.argCurrentStyle.styleUrl != null) {
            this.argStyles.add(this.argCurrentStyle);
        } else if (this.argStyles.isEmpty()) {
            this.argCurrentStyle.styleUrl = "resource://styles/standard/elemstyles.mapcss";
            this.argStyles.add(this.argCurrentStyle);
        }
    }

    private void handleOption(Option o) {
        switch (o) {
            case HELP: {
                RenderingCLI.showHelp();
                Lifecycle.exitJosm(true, 0);
                break;
            }
            case DEBUG: {
                this.argDebug = true;
                break;
            }
            case TRACE: {
                this.argTrace = true;
                break;
            }
            default: {
                throw new AssertionError((Object)("Unexpected option index: " + o));
            }
        }
    }

    private void handleOption(Option o, String arg) {
        switch (o) {
            case INPUT: {
                this.argInput = arg;
                break;
            }
            case STYLE: {
                if (this.argCurrentStyle.styleUrl != null) {
                    this.argStyles.add(this.argCurrentStyle);
                    this.argCurrentStyle = new RenderingHelper.StyleData();
                }
                this.argCurrentStyle.styleUrl = arg;
                break;
            }
            case OUTPUT: {
                this.argOutput = arg;
                break;
            }
            case ZOOM: {
                try {
                    this.argZoom = Integer.valueOf(arg);
                }
                catch (NumberFormatException nfe) {
                    throw new OptionParser.OptionParseException(I18n.tr("Expected integer number for option {0}, but got ''{1}''", "--zoom", arg), nfe);
                }
                if (this.argZoom >= 0) break;
                throw new OptionParser.OptionParseException(I18n.tr("Expected integer number >= 0 for option {0}, but got ''{1}''", "--zoom", arg));
            }
            case BOUNDS: {
                if ("auto".equals(arg)) break;
                try {
                    this.argBounds = new Bounds(arg, ",", Bounds.ParseMethod.LEFT_BOTTOM_RIGHT_TOP, false);
                    break;
                }
                catch (IllegalArgumentException iae) {
                    throw new OptionParser.OptionParseException(I18n.tr("Unable to parse {0} parameter: {1}", "--bounds", iae.getMessage()), iae);
                }
            }
            case SETTING: {
                String keyval = arg;
                String[] comp = keyval.split(":", 2);
                if (comp.length != 2) {
                    throw new OptionParser.OptionParseException(I18n.tr("Expected key and value, separated by '':'' character for option {0}, but got ''{1}''", "--setting", arg));
                }
                this.argCurrentStyle.settings.put(comp[0].trim(), comp[1].trim());
                break;
            }
            case SCALE: {
                try {
                    this.argScale = JosmDecimalFormatSymbolsProvider.parseDouble(arg);
                    break;
                }
                catch (NumberFormatException nfe) {
                    throw new OptionParser.OptionParseException(I18n.tr("Expected floating point number for option {0}, but got ''{1}''", "--scale", arg), nfe);
                }
            }
            case ANCHOR: {
                String[] parts = arg.split(",", -1);
                if (parts.length != 2) {
                    throw new OptionParser.OptionParseException(I18n.tr("Expected two coordinates, separated by comma, for option {0}, but got ''{1}''", "--anchor", arg));
                }
                try {
                    double lon = LatLonParser.parseCoordinate(parts[0]);
                    double lat = LatLonParser.parseCoordinate(parts[1]);
                    this.argAnchor = new LatLon(lat, lon);
                    break;
                }
                catch (IllegalArgumentException iae) {
                    throw new OptionParser.OptionParseException(I18n.tr("In option {0}: {1}", "--anchor", iae.getMessage()), iae);
                }
            }
            case WIDTH_M: {
                try {
                    this.argWidthM = JosmDecimalFormatSymbolsProvider.parseDouble(arg);
                }
                catch (NumberFormatException nfe) {
                    throw new OptionParser.OptionParseException(I18n.tr("Expected floating point number for option {0}, but got ''{1}''", "--width-m", arg), nfe);
                }
                if (!(this.argWidthM <= 0.0)) break;
                throw new OptionParser.OptionParseException(I18n.tr("Expected floating point number > 0 for option {0}, but got ''{1}''", "--width-m", arg));
            }
            case HEIGHT_M: {
                try {
                    this.argHeightM = JosmDecimalFormatSymbolsProvider.parseDouble(arg);
                }
                catch (NumberFormatException nfe) {
                    throw new OptionParser.OptionParseException(I18n.tr("Expected floating point number for option {0}, but got ''{1}''", "--height-m", arg), nfe);
                }
                if (!(this.argHeightM <= 0.0)) break;
                throw new OptionParser.OptionParseException(I18n.tr("Expected floating point number > 0 for option {0}, but got ''{1}''", "--width-m", arg));
            }
            case WIDTH_PX: {
                try {
                    this.argWidthPx = Integer.valueOf(arg);
                }
                catch (NumberFormatException nfe) {
                    throw new OptionParser.OptionParseException(I18n.tr("Expected integer number for option {0}, but got ''{1}''", "--width-px", arg), nfe);
                }
                if (this.argWidthPx > 0) break;
                throw new OptionParser.OptionParseException(I18n.tr("Expected integer number > 0 for option {0}, but got ''{1}''", "--width-px", arg));
            }
            case HEIGHT_PX: {
                try {
                    this.argHeightPx = Integer.valueOf(arg);
                }
                catch (NumberFormatException nfe) {
                    throw new OptionParser.OptionParseException(I18n.tr("Expected integer number for option {0}, but got ''{1}''", "--height-px", arg), nfe);
                }
                if (this.argHeightPx > 0) break;
                throw new OptionParser.OptionParseException(I18n.tr("Expected integer number > 0 for option {0}, but got ''{1}''", "--height-px", arg));
            }
            case PROJECTION: {
                this.argProjection = arg;
                break;
            }
            case MAX_IMAGE_SIZE: {
                try {
                    this.argMaxImageSize = Integer.valueOf(arg);
                }
                catch (NumberFormatException nfe) {
                    throw new OptionParser.OptionParseException(I18n.tr("Expected integer number for option {0}, but got ''{1}''", "--max-image-size", arg), nfe);
                }
                if (this.argMaxImageSize >= 0) break;
                throw new OptionParser.OptionParseException(I18n.tr("Expected integer number >= 0 for option {0}, but got ''{1}''", "--max-image-size", arg));
            }
            default: {
                throw new AssertionError((Object)("Unexpected option index: " + o));
            }
        }
    }

    public static void showHelp() {
        System.out.println(RenderingCLI.getHelp());
    }

    private static String getHelp() {
        return I18n.tr("JOSM rendering command line interface", new Object[0]) + "\n\n" + I18n.tr("Usage", new Object[0]) + ":\n\tjava -jar josm.jar render <options>\n\n" + I18n.tr("Description", new Object[0]) + ":\n" + I18n.tr("Renders data and saves the result to an image file.", new Object[0]) + "\n\n" + I18n.tr("Options", new Object[0]) + ":\n\t--help|-h                 " + I18n.tr("Show this help", new Object[0]) + "\n\t--input|-i <file>         " + I18n.tr("Input data file name (.osm)", new Object[0]) + "\n\t--output|-o <file>        " + I18n.tr("Output image file name (.png); defaults to ''{0}''", "out.png") + "\n\t--style|-s <file>         " + I18n.tr("Style file to use for rendering (.mapcss or .zip)", new Object[0]) + "\n\t                          " + I18n.tr("This option can be repeated to load multiple styles.", new Object[0]) + "\n\t--setting <key>:<value>   " + I18n.tr("Style setting (in JOSM accessible in the style list dialog right click menu)", new Object[0]) + "\n\t                          " + I18n.tr("Applies to the last style loaded with the {0} option.", "--style") + "\n\t--zoom|-z <lvl>           " + I18n.tr("Select zoom level to render. (integer value, 0=entire earth, 18=street level)", new Object[0]) + "\n\t--scale <scale>           " + I18n.tr("Select the map scale", new Object[0]) + "\n\t                          " + I18n.tr("A value of 10000 denotes a scale of 1:10000 (1 cm on the map equals 100 m on the ground; display resolution: 96 dpi)", new Object[0]) + "\n\t                          " + I18n.tr("Options {0} and {1} are mutually exclusive.", "--zoom", "--scale") + "\n\t--bounds|-b auto|<min_lon>,<min_lat>,<max_lon>,<max_lat>\n\t                          " + I18n.tr("Area to render, default value is ''{0}''", "auto") + "\n\t                          " + I18n.tr("With keyword ''{0}'', the downloaded area in the .osm input file will be used (if recorded).", "auto") + "\n\t--anchor <lon>,<lat>      " + I18n.tr("Specify bottom left corner of the rendering area", new Object[0]) + "\n\t                          " + I18n.tr("Used in combination with width and height options to determine the area to render.", new Object[0]) + "\n\t--width-m <number>        " + I18n.tr("Width of the rendered area, in meter", new Object[0]) + "\n\t--height-m <number>       " + I18n.tr("Height of the rendered area, in meter", new Object[0]) + "\n\t--width-px <number>       " + I18n.tr("Width of the target image, in pixel", new Object[0]) + "\n\t--height-px <number>      " + I18n.tr("Height of the target image, in pixel", new Object[0]) + "\n\t--projection <code>       " + I18n.tr("Projection to use, default value ''{0}'' (web-Mercator)", "epsg:3857") + "\n\t--max-image-size <number> " + I18n.tr("Maximum image width/height in pixel (''{0}'' means no limit), default value: {1}", 0, Integer.toString(20000)) + "\n\n" + I18n.tr("To specify the rendered area and scale, the options can be combined in various ways", new Object[0]) + ":\n  * --bounds (--zoom|--scale|--width-px|--height-px)\n  * --anchor (--width-m|--width-px) (--height-m|--height-px) (--zoom|--scale)\n  * --anchor --width-m --height-m (--width-px|--height-px)\n  * --anchor --width-px --height-px (--width-m|--height-m)\n" + I18n.tr("If neither ''{0}'' nor ''{1}'' is given, the default value {2} takes effect and the bounds of the download area in the .osm input file are used.", "bounds", "anchor", "--bounds=auto") + "\n\n" + I18n.tr("Examples", new Object[0]) + ":\n  java -jar josm.jar render -i data.osm -s style.mapcss -z 16\n  josm render -i data.osm -s style.mapcss --scale 5000\n  josm render -i data.osm -s style.mapcss -z 16 -o image.png\n  josm render -i data.osm -s elemstyles.mapcss --setting hide_icons:false -z 16\n  josm render -i data.osm -s style.mapcss -s another_style.mapcss -z 16 -o image.png\n  josm render -i data.osm -s style.mapcss --bounds 21.151,51.401,21.152,51.402 -z 16\n  josm render -i data.osm -s style.mapcss --anchor 21.151,51.401 --width-m 500 --height-m 300 -z 16\n  josm render -i data.osm -s style.mapcss --anchor 21.151,51.401 --width-m 500 --height-m 300 --width-px 1800\n  josm render -i data.osm -s style.mapcss --scale 5000 --projection epsg:4326\n";
    }

    void initialize() {
        Logging.setLogLevel(this.getLogLevel());
        HttpClient.setFactory(Http1Client::new);
        Config.setBaseDirectoriesProvider(JosmBaseDirectories.getInstance());
        Config.setPreferencesInstance(new MemoryPreferences());
        Config.setUrlsProvider(JosmUrls.getInstance());
        Config.getPref().putBoolean("mappaint.auto_reload_local_styles", false);
        String projCode = Optional.ofNullable(this.argProjection).orElse("epsg:3857");
        ProjectionRegistry.setProjection(Projections.getProjectionByCode(projCode.toUpperCase(Locale.US)));
        Territories.initializeInternalData();
    }

    private Level getLogLevel() {
        if (this.argTrace) {
            return Logging.LEVEL_TRACE;
        }
        if (this.argDebug) {
            return Logging.LEVEL_DEBUG;
        }
        return Logging.LEVEL_INFO;
    }

    RenderingArea determineRenderingArea(DataSet ds) {
        Projection proj = ProjectionRegistry.getProjection();
        Double scale = null;
        if (this.argZoom != null) {
            scale = 4.007501668557849E7 / Math.pow(2.0, this.argZoom.intValue()) / 256.0 / proj.getMetersPerUnit();
        }
        Bounds bounds = this.argBounds;
        ProjectionBounds pb = null;
        if (bounds == null) {
            if (this.argAnchor != null) {
                double heightEn;
                double widthEn;
                EastNorth projAnchor = proj.latlon2eastNorth(this.argAnchor);
                double enPerMeter = Double.NaN;
                DoubleSupplier getEnPerMeter = () -> {
                    double shiftMeter = 10.0;
                    EastNorth projAnchorShifted = projAnchor.add(shiftMeter / proj.getMetersPerUnit(), shiftMeter / proj.getMetersPerUnit());
                    LatLon anchorShifted = proj.eastNorth2latlon(projAnchorShifted);
                    return projAnchor.distance(projAnchorShifted) / this.argAnchor.greatCircleDistance(anchorShifted);
                };
                if (scale == null) {
                    if (this.argScale != null) {
                        enPerMeter = getEnPerMeter.getAsDouble();
                        scale = this.argScale * enPerMeter / 3779.5275590551178;
                    } else if (this.argWidthM != null && this.argWidthPx != null) {
                        enPerMeter = getEnPerMeter.getAsDouble();
                        scale = this.argWidthM / (double)this.argWidthPx.intValue() * enPerMeter;
                    } else if (this.argHeightM != null && this.argHeightPx != null) {
                        enPerMeter = getEnPerMeter.getAsDouble();
                        scale = this.argHeightM / (double)this.argHeightPx.intValue() * enPerMeter;
                    } else {
                        throw new IllegalArgumentException(I18n.tr("Argument {0} given, but scale cannot be determined from remaining arguments", "--anchor"));
                    }
                }
                if (this.argWidthM != null) {
                    if (Double.isNaN(enPerMeter)) {
                        enPerMeter = getEnPerMeter.getAsDouble();
                    }
                    widthEn = this.argWidthM * enPerMeter;
                } else if (this.argWidthPx != null) {
                    widthEn = (double)this.argWidthPx.intValue() * scale;
                } else {
                    throw new IllegalArgumentException(I18n.tr("Argument {0} given, expected {1} or {2}", "--anchor", "--width-m", "--width-px"));
                }
                if (this.argHeightM != null) {
                    if (Double.isNaN(enPerMeter)) {
                        enPerMeter = getEnPerMeter.getAsDouble();
                    }
                    heightEn = this.argHeightM * enPerMeter;
                } else if (this.argHeightPx != null) {
                    heightEn = (double)this.argHeightPx.intValue() * scale;
                } else {
                    throw new IllegalArgumentException(I18n.tr("Argument {0} given, expected {1} or {2}", "--anchor", "--height-m", "--height-px"));
                }
                pb = new ProjectionBounds(projAnchor);
                pb.extend(new EastNorth(projAnchor.east() + widthEn, projAnchor.north() + heightEn));
                bounds = new Bounds(proj.eastNorth2latlon(pb.getMin()), false);
                bounds.extend(proj.eastNorth2latlon(pb.getMax()));
            } else {
                if (ds.getDataSourceBounds().isEmpty()) {
                    throw new IllegalArgumentException(I18n.tr("{0} mode, but no bounds found in osm data input file", "--bounds=auto"));
                }
                bounds = ds.getDataSourceBounds().get(0);
            }
        }
        if (pb == null) {
            pb = new ProjectionBounds();
            pb.extend(proj.latlon2eastNorth(bounds.getMin()));
            pb.extend(proj.latlon2eastNorth(bounds.getMax()));
        }
        if (scale == null) {
            if (this.argScale != null) {
                double enPerMeter = pb.getMin().distance(pb.getMax()) / bounds.getMin().greatCircleDistance(bounds.getMax());
                scale = this.argScale * enPerMeter / 3779.5275590551178;
            } else if (this.argWidthPx != null) {
                scale = (pb.maxEast - pb.minEast) / (double)this.argWidthPx.intValue();
            } else if (this.argHeightPx != null) {
                scale = (pb.maxNorth - pb.minNorth) / (double)this.argHeightPx.intValue();
            } else {
                throw new IllegalArgumentException(I18n.tr("Unable to determine scale, one of the options {0}, {1}, {2} or {3} expected", "--zoom", "--scale", "--width-px", "--height-px"));
            }
        }
        RenderingArea ra = new RenderingArea();
        ra.bounds = bounds;
        ra.scale = scale;
        return ra;
    }

    private DataSet loadDataset() throws IOException, IllegalDataException {
        DataSet dataSet;
        block9: {
            if (this.argInput == null) {
                throw new IllegalArgumentException(I18n.tr("Missing argument - input data file ({0})", "--input|-i"));
            }
            InputStream inputStream = Compression.getUncompressedFileInputStream(Paths.get(this.argInput, new String[0]));
            try {
                dataSet = OsmReader.parseDataSet(inputStream, null);
                if (inputStream == null) break block9;
            }
            catch (Throwable throwable) {
                try {
                    if (inputStream != null) {
                        try {
                            inputStream.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (IllegalDataException e) {
                    throw new IllegalDataException(I18n.tr("In .osm data file ''{0}'' - ", this.argInput) + e.getMessage(), e);
                }
            }
            inputStream.close();
        }
        return dataSet;
    }

    private void checkPreconditions(RenderingHelper rh) {
        Dimension imgSize = rh.getImageSize();
        Logging.debug("image size (px): {0}x{1}", imgSize.width, imgSize.height);
        int maxSize = Optional.ofNullable(this.argMaxImageSize).orElse(20000);
        if (maxSize != 0 && (imgSize.width > maxSize || imgSize.height > maxSize)) {
            throw new IllegalArgumentException(I18n.tr("Image dimensions ({0}x{1}) exceeds maximum image size {2} (use option {3} to change limit)", imgSize.width, imgSize.height, maxSize, "--max-image-size"));
        }
    }

    private void writeImageToFile(BufferedImage image) throws IOException {
        String output = Optional.ofNullable(this.argOutput).orElse("out.png");
        ImageIO.write((RenderedImage)image, "png", new File(output));
    }

    static class RenderingArea {
        public Bounds bounds;
        public double scale;

        RenderingArea() {
        }
    }

    private static enum Option {
        HELP(false, 'h'),
        DEBUG(false, '*'),
        TRACE(false, '*'),
        INPUT(true, 'i'),
        STYLE(true, 's'),
        SETTING(true, '*'),
        OUTPUT(true, 'o'),
        ZOOM(true, 'z'),
        SCALE(true, '*'),
        BOUNDS(true, 'b'),
        ANCHOR(true, '*'),
        WIDTH_M(true, '*'),
        HEIGHT_M(true, '*'),
        WIDTH_PX(true, '*'),
        HEIGHT_PX(true, '*'),
        PROJECTION(true, '*'),
        MAX_IMAGE_SIZE(true, '*');

        private final String name = this.name().toLowerCase(Locale.US).replace('_', '-');
        private final boolean requiresArg;
        private final char shortOption;

        private Option(boolean requiresArgument, char shortOption) {
            this.requiresArg = requiresArgument;
            this.shortOption = shortOption;
        }

        public String getName() {
            return this.name;
        }

        public boolean requiresArgument() {
            return this.requiresArg;
        }

        public char getShortOption() {
            return this.shortOption;
        }
    }
}

