/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.watcher;

import java.io.IOException;
import java.io.InputStream;
import java.nio.file.DirectoryStream;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.Arrays;
import java.util.stream.StreamSupport;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.common.Strings;
import org.elasticsearch.common.hash.MessageDigests;
import org.elasticsearch.common.util.CollectionUtils;
import org.elasticsearch.watcher.AbstractResourceWatcher;
import org.elasticsearch.watcher.FileChangesListener;

public class FileWatcher
extends AbstractResourceWatcher<FileChangesListener> {
    private FileObserver rootFileObserver;
    private final Path path;
    private final boolean checkFileContents;
    private static final Logger logger = LogManager.getLogger(FileWatcher.class);
    private static final Observer[] EMPTY_DIRECTORY = new Observer[0];

    public FileWatcher(Path path) {
        this(path, false);
    }

    public FileWatcher(Path path, boolean checkFileContents) {
        this.path = path;
        this.checkFileContents = checkFileContents;
        this.rootFileObserver = new FileObserver(path);
    }

    public void clearState() {
        this.rootFileObserver = new FileObserver(this.path);
        try {
            this.rootFileObserver.init(false);
        }
        catch (IOException iOException) {
            // empty catch block
        }
    }

    @Override
    protected void doInit() throws IOException {
        this.rootFileObserver.init(true);
    }

    @Override
    protected void doCheckAndNotify() throws IOException {
        this.rootFileObserver.checkAndNotify();
    }

    protected boolean fileExists(Path path) {
        return Files.exists(path, new LinkOption[0]);
    }

    protected BasicFileAttributes readAttributes(Path path) throws IOException {
        return Files.readAttributes(path, BasicFileAttributes.class, new LinkOption[0]);
    }

    protected InputStream newInputStream(Path path) throws IOException {
        return Files.newInputStream(path, new OpenOption[0]);
    }

    protected DirectoryStream<Path> listFiles(Path path) throws IOException {
        return Files.newDirectoryStream(path);
    }

    private class FileObserver
    extends Observer {
        private long length;
        private long lastModified;
        private Observer[] children;
        private byte[] digest;

        FileObserver(Path path) {
            super(path);
        }

        @Override
        public void checkAndNotify() throws IOException {
            boolean prevExists = this.exists;
            boolean prevIsDirectory = this.isDirectory;
            long prevLength = this.length;
            long prevLastModified = this.lastModified;
            byte[] prevDigest = this.digest;
            this.exists = FileWatcher.this.fileExists(this.path);
            if (this.exists) {
                BasicFileAttributes attributes = FileWatcher.this.readAttributes(this.path);
                this.isDirectory = attributes.isDirectory();
                if (this.isDirectory) {
                    this.length = 0L;
                    this.lastModified = 0L;
                } else {
                    this.length = attributes.size();
                    this.lastModified = attributes.lastModifiedTime().toMillis();
                }
            } else {
                this.isDirectory = false;
                this.length = 0L;
                this.lastModified = 0L;
            }
            if (prevExists) {
                if (this.exists) {
                    if (this.isDirectory) {
                        if (prevIsDirectory) {
                            this.updateChildren();
                        } else {
                            this.onFileDeleted();
                            this.onDirectoryCreated(false);
                        }
                    } else if (prevIsDirectory) {
                        this.onDirectoryDeleted();
                        this.onFileCreated(false);
                    } else if (prevLastModified != this.lastModified || prevLength != this.length) {
                        if (FileWatcher.this.checkFileContents) {
                            this.digest = this.calculateDigest();
                            if (this.digest == null || !Arrays.equals(prevDigest, this.digest)) {
                                this.onFileChanged();
                            }
                        } else {
                            this.onFileChanged();
                        }
                    }
                } else if (prevIsDirectory) {
                    this.onDirectoryDeleted();
                } else {
                    this.onFileDeleted();
                }
            } else if (this.exists) {
                if (this.isDirectory) {
                    this.onDirectoryCreated(false);
                } else {
                    this.onFileCreated(false);
                }
            }
        }

        private byte[] calculateDigest() {
            byte[] byArray;
            block8: {
                InputStream in = FileWatcher.this.newInputStream(this.path);
                try {
                    byArray = MessageDigests.digest(in, MessageDigests.md5());
                    if (in == null) break block8;
                }
                catch (Throwable throwable) {
                    try {
                        if (in != null) {
                            try {
                                in.close();
                            }
                            catch (Throwable throwable2) {
                                throwable.addSuppressed(throwable2);
                            }
                        }
                        throw throwable;
                    }
                    catch (IOException e) {
                        logger.warn("failed to read file [{}] while checking for file changes [{}], will assuming file has been modified", (Object)this.path, (Object)e.toString());
                        return null;
                    }
                }
                in.close();
            }
            return byArray;
        }

        private void init(boolean initial) throws IOException {
            this.exists = FileWatcher.this.fileExists(this.path);
            if (this.exists) {
                BasicFileAttributes attributes = FileWatcher.this.readAttributes(this.path);
                this.isDirectory = attributes.isDirectory();
                if (this.isDirectory) {
                    this.onDirectoryCreated(initial);
                } else {
                    this.length = attributes.size();
                    this.lastModified = attributes.lastModifiedTime().toMillis();
                    if (FileWatcher.this.checkFileContents) {
                        this.digest = this.calculateDigest();
                    }
                    this.onFileCreated(initial);
                }
            }
        }

        private Observer createChild(Path file, boolean initial) throws IOException {
            try {
                FileObserver child = new FileObserver(file);
                child.init(initial);
                return child;
            }
            catch (SecurityException e) {
                logger.debug(() -> Strings.format("Don't have permissions to watch path [%s]", file), (Throwable)e);
                return new DeniedObserver(file);
            }
        }

        private Path[] listFiles() throws IOException {
            try (DirectoryStream<Path> dirs = FileWatcher.this.listFiles(this.path);){
                Path[] pathArray = (Path[])StreamSupport.stream(dirs.spliterator(), false).sorted().toArray(Path[]::new);
                return pathArray;
            }
        }

        private Observer[] listChildren(boolean initial) throws IOException {
            Object[] files = this.listFiles();
            if (!CollectionUtils.isEmpty(files)) {
                Observer[] childObservers = new Observer[files.length];
                for (int i = 0; i < files.length; ++i) {
                    childObservers[i] = this.createChild((Path)files[i], initial);
                }
                return childObservers;
            }
            return EMPTY_DIRECTORY;
        }

        private void updateChildren() throws IOException {
            Object[] files = this.listFiles();
            if (!CollectionUtils.isEmpty(files)) {
                Observer[] newChildren = new Observer[files.length];
                int child = 0;
                int file = 0;
                while (file < files.length || child < this.children.length) {
                    int compare = file >= files.length ? -1 : (child >= this.children.length ? 1 : this.children[child].path.compareTo((Path)files[file]));
                    if (compare == 0) {
                        this.children[child].checkAndNotify();
                        newChildren[file] = this.children[child];
                        ++file;
                        ++child;
                        continue;
                    }
                    if (compare > 0) {
                        newChildren[file] = this.createChild((Path)files[file], false);
                        ++file;
                        continue;
                    }
                    this.deleteChild(child);
                    ++child;
                }
                this.children = newChildren;
            } else {
                for (int child = 0; child < this.children.length; ++child) {
                    this.deleteChild(child);
                }
                this.children = EMPTY_DIRECTORY;
            }
        }

        private void deleteChild(int child) {
            if (this.children[child].exists) {
                if (this.children[child].isDirectory) {
                    this.children[child].onDirectoryDeleted();
                } else {
                    this.children[child].onFileDeleted();
                }
            }
        }

        private void onFileCreated(boolean initial) {
            for (FileChangesListener listener : FileWatcher.this.listeners()) {
                try {
                    if (initial) {
                        listener.onFileInit(this.path);
                        continue;
                    }
                    listener.onFileCreated(this.path);
                }
                catch (Exception e) {
                    logger.warn("cannot notify file changes listener", (Throwable)e);
                }
            }
        }

        @Override
        void onFileDeleted() {
            for (FileChangesListener listener : FileWatcher.this.listeners()) {
                try {
                    listener.onFileDeleted(this.path);
                }
                catch (Exception e) {
                    logger.warn("cannot notify file changes listener", (Throwable)e);
                }
            }
        }

        private void onFileChanged() {
            for (FileChangesListener listener : FileWatcher.this.listeners()) {
                try {
                    listener.onFileChanged(this.path);
                }
                catch (Exception e) {
                    logger.warn("cannot notify file changes listener", (Throwable)e);
                }
            }
        }

        private void onDirectoryCreated(boolean initial) throws IOException {
            for (FileChangesListener listener : FileWatcher.this.listeners()) {
                try {
                    if (initial) {
                        listener.onDirectoryInit(this.path);
                        continue;
                    }
                    listener.onDirectoryCreated(this.path);
                }
                catch (Exception e) {
                    logger.warn("cannot notify file changes listener", (Throwable)e);
                }
            }
            this.children = this.listChildren(initial);
        }

        @Override
        void onDirectoryDeleted() {
            for (int child = 0; child < this.children.length; ++child) {
                this.deleteChild(child);
            }
            for (FileChangesListener listener : FileWatcher.this.listeners()) {
                try {
                    listener.onDirectoryDeleted(this.path);
                }
                catch (Exception e) {
                    logger.warn("cannot notify file changes listener", (Throwable)e);
                }
            }
        }
    }

    private static abstract class Observer {
        final Path path;
        boolean exists;
        boolean isDirectory;

        private Observer(Path path) {
            this.path = path;
        }

        abstract void checkAndNotify() throws IOException;

        abstract void onDirectoryDeleted();

        abstract void onFileDeleted();
    }

    private static class DeniedObserver
    extends Observer {
        private DeniedObserver(Path path) {
            super(path);
        }

        @Override
        void checkAndNotify() throws IOException {
        }

        @Override
        void onDirectoryDeleted() {
        }

        @Override
        void onFileDeleted() {
        }
    }
}

