/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.cluster.routing;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Predicate;
import java.util.stream.Stream;
import org.elasticsearch.cluster.node.DiscoveryNode;
import org.elasticsearch.cluster.routing.ShardRouting;
import org.elasticsearch.cluster.routing.ShardRoutingState;
import org.elasticsearch.common.util.Maps;
import org.elasticsearch.core.Nullable;
import org.elasticsearch.index.Index;
import org.elasticsearch.index.shard.ShardId;

public class RoutingNode
implements Iterable<ShardRouting> {
    private final String nodeId;
    @Nullable
    private final DiscoveryNode node;
    private final LinkedHashMap<ShardId, ShardRouting> shards;
    private final LinkedHashSet<ShardRouting> initializingShards;
    private final LinkedHashSet<ShardRouting> relocatingShards;
    private final LinkedHashSet<ShardRouting> startedShards;
    private final Map<Index, Set<ShardRouting>> shardsByIndex;
    private static final ShardRouting[] EMPTY_SHARD_ROUTING_ARRAY = new ShardRouting[0];

    RoutingNode(String nodeId, @Nullable DiscoveryNode node, int sizeGuess) {
        this.nodeId = nodeId;
        this.node = node;
        this.shards = Maps.newLinkedHashMapWithExpectedSize(sizeGuess);
        this.relocatingShards = new LinkedHashSet();
        this.initializingShards = new LinkedHashSet();
        this.startedShards = new LinkedHashSet();
        this.shardsByIndex = Maps.newHashMapWithExpectedSize(sizeGuess);
        assert (this.invariant());
    }

    private RoutingNode(RoutingNode original) {
        this.nodeId = original.nodeId;
        this.node = original.node;
        this.shards = new LinkedHashMap<ShardId, ShardRouting>(original.shards);
        this.relocatingShards = new LinkedHashSet<ShardRouting>(original.relocatingShards);
        this.initializingShards = new LinkedHashSet<ShardRouting>(original.initializingShards);
        this.startedShards = new LinkedHashSet<ShardRouting>(original.startedShards);
        this.shardsByIndex = Maps.copyOf(original.shardsByIndex, HashSet::new);
        assert (this.invariant());
    }

    RoutingNode copy() {
        return new RoutingNode(this);
    }

    @Override
    public Iterator<ShardRouting> iterator() {
        return Collections.unmodifiableCollection(this.shards.values()).iterator();
    }

    @Nullable
    public DiscoveryNode node() {
        return this.node;
    }

    @Nullable
    public ShardRouting getByShardId(ShardId id) {
        return this.shards.get(id);
    }

    public boolean hasIndex(Index index) {
        return this.shardsByIndex.containsKey(index);
    }

    public String nodeId() {
        return this.nodeId;
    }

    public int size() {
        return this.shards.size();
    }

    void add(ShardRouting shard) {
        this.addInternal(shard, true);
    }

    void addWithoutValidation(ShardRouting shard) {
        this.addInternal(shard, false);
    }

    private void addInternal(ShardRouting shard, boolean validate) {
        ShardRouting existing = this.shards.putIfAbsent(shard.shardId(), shard);
        if (existing != null) {
            IllegalStateException e = new IllegalStateException("Trying to add a shard " + String.valueOf(shard.shardId()) + " to a node [" + this.nodeId + "] where it already exists. current [" + String.valueOf(this.shards.get(shard.shardId())) + "]. new [" + String.valueOf(shard) + "]");
            assert (false) : e;
            throw e;
        }
        if (shard.initializing()) {
            this.initializingShards.add(shard);
        } else if (shard.relocating()) {
            this.relocatingShards.add(shard);
        } else if (shard.started()) {
            this.startedShards.add(shard);
        }
        this.shardsByIndex.computeIfAbsent(shard.index(), k -> new HashSet()).add(shard);
        assert (!validate || this.invariant());
    }

    void update(ShardRouting oldShard, ShardRouting newShard) {
        if (!this.shards.containsKey(oldShard.shardId())) {
            return;
        }
        ShardRouting previousValue = this.shards.put(newShard.shardId(), newShard);
        assert (previousValue == oldShard) : "expected shard " + String.valueOf(previousValue) + " but was " + String.valueOf(oldShard);
        if (oldShard.initializing()) {
            exist = this.initializingShards.remove(oldShard);
            assert (exist) : "expected shard " + String.valueOf(oldShard) + " to exist in initializingShards";
        } else if (oldShard.relocating()) {
            exist = this.relocatingShards.remove(oldShard);
            assert (exist) : "expected shard " + String.valueOf(oldShard) + " to exist in relocatingShards";
        } else if (oldShard.started()) {
            exist = this.startedShards.remove(oldShard);
            assert (exist) : "expected shard " + String.valueOf(oldShard) + " to exist in startedShards";
        }
        Set<ShardRouting> byIndex = this.shardsByIndex.get(oldShard.index());
        byIndex.remove(oldShard);
        byIndex.add(newShard);
        if (newShard.initializing()) {
            this.initializingShards.add(newShard);
        } else if (newShard.relocating()) {
            this.relocatingShards.add(newShard);
        } else if (newShard.started()) {
            this.startedShards.add(newShard);
        }
        assert (this.invariant());
    }

    void remove(ShardRouting shard) {
        ShardRouting previousValue = (ShardRouting)this.shards.remove(shard.shardId());
        assert (previousValue == shard) : "expected shard " + String.valueOf(previousValue) + " but was " + String.valueOf(shard);
        if (shard.initializing()) {
            exist = this.initializingShards.remove(shard);
            assert (exist) : "expected shard " + String.valueOf(shard) + " to exist in initializingShards";
        } else if (shard.relocating()) {
            exist = this.relocatingShards.remove(shard);
            assert (exist) : "expected shard " + String.valueOf(shard) + " to exist in relocatingShards";
        } else if (shard.started()) {
            exist = this.startedShards.remove(shard);
            assert (exist) : "expected shard " + String.valueOf(shard) + " to exist in startedShards";
        }
        Set<ShardRouting> byIndex = this.shardsByIndex.get(shard.index());
        byIndex.remove(shard);
        if (byIndex.isEmpty()) {
            this.shardsByIndex.remove(shard.index());
        }
        assert (this.invariant());
    }

    public ShardRouting[] initializing() {
        return this.initializingShards.toArray(EMPTY_SHARD_ROUTING_ARRAY);
    }

    public ShardRouting[] relocating() {
        return this.relocatingShards.toArray(EMPTY_SHARD_ROUTING_ARRAY);
    }

    public ShardRouting[] started() {
        return this.startedShards.toArray(EMPTY_SHARD_ROUTING_ARRAY);
    }

    public int numberOfShardsWithState(ShardRoutingState state) {
        return this.internalGetShardsWithState(state).size();
    }

    public Stream<ShardRouting> shardsWithState(ShardRoutingState state) {
        return this.internalGetShardsWithState(state).stream();
    }

    public Stream<ShardRouting> shardsWithState(String index, ShardRoutingState ... states) {
        return Stream.of(states).flatMap(state -> this.shardsWithState(index, (ShardRoutingState)((Object)state)));
    }

    public Stream<ShardRouting> shardsWithState(String index, ShardRoutingState state) {
        return this.shardsWithState(state).filter(shardRouting -> Objects.equals(shardRouting.getIndexName(), index));
    }

    private LinkedHashSet<ShardRouting> internalGetShardsWithState(ShardRoutingState state) {
        return switch (state) {
            default -> throw new IncompatibleClassChangeError();
            case ShardRoutingState.UNASSIGNED -> throw new IllegalArgumentException("Unassigned shards are not linked to a routing node");
            case ShardRoutingState.INITIALIZING -> this.initializingShards;
            case ShardRoutingState.STARTED -> this.startedShards;
            case ShardRoutingState.RELOCATING -> this.relocatingShards;
        };
    }

    public int numberOfOwningShards() {
        return this.shards.size() - this.relocatingShards.size();
    }

    public int numberOfOwningShardsForIndex(Index index) {
        Set<ShardRouting> shardRoutings = this.shardsByIndex.get(index);
        if (shardRoutings == null) {
            return 0;
        }
        return Math.toIntExact(shardRoutings.stream().filter(Predicate.not(ShardRouting::relocating)).count());
    }

    public String prettyPrint() {
        StringBuilder sb = new StringBuilder();
        sb.append("-----node_id[").append(this.nodeId).append("][").append(this.node == null ? "X" : "V").append("]\n");
        for (ShardRouting entry : this.shards.values()) {
            sb.append("--------").append(entry.shortSummary()).append('\n');
        }
        return sb.toString();
    }

    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("routingNode ([");
        if (this.node != null) {
            sb.append(this.node.getName());
            sb.append("][");
            sb.append(this.node.getId());
            sb.append("][");
            sb.append(this.node.getHostName());
            sb.append("][");
            sb.append(this.node.getHostAddress());
        } else {
            sb.append("null");
        }
        sb.append("], [");
        sb.append(this.shards.size());
        sb.append(" assigned shards])");
        return sb.toString();
    }

    public ShardRouting[] copyShards() {
        return this.shards.values().toArray(EMPTY_SHARD_ROUTING_ARRAY);
    }

    public Index[] copyIndices() {
        return this.shardsByIndex.keySet().toArray(Index.EMPTY_ARRAY);
    }

    public boolean isEmpty() {
        return this.shards.isEmpty();
    }

    boolean invariant() {
        ArrayList<ShardRouting> shardRoutingsInitializing = new ArrayList<ShardRouting>(this.shards.size());
        ArrayList<ShardRouting> shardRoutingsRelocating = new ArrayList<ShardRouting>(this.shards.size());
        ArrayList<ShardRouting> shardRoutingsStarted = new ArrayList<ShardRouting>(this.shards.size());
        Map<Index, Set> shardRoutingsByIndex = Maps.newHashMapWithExpectedSize(this.shards.size());
        for (ShardRouting shard : this.shards.values()) {
            switch (shard.state()) {
                case INITIALIZING: {
                    shardRoutingsInitializing.add(shard);
                    break;
                }
                case RELOCATING: {
                    shardRoutingsRelocating.add(shard);
                    break;
                }
                case STARTED: {
                    shardRoutingsStarted.add(shard);
                }
            }
            shardRoutingsByIndex.computeIfAbsent(shard.index(), k -> new HashSet(10)).add(shard);
        }
        assert (this.initializingShards.size() == shardRoutingsInitializing.size() && this.initializingShards.containsAll(shardRoutingsInitializing));
        assert (this.relocatingShards.size() == shardRoutingsRelocating.size() && this.relocatingShards.containsAll(shardRoutingsRelocating));
        assert (this.startedShards.size() == shardRoutingsStarted.size() && this.startedShards.containsAll(shardRoutingsStarted));
        assert (shardRoutingsByIndex.equals(this.shardsByIndex));
        return true;
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        RoutingNode that = (RoutingNode)o;
        return this.nodeId.equals(that.nodeId) && Objects.equals(this.node, that.node) && this.shards.equals(that.shards);
    }

    public int hashCode() {
        return Objects.hash(this.nodeId, this.node, this.shards);
    }
}

