/**
 * Copyright (c) 2025, Fabian Groffen. All rights reserved.
 *
 * See LICENSE for the license.
 */

#include <ldns/ldns.h>

#include "util.h"

const char *FGCOLOUR_RED;
const char *FGCOLOUR_GREEN;
const char *FGCOLOUR_YELLOW;
const char *FGCOLOUR_BLUE;
const char *FGCOLOUR_MAGENTA;
const char *FGCOLOUR_CYAN;
const char *FGCOLOUR_NORM;

const char *FGCOLOUR_BRRED;
const char *FGCOLOUR_BRGREEN;
const char *FGCOLOUR_BRYELLOW;
const char *FGCOLOUR_BRBLUE;
const char *FGCOLOUR_BRMAGENTA;
const char *FGCOLOUR_BRCYAN;

void
util_enable_colours
(
    bool enable
)
{
    if (enable)
    {
        FGCOLOUR_RED       = "\x1b[0;31m";
        FGCOLOUR_GREEN     = "\x1b[0;32m";
        FGCOLOUR_YELLOW    = "\x1b[0;33m";
        FGCOLOUR_BLUE      = "\x1b[0;34m";
        FGCOLOUR_MAGENTA   = "\x1b[0;35m";
        FGCOLOUR_CYAN      = "\x1b[0;36m";
        FGCOLOUR_NORM      = "\x1b[0;0m";

        FGCOLOUR_BRRED     = "\x1b[1;31m";
        FGCOLOUR_BRGREEN   = "\x1b[1;32m";
        FGCOLOUR_BRYELLOW  = "\x1b[1;33m";
        FGCOLOUR_BRBLUE    = "\x1b[1;34m";
        FGCOLOUR_BRMAGENTA = "\x1b[1;35m";
        FGCOLOUR_BRCYAN    = "\x1b[1;36m";
    }
    else
    {
        FGCOLOUR_RED       = "";
        FGCOLOUR_GREEN     = "";
        FGCOLOUR_YELLOW    = "";
        FGCOLOUR_BLUE      = "";
        FGCOLOUR_MAGENTA   = "";
        FGCOLOUR_CYAN      = "";
        FGCOLOUR_NORM      = "";

        FGCOLOUR_BRRED     = "";
        FGCOLOUR_BRGREEN   = "";
        FGCOLOUR_BRYELLOW  = "";
        FGCOLOUR_BRBLUE    = "";
        FGCOLOUR_BRMAGENTA = "";
        FGCOLOUR_BRCYAN    = "";
    }
}

/**
 * Convert an IP-address into a reversed arpa. domain name.
 * Frees input when an address, leaves input as is otherwise.  Returns
 * the created reversed address, or the original input.
 */
ldns_rdf *
util_addr2dname
(
    ldns_rdf *input,
    bool      useip6intdomain
)
{
    ldns_rdf *ret;
    uint8_t  *btes;
    size_t    btelen;

    /* NOTE: ldns_rdf_address_reverse should/could work for this when
     * useip6intdomain is false, but it seems its implementation is very
     * complex using many mallocs/frees, and using loops for ipv6
     * instead of the simple unroll done here */

    if (ldns_rdf2wire(&btes, input, &btelen) != LDNS_STATUS_OK)
        return input;

    ret = input;
    switch (ldns_rdf_get_type(input))
    {
        case LDNS_RDF_TYPE_A:
            if (btelen == 4)
            {
                uint8_t *ipdata = btes;
                char     iprev[255 + 1];

                snprintf(iprev, sizeof(iprev), "%u.%u.%u.%u.in-addr.arpa.",
                         ipdata[3], ipdata[2], ipdata[1], ipdata[0]);
                ret = ldns_dname_new_frm_str(iprev);
                ldns_rdf_deep_free(input);
            }
            break;
        case LDNS_RDF_TYPE_AAAA:
            if (btelen == 16)
            {
                uint8_t *ipdata = btes;
                char     iprev[255 + 1];

                snprintf(iprev, sizeof(iprev),
                         "%c.%c.%c.%c." "%c.%c.%c.%c."
                         "%c.%c.%c.%c." "%c.%c.%c.%c."
                         "%c.%c.%c.%c." "%c.%c.%c.%c."
                         "%c.%c.%c.%c." "%c.%c.%c.%c."
                         "ip6.%s.",
                         ldns_int_to_hexdigit(ipdata[15] & 0xF),
                         ldns_int_to_hexdigit(ipdata[15] >> 4),
                         ldns_int_to_hexdigit(ipdata[14] & 0xF),
                         ldns_int_to_hexdigit(ipdata[14] >> 4),
                         ldns_int_to_hexdigit(ipdata[13] & 0xF),
                         ldns_int_to_hexdigit(ipdata[13] >> 4),
                         ldns_int_to_hexdigit(ipdata[12] & 0xF),
                         ldns_int_to_hexdigit(ipdata[12] >> 4),

                         ldns_int_to_hexdigit(ipdata[11] & 0xF),
                         ldns_int_to_hexdigit(ipdata[11] >> 4),
                         ldns_int_to_hexdigit(ipdata[10] & 0xF),
                         ldns_int_to_hexdigit(ipdata[10] >> 4),
                         ldns_int_to_hexdigit(ipdata[ 9] & 0xF),
                         ldns_int_to_hexdigit(ipdata[ 9] >> 4),
                         ldns_int_to_hexdigit(ipdata[ 8] & 0xF),
                         ldns_int_to_hexdigit(ipdata[ 8] >> 4),

                         ldns_int_to_hexdigit(ipdata[ 7] & 0xF),
                         ldns_int_to_hexdigit(ipdata[ 7] >> 4),
                         ldns_int_to_hexdigit(ipdata[ 6] & 0xF),
                         ldns_int_to_hexdigit(ipdata[ 6] >> 4),
                         ldns_int_to_hexdigit(ipdata[ 5] & 0xF),
                         ldns_int_to_hexdigit(ipdata[ 5] >> 4),
                         ldns_int_to_hexdigit(ipdata[ 4] & 0xF),
                         ldns_int_to_hexdigit(ipdata[ 4] >> 4),

                         ldns_int_to_hexdigit(ipdata[ 3] & 0xF),
                         ldns_int_to_hexdigit(ipdata[ 3] >> 4),
                         ldns_int_to_hexdigit(ipdata[ 2] & 0xF),
                         ldns_int_to_hexdigit(ipdata[ 2] >> 4),
                         ldns_int_to_hexdigit(ipdata[ 1] & 0xF),
                         ldns_int_to_hexdigit(ipdata[ 1] >> 4),
                         ldns_int_to_hexdigit(ipdata[ 0] & 0xF),
                         ldns_int_to_hexdigit(ipdata[ 0] >> 4),

                         useip6intdomain ? "int" : "arpa");
                ret = ldns_dname_new_frm_str(iprev);
                ldns_rdf_deep_free(input);
            }
            break;
        default:
            /* nothing, keep original input */
            break;
    }

    free(btes);

    return ret;
}

/**
 * Create a dname from the input string, respecting ndots count.  The
 * result of this function is an rdf structure setup with absolute or
 * relative dname based on the input and ndots.
 */
ldns_rdf *
util_dname_frm_str
(
    const char *input,
    uint8_t     ndots,
    bool        useip6intdomain
)
{
    ldns_rdf      *dname;
    const char    *domstr;
    uint8_t        dots   = 0;

    if (ldns_str2rdf_a(&dname, input) == LDNS_STATUS_OK ||
        ldns_str2rdf_aaaa(&dname, input) == LDNS_STATUS_OK)
    {
        dname = util_addr2dname(dname, useip6intdomain);
        ndots = 0;  /* always absolute */
    }
    else
    {
        dname = ldns_dname_new_frm_str(input);
    }

    /* get the number of dots in the input, for ndots evaluation */
    for (domstr = input; dots >= ndots && *domstr != '\0'; domstr++)
    {
        if (*domstr == '.')
            dots++;
    }

    /* ldns_dname_new_frm_str turns the input into an absolute
     * domain always (currently), so if the input has less dots than
     * ndots, drop the trailing dot that makes it absolute */
    if (dots < ndots &&
        ldns_dname_absolute(dname) /* in case it changes in the future */)
    {
        ldns_rdf_set_size(dname, ldns_rdf_size(dname) - 1);
    }

    return dname;
}

/**
 * Turn input string into a list of rdf structures containing addresses.
 * The list is NULL-terminated.  When the input is an address (IPv4 or
 * IPv6) the result is a single rdf structure for the address.
 * Otherwise the input is resolved using res, and all found A and AAAA
 * answers are returned from this function.  When nothing could be
 * resolved, NULL is returned.  An empty list, is not a possible return
 * from this function.
 */
ldns_rdf **
util_addr_frm_str
(
    ldns_resolver *res,
    const char    *input,
    uint8_t        ndots
)
{
    ldns_rdf   **ret;
    ldns_rdf    *dns;

    if (ldns_str2rdf_a(&dns, input) == LDNS_STATUS_OK ||
        ldns_str2rdf_aaaa(&dns, input) == LDNS_STATUS_OK)
    {
        ret = calloc(sizeof(ret[0]), 2);
        ret[0] = dns;
    }
    else
    {
        ldns_pkt      *pkt;
        ldns_rr_list  *ans;
        ldns_rr_type  *tpe;
        ldns_rr_type   types[] = {
            LDNS_RR_TYPE_A,
            LDNS_RR_TYPE_AAAA,
            (ldns_rr_type)0
        };
        size_t         retlen = 0;  /* empty */

        dns = util_dname_frm_str(input, ndots, false);
        ret = NULL;

        /* ok, need to resolve using res */
        for (tpe = types; *tpe != (ldns_rr_type)0; tpe++)
        {
            size_t i;

            pkt = ldns_resolver_search(res, dns,
                                       *tpe, LDNS_RR_CLASS_IN, LDNS_RD);
            if (pkt == NULL)
                continue;
            ans = ldns_pkt_rr_list_by_type(pkt, *tpe, LDNS_SECTION_ANSWER);
            if (ans == NULL)
                continue;

            ret = realloc(ret,
                          sizeof(ret[0]) * (retlen + 1 +
                                            ldns_rr_list_rr_count(ans)));
            for (i = 0; i < ldns_rr_list_rr_count(ans); i++)
            {
                ldns_rr      *rr   = ldns_rr_list_rr(ans, i);
                ldns_rr_type  rtpe = ldns_rr_get_type(rr);

                if (rtpe != *tpe)
                    continue;

                ret[retlen++] = ldns_rdf_clone(ldns_rr_rdf(rr, 0));
            }

            ldns_rr_list_deep_free(ans);
        }

        if (retlen == 0)
            return NULL;

        ret[retlen] = NULL;
    }

    return ret;
}

/* vim: set ts=4 sw=4 expandtab cinoptions=(0,u0,U1,W2s,l1: */
